/*
 * Copyright (c) 1997 Erez Zadok
 * Copyright (c) 1989 Jan-Simon Pendry
 * Copyright (c) 1989 Imperial College of Science, Technology & Medicine
 * Copyright (c) 1989 The Regents of the University of California.
 * All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Jan-Simon Pendry at Imperial College, London.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      %W% (Berkeley) %G%
 *
 * $Id: info_hes.c,v 5.2.2.1 1992/02/09 15:08:29 jsp beta $
 *
 */

/*
 * Get info from Hesiod
 *
 * Zone transfer code from Bruce Cole <cole@cs.wisc.edu>
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <am_defs.h>
#include <amd.h>

/* these externs come from <hesiod.h> and are not needed here! */
extern char **hes_resolve(const char *name, const char *type);
extern char **hesiod_resolve(void *context, const char *name, const char *type);
extern char *hes_to_bind(const char *name, const char *type);
extern char *hesiod_to_bind(void *context, const char *name, const char *type);
extern int hes_error(void);

#define	HES_PREFIX	"hesiod."
#define	HES_PREFLEN	7

/*
 * Patch up broken system include files
 */
# ifndef C_HS
#  define C_HS	4
# endif /* not C_HS */
# ifndef T_TXT
#  define T_TXT	16
# endif /* T_TXT */

/*
 * Zone transfer...
 */
# define MAXHSNS 8
# define MAX_NSADDR 16
/* How long to wait before timing out zone transfer (secs) */
# define TCP_RETRANS 60

# define hs_server_addr(ns) ((struct in_addr *) nsaddr_list[ns])
# define MAXDATA 8*1024

static char *hs_domain;
static char nsaddr_list[MAX_NSADDR][sizeof(struct in_addr)];
static int hs_nscount;
static int servernum;
static int soacnt;
static mnt_map *hs_map;
static struct timeval hs_timeout;
static voidp hesiod_context;

/* forward declarations */
static int hs_get_ns_list(char *domain);
static int hs_parse(char *msg, char *eom);
static int hs_readresp(int s, char *answer, int anslen);
static int hs_res_selwait(int sock, struct timeval *timeout);
static int hs_res_send(char *buf, int buflen, char *answer, int anslen);
static int hs_res_vcread(int sock, int buflen, char *buf, struct timeval *timeout);
static int hs_strip_our_domain(char *name);
static int hs_zone_transfer(char *domain);
static char *hs_make_value(u_char *cp, int len);


/*
 * No easy way to probe the server - check the map name begins with "hesiod."
 * Note: this name includes 'amu_' so as to not conflict with libhesiod's
 * hesiod_init() function.
 */
int
amu_hesiod_init(mnt_map *, char *map, time_t *tp)
{
#ifdef DEBUG
  dlog("amu_hesiod_init(%s)", map);
#endif /* DEBUG */
  *tp = 0;

  if(!hesiod_context && hesiod_init(&hesiod_context) != 0)
    return ENOENT;

  return strncmp(map, HES_PREFIX, HES_PREFLEN) == 0 ? 0 : ENOENT;
}


/*
 * Do a Hesiod nameserver call.
 * Modify time is ignored by Hesiod - XXX
 */
int
hesiod_search(mnt_map *m, char *map, char *key, char **pval, time_t *tp)
{
  int error;
  char hes_key[MAXPATHLEN];
  char **rvec;

#ifdef DEBUG
  dlog("hesiod_search(m=%x, map=%s, key=%s, pval=%x tp=%x)", m, map, key, pval, tp);
#endif /* DEBUG */

  sprintf(hes_key, "%s.%s", key, map + HES_PREFLEN);

  /*
   * Call the resolver
   */
#ifdef DEBUG
  dlog("hesiod_search: hes_resolve(%s, %s)", hes_key, "automount");
  if (debug_flags & D_FULL)
    _res.options |= RES_DEBUG;
#endif /* DEBUG */

#ifdef HAVE_HESIOD_RESOLVE
  rvec = hesiod_resolve(hesiod_context, hes_key, "automount");
#else /* not HAVE_HESIOD_RESOLVE */
  rvec = hes_resolve(hes_key, "automount");
#endif /* not HAVE_HESIOD_RESOLVE */

  /*
   * If a reply was forthcoming then return
   * it (and free subsequent replies)
   */
  if (rvec && *rvec) {
    *pval = *rvec;
    while (*++rvec)
      free(*rvec);
    return 0;
  }

#ifdef HAVE_HESIOD_RESOLVE
  return errno;
#else /* not HAVE_HESIOD_RESOLVE */
  /*
   * Otherwise reflect the hesiod error into a Un*x error
   */
# ifdef DEBUG
  dlog("hesiod_search: Error: %d", hes_error());
# endif /* DEBUG */
  switch (hes_error()) {
  case HES_ER_NOTFOUND:
    error = ENOENT;
    break;
  case HES_ER_CONFIG:
    error = EIO;
    break;
  case HES_ER_NET:
    error = ETIMEDOUT;
    break;
  default:
    error = EINVAL;
    break;
  }
# ifdef DEBUG
  dlog("hesiod_search: Returning: %d", error);
# endif /* DEBUG */
  return error;
#endif /* not HAVE_HESIOD_RESOLVE */
}
