/*
 * Copyright (c) 1997 Erez Zadok
 * Copyright (c) 1990 Jan-Simon Pendry
 * Copyright (c) 1990 Imperial College of Science, Technology & Medicine
 * Copyright (c) 1990 The Regents of the University of California.
 * All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Jan-Simon Pendry at Imperial College, London.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      %W% (Berkeley) %G%
 *
 * $Id: nfs_subr.c,v 5.2.2.1 1992/02/09 15:08:53 jsp beta $
 *
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <am_defs.h>
#include <amd.h>

/*
 * Convert from UN*X to NFS error code
 */
#define nfs_error(e) ((nfsstat)(e))


static char *
do_readlink(am_node *mp, int *error_return, struct attrstat **attrpp)
{
  char *ln;

  /*
   * If there is a readlink method, then use
   * that, otherwise if a link exists use
   * that, otherwise use the mount point.
   */
  if (mp->am_mnt->mf_ops->readlink) {
    int retry = 0;
    mp = (*mp->am_mnt->mf_ops->readlink) (mp, &retry);
    if (mp == 0) {
      *error_return = retry;
      return 0;
    }
    /* reschedule_timeout_mp(); */
  }

  if (mp->am_link) {
    ln = mp->am_link;
  } else {
    ln = mp->am_mnt->mf_mount;
  }
  if (attrpp)
    *attrpp = &mp->am_attr;

  return ln;
}


voidp
nfsproc_null_2(voidp argp, CLIENT *clp)
{
  static char res;

  return (voidp) &res;
}


struct attrstat *
nfsproc_getattr_2(struct nfs_fh *argp, CLIENT *clp)
{
  static struct attrstat res;
  am_node *mp;
  int retry;

#ifdef DEBUG
  Debug(D_TRACE)
    plog(XLOG_DEBUG, "getattr:");
#endif /* DEBUG */

  mp = fh_to_mp2(argp, &retry);
  if (mp == 0) {

#ifdef DEBUG
    Debug(D_TRACE)
      plog(XLOG_DEBUG, "\tretry=%d", retry);
#endif /* DEBUG */

    if (retry < 0)
      return 0;
    res.status = nfs_error(retry);
  } else {
    struct attrstat *attrp = &mp->am_attr;

#ifdef DEBUG
    Debug(D_TRACE)
      plog(XLOG_DEBUG, "\tstat(%s), size = %d", mp->am_path, attrp->attrstat_u.attributes.size);
#endif /* DEBUG */

    mp->am_stats.s_getattr++;
    return attrp;
  }

#ifndef MNT2_NFS_OPT_SYMTTL
  /*
   * This code is needed to defeat Solaris 2.4's (and newer) symlink values
   * cache.  It forces the last-modifed time of the symlink to be current.
   * It is not needed if the O/S has an nfs flag to turn off the
   * symlink-cache at mount time (such as Irix 5.x and 6.x). -Erez.
   */
  if (++res.attrstat_u.attributes.mtime.useconds == 0)
    ++res.attrstat_u.attributes.mtime.seconds;
#endif /* not MNT2_NFS_OPT_SYMTTL */

  return &res;
}


struct attrstat *
nfsproc_setattr_2(struct sattrargs *argp, CLIENT *clp)
{
  static struct attrstat res;

  if (!fh_to_mp(&argp->file))
    res.status = nfs_error(ESTALE);
  else
    res.status = nfs_error(EROFS);

  return &res;
}


voidp
nfsproc_root_2(voidp argp, CLIENT *clp)
{
  static char res;

  return (voidp) &res;
}


struct diropres *
nfsproc_lookup_2(struct diropargs *argp, CLIENT *clp)
{
  static struct diropres res;
  am_node *mp;
  int retry;

#ifdef DEBUG
  Debug(D_TRACE)
    plog(XLOG_DEBUG, "lookup:");
#endif /* DEBUG */

  mp = fh_to_mp2(&argp->dir, &retry);
  if (mp == 0) {
    if (retry < 0)
      return 0;
    res.status = nfs_error(retry);
  } else {
    int error;
    am_node *ap;
#ifdef DEBUG
    Debug(D_TRACE)
      plog(XLOG_DEBUG, "\tlookuppn(%s, %s)", mp->am_path, argp->name);
#endif /* DEBUG */
    ap = (*mp->am_mnt->mf_ops->lookuppn) (mp, argp->name, &error, VLOOK_CREATE);
    if (ap == 0) {
      if (error < 0) {
#ifdef DEBUG
	dlog("Not sending RPC reply");
#endif /* DEBUG */
	amd_stats.d_drops++;
	return 0;
      }
      res.status = nfs_error(error);
    } else {
      mp_to_fh(ap, &res.diropres_u.diropres.file);
      res.diropres_u.diropres.attributes = ap->am_fattr;
      res.status = NFS_OK;
    }
    mp->am_stats.s_lookup++;
    /* reschedule_timeout_mp(); */
  }

  return &res;
}


void
quick_reply(am_node *mp, int error)
{
  SVCXPRT *transp = mp->am_transp;
  struct diropres res;
  bool_t(*xdr_result) () = xdr_diropres;

  /*
   * If there's a transp structure then we can reply to the client's
   * nfs lookup request.
   */
  if (transp) {
    if (error == 0) {
      /*
       * Construct a valid reply to a lookup request.  Same
       * code as in nfsproc_lookup_2() above.
       */
      mp_to_fh(mp, &res.diropres_u.diropres.file);
      res.diropres_u.diropres.attributes = mp->am_fattr;
      res.status = NFS_OK;
    } else
      /*
       * Return the error that was passed to us.
       */
      res.status = nfs_error(error);

    /*
     * Send off our reply
     */
    if (!svc_sendreply(transp, (XDRPROC_T_TYPE) xdr_result, (SVC_IN_ARG_TYPE) & res))
      svcerr_systemerr(transp);

    /*
     * Free up transp.  It's only used for one reply.
     */
    free(transp);
    mp->am_transp = NULL;
#ifdef DEBUG
    dlog("Quick reply sent for %s", mp->am_mnt->mf_mount);
#endif /* DEBUG */
  }
}


struct readlinkres *
nfsproc_readlink_2(struct nfs_fh *argp, CLIENT *clp)
{
  static struct readlinkres res;
  am_node *mp;
  int retry;

#ifdef DEBUG
  Debug(D_TRACE)
    plog(XLOG_DEBUG, "readlink:");
#endif /* DEBUG */

  mp = fh_to_mp2(argp, &retry);
  if (mp == 0) {
  readlink_retry:
    if (retry < 0)
      return 0;
    res.status = nfs_error(retry);
  } else {
    char *ln = do_readlink(mp, &retry, (struct attrstat **) 0);
    if (ln == 0)
      goto readlink_retry;
    res.status = NFS_OK;
#ifdef DEBUG
    Debug(D_TRACE)
      if (ln)
	plog(XLOG_DEBUG, "\treadlink(%s) = %s", mp->am_path, ln);
#endif /* DEBUG */
    res.readlinkres_u.data = ln;
    mp->am_stats.s_readlink++;
  }

  return &res;
}


struct readres *
nfsproc_read_2(struct readargs *argp, CLIENT *clp)
{
  static struct readres res;

  memset((char *) &res, 0, sizeof(res));
  res.status = nfs_error(EACCES);

  return &res;
}


voidp
nfsproc_writecache_2(voidp argp, CLIENT *clp)
{
  static char res;

  return (voidp) &res;
}


struct attrstat *
nfsproc_write_2(writeargs *argp, CLIENT *clp)
{
  static struct attrstat res;

  if (!fh_to_mp(&argp->file))
    res.status = nfs_error(ESTALE);
  else
    res.status = nfs_error(EROFS);

  return &res;
}


struct diropres *
nfsproc_create_2(createargs *argp, CLIENT *clp)
{
  static struct diropres res;

  if (!fh_to_mp(&argp->where.dir))
    res.status = nfs_error(ESTALE);
  else
    res.status = nfs_error(EROFS);

  return &res;
}


static nfsstat *
unlink_or_rmdir(struct diropargs *argp, CLIENT *clp, int unlinkp)
{
  static nfsstat res;
  int retry;

  am_node *mp = fh_to_mp3(&argp->dir, &retry, VLOOK_DELETE);
  if (mp == 0) {
    if (retry < 0)
      return 0;
    res = nfs_error(retry);
    goto out;
  }

  if (mp->am_fattr.type != NFDIR) {
    res = nfs_error(ENOTDIR);
    goto out;
  }

#ifdef DEBUG
  Debug(D_TRACE)
    plog(XLOG_DEBUG, "\tremove(%s, %s)", mp->am_path, argp->name);
#endif /* DEBUG */

  mp = (*mp->am_mnt->mf_ops->lookuppn) (mp, argp->name, &retry, VLOOK_DELETE);
  if (mp == 0) {
    /*
     * Ignore retries...
     */
    if (retry < 0)
      retry = 0;
    /*
     * Usual NFS workaround...
     */
    else if (retry == ENOENT)
      retry = 0;
    res = nfs_error(retry);
  } else {
    forcibly_timeout_mp(mp);
    res = NFS_OK;
  }

out:
  return &res;
}


nfsstat *
nfsproc_remove_2(struct diropargs *argp, CLIENT *clp)
{
  return unlink_or_rmdir(argp, clp, TRUE);
}


nfsstat *
nfsproc_rename_2(renameargs *argp, CLIENT *clp)
{
  static nfsstat res;

  if (!fh_to_mp(&argp->from.dir) || !fh_to_mp(&argp->to.dir))
    res = nfs_error(ESTALE);
  /*
   * If the kernel is doing clever things with referenced files
   * then let it pretend...
   */
  else if (strncmp(argp->to.name, ".nfs", 4) == 0)
    res = NFS_OK;
  /*
   * otherwise a failure
   */
  else
    res = nfs_error(EROFS);

  return &res;
}


nfsstat *
nfsproc_link_2(linkargs *argp, CLIENT *clp)
{
  static nfsstat res;

  if (!fh_to_mp(&argp->from) || !fh_to_mp(&argp->to.dir))
    res = nfs_error(ESTALE);
  else
    res = nfs_error(EROFS);

  return &res;
}


nfsstat *
nfsproc_symlink_2(symlinkargs *argp, CLIENT *clp)
{
  static nfsstat res;

  if (!fh_to_mp(&argp->from.dir))
    res = nfs_error(ESTALE);
  else
    res = nfs_error(EROFS);

  return &res;
}


struct diropres *
nfsproc_mkdir_2(createargs *argp, CLIENT *clp)
{
  static struct diropres res;

  if (!fh_to_mp(&argp->where.dir))
    res.status = nfs_error(ESTALE);
  else
    res.status = nfs_error(EROFS);

  return &res;
}


nfsstat *
nfsproc_rmdir_2(struct diropargs *argp, CLIENT *clp)
{
  return unlink_or_rmdir(argp, clp, FALSE);
}


struct readdirres *
nfsproc_readdir_2(readdirargs *argp, CLIENT *clp)
{
  static readdirres res;
  static entry e_res[MAX_READDIR_ENTRIES];
  am_node *mp;
  int retry;

#ifdef DEBUG
  Debug(D_TRACE)
    plog(XLOG_DEBUG, "readdir:");
#endif /* DEBUG */

  mp = fh_to_mp2(&argp->dir, &retry);
  if (mp == 0) {
    if (retry < 0)
      return 0;
    res.status = nfs_error(retry);
  } else {
#ifdef DEBUG
    Debug(D_TRACE)
      plog(XLOG_DEBUG, "\treaddir(%s)", mp->am_path);
#endif /* DEBUG */
    res.status = nfs_error((*mp->am_mnt->mf_ops->readdir)
			   (mp, argp->cookie,
			    &res.readdirres_u.reply, e_res, argp->count));
    mp->am_stats.s_readdir++;
  }

  return &res;
}


struct statfsres *
nfsproc_statfs_2(struct nfs_fh *argp, CLIENT *clp)
{
  static statfsres res;
  am_node *mp;
  int retry;

#ifdef DEBUG
  Debug(D_TRACE)
    plog(XLOG_DEBUG, "statfs:");
#endif /* DEBUG */

  mp = fh_to_mp2(argp, &retry);
  if (mp == 0) {
    if (retry < 0)
      return 0;
    res.status = nfs_error(retry);
  } else {
    statfsokres *fp;
#ifdef DEBUG
    Debug(D_TRACE)
      plog(XLOG_DEBUG, "\tstat_fs(%s)", mp->am_path);
#endif /* DEBUG */

    /*
     * just return faked up file system information
     */
    fp = &res.statfsres_u.reply;

    fp->tsize = 1024;
    fp->bsize = 4096;
    fp->blocks = 0; /* set to 1 if you don't want empty automounts */
    fp->bfree = 0;
    fp->bavail = 0;

    res.status = NFS_OK;
    mp->am_stats.s_statfs++;
  }

  return &res;
}
