/*
 * Copyright (c) 1997 Erez Zadok
 * Copyright (c) 1990 Jan-Simon Pendry
 * Copyright (c) 1990 Imperial College of Science, Technology & Medicine
 * Copyright (c) 1990 The Regents of the University of California.
 * All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Jan-Simon Pendry at Imperial College, London.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      %W% (Berkeley) %G%
 *
 * $Id: mount_svr4.c,v 5.2.2.1 1992/02/09 15:10:08 jsp beta $
 *
 */

/*
 * SVR4:
 * Solaris 2.x (SunOS 5.x) Mount helper.
 *      -Erez Zadok <ezk@cs.columbia.edu>
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <am_defs.h>
#include <amu.h>

#undef DEBUG_MOUNT_SVR4

#ifdef DEBUG_MOUNT_SVR4
static void
print_knetconfig(const char *str, const struct knetconfig *kncp)
{
  fprintf(stderr, "%s -> semantics:%lu\n", str, kncp->knc_semantics);
  fprintf(stderr, "%s -> protofmly:%s\n", str, kncp->knc_protofmly);
  fprintf(stderr, "%s -> proto:%s\n", str, kncp->knc_proto);
  fprintf(stderr, "%s -> rdev:%lu\n", str, kncp->knc_rdev);
  fprintf(stderr, "%s -> unused:%s\n", str, "unused not printed");
}

static void
print_netbuf(const char *str, const struct netbuf *nbp)
{
  int i;

  fprintf(stderr, "%s -> maxlen:%d\n", str, nbp->maxlen);
  fprintf(stderr, "%s -> len:%d\n", str, nbp->len);
  for (i=0; i<nbp->len; ++i)
    fprintf(stderr, "%s -> len[%d]:%d\n", str, i, (int)(nbp->buf[i]));
}

/* get string version (in hex) of identifier */
static char *
get_hex_string(u_int len, const char *fhdata)
{
  int i;
  static char buf[128];		/* better not go over it! */
  char str[16];
  short int arr[64];

  if (!fhdata)
    return NULL;
  buf[0] = '\0';
  memset(&arr[0], 0, (64 * sizeof(short int)));
  memcpy(&arr[0], &fhdata[0], len);
  for (i=0; i<len/sizeof(short int); i++) {
    sprintf(str, "%04x", ntohs(arr[i]));
    strcat(buf, str);
  }
  return buf;
}

void
print_nfs_fh(const char *head, caddr_t fh, const char *type)
{
  int len, i;
  am_nfs_fh *fh2p;
  am_nfs_fh3 *fh3p;
  char *dp = NULL;

  if (STREQ(type, MNTTAB_TYPE_NFS3)) {
    fh3p = (am_nfs_fh3 *) fh;
    len = fh3p->fh3_length;
    dp = fh3p->fh3_u.data;
  } else {
    fh2p = (am_nfs_fh *) fh;
    len = FHSIZE;
    dp = fh2p->fh_data;
  }
  printf("\t%s: FH type %s, len=%d, \"%s\"\n",
	 head ? head : "print_nfs_fh",
	 type, len, get_hex_string(len, dp));
  for (i=0; i<len; ++i)
    fprintf(stderr, "\t%s -> len[%2d]: %4d (%08x)\n", "handle",
	    i, (int)(dp[i]), (unsigned int)(dp[i]));
}

void
xprint_nfs_fh3(const char *head, am_nfs_fh3 *fh3p)
{
  int len;
  char *dp;

  len = fh3p->fh3_length;
  dp = fh3p->fh3_u.data;
  plog(XLOG_INFO, "\t%s: FH3 len=%d, \"%s\"\n",
       head ? head : "print_nfs_fh",
       len, get_hex_string(len, dp));
}

static void
print_nfs_args(const nfs_args_t *nap, const char *type)
{
  if (!nap) {
    fprintf(stderr, "NULL nfs_args!\n");
    return;
  }

  fprintf(stderr, "addr:0x%x:\n", (int) nap->addr);
  if (nap->addr)
    print_netbuf("\taddr", nap->addr);

  fprintf(stderr, "syncaddr:0x%x:\n", (int) nap->syncaddr);
  if (nap->syncaddr)
    print_netbuf("\tsyncaddr", nap->syncaddr);

  fprintf(stderr, "knconf:%d:\n", (int) nap->knconf);
  if (nap->knconf)
    print_knetconfig("\tknconf", nap->knconf);

  fprintf(stderr, "hostname:%s:\n", (nap->hostname ? nap->hostname : "null"));
  fprintf(stderr, "netname:%s:\n", (nap->netname ? nap->netname : "null"));
  fprintf(stderr, "fh:0x%x:\n", (int) nap->fh);
  if (nap->fh)
    print_nfs_fh("mount_svr4", nap->fh, type);
  fprintf(stderr, "flags:0x%x:\n", nap->flags);
  fprintf(stderr, "wsize:%d:\n", nap->wsize);
  fprintf(stderr, "rsize:%d:\n", nap->rsize);
  fprintf(stderr, "timeo:%d:\n", nap->timeo);
  fprintf(stderr, "retrans:%d:\n", nap->retrans);
  fprintf(stderr, "acregmin:%d:\n", nap->acregmin);
  fprintf(stderr, "acregmax:%d:\n", nap->acregmax);
  fprintf(stderr, "acdirmin:%d:\n", nap->acdirmin);
  fprintf(stderr, "acdirmax:%d:\n", nap->acdirmax);

  fprintf(stderr, "pathconf:0x%x:\n", (int) nap->pathconf);
  if (nap->pathconf) {
    fprintf(stderr, "need to print struct pathconf\n");
  }
  fflush(stderr);
}
#endif /* DEBUG_MOUNT_SVR4 */

/*
 * Map from conventional mount arguments
 * to Solaris 2.x (SunOS 5.x) style arguments.
 */
int
mount_svr4(char *fsname, char *dir, int flags, MTYPE_TYPE type, caddr_t data)
{

#ifdef DEBUG_MOUNT_SVR4
  fprintf(stderr,
	  "mount_svr4: fsname=\"%s\", dir=\"%s\", flags=0x%x, type=\"%s\"(%d)\n",
	  fsname ? fsname : "null",
	  dir ? dir : "null",
	  flags, type, sysfs(GETFSIND, type));
#endif /* DEBUG_MOUNT_SVR4 */

#if defined(MOUNT_TYPE_NFS3) && defined(MNTTAB_TYPE_NFS3)
  if (STREQ(type, MOUNT_TYPE_NFS3)) {
# ifdef DEBUG_MOUNT_SVR4
    int ret;
    print_nfs_args((nfs_args_t *) data, type);
    ret = mount(fsname, dir, (MNT2_GEN_OPT_DATA | flags),
		type, (char *) data, sizeof(nfs_args_t));
    fprintf(stderr, "mount() returned %d (errno=%d)\n", ret, errno);
    return ret;
# else /* not DEBUG_MOUNT_SVR4 */
    return mount(fsname, dir, (MNT2_GEN_OPT_DATA | flags),
		 type, (char *) data, sizeof(nfs_args_t));
# endif /* not DEBUG_MOUNT_SVR4 */
  }
#endif /* defined(MOUNT_TYPE_NFS3) && defined(MNTTAB_TYPE_NFS3) */

#if defined(MOUNT_TYPE_NFS) && defined(MNTTAB_TYPE_NFS)
  if (STREQ(type, MOUNT_TYPE_NFS)) {
# ifdef DEBUG_MOUNT_SVR4
    int ret;
    print_nfs_args((nfs_args_t *) data, type);
    ret = mount(fsname, dir, (MNT2_GEN_OPT_DATA | flags),
		type, (char *) data, sizeof(nfs_args_t));
    fprintf(stderr, "mount() returned %d (errno=%d)\n", ret, errno);
    return ret;
# else /* not DEBUG_MOUNT_SVR4 */
    return mount(fsname, dir, (MNT2_GEN_OPT_DATA | flags),
		 type, (char *) data, sizeof(nfs_args_t));
# endif /* not DEBUG_MOUNT_SVR4 */
  }
#endif /* defined(MOUNT_TYPE_NFS) && defined(MNTTAB_TYPE_NFS) */

#if defined(MOUNT_TYPE_UFS) && defined(MNTTAB_TYPE_UFS)
  if (STREQ(type, MOUNT_TYPE_UFS))
    return mount(fsname, dir, (MNT2_GEN_OPT_DATA | flags),
		 type, (char *) data, sizeof(ufs_args_t));
#endif /* defined(MOUNT_TYPE_UFS) && defined(MNTTAB_TYPE_UFS) */

#if defined(MOUNT_TYPE_PCFS) && defined(MNTTAB_TYPE_PCFS)
  if (STREQ(type, MOUNT_TYPE_PCFS))
    return mount(fsname, dir, (MNT2_GEN_OPT_DATA | flags),
		 type, (char *) data, sizeof(pcfs_args_t));
#endif /* defined(MOUNT_TYPE_PCFS) && defined(MNTTAB_TYPE_PCFS) */

#if defined(MOUNT_TYPE_CDFS) && defined(MNTTAB_TYPE_CDFS)
  /*
   * HSFS on Solaris allows for 3 HSFSMNT_* flags to be passed
   * as arguments to the mount().  These flags are bit fields in an
   * integer, and that integer is passed as the "data" of this system
   * call.  The flags are described in <sys/fs/hsfs_rrip.h>.  However,
   * Solaris does not have an interface to these.  It does not define
   * a structure hsfs_args or anything that one can figure out what
   * arguments to pass to mount(2) for this type of filesystem.
   * Therefore, until Sun does, no arguments are passed to this mount
   * below.
   * -Erez Zadok <ezk@cs.columbia.edu>.
   */
  if (STREQ(type, MOUNT_TYPE_CDFS))
    return mount(fsname, dir, (MNT2_GEN_OPT_FSS | flags),
		 type, (char *) NULL, 0);
#endif /* defined(MOUNT_TYPE_CDFS) && defined(MNTTAB_TYPE_CDFS) */

#if defined(MOUNT_TYPE_LOFS) && defined(MNTTAB_TYPE_LOFS)
  if (STREQ(type, MOUNT_TYPE_LOFS))
    return mount(fsname, dir, (MNT2_GEN_OPT_FSS | flags),
		 type, (char *) NULL, 0);
#endif /* defined(MOUNT_TYPE_LOFS) && defined(MNTTAB_TYPE_LOFS) */

#if defined(MOUNT_TYPE_AUTOFS) && defined(MNTTAB_TYPE_AUTOFS)
  if (STREQ(type, MOUNT_TYPE_AUTOFS))
    return mount(fsname, dir, (MNT2_GEN_OPT_DATA | flags),
		 type, (char *) data, sizeof(autofs_args_t));
#endif /* defined(MOUNT_TYPE_AUTOFS) && defined(MNTTAB_TYPE_AUTOFS) */

  return EINVAL;
}
