/* ax25lnk.c 1994.5.31 */

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <time.h>
#include <assert.h>

#include "pbdef.h"
#include "ax25.h"
#include "pbkiss.h"
#include "pblib.h"

#define T2    2

#define MOD8(A) (((A)&0x07))

extern struct stevt evttbl[][16];	    /* ax25tbl.c */

struct stqueue ax25queue = {NULL,NULL};	    /* AX.25 queue  */

/*
 * < ax25link > AX.25 link
 */
int ax25link()
{
    int  ax25rcv();
    void ax25evt(),ax25snd(),ax25timer();

    struct stax25cb *ax25;

    ax25 = (struct stax25cb*)ax25queue.head;
    while(ax25) {
        ax25rcv(ax25);
        ax25evt(ax25);
        ax25snd(ax25);
        ax25timer(ax25);
        ax25 = ax25->next;
    }
}

/*
 * < ax25_StartT1 >
 */
VOID ax25_StartT1(ax25)
struct stax25cb *ax25;
{
    ax25->f_t1   = ON;
    ax25->tim_t1 = ax25->frack*(ax25->digi+1);
}

/*
 * < ax25_StopT1 >
 */
VOID ax25_StopT1(ax25)
struct stax25cb *ax25;
{
    ax25->f_t1 = OFF;
}

/*
 * < ax25_StartT3 >
 */
VOID ax25_StartT3(ax25)
struct stax25cb *ax25;
{
    ax25->f_t3   = ON;
    ax25->tim_t3 = ax25->check;
}

/*
 * < ax25_StopT3 >
 */
VOID ax25_StopT3(ax25)
struct stax25cb *ax25;
{
    ax25->f_t3 = OFF;
}

/*
 * < ax25_Connected >
 */
BOOL ax25_Connected(ax25)
struct stax25cb *ax25;
{
    if (ax25->status < S5)
        return(NO);
    else
        return(YES);
}

/*
 * < ax25rcv > rcv packet
 */
int ax25rcv(ax25cb)
struct stax25cb *ax25cb;
{
    BOOL ck_to_adrs(struct stax25cb*,char*,int);
    BOOL ck_from_adrs(struct stax25cb*,char*,int);
    VOID get_from_adrs(struct stax25cb*,char*,int);
    u_char get_ctl(struct stax25cb*,char*,int);
    void clr_cb(struct stax25cb*);
    void rcv_su(struct stax25cb*,struct stframe*);
    void rcv_i(struct stax25cb*,struct stframe*);

    u_int ctl;
    struct stframe *frame;

    frame = (struct stframe*)getq(&rkssfrm);
    if (frame == NULL) {
        return(-1);
    }
    if (!ck_to_adrs(ax25cb,frame->data,frame->frmlen)) {
        insq(&rkssfrm,frame);
	return(-1);
    }
    ctl = get_ctl(ax25cb,frame->data,frame->frmlen);

    if (ax25cb->status == S1 && ax25cb->f_conok) {
        clr_cb(ax25cb);
        get_from_adrs(ax25cb,frame->data,frame->frmlen);
    } else {
        if (!ck_from_adrs(ax25cb,frame->data,frame->frmlen)) {
	    insq(&rkssfrm,frame);
	    return(-1);
	}
    }

    if (ctl & 0x01) {
        rcv_su(ax25cb,frame);
    } else {			/* I frame */
        rcv_i(ax25cb,frame);
    }
    free(frame);

    if (ax25cb->status < S5)
        return(0);

    ax25cb->retries = 0;
				    /* adjust T1 */
    if (ax25cb->win == 0) {
        ax25_StopT1(ax25cb);
        ax25_StartT3(ax25cb);
    } else {
        ax25_StartT1(ax25cb);	    /* restart T1 */
    }
    return(0);
}

BOOL ck_to_adrs(ax25cb,p,n)
struct stax25cb *ax25cb;
u_char p[];
int n;
{
    int i;

    if (n < (1+AXALEN+AXALEN+1)) {
        return(OFF);
    }
    i = 0;
    if ((p[i]>>4) != ax25cb->tncid) {
	return(OFF);
    }
    i++;
    if (cmpadr(ax25cb->myadrs,&p[i]) != 0) {
        return(OFF);
    }
    return(ON);
}

BOOL ck_from_adrs(ax25cb,p,n)
struct stax25cb *ax25cb;
u_char p[];
int n;
{
    int i,j;

    i = 1+AXALEN;
    if (cmpadr(ax25cb->yradrs,&p[i]) != 0) {
        return(OFF);
    }
    i += AXALEN;
    for (j = 0; j < ax25cb->digi; j++) {
	if (cmpadr(ax25cb->digiadrs[ax25cb->digi-j-1],&p[i]) != 0) {
	    return(OFF);
	}
	i += AXALEN;
    }
    if (!(p[i-1] & E)) {
        return(OFF);
    }
    return(ON);
}

VOID get_from_adrs(ax25cb,p,n)
struct stax25cb *ax25cb;
u_char p[];
int n;
{
    int i;

    i = 1+AXALEN;
    adrscpy(ax25cb->yradrs,&p[i]);
    i += AXALEN;
    ax25cb->digi = 0;
    if (!p[i-1] & C) {	    /* digi ? */
        adrscpy(ax25cb->digiadrs[0],&p[i]);
	ax25cb->digi++;
    }
}

u_char get_ctl(ax25cb,p,n)
struct stax25cb *ax25cb;
u_char p[];
int n;
{
    u_char ctl;
    int i,j;

    i = 1+AXALEN+AXALEN;
    for (j = 0; j < MAXDIGIS; j++) {
        if (p[i-1] & E) {	    /* adrs end ? */
	    break;
	}
	i += AXALEN;
    }
    ctl = p[i];
    return(ctl);
}

/*
 * < clr_cb > clear AX.25 control block
 */
clr_cb(ax25cb)
struct stax25cb *ax25cb;
{
    struct stframe *frame;

    ax25cb->status   = S1;
    ax25cb->f_start  = OFF;
    ax25cb->f_stop   = OFF;
    ax25cb->vr       = 0;
    ax25cb->vs       = 0;
    ax25cb->win      = 0;
    ax25cb->f_sndreq = OFF;
    ax25cb->snd_seq  = 0;
    ax25_StopT1(ax25cb);
    ax25_StopT3(ax25cb);
    ax25cb->retries  = 0;
    while(frame = (struct stframe*)getq(&ax25cb->rcvifrm))
        free(frame);
    while(frame = (struct stframe*)getq(&ax25cb->sndifrm))
        free(frame);

    ax25cb->f_conok  = OFF;

    ax25cb->rcv_byte = 0;
    ax25cb->snd_byte = 0;
}

/*
 * < rcv_su > rcv S,U frame
 */
void rcv_su(ax25cb,frame)
struct stax25cb *ax25cb;
struct stframe *frame;
{
    int vsupdat(struct stax25cb*,u_char);
    int getevt(int,int);

    int cmdres,evt,nc,ns;
    int ctl;

    if (frame->data[1+AXALEN-1] & 0x80) {
        cmdres = CMD;
    } else {
        cmdres = RES;
    }
    ctl = frame->data[1+AXALEN+AXALEN+ax25cb->digi*AXALEN]  & 0xff;

    if ((ctl&0xef) == DISC) {
        ax25cb->dreason = DISC_REMOTE;
    } else if ((ctl&0xef) == DM) {
        ax25cb->dreason = DISC_BUSY;
    }

    vsupdat(ax25cb,ctl);		/* V(S) */
    evt = getevt(cmdres,ctl);
    if (evt == (-1))
        return;
    nc = evttbl[evt][ax25cb->status].ctl;
    ns = evttbl[evt][ax25cb->status].status;
    ax25cb->status = ns;
    if (nc == (-1))
        return;
    ax25cb->snd_ctl = nc;
    ax25cb->f_sndreq = ON;		/* send req */
    if (ctl & PF) {
        ax25cb->tim_t2 = 0;
        ax25cb->snd_ctl |= PF;
    } else {
        ax25cb->tim_t2 = T2;
    }
}

/*
 * < rcv_i > rcv I frame
 */
void rcv_i(ax25cb,frame)
struct stax25cb *ax25cb;
struct stframe *frame;
{
    int vsupdat(struct stax25cb*,u_char);
    int getevt(int,int);

    int evt,nc,ns;
    int ctl,n;

    struct stframe *ifrm;

    if (ax25cb->status < S5)
        return;

    ctl = frame->data[1+AXALEN+AXALEN+ax25cb->digi*AXALEN]  & 0xff;
    vsupdat(ax25cb,ctl);		/* V(S) */
    if (ax25cb->vr == MOD8(ctl>>1)) {
	ax25cb->vr = MOD8(ax25cb->vr + 1);
	n = frame->frmlen - (1+AXALEN+AXALEN+ax25cb->digi*AXALEN+1+1);
	ax25cb->rcv_byte += n;
	if ((ifrm = frmalloc(n)) == NULL) {
	    assert(0);
	}
	add_frm(ifrm,frame->data+(1+AXALEN+AXALEN+ax25cb->digi*AXALEN+1+1),n);
	putq(&ax25cb->rcvifrm,ifrm);
    }
    evt = getevt(CMD,ctl);
    if (evt == (-1))
        return;
    nc = evttbl[evt][ax25cb->status].ctl;
    ns = evttbl[evt][ax25cb->status].status;
    ax25cb->status = ns;
    if (nc == (-1))
        return;
    ax25cb->snd_ctl = nc;
    ax25cb->f_sndreq = ON;		/* send req */
    if (ctl & PF) {
        ax25cb->tim_t2 = 0;
        ax25cb->snd_ctl |= PF;
    } else {
        ax25cb->tim_t2 = T2;
    }
}

/*
 * < vsupdat > VS up-date
 */
vsupdat(ax25cb,ctl)
struct stax25cb *ax25cb;
u_char ctl;
{
    struct stframe *frame;
    TINY vs;
    u_char ctl2;

    ctl2 = ctl & 0x0f;
    if ((ctl&0x01) == I || ctl2 == RR || ctl2 == RNR || ctl2 == REJ) {
        vs = MOD8(ctl>>5);
        while(ax25cb->vs != vs) {
	    frame = (struct stframe*)getq(&ax25cb->sndifrm);
	    if (frame == NULL) {
	        break;
	    }
	    ax25cb->snd_byte += frame->frmlen;
	    free(frame);

	    ax25cb->vs = MOD8(ax25cb->vs + 1);	/* V(s) + 1 */
	    ax25cb->win--;
        }
    }
}

/*
 * < ax25evt > AX.25 event
 */
void ax25evt(ax25cb)
struct stax25cb *ax25cb;
{
    int getevt(int,int);

    int evt,nc,ns;

    evt = (-1);
    if (ax25cb->f_start) {
	ax25cb->f_start = OFF;
        clr_cb(ax25cb);
        ax25cb->dreason = 0;
        evt = E_START;		/* TNC start command */

    } else if (ax25cb->f_stop) {
        ax25cb->f_stop = OFF;
	ax25cb->snd_seq = 0;
	ax25cb->dreason = DISC_LOCAL;
        evt = E_STOP;		/* TNC stop command */

    } else if (ax25cb->f_t1 && ax25cb->tim_t1 == 0) {
	ax25_StopT1(ax25cb);
	ax25cb->snd_seq = 0;
        evt = E_T1OUT;
	if (ax25cb->retries++ >= ax25cb->retry) {
	    ax25cb->f_t1 = OFF;
	    ax25cb->dreason = DISC_TIMEOUT;
	    evt = E_N2OVER;
	}

    } else if (ax25cb->f_t3 && ax25cb->tim_t3 == 0) {
	ax25_StopT3(ax25cb);
	ax25cb->snd_seq = 0;
        evt = E_T3OUT;

    } else if (ax25cb->sndifrm.head && ax25cb->f_t1 == OFF) {
        evt = E_IDOL;
    }
    if (evt == (-1))
        return;

    nc = evttbl[evt][ax25cb->status].ctl;
    ns = evttbl[evt][ax25cb->status].status;
    ax25cb->status = ns;
    if (nc == (-1))
        return;

    ax25cb->snd_ctl = nc;
    ax25cb->f_sndreq = ON;		/* send req */
    ax25cb->tim_t2 = 0;			/* T2 0sec */
}

/*
 * < getevt > get event code
 */
int getevt(cmdres,ctl)
int cmdres;
int ctl;
{
    int evt;

    if (!(ctl & 0x01)) {	    /* I */
        evt = E_ICP;
	if (!(ctl & PF))
	    evt = E_IC;
	return(evt);
    }

    if ((ctl & 0x0f) == RR) {
        if (cmdres)
	    evt = E_RRCP;
	else
	    evt =E_RRRF;
	if (!(ctl & PF))
	    evt++;
	return(evt);
    }

    if ((ctl & 0x0f) == REJ) {
        if (cmdres)
	    evt = E_REJCP;
	else
	    evt =E_REJRF;
	if (!(ctl & PF))
	    evt++;
	return(evt);
    }

    if ((ctl & 0x0f) == RNR) {
        if (cmdres)
	    evt = E_RNRCP;
	else
	    evt =E_RNRRF;
	if (!(ctl & PF))
	    evt++;
	return(evt);
    }

    switch(ctl & 0xef) {
        case 0x2f:	    /* SABM */
	    return(E_SABM);
	case 0x43:	    /* DISC */
	    return(E_DISC);
	case 0x0f:	    /* DM   */
	    return(E_DM);
	case 0x63:	    /* UA   */
	    return(E_UA);
	default:
	    break;
    }
    return(-1);
}

/*
 * < ax25snd > send packet
 */
void ax25snd(ax25cb)
struct stax25cb *ax25cb;
{
    int snd_su();
    int snd_i(),snd_i_full();

    int r;

    switch(ax25cb->snd_seq) {
        case 0:
	    if (ax25cb->f_sndreq == OFF) {
	        break;
	    }
	    ax25cb->f_sndreq = OFF;
            ax25cb->snd_seq++;

	case 1:
	    if (ax25cb->tim_t2 != 0) {
	        break;
	    }
	    if (ax25cb->snd_ctl & 0x01) {    /* S frame ? */
	        snd_su(ax25cb);
	        if (ax25cb->snd_ctl & 0x0100) {  /* command ? */
		    ax25_StartT1(ax25cb);
	        }
	        ax25cb->snd_seq  = 0;
	        ax25cb->f_sndreq = OFF;
	    } else {
	        ax25cb->snd_seq++;
	    }
	    break;
	case 2:			    /* I frmae */
	    if (ax25cb->f_fullduplex) {
	        ax25cb->win = 0;
	        ax25cb->snd_seq++;
	    } else {
                snd_i(ax25cb);
		ax25_StartT1(ax25cb);
	        ax25cb->snd_seq  = 0;
	        ax25cb->f_sndreq = OFF;
	    }
	    break;
	case 3:			    /* I frame fullduplex */
	    if (ax25cb->snd_ctl & 0x01) {
	        ax25cb->snd_seq = 0;
		break;
	    }
            r = snd_i_full(ax25cb);
	    if (r > 0) {
		ax25_StartT1(ax25cb);
	    }
	    ax25cb->f_sndreq = OFF;
	    break;
	default:
	    break;
    }
}

/*
 * < snd_su > send S,U frame
 */
int snd_su(ax25cb)
struct stax25cb *ax25cb;
{

    struct stframe *mk_adrs_header(struct stax25cb*,int);

    u_char ctl,c;
    struct stframe *frame;

    frame = mk_adrs_header(ax25cb,1);
    ctl = ax25cb->snd_ctl & 0xff;
    c = ctl & 0x0f;
    if (c == RR || c == RNR || c == REJ) {
        ctl |= (ax25cb->vr << 5);
    }
    add_frm_val(frame,(u_long)ctl,1);
    if (ax25cb->snd_ctl & 0x0100) {   /* command ? */
        frame->data[1+AXALEN-1] |= 0x80;
    } else {
        frame->data[1+AXALEN+AXALEN-1] |= 0x80;
    }
    putq(&skssfrm,frame);
}

/*
 * < snd_i > send I frame
 */
int snd_i(ax25cb)
struct stax25cb *ax25cb;
{
    struct stframe *mk_adrs_header(struct stax25cb*,int);

    struct stframe *frame,*ifrm;
    int i,m;
    u_char ctl,pid;

    pid = 0xf0;
    m = cntq(&ax25cb->sndifrm);
    if (m > (int)ax25cb->maxframe)
        m = (int)ax25cb->maxframe;
    ifrm = (struct stframe*)ax25cb->sndifrm.head;
    ax25cb->win = 0;
    for (i = 0; i < m; i++,ifrm = ifrm->next) {
        if (ifrm == NULL)
	    break;
        frame = mk_adrs_header(ax25cb,ifrm->frmlen+2);
        ctl = 0x00;
	ctl |= (MOD8(ax25cb->vs+i) << 1);
        ctl |= (ax25cb->vr << 5);
	if (i == (m-1)) {	    /* last frame ? */
	    ctl |= PF;
	}
        add_frm_val(frame,(u_long)ctl,1);
        add_frm_val(frame,(u_long)pid,1);
	add_frm(frame,ifrm->data,ifrm->frmlen);
        frame->data[1+AXALEN-1] |= 0x80;	/* command */
        putq(&skssfrm,frame);
	ax25cb->win++;
    }
}

/*
 * < snd_i_full > send I frame fullduplex
 */
int snd_i_full(ax25cb)
struct stax25cb *ax25cb;
{
    struct stframe *mk_adrs_header(struct stax25cb*,int);

    struct stframe *frame,*ifrm;
    int    i;
    u_char ctl,pid;

    if (ax25cb->win >= ax25cb->maxframe) {
        return(0);
    }
    pid = 0xf0;
    ifrm = (struct stframe*)ax25cb->sndifrm.head;
    for (i = 0; i < ax25cb->win; i++,ifrm = ifrm->next) {
	if (ifrm == NULL) {
	    return(0);
	}
    }
    if (ifrm == NULL) {
        return(0);
    }

    frame = mk_adrs_header(ax25cb,ifrm->frmlen+2);
    ctl = (MOD8(ax25cb->vs+i) << 1) | (ax25cb->vr << 5);
    add_frm_val(frame,(u_long)ctl,1);
    add_frm_val(frame,(u_long)pid,1);
    add_frm(frame,ifrm->data,ifrm->frmlen);
    frame->data[1+AXALEN-1] |= 0x80;	/* command */
    putq(&skssfrm,frame);
    ax25cb->win++;
    return(1);
}

struct stframe *mk_adrs_header(ax25cb,n)
struct stax25cb *ax25cb;
int n;
{
    struct stframe *frame;
    int i;

    if ((frame = frmalloc(1+AXALEN+AXALEN+ax25cb->digi*AXALEN+n)) == NULL) {
        assert(0);
    }
    frame->data[frame->frmlen++] = (ax25cb->tncid<<4);
    add_frm(frame,ax25cb->yradrs,AXALEN);
    add_frm(frame,ax25cb->myadrs,AXALEN);
    frame->data[1+AXALEN-1]        |= 0x60;	/* RR bit */
    frame->data[1+AXALEN+AXALEN-1] |= 0x60;	/* RR bit */
    for (i = 0; i < ax25cb->digi; i++) {
        add_frm(frame,ax25cb->digiadrs[i],AXALEN);
    }
    frame->data[frame->frmlen-1] |= E;
    return(frame);
}

/*
 * < ax25timer > timer	1sec
 */
void ax25timer(ax25cb)
struct stax25cb *ax25cb;
{
    static time_t t1;
    time_t t2;

    t2 = clock();
    if (t2 == t1)
        return;
    t1 = t2;

    if (ax25cb->tim_t2 > 0) {
        ax25cb->tim_t2--;
    }

    if (ax25cb->f_t1) {
        if (ax25cb->tim_t1 > 0)
	    ax25cb->tim_t1--;
    }

    if (ax25cb->f_t3) {
        if (ax25cb->tim_t3 > 0)
	    ax25cb->tim_t3--;
    }
}

/* ax25lnk.c */
