

/*******************************
 *
 * calld.c  (from ax25_call.c)
 *
 *******************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <ctype.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include <linux/ax25.h>
#include <linux/rose.h>

#include "axutils.h"
#include "axconfig.h"
#include "rsconfig.h"

#include "fpac.h"

union sockaddr_ham
{
	struct full_sockaddr_ax25 axaddr;
	struct sockaddr_rose rsaddr;
};

#define BUFLEN 4096

void alarm_handler(int sig)
{
}

void err(char *message)
{
	write(STDOUT_FILENO, message, strlen(message));
	sleep(1);
	exit(1);
}

/* Transforms a ROSE connection string to struct sockaddr_rose */
static int rs_convert_call (char *address, struct sockaddr_rose *addr)
{
  char *command, *call, *rose;
  
  command = strdup(address);
  if (command)
  {

    call = strtok (command, " \t\n\r");
    rose = strtok (NULL, " \t\r\n");
    if (strcasecmp(rose, "VIA") == 0)
        rose = strtok (NULL, " \t\r\n");

    addr->srose_family = AF_ROSE;
    addr->srose_ndigis = 0;

    if (convert_call_entry (call, addr->srose_call.ax25_call) == -1)
      {
        free(command);
        return -1;
      }
  
    if (convert_rose_address (rose, addr->srose_addr.rose_addr) == -1)
      {
        free(command);
        return -1;
      }

    free(command);
  }
  return sizeof(struct sockaddr_rose);
}

static char *reason(unsigned char cause, unsigned char diagnostic)
{
	static char *desc;

	switch (cause) 
	{
	case ROSE_DTE_ORIGINATED:
		desc = "Remote Station cleared connection";
		break;
	case ROSE_NUMBER_BUSY:
		if (diagnostic == 0x48)
			desc = "Remote station is connecting to you";
		else
			desc = "Remote Station is busy";
		break;
	case ROSE_INVALID_FACILITY:
		desc = "Invalid X.25 Facility Requested";
		break;
	case ROSE_NETWORK_CONGESTION:
		desc = "Network Congestion";
		break;
	case ROSE_OUT_OF_ORDER:
		desc = "Out of Order, link unavailable";
		break;
	case ROSE_ACCESS_BARRED:
		desc = "Access Barred";
		break;
	case ROSE_NOT_OBTAINABLE:
		desc = "No Route Defined for address specified";
		break;
	case ROSE_REMOTE_PROCEDURE:
		desc = "Remote Procedure Error";
		break;
	case ROSE_LOCAL_PROCEDURE:
		desc = "Local Procedure Error";
		break;
	case ROSE_SHIP_ABSENT:
		desc = "Remote Station not responding";
		break;
	default:
		desc = "Unknown";
		break;
	}

	return desc;
}

int is_rose(char *address)
{
	int i;

	if (strlen(address) != 10)
		return(0);

	for (i = 0 ; *address ; i++, address++)
		if (!isdigit(*address))
			break;
	return(i == 10);
}

int main(int argc, char **argv)
{
	cfg_t cfg;
	int buflen;
	int type;
	char address[256];
	char *ptr;
	char *sender;
	char buffer[4096], *addr;
	fd_set read_fd;
	int nb, offs;
	int n, s = -1, bindlen = 0, addrlen = 0;
	union sockaddr_ham axbind, axconnect;

	/*
	 * Arguments should be "ax25_call port"
	 */
	if (argc < 2) {
		err("ERROR: invalid number of parameters\r");
	}

	if (ax25_config_load_ports() == 0) {
		err("ERROR: problem with axports file\r");
	}

	if (rs_config_load_ports() == 0) {
		err("ERROR: problem with axports file\r");
	}

	/* Read FPAC configuration file */
	if (cfg_open(&cfg) != 0)
	{
		err("ERROR: problem with fpac.conf file\r");
	}

	/* Ask for callsign */

	printf("Callsign : ");
	fflush(stdout);

	offs = 0;
	while (offs < 256)
	{
		nb = read(0, buffer+offs, 256-offs);
		if (nb <= 0)
		{
			/* Disconnected or problem ! */
			return(0);
		}
		offs += nb;
		buffer[offs] = '\0';
		if (strchr(buffer, '\r'))
			break;
	}

	/* Deletes the '\r' */
	if (buffer[offs-1] == '\r')
		buffer[offs-1] = '\0';

	/* extract sender */
	ptr = strchr(buffer, '^');
	if (ptr)
		*ptr++ = '\0';

	/* Deletes the '.' for binary information */
	sender = buffer;
	if (*sender == '.')
		++sender;

	sprintf(address, "*** Connecting %s\r", ptr);
	write(STDOUT_FILENO, address, strlen(address));

	/*
	 *
	 * Calling callsign    = sender
	 * Destination + digis = ptr
	 *
	 */

	/* Check the kind of connection : ax25 or ROSE */
	sscanf( ptr, "%*s %s", address);
	if (strcasecmp(address, "VIA") == 0)
		sscanf( ptr, "%*s %*s %s", address);

	if (is_rose(address))
		type = AF_ROSE;
	else
		type = AF_AX25;

	/*
	 * Parse the passed values for correctness.
	 */

	switch (type)
	{

	case AF_AX25:
		axbind.axaddr.fsa_ax25.sax25_family = AF_AX25;
		axbind.axaddr.fsa_ax25.sax25_ndigis = 1;

		if (convert_call_entry(sender, axbind.axaddr.fsa_ax25.sax25_call.ax25_call) == -1) {
			sprintf(buffer, "ERROR: invalid callsign - %s\r", sender);
			err(buffer);
		}

		if ((addr = ax25_config_get_addr(argv[1])) == NULL) {
			sprintf(buffer, "ERROR: invalid AX.25 port name - %s\r", argv[1]);
			err(buffer);
		}

		if (convert_call_entry(addr, axbind.axaddr.fsa_digipeater[0].ax25_call) == -1) {
			sprintf(buffer, "ERROR: invalid AX.25 port callsign - %s\r", argv[1]);
			err(buffer);
		}

		/* Indicatif appele et digis... */
		if ((addrlen = convert_call (ptr, &axconnect.axaddr)) == -1)
		{
			fprintf (stderr, "Unknown callsign %s\n", ptr);
			return 0;
		}

		/*
		 * Open the socket into the kernel.
		 */
		if ((s = socket(AF_AX25, SOCK_SEQPACKET, 0)) < 0) {
			sprintf(buffer, "ERROR: cannot open AX.25 socket, %s\r", strerror(errno));
			err(buffer);
		}

		bindlen = sizeof(struct full_sockaddr_ax25);

		break;

	case AF_ROSE:
		axbind.rsaddr.srose_family = AF_ROSE;
		axbind.rsaddr.srose_ndigis = 0;

		if (convert_call_entry(sender, axbind.rsaddr.srose_call.ax25_call) == -1) {
			sprintf(buffer, "ERROR: invalid callsign - %s\r", sender);
			err(buffer);
		}

		/* Port */
		addr = rs_config_get_addr (NULL);
		convert_rose_address (addr, axbind.rsaddr.srose_addr.rose_addr);

		if ((addrlen = rs_convert_call(ptr, &axconnect.rsaddr)) == -1)
		{
			fprintf (stderr, "Unknown callsign %s\n", ptr);
			return 0;
		}

		/*
		 * Open the socket into the kernel.
		 */
		if ((s = socket(AF_ROSE, SOCK_SEQPACKET, 0)) < 0) {
			sprintf(buffer, "ERROR: cannot open ROSE socket, %s\r", strerror(errno));
			err(buffer);
		}

		bindlen = sizeof(struct sockaddr_rose);

		break;
	}

	buflen = BUFLEN;
	setsockopt(s, SOL_SOCKET, SO_RCVBUF, &buflen, sizeof(buflen));
	setsockopt(s, SOL_SOCKET, SO_SNDBUF, &buflen, sizeof(buflen));

	/*
	 * Set our AX.25 callsign and AX.25 port callsign accordingly.
	 */
	if (bind(s, (struct sockaddr *)&axbind, addrlen) != 0) {
		sprintf(buffer, "ERROR: cannot bind AX.25 socket, %s\r", strerror(errno));
		err(buffer);
	}

	/*
	 * If no response in 300 seconds, go away.
	 */
	alarm(300);

	signal(SIGALRM, alarm_handler);

	/*
	 * Lets try and connect to the far end.
	 */
	if (connect(s, (struct sockaddr *)&axconnect, addrlen) != 0) {

		if (type == AF_ROSE)
		{
			char origin[80];
			struct rose_facilities_struct facilities;
			struct rose_cause_struct rose_cause;
			
			rose_cause.cause      = ROSE_LOCAL_PROCEDURE;
			rose_cause.diagnostic = 0x12;

			ioctl(s, SIOCRSGCAUSE, &rose_cause);

			origin[0] = '\0';
			if ((ioctl(s, SIOCRSGFACILITIES, &facilities) != -1) &&
				(facilities.fail_call.ax25_call[0]))
			{
				sprintf(origin, " at %s @%s",
					ax2asc(&facilities.fail_call),
					rose2asc(&facilities.fail_addr));
			}

			sprintf(address, "*** No answer from %s%s\r*** %02X%02X - %s\r",
				ptr,
				origin,
				rose_cause.cause, 
				rose_cause.diagnostic, 
				reason(rose_cause.cause, rose_cause.diagnostic));
		}
		else
		{
			switch (errno) {
			case ECONNREFUSED:
				strcpy(address, "*** Connection refused - aborting\r");
				break;
			case ENETUNREACH:
				strcpy(address, "*** No known route - aborting\r");
				break;
			case EINTR:
				strcpy(address, "*** Connection timed out - aborting\r");
				break;
			default:
				sprintf(address, "*** Cannot connect (%s)\r", strerror(errno));
				break;
			}
		}
		err(address);
	}

	/*
	 * We got there.
	 */
	alarm(0);

	sprintf(buffer, "*** Connection done\r");
	write(STDOUT_FILENO, buffer, strlen(buffer));

	/*
	 * Loop until one end of the connection goes away.
	 */
	for (;;) {
		FD_ZERO(&read_fd);
		FD_SET(STDIN_FILENO, &read_fd);
		FD_SET(s, &read_fd);
		
		select(s + 1, &read_fd, NULL, NULL, NULL);

		if (FD_ISSET(s, &read_fd)) 
		{
			if ((n = read(s, buffer, sizeof(buffer))) <= 0) 
			{
				if (type == AF_ROSE)
				{
					int ret;
					char origin[80];
					struct rose_facilities_struct facilities;
					struct rose_cause_struct rose_cause;
					rose_cause.cause      = ROSE_LOCAL_PROCEDURE;
					rose_cause.diagnostic = 0x12;

					ioctl(s, SIOCRSGCAUSE, &rose_cause);

					origin[0] = '\0';
					ret = ioctl(s, SIOCRSGFACILITIES, &facilities);
					if ((ret != -1) && (facilities.fail_call.ax25_call[0]))
					{
						sprintf(origin, " at %s @ %s",
							ax2asc(&facilities.fail_call),
							fpac2asc(&facilities.fail_addr));
					}

					sprintf(address, "\r*** Disconnected%s\r*** %02X%02X - %s\r",
						origin,
						rose_cause.cause, 
						rose_cause.diagnostic, 
						reason(rose_cause.cause, rose_cause.diagnostic));
				}
				else	
					strcpy(address, "\r*** Disconnected\r");
				err(address);
			}
			write(STDOUT_FILENO, buffer, n);
		}

		if (FD_ISSET(STDIN_FILENO, &read_fd)) 
		{
			if ((n = read(STDIN_FILENO, buffer, 250)) <= 0) 
			{
				if (type == AF_ROSE)
				{
					int ret;
					char origin[80];
					struct rose_facilities_struct facilities;
					struct rose_cause_struct rose_cause;
					
					rose_cause.cause      = ROSE_LOCAL_PROCEDURE;
					rose_cause.diagnostic = 0x12;

					ioctl(s, SIOCRSGCAUSE, &rose_cause);

					origin[0] = '\0';
					ret = ioctl(s, SIOCRSGFACILITIES, &facilities);
					if ((ret != -1) && (facilities.fail_call.ax25_call[0]))
					{
						sprintf(origin, " at %s @ %s",
							ax2asc(&facilities.fail_call),
							fpac2asc(&facilities.fail_addr));
					}

					sprintf(address, "\r*** Disconnected%s\r*** %02X%02X - %s\r",
						origin,
						rose_cause.cause, 
						rose_cause.diagnostic, 
						reason(rose_cause.cause, rose_cause.diagnostic));
				}
				else	
					strcpy(address, "\r*** Disconnected\r");
				err(address);
			}
			write(s, buffer, n);
		}
	}

	return 0;
}
