#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <linux/ax25.h>
#include <linux/rose.h>

#include "axutils.h"
#include "axconfig.h"

#define	AX25_HBIT	0x80

void alarm_handler(int sig)
{
}

void err(char *message)
{
	write(STDOUT_FILENO, message, strlen(message));
	exit(1);
}

int main(int argc, char **argv)
{
	char buffer[512], *addr;
	fd_set read_fd;
	int n = 0, s, addrlen, yes = 1;
	struct full_sockaddr_ax25 axbind, axconnect;
	struct sockaddr_rose rosesock, rosepeer;

	/*
	 * Arguments should be "rsdwnlnk ax25port ax25call"
	 */
	if (argc != 3) {
		strcpy(buffer, "ERROR: invalid number of parameters\r");
		err(buffer);
	}

	if (ax25_config_load_ports() == 0) {
		strcpy(buffer, "ERROR: problem with axports file\r");
		err(buffer);
	}

	addrlen = sizeof(struct sockaddr_rose);

	if (getsockname(STDIN_FILENO, (struct sockaddr *)&rosesock, &addrlen) == -1) {
		sprintf(buffer, "ERROR: cannot getsockname - %s\r", strerror(errno));
		err(buffer);
	}

	addrlen = sizeof(struct sockaddr_rose);

	if (getpeername(STDIN_FILENO, (struct sockaddr *)&rosepeer, &addrlen) == -1) {
		sprintf(buffer, "ERROR: cannot getpeername - %s\r", strerror(errno));
		err(buffer);
	}

	/*
	 * Parse the passed values for correctness.
	 */
	axbind.fsa_ax25.sax25_family = AF_AX25;
	axbind.fsa_ax25.sax25_ndigis = 1;
	axbind.fsa_ax25.sax25_call   = rosepeer.srose_call;

	if ((addr = ax25_config_get_addr(argv[1])) == NULL) {
		sprintf(buffer, "ERROR: invalid AX.25 port name - %s\r", argv[1]);
		err(buffer);
	}

	if (convert_call_entry(addr, axbind.fsa_digipeater[0].ax25_call) == -1) {
		sprintf(buffer, "ERROR: invalid AX.25 port callsign - %s\r", argv[1]);
		err(buffer);
	}

	axconnect.fsa_ax25.sax25_family = AF_AX25;
	axconnect.fsa_ax25.sax25_call   = rosesock.srose_call;

	/*
	 *	The path at the far end has a digi in it.
	 */
	if (rosepeer.srose_ndigis == 1) {
		axconnect.fsa_digipeater[n] = rosepeer.srose_digi;
		axconnect.fsa_digipeater[n].ax25_call[6] |= AX25_HBIT;
		n++;
	}

	/*
	 *	Incoming call has a different DNIC
	 */
	if (memcmp(rosepeer.srose_addr.rose_addr, rosesock.srose_addr.rose_addr, 2) != 0) {
		addr = rose2asc(&rosepeer.srose_addr);
		addr[4] = '\0';
		if (convert_call_entry(addr, axconnect.fsa_digipeater[n].ax25_call) == -1) {
			sprintf(buffer, "ERROR: invalid callsign - %s\r", addr);
			err(buffer);
		}
		axconnect.fsa_digipeater[n].ax25_call[6] |= AX25_HBIT;
		n++;		
	}

	/*
	 *	Put the remote address sans DNIC into the digi chain.
	 */
	addr = rose2asc(&rosepeer.srose_addr);
	if (convert_call_entry(addr + 4, axconnect.fsa_digipeater[n].ax25_call) == -1) {
		sprintf(buffer, "ERROR: invalid callsign - %s\r", addr + 4);
		err(buffer);
	}
	axconnect.fsa_digipeater[n].ax25_call[6] |= AX25_HBIT;
	n++;		

	/*
	 *	And my local ROSE callsign.
	 */
	if (convert_call_entry(argv[2], axconnect.fsa_digipeater[n].ax25_call) == -1) {
		sprintf(buffer, "ERROR: invalid callsign - %s\r", argv[2]);
		err(buffer);
	}
	axconnect.fsa_digipeater[n].ax25_call[6] |= AX25_HBIT;
	n++;

	/*
	 *	A digi has been specified for this end.
	 */
	if (rosesock.srose_ndigis == 1) {
		axconnect.fsa_digipeater[n] = rosesock.srose_digi;
		n++;
	}

	axconnect.fsa_ax25.sax25_ndigis = n;

	addrlen = sizeof(struct full_sockaddr_ax25);

	/*
	 * Open the socket into the kernel.
	 */
	if ((s = socket(AF_AX25, SOCK_SEQPACKET, 0)) < 0) {
		sprintf(buffer, "ERROR: cannot open AX.25 socket, %s\r", strerror(errno));
		err(buffer);
	}

	if (setsockopt(s, SOL_AX25, AX25_IAMDIGI, &yes, sizeof(yes)) == -1) {
		sprintf(buffer, "ERROR: cannot setsockopt, %s\r", strerror(errno));
		err(buffer);
	}

	/*
	 * Set our AX.25 callsign and AX.25 port callsign accordingly.
	 */
	if (bind(s, (struct sockaddr *)&axbind, addrlen) != 0) {
		sprintf(buffer, "ERROR: cannot bind AX.25 socket, %s\r", strerror(errno));
		err(buffer);
	}

	/*
	 * If no response in 30 seconds, go away.
	 */
	alarm(30);

	signal(SIGALRM, alarm_handler);

	/*
	 * Lets try and connect to the far end.
	 */
	if (connect(s, (struct sockaddr *)&axconnect, addrlen) != 0) {
		switch (errno) {
			case ECONNREFUSED:
				strcpy(buffer, "*** Connection refused\r");
				break;
			case ENETUNREACH:
				strcpy(buffer, "*** No known route\r");
				break;
			case EINTR:
				strcpy(buffer, "*** Connection timed out\r");
				break;
			default:
				sprintf(buffer, "ERROR: cannot connect to AX.25 callsign, %s\r", strerror(errno));
				break;
		}

		err(buffer);
	}

	/*
	 * We got there.
	 */
	alarm(0);

	strcpy(buffer, "*** Connected\r");
	write(STDOUT_FILENO, buffer, strlen(buffer));

	/*
	 * Loop until one end of the connection goes away.
	 */
	for (;;) {
		FD_ZERO(&read_fd);
		FD_SET(STDIN_FILENO, &read_fd);
		FD_SET(s, &read_fd);
		
		select(s + 1, &read_fd, NULL, NULL, NULL);

		if (FD_ISSET(s, &read_fd)) {
			if ((n = read(s, buffer, 512)) == -1)
				break;
			write(STDOUT_FILENO, buffer, n);
		}

		if (FD_ISSET(STDIN_FILENO, &read_fd)) {
			if ((n = read(STDIN_FILENO, buffer, 512)) == -1) {
				close(s);
				break;
			}
			write(s, buffer, n);
		}
	}

	return 0;
}
