/*****************************************************************************/

/*
 *	smdiag.c  -- kernel soundcard radio modem driver diagnostics utility.
 *
 *	Copyright (C) 1996  Thomas Sailer (sailer@ife.ee.ethz.ch)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Please note that the GPL allows you to use the driver, NOT the radio.
 *  In order to use the radio, you need a license from the communications
 *  authority of your country.
 *
 *
 * History:
 *   0.1  26.06.96  Started
 */

/*****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <limits.h>
#include <linux/if.h>
#include <netinet/in.h>
#include <linux/if_ether.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <linux/hdlcdrv.h>
#include <linux/soundmodem.h>

/* ---------------------------------------------------------------------- */

static int sock_sm;
static struct ifreq ifr_sm;
static char *progname;
static Display *display = NULL;
static Window window;
static Pixmap pixmap;
static unsigned long col_zeroline;
static unsigned long col_background;
static unsigned long col_trace;
static GC gr_context;
static int xmul;

/* ---------------------------------------------------------------------- */

static int x_error_handler(Display *disp, XErrorEvent *evt)
{
    char err_buf[256], mesg[256], number[256];
    char *mtype = "XlibMessage";

    XGetErrorText(disp, evt->error_code, err_buf, sizeof(err_buf));
    fprintf(stderr, "X Error: %s\n", err_buf);
    XGetErrorDatabaseText(disp, mtype, "MajorCode", "Request Major code %d", 
			  mesg, sizeof(mesg));
    fprintf(stderr, mesg, evt->request_code);
    sprintf(number, "%d", evt->request_code);
    XGetErrorDatabaseText(disp, "XRequest", number, "", err_buf, 
			  sizeof(err_buf));
    fprintf(stderr, " (%s)\n", err_buf);
    abort();
}

/* ---------------------------------------------------------------------- */

static int do_sm_ioctl(int cmd, struct sm_ioctl *par)
{
	struct ifreq ifr = ifr_sm;
	
	ifr.ifr_data = (caddr_t)par;
	par->cmd = cmd;
	return ioctl(sock_sm, SIOCDEVPRIVATE, &ifr);
}

/* ---------------------------------------------------------------------- */

static int restore_ifflags(void)
{
	if (ioctl(sock_sm, SIOCSIFFLAGS, &ifr_sm) < 0) {
		fprintf(stderr, "%s: Error %s (%i), cannot restore interface"
			" flags\n", progname, strerror(errno), errno);
		return -1;
	}
	return 0;
}

/* ---------------------------------------------------------------------- */

static void restore_ifflags_sig(int signal)
{
	printf("signal %i caught\n", signal);
	exit(restore_ifflags() ? 1 : 0);
}

/* ---------------------------------------------------------------------- */

#define WIDTH 512
#define HEIGHT (constell ? 512 : 256)

static int openwindow(char *disp, int constell, int samplesperbit)
{
        XSetWindowAttributes attr;
        XGCValues gr_values;
        XColor color, dummy;
        XSizeHints sizehints;

        if (!(display = XOpenDisplay(NULL)))
                return -1;
        XSetErrorHandler(x_error_handler);
        XAllocNamedColor(display, DefaultColormap(display, 0), "red",
                         &color, &dummy);
	col_zeroline = color.pixel;
	col_background = WhitePixel(display, 0);
	col_trace = BlackPixel(display, 0);
        attr.background_pixel = col_background;
        if (!(window = XCreateWindow(display, XRootWindow(display, 0), 
				     200, 200, WIDTH, HEIGHT, 5, 
				     DefaultDepth(display, 0), 
				     InputOutput, DefaultVisual(display, 0),
				     CWBackPixel, &attr))) {
		fprintf(stderr, "smdiag: unable to open X window\n");
		exit(1);
	}
	if (!(pixmap = XCreatePixmap(display, window, WIDTH, HEIGHT,
				     DefaultDepth(display, 0)))) {
		fprintf(stderr, "smdiag: unable to open offscreen pixmap\n");
		exit(1);
	}
        xmul = WIDTH / (2*(samplesperbit > 0 ? samplesperbit : 1));
        XSelectInput(display, window, KeyPressMask | StructureNotifyMask
		     | ExposureMask) ;
        XAllocNamedColor(display, DefaultColormap(display, 0), "red",
                         &color, &dummy);
        gr_values.foreground = col_trace;
        gr_values.line_width = 1;
        gr_values.line_style = LineSolid;
        gr_context = XCreateGC(display, window, GCForeground | GCLineWidth |
                               GCLineStyle, &gr_values);
        XStoreName(display, window, "diagnostics");
        /*
         * Do not allow the window to be resized
         */
        memset(&sizehints, 0, sizeof(sizehints));
        sizehints.min_width = sizehints.max_width = WIDTH;
        sizehints.min_height = sizehints.max_height = HEIGHT;
        sizehints.flags = PMinSize | PMaxSize;
        XSetWMNormalHints(display, window, &sizehints);
        XMapWindow(display, window);
        XSynchronize(display, 1);
        return 0;
}

#undef WIDTH
#undef HEIGHT

/* ---------------------------------------------------------------------- */

static void closewindow(void)
{
        if (!display)
                return;
        XDestroyWindow(display, window);
        XCloseDisplay(display);
        display = NULL;
}

/* ---------------------------------------------------------------------- */

#define XCOORD(x) ((x) * xm)
#define YCOORD(y) ((SHRT_MAX - (int)(y)) * winattrs.height / USHRT_MAX)

static void drawdata(short *data, int len, int replace, int xm)
{
	int cnt;
        GC gc;
        XGCValues gcv;
        XWindowAttributes winattrs;	

        if (!display || !pixmap)
                return;
        XGetWindowAttributes(display, window, &winattrs);
	gcv.line_width = 1;
        gcv.line_style = LineSolid;
        gc = XCreateGC(display, pixmap, GCLineWidth | GCLineStyle, &gcv);
        XSetState(display, gc, col_background, col_background, GXcopy, 
		  AllPlanes);
	if (replace)
		XFillRectangle(display, pixmap, gc, 0, 0, 
			       winattrs.width, winattrs.height);
	else
		XCopyArea(display, window, pixmap, gr_context, 0, 0, 
			  winattrs.width, winattrs.height, 0, 0);
        XSetForeground(display, gc, col_trace);
	for (cnt = 0; cnt < len-1; cnt++)
		XDrawLine(display, pixmap, gc, XCOORD(cnt), YCOORD(data[cnt]),
			  XCOORD(cnt+1), YCOORD(data[cnt+1]));
        XSetForeground(display, gc, col_zeroline);
	XDrawLine(display, pixmap, gc, 0, YCOORD(0), winattrs.width,
		  YCOORD(0));
	XCopyArea(display, pixmap, window, gr_context, 0, 0, winattrs.width,
                  winattrs.height, 0, 0);
        XFreeGC(display, gc);
	XSync(display, 0);
}

#undef XCOORD
#undef YCOORD

/* ---------------------------------------------------------------------- */

#define XCOORD(x) ((SHRT_MAX - (int)(x)) * winattrs.width / USHRT_MAX)
#define YCOORD(y) ((SHRT_MAX - (int)(y)) * winattrs.height / USHRT_MAX)

static void drawconstell(short *data, int len)
{
	int cnt;
        GC gc;
        XGCValues gcv;
        XWindowAttributes winattrs;	

        if (!display || !pixmap)
                return;
        XGetWindowAttributes(display, window, &winattrs);
	gcv.line_width = 1;
        gcv.line_style = LineSolid;
        gc = XCreateGC(display, pixmap, GCLineWidth | GCLineStyle, &gcv);
        XSetState(display, gc, col_background, col_background, GXcopy, 
		  AllPlanes);
	XCopyArea(display, window, pixmap, gr_context, 0, 0, 
		  winattrs.width, winattrs.height, 0, 0);
        XSetForeground(display, gc, col_trace);
	for (cnt = 0; cnt < len-1; cnt += 2)
		XDrawPoint(display, pixmap, gc, 
			   XCOORD(data[cnt]), YCOORD(data[cnt+1]));
        XSetForeground(display, gc, col_zeroline);
	XDrawLine(display, pixmap, gc, 0, YCOORD(0), winattrs.width, YCOORD(0));
	XDrawLine(display, pixmap, gc, XCOORD(0), 0, XCOORD(0), winattrs.height);
	XCopyArea(display, pixmap, window, gr_context, 0, 0, winattrs.width,
                  winattrs.height, 0, 0);
        XFreeGC(display, gc);
	XSync(display, 0);
}

#undef XCOORD
#undef YCOORD

/* ---------------------------------------------------------------------- */

static void clearwindow(void)
{
        XWindowAttributes winattrs;	
	GC gc;
        XGCValues gcv;

        if (!display || !pixmap)
                return;
        XGetWindowAttributes(display, window, &winattrs);
	gcv.line_width = 1;
        gcv.line_style = LineSolid;
        gc = XCreateGC(display, pixmap, GCLineWidth | GCLineStyle, &gcv);
        XSetState(display, gc, col_background, col_background, GXcopy, 
		  AllPlanes);
	XFillRectangle(display, pixmap, gc, 0, 0, 
		       winattrs.width, winattrs.height);
        XSetForeground(display, gc, col_zeroline);
        XClearArea(display, window, 0, 0, 0, 0, False);
	XCopyArea(display, pixmap, window, gr_context, 0, 0, winattrs.width,
                  winattrs.height, 0, 0);
        XFreeGC(display, gc);	
}

/* ---------------------------------------------------------------------- */

static Bool predicate(Display *display, XEvent *event, char *arg)
{
        return True;
}

/* ---------------------------------------------------------------------- */

static char *getkey(void)
{
        XWindowAttributes winattrs;	
        XEvent evt;
        static char kbuf[32];
        int i;

        if (!display)
                return NULL;
	XSync(display, 0);
        if (!XCheckIfEvent(display, &evt, predicate, NULL)) 
                return NULL;
        switch (evt.type) {
        case KeyPress:
                i = XLookupString((XKeyEvent *)&evt, kbuf, sizeof(kbuf)-1, 
                                  NULL, NULL);
                if (!i)
                        return NULL;
                kbuf[i] = 0;
                return kbuf;
        case DestroyNotify:
                XCloseDisplay(display);
                display = NULL;
                return NULL;
	case Expose:
		XGetWindowAttributes(display, window, &winattrs);
		XCopyArea(display, pixmap, window, gr_context, 0, 0, 
			  winattrs.width, winattrs.height, 0, 0);
		return NULL;
        default:
                return NULL;
        }
        return NULL;
}

/* ---------------------------------------------------------------------- */

static void printmode(unsigned int mode, unsigned int trigger)
{
	printf("Source: %s%s\n", (mode == SM_DIAGMODE_DEMOD) ?
	       "demodulator (eye diagram)" : "input (oscilloscope)",
	       (trigger & SM_DIAGFLAG_DCDGATE) ? " gated with DCD" : "");
}

/* ---------------------------------------------------------------------- */

static const char *usage_str = 
"[-d display] [-i smif] [-c] [-e]\n"
"  -d: display host\n"
"  -i: specify the name of the baycom kernel driver interface\n"
"  -c: toggle carrier trigger\n"
"  -e: eye diagram mode\n\n"
"  -p: constellation plot\n\n";

int main(int argc, char *argv[])
{
	char *name_sm = "sm0";
	char *disp = NULL;
	unsigned int mode = SM_DIAGMODE_INPUT;
	unsigned int trigger = 0;
	struct sm_ioctl smi;
	short data[256];
	char *cp;
	int ret;

	progname = *argv;
	printf("%s: Version 0.1; (C) 1996 by Thomas Sailer HB9JNX/AE4WA\n", *argv);
	while ((ret = getopt(argc, argv, "d:i:ecp")) != -1) {
		switch (ret) {
		case 'd':
			disp = optarg;
			break;
		case 'i':
			name_sm = optarg;
			break;
		case 'e':
			mode = SM_DIAGMODE_DEMOD;
			trigger = SM_DIAGFLAG_DCDGATE;
			break;
		case 'c':
			trigger ^= SM_DIAGFLAG_DCDGATE;
			break;
		case 'p':
			mode = SM_DIAGMODE_CONSTELLATION;
			break;
		default:
			printf("usage: %s %s", *argv, usage_str);
			exit(-1);
		}
	}
	if ((sock_sm = socket(PF_INET, SOCK_PACKET, htons(ETH_P_AX25))) < 0) {
		perror("socket");
		exit(-1);
	}
	strcpy(ifr_sm.ifr_name, name_sm);
	if (ioctl(sock_sm, SIOCGIFFLAGS, &ifr_sm) < 0) {
		perror("ioctl: SIOCGIFFLAGS");
		exit(-1);
	}
	if (!(ifr_sm.ifr_flags & IFF_UP)) {
		fprintf(stderr, "interface %s down\n", ifr_sm.ifr_name);
		exit(1);
	}
	if (!(ifr_sm.ifr_flags & IFF_RUNNING)) {
		fprintf(stderr, "interface %s not running\n", ifr_sm.ifr_name);
		exit(1);
	}
	signal(SIGTERM, restore_ifflags_sig);
	signal(SIGQUIT, restore_ifflags_sig);
	if (atexit((void (*)(void))restore_ifflags)) {
		perror("atexit");
		restore_ifflags();
	}		
	printmode(mode, trigger);
	for (;;) {
		smi.data.diag.mode = mode;
		smi.data.diag.flags = trigger;
		smi.data.diag.datalen = sizeof(data) / sizeof(short);
		smi.data.diag.data = data;
		if (do_sm_ioctl(SMCTL_DIAGNOSE, &smi)) {
			perror("do_sm_ioctl: SMCTL_DIAGNOSE");
			exit(1);
		}
		if (smi.data.diag.mode == mode && 
		    smi.data.diag.flags & SM_DIAGFLAG_VALID) {
			if (!display) {
				openwindow(disp, mode == SM_DIAGMODE_CONSTELLATION,
					   smi.data.diag.samplesperbit);
				clearwindow();
			}
			if (mode == SM_DIAGMODE_CONSTELLATION)
				drawconstell(smi.data.diag.data, smi.data.diag.datalen);
			else
				drawdata(smi.data.diag.data, smi.data.diag.datalen,
					 smi.data.diag.mode == SM_DIAGMODE_INPUT,
					 smi.data.diag.mode == SM_DIAGMODE_INPUT ? 5:xmul);
		} else
			usleep(100000L);
		if (display) {
			if ((cp = getkey())) {
				for (; *cp; cp++) {
					printf("char pressed: '%c'\n", *cp);
					switch (*cp) {
					case 'c':
					case 'C':
						clearwindow();
						printmode(mode, trigger);
						break;
					case 'q':
					case 'Q':
						closewindow();
						break;
					case 'i':
					case 'I':
						if (mode == SM_DIAGMODE_CONSTELLATION)
							break;
						mode = SM_DIAGMODE_INPUT;
						clearwindow();
						printmode(mode, trigger);
						break;
					case 'e':
					case 'E':
						if (mode == SM_DIAGMODE_CONSTELLATION)
							break;
						mode = SM_DIAGMODE_DEMOD;
						clearwindow();
						printmode(mode, trigger);
						break;
					case 'd':
					case 'D':
						trigger ^= SM_DIAGFLAG_DCDGATE;
						clearwindow();
						printmode(mode, trigger);
						break;
					}
				}
			}
			if (!display)
				exit(0);
		}
	}
}

/* ---------------------------------------------------------------------- */
