#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>
#include <syslog.h>
#include <linux/socket.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <linux/ax25.h>

#include "node.h"
#include "io.h"
#include "axutils.h"
#include "axconfig.h"
#include "nrconfig.h"
#include "procutils.h"

char NrId[20]	= "";

void node_perror(char *str, int err)
{
	int oldmode;

	oldmode = set_eolmode(User.fd, EOLMODE_TEXT);
	tprintf("%s} ", NrId);
	if (str)
		usputs(str, User.fd);
	if (str && err != -1)
		usputs(": ", User.fd);
	if (err != -1)
		usputs(strerror(err), User.fd);
	usputs("\n", User.fd);
	usflush(User.fd);
	set_eolmode(User.fd, oldmode);
	syslog(LOG_ERR, "%s: %s\n", str, strerror(err));
}

char *print_node(const char *alias, const char *call)
{
	static char node[17];

	sprintf(node, "%s%s%s",
		!strcmp(alias, "*") ? "" : alias,
		!strcmp(alias, "*") ? "" : ":",
		call);
	return node;
}

static char *parse_string(char *str)
{
	char *cp = str;
	unsigned long num;

	while (*str != '\0' && *str != '\"') {
		if (*str == '\\') {
			str++;
			switch (*str++) {
			case 'n':
				*cp++ = '\n';
				break;
			case 't':
				*cp++ = '\t';
				break;
			case 'v':
				*cp++ = '\v';
				break;
			case 'b':
				*cp++ = '\b';
				break;
			case 'r':
				*cp++ = '\r';
				break;
			case 'f':
				*cp++ = '\f';
				break;
			case 'a':
				*cp++ = '\007';
				break;
			case '\\':
				*cp++ = '\\';
				break;
			case '\"':
				*cp++ = '\"';
				break;
			case 'x':
				num = strtoul(--str, &str, 16);
				*cp++ = (char) num;
				break;
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
				num = strtoul(--str, &str, 8);
				*cp++ = (char) num;
				break;
			case '\0':
				return NULL;
			default:
				*cp++ = *(str - 1);
				break;
			};
		} else {
			*cp++ = *str++;
		}
	}
	if (*str == '\"')
		str++;		/* skip final quote */
	*cp = '\0';		/* terminate string */
	return str;
}

int parse_args(char *argv[],char *cmd)
{
	int ct = 0;
	int quoted;

	while (ct < 31)
	{
		quoted = 0;
		while (*cmd && isspace(*cmd))
			cmd++;
		if (*cmd == 0)
			break;
		if (*cmd == '"') {
			quoted++;
			cmd++;
		}
		argv[ct++] = cmd;
		if (quoted) {
			if ((cmd = parse_string(cmd)) == NULL)
				return 0;
		} else {
			while (*cmd && !isspace(*cmd))
				cmd++;
		}
		if (*cmd)
			*cmd++ = 0;
	}
	argv[ct] = NULL;
	return ct;
}

int cmdparse(struct cmd *cmds, char *cmdline)
{
	struct cmd *cmdp;
	int argc;
	char *argv[32];

	if ((argc = parse_args(argv, cmdline)) == 0 || *argv[0] == '#')
		return 0;
	for (cmdp = cmds; cmdp->function != NULL; cmdp++)
		if (strncasecmp(cmdp->name, argv[0], strlen(argv[0])) == 0)
			break;
	if (cmdp->function == NULL)
		return -1;
	return (*cmdp->function)(argc, argv);
}

/*
 * Do some validity checking for callsign pointed to by `s'.
 */
static int check_call(const char *s)
{
	int len = 0;
	int nums = 0;
	int ssid = 0;
	char *p[1];

	if (s == NULL)
		return -1;
	while (*s && *s != '-') {
		if (!isalnum(*s))
			return -1;
		if (isdigit(*s))
			nums++;
		len++;
		s++;
	}
	if (*s == '-') {
		if (!isdigit(*++s))
			return -1;
		ssid = strtol(s, p, 10);
		if (**p)
			return -1;
	}
	if (len < 4 || len > 6 || !nums || nums > 2 || ssid < 0 || ssid > 15)
		return -1;
	return 0;
}

static void alarm_handler(int sig)
{
	set_eolmode(User.fd, EOLMODE_TEXT);
	tprintf("\n%s} Timeout! Disconnecting...\n", NrId);
	do_bye(0, NULL);
}

static void term_handler(int sig)
{
	set_eolmode(User.fd, EOLMODE_TEXT);
	tprintf("\n%s} System going down! Disconnecting...\n", NrId);
	do_bye(0, NULL);
}

int main(int argc, char *argv[])
{
	struct full_sockaddr_ax25 sax;
	int sax_len = sizeof(struct full_sockaddr_ax25);
	struct sockaddr_in *sin = NULL;
	char *p, buf[256], *pw;
	int paclen;
	struct alias *ap;
	struct ecmd *ep;

	signal(SIGALRM, alarm_handler);
	signal(SIGTERM, term_handler);
	openlog("node", LOG_PID, LOG_DAEMON);
	if (ax25_config_load_ports() == 0) {
		syslog(LOG_ERR, "No AX.25 port data configured\n");
		return 1;
	}
	nr_config_load_ports();
	sprintf(NrId, "%s:%s", nr_config_get_alias(NULL), nr_config_get_addr(NULL));
	if (getpeername(STDOUT_FILENO, (struct sockaddr *)&sax, &sax_len) == -1) {
		if (errno != ENOTSOCK) {
			syslog(LOG_ERR, "getpeername: %m\n");
			return 1;
		}
		User.ul_type = AF_UNSPEC;
	} else
		User.ul_type = sax.fsa_ax25.sax25_family;
	switch (User.ul_type) {
	case AF_AX25:
		strcpy(User.call, ax2asc(&sax.fsa_ax25.sax25_call));
		if (getsockname(STDOUT_FILENO, (struct sockaddr *)&sax, &sax_len) == -1) {
			syslog(LOG_ERR, "getsockname: %m\n");
			return 1;
		}
		strcpy(User.ul_name, ax25_config_get_port(&sax.fsa_digipeater[0]));
		paclen = ax25_config_get_paclen(User.ul_name);
		p = AX25_EOL;
		break;
	case AF_NETROM:
		strcpy(User.call, ax2asc(&sax.fsa_ax25.sax25_call));
		strcpy(User.ul_name, ax2asc(&sax.fsa_digipeater[0]));
		paclen = nr_config_get_paclen(NULL);
		p = NETROM_EOL;
		break;
	case AF_INET:
		sin = (struct sockaddr_in *)&sax;
		strcpy(User.ul_name, inet_ntoa(sin->sin_addr));
		paclen = 1024;
		p = INET_EOL;
		break;
	case AF_UNSPEC:
		strcpy(User.ul_name, "local");
		if ((p = get_call(getuid())) == NULL) {
			syslog(LOG_ERR, "No uid->callsign association found\n");
			return 1;
		}
		strcpy(User.call, p);
		paclen = 1024;
		p = UNSPEC_EOL;
		break;
	default:
		syslog(LOG_ERR, "Unsupported address family\n");
		return 1;
	}
	if (init_io(User.fd, paclen, p) == -1) {
		write(User.fd, "Error initializing I/O.\r\n", 25);
		return 1;
	}
	if (User.ul_type == AF_INET) {
		set_telnetmode(User.fd, 1);
		tn_do_linemode(User.fd);
	}
	if (read_config() == -1) {
		end_io(User.fd);
		return 1;
	}
	if (User.call[0] == 0) {
		tprintf("\n%s (%s)\n\n", VERSION, HostName);
		tprintf("login: ");
		usflush(User.fd);
		if ((p = readline(User.fd)) == NULL) {
			end_io(User.fd);
			return 0;
		}
		strncpy(User.call, p, 9);
		User.call[9] = 0;
		strlwr(User.call);
	}
	if (check_call(User.call) == -1) {
		tprintf("%s} Invalid callsign\n", NrId);
		end_io(User.fd);
		return 1;
	}
	if ((pw = read_perms(User.call, User.ul_type, sin ? sin->sin_addr.s_addr : 0L, argv[1])) == NULL) {
		tprintf("%s} Sorry, I'm not allowed to talk to you...\n", NrId);
		end_io(User.fd);
		return 1;
	} else if (strcmp(pw, "*") != 0) {
		tprintf("Password: ");
		if (User.ul_type == AF_INET) {
			tn_will_echo(User.fd);
			set_eolmode(User.fd, EOLMODE_BINARY);
		}
		usflush(User.fd);
		p = readline(User.fd);
		if (User.ul_type == AF_INET) {
			tn_wont_echo(User.fd);
			set_eolmode(User.fd, EOLMODE_TEXT);
			tprintf("\n");
		}
		if (p == NULL || strcmp(p, pw) != 0) {
			tprintf("\n%s} Login failed\n", NrId);
			end_io(User.fd);
			return 1;
		}
	}
	free(pw);
	tprintf("%s} Welcome to %s network node\n\n", NrId, HostName);
	login_user();
	while (1)
	{
		usflush(User.fd);
		User.state = STATE_IDLE;
		time(&User.cmdtime);
		update_user();
		alarm(IdleTimeout);
		if ((p = readline(User.fd)) == NULL)
			do_bye(0, NULL);
		alarm(IdleTimeout);
		time(&User.cmdtime);
		update_user();
		for (ap = Aliases; ap != NULL; ap = ap->next) {
			if (!strcasecmp(p, ap->name)) {
				strcpy(buf, ap->command);
				p = buf;
				break;
			}
		}
		for (ep = Extcmds; ep != NULL; ep = ep->next) {
			if (!strcasecmp(p, ep->name)) {
				extcmd(ep, p);
				break;
			}
		}
		if (ep == NULL && cmdparse(Nodecmds, p) == -1)
			tprintf("%s} Unknown command. Type ? for a list\n", NrId);
		tprintf("\n");
	}
}
