/*
 * TRK - Satellite tracking program based on Norad SGP/SDP model with
 *       curses interface
 *
 *	by Lapo Pieri IK5NAX  2000-2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send bugs reports, comments, critique, etc, to ik5nax@amsat.org
 */

#include <math.h>
#include "trk.h"

#define thetas 4.652418e-3  /* semidiametro del sole visto dalla terra */

/*
  Mi sembra che funzioni, ma e` stato messo a punto andando a tentoni,
  riscrivere la teoria e rimettere le formule un po' a pulito non sarebbe
  male...
*/
char sat_illum(satpos sat, satpos sun){
double rhoe, rhos, rs, thetae, theta, costheta;

 rhoe=sqrt(sat.x*sat.x+sat.y*sat.y+sat.z*sat.z);
 rs=sqrt(sun.x*sun.x+sun.y*sun.y+sun.z*sun.z);
 rhos=sqrt((sat.x-sun.x)*(sat.x-sun.x)+(sat.y-sun.y)*(sat.y-sun.y)+
	   (sat.z-sun.z)*(sat.z-sun.z));

 thetae=asin(R/rhoe);
 costheta=(sat.x*(sat.x-sun.x)+sat.y*(sat.y-sun.y)+sat.z*(sat.z-sun.z))
   /rhoe/rhos;
 theta=acos(costheta);


 /* Eclisse in ombra */
 /* if(thetae>thetas && theta<thetae-thetas) */
if(theta<thetae-thetas && costheta>0.)
   return 'U'; /* Umbral Eclipse */

 /* Eclisse in penombra */
 if(fabs(thetae-thetas)<theta && theta<thetae+thetas && costheta>0.)
   return 'P'; /* Penumbral Eclipse */

 /* Satellite visibile perche' sorto e con il sole sotto -6 gradi di elev */
 if(sat.el>0. && sun.el<-0.1047) 
   return 'V'; /* Visual Visible */

 /* Satellite non visibile perche' il sole e` troppo alto */
 if(sat.el>0. && sun.el>=-0.1047) 
   return 'S'; /* in Sunlight */

 /* Satellite illuminato ma non in vista */
 return 'I'; /* Illuminated */


 /*
   Non dovrebbe essere difficile scoprire i passaggi sul disco del sole (e poi
   ampliarlo alla luna) controllando che, durante el>0 az,el sat (forse meglio
   farlo con ra,dec?) non si discostino piu` di thetas dai corrispettivi per
   il sole (o la luna)
 */
}


double elev(double t, SAT *sats, OBS *obs){
double theta, rx, ry, rz, rS, rE, rZ, r;
float xo, yo, zo;

theta=tetha(t+J2000, obs->lon/M_PI*180.);
 (void) (*sgp4)(sats, (double)t);
 obsijk(theta, obs, &xo, &yo, &zo);
 
 rx=sats->x[0]*R-xo; ry=sats->x[1]*R-yo; rz=sats->x[2]*R-zo; 
 rS=sin(obs->lat)*cos(theta)*rx+sin(obs->lat)*sin(theta)*ry-cos(obs->lat)*rz;
 rE=-sin(theta)*rx+cos(theta)*ry;
 rZ=cos(obs->lat)*cos(theta)*rx+cos(obs->lat)*sin(theta)*ry+sin(obs->lat)*rz;
 
 r=sqrt(rS*rS+rE*rE+rZ*rZ);
 
return 180./M_PI*asin(rZ/r);
}



void single_sat_calc(double t, SAT *sats, OBS *obs, satpos *asp){
double thetav, rx, ry, rz, rS, rE, rZ, r, vx, vy, vz, az, el;
float xo, yo, zo;
const float oe=72.9212352e-6;
double ra, dec, ssplat, ssplon, h;

  thetav=tetha(t+J2000, obs->lon/M_PI*180.);
  (void) (*sgp4)(sats, (double)t);
  obsijk(thetav, obs, &xo, &yo, &zo);

  rx=sats->x[0]*R-xo; ry=sats->x[1]*R-yo; rz=sats->x[2]*R-zo; 
  asp->x=sats->x[0]*R; asp->y=sats->x[1]*R; asp->z=sats->x[2]*R;
  ijk2sez(rx, ry, rz, obs->lat, thetav, &rS, &rE, &rZ);
  sez2azel(rS, rE, rZ, &az, &el, &r);
  asp->az=az; asp->el=el; asp->range=r;

  vx=sats->dx[0]*R/86400.; vy=sats->dx[1]*R/86400.; vz=sats->dx[2]*R/86400.;
  
  asp->rangerate=((vx+oe*yo)*rx+(vy-oe*xo)*ry+vz*rz)/r;
  asp->norad=sats->norad; asp->class=sats->eclass;
  strcpy(asp->name, sats->name);
  
  asp->phase=sats->mean-sats->peri+2*M_PI;
  asp->phase=fmod(asp->phase, 2.*M_PI);
  
  ijk2radec(rx, ry, rz, &ra, &dec);
  SSP(sats->x[0]*R, sats->x[1]*R, sats->x[2]*R, t+J2000, &ssplat, &ssplon, &h);
  asp->ra=ra; asp->dec=dec; 
  asp->ssplon=ssplon; asp->ssplat=ssplat; asp->height=h;
 
  
#ifdef DEBUG
  mvprintw(16,1,"xs= %8.1f", sats->x[0]*R);
  mvprintw(16,15,"ys= %8.1f", sats->x[1]*R);
  mvprintw(16,29,"zs= %8.1f", sats->x[2]*R);
  mvprintw(17,1,"xo= %8.1f", xo);
  mvprintw(17,15,"yo= %8.1f", yo);
  mvprintw(17,29,"zo= %8.1f", zo);
  mvprintw(18,1,"vx= %8.3f", vx);
  mvprintw(18,15,"vy= %8.3f", vy);
  mvprintw(18,29,"vz= %8.3f", vz);
  mvprintw(19,0,"v0x= %8.3f", oe*xo);
  mvprintw(19,14,"v0y= %8.3f", oe*yo);
  mvprintw(19,28,"v0z= %8.3f", 0.);
#endif

}

/* Tutta da riscrivere, ci sono molti calcoli doppi */
void SSP(double x, double y, double z, double t,
	 double *ssplat, double *ssplon, double *height){
  double ilat, l, c;
  const double f=3.3528107e-3, e2=2*f-f*f, EPS=1e-6;

  *ssplon=atan2(y, x)-tetha(t, 0);
  while(*ssplon>M_PI) *ssplon-=2*M_PI;
  while(*ssplon<-M_PI) *ssplon+=2*M_PI;
  ilat=atan(z/sqrt(x*x+y*y));

  do{
    l=ilat;
    c=1/sqrt(1+e2*sin(l)*sin(l));
    ilat=atan((z-R*c*e2*sin(l))/sqrt(x*x+y*y));
  }while(fabs(l-ilat)>EPS);
  
  ilat=(ilat+l)/2; 
  while(ilat>M_PI/2.) ilat-=M_PI;
  while(ilat<-M_PI/2.) ilat+=M_PI;
 
  *ssplat=ilat;
  *height=sqrt(x*x+y*y)/cos(ilat)-R*c;
}

