/*
 *  This is my version of axl.c, written for the LBBS code to make it
 *    compatable with the kernel AX25 driver.  It appears to work, with
 *    my setup, so it'll probably not work else where :-).
 *
 *  This was inspired by the example code written by Alan Cox (GW4PTS)
 *    axl.c in AX25USER.TGZ from sunacm.swan.ac.uk.
 *
 *
 *  Copyright (C) 1995 by Darryl L. Miles, G7LED.
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *
 *  Just a quickie Feb 1995.
 *  It *was* just a quickie (at the time) but you'd know how these things
 *   just... Apr 1995.
 *
 *  If your AX25/NETROM system is relying on this code for
 *    securetty/firewalling then please be aware this has been coded
 *    with the intent on striving on through system/(mis)configuration
 *    errors in the hope that at worst it will run with a degraded
 *    service.  Rather than leave your system providing no service at
 *    all, if opinions require the old behavior back when let me know
 *    and I'll #ifdef it in.
 *
 *
 *  History:
 *
 *	1.0  Feb 1995	Basic AX25 listening daemon, Multi-port, Call
 *			  matching, etc...
 *
 *	1.1  Feb 1995	Moved entry scanning before fork().
 *			Added setgroups() to plug securetty hole.
 *			Minor fixes + Improved handling.
 *
 *	1.2  Apr 1995   NETROM support added from developing AX25
 *			  028b/029.
 *			Added 'defaults' port setting.
 *			Added FLAG_NODIGIS.
 *
 *	1.3  Jul 1995	Make it a little more intelligent about what to
 *			  do with errors.
 *			Added exec and argv[0] as two different fields,
 *			  much like inetd uses.
 *
 *	1.4  Aug 1995	Confirmed support for AX25 030 (1.3.20 + hacks),
 *			  it appears to work.
 *			It will now bootup even if initial config errors
 *			  occur when setting up and binding (e.g. port(s)
 *			  down), it will skip the port(s) with a problem
 *			  and listen out on those which are left standing.
 *
 *	1.5  Aug 1995	Updated old (buggy) libax25.a function copies in axl.
 *			  Causing all sorts of problems.
 *
 *	1.6  Aug 1995	Reset the 'defaults' entry's when we start parsing
 *			  a new interface.
 *
 *	1.7  TODO	Connection logging called for....
 *
 *
 *  TODO:  Logging (connection, errors, etc...)
 *         Callsign validation check.
 *         Do a /etc/nologin, or similar check.
 *         Flag to setup utmp and run through a ptty, for /bin/login.
 *         Hams might prefer this program to auto create accounts (like WAMPES did/does/will do)
 *         The timing of the 'accept()' might be changed, defered to the
 *           child, then that child fork() itself, to stop race conditions
 *           around 'accept()'.
 *         Maybe a slightly more user friendly flags field in the config, using concated
 *           letter instead of a number of OR'ed bits.
 *         Add a config file to allow/disallow connections/services at
 *           different times of the day to restrict access say.
 *
 *
**/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <grp.h>

#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <linux/ax25.h>
#include <linux/netrom.h>
#include <linux/fs.h>		/* NR_FILE **/

#ifdef	AX25UTILS_PKG
   #include "version.h"
#endif

#ifdef	LIBAX25
   #include "libax25.h"
#else
   #define	AX25_CALLSID	10
   #define	AX25_CALLNSID	7
#endif


#ifndef AX25D_CONF
   #define	AX25D_CONF	"/usr/local/ax25/etc/ax25d"
#endif

#define MAX_ARGS	10

#define	FLAG_UTMP	0x01
#define	FLAG_VALIDCALL	0x02
#define	FLAG_CHKNL	0x01
#define	FLAG_NODIGIS	0x40
#define	FLAG_LOCKOUT	0x80

struct axlist {		/* Have used same struct for quickness **/
   struct axlist *next;	/* Port list **/

   char *portcall;	/* Port call, only set across the port list. **/
   int fd;		/* The listening socket fd. **/

   int af_type;		/* AF_AX25 or AF_NETROM port. **/

   ax25_address bind;
   ax25_address peer;

   struct axlist *ents;	/* Exec line entries. **/
   char *call;		/* Call in listing entries. **/
   char *node;		/* Node call in listing entries. **/
   uid_t uid;		/* UID to run program as. **/
   gid_t gid;		/* GID to run program as. **/
   char *exec;		/* Real exec. **/
   char *shell;		/* Command line. With possible escapes. **/

   int window;		/* Set window to... **/
   unsigned long t1;	/* Set T1 to... (Retrans timer) **/
   unsigned long t2;	/* Set T2 to... (Ack delay) **/
   unsigned long t3;	/* Set T3 to... (Idle Poll timer) **/
   unsigned long n2;	/* Set N2 to... (Retries) **/
   int flags;		/* FLAG_ values ORed... **/
};

static struct axlist *AXL	= (struct axlist *)0;
static char *ConfigFile		= AX25D_CONF;
static char User[AX25_CALLSID];			/* Room for 'GB9ZZZ-15\0' **/
static char Node[AX25_CALLSID];			/* Room for 'GB9ZZZ-15\0' **/
static int Update		= 1;		/* Cause update on bootup **/
static char *Progname		= NULL;

static char *Version_String	= "1.6";
#ifdef	VERSION_KERNEL
static char *Version_Kernel	= VERSION_KERNEL;
#endif
static char *Error_fork		= "fork() failed - Sorry unable to honour your connection.\r";


#ifndef LIBAX25
   static int ax25_aton(ax25_address *, const char *);
   static char *ax25_ntoa(char *, const ax25_address *);
   static int ax25_atonpath(struct full_sockaddr_ax25 *, const char *);
   static char *ax25_stripssid(const char *);
#endif

static int ReadConfig(void);
static void SignalHUP(int);
static void SetupOptions(int, struct axlist *);
static void WorkoutArgs(int, char *, int *, char **);
static struct axlist *ClearList(struct axlist *);

int
main(argc, argv)
int argc;
char *argv[];
{
   struct axlist *axltmp, *paxl, *raxl;
   struct full_sockaddr_ax25 ax25;
   int addrlen;
   int cnt;
   
   if((Progname = strrchr(argv[0], '/')))
      Progname++;
   else
      Progname = argv[0];
   
   while((cnt = getopt(argc, argv, "c:vV")) != EOF) {
      switch(cnt) {
      case 'c':
         ConfigFile = optarg;
         break;
         
      case 'v':
      case 'V':
#ifdef	VERSION_KERNEL
         fprintf(stderr, "%s v%s (for %s), Copyright (C) 1995, Darryl L. Miles (G7LED)\n", Progname, Version_String, Version_Kernel);
#else
         fprintf(stderr, "%s v%s, Copyright (C) 1995, Darryl L. Miles (G7LED)\n", Progname, Version_String);
#endif
         fprintf(stderr, "\n%s comes with ABSOLUTELY NO WARRANTY;\n", Progname);
         fprintf(stderr, "This program is free software; you can redistribute it and/or modify\n");
         fprintf(stderr, "it under the terms of the GNU General Public License as published by\n");
         fprintf(stderr, "the Free Software Foundation; either version 2 of the License, or\n");
         fprintf(stderr, "(at your option) any later version.\n");
         exit(1);
         break;
      
      default:
         fprintf(stderr, "Usage: %s [-V] [-c altfile]\n", Progname);
         exit(1);
         break;
      }
   }

   /* Clean up a possibily mucky parent. **/
   for(cnt = 3; cnt < NR_FILE; cnt++)
      close(cnt);

#if 0
   ReadConfig();
#endif

   signal(SIGHUP, SignalHUP);
   signal(SIGCHLD, SIG_IGN);

   for( ; ; ) {
      int maxfd = -1;
      struct fd_set fdread;

      if(Update) {
         if(ReadConfig() < 0) {
            if(AXL == (struct axlist *)0) {
               fprintf(stderr, "%s: Config file reload error, no ports to listen on, exiting.\n", Progname);
               exit(1);
            } else {
               fprintf(stderr, "%s: Config file reload error, continuing with original port config.\n", Progname);
            }
         }
         Update = 0;
      }

      FD_ZERO(&fdread);

      for(paxl = AXL; paxl && paxl->fd >= 0; paxl = paxl->next) {
         FD_SET(paxl->fd, &fdread);
         if(paxl->fd > maxfd)
            maxfd = paxl->fd;
      }

      if(select(maxfd + 1, &fdread, NULL, NULL, NULL) > 0) {
         for(paxl = AXL; paxl; paxl = paxl->next) {
            if(FD_ISSET(paxl->fd, &fdread)) {
               pid_t pid;
               gid_t grps[2];
               char *argv[MAX_ARGS];
               int argc;
               int new;
               int i;
            
               /* Setting up a non-blocking accept() so is does not hang up
                  - I am not sure at this time why I didn't/don't assign
                  the socket non-blocking to start with.  **/
               /* We really need to setup the netrom window option here so
                  that it's negotiated correctly on accepting the connection. **/
               /* It would be very useful if recvmsg/sendmsg were supported
                  then we can move the call checking up here. **/
               i = 1;
               ioctl(paxl->fd, FIONBIO, &i);
               addrlen = sizeof(ax25);
               new = accept(paxl->fd, (struct sockaddr *)&ax25, &addrlen);
               i = 0;
               ioctl(paxl->fd, FIONBIO, &i);

               if(new < 0 && errno == EWOULDBLOCK)
                  continue;	/* It's gone ??? **/
               else if(new < 0) {
                  fprintf(stderr, "%s: accept(): %s\n", Progname, strerror(errno));
                  fprintf(stderr, "\tclosing socket on %s and continuing\n", paxl->portcall);
                  close(paxl->fd);
                  paxl->fd = -1;
                  continue;
               }

               ax25_ntoa(User, &ax25.fsa_ax25.sax25_call);
               if(paxl->af_type == AF_NETROM && ax25.fsa_ax25.sax25_ndigis == 1)
                  ax25_ntoa(Node, &ax25.fsa_digipeater[0]);
               else
                  Node[0] = '\0';	/* Hmm funny version of NETROM Kernel **/

               for(raxl = paxl->ents; raxl; raxl = raxl->ents) {
                  if(paxl->af_type == AF_NETROM && raxl->node && Node[0] != '\0') {
                     if(strchr(raxl->node, '-') == NULL) {
                        if(strcasecmp(raxl->node, ax25_stripssid(Node)))
                           continue;	/* Found no match (for any SSID) **/
                     } else {
                        if(strcasecmp(raxl->node, Node))
                           continue;	/* Found no match **/
                     }
                  }
                  if(!raxl->call)	/* default **/
                     break;

                  if(strchr(raxl->call, '-') == NULL) {
                     if(!strcasecmp(raxl->call, ax25_stripssid(User)))
                        break;	/* Found a match (for any SSID) **/
                  } else {
                     if(!strcasecmp(raxl->call, User))
                        break;	/* Found a match **/
                  }
               }

               if(!raxl || (raxl->flags & FLAG_LOCKOUT) ||
                (raxl->af_type == AF_AX25 && raxl->flags & FLAG_NODIGIS &&
                ax25.fsa_ax25.sax25_ndigis != 0)) {
                  /* No default, not allowed to connect, not allowed
                     to uplink via digi's **/
                  close(new);
/*                  exit(0); **/
                  continue;
               }
         
               pid = fork();
               switch(pid) {
               case -1:
                  /* Give them a clue, as to what is going wrong. **/
                  write(new, Error_fork, strlen(Error_fork)); 
                  /* I don't think AX25 at the moment will hold the
                     connection open, if the above does not make it
                     through first time. **/
                  close(new);
                  break;			/* Oh well... **/

               case 0:
#if 0	/* Moved before the fork() **/
                  ax25_ntoa(User, &ax25.fsa_ax25.sax25_call);

                  if(paxl->af_type == AF_NETROM && ax25.fsa_ndigis == 1)
                     ax25_ntoa(Node, &ax25.fsa_digipeater[0]);
                  else
                     Node[0] = '\0';	/* Hmm funny version of NETROM **/

                  for(raxl = paxl->ents; raxl; raxl = raxl->ents) {
                     if(paxl->af_type == AF_NETROM && raxl->node && Node[0] != '\0') {
                        if(strchr(raxl->node, '-') == NULL) {
                           if(strcasecmp(raxl->node, ax25_stripssid(Node)))
                              continue;	/* Found no match (for any SSID) **/
                        } else {
                           if(strcasecmp(raxl->node, Node))
                              continue;	/* Found no match **/
                        }
                     }
                     if(!raxl->call)	/* default **/
                        break;
                     if(strchr(raxl->call, '-') == NULL) {
                        if(!strcasecmp(raxl->call, ax25_stripssid(User)))
                           break;	/* Found a match (for any SSID) **/
                     } else {
                        if(!strcasecmp(raxl->call, User))
                           break;	/* Found a match **/
                     }
                  }
               
                  if(!raxl || (raxl->flags & FLAG_LOCKOUT) ||
                   (raxl->af_type == AF_AX25 && (raxl->flags & FLAG_NODIGIS) && ax25.sax25_ndigis != 0)) {
                     /* No default, not allowed to connect, not allowed
                        to uplink via digi's **/
                     close(new);
                     exit(0);
                  }
#endif
                  SetupOptions(new, raxl);
               
                  dup2(new, 0);
                  dup2(new, 1);
                  dup2(new, 2);
                  close(new);

#if 0		/* Linux NR_FILE is now 1024! :-) **/
                  for(cnt = 3; cnt < NR_FILE; cnt++)
                     close(cnt);
#endif
                  /* Might be more efficient if we just went down AXL,
                     we cleaned up our parents left overs on bootup **/
                  for(axltmp = AXL; axltmp; axltmp = axltmp->next)
                     close(axltmp->fd);
               
                  WorkoutArgs(raxl->af_type, raxl->shell, &argc, argv);
               
                  /* Make root secure, before we exec() **/
                  setgroups(0, grps);	/* Strip any supplementary gid's **/
#if 1	/* POSIX prefers this, does it not? **/
                  setgid(raxl->gid);
                  setuid(raxl->uid);
#else
                  setregid(raxl->gid, raxl->gid);
                  setreuid(raxl->uid, raxl->uid);
#endif                  

                  execve(raxl->exec, argv, NULL);
                  exit(1);		/* Oh dear **/
                  break;

               default:
                  close(new);
                  break;
               }
            }
         }
      }
   }

   /* NOT REACHED **/
   exit(1);		/* How did we get here? **/
}


static struct axlist *
ClearList(list)
struct axlist *list;
{
   struct axlist *tp1, *tp2, *tmp;
   
   for(tp1 = list; tp1; ) {
      for(tp2 = tp1->ents; tp2; ) {
         if(tp2->portcall)
            free(tp2->portcall);
         if(tp2->call)
            free(tp2->call);
         if(tp2->node)
            free(tp2->node);
         if(tp2->exec)
            free(tp2->exec);
         if(tp2->shell)
            free(tp2->shell);

         tmp = tp2->ents;
         free(tp2);
         tp2 = tmp;
      }

      if(tp1->portcall)
         free(tp1->portcall);
      if(tp1->call)
         free(tp1->call);
      if(tp1->node)
         free(tp1->node);
      if(tp1->exec)
         free(tp1->exec);
      if(tp1->shell)
         free(tp1->shell);
      close(tp1->fd);

      tmp = tp1->next;
      free(tp1);
      tp1 = tmp;
   }

   return (struct axlist *)0;
}

static void
SignalHUP(code)
int code;
{
   signal(SIGHUP, SignalHUP);
   Update = 1;		/* Schedule a update **/
}

static int
ReadConfig(void)
{
   struct axlist axl_defaults;
   struct axlist *axl_build = (struct axlist *)0;
   struct axlist *axl_port = (struct axlist *)0;	/* Keep GCC happy **/
   struct axlist *axl_ent, *axltmp;
   struct full_sockaddr_ax25 ax25;
   struct passwd *pwd;
   FILE *fp;
   char buffer[2048];
   char *sp, *cp, *call, *node;
   int addrlen;
   int af_type = 0;	/* Keep GCC happy **/
   int line;
   int hunt, error = 0;
   int defaults = 0;

   memset(&axl_defaults, 0, sizeof(axl_defaults));

   if((fp = fopen(ConfigFile, "r")) == NULL)
      return -1;

   line = 0;
   hunt = 1;
   while(fgets(buffer, sizeof(buffer), fp) != NULL) {
      line++;
      for(cp = buffer; *cp && isspace(*cp); cp++)
         ;
      if(*cp == '#')
         continue;
         
      switch(*cp) {
      case '[':		/* AX25 port call **/
         cp++;
         af_type = AF_AX25;
         hunt = 1;
         error = 0;
         break;
      case '<':		/* NETROM iface call **/
         cp++;
         af_type = AF_NETROM;
         hunt = 1;
         error = 0;
         break;
      default:
         if(hunt && error == 0)
            goto BadLine;
         break;
      }

      if(hunt) {	/* We've found a Iface entry **/
         /* Reset 'defaults' entry on the interface **/
         memset(&axl_defaults, 0, sizeof(axl_defaults));
         for( ; *cp && isspace(*cp); cp++)
            ;
         for(sp = cp; *cp /* && !isspace(*cp) */ && *cp != ']' && *cp != '>'; cp++)
            *cp = toupper(*cp);	/* Make it look pretty for strdup() below **/
         if(*cp)
            *cp = '\0';
         else
            goto BadLine;

         if(!(axl_port = calloc(1, sizeof(*axl_port)))) {
            fprintf(stderr, "%s: Malloc() failed\n", Progname);	/* Completely PARANOID :-) **/
            goto Error;
         }
         if(!(axl_port->portcall = strdup(sp))) {
            fprintf(stderr, "%s: Malloc() failed\n", Progname);	/* Completely PARANOID :-) **/
            goto Error;
         }
         axl_port->af_type = af_type;

      
         if((axl_port->fd = socket(axl_port->af_type, SOCK_SEQPACKET, 0)) < 0) {
            fprintf(stderr, "%s: socket(): %s\n", Progname, strerror(errno));

            free(axl_port->portcall);
            free(axl_port);
            error = 1;
            continue;
         }

         addrlen = ax25_atonpath(&ax25, axl_port->portcall);
         
         memcpy(&axl_port->bind, &ax25.fsa_ax25.sax25_call, sizeof(axl_port->bind));
         /* This might be used later to speed up callsign comparisons, if needs be **/
         ax25.fsa_ax25.sax25_family = axl_port->af_type;

         if(bind(axl_port->fd, (struct sockaddr *)&ax25, addrlen) < 0) {
            fprintf(stderr, "%s: bind(): %s '%s'\n", Progname, strerror(errno), axl_port->portcall);

            close(axl_port->fd);
            free(axl_port->portcall);
            free(axl_port);
            error = 1;
            continue;
         }
         if(listen(axl_port->fd, SOMAXCONN) < 0) {
            fprintf(stderr, "%s: listen(): %s\n", Progname, strerror(errno));

            close(axl_port->fd);
            free(axl_port->portcall);
            free(axl_port);
            error = 1;
            continue;
         }

         /* Fix port call **/
         for(cp = axl_port->portcall; *cp && !isspace(*cp); cp++)
            ;
         if(*cp) {
            *cp = '\0';
            if(!(cp = strdup(axl_port->portcall)))
               goto Error;
            free(axl_port->portcall);
            axl_port->portcall = cp;
         }

         /* Add it to the head of the list we are building **/
         if(axl_build == (struct axlist *)0) {
            axl_build = axl_port;
         } else {
            for(axltmp = axl_build; axltmp->next; axltmp = axltmp->next)
               ;
            axltmp->next = axl_port;
         }

         hunt = 0;	/* Next lines will be entries **/

      } else {		/* This is an entry **/
         if(!(axl_ent = calloc(1, sizeof(*axl_ent)))) {
            fprintf(stderr, "%s: Malloc() failed\n", Progname);	/* Completely PARANOID :-) **/
            goto Error;
         }

         axl_ent->af_type = axl_port->af_type;	/* Inherit this **/

         if(axl_ent->af_type == AF_AX25) {
            for(call = cp; *cp && !isspace(*cp); cp++)
               *cp = toupper(*cp);	/* Make it look pretty for strdup() below **/
            if(*cp)
               *cp++ = '\0';
            else {
               free(axl_ent);
               continue;
            }
         } else {
            for(call = cp; *cp && !isspace(*cp) && *cp != '@'; cp++)
               *cp = toupper(*cp);	/* Make it look pretty for strdup() below **/
            if(*cp == '@') {
               *cp++ = '\0';
               for(node = cp; *cp && !isspace(*cp); cp++)
                  *cp = toupper(*cp);	/* Make it look pretty for strdup() below **/
               if(*cp)
                  *cp++ = '\0';
               else {
                  free(axl_ent);
                  continue;
               }
               if(!(axl_ent->node = strdup(node))) {
                  fprintf(stderr, "%s: Malloc() failed\n", Progname);	/* Completely PARANOID :-) **/
                  free(axl_ent);
                  goto Error;
               }
               if(*call == '\0')
                  call = "default";	/* @NODE means default@NODE **/
            } else if(*cp) {
               *cp++ = '\0';
            } else {
               free(axl_ent);
               continue;
            }
         }

         defaults = 0;
         if(!strcasecmp("defaults", call)) {
            defaults = 1;
         } else if(strcasecmp("default", call)) {
            if(!(axl_ent->call = strdup(call))) {
               fprintf(stderr, "%s: Malloc() failed\n", Progname);	/* Completely PARANOID :-) **/
               free(axl_ent);
               goto Error;
            }
         }
         /* Save encoded peer for rainy day **/
         ax25_aton(&axl_ent->peer, call);
            
         /* Window **/
         for( ; *cp && isspace(*cp); cp++)
            ;
         for(sp = cp; *cp && !isspace(*cp); cp++)
            ;
         if(*cp)
            *cp++ = '\0';
         else
            goto BadArgsFree;
         if(defaults == 0) {
            if(*sp != '*')		/* atoi would return 0 anyway but... **/
               axl_ent->window = atoi(sp);
            else
               axl_ent->window = axl_defaults.window;
         } else {
            if(*sp != '*')		/* atoi would return 0 anyway but... **/
               axl_defaults.window = atoi(sp);
         }

         /* T1 **/
         for( ; *cp && isspace(*cp); cp++)
            ;
         for(sp = cp; *cp && !isspace(*cp); cp++)
            ;
         if(*cp)
            *cp++ = '\0';
         else
            goto BadArgsFree;
         if(defaults == 0) {
            if(*sp != '*')		/* atoi would return 0 anyway but... **/
               axl_ent->t1 = atoi(sp);
            else
               axl_ent->t1 = axl_defaults.t1;
         } else {
            if(*sp != '*')		/* atoi would return 0 anyway but... **/
               axl_defaults.t1 = atoi(sp);
         }

         /* T2 **/
         for( ; *cp && isspace(*cp); cp++)
            ;
         for(sp = cp; *cp && !isspace(*cp); cp++)
            ;
         if(*cp)
            *cp++ = '\0';
         else
            goto BadArgsFree;
         if(defaults == 0) {
            if(*sp != '*')		/* atoi would return 0 anyway but... **/
               axl_ent->t2 = atoi(sp);
            else
               axl_ent->t2 = axl_defaults.t2;
         } else {
            if(*sp != '*')		/* atoi would return 0 anyway but... **/
               axl_defaults.t2 = atoi(sp);
         }
            
         /* T3 **/
         for( ; *cp && isspace(*cp); cp++)
            ;
         for(sp = cp; *cp && !isspace(*cp); cp++)
            ;
         if(*cp)
            *cp++ = '\0';
         else
            goto BadArgsFree;
         if(defaults == 0) {
            if(*sp != '*')		/* atoi would return 0 anyway but... **/
               axl_ent->t3 = atoi(sp);
            else
               axl_ent->t3 = axl_defaults.t3;
         } else {
            if(*sp != '*')		/* atoi would return 0 anyway but... **/
               axl_defaults.t3 = atoi(sp);
         }
            
         /* N2 **/
         for( ; *cp && isspace(*cp); cp++)
            ;
         for(sp = cp; *cp && !isspace(*cp); cp++)
            ;
         if(*cp)
            *cp++ = '\0';
         else
            goto BadArgsFree;
         if(defaults == 0) {
            if(*sp != '*')		/* atoi would return 0 anyway but... **/
               axl_ent->n2 = atoi(sp);
            else
               axl_ent->n2 = axl_defaults.n2;
         } else {
            if(*sp != '*')		/* atoi would return 0 anyway but... **/
               axl_defaults.n2 = atoi(sp);
         }
   
         if(defaults == 0) {		/* If 'defaults' ignore the next bit **/
            /* Flags **/
            for( ; *cp && isspace(*cp); cp++)
               ;
            for(sp = cp; *cp && !isspace(*cp); cp++)
               ;
            if(*cp)
               *cp++ = '\0';
            else
               goto BadArgsFree;
            axl_ent->flags = atol(sp);
            
            if(!(axl_ent->flags & FLAG_LOCKOUT)) {
               /* Get username **/
               for( ; *cp && isspace(*cp); cp++)
                  ;
               for(sp = cp; *cp && !isspace(*cp); cp++)
                  ;
               if(*cp)
                  *cp++ = '\0';
               else
                  goto BadArgsFree;

               if((pwd = getpwnam(sp)) == (struct passwd *)0) {
                  fprintf(stderr, "%s: UID for user '%s' is unknown, ignoring entry\n", Progname, sp);
                  goto BadUID;
               }
               axl_ent->uid = pwd->pw_uid;
               axl_ent->gid = pwd->pw_gid;

               /* Get exec file **/
               for( ; *cp && isspace(*cp); cp++)
                  ;
               for(sp = cp; *cp && !isspace(*cp); cp++)
                  ;
               if(*cp)
                  *cp++ = '\0';
               else
                  goto BadArgsFree;
               if(!*sp)
                  goto BadArgsFree;
               if(!(axl_ent->exec = strdup(sp))) {
                  fprintf(stderr, "%s: Malloc() failed\n", Progname);	/* Completely PARANOID :-) **/
                  if(axl_ent->call)
                     free(axl_ent->call);
                  free(axl_ent);
                  goto Error;
               }

               /* Get command line **/
               for( ; *cp && isspace(*cp); cp++)
                  ;
               for(sp = cp; *cp; cp++) {
                  if(*cp == '\n') {
                     *cp = '\0';
                     break;
                  }
               }
               if(!*sp)
                  goto BadArgsFree2;
               if(!(axl_ent->shell = strdup(sp))) {
                  fprintf(stderr, "%s: Malloc() failed\n", Progname);	/* Completely PARANOID :-) **/
                  if(axl_ent->exec)
                     free(axl_ent->exec);
                  if(axl_ent->call)
                     free(axl_ent->call);
                  free(axl_ent);
                  goto Error;
               }
            }

            axl_ent->next = (struct axlist *)0;
            if(axl_port->ents == (struct axlist *)0) {
               axl_port->ents = axl_ent;
            } else {
               for(axltmp = axl_port->ents; axltmp->ents; axltmp = axltmp->ents)
                  ;
               axltmp->ents = axl_ent;
            }
         }
      }

      continue;

BadLine:
      fprintf(stderr, "%s: Bad config entry on line %d.\n", Progname, line);
      continue;

BadUID:
      if(axl_ent->call)
         free(axl_ent->call);
      free(axl_ent);
      continue;

BadArgsFree2:
      if(axl_ent->exec)
         free(axl_ent->exec);
BadArgsFree:
      if(axl_ent->call)
         free(axl_ent->call);
      free(axl_ent);
/* BadArgs: **/
      fprintf(stderr, "%s: Bad config entry on line %d, not enough fields.\n", Progname, line);
      continue;
   }

   fclose(fp);

   AXL = ClearList(AXL);
   AXL = axl_build;		/* Assign our built list to AXL **/
   return 0;


Error:
   axl_build = ClearList(axl_build);
   return -1;
}


static void
WorkoutArgs(af_type, shell, argc, argv)
int af_type;
char *shell;
int *argc;
char **argv;
{
   char buffer[1024];	/* Maximum arg size **/
   char *sp, *cp;
   int cnt;
   int args = 0;
   
   cnt = 0;
   for(cp = shell; *cp; cp++) {
      if(isspace(*cp) && cnt) {
         buffer[cnt] = '\0';
         if(!(argv[args++] = strdup(buffer))) {
            fprintf(stderr, "%s: Malloc() failed\n", Progname);	/* Completely PARANOID :-) **/
            exit(2);
         }
         cnt = 0;
         if(args == MAX_ARGS - 1) {
            argv[args] = NULL;
            *argc = args;
            return;
         }
         continue;
      } else if(isspace(*cp))	/* && !cnt **/
         continue;

      switch(*cp) {
      case '%':
         cp++;
         switch(*cp) {
         case 'U':	/* username in UPPER **/
            for(sp = User; *sp && *sp != '-'; sp++)
               buffer[cnt++] = toupper(*sp);
            break;

         case 'u':	/* USERNAME IN lower **/
            for(sp = User; *sp && *sp != '-'; sp++)
               buffer[cnt++] = tolower(*sp);
            break;

         case 'S':	/* username in UPPER (with SID) **/
            for(sp = User; *sp; sp++)
               buffer[cnt++] = toupper(*sp);
            break;

         case 's':	/* USERNAME IN lower (with SID) **/
            for(sp = User; *sp; sp++)
               buffer[cnt++] = tolower(*sp);
            break;

         case 'P':	/* nodename in UPPER **/
            if(af_type == AF_NETROM) {
               for(sp = Node; *sp && *sp != '-'; sp++)
                  buffer[cnt++] = toupper(*sp);
            } else
               buffer[cnt++] = '%';
            break;

         case 'p':	/* NODENAME IN lower **/
            if(af_type == AF_NETROM) {
               for(sp = Node; *sp && *sp != '-'; sp++)
                  buffer[cnt++] = tolower(*sp);
            } else
               buffer[cnt++] = '%';
            break;

         case 'R':	/* nodename in UPPER (with SID) **/
            if(af_type == AF_NETROM) {
               for(sp = Node; *sp; sp++)
                  buffer[cnt++] = toupper(*sp);
            } else
               buffer[cnt++] = '%';
            break;

         case 'r':	/* NODENAME IN lower (with SID) **/
            if(af_type == AF_NETROM) {
               for(sp = Node; *sp; sp++)
                  buffer[cnt++] = tolower(*sp);
            } else
               buffer[cnt++] = '%';
            break;

         case '\0':
         case '%':
         default:
            buffer[cnt++] = '%';
            break;
         }
         break;

      default:
         buffer[cnt++] = *cp;
         break;
      }
   }

   if(cnt) {
      buffer[cnt] = '\0';
      if(!(argv[args++] = strdup(buffer))) {
         fprintf(stderr, "%s: Malloc() failed\n", Progname);	/* Completely PARANOID :-) **/
         exit(2);
      }
   }

   argv[args] = NULL;
   *argc = args;
   return;
}

static void
SetupOptions(new, axl)
int new;
struct axlist *axl;
{
   if(axl->af_type == AF_AX25) {
      if(axl->window)
         setsockopt(new, SOL_AX25, AX25_WINDOW, &axl->window, sizeof(axl->window));
      if(axl->t1)
         setsockopt(new, SOL_AX25, AX25_T1, &axl->t1, sizeof(axl->t1));
      if(axl->n2)
         setsockopt(new, SOL_AX25, AX25_N2, &axl->n2, sizeof(axl->n2));
      if(axl->t2)
         setsockopt(new, SOL_AX25, AX25_T2, &axl->t2, sizeof(axl->t2));
      if(axl->t3)
         setsockopt(new, SOL_AX25, AX25_T3, &axl->t3, sizeof(axl->t3));
   } else {	/* NETROM **/
#ifdef NETROM_WINDOW
      if(axl->window)
         setsockopt(new, SOL_NETROM, NETROM_WINDOW, &axl->window, sizeof(axl->window));
#endif
      if(axl->t1)
         setsockopt(new, SOL_NETROM, NETROM_T1, &axl->t1, sizeof(axl->t1));
      if(axl->n2)
         setsockopt(new, SOL_NETROM, NETROM_N2, &axl->n2, sizeof(axl->n2));
      if(axl->t2)
         setsockopt(new, SOL_NETROM, NETROM_T2, &axl->t2, sizeof(axl->t2));
#if 0
      if(axl->t3)
         setsockopt(new, SOL_NETROM, NETROM_T3, &axl->t3, sizeof(axl->t3));
#endif
   }
}


/******************************** UTIL STUFF *******************************/

#ifndef	LIBAX25		/* These are nicked out of me libax25.a **/

/* Should arg1 (char *h) really be 'const char *h' ?? **/
static int
ax25_aton(axa, h)
ax25_address *axa;
const char *h;
{
   static const ax25_address ax25_empty = {
    { ' ' << 1, ' ' << 1, ' ' << 1, ' ' << 1, ' ' << 1, ' ' << 1, 0 }
   };
   short cnt;
   
   memcpy(axa, &ax25_empty, sizeof(ax25_address));

   for(cnt = 0; *h && isalnum(*h) && cnt < 6; cnt++, h++) {
      axa->ax25_call[cnt] = ((toupper(*h) << 1) & 0xff);
#if 0
      axa->ax25_call[cnt] = ((islower(*h) ? toupper(*h) << 1 : *h << 1) & 0xff);
#endif
   }
   
   if(*h && *h == '-') {
      cnt = atoi(++h);
      if(cnt < 0 || cnt > 15)
         cnt = 0;	/* should return 0/error or something **/
      axa->ax25_call[6] = (cnt << 1) & 0x1e; 
   }

   return sizeof(struct sockaddr_ax25);
}


/* Should arg1 (char *h) really be 'const char *h' ?? **/
static int
ax25_atonpath(ax, h)
struct full_sockaddr_ax25 *ax;
const char *h;
{
   int addrlen, ndigis;
   char *cp, *tp, *line;
   
   if(!(line = strdup(h)))
      return 0;

   for(cp = line; *cp && isspace(*cp); cp++)
      ;
   for(tp = cp; *cp && !isspace(*cp); cp++)
      ;
   if(*cp)
      *cp++ = '\0';
   if((addrlen = ax25_aton(&ax->fsa_ax25.sax25_call, tp)) == 0) {
      free(line);
      return 0;
   }
   ax->fsa_ax25.sax25_ndigis = 0;

   for(ndigis = 0; *cp && ndigis < AX25_MAX_DIGIS; ) {
      for( ; *cp && isspace(*cp); cp++)
         ;
      for(tp = cp; *cp && !isspace(*cp); cp++)
         ;
     if(*cp)
        *cp++ = '\0';
     if(*tp == '\0')
        break;
        
     if(ndigis == 0 && (!strcasecmp("V", tp) || !strcasecmp("VIA", tp)))
        continue;
     if(ax25_aton(&ax->fsa_digipeater[ndigis], tp) == 0) {
        free(line);
        return 0;
     }
     ax->fsa_ax25.sax25_ndigis++;
     addrlen = sizeof(struct full_sockaddr_ax25);
/*     addrlen += sizeof(ax25_address); - OLD Method **/
     ndigis++;
   }

   return addrlen;
}


/* Up to user to make sure 'h' is at least 'char h[AX25_CALLSID]' **/
static char *
ax25_ntoa(peer, axa)
char *peer;
const ax25_address *axa;
{
   const unsigned char *cp;
   unsigned char *h = peer;
   short cnt;

   for(cnt = 0, cp = (char *)axa; ; cnt++, cp++) {
      if(*cp == '\100')		/* Found a PAD char, skip the rest **/
         cnt = 5;
      else if(cnt < 6)		/* Callsign **/
         *h++ = *cp >> 1;
      else if(cnt == 6) {	/* SSID **/
         *h++ = '-';
         if((*cp & 0x1e) > 19) {
            *h++ = '1';
            *h++ = ((*cp & 0x1e) >> 1) + '&';
         } else {
            *h++ = ((*cp & 0x1e) >> 1) + '0';
         }
         *h = '\0';
         return peer;
      } 
   }
}

static char *
ax25_stripssid(peer)
const char *peer;
{
   static char tmp[AX25_CALLNSID];	/* "GB7ZZZ\0" **/
   int cnt;

   for(cnt = 0; *peer && cnt < (sizeof(tmp) - 1); peer++) {
      if(*peer == '-') {
         tmp[cnt] = '\0';
         return tmp;
      } 
      tmp[cnt++] = *peer;
   }
   if(*peer == '-' || !*peer) {		/* Already stripped **/
      tmp[cnt] = '\0';
      return tmp;
   }
   
   return NULL;		/* I dunno what we've been given... **/
}

#endif	/* LIBAX25 **/
