/* Copyright (C) 1999, 2000 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <qpushbutton.h>
#include <qkeycode.h>
#include <qapplication.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qmessagebox.h>
#include <string.h>
#include <ctype.h>
#include <iostream.h>
#include <strstream.h>

#include "gpl.h"
#include "dialogs.h"
#include "tnc.h"
#include "widgets.h"
#include "buffers.h"
#include "filesend.h"
#include "download.h"

#define COMMAND_SIZE 100

ConnectDialog::ConnectDialog(Tnc* tnc_p_, int standard_size, QWidget* qwidget_p):
                             QDialog(qwidget_p, "Connect", true), tnc_p(tnc_p_) {
    int max_size;
    char* temp_ptr;
    if (tnc_p->tnc_func.active_port && (tnc_p->tnc_func.hfmode == Tnc_func::amtor
					|| tnc_p->tnc_func.hfmode == Tnc_func::lamtor
					|| tnc_p->tnc_func.hfmode == Tnc_func::fec)) {
        temp_ptr = tnc_p->tnc_func.selCall;
	max_size = selCall_SIZE;
    }
    else {
        temp_ptr = tnc_p->tnc_func.hisCall[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port];
	if (!tnc_p->tnc_func.active_port || tnc_p->tnc_func.hfmode == Tnc_func::packet) max_size = CONNECT_SIZE;
	else max_size = hisCall_SIZE;
    }

    QLabel* label_p = new QLabel("Station to connect with?", this);
    if (!label_p) {
        cerr << "Memory allocation error in ConnectDialog::ConnectDialog()" << endl;
	exit(MEM_ERROR);
    }
    if (style() == WindowsStyle) {
        label_p->setGeometry(0, standard_size/2, standard_size * 9, standard_size/2);
    }
    else {
        label_p->setGeometry(0, standard_size/2, (standard_size * 15)/2, standard_size/2);
    }
    label_p->setAlignment(AlignCenter);

    editbox_p = new ConnectEditor(this, max_size, tnc_p);
    if (!editbox_p) {
        cerr << "Memory allocation error in ConnectDialog::ConnectDialog()" << endl;
	exit(MEM_ERROR);
    }
    int box_width = standard_size * 4;
    editbox_p->setGeometry((label_p->width() - box_width)/2, (standard_size * 3)/2, box_width, standard_size);
    setCaption("Connect");

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in ConnectDialog::ConnectDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in ConnectDialog::ConnectDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();

    ok_button_p->setGeometry(standard_size, standard_size * 3,
			     button_size.width(), button_size.height());
    cancel_button_p->setGeometry(label_p->width() - (standard_size + button_size.width()),
				 standard_size * 3, button_size.width(), button_size.height());

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    setFixedSize(label_p->width(), (standard_size * 7)/2 + button_size.height());
    editbox_p->count = strlen(temp_ptr);
    editbox_p->setText(temp_ptr);
    editbox_p->setFocus();

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));
    QObject::connect(editbox_p, SIGNAL(returnPressed()), this, SLOT(accept()));
    QObject::connect(editbox_p, SIGNAL(esc_pressed()), this, SLOT(reject()));
}

void ConnectEditor::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();
    int letter = event->ascii();
    int valid_flag = false;
    if (keycode ==  Key_Return) {
        if (count) emit returnPressed();
	else QApplication::beep();
    }
    else if (keycode == Key_Escape) emit esc_pressed();
    else if (keycode == Key_Backspace && cursorPosition() != 0) {
        count--;
	valid_flag = true;
    }
    else if (keycode == Key_Delete && cursorPosition() < count) {
        count--;
	valid_flag = true;
    }

    else if (keycode == Key_Right || keycode == Key_Left
	       || keycode == Key_Home || keycode == Key_End) {
        valid_flag = true;
    }

    else if (isalnum(letter)){
        if (count < max_size) {
	    count++;
	    letter = toupper(letter);
	    valid_flag = true;
	}
	else QApplication::beep();
    }
    else if (letter == '/'
	     || (letter == '!' && cursorPosition() == 0
		 && tnc_p->tnc_func.active_port
		 && (tnc_p->tnc_func.hfmode == Tnc_func:: pactor
		     || tnc_p->tnc_func.hfmode == Tnc_func:: gtor))
	     || ((letter == ',' || letter == ' ' || letter == '-')
		 && cursorPosition()
		 && (!tnc_p->tnc_func.active_port
		     || tnc_p->tnc_func.hfmode == Tnc_func::packet))) {
        if (count < max_size) {
	    count++;
	    valid_flag = true;
	}
	else QApplication::beep();
    }
    
    if (valid_flag) {
#if QT_VERSION>=200
        QKeyEvent modified_event(QEvent::KeyPress, keycode,
				    letter, event->state(),
				    event->text().upper(),
				    event->isAutoRepeat(), event->count());
#elif QT_VERSION>=140
        QKeyEvent modified_event(Event_KeyPress, keycode,
				    letter, event->state());
#else
#error Wrong Qt version - either 1.4* or 2.* required
#endif
	QLineEdit::keyPressEvent(&modified_event);
    }
}

CallsignDialog::CallsignDialog(Tnc* tnc_p, int standard_size, QWidget* qwidget_p):
                               QDialog(qwidget_p, "Connect", true) {
    QLabel* label_p = new QLabel("Callsign?", this);
    if (!label_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }
    if (style() == WindowsStyle) {
        label_p->setGeometry(0, standard_size/2, standard_size * 9, standard_size/2);
    }
    else {
        label_p->setGeometry(0, standard_size/2, (standard_size * 15)/2, standard_size/2);
    }
    label_p->setAlignment(AlignCenter);

    editbox_p = new CallsignEditor(this, tnc_p);
    if (!editbox_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }
    int box_width = standard_size * 4;
    editbox_p->setGeometry((label_p->width() - box_width)/2, (standard_size * 3)/2, box_width, standard_size);
    setCaption("Callsign");

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setGeometry(standard_size, standard_size * 3,
			     button_size.width(), button_size.height());
    cancel_button_p->setGeometry(label_p->width() - (standard_size + button_size.width()),
				 standard_size * 3, button_size.width(), button_size.height());

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    setFixedSize(label_p->width(), (standard_size * 7)/2 + button_size.height());
    editbox_p->setFocus();

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));
    QObject::connect(editbox_p, SIGNAL(returnPressed()), this, SLOT(accept()));
    QObject::connect(editbox_p, SIGNAL(esc_pressed()), this, SLOT(reject()));
}

void CallsignEditor::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();
    int letter = event->ascii();
    int valid_flag = false;
    if (keycode ==  Key_Return) {
        if (count) emit returnPressed();
	else QApplication::beep();
    }
    else if (keycode == Key_Escape) emit esc_pressed();
    else if (keycode == Key_Backspace && cursorPosition() != 0) {
        count--;
	valid_flag = true;
    }
    else if (keycode == Key_Delete && cursorPosition() < count) {
        count--;
	valid_flag = true;
    }

    else if (keycode == Key_Right || keycode == Key_Left
	       || keycode == Key_Home || keycode == Key_End) {
        valid_flag = true;
    }

    else if (isalnum(letter)){
        if (count < hisCall_SIZE) {
	    count++;
	    letter = toupper(letter);
	    valid_flag = true;
	}
	else QApplication::beep();
    }
    else if (letter == '/'
	     || (letter == '-'
		 && cursorPosition()
		 && (!tnc_p->tnc_func.active_port
		     || tnc_p->tnc_func.hfmode == Tnc_func::packet))) {
        if (count < hisCall_SIZE) {
	    count++;
	    valid_flag = true;
	}
	else QApplication::beep();
    }
       
    if (valid_flag) {
#if QT_VERSION>=200
        QKeyEvent modified_event(QEvent::KeyPress, keycode,
				    letter, event->state(),
				    event->text().upper(),
				    event->isAutoRepeat(), event->count());
#elif QT_VERSION>=140
        QKeyEvent modified_event(Event_KeyPress, keycode,
				    letter, event->state());
#else
#error Wrong Qt version - either 1.4* or 2.* required
#endif
	QLineEdit::keyPressEvent(&modified_event);
    }
}

CommandDialog::CommandDialog(int standard_size, QWidget* qwidget_p):
                               QDialog(qwidget_p, "Command", true) {
    QLabel* label_p = new QLabel("Kam command to send?\n"
				 "(Press F5 or script button to send\nconfiguration script)", this);
    if (!label_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }
    if (style() == WindowsStyle) {
        label_p->setGeometry(0, standard_size/3, standard_size * 12, (standard_size * 3)/2);
    }
    else {
        label_p->setGeometry(0, standard_size/3, standard_size * 10, (standard_size * 3)/2);
    }
    label_p->setAlignment(AlignCenter);

    editbox_p = new CommandEditor(this);
    if (!editbox_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }
    int box_width = standard_size * 5;
    editbox_p->setGeometry((label_p->width() - box_width)/2, standard_size * 2, box_width, standard_size);
    setCaption("Command");

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* script_button_p = new QPushButton("Script", this);
    if (!script_button_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setGeometry(standard_size, (standard_size * 7)/2,
			     button_size.width(), button_size.height());
    script_button_p->setGeometry((label_p->width() - button_size.width())/2,
			         (standard_size * 7)/2, button_size.width(), button_size.height());
    cancel_button_p->setGeometry(label_p->width() - (standard_size + button_size.width()),
				 (standard_size * 7)/2, button_size.width(), button_size.height());

    ok_button_p->setAutoDefault(true);
    script_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    setFixedSize(label_p->width(), standard_size * 4 + button_size.height());
    editbox_p->setFocus();

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(command_accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(command_reject()));
    QObject::connect(script_button_p, SIGNAL(clicked()), this, SLOT(command_script()));
    QObject::connect(editbox_p, SIGNAL(key_pressed(int)), this, SLOT(done(int)));
}

void CommandEditor::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();
    int letter = event->ascii();
    int valid_flag = false;
    if (keycode == Key_Escape) emit key_pressed(CommandDialog::rejected);
    else if (keycode == Key_F5) emit key_pressed(CommandDialog::script);
    else if (keycode == Key_Return && count) emit key_pressed(CommandDialog::accepted);
    else if (keycode == Key_Backspace && cursorPosition() != 0) {
        count--;
	valid_flag = true;
    }
    else if (keycode == Key_Delete && cursorPosition() < count) {
        count--;
	valid_flag = true;
    }

    else if (keycode == Key_Right || keycode == Key_Left
	       || keycode == Key_Home || keycode == Key_End) {
        valid_flag = true;
    }

    else if (letter >= 32 && letter <= 127){
        if (count < COMMAND_SIZE) {
	    count++;
	    letter = toupper(letter);
	    valid_flag = true;
	}
	else QApplication::beep();
    }
       
    if (valid_flag) {
#if QT_VERSION>=200
        QKeyEvent modified_event(QEvent::KeyPress, keycode,
				    letter, event->state(),
				    event->text().upper(),
				    event->isAutoRepeat(), event->count());
#elif QT_VERSION>=140
        QKeyEvent modified_event(Event_KeyPress, keycode,
				    letter, event->state());
#else
#error Wrong Qt version - either 1.4* or 2.* required
#endif
	QLineEdit::keyPressEvent(&modified_event);
    }
}

RstDialog::RstDialog(int standard_size, QWidget* qwidget_p):
                               QDialog(qwidget_p, "RST", true) {
    QLabel* label_p = new QLabel("New RST Report?", this);
    if (!label_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }
    if (style() == WindowsStyle) {
        label_p->setGeometry(0, standard_size/2, standard_size * 9, standard_size/2);
    }
    else {
        label_p->setGeometry(0, standard_size/2, (standard_size * 15)/2, standard_size/2);
    }
    label_p->setAlignment(AlignCenter);

    editbox_p = new RstEditor(this);
    if (!editbox_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }
    int box_width = (standard_size * 4)/3;
    editbox_p->setGeometry((label_p->width() - box_width)/2, (standard_size * 3)/2, box_width, standard_size);
    setCaption("RST Report");

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setGeometry(standard_size, standard_size * 3,
			     button_size.width(), button_size.height());
    cancel_button_p->setGeometry(label_p->width() - (standard_size + button_size.width()),
				 standard_size * 3, button_size.width(), button_size.height());

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));

    setFixedSize(label_p->width(), (standard_size * 7)/2 + button_size.height());
    editbox_p->setFocus();
    QObject::connect(editbox_p, SIGNAL(returnPressed()), this, SLOT(accept()));
    QObject::connect(editbox_p, SIGNAL(esc_pressed()), this, SLOT(reject()));
}

void RstEditor::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();
    int letter = event->ascii();
    int valid_flag = false;
    if (keycode ==  Key_Return) {
        if (count == 3) emit returnPressed();
	else QApplication::beep();
    }
    else if (keycode == Key_Escape) emit esc_pressed();
    else if (keycode == Key_Backspace && cursorPosition() != 0) {
        count--;
	valid_flag = true;
    }
    else if (keycode == Key_Delete && cursorPosition() < count) {
        count--;
	valid_flag = true;
    }

    else if (keycode == Key_Right || keycode == Key_Left
	     || keycode == Key_Home || keycode == Key_End) {
        valid_flag = true;
    }

    else if (isdigit(letter) && count < 3) {
        valid_flag = true;
	count++;
    }

    if (valid_flag) QLineEdit::keyPressEvent(event);
}

PromptDialog::PromptDialog(const char* text, const char* caption, int standard_size, QWidget* widget_p): QDialog(widget_p, caption, true) {

    QLabel* label_p = new QLabel(text, this);
    if (!label_p) {
        cerr << "Memory allocation error in PromptDialog::PromptDialog()" << endl;
	exit(MEM_ERROR);
    }

    if (style() == WindowsStyle) {
        label_p->setGeometry(0, standard_size/2, standard_size * 9, standard_size/2);
    }
    else {
        label_p->setGeometry(0, standard_size/2, (standard_size * 15)/2, standard_size/2);
    }
    
    label_p->setAlignment(AlignCenter);

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in PromptDialog::PromptDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in PromptDialog::PromptDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setGeometry(standard_size, (standard_size * 3)/2,
			     button_size.width(), button_size.height());
    cancel_button_p->setGeometry(label_p->width() - (standard_size + button_size.width()),
				 (standard_size * 3)/2, button_size.width(), button_size.height());
    

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    ok_button_p->setFocus();

    setFixedSize(label_p->width(), standard_size * 2 + button_size.height());
    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));

    setCaption(caption);
}


Autocq_modeDialog::Autocq_modeDialog(int standard_size, QWidget* qwidget_p): QDialog(qwidget_p, 0, true) {

    QButtonGroup* buttongroup_p = new QButtonGroup(this);
    buttongroup_p->setTitle("Auto-CQ Mode?");

    if (style() == WindowsStyle) {
        setFixedSize(standard_size * 9, standard_size * 5);
        buttongroup_p->setGeometry(standard_size/2, standard_size/2, standard_size * 8, standard_size * 3);
    }
    else {
        setFixedSize((standard_size * 15)/2, standard_size * 5);
        buttongroup_p->setGeometry(standard_size/2, standard_size/2, (standard_size * 13)/2, standard_size * 3);
    }

    QRadioButton* radio_button_p;
    radio_button_p = new QRadioButton("Amtor FEC", buttongroup_p);
    if (!radio_button_p) {
        cerr << "Memory allocation error in Autocq_modeDialog::Autocq_modeDialog()" << endl;
	exit(MEM_ERROR);
    }
    radio_button_p->setGeometry(standard_size/2, standard_size/2, standard_size * 3, standard_size);

    radio_button_p = new QRadioButton("Pactor FEC", buttongroup_p);
    if (!radio_button_p) {
        cerr << "Memory allocation error in Autocq_modeDialog::Autocq_modeDialog()" << endl;
	exit(MEM_ERROR);
    }
    radio_button_p->setGeometry(standard_size/2, (standard_size * 3)/2, standard_size * 3, standard_size);
    buttongroup_p->setButton(0);
    
    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setGeometry(standard_size, standard_size * 4,
			     button_size.width(), button_size.height());
    cancel_button_p->setGeometry(width() - (standard_size + button_size.width()),
				 standard_size * 4, button_size.width(), button_size.height());

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    ok_button_p->setFocus();

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));
    QObject::connect(buttongroup_p, SIGNAL(clicked(int)), this, SLOT(mode_button_pressed(int)));

    setCaption("Auto-CQ Mode");

}

void Autocq_modeDialog::mode_button_pressed(int mode) {
    if (mode) prog_func.tor_autocq_mode = Prog_func::pactor;
    else prog_func.tor_autocq_mode = Prog_func::amtor;
}

GplDialog::GplDialog(int size, QWidget* qwidget_p): QDialog(qwidget_p, "GPL", true), standard_size(size) {

    editbox_p = new GplEditor(this);
    if (!editbox_p) {
        cerr << "Memory allocation error in GplDialog::GplDialog()" << endl;
	exit(MEM_ERROR);
    }
    editbox_p->setFont(QFont("courier", 12, QFont::Normal));
    setCaption("kamplus-qt: Conditions, Notices and Disclaimers");

    label_p = new QLabel("Do you accept the Conditions, Notices and Disclaimers shown above?", this);
    if (!label_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }

    label_p->setAlignment(AlignCenter);

    ok_button_p = new QPushButton("Accept", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in GplDialog::GplDialog()" << endl;
	exit(MEM_ERROR);
    }

    cancel_button_p = new QPushButton("Reject", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in GplDialog::GplDialog()" << endl;
	exit(MEM_ERROR);
    }

    button_size = cancel_button_p->sizeHint();
    
    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));

    QFont font("courier", 12, QFont::Normal);
    editbox_p->setFont(font);
    editbox_p->setFocus();
    editbox_p->setReadOnly(true);
    QObject::connect(editbox_p, SIGNAL(esc_pressed()), this, SLOT(reject()));

    setMinimumSize(standard_size * 20, standard_size * 14);
    editbox_p->setText(copyright_msg);
}

void GplDialog::resizeEvent(QResizeEvent* event) {
    int widget_height = event->size().height();
    int widget_width = event->size().width();
    
    editbox_p->setGeometry(2, standard_size/2, widget_width - 4, widget_height - (standard_size * 7)/2);
    label_p->setGeometry(0, widget_height - (standard_size * 5)/2, widget_width, standard_size/2);
    ok_button_p->setGeometry(standard_size, widget_height - (standard_size * 3)/2,
			     button_size.width(), button_size.height());
    cancel_button_p->setGeometry(widget_width - (standard_size + button_size.width()),
				 widget_height - (standard_size * 3)/2, button_size.width(),
				 button_size.height());
}

void GplEditor::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();

    if (keycode == Key_Escape) emit esc_pressed();

    else if (keycode == Key_Right || keycode == Key_Left
	     || keycode == Key_Home || keycode == Key_End
	     || keycode == Key_Up || keycode == Key_Down
	     || keycode == Key_PageUp || keycode == Key_PageDown) {
        QMultiLineEdit::keyPressEvent(event);
    }
}


HelpDialog::HelpDialog(int size): QDialog(0, "Help", false), standard_size(size),
                                                      open_flag(false) {

    char* filename = new char[strlen(prog_func.filedir) + 13];
    strcpy(filename, prog_func.filedir);
    strcat(filename, "/helpfile-qt");
    ifstream helpfile(filename, ios::in | ios::nocreate);
    if (!helpfile) {
	helpfile.clear();
	ostrstream strm;
	strm << filename << " cannot be found/opened" << ends;
	char* message = strm.str();
	QMessageBox::warning(this, "Filesend", message,
				 QMessageBox::Ok | QMessageBox::Default, 0);
	delete[] message;
    }
    else {
        helpfile.seekg(0, ios::end);
	streampos size = helpfile.tellg();
	if (size) {
	    helpfile.seekg(0); // go to beginning of file
	    char letter;
	    char* help_msg = new char[(int)size];
	    if (!help_msg) {
	        cerr << "Memory allocation error in HelpDialog::HelpDialog()" << endl;
		exit(MEM_ERROR);
	    }

	    char* char_p;
	    for (char_p = help_msg; (helpfile.get(letter)); char_p++) {
	        *char_p = letter;
	    }
	    char_p--;
	    *char_p = 0;    // null terminate help_msg
	    open_flag = true;
	    helpfile.close();

	    editbox_p = new HelpEditor(this);
	    if (!editbox_p) {
	        cerr << "Memory allocation error in HelpDialog::HelpDialog()" << endl;
		exit(MEM_ERROR);
	    }
	    editbox_p->setFont(QFont("courier", 12, QFont::Normal));
	    setCaption("kam-qt: Helpfile");

	    cancel_button_p = new QPushButton("Cancel", this);
	    if (!cancel_button_p) {
	        cerr << "Memory allocation error in HelpDialog::HelpDialog()" << endl;
		exit(MEM_ERROR);
	    }

	    button_size = cancel_button_p->sizeHint();
    
	    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(kill_dialog()));

	    QFont font("courier", 12, QFont::Normal);
	    editbox_p->setFont(font);
	    editbox_p->setFocus();
	    editbox_p->setReadOnly(true);
	    QObject::connect(editbox_p, SIGNAL(esc_pressed()), this, SLOT(kill_dialog()));
	    
	    setMinimumSize(standard_size * 20, standard_size * 14);
	    editbox_p->setText(help_msg);
	    delete[] help_msg;
	}
    }
    delete[] filename;
}

void HelpDialog::resizeEvent(QResizeEvent* event) {
    int widget_height = event->size().height();
    int widget_width = event->size().width();
    
    editbox_p->setGeometry(2, standard_size/2, widget_width - 4, widget_height - (standard_size * 5)/2);
    cancel_button_p->setGeometry((widget_width - button_size.width())/2,
				 widget_height - (standard_size * 3)/2, button_size.width(),
				 button_size.height());
}

void HelpEditor::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();

    if (keycode == Key_Escape) emit esc_pressed();

    else if (keycode == Key_Right || keycode == Key_Left
	     || keycode == Key_Home || keycode == Key_End
	     || keycode == Key_Up || keycode == Key_Down
	     || keycode == Key_PageUp || keycode == Key_PageDown) {
        QMultiLineEdit::keyPressEvent(event);
    }
}

CwSpeedDialog::CwSpeedDialog(int standard_size, int cw_speed, QWidget* qwidget_p):
                               QDialog(qwidget_p, "CwSpeed", true) {
    QLabel* label_p = new QLabel("New Speed (5 - 50 WPM)?", this);
    if (!label_p) {
        cerr << "Memory allocation error in CwSpeedDialog::CwSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setGeometry(0, standard_size/2, standard_size * 8, standard_size);
    label_p->setAlignment(AlignCenter);

    qlcdnumber_p = new QLCDNumber(2, this);
    if (!qlcdnumber_p) {
        cerr << "Memory allocation error in CwSpeedDialog::CwSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    qlcdnumber_p->setGeometry(standard_size * 8, standard_size/2, standard_size, standard_size);
    qlcdnumber_p->display(cw_speed);

    int widget_width = label_p->width() + qlcdnumber_p->width() + standard_size/2;
    qslider_p = new QSlider(5, 50, 5, cw_speed, QSlider::Horizontal, this, "CW Speed Slider");
    if (!qslider_p) {
        cerr << "Memory allocation error in CwSpeedDialog::CwSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    qslider_p->setTickmarks(QSlider::Below);
    qslider_p->setTickInterval(5);
    qslider_p->setTracking(true);
    int slider_width = widget_width - standard_size * 2;
    qslider_p->setGeometry((widget_width - slider_width)/2, standard_size * 2, slider_width,
			   qslider_p->sizeHint().height());
    setCaption("CW Speed");

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setGeometry(standard_size, (standard_size * 7)/2,
			     button_size.width(), button_size.height());
    cancel_button_p->setGeometry(widget_width - (standard_size + button_size.width()),
				 (standard_size * 7)/2, button_size.width(), button_size.height());

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    setFixedSize(widget_width, standard_size * 4 + button_size.height());
    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));
    QObject::connect(qslider_p, SIGNAL(valueChanged(int)), this, SLOT(display(int)));
    QObject::connect(qslider_p, SIGNAL(sliderReleased(void)), this, SLOT(set_sliderpos()));
}

DownloadDialog::DownloadDialog(const char* filename, Tnc* tnc_p_, MainScreen* ms_p, ReceiveWin* rw_p,
			        int standard_size):
                                  QDialog(0, "Download", false), tnc_p(tnc_p_),
                                  mainscreen_p(ms_p), receivewin_p(rw_p) {
    stream = tnc_p->tnc_func.active_stream();
    port = tnc_p->tnc_func.active_port;
    ostrstream strm;
    strm << "\nDownloading ";
    if (!port) {
        strm << "Vhf port stream " << (char)((stream | 0x40) + 1) << '\n';
    } 
    else {
        strm << "Hf port ";
	if (tnc_p->tnc_func.hfmode != Tnc_func::packet) strm << "Tor stream\n";
	else strm << "stream " <<  (char)((stream | 0x40) + 1) << '\n';
    }
    strm << "to file: " << filename << '\n' << ends;
    char* text = strm.str();

    QLabel* label_p = new QLabel(text, this);
    if (!label_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    delete[] text;

    const int min_width = standard_size * 8;
    const int hint_width = (label_p->sizeHint()).width() + standard_size;
    const int widget_width = min_width > hint_width ? min_width : hint_width;
    label_p->setGeometry(0, 0, widget_width, standard_size * 2);
    label_p->setAlignment(AlignCenter);

    
    QPushButton* cancel_button_p = new QPushButton("End", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    QSize button_size = cancel_button_p->sizeHint();
    cancel_button_p->setGeometry((widget_width - button_size.width())/2, standard_size * 2,
				    button_size.width(), button_size.height());
    
    label_p = new QLabel(this);
    if (!label_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setGeometry(0, (standard_size * 7)/2, widget_width, standard_size);
    label_p->setAlignment(AlignLeft | AlignVCenter);
    label_p->setFrameStyle(QFrame::Panel | QFrame::Sunken);

    const int line_width = 1;
    label_p->setLineWidth(line_width);

    QLabel* sub_label_p = new QLabel(" Bytes downloaded: ", label_p);
    if (!sub_label_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    const int intro_text_width = sub_label_p->sizeHint().width();
    const int sub_label_height = standard_size - line_width * 2;
    sub_label_p->setGeometry(line_width, line_width, intro_text_width, sub_label_height);

    bytes_label_p = new QLabel(label_p);
    if (!bytes_label_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    bytes_label_p->setGeometry(intro_text_width + line_width, line_width,
			       widget_width - intro_text_width - line_width * 2, sub_label_height);
    bytes_label_p->setAlignment(AlignLeft | AlignVCenter);
    show_bytes(0);

    setFixedSize(widget_width, (standard_size * 9)/2);
    setCaption("kam-qt: Download");
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(end_download()));
}	

void DownloadDialog::end_download(void) {
    DownloadList* download_list_p = tnc_p->tnc_func.download_list_ptr;
    // we need to find the download file object receiving the
    // file on the dialog's stream and port
    download_list_p->reset(DList_enum::bottom_end);
    DownloadFile* dload_file_p;
    while ((dload_file_p = (DownloadFile*)download_list_p->inspect(DList_enum::up)) != 0
	     && !(dload_file_p->get_stream() == stream
		    && dload_file_p->get_port() == port)) {}
    if (dload_file_p) {
        download_list_p->extract();
	mainscreen_p->display_connected_status();
	dload_file_p->dialog_take_ownership(); // we are now responsible for deleting this object
	delete dload_file_p;  // the DownloadFile object used to have ownership of the relevant dialog
	                      // object of this class so it is now safe to do 'delete this'
	delete this;          // this is safe - see above
    }
    else {
        receivewin_p->write("\nOops - can't find the download file object to delete\n"
			    "Please report bug\n");
	QApplication::beep();
    }
}

void DownloadDialog::show_bytes(long bytes) {
    ostrstream strm;
    strm << bytes << ends;
    char* text = strm.str();
    bytes_label_p->setText(text);
    delete[] text;
}

UploadDialog::UploadDialog(const char* filename, long filesize, Tnc* tnc_p_, BufferList& bl,
			   MainScreen* ms_p, ReceiveWin* rw_p, int standard_size):
                                  QDialog(0, "Upload", false), tnc_p(tnc_p_),
                                  buffer_list(bl), mainscreen_p(ms_p), receivewin_p(rw_p) {
    stream = tnc_p->tnc_func.active_stream();
    port = tnc_p->tnc_func.active_port;
    ostrstream strm;
    strm << "Uploading file: " << filename << "\nto ";
    if (!port) {
        strm << "Vhf port stream " << (char)((stream | 0x40) + 1) << '\n';
    } 
    else {
        strm << "Hf port ";
	if (tnc_p->tnc_func.hfmode != Tnc_func::packet) strm << "Tor stream\n";
	else strm << "stream " <<  (char)((stream | 0x40) + 1) << '\n';
    }
    strm << "(File size: ";
    if (filesize > 99999) strm << filesize/1024 << " Kb)" << ends;
    else strm << filesize << " bytes)" << ends;
    char* text = strm.str();

    QLabel* label_p = new QLabel(text, this);
    if (!label_p) {
        cerr << "Memory allocation error in UploadDialog::UploadDialog()" << endl;
	exit(MEM_ERROR);
    }
    delete[] text;

    const int min_width = standard_size * 8;
    const int hint_width = (label_p->sizeHint()).width() + standard_size;
    const int widget_width = min_width > hint_width ? min_width : hint_width;
    label_p->setGeometry(0, 0, widget_width, standard_size * 2);
    label_p->setAlignment(AlignCenter);


    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in UploadDialog::UploadDialog()" << endl;
	exit(MEM_ERROR);
    }
    QSize button_size = cancel_button_p->sizeHint();
    cancel_button_p->setGeometry((widget_width - button_size.width())/2, standard_size * 2,
				    button_size.width(), button_size.height());
    
    label_p = new QLabel(this);
    if (!label_p) {
        cerr << "Memory allocation error in UploadDialog::UploadDialog()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setGeometry(0, (standard_size * 7)/2, widget_width, standard_size);
    label_p->setAlignment(AlignLeft | AlignVCenter);
    label_p->setFrameStyle(QFrame::Panel | QFrame::Sunken);

    const int line_width = 1;
    label_p->setLineWidth(line_width);

    QLabel* sub_label_p = new QLabel(" Bytes sent to Kam: ", label_p);
    if (!sub_label_p) {
        cerr << "Memory allocation error in UploadDialog::UploadDialog()" << endl;
	exit(MEM_ERROR);
    }

    const int intro_text_width = sub_label_p->sizeHint().width();
    const int sub_label_height = standard_size - line_width * 2;
    sub_label_p->setGeometry(line_width, line_width, intro_text_width, sub_label_height);

    bytes_label_p = new QLabel(label_p);
    if (!bytes_label_p) {
        cerr << "Memory allocation error in UploadDialog::UploadDialog()" << endl;
	exit(MEM_ERROR);
    }
    bytes_label_p->setGeometry(intro_text_width + line_width, line_width,
			       widget_width - intro_text_width - line_width * 2, sub_label_height);
    bytes_label_p->setAlignment(AlignLeft | AlignVCenter);
    show_bytes(0);

    setFixedSize(widget_width, (standard_size * 9)/2);
    setCaption("kam-qt: Upload");
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(end_upload()));
}	

void UploadDialog::end_upload(void) {
    if (buffer_list.get_upload_status(stream, port) == BufferList::file) { // check for sanity
        // we now need to find the file buffer sending the
        // message on the dialog's stream and port
        buffer_list.reset(DList_enum::bottom_end);
	FileBuffer* file_buffer_p;
	while ((file_buffer_p = (FileBuffer*)buffer_list.inspect(DList_enum::up)) != 0
	         && !(file_buffer_p->get_stream() == stream
		        && file_buffer_p->get_port() == port)) {}
	if (file_buffer_p) file_buffer_p->end_loading();
	else {
	    receivewin_p->write("\nOops - can't find the file buffer to remove\n"
	                        "Please report bug\n");
	    QApplication::beep();
	}
    }
    // This method simply marks the FileBuffer object as available for removal
    // from the BufferList object, and does not remove the object itself
    // UploadDialog objects are modeless and are deleted by the FilesendBuffer object
    // when that object is deleted in MainScreen::timerEvent 
}

void UploadDialog::show_bytes(long bytes) {
    ostrstream strm;
    strm << bytes << ends;
    char* text = strm.str();
    bytes_label_p->setText(text);
    delete[] text;
}

SettingsDialog::SettingsDialog(int standard_size, Settings& settings, QWidget* qwidget_p):
                                 QDialog(qwidget_p, 0, true) {

    rx_endstop_checkbox_p = new QCheckBox("Provide end stop on receive", this);
    if (!rx_endstop_checkbox_p) {
        cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
	exit(MEM_ERROR);
    }
    rx_endstop_checkbox_p->setGeometry(standard_size, standard_size/2, standard_size * 7, standard_size);
    rx_endstop_checkbox_p->setChecked(settings.rx_endstop_flag);

    cw_rx_checkbox_p = new QCheckBox("Display received CW", this);
    if (!cw_rx_checkbox_p) {
        cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
	exit(MEM_ERROR);
    }
    cw_rx_checkbox_p->setGeometry(standard_size, (standard_size * 3)/2, standard_size * 7, standard_size);
    cw_rx_checkbox_p->setChecked(settings.cw_rx_flag);

    bell_checkbox_p = new QCheckBox("Sound received bell character", this);
    if (!bell_checkbox_p) {
        cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
	exit(MEM_ERROR);
    }
    bell_checkbox_p->setGeometry(standard_size, (standard_size * 5)/2, standard_size * 7, standard_size);
    bell_checkbox_p->setChecked(settings.bell_flag);
    
    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
	exit(MEM_ERROR);
    }

    setFixedSize(standard_size * 9, (standard_size * 11)/2);

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setGeometry(standard_size, standard_size * 4,
			     button_size.width(), button_size.height());
    cancel_button_p->setGeometry(width() - (standard_size + button_size.width()),
				 standard_size * 4, button_size.width(), button_size.height());

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    ok_button_p->setFocus();
    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));

    setCaption("Settings");
}

Settings SettingsDialog:: get_settings(void) const {
    Settings settings;
    settings.rx_endstop_flag = rx_endstop_checkbox_p->isChecked();
    settings.cw_rx_flag = cw_rx_checkbox_p->isChecked();
    settings.bell_flag = bell_checkbox_p->isChecked();
    return settings;
}

PrintMarkDialog::PrintMarkDialog(int standard_size, QWidget* qwidget_p):
                                 QDialog(qwidget_p, "PrintMark", true) {
    QLabel* label_p = new QLabel("Print from the mark?\n"
				 "(Press Cancel button to cancel the print job)", this);
    if (!label_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }
    if (style() == WindowsStyle) {
        label_p->setGeometry(0, standard_size/3, standard_size * 14, (standard_size * 3)/2);
    }
    else {
        label_p->setGeometry(0, standard_size/3, standard_size * 13, (standard_size * 3)/2);
    }
    label_p->setAlignment(AlignCenter);

    QPushButton* ok_button_p = new QPushButton("Ok", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* no_button_p = new QPushButton("No", this);
    if (!no_button_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel job", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setGeometry(standard_size, (standard_size * 5)/2,
			     button_size.width(), button_size.height());
    no_button_p->setGeometry((label_p->width() - button_size.width())/2,
			     (standard_size * 5)/2, button_size.width(), button_size.height());
    cancel_button_p->setGeometry(label_p->width() - (standard_size + button_size.width()),
				 (standard_size * 5)/2, button_size.width(), button_size.height());

    ok_button_p->setAutoDefault(true);
    no_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    ok_button_p->setFocus();

    setFixedSize(label_p->width(), standard_size * 3 + button_size.height());

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(command_accept()));
    QObject::connect(no_button_p, SIGNAL(clicked()), this, SLOT(command_reject()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(command_cancel_job()));
}

void PrintMarkDialog:: keyPressEvent(QKeyEvent* event) {
    if (event->key() == Key_Escape) done(rejected);
    else QDialog::keyPressEvent(event);
}
