/* Copyright (C) 1999, 2000 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

/*

The classes in this file are responsible for

- controlling the keyboard and directing all keyboard input
- via its helper classes declared in keyboard_funcs.h, carrying out
  actions directly controlled from the keyboard
- loading tr_buffer from the keyboard
- if sending CW, checking that the letter entered is a valid morse
  character
- if sending FEC, Amtor or RTTY, checking that the letter entered
  is a valid Baudot character

*/

#ifndef KEYBOARD_H
#define KEYBOARD_H

#define CONNECT_SIZE 40
#if CONNECT_SIZE < hisCall_SIZE || CONNECT_SIZE < selCall_SIZE
#error 
#endif

#include <iostream.h>
#include <fstream.h>
#include <ncurses.h>
// we don't want to use the 'clear' macro in ncurses.h - it will interfere with iostream::clear()
#undef clear
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <stdio.h>
#include "tnc.h"
#include "screen.h"
#include "buffers.h"
#include "filesend.h"
#include "prog_defs.h"
#include "pipes.h"
#include "keyboard_funcs.h"

typedef unsigned char uchar;

extern Prog_func prog_func;

void curses_init(void);

class Keyboard_input {
protected:
    int input_confirm(void);
public:
    virtual void action(void) = 0;
    virtual int action_pending(fd_set*) const; // as a default this checks FD_ISSET on stdin
    virtual ~Keyboard_input(void) {} // in case a derived class has a destructor
};

class Keyboard_input_with_data: public Keyboard_input {
protected:
    static MainScreen* mainscreen_ptr;
    static SendWin* sendwin_ptr;
    static ReceiveWin* receivewin_ptr;
    static DList<Screen>* screenform_ptr;
    static Transmit_buffer* tr_buffer_ptr;
    static Tnc* tnc_ptr;
    static BufferList* buffer_list_ptr;
    static Keyboard_input* keyboard_ptr;
    static Keyboard_input* default_input_ptr;
    static int F_KEYmatrix[3][7];
    static Keyboard_funcs keyboard_funcs;
    static void set_data(MainScreen*, SendWin*, ReceiveWin*, DList<Screen>*,
			   Transmit_buffer*, Tnc*, BufferList*, Keyboard_input*);
    void run_funcs(void) {
	if (keyboard_funcs.connect_script_flag == Keyboard_funcs::running) keyboard_funcs.run_connect_script();
	if (keyboard_funcs.send_parms_flag == Keyboard_funcs::running) keyboard_funcs.send_parms();
    }
public:
    const Keyboard_input* get_keyboard_ptr(void) const {return keyboard_ptr;}
    virtual void action(void) = 0;
    virtual int action_pending(fd_set*) const;
    friend void curses_init(void); // this is a friend, because it needs access to F_KEYmatrix[][]
};

class Keyboard_controller;          // forward declaration
class Keyboard_setup_mode;          // forward declaration

class Exit_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    friend Keyboard_setup_mode;
    PopupWin* popupwin_ptr;
    Exit_query(void) {}            // private class - only Keyboard_controller or Keyboard_setup_mode
    void setup(void);              // can create an object
public:   
    void action(void);
};

class Help_file: public Keyboard_input_with_data {
    friend Keyboard_controller;
    Help_file(void) {}            // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Openfile: public Keyboard_input_with_data {
    friend Keyboard_controller;
    enum Mode {text, binary, s_plus} mode;
    int stop_sending_query_flag;
    PopupWin* popupwin_ptr;
    Openfile(void) {}            // private class - only Keyboard_controller can create an object
    char filename[256];
    int index;
    void setup(Mode);
public:
    void action(void);
};

class Input_command: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Input_command(void) {}      // private class - only Keyboard_controller can create an object
    char buffer[256];
    int index;
    int get_filename_flag;
    void setup(void);
public:
    void action(void);
};

class Callsign: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Callsign(void) {}            // private class - only Keyboard_controller can create an object
    char buffer[hisCall_SIZE + 1];
    int index;
    void setup(void);
public:
    void action(void);
};

class Send_parms_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Send_parms_query(void) {}   // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Connect: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Connect(void) {}            // private class - only Keyboard_controller can create an object
    char buffer[CONNECT_SIZE + 1];
    int index;
    int max_size;
    void setup(void);
public:
    void action(void);
};

class Packet_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Packet_query(void) {}            // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Pactor_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Pactor_query(void) {}            // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Amtor_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Amtor_query(void) {}            // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Ascii_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Ascii_query(void) {}            // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Fec_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Fec_query(void) {}              // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Gmon_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Gmon_query(void) {}             // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Gtor_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Gtor_query(void) {}             // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Lamtor_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Lamtor_query(void) {}           // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Rtty_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Rtty_query(void) {}             // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Tor_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Tor_query(void) {}              // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Cw_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Cw_query(void) {}               // private class - only Keyboard_controller can create an object
    char buffer[3];
    int index;
    void setup(void);
public:
    void action(void);
};

class Disconnect_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Disconnect_query(void) {}       // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Abort_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Abort_query(void) {}           // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Send_message_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    char message_num;
    Send_message_query(void) {}    // private class - only Keyboard_controller can create an object
    void setup(char);
public:
    void action(void);
};

class Autocq_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Autocq_query(void) {}          // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Enter_rst: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Enter_rst(void) {}            // private class - only Keyboard_controller can create an object
    int index;
    void setup(void);
public:
    void action(void);
};

class Download_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    enum Mode {binary, s_plus} mode;
    PopupWin* popupwin_ptr;
    Download_query(void) {}       // private class - only Keyboard_controller can create an object
    int index;
    char filename[256];
    void setup(Mode);
public:
    void action(void);
};

class Printmark_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    enum Mode {set_mark, print} mode;
    PopupWin* popupwin_ptr;
    Printmark_query(void) {}      // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

class Printbuffer_query: public Keyboard_input_with_data {
    friend Keyboard_controller;
    PopupWin* popupwin_ptr;
    Printbuffer_query(void) {}    // private class - only Keyboard_controller can create an object
    void setup(void);
public:
    void action(void);
};

// This class provides keyboard functions when the Kam is operating in its normal (host) mode

class Keyboard_controller: public Keyboard_input_with_data {
private:
    Exit_query exit_query_obj;
    Help_file help_obj;
    Openfile openfile_obj;
    Input_command input_command_obj;
    Callsign callsign_obj;
    Send_parms_query send_parms_query_obj;
    Connect connect_obj;
    Amtor_query amtor_query_obj;
    Ascii_query ascii_query_obj;
    Fec_query fec_query_obj;
    Gmon_query gmon_query_obj;
    Gtor_query gtor_query_obj;
    Lamtor_query lamtor_query_obj;
    Pactor_query pactor_query_obj;
    Packet_query packet_query_obj;
    Rtty_query rtty_query_obj;
    Tor_query tor_query_obj;
    Cw_query cw_query_obj;
    Disconnect_query disconnect_query_obj;
    Abort_query abort_query_obj;
    Send_message_query send_message_query_obj;
    Autocq_query autocq_query_obj;
    Enter_rst enter_rst_obj;
    Download_query download_query_obj;
    Printmark_query printmark_query_obj;
    Printbuffer_query printbuffer_query_obj;
    PopupWin* popupwin_ptr;
public:
    void action(void);
    Keyboard_controller(MainScreen&, SendWin&, ReceiveWin&, DList<Screen>&,
			  Transmit_buffer&, Tnc*, BufferList&);
};

// This class provides keyboard functions when operating in setup (terminal) mode

class Keyboard_setup_mode: public Keyboard_input_with_data {
private:
    Exit_query exit_query_obj;
public:
    void action(void);
    Keyboard_setup_mode(MainScreen&, SendWin&, ReceiveWin&, DList<Screen>&,
			  Transmit_buffer&, Tnc*);
};

class Gpl_query: public Keyboard_input {
    CopyrightScreen& copyright_screen;
    ofstream fileout;
public:
    void action(void);
    Gpl_query(CopyrightScreen& a): copyright_screen(a) {}
};

#endif
