/*
 * %W% %E% %U%  [EXTREL_1.2]
 *
 * VersaTrack orbit calculations are based on those that appear in Dr. Manfred
 * Bester's sattrack program (the Unix(tm) versions 1 and 2).
 *
 * The data from which the maps where generated come from "xsat", an
 * X-Windows program by David A. Curry (N9MSW).
 *
 * Site coordinates come from various sources, including a couple of
 * World Almanacs, and also from both of the programs mentioned above.
 *
 * The following are authors' applicable copyright notices:
 *
 *                                                                               
 * Copyright (c) 1992, 1993, 1994 Manfred Bester. All Rights Reserved.        
 *                                                                           
 * Permission to use, copy, modify, and distribute this software and its      
 * documentation for educational, research and non-profit purposes, without   
 * fee, and without a written agreement is hereby granted, provided that the  
 * above copyright notice and the following three paragraphs appear in all    
 * copies.                                                                    
 *                                                                              
 * Permission to incorporate this software into commercial products may be    
 * obtained from the author, Dr. Manfred Bester, 1636 M. L. King Jr. Way,     
 * Berkeley, CA 94709, USA.                                                   
 *                                                                             
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,  
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF    
 * THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR HAS BEEN ADVISED   
 * OF THE POSSIBILITY OF SUCH DAMAGE.                                         
 *                                                                             
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT       
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A    
 * PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"       
 * BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,  
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.                                   
 *                                                                             
 *                                                                             
 * Copyright 1992 by David A. Curry                                            
 *                                                                             
 * Permission to use, copy, modify, distribute, and sell this software and its 
 * documentation for any purpose is hereby granted without fee, provided that  
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The  
 * author makes no representations about the suitability of this software for  
 * any purpose.  It is provided "as is" without express or implied warranty.   
 *                                                                             
 * David A. Curry, N9MSW                                                       
 * Purdue University                                                           
 * Engineering Computer Network                                                
 * 1285 Electrical Engineering Building                                        
 * West Lafayette, IN 47907                                                    
 * davy@ecn.purdue.edu                                                         
 *                                                                             
 * VersaTrack Copyright (c) 1993, 1994 Siamack Navabpour. All Rights Reserved.
 *
 * Permission is hereby granted to copy, modify and distribute VersaTrack
 * in whole, or in part, for educational, non-profit and non-commercial use
 * only, free of charge or obligation, and without agreement, provided that
 * all copyrights and restrictions noted herein are observed and followed, and
 * additionally, that this and all other copyright notices listed herein
 * appear unaltered in all copies and in all derived work.
 *
 * This notice shall not in any way void or supersede any of the other authors
 * rights or privileges.
 *
 * VersaTrack IS PRESENTED FREE AND "AS IS", WITHOUT ANY WARRANTY OR SUPPORT.
 * YOU USE IT AT YOUR OWN RISK. The author(s) shall not be liable for any
 * direct, indirect, incidental, or consequential damage, loss of profits or
 * other tangible or intangible losses or benefits, arising out of or related
 * to its use. VersaTrack carries no warranty, explicit or implied, including
 * but not limited to those of merchantablity and fitness for a particular
 * purpose.
 *
 * Siamack Navabpour, 12342 Hunter's Chase Dr. Apt. 2114, Austin, TX 78729.
 * sia@bga.com or sia@realtime.com.
 */


#include <windows.h>
#include <stdio.h>
#include <math.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>

#include "vstdefs.h"
#include "constant.h"
#include "vsttype.h"
#include "vstextrn.h"

static int  month_days[] = {
    0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334
};

static char *daynames[]  = {
    "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};

long day_number(year,month,day)
int year, month, day;
{
    long result;
    
    if (year < 50)                 /* allows 4 or 2 digit year specifications */
        year += 2000;
    else
        if (year < 100)
            year += 1900;

    result = ((((long) year - 1901) * 1461) >> 2) 
             + month_days[month-1] + day + 365;

    if (year%4 == 0 && month > 2)         /* correct day number for leap year */
        result++;

    return result;
}

/*
 * siderealtime: calculates sidereal time. For reference see:
 *  "The Astronomical Almanac", 1983, page B6
 *
 *  gmsTime    = Greenwich mean sidereal time
 *  gasTime    = Greenwich apparent sidereal time
 *  lasTime    = local apparent sidereal time
 *  equEquinox = equation of the equinoxes
 */

void siderealtime(tp, timeArg)
track_t *tp;
double timeArg;
{
    double Tu, Ru, utcInterv, gmsTime;
    extern double eqxnutation(track_t *);

    tp->juliandate = timeArg + JULDAT1900 - 0.5;                     /* [d] */
    
    utcInterv  = reduce(tp->juliandate-floor(tp->juliandate)-0.5, ZERO, ONE);  /* [d] */
    Tu         = (tp->juliandate - utcInterv - JULDAT1900) / JULCENT; /* [jcy] */
    Ru         = 2.580556e-5 * Tu*Tu + 2400.051262 * Tu + 6.646065556; /* [h] */

    gmsTime    = (Ru / 24.0 + utcInterv * SIDSOLAR) * TWOPI;         /* [rad] */
    gmsTime    = reduce(gmsTime, ZERO, TWOPI);                       /* [rad] */

    if (fabs(tp->juliandate - tp->lastjdate) >= NUTEUPDATEINT) {
        tp->equinox = eqxnutation(tp);
        tp->lastjdate = tp->juliandate;
    }
    
    tp->gastime= reduce (gmsTime + tp->equinox, ZERO, TWOPI);        /* [rad] */
    tp->lastime = reduce (tp->gastime - tp->sitep->c_lng, ZERO, TWOPI); /* [rad] */
}


static void
daytoymdyd(daynum, year, month, day, yearday)
long daynum;
int  *year, *month, *day, *yearday;

{
    long Y;
    int  M, L;

    Y         = 4 * daynum;
    Y        /= 1461;
    daynum   -= 365 + (((Y - 1) * 1461) >> 2);
    *yearday  = daynum;

    L = 0;
    if (Y%4 == 0 && daynum > month_days[2])
        L = 1;

    M = 1;
    while (daynum > month_days[M] + L)
        M++;

    daynum -= (month_days[M-1]);
    if (M > 2)
        daynum -= L;
   
    *year  = Y + 1900;
    *month = M;
    *day   = daynum;
}


void printDate(fp,time, sitep, flags)
double time;
FILE   *fp;
site_t *sitep;
int flags;
{
    long daynum;
    int  day, month, year, yearDay;

    if (flags & SE_GTR_LOC) {
        time += timeZone;
    }
    else if (flags & SE_GTR_SLT) {
        time += sitep ? sitep->c_timezone : 0.0;
    }

    daynum = (long) time;
    daytoymdyd(daynum, &year, &month, &day, &yearDay);

    fprintf(fp,"%s %02d/%02d/%02d", daynames[daynum%7], month,
        day, year-1900);
}


void printTime(fp, time, sitep, flags)
FILE *fp;
double time;
site_t *sitep;
unsigned int flags;
{
    char  buf[16];

    timeStr(time, sitep, flags, 1, buf);
    buf[8] = 0;
    fprintf(fp,"%s", buf);
}

void timeStr(time, sitep, flags, gtr, str)
double time;
site_t *sitep;
unsigned int flags;
int gtr;
char *str;
{
    double dayTime;
    long   dayNum;
    int    hours, mins, secs, dummyi;
    char *tzstr;

    tzstr = "UTC";
    if (gtr) {
        if (flags & SE_GTR_LOC) {
            time += timeZone;
            tzstr = timeZoneName;
        }
        else if (flags & SE_GTR_SLT) {
            time += sitep ? sitep->c_timezone : 0.0;
            tzstr = sitep ? sitep->c_tzname : "UTC" ;
        }
    }
    else {
        if (flags & SE_RTD_LOC) {
            time += timeZone;
            tzstr = timeZoneName;
        }
        else if (flags & SE_RTD_SLT) {
            time += sitep ? sitep->c_timezone : 0.0;
            tzstr = sitep ? sitep->c_tzname : "UTC" ;
        }
    }
    dayNum  = (long) time;
    dayTime = time - (double) dayNum;
    timetodhms(dayTime, &dummyi, &hours, &mins, &secs);
    sprintf(str,"%02d:%02d:%02d  %s", hours, mins, secs, tzstr);
}

void durStr(time, str)
double time;
char *str;
{
    long   days;
    int    hours, mins, secs, dummyi;

    days  = (long) time;
    timetodhms(time - (double) days, &dummyi, &hours, &mins, &secs);
    sprintf(str,"%02d:%02d:%02d:%02d", days, hours, mins, secs);
}

static void
timetodhms(time, days, hours, mins, secs)
double time;
int    *days, *hours, *mins, *secs;

{
    double timed, timeh, timem, times;

    timed  = time;
    timeh  = 24.0 *   modf(timed,&timed);
    timem  = 60.0 *   modf(timeh,&timeh);
    times  = 60.0 *   modf(timem,&timem);
    times +=          modf(times,&times);

    if (fabs(times - 60.0) < 5.0e-4) {
        times  = 0.0;
        timem += 1.0;
        if (fabs(timem - 60.0) < 1.0e-4) {
            timem  = 0.0;
            timeh += 1.0;
            if (fabs(timeh - 24.0) < 1.0e-4) {
                timeh  = 0.0;
                timed += 1.0;
            }
        }
    }
    *days  = (int) timed;
    *hours = (int) timeh;
    *mins  = (int) timem;
    *secs  = (int) times;
}


void
dateStr(time, sitep, flags, gtr, string)
double time;  /* in day units + fractions */
site_t *sitep;
unsigned int flags; /* time zone difference in day units + day fractions */
int gtr;
char *string;
{
    double dayTime;
    long dayNum;
    int  day, month, year, yearDay, hours, mins, secs, idummy;
    char *tzstr;

    tzstr = "UTC";
    if (gtr) {
        if (flags & SE_GTR_LOC) {
            time += timeZone;
            tzstr = timeZoneName;
        }
        else if (flags & SE_GTR_SLT) {
            time += sitep ? sitep->c_timezone : 0.0;
            tzstr = sitep ? sitep->c_tzname : "UTC";
        }
    }
    else {
        if (flags & SE_RTD_LOC) {
            time += timeZone;
            tzstr = timeZoneName;
        }
        else if (flags & SE_RTD_SLT) {
            time += sitep ? sitep->c_timezone : 0.0;
            tzstr = sitep ? sitep->c_tzname : "UTC";
        }
    }
    dayNum = (long) time;
    daytoymdyd(dayNum, &year, &month, &day, &yearDay);
    dayTime = time - (double) dayNum;
    timetodhms(dayTime, &idummy, &hours, &mins, &secs);

    sprintf(string, "%02d/%02d/%02d %02d:%02d:%02d %-3.3s", month, day, year-1900,
        hours, mins, secs, tzstr);
}

double
utctime()
{
    double daynum;
    double hour;
    SYSTEMTIME tm;

    GetSystemTime(&tm);

    daynum    = day_number(tm.wYear, tm.wMonth, tm.wDay);
    hour      = (double) tm.wHour + (double)tm.wMinute / 60.0 +
                (double) tm.wSecond/3600.0 + (double)tm.wMilliseconds / (3600.0 * 1000.0);
    return  daynum + hour / 24.0 + ( 5/SPD ) ;
}

void calendar(year,yearday,pmonth,pday)
int year, yearday, *pmonth, *pday;
{
    int month, day, offset;

    static char monthdays[26] = {
        0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31, /* non-leap */
        0, 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31  /* leap */
    };

    /*
     * is "year" is a leap year ?
     * leap years    : 1600, 1984, 1988, 1992, 1996, 2000 etc.  
     * non-leap years : 1700, 1800, 1900, 2100 etc.
     */
     
    offset = ((year%4 == 0) && (year%100 == 0) && (year%400 != 0)) ? 13 : 0;

    month = 0;
    while (yearday > 0) {
        month++;
        yearday -= (int) monthdays[month+offset];
    }
    
    day = yearday + (int) monthdays[month+offset];
    
    *pmonth = month;
    *pday   = day;
}


char *monthname(month)
int  month;
{
    static char *name[] = { "???", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
                            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

    return (month < 1 || month > 12) ? name[0] : name[month];
}


double internalTime( yr, mon, day, hour, min, sec)
int yr, mon, day, hour, min, sec;
{
    double t, h;
    
    t  = (double) day_number(yr, mon, day);
    h  = (double) hour + (double) min / 60.0 + (double)sec / 3600.0;
    t += h / 24.0;
    
    return t;
}

double
dateStrtoTime(char * buf)
{
    int mm,dd,yy,hh,min,ss;
    char ch, *tzp;
    double t;

    stripTrailingSpace(buf);
    if (sscanf(buf,"%02d-%02d-%02d %02d:%02d:%02d", &mm,&dd,&yy,&hh,&min,&ss) != 6)
        if (sscanf(buf,"%02d/%02d/%02d %2d:%02d:%02d", &mm,&dd,&yy,&hh,&min,&ss) != 6)
            return -1.0;            

    tzp = timeZoneName;
    t = internalTime(yy, mm, dd, hh, min, ss);

    if (strlen(buf) > 3) {
        ch = buf[strlen(buf)-3];
        if (ch != ':' ) {
            tzp = buf + strlen(buf) - 3;
            if (!cistrcmp(tzp, "LOC") || !cistrcmp(tzp, timeZoneName))
                t -= timeZone;
            else if (cistrcmp(tzp,"UTC") && cistrcmp(tzp,"GMT")) {
                sprintf(tmpbuf, "Timezone name must be one of %s, LOC, UTC, or GMT",
                        timeZoneName);
                usermsg(tmpbuf);
                return -1.0;
            }
        }
    }
    return t;
}

void
dateOnlyStr(time, sitep, flags, gtr, string)
double time;  /* in day units + fractions */
site_t *sitep;
unsigned int flags; /* time zone difference in day units + day fractions */
int gtr;
char *string;
{
    long dayNum;
    int  day, month, year, yearDay;
    char *tzstr;

    tzstr = "UTC";
    if (gtr) {
        if (flags & SE_GTR_LOC) {
            time += timeZone;
            tzstr = timeZoneName;
        }
        else if (flags & SE_GTR_SLT) {
            time += sitep ? sitep->c_timezone : 0.0;
            tzstr = sitep ? sitep->c_tzname : "UTC";
        }
    }
    else {
        if (flags & SE_RTD_LOC) {
            time += timeZone;
            tzstr = timeZoneName;
        }
        else if (flags & SE_RTD_SLT) {
            time += sitep ? sitep->c_timezone : 0.0;
            tzstr = sitep ? sitep->c_tzname : "UTC";
        }
    }
    dayNum = (long) time;    
    daytoymdyd(dayNum, &year, &month, &day, &yearDay);
        
    sprintf(string,"%s %s %02d %d", daynames[dayNum%7], monthname(month), day, year);
}
 

