/*
 * %W% %E% %U%  [EXTREL_1.2]
 *
 * VersaTrack orbit calculations are based on those that appear in Dr. Manfred
 * Bester's sattrack program (the Unix(tm) versions 1 and 2).
 *
 * The data from which the maps where generated come from "xsat", an
 * X-Windows program by David A. Curry (N9MSW).
 *
 * Site coordinates come from various sources, including a couple of
 * World Almanacs, and also from both of the programs mentioned above.
 *
 * The following are authors' applicable copyright notices:
 *
 *                                                                               
 * Copyright (c) 1992, 1993, 1994 Manfred Bester. All Rights Reserved.        
 *                                                                           
 * Permission to use, copy, modify, and distribute this software and its      
 * documentation for educational, research and non-profit purposes, without   
 * fee, and without a written agreement is hereby granted, provided that the  
 * above copyright notice and the following three paragraphs appear in all    
 * copies.                                                                    
 *                                                                              
 * Permission to incorporate this software into commercial products may be    
 * obtained from the author, Dr. Manfred Bester, 1636 M. L. King Jr. Way,     
 * Berkeley, CA 94709, USA.                                                   
 *                                                                             
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,  
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF    
 * THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR HAS BEEN ADVISED   
 * OF THE POSSIBILITY OF SUCH DAMAGE.                                         
 *                                                                             
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT       
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A    
 * PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"       
 * BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,  
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.                                   
 *                                                                             
 *                                                                             
 * Copyright 1992 by David A. Curry                                            
 *                                                                             
 * Permission to use, copy, modify, distribute, and sell this software and its 
 * documentation for any purpose is hereby granted without fee, provided that  
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The  
 * author makes no representations about the suitability of this software for  
 * any purpose.  It is provided "as is" without express or implied warranty.   
 *                                                                             
 * David A. Curry, N9MSW                                                       
 * Purdue University                                                           
 * Engineering Computer Network                                                
 * 1285 Electrical Engineering Building                                        
 * West Lafayette, IN 47907                                                    
 * davy@ecn.purdue.edu                                                         
 *                                                                             
 * VersaTrack Copyright (c) 1993, 1994 Siamack Navabpour. All Rights Reserved.
 *
 * Permission is hereby granted to copy, modify and distribute VersaTrack
 * in whole, or in part, for educational, non-profit and non-commercial use
 * only, free of charge or obligation, and without agreement, provided that
 * all copyrights and restrictions noted herein are observed and followed, and
 * additionally, that this and all other copyright notices listed herein
 * appear unaltered in all copies and in all derived work.
 *
 * This notice shall not in any way void or supersede any of the other authors
 * rights or privileges.
 *
 * VersaTrack IS PRESENTED FREE AND "AS IS", WITHOUT ANY WARRANTY OR SUPPORT.
 * YOU USE IT AT YOUR OWN RISK. The author(s) shall not be liable for any
 * direct, indirect, incidental, or consequential damage, loss of profits or
 * other tangible or intangible losses or benefits, arising out of or related
 * to its use. VersaTrack carries no warranty, explicit or implied, including
 * but not limited to those of merchantablity and fitness for a particular
 * purpose.
 *
 * Siamack Navabpour, 12342 Hunter's Chase Dr. Apt. 2114, Austin, TX 78729.
 * sia@bga.com or sia@realtime.com.
 */


#include <windows.h>
#include <stdio.h>
#include <math.h>
#include <sys/types.h>

#include "vstdefs.h"
#include "constant.h"
#include "vsttype.h"
#include "vstextrn.h"

static int prevx = -1;
static int prevy = -1;
static double dummya;
extern void convcoord(double, double, int *, int *);

static BOOL
DurationTerminate() /* to prevent error caused by compiler optimization */
{
    return duration_stop != 0;
}    


void
DrawLine(lat, lon)
double lat, lon;
{
#undef ABS
#undef MIN
#undef MAX
#define ABS(x)      (((x) < 0) ? -(x) : (x))
#define MIN(x,y)    (((x) < (y)) ? (x) : (y))
#define MAX(x,y)    (((x) < (y)) ? (y) : (x))
    RECT r;
    int x, y;

    convcoord(lat, lon, &x, &y);
    display_lock();
    if (prevx < 0 || (ABS(prevx-x) > 80 || ABS(prevy-y) > 40)) {
        MoveToEx(hCompatDC, x, y, NULL);
        prevx = x;
        prevy = y;
    }
    LineTo(hCompatDC, x, y);
	display_unlock();
    r. left = MIN(x, prevx) -1;
    r. right = MAX(x,prevx) +1;
    r. top = MIN(y, prevy) -1;
    r. bottom = MAX( y, prevy) + 1;
    prevx = x;
    prevy = y;
    RedrawScreen(&r); /* NULL TRUE */
}                            

static WINAPI
duration_thread(sp)
select_t *sp;
{
    extern void blankBottom(void);
    double predTime, predStopTime, slowStepTime, fastStepTime;
    double lastTime, tzdiff;
    int lineno;
    BOOL firstLine, was_visible, finishPass;
    track_t *tp;
    result_t *rp;
    satellite_t *satp;
    site_t *sitep;
    char *tzstr;
    SIZE sz;
    int tx=4;

    if (!sp)
        return 0;

    tp = sp->sl_tp;
    satp = tp->satp;
    sitep = tp->sitep;
    if (!satp || !sitep)
        return 0;
    rp = sp->sl_rp;

    gtrFile = NULL;
    if (gtrfilename[0]) {
        gtrFile = fopen(gtrfilename, "w");
        if (gtrFile == NULL) {
            sprintf(tmpbuf,"Unable to open \"%s\" for writing GT data",gtrfilename);
            usermsg(tmpbuf);
        }
    }

    firstLine    = FALSE;
    was_visible = 0;
    lineno = 0;
    firstLine = 1;

    predTime = tp->starttime;

    if (predTime == -1.0)
        predTime = utctime();

    lastTime = predTime;

    predStopTime = predTime + tp->duration;

    if (predStopTime < tp->stoptime)
        predStopTime = tp->stoptime;

    predict_init(modelflag, sp, predTime);

    if (satp->s_flags & SF_GEOSTAT) {
        prevx = -1;
        sprintf(tmpbuf, "%s is Geostationary. Please use Real-time display.", satp->s_name);
        usermsg(tmpbuf);
        PostMessage(Gwnd, GTR_QUIT, (WPARAM) 0, (LPARAM) sp);
        Sleep(0);
        ExitThread(0);
    }
    prevx = prevy = -1;

    slowStepTime = tp->steptime; /* 1.0 / tp->epochmeanmotion / 2000.0; */
    fastStepTime = slowStepTime * 3.0;

    MakePen(hCompatDC, PenColor);
    SetROP2(hCompatDC, R2_COPYPEN);
    blankBottom();

    sprintf(tmpbuf,"%s Ground Track", satp->s_name);
    DrawTextStr(hCompatDC, tx, tRect.top+2, SRCCOPY, CBLACK, 8, hHelv, tmpbuf, &sz);
    tx += sz.cx+26;
    DrawTextStr(hCompatDC, tx, tRect.top+2, SRCCOPY, CBLACK, 8, hHelv, "Start:", &sz);
    tx += sz.cx+7;
    dateStr(predTime, tp->sitep, sp->flags, 1, tmpbuf);
    DrawTextStr(hCompatDC, tx, tRect.top+2, SRCCOPY, CBLACK, 8, hHelv, tmpbuf, &sz);
    tx += sz.cx+16;
    DrawTextStr(hCompatDC, tx, tRect.top+2, SRCCOPY, CBLACK, 8, hHelv, "End:", &sz);
    tx += sz.cx+7;
    dateStr(predStopTime, tp->sitep, sp->flags, 1, tmpbuf);
    DrawTextStr(hCompatDC, tx, tRect.top+2, SRCCOPY, CBLACK, 8, hHelv, tmpbuf, &sz);
    setwsize(Gwnd, TRUE);
	RedrawScreen(&tRect);
    lineno = 0;
    if (gtrFile) {
        dateStr(predTime, tp->sitep, sp->flags, 1, tmpbuf);
        fprintf(gtrFile, "\nSatellite Name:      %s",satp->s_name);
        if (satp->s_modep && satp->s_modep->alias)
	        fprintf(gtrFile," (%s)", satp->s_modep->alias);
        fprintf(gtrFile, "\nElement Set Number:  %d\n",satp->s_elementset);
        fprintf(gtrFile, "Elset Epoch Time:    %s UTC\n",satp->s_epochstr);
        fprintf(gtrFile, "Prediction Start:    %s\n", tmpbuf);
        fprintf(gtrFile, "Duration:            %-.2f days\n", tp->duration);
        fprintf(gtrFile, "Starting Orbit No.:  %d\n", (int) rp->r_orbitnum);
	   	sprintf(tmpbuf, "(%-.4f %c %-.4f %c)", fabs(sitep->c_lat * CRD),
	    	sitep->c_lat < 0.0 ? 'S' : 'N', fabs(sitep->c_lng * CRD),
			sitep->c_lng > 0.0 ? 'W' : 'E');
        fprintf(gtrFile, "Ground Station:      %-.31s, %-.39s %s\n",
			sitep->c_name, sitep->c_locale, tmpbuf);
        tzstr = "UTC";
        tzdiff = 0;
        
        if (sp->flags & SE_GTR_LOC) {
            tzstr = timeZoneName;
            tzdiff = timeZone * 24.0;
        }
        else if (sp->flags & SE_GTR_SLT) {
            tzstr = sitep->c_tzname;
            tzdiff = sitep->c_timezone * 24.0;
        }
        fprintf(gtrFile, "Report Time in:      %s (UTC %+2.2lf hrs)\n", tzstr, tzdiff);
    }
    firstLine    = FALSE;
    was_visible =  FALSE;
    lineno = 8;

    do {
        if (predict(modelflag, sp, predTime) == 0)
            break;

        if (DurationTerminate())
            break;

        DrawLine(rp->r_lat, rp->r_long);

        if (rp->r_elevation >= minElevation) {

            if (++lineno > 55) {
                firstLine = 1;
                lineno = 0;
                if (gtrFile) fprintf(gtrFile,"\f");
            }
            if (gtrFile && ( firstLine || !was_visible)) {
                if (firstLine && was_visible) {
	            	fprintf(gtrFile,"\n\n\n"), lineno += 2;
                }
                else {
                    fprintf(gtrFile,"\n\n");
                    printDate(gtrFile, predTime, sitep, sp->flags);
                    fprintf(gtrFile,"  ---  Orbit %ld\n\n",rp->r_orbitnum);
                    if (sp->flags & SE_GTR_LOC)
                        fprintf(gtrFile,"   %-3.3s",timeZoneName);
                    else if (sp->flags & SE_GTR_UTC)
                        fprintf(gtrFile,"   UTC");
                    else
                        fprintf(gtrFile,"   %3s",sitep->c_tzname);
                    fprintf(gtrFile,"   Azimuth  Elev    Range   Lat    Long    Height  Doppler  Loss  Phase InSun\n");
                    fprintf(gtrFile,"          [deg]  [deg]     [km]  [deg]   [deg]    [km]    [kHz]   [dB]\n\n");
                    lineno += 6;
                }
                firstLine = 0;
            }

            if (gtrFile) {
                printTime(gtrFile, predTime, sitep, sp->flags);
                fprintf(gtrFile,"  %5.1lf  %5.1lf  %7.1lf", rp->r_azimuth,
                    rp->r_elevation, rp->r_range);
                fprintf(gtrFile,"  %+5.1lf  %+6.1lf  %7.1lf", rp->r_lat,
                    rp->r_long, rp->r_height);
                fprintf(gtrFile,"  %5.llf  %5.1lf  %3d   %c\n", rp->r_doppler,
                    rp->r_pathloss, rp->r_phase, rp->r_insun);
            }
            was_visible = TRUE;
        }
        else {
            was_visible = FALSE;
        }
        predTime   +=  (rp->r_elevation < VLOWELEV) ? fastStepTime : tp->steptime;

        finishPass  = (rp->r_elevation >= tp->minelevation) ? TRUE : FALSE;
        
    } while (predTime <= predStopTime || finishPass);

	if (DurationTerminate() && gtrFile)
	    fprintf(gtrFile,"\n***** TERMINATED BY USER COMMAND *****\n");
	    
    prevx = -1;

    if (gtrFile)
        fclose(gtrFile);

    PostMessage(Gwnd, GTR_QUIT, (WPARAM) 0, (LPARAM) sp);
    Sleep(0);
    Sleep(0);
    ExitThread(0);
    return 0;
}

void
duration_predict(sp)
select_t *sp;
{
    long thid;
    extern char *ErrorString(int);

    if (!sp || duration_running || duration_stop)
        return;

    WaitCursor();
    
    gtr_thread_handle = CreateThread(NULL, 0, (PTHREAD_START_ROUTINE)
        duration_thread, (void *) sp, CREATE_SUSPENDED, &thid);
        
    if ((gtr_thread_handle == NULL) || (gtr_thread_handle ==INVALID_HANDLE_VALUE)) {
		sprintf(tmpbuf,"Cannot create GTR thread: %s",ErrorString(GetLastError()));  
        NormCursor();
        usermsg(tmpbuf);
    }
    else {
        duration_running = 1;
	    ResumeThread(gtr_thread_handle);
	}
}


void
continuous_display_init(sp)
select_t *sp;
{
    extern void rtd_allbegin();
    int r;

	if (!selInfo) {
	    usermsg("Active List is empty");
	    return;
	}
    WaitCursor();
    
    SetROP2(hCompatDC, R2_XORPEN);
    MakePen(hCompatDC, CBLACK);
    MakeBrush(hCompatDC, BrushColor);

    if (!sp) {
        for (sp = selInfo; sp; sp = sp->sl_next)
            if ((r =rtd_start(Gwnd, hInst, sp)) == FALSE)
            	break;
    }
    else
        r = rtd_start(Gwnd, hInst, sp);
        
    EnableWindow(Gwnd, TRUE);
    SetFocus(Gwnd);
    
    NormCursor();
}

void
continuous_display_finish(sp)
select_t *sp;
{
    select_t *spp;
    int n=0;

	if (!selInfo) return;
    
    WaitCursor();

    if (!sp) {
        for (spp = selInfo; spp != 0; spp = spp->sl_next)
            n |= rtd_finish(spp);
    }
    else
        n = rtd_finish(sp);
    if (!n) {
        SetFocus(Gwnd);
        NormCursor();
    }
}

 
