/*
 * %W% %E% %U%  [EXTREL_1.2]
 *
 * VersaTrack orbit calculations are based on those that appear in Dr. Manfred
 * Bester's sattrack program (the Unix(tm) versions 1 and 2).
 *
 * The data from which the maps where generated come from "xsat", an
 * X-Windows program by David A. Curry (N9MSW).
 *
 * Site coordinates come from various sources, including a couple of
 * World Almanacs, and also from both of the programs mentioned above.
 *
 * The following are authors' applicable copyright notices:
 *
 *                                                                               
 * Copyright (c) 1992, 1993, 1994 Manfred Bester. All Rights Reserved.        
 *                                                                           
 * Permission to use, copy, modify, and distribute this software and its      
 * documentation for educational, research and non-profit purposes, without   
 * fee, and without a written agreement is hereby granted, provided that the  
 * above copyright notice and the following three paragraphs appear in all    
 * copies.                                                                    
 *                                                                              
 * Permission to incorporate this software into commercial products may be    
 * obtained from the author, Dr. Manfred Bester, 1636 M. L. King Jr. Way,     
 * Berkeley, CA 94709, USA.                                                   
 *                                                                             
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,  
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF    
 * THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR HAS BEEN ADVISED   
 * OF THE POSSIBILITY OF SUCH DAMAGE.                                         
 *                                                                             
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT       
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A    
 * PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"       
 * BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,  
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.                                   
 *                                                                             
 *                                                                             
 * Copyright 1992 by David A. Curry                                            
 *                                                                             
 * Permission to use, copy, modify, distribute, and sell this software and its 
 * documentation for any purpose is hereby granted without fee, provided that  
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The  
 * author makes no representations about the suitability of this software for  
 * any purpose.  It is provided "as is" without express or implied warranty.   
 *                                                                             
 * David A. Curry, N9MSW                                                       
 * Purdue University                                                           
 * Engineering Computer Network                                                
 * 1285 Electrical Engineering Building                                        
 * West Lafayette, IN 47907                                                    
 * davy@ecn.purdue.edu                                                         
 *                                                                             
 * VersaTrack Copyright (c) 1993, 1994 Siamack Navabpour. All Rights Reserved.
 *
 * Permission is hereby granted to copy, modify and distribute VersaTrack
 * in whole, or in part, for educational, non-profit and non-commercial use
 * only, free of charge or obligation, and without agreement, provided that
 * all copyrights and restrictions noted herein are observed and followed, and
 * additionally, that this and all other copyright notices listed herein
 * appear unaltered in all copies and in all derived work.
 *
 * This notice shall not in any way void or supersede any of the other authors
 * rights or privileges.
 *
 * VersaTrack IS PRESENTED FREE AND "AS IS", WITHOUT ANY WARRANTY OR SUPPORT.
 * YOU USE IT AT YOUR OWN RISK. The author(s) shall not be liable for any
 * direct, indirect, incidental, or consequential damage, loss of profits or
 * other tangible or intangible losses or benefits, arising out of or related
 * to its use. VersaTrack carries no warranty, explicit or implied, including
 * but not limited to those of merchantablity and fitness for a particular
 * purpose.
 *
 * Siamack Navabpour, 12342 Hunter's Chase Dr. Apt. 2114, Austin, TX 78729.
 * sia@bga.com or sia@realtime.com.
 */


#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "resource.h"
#include "vstdefs.h"
#include "constant.h"
#include "vsttype.h"
#include "vstextrn.h"

#define DEFFLAGS    (SE_RTD_DIC|SE_GTR_UTC|SE_RTD_UTC|SE_RTD_POA|SE_RTD_BOA)
extern void flagStr(unsigned int, char *);

struct flags_t {
    char *name;
    unsigned int orvalue, andvalue;
};

static struct flags_t Flagtab[16] = {
    "BOA", SE_RTD_BOA, 0,
    "POA", SE_RTD_POA, 0,
    "BE", SE_RTD_BEEP, 0,
    "ICO", SE_RTD_DIC, 0,
    "BMP", SE_RTD_BMP, 0,
    "GU", SE_GTR_UTC,  SE_GTR_LOC | SE_GTR_SLT,
    "GL", SE_GTR_LOC,  SE_GTR_UTC | SE_GTR_SLT,
    "GS", SE_GTR_SLT,  SE_GTR_UTC | SE_GTR_LOC,
    "RU", SE_RTD_UTC,  SE_RTD_LOC | SE_RTD_SLT,
    "RL", SE_RTD_LOC,  SE_RTD_UTC | SE_RTD_SLT,
    "RS", SE_RTD_SLT,  SE_RTD_UTC | SE_RTD_LOC,
    "HIC",SE_RTD_HIC,  SE_RTD_DIC,
    "SIM", SE_RTD_SIM, 0,
    "UA", SE_RTD_USERA, 0,
    "UB", SE_RTD_USERB, 0,
    0, 0, 0
};


/*
 * List the active list.
 */
 
void
listSelections(hwnd)
HWND hwnd; 
{
    register select_t *sp;

    for (sp = selInfo; sp ; sp = sp->sl_next)
        sp->sl_index =  SendMessage(hwnd, LB_ADDSTRING, (WPARAM)0, (LPARAM) sp->sl_buf);
}


select_t *
addSelection(satp, sitep, start, dur, step, minelev, update, flags)
satellite_t *satp;
site_t *sitep;
double start, dur, step, minelev;
int update;
unsigned int flags;
{
    select_t *sp,*spp;
    track_t *tp;
    
    if (!satp || !sitep) {
        usermsg("must select a satellite and site first!");
        return NULL;
    }

    if ((sp = findSelectionbyName(satp, sitep)) != NULL) {
        usermsg("Duplicate entry already exists.");
        return NULL;
    }

    spp = (select_t *) safeAlloc(sizeof(select_t));
    tp = (track_t *) safeAlloc(sizeof(track_t));

    if (!spp || !tp)
        return NULL;

    if (!(spp->sl_rp = (result_t *) safeAlloc(sizeof(result_t))))
        return NULL;
    
    InitializeCriticalSection(&spp->rtd_mutex);
    
    tp->satp = satp;
    tp->sitep = sitep;

    tp->steptime = step;
    tp->minelevation = (float)minelev;
    tp->starttime = start;
    tp->duration = dur;
    spp->sl_tp = tp;
    spp->rtdhwnd = 0;
    spp->rtd_thread_handle = NULL;
    spp->sl_index = -1;
    spp->flags = flags ? flags : DEFFLAGS;
    spp->updatetime = update ? update : 30;
    spp->sl_next = NULL;

    sprintf(tmpbuf, "%-14.14s %-.31s, %-.39s", satp->s_name, sitep->c_name, sitep->c_locale);
    strncpy(spp->sl_buf, tmpbuf, sizeof(spp->sl_buf));

    if (selInfo == NULL)
        selInfo = spp;
    else {
        for (sp = selInfo; sp->sl_next; sp = sp->sl_next) ;
        sp->sl_next = spp;
    }
    return spp;
}

BOOL
modifySelectionbyName(hwnd, selp, satp, sitep, start, dur, step, minelev, update, flags)
HWND hwnd;
select_t *selp;
satellite_t *satp;
site_t *sitep;
double start, dur, step, minelev;
int update;
unsigned int flags;
{
    track_t *tp;
    
    if (!selp)
        return FALSE;


    if (selp->running || selp->rtdhwnd) {
#ifdef _DEBUG_
        diag("modifySelection: rtd running: %d  hwnd %x\n", selp->running, selp->rtdhwnd);
#endif /* _DEBUG_ */
        usermsg("Entry currently in use. Cannot modify.");
        return FALSE;
    }
    if (!satp || !sitep) {
        usermsg("must select satellite and site first!");
        return FALSE;
    }
    tp = selp->sl_tp;

    tp->satp = satp;
    tp->sitep = sitep;
    
    tp->starttime = start;
    tp->duration = (dur <= 0.0) ? atof(DURATION_DEFAULT) : dur;
    tp->steptime = (step <= 0.0) ? atof(STEPTIME_DEFAULT)/SPD : step;
    selp->updatetime = (update <=0 ) ? atoi(UPDATE_DEFAULT) : update;
    tp->minelevation = (float) minelev;
    selp->flags = flags;

    sprintf(tmpbuf, "%-14.14s %-.31s, %-.39s", satp->s_name, sitep->c_name, sitep->c_locale);
    strncpy(selp->sl_buf, tmpbuf, sizeof(selp->sl_buf));

    SendMessage(hwnd, LB_DELETESTRING, (WPARAM) selp->sl_index, (LPARAM) 0);
    SendMessage(hwnd, LB_INSERTSTRING, (WPARAM) selp->sl_index, (LPARAM) selp->sl_buf);

    return TRUE;
}


int
deleteSelection(select_t * sp)
{
    select_t *spp, *psp;
    int index;
    track_t *tp;
    
    if (selInfo == NULL) {
        usermsg("Active list is empty");
        return -1;
    }
    if (sp == NULL) {
        usermsg("No current selection");
		return -1;
	}
    for (psp = NULL, spp = selInfo; spp ; spp = spp->sl_next) {
        if (spp == sp) {
            if (spp->running || spp->rtdhwnd) {
                tp = spp->sl_tp;
                sprintf(tmpbuf,"Entry for (%-.31s , %-.39s) is currently in use.",
                    tp->satp->s_name, tp->sitep->c_name);
                usermsg(tmpbuf);
            }
            sp = spp->sl_next;
            index = spp->sl_index;
            if (psp) {
                psp->sl_next = sp;
            }
            else
                selInfo = sp;
                
            if (spp == selInfo)
                selInfo = NULL;
           
            if (currentSel == spp)
                currentSel  = NULL;
                
            for (; sp ; sp = sp->sl_next)
                sp->sl_index -= 1;
                
            safeFree(spp->sl_tp);
            safeFree(spp->sl_rp);
                        
            DeleteCriticalSection(&spp->rtd_mutex);
            safeFree(spp);
            
            return index;
        }
        psp = spp;
    }
    usermsg("Entry not found");
    return -1;
}

 
select_t *
findSelectionbyName(satp, sitep)
satellite_t *satp;
site_t *sitep;
{
    select_t *sp;
    register track_t *tp;
    
    if (satp && sitep) {
        for (sp = selInfo; sp ; sp = sp->sl_next) {
            tp = sp->sl_tp;
	    if (satp == tp->satp && sitep == tp->sitep) {
	        if ((sp != currentSel) && ((serverInfo[0].srv_state == ST_SRV_RUNNING)||
	            (serverInfo[1].srv_state == ST_SRV_RUNNING)))
	            if (!yesno("Radio/Rotator Control running! Are you sure you want \
to change the current satellite ?"))
                        return NULL;
                return currentSel = sp;
    	    }
        }
    }
    return (select_t *)NULL;
}

select_t *
findSelectionbyID(id)
register int id;
{
    register select_t *sp;
    
    for (sp = selInfo; sp ; sp = sp->sl_next)
        if (sp->sl_index == id) {
            if ((sp != currentSel) && ((serverInfo[0].srv_state == ST_SRV_RUNNING)||
                (serverInfo[1].srv_state == ST_SRV_RUNNING)))
	            if (!yesno("Radio/Rotator Control running! Are you sure you want \
to change the current satellite ?"))
                        return NULL;
            return currentSel = sp;
        }
    return (select_t *) NULL;
}


/*
 * convert flag bits to comma separated list of character "abreviations".
 */ 
void
flagStr(flags, str)
unsigned int flags;
char *str;
{
    struct flags_t *fsp;
    
    str[0] = 0;
    for (fsp = Flagtab; fsp->name; fsp++) {
        if (flags & fsp->orvalue) {
            if (str[0])
                strcat(str, ",");
            strcat(str, fsp->name);
        }
    }
}

/*
 *  Read the active list from active.dat.
 */

void readactiveDB()
{    
    char *sitenamep, *satnamep, *cp;
    char *line, linebuf[240];
    double minelev, steptime, duration, ttime;
    char *delim = " \t\n;,";
    FILE *fp;
    int lineno, update;
    unsigned int flags;
    site_t *sitep;
    satellite_t *satp;
    struct flags_t *flp;
    extern satellite_t *findSatbyName(char *);
    
    sprintf(linebuf,"%s\\%s", datadir, activefilename);
    if ((fp = fopen(linebuf, "r")) == NULL) {
        sprintf(tmpbuf,"Can't open %s", linebuf);
#ifdef _DEBUG_
        diag(tmpbuf);
#endif /* _DEBUG_ */
        usermsg(tmpbuf);
    }

    lineno = 0;
    if (!timeZoneName[0]) {
        strcpy(timeZoneName,"UTC");
        timeZone = 0.0;
    }
#ifdef _DEBUG_
	diag("Active list:\n");
#endif /* _DEBUG_ */

    while (fp && !feof(fp)) {
        linebuf[0] = 0;
        if (!(line = getline(linebuf, sizeof(linebuf), fp)) || (linebuf[0] == 0))
            break;
        lineno++;
        if (line[0] == '#' || line[0] == '\n')
            continue;
            
        satnamep = strtok(line, ":");

        sitenamep = strtok(NULL, ":");
        sitenamep = stripLeadingSpace(sitenamep);

        if (sitenamep) {
            stripTrailingSpace(sitenamep);
        	if (strchr(sitenamep,'*'))
	            sitenamep = SiteName;
	    }
	    else
	    	sitenamep = "???";
	    	
        if (satnamep)
            stripTrailingSpace(satnamep);
        else
            satnamep = "???";

        if ((satp =  findSatbyName(satnamep)) == NULL) {
            sprintf(tmpbuf, "%s line %d: can't find satellite \"%s\".",
                activefilename, lineno, satnamep);
            usermsg(tmpbuf);
            continue;
        }

        if ((sitep = findSitebyName(sitenamep)) == NULL) {
            sprintf(tmpbuf, "%s line %d: can't find site \"%s\".",
                activefilename, lineno, sitenamep);
            usermsg(tmpbuf);
            continue;
        }
		
        ttime = -1.0;
        cp = strtok(NULL, ",;!|");
        if (cp) {
            cp = stripLeadingSpace(cp);
            stripTrailingSpace(cp);
            if (*cp != '*') {
                ttime = dateStrtoTime(cp);
                if (ttime < 0.0) {
                    sprintf(tmpbuf,"Invalid date/time format in %s line %d\n",
                        activefilename, lineno);
                    usermsg(tmpbuf);
                    ttime = -1.0;
                }
            }
        }
        cp = strtok(NULL, delim);
        cp = stripLeadingSpace(cp);
        duration = atof((cp && *cp != '*') ? cp : DURATION_DEFAULT);

        cp = strtok(NULL, delim);
        cp = stripLeadingSpace(cp);
        steptime = atof((cp && *cp != '*') ? cp : STEPTIME_DEFAULT) / SPD;

        cp = strtok(NULL, delim);
        cp = stripLeadingSpace(cp);
        minelev = atof((cp && *cp != '*') ? cp : MINELEVATION_DEFAULT ) * CDR;

        cp = strtok(NULL, delim);
        cp = stripLeadingSpace(cp);
        update = atoi((cp && *cp != '*') ? cp :  UPDATE_DEFAULT);

        cp = strtok(NULL, delim);
        flags = SE_RTD_KM;
        while (cp) {
            cp = stripLeadingSpace(cp);
            for (flp = Flagtab; flp->name; flp++) {
                if (cistrcmp(cp, flp->name) == 0) {
                    flags &= ~ flp->andvalue;
                    flags |= flp->orvalue;
                }
            }
            cp = strtok(NULL, delim);
        } while (cp);
#ifdef _DEBUG_
		diag("%02d: %s, %s\n", lineno, satp->s_name, sitep->c_name);
#endif /* _DEBUG_ */
        addSelection(satp, sitep, ttime, duration, steptime, minelev, update, flags);
    }

    if (selInfo)
        currentSel = selInfo;
    if (fp)
        fclose(fp);
}

void
saveactivelist()
{
    FILE *fp;
    select_t *selp;
    track_t *tp;
    int lineno, i;
    char *sitenamep;
    char start[40], step[10], dur[10], minelev[10], update[10];

    if (!selInfo)
        return;

    sprintf(tmpbuf,"%s\\%s", datadir,activefilename);

    if ((fp = fopen(tmpbuf,"w")) == NULL) {
        sprintf(string,"Cannot create/write \"%s\"\n",tmpbuf);
#ifdef _DEBUG_
        diag(string);
#endif /* _DEBUG_ */
        usermsg(string);
        return;
    }

    fprintf(fp,"#TZNAME= %s, TZDIFF= %-.3f\n", timeZoneName, timeZone * 24.0);
    fprintf(fp,"#\n# Satellite, site,start time, duration, step time, min. elev., update time, flags\n");

    for (lineno = 4, selp = selInfo; selp; selp = selp->sl_next, lineno++) {
        tp = selp->sl_tp;

        if (selp->updatetime == 30)
            strcpy(update, "*");
        else
            sprintf(update,"%d", selp->updatetime);

        if (tp->starttime == -1.0)
            strcpy(start,"*");
        else
            dateStr(tp->starttime, tp->sitep, SE_GTR_UTC, 0, start);

        if (tp->steptime > (29.0/SPD) && tp->steptime < (31.0/SPD) )
            strcpy(step,"*");
        else
            sprintf(step,"%.0f", tp->steptime * SPD);

        if (tp->duration == 1.0)
            strcpy(dur, "*");
        else
            sprintf(dur,"%.4f", tp->duration);

        if (tp->minelevation > (-0.2*CDR) && tp->minelevation < (0.2*CDR))
            strcpy(minelev, "*");
        else
            sprintf(minelev, "%.0f", tp->minelevation * CRD);

        sprintf(tmpbuf, "%s, %s", tp->sitep->c_name, tp->sitep->c_locale);
        sitenamep = tmpbuf;
        if (cistrcmp(sitenamep, SiteName) == 0)
            sitenamep = "*";

        for (i=0; i<BUFLEN; i++) textbuf[i] = 0;
        sprintf(textbuf, "%-.31s: %-.39s:", tp->satp->s_name, sitenamep);
        for (i=strlen(textbuf); i<30; i++) textbuf[i] = ' ';

        flagStr(selp->flags, tmpbuf);
        fprintf(fp, "%s  %s,\t%s\t%s\t%s\t%s\t%s\n", textbuf,  start, dur, step,
            minelev, update, tmpbuf);
        fflush(fp);

        if (ferror(fp))
            break;
    }

    if (ferror(fp))
        usermsg("Error writing selections to file");

    fclose(fp);

    sprintf(tmpbuf, "%d entries in active list", lineno-4);
    usermsg(tmpbuf);
#ifdef _DEBUG_
    diag("wrote %d entries to defaults file\n", lineno-4);
#endif
}
