/*
 * %W% %E% %U%
 * Sample code for communication with external device via a supported
 * serial port. The routines in this file are used for setting up
 * the serial port parameters. Externally visible interface is
 * SetPortParams(). A dialog box is used to solicit changes.
 * Results will be used to update/set a tty_t struct.
 */

#include "resource.h"
#include "radextrn.h"
#include "vsttype.h"

ttyadj_t ttyadj = { 0 };

radparam_t rigInfo = {
    {
        2, 4800, 8, NOPARITY, ONESTOPBIT, 0, 0, { 0 }, &ttyadj
    },
    { 0 }
};

static int portflags;
static char portname[48];
static char baudrate[16];
static char icdelay[16];
static char optionals[32];
static tty_t *ttyp;

#define RAD_RTSCTS       0x000001 /* MUST BE SAME AS TTY_RTSCTS */
#define RAD_DSRDTR       0x000002 /* MUST BE SAME AS TTY_DSRDTR */
#define RAD_XONXOFF      0x000004 /* MUST BE SAME AS TTY_XONXOFF */
#define RAD_RFU          0x000008
#define RAD_DBITS_7      0x000010
#define RAD_DBITS_8      0x000020
#define RAD_STOP_1       0x000040
#define RAD_STOP_15      0x000080
#define RAD_STOP_2       0x000100
#define RAD_PARITY_NONE  0x000200
#define RAD_PARITY_ODD   0x000400
#define RAD_PARITY_EVEN  0x000800
#define RAD_PARITY_MARK  0x001000
#define RAD_PARITY_SPACE 0x002000
#define RAD_DBITS_6      0x004000

button_t BtnTab[] = {
    RAD_DBITS_6,      IDB_DATABITS_6,  0,
    RAD_DBITS_7,      IDB_DATABITS_7,  0,
    RAD_DBITS_8,      IDB_DATABITS_8,  0,
    RAD_STOP_1,       IDB_STOPBITS_1,  0,
    RAD_STOP_15,      IDB_STOPBITS_15, 0,
    RAD_STOP_2,       IDB_STOPBITS_2,  0,
    RAD_PARITY_NONE,  IDB_PARITY_NONE, 0,
    RAD_PARITY_ODD,   IDB_PARITY_ODD, 0,
    RAD_PARITY_EVEN,  IDB_PARITY_EVEN, 0,
    RAD_PARITY_MARK,  IDB_PARITY_MARK, 0,
    RAD_PARITY_SPACE, IDB_PARITY_SPACE, 0,
    RAD_RTSCTS,       IDB_HS_RTSCTS, 0,
    RAD_DSRDTR,       IDB_HS_DTR, 0,
    RAD_XONXOFF,      IDB_HS_XONXOFF, 0,
    -1,             -1, 0,
};


static void
ttytocontrols()
{
    register tty_t *tp = ttyp;
    
    portflags = tp->tt_flags & 07;
    sprintf(portname,"COM%d", tp->tt_portno);
    sprintf(baudrate,"%d", tp->tt_baudrate);
    sprintf(icdelay,"%d", tp->tt_icdelay);
    strncpy(optionals, tp->tt_optparams, sizeof(optionals) -1);
    
    if (tp->tt_stopbits == ONESTOPBIT)
        portflags |= RAD_STOP_1;
    if (tp->tt_stopbits == ONE5STOPBITS)
        portflags |= RAD_STOP_15;
    if (tp->tt_stopbits == TWOSTOPBITS)
        portflags |= RAD_STOP_2;

    if (tp->tt_databits == 6)
        portflags |= RAD_DBITS_6;
    if (tp->tt_databits == 7)
        portflags |= RAD_DBITS_7;
    if (tp->tt_databits == 8)
        portflags |= RAD_DBITS_8;

    if (tp->tt_parity == NOPARITY)
        portflags |= RAD_PARITY_NONE;
    if (tp->tt_parity == ODDPARITY)
        portflags |= RAD_PARITY_ODD;
    if (tp->tt_parity == EVENPARITY)
        portflags |= RAD_PARITY_EVEN;
    if (tp->tt_parity == MARKPARITY)
        portflags |= RAD_PARITY_MARK;
    if (tp->tt_parity == SPACEPARITY)
        portflags |= RAD_PARITY_SPACE;
}

static void
controlstotty()
{
    register tty_t *tp = ttyp;
#ifdef _DEBUG_
    diag("Baud Rate %s port %s\n", baudrate, portname);
#endif /* _DEBUG_ */

    sscanf(baudrate,"%d", &tp->tt_baudrate);
    
    UpperCase(portname);
    sscanf(portname,"COM%d",&tp->tt_portno);
    
    sscanf(icdelay,"%d", &tp->tt_icdelay);
    
    if (tp->tt_optparams)
        GlobalFree((HANDLE)tp->tt_optparams);
        
    strncpy(tp->tt_optparams, optionals, sizeof(tp->tt_optparams) -1);
    
    if (portflags & RAD_DBITS_6)
        tp->tt_databits = 6;
    if (portflags & RAD_DBITS_7)
        tp->tt_databits = 7;
    if (portflags & RAD_DBITS_8)
        tp->tt_databits = 8;
        
    if (portflags & RAD_STOP_1)
        tp->tt_stopbits = ONESTOPBIT;
    if (portflags & RAD_STOP_15)
        tp->tt_stopbits = ONE5STOPBITS;
    if (portflags & RAD_STOP_2)
        tp->tt_stopbits = TWOSTOPBITS;

    if (portflags & RAD_PARITY_NONE)
        tp->tt_parity = NOPARITY;
    if (portflags & RAD_PARITY_ODD)
        tp->tt_parity = ODDPARITY;
    if (portflags & RAD_PARITY_EVEN)
        tp->tt_parity = EVENPARITY;
    if (portflags & RAD_PARITY_MARK)
        tp->tt_parity = MARKPARITY;
    if (portflags & RAD_PARITY_SPACE)
        tp->tt_parity = SPACEPARITY;
    tp->tt_flags = portflags & 07;
}
    

void
setbutton(HWND hwnd, int id, BOOL flag)
{
    SendDlgItemMessage(hwnd, id, BM_SETSTATE, flag ? (WPARAM) 1 : 0, (LPARAM) 0);
}
    

void
setbuttons(HWND hwnd, int state, button_t *btab)
{
    register button_t *btp;
    int p;

    for (btp = btab; btp->fval != -1; btp++) {
        p = btp->fval & state;
        setbutton(hwnd, btp->bval, p ? TRUE : FALSE);
    }
}


static int
checkbuttons(hwnd, wParam)
HWND hwnd;
WPARAM wParam;
{
    int cmd;

    cmd = LOWORD(wParam);
    
    switch (cmd) {
    case IDOK:
    case IDCANCEL:
        return cmd;
    default:
        break;
    }
    
    if (HIWORD(wParam) != BN_CLICKED)
        return -1;

    switch (cmd) {
    case IDB_STOPBITS_1:
        portflags &= ~(RAD_STOP_2|RAD_STOP_15);
        portflags |= RAD_STOP_1;
        goto bset;

    case IDB_STOPBITS_15:
        portflags &= ~(RAD_STOP_1|RAD_STOP_2);
        portflags |= RAD_STOP_15;
        goto bset;

    case IDB_STOPBITS_2:
        portflags &= ~(RAD_STOP_1|RAD_STOP_15);
        portflags |= RAD_STOP_2;
        goto bset;

    case IDB_PARITY_NONE:
        portflags &= ~(RAD_PARITY_ODD|RAD_PARITY_EVEN|RAD_PARITY_MARK|RAD_PARITY_SPACE);
        portflags |= RAD_PARITY_NONE;
        goto bset;

    case IDB_PARITY_ODD:
        portflags &= ~(RAD_PARITY_NONE|RAD_PARITY_EVEN|RAD_PARITY_MARK|RAD_PARITY_SPACE);
        portflags |= RAD_PARITY_ODD;
        goto bset;
        
    case IDB_PARITY_EVEN:
        portflags &= ~(RAD_PARITY_NONE|RAD_PARITY_ODD|RAD_PARITY_MARK|RAD_PARITY_SPACE);
        portflags |= RAD_PARITY_EVEN;
        goto bset;
        
    case IDB_PARITY_MARK:
        portflags &= ~(RAD_PARITY_NONE|RAD_PARITY_ODD|RAD_PARITY_EVEN|RAD_PARITY_SPACE);
        portflags |= RAD_PARITY_MARK;
        goto bset;
        
    case IDB_PARITY_SPACE:
        portflags &= ~(RAD_PARITY_NONE|RAD_PARITY_ODD|RAD_PARITY_EVEN|RAD_PARITY_MARK);
        portflags |= RAD_PARITY_SPACE;
        goto bset;

    case IDB_HS_RTSCTS:
        portflags ^= RAD_RTSCTS;
        goto bset;
        
    case IDB_HS_DTR:
        portflags ^= RAD_DSRDTR;
        goto bset;
        
    case IDB_HS_XONXOFF:
        portflags ^= RAD_XONXOFF;
        goto bset;
        
    case IDB_DATABITS_6:
        portflags &= ~(RAD_DBITS_7|RAD_DBITS_8);
        portflags |= RAD_DBITS_6;
        goto bset;
        
    case IDB_DATABITS_7:
        portflags &= ~(RAD_DBITS_6|RAD_DBITS_8);
        portflags |= RAD_DBITS_7;
        goto bset;
            
    case IDB_DATABITS_8:
        portflags &= ~(RAD_DBITS_6|RAD_DBITS_7);
        portflags |= RAD_DBITS_8;
        /* Fall Through to bset */

    bset:
        setbuttons(hwnd, portflags, BtnTab);
        break;
        
     default: /* Warning: must return -1 */
        break;
     }
     return -1;
}


static void
setControls(hwnd)
HWND hwnd;
{    
    SendDlgItemMessage(hwnd, IDC_PORTPARAMS_PORT, WM_SETTEXT, (WPARAM) 0,
        (LPARAM) portname);
    SendDlgItemMessage(hwnd, IDC_PORTPARAMS_BAUD, WM_SETTEXT, (WPARAM) 0,
        (LPARAM) baudrate);
    SendDlgItemMessage(hwnd, IDC_PORTPARAMS_OPTIONALS, WM_SETTEXT, (WPARAM) 0,
        (LPARAM) optionals);
    SendDlgItemMessage(hwnd, IDC_PORTPARAMS_ICDELAY, WM_SETTEXT, (WPARAM) 0,
        (LPARAM) icdelay);

    setbuttons(hwnd, portflags, BtnTab);
}

static void
getControls(hwnd)
HWND hwnd;
{
    extern char *GetEditText(HWND, int, char *, int);

    GetEditText(hwnd, IDC_PORTPARAMS_PORT, portname, sizeof(portname));
    GetEditText(hwnd, IDC_PORTPARAMS_BAUD, baudrate, sizeof(baudrate));
    GetEditText(hwnd, IDC_PORTPARAMS_ICDELAY, icdelay, sizeof(icdelay));
    GetEditText(hwnd, IDC_PORTPARAMS_OPTIONALS, optionals, sizeof(optionals));
}

static BOOL CALLBACK
PortParamsProc(hwnd, message, wParam, lParam)
HWND hwnd;
UINT message;
WPARAM wParam;
LPARAM lParam;
{
	POINT *p;
	int cmd;

	if (message == WM_MOUSEMOVE)
		return TRUE;

	switch(message) {
	    
	case WM_INITDIALOG:
		p = DialogPos(Gwnd, hwnd);
		SendMessage(hwnd, DM_SETDEFID, (WPARAM) IDNONE, (LPARAM)0 );
		SendMessage(GetDlgItem(hwnd, IDOK), WM_SETFOCUS, 0, (LPARAM)0);
		SetWindowPos(hwnd, 0, (int)p->x, (int)p->y, 0, 0,
			SWP_NOSIZE | SWP_NOZORDER | SWP_SHOWWINDOW);
		setControls(hwnd);
		return TRUE;

		case WM_COMMAND:
			if ((cmd = checkbuttons(hwnd, wParam)) >= 0) {
				switch (cmd) {
				case IDOK:
					getControls(hwnd);
					EndDialog(hwnd, (WPARAM)TRUE);
					return TRUE;

				case IDCANCEL:
					EndDialog(hwnd, (WPARAM)FALSE);
					return TRUE;

				default:
					break;
				}
			}
			break;

        default:
			break;
		}
		return FALSE;
}

       
char *
GetEditText(hwnd, ItemId, buf, maxlen)
HWND hwnd;
int ItemId;
char *buf;
int maxlen;
{
    int i;
    char tbuf[80], *cp;
    
    i = SendDlgItemMessage(hwnd, ItemId, (UINT)WM_GETTEXTLENGTH, (WPARAM) 0, (LPARAM) 0);
    if (i <=0 || i >= sizeof(tbuf)-1)
        return NULL;
    else if (i > 0) {
        SendDlgItemMessage(hwnd, ItemId, (UINT)WM_GETTEXT, (WPARAM) i+1, (LPARAM) tbuf);
        tbuf[i+1] = 0; 
        cp = stripLeadingSpace(tbuf);
        if (!buf) {
            stripTrailingSpace(cp);
            buf = saveString(cp);
        }
        else {
            strncpy(buf, cp, MIN(sizeof(tbuf), maxlen));
            stripTrailingSpace(buf);
        }
    }
    return buf;
}

void
PortParams()
{
	int r;
	extern BOOL TTYProc(tty_t *);

	ttyp = &rigInfo.tty;
	ttytocontrols();

	r = DialogBox(NULL, MAKEINTRESOURCE(IDD_PORTPARAMS), Gwnd, PortParamsProc);

#ifdef _DEBUG_
	diag("End Dialog\n");
#endif /* _DEBUG_ */

	if (r)
		controlstotty();
}
