//
// This file contains proprietary information of Jesse Buckwalter.
// Copying or reproduction without prior written approval is prohibited.
//
// Copyright (c) 1993, 1994, 1995
// Jesse Buckwalter
// 525 Third Street
// Annapolis, MD 21403
// (410) 263-8652
//

#ifndef  __TIMER_H
#include "timer.h"
#endif

#ifndef  __DOS_H
#include <dos.h>
#endif

#ifndef  __CONIO_H
#include <conio.h>
#endif

#ifndef  __STDIO_H
#include <stdio.h>
#endif

#ifndef  __TIME_H
#include <time.h>
#endif

#ifndef  __CALLUP_H
#include "callup.h"
#endif

int            Timer::INT1C = 0x1c;               // DOS tick timer vector
float          Timer::CLKTCK = 18.2;              // clock ticks per second
CastIsr        Timer::oldTickISR = 0;             // pointer to tick timer ISR
Window         Timer::defaultWin( 1, 1, 80, 25, RED, YELLOW );
unsigned long  Timer::elapsedSecs;                // elapsed time in seconds
unsigned long  Timer::elapsedTicks;               // elapsed time in ticks
unsigned long  Timer::startSecs;                  // start time in seconds
unsigned       Timer::showTimerActivity = 0;      // show timer messages
unsigned       Timer::showLoopSpeed = 0;          // show freq of cycle() calls
unsigned       Timer::timerCount = 0;             // count of Timers
Timer*         Timer::timerChainHead = 0;         // head of Timer chain
Window*        Timer::win = &Timer::defaultWin;   // window output object

void interrupt Timer::newTickISR( ... )
// --------------------------------------------------------------------------
//  Tick timer interrupt service routine
// --------------------------------------------------------------------------
{
   enable();
   Timer::elapsedTicks++;
   Timer::oldTickISR();
}

Timer::Timer( const char* name ) : name ( name ), duration( 0 ),
                                   state( Timer::Stopped ),
                                   cu( 0 )
// --------------------------------------------------------------------------
//  Constructor
// --------------------------------------------------------------------------
{
   if (!oldTickISR)                              // if we haven't hooked up
      installISR();                              // hook in our ISR
   timerCount++;                                 // count the timers
}

Timer::Timer( const char*         name,
                    CallUp&       cu,
              const unsigned long ms )
            :
              name( name ), cu( &cu ), duration( ms ),
              state( Timer::Stopped )
// --------------------------------------------------------------------------
//  Constructor
// --------------------------------------------------------------------------
{
   if (!oldTickISR)                              // if we haven't hooked up
      installISR();                              // hook in our ISR
   timerCount++;                                 // count the timers
}

Timer::~Timer()
// --------------------------------------------------------------------------
//  Destructor
// --------------------------------------------------------------------------
{
   stop();                                       // stop the timer
   timerCount--;                                 // reduce the count
   if (!timerCount)                              // if nobody is left
      removeISR();                               // unhook our ISR
}

void Timer::cycle()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   if (elapsedTicks >= 182)                      // correct the elapsed time
   {                                             // base using time() every
      elapsedSecs = ::time( 0 ) - startSecs;     // 10 seconds (182 ticks);
      elapsedTicks = 0;                          // reset the tick counter
   }

   unsigned long currentTimeMs = msTime();
   while (timerChainHead && timerChainHead->stopTimeMs <= currentTimeMs)
   {
      if (showTimerActivity)
      {
         sprintf( win->buf, "%s timer expired", timerChainHead->name );
         win->print( win->buf );
      }
      Timer *tp = timerChainHead;                // keep a pointer to timer
      timerChainHead = timerChainHead->next;     // remove timer from chain
      tp->state = Timer::Expired;                // change the timer state
      if (tp->cu)                                // if provided
         tp->cu->func();                         // call the callup function
   }

   static unsigned loopCnt = 0;
   static unsigned long oldTimeMs = currentTimeMs;
   loopCnt++;                                    // increment loop count
   if (loopCnt >= 10000)                         // every 10000 loops, calc
   {                                             // number of loops per second
      if (showLoopSpeed)
      {
         sprintf( win->buf, "Average loop : %.2f ms",
                            (currentTimeMs - oldTimeMs) / 10000.0 );
         win->print( win->buf );
      }
      oldTimeMs = currentTimeMs;
      loopCnt = 0;
   }
}

unsigned long Timer::dosTime()
// --------------------------------------------------------------------------
//  Return the DOS time, updated every ten seconds
// --------------------------------------------------------------------------
{
   return elapsedTicks / CLKTCK + elapsedSecs + startSecs;
}

void Timer::installISR()
// --------------------------------------------------------------------------
//  Install our tick timer interrupt service routine.
// --------------------------------------------------------------------------
{
   if (oldTickISR)                               // don't do it twice
      return;
   oldTickISR = getvect( INT1C );                // remember the old ISR
   disable();                                    // disable interrupts
   setvect( INT1C, newTickISR );                 // hook in our ISR
   enable();                                     // enable interrutps
   startSecs = ::time( 0 );                      // rember when we started
   elapsedSecs = 0;                              // zero the counters
   elapsedTicks = 0;
}

unsigned long Timer::msLeft() const
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   if (state == Started)
      return stopTimeMs - msTime();
   else
      return 0;
}

unsigned long Timer::msTime()
// --------------------------------------------------------------------------
//  Return the elapsed clock time in milliseconds
// --------------------------------------------------------------------------
{
   return 1000*( elapsedTicks / CLKTCK + elapsedSecs );
}

void Timer::removeISR()
// --------------------------------------------------------------------------
//  Remove our tick timer interrupt service routine.  This function MUST
//  be called before exiting a program.  Otherwise the system will crash.
// --------------------------------------------------------------------------
{
   if (!oldTickISR)
      return;
   disable();
   setvect( INT1C, oldTickISR );
   enable();
   oldTickISR = 0;
}

void Timer::set( CallUp& cu, const unsigned long ms )
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   this->cu = &cu;
   duration = ms;

   // If we should decide that we need to ensure that the timer is stopped
   // at this point, USE stop() to do it!  DON'T change the state to
   // Timer::Stopped because if the timer is running, changing the state
   // will not take the timer out of the timer chain.  Leaving it in the
   // chain with a state other than Started will cause a loop that will lock
   // up the CPU.
}

void Timer::start()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   if (state == Timer::Started)                  // if already running, stop
      stop();                                    // it because it may need to
                                                 // go to a different position
                                                 // in the timer chain
   if (showTimerActivity)
   {
      sprintf( win->buf, "%s timer started, %lu ms", name, duration );
      win->print( win->buf );
   }
   if (duration)                                 // ignore zero durations
   {
      stopTimeMs = msTime() + duration;          // set the stop time
      state = Timer::Started;                    // set the state
      if (timerChainHead == 0)                   // if chain is empty
      {
         timerChainHead = this;                  // point head to us
         next = 0;                               // indicate the chain end
      }
      else
      {
         // Insert the timer in the chain after all timers that will expire
         // sooner

         Timer* tp1;
         Timer* tp2;
         tp1 = tp2 = timerChainHead;
         while (tp2->next != 0 && stopTimeMs > tp2->stopTimeMs)
         {
            tp1 = tp2;
            tp2 = tp2->next;
         }
         if (stopTimeMs > tp2->stopTimeMs)             // if at tail of chain
         {
            tp2->next = this;
            next = 0;
         }
         else
            if (tp2 == timerChainHead)                 // if at head of chain
            {
               next = timerChainHead;
               timerChainHead = this;
            }
            else                                       // if in middle of chain
            {
               tp1->next = this;
               next = tp2;
            }
      }
   }
}

void Timer::start( const unsigned long ms )
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   duration = ms;
   start();
}

void Timer::stop()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   if (showTimerActivity)
   {
      sprintf( win->buf, "%s timer stopped", name );
      win->print( win->buf );
   }
   state = Timer::Stopped;                       // set timer state
   if (timerChainHead)                           // if not an empty chain
      if (this == timerChainHead)                // if we're at the head
         timerChainHead = next;                  // make the next one the head
      else
      {
         Timer *tp1, *tp2;                       // if we're not at the head
         tp1 = tp2 = timerChainHead;             // locate the previous timer
         while (tp2 != this && tp2->next != 0)
         {
            tp1 = tp2;
            tp2 = tp2->next;
         }
         if (tp2 == this)                        // if we're in chain
            tp1->next = next;                    // remove us from chain
         // else                                 // we're not in the chain
      }
}

