//
// This file contains proprietary information of Jesse Buckwalter.
// Copying or reproduction without prior written approval is prohibited.
//
// Copyright (c) 1993, 1994, 1995
// Jesse Buckwalter
// 525 Third Street
// Annapolis, MD 21403
// (410) 263-8652
//

// Link Access Procedures Balanced (LAPB), the upper sublayer of
// AX.25 Level 2

// These functions were developed from Phil Karn's (KA9Q) NOS source code
// dated 920621.

#ifndef  __LAPB_H
#include "lapb.h"
#endif

#ifndef  __MEM_H
#include "mem.h"
#endif

#ifndef  __AX25OBJ_H
#include "ax25obj.h"
#endif

#ifndef  __AX25USER_H
#include "ax25User.h"
#endif

#ifndef  __CALLUP_H
#include "callup.h"
#endif

#ifndef  __CONFIG_H
#include "config.h"
#endif

#ifndef  __LOG_H
#include "log.h"
#endif

#ifndef  __MSGBUF_H
#include "msgbuf.h"
#endif

#ifndef  __PKTQUEUE_H
#include "pktqueue.h"
#endif

#ifndef  __PROTOCOL_H
#include "protocol.h"
#endif

#ifndef  __TIMER_H
#include "timer.h"
#endif

LAPB::LAPB( Config& cfg, Window& win, Log& log )
          :
          cfg( cfg ), win( win ), log( log ),
          axp( 0 ), linkTbl( 0 ),
          t1CallUp( *this ), t3CallUp( *this )
// --------------------------------------------------------------------------
//  Constructor
// --------------------------------------------------------------------------
{
   n2 = cfg.n2;                        // retry limit
   maxframe = cfg.maxFrame;            // transmit flow control level, frames
   mdev = 0;                           // mean rtt deviation, ms
   paclen = cfg.axPacLen;              // maximum outbound packet size, bytes
   proto = cfg.axVersion;              // protocol version
   pthresh = cfg.pThresh;              // poll threshold, bytes
   reason = 0;                         // reason for connection closing
   response = 0;                       // response owed to other end
   rejsent = 0;                        // REJ frame has been sent
   remotebusy = 0;                     // remote sent RNR
   retrans = 0;                        // a retransmission has occurred
   retries = 0;                        // retry counter
   rtt_run = 0;                        // round trip "timer" is running
   rtt_time = 0;                       // stored clock values for RTT, ticks
   rtt_seq = 0;                        // sequence number being timed
   rxasm = 0;                          // receive reassembly buffer
   rxq = new PktQueue;                 // receive queue
   segremain = 0;                      // segmenter state
   srt = cfg.axIRTT;                   // smoothed round-trip time, ms
   state = LAPB::DISCONNECTED;         // link state
   txq = new PktQueue;                 // transmit queue
   unack = 0;                          // number of unacked frames
   vr = 0;                             // our receive state variable
   vs = 0;                             // our send state variable
   window = cfg.axWindow;              // local flow control limit, bytes

   framesToQueue = maxframe - unack;
   t1 = new Timer( "T1", t1CallUp, 2 * cfg.axIRTT );
   t3 = new Timer( "T3", t3CallUp, 2 * cfg.t3Init );
}

LAPB::~LAPB()
// --------------------------------------------------------------------------
//  Destructor
// --------------------------------------------------------------------------
{
   delete t1;
   delete t3;
   delete rxasm;
}

int LAPB::ackOurs( unsigned n )
// --------------------------------------------------------------------------
//  Handle incoming acknowledgements for frames we've sent.  Free frames
//  being acknowledged.  Return -1 to cause a frame reject if number is bad,
//  0 otherwise
// --------------------------------------------------------------------------
{
   // Free up acknowledged frames by purging frames from the I-frame
   // transmit queue. Start at the remote end's last reported V(r)
   // and keep going until we reach the new sequence number.  If we try to
   // free a null pointer, then we have a frame reject condition.

   unsigned oldest = (vs - unack) & MMASK; // Seq number of oldest
                                                 // unacked I-frame
   int acked = 0;                                // count of frames acked by this ACK
   while (unack != 0 && oldest != n)
   {
      MBuf* bp;
      if ((bp = txq->dequeue()) == 0)
         return -1;                              // acking unsent frame
      delete bp;
      unack--;
      acked++;
      if (rtt_run && rtt_seq == oldest)
      {
         // A frame being timed has been acked

         rtt_run = 0;

         // Update only if frame wasn't retransmitted

         if (!retrans)
         {
            unsigned long rtt = t1->msTime() - rtt_time;
            unsigned long abserr = (rtt > srt) ? rtt - srt : srt - rtt;

            // Run SRT and mdev integrators

            srt = (srt*7 + rtt + 4) >> 3;
            mdev = (mdev*3 + abserr + 2) >> 2;
            unsigned long newInterval = 4*mdev + srt;

            // Update timeout.  This has been temporarily coded to adapt
            // to a shorter interval but not a longer one.

            if ( newInterval < 2 * cfg.axIRTT )
               t1->set( newInterval );
            else
               if ( newInterval >= 2 * cfg.axIRTT )
                  t1->set( 2 * cfg.axIRTT );
         }
      }
      retrans = 0;
      retries = 0;
      oldest = (oldest + 1) & MMASK;
   }
   if (unack == 0)
   {
      // All frames acked, stop timeout

      t1->stop();
      t3->stop();
   }
   else
      if (acked != 0)
         t1->start();                            // partial ACK; restart timer

   framesToQueue = maxframe - unack;
   return 0;
}

unsigned LAPB::bytesToSend()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   return paclen * framesToQueue;
}

void LAPB::clrEx()
// --------------------------------------------------------------------------
// Clear exception conditions
// --------------------------------------------------------------------------
{
   remotebusy = 0;
   rejsent = 0;
   response = 0;
   t3->stop();
}

void LAPB::connect()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   switch( state )
   {
      case DISCONNECTED:
         estLink();
         setState( SETUP );
         break;
      case SETUP:
         txq->free();
         break;
      case DISCPENDING:                          // ignore
         break;
      case RECOVERY:
      case CONNECTED:
         txq->free();
         estLink();
         setState( SETUP );
         break;
   }
}

void LAPB::disc()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   switch (state)
   {
      case DISCONNECTED:
      case LISTEN:
         break;                                  // ignored
      case SETUP:
         reason = LB_NORMAL;                     // note: fall through
      case DISCPENDING:
         setState( DISCONNECTED );
         break;
      case CONNECTED:
      case RECOVERY:
         txq->free();
         retries = 0;
         sendCtl( COMMAND, DISC | PF );
         t3->stop();
         t1->start();
         setState( DISCPENDING );
         break;
   }
}

void LAPB::enqResp()
// --------------------------------------------------------------------------
// Enquiry response
// --------------------------------------------------------------------------
{
   char ctl = rxq->len() >= window ? RNR | PF : RR | PF;
   sendCtl( RESPONSE, ctl );
   response = 0;
   t3->stop();
}

void LAPB::enqueue( MBuf& mb )
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   txq->enqueue( &mb );
}

void LAPB::estLink()
// --------------------------------------------------------------------------
// Establish data link
// --------------------------------------------------------------------------
{
   clrEx();
   retries = 0;
   sendCtl( COMMAND, SABM | PF);
   t3->stop();
   t1->start();
}

void LAPB::handleIt( int pid, MBuf& mb )
// --------------------------------------------------------------------------
// handleIt
// --------------------------------------------------------------------------
{
   if (!linkTbl)
      return;
   L3Protocol* ip = (L3Protocol*) &linkTbl->lookup( pid );
   if (ip)
   {
      Level3Iface* l3If = new Level3Iface( axp, axp->remote, axp->local,
                                           &mb, 0 );
      ip->recvIn( *l3If );                       // kick it up
      delBuf = 0;
   }
}

int LAPB::input( int   cmdrsp,                   // command/response flag
                 MBuf& mb )                      // rest of frame, starting with ctl
// --------------------------------------------------------------------------
//  Process incoming frames
// --------------------------------------------------------------------------
{
   if (!&mb || !axp)
   {
      delete &mb;
      return -1;
   }

   int control;
   if ((control = mb.pullChar()) == -1)          // extract the various parts of the
   {                                             // control field for easy use
      delete &mb;                                // probably not necessary
      return -1;
   }
   delBuf = 1;
   unsigned type = Ax25::ftype( control );       // specific type (I/RR/RNR/etc) of
                                                 // frame
   int fClass = type & 0x3;                      // general class (I/S/U) of frame

   char poll = 0;
   char final = 0;
   char pf = control & PF;                       // extracted poll/final bit
   if (pf)                                       // check for polls and finals
      switch (cmdrsp)
      {
         case COMMAND:
            poll = 1;
            break;
         case RESPONSE:
            final = 1;
            break;
      }
   
   unsigned nr;                                  // ACK number of incoming frame
   unsigned ns;                                  // seq number of incoming frame
   switch (fClass)                               // extract sequence nums, if present
   {
      case I:
      case I+2:
         ns = (control >> 1) & MMASK;
      case S:                                    // note fall-thru
         nr = (control >> 5) & MMASK;
         break;
   }

   // This section follows the SDL diagrams by K3NA fairly closely

   unsigned tmp;
   switch (state)
   {
      case DISCONNECTED:
         switch (type)
         {
            case SABM:                           // init or reset link
               sendCtl( RESPONSE, UA|pf );       // always accept
               clrEx();
               unack = vr = vs = 0;
               setState( CONNECTED );               // resets state counters
               srt = cfg.axIRTT;
               mdev = 0;
               t1->set( 2 * cfg.axIRTT );
               t3->start();
               break;
            case DM:                             // ignore to avoid
               break;                            // infinite loops
            default:                             // all others get DM
               if (poll)
                  sendCtl( RESPONSE, DM | pf );
               break;
         }
         break;
      case SETUP:
         switch (type)
         {
            case SABM:                           // simultaneous open
               sendCtl( RESPONSE, UA | pf );
               break;
            case DISC:
               sendCtl( RESPONSE, DM | pf );
               break;
            case UA:                             // connection accepted
               t1->stop();                       // Note: xmit Q not cleared
               t3->start();
               unack = vr = vs = 0;
               setState( CONNECTED );
               break;
            case DM:                             // connection refused
               txq->free();                      // purge xmit queue
               t1->stop();
               reason = LB_DM;
               setState( DISCONNECTED );
               break;
            default:                             // all other frames
               break;                            // ignored
         }
         break;
      case DISCPENDING:
         switch (type)
         {
            case SABM:
               sendCtl( RESPONSE, DM | pf );
               break;
            case DISC:
               sendCtl( RESPONSE, UA | pf );
               break;
            case UA:
            case DM:
               t1->stop();
               setState( DISCONNECTED );
               break;
            default:                             // respond with DM only
               if (poll)                         // to command polls
                  sendCtl( RESPONSE, DM | pf );
               break;
         }
         break;
      case CONNECTED:
         switch (type)
         {
            case SABM:
               sendCtl( RESPONSE, UA | pf );
               clrEx();
               txq->free();                      // purge xmit queue
               t1->stop();
               t3->start();
               unack = vr = vs = 0;
               setState( CONNECTED );            // purge queues
               break;
            case DISC:
               txq->free();                      // purge xmit queue
               sendCtl( RESPONSE, UA | pf );
               t1->stop();
               t3->stop();
               reason = LB_NORMAL;
               setState( DISCONNECTED );
               break;
            case DM:
               reason = LB_DM;
               setState( DISCONNECTED );
               break;
            case UA:
               estLink();
               setState( SETUP );                // re-establish link
               break;
            case FRMR:
               estLink();
               setState( SETUP );                // re-establish link
               break;
            case RR:
            case RNR:
               remotebusy = (control == RNR) ? 1 : 0;
               if (poll)
                  enqResp();
               ackOurs( nr );
               if (framesToQueue && axp->tCallUp)
                  axp->tCallUp->func();
               break;
            case REJ:
               remotebusy = 0;
               if (poll)
                  enqResp();
               ackOurs( nr );
               t1->stop();
               t3->start();

               // This may or may not actually invoke transmission,
               // depending on whether this REJ was caused by
               // our losing his prior ACK.

               invRex();
               if (framesToQueue && axp->tCallUp)     // do after retransmission
                  axp->tCallUp->func();
               break;
            case I:
               ackOurs( nr);                     // * == -1)
               if (framesToQueue && axp->tCallUp)
                  axp->tCallUp->func();
               if (rxq->len() >= window)
               {
                  // Too bad he didn't listen to us; he'll
                  // have to resend the frame later. This
                  // drastic action is necessary to avoid
                  // deadlock.

                  if (poll)
                     sendCtl( RESPONSE, RNR | pf );
                  break;
               }

               // Reject or ignore I-frames with receive sequence number
               // errors

               if (ns != vr)
               {
                  if (proto == Ax25::V1 || !rejsent)
                  {
                     rejsent = 1;
                     sendCtl( RESPONSE, REJ | pf );
                  }
                  else
                     if (poll)
                        enqResp();
                  response = 0;
                  break;
               }
               rejsent = 0;
               vr = (vr+1) & MMASK;
               tmp = rxq->len() >= window ? RNR : RR;
               if (poll)
                  sendCtl( RESPONSE, tmp | PF );
               else
                  response = tmp;
               procData( mb );
               break;
            default:                             // all others ignored
               break;
         }
         break;
      case RECOVERY:
         switch (type)
         {
            case SABM:
               sendCtl( RESPONSE, UA | pf );
               clrEx();
               t1->stop();
               t3->start();
               unack = vr = vs = 0;
               setState( CONNECTED );            // purge queues
               break;
            case DISC:
               txq->free();                      // purge xmit queue
               sendCtl( RESPONSE, UA | pf );
               t1->stop();
               t3->stop();
               response = UA;
               reason = LB_NORMAL;
               setState( DISCONNECTED );
               break;
            case DM:
               reason = LB_DM;
               setState( DISCONNECTED );
               break;
            case UA:
               estLink();
               setState( SETUP );                // re-establish
               break;
            case FRMR:
               estLink();
               setState( SETUP );                // re-establish link
               break;
            case RR:
            case RNR:
               remotebusy = (control == RNR) ? 1 : 0;
               if (proto == Ax25::V1 || final)
               {
                  t1->stop();
                  ackOurs( nr );
                  if (unack != 0)
                     invRex();
                  else
                  {
                     t3->start();
                     setState( CONNECTED );
                  }
                  if (framesToQueue && axp->tCallUp)  // do after retransmission
                     axp->tCallUp->func();
               }
               else
               {
                  if (poll)
                     enqResp();
                  ackOurs( nr );
                  if (framesToQueue && axp->tCallUp)
                     axp->tCallUp->func();

                  // Keep timer running even if all frames
                  // were acked, since we must see a Final

                  if (t1->getState() != Timer::Started)
                     t1->start();
               }
               break;
            case REJ:
               remotebusy = 0;

               // Don't insist on a Final response from the old proto

               if (proto == Ax25::V1 || final)
               {
                  t1->stop();
                  ackOurs( nr );
                  if (unack != 0)
                     invRex();
                  else
                  {
                     t3->start();
                     setState( CONNECTED );
                  }
                  if (framesToQueue && axp->tCallUp)  // do after retransmission
                     axp->tCallUp->func();
               }
               else
               {
                  if (poll)
                     enqResp();
                  ackOurs( nr );
                  if (unack != 0)                // this is certain to trigger
                     invRex();                   //  output
                  if (framesToQueue && axp->tCallUp)  // do after retransmission
                     axp->tCallUp->func();

                  // A REJ that acks everything but doesn't
                  // have the F bit set can cause a deadlock.
                  // So make sure the timer is running.

                  if (t1->getState() != Timer::Started)
                     t1->start();
               }
               break;
            case I:
               ackOurs( nr );                    // * == -1)
               if (framesToQueue && axp->tCallUp)
                  axp->tCallUp->func();

               // Make sure timer is running, since an I frame
               // cannot satisfy a poll

               if (t1->getState() != Timer::Started)
                  t1->start();
               if (rxq->len() >= window)
               {
                  // Too bad he didn't listen to us; he'll
                  // have to resend the frame later. This
                  // drastic action is necessary to avoid
                  // memory deadlock.

                  sendCtl( RESPONSE, RNR | pf );
                  break;
               }

               // Reject or ignore I-frames with receive sequence number
               // errors

               if (ns != vr)
               {
                  if (proto == Ax25::V1 || !rejsent)
                  {
                     rejsent = 1;
                     sendCtl( RESPONSE, REJ | pf );
                  }
                  else
                     if (poll)
                        enqResp();

                  response = 0;
                  break;
               }
               rejsent = 0;
               vr = (vr+1) & MMASK;
               tmp = rxq->len() >= window ? RNR : RR;
               if (poll)
               {
                  sendCtl( RESPONSE, tmp | PF );
               }
               else
                  response = tmp;
               procData( mb );
               break;
            default:
               break;                            // ignored
         }
         break;
   }
   if (delBuf)                                   // if not processed upstairs
      delete &mb;

   // See if we can send some data, perhaps piggybacking an ack.
   // If successful, output() will clear response.

   output();
   if (response != 0)
   {
      sendCtl( RESPONSE, response);
      response = 0;
   }
   return 0;
}

void LAPB::invRex()
// --------------------------------------------------------------------------
// Invoke retransmission
// --------------------------------------------------------------------------
{
   vs -= unack;
   vs &= MMASK;
   unack = 0;
}

void LAPB::link( Ax25& ax, LinkTable& linkTbl )
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   axp = &ax;
   this->linkTbl = &linkTbl;
}

int LAPB::output()
// --------------------------------------------------------------------------
// Start data transmission on link, if possible.  Return number of frames
// sent.
// --------------------------------------------------------------------------
{
   if (!axp || (state != RECOVERY && state != CONNECTED) || remotebusy)
      return 0;

   // Dig into the send queue for the first unsent frame. (Skip the
   // unacknowledged frames at the head of the queue.)

                                                 // start iteration sequence
   MBuf* bp = txq->initIterator( unack );

   // Start at first unsent I-frame, stop when either the
   // number of unacknowledged frames reaches the maxframe limit,
   // or when there are no more frames to send

   int sent = 0;
   while (bp != 0 && unack < maxframe)
   {
      char control = I | (vs++ << 1) | (vr << 5);
      vs &= MMASK;
      MBuf* tbp = new MBuf( *bp );
      if (!tbp)
         return sent;                            // probably out of memory
      axp->sendFrame( COMMAND, control, *tbp );
      unack++;

      // We're implicitly acking any data he's sent, so stop any
      // delayed ack

      response = 0;
      if (t1->getState() != Timer::Started)
      {
         t3->stop();
         t1->start();
      }
      sent++;
      bp = txq->next();                          // next in iteration sequence
      if (!rtt_run)
      {
         // Start round trip timer

         rtt_seq = (control >> 1) & MMASK;
         rtt_time = t1->msTime();
         rtt_run = 1;
      }
   }
   return sent;
}

void LAPB::procData( MBuf& mb )
// --------------------------------------------------------------------------
// Process a valid incoming I frame
// --------------------------------------------------------------------------
{
   // Extract level 3 PID

   int pid;
   if ((pid = mb.pullChar()) == -1)
      return;                                    // no PID

   int seq;
   if (segremain)
   {
      seq = mb.pullChar();                       // reassembly in progress; continue
      if (pid == Ax25::PID_SEGMENT && (seq & SEG_REM) == segremain - 1)
      {
         rxasm->append( mb );                    // correct, in-order segment
         if ((segremain = (seq & SEG_REM)) == 0)
         {
            mb = *rxasm;                         // done; kick it upstairs
            rxasm = 0;
            pid = mb.pullChar();
            handleIt( pid, mb );
         }
      }
      else
      {
         // Error!
         delete rxasm;
         rxasm = 0;
         segremain = 0;
      }
   }
   else                                          // No reassembly in progress
      if (pid == Ax25::PID_SEGMENT)
      {
         seq = mb.pullChar();                    // Start reassembly
         if (!(seq & SEG_FIRST))
            ;                                    // not first seg - error!
         else
         {
            segremain = seq & SEG_REM;// Put first segment on list
            rxasm = &mb;
         }
      }
      else
         handleIt( pid, mb );                    // Normal frame; send upstairs
}

void LAPB::sCall() const
// --------------------------------------------------------------------------
// LAPB status change function
// --------------------------------------------------------------------------
{
   static char* lapbReason[] = { "LAPB state: disconnected, normal close",
                               "LAPB state: disconnected, connection refused",
                               "LAPB state: disconnected, excessive retries" };
   if (cfg.debug)
   {
      const char *lapbState[7] = {  " -- error --   ",
                                    "Disconnected   ",
                                    "Listen         ",
                                    "Setup          ",
                                    "Disc-Pending   ",
                                    "Connected      ",
                                    "Recovery       "};
      sprintf( win.buf, "[LAPB_OLD : %s] [LAPB_NEW : %s]",
               lapbState[ oldState ], lapbState[ state ] );
      log.write( win.buf );
      win.print( win.buf );
   }

   char* cp;
   switch (oldState)
   {
      case DISCONNECTED:
         switch (state)
         {
            case SETUP:
               win.print( "LAPB state: setup" );
               break;
            case CONNECTED:
               win.print( "LAPB state: connected" ); // display on monitor
               log.write( "LAPB state: connected"  ); // add to log file
               break;
            default:
               win.print( "LAPB state: unexpected state change" ); // display on monitor
               log.write( "LAPB state: unexpected state change"  ); // add to log file
         }
         break;
      case SETUP:
         switch (state)
         {
            case CONNECTED:
               win.print( "LAPB state: connected" ); // display on monitor
               log.write( "LAPB state: connected" ); // add to log file
               break;
            case DISCONNECTED:
               cp = lapbReason[ reason ];
               win.print( cp );                  // display on monitor
               log.write( cp );                  // add to log file
               break;
            default:
               win.print( "LAPB state: unexpected state change" ); // display on monitor
               log.write( "LAPB state: unexpected state change" );  // add to log file
         }
         break;
      case DISCPENDING:
         switch (state)
         {
            case DISCONNECTED:
               cp = lapbReason[ reason ];
               win.print( cp );                  // display on monitor
               log.write( cp );                  // add to log file
               break;
            default:
               cp = "LAPB state: unexpected state change";
               win.print( cp );                  // display on monitor
               log.write( cp );                  // add to log file
         }
         break;
      case RECOVERY:
      case CONNECTED:
         switch (state)
         {
            case SETUP:
               win.print( "LAPB state: setup" );
               break;
            case CONNECTED:
               win.print( "LAPB state: connected" ); // display on monitor
               break;
            case DISCONNECTED:
               cp = lapbReason[ reason ];
               win.print( cp );                  // display on monitor
               log.write( cp );                  // add to log file
               break;
            case RECOVERY:
               win.print( "LAPB state: recovery" );
               break;
            case DISCPENDING:
               win.print( "LAPB state: disconnect pending" );
         }
   }
}

int LAPB::sendCtl( int cmdrsp, int cmd )
// --------------------------------------------------------------------------
// Send S or U frame to currently connected station
// --------------------------------------------------------------------------
{
   if (!axp)
      return 0;
   if ((Ax25::ftype( (char)cmd ) & 0x3) == S)    // insert V(R) if S frame
      cmd |= (vr << 5);
   MBuf* bp = new MBuf( 16 );                    // allocate a message buffer
   return axp->sendFrame( cmdrsp, cmd, *bp );
}

void LAPB::setState( int s )
// --------------------------------------------------------------------------
// Set new link state
// --------------------------------------------------------------------------
{
   oldState = state;
   state = s;
   if (s == DISCONNECTED)
   {
      t1->stop();
      t3->stop();
      txq->free();
   }

   // Don't bother the client unless the state is really changing

   if (oldState != s)
   {
      sCall();
      if (axp->sCallUp)
         axp->sCallUp->func();
   }
}

/*
MBuf& LAPB::segmenter( MBuf&     mb,             // Complete packet
                       unsigned  ssize)          // Max size of frame segments
// --------------------------------------------------------------------------
// New-style frame segmenter.
//    Returns queue of segmented fragments, or original packet if small
//    enough.
// --------------------------------------------------------------------------
{
   // See if packet is too small to segment. Note 1-byte grace factor
   // so the PID will not cause segmentation of a 256-byte IP datagram.

   unsigned len = len_p( bp );
   if (len <= ssize + 1)
      return bp;                                 // too small to segment

   ssize -= 2;                                   // ssize now equal to data portion size
   int segments = 1 + (len - 1) / ssize;         // # segments
   unsigned offset = 0;
   MBuf* result = 0;
   MBuf* bptmp;
   while (segments != 0)
   {
      offset += dup_p( &bptmp, bp, offset, ssize);
      if (bptmp == 0)
      {
         free_q( &result );
         break;
      }
      bptmp = pushdown( bptmp, 2);               // make room for segmentation header
      bptmp->data[0] = Ax25::PID_SEGMENT;
      bptmp->data[1] = --segments;
      if (offset == ssize)
         bptmp->data[1] |= SEG_FIRST;
      enqueue( &result, bptmp );
   }
   delete &mb;
   return result;
}
*/
