//
// This file contains proprietary information of Jesse Buckwalter.
// Copying or reproduction without prior written approval is prohibited.
//
// Copyright (c) 1993, 1994, 1995
// Jesse Buckwalter
// 525 Third Street
// Annapolis, MD 21403
// (410) 263-8652
//

#ifndef  __DIRBCST_H
#include "dirbcst.h"
#endif

#ifndef  __STDLIB_H
#include <stdlib.h>
#endif

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __AX25OBJ_H
#include "ax25obj.h"
#endif

#ifndef  __CONFIG_H
#include "config.h"
#endif

#ifndef  __CALLUP_H
#include "callup.h"
#endif

#ifndef  __DIRECTRY_H
#include "directry.h"
#endif

#ifndef  __DIRFILE_H
#include "dirfile.h"
#endif

#ifndef  __MISC_H
#include "misc.h"
#endif

#ifndef  __MSGBUF_H
#include "msgbuf.h"
#endif

#ifndef  __MSGLINE_H
#include "msgline.h"
#endif

#ifndef  __PFHEADER_H
#include "pfheader.h"
#endif

#ifndef  __SATLINK_H
#include "satlink.h"
#endif

#ifndef  __SEMANTIC_H
#include "semantic.h"
#endif

#ifndef  __SLIST_H
#include "slist.h"
#endif

#ifndef  __SYNTAX_H
#include "syntax.h"
#endif

#ifndef  __TIMER_H
#include "timer.h"
#endif

#ifndef  __WINDOW_H
#include "window.h"
#endif

DirFrHdr* DirFrHdr::ZERO = 0;

DirBcstMgr::DirBcstMgr( char&              bcstAddr,
                        char&              myAddr,
                        Protocol&          logProto,
                        DirFileMgr&        dfm,
                        Directory&         dir,
                        SyntaxAnalyzer&    parser,      // select equation parser
                        SemanticsAnalyzer& sem,         // eqn semantics analyzer
                        StatusList&        sList,       // download file status list
                        Config&            cfg,
                        Window&            frWin,
                        Window&            iWin,
                        StatusLine&        sLine )
                      :
                        L3Protocol(),
                        bcstAddr( bcstAddr ),
                        myAddr( myAddr ),
                        logProto( logProto ),
                        dfm( dfm ),
                        dir( dir ),
                        parser( parser ),
                        sem( sem ),
                        sList( sList ),
                        frWin( frWin ),
                        iWin( iWin ),
                        sLine( sLine ),
                        axp( Ax25::ZERO ),
                        dirHoleCount( 0 ),
                        dirHoleListSize( 0 ),
                        eqnAuto( 0 ),
                        eqnNever( 0 ),
                        eqnPriority( 0 ),
                        upToDate( 0 )
// --------------------------------------------------------------------------
//  Constructor
// --------------------------------------------------------------------------
{
   autoSelect = cfg.autoSelect;
   bcstPacLen = cfg.bcstPaclen;
   debug = &cfg.debug;
   exitOnSatCallError = cfg.exitOnSatCallError;
   highlightBg = cfg.highlight1Bg;
   highlightFg = cfg.highlight1Fg;
   loggingStatus = &cfg.loggingStatus;
   logDirFrames = cfg.logDirFrames;
   logInFrames = &cfg.logInFrames;
   maxPfhSize = cfg.maxPfhSize;
   monitor = &cfg.monitor;
   mycall = cfg.mycall;
   pfhDir = cfg.workDir;
   showData = &cfg.showData;
   showDataHex = &cfg.showDataHex;
   simulate = cfg.simulate;
   trace = &cfg.trace;

   dirRqstWaitTimer = new Timer( "dirRqstWait", NULCALLUP, cfg.dirUpdateMs );
   if (!dirRqstWaitTimer)
      frWin.printFatal( "Out of memory" );
   dirData = (char*) (new char[ maxPfhSize ]);
   if (!dirData)
      frWin.printFatal( "Out of memory" );
   dHoles = new DPair[ DIR_LIST_SIZE_START ];
   if (!dHoles)
      frWin.printFatal( "Out of memory" );
   dirHoleListSize = DIR_LIST_SIZE_START;
   readHoles();

   for (int i = 0; i < parser.eqnCount; i++)    // get the pointers to the
   {                                             // auto, priority, and
      Equation* eqn = parser.eqnList[ i ];      // never equations from the
      switch (eqn->id)                           // syntax analyzer's list of
      {                                          // equations
         case LexicalAnalyzer::AUTO :
            eqnAuto = eqn;
            break;
         case LexicalAnalyzer::PRIORITY :
            eqnPriority = eqn;
            break;
         case LexicalAnalyzer::NEVER :
            eqnNever = eqn;
            break;
      }
   }
}

DirBcstMgr::~DirBcstMgr()
// --------------------------------------------------------------------------
//  Destructor
// --------------------------------------------------------------------------
{
   writeHoles();
   delete dirData;
   delete dirRqstWaitTimer;
   delete dHoles;
}

void DirBcstMgr::addHole( DPair* newp )
// -------------------------------------------------------------------------
//  Add a hole to the directory hole-list
// -------------------------------------------------------------------------
{
   if (dirHoleCount == dirHoleListSize)
   {
      if ( (long( dirHoleListSize ) + DIR_LIST_SIZE_INC) * sizeof( DPair )
            > 0xffff )
         frWin.printFatal( "dirAddHole : holeList exceeded 0xffff bytes" );
      unsigned newSize = dirHoleListSize + DIR_LIST_SIZE_INC; // inc list size
      DPair* newHoles = new DPair[ newSize ];
      if (newHoles)
      {
         memcpy( newHoles, dHoles, dirHoleListSize * sizeof( DPair ) );
         delete dHoles;
         dHoles = newHoles;
         dirHoleListSize = newSize;
      }
      else
      {
         delete dHoles;
         sprintf( frWin.buf, "dirAddHole : no memory to reallocate %lu bytes for hole list",
                  dirHoleListSize * sizeof( DPair ) );
         frWin.printFatal( frWin.buf );
      }
   }
   DPair a, b;
   memcpy( &a, newp, sizeof( a ) );              // make a copy so original not altered
   for (int i = 0; i < dirHoleCount; i++)      // insert the new record
      if (a.tOld < dHoles[ i ].tOld)
      {                                          // swap a and dHoles[i]
         memcpy( &b,           &dHoles[ i ], sizeof( a ) );
         memcpy( &dHoles[ i ], &a,           sizeof( a ) );
         memcpy( &a,           &b,           sizeof( a ) );
      }
   memcpy( &dHoles[ dirHoleCount ], &a, sizeof( a ) );
   dirHoleCount++;
}

void DirBcstMgr::deleteHole( int i )
// -------------------------------------------------------------------------
//  Delete a hole from directory hole-list
// -------------------------------------------------------------------------
{
   if (i >= dirHoleCount)
      frWin.printFatal( "dirDeleteHole : hole delete index exceeds list size" );
   if (dirHoleCount == 0)
      frWin.printFatal( "dirDeleteHole : attempting hole delete in empty list" );

   dirHoleCount--;
   for (; i < dirHoleCount; i++)
      memcpy(&dHoles[i], &dHoles[i+1], sizeof(dHoles[i]));
}


DirFrHdr& DirBcstMgr::decodeHdr( MBuf& mb )
// --------------------------------------------------------------------------
//  Decode directory frame header and remove it
// --------------------------------------------------------------------------
{

   if (mb.cnt < 11)                              // too short for dir frame
      return NOFRHDR;

   static DirFrHdr hdr;
   hdr.lastFrame = *mb.data & 0x20;              // flag bit 5
   hdr.newestFile = mb.pullChar() & 0x40;        // flag bit 6
   hdr.fileId = 0;
   int i;
   for (i = 0; i < 4; i++)                       // reverse the bytes
      hdr.fileId |= (unsigned long) mb.pullChar() << (8 * i);
   hdr.offset = 0;
   for (i = 0; i < 4; i++)                       // reverse the bytes
      hdr.offset |= (unsigned long) mb.pullChar() << (8 * i);
   hdr.tOld = 0;
   for (i = 0; i < 4; i++)                       // reverse the bytes
      hdr.tOld |= (unsigned long) mb.pullChar() << (8 * i);
   hdr.tNew = 0;
   for (i = 0; i < 4; i++)                       // reverse the bytes
      hdr.tNew |= (unsigned long) mb.pullChar() << (8 * i);
   return hdr;
}

unsigned DirBcstMgr::dirUpToDate( const unsigned long tNew )
// -------------------------------------------------------------------------
// If the directory contains only one hole and the parameter tNew is less
// than or equal to tOld of the hole, then the directory is up-to-date
// -------------------------------------------------------------------------
{
   return (dirHoleCount == 1 && tNew <= dHoles[ 0 ].tOld);
}

int DirBcstMgr::needEntry( DirFrHdr& hdr )
// -------------------------------------------------------------------------
//  Determine if the hole list indicates that the entry is needed.
//  The upToDate variable is updated only if the entry is
//  needed.  upToDate indicates that the entry will bring the
//  directory up-to-date with no outstanding entries.
// -------------------------------------------------------------------------
{
   DPair         new_h;
   unsigned long tOld, tNew;
   int           i;

   tOld = hdr.tOld;
   tNew = hdr.tNew;

   for (i = 0; i < dirHoleCount; i++)          // get to the right point in the list.
      if (tOld <= dHoles[i].tNew)
         break;
   if (i == dirHoleCount)                      // tOld > tNew limit of last hole
      return 0;                                  // should never get here

   if (tOld <= dHoles[i].tOld)                 // tOld <= lower limit
   {
      if (tNew < dHoles[i].tOld)               // interval < hole
         return 0;
      else
      // (tNew >= dHoles[i].tOld)
         if (tNew >= dHoles[i].tNew)           // interval spans entire hole
         {
            deleteHole(i);
            return 1;
         }
         else
         // (tNew < dHoles[i].tNew)            // interval includes lower limit
         {
            dHoles[i].tOld = tNew + 1;
            return 1;
         }
   }
   else
   // (tOld > dHoles[i].tOld)                  // tOld > lower limit
   {
      if (tNew >= dHoles[i].tNew)              // interval includes upper limit
      {
         dHoles[i].tNew = tOld - 1;
         return 1;
      }
      else
      // (tNew < dHoles[i].tNew)               // data doesn't include lower or upper limit
      {
         new_h.tOld = tNew + 1;
         new_h.tNew = dHoles[i].tNew;
         dHoles[i].tNew = tOld - 1;            // modify current hole for lower hole
         addHole( &new_h );                    // add new hole for upper hole
         return 1;
      }
   }
}

void DirBcstMgr::readHoles()
// -------------------------------------------------------------------------
//  Read directory hole file
// -------------------------------------------------------------------------
{
   char      filename[MAXPATH];
   DPair     hole;
   FILE     *hole_file;
   char      in_str[MAXLINE];

   // If the directory hole file doesn't exist, create it

   sprintf( filename, "%sPFHDIR.HOL", pfhDir );
   if ((hole_file = fopen(filename, "rt")) == NULL)
   {
      if ((hole_file = fopen(filename, "w+t")) == NULL)
      {
         sprintf( frWin.buf, "Cannot create directory hole file <%s>\n",
                  filename );
         frWin.printFatal( frWin.buf );
      }
      fprintf(hole_file, "0 pfh header received\n");
      fprintf(hole_file, "0 pfh file length\n");
      fprintf(hole_file, "1 holes\n");
      fprintf(hole_file, "%lu, %lu\n", 0L, 0xffffffffL);
      fseek(hole_file, 0, SEEK_SET);
   }

   if (!fgets( in_str, MAXLINE, hole_file ) ||   // number of pfh
       !fgets( in_str, MAXLINE, hole_file ) ||   // file length
       !fgets( in_str, MAXLINE, hole_file )   )  // number of hole records
   {
      sprintf( frWin.buf, "Bad directory hole file <%s>\n", filename );
      frWin.printFatal( frWin.buf );
   }

   while (fgets(in_str, MAXLINE, hole_file) != NULL && in_str[0] != '\n')
   {
      sscanf(in_str, "%lu,%lu", &hole.tOld, &hole.tNew);
      addHole( &hole );
   }
   fclose (hole_file);
   sLine.dirStatus( dirHoleCount );
}

void DirBcstMgr::recvIn( Level3Iface& l3If )
//--------------------------------------------------------------------------
//--------------------------------------------------------------------------
{
   MBuf* bp = l3If.bp;                          // data buffer pointer
   if (!Ax25::addrEq( l3If.src, &bcstAddr ))
   {
      char call[ 11 ];
      Ax25::pAx25( call, l3If.src );
      sprintf( iWin.buf, "ERROR - Directories from %s heard but ignored.", call );
      iWin.print( iWin.buf );
      delete bp;
      delete &l3If;
      if (exitOnSatCallError)
      {
         static int code = SatLink::EXITONSATERROR;
         iWin.print( "Program exit requested." );
         satLink->exitProgram( code );
      }
      return;
   }
   if (memcmp( bp->data, "NO -2", 5 ) == 0)      // if permanent problem
      if (memcmp( bp->data + 6, mycall, strlen( mycall ) ) == 0) // if it's to us
            dirRqstWaitTimer->start();           // inhibit directory requests
                                                 // by starting the request
                                                 // "wait" timer;
                                                 // else ignore

   // If the flag is "OK" and directed to us, we don't inhibit the retry
   // at this point.  Retries are triggered after examining the
   // broadcast queue frame and seeing we aren't in the queue.

   if (memcmp( bp->data, "NO", 2 ) == 0 || memcmp( bp->data, "OK", 2 ) == 0)
   {
      if (!*trace && *monitor)
      {
         bp->data[ bp->cnt ] = 0;                // make it a string;
         frWin.print( bp->data );
         char* callPtr = strstr( bp->data, mycall );
         if (callPtr)
         {
            unsigned offset = (unsigned)(callPtr - bp->data);
            frWin.highlight( strlen( bp->data ), offset, strlen( mycall ), highlightFg, highlightBg );
         }
      }
   }
   else
   {
      if (!*trace && *monitor)
         if (*showDataHex)
            frWin.printHex( bp->data, bp->cnt ); // display data in hex
         else
            if (*showData)
               frWin.print( bp->data, bp->cnt ); // display data in non-hex
      saveFrame( *bp );                          // this does not "free" bp
      if (!simulate && *loggingStatus && !*logInFrames && logDirFrames)
         logProto.logRecvIn();
   }
   delete bp;
   delete &l3If;
}

void DirBcstMgr::rqstFill()
// --------------------------------------------------------------------------
//  Send a directory fill request
// --------------------------------------------------------------------------
{
   typedef struct dir_req_header
   {
      unsigned char flags;
      int           block_size;                  // according to specs
   } dir_req_header;

   dir_req_header*  drh;
   int              numpairs;
   MBuf*            bp;
   char*            cp;
   int              pid;

                                                 // should be at least 1
   numpairs = dirHoleCount < MAX_PAIRS ? dirHoleCount : MAX_PAIRS;
   bp = new MBuf( sizeof( dir_req_header ) + numpairs * sizeof( DPair ) + 17 );
   drh = (dir_req_header*) bp->data;            // point to the data
   drh->flags = 0x10;                           // CC bits = 00
   drh->block_size = bcstPacLen;
   cp = bp->data + sizeof(dir_req_header);      // set working pointer
   memcpy( cp, dHoles, numpairs * sizeof(DPair) );
   bp->cnt = sizeof(dir_req_header) + numpairs * sizeof(DPair);
   pid = 0xbd;                                  // pid: bcst directory
   if (*debug > 1)
   {
      frWin.print( "Directory request for holes:" );
      DPair* dp = dHoles;
      for (int i = 0; i < numpairs; dp++, i++ )
      {
         sprintf( frWin.buf, "old: %10lu, new: %10lu", dp->tOld, dp->tNew );
         frWin.print( frWin.buf );
      }
   }

   if (*debug > 1)
   {
      frWin.print( "broadcast command:" );
      frWin.printHex( bp->data, bp->cnt );
   }
   if (axp)
      axp->output( &bcstAddr, &myAddr, pid, *bp );// send AX.25 UI frame
}

void DirBcstMgr::saveFrame( MBuf& mb )
// --------------------------------------------------------------------------
// Save the data from a broadcast directory frame
// --------------------------------------------------------------------------
{
   // AX.25 header should have striped off at this point.  The remainder
   // should be a directory broadcast frame.

   recvBytesIn += mb.cnt;                        // inc total dir bytes received
   if (calcrc( mb.data, mb.cnt ) != 0)           // if CRC error
   {
      recvErrCnt++;                              // increment error count
      if (eCallUp)                               // if we need to report the
         eCallUp->func();
      iWin.print( "Bad directory frame CRC." );
      return;
   }
   mb.cnt -= 2;                                  // exclude the CRC
   DirFrHdr& hdr = decodeHdr( mb );              // decoded frame header info
   if (&hdr == &NOFRHDR)                         // and remove header
   {
      recvErrCnt++;                              // increment error count
      if (eCallUp)                               // if we need to report the
         eCallUp->func();
      iWin.print( "Bad directory frame header.");
      return;
   }
   if (*debug > 1)
   {
      sprintf( frWin.buf, "file %8lx, offset %lu, tO %lu, tN %lu, E-bit %c, N-bit %c",
               hdr.fileId, hdr.offset, hdr.tOld, hdr.tNew,
               hdr.lastFrame ? 'T' : 'F', hdr.newestFile ? 'T' : 'F');
      frWin.print( frWin.buf );
   }

   if (mb.cnt <= 0)
   {
      recvErrCnt++;                              // increment error count
      if (eCallUp)                               // if we need to report the
         eCallUp->func();
      iWin.print( "No data found in directory frame." );
      return;
   }

   // Put frame into directory data accumulator.  This is necessary
   // because the file headers can span directory frames.

   dirCp = dirData + (int) hdr.offset;           // set working pointer
   if (dirCp + mb.cnt > dirData + maxPfhSize)    // if it won't fit
   {
      recvErrCnt++;                              // increment error count
      if (eCallUp)                               // if we need to report the
         eCallUp->func();
      iWin.print( "WARNING: pfh length > maxpfhsize" );
      return;
   }
   if (!hdr.offset)                              // if start of pfh
      dirCnt = 0;                                // reset the count
   memcpy( dirCp, mb.data, mb.cnt );             // load accumulator
   dirCnt += mb.cnt;                             // up the count
   if (!hdr.lastFrame)                           // if we don't have the
      return;                                    // entire pfh

   if (dirCnt != hdr.offset + mb.cnt)            // if the length isn't good
   {
      recvErrCnt++;                              // increment error count
      if (eCallUp)                               // if we need to report the
         eCallUp->func();
      iWin.print( "Bad directory frame.");
      return;
   }

   PFileHeader pfh( iWin );
   int len = pfh.get( dirData, dirCnt );         // decoded file header info
   if (!len)                                     // if we don't have good pfh
      return;                                    // shouldn't get here
   if (pfh.header_checksum != pfh.csum)          // if bad checksum
   {
      pfh.display();                             // display the pfh
      recvErrCnt++;                              // increment error count
      if (eCallUp)                               // if we need to report the
         eCallUp->func();
      iWin.print( "PFH checksum error" );        // display in info window
      return;
   }

   // Check the hole list to determine if the directory entry is needed.
   // Update the holelist in any case.  Also check the recent directories
   // heard list, and don't save if it is already on the list.  We may have
   // gotten the pfh from a file we were grabbing.

   if (needEntry( hdr ) && !dfm.onDirList( hdr.fileId ))
   {
      pfh.display();                             // display new dir entries

      // Evaluate select equations at this point.  Flag any files for
      // downloading in the slist.

      if (autoSelect)                            // autoselect from .CFG
      {
         // It the file isn't on the sList or the status is the default
         // value of GRAB, then do the other checks.  Otherwise, don't muck
         // with the status.

         int status = sList.status( pfh.file_number );
         if (status < 0 || status == sList.GRAB )
            if (eqnPriority && sem.resolve( eqnPriority, &pfh ))
            {
               sList.set( pfh.file_number, sList.PRIORITY );
               iWin.print( "Activating P file." );
            }
            else
            if (eqnAuto && sem.resolve( eqnAuto, &pfh ))
            {                                    // mark as auto download
               sList.set( pfh.file_number, sList.AUTOMATIC );
               iWin.print( "Activating A file." );
            }
            else
            if (eqnNever && sem.resolve( eqnNever, &pfh ))
            {                                    // mark as auto download
               sList.set( pfh.file_number, sList.NEVER );
            }
      }
      dfm.saveData( dirData, len );              // put in dir file
      dir.addPfh( pfh );                         // put in dir display
   }
   sLine.dirTotal( recvBytesIn );                // display the count

   if (hdr.newestFile)                           // check entries with the
      if (dirUpToDate( hdr.tNew ))               // newest-file-flag set
         upToDate = 1;
      else
         upToDate = 0;

   if (upToDate)                                 // if dir is up-to-date
   {
      if (dirRqstWaitTimer->getState() != Timer::Started)
         dirRqstWaitTimer->start();              // start request wait timer
      sLine.dirStatus( 0 );
   }
   else
      sLine.dirStatus( dirHoleCount );
}

void DirBcstMgr::writeHoles()
// -------------------------------------------------------------------------
//  Update directory hole files from hole list
// -------------------------------------------------------------------------
{
   FILE* hole_file = 0;
   int   i;
   char  old_name[ MAXPATH + 2 ];
   char  new_name[ MAXPATH + 2 ];

   sprintf( old_name, "%sPFHDIR.HOL", pfhDir );
   sprintf( new_name, "%sPFHDIR.OLH", pfhDir );
   remove( new_name );
   if (rename( old_name, new_name ) != 0)
   {
      sprintf( frWin.buf, "dir_write_holes : error renaming %s to %s, %s",
               old_name, new_name, sys_errlist[ errno ] );
      frWin.printFatal( frWin.buf );
   }
   if ((hole_file = fopen( old_name, "wt" )) == 0)
   {
      sprintf( frWin.buf, "dirWriteHoles : error creating PFHDIR.HOL, %s",
               sys_errlist[errno] );
      frWin.printFatal( frWin.buf );
   }
   fprintf( hole_file, "0 pfh header received\n" );
   fprintf( hole_file, "0 pfh file length\n" );
   fprintf( hole_file, "%u holes\n", dirHoleCount );

   for (i = 0; i < dirHoleCount; i++)
      fprintf( hole_file, "%lu, %lu\n", dHoles[i].tOld, dHoles[i].tNew );
   fclose( hole_file );
}
