//
// This file contains proprietary information of Jesse Buckwalter.
// Copying or reproduction without prior written approval is prohibited.
//
// Copyright (c) 1993, 1994, 1995
// Jesse Buckwalter
// 525 Third Street
// Annapolis, MD 21403
// (410) 263-8652
//

#ifndef  __ASYINTF_H
#include "asyintf.h"
#endif

#ifndef  __DOS_H
#include <dos.h>
#endif

#ifndef __CTYPE_H
#include <ctype.h>
#endif

#ifndef  __STDIO_H
#include <stdio.h>
#endif

#ifndef  __8250_H
#include "8250.h"
#endif

#ifndef  __MSGBUF_H
#include "msgbuf.h"
#endif

#ifndef  __WINDOW_H
#include "window.h"
#endif

// NOTE:  All of the return codes listed in the comments are NOT valid.
//        They are a holdover from the commercial library implementation.

AsyncIntf::AsyncIntf( unsigned& showDataHex, const unsigned port )
                    :
                      asy( 0 ), HwDevice(),
                      framingErr( 0 ), overflowErr( 0 ),
                      overrunErr( 0 ), parityErr( 0 ), speed( 9600 ),
                      parity( 'N' ), dataBits( 8 ), stopBits( 1 ),
                      inQSize( 512 ), outQSize( 512 ), ioBuffer( 0 ),
                      recvOutSize( 512 )
// -------------------------------------------------------------------------
//  Constructor
// -------------------------------------------------------------------------
{
   this->showDataHex = &showDataHex;
   this->port = port > MAXPORT ? 1 : port;       // invalid defaults to 1
}

AsyncIntf::~AsyncIntf()
// -------------------------------------------------------------------------
//  Destructor
// -------------------------------------------------------------------------
{
   if (ioBuffer)                                 // if port is open
      close();                                   // close it
   printf( "\n" );                               // print errors
   printf( "com port %u overflow errors: %4u\n", port, overflowErr );
   printf( "com port %u overrun errors:  %4u\n", port, overrunErr );
   printf( "com port %u parity  errors:  %4u\n", port, parityErr );
   printf( "com port %u framing errors:  %4u\n\n", port, framingErr );
}

//unsigned AsyncIntf::bufLeft( const char io )
// -------------------------------------------------------------------------
//  Return the number of bytes of space remaining in the selected buffer.
//  The INPUT buffer is checked if <io> is (I), and the output buffer is
//  interrogated if <io> is (O).  Any other "action code" will return zero.
//  Use this function when it is important to avoid program delays due to
//  calls to output functions or to give priority to reception of data (to
//  prevent overflows).
//
//  io:   Action code: I = input, O = output, B = both
//		    Returns zero if action code unrecognized.
// -------------------------------------------------------------------------
//{}

int AsyncIntf::close()
// -------------------------------------------------------------------------
//  NOTE:  close() shuts down a communications channel immediately,
//    even if there is data present in the input or output buffers.
//    Therefore, you may wish to call the waitForClear function
//    before closing the ports.
//
//  The returned values are:
//           0 - Port successfully closed
//           2 - Invalid port number specified.
//           3 - Port is not open.
//           4 - The interrupt sharing signature
//               has been lost. This only occurs on
//               PS/2 class machines supporting
//               interrupt sharing.
// -------------------------------------------------------------------------
{
   int errCode = asy->close();
   overrunErr = asy->errOverrun;
   parityErr = asy->errParity;
   framingErr = asy->errFraming;
   overflowErr = asy->fifo.overflow;
   delete asy;
   delete ioBuffer;                              // deallocate buffer
   ioBuffer = 0;
   set( devDTR, 1 );                             // raise DTR on exit
   return errCode;
}

//int AsyncIntf::CTSstat()
// --------------------------------------------------------------------------
//  CTSstat() provides a means to interrogate the clear-to-send hardware
//  line.  In a typical arrangement, when CTS is asserted, this
//  signals the host (this computer) that the receiver is ready to accept
//  data (in contrast to the DSR line, which signals the receiver as
//  on-line but not nessesarily ready to accept data).  An automated
//  mechanism ( see set() ) is provided to do this, but in cases where
//  this is undesirable or inappropriate, the CTSstat function can be used
//  to intterrogate this line manually.
// --------------------------------------------------------------------------
//{}

int AsyncIntf::dataReady()
// --------------------------------------------------------------------------
// Returns the status of the input buffer.  Returns 1 of data is in the
// buffer; otherwise, returns 0.
// --------------------------------------------------------------------------
{
   return asy->iQSize() ? 1 : 0;
}

//int AsyncIntf::DCDstat()
// --------------------------------------------------------------------------
//  Returns the status of the data carrier detect (DCD) line from the rem-
//  ote device, typically a modem.  When asserted (DCDstat returns 1),
//  the modem indicates that it has successfuly linked with another modem
//  device at another site.
//
// --------------------------------------------------------------------------
//{}

//int AsyncIntf::DSRstat()
// --------------------------------------------------------------------------
//  Returns status of Data Set Ready (DSR) signal line.
//  The data set ready (DSR) line is typically used by a remote station
//  to signal the host system that it is on-line (although not nessesarily
//  ready to receive data yet - see CTSstat().  The DCE has the DSR line
// asserted if DSRstat returns 1.
// --------------------------------------------------------------------------
//{}

unsigned AsyncIntf::errCnt()
// --------------------------------------------------------------------------
//   Return the total of all port errors
// --------------------------------------------------------------------------
{
   return asy->errOverrun + asy->errParity + asy->errFraming +
          asy->fifo.overflow;
}

//int AsyncIntf::exist( const unsigned port )
// --------------------------------------------------------------------------
//  Function exist determines if the indicated port indeed exists.
//  Use this function before attempts to open a port for I/O.
// --------------------------------------------------------------------------
//{}

int AsyncIntf::flush( const char io ) const
// --------------------------------------------------------------------------
//  flush() allows the user to completely clear the contents of either
//  the input (receive) and/or output (transmit) buffers.  The "action
//  code" passed in <io> determines if the input (I) or output (O) buffer
//  is cleared.  Action code (B) will clear both buffers.  This is useful
//  if you wish to cancel a transmitted message or ignore part of a
//  received message.
//
//  io:   Action code: I = input, O = output, B = both
//
//  The values returned are:
//           0 - Queues successfully drained.
//           1 - Invalid action code.
//           2 - Invalid port number.
//           3 - Port not opened.
//           7 - Output queue is full. This error
//               can be returned if local flow
//               control is enabled and flushing the
//               input queue causes an XON to be
//               transmitted.
// --------------------------------------------------------------------------
{
   char iou = toupper( io );
   if (iou == 'I' || iou == 'B')
      asy->iFlush();
   if (iou == 'O' || iou == 'B')
      asy->oFlush();
   return 0;
}

int AsyncIntf::open()
// --------------------------------------------------------------------------
//  The calling functions must be able to handle the size of the
//  MBuf returned by recvOut( MBuf*& bp ) which will be recvOutSize.
// --------------------------------------------------------------------------
{
   // The calling functions must be able to handle a buffer if recvOutSize
   // return by the recvOut( MBuf*& bp ) function.

   return open( inQSize, outQSize, 0, 0, recvOutSize );
}

int AsyncIntf::open( const unsigned inSize, const unsigned outSize,
                     const unsigned irq,    const unsigned addr,
                     const unsigned recvOutSize )
// -------------------------------------------------------------------------
//  Allocate buffers and open the port.
//
//  inBufferSize:  requested size of input (receive) buffer
//  outBufferSize: requested size of output (transmit) buffer
//  irq:           interrupt level for the port.  A value of zero specifies
//                 the default values for port 1 and 2 (IRQ4 and IRQ 3
//                 respectively.
//  addr:          address of the com port.  A value of zero specifies the
//                 default values for port 1 and 2 (0x3f8 and 0x2f8
//                 respectively).
//  recvOutSize:   size of the receive buffers that are returned from the
//                 recvOut( MBuf*& bp ) function.  The calling functions must
//                 be able to handle a buffer this size.
//
//  The values returned are:
//           0 - Port successfully opened.
//           1 - Can't allocate buffer.
//           2 - Invalid port number specified.
//           4 - Invalid parameter.  Either a bad
//               buffer address or port address was
//               specified, or open_line_state was
//               set to an invalid value.
//           6 - No serial port found at the
//               specified port location.
//           9 - Port is already opened.  The port
//               must be closed before it can be
//               reopened.
//          14 - No default port address available.
//          15 - No default IRQ line available.
//          16 - Buffer address or queue size
//               invalid.
//          17 - Error in parameter initialization.
//          18 - Can't create port object.
// -------------------------------------------------------------------------
{
   if (ioBuffer)                                 // if buffer allocated
      return 9;                                  // interface is already open

   this->recvOutSize = recvOutSize;              // set output buffer size

   // Allocate memory for I/O buffers

   if ((ioBuffer = new char[ inSize + outSize + 128 ]) == 0)
      return 1;

   if ((asy = new Asy( port, addr, irq )) == 0)
      return 18;

   int errCode;
   if ((errCode = asy->open( *ioBuffer, inSize + outSize + 128 )) != 0)
   {
      delete ioBuffer;
      ioBuffer = 0;
      return errCode;
   }

   // Set up default parameters for port

   if ((errCode = params( speed, parity, dataBits, stopBits )) != 0)
   {
      delete ioBuffer;
      ioBuffer = 0;
      return 17;
   }

   return 0;
}

int AsyncIntf::params( const unsigned baud, const char pty,
                       const unsigned data, const unsigned stop)
// -------------------------------------------------------------------------
//  params() is used to configure an opened port for the desired
//  communications parameters: baud, data size, parity, and # of stop bits.
//  A call to this function will set up the port appropriately, as well as
//  assert the DTR, RTS and OUT2 control lines and clear all buffers.
//
//  baud:     desired baud; ASY_MINBAUD <= baud <= ASY_MAXBUAD
//  pty:      type of parity
//            May be N)one, E)ven, or O)dd.
//            N)one selected if type is unknown.
//  data:     data size in bits.  Must be 5 - 8 bits.
//              8-bit byte used if out of range.
//  stop:     number of bit times to let line quiesce. Range (1-2).
//              1 stop bit used if out of range.
// --------------------------------------------------------------------------
{
   int errCode = 0;
   if ((errCode = asy->setOpt( asy->BAUDRATE, baud )) != 0)
      return errCode;

   if ((errCode = asy->setOpt( asy->PARITYOPT, pty )) != 0)
      return errCode;

   if ((errCode = asy->setOpt( asy->DATABITS, data )) != 0)
      return errCode;

   if ((errCode = asy->setOpt( asy->STOPBITS, stop )) != 0)
      return errCode;

   if ((errCode = asy->setOpt( asy->CTSmode, 0 )) != 0) // remove CTS rqmt
      return errCode;

   asy->iFlush();                                // flush input queue
   asy->fifo.overflow = 0;                       // clean the slate
   asy->oFlush();                                // flush output queue
   return 0;
}

void AsyncIntf::poll()
// --------------------------------------------------------------------------
// Trigger the receiver function.  If the pointer isn't null, pass
// the buffer to the higher level protocol interface.
// --------------------------------------------------------------------------
{
   if (!asy->iQSize())                           // if no data
      return;
   MBuf* bp;
   recvOut( bp );
   if (bp)
      if (hiProto)
         hiProto->recvIn( *bp );
      else
         delete bp;
}

int AsyncIntf::recvOut( MBuf*& bp )
// --------------------------------------------------------------------------
//  Construct and fill an MBuf from the port.
//
//  The values returned are:
//           0 - Character buffer successfully returned.
//           1 - Can't allocate receive buffer.
//           2 - Invalid port number specified.
//           3 - Port is not open.
//           7 - Output queue is full.  This only
//               occurs if XON/XOFF is enabled.
//          10 - The input queue is empty; there
//               are no characters to read.
// --------------------------------------------------------------------------
{
   bp = 0;                                       // zero pointer to start
   if (!ioBuffer)                                // if interface not open
      return 3;

   bp = new MBuf( recvOutSize );
   if (!bp)
      return 1;

   return recvOut( bp->data, bp->size, bp->cnt );
}

int AsyncIntf::recvOut( char* bp, const unsigned len, unsigned& cnt )
// --------------------------------------------------------------------------
//  Fill a specified array from the port.
//
//  The values returned are:
//           0 - Character buffer successfully returned.
//           1 - Invalid receive buffer.
//           2 - Invalid port number specified.
//           3 - Port is not open.
//           7 - Output queue is full.  This only
//               occurs if XON/XOFF is enabled.
//          10 - The input queue is empty; there
//               are no characters to read.
// --------------------------------------------------------------------------
{
   cnt = 0;
   if (!ioBuffer)                                // if interface not open
      return 3;
   if (!bp)                                      // if invalid pointer
      return 1;

   unsigned iQSize;
   cnt = asy->read( bp, len, iQSize );

   if (cnt && (*showDataHex & RECV_OUT) && frWin)
      frWin->printHex( bp, cnt );

   return 0;
}

int AsyncIntf::recvOut( char& cp )
// -------------------------------------------------------------------------
//  Returns character from input buffer.
//
//  The values returned are:
//           0 - Character successfully returned.
//           1 - Same as 10.
//           2 - Invalid port number specified.
//           3 - Port is not open.
//           7 - Output queue is full.  This only
//               occurs if XON/XOFF is enabled.
//          10 - The input queue is empty; there
//               are no characters to read.
// -------------------------------------------------------------------------
{
   if (!ioBuffer)                                // if interface not open
      return  3;

   unsigned iQSize;
   int errCode = asy->readCh( &cp, iQSize );
   if (errCode)
      return errCode;

   if ((*showDataHex & RECV_OUT) && frWin)
      frWin->printHex( &cp, 1 );
   return 0;
}

//  int AsyncIntf::recvW( char& cp )
// -------------------------------------------------------------------------
//  Returns character from input buffer.  Waits for a charecter to be
//  received if the buffer is empty.  If the port is not open, the function
//  returns 1; otherwise 0.
// -------------------------------------------------------------------------
//{}

//int AsyncIntf::RIstat()
// --------------------------------------------------------------------------
//  Returns the status of the ring indicator (RI) line.  This line is
//  typically used only by modems and indicates that the modem has detected
//  an incoming call if RIstat returns 1.
//
// --------------------------------------------------------------------------
//{}

int AsyncIntf::set( enum devVal param, const unsigned long value )
// -------------------------------------------------------------------------
//  Set an interface parameter to the specified value.  Return 0 if
//  processed OK, otherwise return 1;
//
//  param : parameter to be set
//
//  value : value parameter is to be set to
// -------------------------------------------------------------------------
{
   switch( param )
   {
      // devDTR provides a means to control the port's DTR (Data Terminal
      // Ready) signal line.  When <value> is 1, the DTR line is raised,
      // signalling to remote equipment that the host is "on-line" (although
      // not nessesarily ready to receive data - see devRTS).

      case devDTR :
         break;

      // devRTS provides a means to control the port's RTS (Request To Send)
      // line.  If RTS handshaking is disabled (see devRTSmode), this
      // option may be used.  The RTS should not set if RTS handshaking is
      // enabled.

      case devRTS :
         break;

      // devCTSmode enables or disables automatic CTS handshaking.
      // If <value> is 1, CTS handshaking is enabled, which means that
      // if the remote drops the CTS line, the transmitter will be disabled
      // until the CTS line is asserted again.  Automatic handshake is disabled
      // if <value> is FALSE.  CTS handshaking and "software" handshaking
      // (provided by the softhandshake function) are compatable and may be
      // used in any combination.

      case devCTSmode :
         if (asy->setOpt( asy->CTSmode, (unsigned) value ) != 0)
            return 1;
         break;

      // devOUT1 provided for reasons of completeness only, since the
      // standard PC/XT/AT configurations do not use this control signal.
      // If <value> is 1, the OUT1 signal line on the 8250 will be set to a
      // LOW logic level (inverted logic).  The OUT1 signal is present on
      // pin 34 of the 8250 (but not on the port itself).

      case devOUT1 :
         break;

      // The OUT2 signal line, although not available on the port itself, is
      // used to gate the 8250 <INTRPT> interrupt line and thus acts as a
      // redundant means of controlling 8250 interrupts.  When <value> is
      // 1, the OUT2 line on the 8250 is lowered, which allows the
      // passage of the INTRPT signal through a gating arrangement,
      // allowing the 8250 to generate interrupts.  Int's can be disabled by
      // unASSERTing this line.

      case devOUT2 :
         break;

      // devRTSMode enables or disables automated RTS handshaking.  If <value>
      // is 1, automated RTS handshaking is enabled.  If enabled, the RTS line
      // will be DROPPED when the # of buffer bytes used reaches or exceeds
      // that of <RTSoff>.  The RTS line will then be raised when the
      // buffer is emptied down to the <RTSon> usage point.  If
      // <RTSon> > <RTSoff> then <RTSOn> will be the same as <RTSOff>.

      case devRTSmode :
         break;

      // devSOFTFLOW controls the usage of sftware (control-character) flow
      // control on transmission.  If software flow control is enabled
      // <value = 1> transmission will be halted if the character in "stop"
      // is received.  Transmission is re-enabled if the "start" character is
      // received.  Both the "start" and "stop" characters MUST be CONTROL
      // characters (i.e. start and stop must both be < 0x20.  Also, "start"
      // and "stop" cannot be the same character.  If either one of these
      // restrictions are violated, the defaults (^Q for start and ^S for
      // stop) will be used.

      case devSOFTFLOW :
         break;
      case devSTOPCHAR :
         break;
      case devSTARTCHAR :
         break;

      case devDelay:
         delayMs = unsigned( value );
         break;
      case devInterval:
         interval = unsigned( value );
         break;
   }
   return 0;
}

int AsyncIntf::send( MBuf& buf )
// --------------------------------------------------------------------------
//  Sends all the data in an MBuf and deletes it.
//  The values returned are:
//           0 - Bytes successfully written.
//           2 - Invalid port number specified.
//           3 - Port is not open.
//           7 - The output queue is full.
// --------------------------------------------------------------------------
{
   int errCode = send( buf.data, buf.cnt );
   delete &buf;
   return errCode;
}

int AsyncIntf::send( const char c )
// -------------------------------------------------------------------------
//  Sends a character.  If successful, return 0; if the port isn't open or
//  the comm function indicates an error, return 1.
//  The values returned are:
//           0 - Character successfully written.
//           2 - Invalid port number specified.
//           3 - Port is not open.
//           7 - The output queue is full.
// -------------------------------------------------------------------------
{
   if (!ioBuffer)                                  // if interface not open
      return  3;                                   // return error

   if ((*showDataHex & SEND_OUT) && frWin)
      frWin->printHex( &c, 1 );

   return asy->write( (char*) &c, 1 ) == 1 ? 0 : 1;
}

int AsyncIntf::send( const char* data, const unsigned cnt )
// --------------------------------------------------------------------------
//  Sends all the data in an array.  If successful, return 0; if the port
//  isn't open or the comm function indicates an error, return 1.
//  The values returned are:
//           0 - Bytes successfully written.
//           2 - Invalid port number specified.
//           3 - Port is not open.
//           7 - The output queue is full.
// --------------------------------------------------------------------------
{
   if (!ioBuffer)                                  // if interface not open
      return  3;                                   // return error

   if ((*showDataHex & SEND_OUT) && frWin)
      frWin->printHex( data, cnt );

   if (!delayMs)
      return asy->write( data, cnt );

   int rtnCode = 0;
   for (unsigned i = cnt; i > 0; i--)
   {
      if ((rtnCode = asy->write( data++, 1 )) != 0)
         break;
      if (interval && !(i % interval))
         delay( delayMs );
   }
   return rtnCode;
}

//void AsyncIntf::sendLn( const MBuf &buf )
// --------------------------------------------------------------------------
//  Sends an MBuf out the port with CR and LF appended.
// --------------------------------------------------------------------------
//{
//   send( buf );
//   send( "\r\n", 2 );
//}

//  int AsyncIntf::sendW( const char c )
// -------------------------------------------------------------------------
//  Works as send( const char ), but will wait until at least 1 free
//  position is available in the output buffer before attempting to place
//  the character "c" in it.   Allows the programmer to send characters
//  without regard to available buffer space.
// -------------------------------------------------------------------------
//{}

//void AsyncIntf::waitForClear()
// -------------------------------------------------------------------------
//  A call to waitForClear() will stop processing until the selected
//  output buffer is completely emptied.  Typically used just before a call
//  to the closecom function to prevent premature cut-off of messages in
//  transit.
// -------------------------------------------------------------------------
//{}

