//
// This file contains proprietary information of Jesse Buckwalter.
// Copying or reproduction without prior written approval is prohibited.
//
// Copyright (c) 1993, 1994, 1995
// Jesse Buckwalter
// 525 Third Street
// Annapolis, MD 21403
// (410) 263-8652
//

#ifndef  __8250_H
#include "8250.h"
#endif

#ifndef  __DOS_H
#include <dos.h>
#endif

#pragma option -k- // Do not use standard stack frame for these functions.
                   // The standard frame creates too much overhead.

long    Asy::BAUDCLK = 115200L;                  // 1.8432 Mhz / 16
long    Asy::MAXBPS = 115200L;
Asy*    Asy::asyList[ Asy::ASY_MAX ];
CastISR Asy::portISRs[ Asy::ASY_MAX ] = { Asy::asy0vec, Asy::asy1vec,
                                          Asy::asy2vec, Asy::asy3vec,
                                          Asy::asy4vec, Asy::asy5vec,
                                          Asy::asy6vec, Asy::asy7vec };

Asy::Asy( const int portNum, const int portAddr, const int portIRQ )
        :
          baud( 9600 ), busy( 0 ), ctsReq( 0 ), dataBits( 8 ),
          errBreak( 0 ), errFraming( 0 ), errOverrun( 0 ), errParity( 0 ),
          fifoDepth( 14 ), fifoTimeOuts( 0 ),
          is16550a( 0 ), overrun( 0 ), parity( 0 ), portOpen( 0 ),
          rlsdReq( 0 ), rxChar( 0 ), rxInts( 0 ), stInts( 0 ), stopBits( 1 ),
          txChar( 0 ), txInts( 0 )
// --------------------------------------------------------------------------
//  Constructor
// --------------------------------------------------------------------------
{
   const unsigned defaultAddr[ 4 ] = { 0x03f8, 0x02f8, 0x03e8, 0x02e8 };
   const unsigned defaultIRQ[ 4 ] = { 4, 3, 4, 3 };

   if ((port = portNum - 1) > ASY_MAX - 1)
      port = 0;
   if (!portAddr)
      base = defaultAddr[ port ];
   if (!portIRQ)
      irq = defaultIRQ[ port ];
   asyList[ port ] = this;                       // put us in the list;

   disable();

   saveVec = getIRQ( irq );                      // save original interrupt
   saveMask = getMask( irq );                    // vector, mask state,

   saveLCR = inportb( base+LCR );                // control bits
   saveIER = inportb( base+IER );
   saveMCR = inportb( base+MCR );
   readMSR( MSR_CLEAR );
   saveMSR = msr;
   disable();
   saveIIR = inportb( base+IIR );

   setBit( base+LCR, LCR_DLAB );                 // save speed bytes
   saveDivL = inportb( base+DLL );
   saveDivH = inportb( base+DLM );
   clrBit( base+LCR, LCR_DLAB );
   setIRQ( irq, portISRs[ port ] );              // set interrupt vector to SIO handler

   // determine if 16550A, turn on FIFO mode and clear RX and TX FIFOs

   outportb( base+FCR, char( FIFO_ENABLE ) );

   // According to National ap note AN-493, the FIFO in the 16550 chip
   // is broken and must not be used.  To determine if this is a 16550A
   // (which has a good FIFO implementation) check that both bits 7
   // and 6 of the IIR are 1 after setting the fifo enable bit. If
   // not, don't try to use the FIFO.


   if ((inportb( base+IIR ) & IIR_FIFO_ENABLED) == IIR_FIFO_ENABLED)
   {
      is16550a = 1;
      outportb( base+FCR, FIFO_SETUP );
   }
   else
   {
      // Chip is not a 16550A.  In case it's a 16550 (which has a
      // broken FIFO), turn off the FIFO bit.

      outportb( base+FCR, 0 );
      is16550a = 0;
   }
   outportb( base+IER, 0 );                      // turn off interrupts
   setBit( base+MCR, 0 );                        // reset 8250 master interrupt

   enable();
}

Asy::~Asy()
// --------------------------------------------------------------------------
//  Destructor
// --------------------------------------------------------------------------
{
   disable();

   // Restore original interrupt vector and 8259 mask state

   setIRQ( irq, saveVec );
   if (saveMask)
      maskOn( irq );
   else
      maskOff( irq );

   if (is16550a)
   {
      // Purge hardware FIFOs and disable if we weren't already
      // in FIFO mode when we entered. Apparently some
      // other comm programs can't handle 16550s in
      // FIFO mode; they expect 16450 compatibility mode.

      outportb( base+FCR, FIFO_SETUP );
      if ((saveIIR & IIR_FIFO_ENABLED) != IIR_FIFO_ENABLED)
         outportb( base+FCR, 0 );
   }
   setBit( base+LCR, LCR_DLAB );                 // restore speed regs
   outportb( base+DLL, saveDivL );               // low byte
   outportb( base+DLM, saveDivH);                // hi byte
   clrBit( base+LCR, LCR_DLAB );

   outportb( base+LCR, saveLCR );                // restore control regs
   outportb( base+IER, saveIER );
   outportb( base+MCR, saveMCR );

   enable();
}

int Asy::checkInt()
// --------------------------------------------------------------------------
//  Interrupt handler for this Asy instance; called by ::asy?vec()
//  If interrupt was processed, return 1, otherwise return 0.
// --------------------------------------------------------------------------
{
   busy = 1;
   char intOccured = 0;;
   while( 1 )
   {
      char iir = inportb( base+IIR );            // read Interrupt Identification Register
      iir &= IIR_ID_MASK;                        // ignore timeout interrupt
      intOccured |= !(iir & IIR_IP);             // remember if any interrupts
                                                 //  during this call to checkInt()
      if (!(iir & IIR_IP))                       // if interrupt detected
      {                                          //  in this loop iteration
         switch (iir)
         {
            case IIR_RDA:                        // receiver interrupt
               rxInt();
               break;
            case IIR_THRE:                       // transmit interrupt
               txInt();
               break;
            case IIR_MSTAT:                      // modem status change interrupt
               stInt();
               break;
            case IIR_RLS:                        // error interrupt
               break;
         }
         if (iir & IIR_FIFO_TIMEOUT)             // should happen at end of a single packet
            fifoTimeOuts++;

         // Test for an interrupt of higher priority than a write.  May have a
         // line status or a read which may have masked a write interrupt.  So
         // write if possible.  In case an interrupt occured during processing,
         // it must also be serviced.

         if (iir > IIR_THRE)
            if (ctsReq)                          // if CTS required
            {
               readMSR( MSR_NO_CLEAR );
               if (msr & MSR_CTS)                // if CTS asserted
                  txInt();                       // kick the transmitter
            }
            else
               txInt();                          // leave interrupts enabled
      }
      else                                       // no interrupt occured
         break;
   }
   busy = 0;
   return !intOccured;
}

int Asy::close()
// --------------------------------------------------------------------------
//  Close port
// --------------------------------------------------------------------------
{
   if (!portOpen)
      return 1;
   disable();

// idleTimer->stop();                            // stop the idle timer, if running

   inportb( base+RBR );                          // purge receive data buffer
   outportb( base+IER, 0 );                      // turn off interrupts
   setBit( base+MCR, 0 );                        // reset 8250 master interrupt
   enable();
   return 0;
}

unsigned long Asy::getOpt( int option )
// --------------------------------------------------------------------------
//  Get transmission options
// --------------------------------------------------------------------------
{
   switch (option)
   {
      case BAUDRATE:
         return baud;
      case PARITYOPT:
         switch (parity)
         {
            case NOPARITY   : return 'n';
            case EVENPARITY : return 'e';
            case ODDPARITY  : return 'o';
         }
      case STOPBITS:
         switch (stopBits)
         {
            case STOPBIT1 : return 0;
            case STOPBIT2 : return 1;
         }
      case DATABITS:
         return dataBits + 5;
      case DTR:
         return (inportb( base+MCR ) & MCR_DTR ) ? 1 : 0;
      case RTS:
         return (inportb( base+MCR ) & MCR_RTS ) ? 1 : 0;
      case CTSmode:
         return ctsReq;
      case RLSDmode:
         return rlsdReq;
      case FIFODEPTH:
         return fifoDepth;
      case OUT1:
      case OUT2:
      case RTSmode:
      case SOFTFLOW:
      case STARTCHAR:
      case STOPCHAR:
      default:
         return 0xffffffffL;
   }
}

void Asy::iFlush()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   disable();
   fifo.cnt = 0;
   fifo.rp = fifo.wp = fifo.buf;
   enable();
}

void Asy::oFlush()
// --------------------------------------------------------------------------
// --------------------------------------------------------------------------
{
   disable();
   dma.cnt = 0;
   dma.busy = 0;
   enable();
}

int Asy::open( char& buffer, unsigned size )
// --------------------------------------------------------------------------
//  Open port
// --------------------------------------------------------------------------
{
   // Set up receiver FIFO

   fifo.wp = fifo.rp = fifo.buf = &buffer;
   fifo.bufSize = size;

   if (portOpen)                                 // see if already open
      return 1;
   inportb( base+RBR );                          // purge the receive data buffer
   disable();

   // Turn on receive interrupts and optionally modem interrupts;
   // leave transmit interrupts off until we actually send data.

   if (rlsdReq || ctsReq)
      outportb( base+IER, IER_MS | IER_DAV );
   else
      outportb( base+IER, IER_DAV );

   // Turn on 8250 master interrupt enable (connected to OUT2)

   setBit( base+MCR, MCR_OUT2 );
   int rtnCode = maskOn( irq );                  // tell PIC to allow interrupts
   enable();
   return rtnCode;
}

unsigned Asy::iQSize()
// --------------------------------------------------------------------------
//  Return the number of bytes in the fifo buffer
// --------------------------------------------------------------------------
{
   disable();
   unsigned tmp = fifo.cnt;
   enable();
   return tmp;
}

int Asy::read( char* bp, unsigned len, unsigned& iQSize )
// --------------------------------------------------------------------------
//  Read into a data array;  return count of actual bytes read.
//
//  len:  maximum number of bytes to be read
// --------------------------------------------------------------------------
{
   int cnt;
   for (cnt = 0; cnt < len; cnt++)
   {
      if (readCh( bp++, iQSize ) != 0)
         break;
   }
   return cnt;
}

int Asy::readCh( char* c, unsigned& iQSize )
// --------------------------------------------------------------------------
// Blocking read from asynch line.  Returns 0 if successful; otherwise 1.
// --------------------------------------------------------------------------
{
   // Atomic read and decrement of fifo.cnt

   disable();
   unsigned tmp = fifo.cnt;
   if (tmp)
      fifo.cnt--;
   enable();
   iQSize = tmp;

   if (!tmp)
      return 1;
   *c = *fifo.rp++;
   if (fifo.rp >= &fifo.buf[ fifo.bufSize ])
      fifo.rp = fifo.buf;
   return 0;
}

void Asy::readLSR()
// -----------------------------------------------------------------------------
//  Read the line status register on the serial port indicated by base,
//  recording any errors detected.  Interrupts are disabled on exit.
// -----------------------------------------------------------------------------
{
   disable();                                    // reset interrupt
   lsr = inportb( base+LSR );                    // read Line Statue Register

   if (lsr & LSR_BI)                             // break interrupt?
      errBreak++;
   if (lsr & LSR_OE)                             // overrun error?
      errOverrun++;
   if (lsr & LSR_FE)                             // framing error?
      errFraming++;
   if  (lsr & LSR_PE)                            // parity error?
      errParity++;
   if (lsr & LSR_ERROR)                          // record if error found
      portStatus |= ERROR_FOUND;
}

void Asy::readMSR( const char clearFlag )
// -----------------------------------------------------------------------------
//  Reads the modem status register on the serial port indicated by base,
//  recording any changes detected.
//
//  On exit:
//
//     Interrupts are enabled.
// -----------------------------------------------------------------------------
{
   disable();                                    // reset interrupts
   char newMsr = inportb( base+MSR );            // read Modem Status Register
   enable();
   msr = ((msr & 0x0f)                           // clear former upper four bits
          | newMsr )                             // combine new upper bits
                                                 //  with accumulated deltas
          & clearFlag;                           // possibly clearing deltas
   if (rlsdReq && (msr & MSR_DRLSD))
   {                                             // RLSD just changed and we care
      if (msr & MSR_RLSD)                        // keep count
      {
         answers++;
         portStatus |= CARRIER_DET;
      }
      else
      {
         remdrops++;
         portStatus &= ~CARRIER_DET;
      }
   }

   // Don't bother to do anything if DATA SET READY or RING INDICATOR changed.
}

void Asy::rxInt()
// --------------------------------------------------------------------------
//  Process 8250 receiver interrupts
// --------------------------------------------------------------------------
{
   rxInts++;
   for (;;)
   {
      readLSR();
      if (lsr & LSR_DR)                          // if data ready
      {
         rxChar++;
         char c = inportb( base+RBR );           // read receive buffer register

         // If XON/XOFF received, handle here

         if (fifo.cnt != fifo.bufSize)           // if room in the queue
         {
            *fifo.wp++ = c;                      // put char in queue
            if (fifo.wp >= &fifo.buf[ fifo.bufSize ])
               fifo.wp = fifo.buf;               // wrap around
            fifo.cnt++;
            if (fifo.cnt > fifo.hiWat)           // track the high water mark
               fifo.hiWat = fifo.cnt;

            // If queue if filling up, handle XON/XOFF here
            //                         handle hardware flow control here
         }
         else                                    // else we lost the character
         {
            fifo.overflow++;                     // overflow
            portStatus |= (BUFFER_FULL | ERROR_FOUND);
            return;
         }
      }
      else
        break;
   }
}

int Asy::setOpt( int option, unsigned long value )
// --------------------------------------------------------------------------
//  Set transmission options
// --------------------------------------------------------------------------
{
   char newMask;
   switch (option)
   {
      case BAUDRATE:
         break;
      case PARITYOPT:
         switch (char( value ))                  // set parity mask
         {
            case 'N': case 'n': parity = NOPARITY; break;
            case 'E': case 'e': parity = EVENPARITY; break;
            case 'O': case 'o': parity = ODDPARITY; break;
            default : return 1;
         }
         newMask = parity | 0xc7;
         break;

      case STOPBITS:
         switch (char( value ))                  // set stop-bit mask
         {
            case 0  : stopBits = STOPBIT1; break;
            case 1  : stopBits = STOPBIT2; break;
            default : return 1;
         }
         newMask = stopBits | 0xfb;
         break;

      case DATABITS:
         if (value < 5 || value > 8)             // set data-size mask
            return 1;
         dataBits = newMask = char( value ) - 5;
         break;

      case DTR:
      case OUT1:
      case OUT2:
      case RTS:
      case DOWN:
      case UP:
      case CTSmode:
      case RLSDmode:
      case FIFODEPTH:
         break;
      case RTSmode:
      case SOFTFLOW:
      case STARTCHAR:
      case STOPCHAR:
      default:
         return 1;
   }

   switch (option)
   {
      case PARITYOPT:
      case STOPBITS:
      case DATABITS:
         disable();                              // set line-control register
         outportb( base+LCR, newMask & inportb( base+LCR ) );
         enable();
         break;
      case BAUDRATE:
         return speed( value );
      case CTSmode:
         ctsReq = int( value ) ? 1 : 0;
         break;
      case RLSDmode:
         rlsdReq = int( value ) ? 1 : 0;
         break;
      case DTR:
         writeBit( base+MCR, MCR_DTR, int( value ) );
         break;
      case DOWN:
         clrBit( base+MCR, MCR_RTS );
         clrBit( base+MCR, MCR_DTR );
         break;
      case UP:
         setBit( base+MCR, MCR_RTS );
         setBit( base+MCR, MCR_DTR );
         break;
      case RTS:
         writeBit( base+MCR, MCR_RTS, int( value ) );
         break;
      case FIFODEPTH:
         fifoDepth = value <= FIFO_SIZE_14 ? unsigned( value ) : FIFO_SIZE_14;
         break;
   }
   return 0;
}

void Asy::stInt()
// --------------------------------------------------------------------------
// Handle 8250 modem status change
//   If the signals are unchanging, we ignore them.
//   If they change, we use them to condition the line.
// --------------------------------------------------------------------------
{
   stInts++;
   readMSR( MSR_NO_CLEAR );

   if (ctsReq && (msr & MSR_DCTS))
      if (msr & MSR_CTS)                         // CTS has changed and we care
      {
         if (dma.busy)                           // CTS went up
         {
            setBit( base+IER, IER_TxE );         // enable transmit interrupts
            txInt();                             // kick the transmitter
         }
      }
      else
         clrBit( base+IER, IER_TxE );            // CTS now dropped, disable
                                                 // transmit interrupts

   if (msr & (MSR_TERI | MSR_RI))                // if RI, raise DTR and RTS
      setOpt( UP, 0L );
}

int Asy::speed( unsigned long bps )
// --------------------------------------------------------------------------
//  Set asynch line speed.
//  Return 0 if success, 1 on error.
// --------------------------------------------------------------------------
{
   if (bps == 0 || bps > MAXBPS)
      return 1;
   baud = bps;
   long divisor = BAUDCLK / bps;
   disable();
   inportb( base+RBR );                          // purge receive data buffer
   if (is16550a)                                 // clear tx+rx fifos
      outportb( base+FCR, char( FIFO_SETUP ) );
   setBit( base+LCR, LCR_DLAB );                 // turn on divisor latch access bit
                                                 // load the two bytes of the register
   outportb( base+DLL, char( divisor & 0xff ) );     // low byte
   outportb( base+DLM, char( divisor >> 8) & 0xff ); // hi byte
   clrBit( base+LCR, LCR_DLAB );                 // turn off divisor latch access bit
   enable();
   return 0;
}

void Asy::txInt()
// --------------------------------------------------------------------------
//  Handle 8250 transmitter interrupts
// --------------------------------------------------------------------------
{
   txInts++;

   // If XON/XOFF scheduled, check to see if blocked by XOFF

// if (portStatus & REM_XOFF)
// {
//    enable();
//    return;
// }

   readLSR();                                    // read Line Status Register
                                                 // also disable interrupts
   if (!(lsr & LSR_THRE))                        // if THR not empty
      return;

   // Now we know that it's possible to transmit.
   //
   // Check for possible output:  first check XOFF, and send it if
   // appropriate.

// if (portStatus & SEND_XOFF)
// {
//    portStatus &= ~(SEND_XOFF | XON_SENT);
//    portStatus |= (XOFF_SENT | ERROR_FOUND);
//    put XOFF at front of transmit queue;
// }
// else
// {
//    if (portStatus & SEND_XOFF)
//    {
//       portStatus &= ~SEND_XON;                // turn off flag
//                                               // set flag to indicate that
//       portStatus |= (XON_SENT | ERROR_FOUND); // XON has been sent
//       portError  |= STAT_L_ON;
//       put XON at front of transmit queue;
//    }
// }

   // If it's a 16550A, load up to 16 chars into the tx hw fifo
   // at once. With an 8250, it can be one char at most.

   if (is16550a)
   {
      int count = dma.cnt < fifoDepth ? dma.cnt : fifoDepth;

      // 16550A: LSR_THRE will drop after the first char loaded
      // so we can't look at this bit to determine if the hw fifo is
      // full. There seems to be no way to determine if the tx fifo
      // is full (any clues?). So we should never get here while the
      // fifo isn't empty yet.

      txChar += count;
      dma.cnt -= count;
      while (count-- != 0)
         outportb( base+THR, *dma.data++ );
   }
   else // 8250
   {
      do
      {
         txChar++;
         outportb( base+THR, *dma.data++ );
         readLSR();
      } while (--dma.cnt != 0 && (lsr & LSR_THRE));
   }
   if (dma.cnt == 0)
   {
      dma.busy = 0;
      clrBit( base+IER, IER_TxE );               // disable further transmit interrupts
   }
   enable();
}

int Asy::write( const char* buf, unsigned cnt )
// --------------------------------------------------------------------------
//  Send a buffer on the serial transmitter and wait for completion
// --------------------------------------------------------------------------
{
   if (dma.busy)
      return -1;                                 // already busy

   dma.data = buf;
   dma.cnt = cnt;
   dma.busy = 1;

   // If CTS flow control is disabled or CTS is true,
   // enable transmit interrupts here so we'll take an immediate
   // interrupt to get things going. Otherwise let the
   // modem control interrupt enable transmit interrupts
   // when CTS comes up. If we do turn on TxE,
   // "kick start" the transmitter interrupt routine, in case just
   // setting the interrupt enable bit doesn't cause an interrupt

   if (!ctsReq || (msr & MSR_CTS))
   {
      setBit( base+IER, IER_TxE );
      txInt();
   }

   for(;;)                                       // wait for completion
   {
      disable();
      int tmp = dma.busy;
      enable();
      if( tmp == 0)
         break;
   }
   return cnt - dma.cnt;                         // bytes actually sent
}
