/* Copyright (C) 1999 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

/*

The classes in this file are responsible for

- unloading tr_buffer (the keyboard classes are responsible for loading tr_buffer)
- sending to sendpipe and reading from receivepipe (in other words, it is the
  interface to the Kam Plus TNC)
- processing frames (including information and status frames received from
  the TNC - for which purpose it uses various methods in the classes in 
  screen.h to display the information)
- sending commands to the TNC
- unloading file_buffers and sending the message to the TNC
- converting between Latin-1 and CP437 (and vice-versa) and betweeen
- converting between Latin-1 and CP437 (and vice-versa) and betweeen
  Unix linefeed ('\n') and Coms carriage return ('\r')

*/

#ifndef TNC_H
#define TNC_H

#include <qapplication.h>
#include "prog_defs.h"
#include "pipes.h"
#include "buffers.h"
#include "dialogs.h"
#include "download.h"

typedef unsigned char uchar;

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

extern Prog_func prog_func;

class MainScreen;
class ReceiveWin;

// the following are commands required by the Kam to change modes and from transmit
// to receive, etc

const char packetCMD = 'X';
const char pactorCMD[] = "PACTOR";
const char rttyCMD[] = "RTTY";
const char asciiCMD[] = "ASCII";
const char amtorCMD[] = "AMTOR";
const char lamtorCMD[] = "LAMTOR";
const char fecCMD[] = "FEC";
const char gtorCMD[] = "GTOR";
const char gmonCMD[] = "GMON";
const char torCMD[] = "TOR";
const char cwCMD[] = "CW";
const char packet_connectCMD[] = "CONNECT ";
const char pactor_connectCMD[] = "PACTOR ";
const char amtor_connectCMD[] = "AMTOR ";
const char gtor_connectCMD[] = "GTOR ";

const char txCMD = 'T';
const char rxCMD = 'E';
const char immediate_rxCMD = 'R';
const char tor_disconnectCMD = 'D';
const char packet_disconnectCMD[] = "DISCONNECT";
const char aborttxCMD = 'A';
const char statusCMD[] = "STATUS";

const unsigned char FEND = 0xc0;
const unsigned char FESC = 0xdb;
const unsigned char TFEND = 0xdc;
const unsigned char TFESC = 0xdd;

////

struct hostframe {
    char command;
    char port;
    char stream;
    char data[513]; // we have no control over the size of received frames; 256 bytes is maximum
                    // and an array size of 513 is ultra-conservative to allow for missed
                    // FESC characters because of data overruns 
    int data_count;
    int FEND_count;
    int bytes_afterFEND;
    int got_FESC;
};

struct Tnc_func {
    enum Hfmode {packet, pactor, rtty, ascii, amtor,lamtor, fec, gtor, gmon, tor, cw} hfmode;
    enum {pactor_gtor, not_pactor_gtor} tor_connected_mode;
    enum connected_status {disconnected, connected}; connected_status stream_status[MAXUSERS][2];
    char hisCall[MAXUSERS][2][hisCall_SIZE + 1];
    enum {off, on} hisCall_lock, speed_lock;
    enum {no, yes} have_sent_lock;
    enum {rx, tx} tx_status;
    char selCall[selCall_SIZE + 1];
    int active_hf_stream;
    int active_vhf_stream;
    int active_port;
    int active_stream(void) const {return active_port ? active_hf_stream : active_vhf_stream;}
    int capturefile_flag;
    ofstream capturefile;
    struct{int port, stream;} capture_stream;
    DownloadList* download_list_ptr;
    PrintList* print_list_ptr;
    int got_disconnect_flag;
    struct{int port, stream;} disconnect_stream;
    struct {int vhf_paclen, hf_paclen;} paclen;
    int no_rx_cw_flag;
    int rx_bell_flag;
    int slow_write_flag;
    int sending_cw_flag;
    int active_paclen(void) const {if (!active_port) return paclen.vhf_paclen;
                                   else if (hfmode == packet) return paclen.hf_paclen;
                                   else return paclen.hf_paclen - 1;} // leave room for final '\n' after '\r' in frame
};

// Note: If Kam port 1 (VHF), is being displayed in the receive window, active_port
// will contain 0, and if Kam port 2 (HF) is being displayed, it will contain 1.

// Active_hf_stream will contain a number representing the hf stream currently
// being displayed, or if a vhf stream is being displayed, the last hf stream
// displayed.  Active_vhf_stream will do the same for the vhf stream.
// The number contained will be 0 for stream A, 1 for stream B, and so on.

// Function int active_stream() is a short-hand way of getting the current active
// stream of the current port being displayed in the receive window.


// The following abstract base class provides virtual methods process_received_byte()
// and send_to_tnc() to make it slightly easier for the user to switch between normal
// mode (running in Kam host mode) and setup mode (running in Kam terminal mode).
// Various other methods are also declared virtual and provide dummy functions in
// order to make it easier to provide a generic means of using the normal mode and
// setup mode derived classes

class Tnc_base {
protected:
    Transmit_buffer& tr_buffer;
    Pipe_fifo& receive_pipe;
    Pipe_fifo& send_pipe;
    char combuffer[PIPE_BUF];
    MainScreen* mainscreen_p;           // this win pointer is for process_rx_frame()
    ReceiveWin* receivewin_p;           // ditto
public:
    virtual void process_received_bytes(void) = 0;
    virtual void send_to_tnc(int) = 0;
    void set_wins(MainScreen* a, ReceiveWin* b) {mainscreen_p = a; receivewin_p = b;}
    Tnc_base(Transmit_buffer& a, Pipe_fifo& b, Pipe_fifo& c):
        tr_buffer(a), receive_pipe(b), send_pipe(c) {}
    virtual ~Tnc_base(void) {} // in case a derived class has a destructor
};

// This class provides the Kam host mode functions when the program is running in 
// normal mode

class Tnc: public Tnc_base {
  // this flag indicates that we have sent the command 'STATUS' to the tnc
  // to check free bytes in the Kam buffer; this flags stops us sending two
  // such requests at the same time
    int check_status_flag;

  // this flag indicates that the frame being/just received is an information
  // (command byte = '?') frame
    int info_requested_flag;

    int rx_frame_complete;
    hostframe rx_frame;  // holds frame currently being received
    struct {int packet, tor;} kam_freebytes;
    int unsent_frame_flag;
    int unsent_file_frame_flag;
    int tx_frame_index;
    int file_frame_index;
    char tx_frame[MAX_FRAMESIZE + 6];  // include a byte for a final translated FEND or FESC in data part of frame
    char file_frame[MAX_FRAMESIZE + 6];  // include a byte for a final translated FEND or FESC in data part of frame
    int CMDflag;
    char* const tx_data;                // points to start of data in tx_frame
    char* const file_data;                // points to start of data in file_frame
    char CMDchar;
    int de_flag;
    int find_call_index;
    int modechange_count;
    int connected_status_count;
    int can_send_packet_file_flag;
    int can_send_tor_file_flag;
    char callbuffer[CALLBUFFER_SIZE + 1];
    void process_rx_frame(void);
    void parse_status(char*);
    void read_state_message(char*, int, int);
    uchar cp437_to_Latin1(uchar);
    uchar Latin1_to_cp437(uchar);
public:
    Tnc_func tnc_func;
    void process_received_bytes(void);
    void send_to_tnc(int);
    void send_file(FileBuffer*, const int, const int);
    int get_packet_freebytes(void) const {return kam_freebytes.packet;}
    int get_tor_freebytes(void) const {return kam_freebytes.tor;}
    int get_active_freebytes(void) const {return tnc_func.active_port && tnc_func.hfmode != Tnc_func::packet ?
					    kam_freebytes.tor : kam_freebytes.packet;}
    int get_hf_freebytes(void) const {return tnc_func.hfmode != Tnc_func::packet ?
					    kam_freebytes.tor : kam_freebytes.packet;}
    void find_freebytes(void);
    void get_info(void);
    int get_unsent_frame_flag(void) const {return unsent_frame_flag;}
    int get_unsent_file_frame_flag(void) const {return unsent_file_frame_flag;}
    void find_call(char);
    int is_validcall(const char*);
    void make_selcall(const char*, char*);
    void send_kamcommand(const char* text, // if ascii_stream == -1 or ascii_port == -1
			                   // the current active port or stream is used
			 char ascii_port = -1, char ascii_stream = -1);
    void send_specialcommand(char);
    Tnc(Transmit_buffer&, Pipe_fifo&, Pipe_fifo&, int, int, int, int, int);
    ~Tnc(void);
};

// This class provides Kam terminal mode functions in order to set up the Kam

class Tnc_setup: public Tnc_base {
public:
    void process_received_bytes(void);
    void send_to_tnc(int);
    Tnc_setup(Transmit_buffer& a, Pipe_fifo& b, Pipe_fifo& c): Tnc_base(a, b, c) {}
};

#endif
