/* Copyright (C) 1999 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#ifndef PIPES_H
#define PIPES_H

#include <unistd.h>
#include <iostream.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>
#include <limits.h>
#include <string.h>
#include "prog_defs.h"

/*

This class provides a simplified front end to unix write() and read()
of fifo pipes.  The constructor of the class may be passed an
enumerator to indicate whether the pipe is to read in non-blocking
mode.

The write(char*) method taking a string must have a string less than
PIPE_BUF characters in length, or it will fail.  With the write(char*,
int) method, taking an array and int arguments, the value of int
(indicating the number of chars in the array to be written) should
usually be less than PIPE_BUF, as POSIX write() guarantees that if it
is less than PIPE_BUF, either all will be written or, if the pipe is
too full, none will be written.  This prevents data interleaving if a
number of independent writes to the fifo are made.  If the pipe is too
full, write will return immediately with 0 if the pipe has been set to
read in non-blocking mode.

PIPE_BUF is defined in limits.h, and is at least 512 bytes, and usually 4096 bytes,
but may be calculated from other files included by limits.h.

Where a pipe is used to communicate between parent and child after a
call to fork(), the pipe will normally be used unidirectionally unless
guards or semaphores are used to prevent a process reading data
intended for the other.  However, because each process inherits its
own duplicate of the file descriptors, this cannot be enforced without
closing the read or write file descriptor for the process for which
the reading or writing is to be prohibited.  This can be done by the
process concerned calling the methods Pipe_fifo::make_writeonly() or
Pipe_fifo::make_readonly() after the fork.  If an attempt is made to
read or write to a descriptor closed in this way, the
Pipe_fifo::read() or Pipe_fifo::write() method will ensure that no
read or write will take place, and instead a -1 will be returned.

*/


class Pipe_fifo {
public:
    enum fifo_mode{block, non_block};
private:
    int read_fd;
    int write_fd;
public:
    void set_read_fds(fd_set* readfds_ptr) {FD_SET(read_fd, readfds_ptr);} // calls FD_SET() to enable use of select()
    int isset_read_fds(fd_set* readfds_ptr) {return FD_ISSET(read_fd, readfds_ptr);} // sees if fildes set in readfds
    void set_write_fds(fd_set* writefds_ptr) {FD_SET(write_fd, writefds_ptr);} // calls FD_SET() to enable use of select()
    int isset_write_fds(fd_set* writefds_ptr) {return FD_ISSET(write_fd, writefds_ptr);} // sees if fildes set in writefds
    int read(char*, int);            // returns number of chars read (0 on empty non-blocking fifo)
    int read(void);                  // returns char (or -1 on empty non-blocking fifo)
    int write(const char*);           // the string must not be longer than PIPE_BUF (defined
                                      // in limits.h) and it returns -1 if it is - otherwise
                                      // it returns the number of chars written.  The string
				      // must be null terminated.
    int write(const char*, int);      // returns the number of chars written
    int write(char item) {return write(&item, 1);} // returns 1 if char written, else 0
    void make_writeonly(void);
    void make_readonly(void);
    int get_read_fd(void) const {return read_fd;}
    int get_write_fd(void) const {return write_fd;}
    Pipe_fifo(fifo_mode = block);
    ~Pipe_fifo(void) {make_writeonly(); make_readonly();}
};

#endif
