/*
 *	6pack driver version 0.4.2, 1999/08/22
 *
 *	This module:
 *		This module is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 * 		This module implements the AX.25 protocol for kernel-based
 *		devices like TTYs. It interfaces between a raw TTY, and the
 *		kernel's AX.25 protocol layers, just like slip.c.
 *		AX.25 needs to be separated from slip.c while slip.c is no
 *		longer a static kernel device since it is a module.
 *
 *	Author: Andreas Knsgen <ajk@ccac.rwth-aachen.de>
 *
 *	Lots of stuff has been taken from mkiss.c, written by
 *	Hans Alblas <hans@esrac.ele.tue.nl>
 *
 *	with the fixes from
 *	
 *	Jonathan (G4KLX)	Fixed to match Linux networking changes - 2.1.15.
 *	Matthias (DG2FEF)       Fixed bug in ax25_close(): dev_lock_wait() was
 *                              called twice, causing a deadlock.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <asm/system.h>
#include <asm/segment.h>
#include <asm/bitops.h>
#include <asm/uaccess.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/interrupt.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/tty.h>
#include <linux/errno.h>
#include <linux/netdevice.h>
#include <linux/major.h>
#include <linux/init.h>

#include <linux/timer.h>

#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>

#include <net/ax25.h>

#include "6pack.h"

#ifdef CONFIG_INET
#include <linux/ip.h>
#include <linux/tcp.h>
#endif

#ifdef MODULE
#define AX25_VERSION    "AX25-MODULAR-NET3.019-NEWTTY"
#define	min(a,b)	(a < b ? a : b)
#else
#define	AX25_VERSION	"AX25-NET3.019-NEWTTY"
#endif

#define NR_6PACK 4
#define SIXP_SERIAL_TYPE_NORMAL 1

struct sixpack_channel {
	int magic;		/* magic word */
	int init;		/* channel exists? */
	struct tty_struct *tty; /* link to tty control structure */
};

typedef struct ax25_ctrl {
	char if_name[8];	/* "sp0\0" .. "sp99999\0"	*/
	struct ax_disp ctrl;	/* 				*/
	struct device  dev;	/* the device			*/
} ax25_ctrl_t;

static ax25_ctrl_t **ax25_ctrls = NULL;

int ax25_maxdev = AX25_MAXDEV;		/* Can be overridden with insmod! */

static struct tty_ldisc	ax_ldisc;
static struct tty_driver sixpack_driver;
static int sixpack_refcount;
static struct tty_struct *sixpack_table[NR_6PACK];
static struct termios *sixpack_termios[NR_6PACK];
static struct termios *sixpack_termios_locked[NR_6PACK];
struct sixpack_channel SIXP_Info[NR_6PACK];

static int ax25_init(struct device *);
static int sixpack_init(void);
static int sixpack_write(struct tty_struct *, int, const unsigned char *, int);
static int tnc_init(struct ax_disp *ax);
static void start_tx_timer(struct ax_disp *);
static void tx_schedule(unsigned long);
static void xmit_on_air(struct ax_disp *);
static int sixpack_encaps(unsigned char *tx_buf, unsigned char *tx_buf_raw, int length, unsigned char tx_delay);
static void sixpack_decaps(struct ax_disp *, unsigned char);

static void decode_prio_command(unsigned char, struct ax_disp *);
static void decode_std_command(unsigned char, struct ax_disp *);
static void decode_data(unsigned char, struct ax_disp *);
static void resync_tnc(unsigned long);


/*---------------------------------------------------------------------------*/

/* Find a free channel, and link in this `tty' line. */
static inline struct ax_disp *ax_alloc(void)
{
	ax25_ctrl_t *axp;
	int i;

	if (ax25_ctrls == NULL)		/* Master array missing ! */
		return NULL;

	for (i = 0; i < ax25_maxdev; i++) {
		axp = ax25_ctrls[i];

		/* Not allocated ? */
		if (axp == NULL)
			break;

		/* Not in use ? */
		if (!test_and_set_bit(AXF_INUSE, &axp->ctrl.flags))
			break;
	}

	/* Sorry, too many, all slots in use */
	if (i >= ax25_maxdev)
		return NULL;

	/* If no channels are available, allocate one */
	if (axp == NULL && (ax25_ctrls[i] = kmalloc(sizeof(ax25_ctrl_t), GFP_KERNEL)) != NULL) {
		axp = ax25_ctrls[i];
		memset(axp, 0, sizeof(ax25_ctrl_t));

		/* Initialize channel control data */
		set_bit(AXF_INUSE, &axp->ctrl.flags);
		sprintf(axp->if_name, "sp%d", i++);
		axp->ctrl.tty      = NULL;
		axp->dev.name      = axp->if_name;
		axp->dev.base_addr = i;
		axp->dev.priv      = (void *)&axp->ctrl;
		axp->dev.next      = NULL;
		axp->dev.init      = ax25_init;
	}

	if (axp != NULL) {
		/*
		 * register device so that it can be ifconfig'ed
		 * ax25_init() will be called as a side-effect
		 * SIDE-EFFECT WARNING: ax25_init() CLEARS axp->ctrl !
		 */
		if (register_netdev(&axp->dev) == 0) {
			/* (Re-)Set the INUSE bit.   Very Important! */
			set_bit(AXF_INUSE, &axp->ctrl.flags);
			axp->ctrl.dev = &axp->dev;
			axp->dev.priv = (void *)&axp->ctrl;

			return &axp->ctrl;
		} else {
			clear_bit(AXF_INUSE,&axp->ctrl.flags);
			printk(KERN_ERR "6pack: ax_alloc() - register_netdev() failure.\n");
		}
	}

	return NULL;
}

/* Free an AX25 channel. */
static inline void ax_free(struct ax_disp *ax)
{
	/* Free all AX25 frame buffers. */
	if (ax->rbuff)
		kfree(ax->rbuff);
	ax->rbuff = NULL;
	if (ax->xbuff)
		kfree(ax->xbuff);
	ax->xbuff = NULL;
	if (!test_and_clear_bit(AXF_INUSE, &ax->flags))
		printk(KERN_ERR "6pack: %s: ax_free for already free unit.\n", ax->dev->name);
}

static void ax_changedmtu(struct ax_disp *ax)
{
	struct device *dev = ax->dev;
	unsigned char *xbuff, *rbuff, *oxbuff, *orbuff;
	int len;
	unsigned long flags;

	len = dev->mtu * 2;

	/*
	 * allow for arrival of larger UDP packets, even if we say not to
	 * also fixes a bug in which SunOS sends 512-byte packets even with
	 * an MSS of 128
	 */
	if (len < 576 * 2)
		len = 576 * 2;

	xbuff = kmalloc(len + 4, GFP_ATOMIC);
	rbuff = kmalloc(len + 4, GFP_ATOMIC);

	if (xbuff == NULL || rbuff == NULL)  {
		printk(KERN_ERR "6pack: %s: unable to grow ax25 buffers, MTU change cancelled.\n",
		       ax->dev->name);
		dev->mtu = ax->mtu;
		if (xbuff != NULL)
			kfree(xbuff);
		if (rbuff != NULL)
			kfree(rbuff);
		return;
	}

	save_flags(flags);
	cli();

	oxbuff    = ax->xbuff;
	ax->xbuff = xbuff;
	orbuff    = ax->rbuff;
	ax->rbuff = rbuff;

	if (ax->xleft) {
		if (ax->xleft <= len) {
			memcpy(ax->xbuff, ax->xhead, ax->xleft);
		} else  {
			ax->xleft = 0;
			ax->tx_dropped++;
		}
	}

	ax->xhead = ax->xbuff;

	if (ax->rcount) {
		if (ax->rcount <= len) {
			memcpy(ax->rbuff, orbuff, ax->rcount);
		} else  {
			ax->rcount = 0;
			ax->rx_over_errors++;
			set_bit(AXF_ERROR, &ax->flags);
		}
	}

	ax->mtu      = dev->mtu + 73;
	ax->buffsize = len;

	restore_flags(flags);

	if (oxbuff != NULL)
		kfree(oxbuff);
	if (orbuff != NULL)
		kfree(orbuff);
}


/* Set the "sending" flag.  This must be atomic, hence the ASM. */
static inline void ax_lock(struct ax_disp *ax)
{
	if (test_and_set_bit(0, (void *)&ax->dev->tbusy))
		printk(KERN_ERR "6pack: %s: trying to lock already locked device!\n", ax->dev->name);
}


/* Clear the "sending" flag.  This must be atomic, hence the ASM. */
static inline void ax_unlock(struct ax_disp *ax)
{
	if (!test_and_clear_bit(0, (void *)&ax->dev->tbusy))
		printk(KERN_ERR "6pack: %s: trying to unlock already unlocked device!\n", ax->dev->name);
}

/* Send one completely decapsulated AX.25 packet to the AX.25 layer. */
static void ax_bump(struct ax_disp *ax)
{
	struct ax_disp *tmp_ax;
	struct sk_buff *skb;
	struct sixpack_channel *sixpack;
	int count;
	
        tmp_ax = ax;

	if (ax->rbuff[0] > 0x0f) {
		if (ax->sixpack != NULL) {
			sixpack = ax->sixpack->tty->driver_data;
			if (sixpack->magic == SIXP_DRIVER_MAGIC)
				tmp_ax = ax->sixpack;
		} 
 	}

	count = ax->rcount;

	if ((skb = dev_alloc_skb(count)) == NULL) {
		printk(KERN_ERR "6pack: %s: memory squeeze, dropping packet.\n", ax->dev->name);
		ax->rx_dropped++;
		return;
	}
	ax->cooked_buf[0] = 0; /* KISS command to indicate a data packet */
	skb->dev      = tmp_ax->dev;
	memcpy(skb_put(skb,count), ax->cooked_buf, count);
	skb->mac.raw  = skb->data;
	skb->protocol = htons(ETH_P_AX25);
	netif_rx(skb);
	tmp_ax->rx_packets++;
}

/* Encapsulate one AX.25 packet and stuff into a TTY queue. */

static void
ax_encaps(struct ax_disp *ax, unsigned char *icp, int len)
{
	unsigned char *p;
	int count;
	struct ax_disp *tmp_ax;
	struct sixpack_channel *sixpack = ax->tty->driver_data;

	if (ax->mtu != ax->dev->mtu + 73)	/* Someone has been ifconfigging */
		ax_changedmtu(ax);

	if (len > ax->mtu) {		/* Sigh, shouldn't occur BUT ... */
		len = ax->mtu;
		printk(KERN_ERR "6pack: %s: truncating oversized transmit packet!\n", ax->dev->name);
		ax->tx_dropped++;
		ax_unlock(ax);
		return;
	}

	p = icp;

	if (p[0] > 5)
	{
		printk(KERN_DEBUG "%s: invalid KISS command -- dropped\n", ax->dev->name);
		ax_unlock(ax);
		return;
	}

	if (sixpack->magic != SIXP_DRIVER_MAGIC)
		tmp_ax = ax;
	else
		tmp_ax = ax->sixpack;

	if ((p[0] != 0) && (len > 2))
	{
		printk(KERN_DEBUG "%s: KISS command packet too long -- dropped\n", ax->dev->name);
		ax_unlock(ax);
		return;
	}

	if ((p[0] == 0) && (len < 15))
	{
		printk(KERN_ERR "%s: bad AX.25 packet to transmit -- dropped\n", ax->dev->name);
		ax->tx_dropped++;
		ax_unlock(ax);
		return;
	}

	switch(p[0])
	{
		case 1:	tmp_ax->tx_delay = p[1];	break;
		case 2:	tmp_ax->persistance = p[1];	break;
		case 3: tmp_ax->slottime = p[1];	break;
		case 4: printk(KERN_INFO "6pack: got KISS command 4 -- ignored\n");
			break;
		case 5: tmp_ax->duplex = p[1];		break;
	}

	if (p[0] != 0) {
		ax_unlock(ax);
		return;
	}
	
	count = sixpack_encaps(p, (unsigned char *) tmp_ax->xbuff, len, tmp_ax->tx_delay);
	tmp_ax->xleft = count;
	tmp_ax->status2 = count;
	tmp_ax->xhead = tmp_ax->xbuff;
	
	/* in case of DAMA or fullduplex operation, we don't take care
	   about the state of the DCD or of any timers, as the determination
	   of the correct time to send is the job of the AX.25 layer. We send
	   immediately after data has arrived. */

	if (tmp_ax->duplex == 1) { /* DAMA / fullduplex */
		xmit_on_air(tmp_ax);
	} else { /* CSMA */
		start_tx_timer(tmp_ax);
	} /* if ax->duplex */
} /* ax_encaps */

/* the next two functions perform the persistence/slottime algorithm for CSMA access.
   If the persistence check was successful, write the data to the serial driver.
   Note that in case of DAMA operation, the data is not sent here. */

/* set timer to time out slottime */
static void 
start_tx_timer(struct ax_disp *ax)
{
	del_timer(&(ax->tx_t));
	ax->tx_t.data = (unsigned long) ax;
	ax->tx_t.function = tx_schedule;
	ax->tx_t.expires = jiffies + (((ax->slottime)+1)*HZ)/100;
	add_timer(&(ax->tx_t));
}

/* compute random number and check if it is ok to send */
static void tx_schedule(unsigned long channel)
{
	struct ax_disp *ax = (struct ax_disp *) channel;
	static unsigned char random;
	
	random = random * 17 + 41;

	if (((ax->status1 & SIXP_DCD_MASK) == 0) && (random < ax->persistance)) 
		xmit_on_air(ax);
	else
		start_tx_timer(ax);
}

static void
xmit_on_air(struct ax_disp *ax)
{
	int actual;

	ax->tty->flags |= (1 << TTY_DO_WRITE_WAKEUP);
	ax->led_state = SIXP_STA_LED_ON;
	ax->tty->driver.write(ax->tty, 0, &(ax->led_state), 1);
	ax->tx_enable = 1;
	actual = ax->tty->driver.write(ax->tty, 0, ax->xbuff, ax->status2);
	ax->tx_packets++;
	ax->dev->trans_start = jiffies;
	ax->xleft -= actual;
	ax->xhead = ax->xbuff + actual;
	ax->led_state = SIXP_LED_OFF;
	ax->tty->driver.write(ax->tty, 0, &(ax->led_state), 1);
} /* xmit_on_air */

/*
 * Called by the driver when there's room for more data.  If we have
 * more packets to send, we send them here.
 */
static void ax25_write_wakeup(struct tty_struct *tty)
{
	int actual;
	struct ax_disp *ax = (struct ax_disp *)tty->disc_data;
	struct sixpack_channel *sixpack;

	/* First make sure we're connected. */
	if (ax == NULL || ax->magic != AX25_MAGIC || !ax->dev->start)
		return;
	if (ax->xleft <= 0)  {
		/* Now serial buffer is almost free & we can start
		 * transmission of another packet
		 */
		tty->flags &= ~(1 << TTY_DO_WRITE_WAKEUP);

		if (ax->sixpack != NULL) {
			sixpack= ax->sixpack->tty->driver_data;
	        	if (sixpack->magic  == SIXP_DRIVER_MAGIC)
				ax_unlock(ax->sixpack);
	        }

		ax_unlock(ax);
		mark_bh(NET_BH);
		return;
	}

	actual = tty->driver.write(tty, 0, ax->xhead, ax->xleft);
	ax->xleft -= actual;
	ax->xhead += actual;
}

/* Encapsulate an AX.25 packet and kick it into a TTY queue. */
static int ax_xmit(struct sk_buff *skb, struct device *dev)
{
	struct ax_disp *ax = (struct ax_disp*)dev->priv;
	struct sixpack_channel *sixpack = ax->tty->driver_data;
	struct ax_disp *tmp_ax;

	tmp_ax = NULL;

	if (sixpack->magic  == SIXP_DRIVER_MAGIC) {
		if (skb->data[0] < 0x10)
			skb->data[0] = skb->data[0] + 0x10;
		tmp_ax = ax->sixpack;
	}

	if (!dev->start)  {
		printk(KERN_ERR "6pack: %s: xmit call when iface is down\n", dev->name);
		return 1;
	}

	if (tmp_ax != NULL)
		if (tmp_ax->dev->tbusy)
			return 1;

	if (tmp_ax != NULL)
		if (dev->tbusy) {
			printk(KERN_ERR "6pack: dev busy while serial dev is free\n");
			ax_unlock(ax);
	        }

	if (dev->tbusy) {
		/*
		 * May be we must check transmitter timeout here ?
		 *      14 Oct 1994 Dmitry Gorodchanin.
		 */
		if (jiffies - dev->trans_start  < 20 * HZ) {
			/* 20 sec timeout not reached */
			return 1;
		}

		printk(KERN_ERR "6pack: %s: transmit timed out, %s?\n", dev->name,
		       (ax->tty->driver.chars_in_buffer(ax->tty) || ax->xleft) ?
		       "bad line quality" : "driver error");

		ax->xleft = 0;
		ax->tty->flags &= ~(1 << TTY_DO_WRITE_WAKEUP);
		ax_unlock(ax);
	}

	/* We were not busy, so we are now... :-) */
	if (skb != NULL) {
		ax_lock(ax);
		if (tmp_ax != NULL)
			ax_lock(tmp_ax);
		ax_encaps(ax, skb->data, skb->len);
		kfree_skb(skb);
	}

	return 0;
}

#if defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)

/* Return the frame type ID */
static int ax_header(struct sk_buff *skb, struct device *dev, unsigned short type,
	  void *daddr, void *saddr, unsigned len)
{
#ifdef CONFIG_INET
	if (type != htons(ETH_P_AX25))
		return ax25_encapsulate(skb, dev, type, daddr, saddr, len);
#endif
	return 0;
}


static int ax_rebuild_header(struct sk_buff *skb)
{
#ifdef CONFIG_INET
	return ax25_rebuild_header(skb);
#else
	return 0;
#endif
}

#endif	/* CONFIG_{AX25,AX25_MODULE} */

/* Open the low-level part of the AX25 channel. Easy! */
static int ax_open(struct device *dev)
{
	struct ax_disp *ax = (struct ax_disp*)dev->priv;
	unsigned long len;

	if (ax->tty == NULL)
		return -ENODEV;

	/*
	 * Allocate the frame buffers:
	 *
	 * rbuff	Receive buffer.
	 * xbuff	Transmit buffer.
	 * cbuff        Temporary compression buffer.
	 */
	len = dev->mtu * 2;

	/*
	 * allow for arrival of larger UDP packets, even if we say not to
	 * also fixes a bug in which SunOS sends 512-byte packets even with
	 * an MSS of 128
	 */
	if (len < 576 * 2)
		len = 576 * 2;

	if ((ax->rbuff = kmalloc(len + 4, GFP_KERNEL)) == NULL)
		goto norbuff;

	if ((ax->xbuff = kmalloc(len + 4, GFP_KERNEL)) == NULL)
		goto noxbuff;

	ax->mtu	     = dev->mtu + 73;
	ax->buffsize = len;
	ax->rcount   = 0;
	ax->xleft    = 0;

	ax->flags   &= (1 << AXF_INUSE);      /* Clear ESCAPE & ERROR flags */
	dev->tbusy  = 0;
	dev->start  = 1;

	ax->duplex = 0;
	ax->tx_delay    = SIXP_TXDELAY;
	ax->persistance = SIXP_PERSIST;
	ax->slottime    = SIXP_SLOTTIME;

	ax->led_state   = SIXP_LED_OFF;
	ax->status      = 1;
	ax->status1     = 1;
	ax->status2     = 0;
	ax->tnc_ok      = 0;
	ax->tx_enable   = 0;
	
	init_timer(&ax->tx_t);
	init_timer(&ax->resync_t);
	
	return 0;

	/* Cleanup */
	kfree(ax->xbuff);

noxbuff:
	kfree(ax->rbuff);

norbuff:
	return -ENOMEM;
}


/* Close the low-level part of the AX25 channel. Easy! */
static int ax_close(struct device *dev)
{
	struct ax_disp *ax = (struct ax_disp*)dev->priv;

	if (ax->tty == NULL)
		return -EBUSY;

	del_timer(&(ax->tx_t));
	del_timer(&(ax->resync_t));

	ax->tty->flags &= ~(1 << TTY_DO_WRITE_WAKEUP);

	dev->tbusy = 1;
	dev->start = 0;

	return 0;
}

static int ax25_receive_room(struct tty_struct *tty)
{
	return 65536;  /* We can handle an infinite amount of data. :-) */
}

/*
 * Handle the 'receiver data ready' interrupt.
 * This function is called by the 'tty_io' module in the kernel when
 * a block of data has been received, which can now be decapsulated
 * and sent on to the AX.25 layer for further processing.
 */
static void ax25_receive_buf(struct tty_struct *tty, const unsigned char *cp, char *fp, int count)
{
	struct ax_disp *ax = (struct ax_disp *)tty->disc_data;
	
	if (ax == NULL || ax->magic != AX25_MAGIC || !ax->dev->start)
		return;

	/*
	 * Argh! mtu change time! - costs us the packet part received
	 * at the change
	 */
	if (ax->mtu != ax->dev->mtu + 73)
		ax_changedmtu(ax);

	/* Read the characters out of the buffer */
	while (count--) {
		if (fp != NULL && *fp++) {
			if (!test_and_set_bit(AXF_ERROR, &ax->flags))
				ax->rx_errors++;
			cp++;
			continue;
		}	
		sixpack_decaps(ax, *cp++);
	}
}

static int ax25_open(struct tty_struct *tty)
{
	struct ax_disp *ax = (struct ax_disp *)tty->disc_data;
	struct ax_disp *tmp_ax;
	struct sixpack_channel *sixpack;
	int err, cnt;

	/* First make sure we're not already connected. */
	if (ax && ax->magic == AX25_MAGIC)
		return -EEXIST;

	/* OK.  Find a free AX25 channel to use. */
	if ((ax = ax_alloc()) == NULL)
		return -ENFILE;

	ax->tty = tty;
	tty->disc_data = ax;

	ax->sixpack = NULL;
	tmp_ax    = NULL;

	if (tty->driver.flush_buffer)
		tty->driver.flush_buffer(tty);
	if (tty->ldisc.flush_buffer)
		tty->ldisc.flush_buffer(tty);

	/* Restore default settings */
	ax->dev->type = ARPHRD_AX25;

	/* Perform the low-level AX25 initialization. */
	if ((err = ax_open(ax->dev)))
		return err;

	sixpack= ax->tty->driver_data;

	if (sixpack->magic  == SIXP_DRIVER_MAGIC) {
		for (cnt = 1; cnt < ax25_maxdev; cnt++) {
			if (ax25_ctrls[cnt]) {
				if (ax25_ctrls[cnt]->dev.start) {
					if (ax == &ax25_ctrls[cnt]->ctrl) {
						cnt--;
						tmp_ax = &ax25_ctrls[cnt]->ctrl;
						break;
					}
				}
			}
		}
	}

	if (tmp_ax != NULL) {
		ax->sixpack     = tmp_ax;
		tmp_ax->sixpack = ax;
	}

	MOD_INC_USE_COUNT;

	tnc_init(ax);

	/* Done.  We have linked the TTY line to a channel. */
	return ax->dev->base_addr;
}

static void ax25_close(struct tty_struct *tty)
{
	struct ax_disp *ax = (struct ax_disp *)tty->disc_data;
	int sixpack ;

	/* First make sure we're connected. */
	if (ax == NULL || ax->magic != AX25_MAGIC)
		return;

	sixpack = ax->mode;

	dev_close(ax->dev);

	tty->disc_data = 0;
	ax->tty        = NULL;

	/* VSV = very important to remove timers */
	ax_free(ax);
	unregister_netdev(ax->dev);

	MOD_DEC_USE_COUNT;
}


static struct net_device_stats *ax_get_stats(struct device *dev)
{
	static struct net_device_stats stats;
	struct ax_disp *ax = (struct ax_disp*)dev->priv;

	memset(&stats, 0, sizeof(struct net_device_stats));

	stats.rx_packets     = ax->rx_packets;
	stats.tx_packets     = ax->tx_packets;
	stats.rx_dropped     = ax->rx_dropped;
	stats.tx_dropped     = ax->tx_dropped;
	stats.tx_errors      = ax->tx_errors;
	stats.rx_errors      = ax->rx_errors;
	stats.rx_over_errors = ax->rx_over_errors;

	return &stats;
}


static int
tnc_init(struct ax_disp *ax)
{
	static unsigned char inbyte;
	
	inbyte = SIXP_INIT_CMD;
	ax->tty->driver.write(ax->tty, 0, &inbyte, 1);

	del_timer(&(ax->resync_t));
	ax->resync_t.data = (unsigned long) ax;
	ax->resync_t.function = resync_tnc;
	ax->resync_t.expires = jiffies + SIXP_RESYNC_TIMEOUT;
	add_timer(&(ax->resync_t));

	return 0;
}


/************************************************************************
 *			   STANDARD ENCAPSULATION	        	 *
 ************************************************************************/

/* encode an AX.25 packet into 6pack */

static int sixpack_encaps(unsigned char *tx_buf, unsigned char *tx_buf_raw, int length, unsigned char tx_delay)
{
	int count = 0;
	unsigned char checksum = 0, buf[400];
	int raw_count = 0;

	tx_buf_raw[raw_count++] = SIXP_PRIO_CMD_MASK | SIXP_TX_MASK;
	tx_buf_raw[raw_count++] = SIXP_SEOF;

	buf[0] = tx_delay;
	for(count = 1; count < length; count++)
		buf[count] = tx_buf[count];

	for(count = 0; count < length; count++)
		checksum += buf[count];
	buf[length] = (unsigned char)0xff - checksum;
	
	for(count = 0; count <= length; count++) {
		if((count % 3) == 0) {
			tx_buf_raw[raw_count++] = (buf[count] & 0x3f);
			tx_buf_raw[raw_count] = ((buf[count] >> 2) & 0x30);
		}
		else if((count % 3) == 1) {
			tx_buf_raw[raw_count++] |= (buf[count] & 0x0f);
			tx_buf_raw[raw_count] =
				((buf[count] >> 2) & 0x3c);
		} else {
			tx_buf_raw[raw_count++] |= (buf[count] & 0x03);
			tx_buf_raw[raw_count++] =
				(buf[count] >> 2);
		} /* else */
	} /* for */
	if ((length % 3) != 2)
		raw_count++;
	tx_buf_raw[raw_count++] = SIXP_SEOF;
	return(raw_count);
}

static void sixpack_decaps(struct ax_disp *ax, unsigned char inbyte)
{
	if (inbyte == SIXP_TNC_FOUND) {
		printk(KERN_INFO "6pack: TNC found.\n");
		ax->tnc_ok = TRUE;
		del_timer(&(ax->resync_t));
	}
	if((inbyte & SIXP_PRIO_CMD_MASK) != 0)
		decode_prio_command(inbyte, ax);
	else if((inbyte & SIXP_STD_CMD_MASK) != 0)
		decode_std_command(inbyte, ax);
	else {
		if ((ax->status & SIXP_RX_DCD_MASK) == SIXP_RX_DCD_MASK)
			decode_data(inbyte, ax);
	} /* else */
}

/* identify and execute a 6pack priority command byte */

void decode_prio_command(unsigned char cmd, struct ax_disp *ax)
{
	unsigned char channel;

	channel = cmd & SIXP_CHN_MASK;
	if ((cmd & SIXP_PRIO_DATA_MASK) != 0) {     /* idle ? */

	/* RX and DCD flags can only be set in the same prio command,
	   if the DCD flag has been set without the RX flag in the previous
	   prio command. If DCD has not been set before, something in the
	   transmission has gone wrong. In this case, RX and DCD are
	   cleared in order to prevent the decode_data routine from
	   reading further data that might be corrupt. */

		if (((ax->status & SIXP_DCD_MASK) == 0) &&
			((cmd & SIXP_RX_DCD_MASK) == SIXP_RX_DCD_MASK)) {
				if (ax->status != 1)
					printk(KERN_DEBUG "6pack: protocol violation\n");
				else
					ax->status = 0;
				cmd &= !SIXP_RX_DCD_MASK;
		}
		ax->status = cmd & SIXP_PRIO_DATA_MASK;
	} /* if */

	/* needed to trigger the TNC watchdog */
	ax->tty->driver.write(ax->tty, 0, &(ax->led_state), 1);

        /* if the state byte has been received, the TNC is present,
           so the resync timer can be reset. */

	if (ax->tnc_ok == 1) {
		del_timer(&(ax->resync_t));
		ax->resync_t.data = (unsigned long) ax;
		ax->resync_t.function = resync_tnc;
		ax->resync_t.expires = jiffies + SIXP_INIT_RESYNC_TIMEOUT;
		add_timer(&(ax->resync_t));
	}

	ax->status1 = cmd & SIXP_PRIO_DATA_MASK;
}

/* try to resync the TNC. Called by the resync timer defined in
  decode_prio_command */

static void
resync_tnc(unsigned long channel)
{
	static char resync_cmd = SIXP_INIT_CMD;
	struct ax_disp *ax = (struct ax_disp *) channel;

	printk(KERN_INFO "6pack: resyncing TNC\n");

	/* clear any data that might have been received */
	
	ax->rx_count = 0;
	ax->rx_count_cooked = 0;

	/* reset state machine */

	ax->status = 1;
	ax->status1 = 1;
	ax->status2 = 0;
	ax->tnc_ok = 0;
	
	/* resync the TNC */

	ax->led_state = SIXP_LED_OFF;
	ax->tty->driver.write(ax->tty, 0, &(ax->led_state), 1);
	ax->tty->driver.write(ax->tty, 0, &resync_cmd, 1);


	/* Start resync timer again -- the TNC might be still absent */

	del_timer(&(ax->resync_t));
	ax->resync_t.data = (unsigned long) ax;
	ax->resync_t.function = resync_tnc;
	ax->resync_t.expires = jiffies + SIXP_RESYNC_TIMEOUT;
	add_timer(&(ax->resync_t));
}



/* identify and execute a standard 6pack command byte */

void decode_std_command(unsigned char cmd, struct ax_disp *ax)
{
	unsigned char checksum = 0, channel;
	short i;

	channel = cmd & SIXP_CHN_MASK;
	switch(cmd & SIXP_CMD_MASK) {     /* normal command */
		case SIXP_SEOF:
			if ((ax->rx_count == 0) && (ax->rx_count_cooked == 0)) {
				if ((ax->status & SIXP_RX_DCD_MASK) ==
					SIXP_RX_DCD_MASK) {
					ax->led_state = SIXP_CON_LED_ON;
					ax->tty->driver.write(ax->tty, 0, &(ax->led_state), 1);
				} /* if */
			} else {
				ax->led_state = SIXP_LED_OFF;
				ax->tty->driver.write(ax->tty, 0, &(ax->led_state), 1);
				/* fill trailing bytes with zeroes */
				if (ax->rx_count == 2) {
					decode_data(0, ax);
					decode_data(0, ax);
					ax->rx_count_cooked -= 2;
				}
				else if (ax->rx_count == 3) {
					decode_data(0, ax);
					ax->rx_count_cooked -= 1;
				}
				for (i=0; i< ax->rx_count_cooked; i++)
					checksum += ax->cooked_buf[i];
				if (checksum != SIXP_CHKSUM) {
					printk(KERN_DEBUG "6pack: bad checksum %2.2x\n", checksum);
				} else {
					ax->rcount = ax->rx_count_cooked-1;
					ax_bump(ax);
				} /* else */
				ax->rx_count_cooked = 0;
			} /* else */
			break;
		case SIXP_TX_URUN:
			printk(KERN_DEBUG "6pack: TX underrun\n");
			break;
		case SIXP_RX_ORUN:
			printk(KERN_DEBUG "6pack: RX overrun\n");
			break;
		case SIXP_RX_BUF_OVL:
			printk(KERN_DEBUG "6pack: RX buffer overflow\n");
	} /* switch */
} /* function */

/* decode 4 sixpack-encoded bytes into 3 data bytes */

void decode_data(unsigned char inbyte, struct ax_disp *ax)
{

	unsigned char *buf;

	if (ax->rx_count != 3)
		ax->raw_buf[ax->rx_count++] = inbyte;
	else {
		buf = ax->raw_buf;
		ax->cooked_buf[ax->rx_count_cooked++] =
			buf[0] | ((buf[1] << 2) & 0xc0);
		ax->cooked_buf[ax->rx_count_cooked++] =
			(buf[1] & 0x0f) | ((buf[2] << 2) & 0xf0);
		ax->cooked_buf[ax->rx_count_cooked++] =
			(buf[2] & 0x03) | (inbyte << 2);
		ax->rx_count = 0;
	}
}

int ax_set_mac_address(struct device *dev, void *addr)
{
	if (copy_from_user(dev->dev_addr, addr, AX25_ADDR_LEN))
		return -EFAULT;
	return 0;
}

static int ax_set_dev_mac_address(struct device *dev, void *addr)
{
	struct sockaddr *sa = addr;

	memcpy(dev->dev_addr, sa->sa_data, AX25_ADDR_LEN);

	return 0;
}


/* Perform I/O control on an active ax25 channel. */
static int ax25_disp_ioctl(struct tty_struct *tty, void *file, int cmd, void *arg)
{
	struct ax_disp *ax = (struct ax_disp *)tty->disc_data;
	unsigned int tmp;

	/* First make sure we're connected. */
	if (ax == NULL || ax->magic != AX25_MAGIC)
		return -EINVAL;

	switch (cmd) {
	 	case SIOCGIFNAME:
			if (copy_to_user(arg, ax->dev->name, strlen(ax->dev->name) + 1))
				return -EFAULT;
			return 0;

		case SIOCGIFENCAP:
			put_user(4, (int *)arg);
			return 0;

		case SIOCSIFENCAP:
			get_user(tmp, (int *)arg);
	 		ax->mode = tmp;
			ax->dev->addr_len        = AX25_ADDR_LEN;	  /* sizeof an AX.25 addr */
			ax->dev->hard_header_len = AX25_KISS_HEADER_LEN + AX25_MAX_HEADER_LEN + 3;
			ax->dev->type            = ARPHRD_AX25;
			return 0;

		 case SIOCSIFHWADDR:
			return ax_set_mac_address(ax->dev, arg);

		default:
			return -ENOIOCTLCMD;
	}
}

static int ax_open_dev(struct device *dev)
{
	struct ax_disp *ax = (struct ax_disp*)dev->priv;

	if (ax->tty==NULL)
		return -ENODEV;

	return 0;
}

/* Initialize AX25 control device -- register AX25 line discipline */
__initfunc(int sixpack_init_ctrl_dev(void))
{
	int status;

	if (ax25_maxdev < 4) ax25_maxdev = 4; /* Sanity */

	if ((ax25_ctrls = kmalloc(sizeof(void*) * ax25_maxdev, GFP_KERNEL)) == NULL) {
		printk(KERN_ERR "6pack: Can't allocate ax25_ctrls[] array !  No sixpack available\n");
		return -ENOMEM;
	}

	/* Clear the pointer array, we allocate devices when we need them */
	memset(ax25_ctrls, 0, sizeof(void*) * ax25_maxdev); /* Pointers */

	/* Fill in our line protocol discipline, and register it */
	memset(&ax_ldisc, 0, sizeof(ax_ldisc));
	ax_ldisc.magic  = TTY_LDISC_MAGIC;
	ax_ldisc.name   = "6pack";
	ax_ldisc.flags  = 0;
	ax_ldisc.open   = ax25_open;
	ax_ldisc.close  = ax25_close;
	ax_ldisc.read   = NULL;
	ax_ldisc.write  = NULL;
	ax_ldisc.ioctl  = (int (*)(struct tty_struct *, struct file *, unsigned int, unsigned long))ax25_disp_ioctl;
	ax_ldisc.poll   = NULL;

	ax_ldisc.receive_buf  = ax25_receive_buf;
	ax_ldisc.receive_room = ax25_receive_room;
	ax_ldisc.write_wakeup = ax25_write_wakeup;

	if ((status = tty_register_ldisc(N_6PACK, &ax_ldisc)) != 0)
		printk(KERN_ERR "6pack: can't register line discipline (err = %d)\n", status);

	sixpack_init();

#ifdef MODULE
	return status;
#else
	/*
	 * Return "not found", so that dev_init() will unlink
	 * the placeholder device entry for us.
	 */
	return ENODEV;
#endif
}


/* Initialize the driver.  Called by network startup. */

static int ax25_init(struct device *dev)
{
	struct ax_disp *ax = (struct ax_disp*)dev->priv;

	static char ax25_bcast[AX25_ADDR_LEN] =
		{'Q'<<1,'S'<<1,'T'<<1,' '<<1,' '<<1,' '<<1,'0'<<1};
	static char ax25_test[AX25_ADDR_LEN] =
		{'L'<<1,'I'<<1,'N'<<1,'U'<<1,'X'<<1,' '<<1,'1'<<1};

	if (ax == NULL)		/* Allocation failed ?? */
		return -ENODEV;

	/* Set up the "AX25 Control Block". (And clear statistics) */
	memset(ax, 0, sizeof (struct ax_disp));
	ax->magic  = AX25_MAGIC;
	ax->dev	   = dev;

	/* Finish setting up the DEVICE info. */
	dev->mtu             = AX_MTU;
	dev->hard_start_xmit = ax_xmit;
	dev->open            = ax_open_dev;
	dev->stop            = ax_close;
	dev->get_stats	     = ax_get_stats;
#ifdef HAVE_SET_MAC_ADDR
	dev->set_mac_address = ax_set_dev_mac_address;
#endif
	dev->hard_header_len = 0;
	dev->addr_len        = 0;
	dev->type            = ARPHRD_AX25;
	dev->tx_queue_len    = 10;

	memcpy(dev->broadcast, ax25_bcast, AX25_ADDR_LEN);
	memcpy(dev->dev_addr,  ax25_test,  AX25_ADDR_LEN);

#if defined(CONFIG_AX25) || defined(CONFIG_AX25_MODULE)
	dev->hard_header     = ax_header;
	dev->rebuild_header  = ax_rebuild_header;
#endif

	dev_init_buffers(dev);

	/* New-style flags. */
	dev->flags      = 0;

	return 0;
}

static int sixpack_open(struct tty_struct *tty, struct file *filp)
{
	struct sixpack_channel *sixpack;
	int chan;

	chan = MINOR(tty->device) - tty->driver.minor_start;

	if (chan < 0 || chan >= NR_6PACK)
		return -ENODEV;

	sixpack = &SIXP_Info[chan];

	sixpack->magic = SIXP_DRIVER_MAGIC;
	sixpack->init  = 1;
	sixpack->tty   = tty;

	tty->driver_data = sixpack;

	tty->termios->c_iflag  = IGNBRK | IGNPAR;
	tty->termios->c_cflag  = B9600 | CS8 | CLOCAL;
	tty->termios->c_cflag &= ~CBAUD;

	return 0;
}

static void sixpack_close(struct tty_struct *tty, struct file * filp)
{
	struct sixpack_channel *sixpack = tty->driver_data;

	if (sixpack == NULL || sixpack->magic != SIXP_DRIVER_MAGIC)
                return;

	sixpack->tty   = NULL;
	sixpack->init  = 0;
	tty->stopped = 0;
}

static int sixpack_write(struct tty_struct *tty, int from_user, const unsigned char *buf, int count)
{
	return 0;
}

static int sixpack_ioctl(struct tty_struct *tty, struct file *file, unsigned int cmd, unsigned long arg)
{
	/* Ignore serial ioctl's */
	switch (cmd) {
		case TCSBRK:
		case TIOCMGET:
		case TIOCMBIS:
		case TIOCMBIC:
		case TIOCMSET:
		case TCSETS:
		case TCSETSF:		/* should flush first, but... */
		case TCSETSW:		/* should wait until flush, but... */
			return 0;
		default:
			return -ENOIOCTLCMD;
	}
}


static void sixpack_dummy(struct tty_struct *tty)
{
	struct sixpack_channel *sixpack = tty->driver_data;

	if (tty == NULL)
		return;

	if (sixpack == NULL)
		return;
}

static void sixpack_dummy2(struct tty_struct *tty, unsigned char ch)
{
	struct sixpack_channel *sixpack = tty->driver_data;

	if (tty == NULL)
		return;

	if (sixpack == NULL)
		return;
}


static int sixpack_write_room(struct tty_struct * tty)
{
	struct sixpack_channel *sixpack = tty->driver_data;

	if (tty == NULL)
		return 0;

	if (sixpack == NULL)
		return 0;

	return 65536;  /* We can handle an infinite amount of data. :-) */
}


static int sixpack_chars_in_buffer(struct tty_struct *tty)
{
	struct sixpack_channel *sixpack = tty->driver_data;

	if (tty == NULL)
		return 0;

	if (sixpack == NULL)
		return 0;

	return 0;
}


static void sixpack_set_termios(struct tty_struct *tty, struct termios *old_termios)
{
	/* we don't do termios */
}

/* ******************************************************************** */
/* * 			Init 6pack driver 			      * */
/* ******************************************************************** */

__initfunc(static int sixpack_init(void))
{
	memset(&sixpack_driver, 0, sizeof(struct tty_driver));

	sixpack_driver.magic       = SIXP_DRIVER_MAGIC;
	sixpack_driver.name        = "6pack";
	sixpack_driver.major       = SIXP_MAJOR;
	sixpack_driver.minor_start = 0;
	sixpack_driver.num         = NR_6PACK;
	sixpack_driver.type        = TTY_DRIVER_TYPE_SERIAL;
	sixpack_driver.subtype     = SIXP_SERIAL_TYPE_NORMAL;	/* not needed */

	sixpack_driver.init_termios         = tty_std_termios;
	sixpack_driver.init_termios.c_iflag = IGNBRK | IGNPAR;
	sixpack_driver.init_termios.c_cflag = B9600 | CS8 | CLOCAL;

	sixpack_driver.flags           = TTY_DRIVER_REAL_RAW;
	sixpack_driver.refcount        = &sixpack_refcount;
	sixpack_driver.table           = sixpack_table;
	sixpack_driver.termios         = (struct termios **)sixpack_termios;
	sixpack_driver.termios_locked  = (struct termios **)sixpack_termios_locked;

	sixpack_driver.ioctl           = sixpack_ioctl;
	sixpack_driver.open            = sixpack_open;
	sixpack_driver.close           = sixpack_close;
	sixpack_driver.write           = sixpack_write;
	sixpack_driver.write_room      = sixpack_write_room;
	sixpack_driver.chars_in_buffer = sixpack_chars_in_buffer;
	sixpack_driver.set_termios     = sixpack_set_termios;

	/* some unused functions */
	sixpack_driver.flush_buffer = sixpack_dummy;
	sixpack_driver.throttle     = sixpack_dummy;
	sixpack_driver.unthrottle   = sixpack_dummy;
	sixpack_driver.stop         = sixpack_dummy;
	sixpack_driver.start        = sixpack_dummy;
	sixpack_driver.hangup       = sixpack_dummy;
	sixpack_driver.flush_chars  = sixpack_dummy;
	sixpack_driver.put_char     = sixpack_dummy2;

	if (tty_register_driver(&sixpack_driver)) {
		printk(KERN_ERR "6pack: couldn't register Mkiss device\n");
		return -EIO;
	}

	printk(KERN_INFO "AX.25 6pack device version 0.4.2 enabled\n");

	return 0;
}

#ifdef MODULE
EXPORT_NO_SYMBOLS;

MODULE_PARM(ax25_maxdev, "i");
MODULE_PARM_DESC(ax25_maxdev, "number of 6pack devices");

MODULE_AUTHOR("Andreas Knsgen DG3KQ <ajk@ccac.rwth-aachen.de>");
MODULE_DESCRIPTION("6pack driver for AX.25 over TTYs");

int init_module(void)
{
	return sixpack_init_ctrl_dev();
}

void cleanup_module(void)
{
	int i;

	if (ax25_ctrls != NULL) {
		for (i = 0; i < ax25_maxdev; i++) {
			if (ax25_ctrls[i]) {
				/*
				 * VSV = if dev->start==0, then device
				 * unregistred while close proc.
				 */
				if (ax25_ctrls[i]->dev.start)
					unregister_netdev(&(ax25_ctrls[i]->dev));

				kfree(ax25_ctrls[i]);
				ax25_ctrls[i] = NULL;
			}
		}

		kfree(ax25_ctrls);
		ax25_ctrls = NULL;
	}

	if ((i = tty_register_ldisc(N_6PACK, NULL)))
		printk(KERN_ERR "6pack: can't unregister line discipline (err = %d)\n", i);

	if (tty_unregister_driver(&sixpack_driver))	/* remove devive */
		printk(KERN_ERR "6pack: can't unregister 6pack device\n");
}

#endif /* MODULE */
