
//////////////////////////////////////////////////////////////
//
// FBB driver for TNC2 in WA8DED host mode
//
// File : fbb_ded.dll
//
// (C) F6FBB 1999-2000
//
//////////////////////////////////////////////////////////////
//
//  Version history
//
//	    - 06/01/2001 - EB5AGF
//		Added version resource and code to read it at runtime
//
//////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <stdio.h>
#include <winver.h>
#include "../fbb_drv.h"

#define uchar unsigned char

#define	FBBDLLNAME	"ded.dll"

//------------

int DED_MAJOR = 0;
int DED_MINOR = 0;

typedef struct DATAOUT_t {
	char	*pData;
	int		nChannel;
	int		nType;
	int		nLen;
	struct	DATAOUT_t *pNext;
} DATAOUT;

typedef struct DATAIN_t {
	char	*pData;
	int		nChannel;
	int		nType;
	int		nLen;
	DRVUI	*pUi;
	struct	DATAIN_t *pNext;
} DATAIN;

struct	FBBPORT_t;

typedef struct FBBDRV_t
{
	HWND	hWnd;
	int		nPort;
	int		nNbChan;
	int		nNbMultCh;
	int		nLastCmd;
	int		nNbResync;
	int		nNbInit;
	BOOL	bNext;
	uchar	*nNbFrm;
	uchar	*nNbRet;
	uchar	*nState;
	char	szMyCall[10];
	DRVUI	*pUi;
	DATAOUT	*pTncHead;	// Head of data_out list
	DATAOUT	*pTncTail;	// Tail of data_out list
	DATAIN	*pHostHead;	// Head of data_in list
	DATAIN	*pHostTail;	// Tail of data_in list
	struct	FBBPORT_t	*pPort;
	struct	FBBDRV_t	*pNext;
} FBBDRV;

typedef struct FBBPORT_t
{
	HANDLE	hDev;
	HANDLE	hThread;
	DWORD	dwThreadId;
	BOOL	bUp;
	int		nWait;
	int		nCom;
	int		nCurTnc;
	int		nNbTnc;
	struct	FBBDRV_t	*pTnc[4];
	CRITICAL_SECTION	hCritical;
} FBBPORT;


//////////////////////////////////////////////////////////////
// Local prototypes
//////////////////////////////////////////////////////////////

static BOOL SetupConnection(HANDLE hDev, int nBaudRate);
static DWORD WINAPI DedLoop( LPSTR lpData );
static FBBDRV *SearchPort(int nPort);
static BOOL StartHostMode(FBBDRV *pFbbDrv);
static BOOL EndHostMode(FBBDRV *pFbbDrv);
static BOOL WriteTncBlock(FBBDRV *pFbbDrv, LPSTR lpByte , DWORD dwBytesToWrite);
static int ReadTncChar(FBBDRV *pFbbDrv);
static DATAIN *GetHost(FBBDRV *pFbbDrv);
static BOOL ToTnc(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi);
static BOOL GetDrvVersion(int *nVMaj, int *nVMin);

//////////////////////////////////////////////////////////////
// Global variables
//////////////////////////////////////////////////////////////

FBBDRV	*pDrvHead = NULL;	// Head of port structures

//////////////////////////////////////////////////////////////
//
// Exported functions (5).
// They are :
// BOOL OpenFbbDriver(int nPort, HWND hWnd, void *pDrvInfo);
// BOOL CloseFbbDriver(int nPort);
// BOOL ReadFbbDriver(int nPort, int *nType, int *nChan, char *pszBuf, int *nLen, DRVUI *pUi);
// BOOL WriteFbbDriver(int nPort, int *nType, int nChan, char *pszBuf, int nLen, DRVUI *pUi);
// BOOL StatFbbDriver(int nPort, int nChan, int nCmd, void *pPtr, int nLen);
//
//////////////////////////////////////////////////////////////


BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
    switch (ul_reason_for_call)
	{
		case DLL_PROCESS_ATTACH:
		case DLL_THREAD_ATTACH:
			GetDrvVersion(&DED_MAJOR, &DED_MINOR);
		case DLL_THREAD_DETACH:
			break;
		case DLL_PROCESS_DETACH:
			// Clean all linked ports
			while (pDrvHead)
				CloseFbbDriver(pDrvHead->nPort);
			break;
    }
    return TRUE;
}


static FBBPORT *FindFirstMux(int nCom)
{
	FBBDRV *pTmp = pDrvHead;

	while (pTmp)
	{
		if (pTmp->pPort->nCom == nCom)
			return pTmp->pPort;
		pTmp = pTmp->pNext;
	}
	return NULL;
}

BOOL WINAPI OpenFbbDriver(int nPort, HWND hWnd, void *pDrvInfo)
{
	char	szPort[ 15 ];
	BOOL	fRetVal ;
	FBBDRV	*pFbbDrv;
	HANDLE	hFbbThread;
	DWORD	dwThreadID;
	DRVINFO *pInfo;
	FBBPORT	*pPort;
	COMMTIMEOUTS  CommTimeOuts ;
	
	// Already used ?
	pFbbDrv = SearchPort(nPort);
	if (pFbbDrv)
		return FALSE;
	
	pInfo = (DRVINFO *)pDrvInfo;
	
	if (pInfo->nMultCh > 3)
		return FALSE;
	
	pPort = FindFirstMux(pInfo->nCom);
	
	if (pPort == NULL)
	{
		HANDLE	hDev;

		wsprintf( szPort, "COM%d", pInfo->nCom ) ;
		
		// open COMM device
		
		if ((hDev =	CreateFile(
			szPort, 
			GENERIC_READ | GENERIC_WRITE,
			0,                    // exclusive access
			NULL,                 // no security attrs
			OPEN_EXISTING,
			FILE_ATTRIBUTE_NORMAL, // |
			//FILE_FLAG_OVERLAPPED, // overlapped I/O
			NULL )) == (HANDLE) -1 )
		{
			return FALSE;
		}
		
		
		// setup device buffers
		SetupComm(hDev, 1024, 1024 ) ;

		//DEBUG
		// drop DTR
		EscapeCommFunction(pPort->hDev, CLRDTR ) ;
		//DEBUG

		// purge any information in the buffer
		PurgeComm(hDev, PURGE_TXCLEAR | PURGE_RXCLEAR ) ;
		
		// Set time out
		CommTimeOuts.ReadIntervalTimeout = 0 ;
		CommTimeOuts.ReadTotalTimeoutMultiplier = 0 ;
		CommTimeOuts.ReadTotalTimeoutConstant = 200 ;
		
		// CBR_9600 is approximately 1byte/ms. For our purposes, allow
		// double the expected time per character for a fudge factor.
		CommTimeOuts.WriteTotalTimeoutMultiplier = 2*CBR_9600/pInfo->nBaudrate;
		CommTimeOuts.WriteTotalTimeoutConstant = 0 ;
		
		SetCommTimeouts(hDev, &CommTimeOuts ) ;
		
		if (!SetupConnection(hDev, pInfo->nBaudrate))
			fRetVal = FALSE;
		
		EscapeCommFunction(hDev, SETDTR );

		pPort = (FBBPORT *)LocalAlloc(LPTR, sizeof(FBBPORT));
		if (pPort == NULL)
			return FALSE;

		pPort->hDev = hDev;
		pPort->nCom = pInfo->nCom;
		pPort->bUp  = TRUE;
	}

	pFbbDrv = (FBBDRV *)LocalAlloc(LPTR, sizeof(FBBDRV));
	if (pFbbDrv == NULL)
	{
		if (pPort->nNbTnc == 0)
			LocalFree(pPort);

		return FALSE;
	}

	pPort->pTnc[pPort->nNbTnc] = pFbbDrv;

	pFbbDrv->pPort     = pPort;	
	pFbbDrv->pUi       = NULL;
	pFbbDrv->bNext     = FALSE;
	pFbbDrv->nNbInit   = 5;
	pFbbDrv->hWnd      = hWnd;
	pFbbDrv->nPort     = nPort;
	pFbbDrv->nNbChan   = pInfo->nNbChan;
	pFbbDrv->nNbMultCh = pInfo->nMultCh;
	pFbbDrv->nNbResync = 0;
	pFbbDrv->nNbFrm    = (uchar *)LocalAlloc(LPTR, pInfo->nNbChan+1);
	pFbbDrv->nNbRet    = (uchar *)LocalAlloc(LPTR, pInfo->nNbChan+1);
	pFbbDrv->nState    = (uchar *)LocalAlloc(LPTR, pInfo->nNbChan+1);

	if (pFbbDrv->nNbFrm == NULL || pFbbDrv->nNbRet == NULL || pFbbDrv->nState == NULL)
	{
		if (pPort->nNbTnc == 0)
			LocalFree(pPort);

		// Free the strucure
		LocalFree(pFbbDrv);
		
		return FALSE;
	}
	
	strcpy(pFbbDrv->szMyCall, pInfo->szMyCall);
	
	if (hWnd)
		WM_NOTIFY_MSG = RegisterWindowMessage(FBBDRV_NOTIFY);
	
	// Insert the structure in the list
	pFbbDrv->pNext = pDrvHead;
	pDrvHead = pFbbDrv;
	
	// The thread is only created for the first port of the mux
	if (pPort->nNbTnc == 0)
	{
		InitializeCriticalSection(&pPort->hCritical);
	
		// Create a thread to process the DED protocol.
		hFbbThread = CreateThread( 
			(LPSECURITY_ATTRIBUTES) NULL,
			0,
			(LPTHREAD_START_ROUTINE) DedLoop,
			(LPVOID) pPort,
			0, 
			&dwThreadID );

		pPort->dwThreadId = dwThreadID;
		pPort->hThread    = hFbbThread;
	}
	
	if ((hFbbThread == NULL) || (fRetVal == FALSE))
	{
		// The com port is only freeed for the first port of the mux
		if (pPort->nNbTnc == 0)
		{
			EscapeCommFunction(pPort->hDev, CLRDTR );
			CloseHandle(pPort->hDev);
			DeleteCriticalSection(&pPort->hCritical);
			LocalFree(pPort);
		}
		
		// Remove structure from list
		pDrvHead = pDrvHead->pNext;
		
		// Free the strucure
		LocalFree(pFbbDrv);
		
		return FALSE;
	}
	
	pPort->nNbTnc++;
	
	return TRUE;
	
} // end of OpenFbbDriver()

BOOL WINAPI CloseFbbDriver(int nPort)
{
	FBBDRV *pFbbDrv = SearchPort(nPort);
	FBBPORT *pPort;

	if (pFbbDrv == NULL)
		return FALSE;
	
	// TNC to command mode
	EndHostMode(pFbbDrv);
	
	pPort = pFbbDrv->pPort;

	if (--pPort->nNbTnc == 0)
	{
		// block until thread has been halted
		pPort->bUp = FALSE;
		WaitForSingleObject(pPort->hThread, INFINITE);
	
		// drop DTR
		EscapeCommFunction(pPort->hDev, CLRDTR ) ;
	
		// purge any outstanding reads/writes and close device handle
		PurgeComm(pPort->hDev, PURGE_TXCLEAR | PURGE_RXCLEAR ) ;
		CloseHandle(pPort->hDev) ;

		DeleteCriticalSection(&pPort->hCritical);
	
		LocalFree(pPort);
		pPort = NULL;
	}
	else
	{
		// Thread must be running out of the loop
		pPort->nWait = 1;
		while (pPort->nWait != 2);
	}

	// Remove structure from list
	if (pDrvHead == pFbbDrv)
		pDrvHead = pDrvHead->pNext;
	else
	{
		FBBDRV *pTmp = pDrvHead;
		while (pTmp->pNext)
		{
			if (pTmp->pNext == pFbbDrv)
			{
				pTmp->pNext = pTmp->pNext->pNext;
				break;
			}
			pTmp = pTmp->pNext;
		}
	}

	LocalFree(pFbbDrv->nNbFrm);
	LocalFree(pFbbDrv->nNbRet);
	LocalFree(pFbbDrv->nState);
	LocalFree(pFbbDrv);

	if (pPort)
	{
		// restart the thread
		pPort->nWait = 0;
	}

	return ( TRUE ) ;
	
} // end of CloseFbbDriver()

BOOL WINAPI ReadFbbDriver(int *nPort, int *nType, int *nChan, char *pszBuf, int *nLen, DRVUI *pUi)
{
	DATAIN *pPtr;
	FBBDRV *pFbbDrv = SearchPort(*nPort);
	if (pFbbDrv == NULL)
		return FALSE;
	
	pPtr = GetHost(pFbbDrv);
	if (pPtr == NULL)
		return FALSE;

	if (pPtr->nLen)
	{
		memcpy(pszBuf, pPtr->pData, pPtr->nLen);
		LocalFree(pPtr->pData);
	}
	*nLen = pPtr->nLen;
	*nChan = pPtr->nChannel;
	*nType = pPtr->nType;
	
	if (pPtr->pUi)
	{
		*pUi = *pPtr->pUi;
		LocalFree(pPtr->pUi);
	}
	LocalFree(pPtr);
	
	return TRUE;
} // end of ReadFbbDriver()

BOOL WINAPI WriteFbbDriver(int nPort, int nType, int nChan, char *pszBuf, int nLen, DRVUI *pUi)
{
	FBBDRV *pFbbDrv = SearchPort(nPort);
	if (pFbbDrv == NULL)
		return FALSE;

	return ToTnc(pFbbDrv, nChan, nType, pszBuf, nLen, pUi);
} // end of WriteFbbDriver()

BOOL WINAPI StatFbbDriver(int nPort, int nCmd, int nChan, void *pPtr, int nLen)
{
	FBBDRV *pFbbDrv = SearchPort(nPort);
	if (pFbbDrv == NULL)
		return 0;
	
	switch (nCmd)
	{
	case DRV_SNDCMD:
	case DRV_ECHOCMD:
		return (ToTnc(pFbbDrv, nChan, DRV_COMMAND, (char *)pPtr, -1, NULL));
	case DRV_PORTCMD:
		return (ToTnc(pFbbDrv, 0, DRV_COMMAND, (char *)pPtr, -1, NULL));
	case DRV_PACLEN:
		*((int *) pPtr) = 250;
		return TRUE;
	case DRV_VERSION:
		wsprintf((char *)pPtr, 
			"v%d.%02d FBB driver for WA8DED (F6FBB-%s)", 
			DED_MAJOR, DED_MINOR, __DATE__);
		return TRUE;
	}

	return FALSE;
} // end of StatFbbDriver()


//////////////////////////////////////////////////////////////
//
// Local functions
//
//////////////////////////////////////////////////////////////

static FBBDRV *SearchPort(int nPort)
{
	FBBDRV *pTmp = pDrvHead;
	
	while (pTmp)
	{
		if (pTmp->nPort == nPort)
			break;
		pTmp = pTmp->pNext;
	}
	
	return pTmp;
}

static BOOL ToTnc(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi)
{
	DATAOUT *pPtr;
	
	if (nLen == -1)
		nLen = strlen(pData);

	if (nLen <= 0 || nLen > 256)
		return FALSE;
	
	if (pUi && nType == DRV_UNPROTO)
	{
		// Set the unproto path
		char szStr[256];
		int nUiLen = wsprintf(szStr, "C%s %s", pUi->szTo, pUi->szVia);
		ToTnc(pFbbDrv, 0, DRV_COMMAND, szStr, nUiLen, NULL);
	}

	pPtr           = (DATAOUT *)LocalAlloc( LMEM_FIXED, sizeof(DATAOUT));
	pPtr->pData    = (char *)LocalAlloc( LMEM_FIXED, nLen);
	pPtr->nChannel = nChannel;
	pPtr->nType    = nType;
	pPtr->nLen     = nLen;
	pPtr->pNext    = NULL;

	memcpy(pPtr->pData, pData, nLen);

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->pPort->hCritical);
	
	// Add the structure at end of list
	if (pFbbDrv->pTncHead == NULL)
		pFbbDrv->pTncHead = pPtr;
	else
		pFbbDrv->pTncTail->pNext = pPtr;
	
	// Update tail information
	pFbbDrv->pTncTail = pPtr;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->pPort->hCritical);

	return TRUE;
}

static void ToHost(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi)
{
	DATAIN *pPtr;
	
	if (nLen < 0 || nLen > 256)
		return;
	
	pPtr           = (DATAIN *)LocalAlloc( LMEM_FIXED, sizeof(DATAIN));
	pPtr->nChannel = nChannel;
	pPtr->nType    = nType;
	pPtr->nLen     = nLen;
	pPtr->pUi      = pUi;
	pPtr->pNext    = NULL;

	if (nLen > 0)
	{
		pPtr->pData    = (char *)LocalAlloc( LMEM_FIXED, nLen);
		memcpy(pPtr->pData, pData, nLen);
	}
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->pPort->hCritical);
	
	if (pFbbDrv->pHostHead == NULL)
		pFbbDrv->pHostHead = pPtr;
	else
		pFbbDrv->pHostTail->pNext = pPtr;
	
	// Update tail information
	pFbbDrv->pHostTail = pPtr;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->pPort->hCritical);
	
	// Send Notification message
	if (pFbbDrv->hWnd)
		PostMessage(pFbbDrv->hWnd, WM_NOTIFY_MSG, nType, MAKELONG(pFbbDrv->nPort, nChannel));
}

static DATAOUT *GetTnc(FBBDRV *pFbbDrv)
{
	DATAOUT *pPtr;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->pPort->hCritical);
	
	pPtr = pFbbDrv->pTncHead;
	if (pPtr)
		pFbbDrv->pTncHead = pFbbDrv->pTncHead->pNext;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->pPort->hCritical);
	
	return pPtr;
}

static DATAIN *GetHost(FBBDRV *pFbbDrv)
{
	DATAIN *pPtr;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->pPort->hCritical);
	
	pPtr = pFbbDrv->pHostHead;
	if (pPtr)
		pFbbDrv->pHostHead = pFbbDrv->pHostHead->pNext;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->pPort->hCritical);
	
	return pPtr;
}

static int TncQueueSize(FBBDRV *pFbbDrv)
{
	DATAOUT *pPtr;
	int nNb = 0;
	
	if (pFbbDrv->nNbResync)
		return 0;

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->pPort->hCritical);
	
	pPtr = pFbbDrv->pTncHead;
	while (pPtr)
	{
		pPtr = pPtr->pNext;
		++nNb;
	}
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->pPort->hCritical);
	
	return nNb;
}

static void HostInfo(FBBDRV *pFbbDrv, int nChannel, char *sFormat, ...)
{
	char	szStr[512];
	va_list pArgPtr;
	int nCnt;
	
	va_start(pArgPtr, sFormat);
	nCnt = wvsprintf(szStr, sFormat, pArgPtr);
	va_end(pArgPtr);
	
	if (nCnt > 0)
		ToHost(pFbbDrv, nChannel, DRV_INFO, szStr, nCnt, NULL);
}

static void HostEcho(FBBDRV *pFbbDrv, int nChannel, char *szTxt)
{
	int nCnt = strlen(szTxt);

	if (nCnt > 0)
		ToHost(pFbbDrv, nChannel, DRV_ECHOCMD, szTxt, nCnt, NULL);
}

static BOOL InitPort(FBBDRV *pFbbDrv)
{
	int nNb;
	char szBuf[256];

	char *szInit[] = {
		"MIUSC",
		"I",
		"M",
		NULL
	};

	wsprintf(szBuf, "i%s", pFbbDrv->szMyCall);
	ToTnc(pFbbDrv, 0, DRV_COMMAND, szBuf, -1, NULL);

	for (nNb = 0 ; szInit[nNb] ; nNb++)
		ToTnc(pFbbDrv, 0, DRV_COMMAND, szInit[nNb], -1, NULL);
	return TRUE;
}

static BOOL StartHostMode(FBBDRV *pFbbDrv)
{
	//	int nLen;
	int nChar;
	int nNb;
	BOOL bOk;
	char *pszInit = "\030\033JHOST\r\033MN\r";
	char *pszHost = "\033JHOST1\r";
	//	char szBuffer[80];
	
	HostInfo(pFbbDrv, 0, "StartHostMode");

	// Check if TNC is alive
	for (nNb = 0 ; nNb < 4 ; nNb++)
	{
		bOk = FALSE;
		WriteTncBlock(pFbbDrv, pszInit, strlen(pszInit));
		while ((nChar = ReadTncChar(pFbbDrv)) >= 0)
		{
			Sleep(10);
			bOk = TRUE;
		}
		
		if (bOk)
			break;
	}
	
	if (nNb == 4)
		return FALSE;
	
	// Go to hostmode
	WriteTncBlock(pFbbDrv, pszHost, strlen(pszHost));
	
	// Be sure the COM is empty ...
	
	while (ReadTncChar(pFbbDrv) >= 0)
		Sleep(10);
	
	// Should initialise the port now ...
	InitPort(pFbbDrv);
	
	HostInfo(pFbbDrv, 0, "StartHostMode OK");
	return TRUE;
}

static BOOL EndHostMode(FBBDRV *pFbbDrv)
{
	if (pFbbDrv->nNbInit != -1)
		return FALSE;

	// No monitoring
	ToTnc(pFbbDrv, 0, DRV_COMMAND, "MN", -1, NULL);

	// Command mode 
	ToTnc(pFbbDrv, 0, DRV_COMMAND, "JHOST0", -1, NULL);
	
	// Wait until all TNC queue is processed
	while (pFbbDrv->nNbInit == -1)
	{
		if (TncQueueSize(pFbbDrv) == 0)
			break;
		
		// Wait some time...
		Sleep(100);
	}
	
	return TRUE;
}

BOOL SetupConnection(HANDLE hDev, int nBaudrate)
{
	BOOL       fRetVal ;
	DCB        dcb ;
	
	dcb.DCBlength = sizeof( DCB ) ;
	
	GetCommState(hDev, &dcb ) ;
	
	dcb.BaudRate = nBaudrate ;
	dcb.ByteSize = 8 ;
	dcb.Parity   = NOPARITY ;
	dcb.StopBits = ONESTOPBIT ;
	
	
	// setup hardware flow control
	
	dcb.fOutxDsrFlow = FALSE ;
	dcb.fDtrControl = DTR_CONTROL_DISABLE ;
	
	dcb.fOutxCtsFlow = FALSE ;
	dcb.fRtsControl = RTS_CONTROL_DISABLE ;
	
	// setup software flow control
	
	dcb.fInX = dcb.fOutX = FALSE ;
	
	// other various settings
	
	dcb.fBinary = TRUE ;
	
	fRetVal = SetCommState(hDev, &dcb ) ;
	
	return ( fRetVal ) ;
	
} // end of SetupConnection()

static int ReadTncChar(FBBDRV *pFbbDrv)
{
	BOOL       fReadStat ;
	COMSTAT    ComStat ;
	DWORD      dwErrorFlags;
	DWORD      dwLength;
	BYTE		nChar;

	// Read one character
	// I don't know how to read more than one character
	// with a timeout, to get any number of bytes
	ClearCommError(pFbbDrv->pPort->hDev, &dwErrorFlags, &ComStat ) ;
	fReadStat = ReadFile(pFbbDrv->pPort->hDev, &nChar,
		1, &dwLength, NULL) ;
	if (!fReadStat)
	{
		ClearCommError(pFbbDrv->pPort->hDev, &dwErrorFlags, &ComStat ) ;
		if (dwErrorFlags > 0)
		{
			HostInfo(pFbbDrv, 0, "<CE-%u>\r", dwErrorFlags ) ;
		}
		return -1;
	}
	
	if (dwLength == 0)
	{
		return -1;
	}
	
	return ( nChar ) ;	
} // end of ReadTncChar()

static BOOL WriteTncBlock(FBBDRV *pFbbDrv, LPSTR lpByte , DWORD dwBytesToWrite)
{
	
	BOOL        fWriteStat ;
	DWORD       dwBytesWritten ;
	DWORD       dwErrorFlags;
	//	DWORD   	dwError;
	DWORD       dwBytesSent=0;
	COMSTAT     ComStat;
	
	fWriteStat = WriteFile(pFbbDrv->pPort->hDev, lpByte, dwBytesToWrite,
		&dwBytesWritten, NULL) ;
	if (!fWriteStat)
	{
		// some other error occurred
		ClearCommError(pFbbDrv->pPort->hDev, &dwErrorFlags, &ComStat ) ;
		if (dwErrorFlags > 0)
		{
			HostInfo(pFbbDrv, 0, "<CE-%u>\r", dwErrorFlags ) ;
		}
		return ( FALSE );
	}
	return ( TRUE ) ;
	
} // end of WriteTncBlock()


static void DedUiHeader(int nPort, char *szBuf, DRVUI *pUi)
{
	char *pPtr;
	char *pScan;

	memset (pUi, 0, sizeof(DRVUI));

	pUi->nPort = nPort;

	pScan = szBuf;

	pPtr = strtok (pScan, " ");	/* fm */

	if (pPtr == NULL)
		return;

	pPtr = strtok (NULL, " ");	/* exped */

	if (pPtr == NULL)
		return;
	strncpy (pUi->szFrom, pPtr, 9);
	pUi->szFrom[9] = '\0';

	pPtr = strtok (NULL, " ");	/* to */

	if (pPtr == NULL)
		return;

	pPtr = strtok (NULL, " ");	/* desti */

	if (pPtr == NULL)
		return;
	strncpy (pUi->szTo, pPtr, 9);
	pUi->szTo[9] = '\0';

	pPtr = strtok (NULL, " ");	/* via ou ctl */

	if (pPtr == NULL)
		return;

	if (strcmp (pPtr, "via") == 0)
	{
		for (;;)
		{
			if (*pUi->szVia)
				strcat (pUi->szVia, " ");

			pPtr = strtok (NULL, " ");	/* digis */

			if (pPtr == NULL)
				return;

			if (strcmp (pPtr, "ctl") == 0)
				break;

			strncat (pUi->szVia, pPtr, 9);
		}
		pUi->szVia[80] = '\0';
	}

	pPtr = strtok (NULL, " ");	/* controle */

	if (pPtr == NULL)
		return;
	strncpy (pUi->szCtl, pPtr, 9);
	pUi->szCtl[9] = '\0';

	pUi->bUi = (strncmp (pPtr, "UI", 2) == 0);

	pPtr = strtok (NULL, " ");	/* pid */

	if (pPtr == NULL)
		return;

	pPtr = strtok (NULL, " ");	/* pid */

	if (pPtr == NULL)
		return;
	sscanf (pPtr, "%x", &pUi->nPid);
}

static void SendTncRequest(FBBPORT *pPort)
{
	static int nChan = 0;
	int nLen;
	char szBuf[259];
	DATAOUT *pPtr;
	FBBDRV *pFbbDrv;
	
	pFbbDrv = pPort->pTnc[pPort->nCurTnc];

	if (pFbbDrv->bNext)
	{
		// Got header then get data of monitoring
		szBuf[0] = nChan;
		szBuf[1] = 1;
		szBuf[2] = 0;
		szBuf[3] = 'G';
		pFbbDrv->nLastCmd = 'G';

		nLen = 4;

		pFbbDrv->bNext = FALSE;
	}
	else if (pPtr = GetTnc(pFbbDrv))
	{
		switch(pPtr->nType)
		{
		case DRV_COMMAND:
			szBuf[0] = pPtr->nChannel;
			szBuf[1] = 1;
			break;
		case DRV_DATA:
			szBuf[0] = pPtr->nChannel;
			szBuf[1] = 0;
			break;
		case DRV_UNPROTO:
			szBuf[0] = 0;
			szBuf[1] = 0;
		}
		szBuf[2] = pPtr->nLen - 1;
		memcpy(szBuf+3, pPtr->pData, pPtr->nLen);
		pFbbDrv->nLastCmd = 0;

		nLen = pPtr->nLen + 3;


		LocalFree(pPtr->pData);
		LocalFree(pPtr);
	}
	else
	{
		// Polling for link status
		szBuf[0] = nChan;
		szBuf[1] = 1;
		szBuf[2] = 0;
		szBuf[3] = 'L';
		pFbbDrv->nLastCmd = 'L';

		nLen = 4;

		if (++nChan > pFbbDrv->nNbChan)
			nChan = 0;

		// Next time is Get frame
		pFbbDrv->bNext = TRUE;
	}
	
	WriteTncBlock(pFbbDrv, szBuf, nLen);
}

static void ReadTncAnswer(FBBPORT *pPort)
{
	int nChan;
	int nCode;
	int nChar;
	int nLen;
	int nNb;
	char *pPtr;
	char szBuffer[256];
	FBBDRV *pFbbDrv;

	pFbbDrv = pPort->pTnc[pPort->nCurTnc];

	if (pFbbDrv->nNbResync > 0)
		return;
	
	nChan = ReadTncChar( pFbbDrv);
	if (nChan == -1)
	{
		// Resynchro
		char bChar = '\001';
		
		while (nChan == -1)
		{
			WriteTncBlock(pFbbDrv, &bChar, 1);
			nChan = ReadTncChar( pFbbDrv);
			HostInfo(pFbbDrv, 0, "Resync:%d", ++(pFbbDrv->nNbResync));
			if (pFbbDrv->nNbResync == 256)
			{
				// Try re-init the TNC
				pFbbDrv->nNbInit = 5;
			}

			if (!pFbbDrv->pPort->bUp || (pFbbDrv->nNbResync == 256))
				return;
		}
	}
	
	pFbbDrv->nNbResync = 0;

	nCode = ReadTncChar( pFbbDrv);
	pPtr = szBuffer;
	
	switch (nCode)
	{
	case 0:
		if (pFbbDrv->nLastCmd == 'L')
		{
			// Should never occur
			PurgeComm(pFbbDrv->pPort->hDev, PURGE_TXCLEAR | PURGE_RXCLEAR ) ;
		}
		break;
	case 1:
		for (;;)
		{
			nChar = ReadTncChar(pFbbDrv);
			if (nChar == -1)
				return;
			*pPtr = nChar;
			if (*pPtr == '\0')
				break;
			++pPtr;
		}
		if (pFbbDrv->nLastCmd != 'L')
		{
			HostEcho(pFbbDrv, nChan, szBuffer);
		}
		else
		{
			DRVSTATS DrvStats;
			int nbmes, nbtra, nbatt, nback, nbret, con;

			nbatt = nback = nbret = con = 0;

			sscanf (szBuffer, "%d%d%d%d%d%d",
						&nbmes, &nbtra, &nbatt, &nback, &nbret, &con);

			DrvStats.nNbFrames = nbmes + nbtra;
			DrvStats.nNbRetry  = nbret;
			switch(con)
			{
			case 0:
				DrvStats.nState = DRV_DISCONNECTED;
				break;
			case 1:
				DrvStats.nState = DRV_CONNPENDING;
				break;
			case 3:
				DrvStats.nState = DRV_DISCPENDING;
				break;
			default:
				DrvStats.nState = DRV_CONNECTED;
				break;
			}
			DrvStats.nBuffers  = 500;
			DrvStats.nFrameNb  = 0;
			if (DrvStats.nNbFrames != pFbbDrv->nNbFrm[nChan] || 
				DrvStats.nNbRetry  != pFbbDrv->nNbRet[nChan] || 
				DrvStats.nState    != pFbbDrv->nState[nChan])
			{
				pFbbDrv->nNbFrm[nChan] = DrvStats.nNbFrames;
				pFbbDrv->nNbRet[nChan] = DrvStats.nNbRetry;
				pFbbDrv->nState[nChan] = DrvStats.nState;
				ToHost(pFbbDrv, nChan, DRV_STATS, (char *)&DrvStats, sizeof(DRVSTATS), NULL);
			}
		}
		break;
	case 2:
		for (;;)
		{
			nChar = ReadTncChar(pFbbDrv);
			if (nChar == -1)
				return;
			*pPtr = nChar;
			if (*pPtr == '\0')
				break;
			++pPtr;
		}
		HostEcho(pFbbDrv, nChan, szBuffer);
		break;
	case 3:
		for (;;)
		{
			nChar = ReadTncChar(pFbbDrv);
			if (nChar == -1)
				return;
			*pPtr = nChar;
			if (*pPtr == '\0')
				break;
			++pPtr;
		}
		ToHost(pFbbDrv, nChan, DRV_COMMAND, szBuffer, strlen(szBuffer), NULL);
		break;
	case 4:
		for (;;)
		{
			nChar = ReadTncChar(pFbbDrv);
			if (nChar == -1)
				return;
			*pPtr = nChar;
			if (*pPtr == '\0')
				break;
			++pPtr;
		}
		pFbbDrv->pUi = (DRVUI *)LocalAlloc(LMEM_FIXED, sizeof(DRVUI));
		if (pFbbDrv->pUi)
			DedUiHeader(pFbbDrv->nPort, szBuffer, pFbbDrv->pUi);
		ToHost(pFbbDrv, nChan, DRV_UNPROTO, NULL, 0, pFbbDrv->pUi);
		pFbbDrv->pUi = NULL;
		break;
	case 5:
		for (;;)
		{
			nChar = ReadTncChar(pFbbDrv);
			if (nChar == -1)
				return;
			*pPtr = nChar;
			if (*pPtr == '\0')
				break;
			++pPtr;
		}
		pFbbDrv->pUi = (DRVUI *)LocalAlloc(LMEM_FIXED, sizeof(DRVUI));
		if (pFbbDrv->pUi)
			DedUiHeader(pFbbDrv->nPort, szBuffer, pFbbDrv->pUi);
		pFbbDrv->bNext = TRUE;
		break;
	case 6:
		nLen = ReadTncChar(pFbbDrv) + 1;
		for (nNb = 0 ; nNb < nLen ; nNb++)
		{
			nChar = ReadTncChar(pFbbDrv);
			if (nChar == -1)
				return;
			szBuffer[nNb] = nChar;
		}
		szBuffer[nNb] = '\0';
		ToHost(pFbbDrv, nChan, DRV_UNPROTO, szBuffer, nLen, pFbbDrv->pUi);
		pFbbDrv->pUi = NULL;
		break;
	case 7:
		nLen = ReadTncChar(pFbbDrv) + 1;
		for (nNb = 0 ; nNb < nLen ; nNb++)
			szBuffer[nNb] = ReadTncChar(pFbbDrv);
		szBuffer[nNb] = '\0';
		ToHost(pFbbDrv, nChan, DRV_DATA, szBuffer, nLen, pFbbDrv->pUi);
		break;
	default:
		// Empty the queues
		PurgeComm(pFbbDrv->pPort->hDev, PURGE_TXCLEAR | PURGE_RXCLEAR ) ;
		break;
	}
}

static BOOL NextTnc(FBBPORT *pPort)
{
	FBBDRV *pFbbDrv;
	static int nMuxPrec = -1;

	if (pPort->nWait)
	{
		// Keep the thread out of the loop
		if (pPort->nWait == 1)
			pPort->nWait = 2;
		Sleep(100);
		return FALSE;
	}

	// Must be thread safe
	EnterCriticalSection(&pPort->hCritical);

	// Next TNC
	if (++pPort->nCurTnc == pPort->nNbTnc)
		pPort->nCurTnc = 0;

	pFbbDrv = pPort->pTnc[pPort->nCurTnc];

	// Must be thread safe
	LeaveCriticalSection(&pPort->hCritical);

	if (pFbbDrv->nNbInit == 0)
	{
		// TNC stopped
		Sleep(10);
		return FALSE;
	}

	// Optimise
	if (pFbbDrv->nNbMultCh != nMuxPrec)
	{
		// Set the signals for the MUX
		nMuxPrec = pFbbDrv->nNbMultCh;
		EscapeCommFunction(pPort->hDev, (nMuxPrec & 1) ? CLRDTR : SETDTR);
		EscapeCommFunction(pPort->hDev, (nMuxPrec & 2) ? CLRRTS : SETRTS);
	}

	if (pFbbDrv->nNbInit > 0)
	{
		// Init TNC
		--pFbbDrv->nNbInit;

		//DEBUG
		int nNbInitBack = pFbbDrv->nNbInit;
		pFbbDrv->nNbInit = -1;
		EndHostMode(pFbbDrv);
		pFbbDrv->nNbInit = nNbInitBack;
		//DEBUG

		if (!StartHostMode(pFbbDrv))
			return FALSE;

		pFbbDrv->nNbInit = -1;
		pFbbDrv->nNbResync = 0;
	}

	return TRUE;
}

// TNC Thread

DWORD WINAPI DedLoop(LPSTR lpData)
{
	FBBPORT   *pPort = (FBBPORT*) lpData ;
	
	while (pPort->bUp)
	{
		// Setup the TNC to hostmode
		if (NextTnc(pPort))
		{
			// Send the request
			SendTncRequest(pPort);
		
			// Read the answer
			ReadTncAnswer(pPort);
		}
	}
	
	return( TRUE ) ;
	
} // end of DedLoop()

BOOL GetDrvVersion(int *nVMaj, int *nVMin)
{
	DWORD dwVersionInfoSize;
	DWORD dwTmpHandle;
	LPVOID lpVersionInfo; 
	BOOL bRet = false;


	dwVersionInfoSize = GetFileVersionInfoSize(FBBDLLNAME, &dwTmpHandle);

	if(dwVersionInfoSize)
	{
		lpVersionInfo = LocalAlloc(LPTR, dwVersionInfoSize);
		if(lpVersionInfo)
		{
			if(GetFileVersionInfo(FBBDLLNAME, dwTmpHandle, dwVersionInfoSize, lpVersionInfo))
			{
				LPVOID lpBuffer = LocalAlloc(LPTR, dwVersionInfoSize);
				UINT dwBytes;

				if( VerQueryValue(lpVersionInfo, 
						TEXT("\\StringFileInfo\\000004B0\\FileVersion"), 
						&lpBuffer, 
						&dwBytes)) 
				{
						sscanf((TCHAR *)lpBuffer, "%d,%d", nVMaj, nVMin);
						bRet = true;
				}

				LocalFree(lpBuffer);
			}

			LocalFree(lpVersionInfo);
 		}
 	}

	return bRet;
}
