
//////////////////////////////////////////////////////////////
//
// FBB driver for TFWIN32 DLL host mode
//
// File : fbb_tfw.dll
//
// (C) F6FBB 1999-2000
//
//////////////////////////////////////////////////////////////
//
//  Version history
//
//
//////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <stdio.h>
#include <winver.h>
#include "../fbb_drv.h"

#define uchar unsigned char

#define	FBBDLLNAME	"tfw.dll"

//------------

int TFW_MAJOR = 0;
int TFW_MINOR = 0;

typedef struct DATAOUT_t {
	char	*pData;
	int		nChannel;
	int		nType;
	int		nLen;
	struct	DATAOUT_t *pNext;
} DATAOUT;

typedef struct DATAIN_t {
	char	*pData;
	int		nChannel;
	int		nType;
	int		nLen;
	DRVUI	*pUi;
	struct	DATAIN_t *pNext;
} DATAIN;

struct	FBBPORT_t;

typedef struct FBBDRV_t
{
	HWND	hWnd;
	int		nPort;
	int		nNbChan;
	int		nNbMultCh;
	int		nLastCmd;
	int		nNbResync;
	int		nNbInit;
	BOOL	bNext;
	uchar	*nNbFrm;
	uchar	*nNbRet;
	uchar	*nState;
	char	szMyCall[10];
	DRVUI	*pUi;
	DATAOUT	*pTncHead;	// Head of data_out list
	DATAOUT	*pTncTail;	// Tail of data_out list
	DATAIN	*pHostHead;	// Head of data_in list
	DATAIN	*pHostTail;	// Tail of data_in list
	struct	FBBPORT_t	*pPort;
	struct	FBBDRV_t	*pNext;
} FBBDRV;

typedef __declspec(dllimport) unsigned (* TFOPEN)(HWND);
typedef __declspec(dllimport) BOOL (* TFCLOSE)(void);
typedef __declspec(dllimport) BOOL (* TFCHECK)(void);
typedef __declspec(dllimport) int  (* TFGET)(void);
typedef __declspec(dllimport) BOOL (* TFPUT)(char);

typedef struct FBBPORT_t
{
	//	HANDLE	hDev;
	HANDLE	hThread;
	DWORD	dwThreadId;
	BOOL	bUp;
	TFOPEN	TfOpen;
	TFCLOSE	TfClose;
	TFCHECK	TfChck;
	TFGET	TfGet;
	TFPUT	TfPut;
	int		nWait;
	int		nCom;
	int		nCurTnc;
	int		nNbTnc;
	//	struct	FBBDRV_t	*pTnc[4];
	struct	FBBDRV_t	*pDrv;
	CRITICAL_SECTION	hCritical;
} FBBPORT;


//////////////////////////////////////////////////////////////
// Local prototypes
//////////////////////////////////////////////////////////////

static BOOL SetupConnection(HANDLE hDev, int nBaudRate);
static DWORD WINAPI TfwLoop( LPSTR lpData );
static FBBDRV *SearchPort(int nPort);
static BOOL StartHostMode(FBBDRV *pFbbDrv);
static BOOL EndHostMode(FBBDRV *pFbbDrv);
static BOOL WriteTncBlock(FBBDRV *pFbbDrv, LPSTR lpByte , DWORD dwBytesToWrite);
static int ReadTncChar(FBBDRV *pFbbDrv);
static DATAIN *GetHost(FBBDRV *pFbbDrv);
static BOOL ToTnc(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi);
static BOOL GetDrvVersion(int *nVMaj, int *nVMin);
static void SendTncRequest(FBBPORT *pPort);

//////////////////////////////////////////////////////////////
// Global variables
//////////////////////////////////////////////////////////////

FBBDRV	*pDrvHead = NULL;	// Head of port structures

//////////////////////////////////////////////////////////////
//
// Exported functions (5).
// They are :
// BOOL OpenFbbDriver(int nPort, HWND hWnd, void *pDrvInfo);
// BOOL CloseFbbDriver(int nPort);
// BOOL ReadFbbDriver(int nPort, int *nType, int *nChan, char *pszBuf, int *nLen, DRVUI *pUi);
// BOOL WriteFbbDriver(int nPort, int *nType, int nChan, char *pszBuf, int nLen, DRVUI *pUi);
// BOOL StatFbbDriver(int nPort, int nChan, int nCmd, void *pPtr, int nLen);
//
//////////////////////////////////////////////////////////////


BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
    switch (ul_reason_for_call)
	{
		case DLL_PROCESS_ATTACH:
		case DLL_THREAD_ATTACH:
			GetDrvVersion(&TFW_MAJOR, &TFW_MINOR);
		case DLL_THREAD_DETACH:
			break;
		case DLL_PROCESS_DETACH:
			// Clean all linked ports
			while (pDrvHead)
				CloseFbbDriver(pDrvHead->nPort);
			break;
    }
    return TRUE;
}


/*
static FBBPORT *FindFirstMux(int nCom)
{
	FBBDRV *pTmp = pDrvHead;

	while (pTmp)
	{
		if (pTmp->pPort->nCom == nCom)
			return pTmp->pPort;
		pTmp = pTmp->pNext;
	}
	return NULL;
}
*/

BOOL WINAPI OpenFbbDriver(int nPort, HWND hWnd, void *pDrvInfo)
{
	char	szPort[ 15 ];
	FBBDRV	*pFbbDrv;
	HMODULE	hinstDLL;
	HANDLE	hFbbThread;
	DWORD	dwThreadID;
	DRVINFO *pInfo;
	FBBPORT	*pPort = NULL;
	
	// Already used ?
	pFbbDrv = SearchPort(nPort);
	if (pFbbDrv)
		return FALSE;
	
	pInfo = (DRVINFO *)pDrvInfo;
	
	// Up to eight ports
	if (pInfo->nMultCh > 7)
		return FALSE;
	
	if (pFbbDrv)
		pPort = pFbbDrv->pPort;

	// One com port allowed for TfWin
	if (pPort && pPort->nCom != pInfo->nCom)
		return FALSE;
	
	if (pPort == NULL)
	{
		TFOPEN	TfOpen;
		TFCLOSE	TfClose;
		TFCHECK	TfChck;
		TFGET	TfGet;
		TFPUT	TfPut;
		
		wsprintf( szPort, "COM%d", pInfo->nCom ) ;
		
		// open TfWin
		
		hinstDLL = LoadLibrary("TFWIN32.DLL");
		if (hinstDLL <= (HMODULE)HINSTANCE_ERROR)
		{
			char	Text[80];
			wsprintf(Text, "Cannot load TFWIN32.DLL : Error %d", hinstDLL);
			MessageBox( NULL, Text, "Fbb TfWin driver", MB_ICONEXCLAMATION ) ;
			return FALSE;
		}
		
		TfOpen = (TFOPEN)GetProcAddress(hinstDLL, "TfOpen");
		if (TfOpen == NULL)
			return FALSE;
		
		TfClose = (TFCLOSE)GetProcAddress(hinstDLL, "TfClose");
		if (TfClose == NULL)
			return FALSE;
		
		TfChck = (TFCHECK)GetProcAddress(hinstDLL, "TfChck");
		if (TfChck == NULL)
			return FALSE;
		
		TfGet = (TFGET)GetProcAddress(hinstDLL, "TfGet");
		if (TfGet == NULL)
			return FALSE;
		
		TfPut = (TFPUT)GetProcAddress(hinstDLL, "TfPut");
		if (TfPut == NULL)
			return FALSE;
		
		if (!(*TfOpen)(NULL))
		{
			char	Text[80];
			wsprintf(Text, "Cannot init TFWIN32.DLL : Configuration file error ?");
			MessageBox( NULL, Text, "Fbb TfWin driver", MB_ICONEXCLAMATION ) ;
			return FALSE;
		}
		
		pPort = (FBBPORT *)LocalAlloc(LPTR, sizeof(FBBPORT));
		if (pPort == NULL)
			return FALSE;
		
		pPort->TfOpen = TfOpen;
		pPort->TfClose = TfClose;
		pPort->TfChck = TfChck;
		pPort->TfGet = TfGet;
		pPort->TfPut = TfPut;
		pPort->nCom = pInfo->nCom;
		pPort->nCurTnc = 0;
		pPort->bUp  = TRUE;
	}
	
	pFbbDrv = (FBBDRV *)LocalAlloc(LPTR, sizeof(FBBDRV));
	if (pFbbDrv == NULL)
	{
		LocalFree(pPort);		
		return FALSE;
	}
	
	pPort->pDrv = pFbbDrv;
	
	pFbbDrv->pPort     = pPort;	
	pFbbDrv->pUi       = NULL;
	pFbbDrv->bNext     = FALSE;
	pFbbDrv->nNbInit   = 5;
	pFbbDrv->hWnd      = hWnd;
	pFbbDrv->nPort     = nPort;
	pFbbDrv->nNbChan   = pInfo->nNbChan;
	pFbbDrv->nNbMultCh = pInfo->nMultCh;
	pFbbDrv->nNbResync = 0;
	pFbbDrv->nNbFrm    = (uchar *)LocalAlloc(LPTR, pInfo->nNbChan+1);
	pFbbDrv->nNbRet    = (uchar *)LocalAlloc(LPTR, pInfo->nNbChan+1);
	pFbbDrv->nState    = (uchar *)LocalAlloc(LPTR, pInfo->nNbChan+1);
	
	if (pFbbDrv->nNbFrm == NULL || pFbbDrv->nNbRet == NULL || pFbbDrv->nState == NULL)
	{
		if (pPort->nNbTnc == 0)
			LocalFree(pPort);
		
		// Free the strucure
		LocalFree(pFbbDrv);
		
		return FALSE;
	}
	
	strcpy(pFbbDrv->szMyCall, pInfo->szMyCall);
	
	if (hWnd)
		WM_NOTIFY_MSG = RegisterWindowMessage(FBBDRV_NOTIFY);
	
	// Insert the structure in the list
	pFbbDrv->pNext = pDrvHead;
	pDrvHead = pFbbDrv;
	
	// The thread is only created for the first port of the mux
	if (pPort->nNbTnc == 0)
	{
		InitializeCriticalSection(&pPort->hCritical);
		
		// Create a thread to process the Tfw protocol.
		hFbbThread = CreateThread( 
			(LPSECURITY_ATTRIBUTES) NULL,
			0,
			(LPTHREAD_START_ROUTINE) TfwLoop,
			(LPVOID) pPort,
			0, 
			&dwThreadID );
		
		pPort->dwThreadId = dwThreadID;
		pPort->hThread    = hFbbThread;
	}
	
	if (hFbbThread == NULL)
	{
		// The com port is only freeed for the first port of the mux
		if (pPort->nNbTnc == 0)
		{
			if (pPort->TfClose)
				(*pPort->TfClose)();
			DeleteCriticalSection(&pPort->hCritical);
			LocalFree(pPort);
			pPort = NULL;
		}
		
		// Remove structure from list
		pDrvHead = pDrvHead->pNext;
		
		// Free the strucure
		LocalFree(pFbbDrv);
		pFbbDrv = NULL;
		
		return FALSE;
	}
	
	pPort->nNbTnc++;
	
	return TRUE;
	
} // end of OpenFbbDriver()

BOOL WINAPI CloseFbbDriver(int nPort)
{
	FBBDRV *pFbbDrv = SearchPort(nPort);
	FBBPORT *pPort;

	if (pFbbDrv == NULL)
		return FALSE;
		
	// TNC to command mode
	EndHostMode(pFbbDrv);
	
	pPort = pFbbDrv->pPort;

	// block until thread has been halted
	pPort->bUp = FALSE;
	WaitForSingleObject(pPort->hThread, INFINITE);

	// Close TfWin
	if (pPort->TfClose)
		(*pPort->TfClose)();

	DeleteCriticalSection(&pPort->hCritical);
	
	LocalFree(pPort);
	pPort = NULL;

	// Remove structure from list
	if (pDrvHead == pFbbDrv)
		pDrvHead = pDrvHead->pNext;
	else
	{
		FBBDRV *pTmp = pDrvHead;
		while (pTmp->pNext)
		{
			if (pTmp->pNext == pFbbDrv)
			{
				pTmp->pNext = pTmp->pNext->pNext;
				break;
			}
			pTmp = pTmp->pNext;
		}
	}

	LocalFree(pFbbDrv->nNbFrm);
	LocalFree(pFbbDrv->nNbRet);
	LocalFree(pFbbDrv->nState);
	LocalFree(pFbbDrv);

	return ( TRUE ) ;
	
} // end of CloseFbbDriver()

BOOL WINAPI ReadFbbDriver(int *nPort, int *nType, int *nChan, char *pszBuf, int *nLen, DRVUI *pUi)
{
	DATAIN *pPtr;
	FBBDRV *pFbbDrv = SearchPort(*nPort);
	if (pFbbDrv == NULL)
		return FALSE;
	
	pPtr = GetHost(pFbbDrv);
	if (pPtr == NULL)
		return FALSE;

	if (pPtr->nLen)
	{
		memcpy(pszBuf, pPtr->pData, pPtr->nLen);
		LocalFree(pPtr->pData);
	}
	*nLen = pPtr->nLen;
	*nChan = pPtr->nChannel;
	*nType = pPtr->nType;
	
	if (pPtr->pUi)
	{
		*pUi = *pPtr->pUi;
		LocalFree(pPtr->pUi);
	}
	LocalFree(pPtr);
	
	return TRUE;
} // end of ReadFbbDriver()

BOOL WINAPI WriteFbbDriver(int nPort, int nType, int nChan, char *pszBuf, int nLen, DRVUI *pUi)
{
	FBBDRV *pFbbDrv = SearchPort(nPort);
	if (pFbbDrv == NULL)
		return FALSE;

	return ToTnc(pFbbDrv, nChan, nType, pszBuf, nLen, pUi);
} // end of WriteFbbDriver()

BOOL WINAPI StatFbbDriver(int nPort, int nCmd, int nChan, void *pPtr, int nLen)
{
	FBBDRV *pFbbDrv = SearchPort(nPort);
	if (pFbbDrv == NULL)
		return 0;
	
	switch (nCmd)
	{
	case DRV_SNDCMD:
	case DRV_ECHOCMD:
		return (ToTnc(pFbbDrv, nChan, DRV_COMMAND, (char *)pPtr, -1, NULL));
	case DRV_PORTCMD:
		return (ToTnc(pFbbDrv, 0, DRV_COMMAND, (char *)pPtr, -1, NULL));
	case DRV_PACLEN:
		*((int *) pPtr) = 250;
		return TRUE;
	case DRV_VERSION:
		wsprintf((char *)pPtr, 
			"v%d.%02d FBB driver for WA8Tfw (F6FBB-%s)", 
			TFW_MAJOR, TFW_MINOR, __DATE__);
		return TRUE;
	}

	return FALSE;
} // end of StatFbbDriver()


//////////////////////////////////////////////////////////////
//
// Local functions
//
//////////////////////////////////////////////////////////////

static FBBDRV *SearchPort(int nPort)
{
	FBBDRV *pTmp = pDrvHead;
	
	while (pTmp)
	{
		if (pTmp->nPort == nPort)
			break;
		pTmp = pTmp->pNext;
	}
	
	return pTmp;
}

static BOOL ToTnc(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi)
{
	DATAOUT *pPtr;
	
	if (nLen == -1)
		nLen = strlen(pData);

	if (nLen <= 0 || nLen > 256)
		return FALSE;
	
	if (pUi && nType == DRV_UNPROTO)
	{
		// Set the unproto path
		char szStr[256];
		int nUiLen = wsprintf(szStr, "C%s %s", pUi->szTo, pUi->szVia);
		ToTnc(pFbbDrv, 0, DRV_COMMAND, szStr, nUiLen, NULL);
	}

	pPtr           = (DATAOUT *)LocalAlloc( LPTR, sizeof(DATAOUT));
	pPtr->pData    = (char *)LocalAlloc( LPTR, nLen);
	pPtr->nChannel = nChannel;
	pPtr->nType    = nType;
	pPtr->nLen     = nLen;
	pPtr->pNext    = NULL;

	memcpy(pPtr->pData, pData, nLen);

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->pPort->hCritical);
	
	// Add the structure at end of list
	if (pFbbDrv->pTncHead == NULL)
		pFbbDrv->pTncHead = pPtr;
	else
		pFbbDrv->pTncTail->pNext = pPtr;
	
	// Update tail information
	pFbbDrv->pTncTail = pPtr;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->pPort->hCritical);

	return TRUE;
}

static void ToHost(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi)
{
	DATAIN *pPtr;
	
	if (nLen < 0 || nLen > 256)
		return;
	
	pPtr           = (DATAIN *)LocalAlloc( LPTR, sizeof(DATAIN));
	pPtr->nChannel = nChannel;
	pPtr->nType    = nType;
	pPtr->nLen     = nLen;
	pPtr->pUi      = pUi;
	pPtr->pNext    = NULL;

	if (nLen > 0)
	{
		pPtr->pData    = (char *)LocalAlloc( LPTR, nLen);
		memcpy(pPtr->pData, pData, nLen);
	}
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->pPort->hCritical);
	
	if (pFbbDrv->pHostHead == NULL)
		pFbbDrv->pHostHead = pPtr;
	else
		pFbbDrv->pHostTail->pNext = pPtr;
	
	// Update tail information
	pFbbDrv->pHostTail = pPtr;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->pPort->hCritical);
	
	// Send Notification message
	if (pFbbDrv->hWnd)
		PostMessage(pFbbDrv->hWnd, WM_NOTIFY_MSG, nType, MAKELONG(pFbbDrv->nPort, nChannel));
}

static DATAOUT *GetTnc(FBBDRV *pFbbDrv)
{
	DATAOUT *pPtr;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->pPort->hCritical);
	
	pPtr = pFbbDrv->pTncHead;
	if (pPtr)
		pFbbDrv->pTncHead = pFbbDrv->pTncHead->pNext;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->pPort->hCritical);
	
	return pPtr;
}

static DATAIN *GetHost(FBBDRV *pFbbDrv)
{
	DATAIN *pPtr;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->pPort->hCritical);
	
	pPtr = pFbbDrv->pHostHead;
	if (pPtr)
		pFbbDrv->pHostHead = pFbbDrv->pHostHead->pNext;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->pPort->hCritical);
	
	return pPtr;
}

static int TncQueueSize(FBBDRV *pFbbDrv)
{
	DATAOUT *pPtr;
	int nNb = 0;
	
	if (pFbbDrv->nNbResync)
		return 0;

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->pPort->hCritical);
	
	pPtr = pFbbDrv->pTncHead;
	while (pPtr)
	{
		pPtr = pPtr->pNext;
		++nNb;
	}
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->pPort->hCritical);
	
	return nNb;
}

static void HostInfo(FBBDRV *pFbbDrv, int nChannel, char *sFormat, ...)
{
	char	szStr[512];
	va_list pArgPtr;
	int nCnt;
	
	va_start(pArgPtr, sFormat);
	nCnt = wvsprintf(szStr, sFormat, pArgPtr);
	va_end(pArgPtr);
	
	if (nCnt > 0)
		ToHost(pFbbDrv, nChannel, DRV_INFO, szStr, nCnt, NULL);
}

static void HostEcho(FBBDRV *pFbbDrv, int nChannel, char *szTxt)
{
	int nCnt = strlen(szTxt);

	if (nCnt > 0)
		ToHost(pFbbDrv, nChannel, DRV_ECHOCMD, szTxt, nCnt, NULL);
}

static BOOL InitPort(FBBDRV *pFbbDrv)
{
	int nNb;
	char szBuf[256];

	char *szInit[] = {
		"MIUSC",
		"I",
		"M",
		NULL
	};

	wsprintf(szBuf, "i%s", pFbbDrv->szMyCall);
	ToTnc(pFbbDrv, 0, DRV_COMMAND, szBuf, -1, NULL);

	for (nNb = 0 ; szInit[nNb] ; nNb++)
		ToTnc(pFbbDrv, 0, DRV_COMMAND, szInit[nNb], -1, NULL);
	return TRUE;
}

static BOOL StartHostMode(FBBDRV *pFbbDrv)
{
	//	int nLen;
	int nChar;
	int nNb;
	BOOL bOk;
	char *pszInit = "\030\033JHOST\r\033MN\r";
	char *pszHost = "\033JHOST1\r";
	//	char szBuffer[80];
	
	HostInfo(pFbbDrv, 0, "StartHostMode");

	// Check if TNC is alive
	for (nNb = 0 ; nNb < 4 ; nNb++)
	{
		bOk = FALSE;
		WriteTncBlock(pFbbDrv, pszInit, strlen(pszInit));
		while ((nChar = ReadTncChar(pFbbDrv)) >= 0)
		{
			Sleep(10);
			bOk = TRUE;
		}
		
		if (bOk)
			break;
	}
	
	if (nNb == 4)
		return FALSE;
	
	// Go to hostmode
	WriteTncBlock(pFbbDrv, pszHost, strlen(pszHost));
	
	// Be sure the COM is empty ...
	
	while (ReadTncChar(pFbbDrv) >= 0)
		Sleep(10);
	
	// Should initialise the port now ...
	InitPort(pFbbDrv);
	
	HostInfo(pFbbDrv, 0, "StartHostMode OK");
	return TRUE;
}

static BOOL EndHostMode(FBBDRV *pFbbDrv)
{
	if (pFbbDrv->nNbInit != -1)
		return FALSE;

	// No monitoring
	ToTnc(pFbbDrv, 0, DRV_COMMAND, "MN", -1, NULL);

	// Command mode 
	ToTnc(pFbbDrv, 0, DRV_COMMAND, "JHOST0", -1, NULL);
	
	// Wait until all TNC queue is processed
	while (pFbbDrv->nNbInit == -1)
	{
		SendTncRequest(pFbbDrv->pPort);
		if (TncQueueSize(pFbbDrv) == 0)
			break;
		
		// Wait some time...
		Sleep(100);
	}
	
	return TRUE;
}

BOOL SetupConnection(HANDLE hDev, int nBaudrate)
{
	BOOL       fRetVal ;
	DCB        dcb ;
	
	dcb.DCBlength = sizeof( DCB ) ;
	
	GetCommState(hDev, &dcb ) ;
	
	dcb.BaudRate = nBaudrate ;
	dcb.ByteSize = 8 ;
	dcb.Parity   = NOPARITY ;
	dcb.StopBits = ONESTOPBIT ;
	
	
	// setup hardware flow control
	
	dcb.fOutxDsrFlow = FALSE ;
	dcb.fDtrControl = DTR_CONTROL_DISABLE ;
	
	dcb.fOutxCtsFlow = FALSE ;
	dcb.fRtsControl = RTS_CONTROL_DISABLE ;
	
	// setup software flow control
	
	dcb.fInX = dcb.fOutX = FALSE ;
	
	// other various settings
	
	dcb.fBinary = TRUE ;
	
	fRetVal = SetCommState(hDev, &dcb ) ;
	
	return ( fRetVal ) ;
	
} // end of SetupConnection()

static int ReadTncChar(FBBDRV *pFbbDrv)
{
	if ((*pFbbDrv->pPort->TfChck)())
		return (*pFbbDrv->pPort->TfGet)();

	return -1;
} // end of ReadTncChar()

static BOOL WriteTncBlock(FBBDRV *pFbbDrv, LPSTR lpByte , DWORD dwBytesToWrite)
{
	DWORD i;

	for (i = 0 ; i < dwBytesToWrite ; i++)
	{
		(*pFbbDrv->pPort->TfPut)(lpByte[i]);
	}

	return ( TRUE ) ;	
} // end of WriteTncBlock()


static void TfwUiHeader(int nPort, char *szBuf, DRVUI *pUi)
{
	char *pPtr;
	char *pScan;

	memset (pUi, 0, sizeof(DRVUI));

	if (isdigit(*szBuf))
		szBuf += 3;

	pUi->nPort = nPort;

	pScan = szBuf;

	pPtr = strtok (pScan, " ");	/* fm */

	if (pPtr == NULL)
		return;

	pPtr = strtok (NULL, " ");	/* exped */

	if (pPtr == NULL)
		return;
	strncpy (pUi->szFrom, pPtr, 9);
	pUi->szFrom[9] = '\0';

	pPtr = strtok (NULL, " ");	/* to */

	if (pPtr == NULL)
		return;

	pPtr = strtok (NULL, " ");	/* desti */

	if (pPtr == NULL)
		return;
	strncpy (pUi->szTo, pPtr, 9);
	pUi->szTo[9] = '\0';

	pPtr = strtok (NULL, " ");	/* via ou ctl */

	if (pPtr == NULL)
		return;

	if (strcmp (pPtr, "via") == 0)
	{
		for (;;)
		{
			if (*pUi->szVia)
				strcat (pUi->szVia, " ");

			pPtr = strtok (NULL, " ");	/* digis */

			if (pPtr == NULL)
				return;

			if (strcmp (pPtr, "ctl") == 0)
				break;

			strncat (pUi->szVia, pPtr, 9);
		}
		pUi->szVia[80] = '\0';
	}

	pPtr = strtok (NULL, " ");	/* controle */

	if (pPtr == NULL)
		return;
	strncpy (pUi->szCtl, pPtr, 9);
	pUi->szCtl[9] = '\0';

	pUi->bUi = (strncmp (pPtr, "UI", 2) == 0);

	pPtr = strtok (NULL, " ");	/* pid */

	if (pPtr == NULL)
		return;

	pPtr = strtok (NULL, " ");	/* pid */

	if (pPtr == NULL)
		return;
	sscanf (pPtr, "%x", &pUi->nPid);
}

static void SendTncRequest(FBBPORT *pPort)
{
	static int nChan = 0;
	int nLen;
	char szBuf[259];
	DATAOUT *pPtr;
	FBBDRV *pFbbDrv;
	
	pFbbDrv = pPort->pDrv;

	if (pFbbDrv->bNext)
	{
		// Got header then get data of monitoring
		szBuf[0] = nChan;
		szBuf[1] = 1;
		szBuf[2] = 0;
		szBuf[3] = 'G';
		pFbbDrv->nLastCmd = 'G';

		nLen = 4;

		pFbbDrv->bNext = FALSE;
	}
	else if (pPtr = GetTnc(pFbbDrv))
	{
		switch(pPtr->nType)
		{
		case DRV_COMMAND:
			szBuf[0] = pPtr->nChannel;
			szBuf[1] = 1;
			break;
		case DRV_DATA:
			szBuf[0] = pPtr->nChannel;
			szBuf[1] = 0;
			break;
		case DRV_UNPROTO:
			szBuf[0] = 0;
			szBuf[1] = 0;
		}
		szBuf[2] = pPtr->nLen - 1;
		memcpy(szBuf+3, pPtr->pData, pPtr->nLen);
		pFbbDrv->nLastCmd = 0;

		nLen = pPtr->nLen + 3;


		LocalFree(pPtr->pData);
		LocalFree(pPtr);
	}
	else
	{
		// Polling for link status
		szBuf[0] = nChan;
		szBuf[1] = 1;
		szBuf[2] = 0;
		szBuf[3] = 'L';
		pFbbDrv->nLastCmd = 'L';

		nLen = 4;

		if (++nChan > pFbbDrv->nNbChan)
			nChan = 0;

		// Next time is Get frame
		pFbbDrv->bNext = TRUE;
	}
	
	WriteTncBlock(pFbbDrv, szBuf, nLen);
}

static void ReadTncAnswer(FBBPORT *pPort)
{
	int nChan;
	int nCode;
	int nChar;
	int nLen;
	int nNb;
	char *pPtr;
	char szBuffer[256];
	FBBDRV *pFbbDrv;

	pFbbDrv = pPort->pDrv;

	if (pFbbDrv->nNbResync > 0)
		return;
	
	nChan = ReadTncChar( pFbbDrv);
	if (nChan == -1)
	{
		// Resynchro
		char bChar = '\001';
		
		while (nChan == -1)
		{
			WriteTncBlock(pFbbDrv, &bChar, 1);
			nChan = ReadTncChar( pFbbDrv);
			HostInfo(pFbbDrv, 0, "Resync:%d", ++(pFbbDrv->nNbResync));
			if (pFbbDrv->nNbResync == 256)
			{
				// Try re-init the TNC
				pFbbDrv->nNbInit = 5;
			}

			if (!pFbbDrv->pPort->bUp || (pFbbDrv->nNbResync == 256))
				return;
		}
	}
	
	pFbbDrv->nNbResync = 0;

	nCode = ReadTncChar( pFbbDrv);
	pPtr = szBuffer;
	
	switch (nCode)
	{
	case 0:
		if (pFbbDrv->nLastCmd == 'L')
		{
			// Should never occur
		}
		break;
	case 1:
		for (;;)
		{
			nChar = ReadTncChar(pFbbDrv);
			if (nChar == -1)
				return;
			*pPtr = nChar;
			if (*pPtr == '\0')
				break;
			++pPtr;
		}
		if (pFbbDrv->nLastCmd != 'L')
		{
			HostEcho(pFbbDrv, nChan, szBuffer);
		}
		else
		{
			DRVSTATS DrvStats;
			int nbmes, nbtra, nbatt, nback, nbret, con;

			nbatt = nback = nbret = con = 0;

			sscanf (szBuffer, "%d%d%d%d%d%d",
						&nbmes, &nbtra, &nbatt, &nback, &nbret, &con);

			DrvStats.nNbFrames = nbmes + nbtra;
			DrvStats.nNbRetry  = nbret;
			switch(con)
			{
			case 0:
				DrvStats.nState = DRV_DISCONNECTED;
				break;
			case 1:
				DrvStats.nState = DRV_CONNPENDING;
				break;
			case 3:
				DrvStats.nState = DRV_DISCPENDING;
				break;
			default:
				DrvStats.nState = DRV_CONNECTED;
				break;
			}
			DrvStats.nBuffers  = 500;
			DrvStats.nFrameNb  = 0;
			if (DrvStats.nNbFrames != pFbbDrv->nNbFrm[nChan] || 
				DrvStats.nNbRetry  != pFbbDrv->nNbRet[nChan] || 
				DrvStats.nState    != pFbbDrv->nState[nChan])
			{
				pFbbDrv->nNbFrm[nChan] = DrvStats.nNbFrames;
				pFbbDrv->nNbRet[nChan] = DrvStats.nNbRetry;
				pFbbDrv->nState[nChan] = DrvStats.nState;
				ToHost(pFbbDrv, nChan, DRV_STATS, (char *)&DrvStats, sizeof(DRVSTATS), NULL);
			}
		}
		break;
	case 2:
		for (;;)
		{
			nChar = ReadTncChar(pFbbDrv);
			if (nChar == -1)
				return;
			*pPtr = nChar;
			if (*pPtr == '\0')
				break;
			++pPtr;
		}
		HostEcho(pFbbDrv, nChan, szBuffer);
		break;
	case 3:
		for (;;)
		{
			nChar = ReadTncChar(pFbbDrv);
			if (nChar == -1)
				return;
			*pPtr = nChar;
			if (*pPtr == '\0')
				break;
			++pPtr;
		}
		ToHost(pFbbDrv, nChan, DRV_COMMAND, szBuffer, strlen(szBuffer), NULL);
		break;
	case 4:
		for (;;)
		{
			nChar = ReadTncChar(pFbbDrv);
			if (nChar == -1)
				return;
			*pPtr = nChar;
			if (*pPtr == '\0')
				break;
			++pPtr;
		}
		pFbbDrv->pUi = (DRVUI *)LocalAlloc(LPTR, sizeof(DRVUI));
		if (pFbbDrv->pUi)
			TfwUiHeader(pFbbDrv->nPort, szBuffer, pFbbDrv->pUi);
		ToHost(pFbbDrv, nChan, DRV_UNPROTO, NULL, 0, pFbbDrv->pUi);
		pFbbDrv->pUi = NULL;
		break;
	case 5:
		for (;;)
		{
			nChar = ReadTncChar(pFbbDrv);
			if (nChar == -1)
				return;
			*pPtr = nChar;
			if (*pPtr == '\0')
				break;
			++pPtr;
		}
		pFbbDrv->pUi = (DRVUI *)LocalAlloc(LPTR, sizeof(DRVUI));
		if (pFbbDrv->pUi)
			TfwUiHeader(pFbbDrv->nPort, szBuffer, pFbbDrv->pUi);
		pFbbDrv->bNext = TRUE;
		break;
	case 6:
		nLen = ReadTncChar(pFbbDrv) + 1;
		for (nNb = 0 ; nNb < nLen ; nNb++)
		{
			nChar = ReadTncChar(pFbbDrv);
			if (nChar == -1)
				return;
			szBuffer[nNb] = nChar;
		}
		szBuffer[nNb] = '\0';
		ToHost(pFbbDrv, nChan, DRV_UNPROTO, szBuffer, nLen, pFbbDrv->pUi);
		pFbbDrv->pUi = NULL;
		break;
	case 7:
		nLen = ReadTncChar(pFbbDrv) + 1;
		for (nNb = 0 ; nNb < nLen ; nNb++)
			szBuffer[nNb] = ReadTncChar(pFbbDrv);
		szBuffer[nNb] = '\0';
		ToHost(pFbbDrv, nChan, DRV_DATA, szBuffer, nLen, pFbbDrv->pUi);
		break;
	default:
		// Empty the queues
		break;
	}
}

static BOOL NextTnc(FBBPORT *pPort)
{
	FBBDRV *pFbbDrv;
	static int nMuxPrec = -1;

	if (pPort->nWait)
	{
		// Keep the thread out of the loop
		if (pPort->nWait == 1)
			pPort->nWait = 2;
		Sleep(100);
		return FALSE;
	}

	Sleep(20);

	// Must be thread safe
	EnterCriticalSection(&pPort->hCritical);

	// Next TNC
	if (++pPort->nCurTnc == pPort->nNbTnc)
		pPort->nCurTnc = 0;

	pFbbDrv = pPort->pDrv;

	// Must be thread safe
	LeaveCriticalSection(&pPort->hCritical);

	if (pFbbDrv->nNbInit == 0)
	{
		// TNC stopped
		Sleep(10);
		return FALSE;
	}

	// Optimise
	if (pFbbDrv->nNbMultCh != nMuxPrec)
	{
		/*
		// Set the signals for the MUX
		nMuxPrec = pFbbDrv->nNbMultCh;
		EscapeCommFunction(pPort->hDev, (nMuxPrec & 1) ? CLRDTR : SETDTR);
		EscapeCommFunction(pPort->hDev, (nMuxPrec & 2) ? CLRRTS : SETRTS);
		*/
	}

	if (pFbbDrv->nNbInit > 0)
	{
		// Init TNC
		--pFbbDrv->nNbInit;

		//DEBUG
		int nNbInitBack = pFbbDrv->nNbInit;
		pFbbDrv->nNbInit = -1;
		EndHostMode(pFbbDrv);
		pFbbDrv->nNbInit = nNbInitBack;
		//DEBUG

		if (!StartHostMode(pFbbDrv))
			return FALSE;

		pFbbDrv->nNbInit = -1;
		pFbbDrv->nNbResync = 0;
	}

	return TRUE;
}

// TNC Thread

DWORD WINAPI TfwLoop(LPSTR lpData)
{
	FBBPORT   *pPort = (FBBPORT*) lpData ;
	
	while (pPort->bUp)
	{
		// Setup the TNC to hostmode
		if (NextTnc(pPort))
		{
			// Send the request
			SendTncRequest(pPort);
		
			// Read the answer
			ReadTncAnswer(pPort);
		}
	}
	
	return( TRUE ) ;
	
} // end of TfwLoop()

BOOL GetDrvVersion(int *nVMaj, int *nVMin)
{
	DWORD dwVersionInfoSize;
	DWORD dwTmpHandle;
	LPVOID lpVersionInfo; 
	BOOL bRet = false;


	dwVersionInfoSize = GetFileVersionInfoSize(FBBDLLNAME, &dwTmpHandle);

	if(dwVersionInfoSize)
	{
		lpVersionInfo = LocalAlloc(LPTR, dwVersionInfoSize);
		if(lpVersionInfo)
		{
			if(GetFileVersionInfo(FBBDLLNAME, dwTmpHandle, dwVersionInfoSize, lpVersionInfo))
			{
				LPVOID lpBuffer = LocalAlloc(LPTR, dwVersionInfoSize);
				UINT dwBytes;

				if( VerQueryValue(lpVersionInfo, 
						TEXT("\\StringFileInfo\\000004B0\\FileVersion"), 
						&lpBuffer, 
						&dwBytes)) 
				{
						sscanf((TCHAR *)lpBuffer, "%d,%d", nVMaj, nVMin);
						bRet = true;
				}

				LocalFree(lpBuffer);
			}

			LocalFree(lpVersionInfo);
 		}
 	}

	return bRet;
}
