
//////////////////////////////////////////////////////////////
//
// FBB driver for PTC-II host mode
//
// File : fbb_ptc.cpp
//
// (C) F6FBB 1999
//
//////////////////////////////////////////////////////////////
//
//  Version history
//
//	    - 06/01/2001 - EB5AGF
//		Added version resource and code to read it at runtime
//
//////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <winver.h>
#include "../fbb_drv.h"

#define	FBBDLLNAME	"ptc.dll"

/* Header characters */
#define HST_CHAR	170
#define PACTOR_ST	254
#define PACTOR_CH	31
#define MAXCHAN		32

#define PACTOR	0
#define MODEM1	1
#define MODEM2	2
#define NBMODEM	3

#define G_PHASE 1
#define L_PHASE 2
#define P_PHASE 3

#define uchar unsigned char

#define PACTOR_BUSY(p)		(((p)->nStatus == 0xf7) ? 1 : 0)
#define PACTOR_ONLINE(p)	((((p)->nStatus & 0x07) != 0x07) ? 1 : 0)
#define ISS(p)				(((p)->nStatus & 0x08) ? 1 : 0)

//------------

int PTC_MAJOR = 0;
int PTC_MINOR = 0;

typedef struct DATAOUT_t {
	uchar	*pData;
	int		nChannel;
	int		nType;
	int		nLen;
	struct	DATAOUT_t *pNext;
} DATAOUT;

typedef struct DATAIN_t {
	uchar	*pData;
	int		nCPort;
	int		nType;
	int		nLen;
	DRVUI	*pUi;
	struct	DATAIN_t *pNext;
} DATAIN;

typedef struct Timer {
	long nCount;
	void (*pProc)(void *);
	void *pData;
	struct Timer *pNext;
} TIMER;

typedef struct {
	char szCall[14];
	TIMER *pTimer;
	TIMER *pIss;
} PACCALL;

typedef struct FBBDRV_t
{
	HWND	hWnd;
	int		nPort[NBMODEM];	// Port for modems
	int		nNbChan;		// Total number of packet channels
	int		nNbChan1;		// Nb of packet channels on 1st port
	int		nNbChan2;		// Nb of packet channels on 2nd port
	int		nCurChan1;		// Active nb of packet channels on 1st port
	int		nCurChan2;		// Active nb of packet channels on 2nd port
	int		nChanL;
	int		nChanG;
	int		nPacStatus;
	int		nInitP;
	long	nFbbCPort[MAXCHAN];
	uchar	nChanList[MAXCHAN];
	uchar	nNbFrm[MAXCHAN];
	uchar	nNbRet[MAXCHAN];
	uchar	nState[MAXCHAN];
	uchar	nAux[MAXCHAN];
	HANDLE	hDev;
	HANDLE	hThread;
	DWORD	dwThreadId;
	TIMER	*pTimerHead;
	int		nCom;
	int		nBaudrate;
	int		nLastCmd;
	int		nPhase;
	int		nStatus;
	int		nStatusPrec;
	BOOL	bAgain;
	BOOL	bInit;
	BOOL	bCounter;
	BOOL	bUp;
	BOOL	bPactor;
	BOOL	bIssDelta;
	char	szMyCall[10];
	uchar	szSend[520];// Last data sent
	int		nSize;		// Size of the last data
	DRVUI	*pUi;
	PACCALL	*pPacCall;	// Pactor Call parameters
	DATAOUT	*pTncHead;	// Head of data_out list
	DATAOUT	*pTncTail;	// Tail of data_out list
	DATAIN	*pHostHead;	// Head of data_in list
	DATAIN	*pHostTail;	// Tail of data_in list
	CRITICAL_SECTION	hCritical;
	struct	FBBDRV_t *pNext;
} FBBDRV;


//////////////////////////////////////////////////////////////
// Local prototypes
//////////////////////////////////////////////////////////////

static void HstIss(void *pData);
static void HstBreak(FBBDRV *pFbbDrv);
static void HstOver(FBBDRV *pFbbDrv);
static BOOL PtcInit(FBBDRV *pFbbDrv, int nChan, char *pPtr);
static BOOL SetupConnection(HANDLE hDev, int nBaudRate);
static DWORD WINAPI DedLoop( LPSTR lpData );
static FBBDRV *SearchPort(int nPort);
static FBBDRV *SearchCom(int nCom);
static int PtcPort(FBBDRV *pFbbDrv, int nFbbPort);
static int TncQueueSize(FBBDRV *pFbbDrv, int nChan, int nType);
static BOOL StartHostMode(FBBDRV *pFbbDrv, int nPort);
static BOOL EndHostMode(FBBDRV *pFbbDrv);
static BOOL WriteTncBlock(FBBDRV *pFbbDrv, LPSTR lpByte , DWORD dwBytesToWrite);
static int ReadTncChar(FBBDRV *pFbbDrv);
static DATAIN *GetHost(FBBDRV *pFbbDrv);
static BOOL ToTnc(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi);
static void InitCrc(void);
static void SendTncRequest(FBBDRV *pFbbDrv);
static TIMER *AddTimer(FBBDRV *pFbbDrv, long nCount, void (*pProc)(void *), void *pData);
static void *DelTimer(FBBDRV *pFbbDrv, TIMER *pTimer);
static int Tnc2FbbChannel(FBBDRV *pFbbDrv, int nChan);
static int Tnc2FbbPort(FBBDRV *pFbbDrv, int nChan);
static int Fbb2TncChannel(FBBDRV *pFbbDrv, int nFbbPort, int nFbbChan);
static BOOL InitPort(FBBDRV *pFbbDrv);
static void PactorConnect(FBBDRV *pFbbDrv, char *szCall);
static BOOL GetDrvVersion(int *nVMaj, int *nVMin);


//////////////////////////////////////////////////////////////
// Global variables
//////////////////////////////////////////////////////////////

FBBDRV	*pDrvHead = NULL;	// Head of port structures

unsigned short nzCrcTab[256];	// CRC table
#define updcrc(cp, crc) (((crc >> 8) & 0xff) ^ nzCrcTab[((crc ^ (cp & 0xff)) & 0xff)])

//////////////////////////////////////////////////////////////
//
// Exported functions (5).
// They are :
// BOOL OpenFbbDriver(int nPort, HWND hWnd, void *pDrvInfo);
// BOOL CloseFbbDriver(int nPort);
// BOOL ReadFbbDriver(int nPort, int *nType, int *nChan, char *pszBuf, int *nLen, DRVUI *pUi);
// BOOL WriteFbbDriver(int nPort, int *nType, int nChan, char *pszBuf, int nLen, DRVUI *pUi);
// BOOL StatFbbDriver(int nPort, int nChan, int nCmd, void *pPtr, int nLen);
//
//////////////////////////////////////////////////////////////


BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
    switch (ul_reason_for_call)
	{
		case DLL_PROCESS_ATTACH:
		case DLL_THREAD_ATTACH:
			GetDrvVersion(&PTC_MAJOR, &PTC_MINOR);
		case DLL_THREAD_DETACH:
			break;
		case DLL_PROCESS_DETACH:
			// Clean all linked ports
			while (pDrvHead)
				CloseFbbDriver(pDrvHead->nPort[0]);
			break;
    }
    return TRUE;
}


BOOL WINAPI OpenFbbDriver(int nPort, HWND hWnd, void *pDrvInfo)
{
	char	szPort[ 15 ];
	BOOL	fRetVal ;
	FBBDRV	*pFbbDrv;
	HANDLE	hDev;
	HANDLE	hFbbThread;
	DWORD	dwThreadID;
	DRVINFO *pInfo;
	COMMTIMEOUTS  CommTimeOuts ;
	
	
	pInfo = (DRVINFO *)pDrvInfo;
	if (pInfo->nMultCh > 2)
		return FALSE;
	
	// COM already used ?
	pFbbDrv = SearchCom(pInfo->nCom);
	if (!pFbbDrv)
	{
		// open COMM device
		
		wsprintf( szPort, "COM%d", pInfo->nCom ) ;
	
		if ((hDev =	CreateFile(
			szPort, 
			GENERIC_READ | GENERIC_WRITE,
			0,                    // exclusive access
			NULL,                 // no security attrs
			OPEN_EXISTING,
			FILE_ATTRIBUTE_NORMAL, // |
			//FILE_FLAG_OVERLAPPED, // overlapped I/O
			NULL )) == (HANDLE) -1 )
		{
			return FALSE;
		}

		// setup device buffers
		SetupComm(hDev, 1024, 1024 ) ;
		
		// purge any information in the buffer
		PurgeComm(hDev, PURGE_TXCLEAR | PURGE_RXCLEAR ) ;
		
		// Set time out
		CommTimeOuts.ReadIntervalTimeout = 0 ;
		CommTimeOuts.ReadTotalTimeoutMultiplier = 0 ;
		CommTimeOuts.ReadTotalTimeoutConstant = 200 ;
		
		// CBR_9600 is approximately 1byte/ms. For our purposes, allow
		// double the expected time per character for a fudge factor.
		CommTimeOuts.WriteTotalTimeoutMultiplier = 2*CBR_9600/pInfo->nBaudrate;
		CommTimeOuts.WriteTotalTimeoutConstant = 0 ;
		
		SetCommTimeouts(hDev, &CommTimeOuts ) ;
		
		if (!SetupConnection(hDev, pInfo->nBaudrate))
			fRetVal = FALSE;
		
		EscapeCommFunction(hDev, SETDTR );
		EscapeCommFunction(hDev, SETRTS );
		
		pFbbDrv = (FBBDRV *)LocalAlloc(LPTR, sizeof(FBBDRV));
		if (pFbbDrv == NULL)
			return FALSE;
		
		pFbbDrv->pUi     = NULL;
		pFbbDrv->bAgain  = FALSE;
		pFbbDrv->hDev    = hDev;
		pFbbDrv->bUp     = TRUE;
		pFbbDrv->hWnd    = hWnd;
		pFbbDrv->nCom    = pInfo->nCom;
		pFbbDrv->nPhase  = G_PHASE;
		pFbbDrv->nInitP  = nPort;
		
		strcpy(pFbbDrv->szMyCall, pInfo->szMyCall);
		
		// Insert the structure in the list
		pFbbDrv->pNext = pDrvHead;
		pDrvHead = pFbbDrv;

		if (hWnd)
			WM_NOTIFY_MSG = RegisterWindowMessage(FBBDRV_NOTIFY);
		
		// Init CRC table
		InitCrc();
		InitializeCriticalSection(&pFbbDrv->hCritical);
		
		if (StartHostMode(pFbbDrv, pInfo->nMultCh))
		{
			
			// Create a thread to process the DED protocol.
			hFbbThread = CreateThread( 
				(LPSECURITY_ATTRIBUTES) NULL,
				0,
				(LPTHREAD_START_ROUTINE) DedLoop,
				(LPVOID) pFbbDrv,
				0, 
				&dwThreadID );
		}
		else
			fRetVal = FALSE;
		
		
		if ((hFbbThread == NULL) || (fRetVal == FALSE))
		{
			EscapeCommFunction(hDev, CLRDTR );
			EscapeCommFunction(hDev, CLRRTS );
			CloseHandle(hDev);
			DeleteCriticalSection(&pFbbDrv->hCritical);
			
			// Remove structure from list
			pDrvHead = pDrvHead->pNext;
			
			// Free the structure
			LocalFree(pFbbDrv);

			return FALSE;
		}
		
		pFbbDrv->dwThreadId	= dwThreadID;
		pFbbDrv->hThread	= hFbbThread;
		
	}

	if (pFbbDrv->nPort[pInfo->nMultCh])
	{
		// Port already initialized
		return FALSE;
	}

	pFbbDrv->nPort[pInfo->nMultCh] = nPort;
	switch (pInfo->nMultCh)
	{
	case PACTOR:
		pFbbDrv->bPactor = TRUE;
		break;
	case MODEM1:
		pFbbDrv->nNbChan += pInfo->nNbChan;
		pFbbDrv->nNbChan1 = pInfo->nNbChan;
		break;
	case MODEM2:
		pFbbDrv->nNbChan += pInfo->nNbChan;
		pFbbDrv->nNbChan2 = pInfo->nNbChan;
		break;
	}
	
	// Should initialise the port now ...
	InitPort(pFbbDrv);
	
	return TRUE;
	
} // end of OpenFbbDriver()

BOOL WINAPI CloseFbbDriver(int nPort)
{
	int nI;
	FBBDRV *pFbbDrv = SearchPort(nPort);
	if (pFbbDrv == NULL)
		return FALSE;

	// Close COM and shutdown TNC only if last port is closed
	pFbbDrv->nPort[PtcPort(pFbbDrv, nPort)] = 0;
	for (nI = 0 ; nI < NBMODEM ; nI++)
		if (pFbbDrv->nPort[nI])
			return TRUE;

	// TNC to command mode
	EndHostMode(pFbbDrv);
	
	// block until thread has been halted	
	pFbbDrv->bUp = FALSE;
	WaitForSingleObject(pFbbDrv->hThread, INFINITE);
	
	// drop DTR
	EscapeCommFunction(pFbbDrv->hDev, CLRDTR ) ;
	EscapeCommFunction(pFbbDrv->hDev, CLRRTS ) ;
	
	// purge any outstanding reads/writes and close device handle
	PurgeComm(pFbbDrv->hDev, PURGE_TXCLEAR | PURGE_RXCLEAR ) ;
	CloseHandle(pFbbDrv->hDev) ;
	
	DeleteCriticalSection(&pFbbDrv->hCritical);
	
	// Remove structure from list
	if (pDrvHead == pFbbDrv)
		pDrvHead = pDrvHead->pNext;
	else
	{
		FBBDRV *pTmp = pDrvHead;
		while (pTmp->pNext)
		{
			if (pTmp->pNext == pFbbDrv)
			{
				pTmp->pNext = pTmp->pNext->pNext;
				break;
			}
			pTmp = pTmp->pNext;
		}
	}

	LocalFree(pFbbDrv);

	return ( TRUE ) ;
	
} // end of CloseFbbDriver()

BOOL WINAPI ReadFbbDriver(int *nPort, int *nType, int *nChan, char *pszBuf, int *nLen, DRVUI *pUi)
{
	DATAIN *pPtr;
	FBBDRV *pFbbDrv = SearchPort(*nPort);
	if (pFbbDrv == NULL)
		return FALSE;
	
	pPtr = GetHost(pFbbDrv);
	if (pPtr == NULL)
		return FALSE;

	if (pPtr->nLen)
	{
		memcpy(pszBuf, pPtr->pData, pPtr->nLen);
		LocalFree(pPtr->pData);
	}
	*nLen  = pPtr->nLen;
	*nChan = HIWORD(pPtr->nCPort);
	*nPort = LOWORD(pPtr->nCPort);
	*nType = pPtr->nType;
	
	if (pPtr->pUi)
	{
		*pUi = *pPtr->pUi;
		LocalFree(pPtr->pUi);
	}
	LocalFree(pPtr);
	
	return TRUE;
} // end of ReadFbbDriver()

BOOL WINAPI WriteFbbDriver(int nPort, int nType, int nChan, char *pszBuf, int nLen, DRVUI *pUi)
{
	int nPtcPort;
	int nTncChan;
	char szStr[80];
	FBBDRV *pFbbDrv = SearchPort(nPort);

	if (pFbbDrv == NULL)
		return FALSE;

	switch (nType)
	{
	case DRV_DATA:
		nTncChan = Fbb2TncChannel(pFbbDrv, nPort, nChan);
		if (nTncChan == PACTOR_CH && TncQueueSize(pFbbDrv, PACTOR_CH, DRV_DATA) == 0)
			HstBreak(pFbbDrv);
		return ToTnc(pFbbDrv, nTncChan, DRV_DATA, pszBuf, nLen, NULL);
	case DRV_COMMAND:
		nPtcPort = PtcPort(pFbbDrv, nPort);
		if (nPtcPort != PACTOR && toupper(pszBuf[0]) == 'C')
		{
			// It is a packet outgoing connection. We
			// allocate a new channel : we look for a
			// free channel starting from the last channel

			for (nTncChan = pFbbDrv->nNbChan ; nTncChan > 0 ; nTncChan--)
			{
				if (pFbbDrv->nFbbCPort[nTncChan] == 0)
				{
					pFbbDrv->nFbbCPort[nTncChan] = MAKELONG(nPort, nChan);
					break;
				}
			}

			if (nTncChan == 0)
			{
				// Send busy message ?
				return FALSE;
			}
		}
		else
		{
			nTncChan = Fbb2TncChannel(pFbbDrv, nPort, nChan);
		}
		return ToTnc(pFbbDrv, nTncChan, DRV_COMMAND, szStr, -1, NULL);

	case DRV_UNPROTO:
		nPtcPort = PtcPort(pFbbDrv, nPort);

		/* Select the port */
		if (nPtcPort == 0)
		{
			// No beacon to pactor port
			return FALSE;
		}

		wsprintf (szStr, "%%P%d", nPtcPort);
		nTncChan = Fbb2TncChannel(pFbbDrv, nPort, nChan);
		ToTnc(pFbbDrv, 0, DRV_COMMAND, szStr, -1, NULL);
		return ToTnc(pFbbDrv, nTncChan, nType, pszBuf, nLen, pUi);
	}

	return FALSE;
} // end of WriteFbbDriver()

BOOL WINAPI StatFbbDriver(int nPort, int nCmd, int nChan, void *pPtr, int nLen)
{
	char szStr[80];
	int nPtcPort;
	FBBDRV *pFbbDrv = SearchPort(nPort);

	if (pFbbDrv == NULL)
		return 0;
	
	switch (nCmd)
	{
	case DRV_SNDCMD:
	case DRV_ECHOCMD:
		if (nLen == 0)
			return 0;
	
		nPtcPort = PtcPort(pFbbDrv, nPort);

		if (nPtcPort == 0)
		{
			char *pStr = (char *)pPtr;

			// Pactor port
			nChan = PACTOR_CH;

			if (*pStr == 'C')
			{
				// Ignore connections if not in DISC state.
				if (pFbbDrv->nPacStatus == DRV_DISCONNECTED)
				{
					// Connect on pactor must be delayed
					pStr[nLen] = '\0';
					PactorConnect(pFbbDrv, pStr);
					return TRUE;
				}
				else
				{
					return FALSE;
				}
			}
		}
		else
		{
			/* Select the port */
			wsprintf (szStr, "%%P%d", nPtcPort);
			ToTnc(pFbbDrv, 0, DRV_COMMAND, szStr, -1, NULL);
		}
		return (ToTnc(pFbbDrv, nChan, DRV_COMMAND, (char *)pPtr, nLen, NULL));
	case DRV_INIT:
		if (nLen == 0)
			return 0;
		return PtcInit(pFbbDrv, nChan, (char *)pPtr);
	case DRV_PORTCMD:
		if (nLen == 0)
			return 0;
		return (ToTnc(pFbbDrv, 0, DRV_COMMAND, (char *)pPtr, nLen, NULL));
	case DRV_PACLEN:
		if (nLen == 0)
			return 0;
		*((int *) pPtr) = 250;
		return TRUE;
	case DRV_VERSION:
		wsprintf((char *)pPtr, 
			"v%d.%02d FBB driver for PTC-II (F6FBB-%s)", 
			PTC_MAJOR, PTC_MINOR, __DATE__);
		return TRUE;
	}

	return FALSE;
} // end of StatFbbDriver()


//////////////////////////////////////////////////////////////
//
// Local functions
//
//////////////////////////////////////////////////////////////

static BOOL PtcInit(FBBDRV *pFbbDrv, int nChan, char *pPtr)
{
	BOOL bRes = FALSE;

	if (*pPtr == '!')
	{
		*pPtr = '#';
		ToTnc (pFbbDrv, nChan, DRV_COMMAND, pPtr, -1, NULL);
		return TRUE;
	}

	return bRes;
}

static void InitCrc(void)
{
	unsigned int nI, nJ;
	unsigned short nAccu, nData;

	for (nI = 0 ; nI < 256 ; nI++)
	{
		nAccu = 0;
		nData = nI;
		for (nJ = 0 ; nJ < 8 ; nJ++)
		{
			if ((nData ^ nAccu) & 1)
				nAccu = (nAccu >> 1) ^ 0x8408;
			else
				nAccu = nAccu >> 1;
			nData >>= 1;
		}
		nzCrcTab[nI] = nAccu;
	}
}

static int Tnc2FbbChannel(FBBDRV *pFbbDrv, int nChan)
{
	// Remaps logical channels/ports to FBB physical channels/ports
	if (nChan == PACTOR_CH)
		return 1;

	return HIWORD(pFbbDrv->nFbbCPort[nChan]);
}

static int Tnc2FbbPort(FBBDRV *pFbbDrv, int nChan)
{
	// Remaps logical channels/ports to FBB physical channels/ports
	if (nChan == PACTOR_CH)
		return 1;

	return LOWORD(pFbbDrv->nFbbCPort[nChan]);
}

static int Fbb2TncChannel(FBBDRV *pFbbDrv, int nFbbPort, int nFbbChan)
{
	int nI;
	long nCPort = MAKELONG(nFbbPort, nFbbChan);

	// Remaps FBB physical channels/ports to logical channels/ports

	if (nFbbChan == 0)
		return 0;

	if (PtcPort(pFbbDrv, nFbbPort) == PACTOR)
		return PACTOR_CH;

	for (nI = 1 ; nI < PACTOR_CH ; nI++)
		if (pFbbDrv->nFbbCPort[nI] == nCPort)
			return nI;

	return 0;
}

static int PtcPort(FBBDRV *pFbbDrv, int nFbbPort)
{
	int nI;

	for (nI = 0 ; nI < NBMODEM ; nI++)
	{
		if (pFbbDrv->nPort[nI] == nFbbPort)
			return nI;
	}
	return 0;
}

static FBBDRV *SearchPort(int nPort)
{
	int nI;
	FBBDRV *pTmp = pDrvHead;
	
	while (pTmp)
	{
		for (nI = 0 ; nI < NBMODEM ; nI++)
			if (pTmp->nPort[nI] == nPort)
				return pTmp;
		pTmp = pTmp->pNext;
	}
	
	return NULL;
}

static FBBDRV *SearchCom(int nCom)
{
	FBBDRV *pTmp = pDrvHead;
	
	while (pTmp)
	{
		if (pTmp->nCom == nCom)
			return pTmp;
		pTmp = pTmp->pNext;
	}
	
	return NULL;
}

static BOOL ToTnc(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi)
{
	DATAOUT *pPtr;
	
	if (nLen == -1)
		nLen = strlen(pData);

	if (nLen <= 0 || nLen > 256)
		return FALSE;
	
	if (pUi && nType == DRV_UNPROTO)
	{
		int nUiLen;
		char szStr[256];

		if (pUi->nPort == 0)
		{
			// No beacon on the pactor port
			return FALSE;
		}

		// Set the unproto path
		nUiLen = wsprintf(szStr, "C%s %s", pUi->szTo, pUi->szVia);
		ToTnc(pFbbDrv, 0, DRV_COMMAND, szStr, nUiLen, NULL);
	}

	pPtr           = (DATAOUT *)LocalAlloc( LPTR, sizeof(DATAOUT));
	pPtr->pData    = (uchar *)LocalAlloc( LPTR, nLen);
	pPtr->nChannel = nChannel;
	pPtr->nType    = nType;
	pPtr->nLen     = nLen;
	pPtr->pNext    = NULL;

	memcpy(pPtr->pData, pData, nLen);

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	// Add the structure at end of list
	if (pFbbDrv->pTncHead == NULL)
		pFbbDrv->pTncHead = pPtr;
	else
		pFbbDrv->pTncTail->pNext = pPtr;
	
	// Update tail information
	pFbbDrv->pTncTail = pPtr;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);

	return TRUE;
}

static void ToHost(FBBDRV *pFbbDrv, int nChannel, int nType, char *pData, int nLen, DRVUI *pUi)
{
	DATAIN *pPtr;
	int nFbbCPort;
	BOOL bDisc = FALSE;
	
	if (nLen == -1)
		nLen = strlen(pData);
	
	if (nLen < 0 || nLen > 256)
		return;
	
	if (nChannel == PACTOR_CH)
	{
		nFbbCPort = MAKELONG(pFbbDrv->nPort[PACTOR], 1);
	}
	else 
	{
		if (nType == DRV_COMMAND)
		{
			int nPort;

			// First, get the port in the status line
			char *pScan = strchr(pData, ':');
			if (pScan == NULL)
			{
				// No port indication, problem !
				return;
			}
			--pScan;
			
			nPort = (*pScan) - '0';
			
			// Delete the port information
			memmove(pScan, pScan+2, strlen(pScan+1));
			
			// Points to the "CONNECTED" word
			pScan = pData;
			while (!isspace(*pScan))
				++pScan;

			while (isspace(*pScan))
				++pScan;

			if (toupper(*pScan) == 'C')
			{
				int nFbbChan;

				// in case of Connection, we allocate a new channel
				// if it is a packet incoming connection. We look
				// for a free channel starting from the first 
				// channel of the fbb port
				
				nFbbChan = 1;
				if (nPort == MODEM2)
					nFbbChan += pFbbDrv->nNbChan1;
				
				for ( ; nFbbChan <= pFbbDrv->nNbChan ; nFbbChan++)
				{
					int nI;

					nFbbCPort = MAKELONG(pFbbDrv->nPort[nPort], nFbbChan);
					for (nI = 1 ; nI <= pFbbDrv->nNbChan ; nI++)
					{
						if (pFbbDrv->nFbbCPort[nI] == nFbbCPort)
						{
							// FBB Channel in use
							break;
						}
					}
					if (nI > pFbbDrv->nNbChan)
					{
						// Current port/chennel is free
						pFbbDrv->nFbbCPort[nChannel] = nFbbCPort;
						break;
					}
				}
				if (nFbbChan > pFbbDrv->nNbChan)
				{
					// No available channel on this port.. Disconnect
					ToTnc(pFbbDrv, nChannel, DRV_DATA, "Sorry, no available channel. Disconnecting...", -1, NULL);
					ToTnc(pFbbDrv, nChannel, DRV_COMMAND, "D", -1, NULL);
					return;
				}
				if (nChannel == PACTOR_CH)
					pFbbDrv->nPacStatus = DRV_CONNECTED;
			}
			else if (toupper(*pScan) == 'D')
			{
				bDisc = TRUE;
			}
		}
		if (pUi)
		{
			// Check if port is managed
			if (pUi->nPort == 0)
				return;
			nFbbCPort = MAKELONG(pFbbDrv->nPort[pUi->nPort], 0);
		}
		else if (nType == DRV_INFO)
		{
			if (nChannel)
				nFbbCPort = pFbbDrv->nFbbCPort[nChannel];
			else
				nFbbCPort = MAKELONG(pFbbDrv->nInitP, 1);
		}
		else
		{
			nFbbCPort = pFbbDrv->nFbbCPort[nChannel];
		}
	}

	pPtr         = (DATAIN *)LocalAlloc( LPTR, sizeof(DATAIN));
	pPtr->nCPort = nFbbCPort;
	pPtr->nType  = nType;
	pPtr->nLen   = nLen;
	pPtr->pUi    = pUi;
	pPtr->pNext  = NULL;
	
	if (nLen > 0)
	{
		pPtr->pData    = (uchar *)LocalAlloc( LPTR, nLen);
		memcpy(pPtr->pData, pData, nLen);
	}
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	if (pFbbDrv->pHostHead == NULL)
		pFbbDrv->pHostHead = pPtr;
	else
		pFbbDrv->pHostTail->pNext = pPtr;
	
	// Update tail information
	pFbbDrv->pHostTail = pPtr;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	if (bDisc)
	{
		// Disconnection : free the channel
		pFbbDrv->nFbbCPort[nChannel] = 0;
		if (nChannel == PACTOR_CH)
		{
			DelTimer(pFbbDrv, pFbbDrv->pPacCall->pIss);
			LocalFree(pFbbDrv->pPacCall);
			pFbbDrv->pPacCall = NULL;
			pFbbDrv->nPacStatus = DRV_DISCONNECTED;

			ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, "%L1", -1, NULL);
		}
		pFbbDrv->nPacStatus = DRV_DISCONNECTED;
	}

	// Send Notification message
	if (pFbbDrv->hWnd)
		PostMessage(pFbbDrv->hWnd, WM_NOTIFY_MSG, nType, nFbbCPort);
}

static DATAOUT *GetTnc(FBBDRV *pFbbDrv)
{
	DATAOUT *pPtr, *pTmp;
	int	nChan;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	pPtr = pFbbDrv->pTncHead;
	if (pPtr)
	{
		if (pPtr->nType != DRV_DELAY)
		{
			pFbbDrv->pTncHead = pFbbDrv->pTncHead->pNext;
		}
		else
		{
			nChan = pPtr->nChannel;

			if (atoi((char *)pPtr->pData) < time(NULL))
			{
				// End of delay
				pTmp = pPtr;
				pPtr = pPtr->pNext;
				pFbbDrv->pTncHead = pPtr->pNext;

				// Free the delay structure
				LocalFree(pTmp->pData);
				LocalFree(pTmp);
			}
			else
			{
				// Look for commands on other channels
				pTmp = pPtr;
				pPtr = pPtr->pNext;
				while (pPtr)
				{
					if (pPtr->nChannel != nChan)
					{
						// Extract the structure for the list
						pTmp->pNext = pPtr->pNext;
						break;
					}
					pTmp = pPtr;
					pPtr = pPtr->pNext;
				}
			}
		}
	}

	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	return pPtr;
}

static DATAIN *GetHost(FBBDRV *pFbbDrv)
{
	DATAIN *pPtr;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	pPtr = pFbbDrv->pHostHead;
	if (pPtr)
		pFbbDrv->pHostHead = pFbbDrv->pHostHead->pNext;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	return pPtr;
}

static int TncQueueSize(FBBDRV *pFbbDrv, int nChan, int nType)
{
	DATAOUT *pPtr;
	int nNb = 0;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	pPtr = pFbbDrv->pTncHead;
	while (pPtr)
	{
		pPtr = pPtr->pNext;
		if (nChan == 0 || nChan == pPtr->nChannel)
			if (nType == 0 || nType == pPtr->nType)
				++nNb;
	}
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	return nNb;
}

static void HostInfo(FBBDRV *pFbbDrv, int nChannel, char *sFormat, ...)
{
	char	szStr[512];
	va_list pArgPtr;
	int nCnt;
	
	va_start(pArgPtr, sFormat);
	nCnt = wvsprintf(szStr, sFormat, pArgPtr);
	va_end(pArgPtr);
	
	if (nCnt > 0)
		ToHost(pFbbDrv, nChannel, DRV_INFO, szStr, nCnt, NULL);
}

static BOOL InitPort(FBBDRV *pFbbDrv)
{
	int nNb;
	char szBuf[256];
	char szCall[10];
	char *szPtr;

	char *szInit[] = {
		"MIUSC",
		"I",
		"M",
		NULL
	};

	for (nNb = 0 ; szInit[nNb] ; nNb++)
		ToTnc(pFbbDrv, 0, DRV_COMMAND, szInit[nNb], -1, NULL);

	// Packet callsign
	wsprintf (szBuf, "I %s", pFbbDrv->szMyCall);
	ToTnc (pFbbDrv, 0, DRV_COMMAND, szBuf, -1, NULL);

	// Pactor callsign
	strcpy(szCall, pFbbDrv->szMyCall);
	szPtr = strrchr(szCall, '-');
	if (szPtr)
		*szPtr = '\0';
	wsprintf (szBuf, "I %s", szCall);
	ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, szBuf, -1, NULL);

	wsprintf (szBuf, "Y %d", pFbbDrv->nNbChan);
	ToTnc (pFbbDrv, 0, DRV_COMMAND, szBuf, -1, NULL);

	wsprintf (szBuf, "T 1:300");
	ToTnc (pFbbDrv, 0, DRV_COMMAND, szBuf, -1, NULL);
	wsprintf (szBuf, "T 2:300");
	ToTnc (pFbbDrv, 0, DRV_COMMAND, szBuf, -1, NULL);

	ToTnc (pFbbDrv, 0, DRV_COMMAND, "%%L 0", -1, NULL);

	ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, "#PD 0",   -1, NULL);
	ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, "#UML 0",  -1, NULL);
	ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, "#BOX 0",  -1, NULL);
	ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, "#REM 0",  -1, NULL);
	ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, "#MA 0",   -1, NULL);
	ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, "#CMSG 0", -1, NULL);
	ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, "#ST 2",   -1, NULL);

	ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, "#PAC PRB 0",  -1, NULL);

	return TRUE;
}

static BOOL StartHostMode(FBBDRV *pFbbDrv, int nPort)
{
	//	int nLen;
	int nChar;
	int nNb;
	BOOL bOk;
	char *pszInit = "\r\033\rRESTART\r";
	char *pszPact = "\r\033\rPTC %d\r";
	char *pszHost = "\033JHOST4\r";
	char szBuffer[80];
	
	HostInfo(pFbbDrv, 0, "StartHostMode");

	// Check if TNC is alive
	for (nNb = 0 ; nNb < 4 ; nNb++)
	{
		bOk = FALSE;
		WriteTncBlock(pFbbDrv, pszInit, strlen(pszInit));
		Sleep(2000);
		while ((nChar = ReadTncChar(pFbbDrv)) >= 0)
		{
			Sleep(2);
			bOk = TRUE;
		}
		
		if (bOk)
			break;

		// TNC does not answer. Try going out from HOST mode
		ToTnc(pFbbDrv, 0, DRV_COMMAND, "JHOST0", -1, NULL);
		SendTncRequest(pFbbDrv);

		Sleep(1000);
	}
	
	if (nNb == 4)
		return FALSE;
	
	sprintf (szBuffer, pszPact, PACTOR_CH);

	for (nNb = 0 ; nNb < 4 ; nNb++)
	{
		bOk = FALSE;
		WriteTncBlock(pFbbDrv, szBuffer, strlen(szBuffer));
		Sleep(100);
		while ((nChar = ReadTncChar(pFbbDrv)) >= 0)
		{
			Sleep(2);
			bOk = TRUE;
		}
		
		if (bOk)
			break;
	}
	
	if (nNb == 4)
		return FALSE;
	
	// Go to hostmode
	WriteTncBlock(pFbbDrv, pszHost, strlen(pszHost));
	
	// Be sure the COM is empty ...
	
	while (ReadTncChar(pFbbDrv) >= 0)
		Sleep(10);
	
	HostInfo(pFbbDrv, 0, "StartHostMode OK");

	// Init PTC HostMode counter
	pFbbDrv->bInit = TRUE;

	return TRUE;
}

static BOOL EndHostMode(FBBDRV *pFbbDrv)
{
	int nI;

	// No monitoring
	ToTnc(pFbbDrv, 0, DRV_COMMAND, "MN", -1, NULL);

	// Command mode 
	ToTnc(pFbbDrv, 0, DRV_COMMAND, "JHOST0", -1, NULL);
	
	// Wait until all TNC queue is processed
	// Possible deadlock if resync mode !
	// Wait only 10 seconds
	for (nI = 0 ; nI < 100 ; nI++)
	{
		if (TncQueueSize(pFbbDrv, 0, 0) == 0)
			break;
		
		// Wait some time...
		Sleep(100);
	}
	
	return TRUE;
}

BOOL SetupConnection(HANDLE hDev, int nBaudrate)
{
	BOOL       fRetVal ;
	DCB        dcb ;
	
	dcb.DCBlength = sizeof( DCB ) ;
	
	GetCommState(hDev, &dcb ) ;
	
	dcb.BaudRate = nBaudrate ;
	dcb.ByteSize = 8 ;
	dcb.Parity   = NOPARITY ;
	dcb.StopBits = ONESTOPBIT ;
	
	
	// setup hardware flow control
	
	dcb.fOutxDsrFlow = FALSE ;
	dcb.fDtrControl = DTR_CONTROL_DISABLE ;
	
	dcb.fOutxCtsFlow = FALSE ;
	dcb.fRtsControl = RTS_CONTROL_DISABLE ;
	
	// setup software flow control
	
	dcb.fInX = dcb.fOutX = FALSE ;
	
	// other various settings
	
	dcb.fBinary = TRUE ;
	
	fRetVal = SetCommState(hDev, &dcb ) ;
	
	return ( fRetVal ) ;
	
} // end of SetupConnection()

static int ReadTncChar(FBBDRV *pFbbDrv)
{
	BOOL       fReadStat ;
	COMSTAT    ComStat ;
	DWORD      dwErrorFlags;
	DWORD      dwLength;
	BYTE		nChar;

	// Read one character
	// I don't know how to read more than one character
	// with a timeout, to get any number of bytes
	ClearCommError(pFbbDrv->hDev, &dwErrorFlags, &ComStat ) ;
	fReadStat = ReadFile(pFbbDrv->hDev, &nChar,
		1, &dwLength, NULL) ;
	if (!fReadStat)
	{
		ClearCommError(pFbbDrv->hDev, &dwErrorFlags, &ComStat ) ;
		if (dwErrorFlags > 0)
		{
			HostInfo(pFbbDrv, 0, "<CE-%u>", dwErrorFlags ) ;
		}
		return -1;
	}
	
	if (dwLength == 0)
	{
		return -1;
	}
	
	return ( nChar ) ;	
} // end of ReadTncChar()

static BOOL WriteTncBlock(FBBDRV *pFbbDrv, LPSTR lpByte , DWORD dwBytesToWrite)
{
	
	BOOL        fWriteStat ;
	DWORD       dwBytesWritten ;
	DWORD       dwErrorFlags;
	//	DWORD   	dwError;
	DWORD       dwBytesSent=0;
	COMSTAT     ComStat;
	
	fWriteStat = WriteFile(pFbbDrv->hDev, lpByte, dwBytesToWrite,
		&dwBytesWritten, NULL) ;
	if (!fWriteStat)
	{
		// some other error occurred
		ClearCommError(pFbbDrv->hDev, &dwErrorFlags, &ComStat ) ;
		if (dwErrorFlags > 0)
		{
			HostInfo(pFbbDrv, 0, "<CE-%u>", dwErrorFlags ) ;
		}
		return ( FALSE );
	}
	return ( TRUE ) ;
	
} // end of WriteTncBlock()


static void DedUiHeader(FBBDRV *pFbbDrv, char *szBuf, DRVUI *pUi)
{
	char *pPtr;
	char *pScan;
	int nPort;

	memset (pUi, 0, sizeof(DRVUI));

	pScan = szBuf;

	// Read the monitoring port
	nPort = *pScan - '0';
	// Skip ':'
	if (*pScan == ':')
		++pScan;

	pPtr = strtok (pScan, " ");	/* fm */

	if (pPtr == NULL)
		return;

	// look-up local port to Fbb Port
	pUi->nPort = pFbbDrv->nPort[nPort];
	
	pPtr = strtok (NULL, " ");	/* exped */

	if (pPtr == NULL)
		return;
	strncpy (pUi->szFrom, pPtr, 9);
	pUi->szFrom[9] = '\0';

	pPtr = strtok (NULL, " ");	/* to */

	if (pPtr == NULL)
		return;

	pPtr = strtok (NULL, " ");	/* desti */

	if (pPtr == NULL)
		return;
	strncpy (pUi->szTo, pPtr, 9);
	pUi->szTo[9] = '\0';

	pPtr = strtok (NULL, " ");	/* via ou ctl */

	if (pPtr == NULL)
		return;

	if (strcmp (pPtr, "via") == 0)
	{
		for (;;)
		{
			if (*pUi->szVia)
				strcat (pUi->szVia, " ");

			pPtr = strtok (NULL, " ");	/* digis */

			if (pPtr == NULL)
				return;

			if (strcmp (pPtr, "ctl") == 0)
				break;

			strncat (pUi->szVia, pPtr, 9);
		}
		pUi->szVia[80] = '\0';
	}

	pPtr = strtok (NULL, " ");	/* controle */

	if (pPtr == NULL)
		return;
	strncpy (pUi->szCtl, pPtr, 9);
	pUi->szCtl[9] = '\0';

	pUi->bUi = (strncmp (pPtr, "UI", 2) == 0);

	pPtr = strtok (NULL, " ");	/* pid */

	if (pPtr == NULL)
		return;

	pPtr = strtok (NULL, " ");	/* pid */

	if (pPtr == NULL)
		return;
	sscanf (pPtr, "%x", &pUi->nPid);
}

static void PactorPhase(void *pPtr)
{
	FBBDRV *pFbbDrv = (FBBDRV *)pPtr;

	switch(pFbbDrv->nPacStatus)
	{
	case DRV_CONNWAIT:
		pFbbDrv->nPacStatus = DRV_CONNCHECK;
		pFbbDrv->pPacCall->pTimer = AddTimer(pFbbDrv, 10, PactorPhase, (void *)pFbbDrv);
		break;
	
	case DRV_CONNCHECK:
		// Pactor connection
		ToTnc(pFbbDrv, PACTOR_CH, DRV_COMMAND, "%L1", -1, NULL);
		ToTnc(pFbbDrv, PACTOR_CH, DRV_COMMAND, pFbbDrv->pPacCall->szCall, -1, NULL);

		LocalFree(pFbbDrv->pPacCall);
		pFbbDrv->pPacCall = NULL;
		pFbbDrv->nPacStatus = DRV_CONNPENDING;
		break;
	}
}

static void PactorConnect(FBBDRV *pFbbDrv, char *szCall)
{
	pFbbDrv->pPacCall = (PACCALL *)LocalAlloc(LPTR, sizeof(PACCALL));

	if (pFbbDrv->pPacCall == NULL)
		return;

	ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, "%L0", -1, NULL);
	pFbbDrv->nPacStatus = DRV_CONNWAIT;
	strcpy(pFbbDrv->pPacCall->szCall, szCall);
	pFbbDrv->pPacCall->pTimer = AddTimer(pFbbDrv, 5, PactorPhase, (void *)pFbbDrv);
}

static void SendTncRequest(FBBDRV *pFbbDrv)
{
	unsigned short crc;
	int nI;
	int nLen;
	int nSize;
	uchar *szSend = pFbbDrv->szSend;
	uchar *pData;
	DATAOUT *pPtr = NULL;
	
	// Header
	szSend[0] = (uchar)HST_CHAR;
	szSend[1] = (uchar)HST_CHAR;

	if (!pFbbDrv->bAgain && (pPtr = GetTnc(pFbbDrv)))
	{
		switch(pPtr->nType)
		{
		case DRV_COMMAND:
			if (pPtr->nChannel == PACTOR_CH)
			{
				// Special case of connection and disconnection of the PACTOR channel
				switch(toupper(pPtr->pData[0]))
				{
				case 'C':
					// Ignore connections if not in DISC state.
					/*if (pFbbDrv->nPacStatus == DRV_DISCONNECTED)
					{
						// Connect on pactor must be delayed
						pPtr->pData[pPtr->nLen] = '\0';
						PactorConnect(pFbbDrv, (char *)pPtr->pData);
					}
					LocalFree(pPtr->pData);
					LocalFree(pPtr);
					return;*/
					break;
				case 'D':
					if (pFbbDrv->nPacStatus == DRV_CONNWAIT || pFbbDrv->nPacStatus == DRV_CONNCHECK)
					{
						if (pFbbDrv->pPacCall)
						{
							char szMsg[80];

							DelTimer(pFbbDrv, pFbbDrv->pPacCall->pTimer);
							DelTimer(pFbbDrv, pFbbDrv->pPacCall->pIss);
							LocalFree(pPtr->pData);
							LocalFree(pPtr);
							wsprintf(szMsg, "(%d) DISCONNECTED fm %s", 
									PACTOR_CH, &pFbbDrv->pPacCall->szCall[1]);
							LocalFree(pFbbDrv->pPacCall);
							pFbbDrv->pPacCall = NULL;
							pFbbDrv->nPacStatus = DRV_DISCONNECTED;

							ToHost(pFbbDrv, PACTOR_CH, DRV_COMMAND, szMsg, -1, NULL);
							ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, "%L1", -1, NULL);
							return;
						}
					}
					break;
				}
			}
			szSend[2] = pPtr->nChannel;
			szSend[3] = 1;	// Command
			break;
		case DRV_DATA:
			szSend[2] = pPtr->nChannel;
			szSend[3] = 0;	// Data
			break;
		case DRV_UNPROTO:
			szSend[2] = 0;
			szSend[3] = 0;	// Data
		}

		nLen = pPtr->nLen;
		pData = pPtr->pData;
		pFbbDrv->nLastCmd = 0;
	}
	else
	{
		// Prepare for next request
		if (!pFbbDrv->bAgain)
		{
			switch (pFbbDrv->nPhase)
			{
			case G_PHASE:
				if (pFbbDrv->nChanList[0] != 0)
				{
					pFbbDrv->nChanG = pFbbDrv->nChanList[0] - 1;
					memmove(pFbbDrv->nChanList, pFbbDrv->nChanList+1, MAXCHAN-1);
				}
				else
				{
					pFbbDrv->nPhase = L_PHASE;
				}
				break;
			case L_PHASE:
				if (++pFbbDrv->nChanL > pFbbDrv->nNbChan)
				{
					if (!pFbbDrv->bPactor || pFbbDrv->nChanL > PACTOR_CH)
					{
						pFbbDrv->nChanL = 0;
						pFbbDrv->nChanG = 255;
					}
					else
					{
						pFbbDrv->nChanL = PACTOR_CH;
					}
					pFbbDrv->nPhase = G_PHASE;
				}
				break;

			case P_PHASE:
				pFbbDrv->nPhase = G_PHASE;
				pFbbDrv->nChanG = 255;
				break;
			}
		}

		switch (pFbbDrv->nPhase)
		{
		case G_PHASE:
			szSend[2] = pFbbDrv->nChanG;
			szSend[3] = 1;		// Command
	
			pData = (uchar *)"G";
			nLen = 1;
	
			pFbbDrv->nLastCmd = 'G';
			break;

		case L_PHASE:
			// Polling for link status
			szSend[2] = pFbbDrv->nChanL;
			szSend[3] = 1;
	
			pData = (uchar *)"L";
			nLen = 1;
	
			pFbbDrv->nLastCmd = 'L';
			break;
		}
	}

	// Add counter
	szSend[3] |= (pFbbDrv->bCounter) ? 0x00 : 0x80;
	pFbbDrv->bCounter = !pFbbDrv->bCounter;
	if (!pFbbDrv->bInit)
	{
		// To synchronize the first time
		szSend[3] |= 0x40;
		pFbbDrv->bInit = TRUE;
	}

	nSize = 4;
	nLen;
	szSend[nSize++] = nLen-1;
	if ((nLen-1) == HST_CHAR)
		szSend[nSize++] = 0;

	// Add data
	for (nI = 0 ; nI < nLen ; nI++)
	{
		szSend[nSize++] = pData[nI];
		if (pData[nI] == HST_CHAR)
			szSend[nSize++] = 0;
	}

	// Compute CRC
	crc = 0xffff;
	crc = updcrc(szSend[2], crc);
	crc = updcrc(szSend[3], crc);
	crc = updcrc(szSend[4], crc);
	for (nI = 0 ; nI < nLen ; nI++)
		crc = updcrc(pData[nI], crc);
	crc = ~crc;

	/* Send crc */
	szSend[nSize++] = crc & 0xff;
	if ((crc & 0xff) == HST_CHAR)
	{
		szSend[nSize++] = 0;
	}

	szSend[nSize++] = crc >> 8;
	if ((crc >> 8) == HST_CHAR)
	{
		szSend[nSize++] = 0;
	}
	
	pFbbDrv->nSize = nSize;

	if (pPtr)
	{
		LocalFree(pPtr->pData);
		LocalFree(pPtr);
	}

	WriteTncBlock(pFbbDrv, (char *)szSend, nSize);

	pFbbDrv->bAgain = FALSE;
}

// Watchdog timer
static void HstIss(void *pData)
{
	FBBDRV *pFbbDrv = (FBBDRV *)pData;
	
	pFbbDrv->pPacCall->pIss = NULL;

	if (ISS(pFbbDrv))
		HstOver(pFbbDrv);
	else
		HstBreak(pFbbDrv);
}

static void HstBreak(FBBDRV *pFbbDrv)
{
	// break_in
	ToTnc(pFbbDrv, PACTOR_CH, DRV_COMMAND, "%I", -1, NULL);
}

static void HstOver(FBBDRV *pFbbDrv)
{
	// ChOver
	if (ISS(pFbbDrv))
		ToTnc(pFbbDrv, PACTOR_CH, DRV_COMMAND, "%O", -1, NULL);
}

static int IsPactorData(FBBDRV *pFbbDrv)
{
	DATAOUT *pPtr = pFbbDrv->pTncHead;

	while (pPtr)
	{
		if (pPtr->nType == DRV_DATA && pPtr->nChannel == PACTOR_ST)
			return(1);
		pPtr = pPtr->pNext;
	}
	return(0);
}

static void PactorStatus(FBBDRV *pFbbDrv)
{
	static int prec = -1;

	if (pFbbDrv->nStatusPrec != pFbbDrv->nStatus || pFbbDrv->nPacStatus != prec)
	{

	pFbbDrv->nStatusPrec = pFbbDrv->nStatus;
	prec = pFbbDrv->nPacStatus;
	HostInfo(pFbbDrv, 0, "S:%02x %d %c %c %c",
		pFbbDrv->nStatus & 0xff, 
		pFbbDrv->nPacStatus, 
		PACTOR_BUSY(pFbbDrv)?'B':'.',
		PACTOR_ONLINE(pFbbDrv) ? 'O':'.',
		ISS(pFbbDrv) ? 'I':'.');
	}

	if (pFbbDrv->pPacCall)
	{
		// Call pending

		if (pFbbDrv->nPacStatus == DRV_CONNCHECK)
		{
			if (PACTOR_BUSY(pFbbDrv))
			{
			char szMsg[80];

			// Frequency is busy : stop connection sequence
			DelTimer(pFbbDrv, pFbbDrv->pPacCall->pTimer);
			DelTimer(pFbbDrv, pFbbDrv->pPacCall->pIss);
			wsprintf(szMsg, "(%d) DISCONNECTED fm %s", 
					PACTOR_CH, &pFbbDrv->pPacCall->szCall[1]);
			LocalFree(pFbbDrv->pPacCall);
			pFbbDrv->pPacCall = NULL;
			pFbbDrv->nPacStatus = DRV_DISCONNECTED;

			ToHost(pFbbDrv, PACTOR_CH, DRV_COMMAND, szMsg, -1, NULL);
			ToTnc (pFbbDrv, PACTOR_CH, DRV_COMMAND, "%L1", -1, NULL);
			}
		}

		// Check Tx timeout
		if (PACTOR_ONLINE(pFbbDrv))
		{
			if (ISS(pFbbDrv))
			{
				if (TncQueueSize(pFbbDrv, PACTOR_CH, DRV_DATA) == 0)
				{
					if (pFbbDrv->pPacCall->pIss == NULL)
						pFbbDrv->pPacCall->pIss = AddTimer(pFbbDrv, 10, HstIss, (void *)pFbbDrv);
				}
				else
				{
					DelTimer(pFbbDrv, pFbbDrv->pPacCall->pIss);
					pFbbDrv->pPacCall->pIss = NULL;
				}
			}
			else
			{
				if (TncQueueSize(pFbbDrv, PACTOR_CH, DRV_DATA))
				{
					if (pFbbDrv->pPacCall->pIss == NULL)
						pFbbDrv->pPacCall->pIss = AddTimer(pFbbDrv, 10, HstIss, (void *)pFbbDrv);
				}
				else
				{
					DelTimer(pFbbDrv, pFbbDrv->pPacCall->pIss);
					pFbbDrv->pPacCall->pIss = NULL;
				}
			}
		}
	}
}

static BOOL Resync(FBBDRV *pFbbDrv)
{
	static int nResync = 0;

	// Resynchro : Answer is not received -> Resend the last packet
	WriteTncBlock(pFbbDrv, (char *)pFbbDrv->szSend, pFbbDrv->nSize);
	HostInfo(pFbbDrv, 0, "Resync:%d", ++nResync);

	return FALSE;
}

static BOOL DecodeBlock(FBBDRV *pFbbDrv, uchar *szBuf, int nSize)
{
	int nNb;
	int nChan;
	int nCode;
	int nLen;
	char szBuffer[256];
	char *pPtr = szBuffer;

	nChan = *szBuf++;
	nCode = *szBuf++;
	nSize -= 2;

	switch (nCode)
	{
	case 0:
		if (pFbbDrv->nLastCmd == 'L')
		{
			// Should never occur
			return FALSE;
		}
		break;
	case 1:
		if (nChan == 255)
		{
			pPtr = (char *)&pFbbDrv->nChanList[0];
		}
		for (;;)
		{
			*pPtr = *szBuf++;
			if (*pPtr == '\0')
				break;
			++pPtr;
			if (--nSize == 0)
				return FALSE;
		}
		if (pFbbDrv->nLastCmd != 'L')
		{
			if (nChan != 255)
				HostInfo(pFbbDrv, nChan, "Erreur L : %s", szBuffer);
		}
		else if (nChan < MAXCHAN)
		{
			DRVSTATS DrvStats;
			int nbmes, nbtra, nbatt, nback, nbret, con;

			nbatt = nback = nbret = con = 0;

			sscanf (szBuffer, "%d%d%d%d%d%d",
						&nbmes, &nbtra, &nbatt, &nback, &nbret, &con);

			memset(&DrvStats, 0, sizeof(DRVSTATS));

			DrvStats.nNbFrames = nbmes + nbtra;
			DrvStats.nNbRetry  = nbret;
			DrvStats.nState = DRV_DISCONNECTED;
			if (nChan == PACTOR_CH)
			{
				DrvStats.nAux |= DRV_AUXPACTOR;
				if (ISS(pFbbDrv))
					DrvStats.nAux |= DRV_AUXISS;
			}

			switch(con)
			{
			case 0:
				if (nChan == PACTOR_CH)
				{
					if (pFbbDrv->nPacStatus == DRV_CONNWAIT)
						DrvStats.nState = DRV_CONNWAIT;
					else if (pFbbDrv->nPacStatus == DRV_CONNCHECK)
						DrvStats.nState = DRV_CONNCHECK;
					else
					{
						if (pFbbDrv->nPacStatus != DRV_DISCONNECTED)
						{
							pFbbDrv->nPacStatus = DRV_DISCONNECTED;
						}
						DrvStats.nState = DRV_DISCONNECTED;
					}
				}
				else
				{
					DrvStats.nState = DRV_DISCONNECTED;
				}
				break;
			case 1:
				DrvStats.nState = DRV_CONNPENDING;
				break;
			case 3:
				DrvStats.nState = DRV_DISCPENDING;
				break;
			default:
				if ((pFbbDrv->nStatus & 0xe0) != 0xe0)
				{
					// Avoid listen = connected 
					DrvStats.nState = DRV_CONNECTED;
				}
				break;
			}
			DrvStats.nBuffers  = 500;
			DrvStats.nFrameNb  = 0;

			if (DrvStats.nNbFrames != pFbbDrv->nNbFrm[nChan] || 
				DrvStats.nNbRetry  != pFbbDrv->nNbRet[nChan] || 
				DrvStats.nAux      != pFbbDrv->nAux[nChan]   || 
				DrvStats.nState    != pFbbDrv->nState[nChan])
			{
				pFbbDrv->nNbFrm[nChan] = DrvStats.nNbFrames;
				pFbbDrv->nNbRet[nChan] = DrvStats.nNbRetry;
				pFbbDrv->nAux[nChan]   = DrvStats.nAux;
				pFbbDrv->nState[nChan] = DrvStats.nState;
				ToHost(pFbbDrv, nChan, DRV_STATS, (char *)&DrvStats, sizeof(DRVSTATS), NULL);
			}
		}
		break;
	case 2:
		for (;;)
		{
			*pPtr = *szBuf++;
			if (*pPtr == '\0')
				break;
			++pPtr;
			if (--nSize == 0)
				return FALSE;
		}
		HostInfo(pFbbDrv, nChan, "%s", szBuffer);
		break;
	case 3:
		for (;;)
		{
			*pPtr = *szBuf++;
			if (*pPtr == '\0')
				break;
			++pPtr;
			if (--nSize == 0)
				return FALSE;
		}
		ToHost(pFbbDrv, nChan, DRV_COMMAND, szBuffer, -1, NULL);
		break;
	case 4:
		for (;;)
		{
			*pPtr = *szBuf++;
			if (*pPtr == '\0')
				break;
			++pPtr;
			if (--nSize == 0)
				return FALSE;
		}
		pFbbDrv->pUi = (DRVUI *)LocalAlloc(LPTR, sizeof(DRVUI));
		if (pFbbDrv->pUi)
			DedUiHeader(pFbbDrv, szBuffer, pFbbDrv->pUi);
		ToHost(pFbbDrv, nChan, DRV_UNPROTO, NULL, 0, pFbbDrv->pUi);
		pFbbDrv->pUi = NULL;
		break;
	case 5:
		for (;;)
		{
			*pPtr = *szBuf++;
			if (*pPtr == '\0')
				break;
			if (--nSize == 0)
				return FALSE;
			++pPtr;
		}
		pFbbDrv->pUi = (DRVUI *)LocalAlloc(LPTR, sizeof(DRVUI));
		if (pFbbDrv->pUi)
			DedUiHeader(pFbbDrv, szBuffer, pFbbDrv->pUi);
		pFbbDrv->bAgain = TRUE;
		break;
	case 6:
		nLen = 1 + *szBuf++;
		for (nNb = 0 ; nNb < nLen ; nNb++)
		{
			szBuffer[nNb] = *szBuf++;
			if (--nSize == 0)
				return FALSE;
		}
		szBuffer[nNb] = '\0';
		ToHost(pFbbDrv, nChan, DRV_UNPROTO, szBuffer, nLen, pFbbDrv->pUi);
		pFbbDrv->pUi = NULL;
		break;
	case 7:
		nLen = 1 + *szBuf++;
		for (nNb = 0 ; nNb < nLen ; nNb++)
		{
			szBuffer[nNb] = *szBuf++;
			if (--nSize == 0)
				return FALSE;
		}
		szBuffer[nNb] = '\0';
		if (nChan == 254)
		{
			// Pactor status
			pFbbDrv->nStatus = (int)szBuffer[0] & 0xff;
		}
		else if (pFbbDrv->nPacStatus == DRV_DISCONNECTED && nChan == PACTOR_CH)
		{
			pFbbDrv->pUi = (DRVUI *)LocalAlloc(LPTR, sizeof(DRVUI));
			if (pFbbDrv->pUi)
			{
				pFbbDrv->pUi->nPort = pFbbDrv->nPort[PACTOR];
				strcpy (pFbbDrv->pUi->szFrom, "PACTOR");
				pFbbDrv->pUi->bUi = 1;
				ToHost(pFbbDrv, nChan, DRV_UNPROTO, szBuffer, nLen, pFbbDrv->pUi);
			}
		}
		else
		{
			ToHost(pFbbDrv, nChan, DRV_DATA, szBuffer, nLen, NULL);
		}
		break;
	default:
		// Empty the queues
		return FALSE;
	}

	return TRUE;
}

static void ReadTncAnswer(FBBDRV *pFbbDrv)
{
	int nChar;
	int nLen;
	int nPrev;
	uchar szBuffer[300];
	unsigned short crc;
	
	// Fills a buffer
	nLen = 0;
	nChar = 0;

	while (pFbbDrv->bUp)
	{
		nPrev = nChar;
		nChar = ReadTncChar(pFbbDrv);
		if (nChar == -1)
		{
			Resync(pFbbDrv);
			continue;
		}
		else if (nChar == HST_CHAR)
		{
			if (nPrev == HST_CHAR)
			{
				// Header received
				nLen = 0;
				crc = 0xffff;
				continue;
			}
			else
			{
				szBuffer[nLen++] = nChar;
				crc = updcrc(nChar, crc);
			}
		}
		else if (nPrev == HST_CHAR)
		{
			if (nLen == 0)
			{
				szBuffer[nLen++] = nChar;
				crc = updcrc(nChar, crc);
			}
			// Character must be 0 after a HST_CHAR
			else if (nChar != '\0')
			{
				// Should never occur
				nLen = 0;
			}
		}
		else
		{
			szBuffer[nLen++] = nChar;
			crc = updcrc(nChar, crc);
		}

		if (crc == 0xf0b8)
		{
			// Try to decode the block
			if (DecodeBlock(pFbbDrv, szBuffer, nLen-2))
				return;
		}

		// End of block ?
		if (nLen == 260)
		{
			nLen = 0;
		}
	}
}

// Timer functions

static TIMER *AddTimer(FBBDRV *pFbbDrv, long nCount, void (*pProc)(void *), void *pData)
{
	TIMER *pTimer;

	pTimer = (TIMER *)LocalAlloc(LPTR, sizeof(TIMER));
	if (pTimer == NULL)
		return NULL;

	pTimer->nCount = nCount;
	pTimer->pProc  = pProc;
	pTimer->pData  = pData;

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);

	pTimer->pNext = pFbbDrv->pTimerHead;
	pFbbDrv->pTimerHead = pTimer;

	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);

	return pTimer;
}

static void *DelTimer(FBBDRV *pFbbDrv, TIMER *pTimer)
{
	TIMER *pScan;
	TIMER *pPrec;
	void *pPtr;

	if (pTimer == NULL)
		return NULL;

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);

	for (pPrec = NULL, pScan = pFbbDrv->pTimerHead ; pScan ; pScan = pScan->pNext)
	{
		if (pScan == pTimer)
		{
			if (pPrec)
				pPrec->pNext = pTimer->pNext;
			else
				pFbbDrv->pTimerHead = pTimer->pNext;
			break;
		}
		pPrec = pScan;
	}

	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);

	pPtr = pTimer->pData;
	LocalFree(pTimer);

	return pPtr;
}

static void Timer(FBBDRV *pFbbDrv)
{
	static time_t nPrec = 0L;
	time_t nDelta;
	TIMER *pScan;

	nDelta = time(NULL) - nPrec;
	if (nDelta == 0L)
		return;

	nPrec += nDelta;

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);

	for (pScan = pFbbDrv->pTimerHead ; pScan ; pScan = pScan->pNext)
	{
		if (pScan->nCount > nDelta)
			pScan->nCount -= nDelta;
		else
			pScan->nCount = 0L;
	}

	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);

	// Now scan for expired timers

	for (;;)
	{
		// Must be thread safe
		EnterCriticalSection(&pFbbDrv->hCritical);

		for (pScan = pFbbDrv->pTimerHead ; pScan ; pScan = pScan->pNext)
		{
			if (pScan->nCount == 0L)
				break;
		}

		// Must be thread safe
		LeaveCriticalSection(&pFbbDrv->hCritical);

		if (pScan)
		{
			// Call the procedure
			if (pScan->pProc)
				(*pScan->pProc)(pScan->pData);

			// Delete the timer
			DelTimer(pFbbDrv, pScan);
		}
		else
		{
			// No more expired timer
			break;
		}
	}
}

// TNC Thread

DWORD WINAPI DedLoop(LPSTR lpData)
{
	FBBDRV   *pFbbDrv = (FBBDRV*) lpData ;
	
	while ( pFbbDrv->bUp )
	{
		// Send the request
		SendTncRequest(pFbbDrv);
		
		// Read the answer
		ReadTncAnswer(pFbbDrv);

		// Timer
		Timer(pFbbDrv);

		// Status of the Pactor channel
		PactorStatus(pFbbDrv);
	}
	
	pFbbDrv->dwThreadId = 0 ;
	pFbbDrv->hThread    = NULL ;
	
	return( TRUE ) ;
	
} // end of DedLoop()


BOOL GetDrvVersion(int *nVMaj, int *nVMin)
{
	DWORD dwVersionInfoSize;
	DWORD dwTmpHandle;
	LPVOID lpVersionInfo; 
	BOOL bRet = false;


	dwVersionInfoSize = GetFileVersionInfoSize(FBBDLLNAME, &dwTmpHandle);

	if(dwVersionInfoSize)
	{
		lpVersionInfo = LocalAlloc(LPTR, dwVersionInfoSize);
		if(lpVersionInfo)
		{
			if(GetFileVersionInfo(FBBDLLNAME, dwTmpHandle, dwVersionInfoSize, lpVersionInfo))
			{
				LPVOID lpBuffer = LocalAlloc(LPTR, dwVersionInfoSize);
				UINT dwBytes;

				if( VerQueryValue(lpVersionInfo, 
						TEXT("\\StringFileInfo\\000004B0\\FileVersion"), 
						&lpBuffer, 
						&dwBytes)) 
				{
						sscanf((TCHAR *)lpBuffer, "%d,%d", nVMaj, nVMin);
						bRet = true;
				}

				LocalFree(lpBuffer);
			}

			LocalFree(lpVersionInfo);
 		}
 	}

	return bRet;
}
