
//////////////////////////////////////////////////////////////
//
// FBB driver for KAM-Pactor mode
//
// File : fbb_kap.dll
//
// (C) F6FBB 2000
//
//////////////////////////////////////////////////////////////
//
//  Version history
//
//	    - 06/01/2001 - EB5AGF
//		Added version resource and code to read it at runtime
//
//////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <winver.h>
#include "../fbb_drv.h"

#define uchar unsigned char

#define	FEND	0xc0
#define	FESC	0xdb
#define	TFEND	0xdc
#define	TFESC	0xdd

#define DRV_HOST 99

#define FBBDLLNAME "kap.dll"

//------------

int KAP_MAJOR = 1;
int KAP_MINOR = 1;

typedef struct DATAOUT_t {
	char	*pData;
	int		nType;
	int		nLen;
	struct	DATAOUT_t *pNext;
} DATAOUT;

typedef struct DATAIN_t {
	char	*pData;
	int		nType;
	int		nLen;
	DRVUI	*pUi;
	struct	DATAIN_t *pNext;
} DATAIN;

typedef struct Timer {
	long nCount;
	void (*pProc)(void *);
	void *pData;
	struct Timer *pNext;
} TIMER;

typedef struct {
	char szCommand[40];
	TIMER *pTimer;
	TIMER *pIss;
} PACCALL;

typedef struct FBBDRV_t
{
	HWND	hWnd;		// FBB Window
	int		nPort;		// FBB Port
	int		nState;
	int		nCom;
	int		nNbCmd;
	BOOL	bUp;
	BOOL	bIss;
	char	szCall[10];	// Remote call
	char	szMyCall[10];	// BBS callsign
	HANDLE	hDev;
	HANDLE	hThread;
	DWORD	dwThreadId;
	DRVUI	*pUi;
	DATAOUT	*pTncHead;	// Head of data_out list
	DATAOUT	*pTncTail;	// Tail of data_out list
	DATAIN	*pHostHead;	// Head of data_in list
	DATAIN	*pHostTail;	// Tail of data_in list
	TIMER	*pTimerHead;
	PACCALL	*pPacCall;	// Pactor Call parameters
	DRVSTATS pStats;
	CRITICAL_SECTION	hCritical;
	struct	FBBDRV_t *pNext;
} FBBDRV;


//////////////////////////////////////////////////////////////
// Local prototypes
//////////////////////////////////////////////////////////////

static DWORD WINAPI KapLoop( LPSTR lpData );
static DATAIN *GetHost(FBBDRV *pFbbDrv);
static BOOL ToTnc(FBBDRV *pFbbDrv, int nType, char *pData, int nLen, DRVUI *pUi);
static BOOL KapInit(FBBDRV *pFbbDrv, char *pPtr);
static BOOL Disconnected(FBBDRV *pFbbDrv);
static BOOL ConnectTo(FBBDRV *pFbbDrv, char *szRemoteCall);
static void ToHost(FBBDRV *pFbbDrv, int nType, char *pData, int nLen, DRVUI *pUi);
static FBBDRV *SearchPort(int nPort);
BOOL SetupConnection(HANDLE hDev, int nBaudrate);
static BOOL StartHostMode(FBBDRV *pFbbDrv, int nPort);
static BOOL EndHostMode(FBBDRV *pFbbDrv);
static void HostInfo(FBBDRV *pFbbDrv, char *sFormat, ...);
static char *GetCommand(char *pData, char *szCommand);
// static BOOL PactorConnect(FBBDRV *pFbbDrv, char *szCommand, char *szCall);
//static TIMER *AddTimer(FBBDRV *pFbbDrv, long nCount, void (*pProc)(void *), void *pData);
static BOOL WriteTncBlock(FBBDRV *pFbbDrv, LPSTR lpByte , DWORD dwBytesToWrite);
static int ReadTncChar(FBBDRV *pFbbDrv);
static BOOL InitPort(FBBDRV *pFbbDrv);
static int TncQueueSize(FBBDRV *pFbbDrv);
static BOOL GetDrvVersion(int *nVMaj, int *nVMin);

//////////////////////////////////////////////////////////////
// Global variables
//////////////////////////////////////////////////////////////

FBBDRV	*pDrvHead = NULL;	// Head of port structures

//////////////////////////////////////////////////////////////
//
// Exported functions (5).
// They are :
// BOOL OpenFbbDriver(int nPort, HWND hWnd, void *pDrvInfo);
// BOOL CloseFbbDriver(int nPort);
// BOOL ReadFbbDriver(int nPort, int *nType, int *nChan, char *pszBuf, int *nLen, DRVUI *pUi);
// BOOL WriteFbbDriver(int nPort, int *nType, int nChan, char *pszBuf, int nLen, DRVUI *pUi);
// BOOL StatFbbDriver(int nPort, int nChan, int nCmd, void *pPtr, int nLen);
//
//////////////////////////////////////////////////////////////


BOOL APIENTRY DllMain( HANDLE hModule, 
					  DWORD  ul_reason_for_call, 
					  LPVOID lpReserved
					  )
{
    switch (ul_reason_for_call)
	{
	case DLL_PROCESS_ATTACH:
	case DLL_THREAD_ATTACH:
		GetDrvVersion(&KAP_MAJOR, &KAP_MINOR);
	case DLL_THREAD_DETACH:
		break;
	case DLL_PROCESS_DETACH:
		// Clean all linked ports
		while (pDrvHead)
			CloseFbbDriver(pDrvHead->nPort);
		break;
    }
    return TRUE;
}


BOOL WINAPI OpenFbbDriver(int nPort, HWND hWnd, void *pDrvInfo)
{
	char	szPort[ 15 ];
	BOOL	fRetVal ;
	FBBDRV	*pFbbDrv;
	HANDLE	hDev;
	HANDLE	hFbbThread;
	DWORD	dwThreadID;
	DRVINFO *pInfo;
	COMMTIMEOUTS  CommTimeOuts ;
	
	
	pInfo = (DRVINFO *)pDrvInfo;
	if (pInfo->nMultCh > 2)
		return FALSE;
	
	wsprintf( szPort, "COM%d", pInfo->nCom ) ;
	
	// open COMM device
	if ((hDev =	CreateFile(
		szPort, 
		GENERIC_READ | GENERIC_WRITE,
		0,                    // exclusive access
		NULL,                 // no security attrs
		OPEN_EXISTING,
		FILE_ATTRIBUTE_NORMAL, // |
		//FILE_FLAG_OVERLAPPED, // overlapped I/O
		NULL )) == (HANDLE) -1 )
	{
		return FALSE;
	}
	
	// setup device buffers
	SetupComm(hDev, 1024, 1024 ) ;
	
	// purge any information in the buffer
	PurgeComm(hDev, PURGE_TXCLEAR | PURGE_RXCLEAR ) ;
	
	// Set time out
	CommTimeOuts.ReadIntervalTimeout = 0 ;
	CommTimeOuts.ReadTotalTimeoutMultiplier = 0 ;
	CommTimeOuts.ReadTotalTimeoutConstant = 200 ;
	
	// CBR_9600 is approximately 1byte/ms. For our purposes, allow
	// double the expected time per character for a fudge factor.
	CommTimeOuts.WriteTotalTimeoutMultiplier = 2*CBR_9600/pInfo->nBaudrate;
	CommTimeOuts.WriteTotalTimeoutConstant = 0 ;
	
	SetCommTimeouts(hDev, &CommTimeOuts ) ;
	
	if (!SetupConnection(hDev, pInfo->nBaudrate))
		fRetVal = FALSE;
	
	EscapeCommFunction(hDev, SETDTR );
	EscapeCommFunction(hDev, SETRTS );
	
	pFbbDrv = (FBBDRV *)LocalAlloc(LPTR, sizeof(FBBDRV));
	if (pFbbDrv == NULL)
		return FALSE;
	
	pFbbDrv->pUi     = NULL;
	pFbbDrv->hDev    = hDev;
	pFbbDrv->bUp     = TRUE;
	pFbbDrv->hWnd    = hWnd;
	pFbbDrv->nCom    = pInfo->nCom;
	pFbbDrv->nPort   = nPort;
	
	strcpy(pFbbDrv->szMyCall, pInfo->szMyCall);
	
	// Insert the structure in the list
	pFbbDrv->pNext = pDrvHead;
	pDrvHead = pFbbDrv;
	
	if (hWnd)
		WM_NOTIFY_MSG = RegisterWindowMessage(FBBDRV_NOTIFY);
	
	InitializeCriticalSection(&pFbbDrv->hCritical);
	
	if (StartHostMode(pFbbDrv, pInfo->nMultCh))
	{
		// Create a thread to process the KAM protocol.
		hFbbThread = CreateThread( 
			(LPSECURITY_ATTRIBUTES) NULL,
			0,
			(LPTHREAD_START_ROUTINE) KapLoop,
			(LPVOID) pFbbDrv,
			0, 
			&dwThreadID );
	}
	else
		fRetVal = FALSE;
	
	
	if ((hFbbThread == NULL) || (fRetVal == FALSE))
	{
		EscapeCommFunction(hDev, CLRDTR );
		EscapeCommFunction(hDev, CLRRTS );
		CloseHandle(hDev);
		DeleteCriticalSection(&pFbbDrv->hCritical);
		
		// Remove structure from list
		pDrvHead = pDrvHead->pNext;
		
		// Free the structure
		LocalFree(pFbbDrv);
		
		return FALSE;
	}
	
	pFbbDrv->dwThreadId	= dwThreadID;
	pFbbDrv->hThread	= hFbbThread;
	
	return TRUE;
	
} // end of OpenFbbDriver()

BOOL WINAPI CloseFbbDriver(int nPort)
{
	FBBDRV *pFbbDrv = SearchPort(nPort);
	if (pFbbDrv == NULL)
		return FALSE;

	// TNC to command mode
	EndHostMode(pFbbDrv);
	
	// block until thread has been halted
	pFbbDrv->bUp = FALSE;
	WaitForSingleObject(pFbbDrv->hThread, INFINITE);
	
	DeleteCriticalSection(&pFbbDrv->hCritical);
	
	// Remove structure from list
	if (pDrvHead == pFbbDrv)
		pDrvHead = pDrvHead->pNext;
	else
	{
		FBBDRV *pTmp = pDrvHead;
		while (pTmp->pNext)
		{
			if (pTmp->pNext == pFbbDrv)
			{
				pTmp->pNext = pTmp->pNext->pNext;
				break;
			}
			pTmp = pTmp->pNext;
		}
	}
	
	LocalFree(pFbbDrv);
	
	return ( TRUE ) ;
	
} // end of CloseFbbDriver()

BOOL WINAPI ReadFbbDriver(int *nPort, int *nType, int *nChan, char *pszBuf, int *nLen, DRVUI *pUi)
{
	DATAIN *pPtr;
	FBBDRV *pFbbDrv = SearchPort(*nPort);
	if (pFbbDrv == NULL)
		return FALSE;
	
	pPtr = GetHost(pFbbDrv);
	if (pPtr == NULL)
		return FALSE;
	
	if (pPtr->nLen)
	{
		memcpy(pszBuf, pPtr->pData, pPtr->nLen);
		LocalFree(pPtr->pData);
	}
	*nLen = pPtr->nLen;
	*nChan = 1;
	*nType = pPtr->nType;
	
	if (pPtr->pUi)
	{
		*pUi = *pPtr->pUi;
		LocalFree(pPtr->pUi);
	}
	LocalFree(pPtr);
	
	return TRUE;
} // end of ReadFbbDriver()

BOOL WINAPI WriteFbbDriver(int nPort, int nType, int nChan, char *pszBuf, int nLen, DRVUI *pUi)
{
	FBBDRV *pFbbDrv = SearchPort(nPort);
	if (pFbbDrv == NULL)
		return FALSE;
	
	return ToTnc(pFbbDrv, nType, pszBuf, nLen, pUi);
} // end of WriteFbbDriver()

BOOL WINAPI StatFbbDriver(int nPort, int nCmd, int nChan, void *pPtr, int nLen)
{
	/*
	static char *szConnect[4] = { "CONNECT", "PACTOR", "GTOR", NULL };
	char szCommand[256];
	char *pCall;
	int niI;
	*/

	FBBDRV *pFbbDrv = SearchPort(nPort);
	if (pFbbDrv == NULL)
		return 0;
	
	switch (nCmd)
	{
	case DRV_INIT:
		return KapInit(pFbbDrv, (char *)pPtr);
	case DRV_SNDCMD:
	case DRV_ECHOCMD:
		/*
		pCall = GetCommand((char *)pPtr, szCommand);
		niI = 0;
		while (szConnect[niI])
		{
			if (strnicmp(szCommand, szConnect[niI], strlen(szCommand)) == 0)
				return PactorConnect(pFbbDrv, szCommand, pCall);
			++niI;
		}
		*/
		return (ToTnc(pFbbDrv, DRV_COMMAND, (char *)pPtr, strlen((char *)pPtr), NULL));
	case DRV_PORTCMD:
		return (ToTnc(pFbbDrv, DRV_COMMAND, (char *)pPtr, strlen((char *)pPtr), NULL));
	case DRV_PACLEN:
		*((int *) pPtr) = 250;
		return TRUE;
/*	case DRV_SYSRDY:
		return (ToTnc(pFbbDrv, DRV_COMMAND, "PACTOR", 6, NULL)); */
	case DRV_VERSION:
		wsprintf((char *)pPtr, 
			"v%d.%02d FBB driver for KAM-Pactor (F6FBB-%s)", 
			KAP_MAJOR, KAP_MINOR, __DATE__);
		return TRUE;
	}
	
	return FALSE;
} // end of StatFbbDriver()


//////////////////////////////////////////////////////////////
//
// Local functions
//
//////////////////////////////////////////////////////////////

static BOOL StartHostMode(FBBDRV *pFbbDrv, int nPort)
{
	//	int nLen;
	int nChar;
	int nNb;
	BOOL bOk;
	char *pszRet = "\r";
	char *pszCmd = "\300Q\300";
	char *pszReset = "\rRESET\r";
	char *pszHost = "INTFACE HOST\r";
	
	HostInfo(pFbbDrv, 0, "StartHostMode");

	// Check if TNC is alive
	for (nNb = 0 ; nNb < 4 ; nNb++)
	{
		bOk = FALSE;
		WriteTncBlock(pFbbDrv, pszRet, strlen(pszRet));
		Sleep(1000);
		while ((nChar = ReadTncChar(pFbbDrv)) >= 0)
		{
			Sleep(2);
			bOk = TRUE;
		}
		
		if (bOk)
			break;

		// TNC does not answer. Try going out from HOST mode
		WriteTncBlock(pFbbDrv, pszCmd, strlen(pszCmd));

		Sleep(1000);
	}
	
	if (nNb == 4)
		return FALSE;
	
	// Go to hostmode
	for (nNb = 0 ; nNb < 4 ; nNb++)
	{
		bOk = FALSE;
		WriteTncBlock(pFbbDrv, pszHost, strlen(pszHost));
		Sleep(100);
		while ((nChar = ReadTncChar(pFbbDrv)) >= 0)
		{
			Sleep(10);
			bOk = TRUE;
		}
		
		if (bOk)
			break;
	}
	
	if (nNb == 4)
		return FALSE;
	
	// Reset the TNC
	WriteTncBlock(pFbbDrv, pszReset, strlen(pszReset));
	Sleep(1000);
	
	// Be sure the COM is empty ...
	while (ReadTncChar(pFbbDrv) >= 0)
		Sleep(10);
	
	HostInfo(pFbbDrv, 0, "StartHostMode OK");

	// Init PTC HostMode counter
//	pFbbDrv->bInit = TRUE;

	return InitPort(pFbbDrv);
}

static BOOL EndHostMode(FBBDRV *pFbbDrv)
{
	int nI;

	// Back to cmd: mode
	ToTnc(pFbbDrv, DRV_HOST, "Q", 1, NULL);

	// Wait until all TNC queue is processed
	// Possible deadlock if resync mode !
	// Wait only 10 seconds
	for (nI = 0 ; nI < 100 ; nI++)
	{
		if (TncQueueSize(pFbbDrv) == 0)
			break;
		
		// Wait some time...
		Sleep(100);
	}

	return TRUE;
}

BOOL SetupConnection(HANDLE hDev, int nBaudrate)
{
	BOOL       fRetVal ;
	DCB        dcb ;
	
	dcb.DCBlength = sizeof( DCB ) ;
	
	GetCommState(hDev, &dcb ) ;
	
	dcb.BaudRate = nBaudrate ;
	dcb.ByteSize = 8 ;
	dcb.Parity   = NOPARITY ;
	dcb.StopBits = ONESTOPBIT ;
	
	
	// setup hardware flow control
	
	dcb.fOutxDsrFlow = FALSE ;
	dcb.fDtrControl = DTR_CONTROL_DISABLE ;
	
	dcb.fOutxCtsFlow = FALSE ;
	dcb.fRtsControl = RTS_CONTROL_DISABLE ;
	
	// setup software flow control
	
	dcb.fInX = dcb.fOutX = FALSE ;
	
	// other various settings
	
	dcb.fBinary = TRUE ;
	
	fRetVal = SetCommState(hDev, &dcb ) ;
	
	return ( fRetVal ) ;
	
} // end of SetupConnection()

static int ReadTncChar(FBBDRV *pFbbDrv)
{
	BOOL       fReadStat ;
	COMSTAT    ComStat ;
	DWORD      dwErrorFlags;
	DWORD      dwLength;
	BYTE		nChar;

	// Read one character
	// I don't know how to read more than one character
	// with a timeout, to get any number of bytes
	ClearCommError(pFbbDrv->hDev, &dwErrorFlags, &ComStat ) ;
	fReadStat = ReadFile(pFbbDrv->hDev, &nChar,
		1, &dwLength, NULL) ;
	if (!fReadStat)
	{
		ClearCommError(pFbbDrv->hDev, &dwErrorFlags, &ComStat ) ;
		if (dwErrorFlags > 0)
		{
			HostInfo(pFbbDrv, "<CE-%u>", dwErrorFlags ) ;
		}
		return -1;
	}
	
	if (dwLength == 0)
	{
		return -1;
	}
	
	return ( nChar ) ;	
} // end of ReadTncChar()

static BOOL WriteTncBlock(FBBDRV *pFbbDrv, LPSTR lpByte , DWORD dwBytesToWrite)
{
	
	BOOL        fWriteStat ;
	DWORD       dwBytesWritten ;
	DWORD       dwErrorFlags;
	//	DWORD   	dwError;
	DWORD       dwBytesSent=0;
	COMSTAT     ComStat;
	
	fWriteStat = WriteFile(pFbbDrv->hDev, lpByte, dwBytesToWrite,
		&dwBytesWritten, NULL) ;
	if (!fWriteStat)
	{
		// some other error occurred
		ClearCommError(pFbbDrv->hDev, &dwErrorFlags, &ComStat ) ;
		if (dwErrorFlags > 0)
		{
			HostInfo(pFbbDrv, "<CE-%u>", dwErrorFlags ) ;
		}
		return ( FALSE );
	}
	return ( TRUE ) ;
	
} // end of WriteTncBlock()

static FBBDRV *SearchPort(int nPort)
{
	FBBDRV *pTmp = pDrvHead;
	
	while (pTmp)
	{
		if (pTmp->nPort == nPort)
			return pTmp;
		pTmp = pTmp->pNext;
	}
	
	return NULL;
}

static BOOL KapInit(FBBDRV *pFbbDrv, char *pPtr)
{
	return ToTnc(pFbbDrv, DRV_COMMAND, pPtr, strlen(pPtr), NULL);
/*
	char cCmd = *pPtr++;

	while (*pPtr && isspace(*pPtr))
		++pPtr;

	switch (toupper(cCmd))
	{
	case 'A':
		strncpy(pFbbDrv->szRemTcpAddr, pPtr, 15);
		pFbbDrv->szRemTcpAddr[15] = '\0';
		return TRUE;
	case 'P':
		pFbbDrv->nRemTcpPort = atoi(pPtr);
		return TRUE;
	}
	return FALSE;
*/
}

static BOOL ConnectTo(FBBDRV *pFbbDrv, char *szRemoteCall)
{

	if (pFbbDrv->nState != DRV_DISCONNECTED)
	{
		ToHost(pFbbDrv, DRV_COMMAND, "CHANNEL ALREADY CONNECTED", -1, NULL);
		return TRUE;
	}

	pFbbDrv->nState = DRV_CONNPENDING;
	strcpy(pFbbDrv->szMyCall, pFbbDrv->szMyCall);
	strcpy(pFbbDrv->szCall, strupr(szRemoteCall));

	return TRUE;
}

static char *GetCommand(char *pData, char *szCommand)
{
	char *pScan = pData;
	int nI = 0;
	while (isgraph(*pScan))
	{
		szCommand[nI++] = *pScan++;
		if (nI == 20)
			break;
	}
	szCommand[nI] = '\0';

	while (*pScan && isspace(*pScan))
		++pScan;

	return pScan;
}

static BOOL ToTnc(FBBDRV *pFbbDrv, int nType, char *pData, int nLen, DRVUI *pUi)
{
	if (nLen == -1)
		nLen = strlen(pData);

	if (nLen <= 0 || nLen > 256)
		return FALSE;
	
	if (nType == DRV_UNPROTO)
		return FALSE;
	
	if (nType == DRV_COMMAND && *pData == '#')
		return FALSE;

	DATAOUT *pPtr;
	
	pPtr           = (DATAOUT *)LocalAlloc( LMEM_FIXED, sizeof(DATAOUT));
	pPtr->pData    = (char *)LocalAlloc( LMEM_FIXED, nLen);
	pPtr->nType    = nType;
	pPtr->nLen     = nLen;
	pPtr->pNext    = NULL;

	memcpy(pPtr->pData, pData, nLen);

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	// Add the structure at end of list
	if (pFbbDrv->pTncHead == NULL)
		pFbbDrv->pTncHead = pPtr;
	else
		pFbbDrv->pTncTail->pNext = pPtr;
	
	// Update tail information
	pFbbDrv->pTncTail = pPtr;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);

	return TRUE;
}

static void ToHost(FBBDRV *pFbbDrv, int nType, char *pData, int nLen, DRVUI *pUi)
{
	DATAIN *pPtr;
	
	if (pData == NULL)
		nLen = 0;
	else if (nLen == -1)
		nLen = strlen(pData);

	if (nLen < 0 || nLen > 256)
		return;

	pPtr           = (DATAIN *)LocalAlloc( LMEM_FIXED, sizeof(DATAIN));
	pPtr->nType    = nType;
	pPtr->nLen     = nLen;
	pPtr->pUi      = pUi;
	pPtr->pNext    = NULL;
	
	if (nLen > 0)
	{
		pPtr->pData    = (char *)LocalAlloc( LMEM_FIXED, nLen);
		memcpy(pPtr->pData, pData, nLen);
	}
	else
		pPtr->pData = NULL;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	if (pFbbDrv->pHostHead == NULL)
		pFbbDrv->pHostHead = pPtr;
	else
		pFbbDrv->pHostTail->pNext = pPtr;
	
	// Update tail information
	pFbbDrv->pHostTail = pPtr;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	// Send Notification message
	if (pFbbDrv->hWnd)
		PostMessage(pFbbDrv->hWnd, WM_NOTIFY_MSG, nType, MAKELONG(pFbbDrv->nPort, 1));
}

static DATAOUT *GetTnc(FBBDRV *pFbbDrv)
{
	DATAOUT *pPtr;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	pPtr = pFbbDrv->pTncHead;
	if (pPtr)
		pFbbDrv->pTncHead = pFbbDrv->pTncHead->pNext;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	return pPtr;
}

static DATAIN *GetHost(FBBDRV *pFbbDrv)
{
	DATAIN *pPtr;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	pPtr = pFbbDrv->pHostHead;
	if (pPtr)
		pFbbDrv->pHostHead = pFbbDrv->pHostHead->pNext;
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	return pPtr;
}

static int TncQueueSize(FBBDRV *pFbbDrv)
{
	DATAOUT *pPtr;
	int nNb = 0;
	
	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);
	
	pPtr = pFbbDrv->pTncHead;
	while (pPtr)
	{
		pPtr = pPtr->pNext;
		++nNb;
	}
	
	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);
	
	return nNb;
}

static void HostInfo(FBBDRV *pFbbDrv, char *sFormat, ...)
{
	char	szStr[512];
	va_list pArgPtr;
	int nCnt;
	
	va_start(pArgPtr, sFormat);
	nCnt = wvsprintf(szStr, sFormat, pArgPtr);
	va_end(pArgPtr);
	
	if (nCnt > 0)
		ToHost(pFbbDrv, DRV_INFO, szStr, nCnt, NULL);
}

static BOOL InitPort(FBBDRV *pFbbDrv)
{
	int nNb;
	char *szInit[] = {
		"MX ON",
		"MC ON",
		"MCOM ON",
		"M ON",
		NULL
	};
	
	for (nNb = 0 ; szInit[nNb] ; nNb++)
		ToTnc(pFbbDrv, DRV_COMMAND, szInit[nNb], strlen(szInit[nNb]), NULL);
	return TRUE;
}

void KamStatus(FBBDRV *pFbbDrv)
{
	static time_t nTimePrec;

	time_t nTime = time(NULL);
	
	if (nTimePrec != nTime)
	{
		nTimePrec = nTime;

		ToTnc(pFbbDrv, DRV_COMMAND, "S", -1, NULL);

		if (pFbbDrv->pStats.nState != DRV_DISCONNECTED)
		{
			ToTnc(pFbbDrv, DRV_COMMAND, "TRI", -1, NULL);
		}
	}
}

static void SendTncRequest(FBBDRV *pFbbDrv)
{
	int nI;
	int nLen;
	int nSize = 0;
	char *pData;
	DATAOUT *pPtr;
	char szSend[1024];
	static int test = 0;
	
	pPtr = GetTnc(pFbbDrv);

	if (pPtr == NULL)
	{
		// Nothing to send
		KamStatus(pFbbDrv);
		return;
	}

	// Header
	szSend[nSize++] = (char)FEND;

	switch(pPtr->nType)
	{
	case DRV_HOST:
		break;
	case DRV_COMMAND:
		szSend[nSize++] = 'C';	// Command
		szSend[nSize++] = '2';	// HF port;
		szSend[nSize++] = 'A';	// 1st stream;
		++pFbbDrv->nNbCmd;
		break;
	case DRV_DATA:
		szSend[nSize++] = 'D';	// Data
		szSend[nSize++] = '2';	// HF port;
		szSend[nSize++] = 'A';	// 1st stream;
		break;
	}

	nLen = pPtr->nLen;
	pData = pPtr->pData;

	// Add data
	for (nI = 0 ; nI < nLen ; nI++)
	{
		switch (pData[nI])
		{
		case FEND:
			szSend[nSize++] = (char)FESC;
			szSend[nSize++] = (char)TFEND;
			break;
		case FESC:
			szSend[nSize++] = (char)FESC;
			szSend[nSize++] = (char)TFESC;
			break;
		default:
			szSend[nSize++] = pData[nI];
			break;
		}
	}

	// End of block
	szSend[nSize++] = (char)FEND;

	LocalFree(pPtr->pData);
	LocalFree(pPtr);

	WriteTncBlock(pFbbDrv, (char *)szSend, nSize);
}

static BOOL Disconnected(FBBDRV *pFbbDrv)
{
	char szMsg[300];

	pFbbDrv->nState = DRV_DISCONNECTED;
	wsprintf(szMsg, "(1) DISCONNECTED fm %s", pFbbDrv->szCall);
	ToHost(pFbbDrv, DRV_COMMAND, szMsg, -1, NULL);

	return TRUE;
}

static void KapGetStatus(FBBDRV *pFbbDrv, char *szLine)
{
	int nI;
	int nStat;
	int nAck;
	int nStream;
	int nPort;

#define NB_KAM_STAT	13

#define DRV_DISCONNECTED	0
#define DRV_CONNECTED		1
#define DRV_DISCPENDING		2
#define DRV_CONNPENDING		3
#define DRV_CONNWAIT		4
#define DRV_CONNCHECK		5
	static struct
	{
		int nState;
		int nLg;
		char *szTxt;
	}
	kam_stat[NB_KAM_STAT] =

	{
		{
			DRV_DISCONNECTED, 9, "DISCONNEC"
		}
		,
		{
			DRV_CONNPENDING, 9, "CONNECT i"
		}
		,
		{
			DRV_CONNECTED, 9, "FRMR in p"
		}
		,
		{
			DRV_DISCPENDING, 9, "DISC in p"
		}
		,
		{
			DRV_CONNECTED, 9, "CONNECTED"
		}
		,
		{
			DRV_CONNECTED, 1, "x"
		}
		,
		{
			DRV_CONNECTED, 9, "Waiting a"
		}
		,
		{
			DRV_CONNECTED, 9, "Device bu"
		}
		,
		{
			DRV_CONNECTED, 9, "Remote de"
		}
		,
		{
			DRV_CONNECTED, 9, "Both devi"
		}
		,
		{
			DRV_CONNECTED, 17, "Waiting ACK and d"
		}
		,
		{
			DRV_CONNECTED, 17, "Waiting ACK and r"
		}
		,
		{
			DRV_CONNECTED, 17, "Waiting ACK and b"
		}
		,
	};


	char *pScan = szLine;
	if (pScan[1] != '/')
		return;

	nStream = pScan[0] - '@';
	nPort = (pScan[2] == 'V') ? 1 : 2;

	pScan += 4;
	if (*pScan == '#')
	{
		while ((*pScan) && (*pScan != '('))
			++pScan;
		++pScan;
		nAck = 0;
		while (isdigit (*pScan))
		{
			nAck *= 10;
			nAck += (*pScan - '0');
			++pScan;
		}
		pScan += 2;
	}
	else
	{
		nAck = 0;
	}

	nStat = 20;
	for (nI = 0; nI < NB_KAM_STAT; nI++)
	{
		if (strncmp (kam_stat[nI].szTxt, pScan, kam_stat[nI].nLg) == 0)
		{
			nStat = kam_stat[nI].nState;
			break;
		}
	}

	if (nStat == DRV_DISCONNECTED)
	{
		nAck = 0;
	}

	if (pFbbDrv->pStats.nState != nStat)
	{
		pFbbDrv->pStats.nState = nStat;
		pFbbDrv->pStats.nAux = 1;
	}

	if (pFbbDrv->pStats.nNbFrames != nAck)
	{
		pFbbDrv->pStats.nNbFrames = nAck;
		pFbbDrv->pStats.nAux = 1;
	}
}

static void	KapStatus(FBBDRV *pFbbDrv, char *pData)
{
	char *pLine;

	pLine = strtok(pData, "\r");
	while (pLine)
	{
		KapGetStatus(pFbbDrv, pLine);
		pLine = strtok(NULL, "\r");
	}
}

static void	KapMonitor(FBBDRV *pFbbDrv, int nPort, int nStream, char *pData)
{
	char *pHead;
	char *pTxt;
	DRVUI ui;
	int nHeadLen;

	// Header
	pHead = strtok(pData, "\r");
	nHeadLen = strlen(pHead);

	// Data
	pTxt = strtok(NULL, "\r");


	// Extract fields for the header line

	char *ptr;
	char *sptr;

	memset (&ui, 0, sizeof (DRVUI));

//	ui.nPort = nPort;
	ui.nPort = pFbbDrv->nPort;

	ptr = strtok (pHead, ">"); /* exped */

	if (ptr == NULL)
		return;
	strncpy (ui.szFrom, ptr, 11); ui.szFrom[11] = '\0';

	ptr = strtok (NULL, ":,");	/* desti */

	if (ptr == NULL)
		return;
	strncpy (ui.szTo, ptr, 11); ui.szTo[11] = '\0';

	ptr = strtok (NULL, ":,");	/* digis */

	if (ptr == NULL)
		return;

	if (*ptr != ' ')
	{
		for (;;)
		{
			strncat (ui.szVia, ptr, 12);
			strcat (ui.szVia, " ");

			ptr = strtok (NULL, ":,");	/* digis */

			if (ptr == NULL)
				return;

			if ((*ptr == '\0') || (*ptr == ' '))
				break;

		}
	}

	++ptr;
	sptr = ptr;

	while ((*sptr) && (*sptr != '>'))
		++sptr;
	*sptr = '\0';

	/* controle */
	*ui.szCtl = '\0';

	if (ptr[0] == '<')
	{
		int pos = 0;
		int version = 1;
		int reponse = 0;

		++ptr;
		if (ptr[0] == '<')
		{
			version = 2;
			/* AX25 Version 2 */
			++ptr;
		}

		sptr = ptr;
		if (*sptr == 'F')
		{
			pos = 4;
		}
		else if (*sptr == 'U')
		{
			pos = 2;
			if (sptr[1] == 'A')
				reponse = 1;
		}
		else if (*sptr == 'C')
		{
			strcpy (ptr, "SABM");
			pos = 4;
		}
		else if (*sptr == 'D')
		{
			strcpy (ptr, "DISC");
			pos = 4;
		}
		else if (*sptr == 'I')
		{
			pos = 3;
		}
		else
		{
			if (*sptr == 'r')
			{
				strupr (sptr);
				reponse = 1;
			}
			if (sptr[1] == 'R')
				pos = 3;
			else
				pos = 4;
		}

		if (version == 1)
		{
			if (reponse)
				sptr[pos] = '\0';
			else
				sptr[pos] = '!';
		}
		else
		{
			if (reponse)
				sptr[pos] = '-';
			else
				sptr[pos] = '+';
		}
		sptr[pos + 1] = '\0';
		strncpy (ui.szCtl, ptr, 4); ui.szCtl[4] = '\0';
	}

	ui.bUi = (strncmp (ui.szCtl, "UI", 2) == 0);

	ui.nPid = 0xf0;

	DRVUI *pUi = (DRVUI *)LocalAlloc(LMEM_FIXED, sizeof(DRVUI));
	if (pUi == NULL)
		return;

	*pUi = ui;
	ToHost(pFbbDrv, DRV_UNPROTO, pTxt, -1, pUi);
}

static void ReadTncAnswer(FBBDRV *pFbbDrv)
{
	int nChar;
	int nLen;
	int nPort;
	int nStream;
	int nStatus;
	BOOL bFesc;
	char szBuf[300];
/*	
	char szAnswer[80];
	char szCall[80];
*/
	char *pData;
	
	nChar = ReadTncChar(pFbbDrv);
	if (nChar == -1)
		return;

	if (nChar != FEND)
		return;

	// Fills a buffer
	nLen = 0;
	bFesc = FALSE;

	while (pFbbDrv->bUp)
	{
		nChar = ReadTncChar(pFbbDrv);
		if (nChar == -1)
		{
//			Resync(pFbbDrv);
			return;
		}
		else if (nChar == FEND)
		{
			if (nLen > 0)
				break;
		}
		else if (nChar == FESC)
		{
			bFesc = TRUE;
		}
		else if (bFesc)
		{
			bFesc = FALSE;
			if (nChar == TFESC)
				szBuf[nLen++] = (char)FESC;
			else if (nChar == TFEND)
				szBuf[nLen++] = (char)FEND;
			else
				return;	// Block error ... Abort
		}
		else
		{
			szBuf[nLen++] = nChar;
		}
	}

	if (!pFbbDrv->bUp)
		return;

	// Decode the block
	nStatus = szBuf[0];
	nPort   = szBuf[1] - '1';
	nStream = szBuf[2];
	pData   = szBuf+3;
	nLen   -= 3;

	pData[nLen] = '\0';

	switch (nStatus)
	{
	case '?':	// Status
		HostInfo(pFbbDrv, "S:%c%c %02x %02x", nStream, pData[0], pData[1] & 0xff, pData[2] & 0xff);
		break;
	case 'C':	// Command answer
		--pFbbDrv->nNbCmd;
		if (strncmp(pData, "FREE BYTES", 10) == 0)
			KapStatus(pFbbDrv, pData);
		else if (strncmp(pData, "TRIES", 5) == 0)
		{
			int nTries = atoi(pData+7);
			if (pFbbDrv->pStats.nNbRetry != nTries)
			{
				pFbbDrv->pStats.nNbRetry = nTries;
				pFbbDrv->pStats.nAux = 1;
			}
		}
		break;
	case 'D':	// Data
		ToHost(pFbbDrv, DRV_DATA, pData, nLen, NULL);
		break;
	case 'I':	// ISS/IRS info
		pFbbDrv->bIss = (*pData == '1');
		break;
	case 'M':	// Monitor
		KapMonitor(pFbbDrv, nPort, nStream, pData);
		break;
	case 'R':	// Connect request
		break;
	case 'S':	// Link state
		/*sscanf(pData, "%s %s", szAnswer, szCall);
		if (strncmp(szCall, "STANDBY", 7) == 0)
			Disconnected(pFbbDrv);
		*/
		if (strncmp(pData, "*** ", 4) != 0)
			break;
		switch (pData[4])
		{
		case 'C':
			// Connected
			pFbbDrv->nState = DRV_CONNECTED;
			sscanf(pData, "%*s %*s %*s %s", pFbbDrv->szCall);
			wsprintf(szBuf, "(1) CONNECTED to %s", pFbbDrv->szCall);
			ToHost(pFbbDrv, DRV_COMMAND, szBuf, -1, NULL);
			break;
		case 'r':
			// retry count
			break;
		case 'D':
			// Disconnected
			pFbbDrv->nState = DRV_DISCONNECTED;
			wsprintf(szBuf, "(1) DISCONNECTED fm KAM");
			ToHost(pFbbDrv, DRV_COMMAND, szBuf, -1, NULL);
			break;
		}
		break;
	case 'T':	// Trace
		break;
	}

	DRVSTATS DrvStats;

	/*
	if (DrvStats.nState != pFbbDrv->pStats.nState)
	{
		pFbbDrv->pStats.nState = DrvStats.nState;
		pFbbDrv->pStats.nAux = 1;
	}
	*/

	if (pFbbDrv->pStats.nAux)
	{
		pFbbDrv->pStats.nAux = 0;
		DrvStats = pFbbDrv->pStats;
		DrvStats.nBuffers  = 500;
		DrvStats.nFrameNb  = 0;

		ToHost(pFbbDrv, DRV_STATS, (char *)&DrvStats, sizeof(DRVSTATS), NULL);
	}
}

#if 0
static void PactorPhase(void *pPtr)
{
	FBBDRV *pFbbDrv = (FBBDRV *)pPtr;

	switch(pFbbDrv->nState)
	{
	case DRV_CONNWAIT:
		pFbbDrv->nState = DRV_CONNCHECK;
		pFbbDrv->pPacCall->pTimer = AddTimer(pFbbDrv, 10, PactorPhase, (void *)pFbbDrv);
		break;
	
	case DRV_CONNCHECK:
		// Pactor connection
		ToTnc(pFbbDrv, DRV_COMMAND, pFbbDrv->pPacCall->szCommand, -1, NULL);
		LocalFree(pFbbDrv->pPacCall);
		pFbbDrv->pPacCall = NULL;
		pFbbDrv->nState = DRV_CONNPENDING;
		break;
	}
}

static BOOL PactorConnect(FBBDRV *pFbbDrv, char *szCommand, char *szCall)
{

	if (*szCall == '\0')
		return FALSE;

	pFbbDrv->pPacCall = (PACCALL *)LocalAlloc(LPTR, sizeof(PACCALL));

	if (pFbbDrv->pPacCall == NULL)
		return FALSE;

	// Back to cmd: mode
	ToTnc(pFbbDrv, DRV_HOST, "X", 1, NULL);

	// New state
	pFbbDrv->nState = DRV_CONNWAIT;

	if (*szCommand == 'C' || *szCommand == 'c')
		strcpy(szCommand, "PACTOR");
	strcpy(pFbbDrv->szCall, szCall);
	wsprintf(pFbbDrv->pPacCall->szCommand, "%s %s", szCommand, szCall);
	pFbbDrv->pPacCall->pTimer = AddTimer(pFbbDrv, 5, PactorPhase, (void *)pFbbDrv);

	return TRUE;
}

// Timer functions

static TIMER *AddTimer(FBBDRV *pFbbDrv, long nCount, void (*pProc)(void *), void *pData)
{
	TIMER *pTimer;

	pTimer = (TIMER *)LocalAlloc(LPTR, sizeof(TIMER));
	if (pTimer == NULL)
		return NULL;

	pTimer->nCount = nCount;
	pTimer->pProc  = pProc;
	pTimer->pData  = pData;

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);

	pTimer->pNext = pFbbDrv->pTimerHead;
	pFbbDrv->pTimerHead = pTimer;

	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);

	return pTimer;
}

static void *DelTimer(FBBDRV *pFbbDrv, TIMER *pTimer)
{
	TIMER *pScan;
	TIMER *pPrec;
	void *pPtr;

	if (pTimer == NULL)
		return NULL;

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);

	for (pPrec = NULL, pScan = pFbbDrv->pTimerHead ; pScan ; pScan = pScan->pNext)
	{
		if (pScan == pTimer)
		{
			if (pPrec)
				pPrec->pNext = pTimer->pNext;
			else
				pFbbDrv->pTimerHead = pTimer->pNext;
			break;
		}
		pPrec = pScan;
	}

	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);

	pPtr = pTimer->pData;
	LocalFree(pTimer);

	return pPtr;
}

static void Timer(FBBDRV *pFbbDrv)
{
	static time_t nPrec = 0L;
	time_t nDelta;
	TIMER *pScan;

	nDelta = time(NULL) - nPrec;
	if (nDelta == 0L)
		return;

	nPrec += nDelta;

	// Must be thread safe
	EnterCriticalSection(&pFbbDrv->hCritical);

	for (pScan = pFbbDrv->pTimerHead ; pScan ; pScan = pScan->pNext)
	{
		if (pScan->nCount > nDelta)
			pScan->nCount -= nDelta;
		else
			pScan->nCount = 0L;
	}

	// Must be thread safe
	LeaveCriticalSection(&pFbbDrv->hCritical);

	// Now scan for expired timers
	for (;;)
	{
		// Must be thread safe
		EnterCriticalSection(&pFbbDrv->hCritical);

		for (pScan = pFbbDrv->pTimerHead ; pScan ; pScan = pScan->pNext)
		{
			if (pScan->nCount == 0L)
				break;
		}

		// Must be thread safe
		LeaveCriticalSection(&pFbbDrv->hCritical);

		if (pScan)
		{
			// Call the procedure
			if (pScan->pProc)
				(*pScan->pProc)(pScan->pData);

			// Delete the timer
			DelTimer(pFbbDrv, pScan);
		}
		else
		{
			// No more expired timer
			break;
		}
	}
}

#endif

// KAM Thread

DWORD WINAPI KapLoop(LPSTR lpData)
{
	FBBDRV	*pFbbDrv = (FBBDRV*) lpData ;

	while ( pFbbDrv->bUp )
	{
		// Send the request
		SendTncRequest(pFbbDrv);
		
		// Read the answer
		ReadTncAnswer(pFbbDrv);

		// Timer
		// Timer(pFbbDrv);

		// Status of the Pactor channel
		// PactorStatus(pFbbDrv);
	}
	
	pFbbDrv->dwThreadId = 0 ;
	pFbbDrv->hThread    = NULL ;
	
	return( TRUE ) ;
		
} // end of KapLoop()


BOOL GetDrvVersion(int *nVMaj, int *nVMin)
{
	DWORD dwVersionInfoSize;
	DWORD dwTmpHandle;
	LPVOID lpVersionInfo; 
	BOOL bRet = false;


	dwVersionInfoSize = GetFileVersionInfoSize(FBBDLLNAME, &dwTmpHandle);

	if(dwVersionInfoSize)
	{
		lpVersionInfo = LocalAlloc(LPTR, dwVersionInfoSize);
		if(lpVersionInfo)
		{
			if(GetFileVersionInfo(FBBDLLNAME, dwTmpHandle, dwVersionInfoSize, lpVersionInfo))
			{
				LPVOID lpBuffer = LocalAlloc(LPTR, dwVersionInfoSize);
				UINT dwBytes;

				if( VerQueryValue(lpVersionInfo, 
						TEXT("\\StringFileInfo\\000004B0\\FileVersion"), 
						&lpBuffer, 
						&dwBytes)) 
				{
						sscanf((TCHAR *)lpBuffer, "%d,%d", nVMaj, nVMin);
						bRet = true;
				}

				LocalFree(lpBuffer);
			}

			LocalFree(lpVersionInfo);
 		}
 	}

	return bRet;
}
