
/*
 *	af_ax25.c
 *
 *	AX.25 sockets specific functions
 */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>

#include "hmalloc.h"

#include "csock.h"
#include "af_ax25.h"
#include "log.h"
#include "config.h"

#ifdef HAVE_AX25

#include <linux/ax25.h>
#include <linux/netrom.h>
#include <linux/rose.h>

#include "axutils.h"
#include "axconfig.h"

/*
 *	Bind() an ax.25 interface and listen() on it
 */

void ax25_listen(struct listenq_t *lq)
{
	log(L_SINFO, "Binding AX.25 interface %s:%s%s",
		lq->port, lq->call, lq->compress ? " (compressed)" : "");
	
	if ((lq->fd = socket(lq->af_type, SOCK_SEQPACKET, 0)) < 0) {
		log(L_CRIT, "ax25_listen(): socket(): %s", strerror(errno));
		exit(1);
	}
	
	if (lq->fd > maxfd)
		maxfd = lq->fd;
	
	if (bind(lq->fd, (struct sockaddr *)&lq->sockaddr, lq->addrlen) < 0) {
		log(L_CRIT, "Could not bind() AX.25 interface %s:%s: %s", 
			lq->port, lq->call, strerror(errno));
		close(lq->fd);
		exit(1);
	}
	
	if (listen(lq->fd, SOMAXCONN) < 0) {
		log(L_CRIT, "Could not listen() on AX.25 interface %s:%s: %s",
			lq->port, lq->call, strerror(errno));
		close(lq->fd);
		exit(1);
	}
	
	FD_SET(lq->fd, &readfds);
}

/*
 *	Start listening on ax25 interfaces
 */

void ax25_listens(void)
{
	struct listenq_t *lq;
	
	for (lq = listenq; (lq); lq = lq->next)
		if (lq->af_type == AF_AX25)
			ax25_listen(lq);
}

/*
 *	Accept an AX.25 connection
 */

void ax25_accept(struct listenq_t *lq)
{
	int i;
	int addrlen, new_fd;
	struct csock_t *s;
	union {
		struct full_sockaddr_ax25 ax25;
	} sockaddr;
	
	i = 1;
	ioctl(lq->fd, FIONBIO, &i);	/* Set up a non-blocking accept() */
	
	addrlen = sizeof(struct full_sockaddr_ax25);
	new_fd = accept(lq->fd, (struct sockaddr *)&sockaddr, &addrlen);
	
	i = 0;
	ioctl(lq->fd, FIONBIO, &i);
	
	if (new_fd < 0) {
		if ((errno == EWOULDBLOCK) || (errno == EINTR))
			return;
		
		log(L_ERR, "AX.25 accept() error: %s", strerror(errno));
		log(L_ERR, "Closing AX.25 interface %s:%s (fd %d)", 
			lq->port, lq->call, lq->fd);
		
		close(lq->fd);
		lq->fd = -1;
		return;
	}
	
	i = 1;
	ioctl(new_fd, FIONBIO, &i);	/* Make the socket non-blocking */
	
	s = sock_alloc(new_fd, AF_AX25, DEF_BUFLEN_AX, DEF_BUFLEN_AX, lq->compress);
	s->eoltype = ax25_eol;
	s->eol = AX25_EOL;
	
	s->call = hstrdup(ax2asc(&sockaddr.ax25.fsa_ax25.sax25_call));
	s->node = NULL;
	
	addrlen = sizeof(struct full_sockaddr_ax25);
	getsockname(new_fd, (struct sockaddr *)&sockaddr, &addrlen);
	s->port = hstrdup(ax25_config_get_port(&sockaddr.ax25.fsa_digipeater[0]));
	
	sock_login(s);
}

/*
 *	Prepare an outgoing ax.25 connection
 */
 
int ax25_prepare_outgoing(struct csock_t *s, int argc, char **argv)
{
	char path[20];
	char *portcall;
	
	if (argc < 2)
		log(L_ERR, "ax25_prepare_outgoing(): Not enough arguments, need at least port and destination (digis optional)");
	
	portcall = ax25_config_get_addr(argv[0]);
	if (!portcall) {
		cs_errno = CS_E_INVPORT;
		log(L_ERR, "ax25_prepare_outgoing(): Invalid port: %s", argv[0]);
		return -1;
	}
	
	sprintf(path, "%s %s", (char *)clucall, portcall);
	convert_call(path, &s->sockaddr.ax25);
	s->sockaddr.ax25.fsa_ax25.sax25_family = AF_AX25;
	s->addrlen = sizeof(struct full_sockaddr_ax25);
	
	if (bind(s->fd, (struct sockaddr *)&s->sockaddr, s->addrlen) == -1) {
		cs_errno = errno;
		log(L_ERR, "ax25_prepare_outgoing(): AF_AX25: bind(): %s", strerror(cs_errno));
		return -1;
	}
	
	if (convert_call_arglist(&argv[1], &s->sockaddr.ax25) == -1) {
		cs_errno = CS_E_INVCALL;
		log(L_ERR, "ax25_prepare_outgoing(): convert_call_arglist failed");
		return -1;
	}
	
	s->call = strdup(argv[1]);
	s->port = strdup(argv[0]);
	s->eoltype = ax25_eol;
	s->eol = AX25_EOL;
	
	return 0;
}

#endif

