
/*
 *	net_user.c
 *
 *	network user list handling
 */
 
#include <string.h>
#include <stdio.h>

#include "net_user.h"
#include "network.h"
#include "net_link.h"
#include "hmalloc.h"
#include "log.h"
#include "cluster.h"
#include "f_nuser.h"

struct nuser_t *nusers;	/* Users */

int nuser_count;		/* How many users on the network */
int kuser_count;		/* How many KNOWN users on the network */
int nuser_max;			/* Users high water mark */
int kuser_max;			/* Known users high water */

int away_strings;		/* Away strings allocated */

/* ====================================================================== */
/*  U S E R   T A B L E   H A N D L I N G                                 */
/* ====================================================================== */

/*
 *	Delete an user from the user list
 */

void free_nuser(struct nuser_t *nu)
{
	*nu->prevp = nu->next;
	if (nu->next)
		nu->next->prevp = nu->prevp;
	if (nu->away_str) {
		hfree(nu->away_str);
		away_strings--;
	}
	hfree(nu);
	nuser_count--;
	kuser_count--;
}

/*
 *	Delete all users on a node
 */

void free_nusersn(struct node_t *n)
{
	struct nuser_t *nu, *next;

	nu = nusers;
	while (nu) {
		next = nu->next;
		if (nu->node == n)
			free_nuser(nu);
		nu = next;
	}
}

/*
 * 	Add an user to the list 
 */

struct nuser_t *insert_nuser(struct nuser_t *nu)
{
	struct nuser_t **prevp, *p;
	
	prevp = &nusers;
	p = nusers;
	while ((p) && (strcmp(p->call, nu->call) < 0)) {
		prevp = &p->next;
		p = p->next;
	}
	
	*prevp = nu;
	nu->prevp = prevp;
	nu->next = p;
	if (nu->next)
		nu->next->prevp = &nu->next;
	nuser_count++;
	kuser_count++;
	
	if (nuser_count > nuser_max)	/* High water mark */
		nuser_max = nuser_count;
	if (kuser_count > kuser_max)	/* High water mark */
		kuser_max = kuser_count;
	
	return nu;
}

/*
 *	Find a record from the user list
 */

struct nuser_t *get_nuser(call_t *call)
{
	struct nuser_t *p;

	p = nusers;
	while ((p) && (strcmp(p->call, (char *)call) != 0))
		p = p->next;
	return p;
}

struct nuser_t *get_nuserh(call_t *call, struct node_t *node)
{
	struct nuser_t *p;

	p = nusers;
	while ((p) && (strcmp(p->call, (char *)call) || p->node != node))
		p = p->next;
	return p;
}

struct nuser_t *guess_nuser(call_t *call)
{
	struct nuser_t *user, *u, *p;
	int l, m;

	p = nusers;
	m = 0;
	l = strlen((char *)call);
	if (l != 0) {
		while ((p)) {
			/* Match substring: */
			if (strstr(p->call, (char *)call)) {
				if (!strcmp((char *)call, p->call))
					/* Whoa, exact match! Accept this, even if there are others */
					return p;
				m++;
				u = p;
			}
			p = p->next;
		}
	}
	if (m != 1)		/* Exactly one match */
		return NULL;
	user = u;
	strcpy((char *)call, u->call);
	return user;
}

/*
 *	Calculate the total users count nuser_count, based on the node list
 */

int count_users(void)
{
	struct node_t *n = nodes;
	
	nuser_count = 0;
	while (n) {
		nuser_count += n->users;
		n = n->next;
	}
	if (nuser_count > nuser_max)	/* High water mark */
		nuser_max = nuser_count;

	return nuser_count;
}

/*
 *	User came back
 */

void net_here(struct link_t *l, call_t *call)
{
	struct nuser_t *nu;

	if (!(nu = get_nuser(call)) || ((l) && (nu->node == localnode)) || 
	   (!nu->here))
		return;

	nu->here = 1;
	nu->away_time = time(NULL);
	if (nu->away_str) {
		hfree(nu->away_str);
		nu->away_str = NULL;
		away_strings--;
	}
	log(L_NUSER, "%s came back.", call);
	user_userhere(nu);
	link_userhere(nu);
}


/*
 *	User went away
 */
 
void net_away(struct link_t *l, call_t *call, char *reason)
{
	struct nuser_t *u;

	if (!(u = get_nuser(call)) || ((l) && (u->node == localnode)) )
		return;

	if (reason) {
		if (u->away_str)	/* The user might just be changing the string! */
			hfree(u->away_str);
		else
			away_strings++;
		u->away_str = hstrdup(reason);
		log(L_NUSER, "%s is away: %s", call, reason);
	} else
		log(L_NUSER, "%s is away.", call);
	
	link_useraway(u);
	u->here = 0;
	u->away_time = time(NULL);
	user_useraway(u);
}

/*
 *	Add an user
 */

void net_adduser(struct nuser_t *nu)
{
	int b;
	struct nuser_t *p, *n;
	char s[256];
	
	p = nu;
	sprintf(s, "PC16^%s", nu->node->call);
	b = 0;
	
	while (p) {
		n = p->next;
		
		if (get_nuserh(&p->call, p->node) || (valid_call(&p->call) == -1))
			hfree(p);
		else {
			b++;
			p->node->users++;
			p->node->users_known = 1;
			insert_nuser(p);
			if (p->node != localnode)
				log(L_NUSER, "%s@%s login", p->call, p->node->call);
			user_useradd(p);
			/* Spread the word */
			sprintf(s + strlen(s), "^%s - ", p->call);
			if (p->here)
				strcat(s, "1");
			else
				strcat(s, "0");
			set_homenode(p, &p->call, &p->node->call, 5);
		}
		p = n;
	}
	
	if (b == 0)
		return;
		
	sprintf(s + strlen(s), "^%s^\n", hops2pcstr(pc_hops));
	if (pc_hops <= 98)
		pc_sendall(nu->node->via, (void *) s);	/* !BUG! u might be freed!  */
}

/*
 *	User login
 */

struct nuser_t *nuser_login(struct link_t *l, call_t *call, struct node_t *n, 
			int here)
{
	struct nuser_t *u;
	
	u = hmalloc(sizeof(struct nuser_t));
	
	strcpy(u->call, (char *)call);
	u->node = n;
	*u->name = '\0';
	u->here = here;
	u->away_str = NULL;
	u->away_time = time(NULL);
	u->sysop = 0;
	u->priviledged = 0;
	u->next = NULL;
	u->prevp = NULL;
	u->time = time(NULL);
	net_adduser(u);
	
	return u;
}

/*
 *	Remove a cluster user
 */

void net_deluser(struct link_t *l, struct nuser_t *nu)
{
	nu->time = time(NULL);
	user_userdel(nu);
	
	if (l)
		log(L_NUSER, "%s@%s logout", nu->call, nu->node->call);
		
	link_userdel(nu);
	
	nu->node->users--;
	free_nuser(nu);
}

/*
 *	User logout by callsign/node
 */

void nuser_logout(struct link_t *l, call_t *call, struct node_t *n, time_t time)
{
	struct nuser_t *u;

	if ((l) && (n == localnode))	/* Check for a loop */
		return;
		
	if (!(u = get_nuserh(call, n)))
		return;

	net_deluser(l, u);
}

