(******************************************************************************
*                                   xmsLib                                    *
******************************************************************************)

Unit xmsLib;

  { An XMS library. Mostly implemented in inline assembly. I found this
    from somewhere, a long time ago, i suppose it's in the public domain
    (not under the GPL). Author unknown. I have added the accounting, and
    an user command to check the results. }

interface

type
   xmsMovePtr = ^xmsMoveStructure;
   xmsMoveStructure = record
      length         : longint; { 32-bit # of bytes to transfer }
      sourceHandle   : word;
      sourceOffset   : longint;
      destHandle     : word;
      destOffset     : longint;
   end; { xmsMoveStructure definition }
   UserString  = string[15];
var
   xmsPresent   : boolean; { true if XMS was detected }
   xmsDisabled  : boolean; { true if XMS usage disabled }
   xmsVersion   : word;
   xmmVersion   : word;
   hmaPresent   : boolean;
   xmsErrorCode : byte;    { if an error exists, it will be placed here }

   xmsUsed      : LongInt; { Bytes used by this program }
   xmsHandles   : Byte;    { XMS handles in use }

procedure detectXMS; { look for xms existance, and sets global library variables }
procedure setXMSHandlerAddress;
procedure getXMSVersionNumber;
function  printXMSVersion : string; { a readable string .. }
function  printXMMVersion : string; { a readable string .. }
procedure queryFreeExtendedMemory(var largestBlock, totalInK : word);
function xmsLargestBlock : word;
function xmsTotalFreeMemory : word;
function allocateXMB(sizeInK : word; var handle : word; Const user : userString ) : boolean;
function freeXMB(handle : word) : boolean;
function moveXMB(structure : xmsMovePtr) : boolean;
function moveXMBlock(len : longint; srcHandle : word; srcOfs : longint;
                     dstHandle : word; dstOfs : longint) : boolean;
function BaseToXMB(len : word; fromPtr : pointer;
                      toHandle : word; toOfs : longint) : boolean;
function XMBtoBase(len : word; toPtr : pointer;
                      fmHandle : word; fmOfs : longint) : boolean;
function lockXMB(handle : word) : boolean;
function unlockXMB(handle : word) : boolean;
function getXMBInformation(handle : word; var lockCount, freeHandles : byte;
                           var sizeInK : word) : boolean;
function reallocXMB(newSizeInK, handle : word) : boolean;
function requestUMB(sizeInParagraphs : word; var segmentOfUMB : word;
                    var sizeAllocatedOrAvailable : word) : boolean;
function releaseUMB(segmentOfUMB : word) : boolean;
function xmsErrorStr : string;
procedure xmsCheck(Result:Boolean);
procedure freeAllXms;

procedure xmsStatus_Cmd(p:Byte);

implementation
uses Dos, BPQ, CStrings, Config;
type
   xmsErrorType  = record
                   errorNumber  : byte;
                   errorMessage : string[39];
                   end;

   xmsHandleType = record
                   handle       : word;
                   size         : word;
                   user         : UserString;
                   {mIn, mOut    : LongInt;}
                   end;

const
   maxXMSErrors = 27;
   xmsErrorArray : array [1 .. maxXMSErrors] of xmsErrorType = (
      (errorNumber : $80; errorMessage :  'Function not implemented'),
      (errorNumber : $81; errorMessage :  'VDISK device detected'),
      (errorNumber : $82; errorMessage :  'A20 Error occured'),
      (errorNumber : $8e; errorMessage :  'General driver error'),
      (errorNumber : $8f; errorMessage :  'Fatal driver error'),
      (errorNumber : $90; errorMessage :  'HMA does not exist'),
      (errorNumber : $91; errorMessage :  'HMA is already in use'),
      (errorNumber : $92; errorMessage :  'Size is smaller then /HMAMIN= parameter'),
      (errorNumber : $93; errorMessage :  'HMA not allocated'),
      (errorNumber : $94; errorMessage :  'A20 line still enabled'),
      (errorNumber : $a0; errorMessage :  'No more free XMS memory'),
      (errorNumber : $a1; errorMessage :  'No more XMS handles'),
      (errorNumber : $a2; errorMessage :  'Invalid handle'),
      (errorNumber : $a3; errorMessage :  'Invalid source handle'),
      (errorNumber : $a4; errorMessage :  'Invalid source offset'),
      (errorNumber : $a5; errorMessage :  'Invalid destination handle'),
      (errorNumber : $a6; errorMessage :  'Invalid destination offset'),
      (errorNumber : $a7; errorMessage :  'Invalid length'),
      (errorNumber : $a8; errorMessage :  'Move resulted in overlap'),
      (errorNumber : $a9; errorMessage :  'Parity error'),
      (errorNumber : $aa; errorMessage :  'Block not locked'),
      (errorNumber : $ab; errorMessage :  'Block locked'),
      (errorNumber : $ac; errorMessage :  'Block lock count overflow'),
      (errorNumber : $ad; errorMessage :  'Lock failure'),
      (errorNumber : $b0; errorMessage :  'Smaller UMB available'),
      (errorNumber : $b1; errorMessage :  'No UMBs available'),
      (errorNumber : $b2; errorMessage :  'Invalid UMB segment number')
      );
   xmsMaxHandles = 30;

var
   xmsAddress : pointer; { used to point to XMS entry address }

   xmsHandle  : Array[1..xmsMaxHandles] of xmsHandleType;  { XMS handles used }

   xms2main, main2xms, xmsAllocs, xmsFrees : LongInt;

(******************************************************************************
*                                  detectXMS                                  *
******************************************************************************)

procedure detectXMS;
Var b : Byte;
begin
     asm
        mov xmsPresent, 0 { no xms available }
        mov ax, $4300
        int $2f { multiplexer interrupt identification }
        cmp al, $80 { well , is there XMM ? }
        jne @noXMSDriver
        mov xmsPresent, 1 { true, we have an xms driver }
            @noXMSDriver:
     end; { asm }

 xmsUsed := 0;
 xmsHandles := 0;
 For b := 1 to 10
  do Begin
     xmsHandle[b].Handle := 0;
     End;

 If xmsPresent
   then Begin
        setXMSHandlerAddress;
        getXMSVersionNumber;
        End;

 xms2main  := 0;
 main2xms  := 0;
 xmsAllocs := 0;
 xmsFrees  := 0;

end; {detectXMS}

(******************************************************************************
*                            setXMSHandlerAddress                             *
******************************************************************************)
procedure setXMSHandlerAddress;
begin
     asm
        mov ax,$4310
        int $2f { ES:BX points to xms driver entry point }
        mov word ptr [xmsAddress], bx
        mov word ptr [xmsAddress + 2], es
     end; { asm }
end; {setXMSHandlerAddress}

(******************************************************************************
*                             getXMSVersionNumber                             *
******************************************************************************)
procedure getXMSVersionNumber;
begin
     asm
        xor ah, ah; { function 0 .. }
        call [xmsAddress]
        mov xmsVersion, ax
        mov xmmVersion, bx
        mov byte ptr hmaPresent, dl { true or false .. }
     end; { asm }
end; {getXMSVersionNumber}

(******************************************************************************
*                               printXMSVersion                               *
******************************************************************************)
function printXMSVersion;
var
   s1, s2  : string;
begin
   str(xmsVersion div $100, s1);
   str(xmsVersion mod $100, s2);
   printXMSVersion := s1 + '.' + s2;
end; {printXMSVersion}

(******************************************************************************
*                               printXMMVersion                               *
******************************************************************************)
function printXMMVersion;
var
   s1, s2, s3  : string;
begin
   str(XMMVersion div $100, s1);
   str((XMMVersion mod $100) div $10, s2);
   str(XMMVersion mod $10, s3);
   printXMMVersion := s1 + '.'+ s2 + s3;
end; {printXMMVersion}

(******************************************************************************
*                           queryFreeExtendedMemory                           *
******************************************************************************)
procedure queryFreeExtendedMemory;
var
   ourLB, ourTIK : word;
begin
   asm
      mov ah, 8
      call [xmsAddress]
      mov ourLB, ax
      mov ourTIK, dx
      mov xmsErrorCode, bl
   end; { asm }
   largestBlock := ourLB;
   totalInK := ourTIK;
end; {queryFreeExtendedMemory}

(******************************************************************************
*                               xmsLargestBlock                               *
******************************************************************************)
function xmsLargestBlock;
var
   lb, tik : word;
begin
   queryFreeExtendedMemory(lb, tik);
   xmsLargestBlock := lb;
end; {xmsLargestBlock}

(******************************************************************************
*                             xmsTotalFreeMemory                              *
******************************************************************************)
function xmsTotalFreeMemory;
var
   lb, tik : word;
begin
   queryFreeExtendedMemory(lb, tik);
   xmsTotalFreeMemory := tik;
end; {xmsTotalFreeMemory}

(******************************************************************************
*                                 allocateXMB                                 *
* if returns True handle has the handle to the memory block                   *
******************************************************************************)
function allocateXMB;
var
   allocGranted : boolean;
   ourHandle    : word;
   b            : Byte;
begin
   asm
      mov ah, 9
      mov dx, sizeInK
      call [xmsAddress]
      mov allocGranted, al { did we make it ? }
      mov ourHandle, dx
      mov xmsErrorCode, bl
   end; { asm }
   allocateXMB := allocGranted;
   if (allocGranted) then
      Begin
      handle := ourHandle;
      b := 1;
      While not (xmsHandle[b].Handle = 0)
       do Inc(b);
      xmsHandle[b].Handle := ourHandle;
      xmsHandle[b].Size := sizeInK;
      xmsHandle[b].User := User;
      xmsUsed := xmsUsed + sizeInK;
      Inc(xmsHandles);
      Inc(xmsAllocs);
      End;
end; {allocateXMB}

(******************************************************************************
*                                   freeXMB                                   *
******************************************************************************)
function freeXMB;
var
   releaseGranted : boolean;
   b              : byte;
begin
   asm
      mov ah, $a
      mov dx, handle
      call [xmsAddress]
      mov releaseGranted, al
      mov xmsErrorCode, bl
   end; { asm }
   freeXMB := releaseGranted;
   If releaseGranted
     then Begin
          b := 1;
          while xmsHandle[b].Handle <> handle
           do Inc(b);
          xmsHandle[b].Handle := 0;
          xmsUsed := xmsUsed - xmsHandle[b].Size;
          Dec(xmsHandles);
          Inc(xmsFrees);
          End;

end; {freeXMB}

(******************************************************************************
*                                   moveXMB                                   *
******************************************************************************)
function moveXMB;
var
   moveGranted : boolean;
   segmento    : word;
   offseto     : word;
begin
   segmento := seg(structure^);
   offseto  := ofs(structure^);
   asm
      push ds
      pop es
      mov si, offseto
      mov ax, segmento
      mov ds, ax
      mov ah, $b
      call [es:xmsAddress]
      push es
      pop ds
      mov moveGranted, al
      mov xmsErrorCode, bl
   end; { asm }
   moveXMB := moveGranted;
end; {moveXMB}

(******************************************************************************
*                                 moveXMBlock                                 *
******************************************************************************)
function moveXMBlock;
var
   struct : xmsMoveStructure;
begin
   with struct do begin
      length := len;
      sourceHandle := srcHandle;
      sourceOffset := srcOfs;
      destHandle := dstHandle;
      destOffset := dstOfs;
   end; { with }
   moveXMBlock := moveXMB(@struct); { go do it ! }
end; {moveXMBlock}

(******************************************************************************
*                                BaseToXMB                                 *
* move fm ptr len bytes to XMB handle, at offset                              *
******************************************************************************)
function BaseToXMB;
var
   l : longint;
begin
   l := longint(fromPtr);
   BaseToXMB := moveXMBlock(len, 0, l, toHandle, toOfs);
   Inc(Main2XMS);
end; {BaseToXMB}

(******************************************************************************
*                                XMBtoBase                                 *
* xmb fmhandle at ofsset fmofs, move to main storage at pointer toptr, len byt*
******************************************************************************)
function XMBtoBase;
var
   l : longint;
begin
   l := longint(toPtr);
   XMBtoBase := moveXMBlock(len, fmHandle, fmOfs, 0, l);
   Inc(XMS2Main);
end; {XMBtoBase}

(******************************************************************************
*                                   lockXMB                                   *
******************************************************************************)
function lockXMB;
var
   lockGranted : boolean;
begin
   asm
      mov ah, $c
      mov dx, handle
      call [xmsAddress]
      mov lockGranted, al
      mov xmsErrorCode, bl
   end; { asm }
   lockXMB := lockGranted;
end; {lockXMB}

(******************************************************************************
*                                  unlockXMB                                  *
******************************************************************************)
function unlockXMB;
var
   unlockGranted : boolean;
begin
   asm
      mov ah, $d
      mov dx, handle
      call [xmsAddress]
      mov unlockGranted, al
      mov xmsErrorCode, bl
   end; { asm }
   unlockXMB := unlockGranted;
end; {unlockXMB}

(******************************************************************************
*                              getXMBInformation                              *
******************************************************************************)
function getXMBInformation;
var
   informationReceived : boolean;
   ourSIK              : word;
   ourFH, ourLC        : byte;
begin
   asm
      mov ah, $e
      mov dx, handle
      call [xmsAddress]
      mov informationReceived, al
      mov ourLC, bh
      mov ourFH, bl
      mov ourSIK, dx
      mov xmsErrorCode, bl
   end; { asm }
   getXMBInformation := informationReceived;
   sizeInK := ourSIK;
   freeHandles := ourFH;
   lockCount := ourLC;
end; {getXMBInformation}

(******************************************************************************
*                                 reallocXMB                                  *
******************************************************************************)
function reallocXMB;
var
   reallocGranted : boolean;
begin
   asm
      mov ah, $f
      mov bx, newSizeInK
      mov dx, handle
      call [xmsAddress]
      mov reallocGranted, al
      mov xmsErrorCode, bl
   end; { asm }
   reallocXMB := reallocGranted;
end; {reallocXMB}

(******************************************************************************
*                                 requestUMB                                  *
******************************************************************************)
function requestUMB;
var
   requestGranted : boolean;
   ourSOUMB, ourSAOA : word;
begin
   asm
      mov ah, $10
      mov dx, sizeInParagraphs
      call [xmsAddress]
      mov requestGranted, al
      mov ourSOUMB, bx
      mov ourSAOA, dx
      mov xmsErrorCode, bl
   end; { asm }
   requestUMB := requestGranted;
   segmentOfUMB := ourSOUMB;
   sizeAllocatedOrAvailable := ourSAOA;
end; {requestUMB}

(******************************************************************************
*                                 releaseUMB                                  *
******************************************************************************)
function releaseUMB;
var
   releaseGranted : boolean;
begin
   asm
      mov ah, $11
      mov dx, segmentOfUMB
      call [xmsAddress]
      mov releaseGranted, al
      mov xmsErrorCode, bl
   end; { asm }
   releaseUMB := releaseGranted;
end; {releaseUMB}

(******************************************************************************
*                                 xmsErrorStr                                 *
******************************************************************************)
function xmsErrorStr;
var
   i, errorFound : byte;
begin
   errorFound := 0;
   for i := 1 to maxXMSErrors do
      if (xmsErrorCode = xmsErrorArray[i].errorNumber) then
         errorFound := i;
   if (errorFound = 0) then
      xmsErrorStr := 'Unknown XMS error'
   else
      xmsErrorStr := xmsErrorArray[errorFound].errorMessage;
end; {xmsErrorStr}

(******************************************************************************
*                              xmsCheck                                       *
******************************************************************************)

procedure xmsCheck(Result:Boolean);
Begin

 If not Result
  then CriticalError('XMS error: ' + xmsErrorStr,12);

End;

(******************************************************************************
*                                 FreeAllXMS                                  *
******************************************************************************)

Procedure freeAllXms;
Var
 b  : byte;
 ok : boolean;
Begin

 For b := 1 to xmsMaxHandles
  do If (xmsHandle[b].handle > 0)
       then ok := freeXMB(xmshandle[b].handle);

End;

(******************************************************************************
*                                 xmsStatus_Cmd                               *
******************************************************************************)

procedure xmsStatus_Cmd(p:Byte);
var
  b       : byte;
  Frees,
  Locks   : byte;
  h, Size : Word;
Begin

 If xmsPresent
   then Begin
        Send(p,'XMS status:' + Cr
             + ' XMS Version ' + PrintXMSVersion + ', XMM version ' + PrintXMMVersion + '. '
                  + Int2Str(XMSTotalFreeMemory) + ' KB free, largest block '
                  + Int2Str(XMSLargestBlock) + ' KB. ' + Cr);
        If AllocateXMB(1,h,'') and GetXMBInformation(h,Locks,Frees,Size)
          and FreeXMB(h)
           then Begin
                Dec(xmsAllocs); Dec(xmsFrees);
                Send(p, ' ' + Int2Str(Frees + 1) + ' handles free, ' + Int2Str(Locks) + ' locked. '
                      + Int2Str(XMSUsed) + ' KB used by Clusse in '
                      + Int2Str(xmsHandles) + ' handles.' + Cr);
                End;
        Send(p,' Moves: ' + Int2Str(main2xms) + ' in, ' + Int2Str(xms2main) + ' out. '
                 + Int2Str(xmsAllocs) + ' allocs, ' + Int2Str(xmsFrees) + ' frees.' + Cr + Cr);

        Send(p,' Handle Size  Used for' + Cr);
        For b := 1 to xmsMaxHandles
         do if xmsHandle[b].Handle > 0
              then Send(p, '  ' + HexW2Str(xmsHandle[b].Handle) + 'h '
                        + PadRight(5,Int2Str(xmsHandle[b].Size) + 'K') + ' ' + xmsHandle[b].User + Cr);
        End
   else If xmsDisabled
          then Send(p,'XMS usage disabled.' + Cr)
          else Send(p,'XMS not detected.' + Cr);


End;

(******************************************************************************
*                                    MAIN                                     *
******************************************************************************)

end.
