//----------------------------------------------------------------------------
//                             PSAUTIL 
//   A collection of program utilities for programming PSA sound cards. 
//      Demo software accompanying the  QEX  August  1994 article:	       
//      "Using a Programmable PC Sound Card for Amateur Radio"         
//                  (c) Johan Forrer, KC7WW			       
//		     26553 Priceview Drive                             
//                    Monroe, OR 97456				       
//----------------------------------------------------------------------------
//                        ACKNOWLEDGEMENTS
//                        ----------------
// This collection of program utilities contains contributions from several
// talented programmers - I merely re-organized things a bit, added comments
// and simplified matters to get you started. This work would not
// have been possible without the efforts of:
//
// Joe Ballantyne, Billy Brackenridge, Milo Street, and Matt Gonzalez, all 
// who works for Echo Speech Corp.
//
// Believe it or not, but I value the support from Analog Devices
// DSP Applications group. This includes many ecellent code examples for
// the ADSP-21xx as well as making the SCOPE tools available.
//
//----------------------------------------------------------------------------
//                         DISCLAIMER
//			   ----------
// The author('s) of this software utility is not responsible or liable
// for any loss or mishaps that may occur due the use of this software.
// It is provided solely for educational purposes and comes with no
// guarantees. What you see is what you get. Use at your discretion.
//								       
//----------------------------------------------------------------------------
//     Compilation/link instructions for Borland C++                                  
//     >bcc -c -ml psautil.cpp
//								       
//----------------------------------------------------------------------------
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <dos.h>

#include "psadefs.h"

//--------------------------------------------------------------------
// Global data
//--------------------------------------------------------------------
unsigned int wssbase=0x530;	// Valid addresses are
				// 530h, 640h, EB0h, or F40h
//---------------------------------------------------------------------
unsigned DSPdata;		// These fields will be filled
unsigned DSPstatus;		// with data when the DSP board
unsigned DSPcontrol;		// hardware is probed
unsigned DSPtype;
unsigned DSPindexed;
unsigned DSPversion;
unsigned DSPverID;
//---------------------------------------------------------------------


//--------------------------------------------------------------------------
// This routine probes the DSP hardware to determine
// where the harware is located and the type of ASIC 
//--------------------------------------------------------------------------
int findPSS()
{
int asic_id = 0;
int port_offset = 0;

//
// First we look for the older style ESC-614 ASIC
// Look at port 220h first:
	asic_id=inpw(PSS_ID_VERS_614);
	if ((asic_id & 0xff00)!=0x4500)	 {
  		port_offset=0x20;
		asic_id=inpw(PSS_ID_VERS_614+port_offset);
		if ((asic_id & 0xff00)!=0x4500)  asic_id=0;
 		}

	if (asic_id) {
//
// Located an ESC-614 ASIC
//
  		printf("Echo ESC614 ASIC detected at %#x.\n",PSS_DATA_614+port_offset);
  		printf("Rev %d ASIC found.\n",asic_id & 0xFF);
  		DSPdata = PSS_DATA_614+port_offset;
  		DSPstatus = PSS_STATUS_614+port_offset;
  		DSPcontrol = PSS_CONTROL_614+port_offset;
  		DSPtype = ESC614;
  		DSPverID= PSS_ID_VERS_614+port_offset;
		DSPversion = asic_id & 0xFF; 
		return 1;
 		} 

//
// Otherwise we look for the latest ESC-615 style ASIC
// Look at port 230h first:
 	else 	{
   		port_offset=0;
   		outpw(PSS_CONTROL_615,ID_VERSION_615);
   		asic_id=inpw(PSS_INDEXED_615);
   		if((asic_id & 0xff00)!=0x4500 && (asic_id & 0xff00)!=0x5000) {
     			port_offset=0x20;
     			outpw(PSS_CONTROL_615+port_offset,ID_VERSION_615);
     			asic_id=inpw(PSS_INDEXED_615+port_offset);
     			if ((asic_id & 0xff00)!=0x4500 && (asic_id & 0xff00)!=0x5000)
      				asic_id=0;
    			}

		if (asic_id) {
//
// Located an ESC-615
//
  			DSPdata = PSS_DATA_615+port_offset;
  			DSPstatus = PSS_STATUS_615+port_offset;
  			DSPcontrol = PSS_CONTROL_615+port_offset;
  			DSPtype = ESC615;
  			DSPindexed = PSS_INDEXED_615 + port_offset;
  			printf("Echo ESC615 detected at %#x.\n",PSS_DATA_615+port_offset);
  			printf("Rev %d ASIC found.\n",asic_id & 0xFF);
			DSPversion = asic_id & 0xFF; 
			return 1;
 			} 
 		else  return 0;	   // Could not detect the ECHO chip
		}
}

//--------------------------------------------------------------------------
// Routine to do the proper interrupt acknowledge sequence
//--------------------------------------------------------------------------

int ackInt()
{
	switch (DSPtype) {
 		case ESC614:	
  			outpw(DSPverID, 0);
  			break;
 		case ESC615:	
  			inpw(DSPindexed);
  			break;
 		default:
  			return 0;
		}

	return 1;
}

//--------------------------------------------------------------------------
// Routines to do port DSP I/O 
//--------------------------------------------------------------------------
unsigned int getdspword()
{
int i;

	for (i=0; i<1024; i++)
       		if(inpw(DSPstatus) & PSS_STAT_RF)
    			return inpw(DSPdata);
	printf("getdspword: Read timeout error\n");
	exit(1);

	return 0;	// To keep compiler happy
}

//--------------------------------------------------------------------------

int putdspword(unsigned int data)
{
int i;
	for (i=0; i<1024; i++)
    		if(inpw(DSPstatus) & PSS_STAT_WE) {
    			outpw(DSPdata, data);
    			return 0;
    			}
	printf("putdspword: Write timeout error\n");
	exit(1);
	return 0; 	// To keep compiler happy
}

//---------------------------------------------------------------------------
void do_hard_reset()
{
time_t timeout;

	outpw(DSPcontrol,PSS_CNTRL_RESET);  // reset the DSP
	timeout=clock()+DELAY;
	while (timeout>clock())
  		outpw(DSPcontrol,0x0000);   // Release the reset

printf("do_hard_reset: Hard reset\n");
}

//---------------------------------------------------------------------------
// Routine to perform a boot
// Hard reset is optional
//---------------------------------------------------------------------------
int bootDSP(char * ps, int force_reset)
{
FILE *fin;
int j, iVersion;
long count=0;
time_t timeout;
//
// Open file containing boot data
//
	if ((fin=fopen(ps,"rb"))==NULL)	{
  		printf("bootDSP: Unable to open %s\n",ps);
  		return 0;
 		}

	j=fgetc(fin);                   // get first byte of boot sequence
	count++;

//
// If hard reset is required, pull on DSP's reset line
//
	if (force_reset)  {
  		outpw(DSPdata,j);  // latch out first byte of boot
  		do_hard_reset();
 		}

//
// Continue with boot process
//
	while (!feof(fin)) {
  		if (!(inpw(DSPstatus) & PSS_STAT_FLAG3)) {
    			timeout=clock()+DELAY;
    			while (timeout>clock() && !(inpw(DSPstatus) & PSS_STAT_FLAG3));
    				if (clock()>timeout) {
      					printf("bootDSP: Timeout error after %d bytes.\a\n",count);
      					fclose(fin);
      					return 0;
     					}
   		}

  	outpw(DSPdata,fgetc(fin));  // next byte
  	count++;
 	}

	outpw(DSPdata,0);   	// this last dummy write is required

//
// Now wait for the DSP to wake up
//
	timeout=DELAY+clock();
	while (!(inpw(DSPstatus) & PSS_STAT_RF) && timeout>clock());

	if (clock()>timeout)  {
  		printf("bootDSP: Timeout error: version number not read.");
  		fclose(fin);
  		return 0;
  		}

	iVersion=inpw(DSPdata);
	printf("bootDSP: DSP loaded with %s ver: %u.%u \n",ps, (iVersion >> 8), iVersion & 255);

   	fclose(fin);
 	return 1;
} 

//--------------------------------------------------------------------------
// AD1848 support routines
//--------------------------------------------------------------------------
// This routine gets the state of the 1848 without changing it at all.
// This means that we read the initial state of the index register.  Then we
// read the status register.  Then we read all of the 16 indexed registers
// on the 1848.  We do this by changing just the low order 4 bits in the index
// register and reading the indexed register data.  When we are done, we 
// restore the index register state to its initial state.  The passed pointer
// must point to at least 18 bytes of storage for the returned state.
// 
// State:
//     0 - Index Address register (also contains MCE and TRD bits)
//     1 - 16 contents of Indexed data register
//--------------------------------------------------------------------------

void get1848state(unsigned char *state)
{
int i,index,initialindex;

	*state++=index=initialindex=(unsigned char)inp(wssbase+WSS_CODECINDEXADDR);
	*state++=(unsigned char)inp(wssbase+WSS_CODECSTATUS);
	index=index&0xFFF0;

	for (i=0;i<16;i++) {
		outp(wssbase+WSS_CODECINDEXADDR,index);
		*state++=(unsigned char)inp(wssbase+WSS_CODECINDEXDATA);
		index++;
		}

	outp(wssbase+WSS_CODECINDEXADDR,initialindex);
}

//--------------------------------------------------------------------------

void set1848state(unsigned char *newstate)
{
int j;
unsigned char state[18];
clock_t delay;

// Get and save the current AD1848 state
	get1848state(state);

// See if we need to set the MCE bit in order to change the state.  If
// not, then simply make the change.  If so, then we must follow a special
// procedure to change the bytes protected by the MCE bit before changing
// the rest of the bytes.
//
	if ((newstate[10]!=state[10])||((newstate[11]&0x00fc)!=
		(state[11]&0x00fc))) { 

// We need to change the sample rate: mute DACs.
		outp(wssbase+WSS_CODECINDEXADDR,(state[0]&0x00f0)|0x0006);
		outp(wssbase+WSS_CODECINDEXDATA,state[8]|0x0080);
		outp(wssbase+WSS_CODECINDEXADDR,(state[0]&0x00f0)|0x0007);
		outp(wssbase+WSS_CODECINDEXDATA,state[9]|0x0080);
//
// If autocalibration then mute AUX1 and AUX2 and digital mix. */
//
		if (newstate[11]&0x0008) {
		   outp(wssbase+WSS_CODECINDEXADDR,(state[0]&0x00f0)|0x0002);
		   outp(wssbase+WSS_CODECINDEXDATA,state[4]|0x0080);
		   outp(wssbase+WSS_CODECINDEXADDR,(state[0]&0x00f0)|0x0003);
		   outp(wssbase+WSS_CODECINDEXDATA,state[5]|0x0080);
		   outp(wssbase+WSS_CODECINDEXADDR,(state[0]&0x00f0)|0x0004);
		   outp(wssbase+WSS_CODECINDEXDATA,state[6]|0x0080);
		   outp(wssbase+WSS_CODECINDEXADDR,(state[0]&0x00f0)|0x0005);
		   outp(wssbase+WSS_CODECINDEXDATA,state[7]|0x0080);	
		   outp(wssbase+WSS_CODECINDEXADDR,(state[0]&0x00f0)|0x000d);
		   outp(wssbase+WSS_CODECINDEXDATA,state[15]&0x00fe);
		   }

// Now change the data format if needed.
		if (newstate[10]!=state[10]) {

// Set the mode change enable bit in the CODECINDEXADDRESS register.
		  outp(wssbase+WSS_CODECINDEXADDR,(state[0]&0x00f0)|0x0048);	

// Update the sample rate register.
		  outp(wssbase+WSS_CODECINDEXDATA,newstate[10]);

// Wait for resyncronization initialization to finish.
		  while (inp(wssbase+WSS_CODECINDEXADDR)==0x0080);
		  }	

// Now change the interface configuration if needed. */
		if (newstate[11]!=state[11]) {

// Set the mode change enable bit in the CODECINDEXADDRESS register. */
		  outp(wssbase+WSS_CODECINDEXADDR,(state[0]&0x00f0)|0x0049);	

// Update the sample rate register.
		  outp(wssbase+WSS_CODECINDEXDATA,newstate[11]);
		  }

// Now reset the MCE bit and at the same time point to the test and
// initialization register so that we can wait for autocalibration to
// finish if it is being done.

		outp(wssbase+WSS_CODECINDEXADDR,(state[0]&0x00b0)|0x000b);		

// If autocalibration or K grade part, then wait for it to finish.  To
// do that correctly, wait for the ACI bit to go high and then low.

		if ((newstate[11]&0x0008)||(state[14]!=0x0009)) { // 48?
		  outp(wssbase+WSS_CODECINDEXADDR,(state[0]&0x00f0)|0x004B);	

// Wait for ACI high
		  while (inp(wssbase+WSS_CODECINDEXDATA)&0x0020==0);

// wait for ACI low
		  while (inp(wssbase+WSS_CODECINDEXDATA)&0x0020);
		  }
		else { 

// We have a J grade part and no autocalibration 
// delay 70ms for 16.9MHz clock and 90ms for 24.5MHz clock.
// See p. 17&24 of AD1848 spec.
		  delay=clock()+((newstate[10]&0x0001)?70:90);
		  while (clock()<delay);
		  }
		}


// The 2 low order bits changed.
	else if (newstate[11]!=state[11]) { 
		outp(wssbase+WSS_CODECINDEXADDR,(state[0]&0x00f0)|0x0009);	
		outp(wssbase+WSS_CODECINDEXDATA,newstate[11]);
	}

// Now we have changed the protected registers (if needed) so now we change
// the rest of the registers.  Note that we only change registers that need
// changing.  Also we skip the 2 MCE protected registers that we changed if
// needed in the above code.
	for (j=2;j<10;j++) {
		if (newstate[j]!=state[j]) {
		  outp(wssbase+WSS_CODECINDEXADDR,(newstate[0]&0x00f0)|(j-2));		
		  outp(wssbase+WSS_CODECINDEXDATA,newstate[j]);
		  }
		}

	for (j=12;j<18;j++) {
		if (newstate[j]!=state[j]) {
		  outp(wssbase+WSS_CODECINDEXADDR,(newstate[0]&0x00f0)|(j-2));
		  outp(wssbase+WSS_CODECINDEXDATA,newstate[j]);
		  }
		}

// Finally we output the desired state of the codec index register.  We
//always output this because we don't know its exact state after the above 
//loops execute.
	outp(wssbase+WSS_CODECINDEXADDR,newstate[0]);
}

//--------------------------------------------------------------------------
// The following routines:
//  		read_data()
//  		write_data()
//  		read_prog()
//  		write_prog()
// 		load_DSP_mem()
// are an extention of the DSP kernel. These allow one to implement simple
// debugging code. The last function is a simple code loader. Its purpose
// is to load individual code modules - its for those that do not have
// the ADI toolkit and thus do not have a linker/sprom splitter to create
// multiple boot pages. A hard way to do things, but its free.
//
//--------------------------------------------------------------------------

unsigned int read_data(unsigned int dm_address)
{
	putdspword(READ_DATA_CMD);         // send read command
	putdspword(dm_address);            // send address
	return(getdspword());
}

//--------------------------------------------------------------------------
void write_data(unsigned int dm_address, unsigned int dm_data)
{
	putdspword(WRITE_DATA_CMD);        // send write command
	putdspword(dm_address);            // send address
	putdspword(dm_data);               // send data
}

//--------------------------------------------------------------------------
unsigned long read_prog(unsigned int pm_address)
{
unsigned long pm_data;

	putdspword(READ_PROG_CMD);         // send read command
	putdspword(pm_address);            // send address
	pm_data = getdspword();       	   // get msw data
	pm_data = (pm_data << 8) | getdspword();           // get px data
	return(pm_data);
}

//--------------------------------------------------------------------------
void write_prog(unsigned int pm_address, unsigned long pm_data)
{
	putdspword(WRITE_PROG_CMD);        // send write command
	putdspword(pm_address);            // send address
	putdspword((unsigned int)(pm_data&0x000000ff)); // send px data
	putdspword((unsigned int)(pm_data>>8));     	// send msw data
}

//--------------------------------------------------------------------------
// This routine makes sure that the DSP test program loaded OK.  It does
// this by sending a PROG_ID_CMD and verifying that the correct string gets
// sent back from the DSP. 
//--------------------------------------------------------------------------
void checkdsptestprog(void)
{
int i;
char id[10];

	putdspword(PROG_ID_CMD);	/* send program id command         */
	for(i=0; i<8; i++)
		id[i]=getdspword();	/* read back test program response */
		if(strcmp(id,"PSA-MON") != 0) {
		printf("checkdsptestprog:Received bad bootstrap ID - Boot load FAILED\n");
		exit(1);
		}
}

//-------------------------------------------------------------------------
// Load DSP memory from a ".CDE" file
//-------------------------------------------------------------------------
int load_DSP_mem(char *filename)
{
long instruction;
unsigned int data_ptr;
FILE *fin;
char in_line[80];

int hextoi(char);

	if ((fin = fopen(filename, "rt")) == NULL){
		printf("load_DSP_mem: Failed to open %s\n", filename);
		return -1;
		}

	for(;;) {
		fscanf(fin,"%s", in_line);		// get a field
		if(feof(fin)) break;

// Get address
		data_ptr = hextoi(in_line[0]);
		data_ptr = data_ptr*16 + hextoi(in_line[1]);
		data_ptr = data_ptr*16 + hextoi(in_line[2]);
		data_ptr = data_ptr*16 + hextoi(in_line[3]);

		if((data_ptr<0x0800)&&(data_ptr>0x03FF)) {
			printf("load_DSP_mem: Cannot load code in reserved code space (0400 - 07FF)\n");
			fclose(fin);
			return -1;
			}

// get instruction = 3 bytes or 6 hex characters 
		fscanf(fin,"%s", in_line);		// get a line
		if(feof(fin)) break;

		instruction = (long)hextoi(in_line[0]);
		instruction = instruction*16 + (long)hextoi(in_line[1]);
		instruction = instruction*16 + (long)hextoi(in_line[2]);
		instruction = instruction*16 + (long)hextoi(in_line[3]);
		instruction = instruction*16 + (long)hextoi(in_line[4]);
		instruction = instruction*16 + (long)hextoi(in_line[5]);

		write_prog(data_ptr, instruction);
		if(instruction!=read_prog(data_ptr)) {
			printf("load_DSP_mem: Write at %X failed??\n", data_ptr);
			fclose(fin);
			return -1;
			}
		}

	return 0;
}

//---------------------------------------------------------------------------
int hextoi(char in_char)
{
int i;

	switch (in_char) {
		case '0':
			i = 0;
			break;
		case '1':
			i = 1;
			break;
		case '2':
			i = 2;
			break;
		case '3':
			i = 3;
			break;
		case '4':
			i = 4;
			break;
		case '5':
			i = 5;
			break;
		case '6':
			i = 6;
			break;
		case '7':
			i = 7;
			break;
		case '8':
			i = 8;
			break;
		case '9':
			i = 9;
			break;
		case 'A':
			i = 10;
			break;
		case 'B':
			i = 11;
			break;
		case 'C':
			i = 12;
			break;
		case 'D':
			i = 13;
			break;
		case 'E':
			i = 14;
			break;
		case 'F':
			i = 15;
			break;
		default:
			return(255);
		}
	return(i);
}
//---------------------------------------------------------------------------
