/*  XNECVIEW - a program for visualizing NEC2 structure and gain files
 *
 *  Copyright (C) 1998-1999, P.T. de Boer -- pa3fwm@amsat.org
 *
 *  Distributed on the conditions of the GPL: see the files README and
 *  COPYING, which accompany this source file.
 *
 *  This module contains all drawing related stuff, like coordinate
 *  transformation and drawing routines based on generic low-level graphics
 *  routines, as provided by the Outdev-struct pointed to by *out.
 */

#include <stdio.h>
#include <math.h>
#include <X11/X.h>
#include <X11/Xlib.h>

#include "xnecview.h"


/* ------------------------- generic drawing macros ------------------------------------- */

#define SetLineAttributes(a,b,c,d) out->SetLineAttributes((a),(b),(c),(d))

#define DrawLine(a,b,c,d) out->DrawLine((a),(b),(c),(d))

#define SetForeground(a) out->SetForeground(a)

#define ClearWindow() out->ClearWindow()

#define DrawStringLL(a,b,c) out->DrawStringLL(a,b,c)     /* draw text specifying lower left corner */
#define DrawStringUL(a,b,c) out->DrawStringUL(a,b,c)     /* draw text specifying upper left corner */

/* ------------------------- coordinate conversion, drawing routines -------------------- */

                               /* these define the projection: */
double phi=ini_phi,theta=ini_theta;    /* angles defining direction of eye */
double zoom=ini_zoom;                  /* zoom factor */
double trx=ini_trx,try=ini_try;        /* 2D-translation, as a fraction of winsize */
int winsizex,winsizey;                 /* size of window in pixels */

                               /* and these are calculated from them: */
double Xx,Xy,Xz,Yx,Yy,Yz;         /* projection matrix */
double Xtr,Ytr;                   /* 2D-translation */
int winsize;                      /* min(winsizex,winsizey) */
Point eyevec;                     /* tip of arrow pointing toward observer */



void calcproj(void)     /* calculate the projection matrix etc. */
{
   double ph,th;
   double scale;

   if (winsizey<winsizex) winsize=winsizey;
   else winsize=winsizex;
   scale=winsize*zoom/(3.0*extr);
   ph=phi*(M_PI/180.);
   th=theta*(M_PI/180.);
   Xz = 0;
   Yz = -scale*sin(th);
   Xx = -scale*sin(ph);
   Xy = scale*cos(ph);
   Yx = scale*cos(ph)*cos(th);
   Yy = scale*sin(ph)*cos(th);
   Xtr = 0.5 + trx*zoom*winsize + 0.5*winsizex;
   Ytr = 0.5 + try*zoom*winsize + 0.5*winsizey;
   eyevec.x = sin(th)*cos(ph);
   eyevec.y = sin(th)*sin(ph);
   eyevec.z = cos(th);
}


void proj(Point *p,double *c)      /* perform the projection: *p defines a point in 3D-space, *c defines a point in the window */
{
   c[0] = Xtr + Xx*p->x + Xy*p->y + Xz*p->z;
   c[1] = Ytr + Yx*p->x + Yy*p->y + Yz*p->z;
}


void interpolate(double *p,double f,double *a,double *b)   /* interpolate linearly between two points in the window (f=0..1 -> *p=*a..*b) */
{
   p[0]=(1-f)*a[0]+f*b[0];
   p[1]=(1-f)*a[1]+f*b[1];
}



/* test whether a line through two points is "ascending" ( / ) as opposed to "descending" ( \ ) */
#define ascending(c1,c2) (((c2)[0]-(c1)[0])*((c2)[1]-(c1)[1])<0)

void draw_axes()		/* draw the axes */
{
   double c0[2],c1[2],c2[2],c3[2];
   Point pp;

   SetForeground(&c_axis);
   SetLineAttributes(0, LineSolid, CapRound, JoinRound);
   pp.x=0; pp.y=0; pp.z=0; proj(&pp,c0);
   pp.x=Axislen*extr; pp.y=0; pp.z=0; proj(&pp,c1);
   pp.x=0; pp.y=Axislen*extr; pp.z=0; proj(&pp,c2);
   pp.x=0; pp.y=0; pp.z=Axislen*extr; proj(&pp,c3);
   DrawLine(c0[0], c0[1], c1[0], c1[1]);
   DrawLine(c0[0], c0[1], c2[0], c2[1]);
   DrawLine(c0[0], c0[1], c3[0], c3[1]);
   if (!quick) {
      if (c1[0]>=c0[0] || !ascending(c0,c1)) DrawStringLL(c1[0]+2,c1[1],"X"); else DrawStringUL(c1[0]+2,c1[1],"X");
      if (c2[0]>=c0[0] || !ascending(c0,c2)) DrawStringLL(c2[0]+2,c2[1],"Y"); else DrawStringUL(c2[0]+2,c2[1],"Y");
      if (c3[0]>=c0[0] || !ascending(c0,c3)) DrawStringLL(c3[0]+2,c3[1],"Z"); else DrawStringUL(c3[0]+2,c3[1],"Z");
   }

   SetLineAttributes(0, LineOnOffDash, CapRound, JoinRound);
   pp.x=0; pp.y=0; pp.z=0; proj(&pp,c0);
   pp.x=-Axislen*extr; pp.y=0; pp.z=0; proj(&pp,c1);
   DrawLine(c0[0], c0[1], c1[0], c1[1]);
   pp.x=0; pp.y=-Axislen*extr; pp.z=0; proj(&pp,c1);
   DrawLine(c0[0], c0[1], c1[0], c1[1]);
   pp.x=0; pp.y=0; pp.z=-Axislen*extr; proj(&pp,c1);
   DrawLine(c0[0], c0[1], c1[0], c1[1]);
}


void draw_wires(void)                /* draw the wires of the antenna */
{
   Wire *wi;
   double c0[2],c1[2];
   char s[20];

   if (numwires==0) return;
   SetForeground(&c_wire);
   SetLineAttributes(2, LineSolid, CapRound, JoinRound);
   for (wi=wires;wi<wires+numwires;wi++) {
      proj(&wi->p0,c0);
      proj(&wi->p1,c1);
      DrawLine(c0[0], c0[1], c1[0], c1[1]);
      if (structplot==SPtags && !quick && wi->itg>0 && wi->dispnr) {
         sprintf(s,"%i",wi->itg);
         if (ascending(c0,c1))
            DrawStringUL((c0[0]+c1[0])/2+2,(c0[1]+c1[1])/2,s);
         else 
            DrawStringLL((c0[0]+c1[0])/2+2,(c0[1]+c1[1])/2,s);
      }
   }
}


void draw_loads(void)                 /* (re)draw loaded wires */
{
   Load *lo;
   double c0[2],c1[2],c2[2],c3[2];

   if (numloads==0) return;
   SetForeground(&c_load);
   SetLineAttributes(3, LineSolid, CapRound, JoinRound);
   for (lo=loads;lo<loads+numloads;lo++) {
      proj(&lo->wire->p0,c0);
      proj(&lo->wire->p1,c1);
      if (lo->wire->ns>0) {
         interpolate(c2,(double)(lo->firstseg-1)/(lo->wire->ns),c0,c1);
         interpolate(c3,(double)lo->lastseg/(lo->wire->ns),c0,c1);
         DrawLine(c2[0], c2[1], c3[0], c3[1]);
      } else {
         DrawLine(c0[0], c0[1], c1[0], c1[1]);
      }
   }
}

void draw_excis(void)                 /* (re)draw wires with excitations */
{
   Exci *ex;
   double c0[2],c1[2],c2[2],c3[2];

   if (numexcis==0) return;
   SetForeground(&c_exci);
   SetLineAttributes(4, LineSolid, CapRound, JoinRound);
   for (ex=excis;ex<excis+numexcis;ex++) {
      proj(&ex->wire->p0,c0);
      proj(&ex->wire->p1,c1);
      if (ex->wire->ns>0) {
         interpolate(c2,(double)(ex->seg-1)/(ex->wire->ns),c0,c1);
         interpolate(c3,(double)ex->seg/(ex->wire->ns),c0,c1);
         DrawLine(c2[0], c2[1], c3[0], c3[1]);
      } else {
         DrawLine(c0[0], c0[1], c1[0], c1[1]);
      }
   }
}


void draw_surfaces(void)
{
   Surface *su;
   double c0[2],c1[2],c2[2],c3[2];
   double ca[2],cc[2];

   if (numsurfaces==0) return;
   SetLineAttributes(0, LineSolid, CapRound, JoinRound);
   for (su=surfaces;su<surfaces+numsurfaces;su++) {
      if ( eyevec.x * (su->pa.x-su->pc.x) + eyevec.y * (su->pa.y-su->pc.y) + eyevec.z * (su->pa.z-su->pc.z) >= 0 )
         SetForeground(&c_surf);
      else
         SetForeground(&c_back);
      switch (su->type) {
         case SU_rect:
         case SU_arb:
            proj(&su->pa,ca);
            proj(&su->pc,cc);
            proj(&su->p0,c0);
            proj(&su->p1,c1);
            c2[0]=2*cc[0]-c0[0]; c2[1]=2*cc[1]-c0[1];
            c3[0]=2*cc[0]-c1[0]; c3[1]=2*cc[1]-c1[1];
            if (su->type!=SU_arb) {
               /* draw border: */
               DrawLine(c0[0], c0[1], c1[0], c1[1]);
               DrawLine(c1[0], c1[1], c2[0], c2[1]);
               DrawLine(c2[0], c2[1], c3[0], c3[1]);
               DrawLine(c3[0], c3[1], c0[0], c0[1]);
            }
            /* draw diagonals: */
            DrawLine(c0[0], c0[1], c2[0], c2[1]);
            DrawLine(c1[0], c1[1], c3[0], c3[1]);
            if (su->type==SU_arb) {
               /* draw additional diagonals */
               DrawLine(0.7071*(c0[0]+c1[0])-0.4142*cc[0], 0.7071*(c0[1]+c1[1])-0.4142*cc[1], 0.7071*(c2[0]+c3[0])-0.4142*cc[0], 0.7071*(c2[1]+c3[1])-0.4142*cc[1]);
               DrawLine(0.7071*(c0[0]+c3[0])-0.4142*cc[0], 0.7071*(c0[1]+c3[1])-0.4142*cc[1], 0.7071*(c2[0]+c1[0])-0.4142*cc[0], 0.7071*(c2[1]+c1[1])-0.4142*cc[1]);
            }
            break;
      }
   }

}



void draw_antenna(int ie)              /* draw the entire antenna */
{
   draw_wires();
   draw_loads();
   draw_excis();
   if (Pending() && ie) return;
   draw_surfaces();
}


void draw_gain_theta(int i)
{
   int j;
   double c0[2],c1[2];

   proj(&gpo[i][0],c1);
   for (j=1;j<numphi;j++) {
      c0[0]=c1[0]; c0[1]=c1[1];
      proj(&gpo[i][j],c1);
      DrawLine(c0[0], c0[1], c1[0], c1[1]);
   }
   proj(&gpo[i][0],c0);
   DrawLine(c0[0], c0[1], c1[0], c1[1]);
}

void draw_gain_phi(int j)
{
   int i;
   double c0[2],c1[2];
   proj(&gpo[0][j],c0);
   for (i=1;i<numtheta;i++) {
      proj(&gpo[i][j],c1);
      DrawLine(c0[0], c0[1], c1[0], c1[1]);
      c0[0]=c1[0]; c0[1]=c1[1];
   }
}


#define R90 (M_PI/2)
#define R180 M_PI
#define R270 (1.5*M_PI)
#define R360 (2*M_PI)

void draw_gain(int ie)
{
   int i,j;

   SetForeground(&c_gain);
   SetLineAttributes(0, LineSolid, CapRound, JoinRound);

   if (gainplot==GP3d && !quick) {

      for (i=0;i<numtheta;i++) draw_gain_theta(i);
      if (Pending() && ie) return;
      for (j=0;j<numphi;j++) draw_gain_phi(j);

   } else {

      if (!scaleplot || theta!=90)      /* suppress if scaleplotting and XY-plane is perpendicular to screen */
         for (i=0;i<numtheta;i++)
            if (fabs(gtheta[i]-R90)<0.001)     /* points in XY-plane? */
               draw_gain_theta(i);

      for (j=0;j<numphi;j++)
         if ((
               ( (fabs((gphi[j]+R90)-R90)<0.001 || fabs((gphi[j])-R180)<0.001)  /* points in XZ-plane? */
                  && (!scaleplot || (phi!=0 && theta!=0)) )  /* suppress if that plane is perpendicular to screen and we're plotting the scale */
            ) || (
               ( fabs((gphi[j])-R90)<0.001 || fabs((gphi[j])-R270)<0.001)       /* points in YZ-plane? */
                  && (!scaleplot || (phi!=90 && theta!=0) )  /* suppress if that plane is perpendicular to screen and we're plotting the scale */
            ))
            draw_gain_phi(j);    

   }
}


void draw_gainscale(int ie)
{
   double a;
   Point po;
   double c0[2],c1[2];
#define Npt 90
#define da R360/Npt
   static double si[Npt],co[Npt];
   static firsttime=1;
   static double radii[numGS][6]=     /* this array determines the radius of all gain scale circles; note: if you change this, also update the labels in GSnumbers[][] */
           {
             {    1,     0.794328,    0.501187,    0.251189,  0.1,       0   },    /* lin:  0, -1, -3, -6, -10 dB */
             {    1,     0.839624,    0.558406,    0.311817,    0.174121,  0   },    /* arrl: 0, -3, -10, -20, -30 dB */
             {    1,     0.75,        0.5,         0.25,      0   }     /* log:  0, -10, -20, -30 dB */
           } ;
   int i;
   double r;
   double *rp;

   if (!scaleplot) return;

   SetForeground(&c_scale);
   SetLineAttributes(0, LineSolid, CapRound, JoinRound);

   if (firsttime) {
      for (i=Npt-1, a=0; i>=0; i--, a+=da) {
         si[i]=sin(a);
         co[i]=cos(a);
      }
      firsttime=0;
   }

   if (phi!=90 && theta!=0) {
      rp=radii[gainscale];
      while ((r=*rp++*extr)) {
         po.x=0; po.y=r; po.z=0;
         proj(&po,c0);
         for (i=0;i<Npt;i++) {
            po.y=r*co[i];
            po.z=r*si[i];
            proj(&po,c1);
            DrawLine(c0[0], c0[1], c1[0], c1[1]);
            c0[0]=c1[0]; c0[1]=c1[1];
         }
      }
      if (Pending() && ie) return;
   }

   if (theta!=90) {
      rp=radii[gainscale];
      while ((r=*rp++*extr)) {
         po.x=r; po.y=0; po.z=0;
         proj(&po,c0);
         for (i=0;i<Npt;i++) {
            po.x=r*co[i];
            po.y=r*si[i];
            proj(&po,c1);
            DrawLine(c0[0], c0[1], c1[0], c1[1]);
            c0[0]=c1[0]; c0[1]=c1[1];
         }
      }
      if (Pending() && ie) return;
   }
   
   if (phi!=0 && theta!=0) {
      rp=radii[gainscale];
      while ((r=*rp++*extr)) {
         po.x=r; po.y=0; po.z=0;
         proj(&po,c0);
         for (i=0;i<Npt;i++) {
            po.x=r*co[i];
            po.z=r*si[i];
            proj(&po,c1);
            DrawLine(c0[0], c0[1], c1[0], c1[1]);
            c0[0]=c1[0]; c0[1]=c1[1];
         }
      }
   }
}


void draw_all(int ie)
{
   if (Pending() && ie) return;
   ClearWindow();
   draw_axes();
   if (Pending() && ie) return;
   if (structplot) draw_antenna(ie);
   if (Pending() && ie) return;
   if (gainplot!=GPnone) {
      draw_gainscale(ie);
      draw_gain(ie);
   }
}


