'
'  This program inputs data from a complex image file and displays the
' image.  The program can deal with two file formats.  The first is
' referred to here as the DOS/PC format and is a binary file.  It is as
' follows.
'
'  First two bytes:  NX, number of columns (size of "down range" in
'                    pixels)--INTEGER.
'
'  3rd and 4th bytes:  NY, number of rows (size of "cross range" in
'                      pixels)--INTEGER.
'
'  Bytes 5 - 8:  floating point sample interval between pixels along a
'                row.
'
'  Bytes 9 - 12:  floating point sample interval between pixels in a
'                 column.
'
'  Bytes 13 - 16:  floating point range zero-pad factor = NX / NX' - 1,
'                  where NX' = FFT size before zero-padding.
'
'  Bytes 17 - 20:  floating point cross range zero-pad factor =
'                  NY / NY' - 1, where NY' = FFT size before zero-padding.
'
'  Byte 21:  character = "L" for linear SAR and "C" for circular SAR.
'
'  Bytes 22 - 25:  floating point minimum range wave number (radians/m).
'
'  Bytes 26 - 29:  floating point maximum range wave number.
'
'  Bytes 30 - 33:  floating point minimum cross range wave number
'                  (radians/m).
'
'  Bytes 34 - 37:  floating point maximum cross range wave number.
'
'  Byte 38:  one-byte character whose ascii value is 0 for a slant plane
'            image and 1 for a ground plane image.
'
'  Byte 39:  one-byte character whose ascii value is 0 for LOS polar
'            interpolation and 1 for stabilized scene polar interpolation.
'
'  Bytes 40 - 43:  floating point minimum SAR frequency (Hz).
'
'  Bytes 44 - 47:  floating point maximum SAR frequency.
'
'  Bytes 48 - 49:  number of SAR frequency samples--INTEGER.
'
'  Bytes 50 - 53:  floating point minimum SAR azimuth angle (degrees).
'
'  Bytes 54 - 57:  floating point maximum SAR azimuth angle.
'
'  Bytes 58 - 59:  number of SAR azimuth angles--INTEGER.
'
'  Bytes 60 - 63:  floating point slant plane "tan(psi)" parameter (psi =
'                  central/stabilized scene depression angle).
'
'  Bytes 64 - 67:  floating point slant plane "tan(eta)" parameter (eta =
'                  "slope (or tilt) angle" = 0 for stabilized scene
'                  images).
'
'  Byte 68:  integer between 0 and 14 (or -2 (aliased as 254)) giving
'            spectral window type.
'
'  Bytes 69 - 72:  Taylor window sidelobe level (dB) if byte 68 = 13 or
'                  14, or four dummy bytes, otherwise.
'
'  Successive bytes:  row-by-row, the complex image data.  Each complex
'                     image value uses 8 bytes.  The first 4 bytes give
'                     the floating point real part and the next 4 bytes
'                     give the floating point imaginary part.  (A row of
'                     data corresponds to a sequence of range data for a
'                     given cross range.  The first row of data in the
'                     file will be at the bottom of the displayed image.)
'
'  In the other format, the image is generally created on a UNIX platform.
' The image data itself is also stored in a binary file.  This file must
' have the extension ".CI".  (If it does not, this program takes the
' format to be as described above.)  In addition to the ci file, the
' "standard data file" with the same name but ".SD" extension must exist
' (in the same disk/directory as the ci file).  This is a text file.  The
' details of these files can be found elsewhere (or by inspection of the
' code below--not all of the sd data is used here).  The format of the
' displayed image is the same for both file types; range is to the right
' and cross range is upwards.
'
'  The program beeps when the image is done being painted on the screen.
' (It takes a while.)  Once the image is displayed, you have several
' options.  The displayed pixels represent attributes between 0 and 255.
' Pressing a sequence of 3 numerical digits in that range causes the
' image to be redrawn showing only pixels with attributes higher than that
' number.  Normally, you do NOT press ENTER after keying in the three
' digits.  However, if you press less than three digits (to specify a
' threshold < 100 or < 10), you have to press ENTER to terminate the
' input sequence.
'
'  If you press S, the image is saved to the file IMAGE.GET in QB's "BSAVE
' format."
'
'  If a rodent is not installed, those are your only options.  Pressing
' any other keys terminates the program and clears the screen.  If a
' rodent *is* installed, you have a few other keystroke-activated options
' available.  Pressing T gives you another way of setting the display
' threshold.  A rodent cursor appears.  Just move it to the pixel that
' represents the minimum attribute you want to display and press any
' rodent button.
'
'  Other rodent options are invoked when you press R or C.  These allow
' you to output complex image sequences.  Move the cursor to the row or
' column that you want to output the sequence for and press a rodent
' button.  If you initially pressed R, a row of complex data is output to
' COMPLEX.ROW.  If you pressed C, a column of complex data is output to
' COMPLEX.COL.  The format of both files is the same.  The first line in
' the file gives the number of data points to follow and the sample
' interval between points, and then comes the real and imaginary parts of
' the complex data.  (The first point in COMPLEX.ROW is at the left of the
' image and the first point in COMPLEX.COL is at the image's bottom.
' Also, the first line in the file may (if FSIGN = 1--see below) contain
' only the sample interval.  Either way, the first line in the file is
' *not* data.)
'
'  Pressing E puts you in a mode where, while holding any active rodent
' button down, you can drag the cursor around and draw a rectangle.  When
' you release the button, the image data within that rectangle gets
' written (i.e., extracted) to the file PORTION.BIN if you're using the
' DOS/PC format, or PORTION.CI (and PORTION.SD) if you're using the ci/sd
' format.  VCI beeps at you when it's done writing the data.  (If you're
' displaying a ci/sd image and you want it extracted/saved in the DOS
' format, SET the DOS environment variable CICONVERT to DOS before running
' the program.)  If you use a rodent button other than the left one, VCI
' will extract a region of the image that is a power of 2 in size.
'
'  Pressing Z does the same thing as E except that, after PORTION.BIN/
' PORTION.CI/PORTION.SD are output, the extracted image is displayed,
' replacing the original one.
'
'  Press M to go into "distance measuring mode."  A rodent cursor appears
' with an instuction asking for "Position 1".  Move the cursor to the
' first point you're interested in and left click.  You now get a request
' for "Position 2".  Move the cursor the second point and left click.  The
' distance between the two points is then displayed at the top.  You can
' repeat this process if you wish.  (The "Position 1" request isn't
' redisplayed, however.)  Right click to exit this mode.  Except for
' possibly F (see below), O, and A, these now represent the complete set of
' key-strokes available that will not terminate the program.
'
'  Press L to turn logarithmic pixel display on and off.  (This keystroke
' will reset the Ctrl-L switch, below, to "nonlinear pixelization.")
'
'  Normally, floating point intensity values are pixelized in a manner
' that doesn't count zero intensities when compressing them to a byte
' value.  This causes very dim intensities to be displayed better.
' Pressing Ctrl-L causes the image to be redisplayed with zero intensity
' values counted in the pixelization, suppressing dim intensities.  (This
' keystroke will reset a logarithmic pixel display to linear.)  Ctrl-L is
' a toggle.  Pressing it again causes the "nonlinear pixelization" to
' again be used.
'
'  Pressing A puts you in a rather bizarre-sounding mode.  It again brings
' up a rodent cursor with which you can draw boxes around various areas of
' the screen, much like if you were extracting the area to PORTION.BIN.
' However, no data is written to a file in this case.  Rather, the box is
' used here merely to identify certain areas of the image.  The purpose of
' selecting these areas is to have VCI scale the complex image (and pixel)
' data in those selected areas.  The scaling is done according to old and
' new values of the angle (with respect to some feature "normal vector")
' at which an incident field is imagined to be incident on the imaged
' objects.  (Presumably, there's something *in* the areas you've
' selected.  The scaling also uses the central SAR frequency and charac-
' teristic size of the object being "transformed.")  In this mode, it
' generally *does* matter which rodent button you use.  You generally want
' to use the left button.  Once you select one area, the rodent cursor
' reappears so you can select another one.  There are two ways of termi-
' nating the selection process.  One is to use the left button for all
' but the last area selected and then use the right button (or middle one
' if your driver uses it) for the last area.  VCI detects which button
' you're using and terminates the selection process when you're done using
' the right or middle buttons.  (So, if you only want to select one area,
' it's okay to use a button other than the left one.)  Another way of
' terminating the selection process is to, once the cursor is displayed,
' just press ENTER, and then press and release the *LEFT* button.  (Do not
' use the right or middle buttons if you are going to use ENTER to termi-
' nate the selection process--that will cause the last area selected to be
' ignored.  Also do not use this method if you're on your first area.)
' There is still a third way of terminating the selection process, and it
' works just like the "ENTER method" except that you press ESCape instead
' of ENTER.  However, it's not just a method of terminating the selection
' process; using ESC aborts the whole scaling operation.  The selection
' process will also terminate automatically and proceed with the scaling
' if you try to select more than 50 areas.  Oh, maybe I should mention
' where VCI gets the angles and other information it needs.  Except for
' the SAR center frequency, which it gets from the input file, it prompts
' you for them once the selection process ends (unless you used ESC to
' end it).  The angles must be input in degrees and the size in meters.
' It first asks you for the old angle, then it asks you for the new one,
' and then it asks you for the other two parameters.  (Be careful typing
' in the answers to the prompts.  Don't try to continue your input onto
' the next line and don't make a mistake and press ENTER before correcting
' it.  The latter will invoke Basic's "Redo from start error", which like
' continuing input onto the next line, will mess up the displayed image.)
' Once you input the new angle, the image data is rescaled and redrawn.
' (The digital pixelization is recalculated anew from the rescaled complex
' data.  Also, once the image is redrawn, you are back in VCI's normal
' display mode, where the normal keystroke/rodent functionallity is
' available.  While in the aspect angle scaling mode, however, keys other
' than ESC and ENTER will be ignored.  These keystrokes, along with using
' the right rodent button, are the only things that will get you back to
' the normal display mode, or put you on a path leading there.)
'
'  If you press O, you may be put in a mode in which you can again use
' the rodent to draw a box around a region of the image that you wish to
' orthorectify (correct for layover).  This process is somewhat slow (even
' in comparison to the slowness of the other options).  After selecting
' the image area to be orthorectified, you will be asked for the height
' that caused the layover.  Only one region of the image can be ortho-
' rectified at a time and it doesn't matter which (supported) rodent
' button you use.  (The height mentioned previously will apply to this
' entire area.)
'
'  Pressing D opens a DOS shell in case you want to temporarily do
' something from the DOS command line (such as rename a file created by
' using the "E" or "Z" options).  Typing EXIT returns you to the image
' displayer.
'
'  If you plan on using the orthorectification option and your image is
' from a "real SAR" (or any other time you need to use a positive sign in
' the exponent of the "fourier kernel"), before running VCI, you NEED to
' make a DOS environment setting.  That setting is:
'
'   SET FSIGN=1
'
'  The computer will beep in the O and A options after the mathematical
' processing is finished.  (This occurs just before redrawing the image
' begins (except for X) and is not to be confused with the beep when the
' the image painting is finished.)
'
'  There are some DOS environment variables that you can set before
' running the program to invoke other options.  Set CENTROID to "ON" if
' you want VCI to draw axes showing the location of the attribute-based
' centroid.  Set it to "POWER" if you want the centroid location based on
' the square of the pixel attributes.
'
'  If you set PALETTE to the name of a file, VCI will output the palette
' data it uses to that file.
'
'  If the DOS environment variable FACETMODEL is set to the name
' (including the path) of a SURFMODL "facet file" or an AWAS/NEC wire grid
' file, this program will also display the 2-D projection of that 3-D
' structure on top of the displayed pixelized image.  To use a SURFMODL
' file, the MODELTYPE environment variable must be set to "SRF" and there
' must not be more than 3 nodes per facet/individual surface.  (There
' would never be less than 3.)  To use a NEC file, MODELTYPE must be set
' to "NEC", and it must be set to "AWAS" to use an AWAS file.  (If
' MODELTYPE isn't NEC or AWAS, it will be taken to be SRF.)  Set the DOS
' variable MODELCOL to the color (1 - 255) to be used to display the
' model.  (If MODELCOL isn't set, the color will be the same as used for
' the rodent.)  If you're using this option, pressing F acts as a toggle
' to turn display of the model on and off.  (Also, the facet/wire grid
' file must not contain any comment statements.)
'
'  Okay, if you haven't figured it out yet:  I have a hard time not adding
' options.  The program now has the additional option of allowing you to
' specify the name of a second image file on the command line.  If this
' option is used, both images are individually pixelized and a combination
' of them is displayed.  How they are combined is determined by the third
' parameter that you must also specify on the command line.  This third
' parameter is called an "action verb."  (Well, actually, it's translated
' to an action verb inside the program.)  The third parameter specified as
' "COMMON" (no quotes) means that only the position-respective pixels that
' are the same in each image are displayed (the images are ANDed
' together).  If you specify "DIF" for the action verb, only the
' differences between the two images are shown (the images are XORed
' together).  If you say "BOTH", the images are combined in OR fashion.
' If this option is to be used, NX and NY must be the same for both
' images, and for the PC/DOS format, the pixel sample intervals must be
' the same.  For the UNIX format, the DataType parameter must be the same
' for both images.  (Execution terminates if these conditions are not
' met.)  Also, although it's not explicit checked for, both images must be
' in the same format.  (A PC/DOS image can't be compared with a UNIX
' image.)  Also, when a second file name is specified, most of the key-
' stroke/rodent functionallity is turned off.  (Pressing "S" to save and
' "M" to measure a distance still work, as does the thresholding option,
' with or without the rodent.)  Finally, if you don't specify the third
' parameter on the command line, OR is assumed.
'
'  Define register variables and functions.
'
DECLARE FUNCTION BESSEL#(N AS LONG,X AS DOUBLE)
DECLARE FUNCTION FACT#(N AS LONG)
DECLARE FUNCTION SINC(X)
'$INCLUDE: 'QBVAUX.INC'
DIM NX AS INTEGER,NY AS INTEGER,INTENSITY AS INTEGER,FDISP AS INTEGER
DIM NXOUT AS INTEGER,NYOUT AS INTEGER,CLA AS LONG,IPAL AS INTEGER,PI AS DOUBLE
DIM COMROW AS INTEGER,COMCOL AS INTEGER,POWER AS INTEGER,RODENT AS INTEGER
DIM DKX AS DOUBLE,DKY AS DOUBLE,BYTE1 AS STRING*1,ISP AS INTEGER,PHIC AS DOUBLE
DIM BYTE2 AS STRING*1,BYTE3 AS STRING*1,BYTE4 AS STRING*1,XLA(1 TO 50)
DIM YLA(1 TO 50),XRA(1 TO 50),YRA(1 TO 50),AREA AS INTEGER,K AS INTEGER
DIM ARGO AS DOUBLE,ARGN AS DOUBLE,OSF AS DOUBLE,NSF AS DOUBLE,NFREQ AS INTEGER
DIM NAZA AS INTEGER,STRVAR AS STRING*1,SARTYPE AS STRING*1,INTPOL AS INTEGER
DIM OFLAG AS INTEGER,FSIGN AS INTEGER,IMAGEADD AS LONG,DT AS DOUBLE
DIM NXBP AS DOUBLE,NYBP AS DOUBLE,NXB AS DOUBLE,NYB AS DOUBLE,KX AS DOUBLE
DIM KY AS DOUBLE,KS AS DOUBLE,INV AS INTEGER,RCOR AS DOUBLE,ICOR AS DOUBLE
DIM S4 AS STRING*4,TYPEFLAG AS INTEGER,NX1 AS INTEGER,NY1 AS INTEGER
DIM BYTES AS LONG,IBYTES AS LONG,DUMCALL AS LONG,HANDLE AS INTEGER
DIM HANDLE1 AS INTEGER,NEWLOC AS LONG,VMHANDLE AS INTEGER,SOURCE AS LONG
DIM DXX AS SINGLE,DY AS SINGLE,KXMIN AS SINGLE,KXMAX AS SINGLE,KYMIN AS SINGLE
DIM KYMAX AS SINGLE,BFR AS SINGLE,EFR AS SINGLE,BAZ AS SINGLE,EAZ AS SINGLE
DIM TANPSI AS SINGLE,TANETA AS SINGLE,RNGPAD AS SINGLE,CRGPAD AS SINGLE
DIM IBASE AS INTEGER,DX1 AS SINGLE,DY1 AS SINGLE,PHIR AS DOUBLE,LOGSW AS INTEGER
DIM SMDATA AS INTEGER,OSDATA AS INTEGER,SMDATA1 AS INTEGER,OSDATA1 AS INTEGER
DIM IWIN AS INTEGER,SLL AS SINGLE,RNBAR AS INTEGER,CNBAR AS INTEGER
DIM SLLREAL AS DOUBLE,TEMPVAR AS DOUBLE,VMHANDLE1 AS INTEGER,SIZE AS LONG
DIM ZOOM AS INTEGER,FER AS INTEGER,NXTEMP AS INTEGER,NYTEMP AS INTEGER
DIM NXBI AS INTEGER,NYBI AS INTEGER,NXNEW AS INTEGER,NYNEW AS INTEGER
DIM EFLAG AS INTEGER,SUMR AS DOUBLE,SUMI AS DOUBLE,NXNEWO AS INTEGER
DIM PIXELSW AS INTEGER
'
'  PI is needed for various things.
'
PI=4#*ATN(1#)
'
'  Set facet/grid model display and "zoom" switches.  (The latter is to
' turn off facet model display if extracted portion of image is to be
' redisplayed.)
'
FDISP=1 : ZOOM=0
'
'  Warn user if rodent isn't installed.
'
ANS$="Y"
IF QRYMOUSE<>-1 THEN
PRINT "You aren't using a rodent.  Do you want to continue (Y/N)?"
5 ANS$=INKEY$ : IF ANS$="" THEN GOTO 5
ANS$=UCASE$(ANS$)
RODENT=0
ELSE
RODENT=1
END IF
IF ANS$="N" THEN GOTO QUITEXEC
'
'  Store allowed screen resolutions and ask user which of those is the
' maximum he/she has support for.
'
DIM MAXRES(1 TO 6) AS INTEGER,MAXIND AS INTEGER,MAXMODE AS INTEGER
MAXRES(1)=22 : MAXRES(2)=20 : MAXRES(3)=18 : MAXRES(4)=16 : MAXRES(5)=14
MAXRES(6)=13
CLS
PRINT "  Which of the following resolutions is the highest your system can ";
PRINT "handle"
PRINT "with 256 colors?  (Don't press ENTER.)"
PRINT
PRINT "1 - 1600 x 1200"
PRINT
PRINT "2 - 1280 x 1024"
PRINT
PRINT "3 - 1024 x 768"
PRINT
PRINT "4 - 800 x 600"
PRINT
PRINT "5 - 640 x 480"
PRINT
PRINT "6 - 320 x 200"
PRINT
PRINT "0 - No 256-color modes available."
PRINT
10 M$=INKEY$ : IF M$="" THEN GOTO 10
MAXIND=VAL(M$)
IF MAXIND<0 OR MAXIND>6 THEN GOTO 10
PRINT MAXIND
IF MAXIND=0 THEN GOTO QUITEXEC
MAXMODE=MAXRES(MAXIND)
'
'  Get sign of exponent in fourier kernel.  (It is used in the orthorecti-
' fication option and in identifying the type of SAR in the UNIX/ci case.)
'
FSIGN=SGN(VAL(LTRIM$(ENVIRON$("FSIGN")))) : IF FSIGN=0 THEN FSIGN=-1
'
'  Set initial value for switch determining how intensities are pixelized.
'
PIXELSW=0
'
'  Set initial value for image scale switch.
'
LOGSW=0
'
'  Find out if "E" extraction option is to save UNIX/ci images in the DOS
' format.
'
CICON$=UCASE$(ENVIRON$("CICONVERT"))
'
'  Get input file(s) and input image(s).
'
CALL PARSE(COMMAND$," ",F$,S1$)
F$=RTRIM$(F$)
IF F$="" THEN
F$="IMAGE.BIN"
F1$=""
ELSE
S1$=LTRIM$(S1$)
CALL PARSE(S1$," ",F1$,S2$)
F1$=RTRIM$(F1$) : S2$=LTRIM$(S2$)
CALL PARSE(S2$," ",ACT$,S1$)
ACT$=UCASE$(RTRIM$(ACT$))
'
'  Define ACT$ in terms of its intrinsic QB meaning and define default.
'
IF ACT$="BOTH" THEN ACT$="OR"
IF ACT$="COMMON" THEN ACT$="AND"
IF ACT$="DIF" THEN ACT$="XOR"
IF ACT$<>"AND" AND ACT$<>"OR" AND ACT$<>"XOR" THEN ACT$="OR"
IF EXIST(F1$)<>1 THEN F1$=""
END IF
IF EXIST(F$)<>1 THEN GOTO 60
'
'  First, find out if image is stored in DOS/PC format or UNIX format.
' (Just use first file specified and assume that second file, if
' specified, is in same format.)
'
REDISPLAY:
CALL PARSE(F$,".",SDFILE$,EXT$)
EXT$=UCASE$(LTRIM$(EXT$))
IF EXT$="SD" THEN EXT$="CI"
IF EXT$="CI" THEN
F$=RTRIM$(SDFILE$)+".CI"
SDFILE$=RTRIM$(SDFILE$)+".SD"
END IF
'
'  Get second support data file name if appropriate.
'
IF F1$<>"" AND EXT$="CI" THEN
CALL PARSE(F1$,".",SDFILE1$,S1$)
F1$=RTRIM$(SDFILE1$)+".CI"
SDFILE1$=RTRIM$(SDFILE1$)+".SD"
END IF
'
'  Size/spatial information is stored in the binary file for the DOS/PC
' format.  Otherwise, it's in the sd text file.
'
IF EXT$<>"CI" THEN
CALL FOPEN(F$,HANDLE,2)
IF HANDLE=0 THEN GOTO QUITEXEC
CALL FREAD(HANDLE,2&,VARSEG(NX),VARPTR(NX))
CALL FREAD(HANDLE,2&,VARSEG(NY),VARPTR(NY))
CALL FREAD(HANDLE,4&,VARSEG(DXX),VARPTR(DXX))
CALL FREAD(HANDLE,4&,VARSEG(DY),VARPTR(DY))
CALL FREAD(HANDLE,4&,VARSEG(RNGPAD),VARPTR(RNGPAD))
CALL FREAD(HANDLE,4&,VARSEG(CRGPAD),VARPTR(CRGPAD))
CALL FREAD(HANDLE,1&,VARSEG(SARTYPE),VARPTR(SARTYPE))
'
'  If this step is really necessary, the SAR data file is questionable;
' proceed anyway.
'
SARTYPE=UCASE$(SARTYPE) : IF SARTYPE<>"C" THEN SARTYPE="L"
CALL FREAD(HANDLE,4&,VARSEG(KXMIN),VARPTR(KXMIN))
CALL FREAD(HANDLE,4&,VARSEG(KXMAX),VARPTR(KXMAX))
CALL FREAD(HANDLE,4&,VARSEG(KYMIN),VARPTR(KYMIN))
CALL FREAD(HANDLE,4&,VARSEG(KYMAX),VARPTR(KYMAX))
CALL FREAD(HANDLE,1&,VARSEG(STRVAR),VARPTR(STRVAR))
ISP=ASC(STRVAR)
CALL FREAD(HANDLE,1&,VARSEG(STRVAR),VARPTR(STRVAR))
INTPOL=ASC(STRVAR)
CALL FREAD(HANDLE,4&,VARSEG(BFR),VARPTR(BFR))
CALL FREAD(HANDLE,4&,VARSEG(EFR),VARPTR(EFR))
SARF=(BFR+EFR)/2
CALL FREAD(HANDLE,2&,VARSEG(NFREQ),VARPTR(NFREQ))
CALL FREAD(HANDLE,4&,VARSEG(BAZ),VARPTR(BAZ))
BAZ0=BAZ
CALL FREAD(HANDLE,4&,VARSEG(EAZ),VARPTR(EAZ))
EAZ0=EAZ
CALL FREAD(HANDLE,2&,VARSEG(NAZA),VARPTR(NAZA))
CALL FREAD(HANDLE,4&,VARSEG(TANPSI),VARPTR(TANPSI))
CALL FREAD(HANDLE,4&,VARSEG(TANETA),VARPTR(TANETA))
CALL FREAD(HANDLE,1&,VARSEG(STRVAR),VARPTR(STRVAR))
IWIN=ASC(STRVAR) : IF IWIN>127 THEN IWIN=IWIN-256
CALL FREAD(HANDLE,4&,VARSEG(SLL),VARPTR(SLL))
IF IWIN=13 OR IWIN=14 THEN
RSLL=SLL : CSLL=SLL
SLLREAL=10#^(SLL/20) : TEMPVAR=LOG(SLLREAL+SQR(SLLREAL^2-1#))/PI
RNBAR=3*INT(2#*TEMPVAR^2+1.000001#) : CNBAR=RNBAR
END IF
'
'  If second file was specified (and exists), input data from it and make
' sure it's at least minimally compatible with first file.
'
IF F1$<>"" THEN
CALL FOPEN(F1$,HANDLE1,2)
IF HANDLE1=0 THEN GOTO QUITEXEC
CALL FREAD(HANDLE1,2&,VARSEG(NX1),VARPTR(NX1))
CALL FREAD(HANDLE1,2&,VARSEG(NY1),VARPTR(NY1))
CALL FREAD(HANDLE1,4&,VARSEG(DX1),VARPTR(DX1))
CALL FREAD(HANDLE1,4&,VARSEG(DY1),VARPTR(DY1))
IF NX1<>NX OR NY1<>NY THEN
CALL FCLOSE(HANDLE)
CALL FCLOSE(HANDLE1)
GOTO QUITEXEC
END IF
IF 2*ABS(DX1-DXX)/(DX1+DXX)>=.001 OR 2*ABS(DY1-DY)/(DY1+DY)>=.001 THEN
CALL FCLOSE(HANDLE)
CALL FCLOSE(HANDLE1)
GOTO QUITEXEC
END IF
FOR I=1 TO 60
CALL FREAD(HANDLE1,1&,VARSEG(BYTE1),VARPTR(BYTE1))
NEXT I
END IF
ELSE
OPEN SDFILE$ FOR INPUT AS #2
'
'  Set dummy values for now.  (If NX and NY don't both change from 0, the
' program terminates.)
'
NX=0 : NY=0 : DXX=1 : DY=1 : DATATYPE$="ICOMPLEX" : TANPSI=0 : TANETA=0 : ISP=0
SARTYPE="C" : INTPOL=0 : SARF=0 : BFR=0 : BAZ=0 : EAZ=0 : NAZA=0 : NFREQ=0
CRGPAD=0 : RNGPAD=0 : WINTYPE$="UNIFORM" : RSLL=0 : CSLL=0 : RNBAR=0 : CNBAR=0
'
'  If FSIGN = 1, a "real SAR" was most likely involved and they typically
' fly in a straight path.
'
IF FSIGN=1 THEN SARTYPE="L"
WHILE NOT EOF(2)
LINE INPUT#2,D$
CALL PARSE(D$,"=",S1$,S2$)
S1$=RTRIM$(S1$) : S2$=LTRIM$(S2$)
IF S1$="ImageWidth" THEN NY=VAL(S2$)
IF S1$="ImageHeight" THEN NX=VAL(S2$)
'
'  Just in case orthorectification is to be done, convert sample intervals
' to meters.
'
IF S1$="SD_RG" THEN DXX=.3048/VAL(S2$)
IF S1$="SD_CR" THEN DY=.3048/VAL(S2$)
IF S1$="DataType" THEN DATATYPE$=UCASE$(S2$)
IF S1$="ZeroPadFactor_CR" THEN CRGPAD=VAL(S2$)
IF S1$="ZeroPadFactor_RG" THEN RNGPAD=VAL(S2$)
IF S1$="TiltAngle" THEN TANETA=TAN(VAL(S2$)*PI/180)
IF S1$="GrazingAngle" THEN TANPSI=TAN(VAL(S2$)*PI/180)
IF S1$="CenterFrequency" THEN SARF=VAL(S2$)
IF S1$="ImagePlane" THEN IF UCASE$(S2$)<>"SLANT" THEN ISP=1
IF S1$="ApertureWidth" THEN BAZ=-VAL(S2$)/2 : EAZ=-BAZ
IF S1$="StartingFrequency" THEN BFR=VAL(S2$)
IF S1$="Weighting" THEN WINTYPE$=S2$
IF S1$="Wgt_SLL_RG" THEN RSLL=VAL(S2$)
IF S1$="Wgt_SLL_CR" THEN CSLL=VAL(S2$)
IF S1$="Wgt_NBAR_RG" THEN RNBAR=VAL(S2$)
IF S1$="Wgt_NBAR_CR" THEN CNBAR=VAL(S2$)
WEND
CLOSE #2
IF WINTYPE$="TAYLOR" THEN
IF RSLL=0 OR CSLL=0 OR RNBAR=0 OR CNBAR=0 THEN WINTYPE$="UNIFORM"
END IF
IF DATATYPE$="COMPLEX" THEN DATATYPE$="FCOMPLEX"
IF DATATYPE$<>"ICOMPLEX" AND DATATYPE$<>"FCOMPLEX" THEN GOTO BADTYPE
IF NX=<0 OR NY<=0 THEN GOTO NOSIZE
EFR=2*SARF-BFR
COSPSI=1/SQR(TANPSI^2+1)
KMIN=4*PI*BFR/2.9979E8 : KXMAX=4*PI*EFR/2.9979E8
KXMIN=KMIN*COS(BAZ*PI/180) : KYMIN=KXMAX*SIN(BAZ*PI/180)
IF ISP=1 THEN KXMIN=KXMIN*COSPSI : KXMAX=KXMAX*COSPSI : KYMIN=KYMIN*COSPSI
KYMAX=-KYMIN
'
'  Convert slant plane azimuth limits to ground plane values.
'
BAZ0=BAZ : EAZ0=EAZ
IF ISP=0 AND SARTYPE="L" THEN BAZ=180*ATN(TAN(BAZ*PI/180)/COSPSI)/PI : EAZ=-BAZ
'
'  If second file was specified (and exists), get minimal support data for
' it and make sure it's compatible with first file.
'
IF F1$<>"" THEN
OPEN SDFILE1$ FOR INPUT AS #2
NX1=0 : NY1=0 : DATATYPE1$="ICOMPLEX"
WHILE NOT EOF(2)
LINE INPUT#2,D$
CALL PARSE(D$,"=",S1$,S2$)
S1$=RTRIM$(S1$) : S2$=LTRIM$(S2$)
IF S1$="ImageWidth" THEN NY1=VAL(S2$)
IF S1$="ImageHeight" THEN NX1=VAL(S2$)
IF S1$="DataType" THEN DATATYPE1$=UCASE$(S2$)
WEND
CLOSE #2
IF DATATYPE1$="COMPLEX" THEN DATATYPE1$="FCOMPLEX"
IF NX1<>NX OR NY1<>NY OR DATATYPE1$<>DATATYPE$ THEN CLOSE : GOTO QUITEXEC
END IF
END IF
'
'  For slant plane image, convert TANPSI and TANETA to their slant plane
' values.  (Technically, the values themselves don't change for the slant
' plane.  Slant plane images merely change how they're used.  I'm just
' making coding simplifications.  Also, I'm not yet sure how meaningful
' this is for a circular SAR.)  Save initial values for possible later
' output.
'
TANPSI0=TANPSI : TANETA0=TANETA
IF ISP=0 THEN
IF EXT$<>"CI" THEN COSPSI=1/SQR(TANPSI^2+1)
TANPSI=TANPSI*COSPSI : TANETA=TANETA*COSPSI
END IF
DELTAKX=(KXMAX-KXMIN)/2 : K0X=(KXMAX+KXMIN)/2 : DELTAKY=(KYMAX-KYMIN)/2
SARF=SARF*1E-9
'
'  Well, okay.  I digressed a little.  Now input the image data.
'
CALL REDIMVMS("A",2&*CLNG(NX)*CLNG(NY),"SINGLE",VMS)
'
'  If VMS is returned from DIMVMS as nonzero, VMS arrays cannot be used.
'
IF VMS<>0 THEN GOTO QUITEXEC
IF F1$<>"" THEN
CALL DIMVMS("C",2&*CLNG(NX)*CLNG(NY),"SINGLE",VMS)
'
'  If there is insufficient virtual memory to store second image (or
' there's any other problem with virtual memory), just act as if second
' image file wasn't specified.
'
IF VMS<>0 THEN F1$=""
END IF
'
'  Get minimum (nonzero) and maximum image strengths in process of
' inputting data.
'
PMAX=-1
PMIN=1E30
'
'  The following is for minimizing the calls to file I/O routines in the
' DOS/PC-based image loading and in the image displaying for both data
' formats.
'
IBYTES=CLNG(8*NX)
REDIM ILINE(1 TO 2*NX)
SMDATA=VARSEG(ILINE(1)) : OSDATA=VARPTR(ILINE(1))
DUMCALL=UVMS("A",VMHANDLE)
IF F1$<>"" THEN
REDIM ILINE1(1 TO 2*NX)
SMDATA1=VARSEG(ILINE1(1)) : OSDATA1=VARPTR(ILINE1(1))
DUMCALL=UVMS("C",VMHANDLE1)
END IF
ON ERROR GOTO FIXDATA
IF EXT$<>"CI" THEN
'
'  DOS/PC-based image
'
CALL FPOINT(VMHANDLE,0&,NEWLOC)
IF F1$<>"" THEN CALL FPOINT(VMHANDLE1,0&,NEWLOC)
FOR J=1 TO NY
CALL FREAD(HANDLE,IBYTES,SMDATA,OSDATA)
CALL FWRITE(VMHANDLE,IBYTES,SMDATA,OSDATA)
'
'  Get second image data if it exists.
'
IF F1$<>"" THEN
CALL FREAD(HANDLE1,IBYTES,SMDATA1,OSDATA1)
CALL FWRITE(VMHANDLE1,IBYTES,SMDATA1,OSDATA1)
END IF
FOR I=1 TO NX
'
'  Store VMS array name for error correction routines.
'
ARRAY$="A"
IBASE=2*I
FER=0
RVALUE=ILINE(IBASE-1)
FER=1
IVALUE=ILINE(IBASE)
FER=2
MAG=SQR(RVALUE^2+IVALUE^2)
IF MAG>PMAX THEN PMAX=MAG
IF MAG>0 AND MAG<PMIN THEN PMIN=MAG
'
'  Process second image data if it exists.
'
IF F1$<>"" THEN
ARRAY$="C"
FER=0
RVALUE=ILINE1(IBASE-1)
FER=1
IVALUE=ILINE1(IBASE)
FER=2
MAG=SQR(RVALUE^2+IVALUE^2)
IF MAG>PMAX THEN PMAX=MAG
IF MAG>0 AND MAG<PMIN THEN PMIN=MAG
END IF
NEXT I
NEXT J
CALL FCLOSE(HANDLE)
IF F1$<>"" THEN CALL FCLOSE(HANDLE1)
ELSE
'
'  UNIX-based image
'
OPEN F$ FOR BINARY AS #1
IF F1$<>"" THEN OPEN F1$ FOR BINARY AS #3
FOR I=1 TO NX
FOR J=1 TO NY
GET#1,,BYTE1
GET#1,,BYTE2
FER=0
IF DATATYPE$="FCOMPLEX" THEN
GET#1,,BYTE3
GET#1,,BYTE4
RVALUE=CVS(BYTE4+BYTE3+BYTE2+BYTE1)
ELSE
RVALUE=CVI(BYTE2+BYTE1)
END IF
GET#1,,BYTE1
GET#1,,BYTE2
FER=1
IF DATATYPE$="FCOMPLEX" THEN
GET#1,,BYTE3
GET#1,,BYTE4
IVALUE=CVS(BYTE4+BYTE3+BYTE2+BYTE1)
ELSE
IVALUE=CVI(BYTE2+BYTE1)
END IF
FER=2
MAG=SQR(RVALUE^2+IVALUE^2)
IF MAG>PMAX THEN PMAX=MAG
IF MAG>0 AND MAG<PMIN THEN PMIN=MAG
CALL PUTSNG("A",2&*CLNG(I+(J-1)*NX)-1&,RVALUE)
CALL PUTSNG("A",2&*CLNG(I+(J-1)*NX),IVALUE)
'
'  Get second image data if it exists.
'
IF F1$<>"" THEN
GET#3,,BYTE1
GET#3,,BYTE2
FER=0
IF DATATYPE$="FCOMPLEX" THEN
GET#3,,BYTE3
GET#3,,BYTE4
RVALUE=CVS(BYTE4+BYTE3+BYTE2+BYTE1)
ELSE
RVALUE=CVI(BYTE2+BYTE1)
END IF
GET#3,,BYTE1
GET#3,,BYTE2
FER=1
IF DATATYPE$="FCOMPLEX" THEN
GET#3,,BYTE3
GET#3,,BYTE4
IVALUE=CVS(BYTE4+BYTE3+BYTE2+BYTE1)
ELSE
IVALUE=CVI(BYTE2+BYTE1)
END IF
FER=2
MAG=SQR(RVALUE^2+IVALUE^2)
IF MAG>PMAX THEN PMAX=MAG
IF MAG>0 AND MAG<PMIN THEN PMIN=MAG
CALL PUTSNG("C",2&*CLNG(I+(J-1)*NX)-1&,RVALUE)
CALL PUTSNG("C",2&*CLNG(I+(J-1)*NX),IVALUE)
END IF
NEXT J
NEXT I
CLOSE #1
IF F1$<>"" THEN CLOSE #3
END IF
ON ERROR GOTO 0
'
'  Find smallest supported screen resolution that will handle picture.
' (If no such resolution can be found, terminate--it is assumed that the
' machine can support any necessary lesser resolution than the maximum
' supported.)
'
MAXIND=0
IF NX<1599 AND NY<1163 THEN MAXIND=1
IF NX<1279 AND NY<987 THEN MAXIND=2
IF NX<1023 AND NY<731 THEN MAXIND=3
IF NX<799 AND NY<563 THEN MAXIND=4
IF NX<639 AND NY<443 THEN MAXIND=5
IF NX<319 AND NY<163 THEN MAXIND=6
IF MAXIND=0 THEN GOTO QUITEXEC
MODE=MAXRES(MAXIND)
IF MODE>MAXMODE THEN GOTO QUITEXEC
'
'  Get size of array necessary to save displayed image.
'
BYTES=4&+CLNG(NY+2)*INT((CSNG(NX+2)*8+7)/8+.001)
IF MODE>13 THEN
SIZE=CLNG(CSNG(BYTES+1&)/2+.001)
SAVEERROR=1
CALL REDIMVMS("IMAGE",SIZE,"INTEGER",SAVEERROR)
ELSE
SIZE=CLNG(CSNG(BYTES+3&)/4+.001)
REDIM IMAGE(1 TO SIZE) AS LONG
END IF
'
'  In case ENTER was pressed when inputting maximum screen resolution,
' purge keyboard buffer.
'
INREGS.AX=&HC06
INREGS.DX=255
CALL INTERRUPTX(&H21,INREGS,OUTREGS)
'
'  Set default pixel display intensity threshold and get CENTROID environ-
' ment variable.
'
THRESHOLD$="0" : CENTROID$=UCASE$(LTRIM$(RTRIM$(ENVIRON$("CENTROID"))))
POWER=1 : IF CENTROID$="POWER" THEN CENTROID$="ON" : POWER=2
'
'  Input facet/wire grid model data if it is to be used.  If virtual
' memory cannot be used to store *facet* model data, act as if no model
' was specified.  (NEC/AWAS wire grid models do not use virtual memory.
' Skip model display if image being displayed is extracted portion of
' previously displayed image.)
'
MODEL$="" : IF ZOOM=0 THEN MODEL$=ENVIRON$("FACETMODEL")
IF MODEL$="" THEN GOTO SETVIDEO
IF EXIST(MODEL$)<>1 THEN GOTO NOMOD
'
'  Get central azimuth angle and ignore model projection request for
' linear SARs if azimuth angles aren't in first or fourth quadrants.
' (PHIR is used to rotate the model in the case of LOS polar
' interpolation.  PHIC is used to get the layover direction for circular
' SARs and also the rotation angle for such SARs.  It is also used to get
' the rotation angle when TANETA = TANPSI = 0.  There is no accounting of
' the fact that layover is not a point to point effect for circular SARs.)
'
IF SARTYPE="L" AND (BAZ<-90 OR EAZ>90) THEN GOTO NOMOD
PHIC=PI*CDBL(BAZ+EAZ)/360#
IF ABS(TANPSI)>.00001 AND SARTYPE="L" THEN
PHIR=-ATN(CDBL(TANETA)/CDBL(TANPSI))
ELSE
'
'  SAR is either moving in ground plane and rotation angle can't be
' determined from TANETA and TANPSI because they're both zero or else a
' circular SAR is involved and TANETA is undefined.  (Note that for either
' type of SAR, PHIC is zero for EXT$ = "CI"--the support data file doesn't
' contain meaningful information regarding absolute azimuth angles.  The
' fact that PHIR can be meaningfully obtained from TANETA and TANPSI for
' ci images from linear SARs as long as the latter quantity is nonzero is
' why the above formula for PHIR is used instead of the seemingly simpler
' process used here.  In other words, if this section of code gets
' executed, the model isn't likely to be projected correctly for ci
' images.)
'
PHIR=-PHIC : IF INTPOL=1 THEN PHIR=0#
'
'  Although PHIC determines the rotation angle for circular SARs with LOS
' polar interpolation, image layover occurs in the down range direction in
' such cases.  (Other than its use in the special case mentioned above, it
' is not used for linear SARs.)
'
IF SARTYPE="C" AND INTPOL=0 THEN PHIC=0#
END IF
MODTYPE$=UCASE$(ENVIRON$("MODELTYPE"))
IF MODTYPE$<>"NEC" AND MODTYPE$<>"AWAS" THEN MODTYPE$="SRF"
'
'   Set color used to display model.
'
MODCOL=VAL(ENVIRON$("MODELCOL"))
IF MODCOL<1 OR MODCOL>255 THEN MODCOL=MCOLOR
OPEN MODEL$ FOR INPUT AS #1
IF MODTYPE$="SRF" THEN
DIM NVERT AS LONG,NSURF AS LONG,VERT1 AS LONG,VERT2 AS LONG,VERT3 AS LONG
DIM NMATL AS INTEGER,LINESTYLE AS INTEGER,NSIDES AS INTEGER,VERTMAX AS INTEGER
LINE INPUT#1,DUMMY$
LINE INPUT#1,DUMMY$
INPUT#1,NMATL,NVERT,NSURF,VERTMAX,NSIDES
IF VERTMAX=3 THEN
FOR I=1 TO NMATL
LINE INPUT#1,DUMMY$
NEXT I
CALL DIMVMS("XMODEL",NVERT,"INTEGER",ERRORCODE)
IF ERRORCODE<>0 THEN
MODEL$=""
ELSE
CALL DIMVMS("YMODEL",NVERT,"INTEGER",ERRORCODE)
IF ERRORCODE<>0 THEN
MODEL$=""
CALL CLRVMS("XMODEL")
ELSE
CALL DIMVMS("VERT1",NSURF,"LONG",ERRORCODE)
IF ERRORCODE<>0 THEN
MODEL$=""
CALL CLRVMS("XMODEL")
CALL CLRVMS("YMODEL")
ELSE
CALL DIMVMS("VERT2",NSURF,"LONG",ERRORCODE)
IF ERRORCODE<>0 THEN
MODEL$=""
CALL CLRVMS("XMODEL")
CALL CLRVMS("YMODEL")
CALL CLRVMS("VERT1")
ELSE
CALL DIMVMS("VERT3",NSURF,"LONG",ERRORCODE)
IF ERRORCODE<>0 THEN
MODEL$=""
CALL CLRVMS("XMODEL")
CALL CLRVMS("YMODEL")
CALL CLRVMS("VERT1")
CALL CLRVMS("VERT2")
END IF
END IF
END IF
END IF
END IF
ELSE
MODEL$=""
END IF
IF MODEL$<>"" THEN
FOR I=1 TO NVERT
INPUT#1,X,Y,Z
'
'   Project and rotate model, and account for layover.
'
CALL ROTATE(X,Y,PHIR)
IF SARTYPE="L" THEN
X=X+Z*TANPSI
Y=Y-Z*TANETA
ELSE
X=X+Z*COS(PHIC)*TANPSI
Y=Y+Z*SIN(PHIC)*TANPSI
END IF
CALL PUTINT("XMODEL",CLNG(I),CINT(X/DXX+CSNG(NX)/2+X0-.5))
CALL PUTINT("YMODEL",CLNG(I),CINT(NY-(Y/DY+CSNG(NY)/2)+Y0-.5))
NEXT I
FOR I=1 TO NSURF
INPUT#1,DMY,DMY,VERT1,VERT2,VERT3
CALL PUTLNG("VERT1",CLNG(I),VERT1)
CALL PUTLNG("VERT2",CLNG(I),VERT2)
CALL PUTLNG("VERT3",CLNG(I),VERT3)
NEXT I
'
'  Set style of line for SURFMODL files based on number of surfaces.
'
LINESTYLE=&H8888 : IF NSURF>15000& THEN LINESTYLE=&H8080
IF NSURF>30000& THEN LINESTYLE=&H8000
END IF
ELSEIF MODTYPE$="NEC" THEN
DIM X1N(1 TO 1392) AS INTEGER,Y1N(1 TO 1392) AS INTEGER
DIM X2N(1 TO 1392) AS INTEGER,Y2N(1 TO 1392) AS INTEGER,NWIRE AS INTEGER
NWIRE=0
LOOKFORGW:
LINE INPUT#1,NECLINE$
HD$=UCASE$(MID$(NECLINE$,1,2))
IF HD$="GE" THEN GOTO GOTWIRES
IF HD$<>"GW" THEN GOTO LOOKFORGW
NWIRE=NWIRE+1
X1=VAL(MID$(NECLINE$,11,10)) : Y1=VAL(MID$(NECLINE$,21,10))
X2=VAL(MID$(NECLINE$,41,10)) : Y2=VAL(MID$(NECLINE$,51,10))
Z1=VAL(MID$(NECLINE$,31,10)) : Z2=VAL(MID$(NECLINE$,61,10))
'
'   Project and rotate model, and account for layover.
'
CALL ROTATE(X1,Y1,PHIR)
IF SARTYPE="L" THEN
X1=X1+Z1*TANPSI
Y1=Y1-Z1*TANETA
ELSE
X1=X1+Z1*COS(PHIC)*TANPSI
Y1=Y1+Z1*SIN(PHIC)*TANPSI
END IF
'
'  Repeat for second point on wire segment.
'
CALL ROTATE(X2,Y2,PHIR)
IF SARTYPE="L" THEN
X2=X2+Z2*TANPSI
Y2=Y2-Z2*TANETA
ELSE
X2=X2+Z2*COS(PHIC)*TANPSI
Y2=Y2+Z2*SIN(PHIC)*TANPSI
END IF
X1N(NWIRE)=CINT(X1/DXX+CSNG(NX)/2+X0-.5)
X2N(NWIRE)=CINT(X2/DXX+CSNG(NX)/2+X0-.5)
Y1N(NWIRE)=CINT(NY-(Y1/DY+CSNG(NY)/2)+Y0-.5)
Y2N(NWIRE)=CINT(NY-(Y2/DY+CSNG(NY)/2)+Y0-.5)
GOTO LOOKFORGW
GOTWIRES:
ELSE
DIM NODE(1 TO 50,1 TO 2) AS INTEGER,XN(1 TO 60),YN(1 TO 60),NSEG AS INTEGER
DIM NNODES AS INTEGER
FOR I=1 TO 5
LINE INPUT#1,DUMMY$
NEXT I
INPUT#1,NNODES
INPUT#1,NSEG
LINE INPUT#1,DUMMY$
LINE INPUT#1,DUMMY$
FOR I=1 TO NNODES
INPUT#1,XN(I),YN(I),Z
'
'   Project and rotate model, and account for layover.
'
CALL ROTATE(XN(I),YN(I),PHIR)
IF SARTYPE="L" THEN
XN(I)=XN(I)+Z*TANPSI
YN(I)=YN(I)-Z*TANETA
ELSE
XN(I)=XN(I)+Z*COS(PHIC)*TANPSI
YN(I)=YN(I)+Z*SIN(PHIC)*TANPSI
END IF
XN(I)=CINT(XN(I)/DXX+CSNG(NX)/2+X0-.5)
YN(I)=CINT(NY-(YN(I)/DY+CSNG(NY)/2)+Y0-.5)
NEXT I
LINE INPUT#1,DUMMY$
FOR I=1 TO NSEG
LINE INPUT#1,DUMMY$
DUMMY$=LTRIM$(DUMMY$)
FOR J=1 TO LEN(DUMMY$)
K=J
IF MID$(DUMMY$,J,1)=" " THEN EXIT FOR
NEXT J
LDUMMY=LEN(DUMMY$)-K
NODE(I,1)=VAL(MID$(DUMMY$,1,K-1))
K=K+1
DUMMY$=MID$(DUMMY$,K,LDUMMY)
FOR J=1 TO LEN(DUMMY$)
K=J
IF MID$(DUMMY$,J,1)=" " THEN EXIT FOR
NEXT J
NODE(I,2)=VAL(MID$(DUMMY$,1,K-1))
NEXT I
END IF
CLOSE #1
GOTO SETVIDEO
NOMOD:
MODEL$=""
SETVIDEO:
'
'  Put screen in appropriate mode.
'
IF MODE=13 THEN
SCREEN 13
CLS
'
'  Setting BITSPIXEL here is basically a "back door" into the QBSVGA
' routines so BPALETTE works even though SVGA mode isn't being used.
'
BITSPIXEL=8
COLOR 255
'
'  Since BSCREEN isn't being used, get number of rodent buttons so rodent
' stuff can be initialized below.
'
IF RODENT=1 THEN
INREGS.AX=0
CALL INTERRUPTX(&H33,INREGS,OUTREGS)
BUTTONS=CSNG(OUTREGS.BX)
END IF
ELSE
'
'  Watch out for bios video access being enforced.
'
BIOACC=ABS(VAL(ENVIRON$("BIOS")))
CALL BSCREEN(MODE,255!,0!,0!)
CALL BCLS(0!)
END IF
'
'  Set line for intsructional/distance-measurement output.
'
DROW=1
'
'  (X0,Y0) = picture (0,0) origin in screen coordinates.
'
X0=1 : Y0=17
'
'  Make sure rodent has sufficient properties.
'
IF RODENT=1 THEN
IF BUTTONS>1 THEN
'
'  Make certain rodent initializations in case BSCREEN didn't get called
' to do that.
'
IF MODE=13 THEN HMAX=319 : VMAX=199 : CALL MOUSINIT(0!)
ELSE
RODENT=0
END IF
END IF
'
'  Define 256-color palette.
'
DATABEGIN:
DATA 0,0,0,20,0,0,21,0,0,22,0,0,23,0,0,24,0,0,25,0,0,26,0,0
DATA 27,0,0,28,0,0,29,0,0,30,0,0,31,0,0,32,0,0,33,0,0,34,0,0
DATA 35,0,0,36,0,0,37,0,0,38,0,0,39,0,0,40,0,0,41,0,0,42,0,0
DATA 43,0,0,44,0,0,45,0,0,46,0,0,47,0,0,48,0,0,49,0,0,50,0,0
DATA 51,0,0,52,0,0,53,0,0,54,0,0,55,0,0,56,0,0,57,0,0,58,0,0
DATA 59,0,0,60,0,0,61,0,0,62,0,0,63,0,0,63,20,0,63,21,0,63,22,0
DATA 63,23,0,63,24,0,63,25,0,63,26,0,63,27,0,63,28,0,63,29,0,63,30,0
DATA 63,31,0,63,32,0,63,33,0,63,34,0,63,35,0,63,36,0,63,37,0,63,38,0
DATA 63,39,0,63,40,0,63,41,0,63,42,0,63,43,0,63,44,0,63,45,0,63,46,0
DATA 63,47,0,63,48,0,63,49,0,63,50,0,63,51,0,63,52,0,63,53,0,63,54,0
DATA 63,55,0,63,56,0,63,57,0,63,58,0,63,59,0,63,60,0,63,61,0,63,62,0
DATA 63,63,0,62,63,0,61,63,0,60,63,0,59,63,0,58,63,0,57,63,0,56,63,0
DATA 55,63,0,54,63,0,53,63,0,52,63,0,51,63,0,50,63,0,49,63,0,48,63,0
DATA 47,63,0,46,63,0,45,63,0,44,63,0,43,63,0,42,63,0,41,63,0,40,63,0
DATA 39,63,0,38,63,0,37,63,0,36,63,0,35,63,0,34,63,0,33,63,0,32,63,0
DATA 31,63,0,30,63,0,29,63,0,28,63,0,27,63,0,26,63,0,25,63,0,24,63,0
DATA 23,63,0,22,63,0,21,63,0,20,63,0,19,63,0,18,63,0,17,63,0,16,63,0
DATA 15,63,0,14,63,0,13,63,0,12,63,0,11,63,0,10,63,0,9,63,0,8,63,0
DATA 7,63,0,6,63,0,5,63,0,4,63,0,3,63,0,2,63,0,1,63,0,0,63,0
DATA 0,63,20,0,63,21,0,63,22,0,63,23,0,63,24,0,63,25,0,63,26,0,63,27
DATA 0,63,28,0,63,29,0,63,30,0,63,31,0,63,32,0,63,33,0,63,34,0,63,35
DATA 0,63,36,0,63,37,0,63,38,0,63,39,0,63,40,0,63,41,0,63,42,0,63,43
DATA 0,63,44,0,63,45,0,63,46,0,63,47,0,63,48,0,63,49,0,63,50,0,63,51
DATA 0,63,52,0,63,53,0,63,54,0,63,55,0,63,56,0,63,57,0,63,58,0,63,59
DATA 0,63,60,0,63,61,0,63,62,0,63,63,0,59,63,0,58,63,0,57,63,0,56,63
DATA 0,55,63,0,54,63,0,53,63,0,52,63,0,51,63,0,50,63,0,49,63,0,48,63
DATA 0,47,63,0,46,63,0,45,63,0,44,63,0,43,63,0,42,63,0,41,63,0,40,63
DATA 0,39,63,0,38,63,0,37,63,0,36,63,0,35,63,0,34,63,0,33,63,0,32,63
DATA 0,31,63,0,30,63,0,29,63,0,28,63,0,27,63,0,26,63,0,25,63,0,24,63
DATA 0,23,63,0,22,63,0,21,63,0,20,63,0,19,63,0,18,63,0,17,63,0,16,63
DATA 0,15,63,0,14,63,0,13,63,0,12,63,0,11,63,0,10,63,0,9,63,0,8,63
DATA 0,7,63,0,6,63,0,5,63,0,4,63,0,3,63,0,2,63,0,1,63,0,0,63
'
'  If PALETTE environment variable exists, take it to specify name of
' file to store palette data for other programs to use.  (Skip outputting
' palette data to file if that's already been done once.)
'
IPAL=0
IF ZOOM=0 THEN
PALFILE$=ENVIRON$("PALETTE")
IF PALFILE$="" OR EXIST(PALFILE$)<>0 THEN GOTO PALSTKNOWN
OPEN PALFILE$ FOR OUTPUT AS #2
IPAL=1
END IF
PALSTKNOWN:
RESTORE DATABEGIN
FOR I=0 TO 255
READ R,G,B
CALL BPALETTE(I,CLNG(R+256*G+65536*B))
IF IPAL=1 THEN PRINT#2,R,G,B
NEXT I
IF IPAL=1 THEN CLOSE #2
'
'  Draw box around image area and display color palette below image area.
' (WP = pixel width of color box in palette display.  DOS environment
' variable BORDER determines color of border.)
'
BORDER=VAL(LTRIM$(ENVIRON$("BORDER")))
IF BORDER=0 THEN BORDER=255
IF BORDER<0 THEN BORDER=0
IF MODE>13 THEN
CALL BLINE(X0-1,Y0-1,CSNG(NX)+X0,CSNG(NY)+Y0,BORDER,"B",&HCCCC)
IF MODE=14 THEN WP=2
IF MODE=16 THEN WP=3
IF MODE=18 THEN WP=4
IF MODE=20 THEN WP=5
IF MODE=22 THEN WP=6
FOR I=0 TO 255
CALL BLINE(I*WP,CSNG(NY)+5+Y0,(I+1)*WP-1,CSNG(NY)+20+Y0,I,"BF",-1)
NEXT I
ELSE
WP=1
LINE (X0-1,Y0-1)-(NX+X0,NY+Y0),BORDER,B,&HCCCC
FOR I=0 TO 255
LINE (I,NY+5+Y0)-(I,NY+20+Y0),I,BF
NEXT I
END IF
'
'  Statement 20 is top of display loop.
'
20 TH=VAL(THRESHOLD$)
THRESHOLD$=""
'
'  Get pixelization scale factor.  (It's done inside the display loop
' because various processes can change PMIN, PMAX, LOGSW, or PIXELSW.)
' Watch out for all image strengths being the same and for that one same
' value being zero.
'
IF PMIN>=PMAX THEN PMIN=.1*PMAX
BIN=0
IF PMAX>0 THEN BIN=((1-LOGSW)*(PMAX-PMIN)+LOGSW*LOG(PMAX/PMIN))/254
IF PMAX>0 AND PIXELSW=1 THEN BIN=PMAX/255
IF BIN=0 THEN
CALL BSCREEN(0!,7!,0!,0!)
PRINT "All image strengths are zero."
STOP
END IF
IF RODENT=1 OR MODEL$<>"" THEN
'
'  Set color for SVGA rodent cursor.  (The idea is to try to avoid using
' the same color as a pixel color.  Otherwise, subroutine BOXDRAG tends to
' erase image pixels.)
'
MCOLOR=CINT(TH)
IF MCOLOR<50 THEN MCOLOR=50
END IF
'
'  Reacquire ILINE's and ILINE1's memory locations in case they moved.
'
SMDATA=VARSEG(ILINE(1)) : OSDATA=VARPTR(ILINE(1))
IF F1$<>"" THEN SMDATA1=VARSEG(ILINE1(1)) : OSDATA1=VARPTR(ILINE1(1))
'
'  Display image and calculate centroid parameters.
'
PIXSUM=0 : CENTX=0 : CENTY=0
SOURCE=IBYTES*CLNG(NY-1)
FOR J=1 TO NY
Y=J-1
CALL FPOINT(VMHANDLE,SOURCE,NEWLOC)
CALL FREAD(VMHANDLE,IBYTES,SMDATA,OSDATA)
IF F1$<>"" THEN
CALL FPOINT(VMHANDLE1,SOURCE,NEWLOC)
CALL FREAD(VMHANDLE1,IBYTES,SMDATA1,OSDATA1)
END IF
SOURCE=SOURCE-IBYTES
FOR I=1 TO NX
IBASE=2*I
X=I-1
RVALUE=ILINE(IBASE-1) : IVALUE=ILINE(IBASE)
CPIX=SQR(RVALUE^2+IVALUE^2)
IF (CPIX>=PMIN AND PIXELSW=0) OR PIXELSW=1 THEN
IF LOGSW=0 AND PIXELSW=0 THEN CPIX=INT((CPIX-PMIN)/BIN+.001)+1
IF LOGSW=1 THEN CPIX=INT(LOG(CPIX/PMIN)/BIN+.001)+1
IF PIXELSW=1 THEN CPIX=INT(CPIX/BIN+.001)
ELSE
CPIX=0
END IF
IF F1$<>"" THEN
'
'  Pixelize second image data and combine with first image according to
' action verb.
'
RVALUE=ILINE1(IBASE-1) : IVALUE=ILINE1(IBASE)
CPIX1=SQR(RVALUE^2+IVALUE^2)
IF (CPIX1>=PMIN AND PIXELSW=0) OR PIXELSW=1 THEN
IF LOGSW=0 AND PIXELSW=0 THEN CPIX1=INT((CPIX1-PMIN)/BIN+.001)+1
IF LOGSW=1 THEN CPIX1=INT(LOG(CPIX1/PMIN)/BIN+.001)+1
IF PIXELSW=1 THEN CPIX1=INT(CPIX1/BIN+.001)
ELSE
CPIX1=0
END IF
IF ACT$="OR" THEN CPIX=CINT(CPIX) OR CINT(CPIX1)
IF ACT$="AND" THEN CPIX=CINT(CPIX) AND CINT(CPIX1)
IF ACT$="XOR" THEN CPIX=CINT(CPIX) XOR CINT(CPIX1)
END IF
IF CPIX<=TH THEN CPIX=0
PIXSUM=PIXSUM+CPIX^POWER : CENTX=CENTX+X*CPIX^POWER
CENTY=CENTY+(NY-Y-1)*CPIX^POWER
IF MODE>13 THEN CALL BPSET(X+X0,Y+Y0,CPIX)
IF MODE=13 THEN PSET(X+X0,Y+Y0),CPIX
NEXT I
NEXT J
IF PIXSUM>0 THEN CENTX=INT(CENTX/PIXSUM) : CENTY=INT(CENTY/PIXSUM)
'
'  Draw centroid axes in inverse video if appropriate.
'
IF CENTROID$="ON" THEN
FOR I=0 TO NX-1
IF MODE>13 THEN
INTENSITY=&HFF AND (NOT BPOINT(I+X0,CSNG(NY)-CENTY-1+Y0))
CALL BPSET(I+X0,CSNG(NY)-CENTY-1+Y0,CSNG(INTENSITY))
ELSE
INTENSITY=&HFF AND (NOT CINT(POINT(I+X0,NY-CENTY-1+Y0)))
PSET(I+X0,NY-CENTY-1+Y0),INTENSITY
END IF
NEXT I
FOR I=0 TO NY-1
IF MODE>13 THEN
INTENSITY=&HFF AND (NOT BPOINT(CENTX+X0,I+Y0))
CALL BPSET(CENTX+X0,I+Y0,CSNG(INTENSITY))
ELSE
INTENSITY=&HFF AND (NOT CINT(POINT(CENTX+X0,I+Y0)))
PSET(CENTX+X0,I+Y0),INTENSITY
END IF
NEXT I
END IF
IF MODEL$="" OR FDISP=0 THEN BEEP
DISPMOD:
'
'  Draw facet/grid model if appropriate to do so.
'
IF MODEL$="" OR FDISP=0 THEN GOTO 30
IF MODTYPE$="SRF" THEN
FOR I=1 TO NSURF
VERT1=GETLNG("VERT1",CLNG(I))
VERT2=GETLNG("VERT2",CLNG(I))
VERT3=GETLNG("VERT3",CLNG(I))
X1=GETINT("XMODEL",VERT1)
Y1=GETINT("YMODEL",VERT1)
X2=GETINT("XMODEL",VERT2)
Y2=GETINT("YMODEL",VERT2)
X3=GETINT("XMODEL",VERT3)
Y3=GETINT("YMODEL",VERT3)
IF MODE=13 THEN
LINE (X1,Y1)-(X2,Y2),MODCOL,,LINESTYLE
LINE (X2,Y2)-(X3,Y3),MODCOL,,LINESTYLE
LINE (X3,Y3)-(X1,Y1),MODCOL,,LINESTYLE
ELSE
CALL BLINE(X1,Y1,X2,Y2,MODCOL,"L",LINESTYLE)
CALL BLINE(X2,Y2,X3,Y3,MODCOL,"L",LINESTYLE)
CALL BLINE(X3,Y3,X1,Y1,MODCOL,"L",LINESTYLE)
END IF
NEXT I
ELSEIF MODTYPE$="NEC" THEN
FOR I=1 TO NWIRE
IF MODE=13 THEN
LINE (X1N(I),Y1N(I))-(X2N(I),Y2N(I)),MODCOL,,&HAAAA
ELSE
X1=X1N(I) : Y1=Y1N(I) : X2=X2N(I) : Y2=Y2N(I)
CALL BLINE(X1,Y1,X2,Y2,MODCOL,"L",&HAAAA)
END IF
NEXT I
ELSE
FOR I=1 TO NSEG
X1=XN(NODE(I,1)) : Y1=YN(NODE(I,1)) : X2=XN(NODE(I,2)) : Y2=YN(NODE(I,2))
IF MODE=13 THEN LINE (X1,Y1)-(X2,Y2),MODCOL,,&HAAAA
IF MODE>13 THEN CALL BLINE(X1,Y1,X2,Y2,MODCOL,"L",&HAAAA)
NEXT I
END IF
BEEP
'
'  Print instructional message.
'
30 IF MODE=13 THEN
LOCATE DROW,1
IF RODENT=1 THEN PRINT "M/R/C for rodent; ###-threshold; S=save.";
IF RODENT=0 THEN PRINT "### for threshold; S to save.";
ELSE
CALL BLOCATE(DROW,1!)
IF RODENT=1 THEN
F$="M/R/C for rodent; ###/T for threshold; S to save.;"
ELSE
F$="### for threshold; S to save.;"
END IF
CALL BPRINT(F$)
END IF
'
'  Wait for input of new threshold or other instructions.
'
40 T$=UCASE$(INKEY$) : IF T$="" THEN GOTO 40
IF T$="F" THEN FDISP=1% AND (NOT FDISP)
IF T$="F" AND FDISP=1 AND MODEL$<>"" THEN GOTO DISPMOD
IF T$="F" AND MODEL$<>"" THEN GOTO 20
IF T$<>"D" THEN GOTO NOSHELL
CALL BSCREEN(0!,7!,0!,0!)
WIDTH 80
CALL STACK("RAMFREE")
PRINT
PRINT "Type EXIT to return to image displayer."
PRINT
SHELL
'
'  Reset threshold string before redisplaying image.
'
THRESHOLD$=LTRIM$(STR$(TH))
GOTO SETVIDEO
NOSHELL:
'
'  In the following, the tests against F1$ are because if a second image
' file is involved, most keypresses just terminate the program.
'
'  If rodent function is being used, reset rodent driver and make
' appropriate constraints on rodent cursor motion.  (RODENT subroutine is
' also called, if appropriate, from within this IF/THEN construct for
' simplicity.)
'
IF INSTR("MTRCEAOZ",T$)<>0 AND RODENT=1 THEN
INREGS.AX=0
CALL INTERRUPTX(&H33,INREGS,OUTREGS)
IF T$<>"T" THEN CALL SETLIM(X0,Y0,X0+CSNG(NX-1),Y0+CSNG(NY-1))
IF T$="T" THEN CALL SETLIM(0!,0!,HMAX,VMAX)
IF T$="M" THEN GOSUB RODENT
END IF
IF (T$<>"E" AND T$<>"Z") OR RODENT=0 OR F1$<>"" THEN GOTO NOEXT
'
'  Portion of image is to be output to new image file, PORTION.BIN/CI, and
' possibly displayed.
'
'  First, reset threshold string.
'
THRESHOLD$=""
CALL BOXDRAG(XL,YL,XR,YR,BUTTON)
'
'  Convert coordinates output by BOXDRAG to array indices.
'
XL=XL-X0+1 : YL=YL-Y0+1 : XR=XR-X0+1 : YR=YR-Y0+1 : YL=NY-YL+1 : YR=NY-YR+1
IF XL>XR THEN SWAP XL,XR
IF YL>YR THEN SWAP YL,YR
NXOUT=CINT(XR-XL+1) : NYOUT=CINT(YR-YL+1)
'
'  If rodent button other than the left one was used, take that as a
' signal to extract a region of the image that is a power of 2 in size in
' both dimensions.
'
IF BUTTON<>0 THEN
NXTEST=LOG(NXOUT)/LOG(2) : NXTEST=2^INT(NXTEST+.001)
NYTEST=LOG(NYOUT)/LOG(2) : NYTEST=2^INT(NYTEST+.001)
IF CINT(NXTEST)<NXOUT THEN NXTEST=2*NXTEST
IF CINT(NYTEST)<NYOUT THEN NYTEST=2*NYTEST
XR=XL+NXTEST-1 : YR=YL+NYTEST-1
IF XR>NX THEN
XL=XL-XR+NX
IF XL<1 THEN XL=1 : NXTEST=NX
XR=XL+NXTEST-1
END IF
IF YR>NY THEN
YL=YL-YR+NY
IF YL<1 THEN YL=1 : NYTEST=NY
YR=YL+NYTEST-1
END IF
'
'  Recalculate NXOUT and NYOUT.
'
NXOUT=CINT(XR-XL+1) : NYOUT=CINT(YR-YL+1)
END IF
IF EXT$<>"CI" OR CICON$="DOS" THEN
IF T$="Z" THEN F$="PORTION.BIN"
'
'  Output data (with current threshold) to PORTION.BIN.  (Delete old file
' if it exists.)
'
IF EXIST("PORTION.BIN")=1 THEN KILL "PORTION.BIN"
OPEN "PORTION.BIN" FOR BINARY AS #1
PUT#1,,NXOUT
PUT#1,,NYOUT
PUT#1,,DXX
PUT#1,,DY
PUT#1,,RNGPAD
PUT#1,,CRGPAD
PUT#1,,SARTYPE
PUT#1,,KXMIN
PUT#1,,KXMAX
PUT#1,,KYMIN
PUT#1,,KYMAX
STRVAR=CHR$(ISP)
PUT#1,,STRVAR
STRVAR=CHR$(INTPOL)
PUT#1,,STRVAR
PUT#1,,BFR
PUT#1,,EFR
PUT#1,,NFREQ
PUT#1,,BAZ0
PUT#1,,EAZ0
PUT#1,,NAZA
PUT#1,,TANPSI0
PUT#1,,TANETA0
STRVAR=CHR$((IWIN+256) MOD 256)
PUT#1,,STRVAR
PUT#1,,SLL
FOR Y=YL TO YR
FOR X=XL TO XR
RVALUE=GETSNG("A",2&*CLNG(X+(Y-1)*NX)-1&)
IVALUE=GETSNG("A",2&*CLNG(X+(Y-1)*NX))
CPIX=SQR(RVALUE^2+IVALUE^2)
IF PIXELSW=0 THEN
IF CPIX>=PMIN THEN CPIX=INT((CPIX-PMIN)/BIN+.001)+1
IF CPIX<PMIN THEN CPIX=0
ELSE
CPIX=INT(CPIX/BIN+.001)
END IF
IF CPIX<=TH THEN RVALUE=0 : IVALUE=0
PUT#1,,RVALUE
PUT#1,,IVALUE
NEXT X
NEXT Y
ELSE
'
'  Write PORTION.CI  (It will be in the FCOMPLEX format, regardless of the
' format at input.)
'
IF T$="Z" THEN F$="PORTION.CI"
IF EXIST("PORTION.CI")=1 THEN KILL "PORTION.CI"
OPEN "PORTION.CI" FOR BINARY AS #1
FOR X=XL TO XR
FOR Y=YL TO YR
RVALUE=GETSNG("A",2&*CLNG(X+(Y-1)*NX)-1&)
IVALUE=GETSNG("A",2&*CLNG(X+(Y-1)*NX))
CPIX=SQR(RVALUE^2+IVALUE^2)
IF PIXELSW=0 THEN
IF CPIX>=PMIN THEN CPIX=INT((CPIX-PMIN)/BIN+.001)+1
IF CPIX<PMIN THEN CPIX=0
ELSE
CPIX=INT(CPIX/BIN+.001)
END IF
IF CPIX<=TH THEN RVALUE=0 : IVALUE=0
S4=MKS$(RVALUE)
FOR I=4 TO 1 STEP -1
BYTE1=MID$(S4,I,1)
PUT#1,,BYTE1
NEXT I
S4=MKS$(IVALUE)
FOR I=4 TO 1 STEP -1
BYTE1=MID$(S4,I,1)
PUT#1,,BYTE1
NEXT I
NEXT Y
NEXT X
'
'  Copy original sd file to PORTION.SD, altering changed data.  TYPEFLAG
' is used to detect the condition in which the DataType parameter wasn't
' included in original sd file.  (It needs to be in the SD file because
' ICOMPLEX is the Case Exec default, not FCOMPLEX.)
'
TYPEFLAG=0
OPEN SDFILE$ FOR INPUT AS #2
OPEN "PORTION.SD" FOR OUTPUT AS #3
WHILE NOT EOF(2)
LINE INPUT#2,D$
IF INSTR(D$,"ImageHeight")>0 THEN
CALL PARSE(D$,"=",S1$,S2$)
D$=S1$+"="+LTRIM$(RTRIM$(STR$(NXOUT)))
END IF
IF INSTR(D$,"ImageWidth")>0 THEN
CALL PARSE(D$,"=",S1$,S2$)
D$=S1$+"="+LTRIM$(RTRIM$(STR$(NYOUT)))
END IF
IF INSTR(D$,"DataType")>0 THEN TYPEFLAG=1 : D$="DataType=FCOMPLEX"
PRINT#3,D$
WEND
CLOSE #2
IF TYPEFLAG=0 THEN PRINT#3,"DataType=FCOMPLEX"
CLOSE #3
END IF
CLOSE #1
BEEP
'
'  Either go back to original image display or display extracted portion
' (after reinputting it).
'
IF T$="E" THEN GOTO 40
'
'  Force a video mode reset upon redisplay.
'
CALL BSCREEN(0!,255!,0!,0!)
ZOOM=1
GOTO REDISPLAY
NOEXT:
IF T$="L" THEN PIXELSW=0 : LOGSW=1-LOGSW : THRESHOLD$="0" : GOTO 20
IF T$="" THEN LOGSW=0 : THRESHOLD$="0" : PIXELSW=1-PIXELSW : GOTO 20
IF T$="R" AND RODENT=1 AND F1$="" THEN
'
'  Output complex image sequence to COMPLEX.ROW.  (If FSIGN = -1 and you
' wish to transform the output sequence to the frequency domain, use what
' would normally be considered to be an inverse FFT (positive sign in
' fourier kernel).)
'
OPEN "COMPLEX.ROW" FOR OUTPUT AS #1
IF FSIGN=1 THEN PRINT#1,DXX
IF FSIGN=-1 THEN PRINT#1,NX,DXX
CALL GETPOS(XP,YP,BUTTON)
COMROW=CINT(YP-Y0)+1
COMROW=NY-COMROW+1
FOR I=1 TO NX
RVALUE=GETSNG("A",2&*(CLNG(I)+CLNG(COMROW-1)*CLNG(NX))-1&)
IVALUE=GETSNG("A",2&*(CLNG(I)+CLNG(COMROW-1)*CLNG(NX)))
PRINT#1,RVALUE,IVALUE
NEXT I
CLOSE #1
ELSEIF T$="C" AND RODENT=1 AND F1$="" THEN
'
'  Output complex image sequence to COMPLEX.COL.
'
OPEN "COMPLEX.COL" FOR OUTPUT AS #1
IF FSIGN=1 THEN PRINT#1,DY
IF FSIGN=-1 THEN PRINT#1,NY,DY
CALL GETPOS(XP,YP,BUTTON)
COMCOL=CINT(XP-X0)+1
FOR J=1 TO NY
RVALUE=GETSNG("A",2&*(CLNG(COMCOL)+CLNG(J-1)*CLNG(NX))-1&)
IVALUE=GETSNG("A",2&*(CLNG(COMCOL)+CLNG(J-1)*CLNG(NX)))
PRINT#1,RVALUE,IVALUE
NEXT J
CLOSE #1
END IF
IF (T$="R" OR T$="C") AND RODENT=1 AND F1$="" THEN GOTO 40
IF (T$="R" OR T$="C") AND RODENT=0 AND F1$="" THEN THRESHOLD$=""
IF T$<>"T" OR RODENT=0 THEN GOTO NOTHRESH
'
'  Use rodent to set pixel display threshold.
'
'  Wait for rodent button press and then process position output from
' GETPOS.
'
CALL GETPOS(XP,YP,BUTTON)
IF MODE>13 THEN THRESHOLD$=STR$(BPOINT(XP,YP)-1)
IF MODE=13 THEN THRESHOLD$=STR$(POINT(XP,YP)-1)
IF VAL(THRESHOLD$)<0 THEN THRESHOLD$="0"
'
'  Replot image with new threshold.
'
GOTO 20
NOTHRESH:
IF T$<>"S" THEN GOTO 50
'
'  Image is to be saved in an array and then to file.
'
'  First, delete old IMAGE.GET file (if any) and reset threshold string.
'
IF EXIST("IMAGE.GET")=1 THEN KILL "IMAGE.GET"
THRESHOLD$=""
'
'   Use conventional QB functions for conventional video mode.
'
IF MODE>13 THEN
'
'  If DIMVMS, above, had a problem with VMS, just go back and wait for
' instructional keypress.
'
IF SAVEERROR<>0 THEN GOTO 40
CALL VGET(X0-1,Y0-1,CSNG(NX)+X0,CSNG(NY)+Y0,"IMAGE")
CALL VBSAVE("IMAGE",BYTES,"IMAGE.GET")
ELSE
GET (X0-1,Y0-1)-(NX+X0,NY+Y0),IMAGE
DEF SEG=VARSEG(IMAGE(1))
BSAVE "IMAGE.GET",VARPTR(IMAGE(1)),BYTES
DEF SEG
END IF
BEEP
GOTO 40
50 IF T$<>"O" OR F1$<>"" THEN GOTO 54
'
'  Correct selected image region for layover.
'
'  First, reset threshold string in case rodent isn't installed or this
' option isn't otherwise available.
'
THRESHOLD$="" : IF RODENT=0 THEN GOTO 55
'
'  Use subroutine BOXDRAG to select area to be orthorectified.
'
CALL BOXDRAG(XL,YL,XR,YR,BUTTON)
'
'  Convert coordinates output by BOXDRAG to array indices.
'
XL=XL-X0+1 : YL=YL-Y0+1 : XR=XR-X0+1 : YR=YR-Y0+1 : YL=NY-YL+1 : YR=NY-YR+1
IF XL>XR THEN SWAP XL,XR
IF YL>YR THEN SWAP YL,YR
IF MODE=13 THEN
LOCATE DROW,1
PRINT "Height (m) causing layover?             ";
LOCATE DROW,28
INPUT "",ZLAY
ELSE
CALL BLOCATE(DROW,1!)
CALL BPRINT("Height (m) causing layover?                      ;")
CALL BLOCATE(DROW,28!)
CALL BINPUT("",ZLAY)
END IF
'
'  OFLAG is used to track whether or not orthorectification really
' occurred.  (If ZLAY is input as zero or the SAR isn't elevated, there's
' no need to consume time performing the orthorectification.)
'
OFLAG=0
IF ABS(ZLAY)>1E-5 AND ABS(TANPSI)>1E-5 AND KXMIN<KXMAX AND KYMIN<KYMAX THEN
'
'  Make sure NX and NY are powers of 2 first.
'
NXTEST=LOG(NX)/LOG(2) : NXTEST=2^INT(NXTEST+.001)
NYTEST=LOG(NY)/LOG(2) : NYTEST=2^INT(NYTEST+.001)
IF CINT(NXTEST)=NX AND CINT(NYTEST)=NY THEN
'
'  Define virtual memory array to store FFT of image.  If REDIMVMS
' operation fails, abort orthorectification process.
'
BVMS=0
CALL REDIMVMS("B",2&*CLNG(NX)*CLNG(NY),"SINGLE",BVMS)
IF BVMS=0 THEN
OFLAG=1
'
'  Go through image array A and FFT range data, after zeroing out data
' outside of XL - XR or YL - YR.  (Data in the original image *within*
' this region is zeroed after copying to XREAL/XIMAG arrays.)
'
REDIM XREAL(1 TO NX) AS DOUBLE,XIMAG(1 TO NX) AS DOUBLE
INV=(1-FSIGN)/2
'
'  Column index only varies from YL to YR here because data outside of
' that in array B is already 0 from the REDIMVMS call, and only the rows
' within that range need to be FFTed.
'
FOR YI=YL TO YR
IF XL>1 THEN
FOR XI=1 TO XL-1
XREAL(XI)=0# : XIMAG(XI)=0#
NEXT XI
END IF
FOR XI=XL TO XR
XREAL(XI)=CDBL(GETSNG("A",2&*CLNG(XI+(YI-1)*NX)-1&))
XIMAG(XI)=CDBL(GETSNG("A",2&*CLNG(XI+(YI-1)*NX)))
CALL PUTSNG("A",2&*CLNG(XI+(YI-1)*NX)-1&,0!)
CALL PUTSNG("A",2&*CLNG(XI+(YI-1)*NX),0!)
NEXT XI
IF XR<=NX-1 THEN
FOR XI=XR+1 TO NX
XREAL(XI)=0# : XIMAG(XI)=0#
NEXT XI
END IF
'
'  If FSIGN = -1, how FFT stores "negative frequencies" at the end of
' the data has to be taken into account before doing the FFT.  If FSIGN =
' 1, that has to be taken into account afterwards.
'
IF FSIGN=-1 THEN
FOR XI=1 TO NX\2
SWAP XREAL(XI),XREAL(XI+NX\2)
SWAP XIMAG(XI),XIMAG(XI+NX\2)
NEXT XI
END IF
CALL FFT(NX,INV,CDBL(DXX))
'
'  Put FFT results in B.
'
IF FSIGN=-1 THEN
FOR XI=1 TO NX
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&,CSNG(XREAL(XI)))
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX),CSNG(XIMAG(XI)))
NEXT XI
ELSE
FOR XI=1 TO NX\2
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&,CSNG(XREAL(XI+NX\2)))
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX),CSNG(XIMAG(XI+NX\2)))
CALL PUTSNG("B",2&*CLNG(XI+NX\2+(YI-1)*NX)-1&,CSNG(XREAL(XI)))
CALL PUTSNG("B",2&*CLNG(XI+NX\2+(YI-1)*NX),CSNG(XIMAG(XI)))
NEXT XI
END IF
NEXT YI
'
'  Okay, range data is fourier transformed.  Repeat above for cross range.
' (Array A is done with for now.  Data is taken out of B, FFTed, and then
' put back in B.  The more noticeable difference, in terms of time, is
' that all columns in B must be FFTed.)
'
REDIM XREAL(1 TO NY) AS DOUBLE,XIMAG(1 TO NY) AS DOUBLE
FOR XI=1 TO NX
IF FSIGN=-1 THEN
FOR YI=1 TO NY\2
XREAL(YI)=CDBL(GETSNG("B",2&*CLNG(XI+(YI+NY\2-1)*NX)-1&))
XIMAG(YI)=CDBL(GETSNG("B",2&*CLNG(XI+(YI+NY\2-1)*NX)))
XREAL(YI+NY\2)=CDBL(GETSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&))
XIMAG(YI+NY\2)=CDBL(GETSNG("B",2&*CLNG(XI+(YI-1)*NX)))
NEXT YI
ELSE
FOR YI=1 TO NY
XREAL(YI)=CDBL(GETSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&))
XIMAG(YI)=CDBL(GETSNG("B",2&*CLNG(XI+(YI-1)*NX)))
NEXT YI
END IF
CALL FFT(NY,INV,CDBL(DY))
IF FSIGN=-1 THEN
FOR YI=1 TO NY
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&,CSNG(XREAL(YI)))
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX),CSNG(XIMAG(YI)))
NEXT YI
ELSE
FOR YI=1 TO NY\2
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&,CSNG(XREAL(YI+NY\2)))
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX),CSNG(XIMAG(YI+NY\2)))
CALL PUTSNG("B",2&*CLNG(XI+(YI+NY\2-1)*NX)-1&,CSNG(XREAL(YI)))
CALL PUTSNG("B",2&*CLNG(XI+(YI+NY\2-1)*NX),CSNG(XIMAG(YI)))
NEXT YI
END IF
NEXT XI
'
'  Okay, array B now stores an FFT of the data originally in A after
' rectangularly windowing the image in A about the region to be ortho-
' rectified.  The spectrum of the original image was likely immersed in a
' "sea of zeros."  However, because of the initial windowing of the image,
' those zeros are no longer necessarilly zero.  That means that the
' spectrum is no longer confined to KXMIN - KXMAX and KYMIN - KYMAX.
' Find out how many zeros were originally on either side of the spectrum
' so that appropriate spectral offsets can be calculated.  (Assume the
' spectrum is centered within the entire FFT.  (This all works best if the
' padding-induced size increase was a multiple of 2.))
'
NXBP=CDBL(NX)/CDBL(RNGPAD+1) : NYBP=CDBL(NY)/CDBL(CRGPAD+1)
NXB=(CDBL(NX)-NXBP)/2# : NYB=(CDBL(NY)-NYBP)/2#
'
'  Get spatial frequency sampling intervals.
'
DKX=2#*PI/CDBL(DXX)/CDBL(NX) : DKY=2#*PI/CDBL(DY)/CDBL(NY)
'
'  Correct phases.
'
FOR XI=1 TO NX
KX=DKX*CDBL(XI-1)+CDBL(KXMIN)-NXB*DKX
FOR YI=1 TO NY
KY=DKY*CDBL(YI-1)+CDBL(KYMIN)-NYB*DKY
IF SARTYPE="C" THEN
KS=SQR(KX^2+KY^2)*CDBL(FSIGN*ZLAY*TANPSI)
ELSE
KS=(KX*CDBL(TANPSI)-KY*CDBL(TANETA))*CDBL(ZLAY*FSIGN)
END IF
RCOR=COS(KS) : ICOR=SIN(KS)
RVALUE=GETSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&)
IVALUE=GETSNG("B",2&*CLNG(XI+(YI-1)*NX))
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&,CSNG(RVALUE*RCOR-IVALUE*ICOR))
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX),CSNG(IVALUE*RCOR+RVALUE*ICOR))
NEXT YI
NEXT XI
'
'  Phases are corrected.  FFT frequency cross range lines (since the
' XREAL and XIMAG arrays are already set up for that) and then do range
' lines.
'
DKY=DKY/2#/PI
INV=(1+FSIGN)/2
FOR XI=1 TO NX
'
'  Get data out of B.
'
IF FSIGN=-1 THEN
FOR YI=1 TO NY
XREAL(YI)=CDBL(GETSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&))
XIMAG(YI)=CDBL(GETSNG("B",2&*CLNG(XI+(YI-1)*NX)))
NEXT YI
ELSE
FOR YI=1 TO NY\2
XREAL(YI)=CDBL(GETSNG("B",2&*CLNG(XI+(YI+NY\2-1)*NX)-1&))
XIMAG(YI)=CDBL(GETSNG("B",2&*CLNG(XI+(YI+NY\2-1)*NX)))
XREAL(YI+NY\2)=CDBL(GETSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&))
XIMAG(YI+NY\2)=CDBL(GETSNG("B",2&*CLNG(XI+(YI-1)*NX)))
NEXT YI
END IF
CALL FFT(NY,INV,DKY)
'
'  Put data back in B.
'
IF FSIGN=-1 THEN
FOR YI=1 TO NY\2
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&,CSNG(XREAL(YI+NY\2)))
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX),CSNG(XIMAG(YI+NY\2)))
CALL PUTSNG("B",2&*CLNG(XI+(YI+NY\2-1)*NX)-1&,CSNG(XREAL(YI)))
CALL PUTSNG("B",2&*CLNG(XI+(YI+NY\2-1)*NX),CSNG(XIMAG(YI)))
NEXT YI
ELSE
FOR YI=1 TO NY
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&,CSNG(XREAL(YI)))
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX),CSNG(XIMAG(YI)))
NEXT YI
END IF
NEXT XI
DKX=DKX/2#/PI
REDIM XREAL(1 TO NX) AS DOUBLE,XIMAG(1 TO NX) AS DOUBLE
FOR YI=1 TO NY
IF FSIGN=-1 THEN
FOR XI=1 TO NX
XREAL(XI)=CDBL(GETSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&))
XIMAG(XI)=CDBL(GETSNG("B",2&*CLNG(XI+(YI-1)*NX)))
NEXT XI
ELSE
FOR XI=1 TO NX\2
XREAL(XI)=CDBL(GETSNG("B",2&*CLNG(XI+NX\2+(YI-1)*NX)-1&))
XIMAG(XI)=CDBL(GETSNG("B",2&*CLNG(XI+NX\2+(YI-1)*NX)))
XREAL(XI+NX\2)=CDBL(GETSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&))
XIMAG(XI+NX\2)=CDBL(GETSNG("B",2&*CLNG(XI+(YI-1)*NX)))
NEXT XI
END IF
CALL FFT(NX,INV,DKX)
IF FSIGN=-1 THEN
FOR XI=1 TO NX\2
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&,CSNG(XREAL(XI+NX\2)))
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX),CSNG(XIMAG(XI+NX\2)))
CALL PUTSNG("B",2&*CLNG(XI+NX\2+(YI-1)*NX)-1&,CSNG(XREAL(XI)))
CALL PUTSNG("B",2&*CLNG(XI+NX\2+(YI-1)*NX),CSNG(XIMAG(XI)))
NEXT XI
ELSE
FOR XI=1 TO NX
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&,CSNG(XREAL(XI)))
CALL PUTSNG("B",2&*CLNG(XI+(YI-1)*NX),CSNG(XIMAG(XI)))
NEXT XI
END IF
NEXT YI
'
'  Okay, orthorectified image corresponding to selected area in original
' image is in B.  Add image B to image A and terminate this excitement.
'
FOR XI=1 TO NX
FOR YI=1 TO NY
RVALUEA=GETSNG("A",2&*CLNG(XI+(YI-1)*NX)-1&)
IVALUEA=GETSNG("A",2&*CLNG(XI+(YI-1)*NX))
RVALUEB=GETSNG("B",2&*CLNG(XI+(YI-1)*NX)-1&)
IVALUEB=GETSNG("B",2&*CLNG(XI+(YI-1)*NX))
CALL PUTSNG("A",2&*CLNG(XI+(YI-1)*NX)-1&,RVALUEA+RVALUEB)
CALL PUTSNG("A",2&*CLNG(XI+(YI-1)*NX),IVALUEA+IVALUEB)
NEXT YI
NEXT XI
END IF
END IF
END IF
'
'  Get new min/max values and redraw image (after clearing portion of
' input prompt that won't get cleared otherwise).
'
IF OFLAG=0 THEN GOTO 30
PMAX=-1
PMIN=1E30
FOR Y=1 TO NY
FOR X=1 TO NX
RVALUE=GETSNG("A",2&*CLNG(X+(Y-1)*NX)-1&)
IVALUE=GETSNG("A",2&*CLNG(X+(Y-1)*NX))
MAG=SQR(RVALUE^2+IVALUE^2)
IF MAG>PMAX THEN PMAX=MAG
IF MAG>0 AND MAG<PMIN THEN PMIN=MAG
NEXT X
NEXT Y
IF MODE>13 THEN CALL LCLEAR(50!)
BEEP : GOTO 20
54 IF T$<>"A" OR F1$<>"" THEN GOTO 55
'
'  Watch out for zero center frequency.
'
IF SARF=0 THEN GOTO 40
'
'  First, reset threshold string in case rodent isn't installed.
'
THRESHOLD$="" : IF RODENT=0 THEN GOTO 55
'
'  Use subroutine BOXDRAG to select specific areas to have their
' intensities scaled.
'
AREA=1
GETAREA:
AR$=UCASE$(INKEY$) : IF AR$="" THEN AR$=" "
IF ASC(AR$)=27 THEN GOTO 40
IF ASC(AR$)=13 THEN GOTO SCALE
CALL BOXDRAG(XLA(AREA),YLA(AREA),XRA(AREA),YRA(AREA),BUTTON)
'
'  Convert coordinates output by BOXDRAG to array indices.
'
XLA(AREA)=XLA(AREA)-X0+1 : YLA(AREA)=YLA(AREA)-Y0+1
XRA(AREA)=XRA(AREA)-X0+1 : YRA(AREA)=YRA(AREA)-Y0+1
YLA(AREA)=NY-YLA(AREA)+1 : YRA(AREA)=NY-YRA(AREA)+1
IF XLA(AREA)>XRA(AREA) THEN SWAP XLA(AREA),XRA(AREA)
IF YLA(AREA)>YRA(AREA) THEN SWAP YLA(AREA),YRA(AREA)
AREA=AREA+1
'
'  Use button-press as alternative method of skipping to scaling calculat-
' ion.
'
IF AREA<51 AND BUTTON=0 THEN GOTO GETAREA
SCALE:
AREA=AREA-1
'
'  Watch out for premature termination.
'
IF AREA<1 THEN GOTO 40
GETPARAM:
'
'  Get aspect angles and other necessary data from user.
'
IF MODE=13 THEN
LOCATE DROW,1
PRINT "Old aspect angle (degrees)?             ";
LOCATE DROW,28
INPUT "",OANGLE
LOCATE DROW,1
PRINT "New aspect angle (degrees)?             ";
LOCATE DROW,28
INPUT "",NANGLE
LOCATE DROW,1
PRINT "Object diameter (m)?                    ";
LOCATE DROW,21
INPUT "",ODIAM
ELSE
CALL BLOCATE(DROW,1!)
CALL BPRINT("Old aspect angle (degrees)?                      ;")
CALL BLOCATE(DROW,28!)
CALL BINPUT("",OANGLE)
CALL LCLEAR(28!)
CALL BLOCATE(DROW,1!)
CALL BINPUT("New aspect angle (degrees)?",NANGLE)
CALL LCLEAR(21!)
CALL BLOCATE(DROW,1!)
CALL BINPUT("Object diameter (m)?",ODIAM)
END IF
IF NANGLE<-90 OR NANGLE>90 OR OANGLE<=-90 OR OANGLE>=90 THEN GOTO GETPARAM
IF ODIAM<0 THEN GOTO GETPARAM
NANGLE=NANGLE*PI/180 : OANGLE=OANGLE*PI/180 : ODIAM=ODIAM/2
ARGO=2#*PI*CDBL(SARF*ODIAM*SIN(OANGLE))/.29979# : OSF=1#
IF ARGO<>0# THEN OSF=ABS(BESSEL(1&,2#*ARGO)/ARGO)
ARGN=2#*PI*CDBL(SARF*ODIAM*SIN(NANGLE))/.29979# : NSF=1#
IF ARGN<>0# THEN NSF=ABS(BESSEL(1&,2#*ARGN)/ARGN)
SFACT=(1+COS(NANGLE))/(1+COS(OANGLE)) : IF OSF<>0# THEN SFACT=SFACT*NSF/OSF
'
'  Go through image, and for each area selected, scale intensities by
' SFACT.  Find new minimum and maximum intensities in process.
'
PMAX=-1
PMIN=1E30
FOR Y=1 TO NY
FOR X=1 TO NX
RVALUE=GETSNG("A",2&*CLNG(X+(Y-1)*NX)-1&)
IVALUE=GETSNG("A",2&*CLNG(X+(Y-1)*NX))
FOR K=1 TO AREA
IF XLA(K)<=X AND X<=XRA(K) AND YLA(K)<=Y AND Y<=YRA(K) THEN
RVALUE=RVALUE*SFACT : IVALUE=IVALUE*SFACT
CALL PUTSNG("A",2&*CLNG(X+(Y-1)*NX)-1&,RVALUE)
CALL PUTSNG("A",2&*CLNG(X+(Y-1)*NX),IVALUE)
END IF
NEXT K
MAG=SQR(RVALUE^2+IVALUE^2)
IF MAG>PMAX THEN PMAX=MAG
IF MAG>0 AND MAG<PMIN THEN PMIN=MAG
NEXT X
NEXT Y
'
'  Redraw image (after clearing portion of input prompt that won't get
' cleared otherwise).
'
IF MODE>13 THEN CALL LCLEAR(50!)
BEEP : GOTO 20
'
'  The threshold is generally a three-digit integer but a <CR> may be used
' to indicate that the threshold is less than 100.
'
55 IF THRESHOLD$<>"" AND T$=CHR$(13) THEN GOTO 20
IF INSTR("0123456789",T$)=0 THEN GOTO QUITEXEC
'
'  Either get next digit in threshold (or look for interrupting keypress)
' or re-display image with new threshold.
'
THRESHOLD$=THRESHOLD$+T$ : IF LEN(THRESHOLD$)<3 THEN GOTO 40
GOTO 20
NOSIZE:
CALL BSCREEN(0!,255!,0!,0!)
PRINT
PRINT "Couldn't find image size in ";SDFILE$;"."
GOTO 60
BADTYPE:
CALL BSCREEN(0!,255!,0!,0!)
PRINT
PRINT "Can't handle ";DATATYPE$;" image."
GOTO 60
QUITEXEC:
'
'  Reset screen and quit.
'
CALL BSCREEN(0!,255!,0!,0!)
60 CALL VMSCLOSE
END
'
'  This subroutine reads two consecutive rodent cursor positions and
' calculates the distance between them in the image being displayed.
'
RODENT:
'
'  Print instructional message.  (This particular one won't get printed
' again until RODENT is called again.)
'
IF MODE=13 THEN
LOCATE DROW,1
PRINT "Position 1?                             ";
ELSE
CALL BLOCATE(DROW,1!)
CALL BPRINT("Position 1?                                      ;")
END IF
GETPOS1:
'
'  Either terminate or get first position.
'
CALL GETPOS(X1,Y1,BUTTON)
'
'  For Mouse Systems mouse, treat middle button like right button.
'
IF BUTTONS>2 AND BUTTON=2 THEN BUTTON=1
IF BUTTON=1 THEN GOTO QUITROD
'
'  Left button must have been pressed.
'
'  Get second positon.  This time, use positional output of GETPOS no
' matter which button was pressed; distance measurement is commited now.
'
'  Print instructional message for second position.
'
IF MODE=13 THEN
LOCATE DROW,1
PRINT "Position 2?              ";
ELSE
CALL BLOCATE(DROW,1!)
CALL BPRINT("Position 2?              ;")
END IF
CALL GETPOS(X2,Y2,BUTTON)
'
'  Complete distance measurement and print it out.
'
DIST=SQR((DXX*(X2-X1))^2+(DY*(Y2-Y1))^2)
IF MODE>13 THEN
CALL BLOCATE(DROW,1!)
CALL BPRINT("DISTANCE ="+STR$(DIST)+";")
ELSE
LOCATE DROW,1
PRINT "DISTANCE =";STR$(DIST);
END IF
'
'  Make new distance measurement (or return to calling routine).
'
GOTO GETPOS1
QUITROD:
'
'  Reset threshold string before returning in case this routine was called
' in middle of changing threshold.
'
THRESHOLD$=""
RETURN 30
FIXDATA:
IF FER=0 THEN
RVALUE=0
IF EXT$<>"CI" THEN CALL PUTSNG(ARRAY$,2&*CLNG(I+(J-1)*NX)-1&,RVALUE)
ELSEIF FER=1 THEN
IVALUE=0
IF EXT$<>"CI" THEN CALL PUTSNG(ARRAY$,2&*CLNG(I+(J-1)*NX),IVALUE)
ELSEIF FER=2 THEN
'
'  This section of code really shouldn't be necessary, but it's here just
' in case I'm wrong.
'
RVALUE=0 : IVALUE=0 : MAG=0
IF EXT$<>"CI" THEN
CALL PUTSNG(ARRAY$,2&*CLNG(I+(J-1)*NX)-1&,RVALUE)
CALL PUTSNG(ARRAY$,2&*CLNG(I+(J-1)*NX),IVALUE)
END IF
ELSE
'
'  This section of code should also never get invoked.
'
CALL BSCREEN(0!,7!,0!,0!)
PRINT "Something bizarre happened."
RESUME 60
END IF
RESUME NEXT
'
'  This subroutine clears the trailling portion of the "DROW line" beginn-
' ing at position BCOL.
'
SUB LCLEAR(BCOL)
DIM I AS INTEGER,ECOL AS INTEGER
SHARED DROW
ECOL=INT((HMAX+1)/8+.001)
IF BCOL<=ECOL THEN
CALL BLOCATE(DROW,BCOL)
FOR I=CINT(BCOL) TO ECOL
CALL BPRINT(" ;")
NEXT I
END IF
END SUB
'
'  This subroutine parses an input string S$ into the separate strings S1$
' and S2$, based on the delimiting string DL$.
'
SUB PARSE(S$,DL$,S1$,S2$)
SI$=LTRIM$(RTRIM$(S$))
N=LEN(SI$)
S1$=SI$
S2$=""
IF N=0 THEN GOTO TERM
I=INSTR(SI$,DL$)
IF I=0 THEN GOTO TERM
S1$=MID$(SI$,1,I-1)
S2$=MID$(SI$,I+LEN(DL$),N-I-LEN(DL$)+1)
TERM:
END SUB
'
'  THIS FUNCTION CALCULATES THE BESSEL FUNCTION OF THE FIRST KIND VIA
' AN ALGORITHM IN "AN ATLAS OF FUNCTIONS," BY SPANIER AND OLDHAM, PAGE
' 527--WITH A FEW MINOR ADAPTATIONS.  IN THE PARAMETER LIST, ORDER IS THE
' INPUT (LONG) ORDER AND Y IS THE INPUT (DOUBLE) ARGUMENT.  (IT WAS
' ADAPTED FROM A FORTRAN ROUTINE--THAT EXPLAINS LABELS LIKE "FIVE" AND
' THE USE OF LONG INTEGERS.)
'
'  YOUR MAIN ROUTINE MUST INCLUDE THE FOLLOWING DECLARE STATEMENT.
'
'   DECLARE FUNCTION BESSEL#(ORDER AS LONG,Y AS DOUBLE)
'
DEFDBL A-H,O-Z
DEFLNG I-N
FUNCTION BESSEL#(ORDER AS LONG,Y AS DOUBLE)
PI=4#*ATN(1#)
'
'  Y IS ALIASED AS X IN CASE Y IS NEGATIVE.
'
X=ABS(Y)
N=ORDER
F1=1#
IF N=0 AND X=0# THEN GOTO FIVE
F1=0#
IF N=0 AND X=0# THEN GOTO FIVE
IF N>=0 AND X<1D-2 THEN GOTO NINE
AN=CDBL(N)
TEST=ABS(AN)
IF X>15#*(TEST+1#) THEN GOTO SEVEN
IF AN>=20#*X THEN GOTO SIX
F1=.5#
H=F1
K=0
TEST=7.5#+.3#*TEST
IF X>TEST THEN GOTO THREE
IF N>=0 THEN GOTO ONE
N=-N
F1=.5#*COS(PI*AN)
H=F1
ONE:
K=K+1
F1=F1+H
H=-H*X^2/4#/CDBL(K)/CDBL(K+N)
F1=F1+H
IF ABS(F1/H)<2D7 THEN GOTO ONE
IF N=0 THEN GOTO FIVE
TWO:
N=N+1
AN=CDBL(N)
F1=2#*F1*AN/X
IF N<=3 THEN GOTO TWO
H=(1#+(2#/7#/AN^2)*(2#/3#/AN^2-1#))/30#/AN^3
H=(H-1#)/12#/AN+AN*(1#+LOG(X/2#/AN))
F1=F1*EXP(H)/SQR(2#*PI*AN)
GOTO FIVE
THREE:
T=SQR(2#/PI/X)
A=PI*(AN+H-(T*X)^2)/2#
F1=0#
FOUR:
F1=F1+T*COS(A)
T=T*(H^2-AN^2)/(2#*H+1#)/X
A=A+PI/2#
H=H+1#
IF H<ABS(AN) THEN GOTO FOUR
IF ABS(F1)>2D7*ABS(T) THEN GOTO FIVE
IF H^2<(2#*H+1#)*X+AN^2 THEN GOTO FOUR
FIVE:
BTEMP=F1
GOTO EIGHT
SEVEN:
BTEMP=SQR(2#/PI/X)*COS(X-PI*AN/2#-PI/4#)
GOTO EIGHT
SIX:
BTEMP=(EXP(1#)*X/2#/AN)^N/SQR(2#*PI*AN)
GOTO EIGHT
NINE:
BTEMP=(X/2#)^N/FACT(N)
EIGHT:
'
'  IF Y IS NEGATIVE, CORRECT SIGN OF BESSEL FUNCTION; OTHERWISE, RETURN.
'
IF Y>=0# THEN GOTO TEN
AM=CDBL(ORDER)
BTEMP=BTEMP*COS(PI*AM)
TEN:
BESSEL=BTEMP
END FUNCTION
'
'  THIS FUNCTION CALCULATES THE FACTORIAL OF N.  IT NEEDS THE FOLLOWING
' DECLARE STATEMENT.
'
'   DECLARE FUNCTION FACT#(N AS LONG)
'
FUNCTION FACT#(N AS LONG)
FTEMP=1#
IF N>0& THEN
FOR I=1& TO N
FTEMP=FTEMP*CDBL(N)
NEXT I
END IF
FACT=FTEMP
END FUNCTION
'
'  This subroutine performs an FFT of order N on data with real values
' stored in XREAL and imaginary values stored in XIMAG (shared arrays)
' The sampling interval for the data to be FFTed is DT.  A forward FFT is
' done for INV = 0 and an inverse one is done if INV = 1.
'
DEFINT I-N
SUB FFT(N AS INTEGER,INV AS INTEGER,DT AS DOUBLE)
SHARED XREAL() AS DOUBLE,XIMAG() AS DOUBLE
DIM AM AS DOUBLE,TR AS DOUBLE,TI AS DOUBLE,UR AS DOUBLE,UI AS DOUBLE
DIM WR AS DOUBLE,WI AS DOUBLE,UTEMPR AS DOUBLE,UTEMPI AS DOUBLE,PI AS DOUBLE
PI=4#*ATN(1#)
AM=LOG(CDBL(N))/LOG(2#)+.1#
M=INT(AM)
NV2=N/2
NM1=N-1
J=1
FOR I=1 TO NM1
IF I>=J THEN GOTO FFTTEN
TR=XREAL(J) : TI=XIMAG(J)
XREAL(J)=XREAL(I) : XIMAG(J)=XIMAG(I)
XREAL(I)=TR : XIMAG(I)=TI
FFTTEN:
K=NV2
TWENTY:
IF K>=J THEN GOTO THIRTY
J=J-K
K=K/2
GOTO TWENTY
THIRTY:
J=J+K
NEXT I
FOR L=1 TO M
LE=2^L
LE1=LE/2
UR=1# : UI=0#
WR=COS(PI/LE1) : WI=-SIN(PI/LE1)
IF INV<>0 THEN WI=-WI
FOR J=1 TO LE1
FOR I=J TO N STEP LE
IP=I+LE1
TR=XREAL(IP)*UR-XIMAG(IP)*UI : TI=XIMAG(IP)*UR+XREAL(IP)*UI
XREAL(IP)=XREAL(I)-TR : XIMAG(IP)=XIMAG(I)-TI
XREAL(I)=XREAL(I)+TR : XIMAG(I)=XIMAG(I)+TI
NEXT I
UTEMPR=UR*WR-UI*WI : UTEMPI=UI*WR+UR*WI
UR=UTEMPR : UI=UTEMPI
NEXT J
NEXT L
FOR I=1 TO N
XREAL(I)=XREAL(I)*DT : XIMAG(I)=XIMAG(I)*DT
NEXT I
END SUB
'
'  This function returns the sinc function for the input value X.  It
' needs the following DECLARE statement.
'
'   DECLARE FUNCTION SINC(X)
'
DEFSNG A-Z
FUNCTION SINC(X)
SINCTEMP=1
IF X<>0 THEN SINCTEMP=SIN(X)/X
SINC=SINCTEMP
END FUNCTION
'
'  This subroutine performs the standard rotational coordinate
' transformation.
'
SUB ROTATE(X,Y,ANGLE AS DOUBLE)
XTEMP=X*COS(ANGLE)-Y*SIN(ANGLE)
Y=X*SIN(ANGLE)+Y*COS(ANGLE)
X=XTEMP
END SUB
'
'  This subroutine puts the character string STRNG$ into the keyboard
' buffer.  If the last character in STRNG$ is a semicolon, a carriage
' return is not added to the end of the stacked data.  Otherwise, a
' carriage return is stacked after STRNG$ is stacked.  (Put two semicolons
' at the end of STRNG$ if you physically want to put a semicolon at the
' end of the buffer.)
'
SUB STACK(STRNG$)
DIM CH(94) AS INTEGER,CL AS INTEGER,C AS INTEGER,I AS INTEGER,L AS INTEGER
DIM CR AS INTEGER,OS AS INTEGER,STCODE(1 TO 5) AS INTEGER
'
'  Store scan codes.
'
CH(0)=&H39
CH(1)=&H02
CH(2)=&H28
CH(3)=&H04
CH(4)=&H05
CH(5)=&H06
CH(6)=&H08
CH(7)=&H28
CH(8)=&H0A
CH(9)=&H0B
CH(10)=&H09
CH(11)=&H0D
CH(12)=&H33
CH(13)=&H0C
CH(14)=&H34
CH(15)=&H35
CH(16)=&H0B
CH(17)=&H02
CH(18)=&H03
CH(19)=&H04
CH(20)=&H05
CH(21)=&H06
CH(22)=&H07
CH(23)=&H08
CH(24)=&H09
CH(25)=&H0A
CH(26)=&H27
CH(27)=&H27
CH(28)=&H33
CH(29)=&H0D
CH(30)=&H34
CH(31)=&H35
CH(32)=&H03
CH(33)=&H1E
CH(34)=&H30
CH(35)=&H2E
CH(36)=&H20
CH(37)=&H12
CH(38)=&H21
CH(39)=&H22
CH(40)=&H23
CH(41)=&H17
CH(42)=&H24
CH(43)=&H25
CH(44)=&H26
CH(45)=&H32
CH(46)=&H31
CH(47)=&H18
CH(48)=&H19
CH(49)=&H10
CH(50)=&H13
CH(51)=&H1F
CH(52)=&H14
CH(53)=&H16
CH(54)=&H2F
CH(55)=&H11
CH(56)=&H2D
CH(57)=&H15
CH(58)=&H2C
CH(59)=&H1A
CH(60)=&H2B
CH(61)=&H1B
CH(62)=&H07
CH(63)=&H0C
CH(64)=&H29
CH(65)=&H1E
CH(66)=&H30
CH(67)=&H2E
CH(68)=&H20
CH(69)=&H12
CH(70)=&H21
CH(71)=&H22
CH(72)=&H23
CH(73)=&H17
CH(74)=&H24
CH(75)=&H25
CH(76)=&H26
CH(77)=&H32
CH(78)=&H31
CH(79)=&H18
CH(80)=&H19
CH(81)=&H10
CH(82)=&H13
CH(83)=&H1F
CH(84)=&H14
CH(85)=&H16
CH(86)=&H2F
CH(87)=&H11
CH(88)=&H2D
CH(89)=&H15
CH(90)=&H2C
CH(91)=&H1A
CH(92)=&H2B
CH(93)=&H1B
CH(94)=&H29
'
'  Define machine code to call interrupt.
'
DEF SEG=VARSEG(STCODE(1))
OS=VARPTR(STCODE(1))
POKE OS,&HB4 : POKE OS+1,5             'MOV AH, 5
POKE OS+2,&HB5                         'MOV CH, [scan code]
'
'  Rest of above opcode will be POKEd in later.
'
POKE OS+4,&HB1                         'MOV CL, [ascii code]
'
'  Rest of above opcode will be POKEd in later.
'
POKE OS+6,&HCD : POKE OS+7,&H16        'INT 16
POKE OS+8,&HCB                         'RETF
L=LEN(STRNG$)                          'Get length of string to be stacked.
CR=1                                   'If last character in STRNG$ is ";",
IF RIGHT$(STRNG$,1)=";" THEN           'don't stack carriage return (and don't
L=L-1                                  'include semicolon in text to be stacked)
STRNG$=LEFT$(STRNG$,L)                 'unless there are two semicolons in a row
IF RIGHT$(STRNG$,1)<>";" THEN CR=0     'at the end.
END IF
IF L>0 THEN
IF CR=0 AND L>15 THEN L=15            'Don't let L be larger than buffer.
IF CR=1 AND L>14 THEN L=14
FOR I=1 TO L                          'Stack STRNG$ one character at a time.
CL=ASC(MID$(STRNG$,I,1))
IF CL=1 THEN CL=60                    'Allow redirection and pipe symbols to
IF CL=2 THEN CL=62                    'be stacked via "gimmick."
IF CL=4 THEN CL=124
IF CL>=32 AND CL<=126 THEN            'Watch out for non-ascii stuff.
C=CL-32
POKE OS+3,CH(C)                       'Finish CH definition.
POKE OS+5,CL                          'Finish CL definition.
CALL ABSOLUTE(OS)                     'Call interrupt.
END IF
NEXT I
END IF
IF CR<>0 THEN
POKE OS+3,&H1C                        'Send carriage return if appropriate to
POKE OS+5,13                          'do so.
CALL ABSOLUTE(OS)
END IF
DEF SEG
END SUB
