/***************************************************************************
 *  Copyright (C) 2009 Nippon Telegraph and Telephone Corporation
 ***************************************************************************/
/**
 * @file	target_db.c
 * @brief   stored procedure for monitoring target DB
 *
 *
 *   -- About "log-filtered file" and "log-filtered-done file" --
 *   We access the table in DB and flat text file during getting stanpshot.
 *   So, we got any problem on getting snapshot, we lost some information.
 *   Autovacuum and checkpoint information are filtered from stdout, and 
 *   store own file. The file will be referred to as "log-filtered file". 
 *   When we get information from "log-filtered file", we extract records 
 *   from these file and diplaying as VIEW on Postgresql and INSERT INTO 
 *   snapshot storing tables.
 *   So if we got any error, aborting the process and lost reading records.
 *   We avoid this situation, store extract records (already VIEWed) into
 *   temp file. The file will be referred to as "log-filtered-done file".
 *   We can save the reading records following example. 
 *   
 *   1. Log-filtered file has 3 records (A,B,C) and we perform stats_autovacuum.
 *
 *   2. 3 records are VIEWed and storing snapshot table.
 *      [log-filtered-file] -> record A,B,C -> [VIEW] -> [snapshot table]
 *
 *   3. All records in log-filtered file move to log-filtered-done file.
 *      [log-filtered-file] -> recoed A,B,C -> [log-filtered-done-file]
 *
 *   4. But unfortunately, we get error at this time... and abort.
 *
 *   5. Try stats_autovacuum again. BTW, Log-filtered-file gets new 2 records (D,E).
 *
 *   6. Reading 2 records in log-filtered file and reading 3 records in
 *      "log-filtered-done file" go on. And VIEWed and storing as a whole.
 *      [log-filtered-file] -> record D,E  --|
 *      [log-filtered-done-file] -> record A,B,C -> [VIEW] -> [snapshot table]
 *
 *   7. All records in log-filtered file move to log-filtered-done file.
 *      Then log-filtered-done file has 5 records (A,B,C,D,E).
 *      [log-filtered-file] -> recoed D,E -> [log-filtered-done-file]
 *
 *   8. OK. We done getting snapshot safely.(transaction end)
 *
 *   9. End of getting snapshot process, we delete log-filtered-done file.
 *
 *   If we got any error on '9', we must deletion log-filtered-done file 
 *   before getting snapshot for avoiding get duplicate information.
 *   But now it's not automatic operation but manual operation...
 */

#include "postgres.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/file.h> 
#include <unistd.h> 
#include <errno.h> 
#ifndef WIN32
#include <sys/utsname.h>
#endif

#include "access/heapam.h"
#include "catalog/pg_type.h"
#include "executor/spi.h"
#include "funcapi.h"
#include "lib/stringinfo.h"
#include "miscadmin.h"
#include "utils/builtins.h"
#include "utils/int8.h"

#include "target_db.h"
#include "target_db_message.h"

/* column attribure definition */
/* autovacuum information */
const AttrType avAttr[] = {
	{"autovacuum_act_time", TIMESTAMPTZOID},
	{"autovacuum_table", TEXTOID},
	{"autovacuum_num_page_remove", INT8OID},
	{"autovacuum_num_page_remain", INT8OID},
	{"autovacuum_num_tup_remove", INT8OID},
	{"autovacuum_num_tup_remain", INT8OID},
	{"autovacuum_duration", FLOAT4OID}
};

/* checkpoint information */
const AttrType cpAttr[] = {
	{"checkpoint_act_time", TIMESTAMPTZOID},
	{"checkpoint_start_trig", TEXTOID},
	{"checkpoint_num_buffers", INT8OID},
	{"checkpoint_create_wal", INT8OID},
	{"checkpoint_delete_wal", INT8OID},
	{"checkpoint_recycle_wal", INT8OID},
	{"checkpoint_write_duration", FLOAT4OID},
	{"checkpoint_sync_duration", FLOAT4OID},
	{"checkpoint_total_duration", FLOAT4OID}
};

/* CPU information (/proc/cpuinfo) */
const AttrType ciAttr[] = {
	{"model_name", TEXTOID},
	{"processors", INT4OID},
	{"cpu_MHz",    FLOAT4OID},
	{"bogomips",   FLOAT4OID}
};

/* Memory information (/proc/meminfo) */
const AttrType miAttr[] = {
	{"mem_total",   INT8OID},
	{"mem_free",    INT8OID},
	{"buffers",     INT8OID},
	{"cached",      INT8OID},
	{"active",      INT8OID},
	{"inactive",    INT8OID},
	{"swap_total",  INT8OID},
	{"swap_free",   INT8OID}
};


PG_MODULE_MAGIC;

PG_FUNCTION_INFO_V1(stats_autovacuuminfo);
PG_FUNCTION_INFO_V1(stats_checkpointinfo);
PG_FUNCTION_INFO_V1(stats_kernel_info);
PG_FUNCTION_INFO_V1(stats_cpu_info);
PG_FUNCTION_INFO_V1(stats_mem_info);

/**
 * @fn	  static int lock_pg_statslog(char *lock_file, const char *funcname)
 * @brief   locking for file access
 *
 *   When we access the filtered-log file, we get lock file for control exclusively
 *
 * @param[in]   lock_file  path to lock file
 * @param[in]   funcname   name of SP invoke
 * @return	  file descriptor of lock file. If not exists -1
 *
 */
static int
lock_pg_statslog(char *lock_file, const char *funcname)
{
	int fd;

	if ((fd = open(lock_file, O_RDWR, S_IRUSR | S_IWUSR)) == -1)
	{
		if (errno == ENOENT)
			return -1;
		else
			ereport(ERROR,
					(errcode_for_file_access(),
					 errmsg(MSG_TARGET_DB_1001, funcname, lock_file)));
	}
	flock(fd, LOCK_EX);

	return fd;
}

/**
 * @fn	  static char *get_lockfile_path(char *path_pgdata, const char *funcname)
 * @brief   getting path to lock file
 *
 *   Lock file exists in 'tmp' directory and the name is 'lock_pg_statsinfo.PID'.
 *   "PID" is same as postmaster pid.
 *
 * @param[in]   path_pgdata  path to $PGDATA
 * @param[in]   funcname     name of SP invoke
 * @return	absolute-path to lock file
 *
 */
static char *
get_lockfile_path(char *path_pgdata, const char *funcname)
{
	StringInfo lock_file;
	StringInfo pid_file;
	FILE *pid_fp = NULL;
	char  pid_str[READBUFSIZE];
	size_t pid_str_len;
	char *ccode;

	/* set absolute-path to postmaster.pid */
	pid_file = makeStringInfo();
	appendStringInfo(pid_file, "%s/%s", path_pgdata, "postmaster.pid");

	/* open postmaster.pid */
	pid_fp = fopen(pid_file->data, "r");
	if (pid_fp == NULL)
		ereport(ERROR,
				(errcode_for_file_access(),
				 errmsg(MSG_TARGET_DB_1001, funcname, pid_file->data)));

	/* get postmaster.pid */
	ccode = fgets(pid_str, READBUFSIZE, pid_fp);
	if (ccode == NULL)
		ereport(ERROR,
				(errcode_for_file_access(),
				 errmsg(MSG_TARGET_DB_1002, funcname, pid_file->data)));
	fclose(pid_fp);

	/* over write LF with '\0' */
	pid_str_len = strlen(pid_str);
	pid_str[pid_str_len - 1] = '\0';

	lock_file = makeStringInfo();
	appendStringInfo(lock_file, "%s.%s", LOCK_FILE_PREFIX, pid_str);

	return lock_file->data;
}

#if PG_VERSION_NUM < 80400
/**
 * @fn	  static text *cstring_to_text(char *str)
 * @brief Convert 'char *' to 'text'
 *
 *   Converts character string in an pointed address to text type.
 *   This routine is the same as cstring_to_text in PG8.4.
 *
 * @param[in]   str  String before converted
 * @return	Address which pointed text-type data after converted
 *
 */
static text *
cstring_to_text(char *str)
{
	text *new_text = NULL;
	int str_len;

	str_len = strlen(str);
	new_text = (text*) palloc(VARHDRSZ + str_len);
	SET_VARSIZE(new_text, str_len + VARHDRSZ);
	memcpy(VARDATA(new_text), str, str_len);

	return new_text;
}

static char *
text_to_cstring(const text *t)
{
        /* must cast away the const, unfortunately */
        text       *tunpacked = pg_detoast_datum_packed((struct varlena *) t);
        int                     len = VARSIZE_ANY_EXHDR(tunpacked);
        char       *result;

        result = (char *) palloc(len + 1);
        memcpy(result, VARDATA_ANY(tunpacked), len);
        result[len] = '\0';

        if (tunpacked != t)
                pfree(tunpacked);

        return result;
}


#endif

/**
 * @fn	  static char *select_pgdata()
 * @brief   getting path to $PGDATA
 *
 *
 * @return	  path to $PGDATA 
 *
 */
static char *
select_pgdata()
{
	char *pgdata = NULL;
	int   spi_result;
	int   result_num;

	if ((spi_result = SPI_connect()) != SPI_OK_CONNECT)
		/* internal error */
		ereport(ERROR, (errmsg(MSG_TARGET_DB_1003, "select_pgdata", spi_result)));

	/* getting DB path from 'pg_setting' table */
	spi_result = SPI_execute(Cmd_select_pgdata, true, 1);
	result_num = SPI_processed;

	if ((spi_result == SPI_OK_SELECT) && (result_num > 0))
	{
		char *val_pgdata;

		val_pgdata = SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1);

		pgdata = (char *)SPI_palloc(strlen(val_pgdata) + 1 + 1024*102);
		memcpy(pgdata, val_pgdata, strlen(val_pgdata)+1);
	}
	SPI_finish();

	return pgdata;
}
/**
 * @fn	  int create_read_file_info(FileInfo **read_file_info, TupleDesc tupdesc,
 *					   char delimiter, char *path, char *fullpath, const char *open_mode, int lock_fd)
 * @brief   create read-file information
 *
 *   Create File info struct, and set some val (delimiter, file path...)
 *   We access some text-files which differ in delimiter and column, so we set 
 *   necessary information in this routine.
 *
 * @param[out]  read_file_info   address to created File info
 * @param[in]   tupdesc		 tuple descriptor
 * @param[in]   delimiter	 delimiter of own file
 * @param[in]   path		 relative path to target file
 * @param[in]   fullpath	 absolute path to target filE
 * @param[in]   open_mode	 open mode using fopen()
 * @param[in]   lock_fd		 lock file descriptor
 * @return	success 0. If failed on fopen(), then the errno
 *
 */
static int
create_read_file_info(
	FileInfo **read_file_info, TupleDesc tupdesc, char delimiter,
	char *path, char *fullpath, const char *open_mode, int lock_fd)
{
	FileInfo *rf_info = NULL;
	int rcode = 0;

	rf_info = (FileInfo *) palloc(sizeof(FileInfo));
	rf_info->tupdesc = tupdesc;
	rf_info->delimiter = delimiter;
	rf_info->path = path;
	rf_info->fullpath = fullpath;
	rf_info->lock_fd = lock_fd;
	rf_info->cur_line_no = 0;

	rf_info->fp = fopen(fullpath, open_mode);
	if (rf_info->fp == NULL)
		if (errno != ENOENT)
		{
			rf_info = NULL;
			rcode = errno;
		}

	*read_file_info = rf_info;

	return rcode;
}

/**
 * @fn	  static char **read_file(FileInfo *rf_info, int natts, bool const_size, const char *funcname)
 * @brief   reading one-record from file
 *
 *   This routine using for reading one-record from specified text-file.
 *   This is common routine on getting autovacuum, checkpoint, and OS info.
 *
 * @param[in]   rf_info  FileInfo for readinf file
 * @param[in]   natts  # of factors in array(max token number of the reading record)
 * @param[in]   const_size  true if token number is static, else false
 * @param[in]   funcname  name of SP invoke
 * @return	address of string arrays (null-terminated if the file is valid)
 *
 */
static char **
read_file(FileInfo *rf_info, int natts, bool const_size, const char *funcname)
{
	char **cols;
	StringInfo token;
	int c;
	int token_num = 0; 

	if ((rf_info == NULL) || (rf_info->fp == NULL))
	{
		unlock_pg_statslog(rf_info->lock_fd);
		ereport(ERROR,
				(errcode_for_file_access(),
				 errmsg(MSG_TARGET_DB_1002, funcname, rf_info->path)));
	}

	/*
	 * create address list for storing tokens 
	 *   - Checking token number, we do extra alloc (+1). 
	 */
	cols = palloc(sizeof(char *) * (natts + 1));
	MemSet(cols, '\0', sizeof(char *) * (natts + 1));

	c = getc(rf_info->fp);
	while ((c != EOF) && (c != '\n') && (token_num <= natts))
	{
		token = makeStringInfo();

		/* Extract tokens */
		while ((c != EOF) &&
			   (c != rf_info->delimiter) &&
			   (c != '\n'))
		{
			/* we continue concatenating string until reading EOF, delimiter, or '\n' */
			appendStringInfoChar(token, (char)c);

			c = getc(rf_info->fp);
		}

		/* reading delimiter（OK. finish reading one token） */
		if (c == rf_info->delimiter)
		{
			cols[token_num] = token->data;
			token_num++;
			c = getc(rf_info->fp);
		}
		/* reading '\n'（OK. finish reading one record） */
		else if (c == '\n')
		{
			cols[token_num] = token->data; 
			token_num++;
			rf_info->cur_line_no++;	
			break;
		}
		/* reading EOF（we read nothing） */
		else
			break;
	}

	/* if natts < numn of token, it's unordinary */
	if (token_num > natts)
	{
		fclose(rf_info->fp);
		unlock_pg_statslog(rf_info->lock_fd);
		ereport(ERROR,
				(errmsg(MSG_TARGET_DB_1007, funcname, rf_info->path, rf_info->cur_line_no)));
	}
	/* 
	 * We access text files which have arbitrary number of token. So if we get 
	 * natts > num of token, checking the num of token is static or not.
	 */
	else if ((token_num < natts) && (token_num > 0))
	{
		/* If the numn of token is static, then abnormal exit */
		if (const_size == true)
		{
			fclose(rf_info->fp);
			unlock_pg_statslog(rf_info->lock_fd);
			ereport(ERROR,
					(errmsg(MSG_TARGET_DB_1007, funcname, rf_info->path, rf_info->cur_line_no)));
		}
	}
	/* If token_num is 0, we have to check whether the record has no info or EOF */
	else if (token_num == 0)
	{
		/* If reading record has only EOF, retrun NULL */
		if (c == EOF)
			return NULL;
	}

	return cols;
}
/**
 * @fn	  int concat_logfile(FileInfo *rf_info, FileInfo *wf_info)
 * @brief   concatenating text files
 *
 *   This routine using for adding filtered-log file records to log-filtered-done file.
 *   Normally we don't have log-filtered-done file, so can skip this routine.
 *
 * @param[in]   rf_info  input file info
 *					   If the input file is opened, the mode must 'r'.
 * @param[in]   wf_info  output file info
 *					   output file have to be opened with 'w' mode.
 * @return	normally    ... RCODE_OK
 *		read error  ... RCODE_NG_READ
 *		write error ... RCODE_NG_WRITE
 *
 */
static int
concat_logfile(FileInfo *rf_info, FileInfo *wf_info)
{
	int rcode;
	size_t read_size = 0;
	size_t write_size = 0;
	char buf[READBUFSIZE];

	/* Normally we don't have input-file(log-filtered-done file), so return quickly.*/
	if (rf_info->fp == NULL)
		return RCODE_OK;

	/* rewind pointer in input-file and forward pointer in output file */
	rewind(rf_info->fp);

	rcode = fseek(wf_info->fp, 0, SEEK_END);
	if (rcode != 0)
		return RCODE_NG_WRITE;

	/*
	 * read one record from input-file and write in output-file
	 */
	read_size = fread(buf, sizeof(char), READBUFSIZE, rf_info->fp);
	while (read_size > 0)
	{
		write_size = fwrite(buf, sizeof(char), read_size, wf_info->fp);
		if (write_size < read_size)
			return RCODE_NG_WRITE;

		read_size = fread(buf, sizeof(char), READBUFSIZE, rf_info->fp);
	}

	if (ferror(rf_info->fp) != 0)
		return RCODE_NG_READ;

	return RCODE_OK;
}

/**
 * @fn	  Datum stats_kernel_info()
 * @brief   getting kernel info
 *
 *   Get kernel info using uname(), and diplat as VIEW.
 *
 * @param[in]    PG_FUNCTION_ARGS  not use
 * @return	 some kernel info
 *
 */
Datum
stats_kernel_info(PG_FUNCTION_ARGS)
{
	char    result[1024];
	struct utsname name;	/* storing area for getting uname() info */
	int rcode;

	rcode = uname(&name);
	if (rcode == -1)
	{
		char *str_errmsg = strerror(errno);
		ereport(ERROR, (errmsg(MSG_TARGET_DB_1004, SP_NAME_STATS_HOSTINFO, str_errmsg)));
	}

	sprintf(result, "%s %s %s %s %s", name.sysname,
					name.nodename,
					name.release,
					name.version,
					name.machine);

	PG_RETURN_TEXT_P(cstring_to_text(result));
}

/**
 * @fn	  Datum stats_cpu_info()
 * @brief   getting cpu info
 *
 *   We get CPU information from '/proc/cpuinfo', and display as VIEW.
 *   Now we get following information.
 *
 *	 - Num of CPU (logical) ... the record has key which string is 'processor'
 *	 - Model name
 *	 - Clock number
 *	 - MIPs  value
 *
 *
 * @param[in]    PG_FUNCTION_ARGS  no use
 * @return	 some cpu info
 *
 */
Datum
stats_cpu_info(PG_FUNCTION_ARGS)
{
        Datum  result; 
	FileInfo  readFileInfo;
	FILE 	*fp;
	char 	**columns;
	char    **columns_res;
	char	*key;
	char	*value;

	text    *key_text;
	text    *value_text;

	char	cpu_model[1024];
	double	cpu_mhz  = 0;
	double	bogomips = 0;
	int	cpu_num  = 0;

        int  cntattr; 
        TupleDesc  tupdesc; 
        HeapTuple  tuple; 


	fp = fopen(path_cpuinfo, "r");
	if (fp == NULL)
		ereport(ERROR,
				(errcode_for_file_access(),
				 errmsg(MSG_TARGET_DB_1001, SP_NAME_STATS_HOSTINFO, path_cpuinfo)));

	readFileInfo.fp = fp;
	readFileInfo.delimiter = ':';
	readFileInfo.lock_fd = -1;
	readFileInfo.path = path_cpuinfo;
	readFileInfo.fullpath = path_cpuinfo;

	cpu_num = 0;

	/*
	 * continue reading until EOF.
	 * If you want to add another item, edit following loop inside.
	 * (Note: /proc/cpuinfo has duplicative info, so following loop is wordily logic.
	 *  But there is no idea... 
	 */
	while ((columns = read_file(&readFileInfo, 2, false, SP_NAME_STATS_HOSTINFO)) != NULL)
	{
		if ((columns[0] == NULL) || (columns[1] == NULL))
			continue;

		/* triming for taken string */
		key_text = trim(cstring_to_text(columns[0]), cstring_to_text(TRIM_TRGT));
		value_text = trim(cstring_to_text(columns[1]), cstring_to_text(TRIM_TRGT));

		key = text_to_cstring(key_text);
		value = text_to_cstring(value_text);

                /* get model info */
                if (strncmp(key, KEY_CPUINFO_MODEL, strlen(KEY_CPUINFO_MODEL)) == 0)
                        strcpy(cpu_model, value);

		/* get Clock info */
		if (strncmp(key, KEY_CPUINFO_MHZ, strlen(KEY_CPUINFO_MHZ)) == 0)
			cpu_mhz = atof(value);

                /* get MIPS value */
                if (strncmp(key, KEY_CPUINFO_BOGOMIPS, strlen(KEY_CPUINFO_BOGOMIPS)) == 0)
                        bogomips = atof(value);
	
                /* check Num of CPU */
		if (strncmp(key, KEY_CPUINFO_PROCESSOR, strlen(KEY_CPUINFO_PROCESSOR)) == 0)
			cpu_num++;
	}

        fclose(fp);

        tupdesc = CreateTemplateTupleDesc(ncpuinfo, false);
        for (cntattr = 0; cntattr < ncpuinfo; cntattr++)
                TupleDescInitEntry(
                        tupdesc, (AttrNumber)(cntattr + 1),
                        ciAttr[cntattr].name, ciAttr[cntattr].oid, -1, 0);


        columns_res = palloc(sizeof(char *) * ncpuinfo);
        columns_res[1] = palloc(sizeof(int));
        columns_res[2] = palloc(sizeof(double));
        columns_res[3] = palloc(sizeof(double));

        columns_res[0] = cpu_model;
        sprintf(columns_res[1], "%d", cpu_num);
        sprintf(columns_res[2], "%.2f", cpu_mhz);
        sprintf(columns_res[3], "%.2f", bogomips);

        tuple = BuildTupleFromCStrings(TupleDescGetAttInMetadata(tupdesc), columns_res);
        result = HeapTupleGetDatum(tuple);

        PG_RETURN_DATUM(result);
}

/**
 * @fn	  Datum stats__mem_info()
 * @brief   getting memory info
 *
 *   We get CPU information from '/proc/meminfo', and display as VIEW.
 *   Now we get following information.
 *	 - Physical total memory size (MemTotal)
 *	 - Physical free  memory size (MemFree)
 *       - Buffered memory size       (Buffers)
 *       - Cached memory size         (Cached)
 *       - Active memory size         (Active)
 *       - Inactive   memory size     (Inactive)
 *	 - Swap total memory size     (SwapTotal)
 *	 - Swap free  memory size     (SwapFree)
 *
 * @return  some mem info
 *
 */

Datum
stats_mem_info(PG_FUNCTION_ARGS)
{
        Datum  result ; 
	FileInfo  readFileInfo;	
	FILE  *fp;
	char  **columns;
	char  **columns_res;
	char  *key;
	char  *value;
	text  *key_text;
	text  *value_text;
	int i;
	
	int64  mem_total = 0;
        int64  mem_free = 0;
        int64  mem_buffers = 0;
        int64  mem_cached = 0;
        int64  mem_active = 0;
        int64  mem_inactive = 0;
	int64  swap_total = 0;
        int64  swap_free = 0;

        int  cntattr; 
        TupleDesc  tupdesc;
        HeapTuple  tuple; 
	AttInMetadata *attinmeta;

	fp = fopen(path_meminfo, "r");
	if (fp == NULL)
		ereport(ERROR,
				(errcode_for_file_access(),
				 errmsg(MSG_TARGET_DB_1001, SP_NAME_STATS_HOSTINFO, path_meminfo)));

	readFileInfo.fp = fp;
	readFileInfo.delimiter = ':';
	readFileInfo.lock_fd = -1;
	readFileInfo.path = path_meminfo;
	readFileInfo.fullpath = path_meminfo;

	/* If you want to add new item, edit following loop inside */
	while ((columns = read_file(&readFileInfo, 2, false, SP_NAME_STATS_HOSTINFO)) != NULL)
	{
		if ((columns[0] == NULL) || (columns[1] == NULL))
			continue;

		key=NULL;
		value=NULL;

		/* trriming for taken string */
		key_text = trim(cstring_to_text(columns[0]), cstring_to_text(TRIM_TRGT));
		value_text = trim(cstring_to_text(columns[1]), cstring_to_text(TRIM_TRGT_KB));

		key = text_to_cstring(key_text);
		value = text_to_cstring(value_text);


		if (strncmp(key, KEY_MEMINFO_MEMTOTAL, strlen(KEY_MEMINFO_MEMTOTAL)) == 0)
		{
                        if (scanint8(value, true, &mem_total) != true)
                                ereport(ERROR,
					(errcode_for_file_access(),
					 errmsg("could not scan %s as mem_total", value)));
		}
                else if (strncmp(key, KEY_MEMINFO_MEMFREE, strlen(KEY_MEMINFO_MEMFREE)) == 0)
                {
                        if (scanint8(value ,true, &mem_free) != true)
                                ereport(ERROR,
					(errcode_for_file_access(),
					 errmsg("could not scan %s as mem_free", value)));
                }
                else if (strncmp(key, KEY_MEMINFO_BUFFERS, strlen(KEY_MEMINFO_BUFFERS)) == 0)
                {
                        if (scanint8(value, true, &mem_buffers) != true)
                                ereport(ERROR,
					(errcode_for_file_access(),
					 errmsg("could not scan %s as mem_buffers", value)));
                }
                else if (strncmp(key, KEY_MEMINFO_CACHED, strlen(KEY_MEMINFO_CACHED)) == 0)
                {
                        if (scanint8(value, true, &mem_cached) != true)
                                ereport(ERROR,
					(errcode_for_file_access(),
					 errmsg("could not scan %s as mem_cached", value)));
                }
                else if (strncmp(key, KEY_MEMINFO_ACTIVE, strlen(KEY_MEMINFO_ACTIVE)) == 0)
                {
                        if (scanint8(value, true, &mem_active) != true)
                                ereport(ERROR,
					(errcode_for_file_access(),
					 errmsg("could not scan %s as mem_active", value)));
                }
                else if (strncmp(key, KEY_MEMINFO_INACTIVE, strlen(KEY_MEMINFO_INACTIVE)) == 0)
                {
                        if (scanint8(value, true, &mem_inactive) != true)
                                ereport(ERROR,
					(errcode_for_file_access(),
					 errmsg("could not scan %s as mem_inactive", value)));
                }
		else if (strncmp(key, KEY_MEMINFO_SWAPTOTAL, strlen(KEY_MEMINFO_SWAPTOTAL)) == 0)
		{
                        if (scanint8(value, true, &swap_total) != true)
                                ereport(ERROR,
					(errcode_for_file_access(),
					 errmsg("could not scan %s as swap_total", value)));
		}
                else if (strncmp(key, KEY_MEMINFO_SWAPFREE, strlen(KEY_MEMINFO_SWAPFREE)) == 0)
                {
                        if (scanint8(value, true, &swap_free) != true)
                                ereport(ERROR,
					(errcode_for_file_access(),
					 errmsg("could not scan %s as swap_free", value)));
                }
	}

	fclose(fp);
        
	tupdesc = CreateTemplateTupleDesc(nmeminfo, false);
        for (cntattr = 0; cntattr < nmeminfo; cntattr++)
                TupleDescInitEntry(
                        tupdesc, (AttrNumber)(cntattr + 1),
                        miAttr[cntattr].name, miAttr[cntattr].oid, -1, 0);

	attinmeta = TupleDescGetAttInMetadata(tupdesc);

        columns_res = palloc(sizeof(char *) * nmeminfo);
	
	for(i=0;i<nmeminfo;i++){
		columns_res[i] = palloc0(LITOSTR_BUFSZ);
	}

	sprintf(columns_res[0], INT64_FORMAT, mem_total);
	sprintf(columns_res[1], INT64_FORMAT, mem_free);
	sprintf(columns_res[2], INT64_FORMAT, mem_buffers);
	sprintf(columns_res[3], INT64_FORMAT, mem_cached);
	sprintf(columns_res[4], INT64_FORMAT, mem_active);
	sprintf(columns_res[5], INT64_FORMAT, mem_inactive);
	sprintf(columns_res[6], INT64_FORMAT, swap_total);
	sprintf(columns_res[7], INT64_FORMAT, swap_free);

        tuple = BuildTupleFromCStrings(attinmeta, columns_res);

        result = HeapTupleGetDatum(tuple);

        PG_RETURN_DATUM(result);
}

/**
 * @fn	  read_filteredlog(FuncCallContext *funcctx, const char *funcname)
 * @brief   reading text-file and creating tuple
 *
 *   Read target text file and build HeapTuple for displaying VIEW.
 *   We read two file. log-filtered file and log-filtered-done file.
 *   So we concatenate these file. Finally, we deletion log-filtered file. 
 *
 * @param[in]   funcctx   function context
 * @param[in]   funcname  name of function invoke
 * @return	VIEW
 *
 */
static Datum
read_filteredlog(FuncCallContext *funcctx, const char *funcname)
{
	ReadLogFileInfo  *readLogInfo;
	FileInfo	*readFileInfo;
	char		**columns;
	int		rcode;

	readLogInfo = (ReadLogFileInfo *)funcctx->user_fctx;

	/*
	 * discriminate target file
	 *   - If trgt_cd is '0' , filtered-log file. Or filtered-log-done file.
	 */
	readFileInfo = (readLogInfo->trgt_cd == READ_TRGT_IS_LOG_FILE)
				 ? readLogInfo->log_info : readLogInfo->done_info;

	/*
	 * reading records.
	 *   - We keep lock(/tmp/xxx.lock) until finish reading all records in two files 
	 *   - On failed, release lock and abnormal exit.
	 */
	while ((readLogInfo->trgt_cd == READ_TRGT_IS_LOG_FILE) ||
		   (readLogInfo->trgt_cd == READ_TRGT_IS_DONE_FILE))
	{
		if ((columns = read_file(readFileInfo, readFileInfo->tupdesc->natts, true, funcname)))
		{
			Datum			result;
			HeapTupleData	tupleData;
			HeapTuple		tuple = &tupleData;

			tuple = BuildTupleFromCStrings(funcctx->attinmeta, columns);
			result = TupleGetDatum(TupleDescGetSlot(readFileInfo->tupdesc), tuple);

			return result;
		}
		else
		{
			/* we reach end of file .. */
			if (readLogInfo->trgt_cd == 0)
			{
				/* If past analysis file is 'log-filtered file', switch log-filtered-done file. */
				readLogInfo->trgt_cd = READ_TRGT_IS_DONE_FILE;
				readFileInfo = readLogInfo->done_info;
			}
			else
			{
				/*
				 * If past analysis file is 'log-filtered-done file', adding record in
				 * filtered-log file to filtered-log-done file.
				 */
				rcode = concat_logfile(readLogInfo->log_info, readLogInfo->done_info);
				if (rcode == RCODE_OK)
				{
					/* OK. All records in log-filtered file adding finish */
					break;
				}
				else
				{
					fclose_logfile(readLogInfo->log_info->fp);
					fclose_logfile(readLogInfo->done_info->fp);
					unlock_pg_statslog(readLogInfo->log_info->lock_fd);
					if (rcode == RCODE_NG_READ)
						ereport(ERROR,
							(errcode_for_file_access(),
							 errmsg(MSG_TARGET_DB_1002, funcname, readLogInfo->log_info->path)));
					else
						ereport(ERROR,
							(errcode_for_file_access(),
							 errmsg(MSG_TARGET_DB_1008, funcname, readLogInfo->done_info->path)));
				}
			}
		}
	}

	fclose_logfile(readLogInfo->log_info->fp);
	fclose_logfile(readLogInfo->done_info->fp);

	/*
	 * Deletion filtered-log file.
	 */
	rcode = unlink(readLogInfo->log_info->fullpath);
	if ((rcode != 0) && (errno != ENOENT))
	{
		unlock_pg_statslog(readFileInfo->lock_fd);
		ereport(ERROR,
				(errcode_for_file_access(),
				 errmsg(MSG_TARGET_DB_1006, funcname, readLogInfo->log_info->path)));
	}

	return (Datum)NULL;
}

/**
 * @fn	  Datum stats_autovacuuminfo(PG_FUNCTION_ARGS)
 * @brief   create autovacuum info VIEW
 *
 *   Create autovacuun information as VIEW. 
 *   If we encountered any crash in past getting snapshot, we reread
 *   lost records from log-filtered-done file.
 *   The path to reading target file is following：
 *
 *     PGDATA/pg_statspack/DB_name-autovacuum.log
 *     PGDATA/pg_statspack/DB_name-autovacuum.log.done
 *
 * @param[in]   PG_FUNCTION_ARGS  not use
 * @return	autovacuum info VIEW
 *
 */
Datum
stats_autovacuuminfo(PG_FUNCTION_ARGS)
{
	FuncCallContext  *funcctx;
	ReadLogFileInfo  *readLogInfo; 
	int		rcode;
	Datum		result;

	if (SRF_IS_FIRSTCALL())
	{
		MemoryContext  oldcontext;
		TupleDesc  tupdesc;
		char  *path_pgdata;
		char  *lock_file;
		char  *dbname;
		int  cntattr;
		StringInfo str_path_log;
		StringInfo str_fullpath_log;
		StringInfo str_path_done;
		StringInfo str_fullpath_done;
		int lock_fd = -1;

		funcctx = SRF_FIRSTCALL_INIT();
		oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);

		dbname = get_dbname();

		/* (Note: it's noisy?) */
		ereport(NOTICE, (errmsg(MSG_TARGET_DB_0001, SP_NAME_STATS_AUTOVACUUMINFO, dbname)));

		tupdesc = CreateTemplateTupleDesc(nautovacuuminfo, false);
		for (cntattr = 0; cntattr < nautovacuuminfo; cntattr++)
			TupleDescInitEntry(
					tupdesc, (AttrNumber)(cntattr + 1),
					avAttr[cntattr].name, avAttr[cntattr].oid, -1, 0);

		funcctx->attinmeta = TupleDescGetAttInMetadata(tupdesc);

		/* alloc areas for next calling */
		readLogInfo = (ReadLogFileInfo *) palloc(sizeof(ReadLogFileInfo));
		readLogInfo->log_info  = (FileInfo *) palloc(sizeof(FileInfo));
		readLogInfo->done_info = (FileInfo *) palloc(sizeof(FileInfo));

		/* set path to $PGDATA  */
		path_pgdata = select_pgdata();
		if (path_pgdata == NULL)
			ereport(ERROR, (errmsg(MSG_TARGET_DB_1005, SP_NAME_STATS_AUTOVACUUMINFO)));

		/* get lock for exclusive control to access autovacuum logs */
		lock_file = get_lockfile_path(path_pgdata, SP_NAME_STATS_AUTOVACUUMINFO);
		lock_fd = lock_pg_statslog(lock_file, SP_NAME_STATS_AUTOVACUUMINFO);

		/*
		 * set path to log-filtered file of autovacuum info
		 */
		str_path_log = makeStringInfo();
		appendStringInfo(str_path_log, "%s/%s-%s", dir_statslog, dbname, file_autovacuuminfo);
		str_fullpath_log = makeStringInfo();
		appendStringInfo(str_fullpath_log, "%s/%s/%s-%s",
				path_pgdata, dir_statslog, dbname, file_autovacuuminfo);

		rcode = create_read_file_info(
				&(readLogInfo->log_info), tupdesc, Delimiter,
				str_path_log->data, str_fullpath_log->data, "r", lock_fd);
		if (rcode != 0)
		{
			unlock_pg_statslog(lock_fd);
			MemoryContextSwitchTo(oldcontext);
			ereport(ERROR,
				(errcode_for_file_access(),
				 errmsg(MSG_TARGET_DB_1001, SP_NAME_STATS_AUTOVACUUMINFO, str_path_log->data)));
		}

		/*
		 * set path to log-filtered-done file of autovacuum info.
		 * Ordinary the file is not exist, so the file->fp is NULL.
		 */
		str_path_done = makeStringInfo();
		appendStringInfo(str_path_done, "%s/%s-%s%s", dir_statslog, dbname, file_autovacuuminfo, suffix_done);
		str_fullpath_done = makeStringInfo();
		appendStringInfo(str_fullpath_done, "%s/%s/%s-%s%s",
				path_pgdata, dir_statslog, dbname, file_autovacuuminfo, suffix_done);

		rcode = create_read_file_info(
				&(readLogInfo->done_info), tupdesc, Delimiter,
				str_path_done->data, str_fullpath_done->data, "a+", lock_fd);
		if (rcode != 0)
		{
			unlock_pg_statslog(lock_fd);
			MemoryContextSwitchTo(oldcontext);
			ereport(ERROR,
				(errcode_for_file_access(),
				 errmsg(MSG_TARGET_DB_1001, SP_NAME_STATS_AUTOVACUUMINFO, str_path_done->data)));
		}

		/*
		 * decision which we read.
		 * First, we check log-filtered file existence.
		 */
		if (readLogInfo->log_info->fp != NULL)
			/* OK. detect log-filtered file. set identification flag */
			readLogInfo->trgt_cd = READ_TRGT_IS_LOG_FILE;
		/* Second, we check log-filtered-done file existence */ 
		else if (readLogInfo->done_info->fp != NULL)
			/* OK. detect log-filtered-done file. set identification flag */
			readLogInfo->trgt_cd = READ_TRGT_IS_DONE_FILE;
		else
		{
			/* We have no file about autovacuum. Return 0 record */
			unlock_pg_statslog(lock_fd);
			MemoryContextSwitchTo(oldcontext);
			SRF_RETURN_DONE(funcctx);
		}

		funcctx->user_fctx = (void *)readLogInfo;
		MemoryContextSwitchTo(oldcontext);
	}

	funcctx = SRF_PERCALL_SETUP();

	/*
	 * create view from reading records
	 */
	result = read_filteredlog(funcctx, SP_NAME_STATS_AUTOVACUUMINFO);
	if (result != (Datum)NULL)
		SRF_RETURN_NEXT(funcctx, result);

	/*
	 * Unlock
	 */
	readLogInfo = (ReadLogFileInfo *)funcctx->user_fctx;
	unlock_pg_statslog(readLogInfo->log_info->lock_fd);

	SRF_RETURN_DONE(funcctx);
}

/**
 * @fn	  Datum stats_checkpointinfo(PG_FUNCTION_ARGS)
 * @brief   Create checkpoint information as VIEW
 *
 *   Create checkpoint information as VIEW.
 *   If we encountered any crash in past getting snapshot, we reread
 *   lost records from log-filtered-done file. 
 *
 *	 PGDATA/pg_statspack/checkpoint.log
 *	 PGDATA/pg_statspack/checkpoint.log.done
 *
 * @param[in]   PG_FUNCTION_ARGS  not use
 * @return	checkpoint info VIEW
 *
 */
Datum
stats_checkpointinfo(PG_FUNCTION_ARGS)
{
	FuncCallContext  *funcctx;
	ReadLogFileInfo  *readLogInfo;
	int		rcode;
	Datum		result;

	if (SRF_IS_FIRSTCALL())
	{
		MemoryContext  oldcontext;
		TupleDesc  tupdesc;
		char  *path_pgdata;
		char  *lock_file;
		int  cntattr;
		StringInfo str_path_log;
		StringInfo str_fullpath_log;
		StringInfo str_path_done;
		StringInfo str_fullpath_done;
		int lock_fd = -1;

		funcctx = SRF_FIRSTCALL_INIT();
		oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);

		/* (Note: it's noisy?) */
		ereport(NOTICE, (errmsg(MSG_TARGET_DB_0002, SP_NAME_STATS_CHECKPOINTINFO)));

		tupdesc = CreateTemplateTupleDesc(ncheckpointinfo, false);
		for (cntattr = 0; cntattr < ncheckpointinfo; cntattr++)
			TupleDescInitEntry(
				tupdesc, (AttrNumber)(cntattr + 1),
				cpAttr[cntattr].name, cpAttr[cntattr].oid, -1, 0);

		funcctx->attinmeta = TupleDescGetAttInMetadata(tupdesc);

		/* alloc areas for next calling */
		readLogInfo = (ReadLogFileInfo *) palloc(sizeof(ReadLogFileInfo));
		readLogInfo->log_info  = (FileInfo *) palloc(sizeof(FileInfo));
		readLogInfo->done_info = (FileInfo *) palloc(sizeof(FileInfo));

		/* set path to $PGDATA */
		path_pgdata = select_pgdata();
		if (path_pgdata == NULL)
			ereport(ERROR, (errmsg(MSG_TARGET_DB_1005, SP_NAME_STATS_CHECKPOINTINFO)));

		/* get lock for exclusive control to access checkpoint logs */
		lock_file = get_lockfile_path(path_pgdata, SP_NAME_STATS_CHECKPOINTINFO);
		lock_fd = lock_pg_statslog(lock_file, SP_NAME_STATS_CHECKPOINTINFO);

		/*
		 * set path to log-filtered file of checkpoint info.
		 */
		str_path_log = makeStringInfo();
		appendStringInfo(str_path_log, "%s/%s", dir_statslog, file_checkpointinfo);
		str_fullpath_log = makeStringInfo();
		appendStringInfo(str_fullpath_log, "%s/%s/%s",
					path_pgdata, dir_statslog, file_checkpointinfo);

		rcode = create_read_file_info(
				&(readLogInfo->log_info), tupdesc, Delimiter,
				str_path_log->data, str_fullpath_log->data, "r", lock_fd);
		if (rcode != 0)
		{
			unlock_pg_statslog(lock_fd);
			MemoryContextSwitchTo(oldcontext);
			ereport(ERROR,
				(errcode_for_file_access(),
				 errmsg(MSG_TARGET_DB_1001, SP_NAME_STATS_CHECKPOINTINFO, str_path_log->data)));
		}

		/*
		 * set path to log-filtered-done file of chckpoint info.
		 * Ordinary the file is not exist, so the file->fp is NULL.
		 */
		str_path_done = makeStringInfo();
		appendStringInfo(str_path_done, "%s/%s%s", dir_statslog, file_checkpointinfo, suffix_done);
		str_fullpath_done = makeStringInfo();
		appendStringInfo(str_fullpath_done, "%s/%s/%s%s",
				path_pgdata, dir_statslog, file_checkpointinfo, suffix_done);

		rcode = create_read_file_info(
				&(readLogInfo->done_info), tupdesc, Delimiter,
				str_path_done->data, str_fullpath_done->data, "a+", lock_fd);
		if (rcode != 0)
		{
			unlock_pg_statslog(lock_fd);
			MemoryContextSwitchTo(oldcontext);
			ereport(ERROR,
				(errcode_for_file_access(),
				 errmsg(MSG_TARGET_DB_1001, SP_NAME_STATS_CHECKPOINTINFO, str_path_done->data)));
		}

		/*
		 * decision which we read.
		 * First, we check log-filtered file existence.
		 */
		if (readLogInfo->log_info->fp != NULL)
			/* OK. detect log-filtered file. set identification flag */
			readLogInfo->trgt_cd = READ_TRGT_IS_LOG_FILE;
		/* Second, we check log-filtered-done file existence */
		else if (readLogInfo->done_info->fp != NULL)
			/* OK. detect log-filtered-done file. set identification flag */
			readLogInfo->trgt_cd = READ_TRGT_IS_DONE_FILE;
		else
		{
			/* We have no file about checkpoint. Return 0 record */
			unlock_pg_statslog(lock_fd);
			MemoryContextSwitchTo(oldcontext);
			SRF_RETURN_DONE(funcctx);
		}

		funcctx->user_fctx = (void *)readLogInfo;
		MemoryContextSwitchTo(oldcontext);
	}

	funcctx = SRF_PERCALL_SETUP();

	/*
	 * create view from reading records
	 */
	result = read_filteredlog(funcctx, SP_NAME_STATS_CHECKPOINTINFO);
	if (result != (Datum)NULL)
		SRF_RETURN_NEXT(funcctx, result);

	/*
	 * Unlock
	 */
	readLogInfo = (ReadLogFileInfo *)funcctx->user_fctx;
	unlock_pg_statslog(readLogInfo->log_info->lock_fd);

	SRF_RETURN_DONE(funcctx);
}
