//////////////////////////////////////////////////////////////////////////
//
// pgScript - PostgreSQL Tools
// RCS-ID:      $Id: pgsApplication.h,v 1.2 2008/08/10 17:45:38 pgunittest Exp $
// Copyright (C) 2002 - 2008, The pgAdmin Development Team
// This software is released under the Artistic Licence
//
//////////////////////////////////////////////////////////////////////////


#ifndef PGSAPP_H_
#define PGSAPP_H_

#include "pgscript/pgScript.h"
#include "pgscript/utilities/pgsThread.h"

class pgsApplication
{
	
public:
	
	static const int default_port = 5432;
	
private:
	
	/** Global symbol table shared between different runs. */
	pgsVarMap m_vars;
	
	/** Allows only one thread to run at once. */
	wxSemaphore m_critical;
	
	/** To protect m_running. */
	wxSemaphore m_mutex;
	
	/** Connection to the database. */
	pgConn * m_connection;
	
	/** Is m_connection provided in the constructor or has it been created. */
	bool m_defined_conn;
	
	/** Number of instances of pgsThread running (should be 0 or 1). */
	int m_running;
	
	/** Detached thread running a pgScript (parses a file or a string). */
	pgsThread * m_thread;
	
	/** pgAdmin specific: post an event to this window when m_thread is done. */
	wxWindow * m_caller;
	
	/** pgAdmin specific: post this event when m_thread is done. */
	long m_event_id;
	
public:
	
	pgsApplication(const wxString & host, const wxString & database,
			const wxString & user, const wxString & password, int port = default_port);
	
	pgsApplication(pgConn * connection);
	
	~pgsApplication();
	
	bool parse_file(const wxString & file, pgsOutputStream & out,
			wxMBConv * conv = &wxConvLocal);
	
	bool parse_string(const wxString & string, pgsOutputStream & out);
	
	/** Is m_thread running? */
	bool is_running();
	
	/** If m_thread is running then wait for it to terminate. */
	void wait();
	
	/** If m_thread is running then delete it. */
	void terminate();
	
	/** Called by m_thread when the thread is finished: is_running() becomes
	 * false and m_event_id is pushed in the event queue if m_caller exists. */
	void complete_execution();
	
	/** Uses a new database connection instead of the previous one. If the
	 * previous one was user-defined then it is deleted otherwise it is just
	 * replaced with the new one. */
	void set_connection(pgConn * conn);
	
	/** Deletes everything in the symbol table. */
	void clear_symbols();
	
#if !defined(NPGADMIN)
	/** Used in pgAdmin integration for sending an event to the caller when the
	 * thread is finishing its task. */
	void set_caller(wxWindow * caller, long event_id);
#endif
	
	/** Retrieves a variable from the symbol table and returns it as a
	 * smart pointer to a pgsVariable (pgsOperand). */
	pgsOperand variable(const wxString & name);
	
	/** Retrieves a variable from the symbol table and returns it as a
	 * string. */
	wxString string(const wxString & name);
	
	/** Loads value for the variable in the symbol table. The value is given
	 * as a pgsOperand. */
	bool variable(const wxString & name, const pgsOperand & value);
	
	/** Loads value for the variable in the symbol table. The value is given
	 * as a string. */
	bool string(const wxString & name, const wxString & value);
	
	/** Tells whether the database connection is valid. */
	bool is_connection_valid() const;
	
private:
	
	/** Common method for parse_file & parse_string: runs the thread. */
	bool run_thread();
	
private:

	pgsApplication(const pgsApplication & that);

	pgsApplication & operator=(const pgsApplication & that);
	
};

#endif /*PGSAPP_H_*/
