#include <stdlib.h>
#include <string.h>

#include "connection.h"
#include "descriptor.h"
#include "statement.h"
#include "common.h"


#ifndef WIN32
/*-----------------------------------------------------------------------------
 * itoa
 *-----------------------------------------------------------------------------
 */
char* itoa (int value, char* string, int radix)
{
	if(radix==10)
		sprintf (string, "%d", value);
	else if(radix == 16)
		sprintf (string, "%x", value);
	else
		sprintf (string, "not yet realized");
 
	return string;
}

char* ltoa (long long value, char* string, int radix)
{
	if(radix==10)
		sprintf (string, "%lld", value);
	else if(radix == 16)
		sprintf (string, "%llx", value);
	else
		sprintf (string, "not yet realized");

	return string;
}

#define WideCharToMultiByte(a,b,c,d,e,f,g,h) wcstombs (e, c, d)

#endif

/*-----------------------------------------------------------------------------
 * GetInt
 *-----------------------------------------------------------------------------
 */
int GetInt(TCHAR** pPtr, TCHAR terminator, SQLINTEGER* pLength, int radix /* <= 10 only! */)
{
	int    ret = 0;
	BOOL   minus = 0;
	TCHAR* ptr = *pPtr;

	while (0 < *pLength)
	{
		if (_T('0') <= *ptr && _T('9') >= *ptr)
			ret = ret*radix + *ptr - _T('0');
		else if (terminator == *ptr || _T('+') ==*ptr)
		{
			(*pPtr)++;
			(*pLength)--;
			break;
		}
		else if (_T('-') == *ptr)
			minus = 1;
		(*pPtr)++;
		ptr++;
		(*pLength)--;
	}

	if (minus)
		ret = -ret;

	return ret;
}

/*-----------------------------------------------------------------------------
 * GetOctetChar
 *-----------------------------------------------------------------------------
 */
TCHAR
GetOctetChar(TCHAR** source, SQLINTEGER* length)
{
	TCHAR chr = _T('\0');

	if (0 < *length && NULL != *source)
	{
		--(*length);
		chr = **source;
		++(*source);
		if (_T('\\') == chr && 0 < *length)
		{
			SQLINTEGER temp_length = 3;
			SQLINTEGER res = GetInt(source, _T('\\'), &temp_length, 8);
			chr = (0 == temp_length) ? (TCHAR) res : _T('\\');
			*length -= (3 - temp_length);
		}
	}

	return chr;
}

/*-----------------------------------------------------------------------------
 * FUNCTION: GetSQLString
 *-----------------------------------------------------------------------------
 */
SQLTCHAR* GetSQLString(SQLTCHAR* Text, SQLINTEGER TextLength, unsigned int Conversion)
{
	TCHAR* ret_text;

	if (SQL_NTS == TextLength)
		TextLength = _tcslen((TCHAR*)Text);

	if (NULL != (ret_text = (TCHAR*)malloc((TextLength+1) * sizeof(TCHAR))))
	{
		switch(Conversion)
		{
			case TC_AS_IS:
				_tcsncpy(ret_text, (TCHAR*)Text, TextLength);
				break;
			case TC_LOWER:
			{
				int i = 0;
				for(;i<TextLength;i++)
					ret_text[i] = (TCHAR)_totlower(Text[i]);
				break;
			}
			default:
				;
		}

		ret_text[TextLength] = _T('\0');
	}

	return (SQLTCHAR*)ret_text;
}

/*----------------------------------------------------------------------------
 * FUNCTION: ReadFromDS
 *
 * DESCRIPTION:   Reads from DataSource only undefined parameters, if parameter
 *              isn't empty - ignore it.
 *----------------------------------------------------------------------------
 */
void ReadFromDS(TCHAR** parameters, TCHAR* DSN, SQLSMALLINT DSNLength)
{
	int    i;
	TCHAR  check_box[10];
	TCHAR  dsn[DSN_MAX_LENGTH+1];
	TCHAR* pDSN;

	if (SQL_NTS == DSNLength)
		pDSN = DSN;
	else
	{
		_tcsncpy(dsn, DSN, DSNLength);
		dsn[DSNLength] = _T('\0');
		pDSN = dsn;
	}

	for (i=0;i<DS_EDIT_PARAMETERS_NUMBER;i++)
		if (_T('\0') == parameters[i][0])
			SQLGetPrivateProfileString(pDSN, c_stDSParameters[i].szKeyName, c_stDSParameters[i].szDefaultValue, parameters[i], c_stDSParameters[i].unMaxValueLength+1, c_szODBC_INI);

	for (;i<DS_CHECKBOX_PARAMETERS_NUMBER;i++)
		if (_T('\0') == parameters[i][0])
		{
			SQLGetPrivateProfileString(pDSN, c_stDSParameters[i].szKeyName, c_stDSParameters[i].szDefaultValue, (SQLTCHAR*)&check_box, sizeof(check_box)/sizeof(TCHAR), c_szODBC_INI);
			if (0 == _tcsicmp(check_box, _T("0"))      ||
				0 == _tcsicmp(check_box, _T("FALSE"))    ||
				0 == _tcsicmp(check_box, _T("NO"))       ||
				0 == _tcsicmp(check_box, _T("N")))
				parameters[i][0] = _T('N');
			else if (0 == _tcsicmp(check_box, _T("1")) ||
				0 == _tcsicmp(check_box, _T("TRUE"))     ||
				0 == _tcsicmp(check_box, _T("YES"))      ||
				0 == _tcsicmp(check_box, _T("Y")))
				parameters[i][0] = _T('Y');
			else
				parameters[i][0] = c_stDSParameters[i].szDefaultValue[0];
			parameters[i][1] = _T('\0');
		}


	for (;i<DS_PARAMETERS_NUMBER;i++)
	{
		if (_T('\0') == parameters[i][0])
			SQLGetPrivateProfileString(pDSN, c_stDSParameters[i].szKeyName, NULL, parameters[i], c_stDSParameters[i].unMaxValueLength+1, c_szODBC_INI);
		if (_T('\0') == parameters[i][0])
		{/* use as default only first word in list */
			const TCHAR* param  = c_stDSParameters[i].szDefaultValue;
			TCHAR*       ptr    = _tcschr(param, _T(','));
			int          length = (NULL != ptr) ? (ptr - param) : _tcslen(param);
			_tcsncpy(parameters[i], param, length);
			parameters[i][length] = _T('\0');
		}
	}
}

/*----------------------------------------------------------------------------
 * FUNCTION: WriteToDS
 *----------------------------------------------------------------------------
 */
void WriteToDS(TCHAR* parameters[], TCHAR* DSN)
{
	int i;

	for (i=0;i<DS_PARAMETERS_NUMBER;i++)
		SQLWritePrivateProfileString(DSN, c_stDSParameters[i].szKeyName, parameters[i], c_szODBC_INI);
}


/*----------------------------------------------------------------------------
 * FUNCTION: AddItem
 *----------------------------------------------------------------------------
 */
SQLRETURN
AddItem(List* pList, SQLHANDLE Handle)
{
	/* is this the first reference? */
	if (NULL == pList->handles)
	{
		pList->handles = (SQLHANDLE*) malloc(LIST_ITEMS_NUMBER*sizeof(SQLHANDLE));
		pList->allocated = LIST_ITEMS_NUMBER;
		pList->used = 0;
	}

	/* is there enough space? */
	if (pList->used == pList->allocated)
	{
		/* realloc memory and copy current data */
		SQLHANDLE* handlesTemp   = (SQLHANDLE*) malloc((pList->allocated+LIST_ITEMS_NUMBER)*sizeof(SQLHANDLE));
		SQLHANDLE* handlesToFree = pList->handles;

		memcpy(handlesTemp, pList->handles, pList->allocated*sizeof(SQLHANDLE));
		pList->allocated += LIST_ITEMS_NUMBER;
		pList->handles = handlesTemp;
		FREE(handlesToFree);
	}

	/* adding */
	pList->handles[pList->used++] = Handle;

	return SQL_SUCCESS;
}


/*----------------------------------------------------------------------------
 * FUNCTION: RemoveItem
 *----------------------------------------------------------------------------
 */
SQLRETURN
RemoveItem(List* pList, SQLHANDLE Handle)
{
	int i;

	if (NULL != pList->handles)
	{
		for (i=0;i<pList->used;i++)
		{
			if (Handle == pList->handles[i])
			{
				if (i != --pList->used)
					memcpy(&pList->handles[i], &pList->handles[i+1], (pList->used-i)*sizeof(SQLHANDLE));
				break;
			}
		}
	}

	return SQL_SUCCESS;
}

/*----------------------------------------------------------------------------
 * FUNCTION: FreeList
 *----------------------------------------------------------------------------
 */
SQLRETURN
FreeList(List* pList, SQLSMALLINT HandleType)
{
	if (NULL != pList->handles)
	{
		if (0 != HandleType)
		{
			int i;

			for (i=pList->used-1;i>=0;i--)
				SQLFreeHandle(HandleType, pList->handles[i]);
		}
		pList->used = 0;
		FREE(pList->handles);
		pList->handles = NULL;
	}

	return SQL_SUCCESS;
}


/*----------------------------------------------------------------------------
 * FUNCTION: GetText
 *
 * DESCRIPTION: 
 *----------------------------------------------------------------------------
 */
TCHAR*
GetText(const TCHAR* szBaseText, const TCHAR* szParameters, ...)
{
	TCHAR*  ret;
	va_list vaList;
	va_list vaCopy;
	
	va_start(vaList, szParameters);
#ifdef WIN32
	vaCopy = vaList;
#else	
	va_copy(vaCopy, vaList);
#endif	
	ret = PrepareText(szBaseText, szParameters, vaCopy);
	va_end(vaList);
	
	return ret;
}


/*----------------------------------------------------------------------------
 * FUNCTION: GetTextFromArray
 *
 * DESCRIPTION: 
 *----------------------------------------------------------------------------
 */
TCHAR*
GetTextFromArray(const TCHAR* szBaseText, int array_size, const TCHAR** szParameters)
{
	if (NULL == szBaseText)
		return NULL;
	else
	{
		const TCHAR* szParameter;
		const TCHAR* pNativeText;
		TCHAR*       pResultText;
		TCHAR*       szResult;
		int          length;
		int          parameters = 0;

		if (0 < array_size)
		{
			/* calculate required space */
			for (length=0;szBaseText[length];length++)
				if(_T('?') == szBaseText[length])
					parameters++;

			length++;
			if (0 != parameters)
			{
				int i;
				parameters = MIN(parameters, array_size);

				/* text uses parameters - check them and calculate their's length */
				for (i=0;i<parameters;i++)
					length += _tcslen(szParameters[i]);
			}
		}
		else
			length = _tcslen(szBaseText) + 1;

		/* allocate required memory and compile result text */
		szResult = (TCHAR*) malloc(length * sizeof(TCHAR));
		if (0 == parameters)
			_tcscpy(szResult, szBaseText);
		else
		{
			szParameter = *szParameters;
			pNativeText = szBaseText;
			pResultText = szResult;
			while(_T('\0') != *pNativeText)
			{
				if (_T('?') == *pNativeText && 0 != parameters)
				{
					while(_T('\0') != *szParameter)
					{
						*pResultText = *szParameter;
						pResultText++;
						szParameter++;
					}
					szParameter = *++szParameters;
					parameters--;
				}
				else
				{
					*pResultText = *pNativeText;
					pResultText++;
				}
				pNativeText++;
			}
			*pResultText = _T('\0');
		}
		return szResult;
	}
}


/*----------------------------------------------------------------------------
 * FUNCTION: PrepareText
 *----------------------------------------------------------------------------
 */
TCHAR*
PrepareText(const TCHAR* szBaseText, const TCHAR* szParameters, va_list vaList)
{
	va_list vaCopy;
  unsigned int unLength;      /* length of the result text */
	unsigned int unParameters;  /* number of parameters required in base text */
	unsigned int unUsedParams = 0;  /* number of parameters used in resulted text */
	const TCHAR* szParameter;
	const TCHAR* pNativeText;
	TCHAR*       pResultText;
	TCHAR*       szResult;

#ifdef WIN32
	vaCopy = vaList;	
#else
	va_copy (vaCopy, vaList);
#endif
    
	if (NULL == szBaseText)
		return NULL;

	if (NULL != szParameters)
	{
		/* calculate required space */
		unParameters = 0;
		for (unLength=0;szBaseText[unLength];unLength++)
			if(_T('?') == szBaseText[unLength])
				unParameters++;

		unLength++;
		if (0 != unParameters)
		{
			/* text uses parameters - check them and calculate their's length */
			szParameter  = szParameters;
			unUsedParams = unParameters;
			while (NULL != szParameter && 0 != unUsedParams)
			{
				unLength += _tcslen(szParameter);
				szParameter = va_arg(vaList, const TCHAR*);
				unUsedParams--;
			}
			unUsedParams = unParameters - unUsedParams;
			unLength -= unUsedParams;
		}
	}
	else
		unLength = _tcslen(szBaseText) + 1;

	/* allocate required memory and compile result text */
	szResult = (TCHAR*) malloc(unLength * sizeof(TCHAR));
	if (0 == unUsedParams)
		_tcscpy(szResult, szBaseText);
	else
	{
		szParameter = szParameters;
		pNativeText = szBaseText;
		pResultText = szResult;
		while(_T('\0') != *pNativeText)
		{
			if (_T('?') == *pNativeText && 0 != unUsedParams)
			{
				while(_T('\0') != *szParameter)
				{
					*pResultText = *szParameter;
					pResultText++;
					szParameter++;
				}
				szParameter = va_arg(vaCopy, const TCHAR*);
				unUsedParams--;
			}
			else
			{
				*pResultText = *pNativeText;
				pResultText++;
			}
			pNativeText++;
		}
		*pResultText = _T('\0');
	}
	return szResult;
}

/*----------------------------------------------------------------------------
 * FUNCTION: FillBufferWithValue
 *
 * RETURN: number of byted returned into buffer.
 *
 * DESCRIPTION: Translates value, recieved in backend's format to the format,
 *              requested by application. 
 *----------------------------------------------------------------------------
 */
SQLLEN FillBufferWithValue(SQLPOINTER TargetValue, SQLLEN     TargetLength, SQLSMALLINT TargetType,
                           TCHAR* SourceValue, SQLINTEGER SourceLength, SQLSMALLINT SourceType
                          )
{
#ifndef UNICODE
	SQLINTEGER   i;
#endif /* UNICODE */
	TCHAR        buffer[50];
	TCHAR*       ptr;
	SQLRETURN    nRet;
	SQLLEN       nResulted;

	if (SQL_DEFAULT == TargetType)
		TargetType = GetCDefaultType(SourceType);

	switch(TargetType)
	{
		case SQL_C_WCHAR:
			switch (SourceType)
			{
				case SQL_BIT:
				{
					WCHAR resulted_bool[2] = {_T('\0'), _T('\0')};
					resulted_bool[0] = (_T('t') == *(TCHAR*)SourceValue || _T('T') == *(TCHAR*)SourceValue) ? L'1' : L'0';
					memcpy(TargetValue, &resulted_bool, (1 < TargetLength) ? sizeof(resulted_bool) : sizeof(WCHAR));
					return sizeof(WCHAR);
				}
				case SQL_BINARY:
				case SQL_VARBINARY:
				case SQL_LONGVARBINARY:
				{/* translate values from backend */
					TCHAR* source_value = (TCHAR*)SourceValue;
					TCHAR* target_value = (TCHAR*)TargetValue;
					nResulted = 0;

					while ((1 < TargetLength) && (0 < SourceLength))
					{
						TCHAR chr = GetOctetChar(&source_value, &SourceLength);
						_tprintf(target_value, _T("%02x"), chr);
						++nResulted;
						target_value += 2;
						TargetLength -= 2;
					}
					break;
				}
				default:
					TargetLength /= sizeof(WCHAR);
					if (0 < (nResulted = MIN(TargetLength-1, SourceLength)))
					{
						WCHAR* result;
#ifdef UNICODE
						/* This is the same type, info stored in ower buffer, use simple copying */
						wcsncpy((TCHAR*)TargetValue, SourceValue, nResulted);
#else
						result = (WCHAR*)TargetValue;
						for(i=0;i<nResulted;i++)
							result[i] = 0x00FF & ((BYTE*)SourceValue)[i];
#endif /* UNICODE */
						result = (WCHAR*)TargetValue;
						result[nResulted] = L'\0';
					}
			}
			return nResulted;
		case SQL_C_CHAR:
		case SQL_C_BINARY:
			switch (SourceType)
			{
				case SQL_BIT:
				{
					char resulted_bool[2] = {'\0', '\0'};
					resulted_bool[0] = (_T('t') == *(TCHAR*)SourceValue || _T('T') == *(TCHAR*)SourceValue) ? '1' : '0';
					memcpy(TargetValue, &resulted_bool, (1 < TargetLength) ? sizeof(resulted_bool) : sizeof(char));
					return sizeof(char);
				}
				case SQL_BINARY:
				case SQL_VARBINARY:
				case SQL_LONGVARBINARY:
				{/* translate values from backend */
					TCHAR* source_value = (TCHAR*)SourceValue;
					CHAR* target_value = (CHAR*)TargetValue;
					nResulted = 0;

					while ((1 < TargetLength) && (0 < SourceLength))
					{
						TCHAR chr = GetOctetChar(&source_value, &SourceLength);
						sprintf(target_value, "%02x", chr);
						++nResulted;
						target_value += 2;
						TargetLength -= 2;
					}
					break;
				}
				default:
					if (0 < (nResulted = MIN(TargetLength-1, SourceLength)))
					{
						BYTE* result;
#ifdef UNICODE
						nResulted = WideCharToMultiByte(CP_ACP, 0, SourceValue, nResulted, (CHAR*)TargetValue, TargetLength-1, NULL, NULL);
#else
					/* This is the same type, info stored in ower buffer, use simple copying */
						strncpy(TargetValue, SourceValue, nResulted);
#endif /* UNICODE */
						result = (BYTE*)TargetValue;
						result[nResulted] = '\0';
					}
			}
			return nResulted;
		case SQL_C_DATE:
		case SQL_C_TYPE_DATE:
			switch(SourceType)
			{
				case SQL_CHAR:
				case SQL_VARCHAR:
				case SQL_LONGVARCHAR:
				case SQL_DATE:
				case SQL_TYPE_DATE:
				{
					SQL_DATE_STRUCT date;
					/* string can be:
					 * 1. 2005-12-31
					 */
					ptr = SourceValue;
					date.year  = GetInt(&ptr, _T('-'), &SourceLength, 10);
					date.month = GetInt(&ptr, _T('-'), &SourceLength, 10);
					date.day   = GetInt(&ptr, _T('-'), &SourceLength, 10);
					memcpy(TargetValue, &date, sizeof(date));
					return sizeof(date);
				}
			}
			break;
		case SQL_C_TIME:
		case SQL_C_TYPE_TIME:
			switch(SourceType)
			{
				case SQL_CHAR:
				case SQL_VARCHAR:
				case SQL_LONGVARCHAR:
				case SQL_TIME:
				case SQL_TYPE_TIME:
				{
					SQL_TIME_STRUCT time;
					/* string can be:
					 * 1. 23:59:59
					 */
					ptr = SourceValue;
					time.hour   = GetInt(&ptr, _T(':'), &SourceLength, 10);
					time.minute = GetInt(&ptr, _T(':'), &SourceLength, 10);
					time.second = GetInt(&ptr, _T(':'), &SourceLength, 10);
					memcpy(TargetValue, &time, sizeof(time));
					return sizeof(time);
				}
			}
			break;
		case SQL_C_TIMESTAMP:
		case SQL_C_TYPE_TIMESTAMP:
			switch(SourceType)
			{
				case SQL_CHAR:
				case SQL_VARCHAR:
				case SQL_LONGVARCHAR:
				case SQL_TIME:
				case SQL_DATE:
				case SQL_TYPE_TIME:
				case SQL_TYPE_DATE:
				case SQL_TIMESTAMP:
				case SQL_TYPE_TIMESTAMP:
				{
					SQL_TIMESTAMP_STRUCT timestamp;
					/* string can be:
					 * 1. 2005-12-31 23:59:59.19999
					 */
					ptr = SourceValue;
					if (SQL_TIME == SourceType || SQL_TYPE_TIME == SourceType)
					{
						timestamp.year   = 0;
						timestamp.month  = 0;
						timestamp.day    = 0;
					}
					else
					{
						timestamp.year   = GetInt(&ptr, _T('-'), &SourceLength, 10);
						timestamp.month  = GetInt(&ptr, _T('-'), &SourceLength, 10);
						timestamp.day    = GetInt(&ptr, _T(' '), &SourceLength, 10);
					}
					timestamp.hour     = GetInt(&ptr, _T(':'), &SourceLength, 10);
					timestamp.minute   = GetInt(&ptr, _T(':'), &SourceLength, 10);
					timestamp.second   = GetInt(&ptr, _T('.'), &SourceLength, 10);
					timestamp.fraction = GetInt(&ptr, _T('.'), &SourceLength, 10);
					memcpy(TargetValue, &timestamp, sizeof(timestamp));
					return sizeof(timestamp);
				}
			}
			break;
		case SQL_C_USHORT:
		case SQL_C_SSHORT:
		case SQL_C_SHORT:
			{
				SQLSMALLINT val;

				_tcsncpy(buffer, SourceValue, SourceLength);
				buffer[SourceLength] = _T('\0');
				val = _ttoi(buffer);
				memcpy(TargetValue, &val, sizeof(val));
				return sizeof(SQLSMALLINT);
			}
		case SQL_C_SLONG:
		case SQL_C_ULONG:
		case SQL_C_LONG:
			{
				SQLINTEGER val;

				_tcsncpy(buffer, SourceValue, SourceLength);
				buffer[SourceLength] = _T('\0');
				val = _ttoi(buffer);
				memcpy(TargetValue, &val, sizeof(SQLINTEGER));
				return sizeof(SQLINTEGER);
			}
		case SQL_C_SBIGINT:
		case SQL_C_UBIGINT:
			{
				SQLBIGINT val;

				_tcsncpy(buffer, SourceValue, SourceLength);
				buffer[SourceLength] = _T('\0');
				val = _ttol(buffer);
				memcpy(TargetValue, &val, sizeof(SQLBIGINT));
				return sizeof(SQLBIGINT);
			}
		case SQL_C_STINYINT:
		case SQL_C_UTINYINT:
		case SQL_C_TINYINT:
			{
				SQLSCHAR val;

				_tcsncpy(buffer, SourceValue, SourceLength);
				buffer[SourceLength] = _T('\0');
				val = _ttoi(buffer);
				memcpy(TargetValue, &val, sizeof(SQLSCHAR));
				return sizeof(SQLSCHAR);
			}
		case SQL_C_FLOAT:
		case SQL_C_DOUBLE:
			{
				SQLFLOAT val;
				TCHAR dec_point = *(localeconv()->decimal_point);

				_tcsncpy(buffer, SourceValue, SourceLength);
				buffer[SourceLength] = _T('\0');

				if ((_T('.') != dec_point) &&
						(ptr = _tcschr(buffer, _T('.')))
				   )
					 *ptr = dec_point;
				
				val = _tcstod(buffer, NULL);
				if (SQL_C_FLOAT == TargetType)
				{
					SQLREAL res = (SQLREAL) val;
					memcpy(TargetValue, &res, sizeof(res));
					return sizeof(res);
				}
				memcpy(TargetValue, &val, sizeof(val));
				return sizeof(val);
			}
		case SQL_C_BIT:
			{
				SQLCHAR val = (_T('T') == *(TCHAR*)SourceValue || _T('t') == *(TCHAR*)SourceValue) ? 1 : 0;
				memcpy(TargetValue, &val, sizeof(SQLCHAR));
				return sizeof(SQLCHAR);
			}
		case SQL_C_NUMERIC:
			{
				int i;
				TCHAR* value;
				TCHAR* point;
				SQL_NUMERIC_STRUCT val;
				SQLCHAR* ptr = val.val;

				memset(val.val, 0, sizeof(val.val));
				_tcsncpy(buffer, SourceValue, SourceLength);
				buffer[SourceLength] = _T('\0');

				/* set information fields */
				val.sign  = (_T('-') == *SourceValue) ? 0 : 1;
				val.scale = (NULL == (point = _tcschr(buffer, _T('.')))) ? 0 : (SourceLength - (point - (TCHAR*)buffer) - 1);
				val.precision = SourceLength - ((0 == val.scale) ? 0 : 1) - ((0 == val.sign) ? 1 : 0);

				/* remove '.' */
				if (0 < val.scale)
				{
					for (point=&buffer[SourceLength - val.scale]; _T('\0') != *point; ++point)
						point[-1] = point[0];

					buffer[--SourceLength] = _T('\0');
				}

				/* make value more usefull for calculations */
				if (_T('-') == buffer[0])
				{
					value = buffer + 1;
					--SourceLength;
				} else {
					value = buffer;
				}

				for (i=0;i<SourceLength;++i)
					value[i] -= _T('0');

				/* convert decimal string into hexadecimal value */
				for (i=0; i < SourceLength;)
				{
					int j;
					int runner;
					
					for (j=i, runner=0; j < SourceLength; ++j)
					{
						runner *= 10;
						runner += value[j];
						value[j] = runner / 256;
						runner %= 256;
					}

					/* add runner as a top value */
					*ptr++ = runner;

					while (0 == value[i] && i < SourceLength)
						++i;
				}

				memcpy(TargetValue, &val, sizeof(val));
				return sizeof(val);
			}
/* INTERVAL_C, SQL_C_GUID */
		default:
			/* unknown TargetType */
			nRet = SQL_ERROR;
	}

	return 0;
}

/*----------------------------------------------------------------------------
 * FUNCTION: ReturnString
 *
 * RETURN: SQL_SUCCESS - string fully returned,
 *         SQL_SUCCESS_WITH_INFO -
 *
 * DESCRIPTION: 
 *
 * WARNING! No error response done - calling function has to set error itself.
 *----------------------------------------------------------------------------
 */
SQLRETURN ReturnString(SQLTCHAR* TargetValue, SQLSMALLINT TargetLength, SQLSMALLINT* SourceLength, const TCHAR* SourceValue, SQLSMALLINT nSourceLength, BOOL in_bytes)
{
	SQLRETURN nRet = SQL_SUCCESS;

	if (NULL == SourceValue)
		SourceValue = _T("");

	if (SQL_NTS == nSourceLength)
		nSourceLength = _tcslen(SourceValue);

	/* return avaible value length if required */
	if (NULL != SourceLength)
		*SourceLength = (in_bytes) ? nSourceLength*sizeof(TCHAR) : nSourceLength;

	if (NULL != TargetValue)
	{
		if (--TargetLength < nSourceLength)
		{
			nRet = SQL_SUCCESS_WITH_INFO;
			nSourceLength = TargetLength;
		}

		_tcsncpy((TCHAR*)TargetValue, SourceValue, nSourceLength);
		TargetValue[nSourceLength] = _T('\0');
	}

	return nRet;
}

/*----------------------------------------------------------------------------
 * FUNCTION: CreateConnectionStringRequest
 *----------------------------------------------------------------------------
 */
SQLRETURN
PrepareConnectionStringRequest(SQLTCHAR* TargetValue, SQLSMALLINT TargetLength, SQLSMALLINT* SourceLength, /*const*/ TCHAR** parameters)
{
	SQLRETURN   nRet   = SQL_SUCCESS_WITH_INFO;
	SQLSMALLINT length = DS_PARAMETERS_NUMBER - NECESSARY_CONNSTR_PARAM;
	                     /* number of '*', added to unnecessary parameters */
	unsigned int i;

	/* calculate required buffer length */
	for (i=FIRST_CONNSTR_PARAM;i<DS_PARAMETERS_NUMBER;i++)
	{
		if (_T('\0') == parameters[i][0])
		{
			length += _tcslen(c_stDSParameters[i].szIdentifier);
			length += _tcslen(c_stDSParameters[i].szKeyName);
			length += STR_SIZEOF(":=?;");
		}
		else if (NECESSARY_CONNSTR_PARAM <= i)
			length -= STR_SIZEOF("*");
	}
	
	/* return ConnectionString length */
	if (NULL != SourceLength)
		*SourceLength = length;

	/* return avaible part of the ConnectionString */
	if (NULL != TargetValue && 0 != TargetLength)
	{
		SQLSMALLINT unToCopy;

		if (length < TargetLength)
			nRet = SQL_SUCCESS;
		else
			length = TargetLength-1;

		/* copy defined parameter values */
		for (i=FIRST_CONNSTR_PARAM; 0 < length;i++)
		{
			if (_T('\0') == parameters[i][0])
			{
				if (NECESSARY_CONNSTR_PARAM <= i)
				{/* add '*' to unnecessary parameter in connection string */
					*TargetValue = _T('*');
					TargetValue++;
					length--;
				}

				if (0 < length)
				{
					unToCopy = MIN(length, (SQLSMALLINT)_tcslen(c_stDSParameters[i].szKeyName));
					_tcsncpy((TCHAR*)TargetValue, c_stDSParameters[i].szKeyName, unToCopy);
					length -= unToCopy;
					if (0 < length)
					{
						TargetValue += unToCopy;
						*TargetValue = _T(':');
						if (0 < --length)
						{
							TargetValue++;
							unToCopy = MIN(length, (SQLSMALLINT) _tcslen(c_stDSParameters[i].szIdentifier));
							_tcsncpy((TCHAR*)TargetValue, c_stDSParameters[i].szIdentifier, unToCopy);
							length -= unToCopy;
							if (0 < length)
							{
								TargetValue += unToCopy;
								unToCopy = MIN(length, (SQLSMALLINT) STR_SIZEOF("=?;"));
								_tcsncpy((TCHAR*)TargetValue, _T("=?;"), unToCopy);
								TargetValue += unToCopy;
								length -= unToCopy;
							}
						}
					}
				}
			}
		}

		*TargetValue = _T('\0');
	}

	return nRet;
}

/*----------------------------------------------------------------------------
 * FUNCTION: CreateConnectionString
 *----------------------------------------------------------------------------
 */
SQLRETURN
PrepareConnectionString(SQLTCHAR* TargetValue, SQLSMALLINT TargetLength,
											  SQLSMALLINT* SourceLength, TCHAR** parameters, TCHAR* dsn)
{
	SQLRETURN   nRet   = SQL_SUCCESS_WITH_INFO;
	SQLSMALLINT length = (NULL == dsn || _T('\0') == dsn[0]) ? STR_SIZEOF("DRIVER={Mammoth ODBCng beta};") : STR_SIZEOF("DSN=;")+_tcslen(dsn);
	SQLSMALLINT unToCopy;

	unsigned int i;

	/* calculate required buffer length */
	for (i=FIRST_CONNSTR_PARAM;i<DS_PARAMETERS_NUMBER;i++)
	{
		if (_T('\0') != parameters[i][0])
		{
			length += _tcslen(parameters[i]);
			length += _tcslen(c_stDSParameters[i].szKeyName);
			length += STR_SIZEOF("=;");
		}
	}
	
	/* return ConnectionString length */
	if (NULL != SourceLength)
		*SourceLength = length*sizeof(TCHAR);

	/* return avaible part of the ConnectionString */
	if (NULL != TargetValue && 0 != TargetLength)
	{
		if (length < TargetLength)
			nRet = SQL_SUCCESS;
		else
			length = TargetLength-1;

		/* first of all - copy DSN or DRIVER key */
		if (NULL == dsn || '\0' == dsn[0])
		{
			/* just copy Driver name */
			unToCopy = MIN(length, STR_SIZEOF("DRIVER={Mammoth ODBCng beta};"));
			_tcsncpy((TCHAR*)TargetValue, _T("DRIVER={Mammoth ODBCng beta};"), unToCopy);
		}
		else
		{
			unToCopy = MIN(length, STR_SIZEOF("DSN="));
			_tcsncpy((TCHAR*)TargetValue, _T("DSN="), unToCopy);
			if (unToCopy < length)
			{
				TargetValue += unToCopy;
				length      -= unToCopy;
				unToCopy = MIN(length, (SQLSMALLINT) _tcslen(dsn));
				_tcsncpy((TCHAR*)TargetValue, dsn, unToCopy);
				if (unToCopy < length)
				{
					length -= unToCopy;
					TargetValue += unToCopy;
					*TargetValue = _T(';');
					unToCopy = 1;
				}
			}
		}
		length      -= unToCopy;
		TargetValue += unToCopy;

		/* copy defined parameter values */
		for (i=FIRST_CONNSTR_PARAM; 0 < length;i++)
		{
			if (_T('\0') != parameters[i][0])
			{
				unToCopy = MIN(length, (SQLSMALLINT)_tcslen(c_stDSParameters[i].szKeyName));
				_tcsncpy((TCHAR*)TargetValue, c_stDSParameters[i].szKeyName, unToCopy);
				length -= unToCopy;
				if (0 < length)
				{
					TargetValue += unToCopy;
					*TargetValue = _T('=');
					if (0 < --length)
					{
						TargetValue++;
						unToCopy = MIN(length, (SQLSMALLINT) _tcslen(parameters[i]));
						_tcsncpy((TCHAR*)TargetValue, parameters[i], unToCopy);
						length -= unToCopy;
						if (0 < length)
						{
							TargetValue += unToCopy;
							*TargetValue = _T(';');
							TargetValue++;
							length--;
						}
					}
				}
			}
		}
		*TargetValue = _T('\0');
	}

	return nRet;
}

/*----------------------------------------------------------------------------
 * FUNCTION: TranslateType
 *----------------------------------------------------------------------------
 */
void TranslateType(CD_REC* common, SQLSMALLINT code, SQLSMALLINT decimalDigits, SQLLEN size, TypeClass typeClass)
{
	switch(typeClass)
	{
		case C_TYPE:
			switch(code)
			{
				/* datetime */
				case SQL_C_TYPE_DATE:
				case SQL_C_TYPE_TIME:
				case SQL_C_TYPE_TIMESTAMP:
					common->type = SQL_DATETIME;
					break;
				/* interval */
				case SQL_C_INTERVAL_MONTH:
				case SQL_C_INTERVAL_YEAR:
				case SQL_C_INTERVAL_YEAR_TO_MONTH:
				case SQL_C_INTERVAL_DAY:
				case SQL_C_INTERVAL_HOUR:
				case SQL_C_INTERVAL_MINUTE:
				case SQL_C_INTERVAL_SECOND:
				case SQL_C_INTERVAL_DAY_TO_HOUR:
				case SQL_C_INTERVAL_DAY_TO_MINUTE:
				case SQL_C_INTERVAL_DAY_TO_SECOND:
				case SQL_C_INTERVAL_HOUR_TO_MINUTE:
				case SQL_C_INTERVAL_HOUR_TO_SECOND:
				case SQL_C_INTERVAL_MINUTE_TO_SECOND:
					common->datetime_interval_precision = 2; /* default precision */
					common->precision = 6;
					common->type = SQL_INTERVAL;
					break;
				case SQL_C_NUMERIC:
					common->precision = 0; /* andyk change */
					common->scale = 0;
				/* no break! */
				default:
					common->type = code;
			}
			break;
		case SQL_TYPE:
			switch (code)
			{
				/* datetime */
				case SQL_TYPE_TIME:
				case SQL_TYPE_TIMESTAMP:
					common->precision = decimalDigits;
				case SQL_TYPE_DATE:
					common->type = SQL_DATETIME;
					common->length = size;
					break;
				/* interval */
				case SQL_INTERVAL_SECOND:
				case SQL_INTERVAL_DAY_TO_SECOND:
				case SQL_INTERVAL_HOUR_TO_SECOND:
				case SQL_INTERVAL_MINUTE_TO_SECOND:
					common->datetime_interval_precision = 2; /* default precision */
					common->precision = decimalDigits;
					common->type = SQL_INTERVAL;
					common->length = size;
					break;
				case SQL_INTERVAL_MONTH:
				case SQL_INTERVAL_YEAR:
				case SQL_INTERVAL_YEAR_TO_MONTH:
				case SQL_INTERVAL_DAY:
				case SQL_INTERVAL_HOUR:
				case SQL_INTERVAL_MINUTE:
				case SQL_INTERVAL_DAY_TO_HOUR:
				case SQL_INTERVAL_DAY_TO_MINUTE:
				case SQL_INTERVAL_HOUR_TO_MINUTE:
					common->datetime_interval_precision = 2; /* default precision */
					common->precision = 6;
					common->type = SQL_INTERVAL;
					/* no break! */
				case SQL_CHAR:
				case SQL_VARCHAR:
				case SQL_LONGVARCHAR:
				case SQL_BINARY:
				case SQL_VARBINARY:
				case SQL_LONGVARBINARY:
					common->length = size;
					break;
				case SQL_DECIMAL:
				case SQL_NUMERIC:
					common->scale = decimalDigits;
					/* no break! */
				case SQL_FLOAT:
				case SQL_REAL:
				case SQL_DOUBLE:
					common->precision = (SQLSMALLINT)size;
				default:
					common->type = code;
			}
			break;
		default:
			;
	}
	common->concise_type = code;
}

/*----------------------------------------------------------------------------
 * FUNCTION: SQLTypeDescriptor
 *----------------------------------------------------------------------------
 */
SQLRETURN
SQLTypeDescriptor(SQLSMALLINT  concise_type,
                  SQLSMALLINT  is_unsigned, /* SQL_TRUE or SQL_FALSE */
                  SQLLEN*      pLength,     /* can contain input length for character data types */
                  SQLSMALLINT* pPrecision,  /* can contain input precision */
                  SQLSMALLINT* pScale,      /* can contain input scale */

                  SQLSMALLINT* pSQLType, 
                  SQLULEN*     pDisplaySize,
                  SQLINTEGER*  pIntervalPrecision,
                  SQLINTEGER*  pNumPrecRadix)
{
	/* initialize return values with default values */
	SQLINTEGER  datetime_interval_precision = 0;
	SQLINTEGER  num_prec_radix              = 0;
	SQLINTEGER  display_size                = 256;
	SQLSMALLINT sql_type                    = concise_type;
	SQLSMALLINT scale                       = (NULL == pScale)     ? 0 : *pScale;
	SQLSMALLINT precision                   = (NULL == pPrecision) ? 0 : *pPrecision;

	switch(concise_type)
	{
		case SQL_DECIMAL:
		case SQL_NUMERIC:
			display_size = precision + 2;
			break;
		case SQL_BIT:
			display_size = 1;
			precision = 1;
			break;
		case SQL_TINYINT:
			display_size = (SQL_FALSE == is_unsigned) ? 4 : 3;
			num_prec_radix = 10;
			precision = 3;
			scale = 0;
			break;
		case SQL_SMALLINT:
			display_size = (SQL_FALSE == is_unsigned) ? 6 : 5;
			num_prec_radix = 10;
			precision = 5;
			scale = 0;
			break;
		case SQL_INTEGER:
			display_size = (SQL_FALSE == is_unsigned) ? 11 : 10;
			num_prec_radix = 10;
			precision = 10;
			scale = 0;
			break;
		case SQL_BIGINT:
			precision = (SQL_FALSE == is_unsigned) ? 19 : 20;
			num_prec_radix = 10;
			display_size = 20;
			scale = 0;
			break;
		case SQL_REAL:
			precision = 7;
			num_prec_radix = 10;
			display_size = 14;
			break;
		case SQL_FLOAT:
		case SQL_DOUBLE:
			precision = 15;
			num_prec_radix = 10;
			display_size = 24;
			break;
		case SQL_TYPE_DATE:
			display_size = STR_SIZEOF("yyyy-mm-dd");
			sql_type = SQL_DATETIME;
			break;
		case SQL_TYPE_TIME:
			display_size = (0 == precision) ? STR_SIZEOF("hh:mm:ss") : (STR_SIZEOF("hh:mm:ss.") + precision);
			sql_type = SQL_DATETIME;
			break;
		case SQL_TYPE_TIMESTAMP:
			display_size = (0 == precision) ? STR_SIZEOF("yyyy-mm-dd hh:mm:ss"): (STR_SIZEOF("yyyy-mm-dd hh:mm:ss.") + precision);
			sql_type = SQL_DATETIME;
			break;
		case SQL_GUID:
			display_size = STR_SIZEOF("aaaaaaaa-bbbb-cccc-dddd-eeeeeeeeeeee");
			break;
		case SQL_BINARY:
		case SQL_VARBINARY:
		case SQL_LONGVARBINARY:
		case SQL_WCHAR:
		case SQL_WVARCHAR:
		case SQL_WLONGVARCHAR:
			display_size = (NULL == pLength) ? 256 : (*pLength << 1);
			break;
		case SQL_CHAR:
		case SQL_VARCHAR:
		case SQL_LONGVARCHAR:
			display_size = (NULL == pLength) ? 256 : *pLength;
			break;
		/* interval */
		case SQL_INTERVAL_MONTH:
		case SQL_INTERVAL_YEAR:
		case SQL_INTERVAL_YEAR_TO_MONTH:
		case SQL_INTERVAL_DAY:
		case SQL_INTERVAL_HOUR:
		case SQL_INTERVAL_MINUTE:
		case SQL_INTERVAL_DAY_TO_HOUR:
		case SQL_INTERVAL_DAY_TO_MINUTE:
		case SQL_INTERVAL_HOUR_TO_MINUTE:
			datetime_interval_precision = 2;
			precision = 6;
			sql_type = SQL_INTERVAL;
			break;
		case SQL_INTERVAL_SECOND:
		case SQL_INTERVAL_DAY_TO_SECOND:
		case SQL_INTERVAL_HOUR_TO_SECOND:
		case SQL_INTERVAL_MINUTE_TO_SECOND:
			datetime_interval_precision = 2;
			sql_type = SQL_INTERVAL;
			break;
	}

	/* return values */
	if (NULL != pSQLType)
		*pSQLType = sql_type;

	if (NULL != pPrecision)
		*pPrecision = precision;

	if (NULL != pDisplaySize)
		*pDisplaySize = (SQLULEN)display_size;

	if (NULL != pIntervalPrecision)
		*pIntervalPrecision = datetime_interval_precision;

	if (NULL != pNumPrecRadix)
		*pNumPrecRadix = num_prec_radix;

	return SQL_SUCCESS;
}

/*----------------------------------------------------------------------------
 * FUNCTION: PrepareParameter
 *----------------------------------------------------------------------------
 */
int*
PrepareParameter(Statement* pStatement, SQLPOINTER ParamValue, SQLUINTEGER ParamLength, SQLSMALLINT ParamType, SQLLEN* ParamIndPtr, SQLSMALLINT DestType, SQLSMALLINT scale)
{
	TCHAR NumBuffer[1500]; /* must be big enough to contain any non-character data */
	TCHAR*     value  = NULL;
	TCHAR*     ptr    = NULL;
	SQLINTEGER length = SQL_NTS;

	if (NULL != ParamIndPtr && SQL_NULL_DATA == *ParamIndPtr)
	{/* null value */
		return (int*) &c_FIELD_NULL;
	}
	else if (NULL == ParamValue)
	{/* nothing to convert */
		SetError(SQL_HANDLE_STMT, pStatement, ERR_NULL_POINTER_BUFFER, NULL);
		return NULL;
	}
	else
	{
		switch(ParamType)
		{
		case SQL_C_TIMESTAMP:
		case SQL_C_TYPE_TIMESTAMP:
			_stprintf((value = NumBuffer), _T("%04u-%02u-%02u %02u:%02u:%02u.%u"), ((SQL_TIMESTAMP_STRUCT*)ParamValue)->year,
			                                                                       ((SQL_TIMESTAMP_STRUCT*)ParamValue)->month,
																																             ((SQL_TIMESTAMP_STRUCT*)ParamValue)->day,
																																             ((SQL_TIMESTAMP_STRUCT*)ParamValue)->hour,
																															          	   ((SQL_TIMESTAMP_STRUCT*)ParamValue)->minute,
																																             ((SQL_TIMESTAMP_STRUCT*)ParamValue)->second,
																																             ((SQL_TIMESTAMP_STRUCT*)ParamValue)->fraction);
			break;
		case SQL_C_TIME:
		case SQL_C_TYPE_TIME:
			_stprintf((value = NumBuffer), _T("%02u:%02u:%02u"), ((SQL_TIME_STRUCT*)ParamValue)->hour, ((SQL_TIME_STRUCT*)ParamValue)->minute, ((SQL_TIME_STRUCT*)ParamValue)->second);
			break;
		case SQL_C_DATE:
		case SQL_C_TYPE_DATE:
			_stprintf((value = NumBuffer), _T("%04u-%02u-%02u"), ((SQL_DATE_STRUCT*)ParamValue)->year, ((SQL_DATE_STRUCT*)ParamValue)->month, ((SQL_DATE_STRUCT*)ParamValue)->day);
			break;
		case SQL_C_SSHORT:
		case SQL_C_USHORT:
		case SQL_C_SHORT:
			_itot(*(SQLSMALLINT*)ParamValue, (value = NumBuffer), 10);
			break;
		case SQL_C_STINYINT:
		case SQL_C_UTINYINT:
		case SQL_C_TINYINT:
			_itot(*(SQLCHAR*)ParamValue, (value = NumBuffer), 10);
			break;
		case SQL_C_SBIGINT:
		case SQL_C_UBIGINT:
			{
				SQLBIGINT ll = *(SQLBIGINT*)ParamValue;
				_ltot(ll, (value = NumBuffer), 10);
			}
			break;
		case SQL_C_SLONG:
		case SQL_C_ULONG: /* = SQL_C_BOOKMARK */
		case SQL_C_LONG:
			_itot(*(SQLINTEGER*)ParamValue, (value = NumBuffer), 10);
			break;
		case SQL_C_WCHAR:
			switch (DestType)
			{
				case SQL_BINARY:
				case SQL_VARBINARY:
				case SQL_LONGVARBINARY:
				/*    When character C data is converted to binary SQL data, each two bytes of character data are 
				 * converted to a single byte (8 bits) of binary data. Each two bytes of character data represent
				 * a number in hexadecimal form.  For example, "01" is converted to a binary 00000001 and "FF" is
				 * converted to a binary 11111111.
				 */
				{
					WCHAR* ptr = (WCHAR*)ParamValue;
					int i;

					length *= 2;
					value = (TCHAR*)malloc(sizeof(TCHAR)*(length+1));

					for (i=0;i<length;i+=4)
					{/* get value */
						int byte;
						if (L'0' <= *ptr && L'9' >= *ptr)
							byte = *ptr - L'0';
						else if (L'a' <= *ptr && L'f' >= *ptr) 
							byte = *ptr - L'a' + 10;
						else if (L'A' <= *ptr && L'F' >= *ptr)
							byte = *ptr - L'A' + 10;
						else
						  byte = 0;
						++ptr;
						byte <<= 4;
						if (L'0' <= *ptr && L'9' >= *ptr)
							byte += *ptr - L'0';
						else if (L'a' <= *ptr && L'f' >= *ptr) 
							byte += *ptr - L'a' + 10;
						else if (L'A' <= *ptr && 'F' >= *ptr)
							byte += *ptr - L'A' + 10;
						_stprintf(&value[i], _T("\\%03o"), byte);
					}
					break;
				}
				default:
					length = (ParamIndPtr && 0 < *ParamIndPtr) ? ((SQL_NTS == *ParamIndPtr) ? wcslen((WCHAR*)ParamValue) : *ParamIndPtr/sizeof(WCHAR)) : ParamLength;
#ifdef UNICODE
					value = (WCHAR*)ParamValue;
#else
					{
						int size;
						if (length < sizeof(NumBuffer)/sizeof(BYTE))
						{
							value = NumBuffer;
							size = sizeof(NumBuffer)/sizeof(BYTE);
						}
						else
							value = malloc(sizeof(BYTE)*(size = (length+1)));
#ifdef WIN32
						WideCharToMultiByte(0, 0, ParamValue, length, value, size, NULL, NULL);
#else
						{
							int i;
							for (i=0;i<length;i++)
								value[i] = 0x00FF & ((BYTE*)ParamValue)[i];
						}
#endif /* WIN32 */
					}
#endif /* UNICODE */
			}
			break;
		case SQL_C_CHAR:
			length = (ParamIndPtr) ? ((0 <= *ParamIndPtr) ? (*ParamIndPtr) : ((SQL_NTS == *ParamIndPtr) ? strlen((CHAR*)ParamValue) : ParamLength)) : (ParamLength);
			if (length < 0)
				length = strlen(ParamValue);

			switch (DestType)
			{
				case SQL_BINARY:
				case SQL_VARBINARY:
				case SQL_LONGVARBINARY:
				/*    When character C data is converted to binary SQL data, each two bytes of character data are 
				 * converted to a single byte (8 bits) of binary data. Each two bytes of character data represent
				 * a number in hexadecimal form.  For example, "01" is converted to a binary 00000001 and "FF" is
				 * converted to a binary 11111111.
				 */
				{
					CHAR* ptr = (CHAR*)ParamValue;
					int i;

					length *= 2;
					value = (TCHAR*)malloc(sizeof(TCHAR)*(length+1));

					for (i=0;i<length;i+=4)
					{/* get value */
						BYTE byte;
						if ('0' <= *ptr && '9' >= *ptr)
							byte = *ptr - '0';
						else if ('a' <= *ptr && 'f' >= *ptr) 
							byte = *ptr - 'a' + 10;
						else if ('A' <= *ptr && 'F' >= *ptr)
							byte = *ptr - 'A' + 10;
						else
						  byte = 0;
						++ptr;
						byte <<= 4;
						if ('0' <= *ptr && '9' >= *ptr)
							byte += *ptr - '0';
						else if ('a' <= *ptr && 'f' >= *ptr) 
							byte += *ptr - 'a' + 10;
						else if ('A' <= *ptr && 'F' >= *ptr)
							byte += *ptr - 'A' + 10;
						_stprintf(&value[i], _T("\\%03o"), byte);
						++ptr;
					}
					break;
				}
				default:
#ifdef UNICODE
			{
				int size;
				if (length < sizeof(NumBuffer)/sizeof(WCHAR))
				{
					value = NumBuffer;
					size = sizeof(NumBuffer)/sizeof(WCHAR);
				}
				else
					value = (WCHAR*)malloc(sizeof(WCHAR)*(size = (length+1)));

				MultiByteToWideChar(0, 0, (CHAR*)ParamValue, length, value, size);
			}
#else
			value = (TCHAR*)ParamValue;
#endif /* UNICODE */
			}
			break;
		case SQL_C_BIT:
			NumBuffer[0] = (0 == *(CHAR*)ParamValue) ? _T('0') : _T('1');
			length = 1;
			value = NumBuffer;
			break;
		case SQL_C_BINARY:
			length = (ParamIndPtr) ? ((0 <= *ParamIndPtr) ? (*ParamIndPtr) : ((SQL_NTS == *ParamIndPtr) ? strlen((CHAR*)ParamValue) : ParamLength)) : (ParamLength);
#ifdef UNICODE
			{
				int size;
				if (length < sizeof(NumBuffer)/sizeof(WCHAR))
				{
					value = NumBuffer;
					size = sizeof(NumBuffer)/sizeof(WCHAR);
				}
				else
					value = (WCHAR*)malloc(sizeof(WCHAR)*(size = (length+1)));

				MultiByteToWideChar(0, 0, (CHAR*)ParamValue, length, value, size);
			}
#else
			value = (TCHAR*)ParamValue;
#endif /* UNICODE */
			break;
		case SQL_C_FLOAT:
		case SQL_C_DOUBLE:
		{
			TCHAR dec_point = *(localeconv()->decimal_point);
			length = _stprintf(value = NumBuffer, _T("%f"), (SQL_C_FLOAT == ParamType) ? *(SQLREAL*)ParamValue : *(SQLDOUBLE*)ParamValue);

			/* if it looks like -13.000000 - remove all '0' */
			while (_T('0') == value[length-1])
				length--;
			if (dec_point == value[length-1])
				length--;
			if ((_T('.') != dec_point) &&
					(ptr = _tcschr(value, dec_point))
			   )
				 *ptr = _T('.');
			break;
		}

		case SQL_C_NUMERIC:
		{
			int i;
			TCHAR  exchange;
			TCHAR* ptr;
			SQL_NUMERIC_STRUCT param;

			memcpy(&param, ParamValue, sizeof(param));
			value = NumBuffer;
			if (0 == param.sign)
			{
				NumBuffer[0] = _T('-');
				++value;
			}

			/* find begining of the real data */
			for (i=SQL_MAX_NUMERIC_LEN-1, *value = _T('0'), ptr = value;i>=0;--i)
				if (0 != param.val[i])
					break;

			/* translate into decimal string by processing this BIG value */
			while (0 <= i)
			{
				int j;
				int runner;
				
				for (j=i, runner=0; 0<=j; --j)
				{
					runner <<= 8;
					runner += param.val[j];
					param.val[j] = runner / 10;
					runner %= 10;
				}

				/* add runner as a top value */
				*ptr++ = runner + _T('0');

				if (0 == param.val[i])
					--i;
			}

			if (_T('0') != *value)
			{/* here we have a decimal string representing the value in the mirrored digits order */
				_stprintf(ptr, _T("e-%d"), scale);

				for (--ptr;value < ptr;++value, --ptr)
				{
					exchange = *value;
					*value = *ptr;
					*ptr = exchange;
				}
			} else {
				value[1] = _T('\0');
			}
			value = NumBuffer;
			break;
		}
/*
		case SQL_C_GUID
*/
		}
	}

	if (SQL_NTS == length)
		length = _tcslen(value);

	if (NULL != (ptr = AddField(pStatement, length+1)))
	{
		_tcsncpy(ptr, value, length);
		ptr[length] = _T('\0');
		if (value != NumBuffer && value != ParamValue)
			FREE(value);
		return (((int*)ptr)-1);
	}

	return NULL;
}

/*----------------------------------------------------------------------------
 * FUNCTION: GetCDefaultType
 *
 * DESCRIPTION: returnes corresponding C-data type for SQL-data type
 *----------------------------------------------------------------------------
 */
SQLSMALLINT
GetCDefaultType(SQLSMALLINT sql_type)
{
	switch(sql_type)
	{
		case SQL_GUID:
		case SQL_CHAR:
		case SQL_VARCHAR:
		case SQL_LONGVARCHAR:
		case SQL_ALL_TYPES:
			return SQL_C_CHAR;
		case SQL_WCHAR:
		case SQL_WVARCHAR:
		case SQL_WLONGVARCHAR:
			return SQL_C_WCHAR;
		case SQL_DECIMAL:
		case SQL_NUMERIC:
			return SQL_C_CHAR;
		case SQL_BIT:
			return SQL_C_BIT;
		case SQL_TINYINT:
			return SQL_C_TINYINT;
		case SQL_SMALLINT:
			return SQL_C_SHORT;
		case SQL_INTEGER:
			return SQL_C_LONG;
		case SQL_BIGINT:
			return SQL_C_SBIGINT;
		case SQL_REAL:
			return SQL_C_FLOAT;
		case SQL_FLOAT:
		case SQL_DOUBLE:
			return SQL_C_DOUBLE;
		case SQL_BINARY:
		case SQL_VARBINARY:
		case SQL_LONGVARBINARY:
			return SQL_C_BINARY;

		/* IDBC v3.0*/
		case SQL_TYPE_DATE:
			return SQL_C_TYPE_DATE;
		case SQL_TYPE_TIME:
			return SQL_C_TYPE_TIME;
		case SQL_TYPE_TIMESTAMP:
			return SQL_C_TYPE_TIMESTAMP;
		
		/* ODBC v2.0 */
		case SQL_DATE:
			return SQL_C_DATE;
		case SQL_TIME:
			return SQL_C_TIME;
		case SQL_TIMESTAMP:
			return SQL_C_TIMESTAMP;
		/* default */
		default:
			return SQL_C_DEFAULT;
	}
}

/*----------------------------------------------------------------------------
 * FUNCTION: GetDefaultType
 *
 * DESCRIPTION: returnes corresponding SQL-data type for C-data type
 *----------------------------------------------------------------------------
 */
SQLSMALLINT
GetDefaultType(SQLSMALLINT c_type)
{
	switch(c_type)
	{
		case SQL_C_CHAR:
			return SQL_CHAR;
		case SQL_C_WCHAR:
			return SQL_WCHAR;
		case SQL_C_BIT:
			return SQL_BIT;
		case SQL_C_NUMERIC:
			return SQL_NUMERIC;
		case SQL_C_TINYINT:
			return SQL_TINYINT;
		case SQL_C_SHORT:
			return SQL_SMALLINT;
		case SQL_C_LONG:
		case SQL_C_SLONG:
		case SQL_C_ULONG:
			return SQL_INTEGER;
		case SQL_C_SBIGINT:
			return SQL_BIGINT;
		case SQL_C_FLOAT:
			return SQL_REAL;
		case SQL_C_DOUBLE:
			return SQL_DOUBLE;
		case SQL_C_BINARY:
			return SQL_BINARY;

		/* IDBC v3.0*/
		case SQL_C_TYPE_DATE:
			return SQL_TYPE_DATE;
		case SQL_C_TYPE_TIME:
			return SQL_TYPE_TIME;
		case SQL_C_TYPE_TIMESTAMP:
			return SQL_TYPE_TIMESTAMP;
		
		/* ODBC v2.0 */
		case SQL_C_DATE:
			return SQL_DATE;
		case SQL_C_TIME:
			return SQL_TIME;
		case SQL_C_TIMESTAMP:
			return SQL_TIMESTAMP;
		/* default */
		default:
			return SQL_DEFAULT;
	}
}

/*----------------------------------------------------------------------------
 * FUNCTION: GetCTypeLength
 *
 * DESCRIPTION: returnes length of the corresponding C-data type in BYTEs,
 *              char_length - means known length in CHARACTERs of character or
 *              binary data type
 *----------------------------------------------------------------------------
 */
SQLLEN
GetCTypeLength(SQLSMALLINT c_type, SQLLEN char_length)
{
	switch(c_type)
	{
		case SQL_C_USHORT:
		case SQL_C_SSHORT:
		case SQL_C_SHORT:
			return sizeof(SQLSMALLINT);
		case SQL_C_SLONG:
		case SQL_C_ULONG: /* = SQL_C_BOOKMARK */
		case SQL_C_LONG:
			return sizeof(SQLINTEGER);
		case SQL_C_FLOAT:
			return sizeof(SQLREAL);
		case SQL_C_DOUBLE:
			return sizeof(SQLDOUBLE);
		case SQL_C_BIT:
		case SQL_C_STINYINT:
		case SQL_C_UTINYINT:
		case SQL_C_TINYINT:
			return sizeof(SQLCHAR);
		case SQL_C_SBIGINT:
		case SQL_C_UBIGINT:
			return sizeof(SQLBIGINT);
		case SQL_C_DATE:
		case SQL_C_TYPE_DATE:
			return sizeof(SQL_DATE_STRUCT);
		case SQL_C_TIME:
		case SQL_C_TYPE_TIME:
			return sizeof(SQL_TIME_STRUCT);
		case SQL_C_TIMESTAMP:
		case SQL_C_TYPE_TIMESTAMP:
			return sizeof(SQL_TIMESTAMP_STRUCT);
		case SQL_C_NUMERIC:
			return sizeof(SQL_NUMERIC_STRUCT);
		case SQL_C_GUID:
			return sizeof(SQLGUID);

		/* variable-length data */
		case SQL_C_BINARY: /* = SQL_C_VARBOOKMARK*/
		/* case SQL_C_XML: */
		case SQL_C_CHAR:
			return sizeof(SQLCHAR)*char_length;
		case SQL_C_WCHAR:
			return sizeof(SQLWCHAR)*char_length;
		
		/* INTERVAL_C */
		default:
			return 0;
	}
}

/*----------------------------------------------------------------------------
 * FUNCTION: SmartAddChar
 *----------------------------------------------------------------------------
 */
SQLRETURN
SmartAddChar(Statement* pStatement, TCHAR chr, TCHAR** dest, SQLINTEGER* dest_length, SQLINTEGER* dest_cursor, BOOL* static_buffer)
{
	if (1 > *dest_length)
	{
		SQLINTEGER new_length = (*dest_length) + 1024;
		TCHAR* new_buffer = malloc(new_length * sizeof(TCHAR));
		if (NULL == new_buffer)
		{
			SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
			return SQL_ERROR;
		}

		memcpy(new_buffer, *dest, (*dest_length)*sizeof(TCHAR));

		if (FALSE != *static_buffer)
			*static_buffer = FALSE;
		else
			FREE(*dest);

		*dest_length = new_length;
		*dest        = new_buffer;
	}

	(*dest)[*dest_cursor] = chr;
	(*dest_cursor)++;

	return SQL_SUCCESS;
}


/*----------------------------------------------------------------------------
 * FUNCTION: SmartBufferCopy
 *----------------------------------------------------------------------------
 */
SQLRETURN
SmartBufferCopy(Statement* pStatement, TCHAR* src, SQLINTEGER from, SQLINTEGER till, TCHAR** dest, SQLINTEGER* dest_length, SQLINTEGER* dest_cursor, BOOL* static_buffer)
{
	SQLINTEGER count = till - from;

	if (count > *dest_length)
	{
		SQLINTEGER new_length = (*dest_length) + count;
		TCHAR* new_buffer = malloc(new_length * sizeof(TCHAR));
		if (NULL == new_buffer)
		{
			SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
			return SQL_ERROR;
		}

		memcpy(new_buffer, *dest, (*dest_length)*sizeof(TCHAR));

		if (FALSE != *static_buffer)
			*static_buffer = FALSE;
		else
			FREE(*dest);

		*dest_length = new_length;
		*dest        = new_buffer;
	}

	memcpy(&(*dest)[*dest_cursor], &src[from], count*sizeof(TCHAR));
	*dest_cursor += count;

	return SQL_SUCCESS;
}

/*----------------------------------------------------------------------------
 * FUNCTION: ProcessEscape
 *----------------------------------------------------------------------------
 */
SQLRETURN
ProcessParameter(Statement* pStatement, TCHAR* src, SQLINTEGER src_length, SQLINTEGER* src_cursor, TCHAR** dest, SQLINTEGER* dest_length, SQLINTEGER* dest_cursor, BOOL* static_buffer)
{
	BOOL       escape = FALSE;
	SQLINTEGER cursor = *src_cursor;
	TCHAR      quote  = src[cursor++];

	for(;cursor < src_length; ++cursor)
		if (_T('\\') == src[cursor])
		{
			escape = (FALSE == escape) ? TRUE : FALSE;
		}
		else if (quote == src[cursor])
		{
			if (FALSE == escape)
				break;
		}

	if (quote == src[cursor])
	{
		SQLRETURN ret = SmartBufferCopy(pStatement, src, *src_cursor, ++cursor, dest, dest_length, dest_cursor, static_buffer);
		*src_cursor = cursor;
		return ret;
	}

	return SQL_ERROR;
}

/*----------------------------------------------------------------------------
 * FUNCTION: ProcessEscape
 *----------------------------------------------------------------------------
 */
SQLRETURN
ProcessEscape(Statement* pStatement, TCHAR* src, SQLINTEGER src_length, SQLINTEGER* src_cursor, TCHAR** dest, SQLINTEGER* dest_length, SQLINTEGER* dest_cursor, BOOL* static_buffer, BOOL parameter, unsigned int* unParametersNumber)
{
	SQLINTEGER   cursor       = *src_cursor;
	ESCAPE_TYPE  escape       = ET_NO_ESCAPES;
	BOOL         skip_compare = FALSE;

	SQLINTEGER   begin;
	SQLINTEGER   last_processed = 0;

	/* skip spaces */
	while (cursor < src_length && _istspace(src[++cursor]));

	/* recognize one of the escape words:
	 * 'd'    - date 'yyyy-mm-dd'
	 * 't'    - time 'hh:mm:ss[,f...]'
	 * 'ts'   - timestamp 'yyyy-mm-dd hh:mm:ss[,f...]'
	 * 'fn'   - function
	 * 'oj'   - outer join
	 * 'call' - procedure call
	 */
	switch(src[cursor])
	{
		case _T('C'):
		case _T('c'): /* call escape */
			if (0 == _tcsnicmp(&src[++cursor], _T("all"), STR_SIZEOF("all")) && _istspace(src[cursor += STR_SIZEOF("all")]))
				escape = ET_PROCEDURE;
			break;
		case _T('D'):
		case _T('d'):	/* date escape */
			if (_istspace(src[++cursor]))
				escape = ET_DATE;
			break;
		case _T('F'):
		case _T('f'):
			if ((_T('n') == src[++cursor] || _T('N') == src[++cursor]) && _istspace(src[++cursor]))
				escape = ET_FUNCTION;								
			break;
		case _T('O'):
		case _T('o'):
			if ((_T('j') == src[++cursor] || _T('J') == src[++cursor]) && _istspace(src[++cursor]))
				escape = ET_OUTER_JOIN;
			break;
		case _T('T'):
		case _T('t'):
			if (_istspace(src[++cursor]))
				escape = ET_TIME;
			else if ((_T('s') == src[cursor] || _T('S') == src[cursor]) && _istspace(src[++cursor]))
				escape = ET_TIMESTAMP;
			break;
	}

	if (TRUE == parameter && (escape == ET_FUNCTION || escape == ET_PROCEDURE))
	{
		SetError(SQL_HANDLE_STMT, pStatement, ERR_WRONG_PARAMETER, NULL);
		return SQL_ERROR;
	}
						
	switch(escape)
	{
		case ET_DATE:
		case ET_TIME:
		case ET_TIMESTAMP:
			/* skip spaces */
			while (cursor < src_length && _istspace(src[++cursor]));

			/* check data */
			if (_T('\'') == src[cursor])
			{
				if (SQL_ERROR == ProcessParameter(pStatement, src, src_length, &cursor, dest, dest_length, dest_cursor, static_buffer))
					return SQL_ERROR;

				/* skip spaces */
				while (cursor < src_length && _istspace(src[cursor]))
					++cursor;

				if (_T('}') == src[cursor])
				{
					*src_cursor = cursor + 1;
					return SQL_SUCCESS;
				}
			}
		
			SetError(SQL_HANDLE_STMT, pStatement, ERR_BAD_QUERY_SYNTAX, NULL);
			return SQL_ERROR;
		case ET_OUTER_JOIN:
			/* skip spaces */
			while (cursor < src_length && _istspace(src[cursor]))
				++cursor;

			last_processed = cursor;
			/* process every parameter - it could be an escape */
			while (cursor < src_length && _T('}') != src[cursor])
			{
				switch (src[cursor])
				{
					case _T('?'):
						if (NULL != unParametersNumber)
						{
							(*unParametersNumber)++;
							src[cursor] = _T('\0');
						}
						++cursor;
					break;

					case _T('}'):
						/* end of escape */
						break;
					case _T('{'):
						if ((SQL_ERROR == SmartBufferCopy(pStatement, src, last_processed, cursor, dest, dest_length, dest_cursor, static_buffer)) ||
						    /* process escape */
						    (SQL_ERROR == ProcessEscape(pStatement, src, src_length, &cursor, dest, dest_length, dest_cursor, static_buffer, parameter, unParametersNumber))
						   )
						{
							SetError(SQL_HANDLE_STMT, pStatement, ERR_BAD_QUERY_SYNTAX, NULL);
							return SQL_ERROR;
						}
						last_processed = cursor;
						break;
					case _T('\''):
					case _T('\"'):
						/* this is a simple parameter */
						if ((SQL_ERROR == SmartBufferCopy(pStatement, src, last_processed, cursor, dest, dest_length, dest_cursor, static_buffer)) ||
						    /* process escape */
						    (SQL_ERROR == ProcessParameter(pStatement, src, src_length, &cursor, dest, dest_length, dest_cursor, static_buffer))
						   )
						{
							SetError(SQL_HANDLE_STMT, pStatement, ERR_BAD_QUERY_SYNTAX, NULL);
							return SQL_ERROR;
						}
						last_processed = cursor;
						break;
					default:
						++cursor;
						break;
				}
			}

			if (SQL_ERROR == SmartBufferCopy(pStatement, src, last_processed, cursor, dest, dest_length, dest_cursor, static_buffer))
			{
				SetError(SQL_HANDLE_STMT, pStatement, ERR_BAD_QUERY_SYNTAX, NULL);
				return SQL_ERROR;
			}

			break;
		case ET_PROCEDURE:
			/* PostgreSQL requires of using SELECT to call a function */
			if (SQL_ERROR == SmartBufferCopy(pStatement, _T(" SELECT "), 0, STR_SIZEOF(" SELECT "), dest, dest_length, dest_cursor, static_buffer))
				return SQL_ERROR;

			skip_compare = TRUE;
			/* NO BREAK! */
		case ET_FUNCTION:
		{
			SQLINTEGER sf = -1;

			/* skip spaces */
			while (cursor < src_length && _istspace(src[++cursor]));

			if (skip_compare)
			{/* mark function name for copy */
				begin = cursor;
				while (cursor < src_length && !_istspace(src[++cursor]) && (_T('(') != src[cursor]));
	
				if (SQL_ERROR == SmartBufferCopy(pStatement, src, begin, cursor, dest, dest_length, dest_cursor, static_buffer))
					return SQL_ERROR;
			}
			else
			{/* try to recognize scalar function */
				SQLINTEGER function_length;

				for (sf = 0; sf < sizeof(c_ScalarFunctions)/sizeof(ScalarFunction); ++sf)
				{
					function_length = _tcslen(c_ScalarFunctions[sf].odbc_function);
					if ((function_length < (src_length - cursor) &&
							(0 == _tcsnicmp(c_ScalarFunctions[sf].odbc_function, &src[cursor], function_length))) &&
							(_istspace(src[cursor + function_length]) || _T('(') == src[cursor + function_length])
						 )
					{
						function_length = _tcslen(c_ScalarFunctions[sf].postgresql_function);
						if (SQL_ERROR == SmartBufferCopy(pStatement, c_ScalarFunctions[sf].postgresql_function, 0, function_length, dest, dest_length, dest_cursor, static_buffer))
							return SQL_ERROR;

						cursor += function_length;
						break;
					}
				}
			}

			/* skip spaces */
			while (cursor < src_length && _istspace(src[cursor]))
				++cursor;

			/* this should be a '(' */
			if (_T('(') != src[cursor++])
			{
				SetError(SQL_HANDLE_STMT, pStatement, ERR_BAD_QUERY_SYNTAX, NULL);
				return SQL_ERROR;
			}

			/* do we need braces? */
			if (sf < (SQLINTEGER)(sizeof(c_ScalarFunctions)/sizeof(ScalarFunction) - 3))
				if (SQL_ERROR == SmartAddChar(pStatement, _T('('), dest, dest_length, dest_cursor, static_buffer))
					return SQL_ERROR;

			last_processed = cursor;
			/* process every parameter - it could be an escape */
			while (cursor < src_length && _T(')') != src[cursor])
			{
				switch (src[cursor])
				{
					case _T('?'):
						if (NULL != unParametersNumber)
						{
							(*unParametersNumber)++;
							src[cursor] = _T('\0');
						}
						++cursor;
					break;

					case _T(')'):
						/* no more parameter */
						break;
					case _T('{'):
						if ((SQL_ERROR == SmartBufferCopy(pStatement, src, last_processed, cursor, dest, dest_length, dest_cursor, static_buffer)) ||
						    /* process escape */
						    (SQL_ERROR == ProcessEscape(pStatement, src, src_length, &cursor, dest, dest_length, dest_cursor, static_buffer, parameter, unParametersNumber))
						   )
						{
							SetError(SQL_HANDLE_STMT, pStatement, ERR_BAD_QUERY_SYNTAX, NULL);
							return SQL_ERROR;
						}
						last_processed = cursor;
						break;
					case _T('\''):
					case _T('\"'):
						/* this is a simple parameter */
						if ((SQL_ERROR == SmartBufferCopy(pStatement, src, last_processed, cursor, dest, dest_length, dest_cursor, static_buffer)) ||
						    /* process escape */
						    (SQL_ERROR == ProcessParameter(pStatement, src, src_length, &cursor, dest, dest_length, dest_cursor, static_buffer))
						   )
						{
							SetError(SQL_HANDLE_STMT, pStatement, ERR_BAD_QUERY_SYNTAX, NULL);
							return SQL_ERROR;
						}
						last_processed = cursor;
						break;
					default:
						++cursor;
						break;
				}
			}

			if (SQL_ERROR == SmartBufferCopy(pStatement, src, last_processed, cursor, dest, dest_length, dest_cursor, static_buffer))
			{
				SetError(SQL_HANDLE_STMT, pStatement, ERR_BAD_QUERY_SYNTAX, NULL);
				return SQL_ERROR;
			}

			++cursor;

			if (sf < (SQLINTEGER)(sizeof(c_ScalarFunctions)/sizeof(ScalarFunction) - 3))
				if (SQL_ERROR == SmartAddChar(pStatement, _T(')'), dest, dest_length, dest_cursor, static_buffer))
					return SQL_ERROR;

			break;
		}
	}

	/* skip spaces */
	while (cursor < src_length && _istspace(src[cursor]))
		++cursor;

	if (_T('}') != src[cursor])
		return SQL_ERROR;

	*src_cursor = cursor + 1;
	return SQL_SUCCESS;
}

/*----------------------------------------------------------------------------
 * FUNCTION: ReplaceEscapes
 *----------------------------------------------------------------------------
 */
unsigned int ReplaceEscapes(Statement* pStatement, SQLTCHAR** StatementTextPtr, SQLINTEGER StaticBufferLength, SQLINTEGER* TextLengthPtr, BOOL ReplaceParams, TCHAR** DestQueryPtr, SQLINTEGER* DestQueryLengthPtr, BOOL parameter)
{
	unsigned int unParametersNumber = 0;

	if (SQL_STMT_NOSCAN_OFF(pStatement))
	{/* should we replace escape sequences? */
		SQLINTEGER last_processed = 0;
		BOOL       escape = FALSE;
		SQLINTEGER i;

		TCHAR  buffer[MAX_QUERY_LENGTH + 1];
		TCHAR* dest_query  = buffer;
		TCHAR* StatementText = *StatementTextPtr;

		int singleQuoter   = 0;
		int doubleQuoter   = 0;
		int escapeBegin    = 0;
		int begin          = 0;

		SQLINTEGER dest_length     = sizeof(buffer)/sizeof(TCHAR);
		SQLINTEGER dest_ptr_length = sizeof(buffer)/sizeof(TCHAR);
		SQLINTEGER dest_cursor     = 0;

		BOOL       static_buffer = TRUE;
		SQLINTEGER TextLength = *TextLengthPtr;

		if (SQL_NTS == TextLength)
			TextLength = _tcslen((TCHAR*)StatementText);

		/* calculate the number of parameters and escape sequences, we count only unquoted escape sequences! */
		for(i=0;i<TextLength;i++)
			switch(StatementText[i])
			{
				case _T('?'): /* parameter to replace */
					if (0 == (singleQuoter || doubleQuoter))
					{
						unParametersNumber++;
						if (ReplaceParams)
							StatementText[i] = _T('\0');
					}
					break;
				case _T('{'):
					escape = TRUE;
					if (0 == (singleQuoter || doubleQuoter))
					{/* escape sequence BEGIN */
					    /* copy everything before escape */
						if ((SQL_ERROR == SmartBufferCopy(pStatement, StatementText, last_processed, i, &dest_query, &dest_length, &dest_cursor, &static_buffer)) ||
						    /* process escape */
								(SQL_ERROR == ProcessEscape(pStatement, StatementText, TextLength, &i, &dest_query, &dest_length, &dest_cursor, &static_buffer, parameter, (ReplaceParams) ? &unParametersNumber : NULL))
						   )
						{
							SetError(SQL_HANDLE_STMT, pStatement, ERR_BAD_QUERY_SYNTAX, NULL);
							return 0;
						}

						/* set new cursor for copy everything */
						last_processed = i;
					}
					break;
				case _T('\''):
					singleQuoter ^= QUOTER_FOUND;
					break;
				case _T('\"'):
					doubleQuoter ^= QUOTER_FOUND;
					break;
			}

		if (escape)
		{
			if (SQL_ERROR == SmartBufferCopy(pStatement, StatementText, last_processed, TextLength, &dest_query, &dest_length, &dest_cursor, &static_buffer))
				return 0;

			memcpy(*StatementTextPtr, dest_query, dest_cursor*sizeof(TCHAR));
			(*StatementTextPtr)[dest_cursor] = _T('0');
			*TextLengthPtr = dest_cursor;
		
		}
	}

	return unParametersNumber;
}

#ifdef UNICODE
int utf8len(WCHAR* String, int Length)
{
	int length;
	int i;

	if (SQL_NTS == Length)
		Length = wcslen(String);

	for(length=0,i=0;i<Length;i++)
	{
		if (0 == (0xFF80 & String[i]))
			length += 1;
		else if (0 == (0xF800 & String[i]))
			length += 2;
		else
			length += 3;
	}
	return length;
}

#endif /* UNICODE */

#define MD5_CODE_LENGTH          4
#define MD5_HASH_LENGTH          16
#define MD5_PASSWORD_LENGTH     (MD5_HASH_LENGTH*2 + sizeof("md5") - 1)
/*=================================================================================================
 *                                MD5 implementation (RFC1321)
 *
 * WARNING! This md5 implementation uses static buffers because of we know the max length of incoming
 *          sequence to hash. Suppose UID_MAX_LENGTH + PWD_MAX_LENGTH < 2^32 bits
 *=================================================================================================
 */

#define F(x,y,z)  (((x) & (y)) | (~(x) & (z)))
#define G(x,y,z)  (((x) & (z)) | ((y) & ~(z)))
#define H(x,y,z)  ((x) ^ (y) ^ (z))
#define I(x,y,z)  ((y) ^ ((x) | ~(z)))
#define SHT_LEFT(x,n) (((x) << (n)) | ((x) >> (32 - (n))))

#define ROUND1(a,b,c,d,k,s,i) ((a) = (b) + SHT_LEFT(((a) + F((b),(c),(d)) + X[(k)] + (i)), (s)))
#define ROUND2(a,b,c,d,k,s,i) ((a) = (b) + SHT_LEFT(((a) + G((b),(c),(d)) + X[(k)] + (i)), (s)))
#define ROUND3(a,b,c,d,k,s,i) ((a) = (b) + SHT_LEFT(((a) + H((b),(c),(d)) + X[(k)] + (i)), (s)))
#define ROUND4(a,b,c,d,k,s,i) ((a) = (b) + SHT_LEFT(((a) + I((b),(c),(d)) + X[(k)] + (i)), (s)))

SQLRETURN
EncryptBytesMD5(const BYTE* input, SQLINTEGER length, BYTE* output)
{
	static const char* hex = "0123456789abcdef";
	uint32 ABCD[4] = {0x67452301, 0xefcdab89, 0x98badcfe, 0x10325476};
	BYTE   buffer[UID_MAX_LENGTH + PWD_MAX_LENGTH + 64];
	BYTE*  hash = (BYTE*)ABCD;
	SQLINTEGER i,l;
	uint32 X[16];

	/* md5 hashing implementation itself */
	/* Step 1. Append Padding Bits */	
	SQLINTEGER buffer_length = (length & ~(SQLINTEGER)0x3F) + 56;
	if (buffer_length <= length)
		buffer_length += 64;

	memset(buffer+length, 0, buffer_length-length);
	memcpy(buffer, input, length);
	buffer[length] = 0x80;
	length <<= 3;

	/* Step 2. Append Length (warning! - only 32 bits) */
	memcpy(buffer+buffer_length, &length, 4);
	memset(buffer+buffer_length+4, 0, 4); /* here should be Length High Word*/

	/* Step 3. Initialize MD Buffer - already initialized in Step 0. */
	
	/* Step 4. Process Message in 16-Word Blocks */
	l = (buffer_length+8)/64;
	for (i=0;i<l;i++)
	{
		uint32 a = ABCD[0],
		       b = ABCD[1],
					 c = ABCD[2],
					 d = ABCD[3];

		memcpy(X, buffer+(i<<6), sizeof(X));

		/* Round 1 */
		ROUND1(a, b, c, d,  0,  7, 0xd76aa478); /* 1 */
		ROUND1(d, a, b, c,  1, 12, 0xe8c7b756); /* 2 */
		ROUND1(c, d, a, b,  2, 17, 0x242070db); /* 3 */
		ROUND1(b, c, d, a,  3, 22, 0xc1bdceee); /* 4 */
		ROUND1(a, b, c, d,  4,  7, 0xf57c0faf); /* 5 */
		ROUND1(d, a, b, c,  5, 12, 0x4787c62a); /* 6 */
		ROUND1(c, d, a, b,  6, 17, 0xa8304613); /* 7 */
		ROUND1(b, c, d, a,  7, 22, 0xfd469501); /* 8 */
		ROUND1(a, b, c, d,  8,  7, 0x698098d8); /* 9 */
		ROUND1(d, a, b, c,  9, 12, 0x8b44f7af); /* 10 */
		ROUND1(c, d, a, b, 10, 17, 0xffff5bb1); /* 11 */
		ROUND1(b, c, d, a, 11, 22, 0x895cd7be); /* 12 */
		ROUND1(a, b, c, d, 12,  7, 0x6b901122); /* 13 */
		ROUND1(d, a, b, c, 13, 12, 0xfd987193); /* 14 */
		ROUND1(c, d, a, b, 14, 17, 0xa679438e); /* 15 */
		ROUND1(b, c, d, a, 15, 22, 0x49b40821); /* 16 */

		/* Round 2 */
		ROUND2(a, b, c, d,  1,  5, 0xf61e2562); /* 17 */
		ROUND2(d, a, b, c,  6,  9, 0xc040b340); /* 18 */
		ROUND2(c, d, a, b, 11, 14, 0x265e5a51); /* 19 */
		ROUND2(b, c, d, a,  0, 20, 0xe9b6c7aa); /* 20 */
		ROUND2(a, b, c, d,  5,  5, 0xd62f105d); /* 21 */
		ROUND2(d, a, b, c, 10,  9, 0x02441453); /* 22 */
		ROUND2(c, d, a, b, 15, 14, 0xd8a1e681); /* 23 */
		ROUND2(b, c, d, a,  4, 20, 0xe7d3fbc8); /* 24 */
		ROUND2(a, b, c, d,  9,  5, 0x21e1cde6); /* 25 */
		ROUND2(d, a, b, c, 14,  9, 0xc33707d6); /* 26 */
		ROUND2(c, d, a, b,  3, 14, 0xf4d50d87); /* 27 */
		ROUND2(b, c, d, a,  8, 20, 0x455a14ed); /* 28 */
		ROUND2(a, b, c, d, 13,  5, 0xa9e3e905); /* 29 */
		ROUND2(d, a, b, c,  2,  9, 0xfcefa3f8); /* 30 */
		ROUND2(c, d, a, b,  7, 14, 0x676f02d9); /* 31 */
		ROUND2(b, c, d, a, 12, 20, 0x8d2a4c8a); /* 32 */

		/* Round 3 */
		ROUND3(a, b, c, d,  5,  4, 0xfffa3942); /* 33 */
		ROUND3(d, a, b, c,  8, 11, 0x8771f681); /* 34 */
		ROUND3(c, d, a, b, 11, 16, 0x6d9d6122); /* 35 */
		ROUND3(b, c, d, a, 14, 23, 0xfde5380c); /* 36 */
		ROUND3(a, b, c, d,  1,  4, 0xa4beea44); /* 37 */
		ROUND3(d, a, b, c,  4, 11, 0x4bdecfa9); /* 38 */
		ROUND3(c, d, a, b,  7, 16, 0xf6bb4b60); /* 39 */
		ROUND3(b, c, d, a, 10, 23, 0xbebfbc70); /* 40 */
		ROUND3(a, b, c, d, 13,  4, 0x289b7ec6); /* 41 */
		ROUND3(d, a, b, c,  0, 11, 0xeaa127fa); /* 42 */
		ROUND3(c, d, a, b,  3, 16, 0xd4ef3085); /* 43 */
		ROUND3(b, c, d, a,  6, 23, 0x04881d05); /* 44 */
		ROUND3(a, b, c, d,  9,  4, 0xd9d4d039); /* 45 */
		ROUND3(d, a, b, c, 12, 11, 0xe6db99e5); /* 46 */
		ROUND3(c, d, a, b, 15, 16, 0x1fa27cf8); /* 47 */
		ROUND3(b, c, d, a,  2, 23, 0xc4ac5665); /* 48 */

		/* Round 4 */
		ROUND4(a, b, c, d,  0,  6, 0xf4292244); /* 49 */
		ROUND4(d, a, b, c,  7, 10, 0x432aff97); /* 50 */
		ROUND4(c, d, a, b, 14, 15, 0xab9423a7); /* 51 */
		ROUND4(b, c, d, a,  5, 21, 0xfc93a039); /* 52 */
		ROUND4(a, b, c, d, 12,  6, 0x655b59c3); /* 53 */
		ROUND4(d, a, b, c,  3, 10, 0x8f0ccc92); /* 54 */
		ROUND4(c, d, a, b, 10, 15, 0xffeff47d); /* 55 */
		ROUND4(b, c, d, a,  1, 21, 0x85845dd1); /* 56 */
		ROUND4(a, b, c, d,  8,  6, 0x6fa87e4f); /* 57 */
		ROUND4(d, a, b, c, 15, 10, 0xfe2ce6e0); /* 58 */
		ROUND4(c, d, a, b,  6, 15, 0xa3014314); /* 59 */
		ROUND4(b, c, d, a, 13, 21, 0x4e0811a1); /* 60 */
		ROUND4(a, b, c, d,  4,  6, 0xf7537e82); /* 61 */
		ROUND4(d, a, b, c, 11, 10, 0xbd3af235); /* 62 */
		ROUND4(c, d, a, b,  2, 15, 0x2ad7d2bb); /* 63 */
		ROUND4(b, c, d, a,  9, 21, 0xeb86d391); /* 64 */

		ABCD[0] += a;
		ABCD[1] += b;
		ABCD[2] += c;
		ABCD[3] += d;
	}

	/* convert hash into hexadecimal representation */
	for(i=sizeof(ABCD)-1;i>=0;i--,hash++)
	{
		*output++ = hex[(*hash >> 4) & 0x0F];
		*output++ = hex[*hash & 0x0F];
	}

	return SQL_SUCCESS;
}


SQLRETURN
EncryptPasswordMD5(const BYTE* pwd, const BYTE* uid, const BYTE* code, BYTE* password)
{
	BYTE buffer[UID_MAX_LENGTH + PWD_MAX_LENGTH];
	BYTE first_crypted[MD5_PASSWORD_LENGTH - (sizeof("md5") - 1) + MD5_CODE_LENGTH];
	SQLINTEGER pwd_length = strlen((const char*)pwd);
	SQLINTEGER uid_length = strlen((const char*)uid);

	/* first iteration */
	memcpy(buffer, pwd, pwd_length);
	memcpy(buffer + pwd_length, uid, uid_length);
	if (SQL_ERROR != EncryptBytesMD5(buffer, pwd_length + uid_length, first_crypted))
	{/* second iteration */
		memcpy(first_crypted + MD5_PASSWORD_LENGTH - (sizeof("md5") - 1), code, MD5_CODE_LENGTH);
		memcpy(password, "md5", sizeof("md5") - 1);
		password[MD5_PASSWORD_LENGTH] = '\0';
		return EncryptBytesMD5(first_crypted, sizeof(first_crypted), password + (sizeof("md5") - 1));
	}

	return SQL_ERROR;
}


SQLRETURN
PostgreTypeToSQLType(SQLINTEGER oid, SQLINTEGER modifier, SQLUINTEGER odbc_version, TCHAR** type_name, SQLSMALLINT* concise_sql_type, SQLINTEGER* length, SQLSMALLINT* precision, BOOL mjet)
{
	SQLSMALLINT data_precision = -1;
	SQLINTEGER  data_length    = -1;

	int typeIndex = PSQL_DATATYPES_NUMBER;
	
	/* find corresponding SQL type */
	if (PSQL_DATATYPES_NUMBER == typeIndex)
	{/* get typeIndex by TypeOID */
		for (typeIndex=0;typeIndex<PSQL_DATATYPES_NUMBER;typeIndex++)
			if (oid == c_PostgreSQLDataTypes[typeIndex].postgresql_type_oid)
				break;
	}

	if (PSQL_DATATYPES_NUMBER == typeIndex)
		/* no type with such name - use unknown type (last record) */
		--typeIndex;

	switch(oid)
	{
		case 25:
		case 17:
		case 18:
		case 19:
		case 1042:
		case 1043:
		case 17071:
		case 17072:
			data_length = (4 <= modifier) ? (modifier - 4) : c_PostgreSQLDataTypes[typeIndex].column_size;
			break;
		case 20:
		case 21:
		case 23:
		case 700:
		case 701:
		case 17069:
			data_precision = 0;
			break;
		case 1083:
		case 1114:
		case 1184:
		case 1266:
			if (0 <= modifier)
				data_precision = (SQLSMALLINT)modifier;
			break;
		case 1700:
			data_precision = (modifier & 0xffff) - 4;
			data_length    = (modifier & 0xffff0000)>>16;
			break;
	}

	/* return requested values */
	if (NULL != concise_sql_type)
	{
		SQLSMALLINT concise = c_PostgreSQLDataTypes[typeIndex].sql_type;
		if (SQL_OV_ODBC2 == odbc_version)
		{
			switch(concise)
			{
				case SQL_TYPE_DATE:
					concise = SQL_DATE;
					break;
				case SQL_TYPE_TIMESTAMP:
					concise = SQL_TIMESTAMP;
					break;
				case SQL_TYPE_TIME:
					concise = SQL_TIME;
					break;
			}
		}

		if (SQL_BIGINT == concise && FALSE != mjet)
		{/* Microsoft doesn't like SQL_BIGINT type */
			concise     = SQL_NUMERIC;
			data_length = 19;
		}

		*concise_sql_type = concise;
	}

	if (NULL != type_name)
		*type_name = (TCHAR*)c_PostgreSQLDataTypes[typeIndex].local_name;

	if (NULL != length)
		*length = data_length;

	if (NULL != precision)
		*precision = data_precision;

	return SQL_SUCCESS;
}

SQLRETURN
DescribeSQLType(SQLSMALLINT concise_sql_type, SQLINTEGER length, SQLSMALLINT precision, SQLSMALLINT* sql_type, SQLINTEGER* data_size, SQLINTEGER* num_prec_radix, SQLULEN* display_size)
{
	SQLSMALLINT type  = concise_sql_type;
	SQLINTEGER  size  = -1;
	SQLINTEGER radix  = -1;
	SQLULEN   display = 0;

	switch(concise_sql_type)
	{
		case SQL_CHAR:
		case SQL_WCHAR:
			display = (0 < (size = (0 <= length) ? length : 0)) ? size : 1024;
			break;
		case SQL_VARCHAR:
		case SQL_WVARCHAR:
			display = (0 < (size = (0 <= length) ? length : 0)) ? size : 8190;
			break;
		case SQL_LONGVARCHAR:
		case SQL_WLONGVARCHAR:
			display = (0 < (size = (0 <= length) ? length : 0)) ? size : 65536;
			break;
		case SQL_BINARY:
			display = (0 < (size = (0 <= length) ? length : 0)) ? (size * 2) : 2048;
			break;
		case SQL_VARBINARY:
			display = (0 < (size = (0 <= length) ? length : 0)) ? (size * 2) : 16380;
			break;
		case SQL_LONGVARBINARY:
			display = (0 < (size = (0 <= length) ? length : 0)) ? (size * 2) : 131072;
			break;
		case SQL_INTEGER:
			size  = 32;
			radix = 2;
			display = 11;
			break;
		case SQL_BIGINT:
			size  = 20;
			radix = 10;
			display = 20;
			break;
		case SQL_FLOAT:
		case SQL_DOUBLE:
			display = 24;
			size  = 64;
			radix = 2;
			break;
		case SQL_REAL:
			display = size  = 14;
			radix = 10;
			break;
		case SQL_TINYINT:
			size  = 8;
			radix = 2;
			display = 4;
			break;
		case SQL_SMALLINT:
			size  = 16;
			radix = 2;
			display = 6;
			break;
		case SQL_DATE:
		case SQL_TYPE_DATE:
			display = size = STR_SIZEOF("yyyy-mm-dd");
			type = SQL_DATETIME;
			break;
		case SQL_TIME:
		case SQL_TYPE_TIME:
			display = size = (0 <= precision) ? STR_SIZEOF("hh:mm:ss") : (STR_SIZEOF("hh:mm:ss.") + precision);
			type = SQL_DATETIME;
			radix = 10;
			break;
		case SQL_TIMESTAMP:
		case SQL_TYPE_TIMESTAMP:
			display = size  = (0 <= precision) ? STR_SIZEOF("yyyy-mm-dd hh:mm:ss"): (STR_SIZEOF("yyyy-mm-dd hh:mm:ss.") + precision);
			type  = SQL_DATETIME;
			radix = 10;
			break;
		case SQL_BIT:
			display = size = 1;
			break;
		case SQL_DECIMAL:
		case SQL_NUMERIC:
			display = (size = length) + 2;
			radix = 10;
			break;
		case SQL_GUID:
			display = STR_SIZEOF("aaaaaaaa-bbbb-cccc-dddd-eeeeeeeeeeee");
			break;
	}

	/* return values */
	if (NULL != sql_type)
		*sql_type = type;

	if (NULL != data_size)
		*data_size = size;

	if (NULL != num_prec_radix)
		*num_prec_radix = radix;

	if (NULL != display_size)
		*display_size = display;

	return SQL_SUCCESS;
}

SQLSMALLINT
GetIntervalSubType(SQLSMALLINT sql_type)
{
	return 3;
}

void EmptyConnectionParameters(TCHAR** parameters)
{
	int i;
	for (i=0;i<DS_PARAMETERS_NUMBER;i++)
		parameters[i][0] = _T('\0');
}

/*----------------------------------------------------------------------------
 * FUNCTION: ParsePattern
 *----------------------------------------------------------------------------
 */
SQLRETURN
ParsePattern(TCHAR* begin, TCHAR* end, Restrictions* restrictions)
{
	TCHAR      prev_char;
	TCHAR*     start;
	SQLINTEGER extrabytes = 0;
	SQLINTEGER patterns   = 0;
	SQLINTEGER schema_length = 0;
	List*      list   = &restrictions->list;
	List*      length = &restrictions->length;

	/* calculate size of memory to allocate and number of patterns to be used */
	for (prev_char=_T('\0'),start=begin;start<end;++start)
	{
		switch(*start)
		{
			case _T('%'):
			case _T('_'):
				++extrabytes;
				break;
			case _T('.'):
				++patterns;
				break;
			case _T('*'): /* simplify mask - remove repeating '*' */
				if (_T('*') == prev_char)
					--extrabytes;
				break;
			default:
				;
		}
		prev_char = *start;
	}

	if (0 == patterns)
		++patterns;

	/* for every possible pattern - make it*/
	for(start=begin;0<patterns;--patterns)
	{
		TCHAR*     schema = NULL;
		TCHAR*     table  = NULL;
		SQLINTEGER table_length  = 0;
		SQLINTEGER table_pattern_length  = 0;
		SQLINTEGER schema_pattern_length = 0;

		/* calculate required space for schema and table */
		for(prev_char=_T('\0');start<end;++start,++schema_length)
		{
			if (_T('_') == *start || _T('%') == *start)
				++schema_length;
			else if (_T('*') == *start && _T('*') == prev_char)
				--schema_length;
			else if (_T('.') == *start)
			{
				++start;
				break;
			}
			prev_char = *start;
		}

		/* 1. prepare schema */
		if (0 == (table_length = end-begin+extrabytes-schema_length)) /* WARNING! this value includes space for zero terminator */
		{/* this is not a schema pattern - but table pattern */
			if (NULL != (schema = (TCHAR*)malloc(sizeof(TCHAR)*(STR_SIZEOF("%")+1))))
				_tcscpy(schema, _T("%"));
			schema_pattern_length |= PATTERN_LONGER;
			table_length = schema_length + 1;
			start = begin;
		}
		else
		{/* we have schema and table patterns */
			if (NULL != (schema = (TCHAR*)malloc(sizeof(TCHAR)*(schema_length+1))))
			{
				/* some characters possible should be escaped */
				TCHAR*     ptr;
				SQLINTEGER i;
				for(ptr=begin,i=0,prev_char=_T('\0');i<schema_length;++ptr,++i)
				{
					if (_T('_') == *ptr || _T('%') == *ptr)
						schema[i++] = _T('\\'); /* adding escape character */

					if (_T('*') == *ptr)
					{
						if ( _T('*') != prev_char)
						{
							schema_pattern_length |= PATTERN_LONGER;
							schema[i] = _T('%');
						}
						else
							--i;
					}
					else
					{
						++schema_pattern_length;
						schema[i] = ((_T('?') != *ptr) ? *ptr : _T('_'));
					}
					prev_char = *ptr;
				}
				schema[i] = _T('\0');
			}
		}

		/* 2. prepare table */
		if (NULL != (table  = (TCHAR*)malloc(sizeof(TCHAR)*(table_length--))))
		{
			/* some characters possible should be escaped */
			TCHAR*     ptr;
			SQLINTEGER i;
			for(ptr=start,i=0,prev_char=_T('\0');i<table_length;++ptr,++i)
			{
				if (_T('_') == *ptr || _T('%') == *ptr)
					table[i++] = _T('\\'); /* adding escape character */

				if (_T('*') == *ptr)
				{
					if (_T('*') != prev_char)
					{
						table_pattern_length |= PATTERN_LONGER;
						table[i] = _T('%');
					}
					else
						--i;
				}
				else
				{
					++table_pattern_length;
					table[i] = ((_T('?') != *ptr) ? *ptr : _T('_'));
				}
				prev_char = *ptr;
			}
			table[i] = _T('\0');
		}

		AddItem(list,   (SQLHANDLE)schema);
		AddItem(length, (SQLHANDLE)(SQLLEN_TO_POINTER)schema_pattern_length);
		AddItem(list,   (SQLHANDLE)table);
		AddItem(length, (SQLHANDLE)(SQLLEN_TO_POINTER)table_pattern_length);
	}

	return SQL_SUCCESS;
}


/*----------------------------------------------------------------------------
 * FUNCTION: CheckPattern
 *
 * WARNING! We assume, that situation with '%%' is impossible - we remove 
 * repeating '*' in ParsePattern
 *----------------------------------------------------------------------------
 */
SQLRETURN
CheckPattern(TCHAR* string_begin, TCHAR* string_end, TCHAR* pattern, SQLINTEGER length)
{
	TCHAR* ptr = pattern;

	if (0 != (PATTERN_LONGER & length))
	{
		length ^= PATTERN_LONGER;

		if (0 == length) /* any string succeeded */
			return SQL_SUCCESS;
		else if ((string_end-string_begin) < length)
			return SQL_ERROR;
	}
	else if ((string_end-string_begin) != length)
		return SQL_ERROR;

	/* check pattern */
	for (;string_begin<string_end;++ptr,++string_begin)
	{
		if (_T('_') == *ptr && (ptr == pattern || _T('\\') != ptr[-1]))
		{/* the best case */
			continue;
		}
		else if (_T('%') == *ptr && (ptr == pattern || _T('\\') != ptr[-1]))
		{
			++ptr;

			while (_T('_') == *ptr)
			{/* WARNING! we don't check (string_begin < string_end) because we have already checked length */
				++string_begin;
				++ptr;
			}

			if (_T('\0') == *ptr) /* no need to check string ending */
				return SQL_SUCCESS;

			if (_T('\\') == *ptr)
				++ptr;

			for (;string_begin<string_end;++string_begin)
				if (_totupper(*string_begin) == _totupper(*ptr))
					break;

			if (string_begin == string_end)
				return SQL_ERROR;
		}
		else if (_T('_') == *string_begin && _T('\\') == *ptr && _T('_')== ptr[1])
		{
			++ptr;
		}
		else if (_totupper(*string_begin) != _totupper(*ptr))
			return SQL_ERROR;
	}

	return ((_T('\0') == *ptr) || (_T('%') == *ptr && _T('\0') == ptr[1])) ? SQL_SUCCESS : SQL_ERROR;
}


/*----------------------------------------------------------------------------
 * FUNCTION: CheckPatterns
 *----------------------------------------------------------------------------
 */
SQLRETURN
CheckPatterns(TCHAR* begin, TCHAR* end, Restrictions* restrictions)
{
	SQLRETURN   nRet = SQL_SUCCESS;
	SQLINTEGER  i;
	TCHAR**     patterns = (TCHAR**)restrictions->list.handles;
	SQLINTEGER* lengthes = (SQLINTEGER*)restrictions->length.handles;
	TCHAR*      schema_begin;
	TCHAR*      schema_end;
	TCHAR*      table_begin;
	TCHAR*      table_end;

	/* 1. get table */
	if (_T('"') == end[-1])
	{
		table_end = --end;
		for(--end;(begin <= end && _T('"') != *end);--end);
		if (begin == end)
			--end;
	}
	else
	{
		table_end = end;
		for(--end;(begin <= end && _T('.') != *end && !_istspace(*end));--end);
		if (begin == end)
			--end;
	}
	table_begin = &end[1];

	if (_T('"') == *end)
		--end;

	while (begin <= end && _istspace(*end))
		--end;

	/* 2. get schema */
	if (begin > end)
		schema_end = (schema_begin = _T("public")) + STR_SIZEOF("public");
	else
	{
		if (_T('.') == *end)
		{
			--end;
			while (begin <= end && _istspace(*end))
				--end;
		}

		if (_T('"') == *end)
		{
			schema_end = end;
			for(--end;(begin <= end && _T('"') != *end);--end);
			if (begin == end && _T('"') != *end)
				--end;
		}
		else
		{
			schema_end = &end[1];
			for(--end;(begin <= end && _T('.') != *end && !_istspace(*end));--end);
			if (begin == end)
				--end;
		}
		schema_begin = &end[1];
	}

	/* 3. possible existance of catalog - ignore */

	for (i=restrictions->list.used-1;0<=i;i-=2)
		if ((SQL_SUCCESS == CheckPattern(schema_begin, schema_end, patterns[i-1], lengthes[i-1])) &&
		    (SQL_SUCCESS == CheckPattern(table_begin, table_end, patterns[i], lengthes[i]))
		   )
			break;

	if (0>i) /* pattern doesn't match */
		nRet = SQL_ERROR;

	return nRet;
}


/*----------------------------------------------------------------------------
 * FUNCTION: PrepareRestriction
 *----------------------------------------------------------------------------
 */
SQLRETURN
PrepareRestriction(Restrictions* restrictions, TCHAR* text)
{/* we can get something like:
  * 1. *          -> "%"."%" (has no sence)
	* 2. ta?le      -> "%"."ta_le"
	* 3. sch?ma.t*  -> "sch_ma"."t%"
	* 4. sh.ma.td   -> "sh"."ma.td",
	*                  "sh.ma"."td"
	*
	*   The most unattractive case - is when income pattern has more then one point - it will give us
	* several cases of division
	*/
	SQLRETURN  nRet     = SQL_SUCCESS;
	POSITION   position = POS_OUTSIDE_READY;
	SQLINTEGER length   = _tcslen(text);
	TCHAR*     begin    = NULL;
	TCHAR*     end      = NULL; /* points to the first character after pattern */

	for(;0<length;++text,--length)
	{
		switch(*text)
		{
			case _T(' '):
				switch(position)
				{
					case POS_OUTSIDE_READY:
						break;
					case POS_INSIDE_SYS:
						end = text;
					/* no break! */
					default:
						position = POS_READY_FOR_DIVISOR;
				}
				break;
			case _T(','):
				switch(position)
				{
					case POS_OUTSIDE_READY:
						length = 0;
						break;
					case POS_INSIDE_SYS:
						end = text;
						/* no break! */
					default:
						if (!SQL_SUCCEEDED(ParsePattern(begin, end, restrictions)))
							length = 0;
						/* we have a pattern here */
						position = POS_OUTSIDE_READY;
						break;
				}
				break;
			default:
				switch(position)
				{
					case POS_READY_FOR_DIVISOR:
						length = 0;
						break;
					case POS_OUTSIDE_READY:
						begin    = text;
						position = POS_INSIDE_SYS;
						break;
					default:
						;
				}

				break;
		}
	}

	if (_T('\0') != *text)
		nRet = SQL_ERROR;
	else if (POS_OUTSIDE_READY != position)
	{/* check for the last item */
		if (POS_INSIDE_SYS == position)
			end = text;

		if (!SQL_SUCCEEDED(ParsePattern(begin, end, restrictions)))
			nRet = SQL_ERROR;
	}

	return nRet;
}
