﻿using System;
using System.Data;
using System.Data.Common;
using System.Configuration;
using System.Collections.Generic;
using System.Text;
using NpgsqlTypes;

namespace fb2pg
{
    /// <summary>
    /// Esta clase gestiona la información de la estructura de las tablas de la base de datos de Firebird
    /// </summary>
    class InfoEstructuraFirebird
    {
        #region · Campos ·
        /// <summary>
        /// Nombre del campo
        /// </summary>
        string campo;
        /// <summary>
        /// Nombre del campo
        /// </summary>
        public string Campo
        {
            get { return campo; }
        }

        /// <summary>
        /// Longitud del campo (no siempre es necesario)
        /// </summary>
        int longitud;
        /// <summary>
        /// Longitud del campo (no siempre es necesario)
        /// </summary>
        public int Longitud
        {
            get { return longitud; }
        }

        /// <summary>
        /// Precisión del campo en tipos de datos numéricos
        /// </summary>
        int precision;
        /// <summary>
        /// Precisión del campo en tipos de datos numéricos
        /// </summary>
        public int Precision
        {
            get { return precision; }
        }

        /// <summary>
        /// Indica si el campo permite valores NULL
        /// </summary>
        bool permiteNull;
        /// <summary>
        /// Indica si el campo permite valores NULL
        /// </summary>
        public bool PermiteNull
        {
            get { return permiteNull; }
        }

        /// <summary>
        /// Valor por defecto que tiene el campo en su definición
        /// </summary>
        string valorInicial;
        /// <summary>
        /// Valor por defecto que tiene el campo en su definición
        /// </summary>
        public string ValorInicial
        {
            get { return this.AjustaValorInicial(); }
        }

        /// <summary>
        /// Nombre del tipo de campo en Firebird
        /// </summary>
        string tipoDB;
        /// <summary>
        /// Nombre del tipo de campo en Firebird
        /// </summary>
        public string TipoDB
        {
            get { return tipoDB.ToLowerInvariant(); }
        }

        /// <summary>
        /// Nombre del tipo de campo en PostgreSQL
        /// </summary>
        public string TipoPG
        {
            get { return this.PgType(this.TipoDB); }
        }

        /// <summary>
        /// Descripción del campo
        /// </summary>
        string descripcion;
        /// <summary>
        /// Descripción del campo
        /// </summary>
        public string Descripcion
        {
            get { return descripcion; }
            set { descripcion = value; }
        }


        #endregion

        #region · Relleno de una lista con los campos de la tabla ·

        /// <summary>
        /// Rellena una lista con los datos de las columnas de la tabla que indica el parámetro <paramref name="nombreTabla"/>
        /// </summary>
        /// <param name="conn">Conexión con la base de datos Firebird</param>
        /// <param name="lista">Lista de objetos que vamos a rellenar</param>
        /// <param name="nombreTabla">Nombre de la tabla que estamos procesando</param>
        /// <returns><b>true</b> si todo va bien, o <b>false</b> si hay algún problema (de momento sólo controlamos que el nombre de la tabla no esté vacío)</returns>
        public static bool RellenaLista(DbConnection conn, List<InfoEstructuraFirebird> lista, string nombreTabla)
        {
            if (lista == null)
                return false;

            if (String.IsNullOrEmpty(nombreTabla))
                return false;

            lista.Clear();
            DataTable tableColumns = conn.GetSchema("Columns", new string[] { null, null, nombreTabla.ToUpperInvariant() });

            int numFilas = tableColumns.Rows.Count;
            for (int i = 0; i < numFilas; i++)
            {
                DataRow row = tableColumns.Rows[i];

                InfoEstructuraFirebird dato = new InfoEstructuraFirebird();
                dato.RellenaCampos(row);
                lista.Add(dato);
            }
            return true;
        }

        #endregion

        #region · Relleno de los campos ·

        /// <summary>
        /// Rellena los campos de la clase con los datos que contiene un <see cref="DataRow"/>
        /// generado con el método "GetSchema("Columns")"
        /// </summary>
        /// <param name="row">Objeto <see cref="DataRow"/> que contiene la definición de la columna</param>
        private void RellenaCampos(DataRow row)
        {
            this.campo = row["COLUMN_NAME"].ToString().ToLowerInvariant();
            this.tipoDB = row["COLUMN_DATA_TYPE"].ToString();

            Int32.TryParse(row["NUMERIC_PRECISION"].ToString(), out this.longitud);
            if (this.longitud == 0)
            {
                Int32.TryParse(row["COLUMN_SIZE"].ToString(), out this.longitud);
                this.longitud = this.LongitudSegunTipo(this.tipoDB, this.longitud);
            }
            Int32.TryParse(row["NUMERIC_SCALE"].ToString(), out this.precision);

            Boolean.TryParse(row["IS_NULLABLE"].ToString(), out this.permiteNull);
            string strDefault = row["COLUMN_DEFAULT"].ToString().ToUpperInvariant().Trim();
            if (!String.IsNullOrEmpty(strDefault))
            {
                int ini = strDefault.IndexOf("DEFAULT");
                if (ini != -1)
                {
                    this.valorInicial = strDefault.Substring(ini + 7).Replace("'", "").Trim();
                    if (this.valorInicial.ToUpperInvariant().Contains("NOW"))
                    {
                        if (this.TipoPG.Equals("date"))
                            this.valorInicial = "CURRENT_DATE";
                        else
                            this.valorInicial = "CURRENT_TIME";
                    }
                }
            }

            this.descripcion = row["DESCRIPTION"].ToString();


            if (!String.IsNullOrEmpty(this.descripcion))
            {
                this.descripcion = this.descripcion.Replace("Ã¡", "á");
                this.descripcion = this.descripcion.Replace("Ã©", "é");
                this.descripcion = this.descripcion.Replace("Ã­", "í");
                this.descripcion = this.descripcion.Replace("Ã³", "ó");
                this.descripcion = this.descripcion.Replace("Ãº", "ú");
                this.descripcion = this.descripcion.Replace("Ã±", "ñ");
                this.descripcion = this.descripcion.Replace("Âº", "º");
                this.descripcion = this.descripcion.Replace("Ãš", "Ú");
            }
        }

        #endregion

        #region · Métodos auxiliares ·

        /// <summary>
        /// Devuelve el tipo de datos de PostgreSQL en función del tipo de Firebird que recibe
        /// </summary>
        /// <param name="FbType">Cadena de texto que contiene el tipo de Firebird</param>
        /// <returns>Cadena de texto que contiene el tipo de PostgreSQL</returns>
        private string PgType(string FbType)
        {
            switch (FbType)
            {
                case "char":
                    return "char";
                case "varchar":
                    return "varchar";
                case "blob sub_type 1":
                    return "text";

                case "date":
                    return "date";
                case "time":
                    return "timetz";
                case "timestamp":
                    return "timestamptz";

                case "smallint":
                    return "int2";
                case "integer":
                    return "int4";
                case "bigint":
                    return "int8";
                case "numeric":
                    return "numeric";
                case "decimal":
                    return "decimal";
                case "float":
                    return "float4";
                case "double precision":
                    return "float8";

                case "blob":
                    return "oid";

                default:
                    throw new ArgumentException("Invalid data type");
            }
        }

        public bool EstipoCaracter()
        {
            NpgsqlDbType tipo = this.TipoDatosPostgresql();

            return (tipo == NpgsqlDbType.Text || tipo == NpgsqlDbType.Char || tipo == NpgsqlDbType.Varchar);
        }

        public bool EsTipoNumerico()
        {
            NpgsqlDbType tipo = this.TipoDatosPostgresql();

            return (tipo == NpgsqlDbType.Smallint || tipo == NpgsqlDbType.Integer ||
                    tipo == NpgsqlDbType.Bigint || tipo == NpgsqlDbType.Real ||
                    tipo == NpgsqlDbType.Double || tipo == NpgsqlDbType.Numeric ||
                    tipo == NpgsqlDbType.Money);
        }

        public bool EsTipoFecha()
        {
            NpgsqlDbType tipo = this.TipoDatosPostgresql();

            return (tipo == NpgsqlDbType.Date || tipo == NpgsqlDbType.Time || tipo == NpgsqlDbType.Timestamp);
        }

        /// <summary>
        /// Devuelve el tipo nativo de PostgreSQL en función del campo <see cref="InfoEstructuraFirebird.TipoPG"/>
        /// </summary>
        /// <returns></returns>
        public NpgsqlDbType TipoDatosPostgresql()
        {
            switch (this.TipoPG)
            {
                case "unknown": return NpgsqlDbType.Text;
                case "refcursor": return NpgsqlDbType.Refcursor;
                //case "char": return NpgsqlDbType.Char;
                case "char": return NpgsqlDbType.Varchar;
                case "bpchar": return NpgsqlDbType.Text;
                case "varchar": return NpgsqlDbType.Varchar;
                case "text": return NpgsqlDbType.Text;
                case "name": return NpgsqlDbType.Text;
                case "bytea": return NpgsqlDbType.Bytea;
                case "bit": return NpgsqlDbType.Bit;
                case "bool": return NpgsqlDbType.Boolean;
                case "int2": return NpgsqlDbType.Smallint;
                case "int4": return NpgsqlDbType.Integer;
                case "int8": return NpgsqlDbType.Bigint;
                case "oid": return NpgsqlDbType.Bigint;
                case "float4": return NpgsqlDbType.Real;
                case "float8": return NpgsqlDbType.Double;
                case "numeric": return NpgsqlDbType.Numeric;
                case "decimal": return NpgsqlDbType.Numeric;
                case "inet": return NpgsqlDbType.Inet;
                case "money": return NpgsqlDbType.Money;
                case "date": return NpgsqlDbType.Date;
                case "time": return NpgsqlDbType.Time;
                case "timetz": return NpgsqlDbType.Time;
                case "timestamp": return NpgsqlDbType.Timestamp;
                case "timestamptz": return NpgsqlDbType.Timestamp;
                case "point": return NpgsqlDbType.Point;
                case "lseg": return NpgsqlDbType.LSeg;
                case "path": return NpgsqlDbType.Path;
                case "box": return NpgsqlDbType.Box;
                case "circle": return NpgsqlDbType.Circle;
                case "polygon": return NpgsqlDbType.Polygon;
            }

            return NpgsqlDbType.Text;

        }

        /// <summary>
        /// Devuelve la longitud del campo, dependiendo de que éste admita o no la definición de una longitud
        /// </summary>
        /// <param name="tipo">Tipo de campo que procesamos</param>
        /// <param name="longitud">Longitud qur debería tener</param>
        /// <returns>Longitud que utilizaremos, o 0 si la definición del capo no la necesita</returns>
        private int LongitudSegunTipo(string tipo, int longitud)
        {
            switch (tipo.ToUpperInvariant())
            {
                case "CHAR":
                    return longitud;
                case "VARCHAR":
                    return longitud;
                case "NUMERIC":
                    return longitud;
                case "DECIMAL":
                    return longitud;
                case "DOUBLE PRECISION":
                    return longitud;
            }
            return 0;
        }

        /// <summary>
        /// Devuelve el valor inicial en función del tipo de campo que contiene el objeto.
        /// </summary>
        /// <returns>Cadena de texto que define el valor por defecto de la columna</returns>
        private string AjustaValorInicial()
        {
            if (!String.IsNullOrEmpty(this.valorInicial))
            {
                string tipo = this.tipoDB.ToUpperInvariant();

                if (tipo.Equals("DATE"))
                    return "CURRENT_DATE";

                if (tipo.Equals("TIME"))
                    return "CURRENT_TIME";

                if (tipo.Equals("TIMESTAMP"))
                    return "CURRENT_TIMESTAMP";

                if (tipo.Equals("CHAR") || tipo.Equals("VARCHAR") || tipo.Equals("BLOB SUB_TYPE 1"))
                    return String.Format("'{0}'", this.valorInicial);
            }
            return this.valorInicial;
        }

        /// <summary>
        /// Devuelve los nombres de los campos que están en la lista, separados por comas
        /// </summary>
        /// <param name="listaCampos">Lista de objetos <see cref="InfoEstructuraFirebird"/></param>
        /// <param name="prefijo">Prefijo que anteponemos al nombre del campo (lo utilizamos en los parámetros de las sentencias SQL)</param>
        /// <returns>Cadena de texto con los nombres de los campos de la lista separados por comas</returns>
        public static string ListaCampos(List<InfoEstructuraFirebird> listaCampos, string prefijo)
        {
            CommaDelimitedStringCollection collection = new CommaDelimitedStringCollection();

            listaCampos.ForEach(
                delegate(InfoEstructuraFirebird c)
                {
                    collection.Add(String.Format("{0}{1}", prefijo, c.Campo));
                });

            return collection.ToString();
        }

        #endregion

    }
}
