-- French towns / Communes françaises

-- Authoritative source / Source faisant autorité :
-- (No permanent link, sorry) / (Pas de lien stable, hélas) 

-- Towns / Communes :
-- http://www.insee.fr/fr/nom_def_met/nomenclatures/cog/dbf/2005/txt/comsimp2005.zip

-- All :
-- http://www.insee.fr/fr/nom_def_met/nomenclatures/cog/cog.telechargement.asp

BEGIN;

-- Database schema / Schéma de la base de données

-- Regions / Régions
CREATE TABLE Regions (
   id SERIAL UNIQUE NOT NULL,
   code VARCHAR(4) UNIQUE NOT NULL, 
   capital VARCHAR(10) NOT NULL, -- REFERENCES Towns (code),
   name TEXT UNIQUE NOT NULL
);

-- Departments / Départements
CREATE TABLE Departments (
   id SERIAL UNIQUE NOT NULL,
   code VARCHAR(4) UNIQUE NOT NULL, -- Not always numeric, for instance Corsica 
                                    -- with 2A and 2B
   capital VARCHAR(10) UNIQUE NOT NULL, -- REFERENCES Towns (code), 
             -- Actually, it is the concatenation of D.code + T.code.
   region VARCHAR(4) NOT NULL REFERENCES Regions (code),
   name TEXT UNIQUE NOT NULL
);

-- Towns / Communes
CREATE TABLE Towns (
   id SERIAL UNIQUE NOT NULL,
   code VARCHAR(10) NOT NULL, -- Only unique inside a department
   article TEXT,
   name TEXT NOT NULL, -- Names are not really unique, for instance 'Sainte-Croix'
   department VARCHAR(4) NOT NULL REFERENCES Departments (code),
   UNIQUE (code, department)
   -- UNIQUE (name, department) -- Not perfectly unique but almost
  );
--
-- PostgreSQL database dump
--

SET client_encoding = 'utf-8';
SET check_function_bodies = false;

SET search_path = public, pg_catalog;

--
-- Data for TOC entry 5 (OID 1241120)
-- Name: regions; Type: TABLE DATA; Schema: public; Owner: bortzmeyer
--

COPY regions (id, code, capital, name) FROM stdin;
1	01	97105	Guadeloupe
2	02	97209	Martinique
3	03	97302	Guyane
4	04	97411	La Réunion
5	11	75056	Île-de-France
6	21	51108	Champagne-Ardenne
7	22	80021	Picardie
8	23	76540	Haute-Normandie
9	24	45234	Centre
10	25	14118	Basse-Normandie
11	26	21231	Bourgogne
12	31	59350	Nord-Pas-de-Calais
13	41	57463	Lorraine
14	42	67482	Alsace
15	43	25056	Franche-Comté
16	52	44109	Pays de la Loire
17	53	35238	Bretagne
18	54	86194	Poitou-Charentes
19	72	33063	Aquitaine
20	73	31555	Midi-Pyrénées
21	74	87085	Limousin
22	82	69123	Rhône-Alpes
23	83	63113	Auvergne
24	91	34172	Languedoc-Roussillon
25	93	13055	Provence-Alpes-Côte d'Azur
26	94	2A004	Corse
\.


--
-- Data for TOC entry 6 (OID 1241134)
-- Name: departments; Type: TABLE DATA; Schema: public; Owner: bortzmeyer
--

COPY departments (id, code, capital, region, name) FROM stdin;
1	01	01053	82	Ain
2	02	02408	22	Aisne
3	03	03190	83	Allier
4	04	04070	93	Alpes-de-Haute-Provence
5	05	05061	93	Hautes-Alpes
6	06	06088	93	Alpes-Maritimes
7	07	07186	82	Ardèche
8	08	08105	21	Ardennes
9	09	09122	73	Ariège
10	10	10387	21	Aube
11	11	11069	91	Aude
12	12	12202	73	Aveyron
13	13	13055	93	Bouches-du-Rhône
14	14	14118	25	Calvados
15	15	15014	83	Cantal
16	16	16015	54	Charente
17	17	17300	54	Charente-Maritime
18	18	18033	24	Cher
19	19	19272	74	Corrèze
20	2A	2A004	94	Corse-du-Sud
21	2B	2B033	94	Haute-Corse
22	21	21231	26	Côte-d'Or
23	22	22278	53	Côtes-d'Armor
24	23	23096	74	Creuse
25	24	24322	72	Dordogne
26	25	25056	43	Doubs
27	26	26362	82	Drôme
28	27	27229	23	Eure
29	28	28085	24	Eure-et-Loir
30	29	29232	53	Finistère
31	30	30189	91	Gard
32	31	31555	73	Haute-Garonne
33	32	32013	73	Gers
34	33	33063	72	Gironde
35	34	34172	91	Hérault
36	35	35238	53	Ille-et-Vilaine
37	36	36044	24	Indre
38	37	37261	24	Indre-et-Loire
39	38	38185	82	Isère
40	39	39300	43	Jura
41	40	40192	72	Landes
42	41	41018	24	Loir-et-Cher
43	42	42218	82	Loire
44	43	43157	83	Haute-Loire
45	44	44109	52	Loire-Atlantique
46	45	45234	24	Loiret
47	46	46042	73	Lot
48	47	47001	72	Lot-et-Garonne
49	48	48095	91	Lozère
50	49	49007	52	Maine-et-Loire
51	50	50502	25	Manche
52	51	51108	21	Marne
53	52	52121	21	Haute-Marne
54	53	53130	52	Mayenne
55	54	54395	41	Meurthe-et-Moselle
56	55	55029	41	Meuse
57	56	56260	53	Morbihan
58	57	57463	41	Moselle
59	58	58194	26	Nièvre
60	59	59350	31	Nord
61	60	60057	22	Oise
62	61	61001	25	Orne
63	62	62041	31	Pas-de-Calais
64	63	63113	83	Puy-de-Dôme
65	64	64445	72	Pyrénées-Atlantiques
66	65	65440	73	Hautes-Pyrénées
67	66	66136	91	Pyrénées-Orientales
68	67	67482	42	Bas-Rhin
69	68	68066	42	Haut-Rhin
70	69	69123	82	Rhône
71	70	70550	43	Haute-Saône
72	71	71270	26	Saône-et-Loire
73	72	72181	52	Sarthe
74	73	73065	82	Savoie
75	74	74010	82	Haute-Savoie
76	75	75056	11	Paris
77	76	76540	23	Seine-Maritime
78	77	77288	11	Seine-et-Marne
79	78	78646	11	Yvelines
80	79	79191	54	Deux-Sèvres
81	80	80021	22	Somme
82	81	81004	73	Tarn
83	82	82121	73	Tarn-et-Garonne
84	83	83137	93	Var
85	84	84007	93	Vaucluse
86	85	85191	52	Vendée
87	86	86194	54	Vienne
88	87	87085	74	Haute-Vienne
89	88	88160	41	Vosges
90	89	89024	26	Yonne
91	90	90010	43	Territoire de Belfort
92	91	91228	11	Essonne
93	92	92050	11	Hauts-de-Seine
94	93	93008	11	Seine-Saint-Denis
95	94	94028	11	Val-de-Marne
96	95	95500	11	Val-d'Oise
97	971	97105	01	Guadeloupe
98	972	97209	02	Martinique
99	973	97302	03	Guyane
100	974	97411	04	La Réunion
\.


--
-- Data for TOC entry 7 (OID 1241154)
-- Name: towns; Type: TABLE DATA; Schema: public; Owner: bortzmeyer
--

COPY towns (id, code, article, name, department) FROM stdin;
1	001	\N	Abergement-Clémenciat	01
2	002	\N	Abergement-de-Varey	01
3	004	\N	Ambérieu-en-Bugey	01
4	005	\N	Ambérieux-en-Dombes	01
5	006	\N	Ambléon	01
6	007	\N	Ambronay	01
7	008	\N	Ambutrix	01
8	009	\N	Andert-et-Condon	01
9	010	\N	Anglefort	01
10	011	\N	Apremont	01
11	012	\N	Aranc	01
12	013	\N	Arandas	01
13	014	\N	Arbent	01
14	015	\N	Arbignieu	01
15	016	\N	Arbigny	01
16	017	\N	Argis	01
17	019	\N	Armix	01
18	021	\N	Ars-sur-Formans	01
19	022	\N	Artemare	01
20	023	\N	Asnières-sur-Saône	01
21	024	\N	Attignat	01
22	025	\N	Bâgé-la-Ville	01
23	026	\N	Bâgé-le-Châtel	01
24	027	\N	Balan	01
25	028	\N	Baneins	01
26	029	\N	Beaupont	01
27	030	\N	Beauregard	01
28	031	\N	Bellignat	01
29	032	\N	Béligneux	01
30	033	\N	Bellegarde-sur-Valserine	01
31	034	\N	Belley	01
32	035	\N	Belleydoux	01
33	036	\N	Belmont-Luthézieu	01
34	037	\N	Bénonces	01
35	038	\N	Bény	01
36	039	\N	Béon	01
37	040	\N	Béréziat	01
38	041	\N	Bettant	01
39	042	\N	Bey	01
40	043	\N	Beynost	01
41	044	\N	Billiat	01
42	045	\N	Birieux	01
43	046	\N	Biziat	01
44	047	\N	Blyes	01
45	049	\N	Boisse	01
46	050	\N	Boissey	01
47	051	\N	Bolozon	01
48	052	\N	Bouligneux	01
49	053	\N	Bourg-en-Bresse	01
50	054	\N	Bourg-Saint-Christophe	01
51	056	\N	Boyeux-Saint-Jérôme	01
52	057	\N	Boz	01
53	058	\N	Brégnier-Cordon	01
54	059	\N	Brénaz	01
55	060	\N	Brénod	01
56	061	\N	Brens	01
57	062	\N	Bressolles	01
58	063	\N	Brion	01
59	064	\N	Briord	01
60	065	\N	Buellas	01
61	066	\N	Burbanche	01
62	067	\N	Ceignes	01
63	068	\N	Cerdon	01
64	069	\N	Certines	01
65	071	\N	Cessy	01
66	072	\N	Ceyzériat	01
67	073	\N	Ceyzérieu	01
68	074	\N	Chalamont	01
69	075	\N	Chaleins	01
70	076	\N	Chaley	01
71	077	\N	Challes	01
72	078	\N	Challex	01
73	079	\N	Champagne-en-Valromey	01
74	080	\N	Champdor	01
75	081	\N	Champfromier	01
76	082	\N	Chanay	01
77	083	\N	Chaneins	01
78	084	\N	Chanoz-Châtenay	01
79	085	\N	Chapelle-du-Châtelard	01
80	087	\N	Charix	01
81	088	\N	Charnoz-sur-Ain	01
82	089	\N	Château-Gaillard	01
83	090	\N	Châtenay	01
84	091	\N	Châtillon-en-Michaille	01
85	092	\N	Châtillon-la-Palud	01
86	093	\N	Châtillon-sur-Chalaronne	01
87	094	\N	Chavannes-sur-Reyssouze	01
88	095	\N	Chavannes-sur-Suran	01
89	096	\N	Chaveyriat	01
90	097	\N	Chavornay	01
91	098	\N	Chazey-Bons	01
92	099	\N	Chazey-sur-Ain	01
93	100	\N	Cheignieu-la-Balme	01
94	101	\N	Chevillard	01
95	102	\N	Chevroux	01
96	103	\N	Chevry	01
97	104	\N	Chézery-Forens	01
98	105	\N	Civrieux	01
99	106	\N	Cize	01
100	107	\N	Cleyzieu	01
101	108	\N	Coligny	01
102	109	\N	Collonges	01
103	110	\N	Colomieu	01
104	111	\N	Conand	01
105	112	\N	Condamine	01
106	113	\N	Condeissiat	01
107	114	\N	Confort	01
108	115	\N	Confrançon	01
109	116	\N	Contrevoz	01
110	117	\N	Conzieu	01
111	118	\N	Corbonod	01
112	119	\N	Corcelles	01
113	121	\N	Corlier	01
114	122	\N	Cormaranche-en-Bugey	01
115	123	\N	Cormoranche-sur-Saône	01
116	124	\N	Cormoz	01
117	125	\N	Corveissiat	01
118	127	\N	Courmangoux	01
119	128	\N	Courtes	01
120	129	\N	Crans	01
121	130	\N	Cras-sur-Reyssouze	01
122	133	\N	Cressin-Rochefort	01
123	134	\N	Crottet	01
124	135	\N	Crozet	01
125	136	\N	Cruzilles-lès-Mépillat	01
126	138	\N	Culoz	01
127	139	\N	Curciat-Dongalon	01
128	140	\N	Curtafond	01
129	141	\N	Cuzieu	01
130	142	\N	Dagneux	01
131	143	\N	Divonne-les-Bains	01
132	144	\N	Dommartin	01
133	145	\N	Dompierre-sur-Veyle	01
134	146	\N	Dompierre-sur-Chalaronne	01
135	147	\N	Domsure	01
136	148	\N	Dortan	01
137	149	\N	Douvres	01
138	150	\N	Drom	01
139	151	\N	Druillat	01
140	152	\N	Échallon	01
141	153	\N	Échenevex	01
142	154	\N	Étrez	01
143	155	\N	Évosges	01
144	156	\N	Faramans	01
145	157	\N	Fareins	01
146	158	\N	Farges	01
147	159	\N	Feillens	01
148	160	\N	Ferney-Voltaire	01
149	162	\N	Flaxieu	01
150	163	\N	Foissiat	01
151	165	\N	Francheleins	01
152	166	\N	Frans	01
153	167	\N	Garnerans	01
154	169	\N	Genouilleux	01
155	170	\N	Géovreissiat	01
156	171	\N	Géovreisset	01
157	172	\N	Germagnat	01
158	173	\N	Gex	01
159	174	\N	Giron	01
160	175	\N	Gorrevod	01
161	176	\N	Grand-Abergement	01
162	177	\N	Grand-Corent	01
163	179	\N	Grièges	01
164	180	\N	Grilly	01
165	181	\N	Groissiat	01
166	182	\N	Groslée	01
167	183	\N	Guéreins	01
168	184	\N	Hautecourt-Romanèche	01
169	185	\N	Hauteville-Lompnes	01
170	186	\N	Hostias	01
171	187	\N	Hotonnes	01
172	188	\N	Illiat	01
173	189	\N	Injoux-Génissiat	01
174	190	\N	Innimond	01
175	191	\N	Izenave	01
176	192	\N	Izernore	01
177	193	\N	Izieu	01
178	194	\N	Jassans-Riottier	01
179	195	\N	Jasseron	01
180	196	\N	Jayat	01
181	197	\N	Journans	01
182	198	\N	Joyeux	01
183	199	\N	Jujurieux	01
184	200	\N	Labalme	01
185	202	\N	Lagnieu	01
186	203	\N	Laiz	01
187	204	\N	Lalleyriat	01
188	205	\N	Lancrans	01
189	206	\N	Lantenay	01
190	207	\N	Lapeyrouse	01
191	208	\N	Lavours	01
192	209	\N	Léaz	01
193	210	\N	Lélex	01
194	211	\N	Lent	01
195	212	\N	Lescheroux	01
196	213	\N	Leyment	01
197	214	\N	Leyssard	01
198	215	\N	Lhôpital	01
199	216	\N	Lhuis	01
200	218	\N	Lochieu	01
201	219	\N	Lompnas	01
202	221	\N	Lompnieu	01
203	224	\N	Loyettes	01
204	225	\N	Lurcy	01
205	227	\N	Magnieu	01
206	228	\N	Maillat	01
207	229	\N	Malafretaz	01
208	230	\N	Mantenay-Montlin	01
209	231	\N	Manziat	01
210	232	\N	Marboz	01
211	233	\N	Marchamp	01
212	234	\N	Marignieu	01
213	235	\N	Marlieux	01
214	236	\N	Marsonnas	01
215	237	\N	Martignat	01
216	238	\N	Massieux	01
217	239	\N	Massignieu-de-Rives	01
218	240	\N	Matafelon-Granges	01
219	241	\N	Meillonnas	01
220	242	\N	Mérignat	01
221	243	\N	Messimy-sur-Saône	01
222	244	\N	Meximieux	01
223	245	\N	Bohas-Meyriat-Rignat	01
224	246	\N	Mézériat	01
225	247	\N	Mijoux	01
226	248	\N	Mionnay	01
227	249	\N	Miribel	01
228	250	\N	Misérieux	01
229	252	\N	Mogneneins	01
230	254	\N	Montagnat	01
231	255	\N	Montagnieu	01
232	257	\N	Montanges	01
233	258	\N	Montceaux	01
234	259	\N	Montcet	01
235	260	\N	Montellier	01
236	261	\N	Monthieux	01
237	262	\N	Montluel	01
238	263	\N	Montmerle-sur-Saône	01
239	264	\N	Montracol	01
240	265	\N	Montréal-la-Cluse	01
241	266	\N	Montrevel-en-Bresse	01
242	267	\N	Nurieux-Volognat	01
243	268	\N	Murs-et-Gélignieux	01
244	269	\N	Nantua	01
245	271	\N	Nattages	01
246	272	\N	Neuville-les-Dames	01
247	273	\N	Neuville-sur-Ain	01
248	274	\N	Neyrolles	01
249	275	\N	Neyron	01
250	276	\N	Niévroz	01
251	277	\N	Nivollet-Montgriffon	01
252	279	\N	Oncieu	01
253	280	\N	Ordonnaz	01
254	281	\N	Ornex	01
255	282	\N	Outriaz	01
256	283	\N	Oyonnax	01
257	284	\N	Ozan	01
258	285	\N	Parcieux	01
259	286	\N	Parves	01
260	288	\N	Péron	01
261	289	\N	Péronnas	01
262	290	\N	Pérouges	01
263	291	\N	Perrex	01
264	292	\N	Petit-Abergement	01
265	293	\N	Peyriat	01
266	294	\N	Peyrieu	01
267	295	\N	Peyzieux-sur-Saône	01
268	296	\N	Pirajoux	01
269	297	\N	Pizay	01
270	298	\N	Plagne	01
271	299	\N	Plantay	01
272	300	\N	Poizat	01
273	301	\N	Polliat	01
274	302	\N	Pollieu	01
275	303	\N	Poncin	01
276	304	\N	Pont-d'Ain	01
277	305	\N	Pont-de-Vaux	01
278	306	\N	Pont-de-Veyle	01
279	307	\N	Port	01
280	308	\N	Pougny	01
281	309	\N	Pouillat	01
282	310	\N	Prémeyzel	01
283	311	\N	Prémillieu	01
284	312	\N	Pressiat	01
285	313	\N	Prévessin-Moëns	01
286	314	\N	Priay	01
287	316	\N	Pugieu	01
288	317	\N	Ramasse	01
289	318	\N	Rancé	01
290	319	\N	Relevant	01
291	320	\N	Replonges	01
292	321	\N	Revonnas	01
293	322	\N	Reyrieux	01
294	323	\N	Reyssouze	01
295	325	\N	Rignieux-le-Franc	01
296	328	\N	Romans	01
297	329	\N	Rossillon	01
298	330	\N	Ruffieu	01
299	331	\N	Saint-Alban	01
300	332	\N	Saint-André-de-Bâgé	01
301	333	\N	Saint-André-de-Corcy	01
302	334	\N	Saint-André-d'Huiriat	01
303	335	\N	Saint-André-le-Bouchoux	01
304	336	\N	Saint-André-sur-Vieux-Jonc	01
305	337	\N	Saint-Bénigne	01
306	338	\N	Saint-Benoît	01
307	339	\N	Saint-Bernard	01
308	340	\N	Saint-Bois	01
309	341	\N	Saint-Champ	01
310	342	\N	Sainte-Croix	01
311	343	\N	Saint-Cyr-sur-Menthon	01
312	344	\N	Saint-Denis-lès-Bourg	01
313	345	\N	Saint-Denis-en-Bugey	01
314	346	\N	Saint-Didier-d'Aussiat	01
315	347	\N	Saint-Didier-de-Formans	01
316	348	\N	Saint-Didier-sur-Chalaronne	01
317	349	\N	Saint-Éloi	01
318	350	\N	Saint-Étienne-du-Bois	01
319	351	\N	Saint-Étienne-sur-Chalaronne	01
320	352	\N	Saint-Étienne-sur-Reyssouze	01
321	353	\N	Sainte-Euphémie	01
322	354	\N	Saint-Genis-Pouilly	01
323	355	\N	Saint-Genis-sur-Menthon	01
324	356	\N	Saint-Georges-sur-Renon	01
325	357	\N	Saint-Germain-de-Joux	01
326	358	\N	Saint-Germain-les-Paroisses	01
327	359	\N	Saint-Germain-sur-Renon	01
328	360	\N	Saint-Jean-de-Gonville	01
329	361	\N	Saint-Jean-de-Niost	01
330	362	\N	Saint-Jean-de-Thurigneux	01
331	363	\N	Saint-Jean-le-Vieux	01
332	364	\N	Saint-Jean-sur-Reyssouze	01
333	365	\N	Saint-Jean-sur-Veyle	01
334	366	\N	Sainte-Julie	01
335	367	\N	Saint-Julien-sur-Reyssouze	01
336	368	\N	Saint-Julien-sur-Veyle	01
337	369	\N	Saint-Just	01
338	370	\N	Saint-Laurent-sur-Saône	01
339	371	\N	Saint-Marcel	01
340	372	\N	Saint-Martin-de-Bavel	01
341	373	\N	Saint-Martin-du-Frêne	01
342	374	\N	Saint-Martin-du-Mont	01
343	375	\N	Saint-Martin-le-Châtel	01
344	376	\N	Saint-Maurice-de-Beynost	01
345	378	\N	Saint-Maurice-de-Gourdans	01
346	379	\N	Saint-Maurice-de-Rémens	01
347	380	\N	Saint-Nizier-le-Bouchoux	01
348	381	\N	Saint-Nizier-le-Désert	01
349	382	\N	Sainte-Olive	01
350	383	\N	Saint-Paul-de-Varax	01
351	384	\N	Saint-Rambert-en-Bugey	01
352	385	\N	Saint-Rémy	01
353	386	\N	Saint-Sorlin-en-Bugey	01
354	387	\N	Saint-Sulpice	01
355	388	\N	Saint-Trivier-de-Courtes	01
356	389	\N	Saint-Trivier-sur-Moignans	01
357	390	\N	Saint-Vulbas	01
358	391	\N	Salavre	01
359	392	\N	Samognat	01
360	393	\N	Sandrans	01
361	396	\N	Sault-Brénaz	01
362	397	\N	Sauverny	01
363	398	\N	Savigneux	01
364	399	\N	Ségny	01
365	400	\N	Seillonnaz	01
366	401	\N	Sergy	01
367	402	\N	Sermoyer	01
368	403	\N	Serrières-de-Briord	01
369	404	\N	Serrières-sur-Ain	01
370	405	\N	Servas	01
371	406	\N	Servignat	01
372	407	\N	Seyssel	01
373	408	\N	Simandre-sur-Suran	01
374	409	\N	Songieu	01
375	410	\N	Sonthonnax-la-Montagne	01
376	411	\N	Souclin	01
377	412	\N	Sulignat	01
378	413	\N	Surjoux	01
379	414	\N	Sutrieu	01
380	415	\N	Talissieu	01
381	416	\N	Tenay	01
382	417	\N	Thézillieu	01
383	418	\N	Thil	01
384	419	\N	Thoiry	01
385	420	\N	Thoissey	01
386	421	\N	Torcieu	01
387	422	\N	Tossiat	01
388	423	\N	Toussieux	01
389	424	\N	Tramoyes	01
390	425	\N	Tranclière	01
391	426	\N	Treffort-Cuisiat	01
392	427	\N	Trévoux	01
393	428	\N	Valeins	01
394	429	\N	Vandeins	01
395	430	\N	Varambon	01
396	431	\N	Vaux-en-Bugey	01
397	432	\N	Verjon	01
398	433	\N	Vernoux	01
399	434	\N	Versailleux	01
400	435	\N	Versonnex	01
401	436	\N	Vesancy	01
402	437	\N	Vescours	01
403	439	\N	Vésines	01
404	441	\N	Vieu-d'Izenave	01
405	442	\N	Vieu	01
406	443	\N	Villars-les-Dombes	01
407	444	\N	Villebois	01
408	445	\N	Villemotier	01
409	446	\N	Villeneuve	01
410	447	\N	Villereversure	01
411	448	\N	Villes	01
412	449	\N	Villette-sur-Ain	01
413	450	\N	Villieu-Loyes-Mollon	01
414	451	\N	Viriat	01
415	452	\N	Virieu-le-Grand	01
416	453	\N	Virieu-le-Petit	01
417	454	\N	Virignin	01
418	456	\N	Vongnes	01
419	457	\N	Vonnas	01
420	001	\N	Abbécourt	02
421	002	\N	Achery	02
422	003	\N	Acy	02
423	004	\N	Agnicourt-et-Séchelles	02
424	005	\N	Aguilcourt	02
425	006	\N	Aisonville-et-Bernoville	02
426	007	\N	Aizelles	02
427	008	\N	Aizy-Jouy	02
428	009	\N	Alaincourt	02
429	010	\N	Allemant	02
430	011	\N	Ambleny	02
431	012	\N	Ambrief	02
432	013	\N	Amifontaine	02
433	014	\N	Amigny-Rouy	02
434	015	\N	Ancienville	02
435	016	\N	Andelain	02
436	017	\N	Anguilcourt-le-Sart	02
437	018	\N	Anizy-le-Château	02
438	019	\N	Annois	02
439	020	\N	Any-Martin-Rieux	02
440	021	\N	Archon	02
441	022	\N	Arcy-Sainte-Restitue	02
442	023	\N	Armentières-sur-Ourcq	02
443	024	\N	Arrancy	02
444	025	\N	Artemps	02
445	026	\N	Artonges	02
446	027	\N	Assis-sur-Serre	02
447	028	\N	Athies-sous-Laon	02
448	029	\N	Attilly	02
449	030	\N	Aubencheul-aux-Bois	02
450	031	\N	Aubenton	02
451	032	\N	Aubigny-aux-Kaisnes	02
452	033	\N	Aubigny-en-Laonnois	02
453	034	\N	Audignicourt	02
454	035	\N	Audigny	02
455	036	\N	Augy	02
456	037	\N	Aulnois-sous-Laon	02
457	038	\N	Autels	02
458	039	\N	Autremencourt	02
459	040	\N	Autreppes	02
460	041	\N	Autreville	02
461	042	\N	Azy-sur-Marne	02
462	043	\N	Bagneux	02
463	044	\N	Bancigny	02
464	046	\N	Barenton-Bugny	02
465	047	\N	Barenton-Cel	02
466	048	\N	Barenton-sur-Serre	02
467	049	\N	Barisis	02
468	050	\N	Barzy-en-Thiérache	02
469	051	\N	Barzy-sur-Marne	02
470	052	\N	Bassoles-Aulers	02
471	053	\N	Baulne-en-Brie	02
472	054	\N	Bazoches-sur-Vesles	02
473	055	\N	Beaumé	02
474	056	\N	Beaumont-en-Beine	02
475	057	\N	Beaurevoir	02
476	058	\N	Beaurieux	02
477	059	\N	Beautor	02
478	060	\N	Beauvois-en-Vermandois	02
479	061	\N	Becquigny	02
480	062	\N	Belleau	02
481	063	\N	Bellenglise	02
482	064	\N	Belleu	02
483	065	\N	Bellicourt	02
484	066	\N	Benay	02
485	067	\N	Bergues-sur-Sambre	02
486	068	\N	Berlancourt	02
487	069	\N	Berlise	02
488	070	\N	Bernot	02
489	071	\N	Berny-Rivière	02
490	072	\N	Berrieux	02
491	073	\N	Berry-au-Bac	02
492	074	\N	Bertaucourt-Epourdon	02
493	075	\N	Berthenicourt	02
494	076	\N	Bertricourt	02
495	077	\N	Berzy-le-Sec	02
496	078	\N	Besmé	02
497	079	\N	Besmont	02
498	080	\N	Besny-et-Loizy	02
499	081	\N	Béthancourt-en-Vaux	02
500	082	\N	Beugneux	02
501	083	\N	Beuvardes	02
502	084	\N	Bézu-le-Guéry	02
503	085	\N	Bézu-Saint-Germain	02
504	086	\N	Bichancourt	02
505	087	\N	Bieuxy	02
506	088	\N	Bièvres	02
507	089	\N	Billy-sur-Aisne	02
508	090	\N	Billy-sur-Ourcq	02
509	091	\N	Blanzy-lès-Fismes	02
510	093	\N	Blérancourt	02
511	094	\N	Blesmes	02
512	095	\N	Bohain-en-Vermandois	02
513	096	\N	Bois-lès-Pargny	02
514	097	\N	Boncourt	02
515	098	\N	Bonneil	02
516	099	\N	Bonnesvalyn	02
517	100	\N	Bony	02
518	101	\N	Bosmont-sur-Serre	02
519	102	\N	Bouconville-Vauclair	02
520	103	\N	Boué	02
521	104	\N	Bouffignereux	02
522	105	\N	Bouresches	02
523	106	\N	Bourg-et-Comin	02
524	107	\N	Bourguignon-sous-Coucy	02
525	108	\N	Bourguignon-sous-Montbavin	02
526	109	\N	Bouteille	02
527	110	\N	Braine	02
528	111	\N	Brancourt-en-Laonnois	02
529	112	\N	Brancourt-le-Grand	02
530	114	\N	Brasles	02
531	115	\N	Braye-en-Laonnois	02
532	116	\N	Braye-en-Thiérache	02
533	117	\N	Bray-Saint-Christophe	02
534	118	\N	Braye	02
535	119	\N	Brécy	02
536	120	\N	Brenelle	02
537	121	\N	Breny	02
538	122	\N	Brie	02
539	123	\N	Brissay-Choigny	02
540	124	\N	Brissy-Hamégicourt	02
541	125	\N	Brumetz	02
542	126	\N	Brunehamel	02
543	127	\N	Bruyères-sur-Fère	02
544	128	\N	Bruyères-et-Montbérault	02
545	129	\N	Bruys	02
546	130	\N	Bucilly	02
547	131	\N	Bucy-le-Long	02
548	132	\N	Bucy-lès-Cerny	02
549	133	\N	Bucy-lès-Pierrepont	02
550	134	\N	Buire	02
551	135	\N	Buironfosse	02
552	136	\N	Burelles	02
553	137	\N	Bussiares	02
554	138	\N	Buzancy	02
555	139	\N	Caillouël-Crépigny	02
556	140	\N	Camelin	02
557	141	\N	Capelle	02
558	142	\N	Castres	02
559	143	\N	Catelet	02
560	144	\N	Caulaincourt	02
561	145	\N	Caumont	02
562	146	\N	Celles-lès-Condé	02
563	147	\N	Celle-sous-Montmirail	02
564	148	\N	Celles-sur-Aisne	02
565	149	\N	Cerizy	02
566	150	\N	Cerny-en-Laonnois	02
567	151	\N	Cerny-lès-Bucy	02
568	152	\N	Cerseuil	02
569	153	\N	Cessières	02
570	154	\N	Chacrise	02
571	155	\N	Chaillevois	02
572	156	\N	Chalandry	02
573	157	\N	Chambry	02
574	158	\N	Chamouille	02
575	159	\N	Champs	02
576	160	\N	Chaourse	02
577	161	\N	Chapelle-Monthodon	02
578	162	\N	Chapelle-sur-Chézy	02
579	163	\N	Charly	02
580	164	\N	Charmel	02
581	165	\N	Charmes	02
582	166	\N	Chartèves	02
583	167	\N	Chassemy	02
584	168	\N	Château-Thierry	02
585	169	\N	Châtillon-lès-Sons	02
586	170	\N	Châtillon-sur-Oise	02
587	171	\N	Chaudardes	02
588	172	\N	Chaudun	02
589	173	\N	Chauny	02
590	174	\N	Chavignon	02
591	175	\N	Chavigny	02
592	176	\N	Chavonne	02
593	177	\N	Chérêt	02
594	178	\N	Chermizy-Ailles	02
595	179	\N	Chéry-Chartreuve	02
596	180	\N	Chéry-lès-Pouilly	02
597	181	\N	Chéry-lès-Rozoy	02
598	182	\N	Chevennes	02
599	183	\N	Chevregny	02
600	184	\N	Chevresis-Monceau	02
601	185	\N	Chézy-en-Orxois	02
602	186	\N	Chézy-sur-Marne	02
603	187	\N	Chierry	02
604	188	\N	Chigny	02
605	189	\N	Chivres-en-Laonnois	02
606	190	\N	Chivres-Val	02
607	191	\N	Chivy-lès-Étouvelles	02
608	192	\N	Chouy	02
609	193	\N	Cierges	02
610	194	\N	Cilly	02
611	195	\N	Ciry-Salsogne	02
612	196	\N	Clacy-et-Thierret	02
613	197	\N	Clairfontaine	02
614	198	\N	Clamecy	02
615	199	\N	Clastres	02
616	200	\N	Clermont-les-Fermes	02
617	201	\N	Cuvres-et-Valsery	02
618	203	\N	Coincy	02
619	204	\N	Coingt	02
620	205	\N	Colligis-Crandelain	02
621	206	\N	Colonfay	02
622	207	\N	Commenchon	02
623	208	\N	Concevreux	02
624	209	\N	Condé-en-Brie	02
625	210	\N	Condé-sur-Aisne	02
626	211	\N	Condé-sur-Suippe	02
627	212	\N	Condren	02
628	213	\N	Connigis	02
629	214	\N	Contescourt	02
630	215	\N	Corbeny	02
631	216	\N	Corcy	02
632	217	\N	Coucy-le-Château-Auffrique	02
633	218	\N	Coucy-lès-Eppes	02
634	219	\N	Coucy-la-Ville	02
635	220	\N	Coulonges-Cohan	02
636	221	\N	Coupru	02
637	222	\N	Courbes	02
638	223	\N	Courboin	02
639	224	\N	Courcelles-sur-Vesles	02
640	225	\N	Courchamps	02
641	226	\N	Courmelles	02
642	227	\N	Courmont	02
643	228	\N	Courtemont-Varennes	02
644	229	\N	Courtrizy-et-Fussigny	02
645	230	\N	Couvrelles	02
646	231	\N	Couvron-et-Aumencourt	02
647	232	\N	Coyolles	02
648	233	\N	Cramaille	02
649	234	\N	Craonne	02
650	235	\N	Craonnelle	02
651	236	\N	Crécy-au-Mont	02
652	237	\N	Crécy-sur-Serre	02
653	238	\N	Crépy	02
654	239	\N	Crézancy	02
655	240	\N	Croix-Fonsommes	02
656	241	\N	Croix-sur-Ourcq	02
657	242	\N	Crouttes-sur-Marne	02
658	243	\N	Crouy	02
659	244	\N	Crupilly	02
660	245	\N	Cuffies	02
661	246	\N	Cugny	02
662	248	\N	Cuirieux	02
663	249	\N	Cuiry-Housse	02
664	250	\N	Cuiry-lès-Chaudardes	02
665	251	\N	Cuiry-lès-Iviers	02
666	252	\N	Cuissy-et-Geny	02
667	253	\N	Cuisy-en-Almont	02
668	254	\N	Cutry	02
669	255	\N	Cys-la-Commune	02
670	256	\N	Dagny-Lambercy	02
671	257	\N	Dallon	02
672	258	\N	Dammard	02
673	259	\N	Dampleux	02
674	260	\N	Danizy	02
675	261	\N	Dercy	02
676	262	\N	Deuillet	02
677	263	\N	Dhuizel	02
678	264	\N	Dizy-le-Gros	02
679	265	\N	Dohis	02
680	266	\N	Dolignon	02
681	267	\N	Dommiers	02
682	268	\N	Domptin	02
683	269	\N	Dorengt	02
684	270	\N	Douchy	02
685	271	\N	Dravegny	02
686	272	\N	Droizy	02
687	273	\N	Dury	02
688	274	\N	Ébouleau	02
689	275	\N	Effry	02
690	276	\N	Englancourt	02
691	277	\N	Épagny	02
692	278	\N	Éparcy	02
693	279	\N	Épaux-Bézu	02
694	280	\N	Épieds	02
695	281	\N	Épine-aux-Bois	02
696	282	\N	Eppes	02
697	283	\N	Erlon	02
698	284	\N	Erloy	02
699	286	\N	Esquéhéries	02
700	287	\N	Essigny-le-Grand	02
701	288	\N	Essigny-le-Petit	02
702	289	\N	Essises	02
703	290	\N	Essômes-sur-Marne	02
704	291	\N	Estrées	02
705	292	\N	Étampes-sur-Marne	02
706	293	\N	Étaves-et-Bocquiaux	02
707	294	\N	Étouvelles	02
708	295	\N	Étréaupont	02
709	296	\N	Étreillers	02
710	297	\N	Étrépilly	02
711	298	\N	Étreux	02
712	299	\N	Évergnicourt	02
713	301	\N	Faucoucourt	02
714	302	\N	Faverolles	02
715	303	\N	Fayet	02
716	304	\N	Fère	02
717	305	\N	Fère-en-Tardenois	02
718	306	\N	Ferté-Chevresis	02
719	307	\N	Ferté-Milon	02
720	308	\N	Fesmy-le-Sart	02
721	309	\N	Festieux	02
722	310	\N	Fieulaine	02
723	311	\N	Filain	02
724	312	\N	Flamengrie	02
725	313	\N	Flavigny-le-Grand-et-Beaurain	02
726	315	\N	Flavy-le-Martel	02
727	316	\N	Fleury	02
728	317	\N	Fluquières	02
729	318	\N	Folembray	02
730	319	\N	Fonsommes	02
731	320	\N	Fontaine-lès-Clercs	02
732	321	\N	Fontaine-lès-Vervins	02
733	322	\N	Fontaine-Notre-Dame	02
734	323	\N	Fontaine-Uterte	02
735	324	\N	Fontenelle	02
736	325	\N	Fontenelle-en-Brie	02
737	326	\N	Fontenoy	02
738	327	\N	Foreste	02
739	328	\N	Fossoy	02
740	329	\N	Fourdrain	02
741	330	\N	Francilly-Selency	02
742	331	\N	Franqueville	02
743	332	\N	Fresnes-en-Tardenois	02
744	333	\N	Fresnes	02
745	334	\N	Fresnoy-le-Grand	02
746	335	\N	Fressancourt	02
747	336	\N	Frières-Faillouël	02
748	337	\N	Froidestrées	02
749	338	\N	Froidmont-Cohartille	02
750	339	\N	Gandelu	02
751	340	\N	Gauchy	02
752	341	\N	Gercy	02
753	342	\N	Gergny	02
754	343	\N	Germaine	02
755	344	\N	Gernicourt	02
756	345	\N	Gibercourt	02
757	346	\N	Gizy	02
758	347	\N	Gland	02
759	348	\N	Glennes	02
760	349	\N	Goudelancourt-lès-Berrieux	02
761	350	\N	Goudelancourt-lès-Pierrepont	02
762	351	\N	Goussancourt	02
763	352	\N	Gouy	02
764	353	\N	Grandlup-et-Fay	02
765	354	\N	Grandrieux	02
766	355	\N	Gricourt	02
767	356	\N	Grisolles	02
768	357	\N	Gronard	02
769	358	\N	Grougis	02
770	359	\N	Grugies	02
771	360	\N	Guignicourt	02
772	361	\N	Guise	02
773	362	\N	Guivry	02
774	363	\N	Guny	02
775	364	\N	Guyencourt	02
776	366	\N	Hannapes	02
777	367	\N	Happencourt	02
778	368	\N	Haramont	02
779	369	\N	Harcigny	02
780	370	\N	Hargicourt	02
781	371	\N	Harly	02
782	372	\N	Hartennes-et-Taux	02
783	373	\N	Hary	02
784	374	\N	Lehaucourt	02
785	375	\N	Hautevesnes	02
786	376	\N	Hauteville	02
787	377	\N	Haution	02
788	378	\N	Hérie	02
789	379	\N	Hérie-la-Viéville	02
790	380	\N	Hinacourt	02
791	381	\N	Hirson	02
792	382	\N	Holnon	02
793	383	\N	Homblières	02
794	384	\N	Houry	02
795	385	\N	Housset	02
796	386	\N	Iron	02
797	387	\N	Itancourt	02
798	388	\N	Iviers	02
799	389	\N	Jaulgonne	02
800	390	\N	Jeancourt	02
801	391	\N	Jeantes	02
802	392	\N	Joncourt	02
803	393	\N	Jouaignes	02
804	395	\N	Jumencourt	02
805	396	\N	Jumigny	02
806	397	\N	Jussy	02
807	398	\N	Juvigny	02
808	399	\N	Juvincourt-et-Damary	02
809	400	\N	Laffaux	02
810	401	\N	Laigny	02
811	402	\N	Lanchy	02
812	403	\N	Landifay-et-Bertaignemont	02
813	404	\N	Landouzy-la-Cour	02
814	405	\N	Landouzy-la-Ville	02
815	406	\N	Landricourt	02
816	407	\N	Laniscourt	02
817	408	\N	Laon	02
818	409	\N	Lappion	02
819	410	\N	Largny-sur-Automne	02
820	411	\N	Latilly	02
821	412	\N	Launoy	02
822	413	\N	Laval-en-Laonnois	02
823	414	\N	Lavaqueresse	02
824	415	\N	Laversine	02
825	416	\N	Lemé	02
826	417	\N	Lempire	02
827	418	\N	Lerzy	02
828	419	\N	Leschelles	02
829	420	\N	Lesdins	02
830	421	\N	Lesges	02
831	422	\N	Lesquielles-Saint-Germain	02
832	423	\N	Leuilly-sous-Coucy	02
833	424	\N	Leury	02
834	425	\N	Leuze	02
835	426	\N	Levergies	02
836	427	\N	Lhuys	02
837	428	\N	Licy-Clignon	02
838	429	\N	Lierval	02
839	430	\N	Liesse-Notre-Dame	02
840	431	\N	Liez	02
841	432	\N	Limé	02
842	433	\N	Lislet	02
843	434	\N	Lizy	02
844	435	\N	Logny-lès-Aubenton	02
845	438	\N	Longpont	02
846	439	\N	Longueval-Barbonval	02
847	440	\N	Lor	02
848	441	\N	Louâtre	02
849	442	\N	Loupeigne	02
850	443	\N	Lucy-le-Bocage	02
851	444	\N	Lugny	02
852	445	\N	Luzoir	02
853	446	\N	Ly-Fontaine	02
854	447	\N	Maast-et-Violaine	02
855	448	\N	Mâchecourt	02
856	449	\N	Macogny	02
857	450	\N	Macquigny	02
858	451	\N	Magny-la-Fosse	02
859	452	\N	Maissemy	02
860	453	\N	Maizy	02
861	454	\N	Malmaison	02
862	455	\N	Malzy	02
863	456	\N	Manicamp	02
864	457	\N	Marchais	02
865	458	\N	Marchais-en-Brie	02
866	459	\N	Marcy	02
867	460	\N	Marcy-sous-Marle	02
868	461	\N	Marest-Dampcourt	02
869	462	\N	Mareuil-en-Dôle	02
870	463	\N	Marfontaine	02
871	464	\N	Margival	02
872	465	\N	Marigny-en-Orxois	02
873	466	\N	Marizy-Sainte-Geneviève	02
874	467	\N	Marizy-Saint-Mard	02
875	468	\N	Marle	02
876	469	\N	Marly-Gomont	02
877	470	\N	Martigny	02
878	471	\N	Martigny-Courpierre	02
879	472	\N	Mauregny-en-Haye	02
880	473	\N	Mayot	02
881	474	\N	Mennessis	02
882	475	\N	Menneville	02
883	476	\N	Mennevret	02
884	477	\N	Mercin-et-Vaux	02
885	478	\N	Merlieux-et-Fouquerolles	02
886	479	\N	Merval	02
887	480	\N	Mesbrecourt-Richecourt	02
888	481	\N	Mesnil-Saint-Laurent	02
889	482	\N	Meurival	02
890	483	\N	Mézières-sur-Oise	02
891	484	\N	Mézy-Moulins	02
892	485	\N	Missy-aux-Bois	02
893	486	\N	Missy-lès-Pierrepont	02
894	487	\N	Missy-sur-Aisne	02
895	488	\N	Molain	02
896	489	\N	Molinchart	02
897	490	\N	Monampteuil	02
898	491	\N	Monceau-le-Neuf-et-Faucouzy	02
899	492	\N	Monceau-lès-Leups	02
900	493	\N	Monceau-le-Waast	02
901	494	\N	Monceau-sur-Oise	02
902	495	\N	Mondrepuis	02
903	496	\N	Monnes	02
904	497	\N	Mons-en-Laonnois	02
905	498	\N	Montaigu	02
906	499	\N	Montbavin	02
907	500	\N	Montbrehain	02
908	501	\N	Montchâlons	02
909	502	\N	Montcornet	02
910	503	\N	Mont-d'Origny	02
911	504	\N	Montescourt-Lizerolles	02
912	505	\N	Montfaucon	02
913	506	\N	Montgobert	02
914	507	\N	Montgru-Saint-Hilaire	02
915	508	\N	Monthenault	02
916	509	\N	Monthiers	02
917	510	\N	Monthurel	02
918	511	\N	Montigny-en-Arrouaise	02
919	512	\N	Montigny-l'Allier	02
920	513	\N	Montigny-le-Franc	02
921	514	\N	Montigny-Lengrain	02
922	515	\N	Montigny-lès-Condé	02
923	516	\N	Montigny-sous-Marle	02
924	517	\N	Montigny-sur-Crécy	02
925	518	\N	Montlevon	02
926	519	\N	Montloué	02
927	520	\N	Mont-Notre-Dame	02
928	521	\N	Montreuil-aux-Lions	02
929	522	\N	Mont-Saint-Jean	02
930	523	\N	Mont-Saint-Martin	02
931	524	\N	Mont-Saint-Père	02
932	525	\N	Morcourt	02
933	526	\N	Morgny-en-Thiérache	02
934	527	\N	Morsain	02
935	528	\N	Mortefontaine	02
936	529	\N	Mortiers	02
937	530	\N	Moulins	02
938	531	\N	Moussy-Verneuil	02
939	532	\N	Moÿ-de-l'Aisne	02
940	533	\N	Muret-et-Crouttes	02
941	534	\N	Muscourt	02
942	535	\N	Nampcelles-la-Cour	02
943	536	\N	Nampteuil-sous-Muret	02
944	537	\N	Nanteuil-la-Fosse	02
945	538	\N	Nanteuil-Notre-Dame	02
946	539	\N	Nauroy	02
947	540	\N	Nesles-la-Montagne	02
948	541	\N	Neufchâtel-sur-Aisne	02
949	542	\N	Neuflieux	02
950	543	\N	Neuilly-Saint-Front	02
951	544	\N	Neuve-Maison	02
952	545	\N	Neuville-Bosmont	02
953	546	\N	Neuville-en-Beine	02
954	547	\N	Neuville-Housset	02
955	548	\N	Neuville-lès-Dorengt	02
956	549	\N	Neuville-Saint-Amand	02
957	550	\N	Neuville-sur-Ailette	02
958	551	\N	Neuville-sur-Margival	02
959	552	\N	Neuvillette	02
960	553	\N	Nizy-le-Comte	02
961	554	\N	Nogentel	02
962	555	\N	Nogent-l'Artaud	02
963	556	\N	Noircourt	02
964	557	\N	Noroy-sur-Ourcq	02
965	558	\N	Nouvion-en-Thiérache	02
966	559	\N	Nouvion-et-Catillon	02
967	560	\N	Nouvion-le-Comte	02
968	561	\N	Nouvion-le-Vineux	02
969	562	\N	Nouvron-Vingré	02
970	563	\N	Noyales	02
971	564	\N	Noyant-et-Aconin	02
972	565	\N	uilly	02
973	566	\N	Ognes	02
974	567	\N	Ohis	02
975	568	\N	Oigny-en-Valois	02
976	569	\N	Oisy	02
977	570	\N	Ollezy	02
978	571	\N	Omissy	02
979	572	\N	Orainville	02
980	573	\N	Orgeval	02
981	574	\N	Origny-en-Thiérache	02
982	575	\N	Origny-Sainte-Benoite	02
983	576	\N	Osly-Courtil	02
984	577	\N	Ostel	02
985	578	\N	Oulches-la-Vallée-Foulon	02
986	579	\N	Oulchy-la-Ville	02
987	580	\N	Oulchy-le-Château	02
988	581	\N	Paars	02
989	582	\N	Paissy	02
990	583	\N	Pancy-Courtecon	02
991	584	\N	Papleux	02
992	585	\N	Parcy-et-Tigny	02
993	586	\N	Parfondeval	02
994	587	\N	Parfondru	02
995	588	\N	Pargnan	02
996	589	\N	Pargny-Filain	02
997	590	\N	Pargny-la-Dhuys	02
998	591	\N	Pargny-les-Bois	02
999	592	\N	Parpeville	02
1000	593	\N	Pasly	02
1001	594	\N	Passy-en-Valois	02
1002	595	\N	Passy-sur-Marne	02
1003	596	\N	Pavant	02
1004	597	\N	Perles	02
1005	598	\N	Pernant	02
1006	599	\N	Pierremande	02
1007	600	\N	Pierrepont	02
1008	601	\N	Pignicourt	02
1009	602	\N	Pinon	02
1010	604	\N	Pithon	02
1011	605	\N	Pleine-Selve	02
1012	606	\N	Plessier-Huleu	02
1013	607	\N	Ploisy	02
1014	608	\N	Plomion	02
1015	609	\N	Ployart-et-Vaurseine	02
1016	610	\N	Pommiers	02
1017	612	\N	Pont-Arcy	02
1018	613	\N	Pontavert	02
1019	614	\N	Pontru	02
1020	615	\N	Pontruet	02
1021	616	\N	Pont-Saint-Mard	02
1022	617	\N	Pouilly-sur-Serre	02
1023	618	\N	Prémont	02
1024	619	\N	Prémontré	02
1025	620	\N	Presles-et-Boves	02
1026	621	\N	Presles-et-Thierny	02
1027	622	\N	Priez	02
1028	623	\N	Prisces	02
1029	624	\N	Proisy	02
1030	625	\N	Proix	02
1031	626	\N	Prouvais	02
1032	627	\N	Proviseux-et-Plesnoy	02
1033	628	\N	Puiseux-en-Retz	02
1034	629	\N	Puisieux-et-Clanlieu	02
1035	631	\N	Quierzy	02
1036	632	\N	Quincy-Basse	02
1037	633	\N	Quincy-sous-le-Mont	02
1038	634	\N	Raillimont	02
1039	635	\N	Ramicourt	02
1040	636	\N	Regny	02
1041	637	\N	Remaucourt	02
1042	638	\N	Remies	02
1043	639	\N	Remigny	02
1044	640	\N	Renansart	02
1045	641	\N	Renneval	02
1046	642	\N	Résigny	02
1047	643	\N	Ressons-le-Long	02
1048	644	\N	Retheuil	02
1049	645	\N	Reuilly-Sauvigny	02
1050	646	\N	Révillon	02
1051	647	\N	Ribeauville	02
1052	648	\N	Ribemont	02
1053	649	\N	Rocourt-Saint-Martin	02
1054	650	\N	Rocquigny	02
1055	651	\N	Rogécourt	02
1056	652	\N	Rogny	02
1057	653	\N	Romeny-sur-Marne	02
1058	654	\N	Romery	02
1059	655	\N	Ronchères	02
1060	656	\N	Roucy	02
1061	657	\N	Rougeries	02
1062	658	\N	Roupy	02
1063	659	\N	Rouvroy	02
1064	660	\N	Rouvroy-sur-Serre	02
1065	661	\N	Royaucourt-et-Chailvet	02
1066	662	\N	Rozet-Saint-Albin	02
1067	663	\N	Rozières-sur-Crise	02
1068	664	\N	Rozoy-Bellevalle	02
1069	665	\N	Grand-Rozoy	02
1070	666	\N	Rozoy-sur-Serre	02
1071	667	\N	Saconin-et-Breuil	02
1072	668	\N	Sains-Richaumont	02
1073	669	\N	Saint-Agnan	02
1074	670	\N	Saint-Algis	02
1075	671	\N	Saint-Aubin	02
1076	672	\N	Saint-Bandry	02
1077	673	\N	Saint-Christophe-à-Berry	02
1078	674	\N	Saint-Clément	02
1079	675	\N	Sainte-Croix	02
1080	676	\N	Saint-Erme-Outre-et-Ramecourt	02
1081	677	\N	Saint-Eugène	02
1082	678	\N	Sainte-Geneviève	02
1083	679	\N	Saint-Gengoulph	02
1084	680	\N	Saint-Gobain	02
1085	681	\N	Saint-Gobert	02
1086	682	\N	Saint-Mard	02
1087	683	\N	Saint-Martin-Rivière	02
1088	684	\N	Saint-Michel	02
1089	685	\N	Saint-Nicolas-aux-Bois	02
1090	686	\N	Saint-Paul-aux-Bois	02
1091	687	\N	Saint-Pierre-Aigle	02
1092	688	\N	Saint-Pierre-lès-Franqueville	02
1093	689	\N	Saint-Pierremont	02
1094	690	\N	Sainte-Preuve	02
1095	691	\N	Saint-Quentin	02
1096	693	\N	Saint-Rémy-Blanzy	02
1097	694	\N	Saint-Simon	02
1098	695	\N	Saint-Thibaut	02
1099	696	\N	Saint-Thomas	02
1100	697	\N	Samoussy	02
1101	698	\N	Sancy-les-Cheminots	02
1102	699	\N	Saponay	02
1103	701	\N	Saulchery	02
1104	702	\N	Savy	02
1105	703	\N	Seboncourt	02
1106	704	\N	Selens	02
1107	705	\N	Selve	02
1108	706	\N	Septmonts	02
1109	707	\N	Septvaux	02
1110	708	\N	Sequehart	02
1111	709	\N	Serain	02
1112	710	\N	Seraucourt-le-Grand	02
1113	711	\N	Serches	02
1114	712	\N	Sergy	02
1115	713	\N	Seringes-et-Nesles	02
1116	714	\N	Sermoise	02
1117	715	\N	Serval	02
1118	716	\N	Servais	02
1119	717	\N	Séry-lès-Mézières	02
1120	718	\N	Silly-la-Poterie	02
1121	719	\N	Sinceny	02
1122	720	\N	Sissonne	02
1123	721	\N	Sissy	02
1124	722	\N	Soissons	02
1125	723	\N	Soize	02
1126	724	\N	Sommelans	02
1127	725	\N	Sommeron	02
1128	726	\N	Sommette-Eaucourt	02
1129	727	\N	Sons-et-Ronchères	02
1130	728	\N	Sorbais	02
1131	729	\N	Soucy	02
1132	730	\N	Soupir	02
1133	731	\N	Sourd	02
1134	732	\N	Surfontaine	02
1135	733	\N	Suzy	02
1136	734	\N	Taillefontaine	02
1137	735	\N	Tannières	02
1138	736	\N	Tartiers	02
1139	737	\N	Tavaux-et-Pontséricourt	02
1140	738	\N	Tergnier	02
1141	739	\N	Terny-Sorny	02
1142	740	\N	Thenailles	02
1143	741	\N	Thenelles	02
1144	742	\N	Thiernu	02
1145	743	\N	Thuel	02
1146	744	\N	Torcy-en-Valois	02
1147	745	\N	Toulis-et-Attencourt	02
1148	746	\N	Travecy	02
1149	747	\N	Trefcon	02
1150	748	\N	Trélou-sur-Marne	02
1151	749	\N	Troësnes	02
1152	750	\N	Trosly-Loire	02
1153	751	\N	Trucy	02
1154	752	\N	Tugny-et-Pont	02
1155	753	\N	Tupigny	02
1156	754	\N	Ugny-le-Gay	02
1157	755	\N	Urcel	02
1158	756	\N	Urvillers	02
1159	757	\N	Vadencourt	02
1160	758	\N	Vailly-sur-Aisne	02
1161	759	\N	Vallée-au-Blé	02
1162	760	\N	Vallée-Mulâtre	02
1163	761	\N	Variscourt	02
1164	762	\N	Vassens	02
1165	763	\N	Vasseny	02
1166	764	\N	Vassogne	02
1167	765	\N	Vaucelles-et-Beffecourt	02
1168	766	\N	Vaudesson	02
1169	767	\N	Vauxrezis	02
1170	768	\N	Vauxaillon	02
1171	769	\N	Vaux-Andigny	02
1172	770	\N	Vauxbuin	02
1173	771	\N	Vauxcéré	02
1174	772	\N	Vaux-en-Vermandois	02
1175	773	\N	Vauxtin	02
1176	774	\N	Vendelles	02
1177	775	\N	Vendeuil	02
1178	776	\N	Vendhuile	02
1179	777	\N	Vendières	02
1180	778	\N	Vendresse-Beaulne	02
1181	779	\N	Vénérolles	02
1182	780	\N	Venizel	02
1183	781	\N	Verdilly	02
1184	782	\N	Verguier	02
1185	783	\N	Grand-Verly	02
1186	784	\N	Petit-Verly	02
1187	785	\N	Vermand	02
1188	786	\N	Verneuil-sous-Coucy	02
1189	787	\N	Verneuil-sur-Serre	02
1190	788	\N	Versigny	02
1191	789	\N	Vervins	02
1192	790	\N	Vesles-et-Caumont	02
1193	791	\N	Veslud	02
1194	792	\N	Veuilly-la-Poterie	02
1195	793	\N	Vézaponin	02
1196	794	\N	Vézilly	02
1197	795	\N	Vic-sur-Aisne	02
1198	796	\N	Vichel-Nanteuil	02
1199	797	\N	Viel-Arcy	02
1200	798	\N	Viels-Maisons	02
1201	799	\N	Vierzy	02
1202	800	\N	Viffort	02
1203	801	\N	Vigneux-Hocquet	02
1204	802	\N	Ville-aux-Bois-lès-Dizy	02
1205	803	\N	Ville-aux-Bois-lès-Pontavert	02
1206	804	\N	Villemontoire	02
1207	805	\N	Villeneuve-Saint-Germain	02
1208	806	\N	Villeneuve-sur-Fère	02
1209	807	\N	Villequier-Aumont	02
1210	808	\N	Villeret	02
1211	809	\N	Villers-Agron-Aiguizy	02
1212	810	\N	Villers-Cotterêts	02
1213	811	\N	Villers-en-Prayères	02
1214	812	\N	Villers-Hélon	02
1215	813	\N	Villers-le-Sec	02
1216	814	\N	Villers-lès-Guise	02
1217	815	\N	Villers-Saint-Christophe	02
1218	816	\N	Villers-sur-Fère	02
1219	817	\N	Ville-Savoye	02
1220	818	\N	Villiers-Saint-Denis	02
1221	819	\N	Vincy-Reuil-et-Magny	02
1222	820	\N	Viry-Noureuil	02
1223	821	\N	Vivaise	02
1224	822	\N	Vivières	02
1225	823	\N	Voharies	02
1226	824	\N	Vorges	02
1227	826	\N	Voulpaix	02
1228	827	\N	Voyenne	02
1229	828	\N	Vregny	02
1230	829	\N	Vuillery	02
1231	830	\N	Wassigny	02
1232	831	\N	Watigny	02
1233	832	\N	Wiège-Faty	02
1234	833	\N	Wimy	02
1235	834	\N	Wissignicourt	02
1236	001	\N	Abrest	03
1237	002	\N	Agonges	03
1238	003	\N	Ainay-le-Château	03
1239	004	\N	Andelaroche	03
1240	005	\N	Archignat	03
1241	006	\N	Arfeuilles	03
1242	007	\N	Arpheuilles-Saint-Priest	03
1243	008	\N	Arronnes	03
1244	009	\N	Aubigny	03
1245	010	\N	Audes	03
1246	011	\N	Aurouër	03
1247	012	\N	Autry-Issards	03
1248	013	\N	Avermes	03
1249	014	\N	Avrilly	03
1250	015	\N	Bagneux	03
1251	016	\N	Barberier	03
1252	017	\N	Barrais-Bussolles	03
1253	018	\N	Bayet	03
1254	019	\N	Beaulon	03
1255	020	\N	Beaune-d'Allier	03
1256	021	\N	Bègues	03
1257	022	\N	Bellenaves	03
1258	023	\N	Bellerive-sur-Allier	03
1259	024	\N	Bert	03
1260	025	\N	Bessay-sur-Allier	03
1261	026	\N	Besson	03
1262	027	\N	Bézenet	03
1263	028	\N	Billezois	03
1264	029	\N	Billy	03
1265	030	\N	Biozat	03
1266	031	\N	Bizeneuille	03
1267	032	\N	Blomard	03
1268	033	\N	Bost	03
1269	034	\N	Boucé	03
1270	035	\N	Bouchaud	03
1271	036	\N	Bourbon-l'Archambault	03
1272	037	\N	Braize	03
1273	038	\N	Bransat	03
1274	039	\N	Bresnay	03
1275	040	\N	Bressolles	03
1276	041	\N	Brethon	03
1277	042	\N	Breuil	03
1278	043	\N	Broût-Vernet	03
1279	044	\N	Brugheas	03
1280	045	\N	Busset	03
1281	046	\N	Buxières-les-Mines	03
1282	047	\N	Celle	03
1283	048	\N	Cérilly	03
1284	049	\N	Cesset	03
1285	050	\N	Chabanne	03
1286	051	\N	Chambérat	03
1287	052	\N	Chamblet	03
1288	053	\N	Chantelle	03
1289	054	\N	Chapeau	03
1290	055	\N	Chapelaude	03
1291	056	\N	Chapelle	03
1292	057	\N	Chapelle-aux-Chasses	03
1293	058	\N	Chappes	03
1294	059	\N	Chareil-Cintrat	03
1295	060	\N	Charmeil	03
1296	061	\N	Charmes	03
1297	062	\N	Charroux	03
1298	063	\N	Chassenard	03
1299	064	\N	Château-sur-Allier	03
1300	065	\N	Châtel-de-Neuvre	03
1301	066	\N	Châtel-Montagne	03
1302	067	\N	Châtelperron	03
1303	068	\N	Châtelus	03
1304	069	\N	Châtillon	03
1305	070	\N	Chavenon	03
1306	071	\N	Chavroches	03
1307	072	\N	Chazemais	03
1308	073	\N	Chemilly	03
1309	074	\N	Chevagnes	03
1310	075	\N	Chezelle	03
1311	076	\N	Chézy	03
1312	077	\N	Chirat-l'Église	03
1313	078	\N	Chouvigny	03
1314	079	\N	Cindré	03
1315	080	\N	Cognat-Lyonne	03
1316	081	\N	Colombier	03
1317	082	\N	Commentry	03
1318	083	\N	Contigny	03
1319	084	\N	Cosne-d'Allier	03
1320	085	\N	Coulandon	03
1321	086	\N	Coulanges	03
1322	087	\N	Couleuvre	03
1323	088	\N	Courçais	03
1324	089	\N	Coutansouze	03
1325	090	\N	Couzon	03
1326	091	\N	Créchy	03
1327	092	\N	Cressanges	03
1328	093	\N	Creuzier-le-Neuf	03
1329	094	\N	Creuzier-le-Vieux	03
1330	095	\N	Cusset	03
1331	096	\N	Deneuille-lès-Chantelle	03
1332	097	\N	Deneuille-les-Mines	03
1333	098	\N	Désertines	03
1334	099	\N	Deux-Chaises	03
1335	100	\N	Diou	03
1336	101	\N	Domérat	03
1337	102	\N	Dompierre-sur-Besbre	03
1338	103	\N	Donjon	03
1339	104	\N	Doyet	03
1340	105	\N	Droiturier	03
1341	106	\N	Durdat-Larequille	03
1342	107	\N	Ébreuil	03
1343	108	\N	Échassières	03
1344	109	\N	Escurolles	03
1345	110	\N	Espinasse-Vozelle	03
1346	111	\N	Estivareilles	03
1347	112	\N	Étroussat	03
1348	113	\N	Ferrières-sur-Sichon	03
1349	114	\N	Ferté-Hauterive	03
1350	115	\N	Fleuriel	03
1351	116	\N	Fourilles	03
1352	117	\N	Franchesse	03
1353	118	\N	Gannat	03
1354	119	\N	Gannay-sur-Loire	03
1355	120	\N	Garnat-sur-Engièvre	03
1356	121	\N	Gennetines	03
1357	122	\N	Gipcy	03
1358	123	\N	Givarlais	03
1359	124	\N	Gouise	03
1360	125	\N	Guillermie	03
1361	126	\N	Hauterive	03
1362	127	\N	Hérisson	03
1363	128	\N	Huriel	03
1364	129	\N	Hyds	03
1365	130	\N	Isle-et-Bardais	03
1366	131	\N	Isserpent	03
1367	132	\N	Jaligny-sur-Besbre	03
1368	133	\N	Jenzat	03
1369	134	\N	Laféline	03
1370	135	\N	Lalizolle	03
1371	136	\N	Lamaids	03
1372	137	\N	Langy	03
1373	138	\N	Lapalisse	03
1374	139	\N	Laprugne	03
1375	140	\N	Lavault-Sainte-Anne	03
1376	141	\N	Lavoine	03
1377	142	\N	Lenax	03
1378	143	\N	Lételon	03
1379	144	\N	Liernolles	03
1380	145	\N	Lignerolles	03
1381	146	\N	Limoise	03
1382	147	\N	Loddes	03
1383	148	\N	Loriges	03
1384	149	\N	Louchy-Montfand	03
1385	150	\N	Louroux-Bourbonnais	03
1386	151	\N	Louroux-de-Beaune	03
1387	152	\N	Louroux-de-Bouble	03
1388	153	\N	Louroux-Hodement	03
1389	154	\N	Luneau	03
1390	155	\N	Lurcy-Lévis	03
1391	156	\N	Lusigny	03
1392	157	\N	Magnet	03
1393	158	\N	Maillet	03
1394	159	\N	Malicorne	03
1395	160	\N	Marcenat	03
1396	161	\N	Marcillat-en-Combraille	03
1397	162	\N	Marigny	03
1398	163	\N	Mariol	03
1399	164	\N	Mayet-d'École	03
1400	165	\N	Mayet-de-Montagne	03
1401	166	\N	Mazerier	03
1402	167	\N	Mazirat	03
1403	168	\N	Meaulne	03
1404	169	\N	Meillard	03
1405	170	\N	Meillers	03
1406	171	\N	Mercy	03
1407	172	\N	Mesples	03
1408	173	\N	Molinet	03
1409	174	\N	Molles	03
1410	175	\N	Monestier	03
1411	176	\N	Monétay-sur-Allier	03
1412	177	\N	Monétay-sur-Loire	03
1413	178	\N	Montaiguët-en-Forez	03
1414	179	\N	Montaigu-le-Blin	03
1415	180	\N	Montbeugny	03
1416	181	\N	Montcombroux-les-Mines	03
1417	182	\N	Monteignet-sur-l'Andelot	03
1418	183	\N	Montet	03
1419	184	\N	Montilly	03
1420	185	\N	Montluçon	03
1421	186	\N	Montmarault	03
1422	187	\N	Montoldre	03
1423	188	\N	Montord	03
1424	189	\N	Montvicq	03
1425	190	\N	Moulins	03
1426	191	\N	Murat	03
1427	192	\N	Nades	03
1428	193	\N	Nassigny	03
1429	194	\N	Naves	03
1430	195	\N	Néris-les-Bains	03
1431	196	\N	Neuilly-en-Donjon	03
1432	197	\N	Neuilly-le-Réal	03
1433	198	\N	Neure	03
1434	200	\N	Neuvy	03
1435	201	\N	Nizerolles	03
1436	202	\N	Noyant-d'Allier	03
1437	203	\N	Paray-le-Frésil	03
1438	204	\N	Paray-sous-Briailles	03
1439	205	\N	Périgny	03
1440	206	\N	Petite-Marche	03
1441	207	\N	Pierrefitte-sur-Loire	03
1442	208	\N	Pin	03
1443	209	\N	Poëzat	03
1444	210	\N	Pouzy-Mésangy	03
1445	211	\N	Prémilhat	03
1446	212	\N	Quinssaines	03
1447	213	\N	Reugny	03
1448	214	\N	Rocles	03
1449	215	\N	Rongères	03
1450	216	\N	Ronnet	03
1451	217	\N	Saint-Angel	03
1452	218	\N	Saint-Aubin-le-Monial	03
1453	219	\N	Saint-Bonnet-de-Four	03
1454	220	\N	Saint-Bonnet-de-Rochefort	03
1455	221	\N	Saint-Bonnet-Tronçais	03
1456	222	\N	Saint-Caprais	03
1457	223	\N	Saint-Christophe	03
1458	224	\N	Saint-Clément	03
1459	225	\N	Saint-Désiré	03
1460	226	\N	Saint-Didier-en-Donjon	03
1461	227	\N	Saint-Didier-la-Forêt	03
1462	228	\N	Saint-Éloy-d'Allier	03
1463	229	\N	Saint-Ennemond	03
1464	230	\N	Saint-Étienne-de-Vicq	03
1465	231	\N	Saint-Fargeol	03
1466	232	\N	Saint-Félix	03
1467	233	\N	Saint-Genest	03
1468	234	\N	Saint-Gérand-de-Vaux	03
1469	235	\N	Saint-Gérand-le-Puy	03
1470	236	\N	Saint-Germain-des-Fossés	03
1471	237	\N	Saint-Germain-de-Salles	03
1472	238	\N	Saint-Hilaire	03
1473	239	\N	Saint-Léger-sur-Vouzance	03
1474	240	\N	Saint-Léon	03
1475	241	\N	Saint-Léopardin-d'Augy	03
1476	242	\N	Saint-Loup	03
1477	243	\N	Saint-Marcel-en-Murat	03
1478	244	\N	Saint-Marcel-en-Marcillat	03
1479	245	\N	Saint-Martin-des-Lais	03
1480	246	\N	Saint-Martinien	03
1481	247	\N	Saint-Menoux	03
1482	248	\N	Saint-Nicolas-des-Biefs	03
1483	249	\N	Saint-Palais	03
1484	250	\N	Saint-Pierre-Laval	03
1485	251	\N	Saint-Plaisir	03
1486	252	\N	Saint-Pont	03
1487	253	\N	Saint-Pourçain-sur-Besbre	03
1488	254	\N	Saint-Pourçain-sur-Sioule	03
1489	255	\N	Saint-Priest-d'Andelot	03
1490	256	\N	Saint-Priest-en-Murat	03
1491	257	\N	Saint-Prix	03
1492	258	\N	Saint-Rémy-en-Rollat	03
1493	259	\N	Saint-Sauvier	03
1494	260	\N	Saint-Sornin	03
1495	261	\N	Sainte-Thérence	03
1496	262	\N	Saint-Victor	03
1497	263	\N	Saint-Voir	03
1498	264	\N	Saint-Yorre	03
1499	265	\N	Saligny-sur-Roudon	03
1500	266	\N	Sanssat	03
1501	267	\N	Saulcet	03
1502	268	\N	Saulzet	03
1503	269	\N	Sauvagny	03
1504	270	\N	Sazeret	03
1505	271	\N	Serbannes	03
1506	272	\N	Servilly	03
1507	273	\N	Seuillet	03
1508	274	\N	Sorbier	03
1509	275	\N	Souvigny	03
1510	276	\N	Sussat	03
1511	277	\N	Target	03
1512	278	\N	Taxat-Senat	03
1513	279	\N	Teillet-Argenty	03
1514	280	\N	Terjat	03
1515	281	\N	Theil	03
1516	282	\N	Theneuille	03
1517	283	\N	Thiel-sur-Acolin	03
1518	284	\N	Thionne	03
1519	285	\N	Tortezais	03
1520	286	\N	Toulon-sur-Allier	03
1521	287	\N	Treban	03
1522	288	\N	Treignat	03
1523	289	\N	Treteau	03
1524	290	\N	Trévol	03
1525	291	\N	Trézelles	03
1526	292	\N	Tronget	03
1527	293	\N	Urçay	03
1528	294	\N	Ussel-d'Allier	03
1529	295	\N	Valignat	03
1530	296	\N	Valigny	03
1531	297	\N	Vallon-en-Sully	03
1532	298	\N	Varennes-sur-Allier	03
1533	299	\N	Varennes-sur-Tèche	03
1534	300	\N	Vaumas	03
1535	301	\N	Vaux	03
1536	302	\N	Veauce	03
1537	303	\N	Venas	03
1538	304	\N	Vendat	03
1539	305	\N	Verneix	03
1540	306	\N	Vernet	03
1541	307	\N	Verneuil-en-Bourbonnais	03
1542	308	\N	Vernusse	03
1543	309	\N	Veurdre	03
1544	310	\N	Vichy	03
1545	311	\N	Vicq	03
1546	312	\N	Vieure	03
1547	313	\N	Vilhain	03
1548	314	\N	Villebret	03
1549	315	\N	Villefranche-d'Allier	03
1550	316	\N	Villeneuve-sur-Allier	03
1551	317	\N	Viplaix	03
1552	318	\N	Vitray	03
1553	319	\N	Voussac	03
1554	320	\N	Ygrande	03
1555	321	\N	Yzeure	03
1556	001	\N	Aiglun	04
1557	004	\N	Allemagne-en-Provence	04
1558	005	\N	Allons	04
1559	006	\N	Allos	04
1560	007	\N	Angles	04
1561	008	\N	Annot	04
1562	009	\N	Archail	04
1563	012	\N	Aubenas-les-Alpes	04
1564	013	\N	Aubignosc	04
1565	016	\N	Authon	04
1566	017	\N	Auzet	04
1567	018	\N	Banon	04
1568	019	\N	Barcelonnette	04
1569	020	\N	Barles	04
1570	021	\N	Barras	04
1571	022	\N	Barrême	04
1572	023	\N	Bayons	04
1573	024	\N	Beaujeu	04
1574	025	\N	Beauvezer	04
1575	026	\N	Bellaffaire	04
1576	027	\N	Bevons	04
1577	028	\N	Beynes	04
1578	030	\N	Blieux	04
1579	031	\N	Bras-d'Asse	04
1580	032	\N	Braux	04
1581	033	\N	Bréole	04
1582	034	\N	Brillanne	04
1583	035	\N	Brunet	04
1584	036	\N	Brusquet	04
1585	037	\N	Caire	04
1586	039	\N	Castellane	04
1587	040	\N	Castellard-Melan	04
1588	041	\N	Castellet	04
1589	042	\N	Castellet-lès-Sausses	04
1590	043	\N	Val-de-Chalvagne	04
1591	045	\N	Céreste	04
1592	046	\N	Chaffaut-Saint-Jurson	04
1593	047	\N	Champtercier	04
1594	049	\N	Château-Arnoux-Saint-Auban	04
1595	050	\N	Châteaufort	04
1596	051	\N	Châteauneuf-Miravail	04
1597	053	\N	Châteauneuf-Val-Saint-Donat	04
1598	054	\N	Châteauredon	04
1599	055	\N	Chaudon-Norante	04
1600	057	\N	Clamensane	04
1601	058	\N	Claret	04
1602	059	\N	Clumanc	04
1603	061	\N	Colmars	04
1604	062	\N	Condamine-Châtelard	04
1605	063	\N	Corbières	04
1606	065	\N	Cruis	04
1607	066	\N	Curbans	04
1608	067	\N	Curel	04
1609	068	\N	Dauphin	04
1610	069	\N	Demandolx	04
1611	070	\N	Digne-les-Bains	04
1612	072	\N	Draix	04
1613	073	\N	Enchastrayes	04
1614	074	\N	Entrages	04
1615	075	\N	Entrepierres	04
1616	076	\N	Entrevaux	04
1617	077	\N	Entrevennes	04
1618	079	\N	Escale	04
1619	081	\N	Esparron-de-Verdon	04
1620	084	\N	Estoublon	04
1621	085	\N	Faucon-du-Caire	04
1622	086	\N	Faucon-de-Barcelonnette	04
1623	087	\N	Fontienne	04
1624	088	\N	Forcalquier	04
1625	090	\N	Fugeret	04
1626	091	\N	Ganagobie	04
1627	092	\N	Garde	04
1628	093	\N	Gigors	04
1629	094	\N	Gréoux-les-Bains	04
1630	095	\N	Hospitalet	04
1631	096	\N	Jausiers	04
1632	097	\N	Javie	04
1633	099	\N	Lambruisse	04
1634	100	\N	Larche	04
1635	101	\N	Lardiers	04
1636	102	\N	Lauzet-Ubaye	04
1637	104	\N	Limans	04
1638	106	\N	Lurs	04
1639	107	\N	Majastres	04
1640	108	\N	Malijai	04
1641	109	\N	Mallefougasse-Augès	04
1642	110	\N	Mallemoisson	04
1643	111	\N	Mane	04
1644	112	\N	Manosque	04
1645	113	\N	Marcoux	04
1646	115	\N	Méailles	04
1647	116	\N	Mées	04
1648	118	\N	Melve	04
1649	120	\N	Meyronnes	04
1650	121	\N	Mézel	04
1651	122	\N	Mirabeau	04
1652	123	\N	Mison	04
1653	124	\N	Montagnac-Montpezat	04
1654	126	\N	Montclar	04
1655	127	\N	Montfort	04
1656	128	\N	Montfuron	04
1657	129	\N	Montjustin	04
1658	130	\N	Montlaux	04
1659	132	\N	Montsalier	04
1660	133	\N	Moriez	04
1661	134	\N	Motte-du-Caire	04
1662	135	\N	Moustiers-Sainte-Marie	04
1663	136	\N	Mure-Argens	04
1664	137	\N	Nibles	04
1665	138	\N	Niozelles	04
1666	139	\N	Noyers-sur-Jabron	04
1667	140	\N	Omergues	04
1668	141	\N	Ongles	04
1669	142	\N	Oppedette	04
1670	143	\N	Oraison	04
1671	144	\N	Palud-sur-Verdon	04
1672	145	\N	Peipin	04
1673	148	\N	Peyroules	04
1674	149	\N	Peyruis	04
1675	150	\N	Piégut	04
1676	151	\N	Pierrerue	04
1677	152	\N	Pierrevert	04
1678	154	\N	Pontis	04
1679	155	\N	Prads-Haute-Bléone	04
1680	156	\N	Puimichel	04
1681	157	\N	Puimoisson	04
1682	158	\N	Quinson	04
1683	159	\N	Redortiers	04
1684	160	\N	Reillanne	04
1685	161	\N	Méolans-Revel	04
1686	162	\N	Revest-des-Brousses	04
1687	163	\N	Revest-du-Bion	04
1688	164	\N	Revest-Saint-Martin	04
1689	166	\N	Riez	04
1690	167	\N	Robine-sur-Galabre	04
1691	169	\N	Rochegiron	04
1692	170	\N	Rochette	04
1693	171	\N	Rougon	04
1694	172	\N	Roumoules	04
1695	173	\N	Saint-André-les-Alpes	04
1696	174	\N	Saint-Benoît	04
1697	175	\N	Sainte-Croix-à-Lauze	04
1698	176	\N	Sainte-Croix-de-Verdon	04
1699	177	\N	Hautes-Duyes	04
1700	178	\N	Saint-Étienne-les-Orgues	04
1701	179	\N	Saint-Geniez	04
1702	180	\N	Saint-Jacques	04
1703	181	\N	Saint-Jeannet	04
1704	182	\N	Saint-Julien-d'Asse	04
1705	183	\N	Saint-Julien-du-Verdon	04
1706	184	\N	Saint-Jurs	04
1707	186	\N	Saint-Laurent-du-Verdon	04
1708	187	\N	Saint-Lions	04
1709	188	\N	Saint-Maime	04
1710	189	\N	Saint-Martin-de-Brômes	04
1711	190	\N	Saint-Martin-les-Eaux	04
1712	191	\N	Saint-Martin-lès-Seyne	04
1713	192	\N	Saint-Michel-l'Observatoire	04
1714	193	\N	Saint-Paul-sur-Ubaye	04
1715	194	\N	Saint-Pierre	04
1716	195	\N	Saint-Pons	04
1717	197	\N	Sainte-Tulle	04
1718	198	\N	Saint-Vincent-les-Forts	04
1719	199	\N	Saint-Vincent-sur-Jabron	04
1720	200	\N	Salignac	04
1721	201	\N	Saumane	04
1722	202	\N	Sausses	04
1723	203	\N	Selonnet	04
1724	204	\N	Senez	04
1725	205	\N	Seyne	04
1726	206	\N	Sigonce	04
1727	207	\N	Sigoyer	04
1728	208	\N	Simiane-la-Rotonde	04
1729	209	\N	Sisteron	04
1730	210	\N	Soleilhas	04
1731	211	\N	Sourribes	04
1732	214	\N	Tartonne	04
1733	216	\N	Thèze	04
1734	217	\N	Thoard	04
1735	218	\N	Thorame-Basse	04
1736	219	\N	Thorame-Haute	04
1737	220	\N	Thuiles	04
1738	222	\N	Turriers	04
1739	224	\N	Ubraye	04
1740	226	\N	Uvernet-Fours	04
1741	227	\N	Vachères	04
1742	228	\N	Valavoire	04
1743	229	\N	Valbelle	04
1744	230	\N	Valensole	04
1745	231	\N	Valernes	04
1746	233	\N	Vaumeilh	04
1747	234	\N	Venterol	04
1748	235	\N	Verdaches	04
1749	236	\N	Vergons	04
1750	237	\N	Vernet	04
1751	240	\N	Villars-Colmars	04
1752	241	\N	Villemus	04
1753	242	\N	Villeneuve	04
1754	244	\N	Volonne	04
1755	245	\N	Volx	04
1756	001	\N	Abriès	05
1757	002	\N	Agnières-en-Dévoluy	05
1758	003	\N	Aiguilles	05
1759	004	\N	Ancelle	05
1760	005	\N	Antonaves	05
1761	006	\N	Argentière-la-Bessée	05
1762	007	\N	Arvieux	05
1763	008	\N	Aspremont	05
1764	009	\N	Aspres-lès-Corps	05
1765	010	\N	Aspres-sur-Buëch	05
1766	011	\N	Avançon	05
1767	012	\N	Baratier	05
1768	013	\N	Barcillonnette	05
1769	014	\N	Barret-sur-Méouge	05
1770	016	\N	Bâtie-Montsaléon	05
1771	017	\N	Bâtie-Neuve	05
1772	018	\N	Bâtie-Vieille	05
1773	019	\N	Beaume	05
1774	020	\N	Bénévent-et-Charbillac	05
1775	021	\N	Bersac	05
1776	022	\N	Bréziers	05
1777	023	\N	Briançon	05
1778	024	\N	Bruis	05
1779	025	\N	Buissard	05
1780	026	\N	Ceillac	05
1781	027	\N	Cervières	05
1782	028	\N	Chabestan	05
1783	029	\N	Chabottes	05
1784	031	\N	Champcella	05
1785	032	\N	Champoléon	05
1786	033	\N	Chanousse	05
1787	034	\N	Châteauneuf-de-Chabre	05
1788	035	\N	Châteauneuf-d'Oze	05
1789	036	\N	Châteauroux-les-Alpes	05
1790	037	\N	Châteauvieux	05
1791	038	\N	Château-Ville-Vieille	05
1792	039	\N	Chauffayer	05
1793	040	\N	Chorges	05
1794	042	\N	Cluse	05
1795	043	\N	Costes	05
1796	044	\N	Crévoux	05
1797	045	\N	Crots	05
1798	046	\N	Embrun	05
1799	047	\N	Éourres	05
1800	048	\N	Épine	05
1801	049	\N	Esparron	05
1802	050	\N	Espinasses	05
1803	051	\N	Étoile-Saint-Cyrice	05
1804	052	\N	Eygliers	05
1805	053	\N	Eyguians	05
1806	054	\N	Fare-en-Champsaur	05
1807	055	\N	Faurie	05
1808	056	\N	Forest-Saint-Julien	05
1809	057	\N	Fouillouse	05
1810	058	\N	Freissinières	05
1811	059	\N	Freissinouse	05
1812	060	\N	Furmeyer	05
1813	061	\N	Gap	05
1814	062	\N	Glaizil	05
1815	063	\N	Grave	05
1816	064	\N	Chapelle-en-Valgaudémar	05
1817	065	\N	Guillestre	05
1818	066	\N	Haute-Beaume	05
1819	067	\N	Infournas	05
1820	068	\N	Jarjayes	05
1821	069	\N	Lagrand	05
1822	070	\N	Laragne-Montéglin	05
1823	071	\N	Lardier-et-Valença	05
1824	072	\N	Laye	05
1825	073	\N	Lazer	05
1826	074	\N	Lettret	05
1827	075	\N	Manteyer	05
1828	076	\N	Méreuil	05
1829	077	\N	Molines-en-Queyras	05
1830	078	\N	Monêtier-Allemont	05
1831	079	\N	Monêtier-les-Bains	05
1832	080	\N	Montbrand	05
1833	081	\N	Montclus	05
1834	082	\N	Mont-Dauphin	05
1835	084	\N	Montgardin	05
1836	085	\N	Montgenèvre	05
1837	086	\N	Montjay	05
1838	087	\N	Montmaur	05
1839	088	\N	Montmorin	05
1840	089	\N	Montrond	05
1841	090	\N	Motte-en-Champsaur	05
1842	091	\N	Moydans	05
1843	092	\N	Neffes	05
1844	093	\N	Névache	05
1845	094	\N	Nossage-et-Bénévent	05
1846	095	\N	Noyer	05
1847	096	\N	Orcières	05
1848	097	\N	Orpierre	05
1849	098	\N	Orres	05
1850	099	\N	Oze	05
1851	100	\N	Pelleautier	05
1852	101	\N	Pelvoux	05
1853	102	\N	Piarre	05
1854	103	\N	Poët	05
1855	104	\N	Poligny	05
1856	106	\N	Prunières	05
1857	107	\N	Puy-Saint-André	05
1858	108	\N	Puy-Saint-Eusèbe	05
1859	109	\N	Puy-Saint-Pierre	05
1860	110	\N	Puy-Saint-Vincent	05
1861	111	\N	Puy-Sanières	05
1862	112	\N	Rabou	05
1863	113	\N	Rambaud	05
1864	114	\N	Réallon	05
1865	115	\N	Remollon	05
1866	116	\N	Réotier	05
1867	117	\N	Ribeyret	05
1868	118	\N	Ribiers	05
1869	119	\N	Risoul	05
1870	120	\N	Ristolas	05
1871	121	\N	Rochebrune	05
1872	122	\N	Roche-de-Rame	05
1873	123	\N	Roche-des-Arnauds	05
1874	124	\N	Rochette	05
1875	126	\N	Rosans	05
1876	127	\N	Rousset	05
1877	128	\N	Saint-André-d'Embrun	05
1878	129	\N	Saint-André-de-Rosans	05
1879	130	\N	Saint-Apollinaire	05
1880	131	\N	Saint-Auban-d'Oze	05
1881	132	\N	Saint-Bonnet-en-Champsaur	05
1882	133	\N	Saint-Chaffrey	05
1883	134	\N	Saint-Clément-sur-Durance	05
1884	135	\N	Sainte-Colombe	05
1885	136	\N	Saint-Crépin	05
1886	138	\N	Saint-Disdier	05
1887	139	\N	Saint-Étienne-en-Dévoluy	05
1888	140	\N	Saint-Étienne-le-Laus	05
1889	141	\N	Saint-Eusèbe-en-Champsaur	05
1890	142	\N	Saint-Firmin	05
1891	143	\N	Saint-Genis	05
1892	144	\N	Saint-Jacques-en-Valgodemard	05
1893	145	\N	Saint-Jean-Saint-Nicolas	05
1894	146	\N	Saint-Julien-en-Beauchêne	05
1895	147	\N	Saint-Julien-en-Champsaur	05
1896	148	\N	Saint-Laurent-du-Cros	05
1897	149	\N	Saint-Léger-les-Mélèzes	05
1898	150	\N	Sainte-Marie	05
1899	151	\N	Saint-Martin-de-Queyrières	05
1900	152	\N	Saint-Maurice-en-Valgodemard	05
1901	153	\N	Saint-Michel-de-Chaillol	05
1902	154	\N	Saint-Pierre-d'Argençon	05
1903	155	\N	Saint-Pierre-Avez	05
1904	156	\N	Saint-Sauveur	05
1905	157	\N	Saint-Véran	05
1906	158	\N	Saix	05
1907	159	\N	Saléon	05
1908	160	\N	Salérans	05
1909	161	\N	Salle les Alpes	05
1910	162	\N	Saulce	05
1911	163	\N	Sauze-du-Lac	05
1912	164	\N	Savines-le-Lac	05
1913	165	\N	Savournon	05
1914	166	\N	Serres	05
1915	167	\N	Sigottier	05
1916	168	\N	Sigoyer	05
1917	169	\N	Sorbiers	05
1918	170	\N	Tallard	05
1919	171	\N	Théus	05
1920	172	\N	Trescléoux	05
1921	173	\N	Upaix	05
1922	174	\N	Val-des-Prés	05
1923	175	\N	Vallouise	05
1924	176	\N	Valserres	05
1925	177	\N	Vars	05
1926	178	\N	Ventavon	05
1927	179	\N	Veynes	05
1928	180	\N	Vigneaux	05
1929	181	\N	Villar-d'Arêne	05
1930	182	\N	Villar-Loubière	05
1931	183	\N	Villar-Saint-Pancrace	05
1932	184	\N	Vitrolles	05
1933	001	\N	Aiglun	06
1934	002	\N	Amirat	06
1935	003	\N	Andon	06
1936	004	\N	Antibes	06
1937	005	\N	Ascros	06
1938	006	\N	Aspremont	06
1939	007	\N	Auribeau-sur-Siagne	06
1940	008	\N	Auvare	06
1941	009	\N	Bairols	06
1942	010	\N	Bar-sur-Loup	06
1943	011	\N	Beaulieu-sur-Mer	06
1944	012	\N	Beausoleil	06
1945	013	\N	Belvédère	06
1946	014	\N	Bendejun	06
1947	015	\N	Berre-les-Alpes	06
1948	016	\N	Beuil	06
1949	017	\N	Bézaudun-les-Alpes	06
1950	018	\N	Biot	06
1951	019	\N	Blausasc	06
1952	020	\N	Bollène-Vésubie	06
1953	021	\N	Bonson	06
1954	022	\N	Bouyon	06
1955	023	\N	Breil-sur-Roya	06
1956	024	\N	Briançonnet	06
1957	025	\N	Broc	06
1958	026	\N	Cabris	06
1959	027	\N	Cagnes-sur-Mer	06
1960	028	\N	Caille	06
1961	029	\N	Cannes	06
1962	030	\N	Cannet	06
1963	031	\N	Cantaron	06
1964	032	\N	Cap-d'Ail	06
1965	033	\N	Carros	06
1966	034	\N	Castagniers	06
1967	035	\N	Castellar	06
1968	036	\N	Castillon	06
1969	037	\N	Caussols	06
1970	038	\N	Châteauneuf-Grasse	06
1971	039	\N	Châteauneuf-Villevieille	06
1972	040	\N	Châteauneuf-d'Entraunes	06
1973	041	\N	Cipières	06
1974	042	\N	Clans	06
1975	043	\N	Coaraze	06
1976	044	\N	Colle-sur-Loup	06
1977	045	\N	Collongues	06
1978	046	\N	Colomars	06
1979	047	\N	Conségudes	06
1980	048	\N	Contes	06
1981	049	\N	Courmes	06
1982	050	\N	Coursegoules	06
1983	051	\N	Croix-sur-Roudoule	06
1984	052	\N	Cuébris	06
1985	053	\N	Daluis	06
1986	054	\N	Drap	06
1987	055	\N	Duranus	06
1988	056	\N	Entraunes	06
1989	057	\N	Escarène	06
1990	058	\N	Escragnolles	06
1991	059	\N	Èze	06
1992	060	\N	Falicon	06
1993	061	\N	Ferres	06
1994	062	\N	Fontan	06
1995	063	\N	Gars	06
1996	064	\N	Gattières	06
1997	065	\N	Gaude	06
1998	066	\N	Gilette	06
1999	067	\N	Gorbio	06
2000	068	\N	Gourdon	06
2001	069	\N	Grasse	06
2002	070	\N	Gréolières	06
2003	071	\N	Guillaumes	06
2004	072	\N	Ilonse	06
2005	073	\N	Isola	06
2006	074	\N	Lantosque	06
2007	075	\N	Levens	06
2008	076	\N	Lieuche	06
2009	077	\N	Lucéram	06
2010	078	\N	Malaussène	06
2011	079	\N	Mandelieu-la-Napoule	06
2012	080	\N	Marie	06
2013	081	\N	Mas	06
2014	082	\N	Massoins	06
2015	083	\N	Menton	06
2016	084	\N	Mouans-Sartoux	06
2017	085	\N	Mougins	06
2018	086	\N	Moulinet	06
2019	087	\N	Mujouls	06
2020	088	\N	Nice	06
2021	089	\N	Opio	06
2022	090	\N	Pégomas	06
2023	091	\N	Peille	06
2024	092	\N	Peillon	06
2025	093	\N	Penne	06
2026	094	\N	Péone	06
2027	095	\N	Peymeinade	06
2028	096	\N	Pierlas	06
2029	097	\N	Pierrefeu	06
2030	098	\N	Puget-Rostang	06
2031	099	\N	Puget-Théniers	06
2032	100	\N	Revest-les-Roches	06
2033	101	\N	Rigaud	06
2034	102	\N	Rimplas	06
2035	103	\N	Roquebillière	06
2036	104	\N	Roquebrune-Cap-Martin	06
2037	105	\N	Roquefort-les-Pins	06
2038	106	\N	Roquesteron	06
2039	107	\N	Roquestéron-Grasse	06
2040	108	\N	Roquette-sur-Siagne	06
2041	109	\N	Roquette-sur-Var	06
2042	110	\N	Roubion	06
2043	111	\N	Roure	06
2044	112	\N	Rouret	06
2045	113	\N	Sainte-Agnès	06
2046	114	\N	Saint-André-de-la-Roche	06
2047	115	\N	Saint-Antonin	06
2048	116	\N	Saint-Auban	06
2049	117	\N	Saint-Blaise	06
2050	118	\N	Saint-Cézaire-sur-Siagne	06
2051	119	\N	Saint-Dalmas-le-Selvage	06
2052	120	\N	Saint-Étienne-de-Tinée	06
2053	121	\N	Saint-Jean-Cap-Ferrat	06
2054	122	\N	Saint-Jeannet	06
2055	123	\N	Saint-Laurent-du-Var	06
2056	124	\N	Saint-Léger	06
2057	125	\N	Saint-Martin-d'Entraunes	06
2058	126	\N	Saint-Martin-du-Var	06
2059	127	\N	Saint-Martin-Vésubie	06
2060	128	\N	Saint-Paul	06
2061	129	\N	Saint-Sauveur-sur-Tinée	06
2062	130	\N	Saint-Vallier-de-Thiey	06
2063	131	\N	Sallagriffon	06
2064	132	\N	Saorge	06
2065	133	\N	Sauze	06
2066	134	\N	Séranon	06
2067	135	\N	Sigale	06
2068	136	\N	Sospel	06
2069	137	\N	Spéracèdes	06
2070	138	\N	Théoule-sur-Mer	06
2071	139	\N	Thiéry	06
2072	140	\N	Tignet	06
2073	141	\N	Toudon	06
2074	142	\N	Touët-de-l'Escarène	06
2075	143	\N	Touët-sur-Var	06
2076	144	\N	Tour	06
2077	145	\N	Tourette-du-Château	06
2078	146	\N	Tournefort	06
2079	147	\N	Tourrette-Levens	06
2080	148	\N	Tourrettes-sur-Loup	06
2081	149	\N	Trinité	06
2082	150	\N	Turbie	06
2083	151	\N	Utelle	06
2084	152	\N	Valbonne	06
2085	153	\N	Valdeblore	06
2086	154	\N	Valderoure	06
2087	155	\N	Vallauris	06
2088	156	\N	Venanson	06
2089	157	\N	Vence	06
2090	158	\N	Villars-sur-Var	06
2091	159	\N	Villefranche-sur-Mer	06
2092	160	\N	Villeneuve-d'Entraunes	06
2093	161	\N	Villeneuve-Loubet	06
2094	162	\N	Brigue	06
2095	163	\N	Tende	06
2096	001	\N	Accons	07
2097	002	\N	Ailhon	07
2098	003	\N	Aizac	07
2099	004	\N	Ajoux	07
2100	005	\N	Alba-la-Romaine	07
2101	006	\N	Albon-d'Ardèche	07
2102	007	\N	Alboussière	07
2103	008	\N	Alissas	07
2104	009	\N	Andance	07
2105	010	\N	Annonay	07
2106	011	\N	Antraigues-sur-Volane	07
2107	012	\N	Arcens	07
2108	013	\N	Ardoix	07
2109	014	\N	Arlebosc	07
2110	015	\N	Arras-sur-Rhône	07
2111	016	\N	Asperjoc	07
2112	017	\N	Assions	07
2113	018	\N	Astet	07
2114	019	\N	Aubenas	07
2115	020	\N	Aubignas	07
2116	022	\N	Baix	07
2117	023	\N	Balazuc	07
2118	024	\N	Banne	07
2119	025	\N	Barnas	07
2120	026	\N	Béage	07
2121	027	\N	Beauchastel	07
2122	028	\N	Beaulieu	07
2123	029	\N	Beaumont	07
2124	030	\N	Beauvène	07
2125	031	\N	Berrias-et-Casteljau	07
2126	032	\N	Berzème	07
2127	033	\N	Bessas	07
2128	034	\N	Bidon	07
2129	035	\N	Boffres	07
2130	036	\N	Bogy	07
2131	037	\N	Borée	07
2132	038	\N	Borne	07
2133	039	\N	Bozas	07
2134	040	\N	Boucieu-le-Roi	07
2135	041	\N	Boulieu-lès-Annonay	07
2136	042	\N	Bourg-Saint-Andéol	07
2137	044	\N	Brossainc	07
2138	045	\N	Burzet	07
2139	047	\N	Cellier-du-Luc	07
2140	048	\N	Chalencon	07
2141	049	\N	Chambon	07
2142	050	\N	Chambonas	07
2143	051	\N	Champagne	07
2144	052	\N	Champis	07
2145	053	\N	Chandolas	07
2146	054	\N	Chanéac	07
2147	055	\N	Charmes-sur-Rhône	07
2148	056	\N	Charnas	07
2149	058	\N	Chassiers	07
2150	059	\N	Châteaubourg	07
2151	060	\N	Châteauneuf-de-Vernoux	07
2152	061	\N	Chauzon	07
2153	062	\N	Chazeaux	07
2154	063	\N	Cheminas	07
2155	064	\N	Cheylard	07
2156	065	\N	Chirols	07
2157	066	\N	Chomérac	07
2158	067	\N	Colombier-le-Cardinal	07
2159	068	\N	Colombier-le-Jeune	07
2160	069	\N	Colombier-le-Vieux	07
2161	070	\N	Cornas	07
2162	071	\N	Coucouron	07
2163	072	\N	Coux	07
2164	073	\N	Crestet	07
2165	074	\N	Creysseilles	07
2166	075	\N	Cros-de-Géorand	07
2167	076	\N	Cruas	07
2168	077	\N	Darbres	07
2169	078	\N	Davézieux	07
2170	079	\N	Désaignes	07
2171	080	\N	Devesset	07
2172	081	\N	Dompnac	07
2173	082	\N	Dornas	07
2174	083	\N	Dunière-sur-Eyrieux	07
2175	084	\N	Eclassan	07
2176	085	\N	Empurany	07
2177	086	\N	Étables	07
2178	087	\N	Fabras	07
2179	088	\N	Faugères	07
2180	089	\N	Félines	07
2181	090	\N	Flaviac	07
2182	091	\N	Fons	07
2183	092	\N	Freyssenet	07
2184	093	\N	Genestelle	07
2185	094	\N	Gilhac-et-Bruzac	07
2186	095	\N	Gilhoc-sur-Ormèze	07
2187	096	\N	Gluiras	07
2188	097	\N	Glun	07
2189	098	\N	Gourdon	07
2190	099	\N	Gras	07
2191	100	\N	Gravières	07
2192	101	\N	Grospierres	07
2193	102	\N	Guilherand-Granges	07
2194	103	\N	Intres	07
2195	104	\N	Issamoulenc	07
2196	105	\N	Issanlas	07
2197	106	\N	Issarlès	07
2198	107	\N	Jaujac	07
2199	108	\N	Jaunac	07
2200	109	\N	Joannas	07
2201	110	\N	Joyeuse	07
2202	111	\N	Juvinas	07
2203	112	\N	Labastide-sur-Bésorgues	07
2204	113	\N	Labastide-de-Virac	07
2205	114	\N	Labatie-d'Andaure	07
2206	115	\N	Labeaume	07
2207	116	\N	Labégude	07
2208	117	\N	Lablachère	07
2209	118	\N	Laboule	07
2210	119	\N	Lac-d'Issarlès	07
2211	120	\N	Lachamp-Raphaël	07
2212	121	\N	Lachapelle-Graillouse	07
2213	122	\N	Lachapelle-sous-Aubenas	07
2214	123	\N	Lachapelle-sous-Chanéac	07
2215	124	\N	Lafarre	07
2216	126	\N	Lagorce	07
2217	127	\N	Lalevade-d'Ardèche	07
2218	128	\N	Lalouvesc	07
2219	129	\N	Lamastre	07
2220	130	\N	Lanarce	07
2221	131	\N	Lanas	07
2222	132	\N	Largentière	07
2223	133	\N	Larnas	07
2224	134	\N	Laurac-en-Vivarais	07
2225	135	\N	Laval-d'Aurelle	07
2226	136	\N	Laveyrune	07
2227	137	\N	Lavillatte	07
2228	138	\N	Lavilledieu	07
2229	139	\N	Laviolle	07
2230	140	\N	Lemps	07
2231	141	\N	Lentillères	07
2232	142	\N	Lespéron	07
2233	143	\N	Limony	07
2234	144	\N	Loubaresse	07
2235	145	\N	Lussas	07
2236	146	\N	Lyas	07
2237	147	\N	Malarce-sur-la-Thines	07
2238	148	\N	Malbosc	07
2239	149	\N	Marcols-les-Eaux	07
2240	150	\N	Mariac	07
2241	151	\N	Mars	07
2242	152	\N	Mauves	07
2243	153	\N	Mayres	07
2244	154	\N	Mazan-l'Abbaye	07
2245	155	\N	Mercuer	07
2246	156	\N	Meyras	07
2247	157	\N	Meysse	07
2248	158	\N	Mézilhac	07
2249	159	\N	Mirabel	07
2250	160	\N	Monestier	07
2251	161	\N	Montpezat-sous-Bauzon	07
2252	162	\N	Montréal	07
2253	163	\N	Montselgues	07
2254	165	\N	Nonières	07
2255	166	\N	Nozières	07
2256	167	\N	Ollières-sur-Eyrieux	07
2257	168	\N	Orgnac-l'Aven	07
2258	169	\N	Ozon	07
2259	170	\N	Pailharès	07
2260	171	\N	Payzac	07
2261	172	\N	Peaugres	07
2262	173	\N	Péreyres	07
2263	174	\N	Peyraud	07
2264	175	\N	Plagnal	07
2265	176	\N	Planzolles	07
2266	177	\N	Plats	07
2267	178	\N	Pont-de-Labeaume	07
2268	179	\N	Pourchères	07
2269	181	\N	Pouzin	07
2270	182	\N	Prades	07
2271	183	\N	Pradons	07
2272	184	\N	Pranles	07
2273	185	\N	Préaux	07
2274	186	\N	Privas	07
2275	187	\N	Prunet	07
2276	188	\N	Quintenas	07
2277	189	\N	Ribes	07
2278	190	\N	Rochecolombe	07
2279	191	\N	Rochemaure	07
2280	192	\N	Rochepaule	07
2281	193	\N	Rocher	07
2282	194	\N	Rochessauve	07
2283	195	\N	Rochette	07
2284	196	\N	Rocles	07
2285	197	\N	Roiffieux	07
2286	198	\N	Rompon	07
2287	199	\N	Rosières	07
2288	200	\N	Roux	07
2289	201	\N	Ruoms	07
2290	202	\N	Sablières	07
2291	203	\N	Sagnes-et-Goudoulet	07
2292	204	\N	Saint-Agrève	07
2293	205	\N	Saint-Alban-d'Ay	07
2294	206	\N	Saint-Alban-en-Montagne	07
2295	207	\N	Saint-Alban-Auriolles	07
2296	208	\N	Saint-Andéol-de-Berg	07
2297	209	\N	Saint-Andéol-de-Fourchades	07
2298	210	\N	Saint-Andéol-de-Vals	07
2299	211	\N	Saint-André-de-Cruzières	07
2300	212	\N	Saint-André-en-Vivarais	07
2301	213	\N	Saint-André-Lachamp	07
2302	214	\N	Saint-Apollinaire-de-Rias	07
2303	215	\N	Saint-Barthélemy-le-Meil	07
2304	216	\N	Saint-Barthélemy-Grozon	07
2305	217	\N	Saint-Barthélemy-le-Plain	07
2306	218	\N	Saint-Basile	07
2307	219	\N	Saint-Bauzile	07
2308	220	\N	Saint-Christol	07
2309	221	\N	Saint-Cierge-la-Serre	07
2310	222	\N	Saint-Cierge-sous-le-Cheylard	07
2311	223	\N	Saint-Cirgues-de-Prades	07
2312	224	\N	Saint-Cirgues-en-Montagne	07
2313	225	\N	Saint-Clair	07
2314	226	\N	Saint-Clément	07
2315	227	\N	Saint-Cyr	07
2316	228	\N	Saint-Désirat	07
2317	229	\N	Saint-Didier-sous-Aubenas	07
2318	230	\N	Saint-Étienne-de-Boulogne	07
2319	231	\N	Saint-Étienne-de-Fontbellon	07
2320	232	\N	Saint-Étienne-de-Lugdarès	07
2321	233	\N	Saint-Étienne-de-Serre	07
2322	234	\N	Saint-Étienne-de-Valoux	07
2323	235	\N	Sainte-Eulalie	07
2324	236	\N	Saint-Félicien	07
2325	237	\N	Saint-Fortunat-sur-Eyrieux	07
2326	238	\N	Saint-Genest-de-Beauzon	07
2327	239	\N	Saint-Genest-Lachamp	07
2328	240	\N	Saint-Georges-les-Bains	07
2329	241	\N	Saint-Germain	07
2330	242	\N	Saint-Gineis-en-Coiron	07
2331	243	\N	Saint-Jacques-d'Atticieux	07
2332	244	\N	Saint-Jean-Chambre	07
2333	245	\N	Saint-Jean-de-Muzols	07
2334	247	\N	Saint-Jean-le-Centenier	07
2335	248	\N	Saint-Jean-Roure	07
2336	249	\N	Saint-Jeure-d'Andaure	07
2337	250	\N	Saint-Jeure-d'Ay	07
2338	251	\N	Saint-Joseph-des-Bancs	07
2339	252	\N	Saint-Julien-Boutières	07
2340	253	\N	Saint-Julien-du-Gua	07
2341	254	\N	Saint-Julien-du-Serre	07
2342	255	\N	Saint-Julien-en-Saint-Alban	07
2343	256	\N	Saint-Julien-Labrousse	07
2344	257	\N	Saint-Julien-le-Roux	07
2345	258	\N	Saint-Julien-Vocance	07
2346	259	\N	Saint-Just	07
2347	260	\N	Saint-Lager-Bressac	07
2348	261	\N	Saint-Laurent-du-Pape	07
2349	262	\N	Saint-Laurent-les-Bains	07
2350	263	\N	Saint-Laurent-sous-Coiron	07
2351	264	\N	Saint-Marcel-d'Ardèche	07
2352	265	\N	Saint-Marcel-lès-Annonay	07
2353	266	\N	Sainte-Marguerite-Lafigère	07
2354	267	\N	Saint-Martial	07
2355	268	\N	Saint-Martin-d'Ardèche	07
2356	269	\N	Saint-Martin-de-Valamas	07
2357	270	\N	Saint-Martin-sur-Lavezon	07
2358	272	\N	Saint-Maurice-d'Ardèche	07
2359	273	\N	Saint-Maurice-d'Ibie	07
2360	274	\N	Saint-Maurice-en-Chalencon	07
2361	275	\N	Saint-Mélany	07
2362	276	\N	Saint-Michel-d'Aurance	07
2363	277	\N	Saint-Michel-de-Boulogne	07
2364	278	\N	Saint-Michel-de-Chabrillanoux	07
2365	279	\N	Saint-Montan	07
2366	280	\N	Saint-Paul-le-Jeune	07
2367	281	\N	Saint-Péray	07
2368	282	\N	Saint-Pierre-de-Colombier	07
2369	283	\N	Saint-Pierre-la-Roche	07
2370	284	\N	Saint-Pierre-Saint-Jean	07
2371	285	\N	Saint-Pierre-sur-Doux	07
2372	286	\N	Saint-Pierreville	07
2373	287	\N	Saint-Pons	07
2374	288	\N	Saint-Priest	07
2375	289	\N	Saint-Privat	07
2376	290	\N	Saint-Prix	07
2377	291	\N	Saint-Remèze	07
2378	292	\N	Saint-Romain-d'Ay	07
2379	293	\N	Saint-Romain-de-Lerps	07
2380	294	\N	Saint-Sauveur-de-Cruzières	07
2381	295	\N	Saint-Sauveur-de-Montagut	07
2382	296	\N	Saint-Sernin	07
2383	297	\N	Saint-Sylvestre	07
2384	298	\N	Saint-Symphorien-sous-Chomérac	07
2385	299	\N	Saint-Symphorien-de-Mahun	07
2386	300	\N	Saint-Thomé	07
2387	301	\N	Saint-Victor	07
2388	302	\N	Saint-Vincent-de-Barrès	07
2389	303	\N	Saint-Vincent-de-Durfort	07
2390	304	\N	Salavas	07
2391	305	\N	Salelles	07
2392	306	\N	Sampzon	07
2393	307	\N	Sanilhac	07
2394	308	\N	Sarras	07
2395	309	\N	Satillieu	07
2396	310	\N	Savas	07
2397	311	\N	Sceautres	07
2398	312	\N	Sécheras	07
2399	313	\N	Serrières	07
2400	314	\N	Silhac	07
2401	315	\N	Souche	07
2402	316	\N	Soyons	07
2403	317	\N	Talencieux	07
2404	318	\N	Tauriers	07
2405	319	\N	Teil	07
2406	321	\N	Thorrenc	07
2407	322	\N	Thueyts	07
2408	323	\N	Toulaud	07
2409	324	\N	Tournon-sur-Rhône	07
2410	325	\N	Ucel	07
2411	326	\N	Usclades-et-Rieutord	07
2412	327	\N	Uzer	07
2413	328	\N	Vagnas	07
2414	329	\N	Valgorge	07
2415	330	\N	Vallon-Pont-d'Arc	07
2416	331	\N	Vals-les-Bains	07
2417	332	\N	Valvignères	07
2418	333	\N	Vanosc	07
2419	334	\N	Vans	07
2420	335	\N	Vaudevant	07
2421	336	\N	Vernon	07
2422	337	\N	Vernosc-lès-Annonay	07
2423	338	\N	Vernoux-en-Vivarais	07
2424	339	\N	Vesseaux	07
2425	340	\N	Veyras	07
2426	341	\N	Villeneuve-de-Berg	07
2427	342	\N	Villevocance	07
2428	343	\N	Vinezac	07
2429	344	\N	Vinzieux	07
2430	345	\N	Vion	07
2431	346	\N	Viviers	07
2432	347	\N	Vocance	07
2433	348	\N	Vogüé	07
2434	349	\N	Voulte-sur-Rhône	07
2435	001	\N	Acy-Romance	08
2436	003	\N	Aiglemont	08
2437	004	\N	Aire	08
2438	005	\N	Alincourt	08
2439	006	\N	Alland'Huy-et-Sausseuil	08
2440	007	\N	Alleux	08
2441	008	\N	Amagne	08
2442	009	\N	Amblimont	08
2443	010	\N	Ambly-Fleury	08
2444	011	\N	Anchamps	08
2445	013	\N	Angecourt	08
2446	014	\N	Annelles	08
2447	015	\N	Antheny	08
2448	016	\N	Aouste	08
2449	017	\N	Apremont	08
2450	018	\N	Ardeuil-et-Montfauxelles	08
2451	019	\N	Grandes-Armoises	08
2452	020	\N	Petites-Armoises	08
2453	021	\N	Arnicourt	08
2454	022	\N	Arreux	08
2455	023	\N	Artaise-le-Vivier	08
2456	024	\N	Asfeld	08
2457	025	\N	Attigny	08
2458	026	\N	Aubigny-les-Pothées	08
2459	027	\N	Auboncourt-Vauzelles	08
2460	028	\N	Aubrives	08
2461	029	\N	Auflance	08
2462	030	\N	Auge	08
2463	031	\N	Aure	08
2464	032	\N	Aussonce	08
2465	033	\N	Authe	08
2466	034	\N	Autrecourt-et-Pourron	08
2467	035	\N	Autruche	08
2468	036	\N	Autry	08
2469	037	\N	Auvillers-les-Forges	08
2470	038	\N	Avançon	08
2471	039	\N	Avaux	08
2472	040	\N	Ayvelles	08
2473	041	\N	Baâlons	08
2474	042	\N	Balaives-et-Butz	08
2475	043	\N	Balan	08
2476	044	\N	Balham	08
2477	045	\N	Ballay	08
2478	046	\N	Banogne-Recouvrance	08
2479	047	\N	Barbaise	08
2480	048	\N	Barby	08
2481	049	\N	Bar-lès-Buzancy	08
2482	052	\N	Bayonville	08
2483	053	\N	Bazeilles	08
2484	055	\N	Beaumont-en-Argonne	08
2485	056	\N	Beffu-et-le-Morthomme	08
2486	057	\N	Belleville-et-Châtillon-sur-Bar	08
2487	058	\N	Belval	08
2488	059	\N	Belval-Bois-des-Dames	08
2489	060	\N	Bergnicourt	08
2490	061	\N	Berlière	08
2491	062	\N	Bertoncourt	08
2492	063	\N	Besace	08
2493	064	\N	Biermes	08
2494	065	\N	Bièvres	08
2495	066	\N	Bignicourt	08
2496	067	\N	Blagny	08
2497	069	\N	Blanchefosse-et-Bay	08
2498	070	\N	Blanzy-la-Salonnaise	08
2499	071	\N	Blombay	08
2500	072	\N	Bosseval-et-Briancourt	08
2501	073	\N	Bossus-lès-Rumigny	08
2502	074	\N	Bouconville	08
2503	075	\N	Boult-aux-Bois	08
2504	076	\N	Boulzicourt	08
2505	077	\N	Bourcq	08
2506	078	\N	Bourg-Fidèle	08
2507	079	\N	Boutancourt	08
2508	080	\N	Bouvellemont	08
2509	081	\N	Bogny-sur-Meuse	08
2510	082	\N	Brécy-Brières	08
2511	083	\N	Brévilly	08
2512	084	\N	Brienne-sur-Aisne	08
2513	085	\N	Brieulles-sur-Bar	08
2514	086	\N	Briquenay	08
2515	087	\N	Brognon	08
2516	088	\N	Bulson	08
2517	089	\N	Buzancy	08
2518	090	\N	Carignan	08
2519	092	\N	Cauroy	08
2520	094	\N	Cernion	08
2521	095	\N	Chagny	08
2522	096	\N	Chalandry-Elaire	08
2523	097	\N	Challerange	08
2524	098	\N	Champigneulle	08
2525	099	\N	Champigneul-sur-Vence	08
2526	100	\N	Champlin	08
2527	101	\N	Chapelle	08
2528	102	\N	Chappes	08
2529	103	\N	Charbogne	08
2530	104	\N	Chardeny	08
2531	105	\N	Charleville-Mézières	08
2532	106	\N	Charnois	08
2533	107	\N	Château-Porcien	08
2534	109	\N	Chatel-Chéhéry	08
2535	110	\N	Châtelet-sur-Sormonne	08
2536	111	\N	Châtelet-sur-Retourne	08
2537	113	\N	Chaumont-Porcien	08
2538	114	\N	Chéhéry	08
2539	115	\N	Chémery-sur-Bar	08
2540	116	\N	Chesne	08
2541	117	\N	Chesnois-Auboncourt	08
2542	119	\N	Cheveuges	08
2543	120	\N	Chevières	08
2544	121	\N	Chilly	08
2545	122	\N	Chooz	08
2546	123	\N	Chuffilly-Roche	08
2547	124	\N	Clavy-Warby	08
2548	125	\N	Cliron	08
2549	126	\N	Condé-lès-Herpy	08
2550	128	\N	Condé-lès-Autry	08
2551	130	\N	Contreuve	08
2552	131	\N	Cornay	08
2553	132	\N	Corny-Machéroménil	08
2554	133	\N	Coucy	08
2555	134	\N	Coulommes-et-Marqueny	08
2556	135	\N	Croix-aux-Bois	08
2557	136	\N	Daigny	08
2558	137	\N	Damouzy	08
2559	138	\N	Deux-Villes	08
2560	139	\N	Deville	08
2561	140	\N	Dom-le-Mesnil	08
2562	141	\N	Dommery	08
2563	142	\N	Donchery	08
2564	143	\N	Doumely-Bégny	08
2565	144	\N	Doux	08
2566	145	\N	Douzy	08
2567	146	\N	Draize	08
2568	147	\N	Dricourt	08
2569	148	\N	Écaille	08
2570	149	\N	Échelle	08
2571	150	\N	Écly	08
2572	151	\N	Écordal	08
2573	152	\N	Élan	08
2574	153	\N	Escombres-et-le-Chesnois	08
2575	154	\N	Estrebay	08
2576	155	\N	Étalle	08
2577	156	\N	Éteignières	08
2578	158	\N	Étrépigny	08
2579	159	\N	Euilly-et-Lombut	08
2580	160	\N	Évigny	08
2581	161	\N	Exermont	08
2582	162	\N	Fagnon	08
2583	163	\N	Faissault	08
2584	164	\N	Falaise	08
2585	165	\N	Faux	08
2586	166	\N	Fépin	08
2587	167	\N	Férée	08
2588	168	\N	Ferté-sur-Chiers	08
2589	169	\N	Flaignes-Havys	08
2590	170	\N	Fleigneux	08
2591	171	\N	Fléville	08
2592	172	\N	Fligny	08
2593	173	\N	Flize	08
2594	174	\N	Floing	08
2595	175	\N	Foisches	08
2596	176	\N	Fossé	08
2597	178	\N	Fraillicourt	08
2598	179	\N	Francheval	08
2599	180	\N	Francheville	08
2600	182	\N	Fréty	08
2601	183	\N	Fromelennes	08
2602	184	\N	Fromy	08
2603	185	\N	Fumay	08
2604	186	\N	Germont	08
2605	187	\N	Gernelle	08
2606	188	\N	Gespunsart	08
2607	189	\N	Girondelle	08
2608	190	\N	Givet	08
2609	191	\N	Givonne	08
2610	192	\N	Givron	08
2611	193	\N	Givry	08
2612	194	\N	Glaire	08
2613	195	\N	Gomont	08
2614	196	\N	Grandchamp	08
2615	197	\N	Grandham	08
2616	198	\N	Grandpré	08
2617	199	\N	Grandville	08
2618	200	\N	Grivy-Loisy	08
2619	201	\N	Gruyères	08
2620	202	\N	Gué-d'Hossus	08
2621	203	\N	Guignicourt-sur-Vence	08
2622	204	\N	Guincourt	08
2623	205	\N	Hagnicourt	08
2624	206	\N	Ham-les-Moines	08
2625	207	\N	Ham-sur-Meuse	08
2626	208	\N	Hannappes	08
2627	209	\N	Hannogne-Saint-Martin	08
2628	210	\N	Hannogne-Saint-Rémy	08
2629	211	\N	Haraucourt	08
2630	212	\N	Harcy	08
2631	214	\N	Hargnies	08
2632	215	\N	Harricourt	08
2633	216	\N	Haudrecy	08
2634	217	\N	Haulmé	08
2635	218	\N	Hautes-Rivières	08
2636	219	\N	Hauteville	08
2637	220	\N	Hauviné	08
2638	222	\N	Haybes	08
2639	223	\N	Herbeuval	08
2640	225	\N	Herpy-l'Arlésienne	08
2641	226	\N	Hierges	08
2642	228	\N	Horgne	08
2643	229	\N	Houdilcourt	08
2644	230	\N	Houldizy	08
2645	232	\N	Illy	08
2646	233	\N	Imécourt	08
2647	234	\N	Inaumont	08
2648	235	\N	Issancourt-et-Rumel	08
2649	236	\N	Jandun	08
2650	237	\N	Joigny-sur-Meuse	08
2651	238	\N	Jonval	08
2652	239	\N	Juniville	08
2653	240	\N	Justine-Herbigny	08
2654	242	\N	Laifour	08
2655	243	\N	Lalobbe	08
2656	244	\N	Lametz	08
2657	245	\N	Lançon	08
2658	246	\N	Landres-et-Saint-Georges	08
2659	247	\N	Landrichamps	08
2660	248	\N	Launois-sur-Vence	08
2661	249	\N	Laval-Morency	08
2662	250	\N	Leffincourt	08
2663	251	\N	Lépron-les-Vallées	08
2664	252	\N	Létanne	08
2665	254	\N	Liart	08
2666	255	\N	Linay	08
2667	256	\N	Liry	08
2668	257	\N	Logny-Bogny	08
2669	259	\N	Longwé	08
2670	260	\N	Lonny	08
2671	261	\N	Louvergny	08
2672	262	\N	Lucquy	08
2673	263	\N	Lumes	08
2674	264	\N	Machault	08
2675	267	\N	Mairy	08
2676	268	\N	Maisoncelle-et-Villers	08
2677	269	\N	Malandry	08
2678	271	\N	Manre	08
2679	272	\N	Maranwez	08
2680	273	\N	Marby	08
2681	274	\N	Marcq	08
2682	275	\N	Margny	08
2683	276	\N	Margut	08
2684	277	\N	Marlemont	08
2685	278	\N	Marquigny	08
2686	279	\N	Mars-sous-Bourcq	08
2687	280	\N	Marvaux-Vieux	08
2688	281	\N	Matton-et-Clémency	08
2689	282	\N	Maubert-Fontaine	08
2690	283	\N	Mazerny	08
2691	284	\N	Mazures	08
2692	286	\N	Ménil-Annelles	08
2693	287	\N	Ménil-Lépinois	08
2694	288	\N	Mesmont	08
2695	289	\N	Messincourt	08
2696	291	\N	Mogues	08
2697	293	\N	Moiry	08
2698	294	\N	Moncelle	08
2699	295	\N	Mondigny	08
2700	296	\N	Montcheutin	08
2701	297	\N	Montcornet	08
2702	298	\N	Montcy-Notre-Dame	08
2703	300	\N	Mont-Dieu	08
2704	301	\N	Montgon	08
2705	302	\N	Monthermé	08
2706	303	\N	Monthois	08
2707	304	\N	Montigny-sur-Meuse	08
2708	305	\N	Montigny-sur-Vence	08
2709	306	\N	Mont-Laurent	08
2710	307	\N	Montmeillant	08
2711	308	\N	Mont-Saint-Martin	08
2712	309	\N	Mont-Saint-Remy	08
2713	310	\N	Mouron	08
2714	311	\N	Mouzon	08
2715	312	\N	Murtin-et-Bogny	08
2716	313	\N	Nanteuil-sur-Aisne	08
2717	314	\N	Neuflize	08
2718	315	\N	Neufmaison	08
2719	316	\N	Neufmanil	08
2720	317	\N	Neuville-à-Maire	08
2721	318	\N	Neuville-aux-Joûtes	08
2722	319	\N	Neuville-lez-Beaulieu	08
2723	320	\N	Neuville-en-Tourne-à-Fuy	08
2724	321	\N	Neuville-Day	08
2725	322	\N	Neuville-lès-This	08
2726	323	\N	Neuville-lès-Wasigny	08
2727	324	\N	Neuvizy	08
2728	325	\N	Noirval	08
2729	326	\N	Nouart	08
2730	327	\N	Nouvion-sur-Meuse	08
2731	328	\N	Nouzonville	08
2732	329	\N	Novion-Porcien	08
2733	330	\N	Novy-Chevrières	08
2734	331	\N	Noyers-Pont-Maugis	08
2735	332	\N	Oches	08
2736	333	\N	Olizy-Primat	08
2737	334	\N	Omicourt	08
2738	335	\N	Omont	08
2739	336	\N	Osnes	08
2740	338	\N	Pauvres	08
2741	339	\N	Perthes	08
2742	340	\N	Poilcourt-Sydney	08
2743	341	\N	Poix-Terron	08
2744	342	\N	Pouru-aux-Bois	08
2745	343	\N	Pouru-Saint-Remy	08
2746	344	\N	Prez	08
2747	346	\N	Prix-lès-Mézières	08
2748	347	\N	Puilly-et-Charbeaux	08
2749	348	\N	Puiseux	08
2750	349	\N	Pure	08
2751	350	\N	Quatre-Champs	08
2752	351	\N	Quilly	08
2753	352	\N	Raillicourt	08
2754	353	\N	Rancennes	08
2755	354	\N	Raucourt-et-Flaba	08
2756	355	\N	Regniowez	08
2757	356	\N	Remaucourt	08
2758	357	\N	Remilly-Aillicourt	08
2759	358	\N	Remilly-les-Pothées	08
2760	360	\N	Renneville	08
2761	361	\N	Renwez	08
2762	362	\N	Rethel	08
2763	363	\N	Revin	08
2764	364	\N	Rilly-sur-Aisne	08
2765	365	\N	Rimogne	08
2766	366	\N	Rocquigny	08
2767	367	\N	Rocroi	08
2768	368	\N	Roizy	08
2769	369	\N	Romagne	08
2770	370	\N	Rouvroy-sur-Audry	08
2771	371	\N	Rubécourt-et-Lamécourt	08
2772	372	\N	Rubigny	08
2773	373	\N	Rumigny	08
2774	374	\N	Sabotterie	08
2775	375	\N	Sachy	08
2776	376	\N	Sailly	08
2777	377	\N	Saint-Aignan	08
2778	378	\N	Saint-Clément-à-Arnes	08
2779	379	\N	Saint-Étienne-à-Arnes	08
2780	380	\N	Saint-Fergeux	08
2781	381	\N	Saint-Germainmont	08
2782	382	\N	Saint-Jean-aux-Bois	08
2783	383	\N	Saint-Juvin	08
2784	384	\N	Saint-Lambert-et-Mont-de-Jeux	08
2785	385	\N	Saint-Laurent	08
2786	386	\N	Saint-Loup-en-Champagne	08
2787	387	\N	Saint-Loup-Terrier	08
2788	388	\N	Saint-Marceau	08
2789	389	\N	Saint-Marcel	08
2790	390	\N	Sainte-Marie	08
2791	391	\N	Saint-Menges	08
2792	392	\N	Saint-Morel	08
2793	393	\N	Saint-Pierre-à-Arnes	08
2794	394	\N	Saint-Pierremont	08
2795	395	\N	Saint-Pierre-sur-Vence	08
2796	396	\N	Saint-Quentin-le-Petit	08
2797	397	\N	Saint-Remy-le-Petit	08
2798	398	\N	Sainte-Vaubourg	08
2799	399	\N	Sapogne-sur-Marche	08
2800	400	\N	Sapogne-et-Feuchères	08
2801	401	\N	Saulces-Champenoises	08
2802	402	\N	Saulces-Monclin	08
2803	403	\N	Sault-lès-Rethel	08
2804	404	\N	Sault-Saint-Remy	08
2805	405	\N	Sauville	08
2806	406	\N	Savigny-sur-Aisne	08
2807	407	\N	Séchault	08
2808	408	\N	Sécheval	08
2809	409	\N	Sedan	08
2810	410	\N	Semide	08
2811	411	\N	Semuy	08
2812	412	\N	Senuc	08
2813	413	\N	Seraincourt	08
2814	415	\N	Sery	08
2815	416	\N	Seuil	08
2816	417	\N	Sévigny-la-Forêt	08
2817	418	\N	Sévigny-Waleppe	08
2818	419	\N	Signy-l'Abbaye	08
2819	420	\N	Signy-le-Petit	08
2820	421	\N	Signy-Montlibert	08
2821	422	\N	Singly	08
2822	424	\N	Sommauthe	08
2823	425	\N	Sommerance	08
2824	426	\N	Son	08
2825	427	\N	Sorbon	08
2826	428	\N	Sorcy-Bauthémont	08
2827	429	\N	Sormonne	08
2828	430	\N	Stonne	08
2829	431	\N	Sugny	08
2830	432	\N	Sury	08
2831	433	\N	Suzanne	08
2832	434	\N	Sy	08
2833	435	\N	Tagnon	08
2834	436	\N	Taillette	08
2835	437	\N	Tailly	08
2836	438	\N	Taizy	08
2837	439	\N	Tannay	08
2838	440	\N	Tarzy	08
2839	441	\N	Termes	08
2840	443	\N	Terron-sur-Aisne	08
2841	444	\N	Tétaigne	08
2842	445	\N	Thelonne	08
2843	446	\N	Thénorgues	08
2844	448	\N	Thilay	08
2845	449	\N	Thin-le-Moutier	08
2846	450	\N	This	08
2847	451	\N	Thour	08
2848	452	\N	Thugny-Trugny	08
2849	453	\N	Toges	08
2850	454	\N	Touligny	08
2851	455	\N	Tourcelles-Chaumont	08
2852	456	\N	Tournavaux	08
2853	457	\N	Tournes	08
2854	458	\N	Tourteron	08
2855	459	\N	Tremblois-lès-Carignan	08
2856	460	\N	Tremblois-lès-Rocroi	08
2857	461	\N	Vandy	08
2858	462	\N	Vaux-Champagne	08
2859	463	\N	Vaux-en-Dieulet	08
2860	464	\N	Vaux-lès-Mouron	08
2861	465	\N	Vaux-lès-Rubigny	08
2862	466	\N	Vaux-lès-Mouzon	08
2863	467	\N	Vaux-Montreuil	08
2864	468	\N	Vaux-Villaine	08
2865	469	\N	Vendresse	08
2866	470	\N	Verpel	08
2867	471	\N	Verrières	08
2868	472	\N	Viel-Saint-Remy	08
2869	473	\N	Vieux-lès-Asfeld	08
2870	475	\N	Villers-Cernay	08
2871	476	\N	Villers-devant-le-Thour	08
2872	477	\N	Villers-devant-Mouzon	08
2873	478	\N	Villers-le-Tilleul	08
2874	479	\N	Villers-le-Tourneur	08
2875	480	\N	Villers-Semeuse	08
2876	481	\N	Villers-sur-Bar	08
2877	482	\N	Villers-sur-le-Mont	08
2878	483	\N	Ville-sur-Lumes	08
2879	484	\N	Ville-sur-Retourne	08
2880	485	\N	Villy	08
2881	486	\N	Vireux-Molhain	08
2882	487	\N	Vireux-Wallerand	08
2883	488	\N	Vivier-au-Court	08
2884	489	\N	Voncq	08
2885	490	\N	Vouziers	08
2886	491	\N	Vrigne-aux-Bois	08
2887	492	\N	Vrigne-Meuse	08
2888	493	\N	Vrizy	08
2889	494	\N	Wadelincourt	08
2890	496	\N	Wagnon	08
2891	497	\N	Warcq	08
2892	498	\N	Warnécourt	08
2893	499	\N	Wasigny	08
2894	500	\N	Wignicourt	08
2895	501	\N	Williers	08
2896	502	\N	Yoncq	08
2897	503	\N	Yvernaumont	08
2898	001	\N	Aigues-Juntes	09
2899	002	\N	Aigues-Vives	09
2900	003	\N	Aiguillon	09
2901	004	\N	Albiès	09
2902	005	\N	Aleu	09
2903	006	\N	Alliat	09
2904	007	\N	Allières	09
2905	008	\N	Alos	09
2906	009	\N	Alzen	09
2907	011	\N	Antras	09
2908	012	\N	Appy	09
2909	013	\N	Arabaux	09
2910	014	\N	Argein	09
2911	015	\N	Arignac	09
2912	016	\N	Arnave	09
2913	017	\N	Arrien-en-Bethmale	09
2914	018	\N	Arrout	09
2915	019	\N	Artigat	09
2916	020	\N	Artigues	09
2917	021	\N	Artix	09
2918	022	\N	Arvigna	09
2919	023	\N	Ascou	09
2920	024	\N	Aston	09
2921	025	\N	Aucazein	09
2922	026	\N	Audressein	09
2923	027	\N	Augirein	09
2924	028	\N	Aulos	09
2925	029	\N	Aulus-les-Bains	09
2926	030	\N	Auzat	09
2927	031	\N	Axiat	09
2928	032	\N	Ax-les-Thermes	09
2929	033	\N	Bagert	09
2930	034	\N	Balacet	09
2931	035	\N	Balaguères	09
2932	037	\N	Barjac	09
2933	038	\N	Bastide-de-Besplas	09
2934	039	\N	Bastide-de-Bousignac	09
2935	040	\N	Bastide-de-Lordat	09
2936	041	\N	Bastide-du-Salat	09
2937	042	\N	Bastide-de-Sérou	09
2938	043	\N	Bastide-sur-l'Hers	09
2939	044	\N	Baulou	09
2940	045	\N	Bédeilhac-et-Aynat	09
2941	046	\N	Bédeille	09
2942	047	\N	Bélesta	09
2943	048	\N	Belloc	09
2944	049	\N	Bénac	09
2945	050	\N	Benagues	09
2946	051	\N	Bénaix	09
2947	052	\N	Besset	09
2948	053	\N	Bestiac	09
2949	054	\N	Betchat	09
2950	055	\N	Bethmale	09
2951	056	\N	Bézac	09
2952	057	\N	Biert	09
2953	058	\N	Bompas	09
2954	059	\N	Bonac-Irazein	09
2955	060	\N	Bonnac	09
2956	061	\N	Bordes-sur-Arize	09
2957	062	\N	Bordes-sur-Lez	09
2958	063	\N	Bosc	09
2959	064	\N	Bouan	09
2960	065	\N	Boussenac	09
2961	066	\N	Brassac	09
2962	067	\N	Brie	09
2963	068	\N	Burret	09
2964	069	\N	Buzan	09
2965	070	\N	Cabannes	09
2966	071	\N	Cadarcet	09
2967	072	\N	Calzan	09
2968	073	\N	Camarade	09
2969	074	\N	Camon	09
2970	075	\N	Campagne-sur-Arize	09
2971	076	\N	Canté	09
2972	077	\N	Capoulet-et-Junac	09
2973	078	\N	Carcanières	09
2974	079	\N	Carla-Bayle	09
2975	080	\N	Carla-de-Roquefort	09
2976	081	\N	Carlaret	09
2977	082	\N	Castelnau-Durban	09
2978	083	\N	Castéras	09
2979	084	\N	Castex	09
2980	085	\N	Castillon-en-Couserans	09
2981	086	\N	Caumont	09
2982	087	\N	Caussou	09
2983	088	\N	Caychax	09
2984	089	\N	Cazals-des-Baylès	09
2985	090	\N	Cazaux	09
2986	091	\N	Cazavet	09
2987	092	\N	Cazenave-Serres-et-Allens	09
2988	093	\N	Celles	09
2989	094	\N	Cérizols	09
2990	095	\N	Cescau	09
2991	096	\N	Château-Verdun	09
2992	097	\N	Clermont	09
2993	098	\N	Contrazy	09
2994	099	\N	Cos	09
2995	100	\N	Couflens	09
2996	101	\N	Coussa	09
2997	102	\N	Coutens	09
2998	103	\N	Crampagna	09
2999	104	\N	Dalou	09
3000	105	\N	Daumazan-sur-Arize	09
3001	106	\N	Dreuilhe	09
3002	107	\N	Dun	09
3003	108	\N	Durban-sur-Arize	09
3004	109	\N	Durfort	09
3005	110	\N	Encourtiech	09
3006	111	\N	Engomer	09
3007	113	\N	Ercé	09
3008	114	\N	Erp	09
3009	115	\N	Esclagne	09
3010	116	\N	Escosse	09
3011	117	\N	Esplas	09
3012	118	\N	Esplas-de-Sérou	09
3013	119	\N	Eycheil	09
3014	120	\N	Fabas	09
3015	121	\N	Ferrières-sur-Ariège	09
3016	122	\N	Foix	09
3017	123	\N	Fornex	09
3018	124	\N	Fossat	09
3019	125	\N	Fougax-et-Barrineuf	09
3020	126	\N	Freychenet	09
3021	127	\N	Gabre	09
3022	128	\N	Gajan	09
3023	129	\N	Galey	09
3024	130	\N	Ganac	09
3025	131	\N	Garanou	09
3026	132	\N	Gaudiès	09
3027	133	\N	Génat	09
3028	134	\N	Gestiès	09
3029	135	\N	Goulier	09
3030	136	\N	Gourbit	09
3031	137	\N	Gudas	09
3032	138	\N	Herm	09
3033	139	\N	Hospitalet-près-l'Andorre	09
3034	140	\N	Ignaux	09
3035	141	\N	Illartein	09
3036	142	\N	Ilhat	09
3037	143	\N	Illier-et-Laramade	09
3038	145	\N	Issards	09
3039	146	\N	Justiniac	09
3040	147	\N	Labatut	09
3041	148	\N	Lacave	09
3042	149	\N	Lacourt	09
3043	150	\N	Lagarde	09
3044	151	\N	Lanoux	09
3045	152	\N	Lapège	09
3046	153	\N	Lapenne	09
3047	154	\N	Larbont	09
3048	155	\N	Larcat	09
3049	156	\N	Larnat	09
3050	157	\N	Laroque-d'Olmes	09
3051	158	\N	Lasserre	09
3052	159	\N	Lassur	09
3053	160	\N	Lavelanet	09
3054	161	\N	Léran	09
3055	162	\N	Lercoul	09
3056	163	\N	Lescousse	09
3057	164	\N	Lescure	09
3058	165	\N	Lesparrou	09
3059	166	\N	Leychert	09
3060	167	\N	Lézat-sur-Lèze	09
3061	168	\N	Lieurac	09
3062	169	\N	Limbrassac	09
3063	170	\N	Lissac	09
3064	171	\N	Lordat	09
3065	172	\N	Loubaut	09
3066	173	\N	Loubens	09
3067	174	\N	Loubières	09
3068	175	\N	Ludiès	09
3069	176	\N	Luzenac	09
3070	177	\N	Madière	09
3071	178	\N	Malegoude	09
3072	179	\N	Malléon	09
3073	180	\N	Manses	09
3074	181	\N	Mas-d'Azil	09
3075	182	\N	Massat	09
3076	183	\N	Mauvezin-de-Prat	09
3077	184	\N	Mauvezin-de-Sainte-Croix	09
3078	185	\N	Mazères	09
3079	186	\N	Méras	09
3080	187	\N	Mercenac	09
3081	188	\N	Mercus-Garrabet	09
3082	189	\N	Mérens-les-Vals	09
3083	190	\N	Mérigon	09
3084	192	\N	Miglos	09
3085	193	\N	Mijanès	09
3086	194	\N	Mirepoix	09
3087	195	\N	Monesple	09
3088	196	\N	Montagagne	09
3089	197	\N	Montaillou	09
3090	198	\N	Montardit	09
3091	199	\N	Montaut	09
3092	200	\N	Montbel	09
3093	201	\N	Montégut-en-Couserans	09
3094	202	\N	Montégut-Plantaurel	09
3095	203	\N	Montels	09
3096	204	\N	Montesquieu-Avantès	09
3097	205	\N	Montfa	09
3098	206	\N	Montferrier	09
3099	207	\N	Montgaillard	09
3100	208	\N	Montgauch	09
3101	209	\N	Montjoie-en-Couserans	09
3102	210	\N	Montoulieu	09
3103	211	\N	Montségur	09
3104	212	\N	Montseron	09
3105	213	\N	Moulin-Neuf	09
3106	214	\N	Moulis	09
3107	215	\N	Nalzen	09
3108	216	\N	Nescus	09
3109	217	\N	Niaux	09
3110	218	\N	Orgeix	09
3111	219	\N	Orgibet	09
3112	220	\N	Orlu	09
3113	221	\N	Ornolac-Ussat-les-Bains	09
3114	222	\N	Orus	09
3115	223	\N	Oust	09
3116	224	\N	Pailhès	09
3117	225	\N	Pamiers	09
3118	226	\N	Pech	09
3119	227	\N	Péreille	09
3120	228	\N	Perles-et-Castelet	09
3121	229	\N	Peyrat	09
3122	230	\N	Pla	09
3123	231	\N	Port	09
3124	232	\N	Prades	09
3125	233	\N	Pradettes	09
3126	234	\N	Pradières	09
3127	235	\N	Prat-Bonrepaux	09
3128	236	\N	Prayols	09
3129	237	\N	Puch	09
3130	238	\N	Pujols	09
3131	239	\N	Quérigut	09
3132	240	\N	Quié	09
3133	241	\N	Rabat-les-Trois-Seigneurs	09
3134	242	\N	Raissac	09
3135	243	\N	Régat	09
3136	244	\N	Rieucros	09
3137	245	\N	Rieux-de-Pelleport	09
3138	246	\N	Rimont	09
3139	247	\N	Rivèrenert	09
3140	249	\N	Roquefixade	09
3141	250	\N	Roquefort-les-Cascades	09
3142	251	\N	Roumengoux	09
3143	252	\N	Rouze	09
3144	253	\N	Sabarat	09
3145	254	\N	Saint-Amadou	09
3146	255	\N	Saint-Amans	09
3147	256	\N	Saint-Bauzeil	09
3148	257	\N	Sainte-Croix-Volvestre	09
3149	258	\N	Saint-Félix-de-Rieutord	09
3150	259	\N	Saint-Félix-de-Tournegat	09
3151	260	\N	Sainte-Foi	09
3152	261	\N	Saint-Girons	09
3153	262	\N	Saint-Jean-d'Aigues-Vives	09
3154	263	\N	Saint-Jean-du-Castillonnais	09
3155	264	\N	Saint-Jean-de-Verges	09
3156	265	\N	Saint-Jean-du-Falga	09
3157	266	\N	Saint-Julien-de-Gras-Capou	09
3158	267	\N	Saint-Lary	09
3159	268	\N	Saint-Lizier	09
3160	269	\N	Saint-Martin-de-Caralp	09
3161	270	\N	Saint-Martin-d'Oydes	09
3162	271	\N	Saint-Michel	09
3163	272	\N	Saint-Paul-de-Jarrat	09
3164	273	\N	Saint-Pierre-de-Rivière	09
3165	274	\N	Saint-Quentin-la-Tour	09
3166	275	\N	Saint-Quirc	09
3167	276	\N	Saint-Victor-Rouzaud	09
3168	277	\N	Saint-Ybars	09
3169	279	\N	Salsein	09
3170	280	\N	Saurat	09
3171	281	\N	Sautel	09
3172	282	\N	Saverdun	09
3173	283	\N	Savignac-les-Ormeaux	09
3174	284	\N	Ségura	09
3175	285	\N	Seix	09
3176	286	\N	Sem	09
3177	287	\N	Senconac	09
3178	289	\N	Lorp-Sentaraille	09
3179	290	\N	Sentein	09
3180	291	\N	Sentenac-d'Oust	09
3181	292	\N	Sentenac-de-Sérou	09
3182	293	\N	Serres-sur-Arget	09
3183	294	\N	Sieuras	09
3184	295	\N	Siguer	09
3185	296	\N	Sinsat	09
3186	297	\N	Sor	09
3187	298	\N	Sorgeat	09
3188	299	\N	Soueix-Rogalle	09
3189	300	\N	Soula	09
3190	301	\N	Soulan	09
3191	302	\N	Suc-et-Sentenac	09
3192	303	\N	Surba	09
3193	304	\N	Suzan	09
3194	305	\N	Tabre	09
3195	306	\N	Tarascon-sur-Ariège	09
3196	307	\N	Taurignan-Castet	09
3197	308	\N	Taurignan-Vieux	09
3198	309	\N	Teilhet	09
3199	310	\N	Thouars-sur-Arize	09
3200	311	\N	Tignac	09
3201	312	\N	Tour-du-Crieu	09
3202	313	\N	Tourtouse	09
3203	314	\N	Tourtrol	09
3204	315	\N	Trémoulet	09
3205	316	\N	Troye-d'Ariège	09
3206	317	\N	Uchentein	09
3207	318	\N	Unac	09
3208	319	\N	Unzent	09
3209	320	\N	Urs	09
3210	321	\N	Ussat	09
3211	322	\N	Ustou	09
3212	323	\N	Vals	09
3213	324	\N	Varilhes	09
3214	325	\N	Vaychis	09
3215	326	\N	Vèbre	09
3216	327	\N	Ventenac	09
3217	328	\N	Verdun	09
3218	329	\N	Vernajoul	09
3219	330	\N	Vernaux	09
3220	331	\N	Vernet	09
3221	332	\N	Verniolle	09
3222	334	\N	Vicdessos	09
3223	335	\N	Villeneuve	09
3224	336	\N	Villeneuve-d'Olmes	09
3225	338	\N	Villeneuve-du-Latou	09
3226	339	\N	Villeneuve-du-Paréage	09
3227	340	\N	Vira	09
3228	341	\N	Viviès	09
3229	342	\N	Sainte-Suzanne	09
3230	002	\N	Ailleville	10
3231	003	\N	Aix-en-Othe	10
3232	004	\N	Allibaudières	10
3233	005	\N	Amance	10
3234	006	\N	Arcis-sur-Aube	10
3235	007	\N	Arconville	10
3236	008	\N	Argançon	10
3237	009	\N	Arrelles	10
3238	010	\N	Arrembécourt	10
3239	011	\N	Arrentières	10
3240	012	\N	Arsonval	10
3241	013	\N	Assenay	10
3242	014	\N	Assencières	10
3243	015	\N	Aubeterre	10
3244	017	\N	Aulnay	10
3245	018	\N	Auxon	10
3246	019	\N	Val-d'Auzon	10
3247	020	\N	Avant-lès-Marcilly	10
3248	021	\N	Avant-lès-Ramerupt	10
3249	022	\N	Avirey-Lingey	10
3250	023	\N	Avon-la-Pèze	10
3251	024	\N	Avreuil	10
3252	025	\N	Bagneux-la-Fosse	10
3253	026	\N	Bailly-le-Franc	10
3254	027	\N	Balignicourt	10
3255	028	\N	Balnot-la-Grange	10
3256	029	\N	Balnot-sur-Laignes	10
3257	030	\N	Barberey-Saint-Sulpice	10
3258	031	\N	Barbuise	10
3259	032	\N	Baroville	10
3260	033	\N	Bar-sur-Aube	10
3261	034	\N	Bar-sur-Seine	10
3262	035	\N	Bayel	10
3263	037	\N	Bercenay-en-Othe	10
3264	038	\N	Bercenay-le-Hayer	10
3265	039	\N	Bergères	10
3266	040	\N	Bernon	10
3267	041	\N	Bertignolles	10
3268	042	\N	Bérulle	10
3269	043	\N	Bessy	10
3270	044	\N	Bétignicourt	10
3271	045	\N	Beurey	10
3272	046	\N	Blaincourt-sur-Aube	10
3273	047	\N	Blignicourt	10
3274	048	\N	Bligny	10
3275	049	\N	Bordes-Aumont	10
3276	050	\N	Bossancourt	10
3277	051	\N	Bouilly	10
3278	052	\N	Boulages	10
3279	053	\N	Bouranton	10
3280	054	\N	Bourdenay	10
3281	055	\N	Bourguignons	10
3282	056	\N	Bouy-Luxembourg	10
3283	057	\N	Bouy-sur-Orvin	10
3284	058	\N	Bragelogne-Beauvoir	10
3285	059	\N	Braux	10
3286	060	\N	Bréviandes	10
3287	061	\N	Brévonnes	10
3288	062	\N	Briel-sur-Barse	10
3289	063	\N	Brienne-la-Vieille	10
3290	064	\N	Brienne-le-Château	10
3291	065	\N	Brillecourt	10
3292	066	\N	Bucey-en-Othe	10
3293	067	\N	Buchères	10
3294	068	\N	Buxeuil	10
3295	069	\N	Buxières-sur-Arce	10
3296	070	\N	Celles-sur-Ource	10
3297	071	\N	Chacenay	10
3298	072	\N	Chaise	10
3299	073	\N	Chalette-sur-Voire	10
3300	074	\N	Chamoy	10
3301	075	\N	Champfleury	10
3302	076	\N	Champignol-lez-Mondeville	10
3303	077	\N	Champigny-sur-Aube	10
3304	078	\N	Champ-sur-Barse	10
3305	079	\N	Channes	10
3306	080	\N	Chaource	10
3307	081	\N	Chapelle-Saint-Luc	10
3308	082	\N	Chapelle-Vallon	10
3309	083	\N	Chappes	10
3310	084	\N	Charmont-sous-Barbuise	10
3311	085	\N	Charmoy	10
3312	086	\N	Charny-le-Bachot	10
3313	087	\N	Chaserey	10
3314	089	\N	Châtres	10
3315	090	\N	Chauchigny	10
3316	091	\N	Chaudrey	10
3317	092	\N	Chauffour-lès-Bailly	10
3318	093	\N	Chaumesnil	10
3319	094	\N	Chavanges	10
3320	095	\N	Chêne	10
3321	096	\N	Chennegy	10
3322	097	\N	Chervey	10
3323	098	\N	Chesley	10
3324	099	\N	Chessy-les-Prés	10
3325	100	\N	Clérey	10
3326	101	\N	Coclois	10
3327	102	\N	Colombé-la-Fosse	10
3328	103	\N	Colombé-le-Sec	10
3329	104	\N	Cormost	10
3330	105	\N	Courcelles-sur-Voire	10
3331	106	\N	Courceroy	10
3332	107	\N	Coursan-en-Othe	10
3333	108	\N	Courtaoult	10
3334	109	\N	Courtenot	10
3335	110	\N	Courteranges	10
3336	111	\N	Courteron	10
3337	112	\N	Coussegrey	10
3338	113	\N	Couvignon	10
3339	114	\N	Crancey	10
3340	115	\N	Creney-près-Troyes	10
3341	116	\N	Crésantignes	10
3342	117	\N	Crespy-le-Neuf	10
3343	118	\N	Croûtes	10
3344	119	\N	Cunfin	10
3345	120	\N	Cussangy	10
3346	121	\N	Dampierre	10
3347	122	\N	Davrey	10
3348	123	\N	Dienville	10
3349	124	\N	Dierrey-Saint-Julien	10
3350	125	\N	Dierrey-Saint-Pierre	10
3351	126	\N	Dolancourt	10
3352	127	\N	Dommartin-le-Coq	10
3353	128	\N	Donnement	10
3354	129	\N	Dosches	10
3355	130	\N	Dosnon	10
3356	131	\N	Droupt-Saint-Basle	10
3357	132	\N	Droupt-Sainte-Marie	10
3358	133	\N	Eaux-Puiseaux	10
3359	134	\N	Échemines	10
3360	135	\N	Éclance	10
3361	136	\N	Éguilly-sous-Bois	10
3362	137	\N	Engente	10
3363	138	\N	Épagne	10
3364	139	\N	Épothémont	10
3365	140	\N	Ervy-le-Châtel	10
3366	141	\N	Essoyes	10
3367	142	\N	Estissac	10
3368	143	\N	Étourvy	10
3369	144	\N	Étrelles-sur-Aube	10
3370	145	\N	Faux-Villecerf	10
3371	146	\N	Fay-lès-Marcilly	10
3372	147	\N	Fays-la-Chapelle	10
3373	148	\N	Ferreux-Quincey	10
3374	149	\N	Feuges	10
3375	150	\N	Fontaine	10
3376	151	\N	Fontaine-les-Grès	10
3377	153	\N	Fontaine-Mâcon	10
3378	154	\N	Fontenay-de-Bossery	10
3379	155	\N	Fontette	10
3380	156	\N	Fontvannes	10
3381	157	\N	Fosse-Corduan	10
3382	158	\N	Fouchères	10
3383	159	\N	Fralignes	10
3384	160	\N	Fravaux	10
3385	161	\N	Fresnay	10
3386	162	\N	Fresnoy-le-Château	10
3387	163	\N	Fuligny	10
3388	164	\N	Gélannes	10
3389	165	\N	Géraudot	10
3390	166	\N	Grandes-Chapelles	10
3391	167	\N	Grandville	10
3392	168	\N	Granges	10
3393	169	\N	Gumery	10
3394	170	\N	Gyé-sur-Seine	10
3395	171	\N	Hampigny	10
3396	172	\N	Herbisse	10
3397	173	\N	Isle-Aumont	10
3398	174	\N	Isle-Aubigny	10
3399	175	\N	Jasseines	10
3400	176	\N	Jaucourt	10
3401	177	\N	Javernant	10
3402	178	\N	Jessains	10
3403	179	\N	Jeugny	10
3404	180	\N	Joncreuil	10
3405	181	\N	Jully-sur-Sarce	10
3406	182	\N	Juvancourt	10
3407	183	\N	Juvanzé	10
3408	184	\N	Juzanvigny	10
3409	185	\N	Lagesse	10
3410	186	\N	Laines-aux-Bois	10
3411	187	\N	Landreville	10
3412	188	\N	Lantages	10
3413	189	\N	Lassicourt	10
3414	190	\N	Laubressel	10
3415	191	\N	Lavau	10
3416	192	\N	Lentilles	10
3417	193	\N	Lesmont	10
3418	194	\N	Lévigny	10
3419	195	\N	Lhuître	10
3420	196	\N	Lignières	10
3421	197	\N	Lignol-le-Château	10
3422	198	\N	Lirey	10
3423	199	\N	Loches-sur-Ource	10
3424	200	\N	Loge-aux-Chèvres	10
3425	201	\N	Loge-Pomblin	10
3426	202	\N	Loges-Margueron	10
3427	203	\N	Longchamp-sur-Aujon	10
3428	204	\N	Longeville-sur-Mogne	10
3429	205	\N	Longpré-le-Sec	10
3430	206	\N	Longsols	10
3431	207	\N	Longueville-sur-Aube	10
3432	208	\N	Louptière-Thénard	10
3433	209	\N	Lusigny-sur-Barse	10
3434	210	\N	Luyères	10
3435	211	\N	Macey	10
3436	212	\N	Machy	10
3437	213	\N	Magnant	10
3438	214	\N	Magnicourt	10
3439	215	\N	Magny-Fouchard	10
3440	216	\N	Mailly-le-Camp	10
3441	217	\N	Maison-des-Champs	10
3442	218	\N	Maisons-lès-Chaource	10
3443	219	\N	Maisons-lès-Soulaines	10
3444	220	\N	Maizières-la-Grande-Paroisse	10
3445	221	\N	Maizières-lès-Brienne	10
3446	222	\N	Maraye-en-Othe	10
3447	223	\N	Marcilly-le-Hayer	10
3448	224	\N	Marigny-le-Châtel	10
3449	225	\N	Marnay-sur-Seine	10
3450	226	\N	Marolles-lès-Bailly	10
3451	227	\N	Marolles-sous-Lignières	10
3452	228	\N	Mathaux	10
3453	229	\N	Maupas	10
3454	230	\N	Mergey	10
3455	231	\N	Mériot	10
3456	232	\N	Merrey-sur-Arce	10
3457	233	\N	Méry-sur-Seine	10
3458	234	\N	Mesgrigny	10
3459	235	\N	Mesnil-la-Comtesse	10
3460	236	\N	Mesnil-Lettre	10
3461	237	\N	Mesnil-Saint-Loup	10
3462	238	\N	Mesnil-Saint-Père	10
3463	239	\N	Mesnil-Sellières	10
3464	240	\N	Messon	10
3465	241	\N	Metz-Robert	10
3466	242	\N	Meurville	10
3467	243	\N	Molins-sur-Aube	10
3468	245	\N	Montaulin	10
3469	246	\N	Montceaux-lès-Vaudes	10
3470	247	\N	Montfey	10
3471	248	\N	Montgueux	10
3472	249	\N	Montiéramey	10
3473	250	\N	Montier-en-l'Isle	10
3474	251	\N	Montigny-les-Monts	10
3475	252	\N	Montmartin-le-Haut	10
3476	253	\N	Montmorency-Beaufort	10
3477	254	\N	Montpothier	10
3478	255	\N	Montreuil-sur-Barse	10
3479	256	\N	Montsuzain	10
3480	257	\N	Morembert	10
3481	258	\N	Morvilliers	10
3482	259	\N	Motte-Tilly	10
3483	260	\N	Moussey	10
3484	261	\N	Mussy-sur-Seine	10
3485	262	\N	Neuville-sur-Seine	10
3486	263	\N	Neuville-sur-Vannes	10
3487	264	\N	Noë-les-Mallets	10
3488	265	\N	Noës-près-Troyes	10
3489	266	\N	Nogent-en-Othe	10
3490	267	\N	Nogent-sur-Aube	10
3491	268	\N	Nogent-sur-Seine	10
3492	269	\N	Nozay	10
3493	270	\N	Onjon	10
3494	271	\N	Origny-le-Sec	10
3495	272	\N	Ormes	10
3496	273	\N	Ortillon	10
3497	274	\N	Orvilliers-Saint-Julien	10
3498	275	\N	Ossey-les-Trois-Maisons	10
3499	276	\N	Paisy-Cosdon	10
3500	277	\N	Palis	10
3501	278	\N	Pargues	10
3502	279	\N	Pars-lès-Chavanges	10
3503	280	\N	Pars-lès-Romilly	10
3504	281	\N	Pavillon-Sainte-Julie	10
3505	282	\N	Payns	10
3506	283	\N	Pel-et-Der	10
3507	284	\N	Périgny-la-Rose	10
3508	285	\N	Perthes-lès-Brienne	10
3509	286	\N	Petit-Mesnil	10
3510	287	\N	Piney	10
3511	288	\N	Plaines-Saint-Lange	10
3512	289	\N	Plancy-l'Abbaye	10
3513	290	\N	Planty	10
3514	291	\N	Plessis-Barbuise	10
3515	293	\N	Poivres	10
3516	294	\N	Poligny	10
3517	295	\N	Polisot	10
3518	296	\N	Polisy	10
3519	297	\N	Pont-Sainte-Marie	10
3520	298	\N	Pont-sur-Seine	10
3521	299	\N	Pouan-les-Vallées	10
3522	300	\N	Pougy	10
3523	301	\N	Pouy-sur-Vannes	10
3524	302	\N	Praslin	10
3525	303	\N	Précy-Notre-Dame	10
3526	304	\N	Précy-Saint-Martin	10
3527	305	\N	Prémierfait	10
3528	306	\N	Proverville	10
3529	307	\N	Prugny	10
3530	308	\N	Prunay-Belleville	10
3531	309	\N	Prusy	10
3532	310	\N	Puits-et-Nuisement	10
3533	312	\N	Racines	10
3534	313	\N	Radonvilliers	10
3535	314	\N	Ramerupt	10
3536	315	\N	Rances	10
3537	316	\N	Rhèges	10
3538	317	\N	Riceys	10
3539	318	\N	Rigny-la-Nonneuse	10
3540	319	\N	Rigny-le-Ferron	10
3541	320	\N	Rilly-Sainte-Syre	10
3542	321	\N	Rivière-de-Corps	10
3543	323	\N	Romilly-sur-Seine	10
3544	324	\N	Roncenay	10
3545	325	\N	Rosières-près-Troyes	10
3546	326	\N	Rosnay-l'Hôpital	10
3547	327	\N	Rothière	10
3548	328	\N	Rouilly-Sacey	10
3549	329	\N	Rouilly-Saint-Loup	10
3550	330	\N	Rouvres-les-Vignes	10
3551	331	\N	Rumilly-lès-Vaudes	10
3552	332	\N	Ruvigny	10
3553	333	\N	Saint-André-les-Vergers	10
3554	334	\N	Saint-Aubin	10
3555	335	\N	Saint-Benoist-sur-Vanne	10
3556	336	\N	Saint-Benoît-sur-Seine	10
3557	337	\N	Saint-Christophe-Dodinicourt	10
3558	338	\N	Saint-Étienne-sous-Barbuise	10
3559	339	\N	Saint-Flavy	10
3560	340	\N	Saint-Germain	10
3561	341	\N	Saint-Hilaire-sous-Romilly	10
3562	342	\N	Saint-Jean-de-Bonneval	10
3563	343	\N	Saint-Julien-les-Villas	10
3564	344	\N	Saint-Léger-près-Troyes	10
3565	345	\N	Saint-Léger-sous-Brienne	10
3566	346	\N	Saint-Léger-sous-Margerie	10
3567	347	\N	Saint-Loup-de-Buffigny	10
3568	348	\N	Saint-Lupien	10
3569	349	\N	Saint-Lyé	10
3570	350	\N	Saint-Mards-en-Othe	10
3571	351	\N	Saint-Martin-de-Bossenay	10
3572	352	\N	Sainte-Maure	10
3573	353	\N	Saint-Mesmin	10
3574	354	\N	Saint-Nabord-sur-Aube	10
3575	355	\N	Saint-Nicolas-la-Chapelle	10
3576	356	\N	Saint-Oulph	10
3577	357	\N	Saint-Parres-aux-Tertres	10
3578	358	\N	Saint-Parres-lès-Vaudes	10
3579	359	\N	Saint-Phal	10
3580	360	\N	Saint-Pouange	10
3581	361	\N	Saint-Remy-sous-Barbuise	10
3582	362	\N	Sainte-Savine	10
3583	363	\N	Saint-Thibault	10
3584	364	\N	Saint-Usage	10
3585	365	\N	Salon	10
3586	366	\N	Saulcy	10
3587	367	\N	Saulsotte	10
3588	368	\N	Savières	10
3589	369	\N	Semoine	10
3590	370	\N	Soligny-les-Étangs	10
3591	371	\N	Sommeval	10
3592	372	\N	Soulaines-Dhuys	10
3593	373	\N	Souligny	10
3594	374	\N	Spoy	10
3595	375	\N	Thennelières	10
3596	376	\N	Thieffrain	10
3597	377	\N	Thil	10
3598	378	\N	Thors	10
3599	379	\N	Torcy-le-Grand	10
3600	380	\N	Torcy-le-Petit	10
3601	381	\N	Torvilliers	10
3602	382	\N	Traînel	10
3603	383	\N	Trancault	10
3604	384	\N	Trannes	10
3605	386	\N	Trouans	10
3606	387	\N	Troyes	10
3607	388	\N	Turgy	10
3608	389	\N	Unienville	10
3609	390	\N	Urville	10
3610	391	\N	Vailly	10
3611	392	\N	Vallant-Saint-Georges	10
3612	393	\N	Vallentigny	10
3613	394	\N	Vallières	10
3614	395	\N	Vanlay	10
3615	396	\N	Vauchassis	10
3616	397	\N	Vauchonvilliers	10
3617	398	\N	Vaucogne	10
3618	399	\N	Vaudes	10
3619	400	\N	Vaupoisson	10
3620	401	\N	Vendeuvre-sur-Barse	10
3621	402	\N	Vendue-Mignot	10
3622	403	\N	Vernonvilliers	10
3623	404	\N	Verpillières-sur-Ource	10
3624	405	\N	Verricourt	10
3625	406	\N	Verrières	10
3626	408	\N	Viâpres-le-Petit	10
3627	409	\N	Villacerf	10
3628	410	\N	Villadin	10
3629	411	\N	Ville-aux-Bois	10
3630	412	\N	Villechétif	10
3631	414	\N	Villeloup	10
3632	415	\N	Villemaur-sur-Vanne	10
3633	416	\N	Villemereuil	10
3634	417	\N	Villemoiron-en-Othe	10
3635	418	\N	Villemorien	10
3636	419	\N	Villemoyenne	10
3637	420	\N	Villenauxe-la-Grande	10
3638	421	\N	Villeneuve-au-Châtelot	10
3639	422	\N	Villeneuve-au-Chemin	10
3640	423	\N	Villeneuve-au-Chêne	10
3641	424	\N	Villeret	10
3642	425	\N	Villery	10
3643	426	\N	Ville-sous-la-Ferté	10
3644	427	\N	Ville-sur-Arce	10
3645	428	\N	Ville-sur-Terre	10
3646	429	\N	Villette-sur-Aube	10
3647	430	\N	Villiers-Herbisse	10
3648	431	\N	Villiers-le-Bois	10
3649	432	\N	Villiers-sous-Praslin	10
3650	433	\N	Villy-en-Trodes	10
3651	434	\N	Villy-le-Bois	10
3652	435	\N	Villy-le-Maréchal	10
3653	436	\N	Vinets	10
3654	437	\N	Virey-sous-Bar	10
3655	438	\N	Vitry-le-Croisé	10
3656	439	\N	Viviers-sur-Artaut	10
3657	440	\N	Voigny	10
3658	441	\N	Vosnon	10
3659	442	\N	Voué	10
3660	443	\N	Vougrey	10
3661	444	\N	Vulaines	10
3662	445	\N	Yèvres-le-Petit	10
3663	001	\N	Aigues-Vives	11
3664	002	\N	Airoux	11
3665	003	\N	Ajac	11
3666	004	\N	Alaigne	11
3667	005	\N	Alairac	11
3668	006	\N	Albas	11
3669	007	\N	Albières	11
3670	008	\N	Alet-les-Bains	11
3671	009	\N	Alzonne	11
3672	010	\N	Antugnac	11
3673	011	\N	Aragon	11
3674	012	\N	Argeliers	11
3675	013	\N	Argens-Minervois	11
3676	014	\N	Armissan	11
3677	015	\N	Arques	11
3678	016	\N	Arquettes-en-Val	11
3679	017	\N	Artigues	11
3680	018	\N	Arzens	11
3681	019	\N	Aunat	11
3682	020	\N	Auriac	11
3683	021	\N	Axat	11
3684	022	\N	Azille	11
3685	023	\N	Badens	11
3686	024	\N	Bages	11
3687	025	\N	Bagnoles	11
3688	026	\N	Baraigne	11
3689	027	\N	Barbaira	11
3690	028	\N	Belcaire	11
3691	029	\N	Belcastel-et-Buc	11
3692	030	\N	Belflou	11
3693	031	\N	Belfort-sur-Rebenty	11
3694	032	\N	Bellegarde-du-Razès	11
3695	033	\N	Belpech	11
3696	034	\N	Belvèze-du-Razès	11
3697	035	\N	Belvianes-et-Cavirac	11
3698	036	\N	Belvis	11
3699	037	\N	Berriac	11
3700	038	\N	Bessède-de-Sault	11
3701	039	\N	Bezole	11
3702	040	\N	Bizanet	11
3703	041	\N	Bize-Minervois	11
3704	042	\N	Blomac	11
3705	043	\N	Bouilhonnac	11
3706	044	\N	Bouisse	11
3707	045	\N	Bouriège	11
3708	046	\N	Bourigeole	11
3709	047	\N	Bousquet	11
3710	048	\N	Boutenac	11
3711	049	\N	Bram	11
3712	050	\N	Brenac	11
3713	051	\N	Brézilhac	11
3714	052	\N	Brousses-et-Villaret	11
3715	053	\N	Brugairolles	11
3716	054	\N	Brunels	11
3717	055	\N	Bugarach	11
3718	056	\N	Cabrespine	11
3719	057	\N	Cahuzac	11
3720	058	\N	Cailhau	11
3721	059	\N	Cailhavel	11
3722	060	\N	Cailla	11
3723	061	\N	Cambieure	11
3724	062	\N	Campagna-de-Sault	11
3725	063	\N	Campagne-sur-Aude	11
3726	064	\N	Camplong-d'Aude	11
3727	065	\N	Camps-sur-l'Agly	11
3728	066	\N	Camurac	11
3729	067	\N	Canet	11
3730	068	\N	Capendu	11
3731	069	\N	Carcassonne	11
3732	070	\N	Carlipa	11
3733	071	\N	Cascastel-des-Corbières	11
3734	072	\N	Cassaigne	11
3735	073	\N	Cassaignes	11
3736	074	\N	Cassés	11
3737	075	\N	Castans	11
3738	076	\N	Castelnaudary	11
3739	077	\N	Castelnau-d'Aude	11
3740	078	\N	Castelreng	11
3741	079	\N	Caudebronde	11
3742	080	\N	Caudeval	11
3743	081	\N	Caunes-Minervois	11
3744	082	\N	Caunette-sur-Lauquet	11
3745	083	\N	Caunettes-en-Val	11
3746	084	\N	Caux-et-Sauzens	11
3747	085	\N	Cavanac	11
3748	086	\N	Caves	11
3749	087	\N	Cazalrenoux	11
3750	088	\N	Cazilhac	11
3751	089	\N	Cenne-Monestiés	11
3752	090	\N	Cépie	11
3753	091	\N	Chalabre	11
3754	092	\N	Citou	11
3755	093	\N	Clat	11
3756	094	\N	Clermont-sur-Lauquet	11
3757	095	\N	Comigne	11
3758	096	\N	Comus	11
3759	097	\N	Conilhac-de-la-Montagne	11
3760	098	\N	Conilhac-Corbières	11
3761	099	\N	Conques-sur-Orbiel	11
3762	100	\N	Corbières	11
3763	101	\N	Coudons	11
3764	102	\N	Couffoulens	11
3765	103	\N	Couiza	11
3766	104	\N	Counozouls	11
3767	105	\N	Cournanel	11
3768	106	\N	Coursan	11
3769	107	\N	Courtauly	11
3770	108	\N	Courtète	11
3771	109	\N	Coustaussa	11
3772	110	\N	Coustouge	11
3773	111	\N	Cruscades	11
3774	112	\N	Cubières-sur-Cinoble	11
3775	113	\N	Cucugnan	11
3776	114	\N	Cumiès	11
3777	115	\N	Cuxac-Cabardès	11
3778	116	\N	Cuxac-d'Aude	11
3779	117	\N	Davejean	11
3780	118	\N	Dernacueillette	11
3781	119	\N	Digne-d'Amont	11
3782	120	\N	Digne-d'Aval	11
3783	121	\N	Donazac	11
3784	122	\N	Douzens	11
3785	123	\N	Duilhac-sous-Peyrepertuse	11
3786	124	\N	Durban-Corbières	11
3787	125	\N	Embres-et-Castelmaure	11
3788	126	\N	Escales	11
3789	127	\N	Escouloubre	11
3790	128	\N	Escueillens-et-Saint-Just-de-Bélengard	11
3791	129	\N	Espéraza	11
3792	130	\N	Espezel	11
3793	131	\N	Fa	11
3794	132	\N	Fabrezan	11
3795	133	\N	Fajac-en-Val	11
3796	134	\N	Fajac-la-Relenque	11
3797	135	\N	Fajolle	11
3798	136	\N	Fanjeaux	11
3799	137	\N	Félines-Termenès	11
3800	138	\N	Fendeille	11
3801	139	\N	Fenouillet-du-Razès	11
3802	140	\N	Ferrals-les-Corbières	11
3803	141	\N	Ferran	11
3804	142	\N	Festes-et-Saint-André	11
3805	143	\N	Feuilla	11
3806	144	\N	Fitou	11
3807	145	\N	Fleury	11
3808	146	\N	Floure	11
3809	147	\N	Fontanès-de-Sault	11
3810	148	\N	Fontcouverte	11
3811	149	\N	Fonters-du-Razès	11
3812	150	\N	Fontiers-Cabardès	11
3813	151	\N	Fontiès-d'Aude	11
3814	152	\N	Fontjoncouse	11
3815	153	\N	Force	11
3816	154	\N	Fournes-Cabardès	11
3817	155	\N	Fourtou	11
3818	156	\N	Fraisse-Cabardès	11
3819	157	\N	Fraissé-des-Corbières	11
3820	158	\N	Gaja-et-Villedieu	11
3821	159	\N	Gaja-la-Selve	11
3822	160	\N	Galinagues	11
3823	161	\N	Gardie	11
3824	162	\N	Generville	11
3825	163	\N	Gincla	11
3826	164	\N	Ginestas	11
3827	165	\N	Ginoles	11
3828	166	\N	Gourvieille	11
3829	167	\N	Gramazie	11
3830	168	\N	Granès	11
3831	169	\N	Greffeil	11
3832	170	\N	Gruissan	11
3833	171	\N	Gueytes-et-Labastide	11
3834	172	\N	Homps	11
3835	173	\N	Hounoux	11
3836	174	\N	Ilhes	11
3837	175	\N	Issel	11
3838	176	\N	Jonquières	11
3839	177	\N	Joucou	11
3840	178	\N	Labastide-d'Anjou	11
3841	179	\N	Labastide-en-Val	11
3842	180	\N	Labastide-Esparbairenque	11
3843	181	\N	Labécède-Lauragais	11
3844	182	\N	Lacombe	11
3845	183	\N	Ladern-sur-Lauquet	11
3846	184	\N	Lafage	11
3847	185	\N	Lagrasse	11
3848	186	\N	Lairière	11
3849	187	\N	Lanet	11
3850	188	\N	Palme	11
3851	189	\N	Laprade	11
3852	190	\N	Redorte	11
3853	191	\N	Laroque-de-Fa	11
3854	192	\N	Lasbordes	11
3855	193	\N	Lasserre-de-Prouille	11
3856	194	\N	Lastours	11
3857	195	\N	Laurabuc	11
3858	196	\N	Laurac	11
3859	197	\N	Lauraguel	11
3860	198	\N	Laure-Minervois	11
3861	199	\N	Lavalette	11
3862	200	\N	Lespinassière	11
3863	201	\N	Leuc	11
3864	202	\N	Leucate	11
3865	203	\N	Lézignan-Corbières	11
3866	204	\N	Lignairolles	11
3867	205	\N	Limousis	11
3868	206	\N	Limoux	11
3869	207	\N	Loupia	11
3870	208	\N	Louvière-Lauragais	11
3871	209	\N	Luc-sur-Aude	11
3872	210	\N	Luc-sur-Orbieu	11
3873	211	\N	Magrie	11
3874	212	\N	Mailhac	11
3875	213	\N	Maisons	11
3876	214	\N	Malras	11
3877	215	\N	Malves-en-Minervois	11
3878	216	\N	Malviès	11
3879	217	\N	Marcorignan	11
3880	218	\N	Marquein	11
3881	219	\N	Marsa	11
3882	220	\N	Marseillette	11
3883	221	\N	Martys	11
3884	222	\N	Mas-Cabardès	11
3885	223	\N	Mas-des-Cours	11
3886	224	\N	Massac	11
3887	225	\N	Mas-Saintes-Puelles	11
3888	226	\N	Mayreville	11
3889	227	\N	Mayronnes	11
3890	228	\N	Mazerolles-du-Razès	11
3891	229	\N	Mazuby	11
3892	230	\N	Mérial	11
3893	231	\N	Mézerville	11
3894	232	\N	Miraval-Cabardes	11
3895	233	\N	Mirepeisset	11
3896	234	\N	Mireval-Lauragais	11
3897	235	\N	Missègre	11
3898	236	\N	Molandier	11
3899	238	\N	Molleville	11
3900	239	\N	Montauriol	11
3901	240	\N	Montazels	11
3902	241	\N	Montbrun-des-Corbières	11
3903	242	\N	Montclar	11
3904	243	\N	Montferrand	11
3905	244	\N	Montfort-sur-Boulzane	11
3906	245	\N	Montgaillard	11
3907	246	\N	Montgradail	11
3908	247	\N	Monthaut	11
3909	248	\N	Montirat	11
3910	249	\N	Montjardin	11
3911	250	\N	Montjoi	11
3912	251	\N	Montlaur	11
3913	252	\N	Montmaur	11
3914	253	\N	Montolieu	11
3915	254	\N	Montréal	11
3916	255	\N	Montredon-des-Corbières	11
3917	256	\N	Montséret	11
3918	257	\N	Monze	11
3919	258	\N	Moussan	11
3920	259	\N	Moussoulens	11
3921	260	\N	Mouthoumet	11
3922	261	\N	Moux	11
3923	262	\N	Narbonne	11
3924	263	\N	Nébias	11
3925	264	\N	Névian	11
3926	265	\N	Niort-de-Sault	11
3927	266	\N	Port-la-Nouvelle	11
3928	267	\N	Ornaisons	11
3929	268	\N	Orsans	11
3930	269	\N	Ouveillan	11
3931	270	\N	Padern	11
3932	271	\N	Palairac	11
3933	272	\N	Palaja	11
3934	273	\N	Paraza	11
3935	274	\N	Pauligne	11
3936	275	\N	Payra-sur-l'Hers	11
3937	276	\N	Paziols	11
3938	277	\N	Pécharic-et-le-Py	11
3939	278	\N	Pech-Luna	11
3940	279	\N	Pennautier	11
3941	280	\N	Pépieux	11
3942	281	\N	Pexiora	11
3943	282	\N	Peyrefitte-du-Razès	11
3944	283	\N	Peyrefitte-sur-l'Hers	11
3945	284	\N	Peyrens	11
3946	285	\N	Peyriac-de-Mer	11
3947	286	\N	Peyriac-Minervois	11
3948	287	\N	Peyrolles	11
3949	288	\N	Pezens	11
3950	289	\N	Pieusse	11
3951	290	\N	Plaigne	11
3952	291	\N	Plavilla	11
3953	292	\N	Pomarède	11
3954	293	\N	Pomas	11
3955	294	\N	Pomy	11
3956	295	\N	Portel-des-Corbières	11
3957	296	\N	Pouzols-Minervois	11
3958	297	\N	Pradelles-Cabardès	11
3959	298	\N	Pradelles-en-Val	11
3960	299	\N	Preixan	11
3961	300	\N	Puginier	11
3962	301	\N	Puichéric	11
3963	302	\N	Puilaurens	11
3964	303	\N	Puivert	11
3965	304	\N	Quillan	11
3966	305	\N	Quintillan	11
3967	306	\N	Quirbajou	11
3968	307	\N	Raissac-d'Aude	11
3969	308	\N	Raissac-sur-Lampy	11
3970	309	\N	Rennes-le-Château	11
3971	310	\N	Rennes-les-Bains	11
3972	311	\N	Ribaute	11
3973	312	\N	Ribouisse	11
3974	313	\N	Ricaud	11
3975	314	\N	Rieux-en-Val	11
3976	315	\N	Rieux-Minervois	11
3977	316	\N	Rivel	11
3978	317	\N	Rodome	11
3979	318	\N	Roquecourbe-Minervois	11
3980	319	\N	Roquefère	11
3981	320	\N	Roquefeuil	11
3982	321	\N	Roquefort-de-Sault	11
3983	322	\N	Roquefort-des-Corbières	11
3984	323	\N	Roquetaillade	11
3985	324	\N	Roubia	11
3986	325	\N	Rouffiac-d'Aude	11
3987	326	\N	Rouffiac-des-Corbières	11
3988	327	\N	Roullens	11
3989	328	\N	Routier	11
3990	329	\N	Rouvenac	11
3991	330	\N	Rustiques	11
3992	331	\N	Saint-Amans	11
3993	332	\N	Saint-André-de-Roquelongue	11
3994	333	\N	Saint-Benoît	11
3995	334	\N	Sainte-Camelle	11
3996	335	\N	Sainte-Colombe-sur-Guette	11
3997	336	\N	Sainte-Colombe-sur-l'Hers	11
3998	337	\N	Saint-Couat-d'Aude	11
3999	338	\N	Saint-Couat-du-Razès	11
4000	339	\N	Saint-Denis	11
4001	340	\N	Sainte-Eulalie	11
4002	341	\N	Saint-Ferriol	11
4003	342	\N	Saint-Frichoux	11
4004	343	\N	Saint-Gaudéric	11
4005	344	\N	Saint-Hilaire	11
4006	345	\N	Saint-Jean-de-Barrou	11
4007	346	\N	Saint-Jean-de-Paracol	11
4008	347	\N	Saint-Julia-de-Bec	11
4009	348	\N	Saint-Julien-de-Briola	11
4010	350	\N	Saint-Just-et-le-Bézu	11
4011	351	\N	Saint-Laurent-de-la-Cabrerisse	11
4012	352	\N	Saint-Louis-et-Parahou	11
4013	353	\N	Saint-Marcel-sur-Aude	11
4014	354	\N	Saint-Martin-des-Puits	11
4015	355	\N	Saint-Martin-de-Villereglan	11
4016	356	\N	Saint-Martin-Lalande	11
4017	357	\N	Saint-Martin-le-Vieil	11
4018	358	\N	Saint-Martin-Lys	11
4019	359	\N	Saint-Michel-de-Lanès	11
4020	360	\N	Saint-Nazaire-d'Aude	11
4021	361	\N	Saint-Papoul	11
4022	362	\N	Saint-Paulet	11
4023	363	\N	Saint-Pierre-des-Champs	11
4024	364	\N	Saint-Polycarpe	11
4025	365	\N	Saint-Sernin	11
4026	366	\N	Sainte-Valière	11
4027	367	\N	Saissac	11
4028	368	\N	Sallèles-Cabardès	11
4029	369	\N	Sallèles-d'Aude	11
4030	370	\N	Salles-d'Aude	11
4031	371	\N	Salles-sur-l'Hers	11
4032	372	\N	Salsigne	11
4033	373	\N	Salvezines	11
4034	374	\N	Salza	11
4035	375	\N	Seignalens	11
4036	376	\N	Serpent	11
4037	377	\N	Serres	11
4038	378	\N	Serviès-en-Val	11
4039	379	\N	Sigean	11
4040	380	\N	Sonnac-sur-l'Hers	11
4041	381	\N	Sougraigne	11
4042	382	\N	Souilhanels	11
4043	383	\N	Souilhe	11
4044	384	\N	Soulatgé	11
4045	385	\N	Soupex	11
4046	386	\N	Talairan	11
4047	387	\N	Taurize	11
4048	388	\N	Termes	11
4049	389	\N	Terroles	11
4050	390	\N	Thézan-des-Corbières	11
4051	391	\N	Tourette-Cabardès	11
4052	392	\N	Tournissan	11
4053	393	\N	Tourouzelle	11
4054	394	\N	Tourreilles	11
4055	395	\N	Trassanel	11
4056	396	\N	Trausse	11
4057	397	\N	Trèbes	11
4058	398	\N	Treilles	11
4059	399	\N	Tréville	11
4060	400	\N	Tréziers	11
4061	401	\N	Tuchan	11
4062	402	\N	Valmigère	11
4063	404	\N	Ventenac-Cabardès	11
4064	405	\N	Ventenac-en-Minervois	11
4065	406	\N	Véraza	11
4066	407	\N	Verdun-en-Lauragais	11
4067	408	\N	Verzeille	11
4068	409	\N	Vignevieille	11
4069	410	\N	Villalier	11
4070	411	\N	Villanière	11
4071	412	\N	Villardebelle	11
4072	413	\N	Villardonnel	11
4073	414	\N	Villar-en-Val	11
4074	415	\N	Villar-Saint-Anselme	11
4075	416	\N	Villarzel-Cabardès	11
4076	417	\N	Villarzel-du-Razès	11
4077	418	\N	Villasavary	11
4078	419	\N	Villautou	11
4079	420	\N	Villebazy	11
4080	421	\N	Villedaigne	11
4081	422	\N	Villedubert	11
4082	423	\N	Villefloure	11
4083	424	\N	Villefort	11
4084	425	\N	Villegailhenc	11
4085	426	\N	Villegly	11
4086	427	\N	Villelongue-d'Aude	11
4087	428	\N	Villemagne	11
4088	429	\N	Villemoustaussou	11
4089	430	\N	Villeneuve-la-Comptal	11
4090	431	\N	Villeneuve-les-Corbières	11
4091	432	\N	Villeneuve-lès-Montréal	11
4092	433	\N	Villeneuve-Minervois	11
4093	434	\N	Villepinte	11
4094	435	\N	Villerouge-Termenès	11
4095	436	\N	Villesèque-des-Corbières	11
4096	437	\N	Villesèquelande	11
4097	438	\N	Villesiscle	11
4098	439	\N	Villespy	11
4099	440	\N	Villetritouls	11
4100	441	\N	Vinassan	11
4101	001	\N	Agen-d'Aveyron	12
4102	002	\N	Aguessac	12
4103	003	\N	Albres	12
4104	004	\N	Almont-les-Junies	12
4105	005	\N	Alpuech	12
4106	006	\N	Alrance	12
4107	007	\N	Ambeyrac	12
4108	008	\N	Anglars-Saint-Félix	12
4109	009	\N	Arnac-sur-Dourdou	12
4110	010	\N	Arques	12
4111	011	\N	Arvieu	12
4112	012	\N	Asprières	12
4113	013	\N	Aubin	12
4114	014	\N	Aurelle-Verlac	12
4115	015	\N	Auriac-Lagast	12
4116	016	\N	Auzits	12
4117	017	\N	Ayssènes	12
4118	018	\N	Balaguier-d'Olt	12
4119	019	\N	Balaguier-sur-Rance	12
4120	020	\N	Balsac	12
4121	021	\N	Bastide-l'Évêque	12
4122	022	\N	Bastide-Pradines	12
4123	023	\N	Bastide-Solages	12
4124	024	\N	Belcastel	12
4125	025	\N	Belmont-sur-Rance	12
4126	026	\N	Bertholène	12
4127	027	\N	Bessuéjouls	12
4128	028	\N	Boisse-Penchot	12
4129	029	\N	Bor-et-Bar	12
4130	030	\N	Bouillac	12
4131	031	\N	Bournazel	12
4132	032	\N	Boussac	12
4133	033	\N	Bozouls	12
4134	034	\N	Brandonnet	12
4135	035	\N	Brasc	12
4136	036	\N	Brommat	12
4137	037	\N	Broquiès	12
4138	038	\N	Brousse-le-Château	12
4139	039	\N	Brusque	12
4140	040	\N	Buzeins	12
4141	041	\N	Cabanès	12
4142	042	\N	Calmels-et-le-Viala	12
4143	043	\N	Calmont	12
4144	044	\N	Camarès	12
4145	045	\N	Camboulazet	12
4146	046	\N	Camjac	12
4147	047	\N	Campagnac	12
4148	048	\N	Campouriez	12
4149	049	\N	Campuac	12
4150	050	\N	Canet-de-Salars	12
4151	051	\N	Cantoin	12
4152	052	\N	Capdenac-Gare	12
4153	053	\N	Capelle-Balaguier	12
4154	054	\N	Capelle-Bleys	12
4155	055	\N	Capelle-Bonance	12
4156	056	\N	Baraqueville	12
4157	057	\N	Cassagnes-Bégonhès	12
4158	058	\N	Cassuéjouls	12
4159	059	\N	Castanet	12
4160	060	\N	Castelmary	12
4161	061	\N	Castelnau-de-Mandailles	12
4162	062	\N	Castelnau-Pégayrols	12
4163	063	\N	Cavalerie	12
4164	064	\N	Cayrol	12
4165	065	\N	Centrès	12
4166	066	\N	Clairvaux-d'Aveyron	12
4167	067	\N	Clapier	12
4168	068	\N	Colombiès	12
4169	069	\N	Combret	12
4170	070	\N	Compeyre	12
4171	071	\N	Compolibat	12
4172	072	\N	Comprégnac	12
4173	073	\N	Comps-la-Grand-Ville	12
4174	074	\N	Condom-d'Aubrac	12
4175	075	\N	Connac	12
4176	076	\N	Conques	12
4177	077	\N	Cornus	12
4178	078	\N	Costes-Gozon	12
4179	079	\N	Coubisou	12
4180	080	\N	Coupiac	12
4181	081	\N	Coussergues	12
4182	082	\N	Couvertoirade	12
4183	083	\N	Cransac	12
4184	084	\N	Creissels	12
4185	085	\N	Crespin	12
4186	086	\N	Cresse	12
4187	087	\N	Cruéjouls	12
4188	088	\N	Curières	12
4189	089	\N	Decazeville	12
4190	090	\N	Druelle	12
4191	091	\N	Drulhe	12
4192	092	\N	Durenque	12
4193	093	\N	Fel	12
4194	094	\N	Entraygues-sur-Truyère	12
4195	095	\N	Escandolières	12
4196	096	\N	Espalion	12
4197	097	\N	Espeyrac	12
4198	098	\N	Estaing	12
4199	099	\N	Fayet	12
4200	100	\N	Firmi	12
4201	101	\N	Flagnac	12
4202	102	\N	Flavin	12
4203	103	\N	Florentin-la-Capelle	12
4204	104	\N	Foissac	12
4205	105	\N	Fouillade	12
4206	106	\N	Gabriac	12
4207	107	\N	Gaillac-d'Aveyron	12
4208	108	\N	Galgan	12
4209	109	\N	Gissac	12
4210	110	\N	Golinhac	12
4211	111	\N	Goutrens	12
4212	112	\N	Graissac	12
4213	113	\N	Gramond	12
4214	114	\N	Grand-Vabre	12
4215	115	\N	Hospitalet-du-Larzac	12
4216	116	\N	Huparlac	12
4217	117	\N	Lacalm	12
4218	118	\N	Lacroix-Barrez	12
4219	119	\N	Laguiole	12
4220	120	\N	Laissac	12
4221	121	\N	Lanuéjouls	12
4222	122	\N	Lapanouse-de-Cernon	12
4223	123	\N	Lapanouse	12
4224	124	\N	Lassouts	12
4225	125	\N	Laval-Roquecezière	12
4226	126	\N	Lavernhe	12
4227	127	\N	Lédergues	12
4228	128	\N	Lescure-Jaoul	12
4229	129	\N	Lestrade-et-Thouels	12
4230	130	\N	Livinhac-le-Haut	12
4231	131	\N	Loubière	12
4232	133	\N	Luc	12
4233	134	\N	Lugan	12
4234	135	\N	Lunac	12
4235	136	\N	Maleville	12
4236	137	\N	Manhac	12
4237	138	\N	Marcillac-Vallon	12
4238	139	\N	Marnhagues-et-Latour	12
4239	140	\N	Martiel	12
4240	141	\N	Martrin	12
4241	142	\N	Mayran	12
4242	143	\N	Mélagues	12
4243	144	\N	Meljac	12
4244	145	\N	Millau	12
4245	146	\N	Monastère	12
4246	147	\N	Montagnol	12
4247	148	\N	Montbazens	12
4248	149	\N	Montclar	12
4249	150	\N	Monteils	12
4250	151	\N	Montézic	12
4251	152	\N	Montfranc	12
4252	153	\N	Montjaux	12
4253	154	\N	Montlaur	12
4254	155	\N	Fondamente	12
4255	156	\N	Montpeyroux	12
4256	157	\N	Montrozier	12
4257	158	\N	Montsalès	12
4258	159	\N	Morlhon-le-Haut	12
4259	160	\N	Mostuéjouls	12
4260	161	\N	Mouret	12
4261	162	\N	Moyrazès	12
4262	163	\N	Murasson	12
4263	164	\N	Mur-de-Barrez	12
4264	165	\N	Muret-le-Château	12
4265	166	\N	Murols	12
4266	167	\N	Najac	12
4267	168	\N	Nant	12
4268	169	\N	Naucelle	12
4269	170	\N	Naussac	12
4270	171	\N	Nauviale	12
4271	172	\N	Nayrac	12
4272	173	\N	Noailhac	12
4273	174	\N	Olemps	12
4274	175	\N	Ols-et-Rinhodes	12
4275	176	\N	Onet-le-Château	12
4276	177	\N	Palmas	12
4277	178	\N	Paulhe	12
4278	179	\N	Peux-et-Couffouleux	12
4279	180	\N	Peyreleau	12
4280	181	\N	Peyrusse-le-Roc	12
4281	182	\N	Pierrefiche	12
4282	183	\N	Plaisance	12
4283	184	\N	Pomayrols	12
4284	185	\N	Pont-de-Salars	12
4285	186	\N	Pousthomy	12
4286	187	\N	Prades-d'Aubrac	12
4287	188	\N	Prades-Salars	12
4288	189	\N	Pradinas	12
4289	190	\N	Prévinquières	12
4290	191	\N	Privezac	12
4291	192	\N	Mounes-Prohencoux	12
4292	193	\N	Pruines	12
4293	194	\N	Quins	12
4294	195	\N	Rebourguil	12
4295	196	\N	Recoules-Prévinquières	12
4296	197	\N	Réquista	12
4297	198	\N	Rieupeyroux	12
4298	199	\N	Rignac	12
4299	200	\N	Rivière-sur-Tarn	12
4300	201	\N	Rodelle	12
4301	202	\N	Rodez	12
4302	203	\N	Roquefort-sur-Soulzon	12
4303	204	\N	Roque-Sainte-Marguerite	12
4304	205	\N	Rouquette	12
4305	206	\N	Roussennac	12
4306	207	\N	Rullac-Saint-Cirq	12
4307	208	\N	Saint-Affrique	12
4308	209	\N	Saint-Amans-des-Cots	12
4309	210	\N	Saint-André-de-Najac	12
4310	211	\N	Saint-André-de-Vézines	12
4311	212	\N	Saint-Beaulize	12
4312	213	\N	Saint-Beauzély	12
4313	214	\N	Saint-Chély-d'Aubrac	12
4314	215	\N	Saint-Christophe-Vallon	12
4315	216	\N	Saint-Côme-d'Olt	12
4316	217	\N	Sainte-Croix	12
4317	218	\N	Saint-Cyprien-sur-Dourdou	12
4318	219	\N	Sainte-Eulalie-d'Olt	12
4319	220	\N	Sainte-Eulalie-de-Cernon	12
4320	221	\N	Saint-Félix-de-Lunel	12
4321	222	\N	Saint-Félix-de-Sorgues	12
4322	223	\N	Sainte-Geneviève-sur-Argence	12
4323	224	\N	Saint-Geniez-d'Olt	12
4324	225	\N	Saint-Georges-de-Luzençon	12
4325	226	\N	Saint-Hippolyte	12
4326	227	\N	Saint-Igest	12
4327	228	\N	Saint-Izaire	12
4328	229	\N	Saint-Jean-d'Alcapiès	12
4329	230	\N	Saint-Jean-Delnous	12
4330	231	\N	Saint-Jean-du-Bruel	12
4331	232	\N	Saint-Jean-et-Saint-Paul	12
4332	233	\N	Saint-Juéry	12
4333	234	\N	Sainte-Juliette-sur-Viaur	12
4334	235	\N	Saint-Just-sur-Viaur	12
4335	236	\N	Saint-Laurent-de-Lévézou	12
4336	237	\N	Saint-Laurent-d'Olt	12
4337	238	\N	Saint-Léons	12
4338	239	\N	Saint-Martin-de-Lenne	12
4339	240	\N	Saint-Parthem	12
4340	241	\N	Sainte-Radegonde	12
4341	242	\N	Saint-Rémy	12
4342	243	\N	Saint-Rome-de-Cernon	12
4343	244	\N	Saint-Rome-de-Tarn	12
4344	245	\N	Saint-Salvadou	12
4345	246	\N	Saint-Santin	12
4346	247	\N	Saint-Saturnin-de-Lenne	12
4347	248	\N	Saint-Sernin-sur-Rance	12
4348	249	\N	Saint-Sever-du-Moustier	12
4349	250	\N	Saint-Symphorien-de-Thénières	12
4350	251	\N	Saint-Victor-et-Melvieu	12
4351	252	\N	Salles-Courbatiès	12
4352	253	\N	Salles-Curan	12
4353	254	\N	Salles-la-Source	12
4354	255	\N	Salmiech	12
4355	256	\N	Salvagnac-Cajarc	12
4356	257	\N	Causse-et-Diège	12
4357	258	\N	Salvetat-Peyralès	12
4358	259	\N	Sanvensa	12
4359	260	\N	Sauclières	12
4360	261	\N	Saujac	12
4361	262	\N	Sauveterre-de-Rouergue	12
4362	263	\N	Savignac	12
4363	264	\N	Sébazac-Concourès	12
4364	265	\N	Sébrazac	12
4365	266	\N	Ségur	12
4366	267	\N	Selve	12
4367	268	\N	Sénergues	12
4368	269	\N	Serre	12
4369	270	\N	Sévérac-le-Château	12
4370	271	\N	Sévérac-l'Église	12
4371	272	\N	Sonnac	12
4372	273	\N	Soulages-Bonneval	12
4373	274	\N	Sylvanès	12
4374	275	\N	Tauriac-de-Camarès	12
4375	276	\N	Tauriac-de-Naucelle	12
4376	277	\N	Taussac	12
4377	278	\N	Tayrac	12
4378	279	\N	Terrisse	12
4379	280	\N	Thérondels	12
4380	281	\N	Toulonjac	12
4381	282	\N	Tournemire	12
4382	283	\N	Trémouilles	12
4383	284	\N	Truel	12
4384	285	\N	Vabre-Tizac	12
4385	286	\N	Vabres-l'Abbaye	12
4386	287	\N	Vailhourles	12
4387	288	\N	Valady	12
4388	289	\N	Valzergues	12
4389	290	\N	Vaureilles	12
4390	291	\N	Verrières	12
4391	292	\N	Versols-et-Lapeyre	12
4392	293	\N	Veyreau	12
4393	294	\N	Vézins-de-Lévézou	12
4394	295	\N	Viala-du-Pas-de-Jaux	12
4395	296	\N	Viala-du-Tarn	12
4396	297	\N	Vibal	12
4397	298	\N	Villecomtal	12
4398	299	\N	Villefranche-de-Panat	12
4399	300	\N	Villefranche-de-Rouergue	12
4400	301	\N	Villeneuve	12
4401	303	\N	Vimenet	12
4402	304	\N	Vitrac-en-Viadène	12
4403	305	\N	Viviez	12
4404	307	\N	Curan	12
4405	001	\N	Aix-en-Provence	13
4406	002	\N	Allauch	13
4407	003	\N	Alleins	13
4408	004	\N	Arles	13
4409	005	\N	Aubagne	13
4410	006	\N	Aureille	13
4411	007	\N	Auriol	13
4412	008	\N	Aurons	13
4413	009	\N	Barben	13
4414	010	\N	Barbentane	13
4415	011	\N	Baux-de-Provence	13
4416	012	\N	Beaurecueil	13
4417	013	\N	Belcodène	13
4418	014	\N	Berre-l'Étang	13
4419	015	\N	Bouc-Bel-Air	13
4420	016	\N	Bouilladisse	13
4421	017	\N	Boulbon	13
4422	018	\N	Cabannes	13
4423	019	\N	Cabriès	13
4424	020	\N	Cadolive	13
4425	021	\N	Carry-le-Rouet	13
4426	022	\N	Cassis	13
4427	023	\N	Ceyreste	13
4428	024	\N	Charleval	13
4429	025	\N	Châteauneuf-le-Rouge	13
4430	026	\N	Châteauneuf-les-Martigues	13
4431	027	\N	Châteaurenard	13
4432	028	\N	Ciotat	13
4433	029	\N	Cornillon-Confoux	13
4434	030	\N	Cuges-les-Pins	13
4435	031	\N	Destrousse	13
4436	032	\N	Éguilles	13
4437	033	\N	Ensuès-la-Redonne	13
4438	034	\N	Eygalières	13
4439	035	\N	Eyguières	13
4440	036	\N	Eyragues	13
4441	037	\N	Fare-les-Oliviers	13
4442	038	\N	Fontvieille	13
4443	039	\N	Fos-sur-Mer	13
4444	040	\N	Fuveau	13
4445	041	\N	Gardanne	13
4446	042	\N	Gémenos	13
4447	043	\N	Gignac-la-Nerthe	13
4448	044	\N	Grans	13
4449	045	\N	Graveson	13
4450	046	\N	Gréasque	13
4451	047	\N	Istres	13
4452	048	\N	Jouques	13
4453	049	\N	Lamanon	13
4454	050	\N	Lambesc	13
4455	051	\N	Lançon-Provence	13
4456	052	\N	Maillane	13
4457	053	\N	Mallemort	13
4458	054	\N	Marignane	13
4459	055	\N	Marseille	13
4460	056	\N	Martigues	13
4461	057	\N	Mas-Blanc-des-Alpilles	13
4462	058	\N	Maussane-les-Alpilles	13
4463	059	\N	Meyrargues	13
4464	060	\N	Meyreuil	13
4465	061	\N	Saint-Pierre-de-Mézoargues	13
4466	062	\N	Mimet	13
4467	063	\N	Miramas	13
4468	064	\N	Mollégès	13
4469	065	\N	Mouriès	13
4470	066	\N	Noves	13
4471	067	\N	Orgon	13
4472	068	\N	Paradou	13
4473	069	\N	Pélissanne	13
4474	070	\N	Penne-sur-Huveaune	13
4475	071	\N	Pennes-Mirabeau	13
4476	072	\N	Peynier	13
4477	073	\N	Peypin	13
4478	074	\N	Peyrolles-en-Provence	13
4479	075	\N	Plan-de-Cuques	13
4480	076	\N	Plan-d'Orgon	13
4481	077	\N	Port-de-Bouc	13
4482	078	\N	Port-Saint-Louis-du-Rhône	13
4483	079	\N	Puyloubier	13
4484	080	\N	Puy-Sainte-Réparade	13
4485	081	\N	Rognac	13
4486	082	\N	Rognes	13
4487	083	\N	Rognonas	13
4488	084	\N	Roque-d'Anthéron	13
4489	085	\N	Roquefort-la-Bédoule	13
4490	086	\N	Roquevaire	13
4491	087	\N	Rousset	13
4492	088	\N	Rove	13
4493	089	\N	Saint-Andiol	13
4494	090	\N	Saint-Antonin-sur-Bayon	13
4495	091	\N	Saint-Cannat	13
4496	092	\N	Saint-Chamas	13
4497	093	\N	Saint-Estève-Janson	13
4498	094	\N	Saint-Étienne-du-Grès	13
4499	095	\N	Saint-Marc-Jaumegarde	13
4500	096	\N	Saintes-Maries-de-la-Mer	13
4501	097	\N	Saint-Martin-de-Crau	13
4502	098	\N	Saint-Mitre-les-Remparts	13
4503	099	\N	Saint-Paul-lès-Durance	13
4504	100	\N	Saint-Rémy-de-Provence	13
4505	101	\N	Saint-Savournin	13
4506	102	\N	Saint-Victoret	13
4507	103	\N	Salon-de-Provence	13
4508	104	\N	Sausset-les-Pins	13
4509	105	\N	Sénas	13
4510	106	\N	Septèmes-les-Vallons	13
4511	107	\N	Simiane-Collongue	13
4512	108	\N	Tarascon	13
4513	109	\N	Tholonet	13
4514	110	\N	Trets	13
4515	111	\N	Vauvenargues	13
4516	112	\N	Velaux	13
4517	113	\N	Venelles	13
4518	114	\N	Ventabren	13
4519	115	\N	Vernègues	13
4520	116	\N	Verquières	13
4521	117	\N	Vitrolles	13
4522	118	\N	Coudoux	13
4523	119	\N	Carnoux-en-Provence	13
4524	001	\N	Ablon	14
4525	002	\N	Acqueville	14
4526	003	\N	Agy	14
4527	004	\N	Aignerville	14
4528	005	\N	Airan	14
4529	006	\N	Amayé-sur-Orne	14
4530	007	\N	Amayé-sur-Seulles	14
4531	008	\N	Amblie	14
4532	009	\N	Amfreville	14
4533	011	\N	Anctoville	14
4534	012	\N	Angerville	14
4535	013	\N	Angoville	14
4536	014	\N	Anguerny	14
4537	015	\N	Anisy	14
4538	016	\N	Annebault	14
4539	019	\N	Arganchy	14
4540	020	\N	Argences	14
4541	021	\N	Arromanches-les-Bains	14
4542	022	\N	Asnelles	14
4543	023	\N	Asnières-en-Bessin	14
4544	024	\N	Auberville	14
4545	025	\N	Aubigny	14
4546	026	\N	Audrieu	14
4547	027	\N	Aunay-sur-Odon	14
4548	028	\N	Auquainville	14
4549	029	\N	Autels-Saint-Bazile	14
4550	030	\N	Authie	14
4551	031	\N	Authieux-Papion	14
4552	032	\N	Authieux-sur-Calonne	14
4553	033	\N	Auvillars	14
4554	034	\N	Avenay	14
4555	035	\N	Balleroy	14
4556	036	\N	Banneville-la-Campagne	14
4557	037	\N	Banneville-sur-Ajon	14
4558	038	\N	Banville	14
4559	039	\N	Barbery	14
4560	040	\N	Barbeville	14
4561	041	\N	Barneville-la-Bertran	14
4562	042	\N	Baron-sur-Odon	14
4563	043	\N	Barou-en-Auge	14
4564	044	\N	Basly	14
4565	045	\N	Basseneville	14
4566	046	\N	Bavent	14
4567	047	\N	Bayeux	14
4568	049	\N	Bazenville	14
4569	050	\N	Bazoque	14
4570	052	\N	Beaulieu	14
4571	053	\N	Beaumais	14
4572	054	\N	Beaumesnil	14
4573	055	\N	Beaumont-en-Auge	14
4574	056	\N	Bauquay	14
4575	057	\N	Bellengreville	14
4576	058	\N	Bellou	14
4577	059	\N	Benerville-sur-Mer	14
4578	060	\N	Bénouville	14
4579	061	\N	Bény-Bocage	14
4580	062	\N	Bény-sur-Mer	14
4581	063	\N	Bernesq	14
4582	064	\N	Bernières-d'Ailly	14
4583	065	\N	Bernières-le-Patry	14
4584	066	\N	Bernières-sur-Mer	14
4585	068	\N	Biéville-Beuville	14
4586	069	\N	Beuvillers	14
4587	070	\N	Beuvron-en-Auge	14
4588	073	\N	Bigne	14
4589	074	\N	Billy	14
4590	075	\N	Bissières	14
4591	076	\N	Blainville-sur-Orne	14
4592	077	\N	Blangy-le-Château	14
4593	078	\N	Blay	14
4594	079	\N	Blonville-sur-Mer	14
4595	080	\N	Bô	14
4596	081	\N	Boissey	14
4597	082	\N	Boissière	14
4598	083	\N	Bonnebosq	14
4599	084	\N	Bonnemaison	14
4600	085	\N	Bonneville-la-Louvet	14
4601	086	\N	Bonneville-sur-Touques	14
4602	087	\N	Bonnil	14
4603	088	\N	Bons-Tassilly	14
4604	089	\N	Bougy	14
4605	090	\N	Boulon	14
4606	091	\N	Bourgeauville	14
4607	092	\N	Bourguébus	14
4608	093	\N	Branville	14
4609	096	\N	Brémoy	14
4610	097	\N	Bretteville-le-Rabet	14
4611	098	\N	Bretteville-l'Orgueilleuse	14
4612	099	\N	Bretteville-sur-Dives	14
4613	100	\N	Bretteville-sur-Laize	14
4614	101	\N	Bretteville-sur-Odon	14
4615	102	\N	Breuil-en-Auge	14
4616	103	\N	Breuil-en-Bessin	14
4617	104	\N	Brévedent	14
4618	105	\N	Brévière	14
4619	106	\N	Bréville-les-Monts	14
4620	107	\N	Bricqueville	14
4621	109	\N	Brouay	14
4622	110	\N	Brucourt	14
4623	111	\N	Bucéels	14
4624	113	\N	Burcy	14
4625	115	\N	Bures-les-Monts	14
4626	116	\N	Bû-sur-Rouvres	14
4627	117	\N	Cabourg	14
4628	118	\N	Caen	14
4629	119	\N	Cagny	14
4630	120	\N	Cahagnes	14
4631	121	\N	Cahagnolles	14
4632	122	\N	Caine	14
4633	123	\N	Cairon	14
4634	124	\N	Cambe	14
4635	125	\N	Cambes-en-Plaine	14
4636	126	\N	Cambremer	14
4637	127	\N	Campagnolles	14
4638	128	\N	Campandré-Valcongrain	14
4639	129	\N	Campeaux	14
4640	130	\N	Campigny	14
4641	131	\N	Canapville	14
4642	132	\N	Canchy	14
4643	134	\N	Canteloup	14
4644	135	\N	Carcagny	14
4645	136	\N	Cardonville	14
4646	137	\N	Carpiquet	14
4647	138	\N	Cartigny-l'Épinay	14
4648	139	\N	Carville	14
4649	140	\N	Castillon	14
4650	141	\N	Castillon-en-Auge	14
4651	142	\N	Castilly	14
4652	143	\N	Caumont-l'Éventé	14
4653	144	\N	Caumont-sur-Orne	14
4654	145	\N	Cauvicourt	14
4655	146	\N	Cauville	14
4656	147	\N	Cernay	14
4657	148	\N	Cerqueux	14
4658	149	\N	Cesny-aux-Vignes-Ouézy	14
4659	150	\N	Cesny-Bois-Halbout	14
4660	151	\N	Champ-du-Boult	14
4661	152	\N	Chapelle-Engerbold	14
4662	153	\N	Chapelle-Haute-Grue	14
4663	154	\N	Chapelle-Yvon	14
4664	155	\N	Cheffreville-Tonnencourt	14
4665	156	\N	Chênedollé	14
4666	157	\N	Cheux	14
4667	158	\N	Chicheboville	14
4668	159	\N	Chouain	14
4669	160	\N	Cintheaux	14
4670	161	\N	Clarbec	14
4671	162	\N	Clécy	14
4672	163	\N	Cléville	14
4673	164	\N	Clinchamps-sur-Orne	14
4674	165	\N	Colleville-sur-Mer	14
4675	166	\N	Colleville-Montgomery	14
4676	167	\N	Colombelles	14
4677	168	\N	Colombières	14
4678	169	\N	Colombiers-sur-Seulles	14
4679	170	\N	Colomby-sur-Thaon	14
4680	171	\N	Combray	14
4681	172	\N	Commes	14
4682	173	\N	Condé-sur-Ifs	14
4683	174	\N	Condé-sur-Noireau	14
4684	175	\N	Condé-sur-Seulles	14
4685	176	\N	Conteville	14
4686	177	\N	Coquainvilliers	14
4687	178	\N	Corbon	14
4688	179	\N	Cordebugle	14
4689	180	\N	Cordey	14
4690	181	\N	Cormelles-le-Royal	14
4691	182	\N	Cormolain	14
4692	183	\N	Cossesseville	14
4693	184	\N	Cottun	14
4694	185	\N	Coudray-Rabut	14
4695	186	\N	Coulombs	14
4696	187	\N	Coulonces	14
4697	188	\N	Coulvain	14
4698	189	\N	Coupesarte	14
4699	190	\N	Courcy	14
4700	191	\N	Courseulles-sur-Mer	14
4701	192	\N	Courson	14
4702	193	\N	Courtonne-la-Meurdrac	14
4703	194	\N	Courtonne-les-Deux-Églises	14
4704	195	\N	Courvaudon	14
4705	196	\N	Crépon	14
4706	197	\N	Cresserons	14
4707	198	\N	Cresseveuille	14
4708	200	\N	Creully	14
4709	201	\N	Crèvecur-en-Auge	14
4710	202	\N	Cricquebuf	14
4711	203	\N	Cricqueville-en-Auge	14
4712	204	\N	Cricqueville-en-Bessin	14
4713	205	\N	Cristot	14
4714	206	\N	Crocy	14
4715	207	\N	Croisilles	14
4716	208	\N	Croissanville	14
4717	209	\N	Crouay	14
4718	210	\N	Croupte	14
4719	211	\N	Culey-le-Patry	14
4720	212	\N	Cully	14
4721	213	\N	Curcy-sur-Orne	14
4722	214	\N	Cussy	14
4723	215	\N	Cuverville	14
4724	216	\N	Damblainville	14
4725	217	\N	Dampierre	14
4726	218	\N	Danestal	14
4727	219	\N	Danvou-la-Ferrière	14
4728	220	\N	Deauville	14
4729	221	\N	Démouville	14
4730	222	\N	Désert	14
4731	223	\N	Détroit	14
4732	224	\N	Deux-Jumeaux	14
4733	225	\N	Dives-sur-Mer	14
4734	226	\N	Donnay	14
4735	227	\N	Douville-en-Auge	14
4736	228	\N	Douvres-la-Délivrande	14
4737	229	\N	Dozulé	14
4738	230	\N	Drubec	14
4739	231	\N	Beaufour-Druval	14
4740	232	\N	Ducy-Sainte-Marguerite	14
4741	235	\N	Écrammeville	14
4742	236	\N	Ellon	14
4743	237	\N	Émiéville	14
4744	238	\N	Englesqueville-en-Auge	14
4745	239	\N	Englesqueville-la-Percée	14
4746	240	\N	Épaney	14
4747	241	\N	Épinay-sur-Odon	14
4748	242	\N	Épron	14
4749	243	\N	Équemauville	14
4750	244	\N	Eraines	14
4751	245	\N	Ernes	14
4752	246	\N	Escoville	14
4753	248	\N	Espins	14
4754	249	\N	Esquay-Notre-Dame	14
4755	250	\N	Esquay-sur-Seulles	14
4756	251	\N	Esson	14
4757	252	\N	Estrées-la-Campagne	14
4758	253	\N	Estry	14
4759	254	\N	Éterville	14
4760	255	\N	Étouvy	14
4761	256	\N	Étréham	14
4762	257	\N	Évrecy	14
4763	258	\N	Falaise	14
4764	259	\N	Familly	14
4765	260	\N	Fauguernon	14
4766	261	\N	Faulq	14
4767	264	\N	Ferrière-Harang	14
4768	265	\N	Fervaques	14
4769	266	\N	Feuguerolles-Bully	14
4770	268	\N	Fierville-Bray	14
4771	269	\N	Fierville-les-Parcs	14
4772	270	\N	Firfol	14
4773	271	\N	Fleury-sur-Orne	14
4774	272	\N	Folie	14
4775	273	\N	Folletière-Abenon	14
4776	274	\N	Fontaine-Étoupefour	14
4777	275	\N	Fontaine-Henry	14
4778	276	\N	Fontaine-le-Pin	14
4779	277	\N	Fontenay-le-Marmion	14
4780	278	\N	Fontenay-le-Pesnel	14
4781	279	\N	Fontenermont	14
4782	280	\N	Formentin	14
4783	281	\N	Formigny	14
4784	282	\N	Foulognes	14
4785	283	\N	Fourches	14
4786	284	\N	Fourneaux-le-Val	14
4787	285	\N	Fournet	14
4788	286	\N	Fourneville	14
4789	287	\N	Frénouville	14
4790	288	\N	Fresne-Camilly	14
4791	289	\N	Fresné-la-Mère	14
4792	290	\N	Fresney-le-Puceux	14
4793	291	\N	Fresney-le-Vieux	14
4794	292	\N	Friardel	14
4795	293	\N	Fumichon	14
4796	294	\N	Garcelles-Secqueville	14
4797	296	\N	Gast	14
4798	297	\N	Gavrus	14
4799	298	\N	Géfosse-Fontenay	14
4800	299	\N	Genneville	14
4801	300	\N	Gerrots	14
4802	301	\N	Giberville	14
4803	302	\N	Glanville	14
4804	303	\N	Glos	14
4805	304	\N	Gonneville-sur-Honfleur	14
4806	305	\N	Gonneville-sur-Mer	14
4807	306	\N	Gonneville-en-Auge	14
4808	307	\N	Goupillières	14
4809	308	\N	Goustranville	14
4810	309	\N	Gouvix	14
4811	310	\N	Grainville-Langannerie	14
4812	311	\N	Grainville-sur-Odon	14
4813	312	\N	Grandcamp-Maisy	14
4814	313	\N	Grandchamp-le-Château	14
4815	316	\N	Grangues	14
4816	317	\N	Graverie	14
4817	318	\N	Graye-sur-Mer	14
4818	319	\N	Grentheville	14
4819	320	\N	Grimbosq	14
4820	322	\N	Guéron	14
4821	324	\N	Hamars	14
4822	325	\N	Hermanville-sur-Mer	14
4823	326	\N	Hermival-les-Vaux	14
4824	327	\N	Hérouville-Saint-Clair	14
4825	328	\N	Hérouvillette	14
4826	329	\N	Heuland	14
4827	330	\N	Heurtevent	14
4828	331	\N	Hiéville	14
4829	332	\N	Hoguette	14
4830	333	\N	Honfleur	14
4831	334	\N	Hôtellerie	14
4832	335	\N	Hotot-en-Auge	14
4833	336	\N	Hottot-les-Bagues	14
4834	337	\N	Houblonnière	14
4835	338	\N	Houlgate	14
4836	339	\N	Hubert-Folie	14
4837	341	\N	Ifs	14
4838	342	\N	Isigny-sur-Mer	14
4839	343	\N	Isles-Bardel	14
4840	344	\N	Janville	14
4841	345	\N	Jort	14
4842	346	\N	Juaye-Mondaye	14
4843	347	\N	Jurques	14
4844	348	\N	Juvigny-sur-Seulles	14
4845	349	\N	Laize-la-Ville	14
4846	350	\N	Lande-sur-Drôme	14
4847	352	\N	Landelles-et-Coupigny	14
4848	353	\N	Landes-sur-Ajon	14
4849	354	\N	Langrune-sur-Mer	14
4850	355	\N	Lantheuil	14
4851	356	\N	Lasson	14
4852	357	\N	Lassy	14
4853	358	\N	Léaupartie	14
4854	359	\N	Lécaude	14
4855	360	\N	Leffard	14
4856	361	\N	Lénault	14
4857	362	\N	Lessard-et-le-Chêne	14
4858	364	\N	Lingèvres	14
4859	365	\N	Lion-sur-Mer	14
4860	366	\N	Lisieux	14
4861	367	\N	Lison	14
4862	368	\N	Lisores	14
4863	369	\N	Litteau	14
4864	370	\N	Molay-Littry	14
4865	371	\N	Livarot	14
4866	372	\N	Livry	14
4867	373	\N	Locheur	14
4868	374	\N	Loges	14
4869	375	\N	Loges-Saulces	14
4870	376	\N	Longraye	14
4871	377	\N	Longues-sur-Mer	14
4872	378	\N	Longueville	14
4873	379	\N	Longvillers	14
4874	380	\N	Loucelles	14
4875	381	\N	Louvagny	14
4876	382	\N	Louvières	14
4877	383	\N	Louvigny	14
4878	384	\N	Luc-sur-Mer	14
4879	385	\N	Magny-en-Bessin	14
4880	386	\N	Magny-la-Campagne	14
4881	387	\N	Magny-le-Freule	14
4882	388	\N	Maisoncelles-la-Jourdan	14
4883	389	\N	Maisoncelles-Pelvey	14
4884	390	\N	Maisoncelles-sur-Ajon	14
4885	391	\N	Maisons	14
4886	393	\N	Maizet	14
4887	394	\N	Maizières	14
4888	395	\N	Malloué	14
4889	396	\N	Maltot	14
4890	397	\N	Mandeville-en-Bessin	14
4891	398	\N	Manerbe	14
4892	399	\N	Manneville-la-Pipard	14
4893	400	\N	Manoir	14
4894	401	\N	Manvieux	14
4895	402	\N	Marais-la-Chapelle	14
4896	403	\N	Marolles	14
4897	404	\N	Martainville	14
4898	405	\N	Martigny-sur-l'Ante	14
4899	406	\N	Martragny	14
4900	407	\N	Mathieu	14
4901	408	\N	May-sur-Orne	14
4902	409	\N	Merville-Franceville-Plage	14
4903	410	\N	Méry-Corbon	14
4904	411	\N	Meslay	14
4905	412	\N	Mesnil-au-Grain	14
4906	413	\N	Mesnil-Auzouf	14
4907	414	\N	Mesnil-Bacley	14
4908	415	\N	Mesnil-Benoist	14
4909	416	\N	Mesnil-Caussois	14
4910	417	\N	Mesnil-Clinchamps	14
4911	418	\N	Mesnil-Durand	14
4912	419	\N	Mesnil-Eudes	14
4913	420	\N	Mesnil-Germain	14
4914	421	\N	Mesnil-Guillaume	14
4915	422	\N	Mesnil-Mauger	14
4916	423	\N	Mesnil-Patry	14
4917	424	\N	Mesnil-Robert	14
4918	425	\N	Mesnil-Simon	14
4919	426	\N	Mesnil-sur-Blangy	14
4920	427	\N	Mesnil-Villement	14
4921	429	\N	Meulles	14
4922	430	\N	Meuvaines	14
4923	431	\N	Mézidon-Canon	14
4924	432	\N	Missy	14
4925	433	\N	Mittois	14
4926	435	\N	Monceaux	14
4927	436	\N	Monceaux-en-Bessin	14
4928	437	\N	Mondeville	14
4929	438	\N	Mondrainville	14
4930	439	\N	Monfréville	14
4931	440	\N	Montamy	14
4932	441	\N	Mont-Bertrand	14
4933	442	\N	Montchamp	14
4934	443	\N	Montchauvet	14
4935	444	\N	Monteille	14
4936	445	\N	Montfiquet	14
4937	446	\N	Montigny	14
4938	448	\N	Montreuil-en-Auge	14
4939	449	\N	Monts-en-Bessin	14
4940	450	\N	Montviette	14
4941	452	\N	Morteaux-Coulibuf	14
4942	453	\N	Mosles	14
4943	454	\N	Mouen	14
4944	455	\N	Moulines	14
4945	456	\N	Moult	14
4946	457	\N	Moutiers-en-Auge	14
4947	458	\N	Moutiers-en-Cinglais	14
4948	459	\N	Moutiers-Hubert	14
4949	460	\N	Moyaux	14
4950	461	\N	Mutrécy	14
4951	462	\N	Neuilly-la-Forêt	14
4952	465	\N	Nonant	14
4953	466	\N	Norolles	14
4954	467	\N	Noron-l'Abbaye	14
4955	468	\N	Noron-la-Poterie	14
4956	469	\N	Norrey-en-Auge	14
4957	471	\N	Notre-Dame-de-Courson	14
4958	473	\N	Notre-Dame-de-Livaye	14
4959	474	\N	Notre-Dame-d'Estrées	14
4960	475	\N	Noyers-Bocage	14
4961	476	\N	Olendon	14
4962	477	\N	Ondefontaine	14
4963	478	\N	Orbec	14
4964	480	\N	Osmanville	14
4965	481	\N	Oubeaux	14
4966	483	\N	Ouffières	14
4967	484	\N	Ouilly-du-Houley	14
4968	486	\N	Ouilly-le-Tesson	14
4969	487	\N	Ouilly-le-Vicomte	14
4970	488	\N	Ouistreham	14
4971	489	\N	Ouville-la-Bien-Tournée	14
4972	491	\N	Parfouru-sur-Odon	14
4973	492	\N	Pennedepie	14
4974	493	\N	Percy-en-Auge	14
4975	494	\N	Périers-en-Auge	14
4976	495	\N	Périers-sur-le-Dan	14
4977	496	\N	Périgny	14
4978	497	\N	Perrières	14
4979	498	\N	Pertheville-Ners	14
4980	499	\N	Petiville	14
4981	500	\N	Pierrefitte-en-Auge	14
4982	501	\N	Pierrefitte-en-Cinglais	14
4983	502	\N	Pierrepont	14
4984	503	\N	Pierres	14
4985	504	\N	Pin	14
4986	505	\N	Placy	14
4987	506	\N	Planquery	14
4988	508	\N	Plessis-Grimoult	14
4989	509	\N	Plumetot	14
4990	510	\N	Pommeraye	14
4991	511	\N	Pont-Bellanger	14
4992	512	\N	Pontécoulant	14
4993	513	\N	Pont-Farcy	14
4994	514	\N	Pont-l'Évêque	14
4995	515	\N	Port-en-Bessin-Huppain	14
4996	516	\N	Potigny	14
4997	517	\N	Poussy-la-Campagne	14
4998	518	\N	Préaux-Saint-Sébastien	14
4999	519	\N	Préaux-Bocage	14
5000	520	\N	Pré-d'Auge	14
5001	521	\N	Presles	14
5002	522	\N	Prêtreville	14
5003	523	\N	Proussy	14
5004	524	\N	Putot-en-Auge	14
5005	525	\N	Putot-en-Bessin	14
5006	527	\N	Biéville-Quétiéville	14
5007	528	\N	Quetteville	14
5008	529	\N	Ranchy	14
5009	530	\N	Ranville	14
5010	531	\N	Rapilly	14
5011	532	\N	Reculey	14
5012	533	\N	Repentigny	14
5013	534	\N	Reux	14
5014	535	\N	Reviers	14
5015	536	\N	Rivière-Saint-Sauveur	14
5016	538	\N	Rocquancourt	14
5017	539	\N	Rocque	14
5018	540	\N	Rocques	14
5019	541	\N	Roque-Baignard	14
5020	542	\N	Rosel	14
5021	543	\N	Rots	14
5022	544	\N	Roucamps	14
5023	545	\N	Roullours	14
5024	546	\N	Rouvres	14
5025	547	\N	Rubercy	14
5026	548	\N	Rucqueville	14
5027	549	\N	Rully	14
5028	550	\N	Rumesnil	14
5029	551	\N	Russy	14
5030	552	\N	Ryes	14
5031	553	\N	Saint-Agnan-le-Malherbe	14
5032	554	\N	Saint-Aignan-de-Cramesnil	14
5033	555	\N	Saint-André-d'Hébertot	14
5034	556	\N	Saint-André-sur-Orne	14
5035	557	\N	Saint-Arnoult	14
5036	558	\N	Saint-Aubin-d'Arquenay	14
5037	559	\N	Saint-Aubin-des-Bois	14
5038	562	\N	Saint-Aubin-sur-Mer	14
5039	563	\N	Saint-Benoît-d'Hébertot	14
5040	564	\N	Saint-Charles-de-Percy	14
5041	565	\N	Saint-Côme-de-Fresné	14
5042	566	\N	Saint-Contest	14
5043	568	\N	Sainte-Croix-Grand-Tonne	14
5044	569	\N	Sainte-Croix-sur-Mer	14
5045	570	\N	Saint-Cyr-du-Ronceray	14
5046	571	\N	Saint-Denis-de-Mailloc	14
5047	572	\N	Saint-Denis-de-Méré	14
5048	573	\N	Saint-Denis-Maisoncelles	14
5049	574	\N	Saint-Désir	14
5050	575	\N	Saint-Étienne-la-Thillaye	14
5051	576	\N	Sainte-Foy-de-Montgommery	14
5052	577	\N	Saint-Gabriel-Brécy	14
5053	578	\N	Saint-Gatien-des-Bois	14
5054	579	\N	Saint-Georges-d'Aunay	14
5055	580	\N	Saint-Georges-en-Auge	14
5056	581	\N	Saint-Germain-d'Ectot	14
5057	582	\N	Saint-Germain-de-Livet	14
5058	583	\N	Saint-Germain-de-Montgommery	14
5059	584	\N	Saint-Germain-de-Tallevende-la-Lande-Vaumont	14
5060	585	\N	Saint-Germain-du-Crioult	14
5061	586	\N	Saint-Germain-du-Pert	14
5062	587	\N	Saint-Germain-la-Blanche-Herbe	14
5063	588	\N	Saint-Germain-Langot	14
5064	589	\N	Saint-Germain-le-Vasson	14
5065	590	\N	Sainte-Honorine-de-Ducy	14
5066	591	\N	Sainte-Honorine-des-Pertes	14
5067	592	\N	Sainte-Honorine-du-Fay	14
5068	593	\N	Saint-Hymer	14
5069	595	\N	Saint-Jean-de-Livet	14
5070	596	\N	Saint-Jean-des-Essartiers	14
5071	597	\N	Saint-Jean-le-Blanc	14
5072	598	\N	Saint-Jouin	14
5073	599	\N	Saint-Julien-de-Mailloc	14
5074	600	\N	Saint-Julien-le-Faucon	14
5075	601	\N	Saint-Julien-sur-Calonne	14
5076	602	\N	Saint-Lambert	14
5077	603	\N	Saint-Laurent-de-Condel	14
5078	604	\N	Saint-Laurent-du-Mont	14
5079	605	\N	Saint-Laurent-sur-Mer	14
5080	606	\N	Saint-Léger-Dubosq	14
5081	607	\N	Saint-Louet-sur-Seulles	14
5082	608	\N	Saint-Loup-de-Fribois	14
5083	609	\N	Saint-Loup-Hors	14
5084	610	\N	Saint-Manvieu-Norrey	14
5085	611	\N	Saint-Manvieu-Bocage	14
5086	613	\N	Saint-Marcouf	14
5087	614	\N	Sainte-Marguerite-d'Elle	14
5088	615	\N	Sainte-Marguerite-des-Loges	14
5089	616	\N	Sainte-Marguerite-de-Viette	14
5090	618	\N	Sainte-Marie-Laumont	14
5091	619	\N	Sainte-Marie-Outre-l'Eau	14
5092	620	\N	Saint-Martin-aux-Chartrains	14
5093	621	\N	Saint-Martin-de-Bienfaite-la-Cressonnière	14
5094	622	\N	Saint-Martin-de-Blagny	14
5095	623	\N	Saint-Martin-de-Fontenay	14
5096	625	\N	Saint-Martin-de-la-Lieue	14
5097	626	\N	Saint-Martin-de-Mailloc	14
5098	627	\N	Saint-Martin-de-Mieux	14
5099	628	\N	Saint-Martin-de-Sallen	14
5100	629	\N	Saint-Martin-des-Besaces	14
5101	630	\N	Saint-Martin-des-Entrées	14
5102	632	\N	Saint-Martin-Don	14
5103	633	\N	Saint-Martin-du-Mesnil-Oury	14
5104	634	\N	Saint-Michel-de-Livet	14
5105	635	\N	Saint-Omer	14
5106	636	\N	Saint-Ouen-des-Besaces	14
5107	637	\N	Saint-Ouen-du-Mesnil-Oger	14
5108	638	\N	Saint-Ouen-le-Houx	14
5109	639	\N	Saint-Ouen-le-Pin	14
5110	640	\N	Saint-Pair	14
5111	643	\N	Saint-Paul-du-Vernay	14
5112	644	\N	Saint-Philbert-des-Champs	14
5113	645	\N	Saint-Pierre-Azif	14
5114	646	\N	Saint-Pierre-Canivet	14
5115	647	\N	Saint-Pierre-de-Mailloc	14
5116	648	\N	Saint-Pierre-des-Ifs	14
5117	649	\N	Saint-Pierre-du-Bû	14
5118	650	\N	Saint-Pierre-du-Fresne	14
5119	651	\N	Saint-Pierre-du-Jonquet	14
5120	652	\N	Saint-Pierre-du-Mont	14
5121	653	\N	Saint-Pierre-la-Vieille	14
5122	654	\N	Saint-Pierre-sur-Dives	14
5123	655	\N	Saint-Pierre-Tarentaine	14
5124	656	\N	Saint-Rémy	14
5125	657	\N	Saint-Samson	14
5126	658	\N	Saint-Sever-Calvados	14
5127	659	\N	Saint-Sylvain	14
5128	660	\N	Saint-Vaast-en-Auge	14
5129	661	\N	Saint-Vaast-sur-Seulles	14
5130	662	\N	Saint-Vigor-des-Mézerets	14
5131	663	\N	Saint-Vigor-le-Grand	14
5132	664	\N	Sallen	14
5133	665	\N	Sallenelles	14
5134	666	\N	Sannerville	14
5135	667	\N	Saon	14
5136	668	\N	Saonnet	14
5137	669	\N	Sassy	14
5138	670	\N	Secqueville-en-Bessin	14
5139	671	\N	Sept-Frères	14
5140	672	\N	Sept-Vents	14
5141	674	\N	Soignolles	14
5142	675	\N	Soliers	14
5143	676	\N	Sommervieu	14
5144	677	\N	Soulangy	14
5145	678	\N	Soumont-Saint-Quentin	14
5146	679	\N	Subles	14
5147	680	\N	Sully	14
5148	681	\N	Surrain	14
5149	682	\N	Surville	14
5150	684	\N	Tessel	14
5151	685	\N	Thaon	14
5152	686	\N	Theil-Bocage	14
5153	687	\N	Theil-en-Auge	14
5154	688	\N	Thiéville	14
5155	689	\N	Thury-Harcourt	14
5156	690	\N	Tierceville	14
5157	691	\N	Tilly-la-Campagne	14
5158	692	\N	Tilly-sur-Seulles	14
5159	693	\N	Tordouet	14
5160	694	\N	Torquesne	14
5161	695	\N	Torteval-Quesnay	14
5162	696	\N	Tortisambert	14
5163	697	\N	Oudon	14
5164	698	\N	Touffréville	14
5165	699	\N	Touques	14
5166	700	\N	Tour-en-Bessin	14
5167	701	\N	Tourgéville	14
5168	702	\N	Tournay-sur-Odon	14
5169	703	\N	Tournebu	14
5170	704	\N	Tourneur	14
5171	705	\N	Tournières	14
5172	706	\N	Tourville-en-Auge	14
5173	707	\N	Tourville-sur-Odon	14
5174	708	\N	Tracy-Bocage	14
5175	709	\N	Tracy-sur-Mer	14
5176	710	\N	Tréprel	14
5177	711	\N	Trévières	14
5178	712	\N	Troarn	14
5179	713	\N	Trois-Monts	14
5180	714	\N	Tronquay	14
5181	715	\N	Trouville-sur-Mer	14
5182	716	\N	Trungy	14
5183	717	\N	Truttemer-le-Grand	14
5184	718	\N	Truttemer-le-Petit	14
5185	719	\N	Urville	14
5186	720	\N	Ussy	14
5187	721	\N	Vacognes-Neuilly	14
5188	722	\N	Vacquerie	14
5189	723	\N	Valsemé	14
5190	724	\N	Varaville	14
5191	726	\N	Vassy	14
5192	727	\N	Vaubadon	14
5193	728	\N	Vaucelles	14
5194	729	\N	Vaudeloges	14
5195	730	\N	Vaudry	14
5196	731	\N	Vauville	14
5197	732	\N	Vaux-sur-Aure	14
5198	733	\N	Vaux-sur-Seulles	14
5199	734	\N	Vendes	14
5200	735	\N	Vendeuvre	14
5201	737	\N	Versainville	14
5202	738	\N	Verson	14
5203	739	\N	Ver-sur-Mer	14
5204	740	\N	Vespière	14
5205	741	\N	Vey	14
5206	742	\N	Vicques	14
5207	743	\N	Victot-Pontfol	14
5208	744	\N	Vienne-en-Bessin	14
5209	745	\N	Vierville-sur-Mer	14
5210	746	\N	Viessoix	14
5211	747	\N	Vieux	14
5212	748	\N	Vieux-Bourg	14
5213	749	\N	Vieux-Fumé	14
5214	750	\N	Vieux-Pont-en-Auge	14
5215	751	\N	Vignats	14
5216	752	\N	Villers-Bocage	14
5217	753	\N	Villers-Canivet	14
5218	754	\N	Villers-sur-Mer	14
5219	755	\N	Villerville	14
5220	756	\N	Villette	14
5221	757	\N	Villiers-le-Sec	14
5222	758	\N	Villons-les-Buissons	14
5223	759	\N	Villy-lez-Falaise	14
5224	760	\N	Villy-Bocage	14
5225	761	\N	Vimont	14
5226	762	\N	Vire	14
5227	763	\N	Vouilly	14
5228	764	\N	Pont-d'Ouilly	14
5229	001	\N	Allanche	15
5230	002	\N	Alleuze	15
5231	003	\N	Ally	15
5232	004	\N	Andelat	15
5233	005	\N	Anglards-de-Saint-Flour	15
5234	006	\N	Anglards-de-Salers	15
5235	007	\N	Anterrieux	15
5236	008	\N	Antignac	15
5237	009	\N	Apchon	15
5238	010	\N	Arches	15
5239	011	\N	Arnac	15
5240	012	\N	Arpajon-sur-Cère	15
5241	013	\N	Auriac-l'Église	15
5242	014	\N	Aurillac	15
5243	015	\N	Auzers	15
5244	016	\N	Ayrens	15
5245	017	\N	Badailhac	15
5246	018	\N	Barriac-les-Bosquets	15
5247	019	\N	Bassignac	15
5248	020	\N	Beaulieu	15
5249	021	\N	Boisset	15
5250	022	\N	Bonnac	15
5251	024	\N	Brageac	15
5252	025	\N	Albepierre-Bredons	15
5253	026	\N	Brezons	15
5254	027	\N	Calvinet	15
5255	028	\N	Carlat	15
5256	029	\N	Cassaniouze	15
5257	030	\N	Cayrols	15
5258	031	\N	Celles	15
5259	032	\N	Celoux	15
5260	033	\N	Cézens	15
5261	034	\N	Chaliers	15
5262	035	\N	Chalinargues	15
5263	036	\N	Chalvignac	15
5264	037	\N	Champagnac	15
5265	038	\N	Champs-sur-Tarentaine-Marchal	15
5266	040	\N	Chanterelle	15
5267	041	\N	Chapelle-d'Alagnon	15
5268	042	\N	Chapelle-Laurent	15
5269	043	\N	Charmensac	15
5270	044	\N	Chastel-sur-Murat	15
5271	045	\N	Chaudes-Aigues	15
5272	046	\N	Chaussenac	15
5273	047	\N	Chavagnac	15
5274	048	\N	Chazelles	15
5275	049	\N	Cheylade	15
5276	050	\N	Claux	15
5277	051	\N	Clavières	15
5278	052	\N	Collandres	15
5279	053	\N	Coltines	15
5280	054	\N	Condat	15
5281	055	\N	Coren	15
5282	056	\N	Crandelles	15
5283	057	\N	Cros-de-Montvert	15
5284	058	\N	Cros-de-Ronesque	15
5285	059	\N	Cussac	15
5286	060	\N	Deux-Verges	15
5287	061	\N	Dienne	15
5288	063	\N	Drugeac	15
5289	064	\N	Escorailles	15
5290	065	\N	Espinasse	15
5291	066	\N	Falgoux	15
5292	067	\N	Fau	15
5293	068	\N	Faverolles	15
5294	069	\N	Ferrières-Saint-Mary	15
5295	070	\N	Fontanges	15
5296	071	\N	Fournoulès	15
5297	072	\N	Freix-Anglards	15
5298	073	\N	Fridefont	15
5299	074	\N	Giou-de-Mamou	15
5300	075	\N	Girgols	15
5301	076	\N	Glénat	15
5302	077	\N	Gourdièges	15
5303	078	\N	Jabrun	15
5304	079	\N	Jaleyrac	15
5305	080	\N	Joursac	15
5306	081	\N	Jou-sous-Monjou	15
5307	082	\N	Junhac	15
5308	083	\N	Jussac	15
5309	084	\N	Labesserette	15
5310	085	\N	Labrousse	15
5311	086	\N	Lacapelle-Barrès	15
5312	087	\N	Lacapelle-del-Fraisse	15
5313	088	\N	Lacapelle-Viescamp	15
5314	089	\N	Ladinhac	15
5315	090	\N	Lafeuillade-en-Vézie	15
5316	091	\N	Landeyrat	15
5317	092	\N	Lanobre	15
5318	093	\N	Lapeyrugue	15
5319	094	\N	Laroquebrou	15
5320	095	\N	Laroquevieille	15
5321	096	\N	Lascelle	15
5322	097	\N	Lastic	15
5323	098	\N	Laurie	15
5324	099	\N	Lavastrie	15
5325	100	\N	Laveissenet	15
5326	101	\N	Laveissière	15
5327	102	\N	Lavigerie	15
5328	103	\N	Leucamp	15
5329	104	\N	Leynhac	15
5330	105	\N	Leyvaux	15
5331	106	\N	Lieutadès	15
5332	107	\N	Lorcières	15
5333	108	\N	Loubaresse	15
5334	110	\N	Lugarde	15
5335	111	\N	Madic	15
5336	112	\N	Malbo	15
5337	113	\N	Mandailles-Saint-Julien	15
5338	114	\N	Marcenat	15
5339	116	\N	Marchastel	15
5340	117	\N	Marcolès	15
5341	118	\N	Marmanhac	15
5342	119	\N	Massiac	15
5343	120	\N	Mauriac	15
5344	121	\N	Maurines	15
5345	122	\N	Maurs	15
5346	123	\N	Méallet	15
5347	124	\N	Menet	15
5348	125	\N	Mentières	15
5349	126	\N	Molèdes	15
5350	127	\N	Molompize	15
5351	128	\N	Monselie	15
5352	129	\N	Montboudif	15
5353	130	\N	Montchamp	15
5354	131	\N	Monteil	15
5355	132	\N	Montgreleix	15
5356	133	\N	Montmurat	15
5357	134	\N	Montsalvy	15
5358	135	\N	Montvert	15
5359	136	\N	Mourjou	15
5360	137	\N	Moussages	15
5361	138	\N	Murat	15
5362	139	\N	Narnhac	15
5363	140	\N	Naucelles	15
5364	141	\N	Neussargues-Moissac	15
5365	142	\N	Neuvéglise	15
5366	143	\N	Nieudan	15
5367	144	\N	Omps	15
5368	145	\N	Oradour	15
5369	146	\N	Pailherols	15
5370	147	\N	Parlan	15
5371	148	\N	Paulhac	15
5372	149	\N	Paulhenc	15
5373	150	\N	Pers	15
5374	151	\N	Peyrusse	15
5375	152	\N	Pierrefort	15
5376	153	\N	Pleaux	15
5377	154	\N	Polminhac	15
5378	155	\N	Pradiers	15
5379	156	\N	Prunet	15
5380	157	\N	Quézac	15
5381	158	\N	Rageade	15
5382	159	\N	Raulhac	15
5383	160	\N	Reilhac	15
5384	161	\N	Rézentières	15
5385	162	\N	Riom-ès-Montagnes	15
5386	163	\N	Roannes-Saint-Mary	15
5387	164	\N	Roffiac	15
5388	165	\N	Rouffiac	15
5389	166	\N	Roumégoux	15
5390	167	\N	Rouziers	15
5391	168	\N	Ruynes-en-Margeride	15
5392	169	\N	Saignes	15
5393	170	\N	Saint-Amandin	15
5394	171	\N	Sainte-Anastasie	15
5395	172	\N	Saint-Antoine	15
5396	173	\N	Saint-Bonnet-de-Condat	15
5397	174	\N	Saint-Bonnet-de-Salers	15
5398	175	\N	Saint-Cernin	15
5399	176	\N	Saint-Chamant	15
5400	178	\N	Saint-Cirgues-de-Jordanne	15
5401	179	\N	Saint-Cirgues-de-Malbert	15
5402	180	\N	Saint-Clément	15
5403	181	\N	Saint-Constant	15
5404	182	\N	Saint-Étienne-Cantalès	15
5405	183	\N	Saint-Étienne-de-Carlat	15
5406	184	\N	Saint-Étienne-de-Maurs	15
5407	185	\N	Saint-Étienne-de-Chomeil	15
5408	186	\N	Sainte-Eulalie	15
5409	187	\N	Saint-Flour	15
5410	188	\N	Saint-Georges	15
5411	189	\N	Saint-Gérons	15
5412	190	\N	Saint-Hippolyte	15
5413	191	\N	Saint-Illide	15
5414	192	\N	Saint-Jacques-des-Blats	15
5415	194	\N	Saint-Julien-de-Toursac	15
5416	195	\N	Saint-Just	15
5417	196	\N	Saint-Mamet-la-Salvetat	15
5418	197	\N	Saint-Marc	15
5419	198	\N	Sainte-Marie	15
5420	199	\N	Saint-Martial	15
5421	200	\N	Saint-Martin-Cantalès	15
5422	201	\N	Saint-Martin-sous-Vigouroux	15
5423	202	\N	Saint-Martin-Valmeroux	15
5424	203	\N	Saint-Mary-le-Plain	15
5425	204	\N	Saint-Paul-des-Landes	15
5426	205	\N	Saint-Paul-de-Salers	15
5427	206	\N	Saint-Pierre	15
5428	207	\N	Saint-Poncy	15
5429	208	\N	Saint-Projet-de-Salers	15
5430	209	\N	Saint-Rémy-de-Chaudes-Aigues	15
5431	211	\N	Saint-Santin-Cantalès	15
5432	212	\N	Saint-Santin-de-Maurs	15
5433	213	\N	Saint-Saturnin	15
5434	214	\N	Saint-Saury	15
5435	215	\N	Saint-Simon	15
5436	216	\N	Saint-Urcize	15
5437	217	\N	Saint-Victor	15
5438	218	\N	Saint-Vincent-de-Salers	15
5439	219	\N	Salers	15
5440	220	\N	Salins	15
5441	221	\N	Sansac-de-Marmiesse	15
5442	222	\N	Sansac-Veinazès	15
5443	223	\N	Sauvat	15
5444	224	\N	Ségalassière	15
5445	225	\N	Ségur-les-Villas	15
5446	226	\N	Sénezergues	15
5447	227	\N	Sériers	15
5448	228	\N	Siran	15
5449	229	\N	Soulages	15
5450	230	\N	Sourniac	15
5451	231	\N	Talizat	15
5452	232	\N	Tanavelle	15
5453	233	\N	Teissières-de-Cornet	15
5454	234	\N	Teissières-lès-Bouliès	15
5455	235	\N	Ternes	15
5456	236	\N	Thiézac	15
5457	237	\N	Tiviers	15
5458	238	\N	Tournemire	15
5459	240	\N	Trémouille	15
5460	241	\N	Trinitat	15
5461	242	\N	Trioulou	15
5462	243	\N	Trizac	15
5463	244	\N	Ussel	15
5464	245	\N	Vabres	15
5465	246	\N	Valette	15
5466	247	\N	Valjouze	15
5467	248	\N	Valuéjols	15
5468	249	\N	Vaulmier	15
5469	250	\N	Vebret	15
5470	251	\N	Védrines-Saint-Loup	15
5471	252	\N	Velzic	15
5472	253	\N	Vernols	15
5473	254	\N	Veyrières	15
5474	255	\N	Vézac	15
5475	256	\N	Vèze	15
5476	257	\N	Vezels-Roussy	15
5477	258	\N	Vic-sur-Cère	15
5478	259	\N	Vieillespesse	15
5479	260	\N	Vieillevie	15
5480	261	\N	Vigean	15
5481	262	\N	Villedieu	15
5482	263	\N	Virargues	15
5483	264	\N	Vitrac	15
5484	265	\N	Ydes	15
5485	266	\N	Yolet	15
5486	267	\N	Ytrac	15
5487	268	\N	Rouget	15
5488	269	\N	Besse	15
5489	001	\N	Abzac	16
5490	002	\N	Adjots	16
5491	003	\N	Agris	16
5492	004	\N	Aignes-et-Puypéroux	16
5493	005	\N	Aigre	16
5494	007	\N	Alloue	16
5495	008	\N	Ambérac	16
5496	009	\N	Ambernac	16
5497	010	\N	Ambleville	16
5498	011	\N	Anais	16
5499	012	\N	Angeac-Champagne	16
5500	013	\N	Angeac-Charente	16
5501	014	\N	Angeduc	16
5502	015	\N	Angoulême	16
5503	016	\N	Ansac-sur-Vienne	16
5504	017	\N	Anville	16
5505	018	\N	Ars	16
5506	019	\N	Asnières-sur-Nouère	16
5507	020	\N	Aubeterre-sur-Dronne	16
5508	021	\N	Aubeville	16
5509	023	\N	Aunac	16
5510	024	\N	Aussac-Vadalle	16
5511	025	\N	Baignes-Sainte-Radegonde	16
5512	026	\N	Balzac	16
5513	027	\N	Barbezières	16
5514	028	\N	Barbezieux-Saint-Hilaire	16
5515	029	\N	Bardenac	16
5516	030	\N	Barret	16
5517	031	\N	Barro	16
5518	032	\N	Bassac	16
5519	033	\N	Bayers	16
5520	034	\N	Bazac	16
5521	035	\N	Beaulieu-sur-Sonnette	16
5522	036	\N	Bécheresse	16
5523	037	\N	Bellon	16
5524	038	\N	Benest	16
5525	039	\N	Bernac	16
5526	040	\N	Berneuil	16
5527	041	\N	Bessac	16
5528	042	\N	Bessé	16
5529	043	\N	Bignac	16
5530	044	\N	Bioussac	16
5531	045	\N	Birac	16
5532	046	\N	Blanzac-Porcheresse	16
5533	047	\N	Blanzaguet-Saint-Cybard	16
5534	048	\N	Boisbreteau	16
5535	049	\N	Bonnes	16
5536	050	\N	Bonneuil	16
5537	051	\N	Bonneville	16
5538	052	\N	Bors(Canton de Montmoreau-Saint-Cybard)	16
5539	053	\N	Bors(Canton de Baignes-Sainte-Radegonde)	16
5540	054	\N	Bouchage	16
5541	055	\N	Bouëx	16
5542	056	\N	Bourg-Charente	16
5543	057	\N	Bouteville	16
5544	058	\N	Boutiers-Saint-Trojan	16
5545	059	\N	Brettes	16
5546	060	\N	Bréville	16
5547	061	\N	Brie	16
5548	062	\N	Brie-sous-Barbezieux	16
5549	063	\N	Brie-sous-Chalais	16
5550	064	\N	Brigueuil	16
5551	065	\N	Brillac	16
5552	066	\N	Brossac	16
5553	067	\N	Bunzac	16
5554	068	\N	Cellefrouin	16
5555	069	\N	Cellettes	16
5556	070	\N	Chabanais	16
5557	071	\N	Chabrac	16
5558	072	\N	Chadurie	16
5559	073	\N	Chalais	16
5560	074	\N	Challignac	16
5561	075	\N	Champagne-Vigny	16
5562	076	\N	Champagne-Mouton	16
5563	077	\N	Champmillon	16
5564	078	\N	Champniers	16
5565	079	\N	Chantillac	16
5566	081	\N	Chapelle	16
5567	082	\N	Charmant	16
5568	083	\N	Charmé	16
5569	084	\N	Charras	16
5570	085	\N	Chasseneuil-sur-Bonnieure	16
5571	086	\N	Chassenon	16
5572	087	\N	Chassiecq	16
5573	088	\N	Chassors	16
5574	089	\N	Châteaubernard	16
5575	090	\N	Châteauneuf-sur-Charente	16
5576	091	\N	Châtignac	16
5577	092	\N	Chavenat	16
5578	093	\N	Chazelles	16
5579	094	\N	Chenommet	16
5580	095	\N	Chenon	16
5581	096	\N	Cherves-Châtelars	16
5582	097	\N	Cherves-Richemont	16
5583	098	\N	Chèvrerie	16
5584	099	\N	Chillac	16
5585	100	\N	Chirac	16
5586	101	\N	Claix	16
5587	102	\N	Cognac	16
5588	103	\N	Combiers	16
5589	104	\N	Condac	16
5590	105	\N	Condéon	16
5591	106	\N	Confolens	16
5592	107	\N	Coulgens	16
5593	108	\N	Coulonges	16
5594	109	\N	Courbillac	16
5595	110	\N	Courcôme	16
5596	111	\N	Courgeac	16
5597	112	\N	Courlac	16
5598	113	\N	Couronne	16
5599	114	\N	Couture	16
5600	115	\N	Cressac-Saint-Genis	16
5601	116	\N	Criteuil-la-Magdeleine	16
5602	117	\N	Curac	16
5603	118	\N	Deviat	16
5604	119	\N	Dignac	16
5605	120	\N	Dirac	16
5606	121	\N	Douzat	16
5607	122	\N	Ébréon	16
5608	123	\N	Échallat	16
5609	124	\N	Écuras	16
5610	125	\N	Édon	16
5611	127	\N	Empuré	16
5612	128	\N	Épenède	16
5613	129	\N	Éraville	16
5614	130	\N	Essards	16
5615	131	\N	Esse	16
5616	132	\N	Étagnac	16
5617	133	\N	Étriac	16
5618	134	\N	Exideuil	16
5619	135	\N	Eymouthiers	16
5620	136	\N	Faye	16
5621	137	\N	Feuillade	16
5622	138	\N	Fléac	16
5623	139	\N	Fleurac	16
5624	140	\N	Fontclaireau	16
5625	141	\N	Fontenille	16
5626	142	\N	Forêt-de-Tessé	16
5627	143	\N	Fouquebrune	16
5628	144	\N	Fouqueure	16
5629	145	\N	Foussignac	16
5630	146	\N	Garat	16
5631	147	\N	Gardes-le-Pontaroux	16
5632	148	\N	Genac	16
5633	149	\N	Genouillac	16
5634	150	\N	Gensac-la-Pallue	16
5635	151	\N	Genté	16
5636	152	\N	Gimeux	16
5637	153	\N	Gondeville	16
5638	154	\N	Gond-Pontouvre	16
5639	155	\N	Gours	16
5640	156	\N	Gourville	16
5641	157	\N	Grand-Madieu	16
5642	158	\N	Grassac	16
5643	160	\N	Guimps	16
5644	161	\N	Guizengeard	16
5645	162	\N	Gurat	16
5646	163	\N	Hiersac	16
5647	164	\N	Hiesse	16
5648	165	\N	Houlette	16
5649	166	\N	Isle-d'Espagnac	16
5650	167	\N	Jarnac	16
5651	168	\N	Jauldes	16
5652	169	\N	Javrezac	16
5653	170	\N	Juignac	16
5654	171	\N	Juillac-le-Coq	16
5655	172	\N	Juillaguet	16
5656	173	\N	Juillé	16
5657	174	\N	Julienne	16
5658	175	\N	Jurignac	16
5659	176	\N	Lachaise	16
5660	177	\N	Ladiville	16
5661	178	\N	Lagarde-sur-le-Né	16
5662	179	\N	Lamérac	16
5663	180	\N	Laprade	16
5664	181	\N	Lessac	16
5665	182	\N	Lesterps	16
5666	183	\N	Lésignac-Durand	16
5667	184	\N	Lichères	16
5668	185	\N	Ligné	16
5669	186	\N	Lignières-Sonneville	16
5670	187	\N	Linars	16
5671	188	\N	Lindois	16
5672	189	\N	Londigny	16
5673	190	\N	Longré	16
5674	191	\N	Lonnes	16
5675	192	\N	Roumazières-Loubert	16
5676	193	\N	Louzac-Saint-André	16
5677	194	\N	Lupsault	16
5678	195	\N	Lussac	16
5679	196	\N	Luxé	16
5680	197	\N	Magdeleine	16
5681	198	\N	Magnac-Lavalette-Villars	16
5682	199	\N	Magnac-sur-Touvre	16
5683	200	\N	Maine-de-Boixe	16
5684	201	\N	Mainfonds	16
5685	202	\N	Mainxe	16
5686	203	\N	Mainzac	16
5687	204	\N	Malaville	16
5688	205	\N	Manot	16
5689	206	\N	Mansle	16
5690	207	\N	Marcillac-Lanville	16
5691	208	\N	Mareuil	16
5692	209	\N	Marillac-le-Franc	16
5693	210	\N	Marsac	16
5694	211	\N	Marthon	16
5695	212	\N	Massignac	16
5696	213	\N	Mazerolles	16
5697	214	\N	Mazières	16
5698	215	\N	Médillac	16
5699	216	\N	Mérignac	16
5700	217	\N	Merpins	16
5701	218	\N	Mesnac	16
5702	220	\N	Métairies	16
5703	221	\N	Mons	16
5704	222	\N	Montboyer	16
5705	223	\N	Montbron	16
5706	224	\N	Montchaude	16
5707	225	\N	Montembuf	16
5708	226	\N	Montignac-Charente	16
5709	227	\N	Montignac-le-Coq	16
5710	228	\N	Montigné	16
5711	229	\N	Montjean	16
5712	230	\N	Montmoreau-Saint-Cybard	16
5713	231	\N	Montrollet	16
5714	232	\N	Mornac	16
5715	233	\N	Mosnac	16
5716	234	\N	Moulidars	16
5717	236	\N	Mouthiers-sur-Boëme	16
5718	237	\N	Mouton	16
5719	238	\N	Moutonneau	16
5720	239	\N	Mouzon	16
5721	240	\N	Nabinaud	16
5722	241	\N	Nanclars	16
5723	242	\N	Nanteuil-en-Vallée	16
5724	243	\N	Nercillac	16
5725	244	\N	Nersac	16
5726	245	\N	Nieuil	16
5727	246	\N	Nonac	16
5728	247	\N	Nonaville	16
5729	248	\N	Oradour	16
5730	249	\N	Oradour-Fanais	16
5731	250	\N	Orgedeuil	16
5732	251	\N	Oriolles	16
5733	252	\N	Orival	16
5734	253	\N	Paizay-Naudouin-Embourie	16
5735	254	\N	Palluaud	16
5736	255	\N	Parzac	16
5737	256	\N	Passirac	16
5738	257	\N	Péreuil	16
5739	258	\N	Pérignac	16
5740	259	\N	Péruse	16
5741	260	\N	Pillac	16
5742	261	\N	Pins	16
5743	262	\N	Plaizac	16
5744	263	\N	Plassac-Rouffiac	16
5745	264	\N	Pleuville	16
5746	267	\N	Poullignac	16
5747	268	\N	Poursac	16
5748	269	\N	Pranzac	16
5749	270	\N	Pressignac	16
5750	271	\N	Puymoyen	16
5751	272	\N	Puyréaux	16
5752	273	\N	Raix	16
5753	274	\N	Rancogne	16
5754	275	\N	Ranville-Breuillaud	16
5755	276	\N	Reignac	16
5756	277	\N	Réparsac	16
5757	279	\N	Rioux-Martin	16
5758	280	\N	Rivières	16
5759	281	\N	Rochefoucauld	16
5760	282	\N	Rochette	16
5761	283	\N	Ronsenac	16
5762	284	\N	Rouffiac	16
5763	285	\N	Rougnac	16
5764	286	\N	Rouillac	16
5765	287	\N	Roullet-Saint-Estèphe	16
5766	289	\N	Roussines	16
5767	290	\N	Rouzède	16
5768	291	\N	Ruelle-sur-Touvre	16
5769	292	\N	Ruffec	16
5770	293	\N	Saint-Adjutory	16
5771	294	\N	Saint-Amant	16
5772	295	\N	Saint-Amant-de-Boixe	16
5773	296	\N	Saint-Amant-de-Bonnieure	16
5774	297	\N	Graves-Saint-Amant	16
5775	298	\N	Saint-Amant-de-Nouère	16
5776	300	\N	Saint-Angeau	16
5777	301	\N	Saint-Aulais-la-Chapelle	16
5778	302	\N	Saint-Avit	16
5779	303	\N	Saint-Bonnet	16
5780	304	\N	Saint-Brice	16
5781	306	\N	Saint-Christophe	16
5782	307	\N	Saint-Ciers-sur-Bonnieure	16
5783	308	\N	Saint-Claud	16
5784	309	\N	Sainte-Colombe	16
5785	310	\N	Saint-Coutant	16
5786	312	\N	Saint-Cybardeaux	16
5787	314	\N	Saint-Eutrope	16
5788	315	\N	Saint-Félix	16
5789	316	\N	Saint-Fort-sur-le-Né	16
5790	317	\N	Saint-Fraigne	16
5791	318	\N	Saint-Front	16
5792	320	\N	Saint-Genis-d'Hiersac	16
5793	321	\N	Saint-Georges	16
5794	322	\N	Saint-Germain-de-Confolens	16
5795	323	\N	Saint-Germain-de-Montbron	16
5796	325	\N	Saint-Gourson	16
5797	326	\N	Saint-Groux	16
5798	328	\N	Saint-Laurent-de-Belzagot	16
5799	329	\N	Saint-Laurent-de-Céris	16
5800	330	\N	Saint-Laurent-de-Cognac	16
5801	331	\N	Saint-Laurent-des-Combes	16
5802	332	\N	Saint-Léger	16
5803	334	\N	Saint-Martial	16
5804	335	\N	Saint-Martin-du-Clocher	16
5805	336	\N	Saint-Mary	16
5806	337	\N	Saint-Maurice-des-Lions	16
5807	338	\N	Saint-Médard	16
5808	339	\N	Auge-Saint-Médard	16
5809	340	\N	Saint-Même-les-Carrières	16
5810	341	\N	Saint-Michel	16
5811	342	\N	Saint-Palais-du-Né	16
5812	343	\N	Saint-Preuil	16
5813	344	\N	Saint-Projet-Saint-Constant	16
5814	345	\N	Saint-Quentin-sur-Charente	16
5815	346	\N	Saint-Quentin-de-Chalais	16
5816	347	\N	Saint-Romain	16
5817	348	\N	Saint-Saturnin	16
5818	349	\N	Sainte-Sévère	16
5819	350	\N	Saint-Séverin	16
5820	351	\N	Saint-Simeux	16
5821	352	\N	Saint-Simon	16
5822	353	\N	Saint-Sornin	16
5823	354	\N	Sainte-Souline	16
5824	355	\N	Saint-Sulpice-de-Cognac	16
5825	356	\N	Saint-Sulpice-de-Ruffec	16
5826	357	\N	Saint-Vallier	16
5827	358	\N	Saint-Yrieix-sur-Charente	16
5828	359	\N	Salles-d'Angles	16
5829	360	\N	Salles-de-Barbezieux	16
5830	361	\N	Salles-de-Villefagnan	16
5831	362	\N	Salles-Lavalette	16
5832	363	\N	Saulgond	16
5833	364	\N	Sauvagnac	16
5834	365	\N	Sauvignac	16
5835	366	\N	Segonzac	16
5836	368	\N	Sers	16
5837	369	\N	Sigogne	16
5838	370	\N	Sireuil	16
5839	371	\N	Sonneville	16
5840	372	\N	Souffrignac	16
5841	373	\N	Souvigné	16
5842	374	\N	Soyaux	16
5843	375	\N	Suaux	16
5844	376	\N	Suris	16
5845	377	\N	Tâche	16
5846	378	\N	Taizé-Aizie	16
5847	379	\N	Taponnat-Fleurignac	16
5848	380	\N	Tâtre	16
5849	381	\N	Theil-Rabier	16
5850	382	\N	Torsac	16
5851	383	\N	Tourriers	16
5852	384	\N	Touvérac	16
5853	385	\N	Touvre	16
5854	386	\N	Touzac	16
5855	387	\N	Triac-Lautrait	16
5856	388	\N	Trois-Palis	16
5857	389	\N	Turgon	16
5858	390	\N	Tusson	16
5859	391	\N	Tuzie	16
5860	392	\N	Valence	16
5861	393	\N	Vars	16
5862	394	\N	Vaux-Lavalette	16
5863	395	\N	Vaux-Rouillac	16
5864	396	\N	Ventouse	16
5865	397	\N	Verdille	16
5866	398	\N	Verneuil	16
5867	399	\N	Verrières	16
5868	400	\N	Verteuil-sur-Charente	16
5869	401	\N	Vervant	16
5870	402	\N	Vibrac	16
5871	403	\N	Vieux-Cérier	16
5872	404	\N	Vieux-Ruffec	16
5873	405	\N	Vignolles	16
5874	406	\N	Vilhonneur	16
5875	408	\N	Villebois-Lavalette	16
5876	409	\N	Villefagnan	16
5877	410	\N	Villegats	16
5878	411	\N	Villejésus	16
5879	412	\N	Villejoubert	16
5880	413	\N	Villiers-le-Roux	16
5881	414	\N	Villognon	16
5882	415	\N	Vindelle	16
5883	416	\N	Vitrac-Saint-Vincent	16
5884	417	\N	Viville	16
5885	418	\N	Vuil-et-Giget	16
5886	419	\N	Vouharte	16
5887	420	\N	Voulgézac	16
5888	421	\N	Vouthon	16
5889	422	\N	Vouzan	16
5890	423	\N	Xambes	16
5891	424	\N	Yviers	16
5892	425	\N	Yvrac-et-Malleyrand	16
5893	002	\N	Agudelle	17
5894	003	\N	Aigrefeuille-d'Aunis	17
5895	004	\N	Île-d'Aix	17
5896	005	\N	Allas-Bocage	17
5897	006	\N	Allas-Champagne	17
5898	007	\N	Anais	17
5899	008	\N	Andilly	17
5900	009	\N	Angliers	17
5901	010	\N	Angoulins	17
5902	011	\N	Annepont	17
5903	012	\N	Annezay	17
5904	013	\N	Antezant-la-Chapelle	17
5905	015	\N	Arces	17
5906	016	\N	Archiac	17
5907	017	\N	Archingeay	17
5908	018	\N	Ardillières	17
5909	019	\N	Ars-en-Ré	17
5910	020	\N	Arthenac	17
5911	021	\N	Arvert	17
5912	022	\N	Asnières-la-Giraud	17
5913	023	\N	Aujac	17
5914	024	\N	Aulnay	17
5915	025	\N	Aumagne	17
5916	026	\N	Authon-Ébéon	17
5917	027	\N	Avy	17
5918	028	\N	Aytré	17
5919	029	\N	Bagnizeau	17
5920	030	\N	Balanzac	17
5921	031	\N	Ballans	17
5922	032	\N	Ballon	17
5923	033	\N	Barde	17
5924	034	\N	Barzan	17
5925	035	\N	Bazauges	17
5926	036	\N	Beaugeay	17
5927	037	\N	Beauvais-sur-Matha	17
5928	038	\N	Bedenac	17
5929	039	\N	Belluire	17
5930	040	\N	Benâte	17
5931	041	\N	Benon	17
5932	042	\N	Bercloux	17
5933	043	\N	Bernay-Saint-Martin	17
5934	044	\N	Berneuil	17
5935	045	\N	Beurlay	17
5936	046	\N	Bignay	17
5937	047	\N	Biron	17
5938	048	\N	Blanzac-lès-Matha	17
5939	049	\N	Blanzay-sur-Boutonne	17
5940	050	\N	Bois	17
5941	051	\N	Bois-Plage-en-Ré	17
5942	052	\N	Boisredon	17
5943	053	\N	Bords	17
5944	054	\N	Boresse-et-Martron	17
5945	055	\N	Boscamnant	17
5946	056	\N	Bougneau	17
5947	057	\N	Bouhet	17
5948	058	\N	Bourcefranc-le-Chapus	17
5949	059	\N	Bourgneuf	17
5950	060	\N	Boutenac-Touvent	17
5951	061	\N	Bran	17
5952	062	\N	Bresdon	17
5953	063	\N	Breuil-la-Réorte	17
5954	064	\N	Breuillet	17
5955	065	\N	Breuil-Magné	17
5956	066	\N	Brie-sous-Archiac	17
5957	067	\N	Brie-sous-Matha	17
5958	068	\N	Brie-sous-Mortagne	17
5959	069	\N	Brives-sur-Charente	17
5960	070	\N	Brizambourg	17
5961	071	\N	Brousse	17
5962	072	\N	Burie	17
5963	073	\N	Bussac-sur-Charente	17
5964	074	\N	Bussac-Forêt	17
5965	075	\N	Cabariot	17
5966	076	\N	Celles	17
5967	077	\N	Cercoux	17
5968	078	\N	Chadenac	17
5969	079	\N	Chaillevette	17
5970	080	\N	Chambon	17
5971	081	\N	Chamouillac	17
5972	082	\N	Champagnac	17
5973	083	\N	Champagne	17
5974	084	\N	Champagnolles	17
5975	085	\N	Champdolent	17
5976	086	\N	Chaniers	17
5977	087	\N	Chantemerle-sur-la-Soie	17
5978	089	\N	Chapelle-des-Pots	17
5979	091	\N	Charron	17
5980	092	\N	Chartuzac	17
5981	093	\N	Château-d'Oléron	17
5982	094	\N	Châtelaillon-Plage	17
5983	095	\N	Chatenet	17
5984	096	\N	Chaunac	17
5985	097	\N	Chay	17
5986	098	\N	Chenac-Saint-Seurin-d'Uzet	17
5987	099	\N	Chepniers	17
5988	100	\N	Chérac	17
5989	101	\N	Cherbonnières	17
5990	102	\N	Chermignac	17
5991	103	\N	Chervettes	17
5992	104	\N	Chevanceaux	17
5993	105	\N	Chives	17
5994	106	\N	Cierzac	17
5995	107	\N	Ciré-d'Aunis	17
5996	108	\N	Clam	17
5997	109	\N	Clavette	17
5998	110	\N	Clérac	17
5999	111	\N	Clion	17
6000	112	\N	Clisse	17
6001	113	\N	Clotte	17
6002	114	\N	Coivert	17
6003	115	\N	Colombiers	17
6004	116	\N	Consac	17
6005	117	\N	Contré	17
6006	118	\N	Corignac	17
6007	119	\N	Corme-Écluse	17
6008	120	\N	Corme-Royal	17
6009	121	\N	Couarde-sur-Mer	17
6010	122	\N	Coulonges	17
6011	124	\N	Courant	17
6012	125	\N	Courcelles	17
6013	126	\N	Courcerac	17
6014	127	\N	Courçon	17
6015	128	\N	Courcoury	17
6016	129	\N	Courpignac	17
6017	130	\N	Coux	17
6018	131	\N	Cozes	17
6019	132	\N	Cramchaban	17
6020	133	\N	Cravans	17
6021	134	\N	Crazannes	17
6022	135	\N	Cressé	17
6023	136	\N	Croix-Chapeau	17
6024	137	\N	Croix-Comtesse	17
6025	138	\N	Dampierre-sur-Boutonne	17
6026	139	\N	Duil-sur-le-Mignon	17
6027	140	\N	Dolus-d'Oléron	17
6028	141	\N	Dompierre-sur-Charente	17
6029	142	\N	Dompierre-sur-Mer	17
6030	143	\N	Douhet	17
6031	145	\N	Échebrune	17
6032	146	\N	Échillais	17
6033	147	\N	Écoyeux	17
6034	148	\N	Écurat	17
6035	149	\N	Éduts	17
6036	150	\N	Églises-d'Argenteuil	17
6037	151	\N	Éguille	17
6038	152	\N	Épargnes	17
6039	153	\N	Esnandes	17
6040	154	\N	Essards	17
6041	155	\N	Étaules	17
6042	156	\N	Expiremont	17
6043	157	\N	Fenioux	17
6044	158	\N	Ferrières	17
6045	159	\N	Fléac-sur-Seugne	17
6046	160	\N	Floirac	17
6047	161	\N	Flotte	17
6048	162	\N	Fontaine-Chalendray	17
6049	163	\N	Fontaines-d'Ozillac	17
6050	164	\N	Fontcouverte	17
6051	165	\N	Fontenet	17
6052	166	\N	Forges	17
6053	167	\N	Fouilloux	17
6054	168	\N	Fouras	17
6055	169	\N	Frédière	17
6056	171	\N	Geay	17
6057	172	\N	Gémozac	17
6058	173	\N	Genétouze	17
6059	174	\N	Genouillé	17
6060	175	\N	Germignac	17
6061	176	\N	Gibourne	17
6062	177	\N	Gicq	17
6063	178	\N	Givrezac	17
6064	179	\N	Gonds	17
6065	180	\N	Gourvillette	17
6066	181	\N	Grandjean	17
6067	182	\N	Grève-sur-Mignon	17
6068	183	\N	Grézac	17
6069	184	\N	Gripperie-Saint-Symphorien	17
6070	185	\N	Gua	17
6071	186	\N	Gué-d'Alleré	17
6072	187	\N	Guitinières	17
6073	188	\N	Haimps	17
6074	189	\N	Hiers-Brouage	17
6075	190	\N	Houmeau	17
6076	191	\N	Jard	17
6077	192	\N	Jarnac-Champagne	17
6078	193	\N	Jarne	17
6079	194	\N	Jarrie	17
6080	195	\N	Jarrie-Audouin	17
6081	196	\N	Jazennes	17
6082	197	\N	Jonzac	17
6083	198	\N	Juicq	17
6084	199	\N	Jussas	17
6085	200	\N	Lagord	17
6086	201	\N	Laigne	17
6087	202	\N	Landes	17
6088	203	\N	Landrais	17
6089	204	\N	Léoville	17
6090	205	\N	Loire-les-Marais	17
6091	206	\N	Loiré-sur-Nie	17
6092	207	\N	Loix	17
6093	208	\N	Longèves	17
6094	209	\N	Lonzac	17
6095	210	\N	Lorignac	17
6096	211	\N	Loulay	17
6097	212	\N	Louzignac	17
6098	213	\N	Lozay	17
6099	214	\N	Luchat	17
6100	215	\N	Lussac	17
6101	216	\N	Lussant	17
6102	217	\N	Macqueville	17
6103	218	\N	Marans	17
6104	219	\N	Marennes	17
6105	220	\N	Marignac	17
6106	221	\N	Marsais	17
6107	222	\N	Marsilly	17
6108	223	\N	Massac	17
6109	224	\N	Matha	17
6110	225	\N	Mathes	17
6111	226	\N	Mazeray	17
6112	227	\N	Mazerolles	17
6113	228	\N	Médis	17
6114	229	\N	Mérignac	17
6115	230	\N	Meschers-sur-Gironde	17
6116	231	\N	Messac	17
6117	232	\N	Meursac	17
6118	233	\N	Meux	17
6119	234	\N	Migré	17
6120	235	\N	Migron	17
6121	236	\N	Mirambeau	17
6122	237	\N	Moëze	17
6123	238	\N	Moings	17
6124	239	\N	Mons	17
6125	240	\N	Montendre	17
6126	241	\N	Montguyon	17
6127	242	\N	Montils	17
6128	243	\N	Montlieu-la-Garde	17
6129	244	\N	Montpellier-de-Médillan	17
6130	245	\N	Montroy	17
6131	246	\N	Moragne	17
6132	247	\N	Mornac-sur-Seudre	17
6133	248	\N	Mortagne-sur-Gironde	17
6134	249	\N	Mortiers	17
6135	250	\N	Mosnac	17
6136	252	\N	Mung	17
6137	253	\N	Muron	17
6138	254	\N	Nachamps	17
6139	255	\N	Nancras	17
6140	256	\N	Nantillé	17
6141	257	\N	Néré	17
6142	258	\N	Neuillac	17
6143	259	\N	Neulles	17
6144	260	\N	Neuvicq	17
6145	261	\N	Neuvicq-le-Château	17
6146	262	\N	Nieul-lès-Saintes	17
6147	263	\N	Nieul-le-Virouil	17
6148	264	\N	Nieul-sur-Mer	17
6149	265	\N	Nieulle-sur-Seudre	17
6150	266	\N	Nouillers	17
6151	267	\N	Nuaillé-d'Aunis	17
6152	268	\N	Nuaillé-sur-Boutonne	17
6153	269	\N	Orignolles	17
6154	270	\N	Ozillac	17
6155	271	\N	Paillé	17
6156	272	\N	Péré	17
6157	273	\N	Pérignac	17
6158	274	\N	Périgny	17
6159	275	\N	Pessines	17
6160	276	\N	Pin	17
6161	277	\N	Saint-Denis-du-Pin	17
6162	278	\N	Pisany	17
6163	279	\N	Plassac	17
6164	280	\N	Plassay	17
6165	281	\N	Polignac	17
6166	282	\N	Pommiers-Moulons	17
6167	283	\N	Pons	17
6168	284	\N	Pont-l'Abbé-d'Arnoult	17
6169	285	\N	Port-d'Envaux	17
6170	286	\N	Portes-en-Ré	17
6171	287	\N	Pouillac	17
6172	288	\N	Poursay-Garnaud	17
6173	289	\N	Préguillac	17
6174	290	\N	Prignac	17
6175	291	\N	Puilboreau	17
6176	292	\N	Puy-du-Lac	17
6177	293	\N	Puyravault	17
6178	294	\N	Puyrolland	17
6179	295	\N	Réaux	17
6180	296	\N	Rétaud	17
6181	297	\N	Rivedoux-Plage	17
6182	298	\N	Rioux	17
6183	299	\N	Rochefort	17
6184	300	\N	Rochelle	17
6185	301	\N	Romazières	17
6186	302	\N	Romegoux	17
6187	303	\N	Ronde	17
6188	304	\N	Rouffiac	17
6189	305	\N	Rouffignac	17
6190	306	\N	Royan	17
6191	307	\N	Sablonceaux	17
6192	308	\N	Saint-Agnant	17
6193	309	\N	Saint-Aigulin	17
6194	310	\N	Saint-André-de-Lidon	17
6195	311	\N	Saint-Augustin	17
6196	312	\N	Saint-Bonnet-sur-Gironde	17
6197	313	\N	Saint-Bris-des-Bois	17
6198	314	\N	Saint-Césaire	17
6199	315	\N	Saint-Christophe	17
6200	316	\N	Saint-Ciers-Champagne	17
6201	317	\N	Saint-Ciers-du-Taillon	17
6202	318	\N	Saint-Clément-des-Baleines	17
6203	319	\N	Sainte-Colombe	17
6204	320	\N	Saint-Coutant-le-Grand	17
6205	321	\N	Saint-Crépin	17
6206	322	\N	Saint-Cyr-du-Doret	17
6207	323	\N	Saint-Denis-d'Oléron	17
6208	324	\N	Saint-Dizant-du-Bois	17
6209	325	\N	Saint-Dizant-du-Gua	17
6210	326	\N	Saint-Eugène	17
6211	327	\N	Saint-Félix	17
6212	328	\N	Saint-Fort-sur-Gironde	17
6213	329	\N	Saint-Froult	17
6214	330	\N	Sainte-Gemme	17
6215	331	\N	Saint-Genis-de-Saintonge	17
6216	332	\N	Saint-Georges-Antignac	17
6217	333	\N	Saint-Georges-de-Didonne	17
6218	334	\N	Saint-Georges-de-Longuepierre	17
6219	335	\N	Saint-Georges-des-Agoûts	17
6220	336	\N	Saint-Georges-des-Coteaux	17
6221	337	\N	Saint-Georges-d'Oléron	17
6222	338	\N	Saint-Georges-du-Bois	17
6223	339	\N	Saint-Germain-de-Lusignan	17
6224	340	\N	Saint-Germain-de-Marencennes	17
6225	341	\N	Saint-Germain-de-Vibrac	17
6226	342	\N	Saint-Germain-du-Seudre	17
6227	343	\N	Saint-Grégoire-d'Ardennes	17
6228	344	\N	Saint-Hilaire-de-Villefranche	17
6229	345	\N	Saint-Hilaire-du-Bois	17
6230	346	\N	Saint-Hippolyte	17
6231	347	\N	Saint-Jean-d'Angély	17
6232	348	\N	Saint-Jean-d'Angle	17
6233	349	\N	Saint-Jean-de-Liversay	17
6234	350	\N	Saint-Julien-de-l'Escap	17
6235	351	\N	Saint-Just-Luzac	17
6236	352	\N	Saint-Laurent-de-la-Barrière	17
6237	353	\N	Saint-Laurent-de-la-Prée	17
6238	354	\N	Saint-Léger	17
6239	355	\N	Sainte-Lheurine	17
6240	356	\N	Saint-Loup	17
6241	357	\N	Saint-Maigrin	17
6242	358	\N	Saint-Mandé-sur-Brédoire	17
6243	359	\N	Saint-Mard	17
6244	360	\N	Sainte-Marie-de-Ré	17
6245	361	\N	Saint-Martial	17
6246	362	\N	Saint-Martial-de-Mirambeau	17
6247	363	\N	Saint-Martial-de-Vitaterne	17
6248	364	\N	Saint-Martial-sur-Né	17
6249	365	\N	Saint-Martin-d'Ary	17
6250	366	\N	Saint-Martin-de-Coux	17
6251	367	\N	Saint-Martin-de-Juillers	17
6252	369	\N	Saint-Martin-de-Ré	17
6253	371	\N	Saint-Maurice-de-Tavernole	17
6254	372	\N	Saint-Médard	17
6255	373	\N	Saint-Médard-d'Aunis	17
6256	374	\N	Sainte-Même	17
6257	375	\N	Saint-Nazaire-sur-Charente	17
6258	376	\N	Saint-Ouen-d'Aunis	17
6259	377	\N	Saint-Ouen	17
6260	378	\N	Saint-Palais-de-Négrignac	17
6261	379	\N	Saint-Palais-de-Phiolin	17
6262	380	\N	Saint-Palais-sur-Mer	17
6263	381	\N	Saint-Pardoult	17
6264	382	\N	Saint-Pierre-d'Amilly	17
6265	383	\N	Saint-Pierre-de-Juillers	17
6266	384	\N	Saint-Pierre-de-l'Île	17
6267	385	\N	Saint-Pierre-d'Oléron	17
6268	386	\N	Saint-Pierre-du-Palais	17
6269	387	\N	Saint-Porchaire	17
6270	388	\N	Saint-Quantin-de-Rançanne	17
6271	389	\N	Sainte-Radegonde	17
6272	390	\N	Sainte-Ramée	17
6273	391	\N	Saint-Rogatien	17
6274	392	\N	Saint-Romain-sur-Gironde	17
6275	393	\N	Saint-Romain-de-Benet	17
6276	394	\N	Saint-Saturnin-du-Bois	17
6277	395	\N	Saint-Sauvant	17
6278	396	\N	Saint-Sauveur-d'Aunis	17
6279	397	\N	Saint-Savinien	17
6280	398	\N	Saint-Seurin-de-Palenne	17
6281	400	\N	Saint-Sever-de-Saintonge	17
6282	401	\N	Saint-Séverin-sur-Boutonne	17
6283	402	\N	Saint-Sigismond-de-Clermont	17
6284	403	\N	Saint-Simon-de-Bordes	17
6285	404	\N	Saint-Simon-de-Pellouaille	17
6286	405	\N	Saint-Sorlin-de-Conac	17
6287	406	\N	Saint-Sornin	17
6288	407	\N	Sainte-Soulle	17
6289	408	\N	Saint-Sulpice-d'Arnoult	17
6290	409	\N	Saint-Sulpice-de-Royan	17
6291	410	\N	Saint-Thomas-de-Conac	17
6292	411	\N	Saint-Trojan-les-Bains	17
6293	412	\N	Saint-Vaize	17
6294	413	\N	Saint-Vivien	17
6295	414	\N	Saint-Xandre	17
6296	415	\N	Saintes	17
6297	416	\N	Saleignes	17
6298	417	\N	Salignac-de-Mirambeau	17
6299	418	\N	Salignac-sur-Charente	17
6300	420	\N	Salles-sur-Mer	17
6301	421	\N	Saujon	17
6302	422	\N	Seigné	17
6303	423	\N	Semillac	17
6304	424	\N	Semoussac	17
6305	425	\N	Semussac	17
6306	426	\N	Seure	17
6307	427	\N	Siecq	17
6308	428	\N	Sonnac	17
6309	429	\N	Soubise	17
6310	430	\N	Soubran	17
6311	431	\N	Soulignonne	17
6312	432	\N	Souméras	17
6313	433	\N	Sousmoulins	17
6314	434	\N	Surgères	17
6315	435	\N	Taillant	17
6316	436	\N	Taillebourg	17
6317	437	\N	Talmont-sur-Gironde	17
6318	438	\N	Tanzac	17
6319	439	\N	Taugon	17
6320	440	\N	Ternant	17
6321	441	\N	Tesson	17
6322	442	\N	Thaims	17
6323	443	\N	Thairé	17
6324	444	\N	Thénac	17
6325	445	\N	Thézac	17
6326	446	\N	Thors	17
6327	447	\N	Thou	17
6328	448	\N	Tonnay-Boutonne	17
6329	449	\N	Tonnay-Charente	17
6330	450	\N	Torxé	17
6331	451	\N	Touches-de-Périgny	17
6332	452	\N	Tremblade	17
6333	453	\N	Trizay	17
6334	454	\N	Tugéras-Saint-Maurice	17
6335	455	\N	Vallée	17
6336	457	\N	Vandré	17
6337	458	\N	Vanzac	17
6338	459	\N	Varaize	17
6339	460	\N	Varzay	17
6340	461	\N	Vaux-sur-Mer	17
6341	462	\N	Vénérand	17
6342	463	\N	Vergeroux	17
6343	464	\N	Vergné	17
6344	465	\N	Vergne	17
6345	466	\N	Vérines	17
6346	467	\N	Vervant	17
6347	468	\N	Vibrac	17
6348	469	\N	Villars-en-Pons	17
6349	470	\N	Villars-les-Bois	17
6350	471	\N	Villedieu	17
6351	472	\N	Villedoux	17
6352	473	\N	Villemorin	17
6353	474	\N	Villeneuve-la-Comtesse	17
6354	476	\N	Villexavier	17
6355	477	\N	Villiers-Couture	17
6356	478	\N	Vinax	17
6357	479	\N	Virollet	17
6358	480	\N	Virson	17
6359	481	\N	Voissay	17
6360	482	\N	Vouhé	17
6361	483	\N	Yves	17
6362	484	\N	Port-des-Barques	17
6363	485	\N	Grand-Village-Plage	17
6364	486	\N	Brée-les-Bains	17
6365	001	\N	Achères	18
6366	002	\N	Ainay-le-Vieil	18
6367	003	\N	Aix-d'Angillon	18
6368	004	\N	Allogny	18
6369	005	\N	Allouis	18
6370	006	\N	Annoix	18
6371	007	\N	Apremont-sur-Allier	18
6372	008	\N	Arçay	18
6373	009	\N	Arcomps	18
6374	010	\N	Ardenais	18
6375	011	\N	Argent-sur-Sauldre	18
6376	012	\N	Argenvières	18
6377	013	\N	Arpheuilles	18
6378	014	\N	Assigny	18
6379	015	\N	Aubigny-sur-Nère	18
6380	016	\N	Aubinges	18
6381	017	\N	Augy-sur-Aubois	18
6382	018	\N	Avord	18
6383	019	\N	Azy	18
6384	020	\N	Bannay	18
6385	021	\N	Bannegon	18
6386	022	\N	Barlieu	18
6387	023	\N	Baugy	18
6388	024	\N	Beddes	18
6389	025	\N	Beffes	18
6390	026	\N	Belleville-sur-Loire	18
6391	027	\N	Bengy-sur-Craon	18
6392	028	\N	Berry-Bouy	18
6393	029	\N	Bessais-le-Fromental	18
6394	030	\N	Blancafort	18
6395	031	\N	Blet	18
6396	032	\N	Boulleret	18
6397	033	\N	Bourges	18
6398	034	\N	Bouzais	18
6399	035	\N	Brécy	18
6400	036	\N	Brinay	18
6401	037	\N	Brinon-sur-Sauldre	18
6402	038	\N	Bruère-Allichamps	18
6403	039	\N	Bué	18
6404	040	\N	Bussy	18
6405	041	\N	Celette	18
6406	042	\N	Celle	18
6407	043	\N	Celle-Condé	18
6408	044	\N	Cerbois	18
6409	045	\N	Chalivoy-Milon	18
6410	046	\N	Chambon	18
6411	047	\N	Chapelle-d'Angillon	18
6412	048	\N	Chapelle-Hugon	18
6413	049	\N	Chapelle-Montlinard	18
6414	050	\N	Chapelle-Saint-Ursin	18
6415	051	\N	Chapelotte	18
6416	052	\N	Charenton-du-Cher	18
6417	053	\N	Charentonnay	18
6418	054	\N	Charly	18
6419	055	\N	Chârost	18
6420	056	\N	Chassy	18
6421	057	\N	Châteaumeillant	18
6422	058	\N	Châteauneuf-sur-Cher	18
6423	059	\N	Châtelet	18
6424	060	\N	Chaumont	18
6425	061	\N	Chaumoux-Marcilly	18
6426	062	\N	Chautay	18
6427	063	\N	Chavannes	18
6428	064	\N	Chéry	18
6429	065	\N	Chezal-Benoît	18
6430	066	\N	Civray	18
6431	067	\N	Clémont	18
6432	068	\N	Cogny	18
6433	069	\N	Colombiers	18
6434	070	\N	Concressault	18
6435	071	\N	Contres	18
6436	072	\N	Cornusse	18
6437	073	\N	Corquoy	18
6438	074	\N	Couargues	18
6439	075	\N	Cours-les-Barres	18
6440	076	\N	Coust	18
6441	077	\N	Couy	18
6442	078	\N	Crézançay-sur-Cher	18
6443	079	\N	Crézancy-en-Sancerre	18
6444	080	\N	Croisy	18
6445	081	\N	Crosses	18
6446	082	\N	Cuffy	18
6447	083	\N	Culan	18
6448	084	\N	Dampierre-en-Crot	18
6449	085	\N	Dampierre-en-Graçay	18
6450	086	\N	Drevant	18
6451	087	\N	Dun-sur-Auron	18
6452	088	\N	Ennordres	18
6453	089	\N	Épineuil-le-Fleuriel	18
6454	090	\N	Étréchy	18
6455	091	\N	Farges-Allichamps	18
6456	092	\N	Farges-en-Septaine	18
6457	093	\N	Faverdines	18
6458	094	\N	Feux	18
6459	095	\N	Flavigny	18
6460	096	\N	Foëcy	18
6461	097	\N	Fussy	18
6462	098	\N	Gardefort	18
6463	099	\N	Garigny	18
6464	100	\N	Genouilly	18
6465	101	\N	Germigny-l'Exempt	18
6466	102	\N	Givardon	18
6467	103	\N	Graçay	18
6468	104	\N	Groises	18
6469	105	\N	Gron	18
6470	106	\N	Grossouvre	18
6471	107	\N	Groutte	18
6472	108	\N	Guerche-sur-l'Aubois	18
6473	109	\N	Henrichemont	18
6474	110	\N	Herry	18
6475	111	\N	Humbligny	18
6476	112	\N	Ids-Saint-Roch	18
6477	113	\N	Ignol	18
6478	114	\N	Ineuil	18
6479	115	\N	Ivoy-le-Pré	18
6480	116	\N	Jalognes	18
6481	117	\N	Jars	18
6482	118	\N	Jouet-sur-l'Aubois	18
6483	119	\N	Jussy-Champagne	18
6484	120	\N	Jussy-le-Chaudrier	18
6485	121	\N	Lantan	18
6486	122	\N	Lapan	18
6487	123	\N	Laverdines	18
6488	124	\N	Lazenay	18
6489	125	\N	Léré	18
6490	126	\N	Levet	18
6491	127	\N	Lignières	18
6492	128	\N	Limeux	18
6493	129	\N	Lissay-Lochy	18
6494	130	\N	Loye-sur-Arnon	18
6495	131	\N	Lugny-Bourbonnais	18
6496	132	\N	Lugny-Champagne	18
6497	133	\N	Lunery	18
6498	134	\N	Lury-sur-Arnon	18
6499	135	\N	Maisonnais	18
6500	136	\N	Marçais	18
6501	137	\N	Mareuil-sur-Arnon	18
6502	138	\N	Marmagne	18
6503	139	\N	Marseilles-lès-Aubigny	18
6504	140	\N	Massay	18
6505	141	\N	Mehun-sur-Yèvre	18
6506	142	\N	Meillant	18
6507	143	\N	Menetou-Couture	18
6508	144	\N	Menetou-Râtel	18
6509	145	\N	Menetou-Salon	18
6510	146	\N	Ménétréol-sous-Sancerre	18
6511	147	\N	Ménétréol-sur-Sauldre	18
6512	148	\N	Méreau	18
6513	149	\N	Méry-ès-Bois	18
6514	150	\N	Méry-sur-Cher	18
6515	151	\N	Montigny	18
6516	152	\N	Montlouis	18
6517	153	\N	Morlac	18
6518	154	\N	Mornay-Berry	18
6519	155	\N	Mornay-sur-Allier	18
6520	156	\N	Morogues	18
6521	157	\N	Morthomiers	18
6522	158	\N	Moulins-sur-Yèvre	18
6523	159	\N	Nançay	18
6524	160	\N	Nérondes	18
6525	161	\N	Neuilly-en-Dun	18
6526	162	\N	Neuilly-en-Sancerre	18
6527	163	\N	Neuvy-Deux-Clochers	18
6528	164	\N	Neuvy-le-Barrois	18
6529	165	\N	Neuvy-sur-Barangeon	18
6530	166	\N	Nohant-en-Goût	18
6531	167	\N	Nohant-en-Graçay	18
6532	168	\N	Noyer	18
6533	169	\N	Nozières	18
6534	170	\N	Oizon	18
6535	171	\N	Orcenais	18
6536	172	\N	Orval	18
6537	173	\N	Osmery	18
6538	174	\N	Osmoy	18
6539	175	\N	Ourouer-les-Bourdelins	18
6540	176	\N	Parassy	18
6541	177	\N	Parnay	18
6542	178	\N	Perche	18
6543	179	\N	Pigny	18
6544	180	\N	Plaimpied-Givaudins	18
6545	181	\N	Plou	18
6546	182	\N	Poisieux	18
6547	183	\N	Pondy	18
6548	184	\N	Précy	18
6549	185	\N	Presly	18
6550	186	\N	Preuilly	18
6551	187	\N	Préveranges	18
6552	188	\N	Primelles	18
6553	189	\N	Quantilly	18
6554	190	\N	Quincy	18
6555	191	\N	Raymond	18
6556	192	\N	Reigny	18
6557	193	\N	Rezay	18
6558	194	\N	Rians	18
6559	195	\N	Sagonne	18
6560	196	\N	Saint-Aignan-des-Noyers	18
6561	197	\N	Saint-Amand-Montrond	18
6562	198	\N	Saint-Ambroix	18
6563	199	\N	Saint-Baudel	18
6564	200	\N	Saint-Bouize	18
6565	201	\N	Saint-Caprais	18
6566	202	\N	Saint-Céols	18
6567	203	\N	Saint-Christophe-le-Chaudry	18
6568	204	\N	Saint-Denis-de-Palin	18
6569	205	\N	Saint-Doulchard	18
6570	206	\N	Saint-Éloy-de-Gy	18
6571	207	\N	Saint-Florent-sur-Cher	18
6572	208	\N	Sainte-Gemme-en-Sancerrois	18
6573	209	\N	Saint-Georges-de-Poisieux	18
6574	210	\N	Saint-Georges-sur-la-Prée	18
6575	211	\N	Saint-Georges-sur-Moulon	18
6576	212	\N	Saint-Germain-des-Bois	18
6577	213	\N	Saint-Germain-du-Puy	18
6578	214	\N	Saint-Hilaire-de-Court	18
6579	215	\N	Saint-Hilaire-de-Gondilly	18
6580	216	\N	Saint-Hilaire-en-Lignières	18
6581	217	\N	Saint-Jeanvrin	18
6582	218	\N	Saint-Just	18
6583	219	\N	Saint-Laurent	18
6584	220	\N	Saint-Léger-le-Petit	18
6585	221	\N	Saint-Loup-des-Chaumes	18
6586	222	\N	Sainte-Lunaise	18
6587	223	\N	Saint-Martin-d'Auxigny	18
6588	224	\N	Saint-Martin-des-Champs	18
6589	225	\N	Saint-Maur	18
6590	226	\N	Saint-Michel-de-Volangis	18
6591	227	\N	Sainte-Montaine	18
6592	228	\N	Saint-Outrille	18
6593	229	\N	Saint-Palais	18
6594	230	\N	Saint-Pierre-les-Bois	18
6595	231	\N	Saint-Pierre-les-Étieux	18
6596	232	\N	Saint-Priest-la-Marche	18
6597	233	\N	Saint-Satur	18
6598	234	\N	Saint-Saturnin	18
6599	235	\N	Sainte-Solange	18
6600	236	\N	Saint-Symphorien	18
6601	237	\N	Sainte-Thorette	18
6602	238	\N	Saint-Vitte	18
6603	239	\N	Saligny-le-Vif	18
6604	240	\N	Sancergues	18
6605	241	\N	Sancerre	18
6606	242	\N	Sancoins	18
6607	243	\N	Santranges	18
6608	244	\N	Saugy	18
6609	245	\N	Saulzais-le-Potier	18
6610	246	\N	Savigny-en-Sancerre	18
6611	247	\N	Savigny-en-Septaine	18
6612	248	\N	Senneçay	18
6613	249	\N	Sens-Beaujeu	18
6614	250	\N	Serruelles	18
6615	251	\N	Sévry	18
6616	252	\N	Sidiailles	18
6617	253	\N	Soulangis	18
6618	254	\N	Soye-en-Septaine	18
6619	255	\N	Subdray	18
6620	256	\N	Subligny	18
6621	257	\N	Sury-près-Léré	18
6622	258	\N	Sury-en-Vaux	18
6623	259	\N	Sury-ès-Bois	18
6624	260	\N	Tendron	18
6625	261	\N	Thaumiers	18
6626	262	\N	Thauvenay	18
6627	263	\N	Thénioux	18
6628	264	\N	Thou	18
6629	265	\N	Torteron	18
6630	266	\N	Touchay	18
6631	267	\N	Trouy	18
6632	268	\N	Uzay-le-Venon	18
6633	269	\N	Vailly-sur-Sauldre	18
6634	270	\N	Vallenay	18
6635	271	\N	Vasselay	18
6636	272	\N	Veaugues	18
6637	273	\N	Venesmes	18
6638	274	\N	Verdigny	18
6639	275	\N	Vereaux	18
6640	276	\N	Vernais	18
6641	277	\N	Verneuil	18
6642	278	\N	Vesdun	18
6643	279	\N	Vierzon	18
6644	280	\N	Vignoux-sous-les-Aix	18
6645	281	\N	Vignoux-sur-Barangeon	18
6646	282	\N	Villabon	18
6647	283	\N	Villecelin	18
6648	284	\N	Villegenon	18
6649	285	\N	Villeneuve-sur-Cher	18
6650	286	\N	Villequiers	18
6651	287	\N	Vinon	18
6652	288	\N	Vorly	18
6653	289	\N	Vornay	18
6654	290	\N	Vouzeron	18
6655	001	\N	Affieux	19
6656	002	\N	Aix	19
6657	003	\N	Albignac	19
6658	004	\N	Albussac	19
6659	005	\N	Allassac	19
6660	006	\N	Alleyrat	19
6661	007	\N	Altillac	19
6662	008	\N	Ambrugeat	19
6663	009	\N	Angles-sur-Corrèze	19
6664	010	\N	Argentat	19
6665	011	\N	Arnac-Pompadour	19
6666	012	\N	Astaillac	19
6667	013	\N	Aubazines	19
6668	014	\N	Auriac	19
6669	015	\N	Ayen	19
6670	016	\N	Bar	19
6671	017	\N	Bassignac-le-Bas	19
6672	018	\N	Bassignac-le-Haut	19
6673	019	\N	Beaulieu-sur-Dordogne	19
6674	020	\N	Beaumont	19
6675	021	\N	Bellechassagne	19
6676	022	\N	Benayes	19
6677	023	\N	Beynat	19
6678	024	\N	Beyssac	19
6679	025	\N	Beyssenac	19
6680	026	\N	Billac	19
6681	027	\N	Bonnefond	19
6682	028	\N	Bort-les-Orgues	19
6683	029	\N	Branceilles	19
6684	030	\N	Brignac-la-Plaine	19
6685	031	\N	Brive-la-Gaillarde	19
6686	032	\N	Brivezac	19
6687	033	\N	Bugeat	19
6688	034	\N	Camps-Saint-Mathurin-Léobazel	19
6689	035	\N	Chabrignac	19
6690	036	\N	Chamberet	19
6691	037	\N	Chamboulive	19
6692	038	\N	Chameyrat	19
6693	039	\N	Champagnac-la-Noaille	19
6694	040	\N	Champagnac-la-Prune	19
6695	041	\N	Chanac-les-Mines	19
6696	042	\N	Chanteix	19
6697	043	\N	Chapelle-aux-Brocs	19
6698	044	\N	Chapelle-aux-Saints	19
6699	045	\N	Chapelle-Saint-Géraud	19
6700	046	\N	Chapelle-Spinasse	19
6701	047	\N	Chartrier-Ferrière	19
6702	048	\N	Chastang	19
6703	049	\N	Chasteaux	19
6704	050	\N	Chauffour-sur-Vell	19
6705	051	\N	Chaumeil	19
6706	052	\N	Chavanac	19
6707	053	\N	Chaveroche	19
6708	054	\N	Chenailler-Mascheix	19
6709	055	\N	Chirac-Bellevue	19
6710	056	\N	Clergoux	19
6711	057	\N	Collonges-la-Rouge	19
6712	058	\N	Combressol	19
6713	059	\N	Concèze	19
6714	060	\N	Condat-sur-Ganaveix	19
6715	061	\N	Cornil	19
6716	062	\N	Corrèze	19
6717	063	\N	Cosnac	19
6718	064	\N	Couffy-sur-Sarsonne	19
6719	065	\N	Courteix	19
6720	066	\N	Cublac	19
6721	067	\N	Curemonte	19
6722	068	\N	Dampniat	19
6723	069	\N	Darazac	19
6724	070	\N	Darnets	19
6725	071	\N	Davignac	19
6726	072	\N	Donzenac	19
6727	073	\N	Égletons	19
6728	074	\N	Église-aux-Bois	19
6729	075	\N	Espagnac	19
6730	076	\N	Espartignac	19
6731	077	\N	Estivals	19
6732	078	\N	Estivaux	19
6733	079	\N	Eyburie	19
6734	080	\N	Eygurande	19
6735	081	\N	Eyrein	19
6736	082	\N	Favars	19
6737	083	\N	Feyt	19
6738	084	\N	Forgès	19
6739	085	\N	Gimel-les-Cascades	19
6740	086	\N	Goulles	19
6741	087	\N	Gourdon-Murat	19
6742	088	\N	Grandsaigne	19
6743	089	\N	Gros-Chastang	19
6744	090	\N	Gumond	19
6745	091	\N	Hautefage	19
6746	092	\N	Jardin	19
6747	093	\N	Jugeals-Nazareth	19
6748	094	\N	Juillac	19
6749	095	\N	Lacelle	19
6750	096	\N	Ladignac-sur-Rondelles	19
6751	097	\N	Lafage-sur-Sombre	19
6752	098	\N	Lagarde-Enval	19
6753	099	\N	Lagleygeolle	19
6754	100	\N	Lagraulière	19
6755	101	\N	Laguenne	19
6756	102	\N	Lamazière-Basse	19
6757	103	\N	Lamazière-Haute	19
6758	104	\N	Lamongerie	19
6759	105	\N	Lanteuil	19
6760	106	\N	Lapleau	19
6761	107	\N	Larche	19
6762	108	\N	Laroche-près-Feyt	19
6763	109	\N	Lascaux	19
6764	110	\N	Latronche	19
6765	111	\N	Laval-sur-Luzège	19
6766	112	\N	Lestards	19
6767	113	\N	Liginiac	19
6768	114	\N	Lignareix	19
6769	115	\N	Ligneyrac	19
6770	116	\N	Liourdres	19
6771	117	\N	Lissac-sur-Couze	19
6772	118	\N	Lonzac	19
6773	119	\N	Lostanges	19
6774	120	\N	Louignac	19
6775	121	\N	Lubersac	19
6776	122	\N	Madranges	19
6777	123	\N	Malemort-sur-Corrèze	19
6778	124	\N	Mansac	19
6779	125	\N	Marcillac-la-Croisille	19
6780	126	\N	Marcillac-la-Croze	19
6781	127	\N	Marc-la-Tour	19
6782	128	\N	Margerides	19
6783	129	\N	Masseret	19
6784	130	\N	Maussac	19
6785	131	\N	Meilhards	19
6786	132	\N	Ménoire	19
6787	133	\N	Mercur	19
6788	134	\N	Merlines	19
6789	135	\N	Mestes	19
6790	136	\N	Meymac	19
6791	137	\N	Meyrignac-l'Église	19
6792	138	\N	Meyssac	19
6793	139	\N	Millevaches	19
6794	140	\N	Monceaux-sur-Dordogne	19
6795	141	\N	Monestier-Merlines	19
6796	142	\N	Monestier-Port-Dieu	19
6797	143	\N	Montaignac-Saint-Hippolyte	19
6798	144	\N	Montgibaud	19
6799	145	\N	Moustier-Ventadour	19
6800	146	\N	Naves	19
6801	147	\N	Nespouls	19
6802	148	\N	Neuvic	19
6803	149	\N	Neuville	19
6804	150	\N	Noailhac	19
6805	151	\N	Noailles	19
6806	152	\N	Nonards	19
6807	153	\N	Objat	19
6808	154	\N	Orgnac-sur-Vézère	19
6809	155	\N	Orliac-de-Bar	19
6810	156	\N	Palazinges	19
6811	157	\N	Palisse	19
6812	158	\N	Pandrignes	19
6813	159	\N	Péret-Bel-Air	19
6814	160	\N	Pérols-sur-Vézère	19
6815	161	\N	Perpezac-le-Blanc	19
6816	162	\N	Perpezac-le-Noir	19
6817	163	\N	Pescher	19
6818	164	\N	Peyrelevade	19
6819	165	\N	Peyrissac	19
6820	166	\N	Pierrefitte	19
6821	167	\N	Confolent-Port-Dieu	19
6822	168	\N	Pradines	19
6823	169	\N	Puy-d'Arnac	19
6824	170	\N	Queyssac-les-Vignes	19
6825	171	\N	Reygade	19
6826	172	\N	Rilhac-Treignac	19
6827	173	\N	Rilhac-Xaintrie	19
6828	174	\N	Roche-Canillac	19
6829	175	\N	Roche-le-Peyroux	19
6830	176	\N	Rosiers-d'Égletons	19
6831	177	\N	Rosiers-de-Juillac	19
6832	178	\N	Sadroc	19
6833	179	\N	Saillac	19
6834	180	\N	Saint-Angel	19
6835	181	\N	Saint-Augustin	19
6836	182	\N	Saint-Aulaire	19
6837	183	\N	Saint-Bazile-de-la-Roche	19
6838	184	\N	Saint-Bazile-de-Meyssac	19
6839	185	\N	Saint-Bonnet-Avalouze	19
6840	186	\N	Saint-Bonnet-Elvert	19
6841	187	\N	Saint-Bonnet-la-Rivière	19
6842	188	\N	Saint-Bonnet-l'Enfantier	19
6843	189	\N	Saint-Bonnet-les-Tours-de-Merle	19
6844	190	\N	Saint-Bonnet-près-Bort	19
6845	191	\N	Saint-Cernin-de-Larche	19
6846	192	\N	Saint-Chamant	19
6847	193	\N	Saint-Cirgues-la-Loutre	19
6848	194	\N	Saint-Clément	19
6849	195	\N	Saint-Cyprien	19
6850	196	\N	Saint-Cyr-la-Roche	19
6851	198	\N	Saint-Éloy-les-Tuileries	19
6852	199	\N	Saint-Étienne-aux-Clos	19
6853	200	\N	Saint-Étienne-la-Geneste	19
6854	201	\N	Saint-Exupéry-les-Roches	19
6855	202	\N	Sainte-Féréole	19
6856	203	\N	Sainte-Fortunade	19
6857	204	\N	Saint-Fréjoux	19
6858	205	\N	Saint-Geniez-ô-Merle	19
6859	206	\N	Saint-Germain-Lavolps	19
6860	207	\N	Saint-Germain-les-Vergnes	19
6861	208	\N	Saint-Hilaire-Foissac	19
6862	209	\N	Saint-Hilaire-les-Courbes	19
6863	210	\N	Saint-Hilaire-Luc	19
6864	211	\N	Saint-Hilaire-Peyroux	19
6865	212	\N	Saint-Hilaire-Taurieux	19
6866	213	\N	Saint-Jal	19
6867	214	\N	Saint-Julien-aux-Bois	19
6868	215	\N	Saint-Julien-le-Pèlerin	19
6869	216	\N	Saint-Julien-le-Vendômois	19
6870	217	\N	Saint-Julien-Maumont	19
6871	218	\N	Saint-Julien-près-Bort	19
6872	219	\N	Sainte-Marie-Lapanouze	19
6873	220	\N	Saint-Martial-de-Gimel	19
6874	221	\N	Saint-Martial-Entraygues	19
6875	222	\N	Saint-Martin-la-Méanne	19
6876	223	\N	Saint-Martin-Sepert	19
6877	225	\N	Saint-Merd-de-Lapleau	19
6878	226	\N	Saint-Merd-les-Oussines	19
6879	227	\N	Saint-Mexant	19
6880	228	\N	Saint-Pantaléon-de-Lapleau	19
6881	229	\N	Saint-Pantaléon-de-Larche	19
6882	230	\N	Saint-Pardoux-Corbier	19
6883	231	\N	Saint-Pardoux-la-Croisille	19
6884	232	\N	Saint-Pardoux-le-Neuf	19
6885	233	\N	Saint-Pardoux-le-Vieux	19
6886	234	\N	Saint-Pardoux-l'Ortigier	19
6887	235	\N	Saint-Paul	19
6888	236	\N	Saint-Priest-de-Gimel	19
6889	237	\N	Saint-Privat	19
6890	238	\N	Saint-Rémy	19
6891	239	\N	Saint-Robert	19
6892	240	\N	Saint-Salvadour	19
6893	241	\N	Saint-Setiers	19
6894	242	\N	Saint-Solve	19
6895	243	\N	Saint-Sornin-Lavolps	19
6896	244	\N	Saint-Sulpice-les-Bois	19
6897	245	\N	Saint-Sylvain	19
6898	246	\N	Saint-Viance	19
6899	247	\N	Saint-Victour	19
6900	248	\N	Saint-Ybard	19
6901	249	\N	Saint-Yrieix-le-Déjalat	19
6902	250	\N	Salon-la-Tour	19
6903	251	\N	Sarran	19
6904	252	\N	Sarroux	19
6905	253	\N	Segonzac	19
6906	254	\N	Ségur-le-Château	19
6907	255	\N	Seilhac	19
6908	256	\N	Sérandon	19
6909	257	\N	Sérilhac	19
6910	258	\N	Servières-le-Château	19
6911	259	\N	Sexcles	19
6912	260	\N	Sioniac	19
6913	261	\N	Sornac	19
6914	262	\N	Soudaine-Lavinadière	19
6915	263	\N	Soudeilles	19
6916	264	\N	Soursac	19
6917	265	\N	Tarnac	19
6918	266	\N	Thalamy	19
6919	268	\N	Toy-Viam	19
6920	269	\N	Treignac	19
6921	270	\N	Troche	19
6922	271	\N	Tudeils	19
6923	272	\N	Tulle	19
6924	273	\N	Turenne	19
6925	274	\N	Ussac	19
6926	275	\N	Ussel	19
6927	276	\N	Uzerche	19
6928	277	\N	Valiergues	19
6929	278	\N	Varetz	19
6930	279	\N	Vars-sur-Roseix	19
6931	280	\N	Végennes	19
6932	281	\N	Veix	19
6933	282	\N	Venarsal	19
6934	283	\N	Veyrières	19
6935	284	\N	Viam	19
6936	285	\N	Vigeois	19
6937	286	\N	Vignols	19
6938	287	\N	Vitrac-sur-Montane	19
6939	288	\N	Voutezac	19
6940	289	\N	Yssandon	19
6941	001	\N	Agencourt	21
6942	002	\N	Agey	21
6943	003	\N	Ahuy	21
6944	004	\N	Aignay-le-Duc	21
6945	005	\N	Aiserey	21
6946	006	\N	Aisey-sur-Seine	21
6947	007	\N	Aisy-sous-Thil	21
6948	008	\N	Alise-Sainte-Reine	21
6949	009	\N	Allerey	21
6950	010	\N	Aloxe-Corton	21
6951	011	\N	Ampilly-les-Bordes	21
6952	012	\N	Ampilly-le-Sec	21
6953	013	\N	Ancey	21
6954	014	\N	Antheuil	21
6955	015	\N	Antigny-la-Ville	21
6956	016	\N	Arceau	21
6957	017	\N	Arcenant	21
6958	018	\N	Arcey	21
6959	020	\N	Arconcey	21
6960	021	\N	Arc-sur-Tille	21
6961	022	\N	Argilly	21
6962	023	\N	Arnay-le-Duc	21
6963	024	\N	Arnay-sous-Vitteaux	21
6964	025	\N	Arrans	21
6965	026	\N	Asnières-en-Montagne	21
6966	027	\N	Asnières-lès-Dijon	21
6967	028	\N	Athée	21
6968	029	\N	Athie	21
6969	030	\N	Aubaine	21
6970	031	\N	Aubigny-en-Plaine	21
6971	032	\N	Aubigny-la-Ronce	21
6972	033	\N	Aubigny-lès-Sombernon	21
6973	034	\N	Autricourt	21
6974	035	\N	Auvillars-sur-Saône	21
6975	036	\N	Auxant	21
6976	037	\N	Auxey-Duresses	21
6977	038	\N	Auxonne	21
6978	039	\N	Avelanges	21
6979	040	\N	Avosnes	21
6980	041	\N	Avot	21
6981	042	\N	Bagnot	21
6982	043	\N	Baigneux-les-Juifs	21
6983	044	\N	Balot	21
6984	045	\N	Barbirey-sur-Ouche	21
6985	046	\N	Bard-le-Régulier	21
6986	047	\N	Bard-lès-Époisses	21
6987	048	\N	Barges	21
6988	049	\N	Barjon	21
6989	050	\N	Baubigny	21
6990	051	\N	Baulme-la-Roche	21
6991	052	\N	Beaulieu	21
6992	053	\N	Beaumont-sur-Vingeanne	21
6993	054	\N	Beaune	21
6994	055	\N	Beaunotte	21
6995	056	\N	Beire-le-Châtel	21
6996	057	\N	Beire-le-Fort	21
6997	058	\N	Belan-sur-Ource	21
6998	059	\N	Bellefond	21
6999	060	\N	Belleneuve	21
7000	061	\N	Bellenod-sur-Seine	21
7001	062	\N	Bellenot-sous-Pouilly	21
7002	063	\N	Beneuvre	21
7003	064	\N	Benoisey	21
7004	065	\N	Bessey-en-Chaume	21
7005	066	\N	Bessey-la-Cour	21
7006	067	\N	Bessey-lès-Cîteaux	21
7007	068	\N	Beurey-Bauguay	21
7008	069	\N	Beurizot	21
7009	070	\N	Bévy	21
7010	071	\N	Bèze	21
7011	072	\N	Bézouotte	21
7012	073	\N	Bierre-lès-Semur	21
7013	074	\N	Billey	21
7014	075	\N	Billy-lès-Chanceaux	21
7015	076	\N	Binges	21
7016	077	\N	Bissey-la-Côte	21
7017	078	\N	Bissey-la-Pierre	21
7018	079	\N	Blagny-sur-Vingeanne	21
7019	080	\N	Blaisy-Bas	21
7020	081	\N	Blaisy-Haut	21
7021	082	\N	Blancey	21
7022	083	\N	Blanot	21
7023	084	\N	Blessey	21
7024	085	\N	Bligny-le-Sec	21
7025	086	\N	Bligny-lès-Beaune	21
7026	087	\N	Bligny-sur-Ouche	21
7027	088	\N	Boncourt-le-Bois	21
7028	089	\N	Bonnencontre	21
7029	090	\N	Boudreville	21
7030	091	\N	Bouhey	21
7031	092	\N	Bouilland	21
7032	093	\N	Bouix	21
7033	094	\N	Bourberain	21
7034	095	\N	Bousselange	21
7035	096	\N	Boussenois	21
7036	097	\N	Boussey	21
7037	098	\N	Boux-sous-Salmaise	21
7038	099	\N	Bouze-lès-Beaune	21
7039	100	\N	Brain	21
7040	101	\N	Braux	21
7041	102	\N	Brazey-en-Morvan	21
7042	103	\N	Brazey-en-Plaine	21
7043	104	\N	Brémur-et-Vaurois	21
7044	105	\N	Bressey-sur-Tille	21
7045	106	\N	Bretenière	21
7046	107	\N	Bretigny	21
7047	108	\N	Brianny	21
7048	109	\N	Brion-sur-Ource	21
7049	110	\N	Brochon	21
7050	111	\N	Brognon	21
7051	112	\N	Broin	21
7052	113	\N	Broindon	21
7053	114	\N	Buffon	21
7054	115	\N	Buncey	21
7055	116	\N	Bure-les-Templiers	21
7056	117	\N	Busseaut	21
7057	118	\N	Busserotte-et-Montenaille	21
7058	119	\N	Bussières	21
7059	120	\N	Bussière-sur-Ouche	21
7060	121	\N	Bussy-la-Pesle	21
7061	122	\N	Bussy-le-Grand	21
7062	123	\N	Buxerolles	21
7063	124	\N	Censerey	21
7064	125	\N	Cérilly	21
7065	126	\N	Cessey-sur-Tille	21
7066	127	\N	Chaignay	21
7067	128	\N	Chailly-sur-Armançon	21
7068	129	\N	Chambain	21
7069	130	\N	Chambeire	21
7070	131	\N	Chamblanc	21
7071	132	\N	Chambuf	21
7072	133	\N	Chambolle-Musigny	21
7073	134	\N	Chamesson	21
7074	135	\N	Champagne-sur-Vingeanne	21
7075	136	\N	Champagny	21
7076	137	\N	Champ-d'Oiseau	21
7077	138	\N	Champdôtre	21
7078	139	\N	Champeau-en-Morvan	21
7079	140	\N	Champignolles	21
7080	141	\N	Champrenault	21
7081	142	\N	Chanceaux	21
7082	143	\N	Channay	21
7083	144	\N	Charencey	21
7084	145	\N	Charigny	21
7085	146	\N	Charmes	21
7086	147	\N	Charny	21
7087	148	\N	Charrey-sur-Saône	21
7088	149	\N	Charrey-sur-Seine	21
7089	150	\N	Chassagne-Montrachet	21
7090	151	\N	Chassey	21
7091	152	\N	Châteauneuf	21
7092	153	\N	Châtellenot	21
7093	154	\N	Châtillon-sur-Seine	21
7094	155	\N	Chaudenay-la-Ville	21
7095	156	\N	Chaudenay-le-Château	21
7096	157	\N	Chaugey	21
7097	158	\N	Chaume-et-Courchamp	21
7098	159	\N	Chaume	21
7099	160	\N	Chaume-lès-Baigneux	21
7100	161	\N	Chaumont-le-Bois	21
7101	162	\N	Chaux	21
7102	163	\N	Chazeuil	21
7103	164	\N	Chazilly	21
7104	165	\N	Chemin-d'Aisey	21
7105	166	\N	Chenôve	21
7106	167	\N	Cheuge	21
7107	168	\N	Chevannay	21
7108	169	\N	Chevannes	21
7109	170	\N	Chevigny-en-Valière	21
7110	171	\N	Chevigny-Saint-Sauveur	21
7111	172	\N	Chivres	21
7112	173	\N	Chorey-les-Beaune	21
7113	175	\N	Cirey-lès-Pontailler	21
7114	176	\N	Civry-en-Montagne	21
7115	177	\N	Clamerey	21
7116	178	\N	Clémencey	21
7117	179	\N	Clénay	21
7118	180	\N	Cléry	21
7119	181	\N	Clomot	21
7120	182	\N	Collonges-lès-Bévy	21
7121	183	\N	Collonges-lès-Premières	21
7122	184	\N	Colombier	21
7123	185	\N	Combertault	21
7124	186	\N	Comblanchien	21
7125	187	\N	Commarin	21
7126	189	\N	Corberon	21
7127	190	\N	Corcelles-les-Arts	21
7128	191	\N	Corcelles-lès-Cîteaux	21
7129	192	\N	Corcelles-les-Monts	21
7130	193	\N	Corgengoux	21
7131	194	\N	Corgoloin	21
7132	195	\N	Cormot-le-Grand	21
7133	196	\N	Corpeau	21
7134	197	\N	Corpoyer-la-Chapelle	21
7135	198	\N	Corrombles	21
7136	199	\N	Corsaint	21
7137	200	\N	Couchey	21
7138	201	\N	Coulmier-le-Sec	21
7139	202	\N	Courban	21
7140	203	\N	Courcelles-Frémoy	21
7141	204	\N	Courcelles-lès-Montbard	21
7142	205	\N	Courcelles-lès-Semur	21
7143	207	\N	Courlon	21
7144	208	\N	Courtivron	21
7145	209	\N	Couternon	21
7146	210	\N	Créancey	21
7147	211	\N	Crécey-sur-Tille	21
7148	212	\N	Crépand	21
7149	213	\N	Crimolois	21
7150	214	\N	Crugey	21
7151	215	\N	Cuiserey	21
7152	216	\N	Culètre	21
7153	217	\N	Curley	21
7154	218	\N	Curtil-Saint-Seine	21
7155	219	\N	Curtil-Vergy	21
7156	220	\N	Cussey-les-Forges	21
7157	221	\N	Cussy-la-Colonne	21
7158	222	\N	Cussy-le-Châtel	21
7159	223	\N	Daix	21
7160	224	\N	Dampierre-en-Montagne	21
7161	225	\N	Dampierre-et-Flée	21
7162	226	\N	Darcey	21
7163	227	\N	Darois	21
7164	228	\N	Détain-et-Bruant	21
7165	229	\N	Diancey	21
7166	230	\N	Diénay	21
7167	231	\N	Dijon	21
7168	232	\N	Dompierre-en-Morvan	21
7169	233	\N	Drambon	21
7170	234	\N	Drée	21
7171	235	\N	Duesme	21
7172	236	\N	Ébaty	21
7173	237	\N	Échalot	21
7174	238	\N	Échannay	21
7175	239	\N	Échenon	21
7176	240	\N	Échevannes	21
7177	241	\N	Échevronne	21
7178	242	\N	Échigey	21
7179	243	\N	Écutigny	21
7180	244	\N	Éguilly	21
7181	245	\N	Épagny	21
7182	246	\N	Épernay-sous-Gevrey	21
7183	247	\N	Époisses	21
7184	248	\N	Éringes	21
7185	249	\N	Esbarres	21
7186	250	\N	Essarois	21
7187	251	\N	Essey	21
7188	252	\N	Étais	21
7189	253	\N	Étalante	21
7190	254	\N	Étang-Vergy	21
7191	255	\N	Étaules	21
7192	256	\N	Étevaux	21
7193	257	\N	Étormay	21
7194	258	\N	Étrochey	21
7195	259	\N	Fain-lès-Montbard	21
7196	260	\N	Fain-lès-Moutiers	21
7197	261	\N	Fauverney	21
7198	262	\N	Faverolles-lès-Lucey	21
7199	263	\N	Fénay	21
7200	264	\N	Fête	21
7201	265	\N	Fixin	21
7202	266	\N	Flacey	21
7203	267	\N	Flagey-Echézeaux	21
7204	268	\N	Flagey-lès-Auxonne	21
7205	269	\N	Flammerans	21
7206	270	\N	Flavignerot	21
7207	271	\N	Flavigny-sur-Ozerain	21
7208	272	\N	Flée	21
7209	273	\N	Fleurey-sur-Ouche	21
7210	274	\N	Foissy	21
7211	275	\N	Foncegrive	21
7212	276	\N	Fontaines-en-Duesmois	21
7213	277	\N	Fontaine-Française	21
7214	278	\N	Fontaine-lès-Dijon	21
7215	279	\N	Fontaines-les-Sèches	21
7216	280	\N	Fontangy	21
7217	281	\N	Fontenelle	21
7218	282	\N	Forléans	21
7219	283	\N	Fraignot-et-Vesvrotte	21
7220	284	\N	Francheville	21
7221	285	\N	Franxault	21
7222	286	\N	Frénois	21
7223	287	\N	Fresnes	21
7224	288	\N	Frôlois	21
7225	289	\N	Fussey	21
7226	290	\N	Gemeaux	21
7227	291	\N	Genay	21
7228	292	\N	Genlis	21
7229	293	\N	Gergueil	21
7230	294	\N	Gerland	21
7231	295	\N	Gevrey-Chambertin	21
7232	296	\N	Gevrolles	21
7233	297	\N	Gilly-lès-Cîteaux	21
7234	298	\N	Gissey-le-Vieil	21
7235	299	\N	Gissey-sous-Flavigny	21
7236	300	\N	Gissey-sur-Ouche	21
7237	301	\N	Glanon	21
7238	302	\N	Gomméville	21
7239	303	\N	Goulles	21
7240	304	\N	Grancey-le-Château-Neuvelle	21
7241	305	\N	Grancey-sur-Ource	21
7242	306	\N	Grenand-lès-Sombernon	21
7243	307	\N	Grésigny-Sainte-Reine	21
7244	308	\N	Grignon	21
7245	309	\N	Griselles	21
7246	310	\N	Grosbois-en-Montagne	21
7247	311	\N	Grosbois-lès-Tichey	21
7248	312	\N	Gurgy-la-Ville	21
7249	313	\N	Gurgy-le-Château	21
7250	314	\N	Hauteroche	21
7251	315	\N	Hauteville-lès-Dijon	21
7252	316	\N	Heuilley-sur-Saône	21
7253	317	\N	Is-sur-Tille	21
7254	318	\N	Ivry-en-Montagne	21
7255	319	\N	Izeure	21
7256	320	\N	Izier	21
7257	321	\N	Jailly-les-Moulins	21
7258	322	\N	Jallanges	21
7259	323	\N	Jancigny	21
7260	324	\N	Jeux-lès-Bard	21
7261	325	\N	Jouey	21
7262	326	\N	Jours-lès-Baigneux	21
7263	327	\N	Jours-en-Vaux	21
7264	328	\N	Juillenay	21
7265	329	\N	Juilly	21
7266	330	\N	Labergement-Foigney	21
7267	331	\N	Labergement-lès-Auxonne	21
7268	332	\N	Labergement-lès-Seurre	21
7269	333	\N	Labruyère	21
7270	334	\N	Lacanche	21
7271	335	\N	Lacour-d'Arcenay	21
7272	336	\N	Laignes	21
7273	337	\N	Lamarche-sur-Saône	21
7274	338	\N	Lamargelle	21
7275	339	\N	Lantenay	21
7276	340	\N	Lanthes	21
7277	341	\N	Lantilly	21
7278	342	\N	Laperrière-sur-Saône	21
7279	343	\N	Larrey	21
7280	344	\N	Lechâtelet	21
7281	345	\N	Léry	21
7282	346	\N	Leuglay	21
7283	347	\N	Levernois	21
7284	348	\N	Licey-sur-Vingeanne	21
7285	349	\N	Liernais	21
7286	350	\N	Lignerolles	21
7287	351	\N	Longchamp	21
7288	352	\N	Longeault	21
7289	353	\N	Longecourt-en-Plaine	21
7290	354	\N	Longecourt-lès-Culêtre	21
7291	355	\N	Longvic	21
7292	356	\N	Losne	21
7293	357	\N	Louesme	21
7294	358	\N	Lucenay-le-Duc	21
7295	359	\N	Lucey	21
7296	360	\N	Lusigny-sur-Ouche	21
7297	361	\N	Lux	21
7298	362	\N	Maconge	21
7299	363	\N	Magnien	21
7300	364	\N	Magny-Lambert	21
7301	365	\N	Magny-la-Ville	21
7302	366	\N	Magny-lès-Aubigny	21
7303	367	\N	Magny-Montarlot	21
7304	368	\N	Magny-lès-Villers	21
7305	369	\N	Magny-Saint-Médard	21
7306	370	\N	Magny-sur-Tille	21
7307	371	\N	Maillys	21
7308	372	\N	Maisey-le-Duc	21
7309	373	\N	Mâlain	21
7310	374	\N	Maligny	21
7311	375	\N	Manlay	21
7312	376	\N	Marandeuil	21
7313	377	\N	Marcellois	21
7314	378	\N	Marcenay	21
7315	379	\N	Marcheseuil	21
7316	380	\N	Marcigny-sous-Thil	21
7317	381	\N	Marcilly-et-Dracy	21
7318	382	\N	Marcilly-Ogny	21
7319	383	\N	Marcilly-sur-Tille	21
7320	384	\N	Marey-lès-Fussey	21
7321	385	\N	Marey-sur-Tille	21
7322	386	\N	Marigny-le-Cahouët	21
7323	387	\N	Marigny-lès-Reullée	21
7324	388	\N	Marliens	21
7325	389	\N	Marmagne	21
7326	390	\N	Marsannay-la-Côte	21
7327	391	\N	Marsannay-le-Bois	21
7328	392	\N	Martrois	21
7329	393	\N	Massingy	21
7330	394	\N	Massingy-lès-Semur	21
7331	395	\N	Massingy-lès-Vitteaux	21
7332	396	\N	Mauvilly	21
7333	397	\N	Mavilly-Mandelot	21
7334	398	\N	Maxilly-sur-Saône	21
7335	399	\N	Meilly-sur-Rouvres	21
7336	400	\N	Meix	21
7337	401	\N	Meloisey	21
7338	402	\N	Menesble	21
7339	403	\N	Ménessaire	21
7340	404	\N	Ménétreux-le-Pitois	21
7341	405	\N	Merceuil	21
7342	406	\N	Mesmont	21
7343	407	\N	Messanges	21
7344	408	\N	Messigny-et-Vantoux	21
7345	409	\N	Meuilley	21
7346	410	\N	Meulson	21
7347	411	\N	Meursanges	21
7348	412	\N	Meursault	21
7349	413	\N	Millery	21
7350	414	\N	Mimeure	21
7351	415	\N	Minot	21
7352	416	\N	Mirebeau-sur-Bèze	21
7353	417	\N	Missery	21
7354	418	\N	Moitron	21
7355	419	\N	Molesme	21
7356	420	\N	Molinot	21
7357	421	\N	Moloy	21
7358	422	\N	Molphey	21
7359	423	\N	Montagny-lès-Beaune	21
7360	424	\N	Montagny-lès-Seurre	21
7361	425	\N	Montbard	21
7362	426	\N	Montberthault	21
7363	427	\N	Montceau-et-Écharnant	21
7364	428	\N	Monthelie	21
7365	429	\N	Montigny-Montfort	21
7366	430	\N	Montigny-Saint-Barthélemy	21
7367	431	\N	Montigny-sur-Armançon	21
7368	432	\N	Montigny-sur-Aube	21
7369	433	\N	Montigny-Mornay-Villeneuve-sur-Vingeanne	21
7370	434	\N	Montlay-en-Auxois	21
7371	435	\N	Montliot-et-Courcelles	21
7372	436	\N	Montmain	21
7373	437	\N	Montmançon	21
7374	438	\N	Montmoyen	21
7375	439	\N	Montoillot	21
7376	440	\N	Montot	21
7377	441	\N	Mont-Saint-Jean	21
7378	442	\N	Morey-Saint-Denis	21
7379	444	\N	Mosson	21
7380	445	\N	Motte-Ternant	21
7381	446	\N	Moutiers-Saint-Jean	21
7382	447	\N	Musigny	21
7383	448	\N	Mussy-la-Fosse	21
7384	449	\N	Nan-sous-Thil	21
7385	450	\N	Nantoux	21
7386	451	\N	Nesle-et-Massoult	21
7387	452	\N	Neuilly-lès-Dijon	21
7388	454	\N	Nicey	21
7389	455	\N	Nod-sur-Seine	21
7390	456	\N	Nogent-lès-Montbard	21
7391	457	\N	Noidan	21
7392	458	\N	Noiron-sous-Gevrey	21
7393	459	\N	Noiron-sur-Bèze	21
7394	460	\N	Noiron-sur-Seine	21
7395	461	\N	Nolay	21
7396	462	\N	Norges-la-Ville	21
7397	463	\N	Normier	21
7398	464	\N	Nuits-Saint-Georges	21
7399	465	\N	Obtrée	21
7400	466	\N	Oigny	21
7401	467	\N	Oisilly	21
7402	468	\N	Orain	21
7403	469	\N	Orgeux	21
7404	470	\N	Origny	21
7405	471	\N	Orret	21
7406	472	\N	Orville	21
7407	473	\N	Ouges	21
7408	474	\N	Pagny-la-Ville	21
7409	475	\N	Pagny-le-Château	21
7410	476	\N	Painblanc	21
7411	477	\N	Panges	21
7412	478	\N	Pasques	21
7413	479	\N	Pellerey	21
7414	480	\N	Pernand-Vergelesses	21
7415	481	\N	Perrigny-lès-Dijon	21
7416	482	\N	Perrigny-sur-l'Ognon	21
7417	483	\N	Pichanges	21
7418	484	\N	Planay	21
7419	485	\N	Plombières-lès-Dijon	21
7420	486	\N	Pluvault	21
7421	487	\N	Pluvet	21
7422	488	\N	Poinçon-lès-Larrey	21
7423	489	\N	Poiseul-la-Grange	21
7424	490	\N	Poiseul-la-Ville-et-Laperrière	21
7425	491	\N	Poiseul-lès-Saulx	21
7426	492	\N	Pommard	21
7427	493	\N	Poncey-lès-Athée	21
7428	494	\N	Poncey-sur-l'Ignon	21
7429	495	\N	Pont	21
7430	496	\N	Pontailler-sur-Saône	21
7431	497	\N	Pont-et-Massène	21
7432	498	\N	Posanges	21
7433	499	\N	Pothières	21
7434	500	\N	Pouillenay	21
7435	501	\N	Pouilly-en-Auxois	21
7436	502	\N	Pouilly-sur-Saône	21
7437	503	\N	Pouilly-sur-Vingeanne	21
7438	504	\N	Prâlon	21
7439	505	\N	Précy-sous-Thil	21
7440	506	\N	Premeaux-Prissey	21
7441	507	\N	Premières	21
7442	508	\N	Prenois	21
7443	510	\N	Prusly-sur-Ource	21
7444	511	\N	Puits	21
7445	512	\N	Puligny-Montrachet	21
7446	513	\N	Quemigny-Poisot	21
7447	514	\N	Quemigny-sur-Seine	21
7448	515	\N	Quetigny	21
7449	516	\N	Quincerot	21
7450	517	\N	Quincey	21
7451	518	\N	Quincy-le-Vicomte	21
7452	519	\N	Recey-sur-Ource	21
7453	520	\N	Remilly-en-Montagne	21
7454	521	\N	Remilly-sur-Tille	21
7455	522	\N	Renève	21
7456	523	\N	Reulle-Vergy	21
7457	524	\N	Riel-les-Eaux	21
7458	525	\N	Roche-en-Brenil	21
7459	526	\N	Rochefort-sur-Brévon	21
7460	527	\N	Rochepot	21
7461	528	\N	Roche-Vanneau	21
7462	529	\N	Roilly	21
7463	530	\N	Rougemont	21
7464	531	\N	Rouvray	21
7465	532	\N	Rouvres-en-Plaine	21
7466	533	\N	Rouvres-sous-Meilly	21
7467	534	\N	Ruffey-lès-Beaune	21
7468	535	\N	Ruffey-lès-Echirey	21
7469	536	\N	Sacquenay	21
7470	537	\N	Saffres	21
7471	538	\N	Saint-Andeux	21
7472	539	\N	Saint-Anthot	21
7473	540	\N	Saint-Apollinaire	21
7474	541	\N	Saint-Aubin	21
7475	542	\N	Saint-Bernard	21
7476	543	\N	Saint-Broing-les-Moines	21
7477	544	\N	Sainte-Colombe	21
7478	545	\N	Sainte-Colombe-sur-Seine	21
7479	546	\N	Saint-Didier	21
7480	547	\N	Saint-Euphrône	21
7481	548	\N	Saint-Germain-de-Modéon	21
7482	549	\N	Saint-Germain-le-Rocheux	21
7483	550	\N	Saint-Germain-lès-Senailly	21
7484	551	\N	Saint-Germain-Source-Seine	21
7485	552	\N	Saint-Hélier	21
7486	553	\N	Saint-Jean-de-Buf	21
7487	554	\N	Saint-Jean-de-Losne	21
7488	555	\N	Saint-Julien	21
7489	556	\N	Saint-Léger-Triey	21
7490	557	\N	Saint-Marc-sur-Seine	21
7491	558	\N	Sainte-Marie-la-Blanche	21
7492	559	\N	Sainte-Marie-sur-Ouche	21
7493	560	\N	Saint-Martin-de-la-Mer	21
7494	561	\N	Saint-Martin-du-Mont	21
7495	562	\N	Saint-Maurice-sur-Vingeanne	21
7496	563	\N	Saint-Mesmin	21
7497	564	\N	Saint-Nicolas-lès-Cîteaux	21
7498	565	\N	Saint-Philibert	21
7499	566	\N	Saint-Pierre-en-Vaux	21
7500	567	\N	Saint-Prix-lès-Arnay	21
7501	568	\N	Saint-Rémy	21
7502	569	\N	Saint-Romain	21
7503	570	\N	Sainte-Sabine	21
7504	571	\N	Saint-Sauveur	21
7505	572	\N	Saint-Seine-en-Bâche	21
7506	573	\N	Saint-Seine-l'Abbaye	21
7507	574	\N	Saint-Seine-sur-Vingeanne	21
7508	575	\N	Saint-Symphorien-sur-Saône	21
7509	576	\N	Saint-Thibault	21
7510	577	\N	Saint-Usage	21
7511	578	\N	Saint-Victor-sur-Ouche	21
7512	579	\N	Salives	21
7513	580	\N	Salmaise	21
7514	581	\N	Samerey	21
7515	582	\N	Santenay	21
7516	583	\N	Santosse	21
7517	584	\N	Saulieu	21
7518	585	\N	Saulon-la-Chapelle	21
7519	586	\N	Saulon-la-Rue	21
7520	587	\N	Saulx-le-Duc	21
7521	588	\N	Saussey	21
7522	589	\N	Saussy	21
7523	590	\N	Savigny-lès-Beaune	21
7524	591	\N	Savigny-le-Sec	21
7525	592	\N	Savigny-sous-Mâlain	21
7526	593	\N	Savilly	21
7527	594	\N	Savoisy	21
7528	595	\N	Savolles	21
7529	596	\N	Savouges	21
7530	597	\N	Segrois	21
7531	598	\N	Seigny	21
7532	599	\N	Selongey	21
7533	600	\N	Semarey	21
7534	601	\N	Semezanges	21
7535	602	\N	Semond	21
7536	603	\N	Semur-en-Auxois	21
7537	604	\N	Senailly	21
7538	605	\N	Sennecey-lès-Dijon	21
7539	606	\N	Ladoix-Serrigny	21
7540	607	\N	Seurre	21
7541	608	\N	Sincey-lès-Rouvray	21
7542	609	\N	Soirans	21
7543	610	\N	Soissons-sur-Nacey	21
7544	611	\N	Sombernon	21
7545	612	\N	Souhey	21
7546	613	\N	Soussey-sur-Brionne	21
7547	614	\N	Spoy	21
7548	615	\N	Sussey	21
7549	616	\N	Tailly	21
7550	617	\N	Talant	21
7551	618	\N	Talmay	21
7552	619	\N	Tanay	21
7553	620	\N	Tarsul	21
7554	621	\N	Tart-l'Abbaye	21
7555	622	\N	Tart-le-Bas	21
7556	623	\N	Tart-le-Haut	21
7557	624	\N	Tellecey	21
7558	625	\N	Ternant	21
7559	626	\N	Terrefondrée	21
7560	627	\N	Thenissey	21
7561	628	\N	Thoires	21
7562	629	\N	Thoisy-la-Berchère	21
7563	630	\N	Thoisy-le-Désert	21
7564	631	\N	Thomirey	21
7565	632	\N	Thorey-en-Plaine	21
7566	633	\N	Thorey-sous-Charny	21
7567	634	\N	Thorey-sur-Ouche	21
7568	635	\N	Thoste	21
7569	636	\N	Thury	21
7570	637	\N	Tichey	21
7571	638	\N	Til-Châtel	21
7572	639	\N	Tillenay	21
7573	640	\N	Torcy-et-Pouligny	21
7574	641	\N	Touillon	21
7575	642	\N	Toutry	21
7576	643	\N	Tréclun	21
7577	644	\N	Trochères	21
7578	645	\N	Trouhans	21
7579	646	\N	Trouhaut	21
7580	647	\N	Trugny	21
7581	648	\N	Turcey	21
7582	649	\N	Uncey-le-Franc	21
7583	650	\N	Urcy	21
7584	651	\N	Val-Suzon	21
7585	652	\N	Vandenesse-en-Auxois	21
7586	653	\N	Vannaire	21
7587	655	\N	Vanvey	21
7588	656	\N	Varanges	21
7589	657	\N	Varois-et-Chaignot	21
7590	658	\N	Vauchignon	21
7591	659	\N	Vaux-Saules	21
7592	660	\N	Veilly	21
7593	661	\N	Velars-sur-Ouche	21
7594	662	\N	Velogny	21
7595	663	\N	Venarey-les-Laumes	21
7596	664	\N	Verdonnet	21
7597	665	\N	Vernois-lès-Vesvres	21
7598	666	\N	Vernot	21
7599	667	\N	Véronnes	21
7600	669	\N	Verrey-sous-Drée	21
7601	670	\N	Verrey-sous-Salmaise	21
7602	671	\N	Vertault	21
7603	672	\N	Vesvres	21
7604	673	\N	Veuvey-sur-Ouche	21
7605	674	\N	Veuxhaulles-sur-Aube	21
7606	675	\N	Vianges	21
7607	676	\N	Vic-de-Chassenay	21
7608	677	\N	Vic-des-Prés	21
7609	678	\N	Vic-sous-Thil	21
7610	679	\N	Vieilmoulin	21
7611	680	\N	Vielverge	21
7612	681	\N	Vieux-Château	21
7613	682	\N	Viévigne	21
7614	683	\N	Viévy	21
7615	684	\N	Vignoles	21
7616	685	\N	Villaines-en-Duesmois	21
7617	686	\N	Villaines-les-Prévôtes	21
7618	687	\N	Villargoix	21
7619	688	\N	Villars-Fontaine	21
7620	689	\N	Villars-et-Villenotte	21
7621	690	\N	Villeberny	21
7622	691	\N	Villebichot	21
7623	692	\N	Villecomte	21
7624	693	\N	Villedieu	21
7625	694	\N	Villeferry	21
7626	695	\N	Villeneuve-les-Convers	21
7627	696	\N	Villeneuve-sous-Charigny	21
7628	698	\N	Villers-la-Faye	21
7629	699	\N	Villers-les-Pots	21
7630	700	\N	Villers-Patras	21
7631	701	\N	Villers-Rotin	21
7632	702	\N	Villey-sur-Tille	21
7633	703	\N	Villiers-en-Morvan	21
7634	704	\N	Villiers-le-Duc	21
7635	705	\N	Villotte-Saint-Seine	21
7636	706	\N	Villotte-sur-Ource	21
7637	707	\N	Villy-en-Auxois	21
7638	708	\N	Villy-le-Moutier	21
7639	709	\N	Viserny	21
7640	710	\N	Vitteaux	21
7641	711	\N	Vix	21
7642	712	\N	Volnay	21
7643	713	\N	Vonges	21
7644	714	\N	Vosne-Romanée	21
7645	715	\N	Voudenay	21
7646	716	\N	Vougeot	21
7647	717	\N	Voulaines-les-Templiers	21
7648	001	\N	Allineuc	22
7649	002	\N	Andel	22
7650	003	\N	Aucaleuc	22
7651	004	\N	Bégard	22
7652	005	\N	Belle-Isle-en-Terre	22
7653	006	\N	Berhet	22
7654	007	\N	Binic	22
7655	008	\N	Bobital	22
7656	009	\N	Bodéo	22
7657	011	\N	Boqueho	22
7658	012	\N	Bouillie	22
7659	013	\N	Bourbriac	22
7660	014	\N	Bourseul	22
7661	015	\N	Bréhand	22
7662	016	\N	Île-de-Bréhat	22
7663	018	\N	Brélidy	22
7664	019	\N	Bringolo	22
7665	020	\N	Broons	22
7666	021	\N	Brusvily	22
7667	023	\N	Bulat-Pestivien	22
7668	024	\N	Calanhel	22
7669	025	\N	Callac	22
7670	026	\N	Calorguen	22
7671	027	\N	Cambout	22
7672	028	\N	Camlez	22
7673	029	\N	Canihuel	22
7674	030	\N	Caouënnec-Lanvézéac	22
7675	031	\N	Carnoët	22
7676	032	\N	Caulnes	22
7677	033	\N	Caurel	22
7678	034	\N	Cavan	22
7679	035	\N	Champs-Géraux	22
7680	036	\N	Chapelle-Blanche	22
7681	037	\N	Chapelle-Neuve	22
7682	038	\N	Châtelaudren	22
7683	039	\N	Chèze	22
7684	040	\N	Coadout	22
7685	041	\N	Coatascorn	22
7686	042	\N	Coatréven	22
7687	043	\N	Coëtlogon	22
7688	044	\N	Coëtmieux	22
7689	045	\N	Cohiniac	22
7690	046	\N	Collinée	22
7691	047	\N	Corlay	22
7692	048	\N	Corseul	22
7693	049	\N	Créhen	22
7694	050	\N	Dinan	22
7695	051	\N	Dolo	22
7696	052	\N	Duault	22
7697	053	\N	Éréac	22
7698	054	\N	Erquy	22
7699	055	\N	Étables-sur-Mer	22
7700	056	\N	Évran	22
7701	057	\N	Faouët	22
7702	058	\N	Ferrière	22
7703	059	\N	Fil	22
7704	060	\N	Gausson	22
7705	061	\N	Glomel	22
7706	062	\N	Gomené	22
7707	063	\N	Gommenec'h	22
7708	064	\N	Gouarec	22
7709	065	\N	Goudelin	22
7710	066	\N	Gouray	22
7711	067	\N	Grâces	22
7712	068	\N	Grâce-Uzel	22
7713	069	\N	Guenroc	22
7714	070	\N	Guingamp	22
7715	071	\N	Guitté	22
7716	072	\N	Gurunhuel	22
7717	073	\N	Harmoye	22
7718	074	\N	Haut-Corlay	22
7719	075	\N	Hémonstoir	22
7720	076	\N	Hénanbihen	22
7721	077	\N	Hénansal	22
7722	078	\N	Hengoat	22
7723	079	\N	Hénon	22
7724	080	\N	Hermitage-Lorge	22
7725	081	\N	Hillion	22
7726	082	\N	Hinglé	22
7727	083	\N	Illifaut	22
7728	084	\N	Jugon-les-Lacs	22
7729	085	\N	Kerbors	22
7730	086	\N	Kerfot	22
7731	087	\N	Kergrist-Moëlou	22
7732	088	\N	Kerien	22
7733	090	\N	Kermaria-Sulard	22
7734	091	\N	Kermoroc'h	22
7735	092	\N	Kerpert	22
7736	093	\N	Lamballe	22
7737	094	\N	Lancieux	22
7738	095	\N	Landebaëron	22
7739	096	\N	Landébia	22
7740	097	\N	Landec	22
7741	098	\N	Landéhen	22
7742	099	\N	Lanfains	22
7743	100	\N	Langast	22
7744	101	\N	Langoat	22
7745	102	\N	Langourla	22
7746	103	\N	Langrolay-sur-Rance	22
7747	104	\N	Languédias	22
7748	105	\N	Languenan	22
7749	106	\N	Langueux	22
7750	107	\N	Laniscat	22
7751	108	\N	Lanleff	22
7752	109	\N	Lanloup	22
7753	110	\N	Lanmérin	22
7754	111	\N	Lanmodez	22
7755	112	\N	Lannebert	22
7756	113	\N	Lannion	22
7757	114	\N	Lanrelas	22
7758	115	\N	Lanrivain	22
7759	116	\N	Lanrodec	22
7760	117	\N	Lantic	22
7761	118	\N	Lanvallay	22
7762	119	\N	Lanvellec	22
7763	121	\N	Lanvollon	22
7764	122	\N	Laurenan	22
7765	123	\N	Léhon	22
7766	124	\N	Lescouët-Gouarec	22
7767	126	\N	Leslay	22
7768	127	\N	Lézardrieux	22
7769	128	\N	Locarn	22
7770	129	\N	Loc-Envel	22
7771	131	\N	Loguivy-Plougras	22
7772	132	\N	Lohuec	22
7773	133	\N	Loscouët-sur-Meu	22
7774	134	\N	Louannec	22
7775	135	\N	Louargat	22
7776	136	\N	Loudéac	22
7777	137	\N	Maël-Carhaix	22
7778	138	\N	Maël-Pestivien	22
7779	139	\N	Magoar	22
7780	140	\N	Malhoure	22
7781	141	\N	Mantallot	22
7782	143	\N	Matignon	22
7783	144	\N	Méaugon	22
7784	145	\N	Mégrit	22
7785	146	\N	Mellionnec	22
7786	147	\N	Merdrignac	22
7787	148	\N	Mérillac	22
7788	149	\N	Merléac	22
7789	150	\N	Merzer	22
7790	151	\N	Meslin	22
7791	152	\N	Minihy-Tréguier	22
7792	153	\N	Moncontour	22
7793	154	\N	Morieux	22
7794	155	\N	Motte	22
7795	156	\N	Moustéru	22
7796	157	\N	Moustoir	22
7797	158	\N	Mûr-de-Bretagne	22
7798	160	\N	Noyal	22
7799	161	\N	Pabu	22
7800	162	\N	Paimpol	22
7801	163	\N	Paule	22
7802	164	\N	Pédernec	22
7803	165	\N	Penguily	22
7804	166	\N	Penvénan	22
7805	167	\N	Perret	22
7806	168	\N	Perros-Guirec	22
7807	169	\N	Peumerit-Quintin	22
7808	170	\N	Plaine-Haute	22
7809	171	\N	Plaintel	22
7810	172	\N	Plancoët	22
7811	173	\N	Planguenoual	22
7812	174	\N	Pléboulle	22
7813	175	\N	Plédéliac	22
7814	176	\N	Plédran	22
7815	177	\N	Pléguien	22
7816	178	\N	Pléhédel	22
7817	179	\N	Fréhel	22
7818	180	\N	Plélan-le-Petit	22
7819	181	\N	Plélauff	22
7820	182	\N	Plélo	22
7821	183	\N	Plémet	22
7822	184	\N	Plémy	22
7823	185	\N	Plénée-Jugon	22
7824	186	\N	Pléneuf-Val-André	22
7825	187	\N	Plérin	22
7826	188	\N	Plerneuf	22
7827	189	\N	Plésidy	22
7828	190	\N	Pleslin-Trigavou	22
7829	191	\N	Plessala	22
7830	192	\N	Plessix-Balisson	22
7831	193	\N	Plestan	22
7832	194	\N	Plestin-les-Grèves	22
7833	195	\N	Pleubian	22
7834	196	\N	Pleudaniel	22
7835	197	\N	Pleudihen-sur-Rance	22
7836	198	\N	Pleumeur-Bodou	22
7837	199	\N	Pleumeur-Gautier	22
7838	200	\N	Pléven	22
7839	201	\N	Plévenon	22
7840	202	\N	Plévin	22
7841	203	\N	Pluc-sur-Lié	22
7842	204	\N	Ploëzal	22
7843	205	\N	Plorec-sur-Arguenon	22
7844	206	\N	Plouagat	22
7845	207	\N	Plouaret	22
7846	208	\N	Plouasne	22
7847	209	\N	Ploubalay	22
7848	210	\N	Ploubazlanec	22
7849	211	\N	Ploubezre	22
7850	212	\N	Plouëc-du-Trieux	22
7851	213	\N	Plouër-sur-Rance	22
7852	214	\N	Plouézec	22
7853	215	\N	Ploufragan	22
7854	216	\N	Plougonver	22
7855	217	\N	Plougras	22
7856	218	\N	Plougrescant	22
7857	219	\N	Plouguenast	22
7858	220	\N	Plouguernével	22
7859	221	\N	Plouguiel	22
7860	222	\N	Plouha	22
7861	223	\N	Plouisy	22
7862	224	\N	Ploulec'h	22
7863	225	\N	Ploumagoar	22
7864	226	\N	Ploumilliau	22
7865	227	\N	Plounérin	22
7866	228	\N	Plounévez-Moëdec	22
7867	229	\N	Plounévez-Quintin	22
7868	231	\N	Plourac'h	22
7869	232	\N	Plourhan	22
7870	233	\N	Plourivo	22
7871	234	\N	Plouvara	22
7872	235	\N	Plouzélambre	22
7873	236	\N	Pludual	22
7874	237	\N	Pluduno	22
7875	238	\N	Plufur	22
7876	239	\N	Plumaudan	22
7877	240	\N	Plumaugat	22
7878	241	\N	Plumieux	22
7879	242	\N	Plurien	22
7880	243	\N	Plusquellec	22
7881	244	\N	Plussulien	22
7882	245	\N	Pluzunet	22
7883	246	\N	Pommeret	22
7884	247	\N	Pommerit-Jaudy	22
7885	248	\N	Pommerit-le-Vicomte	22
7886	249	\N	Pont-Melvez	22
7887	250	\N	Pontrieux	22
7888	251	\N	Pordic	22
7889	253	\N	Pouldouran	22
7890	254	\N	Prat	22
7891	255	\N	Prénessaye	22
7892	256	\N	Quemper-Guézennec	22
7893	257	\N	Quemperven	22
7894	258	\N	Quessoy	22
7895	259	\N	Quévert	22
7896	260	\N	Quillio	22
7897	261	\N	Quintenic	22
7898	262	\N	Quintin	22
7899	263	\N	Quiou	22
7900	264	\N	Roche-Derrien	22
7901	265	\N	Rospez	22
7902	266	\N	Rostrenen	22
7903	267	\N	Rouillac	22
7904	268	\N	Ruca	22
7905	269	\N	Runan	22
7906	271	\N	Saint-Adrien	22
7907	272	\N	Saint-Agathon	22
7908	273	\N	Saint-Alban	22
7909	274	\N	Saint-André-des-Eaux	22
7910	275	\N	Saint-Barnabé	22
7911	276	\N	Saint-Bihy	22
7912	277	\N	Saint-Brandan	22
7913	278	\N	Saint-Brieuc	22
7914	279	\N	Saint-Caradec	22
7915	280	\N	Saint-Carné	22
7916	281	\N	Saint-Carreuc	22
7917	282	\N	Saint-Cast-le-Guildo	22
7918	283	\N	Saint-Clet	22
7919	284	\N	Saint-Connan	22
7920	285	\N	Saint-Connec	22
7921	286	\N	Saint-Denoual	22
7922	287	\N	Saint-Donan	22
7923	288	\N	Saint-Étienne-du-Gué-de-l'Isle	22
7924	289	\N	Saint-Fiacre	22
7925	290	\N	Saint-Gelven	22
7926	291	\N	Saint-Gildas	22
7927	292	\N	Saint-Gilles-du-Mené	22
7928	293	\N	Saint-Gilles-les-Bois	22
7929	294	\N	Saint-Gilles-Pligeaux	22
7930	295	\N	Saint-Gilles-Vieux-Marché	22
7931	296	\N	Saint-Glen	22
7932	297	\N	Saint-Gouéno	22
7933	298	\N	Saint-Guen	22
7934	299	\N	Saint-Hélen	22
7935	300	\N	Saint-Hervé	22
7936	302	\N	Saint-Jacut-de-la-Mer	22
7937	303	\N	Saint-Jacut-du-Mené	22
7938	304	\N	Saint-Jean-Kerdaniel	22
7939	305	\N	Saint-Jouan-de-l'Isle	22
7940	306	\N	Saint-Judoce	22
7941	307	\N	Saint-Julien	22
7942	308	\N	Saint-Juvat	22
7943	309	\N	Saint-Launeuc	22
7944	310	\N	Saint-Laurent	22
7945	311	\N	Saint-Lormel	22
7946	312	\N	Saint-Maden	22
7947	313	\N	Saint-Martin-des-Prés	22
7948	314	\N	Saint-Maudan	22
7949	315	\N	Saint-Maudez	22
7950	316	\N	Saint-Mayeux	22
7951	317	\N	Saint-Méloir-des-Bois	22
7952	318	\N	Saint-Michel-de-Plélan	22
7953	319	\N	Saint-Michel-en-Grève	22
7954	320	\N	Saint-Nicodème	22
7955	321	\N	Saint-Nicolas-du-Pélem	22
7956	322	\N	Saint-Péver	22
7957	323	\N	Saint-Pôtan	22
7958	324	\N	Saint-Quay-Perros	22
7959	325	\N	Saint-Quay-Portrieux	22
7960	326	\N	Saint-Rieul	22
7961	327	\N	Saint-Samson-sur-Rance	22
7962	328	\N	Saint-Servais	22
7963	330	\N	Saint-Thélo	22
7964	331	\N	Sainte-Tréphine	22
7965	332	\N	Saint-Trimoël	22
7966	333	\N	Saint-Vran	22
7967	334	\N	Saint-Igeaux	22
7968	335	\N	Senven-Léhart	22
7969	337	\N	Sévignac	22
7970	338	\N	Squiffiec	22
7971	339	\N	Taden	22
7972	340	\N	Tonquédec	22
7973	341	\N	Tramain	22
7974	342	\N	Trébédan	22
7975	343	\N	Trébeurden	22
7976	344	\N	Trébrivan	22
7977	345	\N	Trébry	22
7978	346	\N	Trédaniel	22
7979	347	\N	Trédarzec	22
7980	348	\N	Trédias	22
7981	349	\N	Trédrez-Locquémeau	22
7982	350	\N	Tréduder	22
7983	351	\N	Treffrin	22
7984	352	\N	Tréfumel	22
7985	353	\N	Trégastel	22
7986	354	\N	Tréglamus	22
7987	356	\N	Trégomeur	22
7988	357	\N	Trégon	22
7989	358	\N	Trégonneau	22
7990	359	\N	Trégrom	22
7991	360	\N	Trégueux	22
7992	361	\N	Tréguidel	22
7993	362	\N	Tréguier	22
7994	363	\N	Trélévern	22
7995	364	\N	Trélivan	22
7996	365	\N	Trémargat	22
7997	366	\N	Trémel	22
7998	367	\N	Tréméloir	22
7999	368	\N	Tréméreuc	22
8000	369	\N	Trémeur	22
8001	370	\N	Tréméven	22
8002	371	\N	Trémorel	22
8003	372	\N	Trémuson	22
8004	373	\N	Tréogan	22
8005	375	\N	Tressignaux	22
8006	376	\N	Trévé	22
8007	377	\N	Tréveneuc	22
8008	378	\N	Trévérec	22
8009	379	\N	Trévou-Tréguignec	22
8010	380	\N	Trévron	22
8011	381	\N	Trézény	22
8012	383	\N	Troguéry	22
8013	384	\N	Uzel	22
8014	385	\N	Vicomté-sur-Rance	22
8015	386	\N	Vieux-Bourg	22
8016	387	\N	Vieux-Marché	22
8017	388	\N	Vildé-Guingalan	22
8018	389	\N	Yffiniac	22
8019	390	\N	Yvias	22
8020	391	\N	Yvignac-la-Tour	22
8021	001	\N	Ahun	23
8022	002	\N	Ajain	23
8023	003	\N	Alleyrat	23
8024	004	\N	Anzême	23
8025	005	\N	Arfeuille-Châtain	23
8026	006	\N	Arrènes	23
8027	007	\N	Ars	23
8028	008	\N	Aubusson	23
8029	009	\N	Auge	23
8030	010	\N	Augères	23
8031	011	\N	Aulon	23
8032	012	\N	Auriat	23
8033	013	\N	Auzances	23
8034	014	\N	Azat-Châtenet	23
8035	015	\N	Azerables	23
8036	016	\N	Banize	23
8037	017	\N	Basville	23
8038	018	\N	Bazelat	23
8039	019	\N	Beissat	23
8040	020	\N	Bellegarde-en-Marche	23
8041	021	\N	Bénévent-l'Abbaye	23
8042	022	\N	Bétête	23
8043	023	\N	Blaudeix	23
8044	024	\N	Blessac	23
8045	025	\N	Bonnat	23
8046	026	\N	Bord-Saint-Georges	23
8047	027	\N	Bosmoreau-les-Mines	23
8048	028	\N	Bosroger	23
8049	029	\N	Bourg-d'Hem	23
8050	030	\N	Bourganeuf	23
8051	031	\N	Boussac	23
8052	032	\N	Boussac-Bourg	23
8053	033	\N	Brionne	23
8054	034	\N	Brousse	23
8055	035	\N	Budelière	23
8056	036	\N	Bussière-Dunoise	23
8057	037	\N	Bussière-Nouvelle	23
8058	038	\N	Bussière-Saint-Georges	23
8059	039	\N	Celle-Dunoise	23
8060	040	\N	Celle-sous-Gouzon	23
8061	041	\N	Cellette	23
8062	042	\N	Ceyroux	23
8063	043	\N	Chamberaud	23
8064	044	\N	Chambon-Sainte-Croix	23
8065	045	\N	Chambon-sur-Voueize	23
8066	046	\N	Chambonchard	23
8067	047	\N	Chamborand	23
8068	048	\N	Champagnat	23
8069	049	\N	Champsanglard	23
8070	050	\N	Chapelle-Baloue	23
8071	051	\N	Chapelle-Saint-Martial	23
8072	052	\N	Chapelle-Taillefert	23
8073	053	\N	Chard	23
8074	054	\N	Charron	23
8075	055	\N	Châtelard	23
8076	056	\N	Châtelus-le-Marcheix	23
8077	057	\N	Châtelus-Malvaleix	23
8078	058	\N	Chauchet	23
8079	059	\N	Chaussade	23
8080	060	\N	Chavanat	23
8081	061	\N	Chénérailles	23
8082	062	\N	Chéniers	23
8083	063	\N	Clairavaux	23
8084	064	\N	Clugnat	23
8085	065	\N	Colondannes	23
8086	066	\N	Compas	23
8087	067	\N	Courtine	23
8088	068	\N	Cressat	23
8089	069	\N	Crocq	23
8090	070	\N	Crozant	23
8091	071	\N	Croze	23
8092	072	\N	Domeyrot	23
8093	073	\N	Dontreix	23
8094	074	\N	Donzeil	23
8095	075	\N	Dun-le-Palestel	23
8096	076	\N	Évaux-les-Bains	23
8097	077	\N	Faux-la-Montagne	23
8098	078	\N	Faux-Mazuras	23
8099	079	\N	Felletin	23
8100	080	\N	Féniers	23
8101	081	\N	Flayat	23
8102	082	\N	Fleurat	23
8103	083	\N	Fontanières	23
8104	084	\N	Forêt-du-Temple	23
8105	086	\N	Fransèches	23
8106	087	\N	Fresselines	23
8107	088	\N	Gartempe	23
8108	089	\N	Genouillac	23
8109	090	\N	Gentioux-Pigerolles	23
8110	091	\N	Gioux	23
8111	092	\N	Glénic	23
8112	093	\N	Gouzon	23
8113	095	\N	Grand-Bourg	23
8114	096	\N	Guéret	23
8115	097	\N	Issoudun-Létrieix	23
8116	098	\N	Jalesches	23
8117	099	\N	Janaillat	23
8118	100	\N	Jarnages	23
8119	101	\N	Jouillat	23
8120	102	\N	Ladapeyre	23
8121	103	\N	Lafat	23
8122	104	\N	Lavaufranche	23
8123	105	\N	Lavaveix-les-Mines	23
8124	106	\N	Lépaud	23
8125	107	\N	Lépinas	23
8126	108	\N	Leyrat	23
8127	109	\N	Linard	23
8128	110	\N	Lioux-les-Monges	23
8129	111	\N	Lizières	23
8130	112	\N	Lourdoueix-Saint-Pierre	23
8131	113	\N	Lupersat	23
8132	114	\N	Lussat	23
8133	115	\N	Magnat-l'Étrange	23
8134	116	\N	Mainsat	23
8135	117	\N	Maison-Feyne	23
8136	118	\N	Maisonnisses	23
8137	119	\N	Malleret	23
8138	120	\N	Malleret-Boussac	23
8139	121	\N	Malval	23
8140	122	\N	Mansat-la-Courrière	23
8141	123	\N	Mars	23
8142	124	\N	Marsac	23
8143	125	\N	Mas-d'Artige	23
8144	126	\N	Masbaraud-Mérignat	23
8145	127	\N	Mautes	23
8146	128	\N	Mazeirat	23
8147	129	\N	Mazière-aux-Bons-Hommes	23
8148	130	\N	Méasnes	23
8149	131	\N	Mérinchal	23
8150	132	\N	Montaigut-le-Blanc	23
8151	133	\N	Montboucher	23
8152	134	\N	Monteil-au-Vicomte	23
8153	136	\N	Mortroux	23
8154	137	\N	Mourioux-Vieilleville	23
8155	138	\N	Moutier-d'Ahun	23
8156	139	\N	Moutier-Malcard	23
8157	140	\N	Moutier-Rozeille	23
8158	141	\N	Naillat	23
8159	142	\N	Néoux	23
8160	143	\N	Noth	23
8161	144	\N	Nouaille	23
8162	145	\N	Nouhant	23
8163	146	\N	Nouzerines	23
8164	147	\N	Nouzerolles	23
8165	148	\N	Nouziers	23
8166	149	\N	Parsac	23
8167	150	\N	Peyrabout	23
8168	151	\N	Peyrat-la-Nonière	23
8169	152	\N	Pierrefitte	23
8170	154	\N	Pionnat	23
8171	155	\N	Pontarion	23
8172	156	\N	Pontcharraud	23
8173	157	\N	Pouge	23
8174	158	\N	Poussanges	23
8175	159	\N	Puy-Malsignat	23
8176	160	\N	Reterre	23
8177	161	\N	Rimondeix	23
8178	162	\N	Roches	23
8179	164	\N	Rougnat	23
8180	165	\N	Royère-de-Vassivière	23
8181	166	\N	Sagnat	23
8182	167	\N	Sannat	23
8183	168	\N	Sardent	23
8184	169	\N	Saunière	23
8185	170	\N	Savennes	23
8186	171	\N	Sermur	23
8187	172	\N	Serre-Bussière-Vieille	23
8188	173	\N	Soubrebost	23
8189	174	\N	Soumans	23
8190	175	\N	Sous-Parsat	23
8191	176	\N	Souterraine	23
8192	177	\N	Saint-Agnant-de-Versillat	23
8193	178	\N	Saint-Agnant-près-Crocq	23
8194	179	\N	Saint-Alpinien	23
8195	180	\N	Saint-Amand	23
8196	181	\N	Saint-Amand-Jartoudeix	23
8197	182	\N	Saint-Avit-de-Tardes	23
8198	183	\N	Saint-Avit-le-Pauvre	23
8199	184	\N	Saint-Bard	23
8200	185	\N	Saint-Chabrais	23
8201	186	\N	Saint-Christophe	23
8202	187	\N	Saint-Dizier-la-Tour	23
8203	188	\N	Saint-Dizier-les-Domaines	23
8204	189	\N	Saint-Dizier-Leyrenne	23
8205	190	\N	Saint-Domet	23
8206	191	\N	Saint-Éloi	23
8207	192	\N	Saint-Étienne-de-Fursac	23
8208	193	\N	Sainte-Feyre	23
8209	194	\N	Sainte-Feyre-la-Montagne	23
8210	195	\N	Saint-Fiel	23
8211	196	\N	Saint-Frion	23
8212	197	\N	Saint-Georges-la-Pouge	23
8213	198	\N	Saint-Georges-Nigremont	23
8214	199	\N	Saint-Germain-Beaupré	23
8215	200	\N	Saint-Goussaud	23
8216	201	\N	Saint-Hilaire-la-Plaine	23
8217	202	\N	Saint-Hilaire-le-Château	23
8218	203	\N	Saint-Julien-la-Genête	23
8219	204	\N	Saint-Julien-le-Châtel	23
8220	205	\N	Saint-Junien-la-Bregère	23
8221	206	\N	Saint-Laurent	23
8222	207	\N	Saint-Léger-Bridereix	23
8223	208	\N	Saint-Léger-le-Guérétois	23
8224	209	\N	Saint-Loup	23
8225	210	\N	Saint-Maixant	23
8226	211	\N	Saint-Marc-à-Frongier	23
8227	212	\N	Saint-Marc-à-Loubaud	23
8228	213	\N	Saint-Marien	23
8229	214	\N	Saint-Martial-le-Mont	23
8230	215	\N	Saint-Martial-le-Vieux	23
8231	216	\N	Saint-Martin-Château	23
8232	217	\N	Saint-Martin-Sainte-Catherine	23
8233	218	\N	Saint-Maurice-près-Crocq	23
8234	219	\N	Saint-Maurice-la-Souterraine	23
8235	220	\N	Saint-Médard-la-Rochette	23
8236	221	\N	Saint-Merd-la-Breuille	23
8237	222	\N	Saint-Michel-de-Veisse	23
8238	223	\N	Saint-Moreil	23
8239	224	\N	Saint-Oradoux-de-Chirouze	23
8240	225	\N	Saint-Oradoux-près-Crocq	23
8241	226	\N	Saint-Pardoux-d'Arnet	23
8242	227	\N	Saint-Pardoux-Morterolles	23
8243	228	\N	Saint-Pardoux-le-Neuf	23
8244	229	\N	Saint-Pardoux-les-Cards	23
8245	230	\N	Saint-Pierre-Chérignat	23
8246	231	\N	Saint-Pierre-de-Fursac	23
8247	232	\N	Saint-Pierre-Bellevue	23
8248	233	\N	Saint-Pierre-le-Bost	23
8249	234	\N	Saint-Priest	23
8250	235	\N	Saint-Priest-la-Feuille	23
8251	236	\N	Saint-Priest-la-Plaine	23
8252	237	\N	Saint-Priest-Palus	23
8253	238	\N	Saint-Quentin-la-Chabanne	23
8254	239	\N	Saint-Sébastien	23
8255	240	\N	Saint-Silvain-Bas-le-Roc	23
8256	241	\N	Saint-Silvain-Bellegarde	23
8257	242	\N	Saint-Silvain-Montaigut	23
8258	243	\N	Saint-Silvain-sous-Toulx	23
8259	244	\N	Saint-Sulpice-le-Dunois	23
8260	245	\N	Saint-Sulpice-le-Guérétois	23
8261	246	\N	Saint-Sulpice-les-Champs	23
8262	247	\N	Saint-Vaury	23
8263	248	\N	Saint-Victor-en-Marche	23
8264	249	\N	Saint-Yrieix-la-Montagne	23
8265	250	\N	Saint-Yrieix-les-Bois	23
8266	251	\N	Tardes	23
8267	252	\N	Tercillat	23
8268	253	\N	Thauron	23
8269	254	\N	Toulx-Sainte-Croix	23
8270	255	\N	Trois-Fonds	23
8271	257	\N	Vallière	23
8272	258	\N	Vareilles	23
8273	259	\N	Verneiges	23
8274	260	\N	Vidaillat	23
8275	261	\N	Viersat	23
8276	262	\N	Vigeville	23
8277	263	\N	Villard	23
8278	264	\N	Villedieu	23
8279	265	\N	Villeneuve	23
8280	266	\N	Villetelle	23
8281	001	\N	Abjat-sur-Bandiat	24
8282	002	\N	Agonac	24
8283	004	\N	Ajat	24
8284	005	\N	Alles-sur-Dordogne	24
8285	006	\N	Allas-les-Mines	24
8286	007	\N	Allemans	24
8287	008	\N	Angoisse	24
8288	009	\N	Anlhiac	24
8289	010	\N	Annesse-et-Beaulieu	24
8290	011	\N	Antonne-et-Trigonant	24
8291	012	\N	Archignac	24
8292	013	\N	Atur	24
8293	014	\N	Aubas	24
8294	015	\N	Audrix	24
8295	016	\N	Augignac	24
8296	018	\N	Auriac-du-Périgord	24
8297	019	\N	Azerat	24
8298	020	\N	Bachellerie	24
8299	021	\N	Badefols-d'Ans	24
8300	022	\N	Badefols-sur-Dordogne	24
8301	023	\N	Baneuil	24
8302	024	\N	Bardou	24
8303	025	\N	Bars	24
8304	026	\N	Bassillac	24
8305	027	\N	Bayac	24
8306	028	\N	Beaumont-du-Périgord	24
8307	029	\N	Beaupouyet	24
8308	030	\N	Beauregard-de-Terrasson	24
8309	031	\N	Beauregard-et-Bassac	24
8310	032	\N	Beauronne	24
8311	033	\N	Beaussac	24
8312	034	\N	Beleymas	24
8313	035	\N	Belvès	24
8314	036	\N	Berbiguières	24
8315	037	\N	Bergerac	24
8316	038	\N	Bertric-Burée	24
8317	039	\N	Besse	24
8318	040	\N	Beynac-et-Cazenac	24
8319	041	\N	Bézenac	24
8320	042	\N	Biras	24
8321	043	\N	Biron	24
8322	044	\N	Blis-et-Born	24
8323	045	\N	Boisse	24
8324	046	\N	Boisseuilh	24
8325	047	\N	Boissière-d'Ans	24
8326	048	\N	Bonneville-et-Saint-Avit-de-Fumadières	24
8327	050	\N	Borrèze	24
8328	051	\N	Bosset	24
8329	052	\N	Bouillac	24
8330	053	\N	Boulazac	24
8331	054	\N	Bouniagues	24
8332	055	\N	Bourdeilles	24
8333	056	\N	Bourdeix	24
8334	057	\N	Bourg-des-Maisons	24
8335	058	\N	Bourg-du-Bost	24
8336	059	\N	Bourgnac	24
8337	060	\N	Bourniquel	24
8338	061	\N	Bourrou	24
8339	062	\N	Bouteilles-Saint-Sébastien	24
8340	063	\N	Bouzic	24
8341	064	\N	Brantôme	24
8342	065	\N	Breuilh	24
8343	066	\N	Brouchaud	24
8344	067	\N	Bugue	24
8345	068	\N	Buisson-de-Cadouin	24
8346	069	\N	Bussac	24
8347	070	\N	Busserolles	24
8348	071	\N	Bussière-Badil	24
8349	073	\N	Calès	24
8350	074	\N	Calviac-en-Périgord	24
8351	075	\N	Campagnac-lès-Quercy	24
8352	076	\N	Campagne	24
8353	077	\N	Campsegret	24
8354	079	\N	Cantillac	24
8355	080	\N	Capdrot	24
8356	081	\N	Carlux	24
8357	082	\N	Carsac-Aillac	24
8358	083	\N	Carsac-de-Gurson	24
8359	084	\N	Carves	24
8360	085	\N	Cassagne	24
8361	086	\N	Castelnaud-la-Chapelle	24
8362	087	\N	Castels	24
8363	088	\N	Cause-de-Clérans	24
8364	089	\N	Cazoulès	24
8365	090	\N	Celles	24
8366	091	\N	Cénac-et-Saint-Julien	24
8367	092	\N	Cendrieux	24
8368	093	\N	Cercles	24
8369	094	\N	Chalagnac	24
8370	095	\N	Chaleix	24
8371	096	\N	Champagnac-de-Belair	24
8372	097	\N	Champagne-et-Fontaine	24
8373	098	\N	Champcevinel	24
8374	099	\N	Champeaux-et-la-Chapelle-Pommier	24
8375	100	\N	Champniers-et-Reilhac	24
8376	101	\N	Champs-Romain	24
8377	102	\N	Chancelade	24
8378	103	\N	Change	24
8379	104	\N	Chantérac	24
8380	105	\N	Chapdeuil	24
8381	106	\N	Chapelle-Aubareil	24
8382	107	\N	Chapelle-Faucher	24
8383	108	\N	Chapelle-Gonaguet	24
8384	109	\N	Chapelle-Grésignac	24
8385	110	\N	Chapelle-Montabourlet	24
8386	111	\N	Chapelle-Montmoreau	24
8387	113	\N	Chapelle-Saint-Jean	24
8388	114	\N	Chassaignes	24
8389	115	\N	Château-l'Évêque	24
8390	116	\N	Châtres	24
8391	117	\N	Chavagnac	24
8392	118	\N	Chenaud	24
8393	119	\N	Cherval	24
8394	120	\N	Cherveix-Cubas	24
8395	121	\N	Chourgnac	24
8396	122	\N	Cladech	24
8397	123	\N	Clermont-de-Beauregard	24
8398	124	\N	Clermont-d'Excideuil	24
8399	126	\N	Colombier	24
8400	127	\N	Coly	24
8401	128	\N	Comberanche-et-Épeluche	24
8402	129	\N	Condat-sur-Trincou	24
8403	130	\N	Condat-sur-Vézère	24
8404	131	\N	Connezac	24
8405	132	\N	Conne-de-Labarde	24
8406	133	\N	Coquille	24
8407	134	\N	Corgnac-sur-l'Isle	24
8408	135	\N	Cornille	24
8409	136	\N	Coubjours	24
8410	137	\N	Coulaures	24
8411	138	\N	Coulounieix-Chamiers	24
8412	139	\N	Coursac	24
8413	140	\N	Cours-de-Pile	24
8414	141	\N	Coutures	24
8415	142	\N	Coux-et-Bigaroque	24
8416	143	\N	Couze-et-Saint-Front	24
8417	144	\N	Creyssac	24
8418	145	\N	Creysse	24
8419	146	\N	Creyssensac-et-Pissot	24
8420	147	\N	Cubjac	24
8421	148	\N	Cunèges	24
8422	150	\N	Daglan	24
8423	151	\N	Doissat	24
8424	152	\N	Domme	24
8425	153	\N	Dornac	24
8426	154	\N	Douchapt	24
8427	155	\N	Douville	24
8428	156	\N	Douze	24
8429	157	\N	Douzillac	24
8430	158	\N	Dussac	24
8431	159	\N	Échourgnac	24
8432	160	\N	Église-Neuve-de-Vergt	24
8433	161	\N	Église-Neuve-d'Issac	24
8434	162	\N	Escoire	24
8435	163	\N	Étouars	24
8436	164	\N	Excideuil	24
8437	165	\N	Eygurande-et-Gardedeuil	24
8438	166	\N	Eyliac	24
8439	167	\N	Eymet	24
8440	168	\N	Plaisance	24
8441	170	\N	Eyvirat	24
8442	171	\N	Eyzerac	24
8443	172	\N	Eyzies-de-Tayac-Sireuil	24
8444	174	\N	Fanlac	24
8445	175	\N	Farges	24
8446	176	\N	Faurilles	24
8447	177	\N	Faux	24
8448	178	\N	Festalemps	24
8449	179	\N	Feuillade	24
8450	180	\N	Firbeix	24
8451	181	\N	Flaugeac	24
8452	182	\N	Fleix	24
8453	183	\N	Fleurac	24
8454	184	\N	Florimont-Gaumier	24
8455	186	\N	Fonroque	24
8456	188	\N	Fossemagne	24
8457	189	\N	Fougueyrolles	24
8458	190	\N	Fouleix	24
8459	191	\N	Fraisse	24
8460	192	\N	Gabillou	24
8461	193	\N	Gageac-et-Rouillac	24
8462	194	\N	Gardonne	24
8463	195	\N	Gaugeac	24
8464	196	\N	Génis	24
8465	197	\N	Ginestet	24
8466	198	\N	Gonterie-Boulouneix	24
8467	199	\N	Gout-Rossignol	24
8468	200	\N	Grand-Brassac	24
8469	202	\N	Granges-d'Ans	24
8470	203	\N	Graulges	24
8471	204	\N	Grèzes	24
8472	205	\N	Grignols	24
8473	206	\N	Grives	24
8474	207	\N	Groléjac	24
8475	208	\N	Grun-Bordas	24
8476	209	\N	Hautefaye	24
8477	210	\N	Hautefort	24
8478	211	\N	Issac	24
8479	212	\N	Issigeac	24
8480	213	\N	Jaure	24
8481	214	\N	Javerlhac-et-la-Chapelle-Saint-Robert	24
8482	215	\N	Jayac	24
8483	216	\N	Jemaye	24
8484	217	\N	Journiac	24
8485	218	\N	Jumilhac-le-Grand	24
8486	219	\N	Labouquerie	24
8487	220	\N	Lacropte	24
8488	221	\N	Rudeau-Ladosse	24
8489	222	\N	Force	24
8490	223	\N	Lalinde	24
8491	224	\N	Lamonzie-Montastruc	24
8492	225	\N	Lamonzie-Saint-Martin	24
8493	226	\N	Lamothe-Montravel	24
8494	227	\N	Lanouaille	24
8495	228	\N	Lanquais	24
8496	229	\N	Lardin-Saint-Lazare	24
8497	230	\N	Larzac	24
8498	231	\N	Lavalade	24
8499	232	\N	Lavaur	24
8500	233	\N	Laveyssière	24
8501	234	\N	Lèches	24
8502	235	\N	Léguillac-de-Cercles	24
8503	236	\N	Léguillac-de-l'Auche	24
8504	237	\N	Lembras	24
8505	238	\N	Lempzours	24
8506	239	\N	Ligueux	24
8507	240	\N	Limeuil	24
8508	241	\N	Limeyrat	24
8509	242	\N	Liorac-sur-Louyre	24
8510	243	\N	Lisle	24
8511	244	\N	Lolme	24
8512	245	\N	Loubejac	24
8513	246	\N	Lunas	24
8514	247	\N	Lusignac	24
8515	248	\N	Lussas-et-Nontronneau	24
8516	249	\N	Manaurie	24
8517	251	\N	Manzac-sur-Vern	24
8518	252	\N	Marcillac-Saint-Quentin	24
8519	253	\N	Mareuil	24
8520	254	\N	Marnac	24
8521	255	\N	Marquay	24
8522	256	\N	Marsac-sur-l'Isle	24
8523	257	\N	Marsalès	24
8524	258	\N	Marsaneix	24
8525	259	\N	Maurens	24
8526	260	\N	Mauzac-et-Grand-Castang	24
8527	261	\N	Mauzens-et-Miremont	24
8528	262	\N	Mayac	24
8529	263	\N	Mazeyrolles	24
8530	264	\N	Ménesplet	24
8531	266	\N	Mensignac	24
8532	267	\N	Mescoules	24
8533	268	\N	Meyrals	24
8534	269	\N	Mialet	24
8535	270	\N	Milhac-d'Auberoche	24
8536	271	\N	Milhac-de-Nontron	24
8537	272	\N	Minzac	24
8538	273	\N	Molières	24
8539	274	\N	Monbazillac	24
8540	276	\N	Monestier	24
8541	277	\N	Monfaucon	24
8542	278	\N	Monmadalès	24
8543	279	\N	Monmarvès	24
8544	280	\N	Monpazier	24
8545	281	\N	Monsac	24
8546	282	\N	Monsaguel	24
8547	283	\N	Monsec	24
8548	284	\N	Montagnac-d'Auberoche	24
8549	285	\N	Montagnac-la-Crempse	24
8550	286	\N	Montagrier	24
8551	287	\N	Montaut	24
8552	288	\N	Montazeau	24
8553	289	\N	Montcaret	24
8554	290	\N	Montferrand-du-Périgord	24
8555	291	\N	Montignac	24
8556	292	\N	Montpeyroux	24
8557	293	\N	Monplaisant	24
8558	294	\N	Montpon-Ménestérol	24
8559	295	\N	Montrem	24
8560	296	\N	Mouleydier	24
8561	297	\N	Moulin-Neuf	24
8562	298	\N	Mouzens	24
8563	299	\N	Mussidan	24
8564	300	\N	Nabirat	24
8565	301	\N	Nadaillac	24
8566	302	\N	Nailhac	24
8567	303	\N	Nanteuil-Auriac-de-Bourzac	24
8568	304	\N	Nantheuil	24
8569	305	\N	Nanthiat	24
8570	306	\N	Nastringues	24
8571	307	\N	Naussannes	24
8572	308	\N	Négrondes	24
8573	309	\N	Neuvic	24
8574	310	\N	Nojals-et-Clotte	24
8575	311	\N	Nontron	24
8576	312	\N	Notre-Dame-de-Sanilhac	24
8577	313	\N	Orliac	24
8578	314	\N	Orliaguet	24
8579	316	\N	Parcoul	24
8580	317	\N	Paulin	24
8581	318	\N	Paunat	24
8582	319	\N	Paussac-et-Saint-Vivien	24
8583	320	\N	Payzac	24
8584	321	\N	Pazayac	24
8585	322	\N	Périgueux	24
8586	323	\N	Petit-Bersac	24
8587	324	\N	Peyrignac	24
8588	325	\N	Peyrillac-et-Millac	24
8589	326	\N	Peyzac-le-Moustier	24
8590	327	\N	Pezuls	24
8591	328	\N	Piégut-Pluviers	24
8592	329	\N	Pizou	24
8593	330	\N	Plazac	24
8594	331	\N	Pomport	24
8595	333	\N	Ponteyraud	24
8596	334	\N	Pontours	24
8597	335	\N	Port-Sainte-Foy-et-Ponchapt	24
8598	336	\N	Prats-de-Carlux	24
8599	337	\N	Prats-du-Périgord	24
8600	338	\N	Pressignac-Vicq	24
8601	339	\N	Preyssac-d'Excideuil	24
8602	340	\N	Prigonrieux	24
8603	341	\N	Proissans	24
8604	343	\N	Puymangou	24
8605	344	\N	Puyrenier	24
8606	345	\N	Queyssac	24
8607	346	\N	Quinsac	24
8608	347	\N	Rampieux	24
8609	348	\N	Razac-d'Eymet	24
8610	349	\N	Razac-de-Saussignac	24
8611	350	\N	Razac-sur-l'Isle	24
8612	351	\N	Ribagnac	24
8613	352	\N	Ribérac	24
8614	353	\N	Rochebeaucourt-et-Argentine	24
8615	354	\N	Roche-Chalais	24
8616	355	\N	Roque-Gageac	24
8617	356	\N	Rouffignac-Saint-Cernin-de-Reilhac	24
8618	357	\N	Rouffignac-de-Sigoulès	24
8619	359	\N	Sadillac	24
8620	360	\N	Sagelat	24
8621	361	\N	Saint-Agne	24
8622	362	\N	Sainte-Alvère	24
8623	363	\N	Saint-Amand-de-Belvès	24
8624	364	\N	Saint-Amand-de-Coly	24
8625	365	\N	Saint-Amand-de-Vergt	24
8626	366	\N	Saint-André-d'Allas	24
8627	367	\N	Saint-André-de-Double	24
8628	368	\N	Saint-Antoine-Cumond	24
8629	369	\N	Saint-Antoine-d'Auberoche	24
8630	370	\N	Saint-Antoine-de-Breuilh	24
8631	371	\N	Saint-Aquilin	24
8632	372	\N	Saint-Astier	24
8633	373	\N	Saint-Aubin-de-Cadelech	24
8634	374	\N	Saint-Aubin-de-Lanquais	24
8635	375	\N	Saint-Aubin-de-Nabirat	24
8636	376	\N	Saint-Aulaye	24
8637	377	\N	Saint-Avit-de-Vialard	24
8638	378	\N	Saint-Avit-Rivière	24
8639	379	\N	Saint-Avit-Sénieur	24
8640	380	\N	Saint-Barthélemy-de-Bellegarde	24
8641	381	\N	Saint-Barthélemy-de-Bussière	24
8642	382	\N	Saint-Capraise-de-Lalinde	24
8643	383	\N	Saint-Capraise-d'Eymet	24
8644	384	\N	Saint-Cassien	24
8645	385	\N	Saint-Cernin-de-Labarde	24
8646	386	\N	Saint-Cernin-de-l'Herm	24
8647	388	\N	Saint-Chamassy	24
8648	389	\N	Saint-Cirq	24
8649	390	\N	Saint-Crépin-d'Auberoche	24
8650	391	\N	Saint-Crépin-de-Richemont	24
8651	392	\N	Saint-Crépin-et-Carlucet	24
8652	393	\N	Sainte-Croix	24
8653	394	\N	Sainte-Croix-de-Mareuil	24
8654	395	\N	Saint-Cybranet	24
8655	396	\N	Saint-Cyprien	24
8656	397	\N	Saint-Cyr-les-Champagnes	24
8657	398	\N	Saint-Estèphe	24
8658	399	\N	Saint-Étienne-de-Puycorbier	24
8659	401	\N	Sainte-Eulalie-d'Ans	24
8660	402	\N	Sainte-Eulalie-d'Eymet	24
8661	403	\N	Saint-Félix-de-Bourdeilles	24
8662	404	\N	Saint-Félix-de-Reillac-et-Mortemart	24
8663	405	\N	Saint-Félix-de-Villadeix	24
8664	406	\N	Sainte-Foy-de-Belvès	24
8665	407	\N	Sainte-Foy-de-Longas	24
8666	408	\N	Saint-Front-d'Alemps	24
8667	409	\N	Saint-Front-de-Pradoux	24
8668	410	\N	Saint-Front-la-Rivière	24
8669	411	\N	Saint-Front-sur-Nizonne	24
8670	412	\N	Saint-Geniès	24
8671	413	\N	Saint-Georges-Blancaneix	24
8672	414	\N	Saint-Georges-de-Montclard	24
8673	415	\N	Saint-Géraud-de-Corps	24
8674	416	\N	Saint-Germain-de-Belvès	24
8675	417	\N	Saint-Germain-des-Prés	24
8676	418	\N	Saint-Germain-du-Salembre	24
8677	419	\N	Saint-Germain-et-Mons	24
8678	420	\N	Saint-Géry	24
8679	421	\N	Saint-Geyrac	24
8680	422	\N	Saint-Hilaire-d'Estissac	24
8681	423	\N	Sainte-Innocence	24
8682	424	\N	Saint-Jean-d'Ataux	24
8683	425	\N	Saint-Jean-de-Côle	24
8684	426	\N	Saint-Jean-d'Estissac	24
8685	427	\N	Saint-Jean-d'Eyraud	24
8686	428	\N	Saint-Jory-de-Chalais	24
8687	429	\N	Saint-Jory-las-Bloux	24
8688	430	\N	Saint-Julien-de-Bourdeilles	24
8689	431	\N	Saint-Julien-de-Crempse	24
8690	432	\N	Saint-Julien-de-Lampon	24
8691	433	\N	Saint-Julien-d'Eymet	24
8692	434	\N	Saint-Just	24
8693	435	\N	Saint-Laurent-des-Bâtons	24
8694	436	\N	Saint-Laurent-des-Hommes	24
8695	437	\N	Saint-Laurent-des-Vignes	24
8696	438	\N	Saint-Laurent-la-Vallée	24
8697	439	\N	Saint-Laurent-sur-Manoire	24
8698	441	\N	Saint-Léon-d'Issigeac	24
8699	442	\N	Saint-Léon-sur-l'Isle	24
8700	443	\N	Saint-Léon-sur-Vézère	24
8701	444	\N	Saint-Louis-en-l'Isle	24
8702	445	\N	Saint-Marcel-du-Périgord	24
8703	446	\N	Saint-Marcory	24
8704	447	\N	Sainte-Marie-de-Chignac	24
8705	448	\N	Saint-Martial-d'Albarède	24
8706	449	\N	Saint-Martial-d'Artenset	24
8707	450	\N	Saint-Martial-de-Nabirat	24
8708	451	\N	Saint-Martial-de-Valette	24
8709	452	\N	Saint-Martial-Viveyrol	24
8710	453	\N	Saint-Martin-de-Fressengeas	24
8711	454	\N	Saint-Martin-de-Gurson	24
8712	455	\N	Saint-Martin-de-Ribérac	24
8713	456	\N	Saint-Martin-des-Combes	24
8714	457	\N	Saint-Martin-l'Astier	24
8715	458	\N	Saint-Martin-le-Pin	24
8716	459	\N	Saint-Maime-de-Péreyrol	24
8717	460	\N	Saint-Méard-de-Drône	24
8718	461	\N	Saint-Méard-de-Gurçon	24
8719	462	\N	Saint-Médard-de-Mussidan	24
8720	463	\N	Saint-Médard-d'Excideuil	24
8721	464	\N	Saint-Mesmin	24
8722	465	\N	Saint-Michel-de-Double	24
8723	466	\N	Saint-Michel-de-Montaigne	24
8724	468	\N	Saint-Michel-de-Villadeix	24
8725	470	\N	Sainte-Mondane	24
8726	471	\N	Sainte-Nathalène	24
8727	472	\N	Saint-Nexans	24
8728	473	\N	Sainte-Orse	24
8729	474	\N	Saint-Pancrace	24
8730	475	\N	Saint-Pantaly-d'Ans	24
8731	476	\N	Saint-Pantaly-d'Excideuil	24
8732	477	\N	Saint-Pardoux-de-Drône	24
8733	478	\N	Saint-Pardoux-et-Vielvic	24
8734	479	\N	Saint-Pardoux-la-Rivière	24
8735	480	\N	Saint-Paul-de-Serre	24
8736	481	\N	Saint-Paul-la-Roche	24
8737	482	\N	Saint-Paul-Lizonne	24
8738	483	\N	Saint-Perdoux	24
8739	484	\N	Saint-Pierre-de-Chignac	24
8740	485	\N	Saint-Pierre-de-Côle	24
8741	486	\N	Saint-Pierre-de-Frugie	24
8742	487	\N	Saint-Pierre-d'Eyraud	24
8743	488	\N	Saint-Pompont	24
8744	489	\N	Saint-Priest-les-Fougères	24
8745	490	\N	Saint-Privat-des-Prés	24
8746	491	\N	Saint-Rabier	24
8747	492	\N	Sainte-Radegonde	24
8748	493	\N	Saint-Raphaël	24
8749	494	\N	Saint-Rémy	24
8750	495	\N	Saint-Romain-de-Monpazier	24
8751	496	\N	Saint-Romain-et-Saint-Clément	24
8752	497	\N	Sainte-Sabine-Born	24
8753	498	\N	Saint-Saud-Lacoussière	24
8754	499	\N	Saint-Sauveur	24
8755	500	\N	Saint-Sauveur-Lalande	24
8756	501	\N	Saint-Seurin-de-Prats	24
8757	502	\N	Saint-Séverin-d'Estissac	24
8758	503	\N	Saint-Sulpice-de-Mareuil	24
8759	504	\N	Saint-Sulpice-de-Roumagnac	24
8760	505	\N	Saint-Sulpice-d'Excideuil	24
8761	507	\N	Sainte-Trie	24
8762	508	\N	Saint-Victor	24
8763	509	\N	Saint-Vincent-de-Connezac	24
8764	510	\N	Saint-Vincent-de-Cosse	24
8765	511	\N	Saint-Vincent-Jalmoutiers	24
8766	512	\N	Saint-Vincent-le-Paluel	24
8767	513	\N	Saint-Vincent-sur-l'Isle	24
8768	514	\N	Saint-Vivien	24
8769	515	\N	Salagnac	24
8770	516	\N	Salignac-Eyvigues	24
8771	517	\N	Salles-de-Belvès	24
8772	518	\N	Salon	24
8773	519	\N	Sarlande	24
8774	520	\N	Sarlat-la-Canéda	24
8775	521	\N	Sarliac-sur-l'Isle	24
8776	522	\N	Sarrazac	24
8777	523	\N	Saussignac	24
8778	524	\N	Savignac-de-Miremont	24
8779	525	\N	Savignac-de-Nontron	24
8780	526	\N	Savignac-Lédrier	24
8781	527	\N	Savignac-les-Églises	24
8782	528	\N	Sceau-Saint-Angel	24
8783	529	\N	Segonzac	24
8784	530	\N	Sencenac-Puy-de-Fourches	24
8785	531	\N	Sergeac	24
8786	532	\N	Serres-et-Montguyard	24
8787	533	\N	Servanches	24
8788	534	\N	Sigoulès	24
8789	535	\N	Simeyrols	24
8790	536	\N	Singleyrac	24
8791	537	\N	Siorac-de-Ribérac	24
8792	538	\N	Siorac-en-Périgord	24
8793	540	\N	Sorges	24
8794	541	\N	Soudat	24
8795	542	\N	Soulaures	24
8796	543	\N	Sourzac	24
8797	544	\N	Tamniès	24
8798	545	\N	Teillots	24
8799	546	\N	Temple-Laguyon	24
8800	547	\N	Terrasson-Lavilledieu	24
8801	548	\N	Teyjat	24
8802	549	\N	Thénac	24
8803	550	\N	Thenon	24
8804	551	\N	Thiviers	24
8805	552	\N	Thonac	24
8806	553	\N	Tocane-Saint-Apre	24
8807	554	\N	Tour-Blanche	24
8808	555	\N	Tourtoirac	24
8809	557	\N	Trélissac	24
8810	558	\N	Trémolat	24
8811	559	\N	Tursac	24
8812	560	\N	Urval	24
8813	561	\N	Valeuil	24
8814	562	\N	Vallereuil	24
8815	563	\N	Valojoulx	24
8816	564	\N	Vanxains	24
8817	565	\N	Varaignes	24
8818	566	\N	Varennes	24
8819	567	\N	Vaunac	24
8820	568	\N	Vélines	24
8821	569	\N	Vendoire	24
8822	570	\N	Verdon	24
8823	571	\N	Vergt	24
8824	572	\N	Vergt-de-Biron	24
8825	573	\N	Verteillac	24
8826	574	\N	Veyrignac	24
8827	575	\N	Veyrines-de-Domme	24
8828	576	\N	Veyrines-de-Vergt	24
8829	577	\N	Vézac	24
8830	579	\N	Vieux-Mareuil	24
8831	580	\N	Villac	24
8832	581	\N	Villamblard	24
8833	582	\N	Villars	24
8834	584	\N	Villefranche-de-Lonchat	24
8835	585	\N	Villefranche-du-Périgord	24
8836	586	\N	Villetoureix	24
8837	587	\N	Vitrac	24
8838	001	\N	Abbans-Dessous	25
8839	002	\N	Abbans-Dessus	25
8840	003	\N	Abbenans	25
8841	004	\N	Abbévillers	25
8842	005	\N	Accolans	25
8843	006	\N	Adam-lès-Passavant	25
8844	007	\N	Adam-lès-Vercel	25
8845	008	\N	Aibre	25
8846	009	\N	Aïssey	25
8847	011	\N	Allenjoie	25
8848	012	\N	Alliés	25
8849	013	\N	Allondans	25
8850	014	\N	Amagney	25
8851	015	\N	Amancey	25
8852	016	\N	Amathay-Vésigneux	25
8853	017	\N	Amondans	25
8854	018	\N	Anteuil	25
8855	019	\N	Appenans	25
8856	020	\N	Arbouans	25
8857	021	\N	Arc-et-Senans	25
8858	022	\N	Arcey	25
8859	024	\N	Arçon	25
8860	025	\N	Arc-sous-Cicon	25
8861	026	\N	Arc-sous-Montenot	25
8862	027	\N	Arguel	25
8863	028	\N	Athose	25
8864	029	\N	Aubonne	25
8865	030	\N	Audeux	25
8866	031	\N	Audincourt	25
8867	032	\N	Autechaux	25
8868	033	\N	Autechaux-Roide	25
8869	034	\N	Auxon-Dessous	25
8870	035	\N	Auxon-Dessus	25
8871	036	\N	Avanne-Aveney	25
8872	038	\N	Avilley	25
8873	039	\N	Avoudrey	25
8874	040	\N	Badevel	25
8875	041	\N	Bannans	25
8876	042	\N	Barboux	25
8877	043	\N	Bart	25
8878	044	\N	Bartherans	25
8879	045	\N	Battenans-les-Mines	25
8880	046	\N	Battenans-Varin	25
8881	047	\N	Baume-les-Dames	25
8882	048	\N	Bavans	25
8883	049	\N	Belfays	25
8884	050	\N	Bélieu	25
8885	051	\N	Belleherbe	25
8886	052	\N	Belmont	25
8887	053	\N	Belvoir	25
8888	054	\N	Berche	25
8889	055	\N	Berthelange	25
8890	056	\N	Besançon	25
8891	057	\N	Bethoncourt	25
8892	058	\N	Beure	25
8893	059	\N	Beutal	25
8894	060	\N	Bians-les-Usiers	25
8895	061	\N	Bief	25
8896	062	\N	Bizot	25
8897	063	\N	Blamont	25
8898	065	\N	Blarians	25
8899	066	\N	Blussangeaux	25
8900	067	\N	Blussans	25
8901	070	\N	Bolandoz	25
8902	071	\N	Bondeval	25
8903	072	\N	Bonnal	25
8904	073	\N	Bonnay	25
8905	074	\N	Bonnétage	25
8906	075	\N	Bonnevaux	25
8907	076	\N	Bonnevaux-le-Prieuré	25
8908	077	\N	Bosse	25
8909	078	\N	Bouclans	25
8910	079	\N	Boujailles	25
8911	082	\N	Bourguignon	25
8912	083	\N	Bournois	25
8913	084	\N	Boussières	25
8914	085	\N	Bouverans	25
8915	086	\N	Braillans	25
8916	087	\N	Branne	25
8917	088	\N	Breconchaux	25
8918	089	\N	Bremondans	25
8919	090	\N	Brères	25
8920	091	\N	Bréseux	25
8921	092	\N	Bretenière	25
8922	093	\N	Bretigney	25
8923	094	\N	Bretigney-Notre-Dame	25
8924	095	\N	Bretonvillers	25
8925	096	\N	Brey-et-Maison-du-Bois	25
8926	097	\N	Brognard	25
8927	098	\N	Buffard	25
8928	099	\N	Bugny	25
8929	100	\N	Bulle	25
8930	101	\N	Burgille	25
8931	102	\N	Burnevillers	25
8932	103	\N	Busy	25
8933	104	\N	By	25
8934	105	\N	Byans-sur-Doubs	25
8935	106	\N	Cademène	25
8936	107	\N	Cendrey	25
8937	108	\N	Cernay-l'Église	25
8938	109	\N	Cessey	25
8939	110	\N	Chaffois	25
8940	111	\N	Chalèze	25
8941	112	\N	Chalezeule	25
8942	113	\N	Chamesey	25
8943	114	\N	Chamesol	25
8944	115	\N	Champagney	25
8945	116	\N	Champlive	25
8946	117	\N	Champoux	25
8947	119	\N	Champvans-les-Moulins	25
8948	120	\N	Chantrans	25
8949	121	\N	Chapelle-des-Bois	25
8950	122	\N	Chapelle-d'Huin	25
8951	123	\N	Charbonnières-les-Sapins	25
8952	124	\N	Charmauvillers	25
8953	125	\N	Charmoille	25
8954	126	\N	Charnay	25
8955	127	\N	Charquemont	25
8956	128	\N	Chasnans	25
8957	129	\N	Chassagne-Saint-Denis	25
8958	130	\N	Châteauvieux-les-Fossés	25
8959	131	\N	Châtelblanc	25
8960	132	\N	Châtillon-Guyotte	25
8961	133	\N	Châtillon-le-Duc	25
8962	134	\N	Châtillon-sur-Lison	25
8963	136	\N	Chaucenne	25
8964	137	\N	Chaudefontaine	25
8965	138	\N	Terres-de-Chaux	25
8966	139	\N	Chaux	25
8967	140	\N	Chaux-lès-Clerval	25
8968	141	\N	Chaux-lès-Passavant	25
8969	142	\N	Chaux-Neuve	25
8970	143	\N	Chay	25
8971	145	\N	Chazot	25
8972	147	\N	Chemaudin	25
8973	148	\N	Chenalotte	25
8974	149	\N	Chenecey-Buillon	25
8975	150	\N	Chevigney-sur-l'Ognon	25
8976	151	\N	Chevigney-lès-Vercel	25
8977	152	\N	Chevillotte	25
8978	153	\N	Chevroz	25
8979	154	\N	Chouzelot	25
8980	155	\N	Cléron	25
8981	156	\N	Clerval	25
8982	157	\N	Cluse-et-Mijoux	25
8983	159	\N	Colombier-Fontaine	25
8984	160	\N	Combes	25
8985	161	\N	Consolation-Maisonnettes	25
8986	162	\N	Corcelles-Ferrières	25
8987	163	\N	Corcelle-Mieslot	25
8988	164	\N	Corcondray	25
8989	166	\N	Côtebrune	25
8990	170	\N	Courcelles-lès-Montbéliard	25
8991	171	\N	Courcelles	25
8992	172	\N	Courchapon	25
8993	173	\N	Cour-Saint-Maurice	25
8994	174	\N	Courtefontaine	25
8995	175	\N	Courtetain-et-Salans	25
8996	176	\N	Courvières	25
8997	177	\N	Crosey-le-Grand	25
8998	178	\N	Crosey-le-Petit	25
8999	179	\N	Crouzet	25
9000	180	\N	Crouzet-Migette	25
9001	181	\N	Cubrial	25
9002	182	\N	Cubry	25
9003	183	\N	Cusance	25
9004	184	\N	Cuse-et-Adrisans	25
9005	185	\N	Cussey-sur-Lison	25
9006	186	\N	Cussey-sur-l'Ognon	25
9007	187	\N	Dambelin	25
9008	188	\N	Dambenois	25
9009	189	\N	Dammartin-les-Templiers	25
9010	190	\N	Dampierre-les-Bois	25
9011	191	\N	Dampierre-sur-le-Doubs	25
9012	192	\N	Dampjoux	25
9013	193	\N	Damprichard	25
9014	194	\N	Dannemarie	25
9015	195	\N	Dannemarie-sur-Crète	25
9016	196	\N	Dasle	25
9017	197	\N	Deluz	25
9018	198	\N	Désandans	25
9019	199	\N	Déservillers	25
9020	200	\N	Devecey	25
9021	201	\N	Dommartin	25
9022	202	\N	Dompierre-les-Tilleuls	25
9023	203	\N	Domprel	25
9024	204	\N	Doubs	25
9025	207	\N	Dung	25
9026	208	\N	Durnes	25
9027	209	\N	Échay	25
9028	210	\N	Échenans	25
9029	211	\N	Échevannes	25
9030	212	\N	École-Valentin	25
9031	213	\N	Écorces	25
9032	214	\N	Écot	25
9033	215	\N	Écouvotte	25
9034	216	\N	Écurcey	25
9035	217	\N	Émagny	25
9036	218	\N	Épenouse	25
9037	219	\N	Épenoy	25
9038	220	\N	Épeugney	25
9039	221	\N	Esnans	25
9040	222	\N	Étalans	25
9041	223	\N	Éternoz	25
9042	224	\N	Étouvans	25
9043	225	\N	Étrabonne	25
9044	226	\N	Étrappe	25
9045	227	\N	Étray	25
9046	228	\N	Étupes	25
9047	229	\N	Évillers	25
9048	230	\N	Exincourt	25
9049	231	\N	Eysson	25
9050	232	\N	Faimbe	25
9051	233	\N	Fallerans	25
9052	234	\N	Ferrières-le-Lac	25
9053	235	\N	Ferrières-les-Bois	25
9054	236	\N	Fertans	25
9055	237	\N	Fesches-le-Châtel	25
9056	238	\N	Fessevillers	25
9057	239	\N	Feule	25
9058	240	\N	Fins	25
9059	241	\N	Flagey	25
9060	242	\N	Flagey-Rigney	25
9061	243	\N	Flangebouche	25
9062	244	\N	Fleurey	25
9063	245	\N	Fontain	25
9064	246	\N	Fontaine-lès-Clerval	25
9065	247	\N	Fontenelle-Montby	25
9066	248	\N	Fontenelles	25
9067	249	\N	Fontenotte	25
9068	250	\N	Foucherans	25
9069	251	\N	Fourbanne	25
9070	252	\N	Fourcatier-et-Maison-Neuve	25
9071	253	\N	Fourg	25
9072	254	\N	Fourgs	25
9073	255	\N	Fournet-Blancheroche	25
9074	256	\N	Frambouhans	25
9075	257	\N	Franey	25
9076	258	\N	Franois	25
9077	259	\N	Frasne	25
9078	261	\N	Froidevaux	25
9079	262	\N	Fuans	25
9080	263	\N	Gellin	25
9081	264	\N	Gémonval	25
9082	265	\N	Geneuille	25
9083	266	\N	Geney	25
9084	267	\N	Gennes	25
9085	268	\N	Germéfontaine	25
9086	269	\N	Germondans	25
9087	270	\N	Gevresin	25
9088	271	\N	Gilley	25
9089	273	\N	Glamondans	25
9090	274	\N	Glay	25
9091	275	\N	Glère	25
9092	276	\N	Gondenans-Montby	25
9093	277	\N	Gondenans-les-Moulins	25
9094	278	\N	Gonsans	25
9095	279	\N	Gouhelans	25
9096	280	\N	Goumois	25
9097	281	\N	Goux-lès-Dambelin	25
9098	282	\N	Goux-les-Usiers	25
9099	283	\N	Goux-sous-Landet	25
9100	284	\N	Grand-Charmont	25
9101	285	\N	Grand'Combe-Châteleu	25
9102	286	\N	Grand'Combe-des-Bois	25
9103	287	\N	Grandfontaine	25
9104	288	\N	Fournets-Luisans	25
9105	289	\N	Grandfontaine-sur-Creuse	25
9106	290	\N	Grange	25
9107	293	\N	Granges-Narboz	25
9108	295	\N	Grangettes	25
9109	296	\N	Gras	25
9110	297	\N	Gratteris	25
9111	298	\N	Grosbois	25
9112	299	\N	Guillon-les-Bains	25
9113	300	\N	Guyans-Durnes	25
9114	301	\N	Guyans-Vennes	25
9115	302	\N	Hautepierre-le-Châtelet	25
9116	303	\N	Hauterive-la-Fresse	25
9117	304	\N	Hérimoncourt	25
9118	305	\N	Hôpital-du-Grosbois	25
9119	306	\N	Hôpital-Saint-Lieffroy	25
9120	307	\N	Hôpitaux-Neufs	25
9121	308	\N	Hôpitaux-Vieux	25
9122	309	\N	Houtaud	25
9123	310	\N	Huanne-Montmartin	25
9124	311	\N	Hyémondans	25
9125	312	\N	Hyèvre-Magny	25
9126	313	\N	Hyèvre-Paroisse	25
9127	314	\N	Indevillers	25
9128	315	\N	Isle-sur-le-Doubs	25
9129	316	\N	Issans	25
9130	317	\N	Jallerange	25
9131	318	\N	Jougne	25
9132	319	\N	Labergement-du-Navois	25
9133	320	\N	Labergement-Sainte-Marie	25
9134	321	\N	Villers-le-Lac	25
9135	322	\N	Laire	25
9136	323	\N	Laissey	25
9137	324	\N	Lanans	25
9138	325	\N	Landresse	25
9139	326	\N	Lantenne-Vertière	25
9140	327	\N	Lanthenans	25
9141	328	\N	Larnod	25
9142	329	\N	Laval-le-Prieuré	25
9143	330	\N	Lavans-Quingey	25
9144	331	\N	Lavans-Vuillafans	25
9145	332	\N	Lavernay	25
9146	333	\N	Laviron	25
9147	334	\N	Levier	25
9148	335	\N	Liebvillers	25
9149	336	\N	Liesle	25
9150	338	\N	Lizine	25
9151	339	\N	Lods	25
9152	340	\N	Lombard	25
9153	341	\N	Lomont-sur-Crête	25
9154	342	\N	Longechaux	25
9155	343	\N	Longemaison	25
9156	344	\N	Longevelle-lès-Russey	25
9157	345	\N	Longevelle-sur-Doubs	25
9158	346	\N	Longeville	25
9159	347	\N	Longeville	25
9160	348	\N	Longevilles-Mont-d'Or	25
9161	349	\N	Loray	25
9162	350	\N	Lougres	25
9163	351	\N	Luhier	25
9164	354	\N	Luxiol	25
9165	355	\N	Magny-Châtelard	25
9166	356	\N	Maîche	25
9167	357	\N	Maisons-du-Bois-Lièvremont	25
9168	359	\N	Malans	25
9169	360	\N	Malbrans	25
9170	361	\N	Malbuisson	25
9171	362	\N	Malpas	25
9172	364	\N	Mamirolle	25
9173	365	\N	Mancenans	25
9174	366	\N	Mancenans-Lizerne	25
9175	367	\N	Mandeure	25
9176	368	\N	Marchaux	25
9177	369	\N	Marvelise	25
9178	370	\N	Mathay	25
9179	371	\N	Mazerolles-le-Salin	25
9180	372	\N	Médière	25
9181	373	\N	Mémont	25
9182	374	\N	Mercey-le-Grand	25
9183	375	\N	Mérey-sous-Montrond	25
9184	376	\N	Mérey-Vieilley	25
9185	377	\N	Mésandans	25
9186	378	\N	Meslières	25
9187	379	\N	Mesmay	25
9188	380	\N	Métabief	25
9189	381	\N	Miserey-Salines	25
9190	382	\N	Moncey	25
9191	383	\N	Moncley	25
9192	384	\N	Mondon	25
9193	385	\N	Montagney-Servigney	25
9194	386	\N	Montancy	25
9195	387	\N	Montandon	25
9196	388	\N	Montbéliard	25
9197	389	\N	Montbéliardot	25
9198	390	\N	Montbenoît	25
9199	391	\N	Mont-de-Laval	25
9200	392	\N	Mont-de-Vougney	25
9201	393	\N	Montécheroux	25
9202	394	\N	Montenois	25
9203	395	\N	Montfaucon	25
9204	397	\N	Montferrand-le-Château	25
9205	398	\N	Montflovin	25
9206	399	\N	Montfort	25
9207	400	\N	Montgesoye	25
9208	401	\N	Montivernage	25
9209	402	\N	Montjoie-le-Château	25
9210	403	\N	Montlebon	25
9211	404	\N	Montmahoux	25
9212	405	\N	Montperreux	25
9213	406	\N	Montrond-le-Château	25
9214	408	\N	Montussaint	25
9215	410	\N	Morre	25
9216	411	\N	Morteau	25
9217	413	\N	Mouthe	25
9218	414	\N	Moutherot	25
9219	415	\N	Mouthier-Haute-Pierre	25
9220	416	\N	Myon	25
9221	417	\N	Naisey-les-Granges	25
9222	418	\N	Nancray	25
9223	419	\N	Nans	25
9224	420	\N	Nans-sous-Sainte-Anne	25
9225	421	\N	Narbief	25
9226	422	\N	Neuchâtel-Urtière	25
9227	424	\N	Nods	25
9228	425	\N	Noël-Cerneux	25
9229	426	\N	Noirefontaine	25
9230	427	\N	Noironte	25
9231	428	\N	Nommay	25
9232	429	\N	Novillars	25
9233	430	\N	Ollans	25
9234	431	\N	Onans	25
9235	432	\N	Orchamps-Vennes	25
9236	433	\N	Orgeans-Blanchefontaine	25
9237	434	\N	Ornans	25
9238	435	\N	Orsans	25
9239	436	\N	Orve	25
9240	437	\N	Osse	25
9241	438	\N	Osselle	25
9242	439	\N	Ougney-Douvot	25
9243	440	\N	Ouhans	25
9244	441	\N	Ouvans	25
9245	442	\N	Oye-et-Pallet	25
9246	443	\N	Palantine	25
9247	444	\N	Palise	25
9248	445	\N	Paroy	25
9249	446	\N	Passavant	25
9250	447	\N	Passonfontaine	25
9251	448	\N	Pelousey	25
9252	449	\N	Péseux	25
9253	450	\N	Pessans	25
9254	451	\N	Petite-Chaux	25
9255	452	\N	Pierrefontaine-lès-Blamont	25
9256	453	\N	Pierrefontaine-les-Varans	25
9257	454	\N	Pirey	25
9258	455	\N	Placey	25
9259	456	\N	Plaimbois-du-Miroir	25
9260	457	\N	Plaimbois-Vennes	25
9261	458	\N	Plains-et-Grands-Essarts	25
9262	459	\N	Planée	25
9263	460	\N	Pointvillers	25
9264	461	\N	Pompierre-sur-Doubs	25
9265	462	\N	Pontarlier	25
9266	463	\N	Pont-de-Roide	25
9267	464	\N	Pontets	25
9268	465	\N	Pont-les-Moulins	25
9269	466	\N	Pouilley-Français	25
9270	467	\N	Pouilley-les-Vignes	25
9271	468	\N	Pouligney-Lusans	25
9272	469	\N	Présentevillers	25
9273	470	\N	Prétière	25
9274	471	\N	Provenchère	25
9275	472	\N	Puessans	25
9276	473	\N	Pugey	25
9277	474	\N	Puy	25
9278	475	\N	Quingey	25
9279	476	\N	Rahon	25
9280	477	\N	Rancenay	25
9281	478	\N	Randevillers	25
9282	479	\N	Rang	25
9283	480	\N	Rantechaux	25
9284	481	\N	Raynans	25
9285	482	\N	Recologne	25
9286	483	\N	Reculfoz	25
9287	485	\N	Rémondans-Vaivre	25
9288	486	\N	Remoray-Boujeons	25
9289	487	\N	Renédale	25
9290	488	\N	Rennes-sur-Loue	25
9291	489	\N	Reugney	25
9292	490	\N	Rigney	25
9293	491	\N	Rignosot	25
9294	492	\N	Rillans	25
9295	493	\N	Rivière-Drugeon	25
9296	494	\N	Rochejean	25
9297	495	\N	Roche-lez-Beaupré	25
9298	496	\N	Roche-lès-Clerval	25
9299	497	\N	Roches-lès-Blamont	25
9300	498	\N	Rognon	25
9301	499	\N	Romain	25
9302	500	\N	Ronchaux	25
9303	501	\N	Rondefontaine	25
9304	502	\N	Roset-Fluans	25
9305	503	\N	Rosières-sur-Barbèche	25
9306	504	\N	Rosureux	25
9307	505	\N	Rougemont	25
9308	506	\N	Rougemontot	25
9309	507	\N	Rouhe	25
9310	508	\N	Roulans	25
9311	509	\N	Routelle	25
9312	510	\N	Ruffey-le-Château	25
9313	511	\N	Rurey	25
9314	512	\N	Russey	25
9315	513	\N	Sainte-Anne	25
9316	514	\N	Saint-Antoine	25
9317	515	\N	Sainte-Colombe	25
9318	516	\N	Saint-Georges-Armont	25
9319	517	\N	Saint-Gorgon-Main	25
9320	518	\N	Saint-Hilaire	25
9321	519	\N	Saint-Hippolyte	25
9322	520	\N	Saint-Juan	25
9323	521	\N	Saint-Julien-lès-Montbéliard	25
9324	522	\N	Saint-Julien-lès-Russey	25
9325	523	\N	Sainte-Marie	25
9326	524	\N	Saint-Maurice-Colombier	25
9327	525	\N	Saint-Point-Lac	25
9328	526	\N	Sainte-Suzanne	25
9329	527	\N	Saint-Vit	25
9330	528	\N	Samson	25
9331	529	\N	Sancey-le-Grand	25
9332	530	\N	Sancey-le-Long	25
9333	531	\N	Santoche	25
9334	532	\N	Saône	25
9335	533	\N	Saraz	25
9336	534	\N	Sarrageois	25
9337	535	\N	Saules	25
9338	536	\N	Sauvagney	25
9339	537	\N	Scey-Maisières	25
9340	538	\N	Séchin	25
9341	539	\N	Seloncourt	25
9342	540	\N	Semondans	25
9343	541	\N	Septfontaines	25
9344	542	\N	Serre-les-Sapins	25
9345	544	\N	Servin	25
9346	545	\N	Silley-Amancey	25
9347	546	\N	Silley-Bléfond	25
9348	547	\N	Sochaux	25
9349	548	\N	Solemont	25
9350	549	\N	Sombacour	25
9351	550	\N	Sommette	25
9352	551	\N	Soulce-Cernay	25
9353	552	\N	Sourans	25
9354	553	\N	Soye	25
9355	554	\N	Surmont	25
9356	555	\N	Taillecourt	25
9357	556	\N	Tallans	25
9358	557	\N	Tallenay	25
9359	558	\N	Tarcenay	25
9360	559	\N	Thiébouhans	25
9361	560	\N	Thise	25
9362	561	\N	Thoraise	25
9363	562	\N	Thulay	25
9364	563	\N	Thurey-le-Mont	25
9365	564	\N	Torpes	25
9366	565	\N	Touillon-et-Loutelet	25
9367	566	\N	Tour-de-Sçay	25
9368	567	\N	Tournans	25
9369	569	\N	Trépot	25
9370	570	\N	Tressandans	25
9371	571	\N	Trévillers	25
9372	572	\N	Trouvans	25
9373	573	\N	Urtière	25
9374	574	\N	Uzelle	25
9375	575	\N	Vaire-Arcier	25
9376	576	\N	Vaire-le-Petit	25
9377	578	\N	Valdahon	25
9378	579	\N	Val-de-Roulans	25
9379	580	\N	Valentigney	25
9380	582	\N	Valleroy	25
9381	583	\N	Valonne	25
9382	584	\N	Valoreille	25
9383	585	\N	Vanclans	25
9384	586	\N	Vandoncourt	25
9385	587	\N	Vauchamps	25
9386	588	\N	Vaucluse	25
9387	589	\N	Vauclusotte	25
9388	590	\N	Vaudrivillers	25
9389	591	\N	Vaufrey	25
9390	592	\N	Vaux-et-Chantegrue	25
9391	593	\N	Vaux-les-Prés	25
9392	594	\N	Velesmes-Essarts	25
9393	595	\N	Vellerot-lès-Belvoir	25
9394	596	\N	Vellerot-lès-Vercel	25
9395	597	\N	Vellevans	25
9396	598	\N	Venise	25
9397	599	\N	Vennans	25
9398	600	\N	Vennes	25
9399	601	\N	Vercel-Villedieu-le-Camp	25
9400	602	\N	Vergranne	25
9401	604	\N	Verne	25
9402	605	\N	Vernierfontaine	25
9403	607	\N	Vernois-lès-Belvoir	25
9404	608	\N	Vernoy	25
9405	609	\N	Verrières-de-Joux	25
9406	610	\N	Verrières-du-Grosbois	25
9407	611	\N	Vèze	25
9408	612	\N	Vieilley	25
9409	613	\N	Viéthorey	25
9410	614	\N	Vieux-Charmont	25
9411	615	\N	Villars-lès-Blamont	25
9412	616	\N	Villars-Saint-Georges	25
9413	617	\N	Villars-sous-Dampjoux	25
9414	618	\N	Villars-sous-Écot	25
9415	619	\N	Villedieu	25
9416	620	\N	Ville-du-Pont	25
9417	621	\N	Villeneuve-d'Amont	25
9418	622	\N	Villers-Buzon	25
9419	623	\N	Villers-Chief	25
9420	624	\N	Villers-Grélot	25
9421	625	\N	Villers-la-Combe	25
9422	626	\N	Villers-Saint-Martin	25
9423	627	\N	Villers-sous-Chalamont	25
9424	628	\N	Villers-sous-Montrond	25
9425	629	\N	Voillans	25
9426	630	\N	Voires	25
9427	631	\N	Vorges-les-Pins	25
9428	632	\N	Voujeaucourt	25
9429	633	\N	Vuillafans	25
9430	634	\N	Vuillecin	25
9431	635	\N	Vyt-lès-Belvoir	25
9432	001	\N	Aix-en-Diois	26
9433	002	\N	Albon	26
9434	003	\N	Aleyrac	26
9435	004	\N	Alixan	26
9436	005	\N	Allan	26
9437	006	\N	Allex	26
9438	007	\N	Ambonil	26
9439	008	\N	Ancône	26
9440	009	\N	Andancette	26
9441	010	\N	Anneyron	26
9442	011	\N	Aouste-sur-Sye	26
9443	012	\N	Arnayon	26
9444	013	\N	Arpavon	26
9445	014	\N	Arthémonay	26
9446	015	\N	Aubenasson	26
9447	016	\N	Aubres	26
9448	017	\N	Aucelon	26
9449	018	\N	Aulan	26
9450	019	\N	Aurel	26
9451	020	\N	Répara-Auriples	26
9452	021	\N	Autichamp	26
9453	022	\N	Ballons	26
9454	023	\N	Barbières	26
9455	024	\N	Barcelonne	26
9456	025	\N	Barnave	26
9457	026	\N	Barret-de-Lioure	26
9458	027	\N	Barsac	26
9459	028	\N	Bathernay	26
9460	030	\N	Bâtie-des-Fonds	26
9461	031	\N	Bâtie-Rolland	26
9462	032	\N	Baume-Cornillane	26
9463	033	\N	Baume-de-Transit	26
9464	034	\N	Baume-d'Hostun	26
9465	035	\N	Beaufort-sur-Gervanne	26
9466	036	\N	Beaumont-en-Diois	26
9467	037	\N	Beaumont-lès-Valence	26
9468	038	\N	Beaumont-Monteux	26
9469	039	\N	Beauregard-Baret	26
9470	040	\N	Beaurières	26
9471	041	\N	Beausemblant	26
9472	042	\N	Beauvallon	26
9473	043	\N	Beauvoisin	26
9474	045	\N	Bégude-de-Mazenc	26
9475	046	\N	Bellecombe-Tarendol	26
9476	047	\N	Bellegarde-en-Diois	26
9477	048	\N	Bénivay-Ollon	26
9478	049	\N	Bésayes	26
9479	050	\N	Bésignan	26
9480	051	\N	Bézaudun-sur-Bîne	26
9481	052	\N	Bonlieu-sur-Roubion	26
9482	054	\N	Bouchet	26
9483	055	\N	Boulc	26
9484	056	\N	Bourdeaux	26
9485	057	\N	Bourg-de-Péage	26
9486	058	\N	Bourg-lès-Valence	26
9487	059	\N	Bouvante	26
9488	060	\N	Bouvières	26
9489	061	\N	Bren	26
9490	062	\N	Brette	26
9491	063	\N	Buis-les-Baronnies	26
9492	064	\N	Chabeuil	26
9493	065	\N	Chabrillan	26
9494	066	\N	Chaffal	26
9495	067	\N	Chalancon	26
9496	068	\N	Chalon	26
9497	069	\N	Chamaloc	26
9498	070	\N	Chamaret	26
9499	071	\N	Chanos-Curson	26
9500	072	\N	Chantemerle-les-Blés	26
9501	073	\N	Chantemerle-lès-Grignan	26
9502	074	\N	Chapelle-en-Vercors	26
9503	075	\N	Charce	26
9504	076	\N	Charens	26
9505	077	\N	Charmes-sur-l'Herbasse	26
9506	078	\N	Charols	26
9507	079	\N	Charpey	26
9508	080	\N	Chastel-Arnaud	26
9509	081	\N	Châteaudouble	26
9510	082	\N	Châteauneuf-de-Bordette	26
9511	083	\N	Châteauneuf-de-Galaure	26
9512	084	\N	Châteauneuf-sur-Isère	26
9513	085	\N	Châteauneuf-du-Rhône	26
9514	086	\N	Châtillon-en-Diois	26
9515	087	\N	Châtillon-Saint-Jean	26
9516	088	\N	Chatuzange-le-Goubet	26
9517	089	\N	Chaudebonne	26
9518	090	\N	Chaudière	26
9519	091	\N	Chauvac-Laux-Montaux	26
9520	092	\N	Chavannes	26
9521	093	\N	Clansayes	26
9522	094	\N	Claveyson	26
9523	095	\N	Cléon-d'Andran	26
9524	096	\N	Clérieux	26
9525	097	\N	Cliousclat	26
9526	098	\N	Cobonne	26
9527	099	\N	Colonzelle	26
9528	100	\N	Combovin	26
9529	101	\N	Comps	26
9530	102	\N	Condillac	26
9531	103	\N	Condorcet	26
9532	104	\N	Cornillac	26
9533	105	\N	Cornillon-sur-l'Oule	26
9534	106	\N	Coucourde	26
9535	107	\N	Crépol	26
9536	108	\N	Crest	26
9537	110	\N	Crozes-Hermitage	26
9538	111	\N	Crupies	26
9539	112	\N	Curnier	26
9540	113	\N	Die	26
9541	114	\N	Dieulefit	26
9542	115	\N	Divajeu	26
9543	116	\N	Donzère	26
9544	117	\N	Échevis	26
9545	118	\N	Épinouze	26
9546	119	\N	Érôme	26
9547	121	\N	Espeluche	26
9548	122	\N	Espenel	26
9549	123	\N	Establet	26
9550	124	\N	Étoile-sur-Rhône	26
9551	125	\N	Eurre	26
9552	126	\N	Eygalayes	26
9553	127	\N	Eygaliers	26
9554	128	\N	Eygluy-Escoulin	26
9555	129	\N	Eymeux	26
9556	130	\N	Eyroles	26
9557	131	\N	Eyzahut	26
9558	133	\N	Fay-le-Clos	26
9559	134	\N	Félines-sur-Rimandoule	26
9560	135	\N	Ferrassières	26
9561	136	\N	Val-Maravel	26
9562	137	\N	Francillon-sur-Roubion	26
9563	138	\N	Garde-Adhémar	26
9564	139	\N	Génissieux	26
9565	140	\N	Geyssans	26
9566	141	\N	Gigors-et-Lozeron	26
9567	142	\N	Glandage	26
9568	143	\N	Grand-Serre	26
9569	144	\N	Grane	26
9570	145	\N	Granges-Gontardes	26
9571	146	\N	Grignan	26
9572	147	\N	Gumiane	26
9573	148	\N	Hauterives	26
9574	149	\N	Hostun	26
9575	150	\N	Izon-la-Bruisse	26
9576	152	\N	Jonchères	26
9577	153	\N	Laborel	26
9578	154	\N	Lachau	26
9579	155	\N	Lapeyrouse-Mornay	26
9580	156	\N	Larnage	26
9581	157	\N	Laupie	26
9582	159	\N	Laval-d'Aix	26
9583	160	\N	Laveyron	26
9584	161	\N	Lemps	26
9585	162	\N	Lens-Lestang	26
9586	163	\N	Léoncel	26
9587	164	\N	Lesches-en-Diois	26
9588	165	\N	Livron-sur-Drôme	26
9589	166	\N	Loriol-sur-Drôme	26
9590	167	\N	Luc-en-Diois	26
9591	168	\N	Lus-la-Croix-Haute	26
9592	169	\N	Malataverne	26
9593	170	\N	Malissard	26
9594	171	\N	Manas	26
9595	172	\N	Manthes	26
9596	173	\N	Marches	26
9597	174	\N	Margès	26
9598	175	\N	Marignac-en-Diois	26
9599	176	\N	Marsanne	26
9600	177	\N	Marsaz	26
9601	178	\N	Menglon	26
9602	179	\N	Mercurol	26
9603	180	\N	Mérindol-les-Oliviers	26
9604	181	\N	Mévouillon	26
9605	182	\N	Mirabel-aux-Baronnies	26
9606	183	\N	Mirabel-et-Blacons	26
9607	184	\N	Miribel	26
9608	185	\N	Mirmande	26
9609	186	\N	Miscon	26
9610	187	\N	Molières-Glandaz	26
9611	188	\N	Mollans-sur-Ouvèze	26
9612	189	\N	Montauban-sur-l'Ouvèze	26
9613	190	\N	Montaulieu	26
9614	191	\N	Montboucher-sur-Jabron	26
9615	192	\N	Montbrison-sur-Lez	26
9616	193	\N	Montbrun-les-Bains	26
9617	194	\N	Montchenu	26
9618	195	\N	Montclar-sur-Gervanne	26
9619	196	\N	Montéléger	26
9620	197	\N	Montélier	26
9621	198	\N	Montélimar	26
9622	199	\N	Montferrand-la-Fare	26
9623	200	\N	Montfroc	26
9624	201	\N	Montguers	26
9625	202	\N	Montjoux	26
9626	203	\N	Montjoyer	26
9627	204	\N	Montlaur-en-Diois	26
9628	205	\N	Montmaur-en-Diois	26
9629	206	\N	Montmeyran	26
9630	207	\N	Montmiral	26
9631	208	\N	Montoison	26
9632	209	\N	Montréal-les-Sources	26
9633	210	\N	Montrigaud	26
9634	211	\N	Montségur-sur-Lauzon	26
9635	212	\N	Montvendre	26
9636	213	\N	Moras-en-Valloire	26
9637	214	\N	Mornans	26
9638	215	\N	Motte-Chalancon	26
9639	216	\N	Motte-de-Galaure	26
9640	217	\N	Motte-Fanjas	26
9641	218	\N	Mours-Saint-Eusèbe	26
9642	219	\N	Mureils	26
9643	220	\N	Nyons	26
9644	221	\N	Omblèze	26
9645	222	\N	Orcinas	26
9646	223	\N	Oriol-en-Royans	26
9647	224	\N	Ourches	26
9648	225	\N	Parnans	26
9649	226	\N	Pègue	26
9650	227	\N	Pelonne	26
9651	228	\N	Pennes-le-Sec	26
9652	229	\N	Penne-sur-l'Ouvèze	26
9653	231	\N	Peyrins	26
9654	232	\N	Peyrus	26
9655	233	\N	Piégon	26
9656	234	\N	Piégros-la-Clastre	26
9657	235	\N	Pierrelatte	26
9658	236	\N	Pierrelongue	26
9659	238	\N	Pilles	26
9660	239	\N	Plaisians	26
9661	240	\N	Plan-de-Baix	26
9662	241	\N	Poët-Célard	26
9663	242	\N	Poët-en-Percip	26
9664	243	\N	Poët-Laval	26
9665	244	\N	Poët-Sigillat	26
9666	245	\N	Pommerol	26
9667	246	\N	Ponet-et-Saint-Auban	26
9668	247	\N	Ponsas	26
9669	248	\N	Pontaix	26
9670	249	\N	Pont-de-Barret	26
9671	250	\N	Pont-de-l'Isère	26
9672	251	\N	Portes-en-Valdaine	26
9673	252	\N	Portes-lès-Valence	26
9674	253	\N	Poyols	26
9675	254	\N	Pradelle	26
9676	255	\N	Prés	26
9677	256	\N	Propiac	26
9678	257	\N	Puygiron	26
9679	258	\N	Puy-Saint-Martin	26
9680	259	\N	Ratières	26
9681	261	\N	Réauville	26
9682	262	\N	Recoubeau-Jansac	26
9683	263	\N	Reilhanette	26
9684	264	\N	Rémuzat	26
9685	266	\N	Rimon-et-Savel	26
9686	267	\N	Rioms	26
9687	268	\N	Rochebaudin	26
9688	269	\N	Rochebrune	26
9689	270	\N	Rochechinard	26
9690	271	\N	Roche-de-Glun	26
9691	272	\N	Rochefort-en-Valdaine	26
9692	273	\N	Rochefort-Samson	26
9693	274	\N	Rochefourchat	26
9694	275	\N	Rochegude	26
9695	276	\N	Roche-Saint-Secret-Béconne	26
9696	277	\N	Roche-sur-Grane	26
9697	278	\N	Roche-sur-le-Buis	26
9698	279	\N	Rochette-du-Buis	26
9699	281	\N	Romans-sur-Isère	26
9700	282	\N	Romeyer	26
9701	283	\N	Rottier	26
9702	284	\N	Roussas	26
9703	285	\N	Rousset-les-Vignes	26
9704	286	\N	Roussieux	26
9705	287	\N	Roynac	26
9706	288	\N	Sahune	26
9707	289	\N	Saillans	26
9708	290	\N	Saint-Agnan-en-Vercors	26
9709	291	\N	Saint-Andéol	26
9710	292	\N	Saint-Auban-sur-l'Ouvèze	26
9711	293	\N	Saint-Avit	26
9712	294	\N	Saint-Bardoux	26
9713	295	\N	Saint-Barthélemy-de-Vals	26
9714	296	\N	Saint-Benoit-en-Diois	26
9715	297	\N	Saint-Bonnet-de-Valclérieux	26
9716	298	\N	Saint-Christophe-et-le-Laris	26
9717	299	\N	Sainte-Croix	26
9718	300	\N	Saint-Dizier-en-Diois	26
9719	301	\N	Saint-Donat-sur-l'Herbasse	26
9720	302	\N	Sainte-Eulalie-en-Royans	26
9721	303	\N	Sainte-Euphémie-sur-Ouvèze	26
9722	304	\N	Saint-Ferréol-Trente-Pas	26
9723	305	\N	Saint-Gervais-sur-Roubion	26
9724	306	\N	Sainte-Jalle	26
9725	307	\N	Saint-Jean-en-Royans	26
9726	308	\N	Saint-Julien-en-Quint	26
9727	309	\N	Saint-Julien-en-Vercors	26
9728	310	\N	Saint-Laurent-d'Onay	26
9729	311	\N	Saint-Laurent-en-Royans	26
9730	312	\N	Saint-Marcel-lès-Sauzet	26
9731	313	\N	Saint-Marcel-lès-Valence	26
9732	314	\N	Saint-Martin-d'Août	26
9733	315	\N	Saint-Martin-en-Vercors	26
9734	316	\N	Saint-Martin-le-Colonel	26
9735	317	\N	Saint-Maurice-sur-Eygues	26
9736	318	\N	Saint-May	26
9737	319	\N	Saint-Michel-sur-Savasse	26
9738	320	\N	Saint-Nazaire-en-Royans	26
9739	321	\N	Saint-Nazaire-le-Désert	26
9740	322	\N	Saint-Pantaléon-les-Vignes	26
9741	323	\N	Saint-Paul-lès-Romans	26
9742	324	\N	Saint-Paul-Trois-Châteaux	26
9743	325	\N	Saint-Rambert-d'Albon	26
9744	326	\N	Saint-Restitut	26
9745	327	\N	Saint-Roman	26
9746	328	\N	Saint-Sauveur-en-Diois	26
9747	329	\N	Saint-Sauveur-Gouvernet	26
9748	330	\N	Saint-Sorlin-en-Valloire	26
9749	331	\N	Saint-Thomas-en-Royans	26
9750	332	\N	Saint-Uze	26
9751	333	\N	Saint-Vallier	26
9752	334	\N	Salettes	26
9753	335	\N	Salles-sous-Bois	26
9754	336	\N	Saou	26
9755	337	\N	Saulce-sur-Rhône	26
9756	338	\N	Sauzet	26
9757	339	\N	Savasse	26
9758	340	\N	Séderon	26
9759	341	\N	Serves-sur-Rhône	26
9760	342	\N	Solérieux	26
9761	343	\N	Souspierre	26
9762	344	\N	Soyans	26
9763	345	\N	Suze-la-Rousse	26
9764	346	\N	Suze	26
9765	347	\N	Tain-l'Hermitage	26
9766	348	\N	Taulignan	26
9767	349	\N	Tersanne	26
9768	350	\N	Teyssières	26
9769	351	\N	Tonils	26
9770	352	\N	Touche	26
9771	353	\N	Tourrettes	26
9772	354	\N	Treschenu-Creyers	26
9773	355	\N	Triors	26
9774	356	\N	Truinas	26
9775	357	\N	Tulette	26
9776	358	\N	Upie	26
9777	359	\N	Vachères-en-Quint	26
9778	360	\N	Valaurie	26
9779	361	\N	Valdrôme	26
9780	362	\N	Valence	26
9781	363	\N	Valouse	26
9782	364	\N	Vassieux-en-Vercors	26
9783	365	\N	Vaunaveys-la-Rochette	26
9784	366	\N	Veaunes	26
9785	367	\N	Venterol	26
9786	368	\N	Vercheny	26
9787	369	\N	Verclause	26
9788	370	\N	Vercoiran	26
9789	371	\N	Véronne	26
9790	372	\N	Vers-sur-Méouge	26
9791	373	\N	Vesc	26
9792	374	\N	Villebois-les-Pins	26
9793	375	\N	Villefranche-le-Château	26
9794	376	\N	Villeperdrix	26
9795	377	\N	Vinsobres	26
9796	378	\N	Volvent	26
9797	379	\N	Granges-les-Beaumont	26
9798	380	\N	Gervans	26
9799	381	\N	Jaillans	26
9800	382	\N	Saint-Vincent-la-Commanderie	26
9801	001	\N	Aclou	27
9802	002	\N	Acon	27
9803	003	\N	Acquigny	27
9804	004	\N	Aigleville	27
9805	005	\N	Ailly	27
9806	006	\N	Aizier	27
9807	007	\N	Ajou	27
9808	008	\N	Alizay	27
9809	009	\N	Ambenay	27
9810	010	\N	Amécourt	27
9811	011	\N	Amfreville-la-Campagne	27
9812	012	\N	Amfreville-les-Champs	27
9813	013	\N	Amfreville-sous-les-Monts	27
9814	014	\N	Amfreville-sur-Iton	27
9815	015	\N	Andé	27
9816	016	\N	Andelys	27
9817	017	\N	Angerville-la-Campagne	27
9818	018	\N	Appeville-Annebault	27
9819	019	\N	Armentières-sur-Avre	27
9820	020	\N	Arnières-sur-Iton	27
9821	021	\N	Asnières	27
9822	022	\N	Aubevoye	27
9823	023	\N	Aulnay-sur-Iton	27
9824	024	\N	Roncenay-Authenay	27
9825	025	\N	Autheuil-Authouillet	27
9826	026	\N	Authevernes	27
9827	027	\N	Authieux	27
9828	028	\N	Authou	27
9829	031	\N	Aviron	27
9830	032	\N	Avrilly	27
9831	033	\N	Bacquepuis	27
9832	034	\N	Bacqueville	27
9833	035	\N	Bailleul-la-Vallée	27
9834	036	\N	Bâlines	27
9835	037	\N	Barc	27
9836	038	\N	Barils	27
9837	039	\N	Barneville-sur-Seine	27
9838	040	\N	Barquet	27
9839	041	\N	Barre-en-Ouche	27
9840	042	\N	Barville	27
9841	043	\N	Baux-de-Breteuil	27
9842	044	\N	Baux-Sainte-Croix	27
9843	045	\N	Bazincourt-sur-Epte	27
9844	046	\N	Bazoques	27
9845	047	\N	Beaubray	27
9846	048	\N	Beauficel-en-Lyons	27
9847	049	\N	Beaumesnil	27
9848	050	\N	Beaumontel	27
9849	051	\N	Beaumont-le-Roger	27
9850	052	\N	Bec-Hellouin	27
9851	053	\N	Bec-Thomas	27
9852	054	\N	Bémécourt	27
9853	055	\N	Bérengeville-la-Campagne	27
9854	056	\N	Bernay	27
9855	057	\N	Bernienville	27
9856	058	\N	Bernières-sur-Seine	27
9857	059	\N	Bernouville	27
9858	060	\N	Berthenonville	27
9859	061	\N	Berthouville	27
9860	062	\N	Berville-en-Roumois	27
9861	063	\N	Berville-la-Campagne	27
9862	064	\N	Berville-sur-Mer	27
9863	065	\N	Beuzeville	27
9864	066	\N	Bézu-la-Forêt	27
9865	067	\N	Bézu-Saint-Éloi	27
9866	068	\N	Bois-Anzeray	27
9867	069	\N	Bois-Arnault	27
9868	070	\N	Boisemont	27
9869	071	\N	Bois-Hellain	27
9870	072	\N	Bois-Jérôme-Saint-Ouen	27
9871	073	\N	Bois-le-Roi	27
9872	074	\N	Boisney	27
9873	075	\N	Bois-Normand-près-Lyre	27
9874	076	\N	Boisset-les-Prévanches	27
9875	077	\N	Boissey-le-Châtel	27
9876	078	\N	Boissière	27
9877	079	\N	Boissy-Lamberville	27
9878	081	\N	Boncourt	27
9879	082	\N	Bonneville-sur-Iton	27
9880	083	\N	Bonneville-Aptot	27
9881	084	\N	Bosc-Bénard-Commin	27
9882	085	\N	Bosc-Bénard-Crescy	27
9883	088	\N	Bosc-Renoult-en-Ouche	27
9884	089	\N	Bosc-Renoult-en-Roumois	27
9885	090	\N	Bosc-Roger-en-Roumois	27
9886	091	\N	Bosgouet	27
9887	092	\N	Bosguérard-de-Marcouville	27
9888	093	\N	Bosnormand	27
9889	094	\N	Bosquentin	27
9890	095	\N	Bosrobert	27
9891	096	\N	Bottereaux	27
9892	097	\N	Bouafles	27
9893	098	\N	Bouchevilliers	27
9894	099	\N	Boulay-Morin	27
9895	100	\N	Boulleville	27
9896	101	\N	Bouquelon	27
9897	102	\N	Bouquetot	27
9898	103	\N	Bourg-Achard	27
9899	104	\N	Bourg-Beaudouin	27
9900	105	\N	Bourgtheroulde-Infreville	27
9901	106	\N	Bournainville-Faverolles	27
9902	107	\N	Bourneville	27
9903	108	\N	Bourth	27
9904	109	\N	Bray	27
9905	110	\N	Brestot	27
9906	111	\N	Bretagnolles	27
9907	112	\N	Breteuil	27
9908	113	\N	Brétigny	27
9909	114	\N	Breuilpont	27
9910	115	\N	Breux-sur-Avre	27
9911	116	\N	Brionne	27
9912	117	\N	Broglie	27
9913	118	\N	Brosville	27
9914	119	\N	Bueil	27
9915	120	\N	Burey	27
9916	121	\N	Bus-Saint-Rémy	27
9917	122	\N	Cahaignes	27
9918	123	\N	Caillouet-Orgeville	27
9919	124	\N	Cailly-sur-Eure	27
9920	125	\N	Calleville	27
9921	126	\N	Campigny	27
9922	127	\N	Canappeville	27
9923	128	\N	Cantiers	27
9924	129	\N	Caorches-Saint-Nicolas	27
9925	130	\N	Capelle-les-Grands	27
9926	131	\N	Carsix	27
9927	132	\N	Caugé	27
9928	133	\N	Caumont	27
9929	134	\N	Cauverville-en-Roumois	27
9930	135	\N	Cesseville	27
9931	136	\N	Chaignes	27
9932	137	\N	Chaise-Dieu-du-Theil	27
9933	138	\N	Chamblac	27
9934	139	\N	Chambord	27
9935	140	\N	Chambray	27
9936	141	\N	Champ-Dolent	27
9937	142	\N	Champenard	27
9938	143	\N	Champignolles	27
9939	144	\N	Champigny-la-Futelaye	27
9940	145	\N	Chanteloup	27
9941	146	\N	Chapelle-Bayvel	27
9942	147	\N	Chapelle-du-Bois-des-Faulx	27
9943	148	\N	Chapelle-Gauthier	27
9944	149	\N	Chapelle-Hareng	27
9945	150	\N	Chapelle-Réanville	27
9946	151	\N	Charleval	27
9947	152	\N	Château-sur-Epte	27
9948	153	\N	Chauvincourt-Provemont	27
9949	154	\N	Chavigny-Bailleul	27
9950	155	\N	Chennebrun	27
9951	156	\N	Chéronvilliers	27
9952	157	\N	Chesne	27
9953	158	\N	Cierrey	27
9954	159	\N	Cintray	27
9955	160	\N	Civières	27
9956	161	\N	Claville	27
9957	162	\N	Collandres-Quincarnon	27
9958	163	\N	Colletot	27
9959	164	\N	Combon	27
9960	165	\N	Conches-en-Ouche	27
9961	166	\N	Condé-sur-Iton	27
9962	167	\N	Condé-sur-Risle	27
9963	168	\N	Connelles	27
9964	169	\N	Conteville	27
9965	170	\N	Cormeilles	27
9966	171	\N	Cormier	27
9967	172	\N	Corneuil	27
9968	173	\N	Corneville-la-Fouquetière	27
9969	174	\N	Corneville-sur-Risle	27
9970	175	\N	Corny	27
9971	176	\N	Coudray	27
9972	177	\N	Coudres	27
9973	179	\N	Courbépine	27
9974	180	\N	Courcelles-sur-Seine	27
9975	181	\N	Courdemanche	27
9976	182	\N	Courteilles	27
9977	183	\N	Couture-Boussey	27
9978	184	\N	Crasville	27
9979	185	\N	Crestot	27
9980	187	\N	Criquebeuf-la-Campagne	27
9981	188	\N	Criquebeuf-sur-Seine	27
9982	189	\N	Croisille	27
9983	190	\N	Croisy-sur-Eure	27
9984	191	\N	Croix-Saint-Leufroy	27
9985	192	\N	Crosville-la-Vieille	27
9986	193	\N	Croth	27
9987	194	\N	Cuverville	27
9988	195	\N	Dame-Marie	27
9989	196	\N	Damps	27
9990	197	\N	Dampsmesnil	27
9991	198	\N	Damville	27
9992	199	\N	Dangu	27
9993	200	\N	Dardez	27
9994	201	\N	Daubeuf-la-Campagne	27
9995	202	\N	Daubeuf-près-Vatteville	27
9996	203	\N	Douains	27
9997	204	\N	Doudeauville-en-Vexin	27
9998	205	\N	Douville-sur-Andelle	27
9999	206	\N	Droisy	27
10000	207	\N	Drucourt	27
10001	208	\N	Duranville	27
10002	209	\N	Écaquelon	27
10003	210	\N	Écardenville-la-Campagne	27
10004	211	\N	Écardenville-sur-Eure	27
10005	212	\N	Écauville	27
10006	213	\N	Écos	27
10007	214	\N	Écouis	27
10008	215	\N	Ecquetot	27
10009	216	\N	Émalleville	27
10010	217	\N	Émanville	27
10011	218	\N	Épaignes	27
10012	219	\N	Épégard	27
10013	220	\N	Épieds	27
10014	221	\N	Épinay	27
10015	222	\N	Épreville-en-Lieuvin	27
10016	223	\N	Épreville-en-Roumois	27
10017	224	\N	Épreville-près-le-Neubourg	27
10018	225	\N	Essarts	27
10019	226	\N	Étrépagny	27
10020	227	\N	Étréville	27
10021	228	\N	Éturqueraye	27
10022	229	\N	Évreux	27
10023	230	\N	Ézy-sur-Eure	27
10024	231	\N	Fains	27
10025	232	\N	Farceaux	27
10026	233	\N	Fatouville-Grestain	27
10027	234	\N	Fauville	27
10028	235	\N	Faverolles-la-Campagne	27
10029	237	\N	Favril	27
10030	238	\N	Ferrières-Haut-Clocher	27
10031	239	\N	Ferrières-Saint-Hilaire	27
10032	240	\N	Ferrière-sur-Risle	27
10033	241	\N	Feuguerolles	27
10034	242	\N	Fidelaire	27
10035	243	\N	Fiquefleur-Équainville	27
10036	244	\N	Flancourt-Catelon	27
10037	245	\N	Fleury-la-Forêt	27
10038	246	\N	Fleury-sur-Andelle	27
10039	247	\N	Flipou	27
10040	248	\N	Folleville	27
10041	249	\N	Fontaine-Bellenger	27
10042	250	\N	Fontaine-Heudebourg	27
10043	251	\N	Fontaine-l'Abbé	27
10044	252	\N	Fontaine-la-Louvet	27
10045	253	\N	Fontaine-la-Soret	27
10046	254	\N	Fontaine-sous-Jouy	27
10047	255	\N	Fontenay	27
10048	256	\N	Forêt-du-Parc	27
10049	257	\N	Forêt-la-Folie	27
10050	258	\N	Fort-Moville	27
10051	259	\N	Foucrainville	27
10052	260	\N	Foulbec	27
10053	261	\N	Fouqueville	27
10054	262	\N	Fourges	27
10055	263	\N	Fourmetot	27
10056	264	\N	Fours-en-Vexin	27
10057	265	\N	Francheville	27
10058	266	\N	Franqueville	27
10059	267	\N	Freneuse-sur-Risle	27
10060	268	\N	Fresne	27
10061	269	\N	Fresne-Cauverville	27
10062	270	\N	Fresne-l'Archevêque	27
10063	271	\N	Fresney	27
10064	273	\N	Gadencourt	27
10065	274	\N	Gaillardbois-Cressenville	27
10066	275	\N	Gaillon	27
10067	276	\N	Gamaches-en-Vexin	27
10068	277	\N	Garencières	27
10069	278	\N	Garennes-sur-Eure	27
10070	279	\N	Gasny	27
10071	280	\N	Gauciel	27
10072	281	\N	Gaudreville-la-Rivière	27
10073	282	\N	Gauville-la-Campagne	27
10074	283	\N	Gisay-la-Coudre	27
10075	284	\N	Gisors	27
10076	285	\N	Giverny	27
10077	286	\N	Giverville	27
10078	287	\N	Glisolles	27
10079	288	\N	Glos-sur-Risle	27
10080	289	\N	Goulafrière	27
10081	290	\N	Goupillières	27
10082	291	\N	Gournay-le-Guérin	27
10083	292	\N	Gouttières	27
10084	293	\N	Gouville	27
10085	294	\N	Grainville	27
10086	295	\N	Grand-Camp	27
10087	296	\N	Grandchain	27
10088	297	\N	Grandvilliers	27
10089	298	\N	Graveron-Sémerville	27
10090	299	\N	Gravigny	27
10091	300	\N	Grosley-sur-Risle	27
10092	301	\N	Grossuvre	27
10093	302	\N	Gros-Theil	27
10094	303	\N	Guernanville	27
10095	304	\N	Guerny	27
10096	305	\N	Guéroulde	27
10097	306	\N	Guichainville	27
10098	307	\N	Guiseniers	27
10099	308	\N	Guitry	27
10100	309	\N	Habit	27
10101	310	\N	Hacqueville	27
10102	311	\N	Harcourt	27
10103	312	\N	Hardencourt-Cocherel	27
10104	313	\N	Harengère	27
10105	315	\N	Harquency	27
10106	316	\N	Hauville	27
10107	317	\N	Haye-Aubrée	27
10108	318	\N	Haye-de-Calleville	27
10109	319	\N	Haye-de-Routot	27
10110	320	\N	Haye-du-Theil	27
10111	321	\N	Haye-le-Comte	27
10112	322	\N	Haye-Malherbe	27
10113	323	\N	Haye-Saint-Sylvestre	27
10114	324	\N	Hébécourt	27
10115	325	\N	Hecmanville	27
10116	326	\N	Hécourt	27
10117	327	\N	Hectomare	27
10118	329	\N	Hennezis	27
10119	330	\N	Herqueville	27
10120	331	\N	Heubécourt-Haricourt	27
10121	332	\N	Heudebouville	27
10122	333	\N	Heudicourt	27
10123	334	\N	Heudreville-en-Lieuvin	27
10124	335	\N	Heudreville-sur-Eure	27
10125	336	\N	Heunière	27
10126	337	\N	Heuqueville	27
10127	338	\N	Hogues	27
10128	339	\N	Hondouville	27
10129	340	\N	Honguemare-Guenouville	27
10130	341	\N	Hosmes	27
10131	342	\N	Houetteville	27
10132	343	\N	Houlbec-Cocherel	27
10133	344	\N	Houlbec-près-le-Gros-Theil	27
10134	345	\N	Houssaye	27
10135	346	\N	Houville-en-Vexin	27
10136	347	\N	Huest	27
10137	348	\N	Igoville	27
10138	349	\N	Illeville-sur-Montfort	27
10139	350	\N	Illiers-l'Évêque	27
10140	351	\N	Incarville	27
10141	353	\N	Irreville	27
10142	354	\N	Iville	27
10143	355	\N	Ivry-la-Bataille	27
10144	356	\N	Jonquerets-de-Livet	27
10145	358	\N	Jouy-sur-Eure	27
10146	359	\N	Juignettes	27
10147	360	\N	Jumelles	27
10148	361	\N	Lande-Saint-Léger	27
10149	362	\N	Landepéreuse	27
10150	363	\N	Landin	27
10151	364	\N	Launay	27
10152	365	\N	Léry	27
10153	366	\N	Letteguives	27
10154	367	\N	Lieurey	27
10155	368	\N	Lignerolles	27
10156	369	\N	Lilly	27
10157	370	\N	Lisors	27
10158	371	\N	Livet-sur-Authou	27
10159	372	\N	Longchamps	27
10160	373	\N	Lorleau	27
10161	374	\N	Louversey	27
10162	375	\N	Louviers	27
10163	376	\N	Louye	27
10164	377	\N	Lyons-la-Forêt	27
10165	378	\N	Madeleine-de-Nonancourt	27
10166	379	\N	Mainneville	27
10167	380	\N	Malleville-sur-le-Bec	27
10168	381	\N	Malouy	27
10169	382	\N	Mandeville	27
10170	383	\N	Mandres	27
10171	384	\N	Manneville-la-Raoult	27
10172	385	\N	Manneville-sur-Risle	27
10173	386	\N	Manoir	27
10174	387	\N	Manthelon	27
10175	388	\N	Marais-Vernier	27
10176	389	\N	Marbeuf	27
10177	390	\N	Marcilly-la-Campagne	27
10178	391	\N	Marcilly-sur-Eure	27
10179	392	\N	Martagny	27
10180	393	\N	Martainville	27
10181	394	\N	Martot	27
10182	395	\N	Mélicourt	27
10183	396	\N	Ménesqueville	27
10184	397	\N	Ménilles	27
10185	398	\N	Menneval	27
10186	399	\N	Mercey	27
10187	400	\N	Merey	27
10188	401	\N	Mesnil-Fuguet	27
10189	402	\N	Mesnil-Hardray	27
10190	403	\N	Mesnil-Jourdain	27
10191	404	\N	Mesnil-Rousset	27
10192	405	\N	Mesnil-sous-Vienne	27
10193	406	\N	Mesnil-sur-l'Estrée	27
10194	407	\N	Mesnil-Verclives	27
10195	408	\N	Mézières-en-Vexin	27
10196	410	\N	Miserey	27
10197	411	\N	Moisville	27
10198	412	\N	Montaure	27
10199	413	\N	Montfort-sur-Risle	27
10200	414	\N	Montreuil-l'Argillé	27
10201	415	\N	Morainville-Jouveaux	27
10202	416	\N	Buis-sur-Damville	27
10203	417	\N	Morgny	27
10204	418	\N	Morsan	27
10205	419	\N	Mouettes	27
10206	420	\N	Mouflaines	27
10207	421	\N	Mousseaux-Neuville	27
10208	422	\N	Muids	27
10209	423	\N	Muzy	27
10210	424	\N	Nagel-Séez-Mesnil	27
10211	425	\N	Nassandres	27
10212	426	\N	Neaufles-Saint-Martin	27
10213	427	\N	Neaufles-Auvergny	27
10214	428	\N	Neubourg	27
10215	429	\N	Neuilly	27
10216	430	\N	Neuve-Grange	27
10217	431	\N	Neuve-Lyre	27
10218	432	\N	Neuville-du-Bosc	27
10219	433	\N	Neuville-sur-Authou	27
10220	434	\N	Noards	27
10221	435	\N	Noë-Poulain	27
10222	436	\N	Nogent-le-Sec	27
10223	437	\N	Nojeon-en-Vexin	27
10224	438	\N	Nonancourt	27
10225	439	\N	Normanville	27
10226	440	\N	Notre-Dame-de-l'Isle	27
10227	441	\N	Notre-Dame-d'Épine	27
10228	442	\N	Notre-Dame-du-Hamel	27
10229	444	\N	Noyer-en-Ouche	27
10230	445	\N	Noyers	27
10231	446	\N	Ormes	27
10232	447	\N	Orvaux	27
10233	448	\N	Pacy-sur-Eure	27
10234	449	\N	Panilleuse	27
10235	451	\N	Parville	27
10236	452	\N	Perriers-la-Campagne	27
10237	453	\N	Perriers-sur-Andelle	27
10238	454	\N	Perruel	27
10239	455	\N	Piencourt	27
10240	456	\N	Pinterville	27
10241	457	\N	Piseux	27
10242	458	\N	Pîtres	27
10243	459	\N	Places	27
10244	460	\N	Plainville	27
10245	462	\N	Planquay	27
10246	463	\N	Plasnes	27
10247	464	\N	Plessis-Grohan	27
10248	465	\N	Plessis-Hébert	27
10249	466	\N	Plessis-Sainte-Opportune	27
10250	467	\N	Pont-Audemer	27
10251	468	\N	Pont-Authou	27
10252	469	\N	Pont-de-l'Arche	27
10253	470	\N	Pont-Saint-Pierre	27
10254	471	\N	Porte-Joie	27
10255	472	\N	Portes	27
10256	473	\N	Port-Mort	27
10257	474	\N	Poses	27
10258	475	\N	Poterie-Mathieu	27
10259	476	\N	Préaux	27
10260	477	\N	Pressagny-l'Orgueilleux	27
10261	478	\N	Prey	27
10262	480	\N	Puchay	27
10263	481	\N	Pullay	27
10264	482	\N	Pyle	27
10265	483	\N	Quatremare	27
10266	484	\N	Quessigny	27
10267	485	\N	Quillebeuf-sur-Seine	27
10268	486	\N	Quittebeuf	27
10269	487	\N	Radepont	27
10270	488	\N	Renneville	27
10271	489	\N	Reuilly	27
10272	490	\N	Richeville	27
10273	491	\N	Roman	27
10274	492	\N	Romilly-la-Puthenaye	27
10275	493	\N	Romilly-sur-Andelle	27
10276	495	\N	Roquette	27
10277	496	\N	Rosay-sur-Lieure	27
10278	497	\N	Rougemontiers	27
10279	498	\N	Rouge-Perriers	27
10280	499	\N	Roussière	27
10281	500	\N	Routot	27
10282	501	\N	Rouvray	27
10283	502	\N	Rugles	27
10284	503	\N	Sacq	27
10285	504	\N	Sacquenville	27
10286	505	\N	Saint-Agnan-de-Cernières	27
10287	506	\N	Saint-Amand-des-Hautes-Terres	27
10288	507	\N	Saint-André-de-l'Eure	27
10289	508	\N	Saint-Antonin-de-Sommaire	27
10290	510	\N	Saint-Aquilin-de-Pacy	27
10291	511	\N	Saint-Aubin-d'Écrosville	27
10292	512	\N	Saint-Aubin-de-Scellon	27
10293	513	\N	Saint-Aubin-des-Hayes	27
10294	514	\N	Saint-Aubin-du-Thenney	27
10295	515	\N	Saint-Aubin-le-Guichard	27
10296	516	\N	Saint-Aubin-le-Vertueux	27
10297	517	\N	Saint-Aubin-sur-Gaillon	27
10298	518	\N	Saint-Aubin-sur-Quillebeuf	27
10299	519	\N	Sainte-Barbe-sur-Gaillon	27
10300	520	\N	Saint-Benoît-des-Ombres	27
10301	521	\N	Saint-Christophe-sur-Avre	27
10302	522	\N	Saint-Christophe-sur-Condé	27
10303	523	\N	Saint-Clair-d'Arcey	27
10304	524	\N	Sainte-Colombe-la-Commanderie	27
10305	525	\N	Sainte-Colombe-près-Vernon	27
10306	526	\N	Sainte-Croix-sur-Aizier	27
10307	527	\N	Saint-Cyr-de-Salerne	27
10308	528	\N	Vaudreuil	27
10309	529	\N	Saint-Cyr-la-Campagne	27
10310	530	\N	Saint-Denis-d'Augerons	27
10311	531	\N	Saint-Denis-des-Monts	27
10312	532	\N	Saint-Denis-du-Béhélan	27
10313	533	\N	Saint-Denis-le-Ferment	27
10314	534	\N	Saint-Didier-des-Bois	27
10315	535	\N	Saint-Élier	27
10316	536	\N	Saint-Éloi-de-Fourques	27
10317	537	\N	Saint-Étienne-du-Vauvray	27
10318	538	\N	Saint-Étienne-l'Allier	27
10319	539	\N	Saint-Étienne-sous-Bailleul	27
10320	540	\N	Sainte-Geneviève-lès-Gasny	27
10321	541	\N	Saint-Georges-du-Mesnil	27
10322	542	\N	Saint-Georges-du-Vièvre	27
10323	543	\N	Saint-Georges-Motel	27
10324	544	\N	Saint-Germain-de-Fresney	27
10325	545	\N	Saint-Germain-de-Pasquier	27
10326	546	\N	Saint-Germain-des-Angles	27
10327	547	\N	Saint-Germain-la-Campagne	27
10328	548	\N	Saint-Germain-sur-Avre	27
10329	549	\N	Saint-Germain-Village	27
10330	550	\N	Saint-Grégoire-du-Vièvre	27
10331	551	\N	Saint-Jean-de-la-Léqueraye	27
10332	552	\N	Saint-Jean-du-Thenney	27
10333	553	\N	Saint-Julien-de-la-Liègue	27
10334	554	\N	Saint-Just	27
10335	555	\N	Saint-Laurent-des-Bois	27
10336	556	\N	Saint-Laurent-du-Tencement	27
10337	557	\N	Saint-Léger-de-Rôtes	27
10338	558	\N	Saint-Léger-du-Gennetey	27
10339	560	\N	Saint-Luc	27
10340	561	\N	Saint-Maclou	27
10341	562	\N	Saint-Marcel	27
10342	563	\N	Saint-Mards-de-Blacarville	27
10343	564	\N	Saint-Mards-de-Fresne	27
10344	565	\N	Sainte-Marguerite-de-l'Autel	27
10345	566	\N	Sainte-Marguerite-en-Ouche	27
10346	567	\N	Sainte-Marie-de-Vatimesnil	27
10347	568	\N	Sainte-Marthe	27
10348	569	\N	Saint-Martin-du-Tilleul	27
10349	570	\N	Saint-Martin-la-Campagne	27
10350	571	\N	Saint-Martin-Saint-Firmin	27
10351	572	\N	Saint-Meslin-du-Bosc	27
10352	573	\N	Saint-Nicolas-d'Attez	27
10353	574	\N	Saint-Nicolas-du-Bosc	27
10354	576	\N	Sainte-Opportune-du-Bosc	27
10355	577	\N	Sainte-Opportune-la-Mare	27
10356	578	\N	Saint-Ouen-d'Attez	27
10357	579	\N	Saint-Ouen-de-Pontcheuil	27
10358	580	\N	Saint-Ouen-de-Thouberville	27
10359	581	\N	Saint-Ouen-des-Champs	27
10360	582	\N	Saint-Ouen-du-Tilleul	27
10361	584	\N	Saint-Paul-de-Fourques	27
10362	586	\N	Saint-Philbert-sur-Boissey	27
10363	587	\N	Saint-Philbert-sur-Risle	27
10364	588	\N	Saint-Pierre-d'Autils	27
10365	589	\N	Saint-Pierre-de-Bailleul	27
10366	590	\N	Saint-Pierre-de-Cernières	27
10367	591	\N	Saint-Pierre-de-Cormeilles	27
10368	592	\N	Saint-Pierre-de-Salerne	27
10369	593	\N	Saint-Pierre-des-Fleurs	27
10370	594	\N	Saint-Pierre-des-Ifs	27
10371	595	\N	Saint-Pierre-du-Bosguérard	27
10372	596	\N	Saint-Pierre-du-Mesnil	27
10373	597	\N	Saint-Pierre-du-Val	27
10374	598	\N	Saint-Pierre-du-Vauvray	27
10375	599	\N	Saint-Pierre-la-Garenne	27
10376	600	\N	Saint-Quentin-des-Isles	27
10377	601	\N	Saint-Samson-de-la-Roque	27
10378	602	\N	Saint-Sébastien-de-Morsent	27
10379	603	\N	Saint-Siméon	27
10380	604	\N	Saint-Sulpice-de-Grimbouville	27
10381	605	\N	Saint-Sylvestre-de-Cormeilles	27
10382	606	\N	Saint-Symphorien	27
10383	607	\N	Saint-Thurien	27
10384	608	\N	Saint-Victor-de-Chrétienville	27
10385	609	\N	Saint-Victor-d'Épine	27
10386	610	\N	Saint-Victor-sur-Avre	27
10387	611	\N	Saint-Vigor	27
10388	612	\N	Saint-Vincent-des-Bois	27
10389	613	\N	Saint-Vincent-du-Boulay	27
10390	614	\N	Sancourt	27
10391	615	\N	Sassey	27
10392	616	\N	Saussaye	27
10393	617	\N	Saussay-la-Campagne	27
10394	618	\N	Sébécourt	27
10395	620	\N	Selles	27
10396	621	\N	Serez	27
10397	622	\N	Serquigny	27
10398	623	\N	Surtauville	27
10399	624	\N	Surville	27
10400	625	\N	Suzay	27
10401	626	\N	Theillement	27
10402	627	\N	Theil-Nolent	27
10403	628	\N	Thevray	27
10404	629	\N	Thiberville	27
10405	630	\N	Thibouville	27
10406	631	\N	Thierville	27
10407	632	\N	Thil	27
10408	633	\N	Thilliers-en-Vexin	27
10409	634	\N	Thomer-la-Sôgne	27
10410	635	\N	Thuit	27
10411	636	\N	Thuit-Anger	27
10412	637	\N	Thuit-Hébert	27
10413	638	\N	Thuit-Signol	27
10414	639	\N	Thuit-Simer	27
10415	640	\N	Tilleul-Dame-Agnès	27
10416	641	\N	Tilleul-Lambert	27
10417	642	\N	Tilleul-Othon	27
10418	643	\N	Tillières-sur-Avre	27
10419	644	\N	Tilly	27
10420	645	\N	Tocqueville	27
10421	646	\N	Torpt	27
10422	647	\N	Tosny	27
10423	648	\N	Tostes	27
10424	649	\N	Touffreville	27
10425	650	\N	Tournedos-Bois-Hubert	27
10426	651	\N	Tournedos-sur-Seine	27
10427	652	\N	Tourneville	27
10428	653	\N	Tourny	27
10429	654	\N	Tourville-la-Campagne	27
10430	655	\N	Tourville-sur-Pont-Audemer	27
10431	656	\N	Toutainville	27
10432	657	\N	Touville	27
10433	658	\N	Tremblay-Omonville	27
10434	659	\N	Trinité	27
10435	660	\N	Trinité-de-Réville	27
10436	661	\N	Trinité-de-Thouberville	27
10437	662	\N	Triqueville	27
10438	663	\N	Troncq	27
10439	664	\N	Tronquay	27
10440	665	\N	Trouville-la-Haule	27
10441	666	\N	Vacherie	27
10442	667	\N	Valailles	27
10443	668	\N	Val-David	27
10444	669	\N	Valletot	27
10445	670	\N	Vandrimare	27
10446	671	\N	Vannecrocq	27
10447	672	\N	Vascuil	27
10448	673	\N	Vatteville	27
10449	674	\N	Vaux-sur-Eure	27
10450	676	\N	Venables	27
10451	677	\N	Venon	27
10452	678	\N	Ventes	27
10453	679	\N	Verneuil-sur-Avre	27
10454	680	\N	Verneusses	27
10455	681	\N	Vernon	27
10456	682	\N	Vesly	27
10457	683	\N	Vézillon	27
10458	684	\N	Vieil-Évreux	27
10459	685	\N	Vieille-Lyre	27
10460	686	\N	Vieux-Port	27
10461	687	\N	Vieux-Villez	27
10462	688	\N	Villalet	27
10463	689	\N	Villegats	27
10464	690	\N	Villers-en-Vexin	27
10465	691	\N	Villers-sur-le-Roule	27
10466	692	\N	Villettes	27
10467	693	\N	Sylvains-les-Moulins	27
10468	694	\N	Villez-sous-Bailleul	27
10469	695	\N	Villez-sur-le-Neubourg	27
10470	696	\N	Villiers-en-Désuvre	27
10471	697	\N	Vironvay	27
10472	698	\N	Vitot	27
10473	699	\N	Voiscreville	27
10474	700	\N	Vraiville	27
10475	701	\N	Val-de-Reuil	27
10476	001	\N	Abondant	28
10477	002	\N	Allaines-Mervilliers	28
10478	003	\N	Allainville	28
10479	004	\N	Allonnes	28
10480	005	\N	Alluyes	28
10481	006	\N	Amilly	28
10482	007	\N	Anet	28
10483	008	\N	Ardelles	28
10484	009	\N	Ardelu	28
10485	010	\N	Argenvilliers	28
10486	012	\N	Arrou	28
10487	013	\N	Aunay-sous-Auneau	28
10488	014	\N	Aunay-sous-Crécy	28
10489	015	\N	Auneau	28
10490	016	\N	Autels-Villevillon	28
10491	017	\N	Autheuil	28
10492	018	\N	Authon-du-Perche	28
10493	019	\N	Baigneaux	28
10494	020	\N	Baignolet	28
10495	021	\N	Bailleau-le-Pin	28
10496	022	\N	Bailleau-l'Évêque	28
10497	023	\N	Bailleau-Armenonville	28
10498	024	\N	Barjouville	28
10499	025	\N	Barmainville	28
10500	026	\N	Baudreville	28
10501	027	\N	Bazoche-Gouet	28
10502	028	\N	Bazoches-en-Dunois	28
10503	029	\N	Bazoches-les-Hautes	28
10504	030	\N	Beauche	28
10505	031	\N	Beaumont-les-Autels	28
10506	032	\N	Beauvilliers	28
10507	033	\N	Belhomert-Guéhouville	28
10508	034	\N	Berchères-Saint-Germain	28
10509	035	\N	Berchères-les-Pierres	28
10510	036	\N	Berchères-sur-Vesgre	28
10511	037	\N	Bérou-la-Mulotière	28
10512	038	\N	Béthonvilliers	28
10513	039	\N	Béville-le-Comte	28
10514	040	\N	Billancelles	28
10515	041	\N	Blandainville	28
10516	042	\N	Bleury	28
10517	044	\N	Boisgasson	28
10518	045	\N	Boissy-en-Drouais	28
10519	046	\N	Boissy-lès-Perche	28
10520	047	\N	Boisville-la-Saint-Père	28
10521	048	\N	Bourdinière-Saint-Loup	28
10522	049	\N	Boncé	28
10523	050	\N	Boncourt	28
10524	051	\N	Bonneval	28
10525	052	\N	Bouglainval	28
10526	053	\N	Boullay-les-Deux-Églises	28
10527	054	\N	Boullay-Mivoye	28
10528	055	\N	Boullay-Thierry	28
10529	056	\N	Boutigny-Prouais	28
10530	057	\N	Bouville	28
10531	058	\N	Bréchamps	28
10532	059	\N	Brezolles	28
10533	060	\N	Briconville	28
10534	061	\N	Brou	28
10535	062	\N	Broué	28
10536	063	\N	Brunelles	28
10537	064	\N	Bû	28
10538	065	\N	Bullainville	28
10539	066	\N	Bullou	28
10540	067	\N	Cernay	28
10541	068	\N	Challet	28
10542	069	\N	Champagne	28
10543	070	\N	Champhol	28
10544	071	\N	Champrond-en-Gâtine	28
10545	072	\N	Champrond-en-Perchet	28
10546	073	\N	Champseru	28
10547	074	\N	Chapelle-d'Aunainville	28
10548	075	\N	Chapelle-du-Noyer	28
10549	076	\N	Chapelle-Forainvilliers	28
10550	077	\N	Chapelle-Fortin	28
10551	078	\N	Chapelle-Guillaume	28
10552	079	\N	Chapelle-Royale	28
10553	080	\N	Charbonnières	28
10554	081	\N	Charonville	28
10555	082	\N	Charpont	28
10556	083	\N	Charray	28
10557	084	\N	Chartainvilliers	28
10558	085	\N	Chartres	28
10559	086	\N	Chassant	28
10560	087	\N	Châtaincourt	28
10561	088	\N	Châteaudun	28
10562	089	\N	Châteauneuf-en-Thymerais	28
10563	090	\N	Châtelets	28
10564	091	\N	Châtelliers-Notre-Dame	28
10565	092	\N	Châtenay	28
10566	093	\N	Châtillon-en-Dunois	28
10567	094	\N	Chaudon	28
10568	095	\N	Chauffours	28
10569	096	\N	Chaussée-d'Ivry	28
10570	098	\N	Cherisy	28
10571	099	\N	Chuisnes	28
10572	100	\N	Cintray	28
10573	101	\N	Civry	28
10574	102	\N	Clévilliers	28
10575	103	\N	Cloyes-sur-le-Loir	28
10576	104	\N	Coltainville	28
10577	105	\N	Combres	28
10578	106	\N	Conie-Molitard	28
10579	107	\N	Corancez	28
10580	108	\N	Cormainville	28
10581	109	\N	Corvées-les-Yys	28
10582	110	\N	Coudray	28
10583	111	\N	Coudray-au-Perche	28
10584	112	\N	Coudreceau	28
10585	113	\N	Coulombs	28
10586	114	\N	Courbehaye	28
10587	115	\N	Courtalain	28
10588	116	\N	Courville-sur-Eure	28
10589	117	\N	Crécy-Couvé	28
10590	118	\N	Croisilles	28
10591	119	\N	Croix-du-Perche	28
10592	120	\N	Crucey-Villages	28
10593	121	\N	Dambron	28
10594	122	\N	Dammarie	28
10595	123	\N	Dampierre-sous-Brou	28
10596	124	\N	Dampierre-sur-Avre	28
10597	126	\N	Dancy	28
10598	127	\N	Dangeau	28
10599	128	\N	Dangers	28
10600	129	\N	Denonville	28
10601	130	\N	Digny	28
10602	132	\N	Donnemain-Saint-Mamès	28
10603	133	\N	Douy	28
10604	134	\N	Dreux	28
10605	135	\N	Droue-sur-Drouette	28
10606	136	\N	Écluzelles	28
10607	137	\N	Écrosnes	28
10608	139	\N	Épeautrolles	28
10609	140	\N	Épernon	28
10610	141	\N	Ermenonville-la-Grande	28
10611	142	\N	Ermenonville-la-Petite	28
10612	143	\N	Escorpain	28
10613	144	\N	Étilleux	28
10614	145	\N	Fains-la-Folie	28
10615	146	\N	Faverolles	28
10616	147	\N	Favières	28
10617	148	\N	Favril	28
10618	149	\N	Ferté-Vidame	28
10619	150	\N	Ferté-Villeneuil	28
10620	151	\N	Fessanvilliers-Mattanvilliers	28
10621	153	\N	Flacey	28
10622	154	\N	Fontaine-la-Guyon	28
10623	155	\N	Fontaine-les-Ribouts	28
10624	156	\N	Fontaine-Simon	28
10625	157	\N	Fontenay-sur-Conie	28
10626	158	\N	Fontenay-sur-Eure	28
10627	159	\N	Framboisière	28
10628	160	\N	Francourville	28
10629	161	\N	Frazé	28
10630	162	\N	Fresnay-le-Comte	28
10631	163	\N	Fresnay-le-Gilmert	28
10632	164	\N	Fresnay-l'Évêque	28
10633	165	\N	Frétigny	28
10634	166	\N	Friaize	28
10635	167	\N	Fruncé	28
10636	168	\N	Gallardon	28
10637	169	\N	Garancières-en-Beauce	28
10638	170	\N	Garancières-en-Drouais	28
10639	171	\N	Garnay	28
10640	172	\N	Gas	28
10641	173	\N	Gasville-Oisème	28
10642	175	\N	Gaudaine	28
10643	176	\N	Gault-Saint-Denis	28
10644	177	\N	Gellainville	28
10645	178	\N	Germainville	28
10646	179	\N	Germignonville	28
10647	180	\N	Gilles	28
10648	182	\N	Gohory	28
10649	183	\N	Gommerville	28
10650	184	\N	Gouillons	28
10651	185	\N	Goussainville	28
10652	187	\N	Guainville	28
10653	188	\N	Gué-de-Longroi	28
10654	189	\N	Guilleville	28
10655	190	\N	Guillonville	28
10656	191	\N	Hanches	28
10657	192	\N	Happonvilliers	28
10658	193	\N	Havelu	28
10659	194	\N	Houville-la-Branche	28
10660	195	\N	Houx	28
10661	196	\N	Illiers-Combray	28
10662	197	\N	Intréville	28
10663	198	\N	Jallans	28
10664	199	\N	Janville	28
10665	200	\N	Jaudrais	28
10666	201	\N	Jouy	28
10667	202	\N	Lamblore	28
10668	203	\N	Landelles	28
10669	204	\N	Langey	28
10670	205	\N	Lanneray	28
10671	206	\N	Laons	28
10672	207	\N	Léthuin	28
10673	208	\N	Levainville	28
10674	209	\N	Lèves	28
10675	210	\N	Levesville-la-Chenard	28
10676	211	\N	Logron	28
10677	212	\N	Loigny-la-Bataille	28
10678	213	\N	Lormaye	28
10679	214	\N	Loupe	28
10680	215	\N	Louville-la-Chenard	28
10681	216	\N	Louvilliers-en-Drouais	28
10682	217	\N	Louvilliers-lès-Perche	28
10683	218	\N	Lucé	28
10684	219	\N	Luigny	28
10685	220	\N	Luisant	28
10686	221	\N	Lumeau	28
10687	222	\N	Luplanté	28
10688	223	\N	Luray	28
10689	224	\N	Lutz-en-Dunois	28
10690	225	\N	Magny	28
10691	226	\N	Maillebois	28
10692	227	\N	Maintenon	28
10693	229	\N	Mainvilliers	28
10694	230	\N	Maisons	28
10695	231	\N	Mancelière	28
10696	232	\N	Manou	28
10697	233	\N	Marboué	28
10698	234	\N	Marchéville	28
10699	235	\N	Marchezais	28
10700	236	\N	Margon	28
10701	237	\N	Marolles-les-Buis	28
10702	239	\N	Marville-Moutiers-Brûlé	28
10703	240	\N	Meaucé	28
10704	241	\N	Mée	28
10705	242	\N	Méréglise	28
10706	243	\N	Mérouville	28
10707	245	\N	Meslay-le-Grenet	28
10708	246	\N	Meslay-le-Vidame	28
10709	247	\N	Mesnil-Simon	28
10710	248	\N	Mesnil-Thomas	28
10711	249	\N	Mévoisins	28
10712	250	\N	Mézières-au-Perche	28
10713	251	\N	Mézières-en-Drouais	28
10714	252	\N	Miermaigne	28
10715	253	\N	Mignières	28
10716	254	\N	Mittainvilliers	28
10717	255	\N	Moinville-la-Jeulin	28
10718	256	\N	Moléans	28
10719	257	\N	Mondonville-Saint-Jean	28
10720	258	\N	Montainville	28
10721	259	\N	Montboissier	28
10722	260	\N	Montharville	28
10723	261	\N	Montigny-le-Chartif	28
10724	262	\N	Montigny-le-Gannelon	28
10725	263	\N	Montigny-sur-Avre	28
10726	264	\N	Montireau	28
10727	265	\N	Montlandon	28
10728	267	\N	Montreuil	28
10729	268	\N	Morainville	28
10730	269	\N	Morancez	28
10731	270	\N	Moriers	28
10732	271	\N	Morvilliers	28
10733	272	\N	Mottereau	28
10734	273	\N	Moulhard	28
10735	274	\N	Moutiers	28
10736	275	\N	Néron	28
10737	276	\N	Neuvy-en-Beauce	28
10738	277	\N	Neuvy-en-Dunois	28
10739	278	\N	Nogent-le-Phaye	28
10740	279	\N	Nogent-le-Roi	28
10741	280	\N	Nogent-le-Rotrou	28
10742	281	\N	Nogent-sur-Eure	28
10743	282	\N	Nonvilliers-Grandhoux	28
10744	283	\N	Nottonville	28
10745	284	\N	Oinville-Saint-Liphard	28
10746	285	\N	Oinville-sous-Auneau	28
10747	286	\N	Ollé	28
10748	287	\N	Orgères-en-Beauce	28
10749	288	\N	Orlu	28
10750	289	\N	Ormoy	28
10751	290	\N	Orrouer	28
10752	291	\N	Ouarville	28
10753	292	\N	Ouerre	28
10754	293	\N	Oulins	28
10755	294	\N	Oysonville	28
10756	295	\N	Ozoir-le-Breuil	28
10757	296	\N	Péronville	28
10758	297	\N	Pézy	28
10759	298	\N	Pierres	28
10760	299	\N	Pinthières	28
10761	300	\N	Poinville	28
10762	301	\N	Poisvilliers	28
10763	302	\N	Pontgouin	28
10764	303	\N	Poupry	28
10765	304	\N	Prasville	28
10766	305	\N	Pré-Saint-Évroult	28
10767	306	\N	Pré-Saint-Martin	28
10768	308	\N	Prudemanche	28
10769	309	\N	Prunay-le-Gillon	28
10770	310	\N	Puisaye	28
10771	311	\N	Puiset	28
10772	312	\N	Puiseux	28
10773	313	\N	Réclainville	28
10774	314	\N	Ressuintes	28
10775	315	\N	Revercourt	28
10776	316	\N	Rohaire	28
10777	317	\N	Roinville	28
10778	318	\N	Romilly-sur-Aigre	28
10779	319	\N	Rouvray-Saint-Denis	28
10780	320	\N	Rouvray-Saint-Florentin	28
10781	321	\N	Rouvres	28
10782	322	\N	Rueil-la-Gadelière	28
10783	323	\N	Saint-Ange-et-Torçay	28
10784	324	\N	Saint-Arnoult-des-Bois	28
10785	325	\N	Saint-Aubin-des-Bois	28
10786	326	\N	Saint-Avit-les-Guespières	28
10787	327	\N	Saint-Bomer	28
10788	329	\N	Saint-Christophe	28
10789	330	\N	Saint-Cloud-en-Dunois	28
10790	331	\N	Saint-Denis-d'Authou	28
10791	332	\N	Sainte-Gemme-Moronval	28
10792	333	\N	Saint-Denis-des-Puits	28
10793	334	\N	Saint-Denis-les-Ponts	28
10794	335	\N	Saint-Éliph	28
10795	336	\N	Saint-Éman	28
10796	337	\N	Saint-Georges-sur-Eure	28
10797	339	\N	Saint-Germain-le-Gaillard	28
10798	340	\N	Saint-Hilaire-sur-Yerre	28
10799	341	\N	Saint-Jean-de-Rebervilliers	28
10800	342	\N	Saint-Jean-Pierre-Fixte	28
10801	343	\N	Saint-Laurent-la-Gâtine	28
10802	344	\N	Saint-Léger-des-Aubées	28
10803	346	\N	Saint-Lubin-de-Cravant	28
10804	347	\N	Saint-Lubin-de-la-Haye	28
10805	348	\N	Saint-Lubin-des-Joncherets	28
10806	349	\N	Saint-Lucien	28
10807	350	\N	Saint-Luperce	28
10808	351	\N	Saint-Maixme-Hauterive	28
10809	352	\N	Saint-Martin-de-Nigelles	28
10810	353	\N	Saint-Maur-sur-le-Loir	28
10811	354	\N	Saint-Maurice-Saint-Germain	28
10812	355	\N	Saint-Ouen-Marchefroy	28
10813	356	\N	Saint-Pellerin	28
10814	357	\N	Saint-Piat	28
10815	358	\N	Saint-Prest	28
10816	359	\N	Saint-Rémy-sur-Avre	28
10817	360	\N	Saint-Sauveur-Marville	28
10818	361	\N	Saint-Symphorien-le-Château	28
10819	362	\N	Saint-Victor-de-Buthon	28
10820	363	\N	Sainville	28
10821	364	\N	Sancheville	28
10822	365	\N	Sandarville	28
10823	366	\N	Santeuil	28
10824	367	\N	Santilly	28
10825	368	\N	Saucelle	28
10826	369	\N	Saulnières	28
10827	370	\N	Saumeray	28
10828	371	\N	Saussay	28
10829	372	\N	Senantes	28
10830	373	\N	Senonches	28
10831	374	\N	Serazereux	28
10832	375	\N	Serville	28
10833	376	\N	Soizé	28
10834	377	\N	Sorel-Moussel	28
10835	378	\N	Souancé-au-Perche	28
10836	379	\N	Soulaires	28
10837	380	\N	Sours	28
10838	382	\N	Terminiers	28
10839	383	\N	Theuville	28
10840	385	\N	Thieulin	28
10841	386	\N	Thimert-Gâtelles	28
10842	387	\N	Thiron Gardais	28
10843	388	\N	Thivars	28
10844	389	\N	Thiville	28
10845	390	\N	Tillay-le-Péneux	28
10846	391	\N	Toury	28
10847	392	\N	Trancrainville	28
10848	393	\N	Tremblay-les-Villages	28
10849	394	\N	Tréon	28
10850	395	\N	Trizay-Coutretot-Saint-Serge	28
10851	396	\N	Trizay-lès-Bonneval	28
10852	397	\N	Umpeau	28
10853	398	\N	Unverre	28
10854	400	\N	Varize	28
10855	401	\N	Vaupillon	28
10856	402	\N	Vérigny	28
10857	403	\N	Ver-lès-Chartres	28
10858	404	\N	Vernouillet	28
10859	405	\N	Vert-en-Drouais	28
10860	406	\N	Viabon	28
10861	407	\N	Vichères	28
10862	408	\N	Vierville	28
10863	409	\N	Vieuvicq	28
10864	410	\N	Villampuy	28
10865	411	\N	Villars	28
10866	412	\N	Villeau	28
10867	414	\N	Villebon	28
10868	415	\N	Villemeux-sur-Eure	28
10869	416	\N	Villeneuve-Saint-Nicolas	28
10870	417	\N	Villiers-le-Morhier	28
10871	418	\N	Villiers-Saint-Orien	28
10872	419	\N	Vitray-en-Beauce	28
10873	421	\N	Voise	28
10874	422	\N	Voves	28
10875	423	\N	Yermenonville	28
10876	424	\N	Yèvres	28
10877	425	\N	Ymeray	28
10878	426	\N	Ymonville	28
10879	001	\N	Argol	29
10880	002	\N	Arzano	29
10881	003	\N	Audierne	29
10882	004	\N	Bannalec	29
10883	005	\N	Baye	29
10884	006	\N	Bénodet	29
10885	007	\N	Berrien	29
10886	008	\N	Beuzec-Cap-Sizun	29
10887	010	\N	Bodilis	29
10888	011	\N	Bohars	29
10889	012	\N	Bolazec	29
10890	013	\N	Botmeur	29
10891	014	\N	Botsorhel	29
10892	015	\N	Bourg-Blanc	29
10893	016	\N	Brasparts	29
10894	017	\N	Brélès	29
10895	018	\N	Brennilis	29
10896	019	\N	Brest	29
10897	020	\N	Briec	29
10898	021	\N	Brignogan-Plage	29
10899	022	\N	Camaret-sur-Mer	29
10900	023	\N	Carantec	29
10901	024	\N	Carhaix-Plouguer	29
10902	025	\N	Cast	29
10903	026	\N	Châteaulin	29
10904	027	\N	Châteauneuf-du-Faou	29
10905	028	\N	Cléden-Cap-Sizun	29
10906	029	\N	Cléden-Poher	29
10907	030	\N	Cléder	29
10908	031	\N	Clohars-Carnoët	29
10909	032	\N	Clohars-Fouesnant	29
10910	033	\N	Cloître-Pleyben	29
10911	034	\N	Cloître-Saint-Thégonnec	29
10912	035	\N	Coat-Méal	29
10913	036	\N	Collorec	29
10914	037	\N	Combrit	29
10915	038	\N	Commana	29
10916	039	\N	Concarneau	29
10917	040	\N	Conquet	29
10918	041	\N	Coray	29
10919	042	\N	Crozon	29
10920	043	\N	Daoulas	29
10921	044	\N	Dinéault	29
10922	045	\N	Dirinon	29
10923	046	\N	Douarnenez	29
10924	047	\N	Drennec	29
10925	048	\N	Edern	29
10926	049	\N	Elliant	29
10927	051	\N	Ergué-Gabéric	29
10928	052	\N	Esquibien	29
10929	053	\N	Faou	29
10930	054	\N	Feuillée	29
10931	055	\N	Folgoët	29
10932	056	\N	Forest-Landerneau	29
10933	057	\N	Forêt-Fouesnant	29
10934	058	\N	Fouesnant	29
10935	059	\N	Garlan	29
10936	060	\N	Gouesnach	29
10937	061	\N	Gouesnou	29
10938	062	\N	Gouézec	29
10939	063	\N	Goulien	29
10940	064	\N	Goulven	29
10941	065	\N	Gourlizon	29
10942	066	\N	Guengat	29
10943	067	\N	Guerlesquin	29
10944	068	\N	Guiclan	29
10945	069	\N	Guilers	29
10946	070	\N	Guiler-sur-Goyen	29
10947	071	\N	Guilligomarc'h	29
10948	072	\N	Guilvinec	29
10949	073	\N	Guimaëc	29
10950	074	\N	Guimiliau	29
10951	075	\N	Guipavas	29
10952	076	\N	Guipronvel	29
10953	077	\N	Guissény	29
10954	078	\N	Hanvec	29
10955	079	\N	Henvic	29
10956	080	\N	Hôpital-Camfrout	29
10957	081	\N	Huelgoat	29
10958	082	\N	Île-de-Batz	29
10959	083	\N	Île-de-Sein	29
10960	084	\N	Île-Molène	29
10961	085	\N	Île-Tudy	29
10962	086	\N	Irvillac	29
10963	087	\N	Juch	29
10964	089	\N	Kergloff	29
10965	090	\N	Kerlaz	29
10966	091	\N	Kerlouan	29
10967	093	\N	Kernilis	29
10968	094	\N	Kernouës	29
10969	095	\N	Kersaint-Plabennec	29
10970	097	\N	Lampaul-Guimiliau	29
10971	098	\N	Lampaul-Plouarzel	29
10972	099	\N	Lampaul-Ploudalmézeau	29
10973	100	\N	Lanarvily	29
10974	101	\N	Landéda	29
10975	102	\N	Landeleau	29
10976	103	\N	Landerneau	29
10977	104	\N	Landévennec	29
10978	105	\N	Landivisiau	29
10979	106	\N	Landrévarzec	29
10980	107	\N	Landudal	29
10981	108	\N	Landudec	29
10982	109	\N	Landunvez	29
10983	110	\N	Langolen	29
10984	111	\N	Lanhouarneau	29
10985	112	\N	Lanildut	29
10986	113	\N	Lanmeur	29
10987	114	\N	Lannéanou	29
10988	115	\N	Lannédern	29
10989	116	\N	Lanneuffret	29
10990	117	\N	Lannilis	29
10991	119	\N	Lanrivoaré	29
10992	120	\N	Lanvéoc	29
10993	122	\N	Laz	29
10994	123	\N	Lennon	29
10995	124	\N	Lesneven	29
10996	125	\N	Leuhan	29
10997	126	\N	Loc-Brévalaire	29
10998	127	\N	Loc-Eguiner-Saint-Thégonnec	29
10999	128	\N	Loc-Eguiner	29
11000	129	\N	Locmaria-Berrien	29
11001	130	\N	Locmaria-Plouzané	29
11002	131	\N	Locmélar	29
11003	132	\N	Locquénolé	29
11004	133	\N	Locquirec	29
11005	134	\N	Locronan	29
11006	135	\N	Loctudy	29
11007	136	\N	Locunolé	29
11008	137	\N	Logonna-Daoulas	29
11009	139	\N	Lopérec	29
11010	140	\N	Loperhet	29
11011	141	\N	Loqueffret	29
11012	142	\N	Lothey	29
11013	143	\N	Mahalon	29
11014	144	\N	Martyre	29
11015	145	\N	Confort-Meilars	29
11016	146	\N	Melgven	29
11017	147	\N	Mellac	29
11018	148	\N	Mespaul	29
11019	149	\N	Milizac	29
11020	150	\N	Moëlan-sur-Mer	29
11021	151	\N	Morlaix	29
11022	152	\N	Motreff	29
11023	153	\N	Névez	29
11024	155	\N	Ouessant	29
11025	156	\N	Pencran	29
11026	158	\N	Penmarch	29
11027	159	\N	Peumérit	29
11028	160	\N	Plabennec	29
11029	161	\N	Pleuven	29
11030	162	\N	Pleyben	29
11031	163	\N	Pleyber-Christ	29
11032	165	\N	Plobannalec-Lesconil	29
11033	166	\N	Ploéven	29
11034	167	\N	Plogastel-Saint-Germain	29
11035	168	\N	Plogoff	29
11036	169	\N	Plogonnec	29
11037	170	\N	Plomelin	29
11038	171	\N	Plomeur	29
11039	172	\N	Plomodiern	29
11040	173	\N	Plonéis	29
11041	174	\N	Plonéour-Lanvern	29
11042	175	\N	Plonévez-du-Faou	29
11043	176	\N	Plonévez-Porzay	29
11044	177	\N	Plouarzel	29
11045	178	\N	Ploudalmézeau	29
11046	179	\N	Ploudaniel	29
11047	180	\N	Ploudiry	29
11048	181	\N	Plouédern	29
11049	182	\N	Plouégat-Guérand	29
11050	183	\N	Plouégat-Moysan	29
11051	184	\N	Plouénan	29
11052	185	\N	Plouescat	29
11053	186	\N	Plouezoc'h	29
11054	187	\N	Plougar	29
11055	188	\N	Plougasnou	29
11056	189	\N	Plougastel-Daoulas	29
11057	190	\N	Plougonvelin	29
11058	191	\N	Plougonven	29
11059	192	\N	Plougoulm	29
11060	193	\N	Plougourvest	29
11061	195	\N	Plouguerneau	29
11062	196	\N	Plouguin	29
11063	197	\N	Plouhinec	29
11064	198	\N	Plouider	29
11065	199	\N	Plouigneau	29
11066	201	\N	Ploumoguer	29
11067	202	\N	Plounéour-Ménez	29
11068	203	\N	Plounéour-Trez	29
11069	204	\N	Plounéventer	29
11070	205	\N	Plounévézel	29
11071	206	\N	Plounévez-Lochrist	29
11072	207	\N	Plourin-lès-Morlaix	29
11073	208	\N	Plourin	29
11074	209	\N	Plouvien	29
11075	210	\N	Plouvorn	29
11076	211	\N	Plouyé	29
11077	212	\N	Plouzané	29
11078	213	\N	Plouzévédé	29
11079	214	\N	Plovan	29
11080	215	\N	Plozévet	29
11081	216	\N	Pluguffan	29
11082	217	\N	Pont-Aven	29
11083	218	\N	Pont-Croix	29
11084	219	\N	Ponthou	29
11085	220	\N	Pont-l'Abbé	29
11086	221	\N	Porspoder	29
11087	222	\N	Port-Launay	29
11088	224	\N	Pouldergat	29
11089	225	\N	Pouldreuzic	29
11090	226	\N	Poullan-sur-Mer	29
11091	227	\N	Poullaouen	29
11092	228	\N	Primelin	29
11093	229	\N	Quéménéven	29
11094	230	\N	Querrien	29
11095	232	\N	Quimper	29
11096	233	\N	Quimperlé	29
11097	234	\N	Rédené	29
11098	235	\N	Relecq-Kerhuon	29
11099	236	\N	Riec-sur-Belon	29
11100	237	\N	Roche-Maurice	29
11101	238	\N	Roscanvel	29
11102	239	\N	Roscoff	29
11103	240	\N	Rosnoën	29
11104	241	\N	Rosporden	29
11105	243	\N	Saint-Coulitz	29
11106	244	\N	Saint-Derrien	29
11107	245	\N	Saint-Divy	29
11108	246	\N	Saint-Eloy	29
11109	247	\N	Saint-Évarzec	29
11110	248	\N	Saint-Frégant	29
11111	249	\N	Saint-Goazec	29
11112	250	\N	Saint-Hernin	29
11113	251	\N	Saint-Jean-du-Doigt	29
11114	252	\N	Saint-Jean-Trolimon	29
11115	254	\N	Saint-Martin-des-Champs	29
11116	255	\N	Saint-Méen	29
11117	256	\N	Saint-Nic	29
11118	257	\N	Saint-Pabu	29
11119	259	\N	Saint-Pol-de-Léon	29
11120	260	\N	Saint-Renan	29
11121	261	\N	Saint-Rivoal	29
11122	262	\N	Saint-Sauveur	29
11123	263	\N	Saint-Ségal	29
11124	264	\N	Saint-Servais	29
11125	265	\N	Sainte-Sève	29
11126	266	\N	Saint-Thégonnec	29
11127	267	\N	Saint-Thois	29
11128	268	\N	Saint-Thonan	29
11129	269	\N	Saint-Thurien	29
11130	270	\N	Saint-Urbain	29
11131	271	\N	Saint-Vougay	29
11132	272	\N	Saint-Yvy	29
11133	273	\N	Santec	29
11134	274	\N	Scaër	29
11135	275	\N	Scrignac	29
11136	276	\N	Sibiril	29
11137	277	\N	Sizun	29
11138	278	\N	Spézet	29
11139	279	\N	Taulé	29
11140	280	\N	Telgruc-sur-Mer	29
11141	281	\N	Tourch	29
11142	282	\N	Trébabu	29
11143	284	\N	Treffiagat	29
11144	285	\N	Tréflaouénan	29
11145	286	\N	Tréflévénez	29
11146	287	\N	Tréflez	29
11147	288	\N	Trégarantec	29
11148	289	\N	Trégarvan	29
11149	290	\N	Tréglonou	29
11150	291	\N	Trégourez	29
11151	292	\N	Tréguennec	29
11152	293	\N	Trégunc	29
11153	294	\N	Tréhou	29
11154	295	\N	Trémaouézan	29
11155	296	\N	Tréméoc	29
11156	297	\N	Tréméven	29
11157	298	\N	Tréogat	29
11158	299	\N	Tréouergat	29
11159	300	\N	Trévoux	29
11160	301	\N	Trézilidé	29
11161	302	\N	Pont-de-Buis-lès-Quimerch	29
11162	001	\N	Afa	2A
11163	004	\N	Ajaccio	2A
11164	006	\N	Alata	2A
11165	008	\N	Albitreccia	2A
11166	011	\N	Altagène	2A
11167	014	\N	Ambiegna	2A
11168	017	\N	Appietto	2A
11169	018	\N	Arbellara	2A
11170	019	\N	Arbori	2A
11171	021	\N	Argiusta-Moriccio	2A
11172	022	\N	Arro	2A
11173	024	\N	Aullène	2A
11174	026	\N	Azilone-Ampaza	2A
11175	027	\N	Azzana	2A
11176	028	\N	Balogna	2A
11177	031	\N	Bastelica	2A
11178	032	\N	Bastelicaccia	2A
11179	035	\N	Belvédère-Campomoro	2A
11180	038	\N	Bilia	2A
11181	040	\N	Bocognano	2A
11182	041	\N	Bonifacio	2A
11183	048	\N	Calcatoggio	2A
11184	056	\N	Campo	2A
11185	060	\N	Cannelle	2A
11186	061	\N	Carbini	2A
11187	062	\N	Carbuccia	2A
11188	064	\N	Cardo-Torgia	2A
11189	065	\N	Cargèse	2A
11190	066	\N	Cargiaca	2A
11191	070	\N	Casaglione	2A
11192	071	\N	Casalabriva	2A
11193	085	\N	Cauro	2A
11194	089	\N	Ciamannacce	2A
11195	090	\N	Coggia	2A
11196	091	\N	Cognocoli-Monticchi	2A
11197	092	\N	Conca	2A
11198	094	\N	Corrano	2A
11199	098	\N	Coti-Chiavari	2A
11200	099	\N	Cozzano	2A
11201	100	\N	Cristinacce	2A
11202	103	\N	Cuttoli-Corticchiato	2A
11203	104	\N	Eccica-Suarella	2A
11204	108	\N	Évisa	2A
11205	114	\N	Figari	2A
11206	115	\N	Foce	2A
11207	117	\N	Forciolo	2A
11208	118	\N	Fozzano	2A
11209	119	\N	Frasseto	2A
11210	127	\N	Giuncheto	2A
11211	128	\N	Granace	2A
11212	129	\N	Grossa	2A
11213	130	\N	Grosseto-Prugna	2A
11214	131	\N	Guagno	2A
11215	132	\N	Guargualé	2A
11216	133	\N	Guitera-les-Bains	2A
11217	139	\N	Lecci	2A
11218	141	\N	Letia	2A
11219	142	\N	Levie	2A
11220	144	\N	Lopigna	2A
11221	146	\N	Loreto-di-Tallano	2A
11222	154	\N	Marignana	2A
11223	158	\N	Mela	2A
11224	160	\N	Moca-Croce	2A
11225	163	\N	Monacia-d'Aullène	2A
11226	174	\N	Murzo	2A
11227	181	\N	Ocana	2A
11228	186	\N	Olivese	2A
11229	189	\N	Olmeto	2A
11230	191	\N	Olmiccia	2A
11231	196	\N	Orto	2A
11232	197	\N	Osani	2A
11233	198	\N	Ota	2A
11234	200	\N	Palneca	2A
11235	203	\N	Partinello	2A
11236	204	\N	Pastricciola	2A
11237	209	\N	Peri	2A
11238	211	\N	Petreto-Bicchisano	2A
11239	212	\N	Piana	2A
11240	215	\N	Pianottoli-Caldarello	2A
11241	228	\N	Pietrosella	2A
11242	232	\N	Pila-Canale	2A
11243	240	\N	Poggiolo	2A
11244	247	\N	Porto-Vecchio	2A
11245	249	\N	Propriano	2A
11246	253	\N	Quasquara	2A
11247	254	\N	Quenza	2A
11248	258	\N	Renno	2A
11249	259	\N	Rezza	2A
11250	262	\N	Rosazia	2A
11251	266	\N	Salice	2A
11252	268	\N	Sampolo	2A
11253	269	\N	Sari-Solenzara	2A
11254	270	\N	Sari-d'Orcino	2A
11255	271	\N	Sarrola-Carcopino	2A
11256	272	\N	Sartène	2A
11257	276	\N	Serra-di-Ferro	2A
11258	278	\N	Serra-di-Scopamène	2A
11259	279	\N	Serriera	2A
11260	282	\N	Soccia	2A
11261	284	\N	Sollacaro	2A
11262	285	\N	Sorbollano	2A
11263	288	\N	Sotta	2A
11264	295	\N	Sant'Andréa-d'Orcino	2A
11265	300	\N	San-Gavino-di-Carbini	2A
11266	308	\N	Sainte-Lucie-de-Tallano	2A
11267	310	\N	Santa-Maria-Figaniella	2A
11268	312	\N	Santa-Maria-Siché	2A
11269	322	\N	Tasso	2A
11270	323	\N	Tavaco	2A
11271	324	\N	Tavera	2A
11272	326	\N	Tolla	2A
11273	330	\N	Ucciani	2A
11274	331	\N	Urbalacone	2A
11275	336	\N	Valle-di-Mezzana	2A
11276	345	\N	Vero	2A
11277	348	\N	Vico	2A
11278	349	\N	Viggianello	2A
11279	351	\N	Villanova	2A
11280	357	\N	Zérubia	2A
11281	358	\N	Zévaco	2A
11282	359	\N	Zicavo	2A
11283	360	\N	Zigliara	2A
11284	362	\N	Zonza	2A
11285	363	\N	Zoza	2A
11286	002	\N	Aghione	2B
11287	003	\N	Aiti	2B
11288	005	\N	Alando	2B
11289	007	\N	Albertacce	2B
11290	009	\N	Aléria	2B
11291	010	\N	Algajola	2B
11292	012	\N	Altiani	2B
11293	013	\N	Alzi	2B
11294	015	\N	Ampriani	2B
11295	016	\N	Antisanti	2B
11296	020	\N	Aregno	2B
11297	023	\N	Asco	2B
11298	025	\N	Avapessa	2B
11299	029	\N	Barbaggio	2B
11300	030	\N	Barrettali	2B
11301	033	\N	Bastia	2B
11302	034	\N	Belgodère	2B
11303	036	\N	Bigorno	2B
11304	037	\N	Biguglia	2B
11305	039	\N	Bisinchi	2B
11306	042	\N	Borgo	2B
11307	043	\N	Brando	2B
11308	045	\N	Bustanico	2B
11309	046	\N	Cagnano	2B
11310	047	\N	Calacuccia	2B
11311	049	\N	Calenzana	2B
11312	050	\N	Calvi	2B
11313	051	\N	Cambia	2B
11314	052	\N	Campana	2B
11315	053	\N	Campi	2B
11316	054	\N	Campile	2B
11317	055	\N	Campitello	2B
11318	057	\N	Canale-di-Verde	2B
11319	058	\N	Canari	2B
11320	059	\N	Canavaggia	2B
11321	063	\N	Carcheto-Brustico	2B
11322	067	\N	Carpineto	2B
11323	068	\N	Carticasi	2B
11324	069	\N	Casabianca	2B
11325	072	\N	Casalta	2B
11326	073	\N	Casamaccioli	2B
11327	074	\N	Casanova	2B
11328	075	\N	Casevecchie	2B
11329	077	\N	Castellare-di-Casinca	2B
11330	078	\N	Castellare-di-Mercurio	2B
11331	079	\N	Castello-di-Rostino	2B
11332	080	\N	Castifao	2B
11333	081	\N	Castiglione	2B
11334	082	\N	Castineta	2B
11335	083	\N	Castirla	2B
11336	084	\N	Cateri	2B
11337	086	\N	Centuri	2B
11338	087	\N	Cervione	2B
11339	088	\N	Chiatra	2B
11340	093	\N	Corbara	2B
11341	095	\N	Corscia	2B
11342	096	\N	Corte	2B
11343	097	\N	Costa	2B
11344	101	\N	Croce	2B
11345	102	\N	Crocicchia	2B
11346	105	\N	Erbajolo	2B
11347	106	\N	Érone	2B
11348	107	\N	Ersa	2B
11349	109	\N	Farinole	2B
11350	110	\N	Favalello	2B
11351	111	\N	Felce	2B
11352	112	\N	Feliceto	2B
11353	113	\N	Ficaja	2B
11354	116	\N	Focicchia	2B
11355	120	\N	Furiani	2B
11356	121	\N	Galéria	2B
11357	122	\N	Gavignano	2B
11358	123	\N	Ghisonaccia	2B
11359	124	\N	Ghisoni	2B
11360	125	\N	Giocatojo	2B
11361	126	\N	Giuncaggio	2B
11362	134	\N	Île-Rousse	2B
11363	135	\N	Isolaccio-di-Fiumorbo	2B
11364	136	\N	Lama	2B
11365	137	\N	Lano	2B
11366	138	\N	Lavatoggio	2B
11367	140	\N	Lento	2B
11368	143	\N	Linguizzetta	2B
11369	145	\N	Loreto-di-Casinca	2B
11370	147	\N	Lozzi	2B
11371	148	\N	Lucciana	2B
11372	149	\N	Lugo-di-Nazza	2B
11373	150	\N	Lumio	2B
11374	152	\N	Luri	2B
11375	153	\N	Manso	2B
11376	155	\N	Matra	2B
11377	156	\N	Mausoléo	2B
11378	157	\N	Mazzola	2B
11379	159	\N	Meria	2B
11380	161	\N	Moïta	2B
11381	162	\N	Moltifao	2B
11382	164	\N	Monacia-d'Orezza	2B
11383	165	\N	Moncale	2B
11384	166	\N	Monte	2B
11385	167	\N	Montegrosso	2B
11386	168	\N	Monticello	2B
11387	169	\N	Morosaglia	2B
11388	170	\N	Morsiglia	2B
11389	171	\N	Muracciole	2B
11390	172	\N	Murato	2B
11391	173	\N	Muro	2B
11392	175	\N	Nessa	2B
11393	176	\N	Nocario	2B
11394	177	\N	Noceta	2B
11395	178	\N	Nonza	2B
11396	179	\N	Novale	2B
11397	180	\N	Novella	2B
11398	182	\N	Occhiatana	2B
11399	183	\N	Ogliastro	2B
11400	184	\N	Olcani	2B
11401	185	\N	Oletta	2B
11402	187	\N	Olmeta-di-Capocorso	2B
11403	188	\N	Olmeta-di-Tuda	2B
11404	190	\N	Olmi-Cappella	2B
11405	192	\N	Olmo	2B
11406	193	\N	Omessa	2B
11407	194	\N	Ortale	2B
11408	195	\N	Ortiporio	2B
11409	199	\N	Palasca	2B
11410	201	\N	Pancheraccia	2B
11411	202	\N	Parata	2B
11412	205	\N	Patrimonio	2B
11413	206	\N	Penta-Acquatella	2B
11414	207	\N	Penta-di-Casinca	2B
11415	208	\N	Perelli	2B
11416	210	\N	Pero-Casevecchie	2B
11417	213	\N	Pianello	2B
11418	214	\N	Piano	2B
11419	216	\N	Piazzali	2B
11420	217	\N	Piazzole	2B
11421	218	\N	Piedicorte-di-Gaggio	2B
11422	219	\N	Piedicroce	2B
11423	220	\N	Piedigriggio	2B
11424	221	\N	Piedipartino	2B
11425	222	\N	Pie-d'Orezza	2B
11426	223	\N	Pietralba	2B
11427	224	\N	Pietracorbara	2B
11428	225	\N	Pietra-di-Verde	2B
11429	226	\N	Pietraserena	2B
11430	227	\N	Pietricaggio	2B
11431	229	\N	Pietroso	2B
11432	230	\N	Piève	2B
11433	231	\N	Pigna	2B
11434	233	\N	Pino	2B
11435	234	\N	Piobetta	2B
11436	235	\N	Pioggiola	2B
11437	236	\N	Poggio-di-Nazza	2B
11438	238	\N	Poggio-di-Venaco	2B
11439	239	\N	Poggio-d'Oletta	2B
11440	241	\N	Poggio-Marinaccio	2B
11441	242	\N	Poggio-Mezzana	2B
11442	243	\N	Polveroso	2B
11443	244	\N	Popolasca	2B
11444	245	\N	Porri	2B
11445	246	\N	Porta	2B
11446	248	\N	Prato-di-Giovellina	2B
11447	250	\N	Prunelli-di-Casacconi	2B
11448	251	\N	Prunelli-di-Fiumorbo	2B
11449	252	\N	Pruno	2B
11450	255	\N	Quercitello	2B
11451	256	\N	Rapaggio	2B
11452	257	\N	Rapale	2B
11453	260	\N	Riventosa	2B
11454	261	\N	Rogliano	2B
11455	263	\N	Rospigliani	2B
11456	264	\N	Rusio	2B
11457	265	\N	Rutali	2B
11458	267	\N	Saliceto	2B
11459	273	\N	Scata	2B
11460	274	\N	Scolca	2B
11461	275	\N	Sermano	2B
11462	277	\N	Serra-di-Fiumorbo	2B
11463	280	\N	Silvareccio	2B
11464	281	\N	Sisco	2B
11465	283	\N	Solaro	2B
11466	286	\N	Sorbo-Ocagnano	2B
11467	287	\N	Sorio	2B
11468	289	\N	Soveria	2B
11469	290	\N	Speloncato	2B
11470	291	\N	Stazzona	2B
11471	292	\N	Sant'Andréa-di-Bozio	2B
11472	293	\N	Sant'Andréa-di-Cotone	2B
11473	296	\N	Sant'Antonino	2B
11474	297	\N	San-Damiano	2B
11475	298	\N	Saint-Florent	2B
11476	299	\N	San-Gavino-d'Ampugnani	2B
11477	301	\N	San-Gavino-di-Tenda	2B
11478	302	\N	San-Giovanni-di-Moriani	2B
11479	303	\N	San-Giuliano	2B
11480	304	\N	San-Lorenzo	2B
11481	305	\N	San-Martino-di-Lota	2B
11482	306	\N	Santa-Lucia-di-Mercurio	2B
11483	307	\N	Santa-Lucia-di-Moriani	2B
11484	309	\N	Santa-Maria-di-Lota	2B
11485	311	\N	Santa-Maria-Poggio	2B
11486	313	\N	San-Nicolao	2B
11487	314	\N	Santo-Pietro-di-Tenda	2B
11488	315	\N	Santo-Pietro-di-Venaco	2B
11489	316	\N	Santa-Reparata-di-Balagna	2B
11490	317	\N	Santa-Reparata-di-Moriani	2B
11491	318	\N	Taglio-Isolaccio	2B
11492	319	\N	Talasani	2B
11493	320	\N	Tallone	2B
11494	321	\N	Tarrano	2B
11495	327	\N	Tomino	2B
11496	328	\N	Tox	2B
11497	329	\N	Tralonca	2B
11498	332	\N	Urtaca	2B
11499	333	\N	Vallecalle	2B
11500	334	\N	Valle-d'Alesani	2B
11501	335	\N	Valle-di-Campoloro	2B
11502	337	\N	Valle-di-Rostino	2B
11503	338	\N	Valle-d'Orezza	2B
11504	339	\N	Vallica	2B
11505	340	\N	Velone-Orneto	2B
11506	341	\N	Venaco	2B
11507	342	\N	Ventiseri	2B
11508	343	\N	Venzolasca	2B
11509	344	\N	Verdèse	2B
11510	346	\N	Vescovato	2B
11511	347	\N	Vezzani	2B
11512	350	\N	Vignale	2B
11513	352	\N	Ville-di-Paraso	2B
11514	353	\N	Ville-di-Pietrabugno	2B
11515	354	\N	Vivario	2B
11516	355	\N	Volpajola	2B
11517	356	\N	Zalana	2B
11518	361	\N	Zilia	2B
11519	364	\N	Zuani	2B
11520	365	\N	San-Gavino-di-Fiumorbo	2B
11521	366	\N	Chisa	2B
11522	001	\N	Aigaliers	30
11523	002	\N	Aigremont	30
11524	003	\N	Aigues-Mortes	30
11525	004	\N	Aigues-Vives	30
11526	005	\N	Aiguèze	30
11527	006	\N	Aimargues	30
11528	007	\N	Alès	30
11529	008	\N	Allègre-les-Fumades	30
11530	009	\N	Alzon	30
11531	010	\N	Anduze	30
11532	011	\N	Angles	30
11533	012	\N	Aramon	30
11534	013	\N	Argilliers	30
11535	014	\N	Arpaillargues-et-Aureillac	30
11536	015	\N	Arphy	30
11537	016	\N	Arre	30
11538	017	\N	Arrigas	30
11539	018	\N	Aspères	30
11540	019	\N	Aubais	30
11541	020	\N	Aubord	30
11542	021	\N	Aubussargues	30
11543	022	\N	Aujac	30
11544	023	\N	Aujargues	30
11545	024	\N	Aulas	30
11546	025	\N	Aumessas	30
11547	026	\N	Avèze	30
11548	027	\N	Bagard	30
11549	028	\N	Bagnols-sur-Cèze	30
11550	029	\N	Barjac	30
11551	030	\N	Baron	30
11552	031	\N	Bastide-d'Engras	30
11553	032	\N	Beaucaire	30
11554	033	\N	Beauvoisin	30
11555	034	\N	Bellegarde	30
11556	035	\N	Belvézet	30
11557	036	\N	Bernis	30
11558	037	\N	Bessèges	30
11559	038	\N	Bez-et-Esparon	30
11560	039	\N	Bezouce	30
11561	040	\N	Blandas	30
11562	041	\N	Blauzac	30
11563	042	\N	Boisset-et-Gaujac	30
11564	043	\N	Boissières	30
11565	044	\N	Bonnevaux	30
11566	045	\N	Bordezac	30
11567	046	\N	Boucoiran-et-Nozières	30
11568	047	\N	Bouillargues	30
11569	048	\N	Bouquet	30
11570	049	\N	Bourdic	30
11571	050	\N	Bragassargues	30
11572	051	\N	Branoux-les-Taillades	30
11573	052	\N	Bréau-et-Salagosse	30
11574	053	\N	Brignon	30
11575	054	\N	Brouzet-lès-Quissac	30
11576	055	\N	Brouzet-lès-Alès	30
11577	056	\N	Bruguière	30
11578	057	\N	Cabrières	30
11579	058	\N	Cadière-et-Cambo	30
11580	059	\N	Cailar	30
11581	060	\N	Caissargues	30
11582	061	\N	Calmette	30
11583	062	\N	Calvisson	30
11584	064	\N	Campestre-et-Luc	30
11585	065	\N	Canaules-et-Argentières	30
11586	066	\N	Cannes-et-Clairan	30
11587	067	\N	Capelle-et-Masmolène	30
11588	068	\N	Cardet	30
11589	069	\N	Carnas	30
11590	070	\N	Carsan	30
11591	071	\N	Cassagnoles	30
11592	072	\N	Castelnau-Valence	30
11593	073	\N	Castillon-du-Gard	30
11594	074	\N	Causse-Bégon	30
11595	075	\N	Caveirac	30
11596	076	\N	Cavillargues	30
11597	077	\N	Cendras	30
11598	079	\N	Chambon	30
11599	080	\N	Chamborigaud	30
11600	081	\N	Chusclan	30
11601	082	\N	Clarensac	30
11602	083	\N	Codognan	30
11603	084	\N	Codolet	30
11604	085	\N	Collias	30
11605	086	\N	Collorgues	30
11606	087	\N	Colognac	30
11607	088	\N	Combas	30
11608	089	\N	Comps	30
11609	090	\N	Concoules	30
11610	091	\N	Congénies	30
11611	092	\N	Connaux	30
11612	093	\N	Conqueyrac	30
11613	094	\N	Corbès	30
11614	095	\N	Corconne	30
11615	096	\N	Cornillon	30
11616	097	\N	Courry	30
11617	098	\N	Crespian	30
11618	099	\N	Cros	30
11619	100	\N	Cruviers-Lascours	30
11620	101	\N	Deaux	30
11621	102	\N	Dions	30
11622	103	\N	Domazan	30
11623	104	\N	Domessargues	30
11624	105	\N	Dourbies	30
11625	106	\N	Durfort-et-Saint-Martin-de-Sossenac	30
11626	107	\N	Estézargues	30
11627	108	\N	Estréchure	30
11628	109	\N	Euzet	30
11629	110	\N	Flaux	30
11630	111	\N	Foissac	30
11631	112	\N	Fons	30
11632	113	\N	Fons-sur-Lussan	30
11633	114	\N	Fontanès	30
11634	115	\N	Fontarèches	30
11635	116	\N	Fournès	30
11636	117	\N	Fourques	30
11637	119	\N	Fressac	30
11638	120	\N	Gagnières	30
11639	121	\N	Gailhan	30
11640	122	\N	Gajan	30
11641	123	\N	Gallargues-le-Montueux	30
11642	124	\N	Garn	30
11643	125	\N	Garons	30
11644	126	\N	Garrigues-Sainte-Eulalie	30
11645	127	\N	Gaujac	30
11646	128	\N	Générac	30
11647	129	\N	Générargues	30
11648	130	\N	Génolhac	30
11649	131	\N	Goudargues	30
11650	132	\N	Grand-Combe	30
11651	133	\N	Grau-du-Roi	30
11652	134	\N	Issirac	30
11653	135	\N	Jonquières-Saint-Vincent	30
11654	136	\N	Junas	30
11655	137	\N	Lamelouze	30
11656	138	\N	Langlade	30
11657	139	\N	Lanuéjols	30
11658	140	\N	Lasalle	30
11659	141	\N	Laudun-l'Ardoise	30
11660	142	\N	Laval-Pradel	30
11661	143	\N	Laval-Saint-Roman	30
11662	144	\N	Lecques	30
11663	145	\N	Lédenon	30
11664	146	\N	Lédignan	30
11665	147	\N	Lézan	30
11666	148	\N	Liouc	30
11667	149	\N	Lirac	30
11668	150	\N	Logrian-Florian	30
11669	151	\N	Lussan	30
11670	152	\N	Mages	30
11671	153	\N	Malons-et-Elze	30
11672	154	\N	Mandagout	30
11673	155	\N	Manduel	30
11674	156	\N	Marguerittes	30
11675	157	\N	Mars	30
11676	158	\N	Martignargues	30
11677	159	\N	Martinet	30
11678	160	\N	Maruéjols-lès-Gardon	30
11679	161	\N	Massanes	30
11680	162	\N	Massillargues-Attuech	30
11681	163	\N	Mauressargues	30
11682	164	\N	Méjannes-le-Clap	30
11683	165	\N	Méjannes-lès-Alès	30
11684	166	\N	Meynes	30
11685	167	\N	Meyrannes	30
11686	168	\N	Mialet	30
11687	169	\N	Milhaud	30
11688	170	\N	Molières-Cavaillac	30
11689	171	\N	Molières-sur-Cèze	30
11690	172	\N	Monoblet	30
11691	173	\N	Mons	30
11692	174	\N	Montaren-et-Saint-Médiers	30
11693	175	\N	Montclus	30
11694	176	\N	Montdardier	30
11695	177	\N	Monteils	30
11696	178	\N	Montfaucon	30
11697	179	\N	Montfrin	30
11698	180	\N	Montignargues	30
11699	181	\N	Montmirat	30
11700	182	\N	Montpezat	30
11701	183	\N	Moulézan	30
11702	184	\N	Moussac	30
11703	185	\N	Mus	30
11704	186	\N	Nages-et-Solorgues	30
11705	187	\N	Navacelles	30
11706	188	\N	Ners	30
11707	189	\N	Nîmes	30
11708	190	\N	Notre-Dame-de-la-Rouvière	30
11709	191	\N	Orsan	30
11710	192	\N	Orthoux-Sérignac-Quilhan	30
11711	193	\N	Parignargues	30
11712	194	\N	Peyremale	30
11713	195	\N	Peyroles	30
11714	196	\N	Pin	30
11715	197	\N	Plans	30
11716	198	\N	Plantiers	30
11717	199	\N	Pommiers	30
11718	200	\N	Pompignan	30
11719	201	\N	Ponteils-et-Brésis	30
11720	202	\N	Pont-Saint-Esprit	30
11721	203	\N	Portes	30
11722	204	\N	Potelières	30
11723	205	\N	Pougnadoresse	30
11724	206	\N	Poulx	30
11725	207	\N	Pouzilhac	30
11726	208	\N	Puechredon	30
11727	209	\N	Pujaut	30
11728	210	\N	Quissac	30
11729	211	\N	Redessan	30
11730	212	\N	Remoulins	30
11731	213	\N	Revens	30
11732	214	\N	Ribaute-les-Tavernes	30
11733	215	\N	Rivières	30
11734	216	\N	Robiac-Rochessadoule	30
11735	217	\N	Rochefort-du-Gard	30
11736	218	\N	Rochegude	30
11737	219	\N	Rogues	30
11738	220	\N	Roquedur	30
11739	221	\N	Roquemaure	30
11740	222	\N	Roque-sur-Cèze	30
11741	223	\N	Rousson	30
11742	224	\N	Rouvière	30
11743	225	\N	Sabran	30
11744	226	\N	Saint-Alexandre	30
11745	227	\N	Saint-Ambroix	30
11746	228	\N	Sainte-Anastasie	30
11747	229	\N	Saint-André-de-Majencoules	30
11748	230	\N	Saint-André-de-Roquepertuis	30
11749	231	\N	Saint-André-de-Valborgne	30
11750	232	\N	Saint-André-d'Olérargues	30
11751	233	\N	Saint-Bauzély	30
11752	234	\N	Saint-Bénézet	30
11753	235	\N	Saint-Bonnet-du-Gard	30
11754	236	\N	Saint-Bonnet-de-Salendrinque	30
11755	237	\N	Saint-Brès	30
11756	238	\N	Saint-Bresson	30
11757	239	\N	Sainte-Cécile-d'Andorge	30
11758	240	\N	Saint-Césaire-de-Gauzignan	30
11759	241	\N	Saint-Chaptes	30
11760	242	\N	Saint-Christol-de-Rodières	30
11761	243	\N	Saint-Christol-lès-Alès	30
11762	244	\N	Saint-Clément	30
11763	245	\N	Saint-Côme-et-Maruéjols	30
11764	246	\N	Sainte-Croix-de-Caderle	30
11765	247	\N	Saint-Denis	30
11766	248	\N	Saint-Dézéry	30
11767	249	\N	Saint-Dionizy	30
11768	250	\N	Saint-Étienne-de-l'Olm	30
11769	251	\N	Saint-Étienne-des-Sorts	30
11770	252	\N	Saint-Félix-de-Pallières	30
11771	253	\N	Saint-Florent-sur-Auzonnet	30
11772	254	\N	Saint-Geniès-de-Comolas	30
11773	255	\N	Saint-Geniès-de-Malgoirès	30
11774	256	\N	Saint-Gervais	30
11775	257	\N	Saint-Gervasy	30
11776	258	\N	Saint-Gilles	30
11777	259	\N	Saint-Hilaire-de-Brethmas	30
11778	260	\N	Saint-Hilaire-d'Ozilhan	30
11779	261	\N	Saint-Hippolyte-de-Caton	30
11780	262	\N	Saint-Hippolyte-de-Montaigu	30
11781	263	\N	Saint-Hippolyte-du-Fort	30
11782	264	\N	Saint-Jean-de-Ceyrargues	30
11783	265	\N	Saint-Jean-de-Crieulon	30
11784	266	\N	Saint-Jean-de-Maruéjols-et-Avéjan	30
11785	267	\N	Saint-Jean-de-Serres	30
11786	268	\N	Saint-Jean-de-Valériscle	30
11787	269	\N	Saint-Jean-du-Gard	30
11788	270	\N	Saint-Jean-du-Pin	30
11789	271	\N	Saint-Julien-de-Cassagnas	30
11790	272	\N	Saint-Julien-de-la-Nef	30
11791	273	\N	Saint-Julien-de-Peyrolas	30
11792	274	\N	Saint-Julien-les-Rosiers	30
11793	275	\N	Saint-Just-et-Vacquières	30
11794	276	\N	Saint-Laurent-d'Aigouze	30
11795	277	\N	Saint-Laurent-de-Carnols	30
11796	278	\N	Saint-Laurent-des-Arbres	30
11797	279	\N	Saint-Laurent-la-Vernède	30
11798	280	\N	Saint-Laurent-le-Minier	30
11799	281	\N	Saint-Mamert-du-Gard	30
11800	282	\N	Saint-Marcel-de-Careiret	30
11801	283	\N	Saint-Martial	30
11802	284	\N	Saint-Martin-de-Valgalgues	30
11803	285	\N	Saint-Maurice-de-Cazevieille	30
11804	286	\N	Saint-Maximin	30
11805	287	\N	Saint-Michel-d'Euzet	30
11806	288	\N	Saint-Nazaire	30
11807	289	\N	Saint-Nazaire-des-Gardies	30
11808	290	\N	Saint-Paulet-de-Caisson	30
11809	291	\N	Saint-Paul-la-Coste	30
11810	292	\N	Saint-Pons-la-Calm	30
11811	293	\N	Saint-Privat-de-Champclos	30
11812	294	\N	Saint-Privat-des-Vieux	30
11813	295	\N	Saint-Quentin-la-Poterie	30
11814	296	\N	Saint-Roman-de-Codières	30
11815	297	\N	Saint-Sauveur-Camprieu	30
11816	298	\N	Saint-Sébastien-d'Aigrefeuille	30
11817	299	\N	Saint-Siffret	30
11818	300	\N	Saint-Théodorit	30
11819	301	\N	Saint-Victor-des-Oules	30
11820	302	\N	Saint-Victor-la-Coste	30
11821	303	\N	Saint-Victor-de-Malcap	30
11822	304	\N	Salazac	30
11823	305	\N	Salindres	30
11824	306	\N	Salinelles	30
11825	307	\N	Salles-du-Gardon	30
11826	308	\N	Sanilhac-Sagriès	30
11827	309	\N	Sardan	30
11828	310	\N	Saumane	30
11829	311	\N	Sauve	30
11830	312	\N	Sauveterre	30
11831	313	\N	Sauzet	30
11832	314	\N	Savignargues	30
11833	315	\N	Saze	30
11834	316	\N	Sénéchas	30
11835	317	\N	Sernhac	30
11836	318	\N	Servas	30
11837	319	\N	Serviers-et-Labaume	30
11838	320	\N	Seynes	30
11839	321	\N	Sommières	30
11840	322	\N	Soudorgues	30
11841	323	\N	Soustelle	30
11842	324	\N	Souvignargues	30
11843	325	\N	Sumène	30
11844	326	\N	Tavel	30
11845	327	\N	Tharaux	30
11846	328	\N	Théziers	30
11847	329	\N	Thoiras	30
11848	330	\N	Tornac	30
11849	331	\N	Tresques	30
11850	332	\N	Trèves	30
11851	333	\N	Uchaud	30
11852	334	\N	Uzès	30
11853	335	\N	Vabres	30
11854	336	\N	Vallabrègues	30
11855	337	\N	Vallabrix	30
11856	338	\N	Vallérargues	30
11857	339	\N	Valleraugue	30
11858	340	\N	Valliguières	30
11859	341	\N	Vauvert	30
11860	342	\N	Vénéjan	30
11861	343	\N	Verfeuil	30
11862	344	\N	Vergèze	30
11863	345	\N	Vernarède	30
11864	346	\N	Vers-Pont-du-Gard	30
11865	347	\N	Vestric-et-Candiac	30
11866	348	\N	Vézénobres	30
11867	349	\N	Vic-le-Fesq	30
11868	350	\N	Vigan	30
11869	351	\N	Villeneuve-lès-Avignon	30
11870	352	\N	Villevieille	30
11871	353	\N	Vissec	30
11872	354	\N	Montagnac	30
11873	355	\N	Saint-Paul-les-Fonts	30
11874	356	\N	Rodilhan	30
11875	001	\N	Agassac	31
11876	002	\N	Aignes	31
11877	003	\N	Aigrefeuille	31
11878	004	\N	Ayguesvives	31
11879	005	\N	Alan	31
11880	006	\N	Albiac	31
11881	007	\N	Ambax	31
11882	008	\N	Anan	31
11883	009	\N	Antichan-de-Frontignes	31
11884	010	\N	Antignac	31
11885	011	\N	Arbas	31
11886	012	\N	Arbon	31
11887	013	\N	Ardiège	31
11888	014	\N	Arguenos	31
11889	015	\N	Argut-Dessous	31
11890	017	\N	Arlos	31
11891	018	\N	Arnaud-Guilhem	31
11892	019	\N	Artigue	31
11893	020	\N	Aspet	31
11894	021	\N	Aspret-Sarrat	31
11895	022	\N	Aucamville	31
11896	023	\N	Aulon	31
11897	024	\N	Auragne	31
11898	025	\N	Aureville	31
11899	026	\N	Auriac-sur-Vendinelle	31
11900	027	\N	Auribail	31
11901	028	\N	Aurignac	31
11902	029	\N	Aurin	31
11903	030	\N	Ausseing	31
11904	031	\N	Ausson	31
11905	032	\N	Aussonne	31
11906	033	\N	Auterive	31
11907	034	\N	Auzas	31
11908	035	\N	Auzeville-Tolosane	31
11909	036	\N	Auzielle	31
11910	037	\N	Avignonet-Lauragais	31
11911	038	\N	Azas	31
11912	039	\N	Bachas	31
11913	040	\N	Bachos	31
11914	041	\N	Bagiry	31
11915	042	\N	Bagnères-de-Luchon	31
11916	043	\N	Balesta	31
11917	044	\N	Balma	31
11918	045	\N	Barbazan	31
11919	046	\N	Baren	31
11920	047	\N	Bax	31
11921	048	\N	Baziège	31
11922	049	\N	Bazus	31
11923	050	\N	Beauchalot	31
11924	051	\N	Beaufort	31
11925	052	\N	Beaumont-sur-Lèze	31
11926	053	\N	Beaupuy	31
11927	054	\N	Beauteville	31
11928	055	\N	Beauville	31
11929	056	\N	Beauzelle	31
11930	057	\N	Belberaud	31
11931	058	\N	Belbèze-de-Lauragais	31
11932	059	\N	Belbèze-en-Comminges	31
11933	060	\N	Bélesta-en-Lauragais	31
11934	061	\N	Bellegarde-Sainte-Marie	31
11935	062	\N	Bellesserre	31
11936	063	\N	Benque	31
11937	064	\N	Benque-Dessous-et-Dessus	31
11938	065	\N	Bérat	31
11939	066	\N	Bessières	31
11940	067	\N	Bezins-Garraux	31
11941	068	\N	Billière	31
11942	069	\N	Blagnac	31
11943	070	\N	Blajan	31
11944	071	\N	Bois-de-la-Pierre	31
11945	072	\N	Boissède	31
11946	073	\N	Bondigoux	31
11947	074	\N	Bonrepos-Riquet	31
11948	075	\N	Bonrepos-sur-Aussonnelle	31
11949	076	\N	Bordes-de-Rivière	31
11950	077	\N	Born	31
11951	078	\N	Boudrac	31
11952	079	\N	Bouloc	31
11953	080	\N	Boulogne-sur-Gesse	31
11954	081	\N	Bourg-d'Oueil	31
11955	082	\N	Bourg-Saint-Bernard	31
11956	083	\N	Boussan	31
11957	084	\N	Boussens	31
11958	085	\N	Boutx	31
11959	086	\N	Bouzin	31
11960	087	\N	Bragayrac	31
11961	088	\N	Brax	31
11962	089	\N	Bretx	31
11963	090	\N	Brignemont	31
11964	091	\N	Bruguières	31
11965	092	\N	Burgalays	31
11966	093	\N	Burgaud	31
11967	094	\N	Buzet-sur-Tarn	31
11968	095	\N	Cabanac-Cazaux	31
11969	096	\N	Cabanac-Séguenville	31
11970	097	\N	Cabanial	31
11971	098	\N	Cadours	31
11972	099	\N	Caignac	31
11973	100	\N	Calmont	31
11974	101	\N	Cambernard	31
11975	102	\N	Cambiac	31
11976	103	\N	Canens	31
11977	104	\N	Capens	31
11978	105	\N	Caragoudes	31
11979	106	\N	Caraman	31
11980	107	\N	Carbonne	31
11981	108	\N	Cardeilhac	31
11982	109	\N	Cassagnabère-Tournas	31
11983	110	\N	Cassagne	31
11984	111	\N	Castagnac	31
11985	112	\N	Castagnède	31
11986	113	\N	Castanet-Tolosan	31
11987	114	\N	Castelbiague	31
11988	115	\N	Castelgaillard	31
11989	116	\N	Castelginest	31
11990	117	\N	Castelmaurou	31
11991	118	\N	Castelnau-d'Estrétefonds	31
11992	119	\N	Castelnau-Picampeau	31
11993	120	\N	Castéra	31
11994	121	\N	Castéra-Vignoles	31
11995	122	\N	Casties-Labrande	31
11996	123	\N	Castillon-de-Larboust	31
11997	124	\N	Castillon-de-Saint-Martory	31
11998	125	\N	Cathervielle	31
11999	126	\N	Caubiac	31
12000	127	\N	Caubous	31
12001	128	\N	Caujac	31
12002	129	\N	Cazaril-Laspènes	31
12003	130	\N	Cazaril-Tambourès	31
12004	131	\N	Cazaunous	31
12005	132	\N	Cazaux-Layrisse	31
12006	133	\N	Cazeaux-de-Larboust	31
12007	134	\N	Cazeneuve-Montaut	31
12008	135	\N	Cazères	31
12009	136	\N	Cépet	31
12010	137	\N	Cessales	31
12011	138	\N	Charlas	31
12012	139	\N	Chaum	31
12013	140	\N	Chein-Dessus	31
12014	141	\N	Ciadoux	31
12015	142	\N	Cier-de-Luchon	31
12016	143	\N	Cier-de-Rivière	31
12017	144	\N	Cierp-Gaud	31
12018	145	\N	Cintegabelle	31
12019	146	\N	Cirès	31
12020	147	\N	Clarac	31
12021	148	\N	Clermont-le-Fort	31
12022	149	\N	Colomiers	31
12023	150	\N	Cornebarrieu	31
12024	151	\N	Corronsac	31
12025	152	\N	Coueilles	31
12026	153	\N	Couladère	31
12027	155	\N	Couret	31
12028	156	\N	Cox	31
12029	157	\N	Cugnaux	31
12030	158	\N	Cuguron	31
12031	159	\N	Cuing	31
12032	160	\N	Daux	31
12033	161	\N	Deyme	31
12034	162	\N	Donneville	31
12035	163	\N	Drémil-Lafage	31
12036	164	\N	Drudas	31
12037	165	\N	Eaunes	31
12038	166	\N	Empeaux	31
12039	167	\N	Encausse-les-Thermes	31
12040	168	\N	Eoux	31
12041	169	\N	Escalquens	31
12042	170	\N	Escanecrabe	31
12043	171	\N	Espanès	31
12044	172	\N	Esparron	31
12045	173	\N	Esperce	31
12046	174	\N	Estadens	31
12047	175	\N	Estancarbon	31
12048	176	\N	Esténos	31
12049	177	\N	Eup	31
12050	178	\N	Fabas	31
12051	179	\N	Faget	31
12052	180	\N	Falga	31
12053	181	\N	Fauga	31
12054	182	\N	Fenouillet	31
12055	183	\N	Figarol	31
12056	184	\N	Flourens	31
12057	185	\N	Folcarde	31
12058	186	\N	Fonbeauzard	31
12059	187	\N	Fonsorbes	31
12060	188	\N	Fontenilles	31
12061	189	\N	Forgues	31
12062	190	\N	Fos	31
12063	191	\N	Fougaron	31
12064	192	\N	Fourquevaux	31
12065	193	\N	Fousseret	31
12066	194	\N	Francarville	31
12067	195	\N	Francazal	31
12068	196	\N	Francon	31
12069	197	\N	Franquevielle	31
12070	198	\N	Fréchet	31
12071	199	\N	Fronsac	31
12072	200	\N	Frontignan-de-Comminges	31
12073	201	\N	Frontignan-Savès	31
12074	202	\N	Fronton	31
12075	203	\N	Frouzins	31
12076	204	\N	Fustignac	31
12077	205	\N	Gagnac-sur-Garonne	31
12078	206	\N	Gaillac-Toulza	31
12079	207	\N	Galié	31
12080	208	\N	Ganties	31
12081	209	\N	Garac	31
12082	210	\N	Gardouch	31
12083	211	\N	Gargas	31
12084	212	\N	Garidech	31
12085	213	\N	Garin	31
12086	215	\N	Gauré	31
12087	216	\N	Gémil	31
12088	217	\N	Génos	31
12089	218	\N	Gensac-de-Boulogne	31
12090	219	\N	Gensac-sur-Garonne	31
12091	220	\N	Gibel	31
12092	221	\N	Gouaux-de-Larboust	31
12093	222	\N	Gouaux-de-Luchon	31
12094	223	\N	Goudex	31
12095	224	\N	Gourdan-Polignan	31
12096	225	\N	Goutevernisse	31
12097	226	\N	Gouzens	31
12098	227	\N	Goyrans	31
12099	228	\N	Gragnague	31
12100	229	\N	Gratens	31
12101	230	\N	Gratentour	31
12102	231	\N	Grazac	31
12103	232	\N	Grenade	31
12104	233	\N	Grépiac	31
12105	234	\N	Grès	31
12106	235	\N	Guran	31
12107	236	\N	Herran	31
12108	237	\N	His	31
12109	238	\N	Huos	31
12110	239	\N	Isle-en-Dodon	31
12111	240	\N	Issus	31
12112	241	\N	Izaut-de-l'Hôtel	31
12113	242	\N	Jurvielle	31
12114	243	\N	Juzes	31
12115	244	\N	Juzet-de-Luchon	31
12116	245	\N	Juzet-d'Izaut	31
12117	246	\N	Labarthe-Inard	31
12118	247	\N	Labarthe-Rivière	31
12119	248	\N	Labarthe-sur-Lèze	31
12120	249	\N	Labastide-Beauvoir	31
12121	250	\N	Labastide-Clermont	31
12122	251	\N	Labastide-Paumès	31
12123	252	\N	Labastide-Saint-Sernin	31
12124	253	\N	Labastidette	31
12125	254	\N	Labège	31
12126	255	\N	Labroquère	31
12127	256	\N	Labruyère-Dorsa	31
12128	258	\N	Lacaugne	31
12129	259	\N	Lacroix-Falgarde	31
12130	260	\N	Laffite-Toupière	31
12131	261	\N	Lafitte-Vigordane	31
12132	262	\N	Lagarde	31
12133	263	\N	Lagardelle-sur-Lèze	31
12134	264	\N	Lagrâce-Dieu	31
12135	265	\N	Lagraulet-Saint-Nicolas	31
12136	266	\N	Lahage	31
12137	267	\N	Lahitère	31
12138	268	\N	Lalouret-Laffiteau	31
12139	269	\N	Lamasquère	31
12140	270	\N	Landorthe	31
12141	271	\N	Lanta	31
12142	272	\N	Lapeyrère	31
12143	273	\N	Lapeyrouse-Fossat	31
12144	274	\N	Larcan	31
12145	275	\N	Laréole	31
12146	276	\N	Larroque	31
12147	277	\N	Lasserre	31
12148	278	\N	Latoue	31
12149	279	\N	Latour	31
12150	280	\N	Latrape	31
12151	281	\N	Launac	31
12152	282	\N	Launaguet	31
12153	283	\N	Lautignac	31
12154	284	\N	Lauzerville	31
12155	285	\N	Lavalette	31
12156	286	\N	Lavelanet-de-Comminges	31
12157	287	\N	Lavernose-Lacasse	31
12158	288	\N	Layrac-sur-Tarn	31
12159	289	\N	Lécussan	31
12160	290	\N	Lège	31
12161	291	\N	Léguevin	31
12162	292	\N	Lescuns	31
12163	293	\N	Lespinasse	31
12164	294	\N	Lespiteau	31
12165	295	\N	Lespugue	31
12166	296	\N	Lestelle-de-Saint-Martory	31
12167	297	\N	Lévignac	31
12168	298	\N	Lez	31
12169	299	\N	Lherm	31
12170	301	\N	Lilhac	31
12171	302	\N	Lodes	31
12172	303	\N	Longages	31
12173	304	\N	Loubens-Lauragais	31
12174	305	\N	Loudet	31
12175	306	\N	Lourde	31
12176	307	\N	Lunax	31
12177	308	\N	Luscan	31
12178	309	\N	Lussan-Adeilhac	31
12179	310	\N	Lux	31
12180	311	\N	Magdelaine-sur-Tarn	31
12181	312	\N	Mailholas	31
12182	313	\N	Malvezie	31
12183	314	\N	Mancioux	31
12184	315	\N	Mane	31
12185	316	\N	Marignac	31
12186	317	\N	Marignac-Lasclares	31
12187	318	\N	Marignac-Laspeyres	31
12188	319	\N	Marliac	31
12189	320	\N	Marquefave	31
12190	321	\N	Marsoulas	31
12191	322	\N	Martisserre	31
12192	323	\N	Martres-de-Rivière	31
12193	324	\N	Martres-Tolosane	31
12194	325	\N	Mascarville	31
12195	326	\N	Massabrac	31
12196	327	\N	Mauran	31
12197	328	\N	Mauremont	31
12198	329	\N	Maurens	31
12199	330	\N	Mauressac	31
12200	331	\N	Maureville	31
12201	332	\N	Mauvaisin	31
12202	333	\N	Mauvezin	31
12203	334	\N	Mauzac	31
12204	335	\N	Mayrègne	31
12205	336	\N	Mazères-sur-Salat	31
12206	337	\N	Melles	31
12207	338	\N	Menville	31
12208	339	\N	Mérenvielle	31
12209	340	\N	Mervilla	31
12210	341	\N	Merville	31
12211	342	\N	Milhas	31
12212	343	\N	Mirambeau	31
12213	344	\N	Miramont-de-Comminges	31
12214	345	\N	Miremont	31
12215	346	\N	Mirepoix-sur-Tarn	31
12216	347	\N	Molas	31
12217	348	\N	Moncaup	31
12218	349	\N	Mondavezan	31
12219	350	\N	Mondilhan	31
12220	351	\N	Mondonville	31
12221	352	\N	Mondouzil	31
12222	353	\N	Monès	31
12223	354	\N	Monestrol	31
12224	355	\N	Mons	31
12225	356	\N	Montaigut-sur-Save	31
12226	357	\N	Montastruc-de-Salies	31
12227	358	\N	Montastruc-la-Conseillère	31
12228	359	\N	Montastruc-Savès	31
12229	360	\N	Montauban-de-Luchon	31
12230	361	\N	Montaut	31
12231	362	\N	Montberaud	31
12232	363	\N	Montbernard	31
12233	364	\N	Montberon	31
12234	365	\N	Montbrun-Bocage	31
12235	366	\N	Montbrun-Lauragais	31
12236	367	\N	Montclar-de-Comminges	31
12237	368	\N	Montclar-Lauragais	31
12238	369	\N	Mont-de-Galié	31
12239	370	\N	Montégut-Bourjac	31
12240	371	\N	Montégut-Lauragais	31
12241	372	\N	Montespan	31
12242	373	\N	Montesquieu-Guittaut	31
12243	374	\N	Montesquieu-Lauragais	31
12244	375	\N	Montesquieu-Volvestre	31
12245	376	\N	Montgaillard-de-Salies	31
12246	377	\N	Montgaillard-Lauragais	31
12247	378	\N	Montgaillard-sur-Save	31
12248	379	\N	Montgazin	31
12249	380	\N	Montgeard	31
12250	381	\N	Montgiscard	31
12251	382	\N	Montgras	31
12252	383	\N	Montjoire	31
12253	384	\N	Montlaur	31
12254	385	\N	Montmaurin	31
12255	386	\N	Montoulieu-Saint-Bernard	31
12256	387	\N	Montoussin	31
12257	388	\N	Montpitol	31
12258	389	\N	Montrabé	31
12259	390	\N	Montréjeau	31
12260	391	\N	Montsaunès	31
12261	392	\N	Mourvilles-Basses	31
12262	393	\N	Mourvilles-Hautes	31
12263	394	\N	Moustajon	31
12264	395	\N	Muret	31
12265	396	\N	Nailloux	31
12266	397	\N	Nénigan	31
12267	398	\N	Nizan-Gesse	31
12268	399	\N	Noé	31
12269	400	\N	Nogaret	31
12270	401	\N	Noueilles	31
12271	402	\N	Odars	31
12272	403	\N	Ondes	31
12273	404	\N	Oô	31
12274	405	\N	Ore	31
12275	406	\N	Palaminy	31
12276	407	\N	Paulhac	31
12277	408	\N	Payssous	31
12278	409	\N	Péchabou	31
12279	410	\N	Pechbonnieu	31
12280	411	\N	Pechbusque	31
12281	412	\N	Péguilhan	31
12282	413	\N	Pelleport	31
12283	414	\N	Peyrissas	31
12284	415	\N	Peyrouzet	31
12285	416	\N	Peyssies	31
12286	417	\N	Pibrac	31
12287	418	\N	Pin-Balma	31
12288	419	\N	Pin-Murelet	31
12289	420	\N	Pinsaguel	31
12290	421	\N	Pins-Justaret	31
12291	422	\N	Plagne	31
12292	423	\N	Plagnole	31
12293	424	\N	Plaisance-du-Touch	31
12294	425	\N	Plan	31
12295	426	\N	Pointis-de-Rivière	31
12296	427	\N	Pointis-Inard	31
12297	428	\N	Polastron	31
12298	429	\N	Pompertuzat	31
12299	430	\N	Ponlat-Taillebourg	31
12300	431	\N	Portet-d'Aspet	31
12301	432	\N	Portet-de-Luchon	31
12302	433	\N	Portet-sur-Garonne	31
12303	434	\N	Poubeau	31
12304	435	\N	Poucharramet	31
12305	436	\N	Pouy-de-Touges	31
12306	437	\N	Pouze	31
12307	438	\N	Pradère-les-Bourguets	31
12308	439	\N	Préserville	31
12309	440	\N	Proupiary	31
12310	441	\N	Prunet	31
12311	442	\N	Puydaniel	31
12312	443	\N	Puymaurin	31
12313	444	\N	Puysségur	31
12314	445	\N	Quint-Fonsegrives	31
12315	446	\N	Ramonville-Saint-Agne	31
12316	447	\N	Razecueillé	31
12317	448	\N	Rebigue	31
12318	449	\N	Régades	31
12319	450	\N	Renneville	31
12320	451	\N	Revel	31
12321	452	\N	Rieucazé	31
12322	453	\N	Rieumajou	31
12323	454	\N	Rieumes	31
12324	455	\N	Rieux	31
12325	456	\N	Riolas	31
12326	457	\N	Roquefort-sur-Garonne	31
12327	458	\N	Roques	31
12328	459	\N	Roquesérière	31
12329	460	\N	Roquettes	31
12330	461	\N	Rouède	31
12331	462	\N	Rouffiac-Tolosan	31
12332	463	\N	Roumens	31
12333	464	\N	Sabonnères	31
12334	465	\N	Saccourvielle	31
12335	466	\N	Saiguède	31
12336	467	\N	Saint-Alban	31
12337	468	\N	Saint-André	31
12338	469	\N	Saint-Araille	31
12339	470	\N	Saint-Aventin	31
12340	471	\N	Saint-Béat	31
12341	472	\N	Saint-Bertrand-de-Comminges	31
12342	473	\N	Saint-Cézert	31
12343	474	\N	Saint-Christaud	31
12344	475	\N	Saint-Clar-de-Rivière	31
12345	476	\N	Saint-Élix-le-Château	31
12346	477	\N	Saint-Élix-Séglan	31
12347	478	\N	Saint-Félix-Lauragais	31
12348	479	\N	Saint-Ferréol-de-Comminges	31
12349	480	\N	Sainte-Foy-d'Aigrefeuille	31
12350	481	\N	Sainte-Foy-de-Peyrolières	31
12351	482	\N	Saint-Frajou	31
12352	483	\N	Saint-Gaudens	31
12353	484	\N	Saint-Geniès-Bellevue	31
12354	485	\N	Saint-Germier	31
12355	486	\N	Saint-Hilaire	31
12356	487	\N	Saint-Ignan	31
12357	488	\N	Saint-Jean	31
12358	489	\N	Saint-Jean-Lherm	31
12359	490	\N	Saint-Jory	31
12360	491	\N	Saint-Julia	31
12361	492	\N	Saint-Julien	31
12362	493	\N	Saint-Lary-Boujean	31
12363	494	\N	Saint-Laurent	31
12364	495	\N	Saint-Léon	31
12365	496	\N	Sainte-Livrade	31
12366	497	\N	Saint-Loup-Cammas	31
12367	498	\N	Saint-Loup-en-Comminges	31
12368	499	\N	Saint-Lys	31
12369	500	\N	Saint-Mamet	31
12370	501	\N	Saint-Marcel-Paulel	31
12371	502	\N	Saint-Marcet	31
12372	503	\N	Saint-Martory	31
12373	504	\N	Saint-Médard	31
12374	505	\N	Saint-Michel	31
12375	506	\N	Saint-Orens-de-Gameville	31
12376	507	\N	Saint-Paul-sur-Save	31
12377	508	\N	Saint-Paul-d'Oueil	31
12378	509	\N	Saint-Pé-d'Ardet	31
12379	510	\N	Saint-Pé-Delbosc	31
12380	511	\N	Saint-Pierre	31
12381	512	\N	Saint-Pierre-de-Lages	31
12382	513	\N	Saint-Plancard	31
12383	514	\N	Saint-Rome	31
12384	515	\N	Saint-Rustice	31
12385	516	\N	Saint-Sauveur	31
12386	517	\N	Saint-Sulpice-sur-Lèze	31
12387	518	\N	Saint-Thomas	31
12388	519	\N	Saint-Vincent	31
12389	520	\N	Sajas	31
12390	521	\N	Saleich	31
12391	522	\N	Salerm	31
12392	523	\N	Salies-du-Salat	31
12393	524	\N	Salles-et-Pratviel	31
12394	525	\N	Salles-sur-Garonne	31
12395	526	\N	Salvetat-Saint-Gilles	31
12396	527	\N	Salvetat-Lauragais	31
12397	528	\N	Saman	31
12398	529	\N	Samouillan	31
12399	530	\N	Sana	31
12400	531	\N	Sarrecave	31
12401	532	\N	Sarremezan	31
12402	533	\N	Saubens	31
12403	534	\N	Saussens	31
12404	535	\N	Sauveterre-de-Comminges	31
12405	536	\N	Saux-et-Pomarède	31
12406	537	\N	Savarthès	31
12407	538	\N	Savères	31
12408	539	\N	Sédeilhac	31
12409	540	\N	Ségreville	31
12410	541	\N	Seilh	31
12411	542	\N	Seilhan	31
12412	543	\N	Sénarens	31
12413	544	\N	Sengouagnet	31
12414	545	\N	Sepx	31
12415	546	\N	Seyre	31
12416	547	\N	Seysses	31
12417	548	\N	Signac	31
12418	549	\N	Sode	31
12419	550	\N	Soueich	31
12420	551	\N	Tarabel	31
12421	552	\N	Terrebasse	31
12422	553	\N	Thil	31
12423	554	\N	Touille	31
12424	555	\N	Toulouse	31
12425	556	\N	Tourreilles	31
12426	557	\N	Tournefeuille	31
12427	558	\N	Toutens	31
12428	559	\N	Trébons-de-Luchon	31
12429	560	\N	Trébons-sur-la-Grasse	31
12430	561	\N	Union	31
12431	562	\N	Urau	31
12432	563	\N	Vacquiers	31
12433	564	\N	Valcabrère	31
12434	565	\N	Valentine	31
12435	566	\N	Vallègue	31
12436	567	\N	Vallesvilles	31
12437	568	\N	Varennes	31
12438	569	\N	Vaudreuille	31
12439	570	\N	Vaux	31
12440	571	\N	Vendine	31
12441	572	\N	Venerque	31
12442	573	\N	Verfeil	31
12443	574	\N	Vernet	31
12444	575	\N	Vieille-Toulouse	31
12445	576	\N	Vieillevigne	31
12446	577	\N	Vignaux	31
12447	578	\N	Vigoulet-Auzil	31
12448	579	\N	Villariès	31
12449	580	\N	Villate	31
12450	581	\N	Villaudric	31
12451	582	\N	Villefranche-de-Lauragais	31
12452	583	\N	Villematier	31
12453	584	\N	Villemur-sur-Tarn	31
12454	585	\N	Villeneuve-de-Rivière	31
12455	586	\N	Villeneuve-Lécussan	31
12456	587	\N	Villeneuve-lès-Bouloc	31
12457	588	\N	Villeneuve-Tolosane	31
12458	589	\N	Villenouvelle	31
12459	590	\N	Binos	31
12460	591	\N	Escoulis	31
12461	592	\N	Larra	31
12462	593	\N	Cazac	31
12463	001	\N	Aignan	32
12464	002	\N	Ansan	32
12465	003	\N	Antras	32
12466	004	\N	Arblade-le-Bas	32
12467	005	\N	Arblade-le-Haut	32
12468	007	\N	Ardizas	32
12469	008	\N	Armentieux	32
12470	009	\N	Armous-et-Cau	32
12471	010	\N	Arrouède	32
12472	012	\N	Aubiet	32
12473	013	\N	Auch	32
12474	014	\N	Augnax	32
12475	015	\N	Aujan-Mournède	32
12476	016	\N	Auradé	32
12477	017	\N	Aurensan	32
12478	018	\N	Aurimont	32
12479	019	\N	Auterive	32
12480	020	\N	Aux-Aussat	32
12481	021	\N	Avensac	32
12482	022	\N	Avéron-Bergelle	32
12483	023	\N	Avezan	32
12484	024	\N	Ayguetinte	32
12485	025	\N	Ayzieu	32
12486	026	\N	Bajonnette	32
12487	027	\N	Barcelonne-du-Gers	32
12488	028	\N	Barcugnan	32
12489	029	\N	Barran	32
12490	030	\N	Bars	32
12491	031	\N	Bascous	32
12492	032	\N	Bassoues	32
12493	033	\N	Bazian	32
12494	034	\N	Bazugues	32
12495	035	\N	Beaucaire	32
12496	036	\N	Beaumarchés	32
12497	037	\N	Beaumont	32
12498	038	\N	Beaupuy	32
12499	039	\N	Beccas	32
12500	040	\N	Bédéchan	32
12501	041	\N	Bellegarde	32
12502	042	\N	Belloc-Saint-Clamens	32
12503	043	\N	Belmont	32
12504	044	\N	Béraut	32
12505	045	\N	Berdoues	32
12506	046	\N	Bernède	32
12507	047	\N	Berrac	32
12508	048	\N	Betcave-Aguin	32
12509	049	\N	Bétous	32
12510	050	\N	Betplan	32
12511	051	\N	Bézéril	32
12512	052	\N	Bezolles	32
12513	053	\N	Bézues-Bajon	32
12514	054	\N	Biran	32
12515	055	\N	Bivès	32
12516	056	\N	Blanquefort	32
12517	057	\N	Blaziert	32
12518	058	\N	Blousson-Sérian	32
12519	059	\N	Bonas	32
12520	060	\N	Boucagnères	32
12521	061	\N	Boulaur	32
12522	062	\N	Bourrouillan	32
12523	063	\N	Bouzon-Gellenave	32
12524	064	\N	Bretagne-d'Armagnac	32
12525	065	\N	Brouilh-Monbert	32
12526	066	\N	Brugnens	32
12527	067	\N	Cabas-Loumassès	32
12528	068	\N	Cadeilhan	32
12529	069	\N	Cadeillan	32
12530	070	\N	Cahuzac-sur-Adour	32
12531	071	\N	Caillavet	32
12532	072	\N	Callian	32
12533	073	\N	Campagne-d'Armagnac	32
12534	074	\N	Cannet	32
12535	075	\N	Cassaigne	32
12536	076	\N	Castelnau-Barbarens	32
12537	077	\N	Castelnau-d'Anglès	32
12538	078	\N	Castelnau-d'Arbieu	32
12539	079	\N	Castelnau-d'Auzan	32
12540	080	\N	Castelnau-sur-l'Auvignon	32
12541	081	\N	Castelnavet	32
12542	082	\N	Castéra-Lectourois	32
12543	083	\N	Castéra-Verduzan	32
12544	084	\N	Castéron	32
12545	085	\N	Castet-Arrouy	32
12546	086	\N	Castex	32
12547	087	\N	Castex-d'Armagnac	32
12548	088	\N	Castillon-Debats	32
12549	089	\N	Castillon-Massas	32
12550	090	\N	Castillon-Savès	32
12551	091	\N	Castin	32
12552	092	\N	Catonvielle	32
12553	093	\N	Caumont	32
12554	094	\N	Caupenne-d'Armagnac	32
12555	095	\N	Caussens	32
12556	096	\N	Cazaubon	32
12557	097	\N	Cazaux-d'Anglès	32
12558	098	\N	Cazaux-Savès	32
12559	099	\N	Cazaux-Villecomtal	32
12560	100	\N	Cazeneuve	32
12561	101	\N	Céran	32
12562	102	\N	Cézan	32
12563	103	\N	Chélan	32
12564	104	\N	Clermont-Pouyguillès	32
12565	105	\N	Clermont-Savès	32
12566	106	\N	Cologne	32
12567	107	\N	Condom	32
12568	108	\N	Corneillan	32
12569	109	\N	Couloumé-Mondebat	32
12570	110	\N	Courrensan	32
12571	111	\N	Courties	32
12572	112	\N	Crastes	32
12573	113	\N	Cravencères	32
12574	114	\N	Cuélas	32
12575	115	\N	Dému	32
12576	116	\N	Duffort	32
12577	117	\N	Duran	32
12578	118	\N	Durban	32
12579	119	\N	Eauze	32
12580	120	\N	Encausse	32
12581	121	\N	Endoufielle	32
12582	122	\N	Esclassan-Labastide	32
12583	123	\N	Escornebuf	32
12584	124	\N	Espaon	32
12585	125	\N	Espas	32
12586	126	\N	Estampes	32
12587	127	\N	Estang	32
12588	128	\N	Estipouy	32
12589	129	\N	Estramiac	32
12590	130	\N	Faget-Abbatial	32
12591	131	\N	Flamarens	32
12592	132	\N	Fleurance	32
12593	133	\N	Fourcès	32
12594	134	\N	Frégouville	32
12595	135	\N	Fustérouau	32
12596	136	\N	Galiax	32
12597	138	\N	Garravet	32
12598	139	\N	Gaudonville	32
12599	140	\N	Gaujac	32
12600	141	\N	Gaujan	32
12601	142	\N	Gavarret-sur-Aulouste	32
12602	143	\N	Gazaupouy	32
12603	144	\N	Gazax-et-Baccarisse	32
12604	145	\N	Gée-Rivière	32
12605	146	\N	Gimbrède	32
12606	147	\N	Gimont	32
12607	148	\N	Giscaro	32
12608	149	\N	Gondrin	32
12609	150	\N	Goutz	32
12610	151	\N	Goux	32
12611	152	\N	Haget	32
12612	153	\N	Haulies	32
12613	154	\N	Homps	32
12614	155	\N	Houga	32
12615	156	\N	Idrac-Respaillès	32
12616	157	\N	Isle-Arné	32
12617	158	\N	Isle-Bouzon	32
12618	159	\N	Isle-de-Noé	32
12619	160	\N	Isle-Jourdain	32
12620	161	\N	Izotges	32
12621	162	\N	Jegun	32
12622	163	\N	Jû-Belloc	32
12623	164	\N	Juillac	32
12624	165	\N	Juilles	32
12625	166	\N	Justian	32
12626	167	\N	Laas	32
12627	168	\N	Labarrère	32
12628	169	\N	Labarthe	32
12629	170	\N	Labarthète	32
12630	171	\N	Labastide-Savès	32
12631	172	\N	Labéjan	32
12632	173	\N	Labrihe	32
12633	174	\N	Ladevèze-Rivière	32
12634	175	\N	Ladevèze-Ville	32
12635	176	\N	Lagarde	32
12636	177	\N	Lagarde-Hachan	32
12637	178	\N	Lagardère	32
12638	180	\N	Lagraulet-du-Gers	32
12639	181	\N	Laguian-Mazous	32
12640	182	\N	Lahas	32
12641	183	\N	Lahitte	32
12642	184	\N	Lalanne	32
12643	185	\N	Lalanne-Arqué	32
12644	186	\N	Lamaguère	32
12645	187	\N	Lamazère	32
12646	188	\N	Lamothe-Goas	32
12647	189	\N	Lannemaignan	32
12648	190	\N	Lannepax	32
12649	191	\N	Lanne-Soubiran	32
12650	192	\N	Lannux	32
12651	193	\N	Larée	32
12652	194	\N	Larressingle	32
12653	195	\N	Larroque-Engalin	32
12654	196	\N	Larroque-Saint-Sernin	32
12655	197	\N	Larroque-sur-l'Osse	32
12656	198	\N	Lartigue	32
12657	199	\N	Lasserade	32
12658	200	\N	Lasséran	32
12659	201	\N	Lasseube-Propre	32
12660	202	\N	Laujuzan	32
12661	203	\N	Lauraët	32
12662	204	\N	Lavardens	32
12663	205	\N	Laveraët	32
12664	206	\N	Laymont	32
12665	207	\N	Leboulin	32
12666	208	\N	Lectoure	32
12667	209	\N	Lelin-Lapujolle	32
12668	210	\N	Lias	32
12669	211	\N	Lias-d'Armagnac	32
12670	212	\N	Ligardes	32
12671	213	\N	Lombez	32
12672	214	\N	Loubédat	32
12673	215	\N	Loubersan	32
12674	216	\N	Lourties-Monbrun	32
12675	217	\N	Louslitges	32
12676	218	\N	Loussous-Débat	32
12677	219	\N	Lupiac	32
12678	220	\N	Luppé-Violles	32
12679	221	\N	Lussan	32
12680	222	\N	Magnan	32
12681	223	\N	Magnas	32
12682	224	\N	Maignaut-Tauzia	32
12683	225	\N	Malabat	32
12684	226	\N	Manas-Bastanous	32
12685	227	\N	Manciet	32
12686	228	\N	Manent-Montané	32
12687	229	\N	Mansempuy	32
12688	230	\N	Mansencôme	32
12689	231	\N	Marambat	32
12690	232	\N	Maravat	32
12691	233	\N	Marciac	32
12692	234	\N	Marestaing	32
12693	235	\N	Margouët-Meymes	32
12694	236	\N	Marguestau	32
12695	237	\N	Marsan	32
12696	238	\N	Marseillan	32
12697	239	\N	Marsolan	32
12698	240	\N	Mascaras	32
12699	241	\N	Mas-d'Auvignon	32
12700	242	\N	Masseube	32
12701	243	\N	Mauléon-d'Armagnac	32
12702	244	\N	Maulichères	32
12703	245	\N	Maumusson-Laguian	32
12704	246	\N	Maupas	32
12705	247	\N	Maurens	32
12706	248	\N	Mauroux	32
12707	249	\N	Mauvezin	32
12708	250	\N	Meilhan	32
12709	251	\N	Mérens	32
12710	252	\N	Miélan	32
12711	253	\N	Miradoux	32
12712	254	\N	Miramont-d'Astarac	32
12713	255	\N	Miramont-Latour	32
12714	256	\N	Mirande	32
12715	257	\N	Mirannes	32
12716	258	\N	Mirepoix	32
12717	260	\N	Monbardon	32
12718	261	\N	Monblanc	32
12719	262	\N	Monbrun	32
12720	263	\N	Moncassin	32
12721	264	\N	Monclar	32
12722	265	\N	Monclar-sur-Losse	32
12723	266	\N	Moncorneil-Grazan	32
12724	267	\N	Monferran-Plavès	32
12725	268	\N	Monferran-Savès	32
12726	269	\N	Monfort	32
12727	270	\N	Mongausy	32
12728	271	\N	Monguilhem	32
12729	272	\N	Monlaur-Bernet	32
12730	273	\N	Monlezun	32
12731	274	\N	Monlezun-d'Armagnac	32
12732	275	\N	Monpardiac	32
12733	276	\N	Montadet	32
12734	277	\N	Montamat	32
12735	278	\N	Montaut	32
12736	279	\N	Montaut-les-Créneaux	32
12737	280	\N	Mont-d'Astarac	32
12738	281	\N	Mont-de-Marrast	32
12739	282	\N	Montégut	32
12740	283	\N	Montégut-Arros	32
12741	284	\N	Montégut-Savès	32
12742	285	\N	Montesquiou	32
12743	286	\N	Montestruc-sur-Gers	32
12744	287	\N	Monties	32
12745	288	\N	Montiron	32
12746	289	\N	Montpézat	32
12747	290	\N	Montréal	32
12748	291	\N	Mormès	32
12749	292	\N	Mouchan	32
12750	293	\N	Mouchès	32
12751	294	\N	Mourède	32
12752	295	\N	Nizas	32
12753	296	\N	Nogaro	32
12754	297	\N	Noilhan	32
12755	298	\N	Nougaroulet	32
12756	299	\N	Noulens	32
12757	300	\N	Orbessan	32
12758	301	\N	Ordan-Larroque	32
12759	302	\N	Ornézan	32
12760	303	\N	Pallanne	32
12761	304	\N	Panassac	32
12762	305	\N	Panjas	32
12763	306	\N	Pauilhac	32
12764	307	\N	Pavie	32
12765	308	\N	Pébées	32
12766	309	\N	Pellefigue	32
12767	310	\N	Perchède	32
12768	311	\N	Pergain-Taillac	32
12769	312	\N	Pessan	32
12770	313	\N	Pessoulens	32
12771	314	\N	Peyrecave	32
12772	315	\N	Peyrusse-Grande	32
12773	316	\N	Peyrusse-Massas	32
12774	317	\N	Peyrusse-Vieille	32
12775	318	\N	Pis	32
12776	319	\N	Plaisance	32
12777	320	\N	Plieux	32
12778	321	\N	Polastron	32
12779	322	\N	Pompiac	32
12780	323	\N	Ponsampère	32
12781	324	\N	Ponsan-Soubiran	32
12782	325	\N	Pouydraguin	32
12783	326	\N	Pouylebon	32
12784	327	\N	Pouy-Loubrin	32
12785	328	\N	Pouy-Roquelaure	32
12786	329	\N	Préchac	32
12787	330	\N	Préchac-sur-Adour	32
12788	331	\N	Preignan	32
12789	332	\N	Préneron	32
12790	333	\N	Projan	32
12791	334	\N	Pujaudran	32
12792	335	\N	Puycasquier	32
12793	336	\N	Puylausic	32
12794	337	\N	Puységur	32
12795	338	\N	Ramouzens	32
12796	339	\N	Razengues	32
12797	340	\N	Réans	32
12798	341	\N	Réjaumont	32
12799	342	\N	Ricourt	32
12800	343	\N	Riguepeu	32
12801	344	\N	Riscle	32
12802	345	\N	Romieu	32
12803	346	\N	Roquebrune	32
12804	347	\N	Roquefort	32
12805	348	\N	Roquelaure	32
12806	349	\N	Roquelaure-Saint-Aubin	32
12807	350	\N	Roquepine	32
12808	351	\N	Roques	32
12809	352	\N	Rozès	32
12810	353	\N	Sabaillan	32
12811	354	\N	Sabazan	32
12812	355	\N	Sadeillan	32
12813	356	\N	Saint-André	32
12814	357	\N	Sainte-Anne	32
12815	358	\N	Saint-Antoine	32
12816	359	\N	Saint-Antonin	32
12817	360	\N	Saint-Arailles	32
12818	361	\N	Saint-Arroman	32
12819	362	\N	Saint-Aunix-Lengros	32
12820	363	\N	Sainte-Aurence-Cazaux	32
12821	364	\N	Saint-Avit-Frandat	32
12822	365	\N	Saint-Blancard	32
12823	366	\N	Saint-Brès	32
12824	367	\N	Saint-Christaud	32
12825	368	\N	Sainte-Christie	32
12826	369	\N	Sainte-Christie-d'Armagnac	32
12827	370	\N	Saint-Clar	32
12828	371	\N	Saint-Créac	32
12829	372	\N	Saint-Cricq	32
12830	373	\N	Sainte-Dode	32
12831	374	\N	Saint-Élix	32
12832	375	\N	Saint-Élix-Theux	32
12833	376	\N	Sainte-Gemme	32
12834	377	\N	Saint-Georges	32
12835	378	\N	Saint-Germé	32
12836	379	\N	Saint-Germier	32
12837	380	\N	Saint-Griède	32
12838	381	\N	Saint-Jean-le-Comtal	32
12839	382	\N	Saint-Jean-Poutge	32
12840	383	\N	Saint-Justin	32
12841	384	\N	Saint-Lary	32
12842	385	\N	Saint-Léonard	32
12843	386	\N	Saint-Lizier-du-Planté	32
12844	387	\N	Saint-Loube	32
12845	388	\N	Sainte-Marie	32
12846	389	\N	Saint-Martin	32
12847	390	\N	Saint-Martin-d'Armagnac	32
12848	391	\N	Saint-Martin-de-Goyne	32
12849	392	\N	Saint-Martin-Gimois	32
12850	393	\N	Saint-Maur	32
12851	394	\N	Saint-Médard	32
12852	395	\N	Sainte-Mère	32
12853	396	\N	Saint-Mézard	32
12854	397	\N	Saint-Michel	32
12855	398	\N	Saint-Mont	32
12856	399	\N	Saint-Orens	32
12857	400	\N	Saint-Orens-Pouy-Petit	32
12858	401	\N	Saint-Ost	32
12859	402	\N	Saint-Paul-de-Baïse	32
12860	403	\N	Saint-Pierre-d'Aubézies	32
12861	404	\N	Saint-Puy	32
12862	405	\N	Sainte-Radegonde	32
12863	406	\N	Saint-Sauvy	32
12864	407	\N	Saint-Soulan	32
12865	408	\N	Salles-d'Armagnac	32
12866	409	\N	Samaran	32
12867	410	\N	Samatan	32
12868	411	\N	Sansan	32
12869	412	\N	Saramon	32
12870	413	\N	Sarcos	32
12871	414	\N	Sarragachies	32
12872	415	\N	Sarraguzan	32
12873	416	\N	Sarrant	32
12874	417	\N	Sauvetat	32
12875	418	\N	Sauveterre	32
12876	419	\N	Sauviac	32
12877	420	\N	Sauvimont	32
12878	421	\N	Savignac-Mona	32
12879	422	\N	Scieurac-et-Flourès	32
12880	423	\N	Séailles	32
12881	424	\N	Ségos	32
12882	425	\N	Ségoufielle	32
12883	426	\N	Seissan	32
12884	427	\N	Sembouès	32
12885	428	\N	Sémézies-Cachan	32
12886	429	\N	Sempesserre	32
12887	430	\N	Sère	32
12888	431	\N	Sérempuy	32
12889	432	\N	Seysses-Savès	32
12890	433	\N	Simorre	32
12891	434	\N	Sion	32
12892	435	\N	Sirac	32
12893	436	\N	Solomiac	32
12894	437	\N	Sorbets	32
12895	438	\N	Tachoires	32
12896	439	\N	Tarsac	32
12897	440	\N	Tasque	32
12898	441	\N	Taybosc	32
12899	442	\N	Terraube	32
12900	443	\N	Termes-d'Armagnac	32
12901	444	\N	Thoux	32
12902	445	\N	Tieste-Uragnoux	32
12903	446	\N	Tillac	32
12904	447	\N	Tirent-Pontéjac	32
12905	448	\N	Touget	32
12906	449	\N	Toujouse	32
12907	450	\N	Tourdun	32
12908	451	\N	Tournan	32
12909	452	\N	Tournecoupe	32
12910	453	\N	Tourrenquets	32
12911	454	\N	Traversères	32
12912	455	\N	Troncens	32
12913	456	\N	Tudelle	32
12914	457	\N	Urdens	32
12915	458	\N	Urgosse	32
12916	459	\N	Valence-sur-Baïse	32
12917	460	\N	Vergoignan	32
12918	461	\N	Verlus	32
12919	462	\N	Vic-Fezensac	32
12920	463	\N	Viella	32
12921	464	\N	Villecomtal-sur-Arros	32
12922	465	\N	Villefranche	32
12923	466	\N	Viozan	32
12924	467	\N	Saint-Caprais	32
12925	468	\N	Aussos	32
12926	001	\N	Abzac	33
12927	002	\N	Aillas	33
12928	003	\N	Ambarès-et-Lagrave	33
12929	004	\N	Ambès	33
12930	005	\N	Andernos-les-Bains	33
12931	006	\N	Anglade	33
12932	007	\N	Arbanats	33
12933	008	\N	Arbis	33
12934	009	\N	Arcachon	33
12935	010	\N	Arcins	33
12936	011	\N	Arès	33
12937	012	\N	Arsac	33
12938	013	\N	Artigues-près-Bordeaux	33
12939	014	\N	Artigues-de-Lussac	33
12940	015	\N	Arveyres	33
12941	016	\N	Asques	33
12942	017	\N	Aubiac	33
12943	018	\N	Aubie-et-Espessas	33
12944	019	\N	Audenge	33
12945	020	\N	Auriolles	33
12946	021	\N	Auros	33
12947	022	\N	Avensan	33
12948	023	\N	Ayguemorte-les-Graves	33
12949	024	\N	Bagas	33
12950	025	\N	Baigneaux	33
12951	026	\N	Balizac	33
12952	027	\N	Barie	33
12953	028	\N	Baron	33
12954	029	\N	Barp	33
12955	030	\N	Barsac	33
12956	031	\N	Bassanne	33
12957	032	\N	Bassens	33
12958	033	\N	Baurech	33
12959	034	\N	Bayas	33
12960	035	\N	Bayon-sur-Gironde	33
12961	036	\N	Bazas	33
12962	037	\N	Beautiran	33
12963	038	\N	Bégadan	33
12964	039	\N	Bègles	33
12965	040	\N	Béguey	33
12966	042	\N	Belin-Béliet	33
12967	043	\N	Bellebat	33
12968	044	\N	Bellefond	33
12969	045	\N	Belvès-de-Castillon	33
12970	046	\N	Bernos-Beaulac	33
12971	047	\N	Berson	33
12972	048	\N	Berthez	33
12973	049	\N	Beychac-et-Caillau	33
12974	050	\N	Bieujac	33
12975	051	\N	Biganos	33
12976	052	\N	Billaux	33
12977	053	\N	Birac	33
12978	054	\N	Blaignac	33
12979	055	\N	Blaignan	33
12980	056	\N	Blanquefort	33
12981	057	\N	Blasimon	33
12982	058	\N	Blaye	33
12983	059	\N	Blésignac	33
12984	060	\N	Bommes	33
12985	061	\N	Bonnetan	33
12986	062	\N	Bonzac	33
12987	063	\N	Bordeaux	33
12988	064	\N	Bossugan	33
12989	065	\N	Bouliac	33
12990	066	\N	Bourdelles	33
12991	067	\N	Bourg	33
12992	068	\N	Bourideys	33
12993	069	\N	Bouscat	33
12994	070	\N	Brach	33
12995	071	\N	Branne	33
12996	072	\N	Brannens	33
12997	073	\N	Braud-et-Saint-Louis	33
12998	074	\N	Brouqueyran	33
12999	075	\N	Bruges	33
13000	076	\N	Budos	33
13001	077	\N	Cabanac-et-Villagrains	33
13002	078	\N	Cabara	33
13003	079	\N	Cadarsac	33
13004	080	\N	Cadaujac	33
13005	081	\N	Cadillac	33
13006	082	\N	Cadillac-en-Fronsadais	33
13007	083	\N	Camarsac	33
13008	084	\N	Cambes	33
13009	085	\N	Camblanes-et-Meynac	33
13010	086	\N	Camiac-et-Saint-Denis	33
13011	087	\N	Camiran	33
13012	088	\N	Camps-sur-l'Isle	33
13013	089	\N	Campugnan	33
13014	090	\N	Canéjan	33
13015	091	\N	Cantenac	33
13016	092	\N	Cantois	33
13017	093	\N	Capian	33
13018	094	\N	Caplong	33
13019	095	\N	Captieux	33
13020	096	\N	Carbon-Blanc	33
13021	097	\N	Carcans	33
13022	098	\N	Cardan	33
13023	099	\N	Carignan-de-Bordeaux	33
13024	100	\N	Cars	33
13025	101	\N	Cartelègue	33
13026	102	\N	Casseuil	33
13027	103	\N	Castelmoron-d'Albret	33
13028	104	\N	Castelnau-de-Médoc	33
13029	105	\N	Castelviel	33
13030	106	\N	Castets-en-Dorthe	33
13031	107	\N	Castillon-de-Castets	33
13032	108	\N	Castillon-la-Bataille	33
13033	109	\N	Castres-Gironde	33
13034	111	\N	Caudrot	33
13035	112	\N	Caumont	33
13036	113	\N	Cauvignac	33
13037	114	\N	Cavignac	33
13038	115	\N	Cazalis	33
13039	116	\N	Cazats	33
13040	117	\N	Cazaugitat	33
13041	118	\N	Cénac	33
13042	119	\N	Cenon	33
13043	120	\N	Cérons	33
13044	121	\N	Cessac	33
13045	122	\N	Cestas	33
13046	123	\N	Cézac	33
13047	124	\N	Chamadelle	33
13048	125	\N	Cissac-Médoc	33
13049	126	\N	Civrac-de-Blaye	33
13050	127	\N	Civrac-sur-Dordogne	33
13051	128	\N	Civrac-en-Médoc	33
13052	129	\N	Cleyrac	33
13053	130	\N	Coimères	33
13054	131	\N	Coirac	33
13055	132	\N	Comps	33
13056	133	\N	Coubeyrac	33
13057	134	\N	Couquèques	33
13058	135	\N	Courpiac	33
13059	136	\N	Cours-de-Monségur	33
13060	137	\N	Cours-les-Bains	33
13061	138	\N	Coutras	33
13062	139	\N	Coutures	33
13063	140	\N	Créon	33
13064	141	\N	Croignon	33
13065	142	\N	Cubnezais	33
13066	143	\N	Cubzac-les-Ponts	33
13067	144	\N	Cudos	33
13068	145	\N	Cursan	33
13069	146	\N	Cussac-Fort-Médoc	33
13070	147	\N	Daignac	33
13071	148	\N	Dardenac	33
13072	149	\N	Daubèze	33
13073	150	\N	Dieulivol	33
13074	151	\N	Donnezac	33
13075	152	\N	Donzac	33
13076	153	\N	Doulezon	33
13077	154	\N	Églisottes-et-Chalaures	33
13078	155	\N	Escaudes	33
13079	156	\N	Escoussans	33
13080	157	\N	Espiet	33
13081	158	\N	Esseintes	33
13082	159	\N	Étauliers	33
13083	160	\N	Eynesse	33
13084	161	\N	Eyrans	33
13085	162	\N	Eysines	33
13086	163	\N	Faleyras	33
13087	164	\N	Fargues	33
13088	165	\N	Fargues-Saint-Hilaire	33
13089	166	\N	Fieu	33
13090	167	\N	Floirac	33
13091	168	\N	Flaujagues	33
13092	169	\N	Floudès	33
13093	170	\N	Fontet	33
13094	171	\N	Fossès-et-Baleyssac	33
13095	172	\N	Fours	33
13096	173	\N	Francs	33
13097	174	\N	Fronsac	33
13098	175	\N	Frontenac	33
13099	176	\N	Gabarnac	33
13100	177	\N	Gaillan-en-Médoc	33
13101	178	\N	Gajac	33
13102	179	\N	Galgon	33
13103	180	\N	Gans	33
13104	181	\N	Gardegan-et-Tourtirac	33
13105	182	\N	Gauriac	33
13106	183	\N	Gauriaguet	33
13107	184	\N	Générac	33
13108	185	\N	Génissac	33
13109	186	\N	Gensac	33
13110	187	\N	Gironde-sur-Dropt	33
13111	188	\N	Giscos	33
13112	189	\N	Gornac	33
13113	190	\N	Goualade	33
13114	191	\N	Gours	33
13115	192	\N	Gradignan	33
13116	193	\N	Grayan-et-l'Hôpital	33
13117	194	\N	Grézillac	33
13118	195	\N	Grignols	33
13119	196	\N	Guillac	33
13120	197	\N	Guillos	33
13121	198	\N	Guîtres	33
13122	199	\N	Gujan-Mestras	33
13123	200	\N	Haillan	33
13124	201	\N	Haux	33
13125	202	\N	Hostens	33
13126	203	\N	Hourtin	33
13127	204	\N	Hure	33
13128	205	\N	Illats	33
13129	206	\N	Isle-Saint-Georges	33
13130	207	\N	Izon	33
13131	208	\N	Jau-Dignac-et-Loirac	33
13132	209	\N	Jugazan	33
13133	210	\N	Juillac	33
13134	211	\N	Labarde	33
13135	212	\N	Labescau	33
13136	213	\N	Brède	33
13137	214	\N	Lacanau	33
13138	215	\N	Ladaux	33
13139	216	\N	Lados	33
13140	218	\N	Lagorce	33
13141	219	\N	Lande-de-Fronsac	33
13142	220	\N	Lamarque	33
13143	221	\N	Lamothe-Landerron	33
13144	222	\N	Lalande-de-Pomerol	33
13145	223	\N	Landerrouat	33
13146	224	\N	Landerrouet-sur-Ségur	33
13147	225	\N	Landiras	33
13148	226	\N	Langoiran	33
13149	227	\N	Langon	33
13150	228	\N	Lansac	33
13151	229	\N	Lanton	33
13152	230	\N	Lapouyade	33
13153	231	\N	Laroque	33
13154	232	\N	Lartigue	33
13155	233	\N	Laruscade	33
13156	234	\N	Latresne	33
13157	235	\N	Lavazan	33
13158	236	\N	Lège-Cap-Ferret	33
13159	237	\N	Léogeats	33
13160	238	\N	Léognan	33
13161	239	\N	Lerm-et-Musset	33
13162	240	\N	Lesparre-Médoc	33
13163	241	\N	Lestiac-sur-Garonne	33
13164	242	\N	Lèves-et-Thoumeyragues	33
13165	243	\N	Libourne	33
13166	244	\N	Lignan-de-Bazas	33
13167	245	\N	Lignan-de-Bordeaux	33
13168	246	\N	Ligueux	33
13169	247	\N	Listrac-de-Durèze	33
13170	248	\N	Listrac-Médoc	33
13171	249	\N	Lormont	33
13172	250	\N	Loubens	33
13173	251	\N	Louchats	33
13174	252	\N	Loupes	33
13175	253	\N	Loupiac	33
13176	254	\N	Loupiac-de-la-Réole	33
13177	255	\N	Lucmau	33
13178	256	\N	Ludon-Médoc	33
13179	257	\N	Lugaignac	33
13180	258	\N	Lugasson	33
13181	259	\N	Lugon-et-l'Île-du-Carnay	33
13182	260	\N	Lugos	33
13183	261	\N	Lussac	33
13184	262	\N	Macau	33
13185	263	\N	Madirac	33
13186	264	\N	Maransin	33
13187	266	\N	Marcenais	33
13188	267	\N	Marcillac	33
13189	268	\N	Margaux	33
13190	269	\N	Margueron	33
13191	270	\N	Marimbault	33
13192	271	\N	Marions	33
13193	272	\N	Marsas	33
13194	273	\N	Martignas-sur-Jalle	33
13195	274	\N	Martillac	33
13196	275	\N	Martres	33
13197	276	\N	Masseilles	33
13198	277	\N	Massugas	33
13199	278	\N	Mauriac	33
13200	279	\N	Mazères	33
13201	280	\N	Mazion	33
13202	281	\N	Mérignac	33
13203	282	\N	Mérignas	33
13204	283	\N	Mesterrieux	33
13205	284	\N	Mios	33
13206	285	\N	Mombrier	33
13207	287	\N	Mongauzy	33
13208	288	\N	Monprimblanc	33
13209	289	\N	Monségur	33
13210	290	\N	Montagne	33
13211	291	\N	Montagoudin	33
13212	292	\N	Montignac	33
13213	293	\N	Montussan	33
13214	294	\N	Morizès	33
13215	295	\N	Mouillac	33
13216	296	\N	Mouliets-et-Villemartin	33
13217	297	\N	Moulis-en-Médoc	33
13218	298	\N	Moulon	33
13219	299	\N	Mourens	33
13220	300	\N	Naujac-sur-Mer	33
13221	301	\N	Naujan-et-Postiac	33
13222	302	\N	Néac	33
13223	303	\N	Nérigean	33
13224	304	\N	Neuffons	33
13225	305	\N	Nizan	33
13226	306	\N	Noaillac	33
13227	307	\N	Noaillan	33
13228	308	\N	Omet	33
13229	309	\N	Ordonnac	33
13230	310	\N	Origne	33
13231	311	\N	Paillet	33
13232	312	\N	Parempuyre	33
13233	314	\N	Pauillac	33
13234	315	\N	Peintures	33
13235	316	\N	Pellegrue	33
13236	317	\N	Périssac	33
13237	318	\N	Pessac	33
13238	319	\N	Pessac-sur-Dordogne	33
13239	320	\N	Petit-Palais-et-Cornemps	33
13240	321	\N	Peujard	33
13241	322	\N	Pian-Médoc	33
13242	323	\N	Pian-sur-Garonne	33
13243	324	\N	Pineuilh	33
13244	325	\N	Plassac	33
13245	326	\N	Pleine-Selve	33
13246	327	\N	Podensac	33
13247	328	\N	Pomerol	33
13248	329	\N	Pompéjac	33
13249	330	\N	Pompignac	33
13250	331	\N	Pondaurat	33
13251	332	\N	Porchères	33
13252	333	\N	Porge	33
13253	334	\N	Portets	33
13254	335	\N	Pout	33
13255	336	\N	Préchac	33
13256	337	\N	Preignac	33
13257	338	\N	Prignac-en-Médoc	33
13258	339	\N	Prignac-et-Marcamps	33
13259	341	\N	Pugnac	33
13260	342	\N	Puisseguin	33
13261	343	\N	Pujols-sur-Ciron	33
13262	344	\N	Pujols	33
13263	345	\N	Puy	33
13264	346	\N	Puybarban	33
13265	347	\N	Puynormand	33
13266	348	\N	Queyrac	33
13267	349	\N	Quinsac	33
13268	350	\N	Rauzan	33
13269	351	\N	Reignac	33
13270	352	\N	Réole	33
13271	353	\N	Rimons	33
13272	354	\N	Riocaud	33
13273	355	\N	Rions	33
13274	356	\N	Rivière	33
13275	357	\N	Roaillan	33
13276	358	\N	Romagne	33
13277	359	\N	Roquebrune	33
13278	360	\N	Roquille	33
13279	361	\N	Ruch	33
13280	362	\N	Sablons	33
13281	363	\N	Sadirac	33
13282	364	\N	Saillans	33
13283	365	\N	Saint-Aignan	33
13284	366	\N	Saint-André-de-Cubzac	33
13285	367	\N	Saint-André-du-Bois	33
13286	369	\N	Saint-André-et-Appelles	33
13287	370	\N	Saint-Androny	33
13288	371	\N	Saint-Antoine	33
13289	372	\N	Saint-Antoine-du-Queyret	33
13290	373	\N	Saint-Antoine-sur-l'Isle	33
13291	374	\N	Saint-Aubin-de-Blaye	33
13292	375	\N	Saint-Aubin-de-Branne	33
13293	376	\N	Saint-Aubin-de-Médoc	33
13294	377	\N	Saint-Avit-de-Soulège	33
13295	378	\N	Saint-Avit-Saint-Nazaire	33
13296	379	\N	Saint-Brice	33
13297	380	\N	Saint-Caprais-de-Blaye	33
13298	381	\N	Saint-Caprais-de-Bordeaux	33
13299	382	\N	Saint-Christoly-de-Blaye	33
13300	383	\N	Saint-Christoly-Médoc	33
13301	384	\N	Saint-Christophe-des-Bardes	33
13302	385	\N	Saint-Christophe-de-Double	33
13303	386	\N	Saint-Cibard	33
13304	387	\N	Saint-Ciers-d'Abzac	33
13305	388	\N	Saint-Ciers-de-Canesse	33
13306	389	\N	Saint-Ciers-sur-Gironde	33
13307	390	\N	Sainte-Colombe	33
13308	391	\N	Saint-Côme	33
13309	392	\N	Sainte-Croix-du-Mont	33
13310	393	\N	Saint-Denis-de-Pile	33
13311	394	\N	Saint-Émilion	33
13312	395	\N	Saint-Estèphe	33
13313	396	\N	Saint-Étienne-de-Lisse	33
13314	397	\N	Sainte-Eulalie	33
13315	398	\N	Saint-Exupéry	33
13316	399	\N	Saint-Félix-de-Foncaude	33
13317	400	\N	Saint-Ferme	33
13318	401	\N	Sainte-Florence	33
13319	402	\N	Sainte-Foy-la-Grande	33
13320	403	\N	Sainte-Foy-la-Longue	33
13321	404	\N	Sainte-Gemme	33
13322	405	\N	Saint-Genès-de-Blaye	33
13323	406	\N	Saint-Genès-de-Castillon	33
13324	407	\N	Saint-Genès-de-Fronsac	33
13325	408	\N	Saint-Genès-de-Lombaud	33
13326	409	\N	Saint-Genis-du-Bois	33
13327	411	\N	Saint-Germain-de-Grave	33
13328	412	\N	Saint-Germain-d'Esteuil	33
13329	413	\N	Saint-Germain-du-Puch	33
13330	414	\N	Saint-Germain-de-la-Rivière	33
13331	415	\N	Saint-Gervais	33
13332	416	\N	Saint-Girons-d'Aiguevives	33
13333	417	\N	Sainte-Hélène	33
13334	418	\N	Saint-Hilaire-de-la-Noaille	33
13335	419	\N	Saint-Hilaire-du-Bois	33
13336	420	\N	Saint-Hippolyte	33
13337	421	\N	Saint-Jean-de-Blaignac	33
13338	422	\N	Saint-Jean-d'Illac	33
13339	423	\N	Saint-Julien-Beychevelle	33
13340	424	\N	Saint-Laurent-Médoc	33
13341	425	\N	Saint-Laurent-d'Arce	33
13342	426	\N	Saint-Laurent-des-Combes	33
13343	427	\N	Saint-Laurent-du-Bois	33
13344	428	\N	Saint-Laurent-du-Plan	33
13345	429	\N	Saint-Léger-de-Balson	33
13346	431	\N	Saint-Léon	33
13347	432	\N	Saint-Loubert	33
13348	433	\N	Saint-Loubès	33
13349	434	\N	Saint-Louis-de-Montferrand	33
13350	435	\N	Saint-Macaire	33
13351	436	\N	Saint-Magne	33
13352	437	\N	Saint-Magne-de-Castillon	33
13353	438	\N	Saint-Maixant	33
13354	439	\N	Saint-Mariens	33
13355	440	\N	Saint-Martial	33
13356	441	\N	Saint-Martin-Lacaussade	33
13357	442	\N	Saint-Martin-de-Laye	33
13358	443	\N	Saint-Martin-de-Lerm	33
13359	444	\N	Saint-Martin-de-Sescas	33
13360	445	\N	Saint-Martin-du-Bois	33
13361	446	\N	Saint-Martin-du-Puy	33
13362	447	\N	Saint-Médard-de-Guizières	33
13363	448	\N	Saint-Médard-d'Eyrans	33
13364	449	\N	Saint-Médard-en-Jalles	33
13365	450	\N	Saint-Michel-de-Castelnau	33
13366	451	\N	Saint-Michel-de-Fronsac	33
13367	452	\N	Saint-Michel-de-Rieufret	33
13368	453	\N	Saint-Michel-de-Lapujade	33
13369	454	\N	Saint-Morillon	33
13370	456	\N	Saint-Palais	33
13371	457	\N	Saint-Pardon-de-Conques	33
13372	458	\N	Saint-Paul	33
13373	459	\N	Saint-Pey-d'Armens	33
13374	460	\N	Saint-Pey-de-Castets	33
13375	461	\N	Saint-Philippe-d'Aiguille	33
13376	462	\N	Saint-Philippe-du-Seignal	33
13377	463	\N	Saint-Pierre-d'Aurillac	33
13378	464	\N	Saint-Pierre-de-Bat	33
13379	465	\N	Saint-Pierre-de-Mons	33
13380	466	\N	Saint-Quentin-de-Baron	33
13381	467	\N	Saint-Quentin-de-Caplong	33
13382	468	\N	Sainte-Radegonde	33
13383	470	\N	Saint-Romain-la-Virvée	33
13384	471	\N	Saint-Sauveur	33
13385	472	\N	Saint-Sauveur-de-Puynormand	33
13386	473	\N	Saint-Savin	33
13387	474	\N	Saint-Selve	33
13388	475	\N	Saint-Seurin-de-Bourg	33
13389	476	\N	Saint-Seurin-de-Cadourne	33
13390	477	\N	Saint-Seurin-de-Cursac	33
13391	478	\N	Saint-Seurin-sur-l'Isle	33
13392	479	\N	Saint-Sève	33
13393	480	\N	Saint-Sulpice-de-Faleyrens	33
13394	481	\N	Saint-Sulpice-de-Guilleragues	33
13395	482	\N	Saint-Sulpice-de-Pommiers	33
13396	483	\N	Saint-Sulpice-et-Cameyrac	33
13397	484	\N	Saint-Symphorien	33
13398	485	\N	Sainte-Terre	33
13399	486	\N	Saint-Trojan	33
13400	487	\N	Saint-Vincent-de-Paul	33
13401	488	\N	Saint-Vincent-de-Pertignas	33
13402	489	\N	Saint-Vivien-de-Blaye	33
13403	490	\N	Saint-Vivien-de-Médoc	33
13404	491	\N	Saint-Vivien-de-Monségur	33
13405	492	\N	Saint-Yzan-de-Soudiac	33
13406	493	\N	Saint-Yzans-de-Médoc	33
13407	494	\N	Salaunes	33
13408	495	\N	Salignac	33
13409	496	\N	Sallebuf	33
13410	498	\N	Salles	33
13411	499	\N	Salles-de-Castillon	33
13412	500	\N	Samonac	33
13413	501	\N	Saucats	33
13414	502	\N	Saugon	33
13415	503	\N	Saumos	33
13416	504	\N	Sauternes	33
13417	505	\N	Sauve	33
13418	506	\N	Sauveterre-de-Guyenne	33
13419	507	\N	Sauviac	33
13420	508	\N	Savignac	33
13421	509	\N	Savignac-de-l'Isle	33
13422	510	\N	Semens	33
13423	511	\N	Sendets	33
13424	512	\N	Sigalens	33
13425	513	\N	Sillas	33
13426	514	\N	Soulac-sur-Mer	33
13427	515	\N	Soulignac	33
13428	516	\N	Soussac	33
13429	517	\N	Soussans	33
13430	518	\N	Tabanac	33
13431	519	\N	Taillan-Médoc	33
13432	520	\N	Taillecavat	33
13433	521	\N	Talais	33
13434	522	\N	Talence	33
13435	523	\N	Targon	33
13436	524	\N	Tarnès	33
13437	525	\N	Tauriac	33
13438	526	\N	Tayac	33
13439	527	\N	Teich	33
13440	528	\N	Temple	33
13441	529	\N	Teste-de-Buch	33
13442	530	\N	Teuillac	33
13443	531	\N	Tizac-de-Curton	33
13444	532	\N	Tizac-de-Lapouyade	33
13445	533	\N	Toulenne	33
13446	534	\N	Tourne	33
13447	535	\N	Tresses	33
13448	536	\N	Tuzan	33
13449	537	\N	Uzeste	33
13450	538	\N	Valeyrac	33
13451	539	\N	Vayres	33
13452	540	\N	Vendays-Montalivet	33
13453	541	\N	Vensac	33
13454	542	\N	Vérac	33
13455	543	\N	Verdelais	33
13456	544	\N	Verdon-sur-Mer	33
13457	545	\N	Vertheuil	33
13458	546	\N	Vignonet	33
13459	547	\N	Villandraut	33
13460	548	\N	Villegouge	33
13461	549	\N	Villenave-de-Rions	33
13462	550	\N	Villenave-d'Ornon	33
13463	551	\N	Villeneuve	33
13464	552	\N	Virelade	33
13465	553	\N	Virsac	33
13466	554	\N	Yvrac	33
13467	555	\N	Marcheprime	33
13468	001	\N	Abeilhan	34
13469	002	\N	Adissan	34
13470	003	\N	Agde	34
13471	004	\N	Agel	34
13472	005	\N	Agonès	34
13473	006	\N	Aigne	34
13474	007	\N	Aigues-Vives	34
13475	008	\N	Aires	34
13476	009	\N	Alignan-du-Vent	34
13477	010	\N	Aniane	34
13478	011	\N	Arboras	34
13479	012	\N	Argelliers	34
13480	013	\N	Aspiran	34
13481	014	\N	Assas	34
13482	015	\N	Assignan	34
13483	016	\N	Aumelas	34
13484	017	\N	Aumes	34
13485	018	\N	Autignac	34
13486	019	\N	Avène	34
13487	020	\N	Azillanet	34
13488	021	\N	Babeau-Bouldoux	34
13489	022	\N	Baillargues	34
13490	023	\N	Balaruc-les-Bains	34
13491	024	\N	Balaruc-le-Vieux	34
13492	025	\N	Bassan	34
13493	026	\N	Beaufort	34
13494	027	\N	Beaulieu	34
13495	028	\N	Bédarieux	34
13496	029	\N	Bélarga	34
13497	030	\N	Berlou	34
13498	031	\N	Bessan	34
13499	032	\N	Béziers	34
13500	033	\N	Boisseron	34
13501	034	\N	Boisset	34
13502	035	\N	Boissière	34
13503	036	\N	Bosc	34
13504	037	\N	Boujan-sur-Libron	34
13505	038	\N	Bousquet-d'Orb	34
13506	039	\N	Bouzigues	34
13507	040	\N	Brenas	34
13508	041	\N	Brignac	34
13509	042	\N	Brissac	34
13510	043	\N	Buzignargues	34
13511	044	\N	Cabrerolles	34
13512	045	\N	Cabrières	34
13513	046	\N	Cambon-et-Salvergues	34
13514	047	\N	Campagnan	34
13515	048	\N	Campagne	34
13516	049	\N	Camplong	34
13517	050	\N	Candillargues	34
13518	051	\N	Canet	34
13519	052	\N	Capestang	34
13520	053	\N	Carlencas-et-Levas	34
13521	054	\N	Cassagnoles	34
13522	055	\N	Castanet-le-Haut	34
13523	056	\N	Castelnau-de-Guers	34
13524	057	\N	Castelnau-le-Lez	34
13525	058	\N	Castries	34
13526	059	\N	Caunette	34
13527	060	\N	Causse-de-la-Selle	34
13528	061	\N	Causses-et-Veyran	34
13529	062	\N	Caussiniojouls	34
13530	063	\N	Caux	34
13531	064	\N	Caylar	34
13532	065	\N	Cazedarnes	34
13533	066	\N	Cazevieille	34
13534	067	\N	Cazilhac	34
13535	068	\N	Cazouls-d'Hérault	34
13536	069	\N	Cazouls-lès-Béziers	34
13537	070	\N	Cébazan	34
13538	071	\N	Ceilhes-et-Rocozels	34
13539	072	\N	Celles	34
13540	073	\N	Cers	34
13541	074	\N	Cessenon-sur-Orb	34
13542	075	\N	Cesseras	34
13543	076	\N	Ceyras	34
13544	077	\N	Clapiers	34
13545	078	\N	Claret	34
13546	079	\N	Clermont-l'Hérault	34
13547	080	\N	Colombières-sur-Orb	34
13548	081	\N	Colombiers	34
13549	082	\N	Combaillaux	34
13550	083	\N	Combes	34
13551	084	\N	Corneilhan	34
13552	085	\N	Coulobres	34
13553	086	\N	Courniou	34
13554	087	\N	Cournonsec	34
13555	088	\N	Cournonterral	34
13556	089	\N	Creissan	34
13557	090	\N	Crès	34
13558	091	\N	Cros	34
13559	092	\N	Cruzy	34
13560	093	\N	Dio-et-Valquières	34
13561	094	\N	Espondeilhan	34
13562	095	\N	Fabrègues	34
13563	096	\N	Faugères	34
13564	097	\N	Félines-Minervois	34
13565	098	\N	Ferrals-les-Montagnes	34
13566	099	\N	Ferrières-les-Verreries	34
13567	100	\N	Ferrières-Poussarou	34
13568	101	\N	Florensac	34
13569	102	\N	Fontanès	34
13570	103	\N	Fontès	34
13571	104	\N	Fos	34
13572	105	\N	Fouzilhon	34
13573	106	\N	Fozières	34
13574	107	\N	Fraisse-sur-Agout	34
13575	108	\N	Frontignan	34
13576	109	\N	Gabian	34
13577	110	\N	Galargues	34
13578	111	\N	Ganges	34
13579	112	\N	Garrigues	34
13580	113	\N	Gigean	34
13581	114	\N	Gignac	34
13582	115	\N	Gorniès	34
13583	116	\N	Grabels	34
13584	117	\N	Graissessac	34
13585	118	\N	Guzargues	34
13586	119	\N	Hérépian	34
13587	120	\N	Jacou	34
13588	121	\N	Joncels	34
13589	122	\N	Jonquières	34
13590	123	\N	Juvignac	34
13591	124	\N	Lacoste	34
13592	125	\N	Lagamas	34
13593	126	\N	Lamalou-les-Bains	34
13594	127	\N	Lansargues	34
13595	128	\N	Laroque	34
13596	129	\N	Lattes	34
13597	130	\N	Laurens	34
13598	131	\N	Lauret	34
13599	132	\N	Lauroux	34
13600	133	\N	Lavalette	34
13601	134	\N	Lavérune	34
13602	135	\N	Lespignan	34
13603	136	\N	Lézignan-la-Cèbe	34
13604	137	\N	Liausson	34
13605	138	\N	Lieuran-Cabrières	34
13606	139	\N	Lieuran-lès-Béziers	34
13607	140	\N	Lignan-sur-Orb	34
13608	141	\N	Livinière	34
13609	142	\N	Lodève	34
13610	143	\N	Loupian	34
13611	144	\N	Lunas	34
13612	145	\N	Lunel	34
13613	146	\N	Lunel-Viel	34
13614	147	\N	Magalas	34
13615	148	\N	Maraussan	34
13616	149	\N	Margon	34
13617	150	\N	Marseillan	34
13618	151	\N	Marsillargues	34
13619	152	\N	Mas-de-Londres	34
13620	153	\N	Matelles	34
13621	154	\N	Mauguio	34
13622	155	\N	Maureilhan	34
13623	156	\N	Mérifons	34
13624	157	\N	Mèze	34
13625	158	\N	Minerve	34
13626	159	\N	Mireval	34
13627	160	\N	Mons	34
13628	161	\N	Montady	34
13629	162	\N	Montagnac	34
13630	163	\N	Montarnaud	34
13631	164	\N	Montaud	34
13632	165	\N	Montbazin	34
13633	166	\N	Montblanc	34
13634	167	\N	Montels	34
13635	168	\N	Montesquieu	34
13636	169	\N	Montferrier-sur-Lez	34
13637	170	\N	Montouliers	34
13638	171	\N	Montoulieu	34
13639	172	\N	Montpellier	34
13640	173	\N	Montpeyroux	34
13641	174	\N	Moulès-et-Baucels	34
13642	175	\N	Mourèze	34
13643	176	\N	Mudaison	34
13644	177	\N	Murles	34
13645	178	\N	Murviel-lès-Béziers	34
13646	179	\N	Murviel-lès-Montpellier	34
13647	180	\N	Nébian	34
13648	181	\N	Neffiès	34
13649	182	\N	Nézignan-l'Évêque	34
13650	183	\N	Nissan-lez-Enserune	34
13651	184	\N	Nizas	34
13652	185	\N	Notre-Dame-de-Londres	34
13653	186	\N	Octon	34
13654	187	\N	Olargues	34
13655	188	\N	Olmet-et-Villecun	34
13656	189	\N	Olonzac	34
13657	190	\N	Oupia	34
13658	191	\N	Pailhès	34
13659	192	\N	Palavas-les-Flots	34
13660	193	\N	Pardailhan	34
13661	194	\N	Paulhan	34
13662	195	\N	Pégairolles-de-Buèges	34
13663	196	\N	Pégairolles-de-l'Escalette	34
13664	197	\N	Péret	34
13665	198	\N	Pérols	34
13666	199	\N	Pézenas	34
13667	200	\N	Pézènes-les-Mines	34
13668	201	\N	Pierrerue	34
13669	202	\N	Pignan	34
13670	203	\N	Pinet	34
13671	204	\N	Plaissan	34
13672	205	\N	Plans	34
13673	206	\N	Poilhes	34
13674	207	\N	Pomérols	34
13675	208	\N	Popian	34
13676	209	\N	Portiragnes	34
13677	210	\N	Pouget	34
13678	211	\N	Poujol-sur-Orb	34
13679	212	\N	Poujols	34
13680	213	\N	Poussan	34
13681	214	\N	Pouzolles	34
13682	215	\N	Pouzols	34
13683	216	\N	Pradal	34
13684	217	\N	Prades-le-Lez	34
13685	218	\N	Prades-sur-Vernazobre	34
13686	219	\N	Prémian	34
13687	220	\N	Puech	34
13688	221	\N	Puéchabon	34
13689	222	\N	Puilacher	34
13690	223	\N	Puimisson	34
13691	224	\N	Puissalicon	34
13692	225	\N	Puisserguier	34
13693	226	\N	Quarante	34
13694	227	\N	Restinclières	34
13695	228	\N	Rieussec	34
13696	229	\N	Riols	34
13697	230	\N	Rives	34
13698	231	\N	Romiguières	34
13699	232	\N	Roquebrun	34
13700	233	\N	Roqueredonde	34
13701	234	\N	Roquessels	34
13702	235	\N	Rosis	34
13703	236	\N	Rouet	34
13704	237	\N	Roujan	34
13705	238	\N	Saint-André-de-Buèges	34
13706	239	\N	Saint-André-de-Sangonis	34
13707	240	\N	Saint-Aunès	34
13708	241	\N	Saint-Bauzille-de-la-Sylve	34
13709	242	\N	Saint-Bauzille-de-Montmel	34
13710	243	\N	Saint-Bauzille-de-Putois	34
13711	244	\N	Saint-Brès	34
13712	245	\N	Saint-Chinian	34
13713	246	\N	Saint-Christol	34
13714	247	\N	Saint-Clément-de-Rivière	34
13715	248	\N	Sainte-Croix-de-Quintillargues	34
13716	249	\N	Saint-Drézéry	34
13717	250	\N	Saint-Étienne-d'Albagnan	34
13718	251	\N	Saint-Étienne-de-Gourgas	34
13719	252	\N	Saint-Étienne-Estréchoux	34
13720	253	\N	Saint-Félix-de-l'Héras	34
13721	254	\N	Saint-Félix-de-Lodez	34
13722	255	\N	Saint-Gély-du-Fesc	34
13723	256	\N	Saint-Geniès-des-Mourgues	34
13724	257	\N	Saint-Geniès-de-Varensal	34
13725	258	\N	Saint-Geniès-de-Fontedit	34
13726	259	\N	Saint-Georges-d'Orques	34
13727	260	\N	Saint-Gervais-sur-Mare	34
13728	261	\N	Saint-Guilhem-le-Désert	34
13729	262	\N	Saint-Guiraud	34
13730	263	\N	Saint-Hilaire-de-Beauvoir	34
13731	264	\N	Saint-Jean-de-Buèges	34
13732	265	\N	Saint-Jean-de-Cornies	34
13733	266	\N	Saint-Jean-de-Cuculles	34
13734	267	\N	Saint-Jean-de-Fos	34
13735	268	\N	Saint-Jean-de-la-Blaquière	34
13736	269	\N	Saint-Jean-de-Minervois	34
13737	270	\N	Saint-Jean-de-Védas	34
13738	271	\N	Saint-Julien	34
13739	272	\N	Saint-Just	34
13740	273	\N	Saint-Martin-de-l'Arçon	34
13741	274	\N	Saint-Martin-de-Londres	34
13742	276	\N	Saint-Mathieu-de-Tréviers	34
13743	277	\N	Saint-Maurice-Navacelles	34
13744	278	\N	Saint-Michel	34
13745	279	\N	Saint-Nazaire-de-Ladarez	34
13746	280	\N	Saint-Nazaire-de-Pézan	34
13747	281	\N	Saint-Pargoire	34
13748	282	\N	Saint-Paul-et-Valmalle	34
13749	283	\N	Saint-Pierre-de-la-Fage	34
13750	284	\N	Saint-Pons-de-Thomières	34
13751	285	\N	Saint-Pons-de-Mauchiens	34
13752	286	\N	Saint-Privat	34
13753	287	\N	Saint-Saturnin-de-Lucian	34
13754	288	\N	Saint-Sériès	34
13755	289	\N	Saint-Thibéry	34
13756	290	\N	Saint-Vincent-de-Barbeyrargues	34
13757	291	\N	Saint-Vincent-d'Olargues	34
13758	292	\N	Salasc	34
13759	293	\N	Salvetat-sur-Agout	34
13760	294	\N	Saturargues	34
13761	295	\N	Saussan	34
13762	296	\N	Saussines	34
13763	297	\N	Sauteyrargues	34
13764	298	\N	Sauvian	34
13765	299	\N	Sérignan	34
13766	300	\N	Servian	34
13767	301	\N	Sète	34
13768	302	\N	Siran	34
13769	303	\N	Sorbs	34
13770	304	\N	Soubès	34
13771	305	\N	Soulié	34
13772	306	\N	Soumont	34
13773	307	\N	Sussargues	34
13774	308	\N	Taussac-la-Billière	34
13775	309	\N	Teyran	34
13776	310	\N	Thézan-lès-Béziers	34
13777	311	\N	Tourbes	34
13778	312	\N	Tour-sur-Orb	34
13779	313	\N	Tressan	34
13780	314	\N	Triadou	34
13781	315	\N	Usclas-d'Hérault	34
13782	316	\N	Usclas-du-Bosc	34
13783	317	\N	Vacquerie-et-Saint-Martin-de-Castries	34
13784	318	\N	Vacquières	34
13785	319	\N	Vailhan	34
13786	320	\N	Vailhauquès	34
13787	321	\N	Valergues	34
13788	322	\N	Valflaunès	34
13789	323	\N	Valmascle	34
13790	324	\N	Valras-Plage	34
13791	325	\N	Valros	34
13792	326	\N	Vélieux	34
13793	327	\N	Vendargues	34
13794	328	\N	Vendémian	34
13795	329	\N	Vendres	34
13796	330	\N	Vérargues	34
13797	331	\N	Verreries-de-Moussans	34
13798	332	\N	Vias	34
13799	333	\N	Vic-la-Gardiole	34
13800	334	\N	Vieussan	34
13801	335	\N	Villemagne-l'Argentière	34
13802	336	\N	Villeneuve-lès-Béziers	34
13803	337	\N	Villeneuve-lès-Maguelone	34
13804	338	\N	Villeneuvette	34
13805	339	\N	Villespassans	34
13806	340	\N	Villetelle	34
13807	341	\N	Villeveyrac	34
13808	342	\N	Viols-en-Laval	34
13809	343	\N	Viols-le-Fort	34
13810	344	\N	Grande-Motte	34
13811	001	\N	Acigné	35
13812	002	\N	Amanlis	35
13813	003	\N	Andouillé-Neuville	35
13814	004	\N	Antrain	35
13815	005	\N	Arbrissel	35
13816	006	\N	Argentré-du-Plessis	35
13817	007	\N	Aubigné	35
13818	008	\N	Availles-sur-Seiche	35
13819	009	\N	Baguer-Morvan	35
13820	010	\N	Baguer-Pican	35
13821	011	\N	Baillé	35
13822	012	\N	Bain-de-Bretagne	35
13823	013	\N	Bains-sur-Oust	35
13824	014	\N	Bais	35
13825	015	\N	Balazé	35
13826	016	\N	Baulon	35
13827	017	\N	Baussaine	35
13828	018	\N	Bazouge-du-Désert	35
13829	019	\N	Bazouges-la-Pérouse	35
13830	021	\N	Beaucé	35
13831	022	\N	Bécherel	35
13832	023	\N	Bédée	35
13833	024	\N	Betton	35
13834	025	\N	Billé	35
13835	026	\N	Bléruais	35
13836	027	\N	Boisgervilly	35
13837	028	\N	Boistrudan	35
13838	029	\N	Bonnemain	35
13839	030	\N	Bosse-de-Bretagne	35
13840	031	\N	Bouëxière	35
13841	032	\N	Bourgbarré	35
13842	033	\N	Bourg-des-Comptes	35
13843	034	\N	Boussac	35
13844	035	\N	Bovel	35
13845	037	\N	Bréal-sous-Montfort	35
13846	038	\N	Bréal-sous-Vitré	35
13847	039	\N	Brécé	35
13848	040	\N	Breteil	35
13849	041	\N	Brie	35
13850	042	\N	Brielles	35
13851	044	\N	Broualan	35
13852	045	\N	Bruc-sur-Aff	35
13853	046	\N	Brulais	35
13854	047	\N	Bruz	35
13855	048	\N	Campel	35
13856	049	\N	Cancale	35
13857	050	\N	Cardroc	35
13858	051	\N	Cesson-Sévigné	35
13859	052	\N	Champeaux	35
13860	053	\N	Chancé	35
13861	054	\N	Chanteloup	35
13862	055	\N	Chantepie	35
13863	056	\N	Chapelle-aux-Filtzméens	35
13864	057	\N	Chapelle-Bouëxic	35
13865	058	\N	Chapelle-Chaussée	35
13866	059	\N	Chapelle-des-Fougeretz	35
13867	060	\N	Chapelle-du-Lou	35
13868	061	\N	Chapelle-Erbrée	35
13869	062	\N	Chapelle-Janson	35
13870	063	\N	Chapelle-Saint-Aubert	35
13871	064	\N	Chapelle-de-Brain	35
13872	065	\N	Chapelle-Thouarault	35
13873	066	\N	Chartres-de-Bretagne	35
13874	067	\N	Chasné-sur-Illet	35
13875	068	\N	Châteaubourg	35
13876	069	\N	Châteaugiron	35
13877	070	\N	Châteauneuf-d'Ille-et-Vilaine	35
13878	071	\N	Châtellier	35
13879	072	\N	Châtillon-en-Vendelais	35
13880	075	\N	Chauvigné	35
13881	076	\N	Chavagne	35
13882	077	\N	Chelun	35
13883	078	\N	Cherrueix	35
13884	079	\N	Chevaigné	35
13885	080	\N	Cintré	35
13886	081	\N	Clayes	35
13887	082	\N	Coësmes	35
13888	083	\N	Coglès	35
13889	084	\N	Comblessac	35
13890	085	\N	Combourg	35
13891	086	\N	Combourtillé	35
13892	087	\N	Cornillé	35
13893	088	\N	Corps-Nuds	35
13894	089	\N	Couyère	35
13895	090	\N	Crevin	35
13896	091	\N	Crouais	35
13897	092	\N	Cuguen	35
13898	093	\N	Dinard	35
13899	094	\N	Dingé	35
13900	095	\N	Dol-de-Bretagne	35
13901	096	\N	Domagné	35
13902	097	\N	Domalain	35
13903	098	\N	Dominelais	35
13904	099	\N	Domloup	35
13905	100	\N	Dompierre-du-Chemin	35
13906	101	\N	Dourdain	35
13907	102	\N	Drouges	35
13908	103	\N	Eancé	35
13909	104	\N	Epiniac	35
13910	105	\N	Erbrée	35
13911	106	\N	Ercé-en-Lamée	35
13912	107	\N	Ercé-près-Liffré	35
13913	108	\N	Essé	35
13914	109	\N	Étrelles	35
13915	110	\N	Feins	35
13916	111	\N	Ferré	35
13917	112	\N	Fleurigné	35
13918	113	\N	Fontenelle	35
13919	114	\N	Forges-la-Forêt	35
13920	115	\N	Fougères	35
13921	116	\N	Fresnais	35
13922	117	\N	Gaël	35
13923	118	\N	Gahard	35
13924	119	\N	Gennes-sur-Seiche	35
13925	120	\N	Gévezé	35
13926	121	\N	Gosné	35
13927	122	\N	Gouesnière	35
13928	123	\N	Goven	35
13929	124	\N	Grand-Fougeray	35
13930	125	\N	Guerche-de-Bretagne	35
13931	126	\N	Guichen	35
13932	127	\N	Guignen	35
13933	128	\N	Guipel	35
13934	129	\N	Guipry	35
13935	130	\N	Hédé	35
13936	131	\N	Hermitage	35
13937	132	\N	Hirel	35
13938	133	\N	Iffendic	35
13939	134	\N	Iffs	35
13940	135	\N	Irodouër	35
13941	136	\N	Janzé	35
13942	137	\N	Javené	35
13943	138	\N	Laignelet	35
13944	139	\N	Laillé	35
13945	140	\N	Lalleu	35
13946	141	\N	Landavran	35
13947	142	\N	Landéan	35
13948	143	\N	Landujan	35
13949	144	\N	Langan	35
13950	145	\N	Langon	35
13951	146	\N	Langouet	35
13952	147	\N	Lanhélin	35
13953	148	\N	Lanrigan	35
13954	149	\N	Lassy	35
13955	150	\N	Lécousse	35
13956	151	\N	Lieuron	35
13957	152	\N	Liffré	35
13958	153	\N	Lillemer	35
13959	154	\N	Livré-sur-Changeon	35
13960	155	\N	Lohéac	35
13961	156	\N	Longaulnay	35
13962	157	\N	Loroux	35
13963	158	\N	Lou-du-Lac	35
13964	159	\N	Lourmais	35
13965	160	\N	Loutehel	35
13966	161	\N	Louvigné-de-Bais	35
13967	162	\N	Louvigné-du-Désert	35
13968	163	\N	Luitré	35
13969	164	\N	Marcillé-Raoul	35
13970	165	\N	Marcillé-Robert	35
13971	166	\N	Marpiré	35
13972	167	\N	Martigné-Ferchaud	35
13973	168	\N	Maure-de-Bretagne	35
13974	169	\N	Maxent	35
13975	170	\N	Mecé	35
13976	171	\N	Médréac	35
13977	172	\N	Meillac	35
13978	173	\N	Melesse	35
13979	174	\N	Mellé	35
13980	175	\N	Mernel	35
13981	176	\N	Messac	35
13982	177	\N	Mézière	35
13983	178	\N	Mézières-sur-Couesnon	35
13984	179	\N	Miniac-Morvan	35
13985	180	\N	Miniac-sous-Bécherel	35
13986	181	\N	Minihic-sur-Rance	35
13987	183	\N	Mondevert	35
13988	184	\N	Montauban-de-Bretagne	35
13989	185	\N	Montautour	35
13990	186	\N	Mont-Dol	35
13991	187	\N	Monterfil	35
13992	188	\N	Montfort-sur-Meu	35
13993	189	\N	Montgermont	35
13994	190	\N	Monthault	35
13995	191	\N	Montours	35
13996	192	\N	Montreuil-des-Landes	35
13997	193	\N	Montreuil-le-Gast	35
13998	194	\N	Montreuil-sous-Pérouse	35
13999	195	\N	Montreuil-sur-Ille	35
14000	196	\N	Mordelles	35
14001	197	\N	Mouazé	35
14002	198	\N	Moulins	35
14003	199	\N	Moussé	35
14004	200	\N	Moutiers	35
14005	201	\N	Muel	35
14006	202	\N	Noë-Blanche	35
14007	203	\N	Nouaye	35
14008	204	\N	Nouvoitou	35
14009	205	\N	Noyal-sous-Bazouges	35
14010	206	\N	Noyal-Châtillon-sur-Seiche	35
14011	207	\N	Noyal-sur-Vilaine	35
14012	208	\N	Orgères	35
14013	209	\N	Ossé	35
14014	210	\N	Pacé	35
14015	211	\N	Paimpont	35
14016	212	\N	Pancé	35
14017	214	\N	Parcé	35
14018	215	\N	Parigné	35
14019	216	\N	Parthenay-de-Bretagne	35
14020	217	\N	Pertre	35
14021	218	\N	Petit-Fougeray	35
14022	219	\N	Pipriac	35
14023	220	\N	Piré-sur-Seiche	35
14024	221	\N	Pléchâtel	35
14025	222	\N	Pleine-Fougères	35
14026	223	\N	Plélan-le-Grand	35
14027	224	\N	Plerguer	35
14028	225	\N	Plesder	35
14029	226	\N	Pleugueneuc	35
14030	227	\N	Pleumeleuc	35
14031	228	\N	Pleurtuit	35
14032	229	\N	Pocé-les-Bois	35
14033	230	\N	Poilley	35
14034	231	\N	Poligné	35
14035	232	\N	Princé	35
14036	233	\N	Québriac	35
14037	234	\N	Quédillac	35
14038	235	\N	Rannée	35
14039	236	\N	Redon	35
14040	237	\N	Renac	35
14041	238	\N	Rennes	35
14042	239	\N	Retiers	35
14043	240	\N	Rheu	35
14044	241	\N	Richardais	35
14045	242	\N	Rimou	35
14046	243	\N	Romagné	35
14047	244	\N	Romazy	35
14048	245	\N	Romillé	35
14049	246	\N	Roz-Landrieux	35
14050	247	\N	Roz-sur-Couesnon	35
14051	248	\N	Sains	35
14052	249	\N	Sainte-Anne-sur-Vilaine	35
14053	250	\N	Saint-Armel	35
14054	251	\N	Saint-Aubin-d'Aubigné	35
14055	252	\N	Saint-Aubin-des-Landes	35
14056	253	\N	Saint-Aubin-du-Cormier	35
14057	254	\N	Saint-Aubin-du-Pavail	35
14058	255	\N	Saint-Benoît-des-Ondes	35
14059	256	\N	Saint-Briac-sur-Mer	35
14060	257	\N	Saint-Brice-en-Coglès	35
14061	258	\N	Saint-Brieuc-des-Iffs	35
14062	259	\N	Saint-Broladre	35
14063	260	\N	Saint-Christophe-des-Bois	35
14064	261	\N	Saint-Christophe-de-Valains	35
14065	262	\N	Sainte-Colombe	35
14066	263	\N	Saint-Coulomb	35
14067	264	\N	Saint-Didier	35
14068	265	\N	Saint-Domineuc	35
14069	266	\N	Saint-Erblon	35
14070	267	\N	Saint-Étienne-en-Coglès	35
14071	268	\N	Saint-Ganton	35
14072	269	\N	Saint-Georges-de-Chesné	35
14073	270	\N	Saint-Georges-de-Gréhaigne	35
14074	271	\N	Saint-Georges-de-Reintembault	35
14075	272	\N	Saint-Germain-du-Pinel	35
14076	273	\N	Saint-Germain-en-Coglès	35
14077	274	\N	Saint-Germain-sur-Ille	35
14078	275	\N	Saint-Gilles	35
14079	276	\N	Saint-Gondran	35
14080	277	\N	Saint-Gonlay	35
14081	278	\N	Saint-Grégoire	35
14082	279	\N	Saint-Guinoux	35
14083	280	\N	Saint-Hilaire-des-Landes	35
14084	281	\N	Saint-Jacques-de-la-Lande	35
14085	282	\N	Saint-Jean-sur-Couesnon	35
14086	283	\N	Saint-Jean-sur-Vilaine	35
14087	284	\N	Saint-Jouan-des-Guérets	35
14088	285	\N	Saint-Just	35
14089	286	\N	Saint-Léger-des-Prés	35
14090	287	\N	Saint-Lunaire	35
14091	288	\N	Saint-Malo	35
14092	289	\N	Saint-Malo-de-Phily	35
14093	290	\N	Saint-Malon-sur-Mel	35
14094	291	\N	Saint-Marcan	35
14095	292	\N	Saint-Marc-le-Blanc	35
14096	293	\N	Saint-Marc-sur-Couesnon	35
14097	294	\N	Sainte-Marie	35
14098	295	\N	Saint-Maugan	35
14099	296	\N	Saint-Médard-sur-Ille	35
14100	297	\N	Saint-Méen-le-Grand	35
14101	299	\N	Saint-Méloir-des-Ondes	35
14102	300	\N	Saint-M'Hervé	35
14103	301	\N	Saint-M'Hervon	35
14104	302	\N	Saint-Onen-la-Chapelle	35
14105	303	\N	Saint-Ouen-la-Rouërie	35
14106	304	\N	Saint-Ouen-des-Alleux	35
14107	305	\N	Saint-Péran	35
14108	306	\N	Saint-Père	35
14109	307	\N	Saint-Pern	35
14110	308	\N	Saint-Pierre-de-Plesguen	35
14111	309	\N	Saint-Rémy-du-Plain	35
14112	310	\N	Saint-Sauveur-des-Landes	35
14113	311	\N	Saint-Séglin	35
14114	312	\N	Saint-Senoux	35
14115	314	\N	Saint-Suliac	35
14116	315	\N	Saint-Sulpice-la-Forêt	35
14117	316	\N	Saint-Sulpice-des-Landes	35
14118	318	\N	Saint-Thual	35
14119	319	\N	Saint-Thurial	35
14120	320	\N	Saint-Uniac	35
14121	321	\N	Saulnières	35
14122	322	\N	Sel-de-Bretagne	35
14123	323	\N	Selle-en-Coglès	35
14124	324	\N	Selle-en-Luitré	35
14125	325	\N	Selle-Guerchaise	35
14126	326	\N	Sens-de-Bretagne	35
14127	327	\N	Servon-sur-Vilaine	35
14128	328	\N	Sixt-sur-Aff	35
14129	329	\N	Sougéal	35
14130	330	\N	Taillis	35
14131	331	\N	Talensac	35
14132	332	\N	Teillay	35
14133	333	\N	Theil-de-Bretagne	35
14134	334	\N	Thorigné-Fouillard	35
14135	335	\N	Thourie	35
14136	336	\N	Tiercent	35
14137	337	\N	Tinténiac	35
14138	338	\N	Torcé	35
14139	339	\N	Trans-la-Forêt	35
14140	340	\N	Treffendel	35
14141	341	\N	Tremblay	35
14142	342	\N	Trémeheuc	35
14143	343	\N	Tresbuf	35
14144	344	\N	Tressé	35
14145	345	\N	Trévérien	35
14146	346	\N	Trimer	35
14147	347	\N	Val-d'Izé	35
14148	348	\N	Vendel	35
14149	350	\N	Vergéal	35
14150	351	\N	Verger	35
14151	352	\N	Vern-sur-Seiche	35
14152	353	\N	Vezin-le-Coquet	35
14153	354	\N	Vieux-Viel	35
14154	355	\N	Vieux-Vy-sur-Couesnon	35
14155	356	\N	Vignoc	35
14156	357	\N	Villamée	35
14157	358	\N	Ville-ès-Nonais	35
14158	359	\N	Visseiche	35
14159	360	\N	Vitré	35
14160	361	\N	Vivier-sur-Mer	35
14161	362	\N	Tronchet	35
14162	363	\N	Pont-Péan	35
14163	001	\N	Aigurande	36
14164	002	\N	Aize	36
14165	003	\N	Ambrault	36
14166	004	\N	Anjouin	36
14167	005	\N	Ardentes	36
14168	006	\N	Argenton-sur-Creuse	36
14169	007	\N	Argy	36
14170	008	\N	Arpheuilles	36
14171	009	\N	Arthon	36
14172	010	\N	Azay-le-Ferron	36
14173	011	\N	Bagneux	36
14174	012	\N	Baraize	36
14175	013	\N	Baudres	36
14176	014	\N	Bazaiges	36
14177	015	\N	Beaulieu	36
14178	016	\N	Bélâbre	36
14179	017	\N	Berthenoux	36
14180	018	\N	Blanc	36
14181	019	\N	Bommiers	36
14182	020	\N	Bonneuil	36
14183	021	\N	Bordes	36
14184	022	\N	Bouesse	36
14185	023	\N	Bouges-le-Château	36
14186	024	\N	Bretagne	36
14187	025	\N	Briantes	36
14188	026	\N	Brion	36
14189	027	\N	Brives	36
14190	028	\N	Buxerette	36
14191	029	\N	Buxeuil	36
14192	030	\N	Buxières-d'Aillac	36
14193	031	\N	Buzançais	36
14194	032	\N	Ceaulmont	36
14195	033	\N	Celon	36
14196	034	\N	Chabris	36
14197	035	\N	Chaillac	36
14198	036	\N	Chalais	36
14199	037	\N	Champenoise	36
14200	038	\N	Champillet	36
14201	040	\N	Chapelle-Orthemale	36
14202	041	\N	Chapelle-Saint-Laurian	36
14203	042	\N	Chasseneuil	36
14204	043	\N	Chassignolles	36
14205	044	\N	Châteauroux	36
14206	045	\N	Châtillon-sur-Indre	36
14207	046	\N	Châtre	36
14208	047	\N	Châtre-Langlin	36
14209	048	\N	Chavin	36
14210	049	\N	Chazelet	36
14211	050	\N	Chezelles	36
14212	051	\N	Chitray	36
14213	052	\N	Chouday	36
14214	053	\N	Ciron	36
14215	054	\N	Cléré-du-Bois	36
14216	055	\N	Clion	36
14217	056	\N	Cluis	36
14218	057	\N	Coings	36
14219	058	\N	Concremiers	36
14220	059	\N	Condé	36
14221	060	\N	Crevant	36
14222	061	\N	Crozon-sur-Vauvre	36
14223	062	\N	Cuzion	36
14224	063	\N	Déols	36
14225	064	\N	Diors	36
14226	065	\N	Diou	36
14227	066	\N	Douadic	36
14228	067	\N	Dunet	36
14229	068	\N	Dun-le-Poëlier	36
14230	069	\N	Écueillé	36
14231	070	\N	Éguzon-Chantôme	36
14232	071	\N	Étrechet	36
14233	072	\N	Faverolles	36
14234	073	\N	Feusines	36
14235	074	\N	Fléré-la-Rivière	36
14236	075	\N	Fontenay	36
14237	076	\N	Fontgombault	36
14238	077	\N	Fontguenand	36
14239	078	\N	Fougerolles	36
14240	079	\N	Francillon	36
14241	080	\N	Frédille	36
14242	081	\N	Gargilesse-Dampierre	36
14243	082	\N	Gehée	36
14244	083	\N	Giroux	36
14245	084	\N	Gournay	36
14246	085	\N	Guilly	36
14247	086	\N	Heugnes	36
14248	087	\N	Ingrandes	36
14249	088	\N	Issoudun	36
14250	089	\N	Jeu-les-Bois	36
14251	090	\N	Jeu-Maloches	36
14252	091	\N	Lacs	36
14253	092	\N	Langé	36
14254	093	\N	Levroux	36
14255	094	\N	Lignac	36
14256	095	\N	Lignerolles	36
14257	096	\N	Lingé	36
14258	097	\N	Liniez	36
14259	098	\N	Lizeray	36
14260	099	\N	Lourdoueix-Saint-Michel	36
14261	100	\N	Lourouer-Saint-Laurent	36
14262	101	\N	Luant	36
14263	102	\N	Luçay-le-Libre	36
14264	103	\N	Luçay-le-Mâle	36
14265	104	\N	Lurais	36
14266	105	\N	Lureuil	36
14267	106	\N	Luzeret	36
14268	107	\N	Lye	36
14269	108	\N	Lys-Saint-Georges	36
14270	109	\N	Magny	36
14271	110	\N	Maillet	36
14272	111	\N	Malicornay	36
14273	112	\N	Mâron	36
14274	113	\N	Martizay	36
14275	114	\N	Mauvières	36
14276	115	\N	Menetou-sur-Nahon	36
14277	116	\N	Ménétréols-sous-Vatan	36
14278	117	\N	Menoux	36
14279	118	\N	Méobecq	36
14280	119	\N	Mérigny	36
14281	120	\N	Mers-sur-Indre	36
14282	121	\N	Meunet-Planches	36
14283	122	\N	Meunet-sur-Vatan	36
14284	123	\N	Mézières-en-Brenne	36
14285	124	\N	Migné	36
14286	125	\N	Migny	36
14287	126	\N	Montchevrier	36
14288	127	\N	Montgivray	36
14289	128	\N	Montierchaume	36
14290	129	\N	Montipouret	36
14291	130	\N	Montlevicq	36
14292	131	\N	Mosnay	36
14293	132	\N	Motte-Feuilly	36
14294	133	\N	Mouhers	36
14295	134	\N	Mouhet	36
14296	135	\N	Moulins-sur-Céphons	36
14297	136	\N	Murs	36
14298	137	\N	Néons-sur-Creuse	36
14299	138	\N	Néret	36
14300	139	\N	Neuillay-les-Bois	36
14301	140	\N	Neuvy-Pailloux	36
14302	141	\N	Neuvy-Saint-Sépulchre	36
14303	142	\N	Niherne	36
14304	143	\N	Nohant-Vic	36
14305	144	\N	Nuret-le-Ferron	36
14306	145	\N	Obterre	36
14307	146	\N	Orsennes	36
14308	147	\N	Orville	36
14309	148	\N	Oulches	36
14310	149	\N	Palluau-sur-Indre	36
14311	150	\N	Parnac	36
14312	151	\N	Parpeçay	36
14313	152	\N	Paudy	36
14314	153	\N	Paulnay	36
14315	154	\N	Pêchereau	36
14316	155	\N	Pellevoisin	36
14317	156	\N	Pérassay	36
14318	157	\N	Pérouille	36
14319	158	\N	Badecon-le-Pin	36
14320	159	\N	Poinçonnet	36
14321	160	\N	Pommiers	36
14322	161	\N	Pont-Chrétien-Chabenet	36
14323	162	\N	Poulaines	36
14324	163	\N	Pouligny-Notre-Dame	36
14325	164	\N	Pouligny-Saint-Martin	36
14326	165	\N	Pouligny-Saint-Pierre	36
14327	166	\N	Préaux	36
14328	167	\N	Preuilly-la-Ville	36
14329	168	\N	Prissac	36
14330	169	\N	Pruniers	36
14331	170	\N	Reboursin	36
14332	171	\N	Reuilly	36
14333	172	\N	Rivarennes	36
14334	173	\N	Rosnay	36
14335	174	\N	Roussines	36
14336	175	\N	Rouvres-les-Bois	36
14337	176	\N	Ruffec	36
14338	177	\N	Sacierges-Saint-Martin	36
14339	178	\N	Saint-Aigny	36
14340	179	\N	Saint-Aoustrille	36
14341	180	\N	Saint-Août	36
14342	181	\N	Saint-Aubin	36
14343	182	\N	Saint-Benoît-du-Sault	36
14344	183	\N	Sainte-Cécile	36
14345	184	\N	Saint-Chartier	36
14346	185	\N	Saint-Christophe-en-Bazelle	36
14347	186	\N	Saint-Christophe-en-Boucherie	36
14348	187	\N	Saint-Civran	36
14349	188	\N	Saint-Cyran-du-Jambot	36
14350	189	\N	Saint-Denis-de-Jouhet	36
14351	190	\N	Sainte-Fauste	36
14352	191	\N	Saint-Florentin	36
14353	192	\N	Saint-Gaultier	36
14354	193	\N	Sainte-Gemme	36
14355	194	\N	Saint-Genou	36
14356	195	\N	Saint-Georges-sur-Arnon	36
14357	196	\N	Saint-Gilles	36
14358	197	\N	Saint-Hilaire-sur-Benaize	36
14359	198	\N	Saint-Lactencin	36
14360	199	\N	Sainte-Lizaigne	36
14361	200	\N	Saint-Marcel	36
14362	201	\N	Saint-Martin-de-Lamps	36
14363	202	\N	Saint-Maur	36
14364	203	\N	Saint-Médard	36
14365	204	\N	Saint-Michel-en-Brenne	36
14366	205	\N	Saint-Pierre-de-Jards	36
14367	206	\N	Saint-Pierre-de-Lamps	36
14368	207	\N	Saint-Plantaire	36
14369	208	\N	Sainte-Sévère-sur-Indre	36
14370	209	\N	Saint-Valentin	36
14371	210	\N	Sarzay	36
14372	211	\N	Sassierges-Saint-Germain	36
14373	212	\N	Saulnay	36
14374	213	\N	Sauzelles	36
14375	214	\N	Sazeray	36
14376	215	\N	Ségry	36
14377	216	\N	Selles-sur-Nahon	36
14378	217	\N	Sembleçay	36
14379	218	\N	Sougé	36
14380	219	\N	Tendu	36
14381	220	\N	Thenay	36
14382	221	\N	Thevet-Saint-Julien	36
14383	222	\N	Thizay	36
14384	223	\N	Tilly	36
14385	224	\N	Tournon-Saint-Martin	36
14386	225	\N	Tranger	36
14387	226	\N	Tranzault	36
14388	227	\N	Urciers	36
14389	228	\N	Valençay	36
14390	229	\N	Varennes-sur-Fouzon	36
14391	230	\N	Vatan	36
14392	231	\N	Velles	36
14393	232	\N	Venduvres	36
14394	233	\N	Vernelle	36
14395	234	\N	Verneuil-sur-Igneraie	36
14396	235	\N	Veuil	36
14397	236	\N	Vicq-Exemplet	36
14398	237	\N	Vicq-sur-Nahon	36
14399	238	\N	Vigoulant	36
14400	239	\N	Vigoux	36
14401	240	\N	Vijon	36
14402	241	\N	Villedieu-sur-Indre	36
14403	242	\N	Villegongis	36
14404	243	\N	Villegouin	36
14405	244	\N	Villentrois	36
14406	245	\N	Villers-les-Ormes	36
14407	246	\N	Villiers	36
14408	247	\N	Vineuil	36
14409	248	\N	Vouillon	36
14410	001	\N	Abilly	37
14411	002	\N	Ambillou	37
14412	003	\N	Amboise	37
14413	004	\N	Anché	37
14414	005	\N	Antogny le Tillac	37
14415	006	\N	Artannes-sur-Indre	37
14416	007	\N	Assay	37
14417	008	\N	Athée-sur-Cher	37
14418	009	\N	Autrèche	37
14419	010	\N	Auzouer-en-Touraine	37
14420	011	\N	Avoine	37
14421	012	\N	Avon-les-Roches	37
14422	013	\N	Avrillé-les-Ponceaux	37
14423	014	\N	Azay-le-Rideau	37
14424	015	\N	Azay-sur-Cher	37
14425	016	\N	Azay-sur-Indre	37
14426	018	\N	Ballan-Miré	37
14427	019	\N	Barrou	37
14428	020	\N	Beaulieu-lès-Loches	37
14429	021	\N	Beaumont-la-Ronce	37
14430	022	\N	Beaumont-en-Véron	37
14431	023	\N	Beaumont-Village	37
14432	024	\N	Benais	37
14433	025	\N	Berthenay	37
14434	026	\N	Betz-le-Château	37
14435	027	\N	Bléré	37
14436	028	\N	Bossay-sur-Claise	37
14437	029	\N	Bossée	37
14438	030	\N	Boulay	37
14439	031	\N	Bourgueil	37
14440	032	\N	Bournan	37
14441	033	\N	Boussay	37
14442	034	\N	Braslou	37
14443	035	\N	Braye-sous-Faye	37
14444	036	\N	Braye-sur-Maulne	37
14445	037	\N	Brèches	37
14446	038	\N	Bréhémont	37
14447	039	\N	Bridoré	37
14448	040	\N	Brizay	37
14449	041	\N	Bueil-en-Touraine	37
14450	042	\N	Candes-Saint-Martin	37
14451	043	\N	Cangey	37
14452	044	\N	Celle-Guenand	37
14453	045	\N	Celle-Saint-Avant	37
14454	046	\N	Céré-la-Ronde	37
14455	047	\N	Cerelles	37
14456	048	\N	Chambon	37
14457	049	\N	Chambourg-sur-Indre	37
14458	050	\N	Chambray-lès-Tours	37
14459	051	\N	Champigny-sur-Veude	37
14460	052	\N	Chançay	37
14461	053	\N	Chanceaux-près-Loches	37
14462	054	\N	Chanceaux-sur-Choisille	37
14463	055	\N	Channay-sur-Lathan	37
14464	056	\N	Chapelle-aux-Naux	37
14465	057	\N	Chapelle-Blanche-Saint-Martin	37
14466	058	\N	Chapelle-sur-Loire	37
14467	059	\N	Charentilly	37
14468	060	\N	Chargé	37
14469	061	\N	Charnizay	37
14470	062	\N	Château-la-Vallière	37
14471	063	\N	Château-Renault	37
14472	064	\N	Chaumussay	37
14473	065	\N	Chaveignes	37
14474	066	\N	Chédigny	37
14475	067	\N	Cheillé	37
14476	068	\N	Chemillé-sur-Dême	37
14477	069	\N	Chemillé-sur-Indrois	37
14478	070	\N	Chenonceaux	37
14479	071	\N	Chezelles	37
14480	072	\N	Chinon	37
14481	073	\N	Chisseaux	37
14482	074	\N	Chouzé-sur-Loire	37
14483	075	\N	Cigogné	37
14484	076	\N	Cinais	37
14485	077	\N	Cinq-Mars-la-Pile	37
14486	078	\N	Ciran	37
14487	079	\N	Civray-de-Touraine	37
14488	080	\N	Civray-sur-Esves	37
14489	081	\N	Cléré-les-Pins	37
14490	082	\N	Continvoir	37
14491	083	\N	Cormery	37
14492	084	\N	Couesmes	37
14493	085	\N	Courçay	37
14494	086	\N	Courcelles-de-Touraine	37
14495	087	\N	Courcoué	37
14496	088	\N	Couziers	37
14497	089	\N	Cravant-les-Côteaux	37
14498	090	\N	Crissay-sur-Manse	37
14499	091	\N	Croix-en-Touraine	37
14500	092	\N	Crotelles	37
14501	093	\N	Crouzilles	37
14502	094	\N	Cussay	37
14503	095	\N	Dame-Marie-les-Bois	37
14504	096	\N	Dierre	37
14505	097	\N	Dolus-le-Sec	37
14506	098	\N	Draché	37
14507	099	\N	Druye	37
14508	100	\N	Épeigné-les-Bois	37
14509	101	\N	Épeigné-sur-Dême	37
14510	102	\N	Essards	37
14511	103	\N	Esves-le-Moutier	37
14512	104	\N	Esvres	37
14513	105	\N	Faye-la-Vineuse	37
14514	106	\N	Ferrière	37
14515	107	\N	Ferrière-Larçon	37
14516	108	\N	Ferrière-sur-Beaulieu	37
14517	109	\N	Fondettes	37
14518	110	\N	Francueil	37
14519	111	\N	Genillé	37
14520	112	\N	Gizeux	37
14521	113	\N	Grand-Pressigny	37
14522	114	\N	Guerche	37
14523	115	\N	Descartes	37
14524	116	\N	Hermites	37
14525	117	\N	Hommes	37
14526	118	\N	Huismes	37
14527	119	\N	Île-Bouchard	37
14528	120	\N	Ingrandes-de-Touraine	37
14529	121	\N	Jaulnay	37
14530	122	\N	Joué-lès-Tours	37
14531	123	\N	Langeais	37
14532	124	\N	Larçay	37
14533	125	\N	Lémeré	37
14534	126	\N	Lerné	37
14535	127	\N	Liège	37
14536	128	\N	Lignières-de-Touraine	37
14537	129	\N	Ligré	37
14538	130	\N	Ligueil	37
14539	131	\N	Limeray	37
14540	132	\N	Loches	37
14541	133	\N	Loché-sur-Indrois	37
14542	134	\N	Louans	37
14543	135	\N	Louestault	37
14544	136	\N	Louroux	37
14545	137	\N	Lublé	37
14546	138	\N	Lussault-sur-Loire	37
14547	139	\N	Luynes	37
14548	140	\N	Luzé	37
14549	141	\N	Luzillé	37
14550	142	\N	Maillé	37
14551	143	\N	Manthelan	37
14552	144	\N	Marçay	37
14553	145	\N	Marcé-sur-Esves	37
14554	146	\N	Marcilly-sur-Maulne	37
14555	147	\N	Marcilly-sur-Vienne	37
14556	148	\N	Marigny-Marmande	37
14557	149	\N	Marray	37
14558	150	\N	Mazières-de-Touraine	37
14559	151	\N	Membrolle-sur-Choisille	37
14560	152	\N	Mettray	37
14561	153	\N	Monnaie	37
14562	154	\N	Montbazon	37
14563	155	\N	Monthodon	37
14564	156	\N	Montlouis-sur-Loire	37
14565	157	\N	Montrésor	37
14566	158	\N	Montreuil-en-Touraine	37
14567	159	\N	Monts	37
14568	160	\N	Morand	37
14569	161	\N	Mosnes	37
14570	162	\N	Mouzay	37
14571	163	\N	Nazelles-Négron	37
14572	165	\N	Neuil	37
14573	166	\N	Neuillé-le-Lierre	37
14574	167	\N	Neuillé-Pont-Pierre	37
14575	168	\N	Neuilly-le-Brignon	37
14576	169	\N	Neuville-sur-Brenne	37
14577	170	\N	Neuvy-le-Roi	37
14578	171	\N	Noizay	37
14579	172	\N	Notre-Dame-d'Oé	37
14580	173	\N	Nouans-les-Fontaines	37
14581	174	\N	Nouâtre	37
14582	175	\N	Nouzilly	37
14583	176	\N	Noyant-de-Touraine	37
14584	177	\N	Orbigny	37
14585	178	\N	Panzoult	37
14586	179	\N	Parçay-Meslay	37
14587	180	\N	Parçay-sur-Vienne	37
14588	181	\N	Paulmy	37
14589	182	\N	Pernay	37
14590	183	\N	Perrusson	37
14591	184	\N	Petit-Pressigny	37
14592	185	\N	Pocé-sur-Cisse	37
14593	186	\N	Pont-de-Ruan	37
14594	187	\N	Ports	37
14595	188	\N	Pouzay	37
14596	189	\N	Preuilly-sur-Claise	37
14597	190	\N	Pussigny	37
14598	191	\N	Razines	37
14599	192	\N	Reignac-sur-Indre	37
14600	193	\N	Restigné	37
14601	194	\N	Reugny	37
14602	195	\N	Riche	37
14603	196	\N	Richelieu	37
14604	197	\N	Rigny-Ussé	37
14605	198	\N	Rillé	37
14606	199	\N	Rilly-sur-Vienne	37
14607	200	\N	Rivarennes	37
14608	201	\N	Rivière	37
14609	202	\N	Roche-Clermault	37
14610	203	\N	Rochecorbon	37
14611	204	\N	Rouziers-de-Touraine	37
14612	205	\N	Saché	37
14613	206	\N	Saint-Antoine-du-Rocher	37
14614	207	\N	Saint-Aubin-le-Dépeint	37
14615	208	\N	Saint-Avertin	37
14616	209	\N	Saint-Bauld	37
14617	210	\N	Saint-Benoît-la-Forêt	37
14618	211	\N	Saint-Branchs	37
14619	212	\N	Sainte-Catherine-de-Fierbois	37
14620	213	\N	Saint-Christophe-sur-le-Nais	37
14621	214	\N	Saint-Cyr-sur-Loire	37
14622	216	\N	Saint-Épain	37
14623	217	\N	Saint-Étienne-de-Chigny	37
14624	218	\N	Saint-Flovier	37
14625	219	\N	Saint-Genouph	37
14626	220	\N	Saint-Germain-sur-Vienne	37
14627	221	\N	Saint-Hippolyte	37
14628	222	\N	Saint-Jean-Saint-Germain	37
14629	223	\N	Saint-Laurent-de-Lin	37
14630	224	\N	Saint-Laurent-en-Gâtines	37
14631	225	\N	Saint-Martin-le-Beau	37
14632	226	\N	Sainte-Maure-de-Touraine	37
14633	227	\N	Saint-Michel-sur-Loire	37
14634	228	\N	Saint-Nicolas-de-Bourgueil	37
14635	229	\N	Saint-Nicolas-des-Motets	37
14636	230	\N	Saint-Ouen-les-Vignes	37
14637	231	\N	Saint-Paterne-Racan	37
14638	232	\N	Saint-Patrice	37
14639	233	\N	Saint-Pierre-des-Corps	37
14640	234	\N	Saint-Quentin-sur-Indrois	37
14641	236	\N	Saint-Règle	37
14642	237	\N	Saint-Roch	37
14643	238	\N	Saint-Senoch	37
14644	240	\N	Saunay	37
14645	241	\N	Savigné-sur-Lathan	37
14646	242	\N	Savigny-en-Véron	37
14647	243	\N	Savonnières	37
14648	244	\N	Sazilly	37
14649	245	\N	Semblançay	37
14650	246	\N	Sennevières	37
14651	247	\N	Sepmes	37
14652	248	\N	Seuilly	37
14653	249	\N	Sonzay	37
14654	250	\N	Sorigny	37
14655	251	\N	Souvigné	37
14656	252	\N	Souvigny-de-Touraine	37
14657	253	\N	Sublaines	37
14658	254	\N	Tauxigny	37
14659	255	\N	Tavant	37
14660	256	\N	Theneuil	37
14661	257	\N	Thilouze	37
14662	258	\N	Thizay	37
14663	259	\N	Tournon-Saint-Pierre	37
14664	260	\N	Tour-Saint-Gelin	37
14665	261	\N	Tours	37
14666	262	\N	Trogues	37
14667	263	\N	Truyes	37
14668	264	\N	Vallères	37
14669	265	\N	Varennes	37
14670	266	\N	Veigné	37
14671	267	\N	Véretz	37
14672	268	\N	Verneuil-le-Château	37
14673	269	\N	Verneuil-sur-Indre	37
14674	270	\N	Vernou-sur-Brenne	37
14675	271	\N	Villaines-les-Rochers	37
14676	272	\N	Villandry	37
14677	273	\N	Ville-aux-Dames	37
14678	274	\N	Villebourg	37
14679	275	\N	Villedômain	37
14680	276	\N	Villedômer	37
14681	277	\N	Villeloin-Coulangé	37
14682	278	\N	Villeperdue	37
14683	279	\N	Villiers-au-Bouin	37
14684	280	\N	Vou	37
14685	281	\N	Vouvray	37
14686	282	\N	Yzeures-sur-Creuse	37
14687	001	\N	Abrets	38
14688	002	\N	Adrets	38
14689	003	\N	Agnin	38
14690	004	\N	Albenc	38
14691	005	\N	Allemond	38
14692	006	\N	Allevard	38
14693	008	\N	Ambel	38
14694	009	\N	Anjou	38
14695	010	\N	Annoisin-Chatelans	38
14696	011	\N	Anthon	38
14697	012	\N	Aoste	38
14698	013	\N	Apprieu	38
14699	014	\N	Arandon	38
14700	015	\N	Artas	38
14701	016	\N	Arzay	38
14702	017	\N	Assieu	38
14703	018	\N	Auberives-en-Royans	38
14704	019	\N	Auberives-sur-Varèze	38
14705	020	\N	Auris	38
14706	021	\N	Autrans	38
14707	022	\N	Avenières	38
14708	023	\N	Avignonet	38
14709	024	\N	Badinières	38
14710	025	\N	Balbins	38
14711	026	\N	Balme-les-Grottes	38
14712	027	\N	Barraux	38
14713	028	\N	Bâtie-Divisin	38
14714	029	\N	Bâtie-Montgascon	38
14715	030	\N	Beaucroissant	38
14716	031	\N	Beaufin	38
14717	032	\N	Beaufort	38
14718	033	\N	Beaulieu	38
14719	034	\N	Beaurepaire	38
14720	035	\N	Beauvoir-de-Marc	38
14721	036	\N	Beauvoir-en-Royans	38
14722	037	\N	Bellegarde-Poussieu	38
14723	038	\N	Belmont	38
14724	039	\N	Bernin	38
14725	040	\N	Besse	38
14726	041	\N	Bessins	38
14727	042	\N	Bévenais	38
14728	043	\N	Bilieu	38
14729	044	\N	Biol	38
14730	045	\N	Biviers	38
14731	046	\N	Bizonnes	38
14732	047	\N	Blandin	38
14733	048	\N	Bonnefamille	38
14734	049	\N	Bossieu	38
14735	050	\N	Bouchage	38
14736	051	\N	Bougé-Chambalud	38
14737	052	\N	Bourg-d'Oisans	38
14738	053	\N	Bourgoin-Jallieu	38
14739	054	\N	Bouvesse-Quirieu	38
14740	055	\N	Brangues	38
14741	056	\N	Bressieux	38
14742	057	\N	Bresson	38
14743	058	\N	Brézins	38
14744	059	\N	Brié-et-Angonnes	38
14745	060	\N	Brion	38
14746	061	\N	Buisse	38
14747	062	\N	Buissière	38
14748	063	\N	Burcin	38
14749	064	\N	Cessieu	38
14750	065	\N	Châbons	38
14751	066	\N	Châlons	38
14752	067	\N	Chamagnieu	38
14753	068	\N	Champagnier	38
14754	069	\N	Champier	38
14755	070	\N	Champ-près-Froges	38
14756	071	\N	Champ-sur-Drac	38
14757	072	\N	Chanas	38
14758	073	\N	Chantelouve	38
14759	074	\N	Chantesse	38
14760	075	\N	Chapareillan	38
14761	076	\N	Chapelle-de-la-Tour	38
14762	077	\N	Chapelle-de-Surieu	38
14763	078	\N	Chapelle-du-Bard	38
14764	080	\N	Charancieu	38
14765	081	\N	Charantonnay	38
14766	082	\N	Charavines	38
14767	083	\N	Charette	38
14768	084	\N	Charnècles	38
14769	085	\N	Charvieu-Chavagneux	38
14770	086	\N	Chasselay	38
14771	087	\N	Chasse-sur-Rhône	38
14772	089	\N	Chassignieu	38
14773	090	\N	Château-Bernard	38
14774	091	\N	Châteauvilain	38
14775	092	\N	Châtelus	38
14776	093	\N	Châtenay	38
14777	094	\N	Châtonnay	38
14778	095	\N	Chatte	38
14779	097	\N	Chavanoz	38
14780	098	\N	Chélieu	38
14781	099	\N	Chevrières	38
14782	100	\N	Cheylas	38
14783	101	\N	Cheyssieu	38
14784	102	\N	Chèzeneuve	38
14785	103	\N	Chichilianne	38
14786	104	\N	Chimilin	38
14787	105	\N	Chirens	38
14788	106	\N	Cholonge	38
14789	107	\N	Chonas-l'Amballan	38
14790	108	\N	Choranche	38
14791	109	\N	Chozeau	38
14792	110	\N	Chuzelles	38
14793	111	\N	Claix	38
14794	112	\N	Clavans-en-Haut-Oisans	38
14795	113	\N	Clelles	38
14796	114	\N	Clonas-sur-Varèze	38
14797	115	\N	Saint-Martin-de-la-Cluze	38
14798	116	\N	Cognet	38
14799	117	\N	Cognin-les-Gorges	38
14800	118	\N	Colombe	38
14801	120	\N	Combe-de-Lancey	38
14802	121	\N	Commelle	38
14803	124	\N	Corbelin	38
14804	125	\N	Cordéac	38
14805	126	\N	Corenc	38
14806	127	\N	Cornillon-en-Trièves	38
14807	128	\N	Corps	38
14808	129	\N	Corrençon-en-Vercors	38
14809	130	\N	Côte-Saint-André	38
14810	131	\N	Côtes-d'Arey	38
14811	132	\N	Côtes-de-Corps	38
14812	133	\N	Coublevie	38
14813	134	\N	Cour-et-Buis	38
14814	135	\N	Courtenay	38
14815	136	\N	Crachier	38
14816	137	\N	Cras	38
14817	138	\N	Crémieu	38
14818	139	\N	Creys-Mépieu	38
14819	140	\N	Crolles	38
14820	141	\N	Culin	38
14821	144	\N	Diémoz	38
14822	145	\N	Dionay	38
14823	146	\N	Dizimieu	38
14824	147	\N	Doissin	38
14825	148	\N	Dolomieu	38
14826	149	\N	Domarin	38
14827	150	\N	Domène	38
14828	151	\N	Échirolles	38
14829	152	\N	Eclose	38
14830	153	\N	Engins	38
14831	154	\N	Entraigues	38
14832	155	\N	Entre-deux-Guiers	38
14833	156	\N	Éparres	38
14834	157	\N	Estrablin	38
14835	158	\N	Eybens	38
14836	159	\N	Eydoche	38
14837	160	\N	Eyzin-Pinet	38
14838	161	\N	Faramans	38
14839	162	\N	Faverges-de-la-Tour	38
14840	163	\N	Ferrière	38
14841	165	\N	Fitilieu	38
14842	166	\N	Flachère	38
14843	167	\N	Flachères	38
14844	169	\N	Fontaine	38
14845	170	\N	Fontanil-Cornillon	38
14846	171	\N	Forteresse	38
14847	172	\N	Four	38
14848	173	\N	Freney-d'Oisans	38
14849	174	\N	Frette	38
14850	175	\N	Froges	38
14851	176	\N	Frontonas	38
14852	177	\N	Garde	38
14853	179	\N	Gières	38
14854	180	\N	Gillonnay	38
14855	181	\N	Goncelin	38
14856	182	\N	Grand-Lemps	38
14857	183	\N	Granieu	38
14858	184	\N	Grenay	38
14859	185	\N	Grenoble	38
14860	186	\N	Gresse-en-Vercors	38
14861	187	\N	Gua	38
14862	188	\N	Herbeys	38
14863	189	\N	Heyrieux	38
14864	190	\N	Hières-sur-Amby	38
14865	191	\N	Huez	38
14866	192	\N	Hurtières	38
14867	193	\N	Isle-d'Abeau	38
14868	194	\N	Izeaux	38
14869	195	\N	Izeron	38
14870	197	\N	Janneyrias	38
14871	198	\N	Jarcieu	38
14872	199	\N	Jardin	38
14873	200	\N	Jarrie	38
14874	203	\N	Laffrey	38
14875	204	\N	Lalley	38
14876	205	\N	Lans-en-Vercors	38
14877	206	\N	Laval	38
14878	207	\N	Lavaldens	38
14879	208	\N	Lavars	38
14880	209	\N	Lentiol	38
14881	210	\N	Leyrieu	38
14882	211	\N	Lieudieu	38
14883	212	\N	Livet-et-Gavet	38
14884	213	\N	Longechenal	38
14885	214	\N	Lumbin	38
14886	215	\N	Luzinay	38
14887	216	\N	Malleval	38
14888	217	\N	Marcieu	38
14889	218	\N	Marcilloles	38
14890	219	\N	Marcollin	38
14891	221	\N	Marnans	38
14892	222	\N	Massieu	38
14893	223	\N	Maubec	38
14894	224	\N	Mayres-Savel	38
14895	225	\N	Méaudre	38
14896	226	\N	Mens	38
14897	228	\N	Merlas	38
14898	229	\N	Meylan	38
14899	230	\N	Meyrié	38
14900	231	\N	Meyrieu-les-Étangs	38
14901	232	\N	Meyssiès	38
14902	235	\N	Miribel-Lanchâtre	38
14903	236	\N	Miribel-les-Échelles	38
14904	237	\N	Mizoën	38
14905	238	\N	Moidieu-Détourbe	38
14906	239	\N	Moirans	38
14907	240	\N	Moissieu-sur-Dolon	38
14908	241	\N	Monestier-d'Ambel	38
14909	242	\N	Monestier-de-Clermont	38
14910	243	\N	Monestier-du-Percy	38
14911	244	\N	Monsteroux-Milieu	38
14912	245	\N	Montagne	38
14913	246	\N	Montagnieu	38
14914	247	\N	Montalieu-Vercieu	38
14915	248	\N	Montaud	38
14916	249	\N	Montbonnot-Saint-Martin	38
14917	250	\N	Montcarra	38
14918	252	\N	Montchaboud	38
14919	253	\N	Mont-de-Lans	38
14920	254	\N	Monteynard	38
14921	255	\N	Montfalcon	38
14922	256	\N	Montferrat	38
14923	257	\N	Montrevel	38
14924	258	\N	Mont-Saint-Martin	38
14925	259	\N	Montseveroux	38
14926	260	\N	Moras	38
14927	261	\N	Morestel	38
14928	262	\N	Morêtel-de-Mailles	38
14929	263	\N	Morette	38
14930	264	\N	Morte	38
14931	265	\N	Motte-d'Aveillans	38
14932	266	\N	Motte-Saint-Martin	38
14933	267	\N	Mottier	38
14934	268	\N	Moutaret	38
14935	269	\N	Mure	38
14936	270	\N	Murette	38
14937	271	\N	Murianette	38
14938	272	\N	Murinais	38
14939	273	\N	Nantes-en-Ratier	38
14940	274	\N	Nantoin	38
14941	275	\N	Serre-Nerpol	38
14942	276	\N	Nivolas-Vermelle	38
14943	277	\N	Notre-Dame-de-Commiers	38
14944	278	\N	Notre-Dame-de-l'Osier	38
14945	279	\N	Notre-Dame-de-Mésage	38
14946	280	\N	Notre-Dame-de-Vaulx	38
14947	281	\N	Noyarey	38
14948	282	\N	Optevoz	38
14949	283	\N	Oris-en-Rattier	38
14950	284	\N	Ornacieux	38
14951	285	\N	Ornon	38
14952	286	\N	Oulles	38
14953	287	\N	Oyeu	38
14954	288	\N	Oytier-Saint-Oblas	38
14955	289	\N	Oz	38
14956	290	\N	Pact	38
14957	291	\N	Pajay	38
14958	292	\N	Paladru	38
14959	293	\N	Panissage	38
14960	294	\N	Panossas	38
14961	295	\N	Parmilieu	38
14962	296	\N	Passage	38
14963	297	\N	Passins	38
14964	298	\N	Péage-de-Roussillon	38
14965	299	\N	Pellafol	38
14966	300	\N	Penol	38
14967	301	\N	Percy	38
14968	302	\N	Périer	38
14969	303	\N	Pierre	38
14970	304	\N	Pierre-Châtel	38
14971	305	\N	Pin	38
14972	306	\N	Pinsot	38
14973	307	\N	Pisieu	38
14974	308	\N	Plan	38
14975	309	\N	Poisat	38
14976	310	\N	Poliénas	38
14977	311	\N	Pommier-de-Beaurepaire	38
14978	312	\N	Pommiers-la-Placette	38
14979	313	\N	Ponsonnas	38
14980	314	\N	Pontcharra	38
14981	315	\N	Pont-de-Beauvoisin	38
14982	316	\N	Pont-de-Chéruy	38
14983	317	\N	Pont-de-Claix	38
14984	318	\N	Pont-Évêque	38
14985	319	\N	Pont-en-Royans	38
14986	320	\N	Porcieu-Amblagnieu	38
14987	321	\N	Prébois	38
14988	322	\N	Presles	38
14989	323	\N	Pressins	38
14990	324	\N	Primarette	38
14991	325	\N	Proveysieux	38
14992	326	\N	Prunières	38
14993	328	\N	Quaix-en-Chartreuse	38
14994	329	\N	Quet-en-Beaumont	38
14995	330	\N	Quincieu	38
14996	331	\N	Réaumont	38
14997	332	\N	Renage	38
14998	333	\N	Rencurel	38
14999	334	\N	Revel	38
15000	335	\N	Revel-Tourdan	38
15001	336	\N	Reventin-Vaugris	38
15002	337	\N	Rives	38
15003	338	\N	Rivière	38
15004	339	\N	Roche	38
15005	340	\N	Roches-de-Condrieu	38
15006	341	\N	Rochetoirin	38
15007	342	\N	Roissard	38
15008	343	\N	Romagnieu	38
15009	344	\N	Roussillon	38
15010	345	\N	Rovon	38
15011	346	\N	Royas	38
15012	347	\N	Roybon	38
15013	348	\N	Ruy	38
15014	349	\N	Sablons	38
15015	350	\N	Sainte-Agnès	38
15016	351	\N	Saint-Agnin-sur-Bion	38
15017	352	\N	Saint-Alban-de-Roche	38
15018	353	\N	Saint-Alban-du-Rhône	38
15019	354	\N	Saint-Albin-de-Vaulserre	38
15020	355	\N	Saint-Andéol	38
15021	356	\N	Saint-André-en-Royans	38
15022	357	\N	Saint-André-le-Gaz	38
15023	358	\N	Sainte-Anne-sur-Gervonde	38
15024	359	\N	Saint-Antoine-l'Abbaye	38
15025	360	\N	Saint-Appolinard	38
15026	361	\N	Saint-Arey	38
15027	362	\N	Saint-Aupre	38
15028	363	\N	Saint-Barthélemy	38
15029	364	\N	Saint-Barthélemy-de-Séchilienne	38
15030	365	\N	Saint-Baudille-de-la-Tour	38
15031	366	\N	Saint-Baudille-et-Pipet	38
15032	367	\N	Saint-Bernard	38
15033	368	\N	Saint-Blaise-du-Buis	38
15034	369	\N	Sainte-Blandine	38
15035	370	\N	Saint-Bonnet-de-Chavagne	38
15036	372	\N	Saint-Bueil	38
15037	373	\N	Saint-Cassien	38
15038	374	\N	Saint-Chef	38
15039	375	\N	Saint-Christophe-en-Oisans	38
15040	376	\N	Saint-Christophe-sur-Guiers	38
15041	377	\N	Saint-Clair-de-la-Tour	38
15042	378	\N	Saint-Clair-du-Rhône	38
15043	379	\N	Saint-Clair-sur-Galaure	38
15044	380	\N	Saint-Didier-de-Bizonnes	38
15045	381	\N	Saint-Didier-de-la-Tour	38
15046	382	\N	Saint-Égrève	38
15047	383	\N	Saint-Étienne-de-Crossey	38
15048	384	\N	Saint-Étienne-de-Saint-Geoirs	38
15049	386	\N	Saint-Geoire-en-Valdaine	38
15050	387	\N	Saint-Geoirs	38
15051	388	\N	Saint-Georges-de-Commiers	38
15052	389	\N	Saint-Georges-d'Espéranche	38
15053	390	\N	Saint-Gervais	38
15054	391	\N	Saint-Guillaume	38
15055	392	\N	Saint-Hilaire-de-Brens	38
15056	393	\N	Saint-Hilaire-de-la-Côte	38
15057	394	\N	Saint-Hilaire-du-Rosier	38
15058	395	\N	Saint-Hilaire	38
15059	396	\N	Saint-Honoré	38
15060	397	\N	Saint-Ismier	38
15061	398	\N	Saint-Jean-d'Avelanne	38
15062	399	\N	Saint-Jean-de-Bournay	38
15063	400	\N	Saint-Jean-de-Moirans	38
15064	401	\N	Saint-Jean-de-Soudain	38
15065	402	\N	Saint-Jean-de-Vaulx	38
15066	403	\N	Saint-Jean-d'Hérans	38
15067	404	\N	Saint-Jean-le-Vieux	38
15068	405	\N	Saint-Joseph-de-Rivière	38
15069	406	\N	Saint-Julien-de-l'Herms	38
15070	407	\N	Saint-Julien-de-Raz	38
15071	408	\N	Saint-Just-Chaleyssin	38
15072	409	\N	Saint-Just-de-Claix	38
15073	410	\N	Saint-Lattier	38
15074	412	\N	Saint-Laurent-du-Pont	38
15075	413	\N	Saint-Laurent-en-Beaumont	38
15076	414	\N	Sainte-Luce	38
15077	415	\N	Saint-Marcel-Bel-Accueil	38
15078	416	\N	Saint-Marcellin	38
15079	417	\N	Sainte-Marie-d'Alloix	38
15080	418	\N	Sainte-Marie-du-Mont	38
15081	419	\N	Saint-Martin-de-Clelles	38
15082	420	\N	Saint-Martin-de-Vaulserre	38
15083	421	\N	Saint-Martin-d'Hères	38
15084	422	\N	Saint-Martin-d'Uriage	38
15085	423	\N	Saint-Martin-le-Vinoux	38
15086	424	\N	Saint-Maurice-en-Trièves	38
15087	425	\N	Saint-Maurice-l'Exil	38
15088	426	\N	Saint-Maximin	38
15089	427	\N	Saint-Michel-de-Saint-Geoirs	38
15090	428	\N	Saint-Michel-en-Beaumont	38
15091	429	\N	Saint-Michel-les-Portes	38
15092	430	\N	Saint-Mury-Monteymond	38
15093	431	\N	Saint-Nazaire-les-Eymes	38
15094	432	\N	Saint-Nicolas-de-Macherin	38
15095	433	\N	Saint-Nizier-du-Moucherotte	38
15096	434	\N	Saint-Ondras	38
15097	435	\N	Saint-Pancrasse	38
15098	436	\N	Saint-Paul-de-Varces	38
15099	437	\N	Saint-Paul-d'Izeaux	38
15100	438	\N	Saint-Paul-lès-Monestier	38
15101	439	\N	Saint-Pierre-d'Allevard	38
15102	440	\N	Saint-Pierre-de-Bressieux	38
15103	442	\N	Saint-Pierre-de-Chartreuse	38
15104	443	\N	Saint-Pierre-de-Chérennes	38
15105	444	\N	Saint-Pierre-de-Méaroz	38
15106	445	\N	Saint-Pierre-de-Mésage	38
15107	446	\N	Saint-Pierre-d'Entremont	38
15108	448	\N	Saint-Prim	38
15109	449	\N	Saint-Quentin-Fallavier	38
15110	450	\N	Saint-Quentin-sur-Isère	38
15111	451	\N	Saint-Romain-de-Jalionas	38
15112	452	\N	Saint-Romain-de-Surieu	38
15113	453	\N	Saint-Romans	38
15114	454	\N	Saint-Sauveur	38
15115	455	\N	Saint-Savin	38
15116	456	\N	Saint-Sébastien	38
15117	457	\N	Saint-Siméon-de-Bressieux	38
15118	458	\N	Saint-Sorlin-de-Morestel	38
15119	459	\N	Saint-Sorlin-de-Vienne	38
15120	460	\N	Saint-Sulpice-des-Rivoires	38
15121	462	\N	Saint-Théoffrey	38
15122	463	\N	Saint-Vérand	38
15123	464	\N	Saint-Victor-de-Cessieu	38
15124	465	\N	Saint-Victor-de-Morestel	38
15125	466	\N	Saint-Vincent-de-Mercuze	38
15126	467	\N	Salagnon	38
15127	468	\N	Salaise-sur-Sanne	38
15128	469	\N	Salette-Fallavaux	38
15129	470	\N	Salle-en-Beaumont	38
15130	471	\N	Sappey-en-Chartreuse	38
15131	472	\N	Sarcenas	38
15132	473	\N	Sardieu	38
15133	474	\N	Sassenage	38
15134	475	\N	Satolas-et-Bonce	38
15135	476	\N	Savas-Mépin	38
15136	478	\N	Séchilienne	38
15137	479	\N	Semons	38
15138	480	\N	Septème	38
15139	481	\N	Sérézin-de-la-Tour	38
15140	483	\N	Sermérieu	38
15141	484	\N	Serpaize	38
15142	485	\N	Seyssinet-Pariset	38
15143	486	\N	Seyssins	38
15144	487	\N	Seyssuel	38
15145	488	\N	Siccieu-Saint-Julien-et-Carisieu	38
15146	489	\N	Siévoz	38
15147	490	\N	Sillans	38
15148	492	\N	Sinard	38
15149	494	\N	Soleymieu	38
15150	495	\N	Sône	38
15151	496	\N	Sonnay	38
15152	497	\N	Sousville	38
15153	498	\N	Succieu	38
15154	499	\N	Susville	38
15155	500	\N	Têche	38
15156	501	\N	Tencin	38
15157	503	\N	Terrasse	38
15158	504	\N	Theys	38
15159	505	\N	Thodure	38
15160	507	\N	Tignieu-Jameyzieu	38
15161	508	\N	Torchefelon	38
15162	509	\N	Tour-du-Pin	38
15163	511	\N	Touvet	38
15164	512	\N	Tramolé	38
15165	513	\N	Treffort	38
15166	514	\N	Tréminis	38
15167	515	\N	Trept	38
15168	516	\N	Tronche	38
15169	517	\N	Tullins	38
15170	518	\N	Valbonnais	38
15171	519	\N	Valencin	38
15172	520	\N	Valencogne	38
15173	521	\N	Valette	38
15174	522	\N	Valjouffrey	38
15175	523	\N	Varacieux	38
15176	524	\N	Varces-Allières-et-Risset	38
15177	525	\N	Vasselin	38
15178	526	\N	Vatilieu	38
15179	527	\N	Vaujany	38
15180	528	\N	Vaulnaveys-le-Bas	38
15181	529	\N	Vaulnaveys-le-Haut	38
15182	530	\N	Vaulx-Milieu	38
15183	531	\N	Velanne	38
15184	532	\N	Vénérieu	38
15185	533	\N	Venon	38
15186	534	\N	Vénosc	38
15187	535	\N	Vernas	38
15188	536	\N	Vernioz	38
15189	537	\N	Verpillière	38
15190	538	\N	Versoud	38
15191	539	\N	Vertrieu	38
15192	540	\N	Veurey-Voroize	38
15193	541	\N	Veyrins-Thuellin	38
15194	542	\N	Veyssilieu	38
15195	543	\N	Vézeronce-Curtin	38
15196	544	\N	Vienne	38
15197	545	\N	Vif	38
15198	546	\N	Vignieu	38
15199	547	\N	Villard-Bonnot	38
15200	548	\N	Villard-de-Lans	38
15201	549	\N	Villard-Notre-Dame	38
15202	550	\N	Villard-Reculas	38
15203	551	\N	Villard-Reymond	38
15204	552	\N	Villard-Saint-Christophe	38
15205	553	\N	Villefontaine	38
15206	554	\N	Villemoirieu	38
15207	555	\N	Villeneuve-de-Marc	38
15208	556	\N	Ville-sous-Anjou	38
15209	557	\N	Villette-d'Anthon	38
15210	558	\N	Villette-de-Vienne	38
15211	559	\N	Vinay	38
15212	560	\N	Virieu	38
15213	561	\N	Viriville	38
15214	562	\N	Vizille	38
15215	563	\N	Voiron	38
15216	564	\N	Voissant	38
15217	565	\N	Voreppe	38
15218	566	\N	Vourey	38
15219	567	\N	Chamrousse	38
15220	001	\N	Abergement-la-Ronce	39
15221	002	\N	Abergement-le-Grand	39
15222	003	\N	Abergement-le-Petit	39
15223	004	\N	Abergement-lès-Thésy	39
15224	006	\N	Aiglepierre	39
15225	007	\N	Alièze	39
15226	008	\N	Amange	39
15227	009	\N	Andelot-en-Montagne	39
15228	010	\N	Andelot-Morval	39
15229	011	\N	Annoire	39
15230	013	\N	Arbois	39
15231	014	\N	Archelange	39
15232	015	\N	Ardon	39
15233	016	\N	Arinthod	39
15234	017	\N	Arlay	39
15235	018	\N	Aromas	39
15236	019	\N	Arsures	39
15237	020	\N	Arsure-Arsurette	39
15238	021	\N	Arthenas	39
15239	022	\N	Asnans-Beauvoisin	39
15240	023	\N	Aubépin	39
15241	024	\N	Audelange	39
15242	025	\N	Augea	39
15243	026	\N	Augerans	39
15244	027	\N	Augisey	39
15245	028	\N	Aumont	39
15246	029	\N	Aumur	39
15247	030	\N	Authume	39
15248	031	\N	Auxange	39
15249	032	\N	Avignon-lès-Saint-Claude	39
15250	034	\N	Balaiseaux	39
15251	035	\N	Balanod	39
15252	036	\N	Balme-d'Épy	39
15253	037	\N	Bans	39
15254	038	\N	Barésia-sur-l'Ain	39
15255	039	\N	Barre	39
15256	040	\N	Barretaine	39
15257	041	\N	Baume-les-Messieurs	39
15258	042	\N	Baverans	39
15259	043	\N	Beaufort	39
15260	045	\N	Beffia	39
15261	046	\N	Bellecombe	39
15262	047	\N	Bellefontaine	39
15263	048	\N	Belmont	39
15264	049	\N	Bersaillin	39
15265	050	\N	Besain	39
15266	051	\N	Biarne	39
15267	052	\N	Bief-des-Maisons	39
15268	053	\N	Bief-du-Fourg	39
15269	054	\N	Biefmorin	39
15270	055	\N	Billecul	39
15271	056	\N	Bletterans	39
15272	057	\N	Blois-sur-Seille	39
15273	058	\N	Blye	39
15274	059	\N	Bois-d'Amont	39
15275	060	\N	Bois-de-Gand	39
15276	061	\N	Boissia	39
15277	062	\N	Boissière	39
15278	063	\N	Bonlieu	39
15279	064	\N	Bonnaud	39
15280	065	\N	Bonnefontaine	39
15281	066	\N	Bornay	39
15282	068	\N	Bouchoux	39
15283	069	\N	Bourcia	39
15284	070	\N	Bourg-de-Sirod	39
15285	072	\N	Bracon	39
15286	073	\N	Brainans	39
15287	074	\N	Brans	39
15288	075	\N	Bréry	39
15289	076	\N	Bretenière	39
15290	077	\N	Bretenières	39
15291	078	\N	Brevans	39
15292	079	\N	Briod	39
15293	080	\N	Broissia	39
15294	081	\N	Buvilly	39
15295	083	\N	Censeau	39
15296	084	\N	Cernans	39
15297	085	\N	Cerniébaud	39
15298	086	\N	Cernon	39
15299	088	\N	Cesancey	39
15300	089	\N	Cézia	39
15301	090	\N	Chaînée-des-Coupis	39
15302	091	\N	Chalesmes	39
15303	092	\N	Chambéria	39
15304	093	\N	Chamblay	39
15305	094	\N	Chamole	39
15306	095	\N	Champagne-sur-Loue	39
15307	096	\N	Champagney	39
15308	097	\N	Champagnole	39
15309	099	\N	Champdivers	39
15310	100	\N	Champrougier	39
15311	101	\N	Champvans	39
15312	102	\N	Chancia	39
15313	103	\N	Chapelle-sur-Furieuse	39
15314	104	\N	Chapelle-Voland	39
15315	105	\N	Chapois	39
15316	106	\N	Charchilla	39
15317	107	\N	Charcier	39
15318	108	\N	Charency	39
15319	109	\N	Charézier	39
15320	110	\N	Charme	39
15321	111	\N	Charnod	39
15322	112	\N	Chassagne	39
15323	113	\N	Chassal	39
15324	114	\N	Château-Chalon	39
15325	115	\N	Château-des-Prés	39
15326	116	\N	Châtelaine	39
15327	117	\N	Chatelay	39
15328	118	\N	Châtel-de-Joux	39
15329	119	\N	Chateley	39
15330	120	\N	Châtelneuf	39
15331	121	\N	Châtenois	39
15332	122	\N	Châtillon	39
15333	123	\N	Chatonnay	39
15334	124	\N	Chaumergy	39
15335	126	\N	Chaumusse	39
15336	127	\N	Chaussenans	39
15337	128	\N	Chaussin	39
15338	129	\N	Chaux-des-Crotenay	39
15339	130	\N	Chaux-des-Prés	39
15340	131	\N	Chaux-du-Dombief	39
15341	132	\N	Chaux-en-Bresse	39
15342	133	\N	Chaux-Champagny	39
15343	134	\N	Chavéria	39
15344	135	\N	Chazelles	39
15345	136	\N	Chemenot	39
15346	137	\N	Chemilla	39
15347	138	\N	Chemin	39
15348	139	\N	Chêne-Bernard	39
15349	140	\N	Chêne-Sec	39
15350	141	\N	Chevigny	39
15351	142	\N	Chevreaux	39
15352	143	\N	Chevrotaine	39
15353	145	\N	Chille	39
15354	146	\N	Chilly-le-Vignoble	39
15355	147	\N	Chilly-sur-Salins	39
15356	148	\N	Chisséria	39
15357	149	\N	Chissey-sur-Loue	39
15358	150	\N	Choisey	39
15359	151	\N	Choux	39
15360	153	\N	Cize	39
15361	154	\N	Clairvaux-les-Lacs	39
15362	155	\N	Clucy	39
15363	156	\N	Cogna	39
15364	157	\N	Coiserette	39
15365	158	\N	Coisia	39
15366	159	\N	Colonne	39
15367	160	\N	Commenailles	39
15368	161	\N	Communailles-en-Montagne	39
15369	162	\N	Condamine	39
15370	163	\N	Condes	39
15371	164	\N	Conliège	39
15372	165	\N	Conte	39
15373	166	\N	Cornod	39
15374	167	\N	Cosges	39
15375	168	\N	Courbette	39
15376	169	\N	Courbouzon	39
15377	170	\N	Courlans	39
15378	171	\N	Courlaoux	39
15379	172	\N	Courtefontaine	39
15380	173	\N	Cousance	39
15381	174	\N	Coyrière	39
15382	175	\N	Coyron	39
15383	176	\N	Cramans	39
15384	177	\N	Crançot	39
15385	178	\N	Crans	39
15386	179	\N	Crenans	39
15387	180	\N	Cressia	39
15388	182	\N	Crissey	39
15389	183	\N	Crotenay	39
15390	184	\N	Crozets	39
15391	185	\N	Cuisia	39
15392	186	\N	Cuttura	39
15393	187	\N	Cuvier	39
15394	188	\N	Dammartin-Marpain	39
15395	189	\N	Damparis	39
15396	190	\N	Dampierre	39
15397	191	\N	Darbonnay	39
15398	192	\N	Denezières	39
15399	193	\N	Deschaux	39
15400	194	\N	Desnes	39
15401	195	\N	Dessia	39
15402	196	\N	Deux-Fays	39
15403	197	\N	Digna	39
15404	198	\N	Dole	39
15405	199	\N	Domblans	39
15406	200	\N	Dompierre-sur-Mont	39
15407	201	\N	Doucier	39
15408	202	\N	Dournon	39
15409	203	\N	Doye	39
15410	204	\N	Dramelay	39
15411	205	\N	Éclans-Nenon	39
15412	206	\N	Écleux	39
15413	207	\N	Écrille	39
15414	208	\N	Entre-deux-Monts	39
15415	209	\N	Val-d'Épy	39
15416	210	\N	Équevillon	39
15417	211	\N	Essards-Taignevaux	39
15418	213	\N	Esserval-Combe	39
15419	214	\N	Esserval-Tartre	39
15420	215	\N	Essia	39
15421	216	\N	Étival	39
15422	217	\N	Étoile	39
15423	218	\N	Étrepigney	39
15424	219	\N	Évans	39
15425	220	\N	Falletans	39
15426	221	\N	Favière	39
15427	222	\N	Fay-en-Montagne	39
15428	223	\N	Ferté	39
15429	224	\N	Fétigny	39
15430	225	\N	Fied	39
15431	226	\N	Florentia	39
15432	227	\N	Foncine-le-Bas	39
15433	228	\N	Foncine-le-Haut	39
15434	229	\N	Fontainebrux	39
15435	230	\N	Fontenu	39
15436	232	\N	Fort-du-Plasne	39
15437	233	\N	Foucherans	39
15438	234	\N	Foulenay	39
15439	235	\N	Fraisans	39
15440	236	\N	Francheville	39
15441	237	\N	Fraroz	39
15442	238	\N	Frasne-les-Meulières	39
15443	239	\N	Frasnée	39
15444	240	\N	Frasnois	39
15445	241	\N	Frébuans	39
15446	243	\N	Froideville	39
15447	244	\N	Frontenay	39
15448	245	\N	Gatey	39
15449	246	\N	Gendrey	39
15450	247	\N	Genod	39
15451	248	\N	Geraise	39
15452	249	\N	Germigney	39
15453	250	\N	Geruge	39
15454	251	\N	Gevingey	39
15455	252	\N	Gevry	39
15456	253	\N	Gigny	39
15457	254	\N	Gillois	39
15458	255	\N	Gizia	39
15459	258	\N	Grande-Rivière	39
15460	259	\N	Grange-de-Vaivre	39
15461	260	\N	Granges-sur-Baume	39
15462	261	\N	Graye-et-Charnay	39
15463	262	\N	Gredisans	39
15464	263	\N	Grozon	39
15465	264	\N	Grusse	39
15466	265	\N	Hautecour	39
15467	266	\N	Hays	39
15468	267	\N	Ivory	39
15469	268	\N	Ivrey	39
15470	269	\N	Jeurre	39
15471	270	\N	Jouhe	39
15472	271	\N	Lac-des-Rouges-Truites	39
15473	272	\N	Ladoye-sur-Seille	39
15474	273	\N	Lains	39
15475	274	\N	Lajoux	39
15476	275	\N	Lamoura	39
15477	277	\N	Larderet	39
15478	278	\N	Largillay-Marsonnay	39
15479	279	\N	Larnaud	39
15480	280	\N	Larrivoire	39
15481	281	\N	Latet	39
15482	282	\N	Latette	39
15483	283	\N	Lavancia-Epercy	39
15484	284	\N	Lavangeot	39
15485	285	\N	Lavans-lès-Dole	39
15486	286	\N	Lavans-lès-Saint-Claude	39
15487	287	\N	Lavans-sur-Valouse	39
15488	288	\N	Lavigny	39
15489	289	\N	Lect	39
15490	290	\N	Légna	39
15491	291	\N	Lemuy	39
15492	292	\N	Lent	39
15493	293	\N	Leschères	39
15494	294	\N	Lézat	39
15495	295	\N	Loisia	39
15496	296	\N	Lombard	39
15497	297	\N	Longchaumois	39
15498	298	\N	Longcochon	39
15499	299	\N	Longwy-sur-le-Doubs	39
15500	300	\N	Lons-le-Saunier	39
15501	301	\N	Loulle	39
15502	302	\N	Louvatange	39
15503	303	\N	Louvenne	39
15504	304	\N	Louverot	39
15505	305	\N	Loye	39
15506	306	\N	Macornay	39
15507	307	\N	Maisod	39
15508	308	\N	Malange	39
15509	309	\N	Mallerey	39
15510	310	\N	Mantry	39
15511	312	\N	Marigna-sur-Valouse	39
15512	313	\N	Marigny	39
15513	314	\N	Marnézia	39
15514	315	\N	Marnoz	39
15515	317	\N	Marre	39
15516	318	\N	Martigna	39
15517	319	\N	Mathenay	39
15518	320	\N	Maynal	39
15519	321	\N	Menétru-le-Vignoble	39
15520	322	\N	Menétrux-en-Joux	39
15521	323	\N	Menotey	39
15522	324	\N	Mérona	39
15523	325	\N	Mesnay	39
15524	326	\N	Mesnois	39
15525	327	\N	Messia-sur-Sorne	39
15526	328	\N	Meussia	39
15527	329	\N	Mièges	39
15528	330	\N	Miéry	39
15529	331	\N	Mignovillard	39
15530	332	\N	Mirebel	39
15531	333	\N	Moirans-en-Montagne	39
15532	334	\N	Moiron	39
15533	335	\N	Moissey	39
15534	336	\N	Molain	39
15535	337	\N	Molamboz	39
15536	338	\N	Molay	39
15537	339	\N	Molinges	39
15538	340	\N	Molpré	39
15539	341	\N	Molunes	39
15540	342	\N	Monay	39
15541	343	\N	Monnetay	39
15542	344	\N	Monnet-la-Ville	39
15543	345	\N	Monnières	39
15544	346	\N	Montagna-le-Reconduit	39
15545	347	\N	Montagna-le-Templier	39
15546	348	\N	Montaigu	39
15547	349	\N	Montain	39
15548	350	\N	Montbarrey	39
15549	351	\N	Montcusel	39
15550	352	\N	Monteplain	39
15551	353	\N	Montfleur	39
15552	354	\N	Montholier	39
15553	355	\N	Montigny-lès-Arsures	39
15554	356	\N	Montigny-sur-l'Ain	39
15555	359	\N	Montmarlon	39
15556	360	\N	Montmirey-la-Ville	39
15557	361	\N	Montmirey-le-Château	39
15558	362	\N	Montmorot	39
15559	363	\N	Montrevel	39
15560	364	\N	Montrond	39
15561	365	\N	Mont-sous-Vaudrey	39
15562	366	\N	Mont-sur-Monnet	39
15563	367	\N	Morbier	39
15564	368	\N	Morez	39
15565	370	\N	Mouchard	39
15566	371	\N	Mouille	39
15567	372	\N	Mournans-Charbonny	39
15568	373	\N	Moussières	39
15569	375	\N	Moutonne	39
15570	376	\N	Moutoux	39
15571	377	\N	Mutigney	39
15572	378	\N	Nanc-lès-Saint-Amour	39
15573	379	\N	Nance	39
15574	380	\N	Nancuise	39
15575	381	\N	Nans	39
15576	382	\N	Nantey	39
15577	385	\N	Neublans-Abergement	39
15578	386	\N	Neuvilley	39
15579	387	\N	Nevy-lès-Dole	39
15580	388	\N	Nevy-sur-Seille	39
15581	389	\N	Ney	39
15582	390	\N	Nogna	39
15583	391	\N	Nozeroy	39
15584	392	\N	Offlanges	39
15585	393	\N	Onglières	39
15586	394	\N	Onoz	39
15587	395	\N	Orbagna	39
15588	396	\N	Orchamps	39
15589	397	\N	Orgelet	39
15590	398	\N	Ougney	39
15591	399	\N	Ounans	39
15592	400	\N	Our	39
15593	401	\N	Oussières	39
15594	402	\N	Pagney	39
15595	403	\N	Pagnoz	39
15596	404	\N	Pannessières	39
15597	405	\N	Parcey	39
15598	406	\N	Pasquier	39
15599	407	\N	Passenans	39
15600	408	\N	Patornay	39
15601	409	\N	Peintre	39
15602	411	\N	Perrigny	39
15603	412	\N	Peseux	39
15604	413	\N	Pesse	39
15605	414	\N	Petit-Mercey	39
15606	415	\N	Petit-Noir	39
15607	417	\N	Piards	39
15608	418	\N	Picarreau	39
15609	419	\N	Pillemoine	39
15610	420	\N	Pimorin	39
15611	421	\N	Pin	39
15612	422	\N	Plainoiseau	39
15613	423	\N	Plaisia	39
15614	424	\N	Planches-en-Montagne	39
15615	425	\N	Planches-près-Arbois	39
15616	426	\N	Plasne	39
15617	427	\N	Plénise	39
15618	428	\N	Plénisette	39
15619	429	\N	Pleure	39
15620	430	\N	Plumont	39
15621	431	\N	Poids-de-Fiole	39
15622	432	\N	Pointre	39
15623	434	\N	Poligny	39
15624	435	\N	Pont-de-Poitte	39
15625	436	\N	Pont-d'Héry	39
15626	437	\N	Pont-du-Navoy	39
15627	438	\N	Ponthoux	39
15628	439	\N	Port-Lesney	39
15629	440	\N	Pratz	39
15630	441	\N	Prémanon	39
15631	442	\N	Prénovel	39
15632	443	\N	Présilly	39
15633	444	\N	Pretin	39
15634	445	\N	Publy	39
15635	446	\N	Pupillin	39
15636	447	\N	Quintigny	39
15637	448	\N	Rahon	39
15638	449	\N	Rainans	39
15639	451	\N	Ranchot	39
15640	452	\N	Rans	39
15641	453	\N	Ravilloles	39
15642	454	\N	Recanoz	39
15643	455	\N	Reithouse	39
15644	456	\N	Relans	39
15645	457	\N	Repôts	39
15646	458	\N	Revigny	39
15647	460	\N	Rixouse	39
15648	461	\N	Rix	39
15649	462	\N	Rochefort-sur-Nenon	39
15650	463	\N	Rogna	39
15651	464	\N	Romain	39
15652	465	\N	Romange	39
15653	466	\N	Rosay	39
15654	467	\N	Rotalier	39
15655	468	\N	Rothonay	39
15656	469	\N	Rouffange	39
15657	470	\N	Rousses	39
15658	471	\N	Ruffey-sur-Seille	39
15659	472	\N	Rye	39
15660	473	\N	Saffloz	39
15661	474	\N	Sainte-Agnès	39
15662	475	\N	Saint-Amour	39
15663	476	\N	Saint-Aubin	39
15664	477	\N	Saint-Baraing	39
15665	478	\N	Saint-Claude	39
15666	479	\N	Saint-Cyr-Montmalin	39
15667	480	\N	Saint-Didier	39
15668	481	\N	Saint-Germain-en-Montagne	39
15669	482	\N	Saint-Germain-lès-Arlay	39
15670	483	\N	Saint-Hymetière	39
15671	484	\N	Saint-Jean-d'Étreux	39
15672	485	\N	Saint-Julien	39
15673	486	\N	Saint-Lamain	39
15674	487	\N	Saint-Laurent-en-Grandvaux	39
15675	488	\N	Saint-Laurent-la-Roche	39
15676	489	\N	Saint-Lothain	39
15677	490	\N	Saint-Loup	39
15678	491	\N	Saint-Lupicin	39
15679	492	\N	Saint-Maur	39
15680	493	\N	Saint-Maurice-Crillat	39
15681	494	\N	Saint-Pierre	39
15682	495	\N	Saint-Thiébaud	39
15683	497	\N	Saizenay	39
15684	498	\N	Salans	39
15685	499	\N	Saligney	39
15686	500	\N	Salins-les-Bains	39
15687	501	\N	Sampans	39
15688	502	\N	Santans	39
15689	503	\N	Sapois	39
15690	504	\N	Sarrogna	39
15691	505	\N	Saugeot	39
15692	506	\N	Savigna	39
15693	507	\N	Séligney	39
15694	508	\N	Sellières	39
15695	509	\N	Senaud	39
15696	510	\N	Septmoncel	39
15697	511	\N	Sergenaux	39
15698	512	\N	Sergenon	39
15699	513	\N	Sermange	39
15700	514	\N	Serre-les-Moulières	39
15701	517	\N	Sirod	39
15702	518	\N	Songeson	39
15703	519	\N	Soucia	39
15704	520	\N	Souvans	39
15705	522	\N	Supt	39
15706	523	\N	Syam	39
15707	524	\N	Tancua	39
15708	525	\N	Tassenières	39
15709	526	\N	Tavaux	39
15710	527	\N	Taxenne	39
15711	528	\N	Thervay	39
15712	529	\N	Thésy	39
15713	530	\N	Thoirette	39
15714	531	\N	Thoiria	39
15715	532	\N	Thoissia	39
15716	533	\N	Toulouse-le-Château	39
15717	534	\N	Tour-du-Meix	39
15718	535	\N	Tourmont	39
15719	537	\N	Trenal	39
15720	538	\N	Uxelles	39
15721	539	\N	Vadans	39
15722	540	\N	Valempoulières	39
15723	542	\N	Valfin-sur-Valouse	39
15724	543	\N	Vannoz	39
15725	544	\N	Varessia	39
15726	545	\N	Vaudioux	39
15727	546	\N	Vaudrey	39
15728	547	\N	Vaux-lès-Saint-Claude	39
15729	548	\N	Vaux-sur-Poligny	39
15730	549	\N	Vercia	39
15731	550	\N	Verges	39
15732	551	\N	Véria	39
15733	552	\N	Vernantois	39
15734	553	\N	Vernois	39
15735	554	\N	Vers-en-Montagne	39
15736	555	\N	Vers-sous-Sellières	39
15737	556	\N	Vertamboz	39
15738	557	\N	Vescles	39
15739	558	\N	Vevy	39
15740	559	\N	Vieille-Loye	39
15741	560	\N	Villard-Saint-Sauveur	39
15742	561	\N	Villards-d'Héria	39
15743	562	\N	Villard-sur-Bienne	39
15744	564	\N	Villechantria	39
15745	565	\N	Villeneuve-d'Aval	39
15746	566	\N	Villeneuve-lès-Charnod	39
15747	567	\N	Villeneuve-sous-Pymont	39
15748	568	\N	Villerserine	39
15749	569	\N	Villers-Farlay	39
15750	570	\N	Villers-les-Bois	39
15751	571	\N	Villers-Robert	39
15752	572	\N	Villette-lès-Arbois	39
15753	573	\N	Villette-lès-Dole	39
15754	574	\N	Villevieux	39
15755	575	\N	Villey	39
15756	576	\N	Vincelles	39
15757	577	\N	Vincent	39
15758	579	\N	Viry	39
15759	581	\N	Vitreux	39
15760	582	\N	Voiteur	39
15761	583	\N	Vosbles	39
15762	584	\N	Vriange	39
15763	585	\N	Vulvoz	39
15764	586	\N	Aresches	39
15765	001	\N	Aire-sur-l'Adour	40
15766	002	\N	Amou	40
15767	003	\N	Angoumé	40
15768	004	\N	Angresse	40
15769	005	\N	Arboucave	40
15770	006	\N	Arengosse	40
15771	007	\N	Argelos	40
15772	008	\N	Argelouse	40
15773	009	\N	Arjuzanx	40
15774	011	\N	Arsague	40
15775	012	\N	Artassenx	40
15776	013	\N	Arthez-d'Armagnac	40
15777	014	\N	Arue	40
15778	015	\N	Arx	40
15779	016	\N	Aubagnan	40
15780	017	\N	Audignon	40
15781	018	\N	Audon	40
15782	019	\N	Aureilhan	40
15783	020	\N	Aurice	40
15784	021	\N	Azur	40
15785	022	\N	Bahus-Soubiran	40
15786	023	\N	Baigts	40
15787	024	\N	Banos	40
15788	025	\N	Bascons	40
15789	026	\N	Bas-Mauco	40
15790	027	\N	Bassercles	40
15791	028	\N	Bastennes	40
15792	029	\N	Bats	40
15793	030	\N	Baudignan	40
15794	031	\N	Bégaar	40
15795	032	\N	Belhade	40
15796	033	\N	Bélis	40
15797	034	\N	Bélus	40
15798	035	\N	Bénesse-lès-Dax	40
15799	036	\N	Bénesse-Maremne	40
15800	037	\N	Benquet	40
15801	038	\N	Bergouey	40
15802	039	\N	Betbezer-d'Armagnac	40
15803	040	\N	Beylongue	40
15804	041	\N	Beyries	40
15805	042	\N	Biarrotte	40
15806	043	\N	Bias	40
15807	044	\N	Biaudos	40
15808	046	\N	Biscarrosse	40
15809	047	\N	Bonnegarde	40
15810	048	\N	Boos	40
15811	049	\N	Bordères-et-Lamensans	40
15812	050	\N	Bostens	40
15813	051	\N	Bougue	40
15814	052	\N	Bourdalat	40
15815	053	\N	Bourriot-Bergonce	40
15816	054	\N	Brassempouy	40
15817	055	\N	Bretagne-de-Marsan	40
15818	056	\N	Brocas	40
15819	057	\N	Buanes	40
15820	058	\N	Cachen	40
15821	059	\N	Cagnotte	40
15822	060	\N	Callen	40
15823	061	\N	Campagne	40
15824	062	\N	Campet-et-Lamolère	40
15825	063	\N	Candresse	40
15826	064	\N	Canenx-et-Réaut	40
15827	065	\N	Capbreton	40
15828	066	\N	Carcarès-Sainte-Croix	40
15829	067	\N	Carcen-Ponson	40
15830	068	\N	Cassen	40
15831	069	\N	Castaignos-Souslens	40
15832	070	\N	Castandet	40
15833	071	\N	Castelnau-Chalosse	40
15834	072	\N	Castelnau-Tursan	40
15835	073	\N	Castelner	40
15836	074	\N	Castel-Sarrazin	40
15837	075	\N	Castets	40
15838	076	\N	Cauna	40
15839	077	\N	Cauneille	40
15840	078	\N	Caupenne	40
15841	079	\N	Cazalis	40
15842	080	\N	Cazères-sur-l'Adour	40
15843	081	\N	Cère	40
15844	082	\N	Classun	40
15845	083	\N	Clèdes	40
15846	084	\N	Clermont	40
15847	085	\N	Commensacq	40
15848	086	\N	Coudures	40
15849	087	\N	Créon-d'Armagnac	40
15850	088	\N	Dax	40
15851	089	\N	Doazit	40
15852	090	\N	Donzacq	40
15853	091	\N	Duhort-Bachen	40
15854	092	\N	Dumes	40
15855	093	\N	Escalans	40
15856	094	\N	Escource	40
15857	095	\N	Estibeaux	40
15858	096	\N	Estigarde	40
15859	097	\N	Eugénie-les-Bains	40
15860	098	\N	Eyres-Moncube	40
15861	099	\N	Fargues	40
15862	100	\N	Frêche	40
15863	101	\N	Gaas	40
15864	102	\N	Gabarret	40
15865	103	\N	Gaillères	40
15866	104	\N	Gamarde-les-Bains	40
15867	105	\N	Garein	40
15868	106	\N	Garrey	40
15869	107	\N	Garrosse	40
15870	108	\N	Gastes	40
15871	109	\N	Gaujacq	40
15872	110	\N	Geaune	40
15873	111	\N	Geloux	40
15874	112	\N	Gibret	40
15875	113	\N	Goos	40
15876	114	\N	Gourbera	40
15877	115	\N	Gousse	40
15878	116	\N	Gouts	40
15879	117	\N	Grenade-sur-l'Adour	40
15880	118	\N	Habas	40
15881	119	\N	Hagetmau	40
15882	120	\N	Hastingues	40
15883	121	\N	Hauriet	40
15884	122	\N	Haut-Mauco	40
15885	123	\N	Herm	40
15886	124	\N	Herré	40
15887	125	\N	Heugas	40
15888	126	\N	Hinx	40
15889	127	\N	Hontanx	40
15890	128	\N	Horsarrieu	40
15891	129	\N	Josse	40
15892	130	\N	Labastide-Chalosse	40
15893	131	\N	Labastide-d'Armagnac	40
15894	132	\N	Labatut	40
15895	133	\N	Labenne	40
15896	134	\N	Labouheyre	40
15897	135	\N	Labrit	40
15898	136	\N	Lacajunte	40
15899	137	\N	Lacquy	40
15900	138	\N	Lacrabe	40
15901	139	\N	Laglorieuse	40
15902	140	\N	Lagrange	40
15903	141	\N	Lahosse	40
15904	142	\N	Laluque	40
15905	143	\N	Lamothe	40
15906	144	\N	Larbey	40
15907	145	\N	Larrivière	40
15908	146	\N	Latrille	40
15909	147	\N	Laurède	40
15910	148	\N	Lauret	40
15911	149	\N	Lencouacq	40
15912	150	\N	Léon	40
15913	151	\N	Lesgor	40
15914	152	\N	Lesperon	40
15915	153	\N	Leuy	40
15916	154	\N	Lévignacq	40
15917	155	\N	Linxe	40
15918	156	\N	Liposthey	40
15919	157	\N	Lit-et-Mixe	40
15920	158	\N	Losse	40
15921	159	\N	Louer	40
15922	160	\N	Lourquen	40
15923	161	\N	Lubbon	40
15924	162	\N	Lucbardez-et-Bargues	40
15925	163	\N	Lüe	40
15926	164	\N	Retjons	40
15927	165	\N	Luglon	40
15928	166	\N	Lussagnet	40
15929	167	\N	Luxey	40
15930	168	\N	Magescq	40
15931	169	\N	Maillas	40
15932	170	\N	Maillères	40
15933	171	\N	Mano	40
15934	172	\N	Mant	40
15935	173	\N	Marpaps	40
15936	174	\N	Mauries	40
15937	175	\N	Maurrin	40
15938	176	\N	Mauvezin-d'Armagnac	40
15939	177	\N	Maylis	40
15940	178	\N	Mazerolles	40
15941	179	\N	Mées	40
15942	180	\N	Meilhan	40
15943	181	\N	Messanges	40
15944	182	\N	Mézos	40
15945	183	\N	Mimbaste	40
15946	184	\N	Mimizan	40
15947	185	\N	Miramont-Sensacq	40
15948	186	\N	Misson	40
15949	187	\N	Moliets-et-Maa	40
15950	188	\N	Momuy	40
15951	189	\N	Monget	40
15952	190	\N	Monségur	40
15953	191	\N	Montaut	40
15954	192	\N	Mont-de-Marsan	40
15955	193	\N	Montégut	40
15956	194	\N	Montfort-en-Chalosse	40
15957	195	\N	Montgaillard	40
15958	196	\N	Montsoué	40
15959	197	\N	Morcenx	40
15960	198	\N	Morganx	40
15961	199	\N	Mouscardès	40
15962	200	\N	Moustey	40
15963	201	\N	Mugron	40
15964	202	\N	Narrosse	40
15965	203	\N	Nassiet	40
15966	204	\N	Nerbis	40
15967	205	\N	Nousse	40
15968	206	\N	Oeyregave	40
15969	207	\N	Oeyreluy	40
15970	208	\N	Onard	40
15971	209	\N	Ondres	40
15972	210	\N	Onesse-et-Laharie	40
15973	211	\N	Orist	40
15974	212	\N	Orthevielle	40
15975	213	\N	Orx	40
15976	214	\N	Ossages	40
15977	215	\N	Ousse-Suzan	40
15978	216	\N	Ozourt	40
15979	217	\N	Parentis-en-Born	40
15980	218	\N	Parleboscq	40
15981	219	\N	Payros-Cazautets	40
15982	220	\N	Pécorade	40
15983	221	\N	Perquie	40
15984	222	\N	Pey	40
15985	223	\N	Peyre	40
15986	224	\N	Peyrehorade	40
15987	225	\N	Philondenx	40
15988	226	\N	Pimbo	40
15989	227	\N	Pissos	40
15990	228	\N	Pomarez	40
15991	229	\N	Pontenx-les-Forges	40
15992	230	\N	Pontonx-sur-l'Adour	40
15993	231	\N	Port-de-Lanne	40
15994	232	\N	Poudenx	40
15995	233	\N	Pouillon	40
15996	234	\N	Pouydesseaux	40
15997	235	\N	Poyanne	40
15998	236	\N	Poyartin	40
15999	237	\N	Préchacq-les-Bains	40
16000	238	\N	Pujo-le-Plan	40
16001	239	\N	Puyol-Cazalet	40
16002	240	\N	Renung	40
16003	242	\N	Rimbez-et-Baudiets	40
16004	243	\N	Rion-des-Landes	40
16005	244	\N	Rivière-Saas-et-Gourby	40
16006	245	\N	Roquefort	40
16007	246	\N	Sabres	40
16008	247	\N	Saint-Agnet	40
16009	248	\N	Saint-André-de-Seignanx	40
16010	249	\N	Saint-Aubin	40
16011	250	\N	Saint-Avit	40
16012	251	\N	Saint-Barthélemy	40
16013	252	\N	Sainte-Colombe	40
16014	253	\N	Saint-Cricq-Chalosse	40
16015	254	\N	Saint-Cricq-du-Gave	40
16016	255	\N	Saint-Cricq-Villeneuve	40
16017	256	\N	Saint-Étienne-d'Orthe	40
16018	257	\N	Sainte-Eulalie-en-Born	40
16019	258	\N	Sainte-Foy	40
16020	259	\N	Saint-Gein	40
16021	260	\N	Saint-Geours-d'Auribat	40
16022	261	\N	Saint-Geours-de-Maremne	40
16023	262	\N	Saint-Gor	40
16024	263	\N	Saint-Jean-de-Lier	40
16025	264	\N	Saint-Jean-de-Marsacq	40
16026	265	\N	Saint-Julien-d'Armagnac	40
16027	266	\N	Saint-Julien-en-Born	40
16028	267	\N	Saint-Justin	40
16029	268	\N	Saint-Laurent-de-Gosse	40
16030	269	\N	Saint-Lon-les-Mines	40
16031	270	\N	Saint-Loubouer	40
16032	271	\N	Sainte-Marie-de-Gosse	40
16033	272	\N	Saint-Martin-de-Hinx	40
16034	273	\N	Saint-Martin-de-Seignanx	40
16035	274	\N	Saint-Martin-d'Oney	40
16036	275	\N	Saint-Maurice-sur-Adour	40
16037	276	\N	Saint-Michel-Escalus	40
16038	277	\N	Saint-Pandelon	40
16039	278	\N	Saint-Paul-en-Born	40
16040	279	\N	Saint-Paul-lès-Dax	40
16041	280	\N	Saint-Perdon	40
16042	281	\N	Saint-Pierre-du-Mont	40
16043	282	\N	Saint-Sever	40
16044	283	\N	Saint-Vincent-de-Paul	40
16045	284	\N	Saint-Vincent-de-Tyrosse	40
16046	285	\N	Saint-Yaguen	40
16047	286	\N	Samadet	40
16048	287	\N	Sanguinet	40
16049	288	\N	Sarbazan	40
16050	289	\N	Sarraziet	40
16051	290	\N	Sarron	40
16052	291	\N	Saubion	40
16053	292	\N	Saubrigues	40
16054	293	\N	Saubusse	40
16055	294	\N	Saugnac-et-Cambran	40
16056	295	\N	Saugnacq-et-Muret	40
16057	296	\N	Seignosse	40
16058	297	\N	Sen	40
16059	298	\N	Serres-Gaston	40
16060	299	\N	Serreslous-et-Arribans	40
16061	300	\N	Seyresse	40
16062	301	\N	Siest	40
16063	302	\N	Sindères	40
16064	303	\N	Solférino	40
16065	304	\N	Soorts-Hossegor	40
16066	305	\N	Sorbets	40
16067	306	\N	Sorde-l'Abbaye	40
16068	307	\N	Sore	40
16069	308	\N	Sort-en-Chalosse	40
16070	309	\N	Souprosse	40
16071	310	\N	Soustons	40
16072	311	\N	Taller	40
16073	312	\N	Tarnos	40
16074	313	\N	Tartas	40
16075	314	\N	Tercis-les-Bains	40
16076	315	\N	Téthieu	40
16077	316	\N	Tilh	40
16078	317	\N	Tosse	40
16079	318	\N	Toulouzette	40
16080	319	\N	Trensacq	40
16081	320	\N	Uchacq-et-Parentis	40
16082	321	\N	Urgons	40
16083	322	\N	Uza	40
16084	323	\N	Vert	40
16085	324	\N	Vicq-d'Auribat	40
16086	325	\N	Vielle-Tursan	40
16087	326	\N	Vielle-Saint-Girons	40
16088	327	\N	Vielle-Soubiran	40
16089	328	\N	Vieux-Boucau-les-Bains	40
16090	329	\N	Vignau	40
16091	330	\N	Villenave	40
16092	331	\N	Villeneuve-de-Marsan	40
16093	332	\N	Ychoux	40
16094	333	\N	Ygos-Saint-Saturnin	40
16095	334	\N	Yzosse	40
16096	001	\N	Ambloy	41
16097	002	\N	Angé	41
16098	003	\N	Areines	41
16099	004	\N	Artins	41
16100	005	\N	Arville	41
16101	006	\N	Autainville	41
16102	007	\N	Authon	41
16103	008	\N	Avaray	41
16104	009	\N	Averdon	41
16105	010	\N	Azé	41
16106	011	\N	Baigneaux	41
16107	012	\N	Baillou	41
16108	013	\N	Bauzy	41
16109	014	\N	Beauchêne	41
16110	015	\N	Beauvilliers	41
16111	016	\N	Billy	41
16112	017	\N	Binas	41
16113	018	\N	Blois	41
16114	019	\N	Boisseau	41
16115	020	\N	Bonneveau	41
16116	022	\N	Bouffry	41
16117	023	\N	Bourré	41
16118	024	\N	Boursay	41
16119	025	\N	Bracieux	41
16120	026	\N	Brévainville	41
16121	027	\N	Briou	41
16122	028	\N	Busloup	41
16123	029	\N	Candé-sur-Beuvron	41
16124	030	\N	Cellé	41
16125	031	\N	Cellettes	41
16126	032	\N	Chailles	41
16127	033	\N	Chambon-sur-Cisse	41
16128	034	\N	Chambord	41
16129	035	\N	Champigny-en-Beauce	41
16130	036	\N	Chaon	41
16131	037	\N	Chapelle-Enchérie	41
16132	038	\N	Chapelle-Montmartin	41
16133	039	\N	Chapelle-Saint-Martin-en-Plaine	41
16134	040	\N	Chapelle-Vendômoise	41
16135	041	\N	Chapelle-Vicomtesse	41
16136	042	\N	Châteauvieux	41
16137	043	\N	Châtillon-sur-Cher	41
16138	044	\N	Châtres-sur-Cher	41
16139	045	\N	Chaumont-sur-Loire	41
16140	046	\N	Chaumont-sur-Tharonne	41
16141	047	\N	Chaussée-Saint-Victor	41
16142	048	\N	Chauvigny-du-Perche	41
16143	049	\N	Chémery	41
16144	050	\N	Cheverny	41
16145	051	\N	Chissay-en-Touraine	41
16146	052	\N	Chitenay	41
16147	053	\N	Choue	41
16148	054	\N	Choussy	41
16149	055	\N	Chouzy-sur-Cisse	41
16150	056	\N	Colombe	41
16151	057	\N	Conan	41
16152	058	\N	Concriers	41
16153	059	\N	Contres	41
16154	060	\N	Cormenon	41
16155	061	\N	Cormeray	41
16156	062	\N	Couddes	41
16157	063	\N	Couffy	41
16158	064	\N	Coulanges	41
16159	065	\N	Coulommiers-la-Tour	41
16160	066	\N	Courbouzon	41
16161	067	\N	Cour-Cheverny	41
16162	068	\N	Courmemin	41
16163	069	\N	Cour-sur-Loire	41
16164	070	\N	Couture-sur-Loir	41
16165	071	\N	Crouy-sur-Cosson	41
16166	072	\N	Crucheray	41
16167	073	\N	Danzé	41
16168	074	\N	Dhuizon	41
16169	075	\N	Droué	41
16170	077	\N	Épiais	41
16171	078	\N	Épuisay	41
16172	079	\N	Essarts	41
16173	080	\N	Faverolles-sur-Cher	41
16174	081	\N	Faye	41
16175	082	\N	Feings	41
16176	083	\N	Ferté-Beauharnais	41
16177	084	\N	Ferté-Imbault	41
16178	085	\N	Ferté-Saint-Cyr	41
16179	086	\N	Fontaines-en-Sologne	41
16180	087	\N	Fontaine-les-Coteaux	41
16181	088	\N	Fontaine-Raoul	41
16182	089	\N	Fontenelle	41
16183	090	\N	Fortan	41
16184	091	\N	Fossé	41
16185	092	\N	Fougères-sur-Bièvre	41
16186	093	\N	Françay	41
16187	094	\N	Fresnes	41
16188	095	\N	Fréteval	41
16189	096	\N	Gault-Perche	41
16190	097	\N	Gièvres	41
16191	098	\N	Gombergean	41
16192	099	\N	Gy-en-Sologne	41
16193	100	\N	Hayes	41
16194	101	\N	Herbault	41
16195	102	\N	Houssay	41
16196	103	\N	Huisseau-en-Beauce	41
16197	104	\N	Huisseau-sur-Cosson	41
16198	105	\N	Josnes	41
16199	106	\N	Lamotte-Beuvron	41
16200	107	\N	Lancé	41
16201	108	\N	Lancôme	41
16202	109	\N	Landes-le-Gaulois	41
16203	110	\N	Langon	41
16204	112	\N	Lassay-sur-Croisne	41
16205	113	\N	Lavardin	41
16206	114	\N	Lestiou	41
16207	115	\N	Lignières	41
16208	116	\N	Lisle	41
16209	118	\N	Loreux	41
16210	119	\N	Lorges	41
16211	120	\N	Lunay	41
16212	121	\N	Madeleine-Villefrouin	41
16213	122	\N	Maray	41
16214	123	\N	Marchenoir	41
16215	124	\N	Marcilly-en-Beauce	41
16216	125	\N	Marcilly-en-Gault	41
16217	126	\N	Mareuil-sur-Cher	41
16218	127	\N	Marolle-en-Sologne	41
16219	128	\N	Marolles	41
16220	129	\N	Maslives	41
16221	130	\N	Maves	41
16222	131	\N	Mazangé	41
16223	132	\N	Méhers	41
16224	133	\N	Membrolles	41
16225	134	\N	Menars	41
16226	135	\N	Mennetou-sur-Cher	41
16227	136	\N	Mer	41
16228	137	\N	Mesland	41
16229	138	\N	Meslay	41
16230	139	\N	Meusnes	41
16231	140	\N	Millançay	41
16232	141	\N	Moisy	41
16233	142	\N	Molineuf	41
16234	143	\N	Mondoubleau	41
16235	144	\N	Monteaux	41
16236	145	\N	Monthou-sur-Bièvre	41
16237	146	\N	Monthou-sur-Cher	41
16238	147	\N	Montils	41
16239	148	\N	Montlivault	41
16240	149	\N	Montoire-sur-le-Loir	41
16241	150	\N	Mont-près-Chambord	41
16242	151	\N	Montrichard	41
16243	152	\N	Montrieux-en-Sologne	41
16244	153	\N	Montrouveau	41
16245	154	\N	Morée	41
16246	155	\N	Muides-sur-Loire	41
16247	156	\N	Mulsans	41
16248	157	\N	Mur-de-Sologne	41
16249	158	\N	Naveil	41
16250	159	\N	Neung-sur-Beuvron	41
16251	160	\N	Neuvy	41
16252	161	\N	Nouan-le-Fuzelier	41
16253	163	\N	Nourray	41
16254	164	\N	Noyers-sur-Cher	41
16255	165	\N	Oigny	41
16256	166	\N	Oisly	41
16257	167	\N	Onzain	41
16258	168	\N	Orçay	41
16259	169	\N	Orchaise	41
16260	170	\N	Ouchamps	41
16261	171	\N	Oucques	41
16262	172	\N	Ouzouer-le-Doyen	41
16263	173	\N	Ouzouer-le-Marché	41
16264	174	\N	Périgny	41
16265	175	\N	Pezou	41
16266	176	\N	Pierrefitte-sur-Sauldre	41
16267	177	\N	Plessis-Dorin	41
16268	178	\N	Plessis-l'Échelle	41
16269	179	\N	Poislay	41
16270	180	\N	Pontlevoy	41
16271	181	\N	Pouillé	41
16272	182	\N	Pray	41
16273	183	\N	Prénouvellon	41
16274	184	\N	Prunay-Cassereau	41
16275	185	\N	Pruniers-en-Sologne	41
16276	186	\N	Rahart	41
16277	187	\N	Renay	41
16278	188	\N	Rhodon	41
16279	189	\N	Rilly-sur-Loire	41
16280	190	\N	Rocé	41
16281	191	\N	Roches	41
16282	192	\N	Roches-l'Évêque	41
16283	193	\N	Romilly	41
16284	194	\N	Romorantin-Lanthenay	41
16285	195	\N	Rougeou	41
16286	196	\N	Ruan-sur-Egvonne	41
16287	197	\N	Saint-Agil	41
16288	198	\N	Saint-Aignan	41
16289	199	\N	Saint-Amand-Longpré	41
16290	200	\N	Sainte-Anne	41
16291	201	\N	Saint-Arnoult	41
16292	202	\N	Saint-Avit	41
16293	203	\N	Saint-Bohaire	41
16294	204	\N	Saint-Claude-de-Diray	41
16295	205	\N	Saint-Cyr-du-Gault	41
16296	206	\N	Saint-Denis-sur-Loire	41
16297	207	\N	Saint-Dyé-sur-Loire	41
16298	208	\N	Saint-Étienne-des-Guérets	41
16299	209	\N	Saint-Firmin-des-Prés	41
16300	210	\N	Sainte-Gemmes	41
16301	211	\N	Saint-Georges-sur-Cher	41
16302	212	\N	Saint-Gervais-la-Forêt	41
16303	213	\N	Saint-Gourgon	41
16304	214	\N	Saint-Hilaire-la-Gravelle	41
16305	215	\N	Saint-Jacques-des-Guérets	41
16306	216	\N	Saint-Jean-Froidmentel	41
16307	217	\N	Saint-Julien-de-Chédon	41
16308	218	\N	Saint-Julien-sur-Cher	41
16309	219	\N	Saint-Laurent-des-Bois	41
16310	220	\N	Saint-Laurent-Nouan	41
16311	221	\N	Saint-Léonard-en-Beauce	41
16312	222	\N	Saint-Loup	41
16313	223	\N	Saint-Lubin-en-Vergonnois	41
16314	224	\N	Saint-Marc-du-Cor	41
16315	225	\N	Saint-Martin-des-Bois	41
16316	226	\N	Saint-Ouen	41
16317	228	\N	Saint-Rimay	41
16318	229	\N	Saint-Romain-sur-Cher	41
16319	230	\N	Saint-Sulpice-de-Pommeray	41
16320	231	\N	Saint-Viâtre	41
16321	232	\N	Salbris	41
16322	233	\N	Sambin	41
16323	234	\N	Santenay	41
16324	235	\N	Sargé-sur-Braye	41
16325	236	\N	Sasnières	41
16326	237	\N	Sassay	41
16327	238	\N	Savigny-sur-Braye	41
16328	239	\N	Seigy	41
16329	240	\N	Seillac	41
16330	241	\N	Selles-Saint-Denis	41
16331	242	\N	Selles-sur-Cher	41
16332	243	\N	Selommes	41
16333	244	\N	Semerville	41
16334	245	\N	Séris	41
16335	246	\N	Seur	41
16336	247	\N	Soings-en-Sologne	41
16337	248	\N	Souday	41
16338	249	\N	Souesmes	41
16339	250	\N	Sougé	41
16340	251	\N	Souvigny-en-Sologne	41
16341	252	\N	Suèvres	41
16342	253	\N	Talcy	41
16343	254	\N	Temple	41
16344	255	\N	Ternay	41
16345	256	\N	Theillay	41
16346	257	\N	Thenay	41
16347	258	\N	Thésée	41
16348	259	\N	Thoré-la-Rochette	41
16349	260	\N	Thoury	41
16350	261	\N	Tourailles	41
16351	262	\N	Tour-en-Sologne	41
16352	263	\N	Tréhet	41
16353	264	\N	Tripleville	41
16354	265	\N	Troo	41
16355	266	\N	Valaire	41
16356	267	\N	Vallières-les-Grandes	41
16357	268	\N	Veilleins	41
16358	269	\N	Vendôme	41
16359	270	\N	Verdes	41
16360	271	\N	Vernou-en-Sologne	41
16361	272	\N	Veuves	41
16362	273	\N	Vievy-le-Rayé	41
16363	274	\N	Villavard	41
16364	275	\N	Ville-aux-Clercs	41
16365	276	\N	Villebarou	41
16366	277	\N	Villebout	41
16367	278	\N	Villechauve	41
16368	279	\N	Villedieu-le-Château	41
16369	280	\N	Villefranche-sur-Cher	41
16370	281	\N	Villefrancur	41
16371	282	\N	Villeherviers	41
16372	283	\N	Villemardy	41
16373	284	\N	Villeneuve-Frouville	41
16374	285	\N	Villeny	41
16375	286	\N	Villeporcher	41
16376	287	\N	Villerable	41
16377	288	\N	Villerbon	41
16378	289	\N	Villermain	41
16379	290	\N	Villeromain	41
16380	291	\N	Villetrun	41
16381	292	\N	Villexanton	41
16382	293	\N	Villiersfaux	41
16383	294	\N	Villiers-sur-Loir	41
16384	295	\N	Vineuil	41
16385	296	\N	Vouzon	41
16386	297	\N	Yvoy-le-Marron	41
16387	001	\N	Aboën	42
16388	002	\N	Ailleux	42
16389	003	\N	Ambierle	42
16390	004	\N	Amions	42
16391	005	\N	Andrézieux-Bouthéon	42
16392	006	\N	Apinac	42
16393	007	\N	Arcinges	42
16394	008	\N	Arcon	42
16395	009	\N	Arthun	42
16396	010	\N	Aveizieux	42
16397	011	\N	Balbigny	42
16398	012	\N	Bard	42
16399	013	\N	Bellegarde-en-Forez	42
16400	014	\N	Belleroche	42
16401	015	\N	Belmont-de-la-Loire	42
16402	016	\N	Bénisson-Dieu	42
16403	017	\N	Bessat	42
16404	018	\N	Bessey	42
16405	019	\N	Boën	42
16406	020	\N	Boisset-lès-Montrond	42
16407	021	\N	Boisset-Saint-Priest	42
16408	022	\N	Bonson	42
16409	023	\N	Bourg-Argental	42
16410	025	\N	Boyer	42
16411	026	\N	Briennon	42
16412	027	\N	Bully	42
16413	028	\N	Burdignes	42
16414	029	\N	Bussières	42
16415	030	\N	Bussy-Albieux	42
16416	031	\N	Caloire	42
16417	032	\N	Cellieu	42
16418	033	\N	Cergne	42
16419	034	\N	Cervières	42
16420	035	\N	Cezay	42
16421	036	\N	Chagnon	42
16422	037	\N	Chalain-d'Uzore	42
16423	038	\N	Chalain-le-Comtal	42
16424	039	\N	Chalmazel	42
16425	040	\N	Chamba	42
16426	041	\N	Chambéon	42
16427	042	\N	Chambles	42
16428	043	\N	Chambuf	42
16429	044	\N	Chambon-Feugerolles	42
16430	045	\N	Chambonie	42
16431	046	\N	Champdieu	42
16432	047	\N	Champoly	42
16433	048	\N	Chandon	42
16434	049	\N	Changy	42
16435	050	\N	Chapelle-en-Lafaye	42
16436	051	\N	Chapelle-Villars	42
16437	052	\N	Charlieu	42
16438	053	\N	Châteauneuf	42
16439	054	\N	Châtelneuf	42
16440	055	\N	Châtelus	42
16441	056	\N	Chavanay	42
16442	058	\N	Chazelles-sur-Lavieu	42
16443	059	\N	Chazelles-sur-Lyon	42
16444	060	\N	Chenereilles	42
16445	061	\N	Cherier	42
16446	062	\N	Chevrières	42
16447	063	\N	Chirassimont	42
16448	064	\N	Chuyer	42
16449	065	\N	Civens	42
16450	066	\N	Cleppé	42
16451	067	\N	Colombier	42
16452	068	\N	Combre	42
16453	069	\N	Commelle-Vernay	42
16454	070	\N	Cordelle	42
16455	071	\N	Coteau	42
16456	072	\N	Côte-en-Couzan	42
16457	073	\N	Cottance	42
16458	074	\N	Coutouvre	42
16459	075	\N	Craintilleux	42
16460	076	\N	Cremeaux	42
16461	077	\N	Croizet-sur-Gand	42
16462	078	\N	Crozet	42
16463	079	\N	Cuinzier	42
16464	081	\N	Cuzieu	42
16465	082	\N	Dancé	42
16466	083	\N	Dargoire	42
16467	084	\N	Débats-Rivière-d'Orpra	42
16468	085	\N	Doizieux	42
16469	086	\N	Écoche	42
16470	087	\N	Écotay-l'Olme	42
16471	088	\N	Épercieux-Saint-Paul	42
16472	089	\N	Essertines-en-Châtelneuf	42
16473	090	\N	Essertines-en-Donzy	42
16474	091	\N	Estivareilles	42
16475	092	\N	Étrat	42
16476	093	\N	Farnay	42
16477	094	\N	Feurs	42
16478	095	\N	Firminy	42
16479	096	\N	Fontanès	42
16480	097	\N	Fouillouse	42
16481	098	\N	Fourneaux	42
16482	099	\N	Fraisses	42
16483	100	\N	Gimond	42
16484	101	\N	Graix	42
16485	102	\N	Grammond	42
16486	103	\N	Grand-Croix	42
16487	104	\N	Gresle	42
16488	105	\N	Grézieux-le-Fromental	42
16489	106	\N	Grézolles	42
16490	107	\N	Gumières	42
16491	108	\N	Hôpital-le-Grand	42
16492	109	\N	Hôpital-sous-Rochefort	42
16493	110	\N	Horme	42
16494	112	\N	Jarnosse	42
16495	113	\N	Jas	42
16496	114	\N	Jeansagnière	42
16497	115	\N	Jonzieux	42
16498	116	\N	Juré	42
16499	117	\N	Lavieu	42
16500	118	\N	Lay	42
16501	119	\N	Leigneux	42
16502	120	\N	Lentigny	42
16503	121	\N	Lérigneux	42
16504	122	\N	Lézigneux	42
16505	123	\N	Lorette	42
16506	124	\N	Lupé	42
16507	125	\N	Luré	42
16508	126	\N	Luriecq	42
16509	127	\N	Mably	42
16510	128	\N	Machézal	42
16511	129	\N	Maclas	42
16512	130	\N	Magneux-Haute-Rive	42
16513	131	\N	Maizilly	42
16514	132	\N	Malleval	42
16515	133	\N	Marcenod	42
16516	134	\N	Marcilly-le-Châtel	42
16517	135	\N	Marclopt	42
16518	136	\N	Marcoux	42
16519	137	\N	Margerie-Chantagret	42
16520	138	\N	Maringes	42
16521	139	\N	Marlhes	42
16522	140	\N	Marols	42
16523	141	\N	Mars	42
16524	142	\N	Merle-Leignec	42
16525	143	\N	Mizérieux	42
16526	145	\N	Montagny	42
16527	146	\N	Montarcher	42
16528	147	\N	Montbrison	42
16529	148	\N	Montchal	42
16530	149	\N	Montrond-les-Bains	42
16531	150	\N	Montverdun	42
16532	151	\N	Mornand-en-Forez	42
16533	152	\N	Nandax	42
16534	153	\N	Neaux	42
16535	154	\N	Néronde	42
16536	155	\N	Nervieux	42
16537	156	\N	Neulise	42
16538	157	\N	Noailly	42
16539	158	\N	Noës	42
16540	159	\N	Noirétable	42
16541	160	\N	Nollieux	42
16542	161	\N	Notre-Dame-de-Boisset	42
16543	162	\N	Ouches	42
16544	163	\N	Pacaudière	42
16545	164	\N	Palogneux	42
16546	165	\N	Panissières	42
16547	166	\N	Parigny	42
16548	167	\N	Pavezin	42
16549	168	\N	Pélussin	42
16550	169	\N	Périgneux	42
16551	170	\N	Perreux	42
16552	171	\N	Pinay	42
16553	172	\N	Planfoy	42
16554	173	\N	Pommiers	42
16555	174	\N	Poncins	42
16556	175	\N	Pouilly-lès-Feurs	42
16557	176	\N	Pouilly-les-Nonains	42
16558	177	\N	Pouilly-sous-Charlieu	42
16559	178	\N	Pradines	42
16560	179	\N	Pralong	42
16561	180	\N	Précieux	42
16562	181	\N	Régny	42
16563	182	\N	Renaison	42
16564	183	\N	Ricamarie	42
16565	184	\N	Riorges	42
16566	185	\N	Rivas	42
16567	186	\N	Rive-de-Gier	42
16568	187	\N	Roanne	42
16569	188	\N	Roche	42
16570	189	\N	Roche-la-Molière	42
16571	191	\N	Roisey	42
16572	192	\N	Rozier-Côtes-d'Aurec	42
16573	193	\N	Rozier-en-Donzy	42
16574	194	\N	Sail-les-Bains	42
16575	195	\N	Sail-sous-Couzan	42
16576	196	\N	Sainte-Agathe-en-Donzy	42
16577	197	\N	Sainte-Agathe-la-Bouteresse	42
16578	198	\N	Saint-Alban-les-Eaux	42
16579	199	\N	Saint-André-d'Apchon	42
16580	200	\N	Saint-André-le-Puy	42
16581	201	\N	Saint-Appolinard	42
16582	202	\N	Saint-Barthélemy-Lestra	42
16583	203	\N	Saint-Bonnet-des-Quarts	42
16584	204	\N	Saint-Bonnet-le-Château	42
16585	205	\N	Saint-Bonnet-le-Courreau	42
16586	206	\N	Saint-Bonnet-les-Oules	42
16587	207	\N	Saint-Chamond	42
16588	208	\N	Saint-Christo-en-Jarez	42
16589	209	\N	Sainte-Colombe-sur-Gand	42
16590	210	\N	Sainte-Croix-en-Jarez	42
16591	211	\N	Saint-Cyprien	42
16592	212	\N	Saint-Cyr-de-Favières	42
16593	213	\N	Saint-Cyr-de-Valorges	42
16594	214	\N	Saint-Cyr-les-Vignes	42
16595	215	\N	Saint-Denis-de-Cabanne	42
16596	216	\N	Saint-Denis-sur-Coise	42
16597	217	\N	Saint-Didier-sur-Rochefort	42
16598	218	\N	Saint-Étienne	42
16599	219	\N	Saint-Étienne-le-Molard	42
16600	220	\N	Saint-Forgeux-Lespinasse	42
16601	221	\N	Sainte-Foy-Saint-Sulpice	42
16602	222	\N	Saint-Galmier	42
16603	223	\N	Saint-Genest-Lerpt	42
16604	224	\N	Saint-Genest-Malifaux	42
16605	225	\N	Genilac	42
16606	226	\N	Saint-Georges-de-Baroille	42
16607	227	\N	Saint-Georges-en-Couzan	42
16608	228	\N	Saint-Georges-Haute-Ville	42
16609	229	\N	Saint-Germain-la-Montagne	42
16610	230	\N	Saint-Germain-Laval	42
16611	231	\N	Saint-Germain-Lespinasse	42
16612	232	\N	Saint-Haon-le-Châtel	42
16613	233	\N	Saint-Haon-le-Vieux	42
16614	234	\N	Saint-Héand	42
16615	235	\N	Saint-Hilaire-Cusson-la-Valmitte	42
16616	236	\N	Saint-Hilaire-sous-Charlieu	42
16617	237	\N	Saint-Jean-Bonnefonds	42
16618	238	\N	Saint-Jean-la-Vêtre	42
16619	239	\N	Saint-Jean-Saint-Maurice-sur-Loire	42
16620	240	\N	Saint-Jean-Soleymieux	42
16621	241	\N	Saint-Jodard	42
16622	242	\N	Saint-Joseph	42
16623	243	\N	Saint-Julien-d'Oddes	42
16624	245	\N	Saint-Julien-la-Vêtre	42
16625	246	\N	Saint-Julien-Molin-Molette	42
16626	247	\N	Saint-Just-en-Bas	42
16627	248	\N	Saint-Just-en-Chevalet	42
16628	249	\N	Saint-Just-la-Pendue	42
16629	251	\N	Saint-Laurent-la-Conche	42
16630	252	\N	Saint-Laurent-Rochefort	42
16631	253	\N	Saint-Léger-sur-Roanne	42
16632	254	\N	Saint-Marcel-de-Félines	42
16633	255	\N	Saint-Marcel-d'Urfé	42
16634	256	\N	Saint-Marcellin-en-Forez	42
16635	257	\N	Saint-Martin-d'Estréaux	42
16636	259	\N	Saint-Martin-la-Plaine	42
16637	260	\N	Saint-Martin-la-Sauveté	42
16638	261	\N	Saint-Martin-Lestra	42
16639	262	\N	Saint-Maurice-en-Gourgois	42
16640	264	\N	Saint-Médard-en-Forez	42
16641	265	\N	Saint-Michel-sur-Rhône	42
16642	266	\N	Saint-Nizier-de-Fornas	42
16643	267	\N	Saint-Nizier-sous-Charlieu	42
16644	268	\N	Saint-Paul-de-Vézelin	42
16645	269	\N	Saint-Paul-d'Uzore	42
16646	270	\N	Saint-Paul-en-Cornillon	42
16647	271	\N	Saint-Paul-en-Jarez	42
16648	272	\N	Saint-Pierre-de-Buf	42
16649	273	\N	Saint-Pierre-la-Noaille	42
16650	274	\N	Saint-Polgues	42
16651	275	\N	Saint-Priest-en-Jarez	42
16652	276	\N	Saint-Priest-la-Prugne	42
16653	277	\N	Saint-Priest-la-Roche	42
16654	278	\N	Saint-Priest-la-Vêtre	42
16655	279	\N	Saint-Just-Saint-Rambert	42
16656	280	\N	Saint-Régis-du-Coin	42
16657	281	\N	Saint-Rirand	42
16658	282	\N	Saint-Romain-d'Urfé	42
16659	283	\N	Saint-Romain-en-Jarez	42
16660	284	\N	Saint-Romain-la-Motte	42
16661	285	\N	Saint-Romain-le-Puy	42
16662	286	\N	Saint-Romain-les-Atheux	42
16663	287	\N	Saint-Sauveur-en-Rue	42
16664	288	\N	Saint-Sixte	42
16665	289	\N	Saint-Symphorien-de-Lay	42
16666	290	\N	Saint-Thomas-la-Garde	42
16667	291	\N	Saint-Thurin	42
16668	293	\N	Saint-Victor-sur-Rhins	42
16669	294	\N	Saint-Vincent-de-Boisset	42
16670	295	\N	Salles	42
16671	296	\N	Salt-en-Donzy	42
16672	297	\N	Salvizinet	42
16673	298	\N	Sauvain	42
16674	299	\N	Savigneux	42
16675	300	\N	Sevelinges	42
16676	301	\N	Soleymieux	42
16677	302	\N	Sorbiers	42
16678	303	\N	Souternon	42
16679	304	\N	Sury-le-Comtal	42
16680	305	\N	Talaudière	42
16681	306	\N	Tarentaise	42
16682	307	\N	Tartaras	42
16683	308	\N	Terrasse-sur-Dorlay	42
16684	310	\N	Thélis-la-Combe	42
16685	311	\N	Tour-en-Jarez	42
16686	312	\N	Tourette	42
16687	313	\N	Trelins	42
16688	314	\N	Tuilière	42
16689	315	\N	Unias	42
16690	316	\N	Unieux	42
16691	317	\N	Urbise	42
16692	318	\N	Usson-en-Forez	42
16693	319	\N	Valeille	42
16694	320	\N	Valfleury	42
16695	321	\N	Valla-sur-Rochefort	42
16696	322	\N	Valla-en-Gier	42
16697	323	\N	Veauche	42
16698	324	\N	Veauchette	42
16699	325	\N	Vendranges	42
16700	326	\N	Véranne	42
16701	327	\N	Vérin	42
16702	328	\N	Verrières-en-Forez	42
16703	329	\N	Versanne	42
16704	330	\N	Villars	42
16705	331	\N	Villemontais	42
16706	332	\N	Villerest	42
16707	333	\N	Villers	42
16708	334	\N	Violay	42
16709	335	\N	Viricelles	42
16710	336	\N	Virigneux	42
16711	337	\N	Vivans	42
16712	338	\N	Vougy	42
16713	339	\N	Chausseterre	42
16714	001	\N	Agnat	43
16715	002	\N	Aiguilhe	43
16716	003	\N	Allègre	43
16717	004	\N	Alleyrac	43
16718	005	\N	Alleyras	43
16719	006	\N	Ally	43
16720	007	\N	Araules	43
16721	008	\N	Arlempdes	43
16722	009	\N	Arlet	43
16723	010	\N	Arsac-en-Velay	43
16724	011	\N	Aubazat	43
16725	012	\N	Aurec-sur-Loire	43
16726	013	\N	Vissac-Auteyrac	43
16727	014	\N	Autrac	43
16728	015	\N	Auvers	43
16729	016	\N	Auzon	43
16730	017	\N	Azérat	43
16731	018	\N	Bains	43
16732	019	\N	Barges	43
16733	020	\N	Bas-en-Basset	43
16734	021	\N	Beaulieu	43
16735	022	\N	Beaumont	43
16736	023	\N	Beaune-sur-Arzon	43
16737	024	\N	Beaux	43
16738	025	\N	Beauzac	43
16739	026	\N	Bellevue-la-Montagne	43
16740	027	\N	Berbezit	43
16741	028	\N	Bessamorel	43
16742	029	\N	Besseyre-Saint-Mary	43
16743	030	\N	Blanzac	43
16744	031	\N	Blassac	43
16745	032	\N	Blavozy	43
16746	033	\N	Blesle	43
16747	034	\N	Boisset	43
16748	035	\N	Bonneval	43
16749	036	\N	Borne	43
16750	037	\N	Bouchet-Saint-Nicolas	43
16751	038	\N	Bournoncle-Saint-Pierre	43
16752	039	\N	Brignon	43
16753	040	\N	Brioude	43
16754	041	\N	Brives-Charensac	43
16755	042	\N	Cayres	43
16756	043	\N	Céaux-d'Allègre	43
16757	044	\N	Cerzat	43
16758	045	\N	Ceyssac	43
16759	046	\N	Chadrac	43
16760	047	\N	Chadron	43
16761	048	\N	Chaise-Dieu	43
16762	049	\N	Chamalières-sur-Loire	43
16763	050	\N	Chambezon	43
16764	051	\N	Chambon-sur-Lignon	43
16765	052	\N	Champagnac-le-Vieux	43
16766	053	\N	Champclause	43
16767	054	\N	Chanaleilles	43
16768	055	\N	Chaniat	43
16769	056	\N	Chanteuges	43
16770	057	\N	Chapelle-Bertin	43
16771	058	\N	Chapelle-d'Aurec	43
16772	059	\N	Chapelle-Geneste	43
16773	060	\N	Charraix	43
16774	061	\N	Chaspinhac	43
16775	062	\N	Chaspuzac	43
16776	063	\N	Chassagnes	43
16777	064	\N	Chassignolles	43
16778	065	\N	Chastel	43
16779	066	\N	Chaudeyrolles	43
16780	067	\N	Chavaniac-Lafayette	43
16781	068	\N	Chazelles	43
16782	069	\N	Chenereilles	43
16783	070	\N	Chilhac	43
16784	071	\N	Chomelix	43
16785	072	\N	Chomette	43
16786	073	\N	Cistrières	43
16787	074	\N	Cohade	43
16788	075	\N	Collat	43
16789	076	\N	Connangles	43
16790	077	\N	Costaros	43
16791	078	\N	Coubon	43
16792	079	\N	Couteuges	43
16793	080	\N	Craponne-sur-Arzon	43
16794	081	\N	Croisances	43
16795	082	\N	Cronce	43
16796	083	\N	Cubelles	43
16797	084	\N	Cussac-sur-Loire	43
16798	085	\N	Desges	43
16799	086	\N	Domeyrat	43
16800	087	\N	Dunières	43
16801	088	\N	Espalem	43
16802	089	\N	Espaly-Saint-Marcel	43
16803	090	\N	Esplantas	43
16804	091	\N	Estables	43
16805	092	\N	Fay-sur-Lignon	43
16806	093	\N	Félines	43
16807	094	\N	Ferrussac	43
16808	095	\N	Fix-Saint-Geneys	43
16809	096	\N	Fontannes	43
16810	097	\N	Freycenet-la-Cuche	43
16811	098	\N	Freycenet-la-Tour	43
16812	099	\N	Frugerès-les-Mines	43
16813	100	\N	Frugières-le-Pin	43
16814	101	\N	Goudet	43
16815	102	\N	Grazac	43
16816	103	\N	Grenier-Montgon	43
16817	104	\N	Grèzes	43
16818	105	\N	Javaugues	43
16819	106	\N	Jax	43
16820	107	\N	Josat	43
16821	108	\N	Jullianges	43
16822	109	\N	Lafarre	43
16823	110	\N	Lamothe	43
16824	111	\N	Landos	43
16825	112	\N	Langeac	43
16826	113	\N	Lantriac	43
16827	114	\N	Lapte	43
16828	115	\N	Laussonne	43
16829	116	\N	Laval-sur-Doulon	43
16830	117	\N	Lavaudieu	43
16831	118	\N	Lavoûte-Chilhac	43
16832	119	\N	Lavoûte-sur-Loire	43
16833	120	\N	Lempdes-sur-Allagnon	43
16834	121	\N	Léotoing	43
16835	122	\N	Lissac	43
16836	123	\N	Lorlanges	43
16837	124	\N	Loudes	43
16838	125	\N	Lubilhac	43
16839	126	\N	Malrevers	43
16840	127	\N	Malvalette	43
16841	128	\N	Malvières	43
16842	129	\N	Mas-de-Tence	43
16843	130	\N	Mazet-Saint-Voy	43
16844	131	\N	Mazerat-Aurouze	43
16845	132	\N	Mazeyrat-d'Allier	43
16846	133	\N	Mercur	43
16847	134	\N	Mézères	43
16848	135	\N	Monastier-sur-Gazeille	43
16849	136	\N	Monistrol-d'Allier	43
16850	137	\N	Monistrol-sur-Loire	43
16851	138	\N	Monlet	43
16852	139	\N	Montclard	43
16853	140	\N	Monteil	43
16854	141	\N	Montfaucon-en-Velay	43
16855	142	\N	Montregard	43
16856	143	\N	Montusclat	43
16857	144	\N	Moudeyres	43
16858	145	\N	Ouides	43
16859	147	\N	Paulhac	43
16860	148	\N	Paulhaguet	43
16861	149	\N	Pébrac	43
16862	150	\N	Pertuis	43
16863	151	\N	Pinols	43
16864	152	\N	Polignac	43
16865	153	\N	Pont-Salomon	43
16866	154	\N	Pradelles	43
16867	155	\N	Prades	43
16868	156	\N	Présailles	43
16869	157	\N	Puy-en-Velay	43
16870	158	\N	Queyrières	43
16871	159	\N	Raucoules	43
16872	160	\N	Rauret	43
16873	162	\N	Retournac	43
16874	163	\N	Riotord	43
16875	164	\N	Roche-en-Régnier	43
16876	165	\N	Rosières	43
16877	166	\N	Saint-André-de-Chalencon	43
16878	167	\N	Saint-Arcons-d'Allier	43
16879	168	\N	Saint-Arcons-de-Barges	43
16880	169	\N	Saint-Austremoine	43
16881	170	\N	Saint-Beauzire	43
16882	171	\N	Saint-Bérain	43
16883	172	\N	Saint-Bonnet-le-Froid	43
16884	173	\N	Saint-Christophe-d'Allier	43
16885	174	\N	Saint-Christophe-sur-Dolaison	43
16886	175	\N	Saint-Cirgues	43
16887	176	\N	Saint-Didier-d'Allier	43
16888	177	\N	Saint-Didier-en-Velay	43
16889	178	\N	Saint-Didier-sur-Doulon	43
16890	180	\N	Saint-Étienne-du-Vigan	43
16891	181	\N	Saint-Étienne-Lardeyrol	43
16892	182	\N	Saint-Étienne-sur-Blesle	43
16893	183	\N	Sainte-Eugénie-de-Villeneuve	43
16894	184	\N	Saint-Ferréol-d'Auroure	43
16895	185	\N	Sainte-Florine	43
16896	186	\N	Saint-Front	43
16897	187	\N	Saint-Geneys-près-Saint-Paulien	43
16898	188	\N	Saint-Georges-d'Aurac	43
16899	189	\N	Saint-Georges-Lagricol	43
16900	190	\N	Saint-Germain-Laprade	43
16901	191	\N	Saint-Géron	43
16902	192	\N	Saint-Haon	43
16903	193	\N	Saint-Hilaire	43
16904	194	\N	Saint-Hostien	43
16905	195	\N	Saint-Ilpize	43
16906	196	\N	Saint-Jean-d'Aubrigoux	43
16907	197	\N	Saint-Jean-de-Nay	43
16908	198	\N	Saint-Jean-Lachalm	43
16909	199	\N	Saint-Jeures	43
16910	200	\N	Saint-Julien-Chapteuil	43
16911	201	\N	Saint-Julien-d'Ance	43
16912	202	\N	Saint-Julien-des-Chazes	43
16913	203	\N	Saint-Julien-du-Pinet	43
16914	204	\N	Saint-Julien-Molhesabate	43
16915	205	\N	Saint-Just-Malmont	43
16916	206	\N	Saint-Just-près-Brioude	43
16917	207	\N	Saint-Laurent-Chabreuges	43
16918	208	\N	Sainte-Marguerite	43
16919	210	\N	Saint-Martin-de-Fugères	43
16920	211	\N	Saint-Maurice-de-Lignon	43
16921	212	\N	Saint-Pal-de-Chalencon	43
16922	213	\N	Saint-Pal-de-Mons	43
16923	214	\N	Saint-Pal-de-Senouire	43
16924	215	\N	Saint-Paul-de-Tartas	43
16925	216	\N	Saint-Paulien	43
16926	217	\N	Saint-Pierre-du-Champ	43
16927	218	\N	Saint-Pierre-Eynac	43
16928	219	\N	Saint-Préjet-Armandon	43
16929	220	\N	Saint-Préjet-d'Allier	43
16930	221	\N	Saint-Privat-d'Allier	43
16931	222	\N	Saint-Privat-du-Dragon	43
16932	223	\N	Saint-Romain-Lachalm	43
16933	224	\N	Sainte-Sigolène	43
16934	225	\N	Saint-Vénérand	43
16935	226	\N	Saint-Vert	43
16936	227	\N	Saint-Victor-Malescours	43
16937	228	\N	Saint-Victor-sur-Arlanc	43
16938	229	\N	Saint-Vidal	43
16939	230	\N	Saint-Vincent	43
16940	231	\N	Salettes	43
16941	232	\N	Salzuit	43
16942	233	\N	Sanssac-l'Église	43
16943	234	\N	Saugues	43
16944	236	\N	Séauve-sur-Semène	43
16945	237	\N	Sembadel	43
16946	238	\N	Séneujols	43
16947	239	\N	Siaugues-Sainte-Marie	43
16948	240	\N	Solignac-sous-Roche	43
16949	241	\N	Solignac-sur-Loire	43
16950	242	\N	Tailhac	43
16951	244	\N	Tence	43
16952	245	\N	Thoras	43
16953	246	\N	Tiranges	43
16954	247	\N	Torsiac	43
16955	249	\N	Valprivas	43
16956	250	\N	Vals-le-Chastel	43
16957	251	\N	Vals-près-le-Puy	43
16958	252	\N	Varennes-Saint-Honorat	43
16959	253	\N	Vastres	43
16960	254	\N	Vazeilles-Limandre	43
16961	255	\N	Vazeilles-près-Saugues	43
16962	256	\N	Venteuges	43
16963	257	\N	Vergezac	43
16964	258	\N	Vergongheon	43
16965	259	\N	Vernassal	43
16966	260	\N	Vernet	43
16967	261	\N	Vézézoux	43
16968	262	\N	Vieille-Brioude	43
16969	263	\N	Vielprat	43
16970	264	\N	Villeneuve-d'Allier	43
16971	265	\N	Villettes	43
16972	267	\N	Vorey	43
16973	268	\N	Yssingeaux	43
16974	001	\N	Abbaretz	44
16975	002	\N	Aigrefeuille-sur-Maine	44
16976	003	\N	Ancenis	44
16977	004	\N	Anetz	44
16978	005	\N	Arthon-en-Retz	44
16979	006	\N	Assérac	44
16980	007	\N	Avessac	44
16981	008	\N	Barbechat	44
16982	009	\N	Basse-Goulaine	44
16983	010	\N	Batz-sur-Mer	44
16984	011	\N	Belligné	44
16985	012	\N	Bernerie-en-Retz	44
16986	013	\N	Besné	44
16987	014	\N	Bignon	44
16988	015	\N	Blain	44
16989	016	\N	Boissière-du-Doré	44
16990	017	\N	Bonnuvre	44
16991	018	\N	Bouaye	44
16992	019	\N	Bouée	44
16993	020	\N	Bouguenais	44
16994	021	\N	Bourgneuf-en-Retz	44
16995	022	\N	Boussay	44
16996	023	\N	Bouvron	44
16997	024	\N	Brains	44
16998	025	\N	Campbon	44
16999	026	\N	Carquefou	44
17000	027	\N	Casson	44
17001	028	\N	Cellier	44
17002	029	\N	Chapelle-Basse-Mer	44
17003	030	\N	Chapelle-des-Marais	44
17004	031	\N	Chapelle-Glain	44
17005	032	\N	Chapelle-Heulin	44
17006	033	\N	Chapelle-Launay	44
17007	034	\N	Chapelle-Saint-Sauveur	44
17008	035	\N	Chapelle-sur-Erdre	44
17009	036	\N	Châteaubriant	44
17010	037	\N	Château-Thébaud	44
17011	038	\N	Chauvé	44
17012	039	\N	Cheix-en-Retz	44
17013	040	\N	Chéméré	44
17014	041	\N	Chevrolière	44
17015	043	\N	Clisson	44
17016	044	\N	Conquereuil	44
17017	045	\N	Cordemais	44
17018	046	\N	Corsept	44
17019	047	\N	Couëron	44
17020	048	\N	Couffé	44
17021	049	\N	Croisic	44
17022	050	\N	Crossac	44
17023	051	\N	Derval	44
17024	052	\N	Donges	44
17025	053	\N	Drefféac	44
17026	054	\N	Erbray	44
17027	055	\N	Baule-Escoublac	44
17028	056	\N	Fay-de-Bretagne	44
17029	057	\N	Fégréac	44
17030	058	\N	Fercé	44
17031	059	\N	Fresnay-en-Retz	44
17032	060	\N	Fresne-sur-Loire	44
17033	061	\N	Frossay	44
17034	062	\N	Gâvre	44
17035	063	\N	Gétigné	44
17036	064	\N	Gorges	44
17037	065	\N	Grand-Auverné	44
17038	066	\N	Grandchamps-des-Fontaines	44
17039	067	\N	Guémené-Penfao	44
17040	068	\N	Guenrouet	44
17041	069	\N	Guérande	44
17042	070	\N	Haie-Fouassière	44
17043	071	\N	Haute-Goulaine	44
17044	072	\N	Herbignac	44
17045	073	\N	Héric	44
17046	074	\N	Indre	44
17047	075	\N	Issé	44
17048	076	\N	Jans	44
17049	077	\N	Joué-sur-Erdre	44
17050	078	\N	Juigné-des-Moutiers	44
17051	079	\N	Landreau	44
17052	080	\N	Lavau-sur-Loire	44
17053	081	\N	Legé	44
17054	082	\N	Ligné	44
17055	083	\N	Limouzinière	44
17056	084	\N	Loroux-Bottereau	44
17057	085	\N	Louisfert	44
17058	086	\N	Lusanger	44
17059	087	\N	Machecoul	44
17060	088	\N	Maisdon-sur-Sèvre	44
17061	089	\N	Malville	44
17062	090	\N	Marne	44
17063	091	\N	Marsac-sur-Don	44
17064	092	\N	Massérac	44
17065	093	\N	Maumusson	44
17066	094	\N	Mauves-sur-Loire	44
17067	095	\N	Meilleraye-de-Bretagne	44
17068	096	\N	Mésanger	44
17069	097	\N	Mesquer	44
17070	098	\N	Missillac	44
17071	099	\N	Moisdon-la-Rivière	44
17072	100	\N	Monnières	44
17073	101	\N	Montagne	44
17074	102	\N	Montbert	44
17075	103	\N	Montoir-de-Bretagne	44
17076	104	\N	Montrelais	44
17077	105	\N	Mouais	44
17078	106	\N	Moutiers-en-Retz	44
17079	107	\N	Mouzeil	44
17080	108	\N	Mouzillon	44
17081	109	\N	Nantes	44
17082	110	\N	Nort-sur-Erdre	44
17083	111	\N	Notre-Dame-des-Landes	44
17084	112	\N	Noyal-sur-Brutz	44
17085	113	\N	Nozay	44
17086	114	\N	Orvault	44
17087	115	\N	Oudon	44
17088	116	\N	Paimbuf	44
17089	117	\N	Pallet	44
17090	118	\N	Pannecé	44
17091	119	\N	Paulx	44
17092	120	\N	Pellerin	44
17093	121	\N	Petit-Auverné	44
17094	122	\N	Petit-Mars	44
17095	123	\N	Pierric	44
17096	124	\N	Pin	44
17097	125	\N	Piriac-sur-Mer	44
17098	126	\N	Plaine-sur-Mer	44
17099	127	\N	Planche	44
17100	128	\N	Plessé	44
17101	129	\N	Pontchâteau	44
17102	130	\N	Pont-Saint-Martin	44
17103	131	\N	Pornic	44
17104	132	\N	Pornichet	44
17105	133	\N	Port-Saint-Père	44
17106	134	\N	Pouillé-les-Côteaux	44
17107	135	\N	Pouliguen	44
17108	136	\N	Préfailles	44
17109	137	\N	Prinquiau	44
17110	138	\N	Puceul	44
17111	139	\N	Quilly	44
17112	140	\N	Regrippière	44
17113	141	\N	Remaudière	44
17114	142	\N	Remouillé	44
17115	143	\N	Rezé	44
17116	144	\N	Riaillé	44
17117	145	\N	Rouans	44
17118	146	\N	Rougé	44
17119	147	\N	Rouxière	44
17120	148	\N	Ruffigné	44
17121	149	\N	Saffré	44
17122	150	\N	Saint-Aignan-Grandlieu	44
17123	151	\N	Saint-André-des-Eaux	44
17124	152	\N	Sainte-Anne-sur-Brivet	44
17125	153	\N	Saint-Aubin-des-Châteaux	44
17126	154	\N	Saint-Brevin-les-Pins	44
17127	155	\N	Saint-Colomban	44
17128	156	\N	Corcoué-sur-Logne	44
17129	157	\N	Saint-Étienne-de-Mer-Morte	44
17130	158	\N	Saint-Étienne-de-Montluc	44
17131	159	\N	Saint-Fiacre-sur-Maine	44
17132	160	\N	Saint-Géréon	44
17133	161	\N	Saint-Gildas-des-Bois	44
17134	162	\N	Saint-Herblain	44
17135	163	\N	Saint-Herblon	44
17136	164	\N	Saint-Hilaire-de-Chaléons	44
17137	165	\N	Saint-Hilaire-de-Clisson	44
17138	166	\N	Saint-Jean-de-Boiseau	44
17139	168	\N	Saint-Joachim	44
17140	169	\N	Saint-Julien-de-Concelles	44
17141	170	\N	Saint-Julien-de-Vouvantes	44
17142	171	\N	Saint-Léger-les-Vignes	44
17143	172	\N	Sainte-Luce-sur-Loire	44
17144	173	\N	Saint-Lumine-de-Clisson	44
17145	174	\N	Saint-Lumine-de-Coutais	44
17146	175	\N	Saint-Lyphard	44
17147	176	\N	Saint-Malo-de-Guersac	44
17148	178	\N	Saint-Mars-de-Coutais	44
17149	179	\N	Saint-Mars-du-Désert	44
17150	180	\N	Saint-Mars-la-Jaille	44
17151	181	\N	Saint-Même-le-Tenu	44
17152	182	\N	Saint-Michel-Chef-Chef	44
17153	183	\N	Saint-Molf	44
17154	184	\N	Saint-Nazaire	44
17155	185	\N	Saint-Nicolas-de-Redon	44
17156	186	\N	Sainte-Pazanne	44
17157	187	\N	Saint-Père-en-Retz	44
17158	188	\N	Saint-Philbert-de-Grand-Lieu	44
17159	189	\N	Sainte-Reine-de-Bretagne	44
17160	190	\N	Saint-Sébastien-sur-Loire	44
17161	191	\N	Saint-Sulpice-des-Landes	44
17162	192	\N	Saint-Viaud	44
17163	193	\N	Saint-Vincent-des-Landes	44
17164	194	\N	Sautron	44
17165	195	\N	Savenay	44
17166	196	\N	Sévérac	44
17167	197	\N	Sion-les-Mines	44
17168	198	\N	Sorinières	44
17169	199	\N	Soudan	44
17170	200	\N	Soulvache	44
17171	201	\N	Sucé-sur-Erdre	44
17172	202	\N	Teillé	44
17173	203	\N	Temple-de-Bretagne	44
17174	204	\N	Thouaré-sur-Loire	44
17175	205	\N	Touches	44
17176	206	\N	Touvois	44
17177	207	\N	Trans-sur-Erdre	44
17178	208	\N	Treffieux	44
17179	209	\N	Treillières	44
17180	210	\N	Trignac	44
17181	211	\N	Turballe	44
17182	212	\N	Vallet	44
17183	213	\N	Varades	44
17184	214	\N	Vay	44
17185	215	\N	Vertou	44
17186	216	\N	Vieillevigne	44
17187	217	\N	Vigneux-de-Bretagne	44
17188	218	\N	Villepot	44
17189	219	\N	Vritz	44
17190	220	\N	Vue	44
17191	221	\N	Chevallerais	44
17192	222	\N	Roche-Blanche	44
17193	223	\N	Geneston	44
17194	224	\N	Grigonnais	44
17195	001	\N	Adon	45
17196	002	\N	Aillant-sur-Milleron	45
17197	004	\N	Amilly	45
17198	005	\N	Andonville	45
17199	006	\N	Ardon	45
17200	008	\N	Artenay	45
17201	009	\N	Aschères-le-Marché	45
17202	010	\N	Ascoux	45
17203	011	\N	Attray	45
17204	012	\N	Audeville	45
17205	013	\N	Augerville-la-Rivière	45
17206	014	\N	Aulnay-la-Rivière	45
17207	015	\N	Autruy-sur-Juine	45
17208	016	\N	Autry-le-Châtel	45
17209	017	\N	Auvilliers-en-Gâtinais	45
17210	018	\N	Auxy	45
17211	019	\N	Baccon	45
17212	020	\N	Bardon	45
17213	021	\N	Barville-en-Gâtinais	45
17214	022	\N	Batilly-en-Gâtinais	45
17215	023	\N	Batilly-en-Puisaye	45
17216	024	\N	Baule	45
17217	025	\N	Bazoches-les-Gallerandes	45
17218	026	\N	Bazoches-sur-le-Betz	45
17219	027	\N	Beauchamps-sur-Huillard	45
17220	028	\N	Beaugency	45
17221	029	\N	Beaulieu-sur-Loire	45
17222	030	\N	Beaune-la-Rolande	45
17223	031	\N	Bellegarde	45
17224	032	\N	Bignon-Mirabeau	45
17225	033	\N	Boësses	45
17226	034	\N	Boigny-sur-Bionne	45
17227	035	\N	Boiscommun	45
17228	036	\N	Boismorand	45
17229	037	\N	Boisseaux	45
17230	038	\N	Bondaroy	45
17231	039	\N	Bonnée	45
17232	040	\N	Bonny-sur-Loire	45
17233	041	\N	Bordeaux-en-Gâtinais	45
17234	042	\N	Bordes	45
17235	043	\N	Bou	45
17236	044	\N	Bougy-lez-Neuville	45
17237	045	\N	Bouilly-en-Gâtinais	45
17238	046	\N	Boulay-les-Barres	45
17239	047	\N	Bouzonville-aux-Bois	45
17240	049	\N	Bouzy-la-Forêt	45
17241	050	\N	Boynes	45
17242	051	\N	Bray-en-Val	45
17243	052	\N	Breteau	45
17244	053	\N	Briare	45
17245	054	\N	Briarres-sur-Essonne	45
17246	055	\N	Bricy	45
17247	056	\N	Bromeilles	45
17248	057	\N	Labrosse	45
17249	058	\N	Bucy-le-Roi	45
17250	059	\N	Bucy-Saint-Liphard	45
17251	060	\N	Bussière	45
17252	061	\N	Cepoy	45
17253	062	\N	Cercottes	45
17254	063	\N	Cerdon	45
17255	064	\N	Cernoy-en-Berry	45
17256	065	\N	Césarville-Dossainville	45
17257	066	\N	Chailly-en-Gâtinais	45
17258	067	\N	Chaingy	45
17259	068	\N	Châlette-sur-Loing	45
17260	069	\N	Chambon-la-Forêt	45
17261	070	\N	Champoulet	45
17262	072	\N	Chanteau	45
17263	073	\N	Chantecoq	45
17264	074	\N	Chapelle-Onzerain	45
17265	075	\N	Chapelle-Saint-Mesmin	45
17266	076	\N	Chapelle-Saint-Sépulcre	45
17267	077	\N	Chapelle-sur-Aveyron	45
17268	078	\N	Chapelon	45
17269	079	\N	Charme	45
17270	080	\N	Charmont-en-Beauce	45
17271	081	\N	Charsonville	45
17272	082	\N	Châteauneuf-sur-Loire	45
17273	083	\N	Château-Renard	45
17274	084	\N	Châtenoy	45
17275	085	\N	Châtillon-Coligny	45
17276	086	\N	Châtillon-le-Roi	45
17277	087	\N	Châtillon-sur-Loire	45
17278	088	\N	Chaussy	45
17279	089	\N	Chécy	45
17280	091	\N	Chevannes	45
17281	092	\N	Chevillon-sur-Huillard	45
17282	093	\N	Chevilly	45
17283	094	\N	Chevry-sous-le-Bignon	45
17284	095	\N	Chilleurs-aux-Bois	45
17285	096	\N	Choux	45
17286	097	\N	Chuelles	45
17287	098	\N	Cléry-Saint-André	45
17288	099	\N	Coinces	45
17289	100	\N	Combleux	45
17290	101	\N	Combreux	45
17291	102	\N	Conflans-sur-Loing	45
17292	103	\N	Corbeilles	45
17293	104	\N	Corquilleroy	45
17294	105	\N	Cortrat	45
17295	106	\N	Coudray	45
17296	107	\N	Coudroy	45
17297	108	\N	Coullons	45
17298	109	\N	Coulmiers	45
17299	110	\N	Courcelles	45
17300	111	\N	Courcy-aux-Loges	45
17301	112	\N	Cour-Marigny	45
17302	113	\N	Courtemaux	45
17303	114	\N	Courtempierre	45
17304	115	\N	Courtenay	45
17305	116	\N	Cravant	45
17306	118	\N	Crottes-en-Pithiverais	45
17307	119	\N	Dadonville	45
17308	120	\N	Dammarie-en-Puisaye	45
17309	121	\N	Dammarie-sur-Loing	45
17310	122	\N	Dampierre-en-Burly	45
17311	123	\N	Darvoy	45
17312	124	\N	Desmonts	45
17313	125	\N	Dimancheville	45
17314	126	\N	Donnery	45
17315	127	\N	Dordives	45
17316	129	\N	Douchy	45
17317	130	\N	Dry	45
17318	131	\N	Échilleuses	45
17319	132	\N	Égry	45
17320	133	\N	Engenville	45
17321	134	\N	Épieds-en-Beauce	45
17322	135	\N	Erceville	45
17323	136	\N	Ervauville	45
17324	137	\N	Escrennes	45
17325	138	\N	Escrignelles	45
17326	139	\N	Estouy	45
17327	141	\N	Faverelles	45
17328	142	\N	Fay-aux-Loges	45
17329	143	\N	Feins-en-Gâtinais	45
17330	144	\N	Férolles	45
17331	145	\N	Ferrières-en-Gâtinais	45
17332	146	\N	Ferté-Saint-Aubin	45
17333	147	\N	Fleury-les-Aubrais	45
17334	148	\N	Fontenay-sur-Loing	45
17335	149	\N	Foucherolles	45
17336	150	\N	Fréville-du-Gâtinais	45
17337	151	\N	Gaubertin	45
17338	152	\N	Gémigny	45
17339	153	\N	Germigny-des-Prés	45
17340	154	\N	Gidy	45
17341	155	\N	Gien	45
17342	156	\N	Girolles	45
17343	157	\N	Givraines	45
17344	158	\N	Gondreville	45
17345	159	\N	Grangermont	45
17346	160	\N	Greneville-en-Beauce	45
17347	161	\N	Griselles	45
17348	162	\N	Guigneville	45
17349	164	\N	Guilly	45
17350	165	\N	Gy-les-Nonains	45
17351	166	\N	Huêtre	45
17352	167	\N	Huisseau-sur-Mauves	45
17353	168	\N	Ingrannes	45
17354	169	\N	Ingré	45
17355	170	\N	Intville-la-Guétard	45
17356	171	\N	Isdes	45
17357	173	\N	Jargeau	45
17358	174	\N	Jouy-en-Pithiverais	45
17359	175	\N	Jouy-le-Potier	45
17360	176	\N	Juranville	45
17361	177	\N	Laas	45
17362	178	\N	Ladon	45
17363	179	\N	Lailly-en-Val	45
17364	180	\N	Langesse	45
17365	181	\N	Léouville	45
17366	182	\N	Ligny-le-Ribault	45
17367	183	\N	Lion-en-Beauce	45
17368	184	\N	Lion-en-Sullias	45
17369	185	\N	Lombreuil	45
17370	186	\N	Lorcy	45
17371	187	\N	Lorris	45
17372	188	\N	Loury	45
17373	189	\N	Louzouer	45
17374	190	\N	Mainvilliers	45
17375	191	\N	Malesherbes	45
17376	192	\N	Manchecourt	45
17377	193	\N	Marcilly-en-Villette	45
17378	194	\N	Mardié	45
17379	195	\N	Mareau-aux-Bois	45
17380	196	\N	Mareau-aux-Prés	45
17381	197	\N	Marigny-les-Usages	45
17382	198	\N	Marsainvilliers	45
17383	199	\N	Melleroy	45
17384	200	\N	Ménestreau-en-Villette	45
17385	201	\N	Mérinville	45
17386	202	\N	Messas	45
17387	203	\N	Meung-sur-Loire	45
17388	204	\N	Mézières-lez-Cléry	45
17389	205	\N	Mézières-en-Gâtinais	45
17390	206	\N	Mignères	45
17391	207	\N	Mignerette	45
17392	208	\N	Montargis	45
17393	209	\N	Montbarrois	45
17394	210	\N	Montbouy	45
17395	211	\N	Montcorbon	45
17396	212	\N	Montcresson	45
17397	213	\N	Montereau	45
17398	214	\N	Montigny	45
17399	215	\N	Montliard	45
17400	216	\N	Mormant-sur-Vernisson	45
17401	217	\N	Morville-en-Beauce	45
17402	218	\N	Moulinet-sur-Solin	45
17403	219	\N	Moulon	45
17404	220	\N	Nancray-sur-Rimarde	45
17405	221	\N	Nangeville	45
17406	222	\N	Nargis	45
17407	223	\N	Nesploy	45
17408	224	\N	Neuville-aux-Bois	45
17409	225	\N	Neuville-sur-Essonne	45
17410	226	\N	Neuvy-en-Sullias	45
17411	227	\N	Nevoy	45
17412	228	\N	Nibelle	45
17413	229	\N	Nogent-sur-Vernisson	45
17414	230	\N	Noyers	45
17415	231	\N	Oison	45
17416	232	\N	Olivet	45
17417	233	\N	Ondreville-sur-Essonne	45
17418	234	\N	Orléans	45
17419	235	\N	Ormes	45
17420	236	\N	Orveau-Bellesauve	45
17421	237	\N	Orville	45
17422	238	\N	Ousson-sur-Loire	45
17423	239	\N	Oussoy-en-Gâtinais	45
17424	240	\N	Outarville	45
17425	241	\N	Ouvrouer-les-Champs	45
17426	242	\N	Ouzouer-des-Champs	45
17427	243	\N	Ouzouer-sous-Bellegarde	45
17428	244	\N	Ouzouer-sur-Loire	45
17429	245	\N	Ouzouer-sur-Trézée	45
17430	246	\N	Pannecières	45
17431	247	\N	Pannes	45
17432	248	\N	Patay	45
17433	249	\N	Paucourt	45
17434	250	\N	Pers-en-Gâtinais	45
17435	251	\N	Pierrefitte-ès-Bois	45
17436	252	\N	Pithiviers	45
17437	253	\N	Pithiviers-le-Vieil	45
17438	254	\N	Poilly-lez-Gien	45
17439	255	\N	Préfontaines	45
17440	256	\N	Presnoy	45
17441	257	\N	Pressigny-les-Pins	45
17442	258	\N	Puiseaux	45
17443	259	\N	Quiers-sur-Bézonde	45
17444	260	\N	Ramoulu	45
17445	261	\N	Rebréchien	45
17446	262	\N	Rouvray-Sainte-Croix	45
17447	263	\N	Rouvres-Saint-Jean	45
17448	264	\N	Rozières-en-Beauce	45
17449	265	\N	Rosoy-le-Vieil	45
17450	266	\N	Ruan	45
17451	267	\N	Saint-Aignan-des-Gués	45
17452	268	\N	Saint-Aignan-le-Jaillard	45
17453	269	\N	Saint-Ay	45
17454	270	\N	Saint-Benoît-sur-Loire	45
17455	271	\N	Saint-Brisson-sur-Loire	45
17456	272	\N	Saint-Cyr-en-Val	45
17457	273	\N	Saint-Denis-de-l'Hôtel	45
17458	274	\N	Saint-Denis-en-Val	45
17459	275	\N	Saint-Firmin-des-Bois	45
17460	276	\N	Saint-Firmin-sur-Loire	45
17461	277	\N	Saint-Florent	45
17462	278	\N	Sainte-Geneviève-des-Bois	45
17463	279	\N	Saint-Germain-des-Prés	45
17464	280	\N	Saint-Gondon	45
17465	281	\N	Saint-Hilaire-les-Andrésis	45
17466	282	\N	Saint-Hilaire-Saint-Mesmin	45
17467	283	\N	Saint-Hilaire-sur-Puiseaux	45
17468	284	\N	Saint-Jean-de-Braye	45
17469	285	\N	Saint-Jean-de-la-Ruelle	45
17470	286	\N	Saint-Jean-le-Blanc	45
17471	287	\N	Saint-Loup-de-Gonois	45
17472	288	\N	Saint-Loup-des-Vignes	45
17473	289	\N	Saint-Lyé-la-Forêt	45
17474	290	\N	Saint-Martin-d'Abbat	45
17475	291	\N	Saint-Martin-sur-Ocre	45
17476	292	\N	Saint-Maurice-sur-Aveyron	45
17477	293	\N	Saint-Maurice-sur-Fessard	45
17478	294	\N	Saint-Michel	45
17479	296	\N	Saint-Péravy-la-Colombe	45
17480	297	\N	Saint-Père-sur-Loire	45
17481	298	\N	Saint-Pryvé-Saint-Mesmin	45
17482	299	\N	Saint-Sigismond	45
17483	300	\N	Sandillon	45
17484	301	\N	Santeau	45
17485	302	\N	Saran	45
17486	303	\N	Sceaux-du-Gâtinais	45
17487	305	\N	Seichebrières	45
17488	306	\N	Selle-en-Hermoy	45
17489	307	\N	Selle-sur-le-Bied	45
17490	308	\N	Semoy	45
17491	309	\N	Sennely	45
17492	310	\N	Sermaises	45
17493	311	\N	Sigloy	45
17494	312	\N	Solterre	45
17495	313	\N	Sougy	45
17496	314	\N	Sully-la-Chapelle	45
17497	315	\N	Sully-sur-Loire	45
17498	316	\N	Sury-aux-Bois	45
17499	317	\N	Tavers	45
17500	320	\N	Thignonville	45
17501	321	\N	Thimory	45
17502	322	\N	Thorailles	45
17503	323	\N	Thou	45
17504	324	\N	Tigy	45
17505	325	\N	Tivernon	45
17506	326	\N	Tournoisis	45
17507	327	\N	Traînou	45
17508	328	\N	Treilles-en-Gâtinais	45
17509	329	\N	Triguères	45
17510	330	\N	Trinay	45
17511	331	\N	Vannes-sur-Cosson	45
17512	332	\N	Varennes-Changy	45
17513	333	\N	Vennecy	45
17514	334	\N	Vieilles-Maisons-sur-Joudry	45
17515	335	\N	Vienne-en-Val	45
17516	336	\N	Viglain	45
17517	337	\N	Villamblain	45
17518	338	\N	Villemandeur	45
17519	339	\N	Villemoutiers	45
17520	340	\N	Villemurlin	45
17521	341	\N	Villeneuve-sur-Conie	45
17522	342	\N	Villereau	45
17523	343	\N	Villevoques	45
17524	344	\N	Villorceau	45
17525	345	\N	Vimory	45
17526	346	\N	Vitry-aux-Loges	45
17527	347	\N	Vrigny	45
17528	348	\N	Yèvre-la-Ville	45
17529	001	\N	Albas	46
17530	002	\N	Albiac	46
17531	003	\N	Alvignac	46
17532	004	\N	Anglars	46
17533	005	\N	Anglars-Juillac	46
17534	006	\N	Anglars-Nozac	46
17535	007	\N	Arcambal	46
17536	008	\N	Arques	46
17537	009	\N	Assier	46
17538	010	\N	Aujols	46
17539	011	\N	Autoire	46
17540	012	\N	Aynac	46
17541	013	\N	Bach	46
17542	014	\N	Bagat-en-Quercy	46
17543	015	\N	Bagnac-sur-Célé	46
17544	016	\N	Baladou	46
17545	017	\N	Bannes	46
17546	018	\N	Bastit	46
17547	019	\N	Beaumat	46
17548	020	\N	Beauregard	46
17549	021	\N	Béduer	46
17550	022	\N	Bélaye	46
17551	023	\N	Belfort-du-Quercy	46
17552	024	\N	Belmont-Bretenoux	46
17553	025	\N	Belmontet	46
17554	026	\N	Belmont-Sainte-Foi	46
17555	027	\N	Berganty	46
17556	028	\N	Bétaille	46
17557	029	\N	Biars-sur-Cère	46
17558	030	\N	Bio	46
17559	031	\N	Blars	46
17560	032	\N	Boissières	46
17561	033	\N	Boulvé	46
17562	034	\N	Bourg	46
17563	035	\N	Boussac	46
17564	036	\N	Bouyssou	46
17565	037	\N	Bouziès	46
17566	038	\N	Bretenoux	46
17567	039	\N	Brengues	46
17568	040	\N	Cabrerets	46
17569	041	\N	Cadrieu	46
17570	042	\N	Cahors	46
17571	043	\N	Cahus	46
17572	044	\N	Caillac	46
17573	045	\N	Cajarc	46
17574	046	\N	Calamane	46
17575	047	\N	Calès	46
17576	048	\N	Calviac	46
17577	049	\N	Calvignac	46
17578	050	\N	Cambayrac	46
17579	051	\N	Cambes	46
17580	052	\N	Camboulit	46
17581	053	\N	Camburat	46
17582	054	\N	Caniac-du-Causse	46
17583	055	\N	Capdenac	46
17584	056	\N	Carayac	46
17585	057	\N	Cardaillac	46
17586	058	\N	Carennac	46
17587	059	\N	Carlucet	46
17588	060	\N	Carnac-Rouffiac	46
17589	061	\N	Cassagnes	46
17590	062	\N	Castelfranc	46
17591	063	\N	Castelnau-Montratier	46
17592	064	\N	Catus	46
17593	065	\N	Cavagnac	46
17594	066	\N	Cazals	46
17595	067	\N	Cazillac	46
17596	068	\N	Cénevières	46
17597	069	\N	Cézac	46
17598	070	\N	Cieurac	46
17599	071	\N	Comiac	46
17600	072	\N	Concorès	46
17601	073	\N	Concots	46
17602	074	\N	Condat	46
17603	075	\N	Corn	46
17604	076	\N	Cornac	46
17605	077	\N	Cours	46
17606	078	\N	Couzou	46
17607	079	\N	Cras	46
17608	080	\N	Crayssac	46
17609	081	\N	Crégols	46
17610	082	\N	Cremps	46
17611	083	\N	Cressensac	46
17612	084	\N	Creysse	46
17613	085	\N	Cuzac	46
17614	086	\N	Cuzance	46
17615	087	\N	Dégagnac	46
17616	088	\N	Douelle	46
17617	089	\N	Duravel	46
17618	090	\N	Durbans	46
17619	091	\N	Escamps	46
17620	092	\N	Esclauzels	46
17621	093	\N	Espagnac-Sainte-Eulalie	46
17622	094	\N	Espédaillac	46
17623	095	\N	Espère	46
17624	096	\N	Espeyroux	46
17625	097	\N	Estal	46
17626	098	\N	Fajoles	46
17627	099	\N	Fargues	46
17628	100	\N	Faycelles	46
17629	101	\N	Felzins	46
17630	102	\N	Figeac	46
17631	103	\N	Flaugnac	46
17632	104	\N	Flaujac-Gare	46
17633	105	\N	Flaujac-Poujols	46
17634	106	\N	Floirac	46
17635	107	\N	Floressas	46
17636	108	\N	Fons	46
17637	109	\N	Fontanes	46
17638	110	\N	Fontanes-du-Causse	46
17639	111	\N	Fourmagnac	46
17640	112	\N	Francoulès	46
17641	113	\N	Frayssinet	46
17642	114	\N	Frayssinet-le-Gélat	46
17643	115	\N	Frayssinhes	46
17644	116	\N	Frontenac	46
17645	117	\N	Gagnac-sur-Cère	46
17646	118	\N	Gignac	46
17647	119	\N	Gigouzac	46
17648	120	\N	Gindou	46
17649	121	\N	Ginouillac	46
17650	122	\N	Gintrac	46
17651	123	\N	Girac	46
17652	124	\N	Glanes	46
17653	125	\N	Gorses	46
17654	126	\N	Goujounac	46
17655	127	\N	Gourdon	46
17656	128	\N	Gramat	46
17657	129	\N	Gréalou	46
17658	130	\N	Grézels	46
17659	131	\N	Grèzes	46
17660	132	\N	Issendolus	46
17661	133	\N	Issepts	46
17662	134	\N	Junies	46
17663	135	\N	Labastide-du-Haut-Mont	46
17664	136	\N	Labastide-du-Vert	46
17665	137	\N	Labastide-Marnhac	46
17666	138	\N	Labastide-Murat	46
17667	139	\N	Labathude	46
17668	140	\N	Laburgade	46
17669	141	\N	Lacam-d'Ourcet	46
17670	142	\N	Lacapelle-Cabanac	46
17671	143	\N	Lacapelle-Marival	46
17672	144	\N	Lacave	46
17673	145	\N	Lachapelle-Auzac	46
17674	146	\N	Ladirat	46
17675	147	\N	Lagardelle	46
17676	148	\N	Lalbenque	46
17677	149	\N	Lamagdelaine	46
17678	150	\N	Lamativie	46
17679	151	\N	Lamothe-Cassel	46
17680	152	\N	Lamothe-Fénelon	46
17681	153	\N	Lanzac	46
17682	154	\N	Laramière	46
17683	155	\N	Larnagol	46
17684	156	\N	Laroque-des-Arcs	46
17685	157	\N	Larroque-Toirac	46
17686	158	\N	Lascabanes	46
17687	159	\N	Latouille-Lentillac	46
17688	160	\N	Latronquière	46
17689	161	\N	Lauresses	46
17690	162	\N	Lauzès	46
17691	163	\N	Laval-de-Cère	46
17692	164	\N	Lavercantière	46
17693	165	\N	Lavergne	46
17694	166	\N	Lebreil	46
17695	167	\N	Lentillac-du-Causse	46
17696	168	\N	Lentillac-Saint-Blaise	46
17697	169	\N	Léobard	46
17698	170	\N	Leyme	46
17699	171	\N	Lherm	46
17700	172	\N	Lhospitalet	46
17701	173	\N	Limogne-en-Quercy	46
17702	174	\N	Linac	46
17703	175	\N	Lissac-et-Mouret	46
17704	176	\N	Livernon	46
17705	177	\N	Loubressac	46
17706	178	\N	Loupiac	46
17707	179	\N	Lugagnac	46
17708	180	\N	Lunan	46
17709	181	\N	Lunegarde	46
17710	182	\N	Luzech	46
17711	183	\N	Marcilhac-sur-Célé	46
17712	184	\N	Marminiac	46
17713	185	\N	Martel	46
17714	186	\N	Masclat	46
17715	187	\N	Mauroux	46
17716	188	\N	Maxou	46
17717	189	\N	Mayrinhac-Lentour	46
17718	190	\N	Mechmont	46
17719	191	\N	Mercuès	46
17720	192	\N	Meyronne	46
17721	193	\N	Miers	46
17722	194	\N	Milhac	46
17723	195	\N	Molières	46
17724	196	\N	Montamel	46
17725	197	\N	Montat	46
17726	198	\N	Montbrun	46
17727	199	\N	Montcabrier	46
17728	200	\N	Montcléra	46
17729	201	\N	Montcuq	46
17730	202	\N	Montdoumerc	46
17731	203	\N	Montet-et-Bouxal	46
17732	204	\N	Montfaucon	46
17733	205	\N	Montgesty	46
17734	206	\N	Montlauzun	46
17735	207	\N	Montredon	46
17736	208	\N	Montvalent	46
17737	209	\N	Nadaillac-de-Rouge	46
17738	210	\N	Nadillac	46
17739	211	\N	Nuzéjouls	46
17740	212	\N	Orniac	46
17741	213	\N	Padirac	46
17742	214	\N	Parnac	46
17743	215	\N	Payrac	46
17744	216	\N	Payrignac	46
17745	217	\N	Pern	46
17746	218	\N	Pescadoires	46
17747	219	\N	Peyrilles	46
17748	220	\N	Pinsac	46
17749	221	\N	Planioles	46
17750	222	\N	Pomarède	46
17751	223	\N	Pontcirq	46
17752	224	\N	Pradines	46
17753	225	\N	Prayssac	46
17754	226	\N	Prendeignes	46
17755	227	\N	Promilhanes	46
17756	228	\N	Prudhomat	46
17757	229	\N	Puybrun	46
17758	230	\N	Puyjourdes	46
17759	231	\N	Puy-l'Évêque	46
17760	232	\N	Quatre-Routes-du-Lot	46
17761	233	\N	Quissac	46
17762	234	\N	Rampoux	46
17763	235	\N	Reilhac	46
17764	236	\N	Reilhaguet	46
17765	237	\N	Reyrevignes	46
17766	238	\N	Rignac	46
17767	239	\N	Roc	46
17768	240	\N	Rocamadour	46
17769	241	\N	Rouffilhac	46
17770	242	\N	Rudelle	46
17771	243	\N	Rueyres	46
17772	244	\N	Sabadel-Latronquière	46
17773	245	\N	Sabadel-Lauzès	46
17774	246	\N	Saignes	46
17775	247	\N	Saillac	46
17776	248	\N	Sainte-Alauzie	46
17777	249	\N	Saint-Bressou	46
17778	250	\N	Saint-Caprais	46
17779	251	\N	Saint-Céré	46
17780	252	\N	Saint-Cernin	46
17781	253	\N	Saint-Chamarand	46
17782	254	\N	Saint-Chels	46
17783	255	\N	Saint-Cirgues	46
17784	256	\N	Saint-Cirq-Lapopie	46
17785	257	\N	Saint-Cirq-Madelon	46
17786	258	\N	Saint-Cirq-Souillaguet	46
17787	259	\N	Saint-Clair	46
17788	260	\N	Sainte-Colombe	46
17789	261	\N	Sainte-Croix	46
17790	262	\N	Saint-Cyprien	46
17791	263	\N	Saint-Daunès	46
17792	264	\N	Saint-Denis-Catus	46
17793	265	\N	Saint-Denis-lès-Martel	46
17794	266	\N	Saint-Félix	46
17795	267	\N	Saint-Germain-du-Bel-Air	46
17796	268	\N	Saint-Géry	46
17797	269	\N	Saint-Hilaire	46
17798	270	\N	Saint-Jean-de-Laur	46
17799	271	\N	Saint-Jean-Lespinasse	46
17800	272	\N	Saint-Jean-Mirabel	46
17801	273	\N	Saint-Laurent-les-Tours	46
17802	274	\N	Saint-Laurent-Lolmie	46
17803	275	\N	Saint-Martin-de-Vers	46
17804	276	\N	Saint-Martin-Labouval	46
17805	277	\N	Saint-Martin-le-Redon	46
17806	278	\N	Saint-Matré	46
17807	279	\N	Saint-Maurice-en-Quercy	46
17808	280	\N	Saint-Médard	46
17809	281	\N	Saint-Médard-de-Presque	46
17810	282	\N	Saint-Médard-Nicourby	46
17811	283	\N	Saint-Michel-de-Bannières	46
17812	284	\N	Saint-Michel-Loubéjou	46
17813	285	\N	Saint-Pantaléon	46
17814	286	\N	Saint-Paul-de-Vern	46
17815	287	\N	Saint-Paul-de-Loubressac	46
17816	288	\N	Saint-Perdoux	46
17817	289	\N	Saint-Pierre-Toirac	46
17818	290	\N	Saint-Projet	46
17819	291	\N	Saint-Sauveur-la-Vallée	46
17820	292	\N	Saint-Simon	46
17821	293	\N	Saint-Sozy	46
17822	294	\N	Saint-Sulpice	46
17823	295	\N	Saint-Vincent-du-Pendit	46
17824	296	\N	Saint-Vincent-Rive-d'Olt	46
17825	297	\N	Salviac	46
17826	298	\N	Sarrazac	46
17827	299	\N	Sauliac-sur-Célé	46
17828	300	\N	Saux	46
17829	301	\N	Sauzet	46
17830	302	\N	Sénaillac-Latronquière	46
17831	303	\N	Sénaillac-Lauzès	46
17832	304	\N	Séniergues	46
17833	305	\N	Sérignac	46
17834	306	\N	Sonac	46
17835	307	\N	Soturac	46
17836	308	\N	Soucirac	46
17837	309	\N	Souillac	46
17838	310	\N	Soulomès	46
17839	311	\N	Sousceyrac	46
17840	312	\N	Strenquels	46
17841	313	\N	Tauriac	46
17842	314	\N	Terrou	46
17843	315	\N	Teyssieu	46
17844	316	\N	Thédirac	46
17845	317	\N	Thégra	46
17846	318	\N	Thémines	46
17847	319	\N	Théminettes	46
17848	320	\N	Tour-de-Faure	46
17849	321	\N	Touzac	46
17850	322	\N	Trespoux-Rassiels	46
17851	323	\N	Ussel	46
17852	324	\N	Uzech	46
17853	325	\N	Vaillac	46
17854	326	\N	Valprionde	46
17855	327	\N	Valroufié	46
17856	328	\N	Varaire	46
17857	329	\N	Vaylats	46
17858	330	\N	Vayrac	46
17859	331	\N	Vers	46
17860	332	\N	Viazac	46
17861	333	\N	Vidaillac	46
17862	334	\N	Vigan	46
17863	335	\N	Villesèque	46
17864	336	\N	Vire-sur-Lot	46
17865	337	\N	Mayrac	46
17866	338	\N	Bessonies	46
17867	339	\N	Saint-Jean-Lagineste	46
17868	340	\N	Saint-Pierre-Lafeuille	46
17869	001	\N	Agen	47
17870	002	\N	Agmé	47
17871	003	\N	Agnac	47
17872	004	\N	Aiguillon	47
17873	005	\N	Allemans-du-Dropt	47
17874	006	\N	Allez-et-Cazeneuve	47
17875	007	\N	Allons	47
17876	008	\N	Ambrus	47
17877	009	\N	Andiran	47
17878	010	\N	Antagnac	47
17879	011	\N	Anthé	47
17880	012	\N	Anzex	47
17881	013	\N	Argenton	47
17882	014	\N	Armillac	47
17883	015	\N	Astaffort	47
17884	016	\N	Aubiac	47
17885	017	\N	Auradou	47
17886	018	\N	Auriac-sur-Dropt	47
17887	019	\N	Bajamont	47
17888	020	\N	Baleyssagues	47
17889	021	\N	Barbaste	47
17890	022	\N	Bazens	47
17891	023	\N	Beaugas	47
17892	024	\N	Beaupuy	47
17893	025	\N	Beauville	47
17894	026	\N	Beauziac	47
17895	027	\N	Bias	47
17896	028	\N	Birac-sur-Trec	47
17897	029	\N	Blanquefort-sur-Briolance	47
17898	030	\N	Blaymont	47
17899	031	\N	Boé	47
17900	032	\N	Bon-Encontre	47
17901	033	\N	Boudy-de-Beauregard	47
17902	034	\N	Bouglon	47
17903	035	\N	Bourgougnague	47
17904	036	\N	Bourlens	47
17905	037	\N	Bournel	47
17906	038	\N	Bourran	47
17907	039	\N	Boussès	47
17908	040	\N	Brax	47
17909	041	\N	Bruch	47
17910	042	\N	Brugnac	47
17911	043	\N	Buzet-sur-Baïse	47
17912	044	\N	Cahuzac	47
17913	045	\N	Calignac	47
17914	046	\N	Calonges	47
17915	047	\N	Cambes	47
17916	048	\N	Cancon	47
17917	049	\N	Casseneuil	47
17918	050	\N	Cassignas	47
17919	051	\N	Castelculier	47
17920	052	\N	Casteljaloux	47
17921	053	\N	Castella	47
17922	054	\N	Castelmoron-sur-Lot	47
17923	055	\N	Castelnaud-de-Gratecambe	47
17924	056	\N	Castelnau-sur-Gupie	47
17925	057	\N	Castillonnès	47
17926	058	\N	Caubeyres	47
17927	059	\N	Caubon-Saint-Sauveur	47
17928	060	\N	Caudecoste	47
17929	061	\N	Caumont-sur-Garonne	47
17930	062	\N	Cauzac	47
17931	063	\N	Cavarc	47
17932	064	\N	Cazideroque	47
17933	065	\N	Clairac	47
17934	066	\N	Clermont-Dessous	47
17935	067	\N	Clermont-Soubiran	47
17936	068	\N	Cocumont	47
17937	069	\N	Colayrac-Saint-Cirq	47
17938	070	\N	Condezaygues	47
17939	071	\N	Coulx	47
17940	072	\N	Courbiac	47
17941	073	\N	Cours	47
17942	074	\N	Couthures-sur-Garonne	47
17943	075	\N	Croix-Blanche	47
17944	076	\N	Cuq	47
17945	077	\N	Cuzorn	47
17946	078	\N	Damazan	47
17947	079	\N	Dausse	47
17948	080	\N	Dévillac	47
17949	081	\N	Dolmayrac	47
17950	082	\N	Dondas	47
17951	083	\N	Doudrac	47
17952	084	\N	Douzains	47
17953	085	\N	Durance	47
17954	086	\N	Duras	47
17955	087	\N	Engayrac	47
17956	088	\N	Escassefort	47
17957	089	\N	Esclottes	47
17958	090	\N	Espiens	47
17959	091	\N	Estillac	47
17960	092	\N	Fals	47
17961	093	\N	Fargues-sur-Ourbise	47
17962	094	\N	Fauguerolles	47
17963	095	\N	Fauillet	47
17964	096	\N	Ferrensac	47
17965	097	\N	Feugarolles	47
17966	098	\N	Fieux	47
17967	099	\N	Fongrave	47
17968	100	\N	Foulayronnes	47
17969	101	\N	Fourques-sur-Garonne	47
17970	102	\N	Francescas	47
17971	103	\N	Fréchou	47
17972	104	\N	Frégimont	47
17973	105	\N	Frespech	47
17974	106	\N	Fumel	47
17975	107	\N	Galapian	47
17976	108	\N	Gaujac	47
17977	109	\N	Gavaudun	47
17978	110	\N	Gontaud-de-Nogaret	47
17979	111	\N	Granges-sur-Lot	47
17980	112	\N	Grateloup-Saint-Gayrand	47
17981	113	\N	Grayssas	47
17982	114	\N	Grézet-Cavagnan	47
17983	115	\N	Guérin	47
17984	117	\N	Hautefage-la-Tour	47
17985	118	\N	Hautesvignes	47
17986	119	\N	Houeillès	47
17987	120	\N	Jusix	47
17988	121	\N	Labastide-Castel-Amouroux	47
17989	122	\N	Labretonie	47
17990	123	\N	Lacapelle-Biron	47
17991	124	\N	Lacaussade	47
17992	125	\N	Lacépède	47
17993	126	\N	Lachapelle	47
17994	127	\N	Lafitte-sur-Lot	47
17995	128	\N	Lafox	47
17996	129	\N	Lagarrigue	47
17997	130	\N	Lagruère	47
17998	131	\N	Lagupie	47
17999	132	\N	Lalandusse	47
18000	133	\N	Lamontjoie	47
18001	134	\N	Lannes	47
18002	135	\N	Laparade	47
18003	136	\N	Laperche	47
18004	137	\N	Laplume	47
18005	138	\N	Laroque-Timbaut	47
18006	139	\N	Lasserre	47
18007	140	\N	Laugnac	47
18008	141	\N	Laussou	47
18009	142	\N	Lauzun	47
18010	143	\N	Lavardac	47
18011	144	\N	Lavergne	47
18012	145	\N	Layrac	47
18013	146	\N	Lédat	47
18014	147	\N	Lévignac-de-Guyenne	47
18015	148	\N	Leyritz-Moncassin	47
18016	150	\N	Longueville	47
18017	151	\N	Loubès-Bernac	47
18018	152	\N	Lougratte	47
18019	154	\N	Lusignan-Petit	47
18020	155	\N	Madaillan	47
18021	156	\N	Marcellus	47
18022	157	\N	Marmande	47
18023	158	\N	Marmont-Pachas	47
18024	159	\N	Mas-d'Agenais	47
18025	160	\N	Masquières	47
18026	161	\N	Massels	47
18027	162	\N	Massoulès	47
18028	163	\N	Mauvezin-sur-Gupie	47
18029	164	\N	Mazières-Naresse	47
18030	165	\N	Meilhan-sur-Garonne	47
18031	167	\N	Mézin	47
18032	168	\N	Miramont-de-Guyenne	47
18033	169	\N	Moirax	47
18034	170	\N	Monbahus	47
18035	171	\N	Monbalen	47
18036	172	\N	Moncaut	47
18037	173	\N	Monclar	47
18038	174	\N	Moncrabeau	47
18039	175	\N	Monflanquin	47
18040	176	\N	Mongaillard	47
18041	177	\N	Monheurt	47
18042	178	\N	Monségur	47
18043	179	\N	Monsempron-Libos	47
18044	180	\N	Montagnac-sur-Auvignon	47
18045	181	\N	Montagnac-sur-Lède	47
18046	182	\N	Montastruc	47
18047	183	\N	Montauriol	47
18048	184	\N	Montaut	47
18049	185	\N	Montayral	47
18050	186	\N	Montesquieu	47
18051	187	\N	Monteton	47
18052	188	\N	Montignac-de-Lauzun	47
18053	189	\N	Montignac-Toupinerie	47
18054	190	\N	Montpezat	47
18055	191	\N	Montpouillan	47
18056	192	\N	Monviel	47
18057	193	\N	Moulinet	47
18058	194	\N	Moustier	47
18059	195	\N	Nérac	47
18060	196	\N	Nicole	47
18061	197	\N	Nomdieu	47
18062	198	\N	Pailloles	47
18063	199	\N	Pardaillan	47
18064	200	\N	Parranquet	47
18065	201	\N	Passage	47
18066	202	\N	Paulhiac	47
18067	203	\N	Penne-d'Agenais	47
18068	204	\N	Peyrière	47
18069	205	\N	Pindères	47
18070	206	\N	Pinel-Hauterive	47
18071	207	\N	Pompiey	47
18072	208	\N	Pompogne	47
18073	209	\N	Pont-du-Casse	47
18074	210	\N	Port-Sainte-Marie	47
18075	211	\N	Poudenas	47
18076	212	\N	Poussignac	47
18077	213	\N	Prayssas	47
18078	214	\N	Puch-d'Agenais	47
18079	215	\N	Pujols	47
18080	216	\N	Puymiclan	47
18081	217	\N	Puymirol	47
18082	218	\N	Puysserampion	47
18083	219	\N	Rayet	47
18084	220	\N	Razimet	47
18085	221	\N	Réaup-Lisse	47
18086	222	\N	Réunion	47
18087	223	\N	Rives	47
18088	224	\N	Romestaing	47
18089	225	\N	Roquefort	47
18090	226	\N	Roumagne	47
18091	227	\N	Ruffiac	47
18092	228	\N	Saint-Antoine-de-Ficalba	47
18093	229	\N	Saint-Astier	47
18094	230	\N	Saint-Aubin	47
18095	231	\N	Saint-Avit	47
18096	232	\N	Saint-Barthélemy-d'Agenais	47
18097	233	\N	Sainte-Bazeille	47
18098	234	\N	Saint-Caprais-de-Lerm	47
18099	235	\N	Saint-Colomb-de-Lauzun	47
18100	236	\N	Sainte-Colombe-de-Duras	47
18101	237	\N	Sainte-Colombe-de-Villeneuve	47
18102	238	\N	Sainte-Colombe-en-Bruilhois	47
18103	239	\N	Saint-Étienne-de-Fougères	47
18104	240	\N	Saint-Étienne-de-Villeréal	47
18105	241	\N	Saint-Eutrope-de-Born	47
18106	242	\N	Saint-Front-sur-Lémance	47
18107	244	\N	Sainte-Gemme-Martaillac	47
18108	245	\N	Saint-Géraud	47
18109	246	\N	Saint-Hilaire-de-Lusignan	47
18110	247	\N	Saint-Jean-de-Duras	47
18111	248	\N	Saint-Jean-de-Thurac	47
18112	249	\N	Saint-Laurent	47
18113	250	\N	Saint-Léger	47
18114	251	\N	Saint-Léon	47
18115	252	\N	Sainte-Livrade-sur-Lot	47
18116	253	\N	Sainte-Marthe	47
18117	254	\N	Saint-Martin-Curton	47
18118	255	\N	Saint-Martin-de-Beauville	47
18119	256	\N	Saint-Martin-de-Villeréal	47
18120	257	\N	Saint-Martin-Petit	47
18121	258	\N	Sainte-Maure-de-Peyriac	47
18122	259	\N	Saint-Maurice-de-Lestapel	47
18123	260	\N	Saint-Maurin	47
18124	262	\N	Saint-Nicolas-de-la-Balerme	47
18125	263	\N	Saint-Pardoux-du-Breuil	47
18126	264	\N	Saint-Pardoux-Isaac	47
18127	265	\N	Saint-Pastour	47
18128	266	\N	Saint-Pé-Saint-Simon	47
18129	267	\N	Saint-Pierre-de-Buzet	47
18130	269	\N	Saint-Pierre-de-Clairac	47
18131	271	\N	Saint-Pierre-sur-Dropt	47
18132	272	\N	Saint-Quentin-du-Dropt	47
18133	273	\N	Saint-Robert	47
18134	274	\N	Saint-Romain-le-Noble	47
18135	275	\N	Saint-Salvy	47
18136	276	\N	Saint-Sardos	47
18137	277	\N	Saint-Sauveur-de-Meilhan	47
18138	278	\N	Saint-Sernin	47
18139	279	\N	Saint-Sixte	47
18140	280	\N	Saint-Sylvestre-sur-Lot	47
18141	281	\N	Saint-Urcisse	47
18142	282	\N	Saint-Vincent-de-Lamontjoie	47
18143	283	\N	Saint-Vite	47
18144	284	\N	Salles	47
18145	285	\N	Samazan	47
18146	286	\N	Sauméjan	47
18147	287	\N	Saumont	47
18148	288	\N	Sauvagnas	47
18149	289	\N	Sauvetat-de-Savères	47
18150	290	\N	Sauvetat-du-Dropt	47
18151	291	\N	Sauvetat-sur-Lède	47
18152	292	\N	Sauveterre-la-Lémance	47
18153	293	\N	Sauveterre-Saint-Denis	47
18154	294	\N	Savignac-de-Duras	47
18155	295	\N	Savignac-sur-Leyze	47
18156	296	\N	Ségalas	47
18157	297	\N	Sembas	47
18158	298	\N	Sénestis	47
18159	299	\N	Sérignac-Péboudou	47
18160	300	\N	Sérignac-sur-Garonne	47
18161	301	\N	Seyches	47
18162	302	\N	Sos	47
18163	303	\N	Soumensac	47
18164	304	\N	Taillebourg	47
18165	305	\N	Tayrac	47
18166	306	\N	Temple-sur-Lot	47
18167	307	\N	Thézac	47
18168	308	\N	Thouars-sur-Garonne	47
18169	309	\N	Tombebuf	47
18170	310	\N	Tonneins	47
18171	311	\N	Tourliac	47
18172	312	\N	Tournon-d'Agenais	47
18173	313	\N	Tourtrès	47
18174	314	\N	Trémons	47
18175	315	\N	Trentels	47
18176	316	\N	Varès	47
18177	317	\N	Verteuil-d'Agenais	47
18178	318	\N	Vianne	47
18179	319	\N	Villebramar	47
18180	320	\N	Villefranche-du-Queyran	47
18181	321	\N	Villeneuve-de-Duras	47
18182	323	\N	Villeneuve-sur-Lot	47
18183	324	\N	Villeréal	47
18184	325	\N	Villeton	47
18185	326	\N	Virazeil	47
18186	327	\N	Xaintrailles	47
18187	328	\N	Saint-Georges	47
18188	001	\N	Albaret-le-Comtal	48
18189	002	\N	Albaret-Sainte-Marie	48
18190	003	\N	Allenc	48
18191	004	\N	Altier	48
18192	005	\N	Antrenas	48
18193	007	\N	Arzenc-d'Apcher	48
18194	008	\N	Arzenc-de-Randon	48
18195	009	\N	Aumont-Aubrac	48
18196	010	\N	Auroux	48
18197	012	\N	Monts-Verts	48
18198	013	\N	Badaroux	48
18199	014	\N	Bagnols-les-Bains	48
18200	015	\N	Pied-de-Borne	48
18201	016	\N	Balsièges	48
18202	017	\N	Banassac	48
18203	018	\N	Barjac	48
18204	019	\N	Barre-des-Cévennes	48
18205	020	\N	Bassurels	48
18206	021	\N	Bastide-Puylaurent	48
18207	022	\N	Bédouès	48
18208	023	\N	Belvezet	48
18209	025	\N	Bessons	48
18210	026	\N	Blavignac	48
18211	027	\N	Bleymard	48
18212	028	\N	Bondons	48
18213	029	\N	Born	48
18214	030	\N	Brenoux	48
18215	031	\N	Brion	48
18216	032	\N	Buisson	48
18217	033	\N	Canilhac	48
18218	034	\N	Canourgue	48
18219	036	\N	Cassagnas	48
18220	037	\N	Chadenet	48
18221	038	\N	Chambon-le-Château	48
18222	039	\N	Chanac	48
18223	040	\N	Chasseradès	48
18224	041	\N	Chastanier	48
18225	042	\N	Chastel-Nouvel	48
18226	043	\N	Châteauneuf-de-Randon	48
18227	044	\N	Chauchailles	48
18228	045	\N	Chaudeyrac	48
18229	046	\N	Chaulhac	48
18230	047	\N	Chaze-de-Peyre	48
18231	048	\N	Cheylard-l'Évêque	48
18232	049	\N	Chirac	48
18233	050	\N	Cocurès	48
18234	051	\N	Collet-de-Dèze	48
18235	053	\N	Cubières	48
18236	054	\N	Cubiérettes	48
18237	055	\N	Cultures	48
18238	056	\N	Esclanèdes	48
18239	057	\N	Estables	48
18240	058	\N	Fage-Montivernoux	48
18241	059	\N	Fage-Saint-Julien	48
18242	060	\N	Fau-de-Peyre	48
18243	061	\N	Florac	48
18244	062	\N	Fontanes	48
18245	063	\N	Fontans	48
18246	064	\N	Fournels	48
18247	065	\N	Fraissinet-de-Fourques	48
18248	066	\N	Fraissinet-de-Lozère	48
18249	067	\N	Gabriac	48
18250	068	\N	Gabrias	48
18251	069	\N	Gatuzières	48
18252	070	\N	Grandrieu	48
18253	071	\N	Grandvals	48
18254	072	\N	Grèzes	48
18255	073	\N	Hermaux	48
18256	074	\N	Hures-la-Parade	48
18257	075	\N	Ispagnac	48
18258	076	\N	Javols	48
18259	077	\N	Julianges	48
18260	078	\N	Lachamp	48
18261	079	\N	Lajo	48
18262	080	\N	Langogne	48
18263	081	\N	Lanuéjols	48
18264	082	\N	Laubert	48
18265	083	\N	Laubies	48
18266	084	\N	Laval-Atger	48
18267	085	\N	Laval-du-Tarn	48
18268	086	\N	Luc	48
18269	087	\N	Malbouzon	48
18270	088	\N	Malène	48
18271	089	\N	Malzieu-Forain	48
18272	090	\N	Malzieu-Ville	48
18273	091	\N	Marchastel	48
18274	092	\N	Marvejols	48
18275	093	\N	Mas-d'Orcières	48
18276	094	\N	Massegros	48
18277	095	\N	Mende	48
18278	096	\N	Meyrueis	48
18279	097	\N	Moissac-Vallée-Française	48
18280	098	\N	Molezon	48
18281	099	\N	Monastier-Pin-Moriès	48
18282	100	\N	Montbel	48
18283	101	\N	Montbrun	48
18284	103	\N	Montrodat	48
18285	104	\N	Nasbinals	48
18286	105	\N	Naussac	48
18287	106	\N	Noalhac	48
18288	107	\N	Palhers	48
18289	108	\N	Panouse	48
18290	110	\N	Paulhac-en-Margeride	48
18291	111	\N	Pelouse	48
18292	112	\N	Pierrefiche	48
18293	115	\N	Pompidou	48
18294	116	\N	Pont-de-Montvert	48
18295	117	\N	Pourcharesses	48
18296	119	\N	Prévenchères	48
18297	120	\N	Prinsuéjols	48
18298	121	\N	Prunières	48
18299	122	\N	Quézac	48
18300	123	\N	Recoules-d'Aubrac	48
18301	124	\N	Recoules-de-Fumas	48
18302	125	\N	Recoux	48
18303	126	\N	Ribennes	48
18304	127	\N	Rieutort-de-Randon	48
18305	128	\N	Rimeize	48
18306	129	\N	Rocles	48
18307	130	\N	Rousses	48
18308	131	\N	Rozier	48
18309	132	\N	Saint-Alban-sur-Limagnole	48
18310	133	\N	Saint-Amans	48
18311	134	\N	Saint-Andéol-de-Clerguemort	48
18312	135	\N	Saint-André-Capcèze	48
18313	136	\N	Saint-André-de-Lancize	48
18314	137	\N	Saint-Bauzile	48
18315	138	\N	Saint-Bonnet-de-Chirac	48
18316	139	\N	Saint-Bonnet-de-Montauroux	48
18317	140	\N	Saint-Chély-d'Apcher	48
18318	141	\N	Mas-Saint-Chély	48
18319	142	\N	Sainte-Colombe-de-Peyre	48
18320	144	\N	Sainte-Croix-Vallée-Française	48
18321	145	\N	Saint-Denis-en-Margeride	48
18322	146	\N	Sainte-Enimie	48
18323	147	\N	Saint-Étienne-du-Valdonnez	48
18324	148	\N	Saint-Étienne-Vallée-Française	48
18325	149	\N	Sainte-Eulalie	48
18326	150	\N	Saint-Flour-de-Mercoire	48
18327	151	\N	Saint-Frézal-d'Albuges	48
18328	152	\N	Saint-Frézal-de-Ventalon	48
18329	153	\N	Saint-Gal	48
18330	154	\N	Saint-Georges-de-Lévéjac	48
18331	155	\N	Saint-Germain-de-Calberte	48
18332	156	\N	Saint-Germain-du-Teil	48
18333	157	\N	Sainte-Hélène	48
18334	158	\N	Saint-Hilaire-de-Lavit	48
18335	160	\N	Saint-Jean-la-Fouillouse	48
18336	161	\N	Saint-Juéry	48
18337	162	\N	Saint-Julien-d'Arpaon	48
18338	163	\N	Saint-Julien-des-Points	48
18339	164	\N	Saint-Julien-du-Tournel	48
18340	165	\N	Saint-Laurent-de-Muret	48
18341	166	\N	Saint-Laurent-de-Trèves	48
18342	167	\N	Saint-Laurent-de-Veyrès	48
18343	168	\N	Saint-Léger-de-Peyre	48
18344	169	\N	Saint-Léger-du-Malzieu	48
18345	170	\N	Saint-Martin-de-Boubaux	48
18346	171	\N	Saint-Martin-de-Lansuscle	48
18347	172	\N	Saint-Maurice-de-Ventalon	48
18348	173	\N	Saint-Michel-de-Dèze	48
18349	174	\N	Saint-Paul-le-Froid	48
18350	175	\N	Saint-Pierre-de-Nogaret	48
18351	176	\N	Saint-Pierre-des-Tripiers	48
18352	177	\N	Saint-Pierre-le-Vieux	48
18353	178	\N	Saint-Privat-de-Vallongue	48
18354	179	\N	Saint-Privat-du-Fau	48
18355	180	\N	Saint-Rome-de-Dolan	48
18356	181	\N	Saint-Saturnin	48
18357	182	\N	Saint-Sauveur-de-Ginestoux	48
18358	183	\N	Saint-Sauveur-de-Peyre	48
18359	184	\N	Saint-Symphorien	48
18360	185	\N	Salelles	48
18361	186	\N	Salle-Prunet	48
18362	187	\N	Salces	48
18363	188	\N	Serverette	48
18364	189	\N	Servières	48
18365	190	\N	Termes	48
18366	191	\N	Tieule	48
18367	192	\N	Trélans	48
18368	193	\N	Vebron	48
18369	194	\N	Vialas	48
18370	195	\N	Vignes	48
18371	197	\N	Villedieu	48
18372	198	\N	Villefort	48
18373	001	\N	Alleuds	49
18374	002	\N	Allonnes	49
18375	003	\N	Ambillou-Château	49
18376	004	\N	Andard	49
18377	005	\N	Andigné	49
18378	006	\N	Andrezé	49
18379	007	\N	Angers	49
18380	008	\N	Angrie	49
18381	009	\N	Antoigné	49
18382	010	\N	Armaillé	49
18383	011	\N	Artannes-sur-Thouet	49
18384	012	\N	Aubigné-sur-Layon	49
18385	013	\N	Auverse	49
18386	014	\N	Aviré	49
18387	015	\N	Avrillé	49
18388	017	\N	Baracé	49
18389	018	\N	Baugé	49
18390	019	\N	Bauné	49
18391	020	\N	Beaucouzé	49
18392	021	\N	Beaufort-en-Vallée	49
18393	022	\N	Beaulieu-sur-Layon	49
18394	023	\N	Beaupréau	49
18395	024	\N	Beausse	49
18396	025	\N	Beauvau	49
18397	026	\N	Bécon-les-Granits	49
18398	027	\N	Bégrolles-en-Mauges	49
18399	028	\N	Béhuard	49
18400	029	\N	Blaison-Gohier	49
18401	030	\N	Blou	49
18402	031	\N	Bocé	49
18403	032	\N	Bohalle	49
18404	033	\N	Boissière-sur-Èvre	49
18405	034	\N	Botz-en-Mauges	49
18406	035	\N	Bouchemaine	49
18407	036	\N	Bouillé-Ménard	49
18408	037	\N	Bourg-d'Iré	49
18409	038	\N	Bourg-l'Évêque	49
18410	039	\N	Bourgneuf-en-Mauges	49
18411	040	\N	Bouzillé	49
18412	041	\N	Brain-sur-Allonnes	49
18413	042	\N	Brain-sur-l'Authion	49
18414	043	\N	Brain-sur-Longuenée	49
18415	044	\N	Breil	49
18416	045	\N	Breille-les-Pins	49
18417	046	\N	Brézé	49
18418	047	\N	Brigné	49
18419	048	\N	Briollay	49
18420	049	\N	Brion	49
18421	050	\N	Brissac-Quincé	49
18422	051	\N	Brissarthe	49
18423	052	\N	Broc	49
18424	053	\N	Brossay	49
18425	054	\N	Candé	49
18426	055	\N	Cantenay-Épinard	49
18427	056	\N	Carbay	49
18428	057	\N	Cernusson	49
18429	058	\N	Cerqueux	49
18430	059	\N	Cerqueux-sous-Passavant	49
18431	060	\N	Chacé	49
18432	061	\N	Challain-la-Potherie	49
18433	062	\N	Chalonnes-sous-le-Lude	49
18434	063	\N	Chalonnes-sur-Loire	49
18435	064	\N	Chambellay	49
18436	065	\N	Champigné	49
18437	066	\N	Champ-sur-Layon	49
18438	067	\N	Champteussé-sur-Baconne	49
18439	068	\N	Champtocé-sur-Loire	49
18440	069	\N	Champtoceaux	49
18441	070	\N	Chanteloup-les-Bois	49
18442	071	\N	Chanzeaux	49
18443	072	\N	Chapelle-du-Genêt	49
18444	073	\N	Chapelle-Hullin	49
18445	074	\N	Chapelle-Rousselin	49
18446	075	\N	Chapelle-Saint-Florent	49
18447	076	\N	Chapelle-Saint-Laud	49
18448	077	\N	Chapelle-sur-Oudon	49
18449	078	\N	Charcé-Saint-Ellier-sur-Aubance	49
18450	079	\N	Chartrené	49
18451	080	\N	Châteauneuf-sur-Sarthe	49
18452	081	\N	Châtelais	49
18453	082	\N	Chaudefonds-sur-Layon	49
18454	083	\N	Chaudron-en-Mauges	49
18455	084	\N	Chaumont-d'Anjou	49
18456	085	\N	Chaussaire	49
18457	086	\N	Chavagnes	49
18458	087	\N	Chavaignes	49
18459	088	\N	Chazé-Henry	49
18460	089	\N	Chazé-sur-Argos	49
18461	090	\N	Cheffes	49
18462	091	\N	Chemellier	49
18463	092	\N	Chemillé	49
18464	093	\N	Chemiré-sur-Sarthe	49
18465	094	\N	Chênehutte-Trèves-Cunault	49
18466	095	\N	Chenillé-Changé	49
18467	096	\N	Cherré	49
18468	097	\N	Cheviré-le-Rouge	49
18469	098	\N	Chigné	49
18470	099	\N	Cholet	49
18471	100	\N	Cizay-la-Madeleine	49
18472	101	\N	Clefs	49
18473	102	\N	Cléré-sur-Layon	49
18474	103	\N	Combrée	49
18475	104	\N	Concourson-sur-Layon	49
18476	105	\N	Contigné	49
18477	106	\N	Corné	49
18478	107	\N	Cornillé-les-Caves	49
18479	108	\N	Cornuaille	49
18480	109	\N	Coron	49
18481	110	\N	Corzé	49
18482	111	\N	Cossé-d'Anjou	49
18483	112	\N	Coudray-Macouard	49
18484	113	\N	Courchamps	49
18485	114	\N	Courléon	49
18486	115	\N	Coutures	49
18487	116	\N	Cuon	49
18488	117	\N	Daguenière	49
18489	119	\N	Daumeray	49
18490	120	\N	Denée	49
18491	121	\N	Dénezé-sous-Doué	49
18492	122	\N	Dénezé-sous-le-Lude	49
18493	123	\N	Distré	49
18494	125	\N	Doué-la-Fontaine	49
18495	126	\N	Drain	49
18496	127	\N	Durtal	49
18497	128	\N	Échemiré	49
18498	129	\N	Écouflant	49
18499	130	\N	Écuillé	49
18500	131	\N	Épieds	49
18501	132	\N	Étriché	49
18502	133	\N	Faveraye-Mâchelles	49
18503	134	\N	Faye-d'Anjou	49
18504	135	\N	Feneu	49
18505	136	\N	Ferrière-de-Flée	49
18506	137	\N	Fief-Sauvin	49
18507	138	\N	Fontaine-Guérin	49
18508	139	\N	Fontaine-Milon	49
18509	140	\N	Fontevraud-l'Abbaye	49
18510	141	\N	Forges	49
18511	142	\N	Fosse-de-Tigné	49
18512	143	\N	Fougeré	49
18513	144	\N	Freigné	49
18514	145	\N	Fuilet	49
18515	147	\N	Gée	49
18516	148	\N	Gené	49
18517	149	\N	Gennes	49
18518	150	\N	Genneteil	49
18519	151	\N	Gesté	49
18520	153	\N	Valanjou	49
18521	154	\N	Grézillé	49
18522	155	\N	Grez-Neuville	49
18523	156	\N	Grugé-l'Hôpital	49
18524	157	\N	Guédéniau	49
18525	158	\N	Hôtellerie-de-Flée	49
18526	159	\N	Huillé	49
18527	160	\N	Ingrandes	49
18528	161	\N	Jaille-Yvon	49
18529	162	\N	Jallais	49
18530	163	\N	Jarzé	49
18531	165	\N	Jubaudière	49
18532	167	\N	Juigné-sur-Loire	49
18533	169	\N	Jumellière	49
18534	170	\N	Juvardeil	49
18535	171	\N	Lande-Chasles	49
18536	172	\N	Landemont	49
18537	173	\N	Lasse	49
18538	174	\N	Lézigné	49
18539	175	\N	Linières-Bouton	49
18540	176	\N	Lion-d'Angers	49
18541	177	\N	Liré	49
18542	178	\N	Loiré	49
18543	179	\N	Longeron	49
18544	180	\N	Longué-Jumelles	49
18545	181	\N	Louerre	49
18546	182	\N	Louresse-Rochemenier	49
18547	183	\N	Louroux-Béconnais	49
18548	184	\N	Louvaines	49
18549	185	\N	Lué-en-Baugeois	49
18550	186	\N	Luigné	49
18551	187	\N	Marans	49
18552	188	\N	Marcé	49
18553	189	\N	Marigné	49
18554	190	\N	Marillais	49
18555	191	\N	Martigné-Briand	49
18556	192	\N	Maulévrier	49
18557	193	\N	May-sur-Èvre	49
18558	194	\N	Mazé	49
18559	195	\N	Mazières-en-Mauges	49
18560	196	\N	Meignanne	49
18561	197	\N	Meigné-le-Vicomte	49
18562	198	\N	Meigné	49
18563	199	\N	Melay	49
18564	200	\N	Membrolle-sur-Longuenée	49
18565	201	\N	Ménitré	49
18566	202	\N	Méon	49
18567	204	\N	Mesnil-en-Vallée	49
18568	205	\N	Miré	49
18569	206	\N	Montfaucon-Montigné	49
18570	207	\N	Montfort	49
18571	208	\N	Montguillon	49
18572	209	\N	Montigné-lès-Rairies	49
18573	211	\N	Montilliers	49
18574	212	\N	Montjean-sur-Loire	49
18575	213	\N	Montpollin	49
18576	214	\N	Montreuil-Juigné	49
18577	215	\N	Montreuil-Bellay	49
18578	216	\N	Montreuil-sur-Loir	49
18579	217	\N	Montreuil-sur-Maine	49
18580	218	\N	Montrevault	49
18581	219	\N	Montsoreau	49
18582	220	\N	Morannes	49
18583	221	\N	Mouliherne	49
18584	222	\N	Mozé-sur-Louet	49
18585	223	\N	Mûrs-Erigné	49
18586	224	\N	Neuillé	49
18587	225	\N	Neuvy-en-Mauges	49
18588	226	\N	Noëllet	49
18589	227	\N	Notre-Dame-d'Allençon	49
18590	228	\N	Noyant	49
18591	229	\N	Noyant-la-Gravoyère	49
18592	230	\N	Noyant-la-Plaine	49
18593	231	\N	Nuaillé	49
18594	232	\N	Nueil-sur-Layon	49
18595	233	\N	Nyoiseau	49
18596	234	\N	Parçay-les-Pins	49
18597	235	\N	Parnay	49
18598	236	\N	Passavant-sur-Layon	49
18599	237	\N	Pellerine	49
18600	238	\N	Pellouailles-les-Vignes	49
18601	239	\N	Pin-en-Mauges	49
18602	240	\N	Plaine	49
18603	241	\N	Plessis-Grammoire	49
18604	242	\N	Plessis-Macé	49
18605	243	\N	Poitevinière	49
18606	244	\N	Pommeraye	49
18607	245	\N	Pontigné	49
18608	246	\N	Ponts-de-Cé	49
18609	247	\N	Possonnière	49
18610	248	\N	Pouancé	49
18611	249	\N	Pouëze	49
18612	250	\N	Prévière	49
18613	251	\N	Pruillé	49
18614	252	\N	Puiset-Doré	49
18615	253	\N	Puy-Notre-Dame	49
18616	254	\N	Querré	49
18617	256	\N	Rablay-sur-Layon	49
18618	257	\N	Rairies	49
18619	258	\N	Renaudière	49
18620	259	\N	Rochefort-sur-Loire	49
18621	260	\N	Romagne	49
18622	261	\N	Rosiers-sur-Loire	49
18623	262	\N	Rou-Marson	49
18624	263	\N	Roussay	49
18625	264	\N	Saint-André-de-la-Marche	49
18626	265	\N	Saint-Aubin-de-Luigné	49
18627	266	\N	Saint-Augustin-des-Bois	49
18628	267	\N	Saint-Barthélemy-d'Anjou	49
18629	268	\N	Sainte-Christine	49
18630	269	\N	Saint-Christophe-du-Bois	49
18631	270	\N	Saint-Christophe-la-Couperie	49
18632	271	\N	Saint-Clément-de-la-Place	49
18633	272	\N	Saint-Clément-des-Levées	49
18634	273	\N	Saint-Crespin-sur-Moine	49
18635	274	\N	Saint-Cyr-en-Bourg	49
18636	276	\N	Saint-Florent-le-Vieil	49
18637	277	\N	Sainte-Gemmes-d'Andigné	49
18638	278	\N	Sainte-Gemmes-sur-Loire	49
18639	279	\N	Saint-Georges-des-Sept-Voies	49
18640	280	\N	Saint-Georges-du-Bois	49
18641	281	\N	Saint-Georges-des-Gardes	49
18642	282	\N	Saint-Georges-sur-Layon	49
18643	283	\N	Saint-Georges-sur-Loire	49
18644	284	\N	Saint-Germain-des-Prés	49
18645	285	\N	Saint-Germain-sur-Moine	49
18646	288	\N	Saint-Jean-de-la-Croix	49
18647	289	\N	Saint-Jean-de-Linières	49
18648	290	\N	Saint-Jean-des-Mauvrets	49
18649	291	\N	Saint-Just-sur-Dive	49
18650	292	\N	Saint-Lambert-du-Lattay	49
18651	294	\N	Saint-Lambert-la-Potherie	49
18652	295	\N	Saint-Laurent-de-la-Plaine	49
18653	296	\N	Saint-Laurent-des-Autels	49
18654	297	\N	Saint-Laurent-du-Mottay	49
18655	298	\N	Saint-Léger-des-Bois	49
18656	299	\N	Saint-Léger-sous-Cholet	49
18657	300	\N	Saint-Lézin	49
18658	301	\N	Saint-Macaire-en-Mauges	49
18659	302	\N	Saint-Macaire-du-Bois	49
18660	303	\N	Saint-Martin-d'Arcé	49
18661	304	\N	Saint-Martin-de-la-Place	49
18662	305	\N	Saint-Martin-du-Bois	49
18663	306	\N	Saint-Martin-du-Fouilloux	49
18664	307	\N	Saint-Mathurin-sur-Loire	49
18665	308	\N	Saint-Melaine-sur-Aubance	49
18666	309	\N	Saint-Michel-et-Chanveaux	49
18667	310	\N	Saint-Paul-du-Bois	49
18668	311	\N	Saint-Philbert-du-Peuple	49
18669	312	\N	Saint-Philbert-en-Mauges	49
18670	313	\N	Saint-Pierre-Montlimart	49
18671	314	\N	Saint-Quentin-en-Mauges	49
18672	315	\N	Saint-Quentin-lès-Beaurepaire	49
18673	316	\N	Saint-Rémy-en-Mauges	49
18674	317	\N	Saint-Rémy-la-Varenne	49
18675	318	\N	Saint-Saturnin-sur-Loire	49
18676	319	\N	Saint-Sauveur-de-Flée	49
18677	320	\N	Saint-Sauveur-de-Landemont	49
18678	321	\N	Saint-Sigismond	49
18679	322	\N	Saint-Sulpice	49
18680	323	\N	Saint-Sylvain-d'Anjou	49
18681	324	\N	Salle-et-Chapelle-Aubry	49
18682	325	\N	Salle-de-Vihiers	49
18683	326	\N	Sarrigné	49
18684	327	\N	Saulgé-l'Hôpital	49
18685	328	\N	Saumur	49
18686	329	\N	Savennières	49
18687	330	\N	Sceaux-d'Anjou	49
18688	331	\N	Segré	49
18689	332	\N	Séguinière	49
18690	333	\N	Seiches-sur-le-Loir	49
18691	334	\N	Sermaise	49
18692	335	\N	Surdres	49
18693	336	\N	Somloire	49
18694	337	\N	Soucelles	49
18695	338	\N	Soulaines-sur-Aubance	49
18696	339	\N	Soulaire-et-Bourg	49
18697	341	\N	Souzay-Champigny	49
18698	342	\N	Tancoigné	49
18699	343	\N	Tessoualle	49
18700	344	\N	Thorigné-d'Anjou	49
18701	345	\N	Thouarcé	49
18702	346	\N	Thoureil	49
18703	347	\N	Tiercé	49
18704	348	\N	Tigné	49
18705	349	\N	Tillières	49
18706	350	\N	Torfou	49
18707	351	\N	Tourlandry	49
18708	352	\N	Toutlemonde	49
18709	353	\N	Trélazé	49
18710	354	\N	Tremblay	49
18711	355	\N	Trémentines	49
18712	356	\N	Trémont	49
18713	358	\N	Turquant	49
18714	359	\N	Ulmes	49
18715	360	\N	Varenne	49
18716	361	\N	Varennes-sur-Loire	49
18717	362	\N	Varrains	49
18718	363	\N	Vauchrétien	49
18719	364	\N	Vaudelnay	49
18720	365	\N	Verchers-sur-Layon	49
18721	366	\N	Vergonnes	49
18722	367	\N	Vern-d'Anjou	49
18723	368	\N	Vernantes	49
18724	369	\N	Vernoil	49
18725	370	\N	Verrie	49
18726	371	\N	Vezins	49
18727	372	\N	Vieil-Baugé	49
18728	373	\N	Vihiers	49
18729	374	\N	Villebernier	49
18730	375	\N	Villedieu-la-Blouère	49
18731	376	\N	Villemoisan	49
18732	377	\N	Villevêque	49
18733	378	\N	Vivy	49
18734	380	\N	Vaulandry	49
18735	381	\N	Yzernay	49
18736	001	\N	Acqueville	50
18737	002	\N	Agneaux	50
18738	003	\N	Agon-Coutainville	50
18739	004	\N	Airel	50
18740	005	\N	Amfreville	50
18741	006	\N	Amigny	50
18742	007	\N	Ancteville	50
18743	008	\N	Anctoville-sur-Boscq	50
18744	009	\N	Angey	50
18745	010	\N	Angoville-au-Plain	50
18746	012	\N	Angoville-sur-Ay	50
18747	013	\N	Anneville-en-Saire	50
18748	014	\N	Anneville-sur-Mer	50
18749	015	\N	Annoville	50
18750	016	\N	Appeville	50
18751	018	\N	Argouges	50
18752	019	\N	Aucey-la-Plaine	50
18753	020	\N	Auderville	50
18754	021	\N	Audouville-la-Hubert	50
18755	022	\N	Aumeville-Lestre	50
18756	023	\N	Auvers	50
18757	024	\N	Auxais	50
18758	025	\N	Avranches	50
18759	026	\N	Azeville	50
18760	027	\N	Bacilly	50
18761	028	\N	Baleine	50
18762	029	\N	Barenton	50
18763	030	\N	Barfleur	50
18764	031	\N	Barneville-Carteret	50
18765	032	\N	Barre-de-Semilly	50
18766	033	\N	Baubigny	50
18767	034	\N	Baudre	50
18768	035	\N	Baudreville	50
18769	036	\N	Baupte	50
18770	037	\N	Bazoge	50
18771	038	\N	Beauchamps	50
18772	039	\N	Beaucoudray	50
18773	040	\N	Beauficel	50
18774	041	\N	Beaumont-Hague	50
18775	042	\N	Beauvoir	50
18776	043	\N	Bellefontaine	50
18777	044	\N	Belval	50
18778	045	\N	Benoîtville	50
18779	046	\N	Bérigny	50
18780	048	\N	Beslon	50
18781	049	\N	Besneville	50
18782	050	\N	Beuvrigny	50
18783	051	\N	Beuzeville-au-Plain	50
18784	052	\N	Beuzeville-la-Bastille	50
18785	054	\N	Biéville	50
18786	055	\N	Biniville	50
18787	056	\N	Bion	50
18788	057	\N	Biville	50
18789	058	\N	Blainville-sur-Mer	50
18790	059	\N	Blosville	50
18791	060	\N	Bloutière	50
18792	061	\N	Boisroger	50
18793	062	\N	Boisyvon	50
18794	063	\N	Bolleville	50
18795	064	\N	Bonneville	50
18796	066	\N	Jullouville	50
18797	069	\N	Bourguenolles	50
18798	070	\N	Boutteville	50
18799	071	\N	Braffais	50
18800	072	\N	Brainville	50
18801	073	\N	Branville-Hague	50
18802	074	\N	Brécey	50
18803	075	\N	Brectouville	50
18804	076	\N	Bréhal	50
18805	077	\N	Bretteville	50
18806	078	\N	Bretteville-sur-Ay	50
18807	079	\N	Breuville	50
18808	080	\N	Brévands	50
18809	081	\N	Bréville-sur-Mer	50
18810	082	\N	Bricquebec	50
18811	083	\N	Bricquebosq	50
18812	084	\N	Bricqueville-la-Blouette	50
18813	085	\N	Bricqueville-sur-Mer	50
18814	086	\N	Brillevast	50
18815	087	\N	Brix	50
18816	088	\N	Brouains	50
18817	089	\N	Brucheville	50
18818	090	\N	Buais	50
18819	092	\N	Cambernon	50
18820	093	\N	Cametours	50
18821	094	\N	Camprond	50
18822	095	\N	Canisy	50
18823	096	\N	Canteloup	50
18824	097	\N	Canville-la-Rocque	50
18825	098	\N	Carantilly	50
18826	099	\N	Carentan	50
18827	100	\N	Carnet	50
18828	101	\N	Carneville	50
18829	102	\N	Carolles	50
18830	103	\N	Carquebut	50
18831	105	\N	Catteville	50
18832	106	\N	Cavigny	50
18833	107	\N	Catz	50
18834	108	\N	Céaux	50
18835	109	\N	Cérences	50
18836	110	\N	Cerisy-la-Forêt	50
18837	111	\N	Cerisy-la-Salle	50
18838	112	\N	Chaise-Baudouin	50
18839	114	\N	Chambres	50
18840	115	\N	Champcervon	50
18841	116	\N	Champcey	50
18842	117	\N	Champeaux	50
18843	118	\N	Champrepus	50
18844	119	\N	Champs-de-Losque	50
18845	120	\N	Chanteloup	50
18846	121	\N	Chapelle-Cécelin	50
18847	123	\N	Chapelle-en-Juger	50
18848	124	\N	Chapelle-Urée	50
18849	125	\N	Chasseguey	50
18850	126	\N	Chavoy	50
18851	127	\N	Chef-du-Pont	50
18852	128	\N	Chefresne	50
18853	129	\N	Cherbourg-Octeville	50
18854	130	\N	Chérencé-le-Héron	50
18855	131	\N	Chérencé-le-Roussel	50
18856	132	\N	Chéris	50
18857	133	\N	Chèvreville	50
18858	134	\N	Chevry	50
18859	135	\N	Clitourps	50
18860	136	\N	Coigny	50
18861	137	\N	Colombe	50
18862	138	\N	Colomby	50
18863	139	\N	Condé-sur-Vire	50
18864	140	\N	Contrières	50
18865	142	\N	Cosqueville	50
18866	143	\N	Coudeville-sur-Mer	50
18867	144	\N	Coulouvray-Boisbenâtre	50
18868	145	\N	Courcy	50
18869	146	\N	Courtils	50
18870	147	\N	Coutances	50
18871	148	\N	Couvains	50
18872	149	\N	Couville	50
18873	150	\N	Crasville	50
18874	151	\N	Créances	50
18875	152	\N	Cresnays	50
18876	153	\N	Cretteville	50
18877	154	\N	Croix-Avranchin	50
18878	155	\N	Crollon	50
18879	156	\N	Crosville-sur-Douve	50
18880	158	\N	Cuves	50
18881	159	\N	Dangy	50
18882	160	\N	Denneville	50
18883	161	\N	Dézert	50
18884	162	\N	Digosville	50
18885	163	\N	Digulleville	50
18886	164	\N	Domjean	50
18887	165	\N	Donville-les-Bains	50
18888	166	\N	Doville	50
18889	167	\N	Dragey-Ronthon	50
18890	168	\N	Ducey	50
18891	169	\N	Écausseville	50
18892	170	\N	Écoquenéauville	50
18893	171	\N	Éculleville	50
18894	172	\N	Émondeville	50
18895	173	\N	Équeurdreville-Hainneville	50
18896	174	\N	Équilly	50
18897	175	\N	Éroudeville	50
18898	176	\N	Étang-Bertrand	50
18899	177	\N	Étienville	50
18900	178	\N	Fermanville	50
18901	179	\N	Ferrières	50
18902	180	\N	Fervaches	50
18903	181	\N	Feugères	50
18904	182	\N	Feuillie	50
18905	183	\N	Fierville-les-Mines	50
18906	184	\N	Flamanville	50
18907	185	\N	Fleury	50
18908	186	\N	Flottemanville	50
18909	187	\N	Flottemanville-Hague	50
18910	188	\N	Folligny	50
18911	189	\N	Fontenay	50
18912	190	\N	Fontenay-sur-Mer	50
18913	191	\N	Foucarville	50
18914	192	\N	Fourneaux	50
18915	193	\N	Fresne-Poret	50
18916	194	\N	Fresville	50
18917	195	\N	Gathemo	50
18918	196	\N	Gatteville-le-Phare	50
18919	197	\N	Gavray	50
18920	198	\N	Geffosses	50
18921	199	\N	Genêts	50
18922	200	\N	Ger	50
18923	202	\N	Giéville	50
18924	203	\N	Glacerie	50
18925	204	\N	Glatigny	50
18926	205	\N	Godefroy	50
18927	206	\N	Gohannière	50
18928	207	\N	Golleville	50
18929	208	\N	Gonfreville	50
18930	209	\N	Gonneville	50
18931	210	\N	Gorges	50
18932	211	\N	Gouberville	50
18933	212	\N	Gourbesville	50
18934	213	\N	Gourfaleur	50
18935	214	\N	Gouvets	50
18936	215	\N	Gouville-sur-Mer	50
18937	216	\N	Graignes	50
18938	217	\N	Grand-Celland	50
18939	218	\N	Granville	50
18940	219	\N	Gratot	50
18941	220	\N	Gréville-Hague	50
18942	221	\N	Grimesnil	50
18943	222	\N	Grosville	50
18944	223	\N	Guéhébert	50
18945	224	\N	Guilberville	50
18946	225	\N	Guislain	50
18947	227	\N	Ham	50
18948	228	\N	Hambye	50
18949	229	\N	Hamelin	50
18950	230	\N	Hardinvast	50
18951	231	\N	Hauteville-sur-Mer	50
18952	232	\N	Hauteville-la-Guichard	50
18953	233	\N	Hautteville-Bocage	50
18954	234	\N	Haye-Bellefond	50
18955	235	\N	Haye-d'Ectot	50
18956	236	\N	Haye-du-Puits	50
18957	237	\N	Haye-Pesnel	50
18958	238	\N	Héauville	50
18959	239	\N	Hébécrevon	50
18960	240	\N	Helleville	50
18961	241	\N	Hémevez	50
18962	242	\N	Herqueville	50
18963	243	\N	Heugueville-sur-Sienne	50
18964	244	\N	Hérenguerville	50
18965	245	\N	Heussé	50
18966	246	\N	Hiesville	50
18967	247	\N	Hocquigny	50
18968	248	\N	Hommet-d'Arthenay	50
18969	249	\N	Houesville	50
18970	250	\N	Houtteville	50
18971	251	\N	Huberville	50
18972	252	\N	Hudimesnil	50
18973	253	\N	Huisnes-sur-Mer	50
18974	254	\N	Husson	50
18975	255	\N	Hyenville	50
18976	256	\N	Isigny-le-Buat	50
18977	257	\N	Jobourg	50
18978	258	\N	Joganville	50
18979	259	\N	Juilley	50
18980	260	\N	Juvigny-le-Tertre	50
18981	261	\N	Lamberville	50
18982	262	\N	Lande-d'Airou	50
18983	263	\N	Lapenty	50
18984	265	\N	Laulne	50
18985	266	\N	Lengronne	50
18986	267	\N	Lessay	50
18987	268	\N	Lestre	50
18988	269	\N	Liesville-sur-Douve	50
18989	270	\N	Lieusaint	50
18990	271	\N	Lingeard	50
18991	272	\N	Lingreville	50
18992	273	\N	Lithaire	50
18993	274	\N	Loges-Marchis	50
18994	275	\N	Loges-sur-Brécey	50
18995	276	\N	Lolif	50
18996	277	\N	Longueville	50
18997	278	\N	Loreur	50
18998	279	\N	Lorey	50
18999	280	\N	Lozon	50
19000	281	\N	Lucerne-d'Outremer	50
19001	282	\N	Luot	50
19002	283	\N	Luzerne	50
19003	284	\N	Macey	50
19004	285	\N	Magneville	50
19005	287	\N	Mancellière-sur-Vire	50
19006	288	\N	Marcey-les-Grèves	50
19007	289	\N	Marchésieux	50
19008	290	\N	Marcilly	50
19009	291	\N	Margueray	50
19010	292	\N	Marigny	50
19011	293	\N	Martigny	50
19012	294	\N	Martinvast	50
19013	295	\N	Maupertuis	50
19014	296	\N	Maupertus-sur-Mer	50
19015	297	\N	Meauffe	50
19016	298	\N	Méautis	50
19017	299	\N	Mesnil	50
19018	300	\N	Mesnil-Adelée	50
19019	301	\N	Mesnil-Amand	50
19020	302	\N	Mesnil-Amey	50
19021	303	\N	Mesnil-Angot	50
19022	304	\N	Mesnil-Aubert	50
19023	305	\N	Mesnil-au-Val	50
19024	308	\N	Mesnilbus	50
19025	310	\N	Mesnil-Eury	50
19026	311	\N	Mesnil-Garnier	50
19027	312	\N	Mesnil-Gilbert	50
19028	313	\N	Mesnil-Herman	50
19029	315	\N	Mesnillard	50
19030	316	\N	Mesnil-Opac	50
19031	317	\N	Mesnil-Ozenne	50
19032	318	\N	Mesnil-Rainfray	50
19033	319	\N	Mesnil-Raoult	50
19034	320	\N	Mesnil-Rogues	50
19035	321	\N	Mesnil-Rouxelin	50
19036	323	\N	Mesnil-Tôve	50
19037	324	\N	Mesnil-Véneron	50
19038	325	\N	Mesnil-Vigot	50
19039	326	\N	Mesnil-Villeman	50
19040	327	\N	Meurdraquière	50
19041	328	\N	Millières	50
19042	329	\N	Milly	50
19043	330	\N	Mobecq	50
19044	332	\N	Moitiers-d'Allonne	50
19045	333	\N	Moitiers-en-Bauptois	50
19046	334	\N	Montabot	50
19047	335	\N	Montaigu-la-Brisette	50
19048	336	\N	Montaigu-les-Bois	50
19049	337	\N	Montanel	50
19050	338	\N	Montbray	50
19051	339	\N	Montchaton	50
19052	340	\N	Montcuit	50
19053	341	\N	Montebourg	50
19054	342	\N	Montfarville	50
19055	343	\N	Montgardon	50
19056	345	\N	Monthuchon	50
19057	347	\N	Montjoie-Saint-Martin	50
19058	348	\N	Montmartin-en-Graignes	50
19059	349	\N	Montmartin-sur-Mer	50
19060	350	\N	Montpinchon	50
19061	351	\N	Montrabot	50
19062	352	\N	Montreuil-sur-Lozon	50
19063	353	\N	Mont-Saint-Michel	50
19064	354	\N	Montsurvent	50
19065	355	\N	Montviron	50
19066	356	\N	Moon-sur-Elle	50
19067	357	\N	Morigny	50
19068	358	\N	Morsalines	50
19069	359	\N	Mortain	50
19070	360	\N	Morville	50
19071	361	\N	Mouche	50
19072	362	\N	Moulines	50
19073	363	\N	Moyon	50
19074	364	\N	Muneville-le-Bingard	50
19075	365	\N	Muneville-sur-Mer	50
19076	368	\N	Nay	50
19077	369	\N	Négreville	50
19078	370	\N	Néhou	50
19079	371	\N	Neufbourg	50
19080	372	\N	Neufmesnil	50
19081	373	\N	Neuville-au-Plain	50
19082	374	\N	Neuville-en-Beaumont	50
19083	375	\N	Néville-sur-Mer	50
19084	376	\N	Nicorps	50
19085	378	\N	Notre-Dame-de-Cenilly	50
19086	379	\N	Notre-Dame-de-Livoye	50
19087	380	\N	Notre-Dame-d'Elle	50
19088	381	\N	Notre-Dame-du-Touchet	50
19089	382	\N	Nouainville	50
19090	384	\N	Octeville-l'Avenel	50
19091	385	\N	Omonville-la-Petite	50
19092	386	\N	Omonville-la-Rogue	50
19093	387	\N	Orglandes	50
19094	388	\N	Orval	50
19095	389	\N	Ouville	50
19096	390	\N	Ozeville	50
19097	391	\N	Parigny	50
19098	393	\N	Percy	50
19099	394	\N	Périers	50
19100	395	\N	Pernelle	50
19101	396	\N	Perques	50
19102	397	\N	Perriers-en-Beauficel	50
19103	398	\N	Perron	50
19104	399	\N	Petit-Celland	50
19105	400	\N	Picauville	50
19106	401	\N	Pierreville	50
19107	402	\N	Pieux	50
19108	403	\N	Pirou	50
19109	404	\N	Placy-Montaigu	50
19110	405	\N	Plessis-Lastelle	50
19111	406	\N	Plomb	50
19112	407	\N	Poilley	50
19113	408	\N	Pontaubault	50
19114	409	\N	Pont-Hébert	50
19115	410	\N	Pontorson	50
19116	411	\N	Ponts	50
19117	412	\N	Portbail	50
19118	413	\N	Précey	50
19119	414	\N	Précorbin	50
19120	415	\N	Prétot-Sainte-Suzanne	50
19121	416	\N	Querqueville	50
19122	417	\N	Quettehou	50
19123	418	\N	Quettetot	50
19124	419	\N	Quettreville-sur-Sienne	50
19125	420	\N	Quibou	50
19126	421	\N	Quinéville	50
19127	422	\N	Raids	50
19128	423	\N	Rampan	50
19129	425	\N	Rauville-la-Bigot	50
19130	426	\N	Rauville-la-Place	50
19131	427	\N	Ravenoville	50
19132	428	\N	Reffuveille	50
19133	429	\N	Regnéville-sur-Mer	50
19134	430	\N	Reigneville-Bocage	50
19135	431	\N	Remilly-sur-Lozon	50
19136	432	\N	Réthoville	50
19137	433	\N	Réville	50
19138	434	\N	Rochelle-Normande	50
19139	435	\N	Rocheville	50
19140	436	\N	Romagny	50
19141	437	\N	Roncey	50
19142	438	\N	Ronde-Haye	50
19143	440	\N	Rouffigny	50
19144	441	\N	Rouxeville	50
19145	442	\N	Rozel	50
19146	443	\N	Sacey	50
19147	444	\N	Saint-Amand	50
19148	445	\N	Saint-André-de-Bohon	50
19149	446	\N	Saint-André-de-l'Épine	50
19150	447	\N	Saint-Aubin-des-Préaux	50
19151	448	\N	Saint-Aubin-de-Terregatte	50
19152	449	\N	Saint-Aubin-du-Perron	50
19153	450	\N	Saint-Barthélemy	50
19154	451	\N	Saint-Brice	50
19155	452	\N	Saint-Brice-de-Landelles	50
19156	453	\N	Sainte-Cécile	50
19157	454	\N	Saint-Christophe-du-Foc	50
19158	455	\N	Saint-Clair-sur-l'Elle	50
19159	456	\N	Saint-Clément-Rancoudray	50
19160	457	\N	Sainte-Colombe	50
19161	458	\N	Saint-Côme-du-Mont	50
19162	460	\N	Sainte-Croix-Hague	50
19163	461	\N	Saint-Cyr	50
19164	462	\N	Saint-Cyr-du-Bailleul	50
19165	463	\N	Saint-Denis-le-Gast	50
19166	464	\N	Saint-Denis-le-Vêtu	50
19167	465	\N	Saint-Ébremond-de-Bonfossé	50
19168	467	\N	Saint-Floxel	50
19169	468	\N	Saint-Fromond	50
19170	469	\N	Sainte-Geneviève	50
19171	470	\N	Saint-Georges-de-Bohon	50
19172	471	\N	Saint-Georges-de-la-Rivière	50
19173	472	\N	Saint-Georges-de-Livoye	50
19174	473	\N	Saint-Georges-d'Elle	50
19175	474	\N	Saint-Georges-de-Rouelley	50
19176	475	\N	Saint-Georges-Montcocq	50
19177	476	\N	Saint-Germain-d'Elle	50
19178	477	\N	Saint-Germain-des-Vaux	50
19179	478	\N	Saint-Germain-de-Tournebut	50
19180	479	\N	Saint-Germain-de-Varreville	50
19181	480	\N	Saint-Germain-le-Gaillard	50
19182	481	\N	Saint-Germain-sur-Ay	50
19183	482	\N	Saint-Germain-sur-Sèves	50
19184	483	\N	Saint-Gilles	50
19185	484	\N	Saint-Hilaire-du-Harcouët	50
19186	485	\N	Saint-Hilaire-Petitville	50
19187	486	\N	Saint-Jacques-de-Néhou	50
19188	487	\N	Saint-James	50
19189	488	\N	Saint-Jean-de-Daye	50
19190	489	\N	Saint-Jean-de-la-Haize	50
19191	490	\N	Saint-Jean-de-la-Rivière	50
19192	491	\N	Saint-Jean-de-Savigny	50
19193	492	\N	Saint-Jean-des-Baisants	50
19194	493	\N	Saint-Jean-des-Champs	50
19195	494	\N	Saint-Jean-du-Corail	50
19196	495	\N	Saint-Jean-du-Corail-des-Bois	50
19197	496	\N	Saint-Jean-le-Thomas	50
19198	497	\N	Saint-Jores	50
19199	498	\N	Saint-Joseph	50
19200	499	\N	Saint-Laurent-de-Cuves	50
19201	500	\N	Saint-Laurent-de-Terregatte	50
19202	502	\N	Saint-Lô	50
19203	503	\N	Saint-Lô-d'Ourville	50
19204	504	\N	Saint-Louet-sur-Vire	50
19205	505	\N	Saint-Loup	50
19206	506	\N	Saint-Malo-de-la-Lande	50
19207	507	\N	Saint-Marcouf	50
19208	508	\N	Sainte-Marie-du-Bois	50
19209	509	\N	Sainte-Marie-du-Mont	50
19210	510	\N	Saint-Martin-d'Aubigny	50
19211	511	\N	Saint-Martin-d'Audouville	50
19212	512	\N	Saint-Martin-de-Bonfossé	50
19213	513	\N	Saint-Martin-de-Cenilly	50
19214	514	\N	Chaulieu	50
19215	515	\N	Saint-Martin-de-Landelles	50
19216	516	\N	Saint-Martin-des-Champs	50
19217	517	\N	Saint-Martin-de-Varreville	50
19218	518	\N	Saint-Martin-le-Bouillant	50
19219	519	\N	Saint-Martin-le-Gréard	50
19220	520	\N	Saint-Martin-le-Hébert	50
19221	521	\N	Saint-Maur-des-Bois	50
19222	522	\N	Saint-Maurice-en-Cotentin	50
19223	523	\N	Sainte-Mère-Église	50
19224	524	\N	Saint-Michel-de-la-Pierre	50
19225	525	\N	Saint-Michel-de-Montjoie	50
19226	528	\N	Saint-Nicolas-de-Pierrepont	50
19227	529	\N	Saint-Nicolas-des-Bois	50
19228	531	\N	Saint-Ovin	50
19229	532	\N	Saint-Pair-sur-Mer	50
19230	533	\N	Saint-Patrice-de-Claids	50
19231	534	\N	Saint-Pellerin	50
19232	535	\N	Sainte-Pience	50
19233	536	\N	Saint-Pierre-d'Arthéglise	50
19234	537	\N	Saint-Pierre-de-Coutances	50
19235	538	\N	Saint-Pierre-de-Semilly	50
19236	539	\N	Saint-Pierre-Église	50
19237	540	\N	Saint-Pierre-Langers	50
19238	541	\N	Saint-Planchers	50
19239	542	\N	Saint-Pois	50
19240	543	\N	Saint-Quentin-sur-le-Homme	50
19241	544	\N	Saint-Rémy-des-Landes	50
19242	545	\N	Saint-Romphaire	50
19243	546	\N	Saint-Samson-de-Bonfossé	50
19244	548	\N	Saint-Sauveur-de-Pierrepont	50
19245	549	\N	Saint-Sauveur-la-Pommeraye	50
19246	550	\N	Saint-Sauveur-Lendelin	50
19247	551	\N	Saint-Sauveur-le-Vicomte	50
19248	552	\N	Saint-Sébastien-de-Raids	50
19249	553	\N	Saint-Senier-de-Beuvron	50
19250	554	\N	Saint-Senier-sous-Avranches	50
19251	556	\N	Sainte-Suzanne-sur-Vire	50
19252	557	\N	Saint-Symphorien-des-Monts	50
19253	558	\N	Saint-Symphorien-le-Valois	50
19254	562	\N	Saint-Vaast-la-Hougue	50
19255	563	\N	Saint-Vigor-des-Monts	50
19256	564	\N	Sainteny	50
19257	565	\N	Sartilly	50
19258	567	\N	Saussemesnil	50
19259	568	\N	Saussey	50
19260	569	\N	Savigny	50
19261	570	\N	Savigny-le-Vieux	50
19262	571	\N	Sébeville	50
19263	572	\N	Sénoville	50
19264	573	\N	Servigny	50
19265	574	\N	Servon	50
19266	575	\N	Sideville	50
19267	576	\N	Siouville-Hague	50
19268	577	\N	Sortosville-en-Beaumont	50
19269	578	\N	Sortosville	50
19270	579	\N	Sottevast	50
19271	580	\N	Sotteville	50
19272	581	\N	Soulles	50
19273	582	\N	Sourdeval	50
19274	583	\N	Sourdeval-les-Bois	50
19275	584	\N	Subligny	50
19276	585	\N	Surtainville	50
19277	586	\N	Surville	50
19278	587	\N	Taillepied	50
19279	588	\N	Tamerville	50
19280	589	\N	Tanis	50
19281	590	\N	Tanu	50
19282	591	\N	Teilleul	50
19283	592	\N	Tessy-sur-Vire	50
19284	593	\N	Teurthéville-Bocage	50
19285	594	\N	Teurthéville-Hague	50
19286	595	\N	Theil	50
19287	596	\N	Théville	50
19288	597	\N	Tirepied	50
19289	598	\N	Tocqueville	50
19290	599	\N	Tollevast	50
19291	600	\N	Tonneville	50
19292	601	\N	Torigni-sur-Vire	50
19293	602	\N	Tourlaville	50
19294	603	\N	Tourville-sur-Sienne	50
19295	604	\N	Tréauville	50
19296	605	\N	Trelly	50
19297	606	\N	Tribehou	50
19298	607	\N	Trinité	50
19299	608	\N	Troisgots	50
19300	609	\N	Turqueville	50
19301	610	\N	Urville	50
19302	611	\N	Urville-Nacqueville	50
19303	612	\N	Vains	50
19304	613	\N	Valcanville	50
19305	614	\N	Valdécie	50
19306	615	\N	Valognes	50
19307	616	\N	Val-Saint-Père	50
19308	617	\N	Varenguebec	50
19309	618	\N	Varouville	50
19310	619	\N	Vast	50
19311	620	\N	Vasteville	50
19312	621	\N	Vaudreville	50
19313	622	\N	Vaudrimesnil	50
19314	623	\N	Vauville	50
19315	624	\N	Vendelée	50
19316	625	\N	Vengeons	50
19317	626	\N	Ver	50
19318	627	\N	Vergoncey	50
19319	628	\N	Vernix	50
19320	629	\N	Vesly	50
19321	630	\N	Vessey	50
19322	631	\N	Veys	50
19323	633	\N	Vicel	50
19324	634	\N	Videcosville	50
19325	635	\N	Vidouville	50
19326	636	\N	Vierville	50
19327	637	\N	Villebaudon	50
19328	638	\N	Villechien	50
19329	639	\N	Villedieu-les-Poêles	50
19330	640	\N	Villiers-le-Pré	50
19331	641	\N	Villiers-Fossard	50
19332	642	\N	Vindefontaine	50
19333	643	\N	Virandeville	50
19334	644	\N	Virey	50
19335	646	\N	Vrétot	50
19336	647	\N	Yquelon	50
19337	648	\N	Yvetot-Bocage	50
19338	001	\N	Ablancourt	51
19339	002	\N	Saint-Martin-d'Ablois	51
19340	003	\N	Aigny	51
19341	004	\N	Allemanche-Launay-et-Soyer	51
19342	005	\N	Allemant	51
19343	006	\N	Alliancelles	51
19344	007	\N	Ambonnay	51
19345	008	\N	Ambrières	51
19346	009	\N	Anglure	51
19347	010	\N	Angluzelles-et-Courcelles	51
19348	012	\N	Anthenay	51
19349	013	\N	Aougny	51
19350	014	\N	Arcis-le-Ponsart	51
19351	015	\N	Argers	51
19352	016	\N	Arrigny	51
19353	017	\N	Arzillières-Neuville	51
19354	018	\N	Athis	51
19355	019	\N	Aubérive	51
19356	020	\N	Aubilly	51
19357	022	\N	Aulnay-l'Aître	51
19358	023	\N	Aulnay-sur-Marne	51
19359	025	\N	Auménancourt	51
19360	027	\N	Auve	51
19361	028	\N	Avenay-Val-d'Or	51
19362	029	\N	Avize	51
19363	030	\N	Ay	51
19364	031	\N	Baconnes	51
19365	032	\N	Bagneux	51
19366	033	\N	Baizil	51
19367	034	\N	Bannay	51
19368	035	\N	Bannes	51
19369	036	\N	Barbonne-Fayel	51
19370	037	\N	Baslieux-lès-Fismes	51
19371	038	\N	Baslieux-sous-Châtillon	51
19372	039	\N	Bassu	51
19373	040	\N	Bassuet	51
19374	041	\N	Baudement	51
19375	042	\N	Baye	51
19376	043	\N	Bazancourt	51
19377	044	\N	Beaumont-sur-Vesle	51
19378	045	\N	Beaunay	51
19379	046	\N	Beine-Nauroy	51
19380	047	\N	Belval-en-Argonne	51
19381	048	\N	Belval-sous-Châtillon	51
19382	049	\N	Bergères-lès-Vertus	51
19383	050	\N	Bergères-sous-Montmirail	51
19384	051	\N	Berméricourt	51
19385	052	\N	Berru	51
19386	053	\N	Berzieux	51
19387	054	\N	Bétheniville	51
19388	055	\N	Bétheny	51
19389	056	\N	Bethon	51
19390	057	\N	Bettancourt-la-Longue	51
19391	058	\N	Bezannes	51
19392	059	\N	Bignicourt-sur-Marne	51
19393	060	\N	Bignicourt-sur-Saulx	51
19394	061	\N	Billy-le-Grand	51
19395	062	\N	Binarville	51
19396	063	\N	Binson-et-Orquigny	51
19397	064	\N	Bisseuil	51
19398	065	\N	Blacy	51
19399	066	\N	Blaise-sous-Arzillières	51
19400	068	\N	Blesme	51
19401	069	\N	Bligny	51
19402	070	\N	Boissy-le-Repos	51
19403	071	\N	Bouchy-Saint-Genest	51
19404	072	\N	Bouilly	51
19405	073	\N	Bouleuse	51
19406	074	\N	Boult-sur-Suippe	51
19407	075	\N	Bourgogne	51
19408	076	\N	Boursault	51
19409	077	\N	Bouvancourt	51
19410	078	\N	Bouy	51
19411	079	\N	Bouzy	51
19412	080	\N	Brandonvillers	51
19413	081	\N	Branscourt	51
19414	082	\N	Braux-Sainte-Cohière	51
19415	083	\N	Braux-Saint-Remy	51
19416	084	\N	Bréban	51
19417	085	\N	Breuil	51
19418	086	\N	Breuil	51
19419	087	\N	Breuvery-sur-Coole	51
19420	088	\N	Brimont	51
19421	089	\N	Brouillet	51
19422	090	\N	Broussy-le-Grand	51
19423	091	\N	Broussy-le-Petit	51
19424	092	\N	Broyes	51
19425	093	\N	Brugny-Vaudancourt	51
19426	094	\N	Brusson	51
19427	095	\N	Buisson	51
19428	097	\N	Bussy-le-Château	51
19429	098	\N	Bussy-le-Repos	51
19430	099	\N	Bussy-Lettrée	51
19431	100	\N	Caure	51
19432	101	\N	Caurel	51
19433	102	\N	Cauroy-lès-Hermonville	51
19434	103	\N	Celle-sous-Chantemerle	51
19435	104	\N	Cernay-en-Dormois	51
19436	105	\N	Cernay-lès-Reims	51
19437	106	\N	Cernon	51
19438	107	\N	Chaintrix-Bierges	51
19439	108	\N	Châlons-en-Champagne	51
19440	109	\N	Châlons-sur-Vesle	51
19441	110	\N	Chaltrait	51
19442	111	\N	Chambrecy	51
19443	112	\N	Chamery	51
19444	113	\N	Champaubert	51
19445	115	\N	Champfleury	51
19446	116	\N	Champguyon	51
19447	117	\N	Champigneul-Champagne	51
19448	118	\N	Champigny	51
19449	119	\N	Champillon	51
19450	120	\N	Champlat-et-Boujacourt	51
19451	121	\N	Champvoisy	51
19452	122	\N	Changy	51
19453	124	\N	Chantemerle	51
19454	125	\N	Chapelaine	51
19455	126	\N	Chapelle-Felcourt	51
19456	127	\N	Chapelle-Lasson	51
19457	128	\N	Chapelle-sous-Orbais	51
19458	129	\N	Charleville	51
19459	130	\N	Charmont	51
19460	132	\N	Charmontois	51
19461	133	\N	Châtelier	51
19462	134	\N	Châtelraould-Saint-Louvent	51
19463	135	\N	Châtillon-sur-Broué	51
19464	136	\N	Châtillon-sur-Marne	51
19465	137	\N	Châtillon-sur-Morin	51
19466	138	\N	Châtrices	51
19467	139	\N	Chaudefontaine	51
19468	140	\N	Chaumuzy	51
19469	141	\N	Chaussée-sur-Marne	51
19470	142	\N	Chavot-Courcourt	51
19471	143	\N	Chemin	51
19472	144	\N	Cheminon	51
19473	145	\N	Chenay	51
19474	146	\N	Cheniers	51
19475	147	\N	Cheppe	51
19476	148	\N	Cheppes-la-Prairie	51
19477	149	\N	Chepy	51
19478	150	\N	Cherville	51
19479	151	\N	Chichey	51
19480	152	\N	Chigny-les-Roses	51
19481	153	\N	Chouilly	51
19482	154	\N	Clamanges	51
19483	155	\N	Clesles	51
19484	156	\N	Cloyes-sur-Marne	51
19485	157	\N	Coizard-Joches	51
19486	158	\N	Val-des-Marais	51
19487	160	\N	Compertrix	51
19488	161	\N	Condé-sur-Marne	51
19489	162	\N	Conflans-sur-Seine	51
19490	163	\N	Congy	51
19491	164	\N	Connantray-Vaurefroy	51
19492	165	\N	Connantre	51
19493	166	\N	Contault	51
19494	167	\N	Coole	51
19495	168	\N	Coolus	51
19496	169	\N	Corbeil	51
19497	170	\N	Corfélix	51
19498	171	\N	Cormicy	51
19499	172	\N	Cormontreuil	51
19500	173	\N	Cormoyeux	51
19501	174	\N	Corribert	51
19502	175	\N	Corrobert	51
19503	176	\N	Corroy	51
19504	177	\N	Coulommes-la-Montagne	51
19505	178	\N	Coupetz	51
19506	179	\N	Coupéville	51
19507	181	\N	Courcelles-Sapicourt	51
19508	182	\N	Courcemain	51
19509	183	\N	Courcy	51
19510	184	\N	Courdemanges	51
19511	185	\N	Courgivaux	51
19512	186	\N	Courjeonnet	51
19513	187	\N	Courlandon	51
19514	188	\N	Courmas	51
19515	190	\N	Courtagnon	51
19516	191	\N	Courtémont	51
19517	192	\N	Courthiézy	51
19518	193	\N	Courtisols	51
19519	194	\N	Courville	51
19520	195	\N	Couvrot	51
19521	196	\N	Cramant	51
19522	197	\N	Croix-en-Champagne	51
19523	198	\N	Crugny	51
19524	199	\N	Cuchery	51
19525	200	\N	Cuis	51
19526	202	\N	Cumières	51
19527	203	\N	Cuperly	51
19528	204	\N	Damery	51
19529	205	\N	Dampierre-au-Temple	51
19530	206	\N	Dampierre-le-Château	51
19531	208	\N	Dampierre-sur-Moivre	51
19532	210	\N	Dizy	51
19533	211	\N	Dommartin-Dampierre	51
19534	212	\N	Dommartin-Lettrée	51
19535	213	\N	Dommartin-sous-Hans	51
19536	214	\N	Dommartin-Varimont	51
19537	215	\N	Dompremy	51
19538	216	\N	Dontrien	51
19539	217	\N	Dormans	51
19540	218	\N	Val-de-Vière	51
19541	219	\N	Drosnay	51
19542	220	\N	Drouilly	51
19543	222	\N	Éclaires	51
19544	223	\N	Écollemont	51
19545	224	\N	Écriennes	51
19546	225	\N	Écueil	51
19547	226	\N	Écury-le-Repos	51
19548	227	\N	Écury-sur-Coole	51
19549	228	\N	Élise-Daucourt	51
19550	229	\N	Épense	51
19551	230	\N	Épernay	51
19552	231	\N	Épine	51
19553	232	\N	Époye	51
19554	233	\N	Escardes	51
19555	234	\N	Esclavolles-Lurey	51
19556	235	\N	Essarts-lès-Sézanne	51
19557	236	\N	Essarts-le-Vicomte	51
19558	237	\N	Esternay	51
19559	238	\N	Étoges	51
19560	239	\N	Étréchy	51
19561	240	\N	Étrepy	51
19562	241	\N	Euvy	51
19563	242	\N	Fagnières	51
19564	243	\N	Faux-Fresnay	51
19565	244	\N	Faux-Vésigneul	51
19566	245	\N	Faverolles-et-Coëmy	51
19567	246	\N	Favresse	51
19568	247	\N	Fèrebrianges	51
19569	248	\N	Fère-Champenoise	51
19570	249	\N	Festigny	51
19571	250	\N	Fismes	51
19572	251	\N	Flavigny	51
19573	252	\N	Fleury-la-Rivière	51
19574	253	\N	Florent-en-Argonne	51
19575	254	\N	Fontaine-Denis-Nuisy	51
19576	255	\N	Fontaine-en-Dormois	51
19577	256	\N	Fontaine-sur-Ay	51
19578	258	\N	Forestière	51
19579	259	\N	Francheville	51
19580	260	\N	Fresne	51
19581	261	\N	Fresne-lès-Reims	51
19582	262	\N	Frignicourt	51
19583	263	\N	Fromentières	51
19584	264	\N	Gault-Soigny	51
19585	265	\N	Gaye	51
19586	266	\N	Germaine	51
19587	267	\N	Germigny	51
19588	268	\N	Germinon	51
19589	269	\N	Giffaumont-Champaubert	51
19590	270	\N	Gigny-Bussy	51
19591	271	\N	Gionges	51
19592	272	\N	Givry-en-Argonne	51
19593	273	\N	Givry-lès-Loisy	51
19594	274	\N	Gizaucourt	51
19595	275	\N	Glannes	51
19596	276	\N	Gourgançon	51
19597	277	\N	Sainte-Marie-du-Lac-Nuisement	51
19598	278	\N	Grandes-Loges	51
19599	279	\N	Granges-sur-Aube	51
19600	280	\N	Gratreuil	51
19601	281	\N	Grauves	51
19602	282	\N	Gueux	51
19603	283	\N	Hans	51
19604	284	\N	Haussignémont	51
19605	285	\N	Haussimont	51
19606	286	\N	Hauteville	51
19607	287	\N	Hautvillers	51
19608	288	\N	Heiltz-le-Hutier	51
19609	289	\N	Heiltz-le-Maurupt	51
19610	290	\N	Heiltz-l'Évêque	51
19611	291	\N	Hermonville	51
19612	292	\N	Herpont	51
19613	293	\N	Heutrégiville	51
19614	294	\N	Hourges	51
19615	295	\N	Huiron	51
19616	296	\N	Humbauville	51
19617	298	\N	Igny-Comblizy	51
19618	299	\N	Isles-sur-Suippe	51
19619	300	\N	Isle-sur-Marne	51
19620	301	\N	Isse	51
19621	302	\N	Istres-et-Bury	51
19622	303	\N	Jâlons	51
19623	304	\N	Janvilliers	51
19624	305	\N	Janvry	51
19625	306	\N	Joiselle	51
19626	307	\N	Jonchery-sur-Suippe	51
19627	308	\N	Jonchery-sur-Vesle	51
19628	309	\N	Jonquery	51
19629	310	\N	Jouy-lès-Reims	51
19630	311	\N	Jussecourt-Minecourt	51
19631	312	\N	Juvigny	51
19632	313	\N	Lachy	51
19633	314	\N	Lagery	51
19634	315	\N	Landricourt	51
19635	316	\N	Larzicourt	51
19636	317	\N	Laval-sur-Tourbe	51
19637	318	\N	Lavannes	51
19638	319	\N	Lenharrée	51
19639	320	\N	Leuvrigny	51
19640	321	\N	Lhéry	51
19641	322	\N	Lignon	51
19642	323	\N	Linthelles	51
19643	324	\N	Linthes	51
19644	325	\N	Lisse-en-Champagne	51
19645	326	\N	Livry-Louvercy	51
19646	327	\N	Loisy-en-Brie	51
19647	328	\N	Loisy-sur-Marne	51
19648	329	\N	Loivre	51
19649	331	\N	Louvois	51
19650	333	\N	Ludes	51
19651	334	\N	Luxémont-et-Villotte	51
19652	336	\N	Maffrécourt	51
19653	337	\N	Magneux	51
19654	338	\N	Mailly-Champagne	51
19655	339	\N	Mairy-sur-Marne	51
19656	340	\N	Maisons-en-Champagne	51
19657	341	\N	Malmy	51
19658	342	\N	Mancy	51
19659	343	\N	Marcilly-sur-Seine	51
19660	344	\N	Mardeuil	51
19661	345	\N	Mareuil-en-Brie	51
19662	346	\N	Mareuil-le-Port	51
19663	347	\N	Mareuil-sur-Ay	51
19664	348	\N	Marfaux	51
19665	349	\N	Margerie-Hancourt	51
19666	350	\N	Margny	51
19667	351	\N	Marigny	51
19668	352	\N	Marolles	51
19669	353	\N	Marsangis	51
19670	354	\N	Marson	51
19671	355	\N	Massiges	51
19672	356	\N	Matignicourt-Goncourt	51
19673	357	\N	Matougues	51
19674	358	\N	Maurupt-le-Montois	51
19675	359	\N	Mécringes	51
19676	360	\N	Meix-Saint-Epoing	51
19677	361	\N	Meix-Tiercelin	51
19678	362	\N	Merfy	51
19679	363	\N	Merlaut	51
19680	364	\N	Méry-Prémecy	51
19681	365	\N	Mesneux	51
19682	367	\N	Mesnil-sur-Oger	51
19683	368	\N	Minaucourt-le-Mesnil-lès-Hurlus	51
19684	369	\N	Murs-Verdey	51
19685	370	\N	Moiremont	51
19686	371	\N	Moivre	51
19687	372	\N	Moncetz-Longevas	51
19688	373	\N	Moncetz-l'Abbaye	51
19689	374	\N	Mondement-Montgivroux	51
19690	375	\N	Montbré	51
19691	376	\N	Montgenost	51
19692	377	\N	Montépreux	51
19693	378	\N	Monthelon	51
19694	379	\N	Montigny-sur-Vesle	51
19695	380	\N	Montmirail	51
19696	381	\N	Montmort-Lucy	51
19697	382	\N	Mont-sur-Courville	51
19698	384	\N	Morangis	51
19699	386	\N	Morsains	51
19700	387	\N	Moslins	51
19701	388	\N	Mourmelon-le-Grand	51
19702	389	\N	Mourmelon-le-Petit	51
19703	390	\N	Moussy	51
19704	391	\N	Muizon	51
19705	392	\N	Mutigny	51
19706	393	\N	Nanteuil-la-Forêt	51
19707	395	\N	Nesle-la-Reposte	51
19708	396	\N	Nesle-le-Repons	51
19709	397	\N	Neuville-aux-Bois	51
19710	398	\N	Neuville-aux-Larris	51
19711	399	\N	Neuville-au-Pont	51
19712	402	\N	Neuvy	51
19713	403	\N	Nogent-l'Abbesse	51
19714	404	\N	Noirlieu	51
19715	406	\N	Norrois	51
19716	407	\N	Noue	51
19717	409	\N	Nuisement-sur-Coole	51
19718	410	\N	uilly	51
19719	411	\N	Oger	51
19720	412	\N	Ognes	51
19721	413	\N	Oiry	51
19722	414	\N	Olizy	51
19723	415	\N	Omey	51
19724	416	\N	Orbais-l'Abbaye	51
19725	417	\N	Orconte	51
19726	418	\N	Ormes	51
19727	419	\N	Outines	51
19728	420	\N	Outrepont	51
19729	421	\N	Oyes	51
19730	422	\N	Pargny-lès-Reims	51
19731	423	\N	Pargny-sur-Saulx	51
19732	424	\N	Passavant-en-Argonne	51
19733	425	\N	Passy-Grigny	51
19734	426	\N	Péas	51
19735	428	\N	Petites-Loges	51
19736	429	\N	Pévy	51
19737	430	\N	Pierre-Morains	51
19738	431	\N	Pierry	51
19739	432	\N	Pleurs	51
19740	433	\N	Plichancourt	51
19741	434	\N	Plivot	51
19742	435	\N	Pocancy	51
19743	436	\N	Pogny	51
19744	437	\N	Poilly	51
19745	438	\N	Poix	51
19746	439	\N	Pomacle	51
19747	440	\N	Pontfaverger-Moronvilliers	51
19748	441	\N	Ponthion	51
19749	442	\N	Possesse	51
19750	443	\N	Potangis	51
19751	444	\N	Pouillon	51
19752	445	\N	Pourcy	51
19753	446	\N	Pringy	51
19754	447	\N	Prosnes	51
19755	448	\N	Prouilly	51
19756	449	\N	Prunay	51
19757	450	\N	Puisieulx	51
19758	451	\N	Queudes	51
19759	452	\N	Rapsécourt	51
19760	453	\N	Recy	51
19761	454	\N	Reims	51
19762	455	\N	Reims-la-Brûlée	51
19763	456	\N	Remicourt	51
19764	457	\N	Reuil	51
19765	458	\N	Reuves	51
19766	459	\N	Réveillon	51
19767	460	\N	Rieux	51
19768	461	\N	Rilly-la-Montagne	51
19769	463	\N	Rivières-Henruel	51
19770	464	\N	Romain	51
19771	465	\N	Romery	51
19772	466	\N	Romigny	51
19773	468	\N	Rosnay	51
19774	469	\N	Rouffy	51
19775	470	\N	Rouvroy-Ripont	51
19776	471	\N	Sacy	51
19777	472	\N	Saint-Amand-sur-Fion	51
19778	473	\N	Saint-Bon	51
19779	474	\N	Saint-Brice-Courcelles	51
19780	475	\N	Saint-Chéron	51
19781	476	\N	Saint-Étienne-au-Temple	51
19782	477	\N	Saint-Étienne-sur-Suippe	51
19783	478	\N	Saint-Eulien	51
19784	479	\N	Saint-Euphraise-et-Clairizet	51
19785	480	\N	Sainte-Gemme	51
19786	482	\N	Saint-Germain-la-Ville	51
19787	483	\N	Saint-Gibrien	51
19788	484	\N	Saint-Gilles	51
19789	485	\N	Saint-Hilaire-au-Temple	51
19790	486	\N	Saint-Hilaire-le-Grand	51
19791	487	\N	Saint-Hilaire-le-Petit	51
19792	488	\N	Saint-Imoges	51
19793	489	\N	Saint-Jean-devant-Possesse	51
19794	490	\N	Saint-Jean-sur-Moivre	51
19795	491	\N	Saint-Jean-sur-Tourbe	51
19796	492	\N	Saint-Just-Sauvage	51
19797	493	\N	Saint-Léonard	51
19798	495	\N	Saint-Loup	51
19799	496	\N	Saint-Lumier-en-Champagne	51
19800	497	\N	Saint-Lumier-la-Populeuse	51
19801	498	\N	Saint-Mard-sur-Auve	51
19802	499	\N	Saint-Mard-lès-Rouffy	51
19803	500	\N	Saint-Mard-sur-le-Mont	51
19804	501	\N	Sainte-Marie-à-Py	51
19805	502	\N	Saint-Martin-aux-Champs	51
19806	503	\N	Saint-Martin-l'Heureux	51
19807	504	\N	Saint-Martin-sur-le-Pré	51
19808	505	\N	Saint-Masmes	51
19809	506	\N	Saint-Memmie	51
19810	507	\N	Sainte-Menehould	51
19811	508	\N	Saint-Ouen-Domprot	51
19812	509	\N	Saint-Pierre	51
19813	510	\N	Saint-Quentin-les-Marais	51
19814	511	\N	Saint-Quentin-le-Verger	51
19815	512	\N	Saint-Quentin-sur-Coole	51
19816	513	\N	Saint-Remy-en-Bouzemont-Saint-Genest-et-Isson	51
19817	514	\N	Saint-Remy-sous-Broyes	51
19818	515	\N	Saint-Remy-sur-Bussy	51
19819	516	\N	Saint-Saturnin	51
19820	517	\N	Saint-Souplet-sur-Py	51
19821	518	\N	Saint-Thierry	51
19822	519	\N	Saint-Thomas-en-Argonne	51
19823	520	\N	Saint-Utin	51
19824	521	\N	Saint-Vrain	51
19825	522	\N	Sapignicourt	51
19826	523	\N	Sarcy	51
19827	524	\N	Saron-sur-Aube	51
19828	525	\N	Sarry	51
19829	526	\N	Saudoy	51
19830	527	\N	Savigny-sur-Ardres	51
19831	528	\N	Scrupt	51
19832	529	\N	Selles	51
19833	530	\N	Sept-Saulx	51
19834	531	\N	Sermaize-les-Bains	51
19835	532	\N	Sermiers	51
19836	533	\N	Servon-Melzicourt	51
19837	534	\N	Serzy-et-Prin	51
19838	535	\N	Sézanne	51
19839	536	\N	Sillery	51
19840	537	\N	Sivry-Ante	51
19841	538	\N	Sogny-aux-Moulins	51
19842	539	\N	Sogny-en-l'Angle	51
19843	542	\N	Soizy-aux-Bois	51
19844	543	\N	Somme-Bionne	51
19845	544	\N	Sommepy-Tahure	51
19846	545	\N	Sommesous	51
19847	546	\N	Somme-Suippe	51
19848	547	\N	Somme-Tourbe	51
19849	548	\N	Somme-Vesle	51
19850	549	\N	Somme-Yèvre	51
19851	550	\N	Sompuis	51
19852	551	\N	Somsois	51
19853	552	\N	Songy	51
19854	553	\N	Souain-Perthes-lès-Hurlus	51
19855	555	\N	Soudé	51
19856	556	\N	Soudron	51
19857	557	\N	Soulanges	51
19858	558	\N	Soulières	51
19859	559	\N	Suippes	51
19860	560	\N	Suizy-le-Franc	51
19861	562	\N	Taissy	51
19862	563	\N	Talus-Saint-Prix	51
19863	564	\N	Tauxières-Mutry	51
19864	565	\N	Thaas	51
19865	566	\N	Thibie	51
19866	567	\N	Thiéblemont-Farémont	51
19867	568	\N	Thil	51
19868	569	\N	Thillois	51
19869	570	\N	Thoult-Trosnay	51
19870	571	\N	Val-de-Vesle	51
19871	572	\N	Tilloy-et-Bellay	51
19872	573	\N	Tinqueux	51
19873	574	\N	Togny-aux-Bufs	51
19874	576	\N	Tours-sur-Marne	51
19875	577	\N	Tramery	51
19876	578	\N	Trécon	51
19877	579	\N	Tréfols	51
19878	580	\N	Trépail	51
19879	581	\N	Treslon	51
19880	582	\N	Trigny	51
19881	583	\N	Trois-Fontaines-l'Abbaye	51
19882	584	\N	Trois-Puits	51
19883	585	\N	Troissy	51
19884	586	\N	Unchair	51
19885	587	\N	Vadenay	51
19886	588	\N	Valmy	51
19887	589	\N	Vanault-le-Châtel	51
19888	590	\N	Vanault-les-Dames	51
19889	591	\N	Vandeuil	51
19890	592	\N	Vandières	51
19891	594	\N	Vassimont-et-Chapelaine	51
19892	595	\N	Vatry	51
19893	596	\N	Vauchamps	51
19894	597	\N	Vauciennes	51
19895	598	\N	Vauclerc	51
19896	599	\N	Vaudemange	51
19897	600	\N	Vaudesincourt	51
19898	601	\N	Vavray-le-Grand	51
19899	602	\N	Vavray-le-Petit	51
19900	603	\N	Vélye	51
19901	604	\N	Ventelay	51
19902	605	\N	Venteuil	51
19903	607	\N	Verdon	51
19904	608	\N	Vernancourt	51
19905	609	\N	Verneuil	51
19906	610	\N	Verrières	51
19907	611	\N	Vert-Toulon	51
19908	612	\N	Vertus	51
19909	613	\N	Verzenay	51
19910	614	\N	Verzy	51
19911	616	\N	Vésigneul-sur-Marne	51
19912	617	\N	Veuve	51
19913	618	\N	Vézier	51
19914	619	\N	Vieil-Dampierre	51
19915	620	\N	Vienne-la-Ville	51
19916	621	\N	Vienne-le-Château	51
19917	622	\N	Ville-Dommange	51
19918	623	\N	Ville-en-Selve	51
19919	624	\N	Ville-en-Tardenois	51
19920	625	\N	Villeneuve-la-Lionne	51
19921	626	\N	Villeneuve-lès-Charleville	51
19922	627	\N	Villeneuve-Renneville-Chevigny	51
19923	628	\N	Villeneuve-Saint-Vistre-et-Villevotte	51
19924	629	\N	Villers-Allerand	51
19925	630	\N	Villers-aux-Bois	51
19926	631	\N	Villers-aux-Nuds	51
19927	632	\N	Villers-en-Argonne	51
19928	633	\N	Villers-Franqueux	51
19929	634	\N	Villers-le-Château	51
19930	635	\N	Villers-le-Sec	51
19931	636	\N	Villers-Marmery	51
19932	637	\N	Villers-sous-Châtillon	51
19933	638	\N	Villeseneux	51
19934	639	\N	Ville-sous-Orbais	51
19935	640	\N	Ville-sur-Tourbe	51
19936	641	\N	Villevenard	51
19937	642	\N	Villiers-aux-Corneilles	51
19938	643	\N	Vinay	51
19939	644	\N	Vincelles	51
19940	645	\N	Vindey	51
19941	646	\N	Virginy	51
19942	647	\N	Vitry-en-Perthois	51
19943	648	\N	Vitry-la-Ville	51
19944	649	\N	Vitry-le-François	51
19945	650	\N	Voilemont	51
19946	651	\N	Voipreux	51
19947	652	\N	Vouarces	51
19948	654	\N	Vouillers	51
19949	655	\N	Vouzy	51
19950	656	\N	Vraux	51
19951	657	\N	Vrigny	51
19952	658	\N	Vroil	51
19953	659	\N	Wargemoulin-Hurlus	51
19954	660	\N	Warmeriville	51
19955	662	\N	Witry-lès-Reims	51
19956	663	\N	Magenta	51
19957	001	\N	Ageville	52
19958	002	\N	Aigremont	52
19959	003	\N	Aillianville	52
19960	004	\N	Aingoulaincourt	52
19961	005	\N	Aizanville	52
19962	006	\N	Allichamps	52
19963	007	\N	Ambonville	52
19964	008	\N	Andelot-Blancheville	52
19965	009	\N	Andilly-en-Bassigny	52
19966	011	\N	Annéville-la-Prairie	52
19967	012	\N	Annonville	52
19968	013	\N	Anrosey	52
19969	014	\N	Aprey	52
19970	015	\N	Arbigny-sous-Varennes	52
19971	016	\N	Arbot	52
19972	017	\N	Arc-en-Barrois	52
19973	019	\N	Arnancourt	52
19974	021	\N	Attancourt	52
19975	022	\N	Aubepierre-sur-Aube	52
19976	023	\N	Auberive	52
19977	025	\N	Audeloncourt	52
19978	027	\N	Aujeurres	52
19979	028	\N	Aulnoy-sur-Aube	52
19980	029	\N	Autigny-le-Grand	52
19981	030	\N	Autigny-le-Petit	52
19982	031	\N	Autreville-sur-la-Renne	52
19983	034	\N	Bailly-aux-Forges	52
19984	035	\N	Baissey	52
19985	036	\N	Balesmes-sur-Marne	52
19986	037	\N	Bannes	52
19987	038	\N	Bassoncourt	52
19988	039	\N	Baudrecourt	52
19989	040	\N	Bay-sur-Aube	52
19990	042	\N	Beauchemin	52
19991	043	\N	Belmont	52
19992	044	\N	Roches-Bettaincourt	52
19993	045	\N	Bettancourt-la-Ferrée	52
19994	047	\N	Beurville	52
19995	050	\N	Biesles	52
19996	051	\N	Bize	52
19997	053	\N	Blaisy	52
19998	055	\N	Blécourt	52
19999	056	\N	Blessonville	52
20000	057	\N	Blumeray	52
20001	058	\N	Bologne	52
20002	059	\N	Bonnecourt	52
20003	060	\N	Bourbonne-les-Bains	52
20004	061	\N	Bourdons-sur-Rognon	52
20005	062	\N	Bourg	52
20006	063	\N	Bourg-Sainte-Marie	52
20007	064	\N	Bourmont	52
20008	065	\N	Bouzancourt	52
20009	066	\N	Brachay	52
20010	067	\N	Brainville-sur-Meuse	52
20011	069	\N	Braux-le-Châtel	52
20012	070	\N	Brennes	52
20013	072	\N	Brethenay	52
20014	074	\N	Breuvannes-en-Bassigny	52
20015	075	\N	Briaucourt	52
20016	076	\N	Bricon	52
20017	079	\N	Brousseval	52
20018	082	\N	Bugnières	52
20019	083	\N	Champsevraine	52
20020	084	\N	Busson	52
20021	085	\N	Buxières-lès-Clefmont	52
20022	087	\N	Buxières-lès-Villiers	52
20023	088	\N	Ceffonds	52
20024	089	\N	Celles-en-Bassigny	52
20025	090	\N	Celsoy	52
20026	091	\N	Cerisières	52
20027	092	\N	Chalancey	52
20028	093	\N	Chalindrey	52
20029	094	\N	Vals-des-Tilles	52
20030	095	\N	Chalvraines	52
20031	097	\N	Chambroncourt	52
20032	099	\N	Chamouilley	52
20033	101	\N	Champigneulles-en-Bassigny	52
20034	102	\N	Champigny-lès-Langres	52
20035	103	\N	Champigny-sous-Varennes	52
20036	104	\N	Chancenay	52
20037	105	\N	Changey	52
20038	106	\N	Chanoy	52
20039	107	\N	Chantraines	52
20040	108	\N	Charmes	52
20041	109	\N	Charmes-en-l'Angle	52
20042	110	\N	Charmes-la-Grande	52
20043	113	\N	Chassigny	52
20044	114	\N	Châteauvillain	52
20045	115	\N	Chatenay-Mâcheron	52
20046	116	\N	Chatenay-Vaudin	52
20047	118	\N	Chatonrupt-Sommermont	52
20048	119	\N	Chaudenay	52
20049	120	\N	Chauffourt	52
20050	121	\N	Chaumont	52
20051	122	\N	Chaumont-la-Ville	52
20052	123	\N	Chevillon	52
20053	125	\N	Chamarandes-Choignes	52
20054	126	\N	Choilley-Dardenay	52
20055	127	\N	Choiseul	52
20056	128	\N	Cirey-lès-Mareilles	52
20057	129	\N	Cirey-sur-Blaise	52
20058	130	\N	Cirfontaines-en-Azois	52
20059	131	\N	Cirfontaines-en-Ornois	52
20060	132	\N	Clefmont	52
20061	133	\N	Clinchamp	52
20062	134	\N	Cohons	52
20063	135	\N	Coiffy-le-Bas	52
20064	136	\N	Coiffy-le-Haut	52
20065	137	\N	Colmier-le-Bas	52
20066	138	\N	Colmier-le-Haut	52
20067	140	\N	Colombey-les-Deux-Églises	52
20068	141	\N	Condes	52
20069	142	\N	Consigny	52
20070	145	\N	Coublanc	52
20071	146	\N	Coupray	52
20072	147	\N	Courcelles-en-Montagne	52
20073	149	\N	Courcelles-sur-Blaise	52
20074	151	\N	Cour-l'Évêque	52
20075	155	\N	Culmont	52
20076	156	\N	Curel	52
20077	157	\N	Curmont	52
20078	158	\N	Cusey	52
20079	159	\N	Cuves	52
20080	160	\N	Daillancourt	52
20081	161	\N	Daillecourt	52
20082	162	\N	Dammartin-sur-Meuse	52
20083	163	\N	Dampierre	52
20084	164	\N	Damrémont	52
20085	165	\N	Dancevoir	52
20086	167	\N	Darmannes	52
20087	168	\N	Dinteville	52
20088	169	\N	Domblain	52
20089	170	\N	Dommarien	52
20090	171	\N	Dommartin-le-Franc	52
20091	172	\N	Dommartin-le-Saint-Père	52
20092	173	\N	Domremy-Landéville	52
20093	174	\N	Doncourt-sur-Meuse	52
20094	175	\N	Donjeux	52
20095	177	\N	Doulaincourt-Saucourt	52
20096	178	\N	Doulevant-le-Château	52
20097	179	\N	Doulevant-le-Petit	52
20098	180	\N	Droyes	52
20099	181	\N	Échenay	52
20100	182	\N	Éclaron-Braucourt-Sainte-Livière	52
20101	183	\N	Ecot-la-Combe	52
20102	184	\N	Effincourt	52
20103	185	\N	Enfonvelle	52
20104	187	\N	Épizon	52
20105	189	\N	Val-d'Esnoms	52
20106	190	\N	Esnouveaux	52
20107	193	\N	Euffigneix	52
20108	194	\N	Eurville-Bienville	52
20109	195	\N	Farincourt	52
20110	196	\N	Faverolles	52
20111	197	\N	Fayl-Billot	52
20112	198	\N	Fays	52
20113	199	\N	Ferrière-et-Lafolie	52
20114	200	\N	Flagey	52
20115	201	\N	Flammerécourt	52
20116	203	\N	Fontaines-sur-Marne	52
20117	204	\N	Forcey	52
20118	205	\N	Foulain	52
20119	206	\N	Frampas	52
20120	207	\N	Frécourt	52
20121	208	\N	Fresnes-sur-Apance	52
20122	211	\N	Froncles	52
20123	212	\N	Fronville	52
20124	213	\N	Genevrières	52
20125	214	\N	Genevroye	52
20126	216	\N	Germaines	52
20127	217	\N	Germainvilliers	52
20128	218	\N	Germay	52
20129	219	\N	Germisay	52
20130	220	\N	Giey-sur-Aujon	52
20131	221	\N	Gillancourt	52
20132	222	\N	Gillaumé	52
20133	223	\N	Gilley	52
20134	225	\N	Goncourt	52
20135	227	\N	Graffigny-Chemin	52
20136	228	\N	Grandchamp	52
20137	229	\N	Grenant	52
20138	230	\N	Gudmont-Villiers	52
20139	231	\N	Guindrecourt-aux-Ormes	52
20140	232	\N	Guindrecourt-sur-Blaise	52
20141	233	\N	Guyonvelle	52
20142	234	\N	Hâcourt	52
20143	235	\N	Hallignicourt	52
20144	237	\N	Harréville-les-Chanteurs	52
20145	239	\N	Heuilley-Cotton	52
20146	240	\N	Heuilley-le-Grand	52
20147	242	\N	Haute-Amance	52
20148	243	\N	Huilliécourt	52
20149	244	\N	Humbécourt	52
20150	245	\N	Humberville	52
20151	246	\N	Humes-Jorquenay	52
20152	247	\N	Illoud	52
20153	248	\N	Is-en-Bassigny	52
20154	249	\N	Isômes	52
20155	250	\N	Joinville	52
20156	251	\N	Jonchery	52
20157	253	\N	Juzennecourt	52
20158	254	\N	Lachapelle-en-Blaisy	52
20159	256	\N	Lafauche	52
20160	257	\N	Laferté-sur-Amance	52
20161	258	\N	Laferté-sur-Aube	52
20162	260	\N	Lamancine	52
20163	262	\N	Lamothe-en-Blaisy	52
20164	264	\N	Laneuvelle	52
20165	265	\N	Bayard-sur-Marne	52
20166	267	\N	Laneuville-au-Pont	52
20167	269	\N	Langres	52
20168	271	\N	Lanques-sur-Rognon	52
20169	272	\N	Lanty-sur-Aube	52
20170	273	\N	Larivière-Arnoncourt	52
20171	274	\N	Latrecey-Ormoy-sur-Aube	52
20172	275	\N	Lavernoy	52
20173	276	\N	Laville-aux-Bois	52
20174	277	\N	Lavilleneuve	52
20175	280	\N	Lecey	52
20176	282	\N	Leffonds	52
20177	284	\N	Leschères-sur-le-Blaiseron	52
20178	285	\N	Leuchey	52
20179	286	\N	Leurville	52
20180	287	\N	Levécourt	52
20181	288	\N	Lezéville	52
20182	289	\N	Liffol-le-Petit	52
20183	290	\N	Loges	52
20184	291	\N	Longchamp	52
20185	292	\N	Longeau-Percey	52
20186	293	\N	Longeville-sur-la-Laines	52
20187	294	\N	Louvemont	52
20188	295	\N	Louvières	52
20189	296	\N	Louze	52
20190	297	\N	Luzy-sur-Marne	52
20191	298	\N	Maâtz	52
20192	300	\N	Magneux	52
20193	301	\N	Maisoncelles	52
20194	302	\N	Maizières	52
20195	303	\N	Maizières-sur-Amance	52
20196	304	\N	Malaincourt-sur-Meuse	52
20197	305	\N	Mandres-la-Côte	52
20198	306	\N	Manois	52
20199	307	\N	Marac	52
20200	308	\N	Maranville	52
20201	310	\N	Marbéville	52
20202	311	\N	Marcilly-en-Bassigny	52
20203	312	\N	Mardor	52
20204	313	\N	Mareilles	52
20205	315	\N	Marnay-sur-Marne	52
20206	316	\N	Mathons	52
20207	318	\N	Melay	52
20208	319	\N	Mennouveaux	52
20209	320	\N	Merrey	52
20210	321	\N	Mertrud	52
20211	322	\N	Meures	52
20212	325	\N	Millières	52
20213	326	\N	Mirbel	52
20214	327	\N	Moëslains	52
20215	328	\N	Montcharvot	52
20216	330	\N	Montheries	52
20217	331	\N	Montier-en-Der	52
20218	332	\N	Val-de-Meuse	52
20219	335	\N	Montot-sur-Rognon	52
20220	336	\N	Montreuil-sur-Blaise	52
20221	337	\N	Montreuil-sur-Thonnance	52
20222	340	\N	Montsaugeon	52
20223	341	\N	Morancourt	52
20224	342	\N	Morionvilliers	52
20225	344	\N	Mouilleron	52
20226	346	\N	Mussey-sur-Marne	52
20227	347	\N	Narcy	52
20228	348	\N	Neuilly-l'Évêque	52
20229	349	\N	Neuilly-sur-Suize	52
20230	350	\N	Neuvelle-lès-Voisey	52
20231	351	\N	Nijon	52
20232	352	\N	Ninville	52
20233	353	\N	Nogent	52
20234	354	\N	Noidant-Chatenoy	52
20235	355	\N	Noidant-le-Rocheux	52
20236	356	\N	Nomécourt	52
20237	357	\N	Noncourt-sur-le-Rongeant	52
20238	358	\N	Noyers	52
20239	359	\N	Nully	52
20240	360	\N	Occey	52
20241	362	\N	Orbigny-au-Mont	52
20242	363	\N	Orbigny-au-Val	52
20243	364	\N	Orcevaux	52
20244	365	\N	Orges	52
20245	366	\N	Ormancey	52
20246	367	\N	Ormoy-lès-Sexfontaines	52
20247	369	\N	Orquevaux	52
20248	370	\N	Osne-le-Val	52
20249	371	\N	Oudincourt	52
20250	372	\N	Outremécourt	52
20251	373	\N	Ozières	52
20252	374	\N	Pailly	52
20253	375	\N	Palaiseul	52
20254	376	\N	Pancey	52
20255	377	\N	Parnoy-en-Bassigny	52
20256	378	\N	Paroy-sur-Saulx	52
20257	379	\N	Pautaines-Augeville	52
20258	380	\N	Peigney	52
20259	383	\N	Perrancey-les-Vieux-Moulins	52
20260	384	\N	Perrogney-les-Fontaines	52
20261	385	\N	Perrusse	52
20262	386	\N	Perthes	52
20263	388	\N	Pierremont-sur-Amance	52
20264	390	\N	Pisseloup	52
20265	391	\N	Planrupt	52
20266	392	\N	Plesnoy	52
20267	393	\N	Poinsenot	52
20268	394	\N	Poinson-lès-Fayl	52
20269	395	\N	Poinson-lès-Grancey	52
20270	396	\N	Poinson-lès-Nogent	52
20271	397	\N	Poiseul	52
20272	398	\N	Poissons	52
20273	399	\N	Pont-la-Ville	52
20274	400	\N	Châtelet-sur-Meuse	52
20275	401	\N	Poulangy	52
20276	403	\N	Praslay	52
20277	405	\N	Prauthoy	52
20278	406	\N	Pressigny	52
20279	407	\N	Prez-sous-Lafauche	52
20280	411	\N	Puellemontier	52
20281	413	\N	Rachecourt-Suzémont	52
20282	414	\N	Rachecourt-sur-Marne	52
20283	415	\N	Rançonnières	52
20284	416	\N	Rangecourt	52
20285	419	\N	Rennepont	52
20286	420	\N	Reynel	52
20287	421	\N	Riaucourt	52
20288	422	\N	Richebourg	52
20289	423	\N	Rimaucourt	52
20290	424	\N	Rivières-le-Bois	52
20291	425	\N	Rivière-les-Fosses	52
20292	426	\N	Rizaucourt-Buchey	52
20293	427	\N	Robert-Magny-Laneuville-à-Rémy	52
20294	428	\N	Rochefort-sur-la-Côte	52
20295	429	\N	Roches-sur-Marne	52
20296	431	\N	Rochetaillée	52
20297	432	\N	Rolampont	52
20298	433	\N	Romain-sur-Meuse	52
20299	436	\N	Rouécourt	52
20300	437	\N	Rouelles	52
20301	438	\N	Rougeux	52
20302	439	\N	Rouvres-sur-Aube	52
20303	440	\N	Rouvroy-sur-Marne	52
20304	442	\N	Rupt	52
20305	443	\N	Sailly	52
20306	444	\N	Saint-Blin	52
20307	445	\N	Saint-Broingt-le-Bois	52
20308	446	\N	Saint-Broingt-les-Fosses	52
20309	447	\N	Saint-Ciergues	52
20310	448	\N	Saint-Dizier	52
20311	449	\N	Saints-Geosmes	52
20312	450	\N	Saint-Loup-sur-Aujon	52
20313	452	\N	Saint-Martin-lès-Langres	52
20314	453	\N	Saint-Maurice	52
20315	455	\N	Saint-Thiébault	52
20316	456	\N	Saint-Urbain-Maconcourt	52
20317	457	\N	Saint-Vallier-sur-Marne	52
20318	459	\N	Sarcey	52
20319	461	\N	Sarrey	52
20320	463	\N	Saudron	52
20321	464	\N	Saulles	52
20322	467	\N	Savigny	52
20323	468	\N	Semilly	52
20324	469	\N	Semoutiers-Montsaon	52
20325	470	\N	Serqueux	52
20326	472	\N	Sexfontaines	52
20327	473	\N	Signéville	52
20328	474	\N	Silvarouvres	52
20329	475	\N	Sommancourt	52
20330	476	\N	Sommerécourt	52
20331	479	\N	Sommevoire	52
20332	480	\N	Soncourt-sur-Marne	52
20333	482	\N	Soulaucourt-sur-Mouzon	52
20334	483	\N	Soyers	52
20335	484	\N	Suzannecourt	52
20336	486	\N	Ternat	52
20337	487	\N	Thilleux	52
20338	488	\N	Thivet	52
20339	489	\N	Thol-lès-Millières	52
20340	490	\N	Thonnance-lès-Joinville	52
20341	491	\N	Thonnance-les-Moulins	52
20342	492	\N	Torcenay	52
20343	493	\N	Tornay	52
20344	494	\N	Treix	52
20345	495	\N	Trémilly	52
20346	497	\N	Troisfontaines-la-Ville	52
20347	499	\N	Vaillant	52
20348	500	\N	Valcourt	52
20349	502	\N	Valleret	52
20350	503	\N	Valleroy	52
20351	504	\N	Terre-Natale	52
20352	505	\N	Vaudrecourt	52
20353	506	\N	Vaudrémont	52
20354	507	\N	Vauxbons	52
20355	509	\N	Vaux-sous-Aubigny	52
20356	510	\N	Vaux-sur-Blaise	52
20357	511	\N	Vaux-sur-Saint-Urbain	52
20358	512	\N	Vecqueville	52
20359	513	\N	Velles	52
20360	514	\N	Verbiesles	52
20361	515	\N	Verseilles-le-Bas	52
20362	516	\N	Verseilles-le-Haut	52
20363	517	\N	Vesaignes-sous-Lafauche	52
20364	518	\N	Vesaignes-sur-Marne	52
20365	519	\N	Vesvres-sous-Chalancey	52
20366	520	\N	Vicq	52
20367	522	\N	Viéville	52
20368	523	\N	Vignes-la-Côte	52
20369	524	\N	Vignory	52
20370	525	\N	Villars-en-Azois	52
20371	526	\N	Villars-Santenoge	52
20372	528	\N	Ville-en-Blaisois	52
20373	529	\N	Villegusien-le-Lac	52
20374	534	\N	Villiers-en-Lieu	52
20375	535	\N	Villiers-le-Sec	52
20376	536	\N	Villiers-lès-Aprey	52
20377	538	\N	Villiers-sur-Suize	52
20378	539	\N	Violot	52
20379	540	\N	Vitry-en-Montagne	52
20380	541	\N	Vitry-lès-Nogent	52
20381	542	\N	Vivey	52
20382	543	\N	Voillecomte	52
20383	544	\N	Voisey	52
20384	545	\N	Voisines	52
20385	546	\N	Voncourt	52
20386	547	\N	Vouécourt	52
20387	548	\N	Vraincourt	52
20388	549	\N	Vroncourt-la-Côte	52
20389	550	\N	Wassy	52
20390	001	\N	Ahuillé	53
20391	002	\N	Alexain	53
20392	003	\N	Ambrières-les-Vallées	53
20393	004	\N	Ampoigné	53
20394	005	\N	Andouillé	53
20395	006	\N	Argenton-Notre-Dame	53
20396	007	\N	Argentré	53
20397	008	\N	Aron	53
20398	009	\N	Arquenay	53
20399	010	\N	Assé-le-Bérenger	53
20400	011	\N	Astillé	53
20401	012	\N	Athée	53
20402	013	\N	Averton	53
20403	014	\N	Azé	53
20404	015	\N	Baconnière	53
20405	016	\N	Bais	53
20406	017	\N	Ballée	53
20407	018	\N	Ballots	53
20408	019	\N	Bannes	53
20409	021	\N	Bazoge-Montpinçon	53
20410	022	\N	Bazouge-de-Chemeré	53
20411	023	\N	Bazouge-des-Alleux	53
20412	025	\N	Bazougers	53
20413	026	\N	Beaulieu-sur-Oudon	53
20414	027	\N	Beaumont-Pied-de-Buf	53
20415	028	\N	Belgeard	53
20416	029	\N	Bierné	53
20417	030	\N	Bignon-du-Maine	53
20418	031	\N	Bigottière	53
20419	032	\N	Blandouet	53
20420	033	\N	Boissière	53
20421	034	\N	Bonchamp-lès-Laval	53
20422	035	\N	Bouchamps-lès-Craon	53
20423	036	\N	Bouère	53
20424	037	\N	Bouessay	53
20425	038	\N	Boulay-les-Ifs	53
20426	039	\N	Bourgneuf-la-Forêt	53
20427	040	\N	Bourgon	53
20428	041	\N	Brains-sur-les-Marches	53
20429	042	\N	Brecé	53
20430	043	\N	Brée	53
20431	045	\N	Brûlatte	53
20432	046	\N	Buret	53
20433	047	\N	Carelles	53
20434	048	\N	Chailland	53
20435	049	\N	Châlons-du-Maine	53
20436	050	\N	Chammes	53
20437	051	\N	Champéon	53
20438	052	\N	Champfrémont	53
20439	053	\N	Champgenéteux	53
20440	054	\N	Changé	53
20441	055	\N	Chantrigné	53
20442	056	\N	Chapelle-Anthenaise	53
20443	057	\N	Chapelle-au-Riboul	53
20444	058	\N	Chapelle-Craonnaise	53
20445	059	\N	Chapelle-Rainsouin	53
20446	061	\N	Charchigné	53
20447	062	\N	Château-Gontier	53
20448	063	\N	Châtelain	53
20449	064	\N	Châtillon-sur-Colmont	53
20450	065	\N	Châtres-la-Forêt	53
20451	066	\N	Chemazé	53
20452	067	\N	Chémeré-le-Roi	53
20453	068	\N	Chérancé	53
20454	069	\N	Chevaigné-du-Maine	53
20455	071	\N	Colombiers-du-Plessis	53
20456	072	\N	Commer	53
20457	073	\N	Congrier	53
20458	074	\N	Contest	53
20459	075	\N	Cosmes	53
20460	076	\N	Cossé-en-Champagne	53
20461	077	\N	Cossé-le-Vivien	53
20462	078	\N	Coudray	53
20463	079	\N	Couesmes-Vaucé	53
20464	080	\N	Couptrain	53
20465	082	\N	Courbeveille	53
20466	083	\N	Courcité	53
20467	084	\N	Craon	53
20468	085	\N	Crennes-sur-Fraubée	53
20469	086	\N	Croixille	53
20470	087	\N	Cropte	53
20471	088	\N	Cuillé	53
20472	089	\N	Daon	53
20473	090	\N	Denazé	53
20474	091	\N	Désertines	53
20475	092	\N	Deux-Évailles	53
20476	093	\N	Dorée	53
20477	094	\N	Entrammes	53
20478	095	\N	Épineux-le-Seguin	53
20479	096	\N	Ernée	53
20480	097	\N	Évron	53
20481	098	\N	Fontaine-Couverte	53
20482	099	\N	Forcé	53
20483	100	\N	Fougerolles-du-Plessis	53
20484	101	\N	Fromentières	53
20485	102	\N	Gastines	53
20486	103	\N	Genest-Saint-Isle	53
20487	104	\N	Gennes-sur-Glaize	53
20488	105	\N	Gesnes	53
20489	106	\N	Gesvres	53
20490	107	\N	Gorron	53
20491	108	\N	Gravelle	53
20492	109	\N	Grazay	53
20493	110	\N	Grez-en-Bouère	53
20494	111	\N	Haie-Traversaine	53
20495	112	\N	Ham	53
20496	113	\N	Hambers	53
20497	114	\N	Hardanges	53
20498	115	\N	Hercé	53
20499	116	\N	Horps	53
20500	117	\N	Houssay	53
20501	118	\N	Housseau-Brétignolles	53
20502	119	\N	Huisserie	53
20503	120	\N	Izé	53
20504	121	\N	Javron-les-Chapelles	53
20505	122	\N	Jublains	53
20506	123	\N	Juvigné	53
20507	124	\N	Laigné	53
20508	125	\N	Landivy	53
20509	126	\N	Larchamp	53
20510	127	\N	Lassay-les-Châteaux	53
20511	128	\N	Laubrières	53
20512	129	\N	Launay-Villiers	53
20513	130	\N	Laval	53
20514	131	\N	Lesbois	53
20515	132	\N	Levaré	53
20516	133	\N	Lignières-Orgères	53
20517	134	\N	Livet	53
20518	135	\N	Livré	53
20519	136	\N	Loigné-sur-Mayenne	53
20520	137	\N	Loiron	53
20521	138	\N	Longuefuye	53
20522	139	\N	Loupfougères	53
20523	140	\N	Louverné	53
20524	141	\N	Louvigné	53
20525	142	\N	Madré	53
20526	143	\N	Maisoncelles-du-Maine	53
20527	144	\N	Marcillé-la-Ville	53
20528	145	\N	Marigné-Peuton	53
20529	146	\N	Martigné-sur-Mayenne	53
20530	147	\N	Mayenne	53
20531	148	\N	Mée	53
20532	150	\N	Ménil	53
20533	151	\N	Méral	53
20534	152	\N	Meslay-du-Maine	53
20535	153	\N	Mézangers	53
20536	154	\N	Montaudin	53
20537	155	\N	Montenay	53
20538	156	\N	Montflours	53
20539	157	\N	Montigné-le-Brillant	53
20540	158	\N	Montjean	53
20541	159	\N	Montourtier	53
20542	160	\N	Montreuil-Poulay	53
20543	161	\N	Montsûrs	53
20544	162	\N	Moulay	53
20545	163	\N	Neau	53
20546	164	\N	Neuilly-le-Vendin	53
20547	165	\N	Niafles	53
20548	168	\N	Nuillé-sur-Vicoin	53
20549	169	\N	Olivet	53
20550	170	\N	Oisseau	53
20551	172	\N	Origné	53
20552	173	\N	Pallu	53
20553	174	\N	Parigné-sur-Braye	53
20554	175	\N	Parné-sur-Roc	53
20555	176	\N	Pas	53
20556	177	\N	Pellerine	53
20557	178	\N	Peuton	53
20558	179	\N	Placé	53
20559	180	\N	Pommerieux	53
20560	181	\N	Pontmain	53
20561	182	\N	Port-Brillet	53
20562	184	\N	Préaux	53
20563	185	\N	Pré-en-Pail	53
20564	186	\N	Quelaines-Saint-Gault	53
20565	187	\N	Ravigny	53
20566	188	\N	Renazé	53
20567	189	\N	Rennes-en-Grenouilles	53
20568	190	\N	Ribay	53
20569	191	\N	Roë	53
20570	192	\N	Rouaudière	53
20571	193	\N	Ruillé-Froid-Fonds	53
20572	194	\N	Ruillé-le-Gravelais	53
20573	195	\N	Sacé	53
20574	196	\N	Saint-Aignan-de-Couptrain	53
20575	197	\N	Saint-Aignan-sur-Roë	53
20576	198	\N	Saint-Aubin-du-Désert	53
20577	199	\N	Saint-Aubin-Fosse-Louvain	53
20578	200	\N	Saint-Baudelle	53
20579	201	\N	Saint-Berthevin	53
20580	202	\N	Saint-Berthevin-la-Tannière	53
20581	203	\N	Saint-Brice	53
20582	204	\N	Saint-Calais-du-Désert	53
20583	205	\N	Saint-Céneré	53
20584	206	\N	Saint-Charles-la-Forêt	53
20585	207	\N	Saint-Christophe-du-Luat	53
20586	208	\N	Saint-Cyr-en-Pail	53
20587	209	\N	Saint-Cyr-le-Gravelais	53
20588	210	\N	Saint-Denis-d'Anjou	53
20589	211	\N	Saint-Denis-de-Gastines	53
20590	212	\N	Saint-Denis-du-Maine	53
20591	213	\N	Saint-Ellier-du-Maine	53
20592	214	\N	Saint-Erblon	53
20593	215	\N	Saint-Fort	53
20594	216	\N	Saint-Fraimbault-de-Prières	53
20595	218	\N	Sainte-Gemmes-le-Robert	53
20596	219	\N	Saint-Georges-Buttavent	53
20597	220	\N	Saint-Georges-le-Fléchard	53
20598	221	\N	Saint-Georges-sur-Erve	53
20599	222	\N	Saint-Germain-d'Anxure	53
20600	223	\N	Saint-Germain-de-Coulamer	53
20601	224	\N	Saint-Germain-le-Fouilloux	53
20602	225	\N	Saint-Germain-le-Guillaume	53
20603	226	\N	Saint-Hilaire-du-Maine	53
20604	228	\N	Saint-Jean-sur-Erve	53
20605	229	\N	Saint-Jean-sur-Mayenne	53
20606	230	\N	Saint-Julien-du-Terroux	53
20607	231	\N	Saint-Laurent-des-Mortiers	53
20608	232	\N	Saint-Léger	53
20609	233	\N	Saint-Loup-du-Dorat	53
20610	234	\N	Saint-Loup-du-Gast	53
20611	235	\N	Sainte-Marie-du-Bois	53
20612	236	\N	Saint-Mars-du-Désert	53
20613	237	\N	Saint-Mars-sur-Colmont	53
20614	238	\N	Saint-Mars-sur-la-Futaie	53
20615	239	\N	Saint-Martin-de-Connée	53
20616	240	\N	Saint-Martin-du-Limet	53
20617	241	\N	Saint-Michel-de-Feins	53
20618	242	\N	Saint-Michel-de-la-Roë	53
20619	243	\N	Saint-Ouën-des-Toits	53
20620	244	\N	Saint-Ouën-des-Vallons	53
20621	245	\N	Saint-Pierre-des-Landes	53
20622	246	\N	Saint-Pierre-des-Nids	53
20623	247	\N	Saint-Pierre-la-Cour	53
20624	248	\N	Saint-Pierre-sur-Erve	53
20625	249	\N	Saint-Pierre-sur-Orthe	53
20626	250	\N	Saint-Poix	53
20627	251	\N	Saint-Quentin-les-Anges	53
20628	252	\N	Saint-Samson	53
20629	253	\N	Saint-Saturnin-du-Limet	53
20630	254	\N	Saint-Sulpice	53
20631	255	\N	Sainte-Suzanne	53
20632	256	\N	Saint-Thomas-de-Courceriers	53
20633	257	\N	Saulges	53
20634	258	\N	Selle-Craonnaise	53
20635	259	\N	Senonnes	53
20636	260	\N	Simplé	53
20637	261	\N	Soucé	53
20638	262	\N	Soulgé-sur-Ouette	53
20639	263	\N	Thubuf	53
20640	264	\N	Thorigné-en-Charnie	53
20641	265	\N	Torcé-Viviers-en-Charnie	53
20642	266	\N	Trans	53
20643	267	\N	Vaiges	53
20644	269	\N	Vautorte	53
20645	270	\N	Vieuvy	53
20646	271	\N	Villaines-la-Juhel	53
20647	272	\N	Villepail	53
20648	273	\N	Villiers-Charlemagne	53
20649	274	\N	Vimarcé	53
20650	276	\N	Voutré	53
20651	001	\N	Abaucourt	54
20652	002	\N	Abbéville-lès-Conflans	54
20653	003	\N	Aboncourt	54
20654	004	\N	Affléville	54
20655	005	\N	Affracourt	54
20656	006	\N	Agincourt	54
20657	007	\N	Aingeray	54
20658	008	\N	Allain	54
20659	009	\N	Allamont	54
20660	010	\N	Allamps	54
20661	011	\N	Allondrelle-la-Malmaison	54
20662	012	\N	Amance	54
20663	013	\N	Amenoncourt	54
20664	014	\N	Ancerviller	54
20665	015	\N	Anderny	54
20666	016	\N	Andilly	54
20667	017	\N	Angomont	54
20668	018	\N	Anoux	54
20669	019	\N	Ansauville	54
20670	020	\N	Anthelupt	54
20671	021	\N	Armaucourt	54
20672	022	\N	Arnaville	54
20673	023	\N	Arracourt	54
20674	024	\N	Arraye-et-Han	54
20675	025	\N	Art-sur-Meurthe	54
20676	026	\N	Athienville	54
20677	027	\N	Atton	54
20678	028	\N	Auboué	54
20679	029	\N	Audun-le-Roman	54
20680	030	\N	Autrepierre	54
20681	031	\N	Autreville-sur-Moselle	54
20682	032	\N	Autrey	54
20683	033	\N	Avillers	54
20684	034	\N	Avrainville	54
20685	035	\N	Avricourt	54
20686	036	\N	Avril	54
20687	037	\N	Azelot	54
20688	038	\N	Azerailles	54
20689	039	\N	Baccarat	54
20690	040	\N	Badonviller	54
20691	041	\N	Bagneux	54
20692	042	\N	Bainville-aux-Miroirs	54
20693	043	\N	Bainville-sur-Madon	54
20694	044	\N	Barbas	54
20695	045	\N	Barbonville	54
20696	046	\N	Barisey-au-Plain	54
20697	047	\N	Barisey-la-Côte	54
20698	048	\N	Baroches	54
20699	049	\N	Baslieux	54
20700	050	\N	Bathelémont-lès-Bauzemont	54
20701	051	\N	Batilly	54
20702	052	\N	Battigny	54
20703	053	\N	Bauzemont	54
20704	054	\N	Bayon	54
20705	055	\N	Bayonville-sur-Mad	54
20706	056	\N	Bazailles	54
20707	057	\N	Beaumont	54
20708	058	\N	Béchamps	54
20709	059	\N	Belleau	54
20710	060	\N	Belleville	54
20711	061	\N	Bénaménil	54
20712	062	\N	Benney	54
20713	063	\N	Bernécourt	54
20714	064	\N	Bertrambois	54
20715	065	\N	Bertrichamps	54
20716	066	\N	Bettainvillers	54
20717	067	\N	Beuveille	54
20718	068	\N	Beuvezin	54
20719	069	\N	Beuvillers	54
20720	070	\N	Bey-sur-Seille	54
20721	071	\N	Bezange-la-Grande	54
20722	072	\N	Bezaumont	54
20723	073	\N	Bicqueley	54
20724	074	\N	Bienville-la-Petite	54
20725	075	\N	Bionville	54
20726	076	\N	Blainville-sur-l'Eau	54
20727	077	\N	Blâmont	54
20728	078	\N	Blémerey	54
20729	079	\N	Blénod-lès-Pont-à-Mousson	54
20730	080	\N	Blénod-lès-Toul	54
20731	081	\N	Boismont	54
20732	082	\N	Boncourt	54
20733	083	\N	Bonviller	54
20734	084	\N	Mont-Bonvillers	54
20735	085	\N	Borville	54
20736	086	\N	Boucq	54
20737	087	\N	Bouillonville	54
20738	088	\N	Bouvron	54
20739	089	\N	Bouxières-aux-Chênes	54
20740	090	\N	Bouxières-aux-Dames	54
20741	091	\N	Bouxières-sous-Froidmont	54
20742	092	\N	Bouzanville	54
20743	093	\N	Brainville	54
20744	094	\N	Bralleville	54
20745	095	\N	Bratte	54
20746	096	\N	Bréhain-la-Ville	54
20747	097	\N	Bréménil	54
20748	098	\N	Brémoncourt	54
20749	099	\N	Briey	54
20750	100	\N	Brin-sur-Seille	54
20751	101	\N	Brouville	54
20752	102	\N	Bruley	54
20753	103	\N	Bruville	54
20754	104	\N	Buissoncourt	54
20755	105	\N	Bulligny	54
20756	106	\N	Bures	54
20757	107	\N	Buriville	54
20758	108	\N	Burthecourt-aux-Chênes	54
20759	109	\N	Ceintrey	54
20760	110	\N	Cerville	54
20761	111	\N	Chaligny	54
20762	112	\N	Chambley-Bussières	54
20763	113	\N	Champenoux	54
20764	114	\N	Champey-sur-Moselle	54
20765	115	\N	Champigneulles	54
20766	116	\N	Chanteheux	54
20767	117	\N	Chaouilley	54
20768	118	\N	Charency-Vezin	54
20769	119	\N	Charey	54
20770	120	\N	Charmes-la-Côte	54
20771	121	\N	Charmois	54
20772	122	\N	Chaudeney-sur-Moselle	54
20773	123	\N	Chavigny	54
20774	124	\N	Chazelles-sur-Albe	54
20775	125	\N	Chenevières	54
20776	126	\N	Chenicourt	54
20777	127	\N	Chenières	54
20778	128	\N	Choloy-Ménillot	54
20779	129	\N	Cirey-sur-Vezouze	54
20780	130	\N	Clayeures	54
20781	131	\N	Clémery	54
20782	132	\N	Clérey-sur-Brenon	54
20783	133	\N	Coincourt	54
20784	134	\N	Colmey	54
20785	135	\N	Colombey-les-Belles	54
20786	136	\N	Conflans-en-Jarnisy	54
20787	137	\N	Cons-la-Grandville	54
20788	138	\N	Cosnes-et-Romain	54
20789	139	\N	Courbesseaux	54
20790	140	\N	Courcelles	54
20791	141	\N	Coyviller	54
20792	142	\N	Crantenoy	54
20793	143	\N	Crépey	54
20794	144	\N	Crévéchamps	54
20795	145	\N	Crévic	54
20796	146	\N	Crézilles	54
20797	147	\N	Crion	54
20798	148	\N	Croismare	54
20799	149	\N	Crusnes	54
20800	150	\N	Custines	54
20801	151	\N	Cutry	54
20802	152	\N	Damelevières	54
20803	153	\N	Dampvitoux	54
20804	154	\N	Deneuvre	54
20805	155	\N	Deuxville	54
20806	156	\N	Diarville	54
20807	157	\N	Dieulouard	54
20808	158	\N	Dolcourt	54
20809	159	\N	Dombasle-sur-Meurthe	54
20810	160	\N	Domèvre-en-Haye	54
20811	161	\N	Domèvre-sur-Vezouze	54
20812	162	\N	Domgermain	54
20813	163	\N	Domjevin	54
20814	164	\N	Dommarie-Eulmont	54
20815	165	\N	Dommartemont	54
20816	166	\N	Dommartin-la-Chaussée	54
20817	167	\N	Dommartin-lès-Toul	54
20818	168	\N	Dommartin-sous-Amance	54
20819	169	\N	Domprix	54
20820	170	\N	Domptail-en-l'Air	54
20821	171	\N	Doncourt-lès-Conflans	54
20822	172	\N	Doncourt-lès-Longuyon	54
20823	173	\N	Drouville	54
20824	174	\N	Écrouves	54
20825	175	\N	Einvaux	54
20826	176	\N	Einville-au-Jard	54
20827	177	\N	Emberménil	54
20828	178	\N	Épiez-sur-Chiers	54
20829	179	\N	Éply	54
20830	180	\N	Erbéviller-sur-Amezule	54
20831	181	\N	Errouville	54
20832	182	\N	Essey-et-Maizerais	54
20833	183	\N	Essey-la-Côte	54
20834	184	\N	Essey-lès-Nancy	54
20835	185	\N	Étreval	54
20836	186	\N	Eulmont	54
20837	187	\N	Euvezin	54
20838	188	\N	Faulx	54
20839	189	\N	Favières	54
20840	190	\N	Fécocourt	54
20841	191	\N	Fenneviller	54
20842	192	\N	Ferrières	54
20843	193	\N	Fey-en-Haye	54
20844	194	\N	Fillières	54
20845	195	\N	Flainval	54
20846	196	\N	Flavigny-sur-Moselle	54
20847	197	\N	Fléville-devant-Nancy	54
20848	198	\N	Fléville-Lixières	54
20849	199	\N	Flin	54
20850	200	\N	Flirey	54
20851	201	\N	Fontenoy-la-Joûte	54
20852	202	\N	Fontenoy-sur-Moselle	54
20853	203	\N	Forcelles-Saint-Gorgon	54
20854	204	\N	Forcelles-sous-Gugney	54
20855	205	\N	Foug	54
20856	206	\N	Fraimbois	54
20857	207	\N	Fraisnes-en-Saintois	54
20858	208	\N	Francheville	54
20859	209	\N	Franconville	54
20860	210	\N	Fréménil	54
20861	211	\N	Frémonville	54
20862	212	\N	Fresnois-la-Montagne	54
20863	213	\N	Friauville	54
20864	214	\N	Frolois	54
20865	215	\N	Frouard	54
20866	216	\N	Froville	54
20867	217	\N	Gélacourt	54
20868	218	\N	Gélaucourt	54
20869	219	\N	Gellenoncourt	54
20870	220	\N	Gémonville	54
20871	221	\N	Gerbécourt-et-Haplemont	54
20872	222	\N	Gerbéviller	54
20873	223	\N	Germiny	54
20874	224	\N	Germonville	54
20875	225	\N	Gézoncourt	54
20876	226	\N	Gibeaumeix	54
20877	227	\N	Giraumont	54
20878	228	\N	Giriviller	54
20879	229	\N	Glonville	54
20880	230	\N	Gogney	54
20881	231	\N	Gondrecourt-Aix	54
20882	232	\N	Gondreville	54
20883	233	\N	Gondrexon	54
20884	234	\N	Gorcy	54
20885	235	\N	Goviller	54
20886	236	\N	Grand-Failly	54
20887	237	\N	Grimonviller	54
20888	238	\N	Gripport	54
20889	239	\N	Griscourt	54
20890	240	\N	Grosrouvres	54
20891	241	\N	Gugney	54
20892	242	\N	Gye	54
20893	243	\N	Hablainville	54
20894	244	\N	Hagéville	54
20895	245	\N	Haigneville	54
20896	246	\N	Halloville	54
20897	247	\N	Hammeville	54
20898	248	\N	Hamonville	54
20899	249	\N	Hannonville-Suzémont	54
20900	250	\N	Haraucourt	54
20901	251	\N	Harbouey	54
20902	252	\N	Haroué	54
20903	253	\N	Hatrize	54
20904	254	\N	Haucourt-Moulaine	54
20905	255	\N	Haudonville	54
20906	256	\N	Haussonville	54
20907	257	\N	Heillecourt	54
20908	258	\N	Hénaménil	54
20909	259	\N	Herbéviller	54
20910	260	\N	Hériménil	54
20911	261	\N	Herserange	54
20912	262	\N	Hoéville	54
20913	263	\N	Homécourt	54
20914	264	\N	Houdelmont	54
20915	265	\N	Houdemont	54
20916	266	\N	Houdreville	54
20917	268	\N	Housséville	54
20918	269	\N	Hudiviller	54
20919	270	\N	Hussigny-Godbrange	54
20920	271	\N	Igney	54
20921	272	\N	Jaillon	54
20922	273	\N	Jarny	54
20923	274	\N	Jarville-la-Malgrange	54
20924	275	\N	Jaulny	54
20925	276	\N	Jeandelaincourt	54
20926	277	\N	Jeandelize	54
20927	278	\N	Jevoncourt	54
20928	279	\N	Jezainville	54
20929	280	\N	Juf	54
20930	281	\N	Jolivet	54
20931	282	\N	Joppécourt	54
20932	283	\N	Jouaville	54
20933	284	\N	Joudreville	54
20934	285	\N	Juvrecourt	54
20935	286	\N	Labry	54
20936	287	\N	Lachapelle	54
20937	288	\N	Lagney	54
20938	289	\N	Laître-sous-Amance	54
20939	290	\N	Laix	54
20940	291	\N	Laluf	54
20941	292	\N	Lamath	54
20942	293	\N	Landécourt	54
20943	294	\N	Landremont	54
20944	295	\N	Landres	54
20945	296	\N	Laneuvelotte	54
20946	297	\N	Laneuveville-aux-Bois	54
20947	298	\N	Laneuveville-derrière-Foug	54
20948	299	\N	Laneuveville-devant-Bayon	54
20949	300	\N	Laneuveville-devant-Nancy	54
20950	301	\N	Lanfroicourt	54
20951	302	\N	Lantéfontaine	54
20952	303	\N	Laronxe	54
20953	304	\N	Laxou	54
20954	305	\N	Lay-Saint-Christophe	54
20955	306	\N	Lay-Saint-Remy	54
20956	307	\N	Lebeuville	54
20957	308	\N	Leintrey	54
20958	309	\N	Lemainville	54
20959	310	\N	Leménil-Mitry	54
20960	311	\N	Lenoncourt	54
20961	312	\N	Lesménils	54
20962	313	\N	Létricourt	54
20963	314	\N	Lexy	54
20964	315	\N	Leyr	54
20965	316	\N	Limey-Remenauville	54
20966	317	\N	Lironville	54
20967	318	\N	Liverdun	54
20968	320	\N	Loisy	54
20969	321	\N	Longlaville	54
20970	322	\N	Longuyon	54
20971	323	\N	Longwy	54
20972	324	\N	Lorey	54
20973	325	\N	Loromontzey	54
20974	326	\N	Lubey	54
20975	327	\N	Lucey	54
20976	328	\N	Ludres	54
20977	329	\N	Lunéville	54
20978	330	\N	Lupcourt	54
20979	331	\N	Magnières	54
20980	332	\N	Maidières	54
20981	333	\N	Mailly-sur-Seille	54
20982	334	\N	Mairy-Mainville	54
20983	335	\N	Maixe	54
20984	336	\N	Maizières	54
20985	337	\N	Malavillers	54
20986	338	\N	Malleloy	54
20987	339	\N	Malzéville	54
20988	340	\N	Mamey	54
20989	341	\N	Mance	54
20990	342	\N	Mancieulles	54
20991	343	\N	Mandres-aux-Quatre-Tours	54
20992	344	\N	Mangonville	54
20993	345	\N	Manoncourt-en-Vermois	54
20994	346	\N	Manoncourt-en-Woëvre	54
20995	348	\N	Manonville	54
20996	349	\N	Manonviller	54
20997	350	\N	Marainviller	54
20998	351	\N	Marbache	54
20999	352	\N	Maron	54
21000	353	\N	Mars-la-Tour	54
21001	354	\N	Marthemont	54
21002	355	\N	Martincourt	54
21003	356	\N	Mattexey	54
21004	357	\N	Maxéville	54
21005	358	\N	Mazerulles	54
21006	359	\N	Méhoncourt	54
21007	360	\N	Ménil-la-Tour	54
21008	362	\N	Mercy-le-Bas	54
21009	363	\N	Mercy-le-Haut	54
21010	364	\N	Méréville	54
21011	365	\N	Merviller	54
21012	366	\N	Messein	54
21013	367	\N	Mexy	54
21014	368	\N	Mignéville	54
21015	369	\N	Millery	54
21016	370	\N	Minorville	54
21017	371	\N	Moineville	54
21018	372	\N	Moivrons	54
21019	373	\N	Moncel-lès-Lunéville	54
21020	374	\N	Moncel-sur-Seille	54
21021	375	\N	Montauville	54
21022	376	\N	Montenoy	54
21023	377	\N	Montigny	54
21024	378	\N	Montigny-sur-Chiers	54
21025	379	\N	Mont-l'Étroit	54
21026	380	\N	Mont-le-Vignoble	54
21027	381	\N	Montreux	54
21028	382	\N	Mont-Saint-Martin	54
21029	383	\N	Mont-sur-Meurthe	54
21030	385	\N	Morfontaine	54
21031	386	\N	Moriviller	54
21032	387	\N	Morville-sur-Seille	54
21033	388	\N	Mouacourt	54
21034	389	\N	Mouaville	54
21035	390	\N	Mousson	54
21036	391	\N	Moutiers	54
21037	392	\N	Moutrot	54
21038	393	\N	Moyen	54
21039	394	\N	Murville	54
21040	395	\N	Nancy	54
21041	396	\N	Neufmaisons	54
21042	397	\N	Neuves-Maisons	54
21043	398	\N	Neuviller-lès-Badonviller	54
21044	399	\N	Neuviller-sur-Moselle	54
21045	400	\N	Nomeny	54
21046	401	\N	Nonhigny	54
21047	402	\N	Norroy-le-Sec	54
21048	403	\N	Norroy-lès-Pont-à-Mousson	54
21049	404	\N	Noviant-aux-Prés	54
21050	405	\N	Ochey	54
21051	406	\N	Ogéviller	54
21052	407	\N	Ognéville	54
21053	408	\N	Olley	54
21054	409	\N	Omelmont	54
21055	410	\N	Onville	54
21056	411	\N	Ormes-et-Ville	54
21057	412	\N	Othe	54
21058	413	\N	Ozerailles	54
21059	414	\N	Pagney-derrière-Barine	54
21060	415	\N	Pagny-sur-Moselle	54
21061	416	\N	Pannes	54
21062	417	\N	Parey-Saint-Césaire	54
21063	418	\N	Parroy	54
21064	419	\N	Parux	54
21065	420	\N	Petit-Failly	54
21066	421	\N	Petitmont	54
21067	422	\N	Pettonville	54
21068	423	\N	Pexonne	54
21069	424	\N	Phlin	54
21070	425	\N	Piennes	54
21071	426	\N	Pierre-la-Treiche	54
21072	427	\N	Pierre-Percée	54
21073	428	\N	Pierrepont	54
21074	429	\N	Pierreville	54
21075	430	\N	Pompey	54
21076	431	\N	Pont-à-Mousson	54
21077	432	\N	Pont-Saint-Vincent	54
21078	433	\N	Port-sur-Seille	54
21079	434	\N	Praye	54
21080	435	\N	Prény	54
21081	436	\N	Preutin-Higny	54
21082	437	\N	Pulligny	54
21083	438	\N	Pulney	54
21084	439	\N	Pulnoy	54
21085	440	\N	Puxe	54
21086	441	\N	Puxieux	54
21087	442	\N	Quevilloncourt	54
21088	443	\N	Raon-lès-Leau	54
21089	444	\N	Raucourt	54
21090	445	\N	Raville-sur-Sânon	54
21091	446	\N	Réchicourt-la-Petite	54
21092	447	\N	Réclonville	54
21093	449	\N	Rehainviller	54
21094	450	\N	Reherrey	54
21095	451	\N	Réhon	54
21096	452	\N	Reillon	54
21097	453	\N	Rembercourt-sur-Mad	54
21098	455	\N	Remenoville	54
21099	456	\N	Réméréville	54
21100	457	\N	Remoncourt	54
21101	458	\N	Repaix	54
21102	459	\N	Richardménil	54
21103	460	\N	Rogéville	54
21104	461	\N	Romain	54
21105	462	\N	Rosières-aux-Salines	54
21106	463	\N	Rosières-en-Haye	54
21107	464	\N	Rouves	54
21108	465	\N	Roville-devant-Bayon	54
21109	466	\N	Royaumeix	54
21110	467	\N	Rozelieures	54
21111	468	\N	Saffais	54
21112	469	\N	Saint-Ail	54
21113	470	\N	Saint-Baussant	54
21114	471	\N	Saint-Boingt	54
21115	472	\N	Saint-Clément	54
21116	473	\N	Saint-Firmin	54
21117	474	\N	Sainte-Geneviève	54
21118	475	\N	Saint-Germain	54
21119	476	\N	Saint-Jean-lès-Longuyon	54
21120	477	\N	Saint-Julien-lès-Gorze	54
21121	478	\N	Saint-Marcel	54
21122	479	\N	Saint-Mard	54
21123	480	\N	Saint-Martin	54
21124	481	\N	Saint-Maurice-aux-Forges	54
21125	482	\N	Saint-Max	54
21126	483	\N	Saint-Nicolas-de-Port	54
21127	484	\N	Sainte-Pôle	54
21128	485	\N	Saint-Pancré	54
21129	486	\N	Saint-Remimont	54
21130	487	\N	Saint-Rémy-aux-Bois	54
21131	488	\N	Saint-Sauveur	54
21132	489	\N	Saint-Supplet	54
21133	490	\N	Saizerais	54
21134	491	\N	Sancy	54
21135	492	\N	Sanzey	54
21136	493	\N	Saulnes	54
21137	494	\N	Saulxerotte	54
21138	495	\N	Saulxures-lès-Nancy	54
21139	496	\N	Saulxures-lès-Vannes	54
21140	497	\N	Saxon-Sion	54
21141	498	\N	Seichamps	54
21142	499	\N	Seicheprey	54
21143	500	\N	Selaincourt	54
21144	501	\N	Seranville	54
21145	502	\N	Serres	54
21146	504	\N	Serrouville	54
21147	505	\N	Sexey-aux-Forges	54
21148	506	\N	Sexey-les-Bois	54
21149	507	\N	Sionviller	54
21150	508	\N	Sivry	54
21151	509	\N	Sommerviller	54
21152	510	\N	Sornéville	54
21153	511	\N	Sponville	54
21154	512	\N	Tanconville	54
21155	513	\N	Tantonville	54
21156	514	\N	Tellancourt	54
21157	515	\N	Thélod	54
21158	516	\N	They-sous-Vaudemont	54
21159	517	\N	Thézey-Saint-Martin	54
21160	518	\N	Thiaucourt-Regniéville	54
21161	519	\N	Thiaville-sur-Meurthe	54
21162	520	\N	Thiébauménil	54
21163	521	\N	Thil	54
21164	522	\N	Thorey-Lyautey	54
21165	523	\N	Thuilley-aux-Groseilles	54
21166	524	\N	Thumeréville	54
21167	525	\N	Tiercelet	54
21168	526	\N	Tomblaine	54
21169	527	\N	Tonnoy	54
21170	528	\N	Toul	54
21171	529	\N	Tramont-Émy	54
21172	530	\N	Tramont-Lassus	54
21173	531	\N	Tramont-Saint-André	54
21174	532	\N	Tremblecourt	54
21175	533	\N	Trieux	54
21176	534	\N	Trondes	54
21177	535	\N	Tronville	54
21178	536	\N	Tucquegnieux	54
21179	537	\N	Ugny	54
21180	538	\N	Uruffe	54
21181	539	\N	Vacqueville	54
21182	540	\N	Val-et-Châtillon	54
21183	541	\N	Valhey	54
21184	542	\N	Valleroy	54
21185	543	\N	Vallois	54
21186	544	\N	Vandelainville	54
21187	545	\N	Vandeléville	54
21188	546	\N	Vandières	54
21189	547	\N	Vanduvre-lès-Nancy	54
21190	548	\N	Vannes-le-Châtel	54
21191	549	\N	Varangéville	54
21192	550	\N	Vathiménil	54
21193	551	\N	Vaucourt	54
21194	552	\N	Vaudémont	54
21195	553	\N	Vaudeville	54
21196	554	\N	Vaudigny	54
21197	555	\N	Vaxainville	54
21198	556	\N	Vého	54
21199	557	\N	Velaine-en-Haye	54
21200	558	\N	Velaine-sous-Amance	54
21201	559	\N	Velle-sur-Moselle	54
21202	560	\N	Veney	54
21203	561	\N	Vennezey	54
21204	562	\N	Verdenal	54
21205	563	\N	Vézelise	54
21206	564	\N	Viéville-en-Haye	54
21207	565	\N	Vigneulles	54
21208	566	\N	Vilcey-sur-Trey	54
21209	567	\N	Villacourt	54
21210	568	\N	Ville-au-Montois	54
21211	569	\N	Ville-au-Val	54
21212	570	\N	Villecey-sur-Mad	54
21213	571	\N	Ville-en-Vermois	54
21214	572	\N	Ville-Houdlémont	54
21215	573	\N	Villers-en-Haye	54
21216	574	\N	Villers-la-Chèvre	54
21217	575	\N	Villers-la-Montagne	54
21218	576	\N	Villers-le-Rond	54
21219	577	\N	Villers-lès-Moivrons	54
21220	578	\N	Villers-lès-Nancy	54
21221	579	\N	Villers-sous-Prény	54
21222	580	\N	Villerupt	54
21223	581	\N	Ville-sur-Yron	54
21224	582	\N	Villette	54
21225	583	\N	Villey-le-Sec	54
21226	584	\N	Villey-Saint-Étienne	54
21227	585	\N	Virecourt	54
21228	586	\N	Viterne	54
21229	587	\N	Vitrey	54
21230	588	\N	Vitrimont	54
21231	589	\N	Vittonville	54
21232	590	\N	Viviers-sur-Chiers	54
21233	591	\N	Voinémont	54
21234	592	\N	Vroncourt	54
21235	593	\N	Waville	54
21236	594	\N	Xammes	54
21237	595	\N	Xermaménil	54
21238	596	\N	Xeuilley	54
21239	597	\N	Xirocourt	54
21240	598	\N	Xivry-Circourt	54
21241	599	\N	Xonville	54
21242	600	\N	Xousse	54
21243	601	\N	Xures	54
21244	602	\N	Han-devant-Pierrepont	54
21245	001	\N	Abainville	55
21246	002	\N	Abaucourt-Hautecourt	55
21247	004	\N	Aincreville	55
21248	005	\N	Amanty	55
21249	007	\N	Ambly-sur-Meuse	55
21250	008	\N	Amel-sur-l'Étang	55
21251	009	\N	Ancemont	55
21252	010	\N	Ancerville	55
21253	011	\N	Andernay	55
21254	012	\N	Apremont-la-Forêt	55
21255	013	\N	Arrancy-sur-Crusne	55
21256	014	\N	Aubréville	55
21257	015	\N	Aulnois-en-Perthois	55
21258	017	\N	Autrécourt-sur-Aire	55
21259	018	\N	Autréville-Saint-Lambert	55
21260	021	\N	Avillers-Sainte-Croix	55
21261	022	\N	Avioth	55
21262	023	\N	Avocourt	55
21263	024	\N	Azannes-et-Soumazannes	55
21264	025	\N	Baâlon	55
21265	026	\N	Badonvilliers-Gérauvilliers	55
21266	027	\N	Bannoncourt	55
21267	028	\N	Bantheville	55
21268	029	\N	Bar-le-Duc	55
21269	030	\N	Baudignécourt	55
21270	031	\N	Baudonvilliers	55
21271	032	\N	Baudrémont	55
21272	033	\N	Baulny	55
21273	034	\N	Bazeilles-sur-Othain	55
21274	035	\N	Bazincourt-sur-Saulx	55
21275	036	\N	Beauclair	55
21276	037	\N	Beaufort-en-Argonne	55
21277	038	\N	Beaulieu-en-Argonne	55
21278	039	\N	Beaumont-en-Verdunois	55
21279	040	\N	Beausite	55
21280	041	\N	Behonne	55
21281	042	\N	Belleray	55
21282	043	\N	Belleville-sur-Meuse	55
21283	044	\N	Belrain	55
21284	045	\N	Belrupt-en-Verdunois	55
21285	046	\N	Beney-en-Woëvre	55
21286	047	\N	Béthelainville	55
21287	048	\N	Béthincourt	55
21288	049	\N	Beurey-sur-Saulx	55
21289	050	\N	Bezonvaux	55
21290	051	\N	Biencourt-sur-Orge	55
21291	053	\N	Billy-sous-Mangiennes	55
21292	054	\N	Bislée	55
21293	055	\N	Blanzée	55
21294	057	\N	Boinville-en-Woëvre	55
21295	058	\N	Boncourt-sur-Meuse	55
21296	059	\N	Bonnet	55
21297	060	\N	Bonzée	55
21298	061	\N	Bouchon-sur-Saulx	55
21299	062	\N	Bouconville-sur-Madt	55
21300	063	\N	Bouligny	55
21301	064	\N	Bouquemont	55
21302	065	\N	Boureuilles	55
21303	066	\N	Bovée-sur-Barboure	55
21304	067	\N	Boviolles	55
21305	068	\N	Brabant-en-Argonne	55
21306	069	\N	Brabant-le-Roi	55
21307	070	\N	Brabant-sur-Meuse	55
21308	071	\N	Brandeville	55
21309	072	\N	Braquis	55
21310	073	\N	Bras-sur-Meuse	55
21311	075	\N	Brauvilliers	55
21312	076	\N	Bréhéville	55
21313	077	\N	Breux	55
21314	078	\N	Brieulles-sur-Meuse	55
21315	079	\N	Brillon-en-Barrois	55
21316	080	\N	Brixey-aux-Chanoines	55
21317	081	\N	Brizeaux	55
21318	082	\N	Brocourt-en-Argonne	55
21319	083	\N	Brouennes	55
21320	084	\N	Broussey-en-Blois	55
21321	085	\N	Broussey-Raulecourt	55
21322	087	\N	Bure	55
21323	088	\N	Burey-en-Vaux	55
21324	089	\N	Burey-la-Côte	55
21325	093	\N	Buxières-sous-les-Côtes	55
21326	094	\N	Buzy-Darmont	55
21327	095	\N	Cesse	55
21328	096	\N	Chaillon	55
21329	097	\N	Chalaines	55
21330	099	\N	Champneuville	55
21331	100	\N	Champougny	55
21332	101	\N	Chardogne	55
21333	102	\N	Charny-sur-Meuse	55
21334	103	\N	Charpentry	55
21335	104	\N	Chassey-Beaupré	55
21336	105	\N	Châtillon-sous-les-Côtes	55
21337	106	\N	Chattancourt	55
21338	107	\N	Chaumont-devant-Damvillers	55
21339	108	\N	Chaumont-sur-Aire	55
21340	109	\N	Chauvency-le-Château	55
21341	110	\N	Chauvency-Saint-Hubert	55
21342	111	\N	Chauvoncourt	55
21343	113	\N	Cheppy	55
21344	114	\N	Chonville-Malaumont	55
21345	115	\N	Cierges-sous-Montfaucon	55
21346	116	\N	Claon	55
21347	117	\N	Clermont-en-Argonne	55
21348	118	\N	Cléry-le-Grand	55
21349	119	\N	Cléry-le-Petit	55
21350	120	\N	Combles-en-Barrois	55
21351	121	\N	Combres-sous-les-Côtes	55
21352	122	\N	Commercy	55
21353	123	\N	Hauts-de-Chée	55
21354	124	\N	Consenvoye	55
21355	125	\N	Contrisson	55
21356	127	\N	Courcelles-en-Barrois	55
21357	128	\N	Courcelles-sur-Aire	55
21358	129	\N	Courouvre	55
21359	132	\N	Cousances-les-Forges	55
21360	133	\N	Couvertpuis	55
21361	134	\N	Couvonges	55
21362	137	\N	Cuisy	55
21363	139	\N	Cumières-le-Mort-Homme	55
21364	140	\N	Cunel	55
21365	141	\N	Dagonville	55
21366	142	\N	Dainville-Bertheléville	55
21367	143	\N	Damloup	55
21368	144	\N	Dammarie-sur-Saulx	55
21369	145	\N	Damvillers	55
21370	146	\N	Dannevoux	55
21371	148	\N	Delouze-Rosières	55
21372	149	\N	Delut	55
21373	150	\N	Demange-aux-Eaux	55
21374	153	\N	Dieppe-sous-Douaumont	55
21375	154	\N	Dieue-sur-Meuse	55
21376	155	\N	Dombasle-en-Argonne	55
21377	156	\N	Dombras	55
21378	157	\N	Dommartin-la-Montagne	55
21379	158	\N	Dommary-Baroncourt	55
21380	159	\N	Dompcevrin	55
21381	160	\N	Dompierre-aux-Bois	55
21382	162	\N	Domremy-la-Canne	55
21383	163	\N	Doncourt-aux-Templiers	55
21384	164	\N	Douaumont	55
21385	165	\N	Doulcon	55
21386	166	\N	Dugny-sur-Meuse	55
21387	167	\N	Dun-sur-Meuse	55
21388	168	\N	Duzey	55
21389	169	\N	Écouviez	55
21390	170	\N	Écurey-en-Verdunois	55
21391	171	\N	Eix	55
21392	172	\N	Éparges	55
21393	173	\N	Épiez-sur-Meuse	55
21394	174	\N	Épinonville	55
21395	175	\N	Érize-la-Brûlée	55
21396	177	\N	Érize-la-Petite	55
21397	178	\N	Érize-Saint-Dizier	55
21398	179	\N	Erneville-aux-Bois	55
21399	180	\N	Esnes-en-Argonne	55
21400	181	\N	Étain	55
21401	182	\N	Éton	55
21402	183	\N	Étraye	55
21403	184	\N	Euville	55
21404	185	\N	Èvres	55
21405	186	\N	Fains-Véel	55
21406	188	\N	Flassigny	55
21407	189	\N	Fleury-devant-Douaumont	55
21408	191	\N	Foameix-Ornel	55
21409	192	\N	Fontaines-Saint-Clair	55
21410	193	\N	Forges-sur-Meuse	55
21411	194	\N	Foucaucourt-sur-Thabas	55
21412	195	\N	Fouchères-aux-Bois	55
21413	196	\N	Frémeréville-sous-les-Côtes	55
21414	197	\N	Fresnes-au-Mont	55
21415	198	\N	Fresnes-en-Woëvre	55
21416	199	\N	Froidos	55
21417	200	\N	Fromeréville-les-Vallons	55
21418	201	\N	Fromezey	55
21419	202	\N	Futeau	55
21420	204	\N	Génicourt-sur-Meuse	55
21421	206	\N	Gercourt-et-Drillancourt	55
21422	207	\N	Géry	55
21423	208	\N	Gesnes-en-Argonne	55
21424	210	\N	Gimécourt	55
21425	211	\N	Gincrey	55
21426	212	\N	Girauvoisin	55
21427	214	\N	Givrauval	55
21428	215	\N	Gondrecourt-le-Château	55
21429	216	\N	Gouraincourt	55
21430	217	\N	Goussaincourt	55
21431	218	\N	Gremilly	55
21432	219	\N	Grimaucourt-en-Woëvre	55
21433	220	\N	Grimaucourt-près-Sampigny	55
21434	221	\N	Guerpont	55
21435	222	\N	Gussainville	55
21436	224	\N	Haironville	55
21437	225	\N	Halles-sous-les-Côtes	55
21438	226	\N	Han-lès-Juvigny	55
21439	228	\N	Hannonville-sous-les-Côtes	55
21440	229	\N	Han-sur-Meuse	55
21441	232	\N	Harville	55
21442	236	\N	Haudainville	55
21443	237	\N	Haudiomont	55
21444	239	\N	Haumont-près-Samogneux	55
21445	241	\N	Heippes	55
21446	242	\N	Hennemont	55
21447	243	\N	Herbeuville	55
21448	244	\N	Herméville-en-Woëvre	55
21449	245	\N	Heudicourt-sous-les-Côtes	55
21450	246	\N	Hévilliers	55
21451	247	\N	Horville-en-Ornois	55
21452	248	\N	Houdelaincourt	55
21453	250	\N	Inor	55
21454	251	\N	Ippécourt	55
21455	252	\N	Iré-le-Sec	55
21456	253	\N	Islettes	55
21457	254	\N	Trois-Domaines	55
21458	255	\N	Jametz	55
21459	256	\N	Jonville-en-Woëvre	55
21460	257	\N	Jouy-en-Argonne	55
21461	258	\N	Geville	55
21462	260	\N	Julvécourt	55
21463	261	\N	Juvigny-en-Perthois	55
21464	262	\N	Juvigny-sur-Loison	55
21465	263	\N	Kur-la-Grande	55
21466	264	\N	Kur-la-Petite	55
21467	265	\N	Labeuville	55
21468	266	\N	Lachalade	55
21469	267	\N	Lachaussée	55
21470	268	\N	Lacroix-sur-Meuse	55
21471	269	\N	Lahaymeix	55
21472	270	\N	Lahayville	55
21473	271	\N	Laheycourt	55
21474	272	\N	Laimont	55
21475	274	\N	Lamorville	55
21476	275	\N	Lamouilly	55
21477	276	\N	Landrecourt-Lempire	55
21478	278	\N	Laneuville-au-Rupt	55
21479	279	\N	Laneuville-sur-Meuse	55
21480	280	\N	Lanhères	55
21481	281	\N	Latour-en-Woëvre	55
21482	282	\N	Lavallée	55
21483	284	\N	Lavincourt	55
21484	285	\N	Lavoye	55
21485	286	\N	Lemmes	55
21486	288	\N	Lérouville	55
21487	289	\N	Levoncourt	55
21488	290	\N	Lignières-sur-Aire	55
21489	291	\N	Ligny-en-Barrois	55
21490	292	\N	Liny-devant-Dun	55
21491	293	\N	Lion-devant-Dun	55
21492	295	\N	Lisle-en-Barrois	55
21493	296	\N	Lisle-en-Rigault	55
21494	297	\N	Lissey	55
21495	298	\N	Loisey-Culey	55
21496	299	\N	Loison	55
21497	300	\N	Longeaux	55
21498	301	\N	Longchamps-sur-Aire	55
21499	302	\N	Longeville-en-Barrois	55
21500	303	\N	Loupmont	55
21501	304	\N	Louppy-le-Château	55
21502	306	\N	Louppy-sur-Loison	55
21503	307	\N	Louvemont-Côte-du-Poivre	55
21504	310	\N	Luzy-Saint-Martin	55
21505	311	\N	Maizeray	55
21506	312	\N	Maizey	55
21507	313	\N	Malancourt	55
21508	315	\N	Mandres-en-Barrois	55
21509	316	\N	Mangiennes	55
21510	317	\N	Manheulles	55
21511	320	\N	Marchéville-en-Woëvre	55
21512	321	\N	Marre	55
21513	322	\N	Marson-sur-Barboure	55
21514	323	\N	Martincourt-sur-Meuse	55
21515	324	\N	Marville	55
21516	325	\N	Maucourt-sur-Orne	55
21517	326	\N	Maulan	55
21518	327	\N	Mauvages	55
21519	328	\N	Maxey-sur-Vaise	55
21520	329	\N	Mécrin	55
21521	330	\N	Méligny-le-Grand	55
21522	331	\N	Méligny-le-Petit	55
21523	332	\N	Menaucourt	55
21524	333	\N	Ménil-aux-Bois	55
21525	334	\N	Ménil-la-Horgne	55
21526	335	\N	Ménil-sur-Saulx	55
21527	336	\N	Merles-sur-Loison	55
21528	338	\N	Milly-sur-Bradon	55
21529	339	\N	Mogeville	55
21530	340	\N	Mognéville	55
21531	341	\N	Moirey-Flabas-Crépion	55
21532	343	\N	Montblainville	55
21533	344	\N	Montbras	55
21534	345	\N	Mont-devant-Sassey	55
21535	346	\N	Montfaucon-d'Argonne	55
21536	347	\N	Monthairons	55
21537	348	\N	Montiers-sur-Saulx	55
21538	349	\N	Montigny-devant-Sassey	55
21539	350	\N	Montigny-lès-Vaucouleurs	55
21540	351	\N	Montmédy	55
21541	352	\N	Montplonne	55
21542	353	\N	Montsec	55
21543	355	\N	Montzéville	55
21544	356	\N	Moranville	55
21545	357	\N	Morgemoulin	55
21546	358	\N	Chanteraine	55
21547	359	\N	Morley	55
21548	360	\N	Mouilly	55
21549	361	\N	Moulainville	55
21550	362	\N	Moulins-Saint-Hubert	55
21551	363	\N	Moulotte	55
21552	364	\N	Mouzay	55
21553	365	\N	Murvaux	55
21554	366	\N	Val-d'Ornain	55
21555	367	\N	Muzeray	55
21556	368	\N	Naives-en-Blois	55
21557	369	\N	Naives-Rosières	55
21558	370	\N	Naix-aux-Forges	55
21559	371	\N	Nançois-le-Grand	55
21560	372	\N	Nançois-sur-Ornain	55
21561	373	\N	Nant-le-Grand	55
21562	374	\N	Nant-le-Petit	55
21563	375	\N	Nantillois	55
21564	376	\N	Nantois	55
21565	377	\N	Nepvant	55
21566	378	\N	Nettancourt	55
21567	379	\N	Neufour	55
21568	380	\N	Neuville-en-Verdunois	55
21569	381	\N	Neuville-lès-Vaucouleurs	55
21570	382	\N	Neuville-sur-Ornain	55
21571	383	\N	Neuvilly-en-Argonne	55
21572	384	\N	Nicey-sur-Aire	55
21573	385	\N	Nixéville-Blercourt	55
21574	386	\N	Nonsard-Lamarche	55
21575	387	\N	Nouillonpont	55
21576	388	\N	Noyers-Auzécourt	55
21577	389	\N	Nubécourt	55
21578	391	\N	Olizy-sur-Chiers	55
21579	394	\N	Ornes	55
21580	395	\N	Osches	55
21581	396	\N	Ourches-sur-Meuse	55
21582	397	\N	Pagny-la-Blanche-Côte	55
21583	398	\N	Pagny-sur-Meuse	55
21584	399	\N	Pareid	55
21585	400	\N	Parfondrupt	55
21586	401	\N	Paroches	55
21587	403	\N	Peuvillers	55
21588	404	\N	Pierrefitte-sur-Aire	55
21589	405	\N	Pillon	55
21590	406	\N	Pintheville	55
21591	407	\N	Pont-sur-Meuse	55
21592	408	\N	Pouilly-sur-Meuse	55
21593	409	\N	Pretz-en-Argonne	55
21594	410	\N	Quincy-Landzécourt	55
21595	411	\N	Rambluzin-et-Benoite-Vaux	55
21596	412	\N	Rambucourt	55
21597	414	\N	Rancourt-sur-Ornain	55
21598	415	\N	Ranzières	55
21599	416	\N	Rarécourt	55
21600	419	\N	Récicourt	55
21601	420	\N	Récourt-le-Creux	55
21602	421	\N	Reffroy	55
21603	422	\N	Regnéville-sur-Meuse	55
21604	423	\N	Rembercourt-Sommaisne	55
21605	424	\N	Remennecourt	55
21606	425	\N	Remoiville	55
21607	426	\N	Resson	55
21608	427	\N	Revigny-sur-Ornain	55
21609	428	\N	Réville-aux-Bois	55
21610	429	\N	Riaville	55
21611	430	\N	Ribeaucourt	55
21612	431	\N	Richecourt	55
21613	433	\N	Rigny-la-Salle	55
21614	434	\N	Rigny-Saint-Martin	55
21615	435	\N	Robert-Espagne	55
21616	436	\N	Roises	55
21617	437	\N	Romagne-sous-les-Côtes	55
21618	438	\N	Romagne-sous-Montfaucon	55
21619	439	\N	Ronvaux	55
21620	442	\N	Raival	55
21621	443	\N	Rouvres-en-Woëvre	55
21622	444	\N	Rouvrois-sur-Meuse	55
21623	445	\N	Rouvrois-sur-Othain	55
21624	446	\N	Rumont	55
21625	447	\N	Rupt-aux-Nonains	55
21626	448	\N	Rupt-devant-Saint-Mihiel	55
21627	449	\N	Rupt-en-Woëvre	55
21628	450	\N	Rupt-sur-Othain	55
21629	452	\N	Saint-Amand-sur-Ornain	55
21630	453	\N	Saint-André-en-Barrois	55
21631	454	\N	Saint-Aubin-sur-Aire	55
21632	456	\N	Saint-Germain-sur-Meuse	55
21633	457	\N	Saint-Hilaire-en-Woëvre	55
21634	458	\N	Saint-Jean-lès-Buzy	55
21635	459	\N	Saint-Joire	55
21636	460	\N	Saint-Julien-sous-les-Côtes	55
21637	461	\N	Saint-Laurent-sur-Othain	55
21638	462	\N	Saint-Maurice-sous-les-Côtes	55
21639	463	\N	Saint-Mihiel	55
21640	464	\N	Saint-Pierrevillers	55
21641	465	\N	Saint-Remy-la-Calonne	55
21642	466	\N	Salmagne	55
21643	467	\N	Sampigny	55
21644	468	\N	Samogneux	55
21645	469	\N	Sassey-sur-Meuse	55
21646	470	\N	Saudrupt	55
21647	471	\N	Saulmory-et-Villefranche	55
21648	472	\N	Saulvaux	55
21649	473	\N	Saulx-lès-Champlon	55
21650	474	\N	Sauvigny	55
21651	475	\N	Sauvoy	55
21652	476	\N	Savonnières-devant-Bar	55
21653	477	\N	Savonnières-en-Perthois	55
21654	479	\N	Seigneulles	55
21655	481	\N	Senon	55
21656	482	\N	Senoncourt-les-Maujouy	55
21657	484	\N	Septsarges	55
21658	485	\N	Sepvigny	55
21659	487	\N	Seuzey	55
21660	488	\N	Silmont	55
21661	489	\N	Sivry-la-Perche	55
21662	490	\N	Sivry-sur-Meuse	55
21663	492	\N	Sommedieue	55
21664	493	\N	Sommeilles	55
21665	494	\N	Sommelonne	55
21666	495	\N	Sorbey	55
21667	496	\N	Sorcy-Saint-Martin	55
21668	497	\N	Souhesmes-Rampont	55
21669	498	\N	Souilly	55
21670	500	\N	Spincourt	55
21671	501	\N	Stainville	55
21672	502	\N	Stenay	55
21673	503	\N	Taillancourt	55
21674	504	\N	Tannois	55
21675	505	\N	Thierville-sur-Meuse	55
21676	506	\N	Thillombois	55
21677	507	\N	Thillot	55
21678	508	\N	Thonne-la-Long	55
21679	509	\N	Thonne-le-Thil	55
21680	510	\N	Thonne-les-Près	55
21681	511	\N	Thonnelle	55
21682	512	\N	Tilly-sur-Meuse	55
21683	514	\N	Trémont-sur-Saulx	55
21684	515	\N	Trésauvaux	55
21685	516	\N	Tréveray	55
21686	517	\N	Seuil-d'Argonne	55
21687	518	\N	Cousances-lès-Triconville	55
21688	519	\N	Tronville-en-Barrois	55
21689	520	\N	Troussey	55
21690	521	\N	Troyon	55
21691	522	\N	Ugny-sur-Meuse	55
21692	523	\N	Vacherauville	55
21693	525	\N	Vadelaincourt	55
21694	526	\N	Vadonville	55
21695	527	\N	Varennes-en-Argonne	55
21696	528	\N	Varnéville	55
21697	530	\N	Valbois	55
21698	531	\N	Vassincourt	55
21699	532	\N	Vaubecourt	55
21700	533	\N	Vaucouleurs	55
21701	534	\N	Vaudeville-le-Haut	55
21702	535	\N	Vaudoncourt	55
21703	536	\N	Vauquois	55
21704	537	\N	Vaux-devant-Damloup	55
21705	540	\N	Vaux-lès-Palameix	55
21706	541	\N	Vavincourt	55
21707	543	\N	Velaines	55
21708	544	\N	Velosnes	55
21709	545	\N	Verdun	55
21710	546	\N	Verneuil-Grand	55
21711	547	\N	Verneuil-Petit	55
21712	549	\N	Véry	55
21713	551	\N	Vigneulles-lès-Hattonchâtel	55
21714	552	\N	Vigneul-sous-Montmédy	55
21715	553	\N	Vignot	55
21716	554	\N	Villécloye	55
21717	555	\N	Ville-devant-Belrain	55
21718	556	\N	Ville-devant-Chaumont	55
21719	557	\N	Ville-en-Woëvre	55
21720	559	\N	Villeroy-sur-Méholle	55
21721	560	\N	Villers-aux-Vents	55
21722	561	\N	Villers-devant-Dun	55
21723	562	\N	Villers-le-Sec	55
21724	563	\N	Villers-lès-Mangiennes	55
21725	565	\N	Villers-sous-Pareid	55
21726	566	\N	Villers-sur-Meuse	55
21727	567	\N	Ville-sur-Cousances	55
21728	568	\N	Ville-sur-Saulx	55
21729	569	\N	Villotte-devant-Louppy	55
21730	570	\N	Villotte-sur-Aire	55
21731	571	\N	Vilosnes-Haraumont	55
21732	572	\N	Vittarville	55
21733	573	\N	Void-Vacon	55
21734	574	\N	Vouthon-Bas	55
21735	575	\N	Vouthon-Haut	55
21736	577	\N	Waly	55
21737	578	\N	Warcq	55
21738	579	\N	Watronville	55
21739	580	\N	Wavrille	55
21740	581	\N	Willeroncourt	55
21741	582	\N	Wiseppe	55
21742	583	\N	Woël	55
21743	584	\N	Woimbey	55
21744	586	\N	Xivray-et-Marvoisin	55
21745	001	\N	Allaire	56
21746	002	\N	Ambon	56
21747	003	\N	Arradon	56
21748	004	\N	Arzal	56
21749	005	\N	Arzon	56
21750	006	\N	Augan	56
21751	007	\N	Auray	56
21752	008	\N	Baden	56
21753	009	\N	Bangor	56
21754	010	\N	Baud	56
21755	011	\N	Béganne	56
21756	012	\N	Beignon	56
21757	013	\N	Belz	56
21758	014	\N	Berné	56
21759	015	\N	Berric	56
21760	016	\N	Bieuzy	56
21761	017	\N	Bignan	56
21762	018	\N	Billiers	56
21763	019	\N	Billio	56
21764	020	\N	Bohal	56
21765	021	\N	Brandérion	56
21766	022	\N	Brandivy	56
21767	023	\N	Brech	56
21768	024	\N	Bréhan	56
21769	025	\N	Brignac	56
21770	026	\N	Bubry	56
21771	027	\N	Buléon	56
21772	028	\N	Caden	56
21773	029	\N	Calan	56
21774	030	\N	Camoël	56
21775	031	\N	Camors	56
21776	032	\N	Campénéac	56
21777	033	\N	Carentoir	56
21778	034	\N	Carnac	56
21779	035	\N	Caro	56
21780	036	\N	Caudan	56
21781	037	\N	Chapelle-Caro	56
21782	038	\N	Chapelle-Gaceline	56
21783	039	\N	Chapelle-Neuve	56
21784	040	\N	Cléguer	56
21785	041	\N	Cléguérec	56
21786	042	\N	Colpo	56
21787	043	\N	Concoret	56
21788	044	\N	Cournon	56
21789	045	\N	Cours	56
21790	046	\N	Crach	56
21791	047	\N	Crédin	56
21792	048	\N	Croisty	56
21793	049	\N	Croixanvec	56
21794	050	\N	Croix-Helléan	56
21795	051	\N	Cruguel	56
21796	052	\N	Damgan	56
21797	053	\N	Elven	56
21798	054	\N	Erdeven	56
21799	055	\N	Étel	56
21800	056	\N	Évriguet	56
21801	057	\N	Faouët	56
21802	058	\N	Férel	56
21803	059	\N	Forges	56
21804	060	\N	Fougerêts	56
21805	061	\N	Gacilly	56
21806	062	\N	Gâvres	56
21807	063	\N	Gestel	56
21808	064	\N	Glénac	56
21809	065	\N	Gourhel	56
21810	066	\N	Gourin	56
21811	067	\N	Grand-Champ	56
21812	068	\N	Grée-Saint-Laurent	56
21813	069	\N	Groix	56
21814	070	\N	Guégon	56
21815	071	\N	Guéhenno	56
21816	072	\N	Gueltas	56
21817	073	\N	Guémené-sur-Scorff	56
21818	074	\N	Guénin	56
21819	075	\N	Guer	56
21820	076	\N	Guern	56
21821	077	\N	Guerno	56
21822	078	\N	Guidel	56
21823	079	\N	Guillac	56
21824	080	\N	Guilliers	56
21825	081	\N	Guiscriff	56
21826	082	\N	Helléan	56
21827	083	\N	Hennebont	56
21828	084	\N	Hézo	56
21829	085	\N	Hdic	56
21830	086	\N	Île-d'Houat	56
21831	087	\N	Île-aux-Moines	56
21832	088	\N	Île-d'Arz	56
21833	089	\N	Inguiniel	56
21834	090	\N	Inzinzac-Lochrist	56
21835	091	\N	Josselin	56
21836	092	\N	Kerfourn	56
21837	093	\N	Kergrist	56
21838	094	\N	Kervignac	56
21839	096	\N	Landaul	56
21840	097	\N	Landévant	56
21841	098	\N	Lanester	56
21842	099	\N	Langoëlan	56
21843	100	\N	Langonnet	56
21844	101	\N	Languidic	56
21845	102	\N	Lanouée	56
21846	103	\N	Lantillac	56
21847	104	\N	Lanvaudan	56
21848	105	\N	Lanvénégen	56
21849	106	\N	Larmor-Baden	56
21850	107	\N	Larmor-Plage	56
21851	108	\N	Larré	56
21852	109	\N	Lauzach	56
21853	110	\N	Lignol	56
21854	111	\N	Limerzel	56
21855	112	\N	Lizio	56
21856	113	\N	Locmalo	56
21857	114	\N	Locmaria	56
21858	115	\N	Locmaria-Grand-Champ	56
21859	116	\N	Locmariaquer	56
21860	117	\N	Locminé	56
21861	118	\N	Locmiquélic	56
21862	119	\N	Locoal-Mendon	56
21863	120	\N	Locqueltas	56
21864	121	\N	Lorient	56
21865	122	\N	Loyat	56
21866	123	\N	Malansac	56
21867	124	\N	Malestroit	56
21868	125	\N	Malguénac	56
21869	126	\N	Marzan	56
21870	127	\N	Mauron	56
21871	128	\N	Melrand	56
21872	129	\N	Ménéac	56
21873	130	\N	Merlevenez	56
21874	131	\N	Meslan	56
21875	132	\N	Meucon	56
21876	133	\N	Missiriac	56
21877	134	\N	Mohon	56
21878	135	\N	Molac	56
21879	136	\N	Monteneuf	56
21880	137	\N	Monterblanc	56
21881	138	\N	Monterrein	56
21882	139	\N	Montertelot	56
21883	140	\N	Moréac	56
21884	141	\N	Moustoir-Ac	56
21885	142	\N	Moustoir-Remungol	56
21886	143	\N	Muzillac	56
21887	144	\N	Naizin	56
21888	145	\N	Néant-sur-Yvel	56
21889	146	\N	Neulliac	56
21890	147	\N	Nivillac	56
21891	148	\N	Nostang	56
21892	149	\N	Noyal-Muzillac	56
21893	150	\N	Noyalo	56
21894	151	\N	Noyal-Pontivy	56
21895	152	\N	Palais	56
21896	153	\N	Péaule	56
21897	154	\N	Peillac	56
21898	155	\N	Pénestin	56
21899	156	\N	Persquen	56
21900	157	\N	Plaudren	56
21901	158	\N	Plescop	56
21902	159	\N	Pleucadeuc	56
21903	160	\N	Pleugriffet	56
21904	161	\N	Ploemel	56
21905	162	\N	Ploemeur	56
21906	163	\N	Ploërdut	56
21907	164	\N	Ploeren	56
21908	165	\N	Ploërmel	56
21909	166	\N	Plouay	56
21910	167	\N	Plougoumelen	56
21911	168	\N	Plouharnel	56
21912	169	\N	Plouhinec	56
21913	170	\N	Plouray	56
21914	171	\N	Pluherlin	56
21915	172	\N	Plumelec	56
21916	173	\N	Pluméliau	56
21917	174	\N	Plumelin	56
21918	175	\N	Plumergat	56
21919	176	\N	Pluneret	56
21920	177	\N	Pluvigner	56
21921	178	\N	Pontivy	56
21922	179	\N	Pont-Scorff	56
21923	180	\N	Porcaro	56
21924	181	\N	Port-Louis	56
21925	182	\N	Priziac	56
21926	183	\N	Quelneuc	56
21927	184	\N	Questembert	56
21928	185	\N	Quéven	56
21929	186	\N	Quiberon	56
21930	187	\N	Quily	56
21931	188	\N	Quistinic	56
21932	189	\N	Radenac	56
21933	190	\N	Réguiny	56
21934	191	\N	Réminiac	56
21935	192	\N	Remungol	56
21936	193	\N	Riantec	56
21937	194	\N	Rieux	56
21938	195	\N	Roche-Bernard	56
21939	196	\N	Rochefort-en-Terre	56
21940	197	\N	Roc-Saint-André	56
21941	198	\N	Rohan	56
21942	199	\N	Roudouallec	56
21943	200	\N	Ruffiac	56
21944	201	\N	Saint	56
21945	202	\N	Saint-Abraham	56
21946	203	\N	Saint-Aignan	56
21947	204	\N	Saint-Allouestre	56
21948	205	\N	Saint-Armel	56
21949	206	\N	Saint-Avé	56
21950	207	\N	Saint-Barthélemy	56
21951	208	\N	Saint-Brieuc-de-Mauron	56
21952	209	\N	Sainte-Brigitte	56
21953	210	\N	Saint-Caradec-Trégomel	56
21954	211	\N	Saint-Congard	56
21955	212	\N	Saint-Dolay	56
21956	213	\N	Saint-Gérand	56
21957	214	\N	Saint-Gildas-de-Rhuys	56
21958	215	\N	Saint-Gonnery	56
21959	216	\N	Saint-Gorgon	56
21960	218	\N	Saint-Gravé	56
21961	219	\N	Saint-Guyomard	56
21962	220	\N	Sainte-Hélène	56
21963	221	\N	Saint-Jacut-les-Pins	56
21964	222	\N	Saint-Jean-Brévelay	56
21965	223	\N	Saint-Jean-la-Poterie	56
21966	224	\N	Saint-Laurent-sur-Oust	56
21967	225	\N	Saint-Léry	56
21968	226	\N	Saint-Malo-de-Beignon	56
21969	227	\N	Saint-Malo-des-Trois-Fontaines	56
21970	228	\N	Saint-Marcel	56
21971	229	\N	Saint-Martin	56
21972	230	\N	Saint-Nicolas-du-Tertre	56
21973	231	\N	Saint-Nolff	56
21974	232	\N	Saint-Perreux	56
21975	233	\N	Saint-Philibert	56
21976	234	\N	Saint-Pierre-Quiberon	56
21977	236	\N	Saint-Servant	56
21978	237	\N	Saint-Thuriau	56
21979	238	\N	Saint-Tugdual	56
21980	239	\N	Saint-Vincent-sur-Oust	56
21981	240	\N	Sarzeau	56
21982	241	\N	Sauzon	56
21983	242	\N	Séglien	56
21984	243	\N	Séné	56
21985	244	\N	Sérent	56
21986	245	\N	Silfiac	56
21987	246	\N	Sourn	56
21988	247	\N	Sulniac	56
21989	248	\N	Surzur	56
21990	249	\N	Taupont	56
21991	250	\N	Théhillac	56
21992	251	\N	Theix	56
21993	252	\N	Tour-du-Parc	56
21994	253	\N	Tréal	56
21995	254	\N	Trédion	56
21996	255	\N	Treffléan	56
21997	256	\N	Tréhorenteuc	56
21998	257	\N	Trinité-Porhoët	56
21999	258	\N	Trinité-sur-Mer	56
22000	259	\N	Trinité-Surzur	56
22001	260	\N	Vannes	56
22002	261	\N	Vraie-Croix	56
22003	262	\N	Bono	56
22004	263	\N	Sainte-Anne-d'Auray	56
22005	264	\N	Kernascléden	56
22006	001	\N	Aboncourt	57
22007	002	\N	Aboncourt-sur-Seille	57
22008	003	\N	Abreschviller	57
22009	004	\N	Achain	57
22010	006	\N	Achen	57
22011	007	\N	Adaincourt	57
22012	008	\N	Adelange	57
22013	009	\N	Ajoncourt	57
22014	010	\N	Alaincourt-la-Côte	57
22015	011	\N	Albestroff	57
22016	012	\N	Algrange	57
22017	013	\N	Alsting	57
22018	014	\N	Altrippe	57
22019	015	\N	Altviller	57
22020	016	\N	Alzing	57
22021	017	\N	Amanvillers	57
22022	018	\N	Amelécourt	57
22023	019	\N	Amnéville	57
22024	020	\N	Ancerville	57
22025	021	\N	Ancy-sur-Moselle	57
22026	022	\N	Angevillers	57
22027	024	\N	Antilly	57
22028	025	\N	Anzeling	57
22029	026	\N	Apach	57
22030	027	\N	Arraincourt	57
22031	028	\N	Argancy	57
22032	029	\N	Arriance	57
22033	030	\N	Arry	57
22034	031	\N	Ars-Laquenexy	57
22035	032	\N	Ars-sur-Moselle	57
22036	033	\N	Arzviller	57
22037	034	\N	Aspach	57
22038	035	\N	Assenoncourt	57
22039	036	\N	Attilloncourt	57
22040	037	\N	Aube	57
22041	038	\N	Audun-le-Tiche	57
22042	039	\N	Augny	57
22043	040	\N	Aulnois-sur-Seille	57
22044	041	\N	Aumetz	57
22045	042	\N	Avricourt	57
22046	043	\N	Ay-sur-Moselle	57
22047	044	\N	Azoudange	57
22048	045	\N	Bacourt	57
22049	046	\N	Baerenthal	57
22050	047	\N	Bambiderstroff	57
22051	048	\N	Bannay	57
22052	049	\N	Ban-Saint-Martin	57
22053	050	\N	Barchain	57
22054	051	\N	Baronville	57
22055	052	\N	Barst	57
22056	053	\N	Bassing	57
22057	054	\N	Baudrecourt	57
22058	055	\N	Bazoncourt	57
22059	056	\N	Bébing	57
22060	057	\N	Béchy	57
22061	058	\N	Behren-lès-Forbach	57
22062	059	\N	Bellange	57
22063	060	\N	Bénestroff	57
22064	061	\N	Béning-lès-Saint-Avold	57
22065	062	\N	Berg-sur-Moselle	57
22066	063	\N	Bérig-Vintrange	57
22067	064	\N	Berling	57
22068	065	\N	Bermering	57
22069	066	\N	Berthelming	57
22070	067	\N	Bertrange	57
22071	069	\N	Berviller-en-Moselle	57
22072	070	\N	Bettange	57
22073	071	\N	Bettborn	57
22074	072	\N	Bettelainville	57
22075	073	\N	Betting-lès-Saint-Avold	57
22076	074	\N	Bettviller	57
22077	075	\N	Beux	57
22078	076	\N	Beyren-lès-Sierck	57
22079	077	\N	Bezange-la-Petite	57
22080	079	\N	Bibiche	57
22081	080	\N	Bickenholtz	57
22082	081	\N	Bidestroff	57
22083	082	\N	Biding	57
22084	083	\N	Bining	57
22085	084	\N	Bioncourt	57
22086	085	\N	Bionville-sur-Nied	57
22087	086	\N	Belles-Forêts	57
22088	087	\N	Bisten-en-Lorraine	57
22089	088	\N	Bistroff	57
22090	089	\N	Bitche	57
22091	090	\N	Blanche-Église	57
22092	091	\N	Bliesbruck	57
22093	092	\N	Blies-Ébersing	57
22094	093	\N	Blies-Guersviller	57
22095	095	\N	Boucheporn	57
22096	096	\N	Boulange	57
22097	097	\N	Boulay-Moselle	57
22098	098	\N	Bourgaltroff	57
22099	099	\N	Bourdonnay	57
22100	100	\N	Bourscheid	57
22101	101	\N	Bousbach	57
22102	102	\N	Bousse	57
22103	103	\N	Bousseviller	57
22104	104	\N	Boust	57
22105	105	\N	Boustroff	57
22106	106	\N	Bouzonville	57
22107	107	\N	Bréhain	57
22108	108	\N	Breidenbach	57
22109	109	\N	Breistroff-la-Grande	57
22110	110	\N	Brettnach	57
22111	111	\N	Bronvaux	57
22112	112	\N	Brouck	57
22113	113	\N	Brouderdorff	57
22114	114	\N	Brouviller	57
22115	115	\N	Brulange	57
22116	116	\N	Buchy	57
22117	117	\N	Buding	57
22118	118	\N	Budling	57
22119	119	\N	Buhl-Lorraine	57
22120	120	\N	Burlioncourt	57
22121	121	\N	Burtoncourt	57
22122	122	\N	Cappel	57
22123	123	\N	Carling	57
22124	124	\N	Cattenom	57
22125	125	\N	Chailly-lès-Ennery	57
22126	126	\N	Chambrey	57
22127	127	\N	Chanville	57
22128	128	\N	Charleville-sous-Bois	57
22129	129	\N	Charly-Oradour	57
22130	130	\N	Château-Bréhain	57
22131	131	\N	Château-Rouge	57
22132	132	\N	Château-Salins	57
22133	133	\N	Château-Voué	57
22134	134	\N	Châtel-Saint-Germain	57
22135	136	\N	Chémery-les-Deux	57
22136	137	\N	Cheminot	57
22137	138	\N	Chenois	57
22138	139	\N	Chérisey	57
22139	140	\N	Chesny	57
22140	141	\N	Chicourt	57
22141	142	\N	Chieulles	57
22142	143	\N	Clouange	57
22143	144	\N	Cocheren	57
22144	145	\N	Coincy	57
22145	146	\N	Coin-lès-Cuvry	57
22146	147	\N	Coin-sur-Seille	57
22147	148	\N	Colligny	57
22148	149	\N	Colmen	57
22149	150	\N	Condé-Northen	57
22150	151	\N	Conthil	57
22151	152	\N	Contz-les-Bains	57
22152	153	\N	Corny-sur-Moselle	57
22153	154	\N	Coume	57
22154	155	\N	Courcelles-Chaussy	57
22155	156	\N	Courcelles-sur-Nied	57
22156	158	\N	Craincourt	57
22157	159	\N	Créhange	57
22158	160	\N	Creutzwald	57
22159	161	\N	Cutting	57
22160	162	\N	Cuvry	57
22161	163	\N	Dabo	57
22162	165	\N	Dalem	57
22163	166	\N	Dalhain	57
22164	167	\N	Dalstein	57
22165	168	\N	Danne-et-Quatre-Vents	57
22166	169	\N	Dannelbourg	57
22167	171	\N	Delme	57
22168	172	\N	Denting	57
22169	173	\N	Desseling	57
22170	174	\N	Destry	57
22171	175	\N	Diane-Capelle	57
22172	176	\N	Diebling	57
22173	177	\N	Dieuze	57
22174	178	\N	Diffembach-lès-Hellimer	57
22175	179	\N	Distroff	57
22176	180	\N	Dolving	57
22177	181	\N	Domnon-lès-Dieuze	57
22178	182	\N	Donjeux	57
22179	183	\N	Donnelay	57
22180	184	\N	Dornot	57
22181	186	\N	Ébersviller	57
22182	187	\N	Éblange	57
22183	188	\N	Éguelshardt	57
22184	189	\N	Eincheville	57
22185	190	\N	Elvange	57
22186	191	\N	Elzange	57
22187	192	\N	Enchenberg	57
22188	193	\N	Ennery	57
22189	194	\N	Entrange	57
22190	195	\N	Epping	57
22191	196	\N	Erching	57
22192	197	\N	Ernestviller	57
22193	198	\N	Erstroff	57
22194	199	\N	Escherange	57
22195	200	\N	Étangs	57
22196	201	\N	Etting	57
22197	202	\N	Etzling	57
22198	203	\N	Évrange	57
22199	204	\N	Failly	57
22200	205	\N	Falck	57
22201	206	\N	Fameck	57
22202	207	\N	Farébersviller	57
22203	208	\N	Farschviller	57
22204	209	\N	Faulquemont	57
22205	210	\N	Fénétrange	57
22206	211	\N	Fèves	57
22207	212	\N	Féy	57
22208	213	\N	Filstroff	57
22209	214	\N	Fixem	57
22210	215	\N	Flastroff	57
22211	216	\N	Fleisheim	57
22212	217	\N	Flétrange	57
22213	218	\N	Fleury	57
22214	219	\N	Flévy	57
22215	220	\N	Flocourt	57
22216	221	\N	Florange	57
22217	222	\N	Folkling	57
22218	224	\N	Folschviller	57
22219	225	\N	Fonteny	57
22220	226	\N	Fontoy	57
22221	227	\N	Forbach	57
22222	228	\N	Fossieux	57
22223	229	\N	Foulcrey	57
22224	230	\N	Fouligny	57
22225	231	\N	Foville	57
22226	232	\N	Francaltroff	57
22227	233	\N	Fraquelfing	57
22228	234	\N	Frauenberg	57
22229	235	\N	Freistroff	57
22230	236	\N	Frémery	57
22231	237	\N	Frémestroff	57
22232	238	\N	Fresnes-en-Saulnois	57
22233	239	\N	Freybouse	57
22234	240	\N	Freyming-Merlebach	57
22235	241	\N	Fribourg	57
22236	242	\N	Gandrange	57
22237	244	\N	Garrebourg	57
22238	245	\N	Gavisse	57
22239	246	\N	Gelucourt	57
22240	247	\N	Gerbécourt	57
22241	248	\N	Givrycourt	57
22242	249	\N	Glatigny	57
22243	250	\N	Goetzenbruck	57
22244	251	\N	Goin	57
22245	252	\N	Gomelange	57
22246	253	\N	Gondrexange	57
22247	254	\N	Gorze	57
22248	255	\N	Gosselming	57
22249	256	\N	Gravelotte	57
22250	257	\N	Grémecey	57
22251	258	\N	Gréning	57
22252	259	\N	Grindorff-Bizing	57
22253	260	\N	Grosbliederstroff	57
22254	261	\N	Gros-Réderching	57
22255	262	\N	Grostenquin	57
22256	263	\N	Grundviller	57
22257	264	\N	Guebenhouse	57
22258	265	\N	Guébestroff	57
22259	266	\N	Guéblange-lès-Dieuze	57
22260	267	\N	Val-de-Guéblange	57
22261	268	\N	Guébling	57
22262	269	\N	Guénange	57
22263	270	\N	Val-de-Bride	57
22264	271	\N	Guenviller	57
22265	272	\N	Guermange	57
22266	273	\N	Guerstling	57
22267	274	\N	Guerting	57
22268	275	\N	Guessling-Hémering	57
22269	276	\N	Guinglange	57
22270	277	\N	Guinkirchen	57
22271	278	\N	Guinzeling	57
22272	280	\N	Guntzviller	57
22273	281	\N	Haboudange	57
22274	282	\N	Hagen	57
22275	283	\N	Hagondange	57
22276	284	\N	Hallering	57
22277	286	\N	Halstroff	57
22278	287	\N	Basse-Ham	57
22279	288	\N	Ham-sous-Varsberg	57
22280	289	\N	Hambach	57
22281	290	\N	Hampont	57
22282	291	\N	Hangviller	57
22283	292	\N	Hannocourt	57
22284	293	\N	Han-sur-Nied	57
22285	294	\N	Hanviller	57
22286	295	\N	Haraucourt-sur-Seille	57
22287	296	\N	Hargarten-aux-Mines	57
22288	297	\N	Harprich	57
22289	298	\N	Harreberg	57
22290	299	\N	Hartzviller	57
22291	300	\N	Haselbourg	57
22292	301	\N	Haspelschiedt	57
22293	302	\N	Hattigny	57
22294	303	\N	Hauconcourt	57
22295	304	\N	Haut-Clocher	57
22296	305	\N	Havange	57
22297	306	\N	Hayange	57
22298	307	\N	Hayes	57
22299	308	\N	Hazembourg	57
22300	309	\N	Heining-lès-Bouzonville	57
22301	310	\N	Hellering-lès-Fénétrange	57
22302	311	\N	Hellimer	57
22303	312	\N	Helstroff	57
22304	313	\N	Hémilly	57
22305	314	\N	Héming	57
22306	315	\N	Henridorff	57
22307	316	\N	Henriville	57
22308	317	\N	Hérange	57
22309	318	\N	Hermelange	57
22310	319	\N	Herny	57
22311	320	\N	Hertzing	57
22312	321	\N	Hesse	57
22313	322	\N	Hestroff	57
22314	323	\N	Hettange-Grande	57
22315	324	\N	Hilbesheim	57
22316	325	\N	Hilsprich	57
22317	326	\N	Hinckange	57
22318	328	\N	Holacourt	57
22319	329	\N	Holling	57
22320	330	\N	Holving	57
22321	331	\N	Hombourg-Budange	57
22322	332	\N	Hombourg-Haut	57
22323	333	\N	Hommarting	57
22324	334	\N	Hommert	57
22325	335	\N	Honskirch	57
22326	336	\N	Hôpital	57
22327	337	\N	Hoste	57
22328	338	\N	Hottviller	57
22329	339	\N	Hultehouse	57
22330	340	\N	Hundling	57
22331	341	\N	Hunting	57
22332	342	\N	Ibigny	57
22333	343	\N	Illange	57
22334	344	\N	Imling	57
22335	345	\N	Inglange	57
22336	346	\N	Insming	57
22337	347	\N	Insviller	57
22338	348	\N	Ippling	57
22339	349	\N	Jallaucourt	57
22340	350	\N	Jouy-aux-Arches	57
22341	351	\N	Jury	57
22342	352	\N	Jussy	57
22343	353	\N	Juvelize	57
22344	354	\N	Juville	57
22345	355	\N	Kalhausen	57
22346	356	\N	Kanfen	57
22347	357	\N	Kappelkinger	57
22348	358	\N	Kédange-sur-Canner	57
22349	359	\N	Kemplich	57
22350	360	\N	Kerbach	57
22351	361	\N	Kerling-lès-Sierck	57
22352	362	\N	Kerprich-aux-Bois	57
22353	364	\N	Kirsch-lès-Sierck	57
22354	365	\N	Kirschnaumen	57
22355	366	\N	Kirviller	57
22356	367	\N	Klang	57
22357	368	\N	Knutange	57
22358	370	\N	Knigsmacker	57
22359	371	\N	Haute-Kontz	57
22360	372	\N	Kuntzig	57
22361	373	\N	Lachambre	57
22362	374	\N	Lafrimbolle	57
22363	375	\N	Lagarde	57
22364	376	\N	Lambach	57
22365	377	\N	Landange	57
22366	379	\N	Landroff	57
22367	380	\N	Laneuveville-lès-Lorquin	57
22368	381	\N	Laneuveville-en-Saulnois	57
22369	382	\N	Langatte	57
22370	383	\N	Languimberg	57
22371	384	\N	Laning	57
22372	385	\N	Laquenexy	57
22373	386	\N	Laudrefang	57
22374	387	\N	Laumesfeld	57
22375	388	\N	Launstroff	57
22376	389	\N	Lelling	57
22377	390	\N	Lemberg	57
22378	391	\N	Lemoncourt	57
22379	392	\N	Lemud	57
22380	393	\N	Lengelsheim	57
22381	394	\N	Léning	57
22382	395	\N	Lesse	57
22383	396	\N	Lessy	57
22384	397	\N	Ley	57
22385	398	\N	Leyviller	57
22386	399	\N	Lezey	57
22387	401	\N	Lidrezing	57
22388	402	\N	Liederschiedt	57
22389	403	\N	Liéhon	57
22390	404	\N	Lindre-Basse	57
22391	405	\N	Lindre-Haute	57
22392	406	\N	Liocourt	57
22393	407	\N	Lixheim	57
22394	408	\N	Lixing-lès-Rouhling	57
22395	409	\N	Lixing-lès-Saint-Avold	57
22396	410	\N	Lhor	57
22397	411	\N	Lommerange	57
22398	412	\N	Longeville-lès-Metz	57
22399	413	\N	Longeville-lès-Saint-Avold	57
22400	414	\N	Lorquin	57
22401	415	\N	Lorry-lès-Metz	57
22402	416	\N	Lorry-Mardigny	57
22403	417	\N	Lostroff	57
22404	418	\N	Loudrefing	57
22405	419	\N	Loupershouse	57
22406	421	\N	Loutzviller	57
22407	422	\N	Louvigny	57
22408	423	\N	Lubécourt	57
22409	424	\N	Lucy	57
22410	425	\N	Luppy	57
22411	426	\N	Luttange	57
22412	427	\N	Lutzelbourg	57
22413	428	\N	Macheren	57
22414	430	\N	Mainvillers	57
22415	431	\N	Maizeroy	57
22416	432	\N	Maizery	57
22417	433	\N	Maizières-lès-Metz	57
22418	434	\N	Maizières-lès-Vic	57
22419	436	\N	Malaucourt-sur-Seille	57
22420	437	\N	Malling	57
22421	438	\N	Malroy	57
22422	439	\N	Manderen	57
22423	440	\N	Manhoué	57
22424	441	\N	Manom	57
22425	442	\N	Many	57
22426	443	\N	Marange-Silvange	57
22427	444	\N	Marange-Zondrange	57
22428	445	\N	Marieulles	57
22429	446	\N	Marimont-lès-Bénestroff	57
22430	447	\N	Marly	57
22431	448	\N	Marsal	57
22432	449	\N	Marsilly	57
22433	451	\N	Marthille	57
22434	452	\N	Maxe	57
22435	453	\N	Maxstadt	57
22436	454	\N	Mécleuves	57
22437	455	\N	Mégange	57
22438	456	\N	Meisenthal	57
22439	457	\N	Menskirch	57
22440	459	\N	Merschweiller	57
22441	460	\N	Merten	57
22442	461	\N	Métairies-Saint-Quirin	57
22443	462	\N	Metting	57
22444	463	\N	Metz	57
22445	464	\N	Metzeresche	57
22446	465	\N	Metzervisse	57
22447	466	\N	Metzing	57
22448	467	\N	Mey	57
22449	468	\N	Mittelbronn	57
22450	469	\N	Mittersheim	57
22451	470	\N	Molring	57
22452	471	\N	Momerstroff	57
22453	472	\N	Moncheux	57
22454	473	\N	Moncourt	57
22455	474	\N	Mondelange	57
22456	475	\N	Mondorff	57
22457	476	\N	Monneren	57
22458	477	\N	Montbronn	57
22459	478	\N	Montdidier	57
22460	479	\N	Montenach	57
22461	480	\N	Montigny-lès-Metz	57
22462	481	\N	Montois-la-Montagne	57
22463	482	\N	Montoy-Flanville	57
22464	483	\N	Morhange	57
22465	484	\N	Morsbach	57
22466	485	\N	Morville-lès-Vic	57
22467	486	\N	Morville-sur-Nied	57
22468	487	\N	Moulins-lès-Metz	57
22469	488	\N	Moussey	57
22470	489	\N	Mouterhouse	57
22471	490	\N	Moyenvic	57
22472	491	\N	Moyeuvre-Grande	57
22473	492	\N	Moyeuvre-Petite	57
22474	493	\N	Mulcey	57
22475	494	\N	Munster	57
22476	495	\N	Narbéfontaine	57
22477	496	\N	Nébing	57
22478	497	\N	Nelling	57
22479	498	\N	Neufchef	57
22480	499	\N	Neufgrange	57
22481	500	\N	Neufmoulins	57
22482	501	\N	Neufvillage	57
22483	502	\N	Neunkirchen-lès-Bouzonville	57
22484	504	\N	Niderhoff	57
22485	505	\N	Niderviller	57
22486	506	\N	Niederstinzel	57
22487	507	\N	Niedervisse	57
22488	508	\N	Nilvange	57
22489	509	\N	Nitting	57
22490	510	\N	Noisseville	57
22491	511	\N	Norroy-le-Veneur	57
22492	512	\N	Nouilly	57
22493	513	\N	Nousseviller-lès-Bitche	57
22494	514	\N	Nousseviller-Saint-Nabor	57
22495	515	\N	Novéant-sur-Moselle	57
22496	516	\N	Oberdorff	57
22497	517	\N	Obergailbach	57
22498	518	\N	Oberstinzel	57
22499	519	\N	Obervisse	57
22500	520	\N	Obreck	57
22501	521	\N	ting	57
22502	523	\N	Ogy	57
22503	524	\N	Ommeray	57
22504	525	\N	Oriocourt	57
22505	526	\N	Ormersviller	57
22506	527	\N	Orny	57
22507	528	\N	Oron	57
22508	529	\N	Ottange	57
22509	530	\N	Ottonville	57
22510	531	\N	Oudrenne	57
22511	532	\N	Pagny-lès-Goin	57
22512	533	\N	Pange	57
22513	534	\N	Peltre	57
22514	535	\N	Petit-Réderching	57
22515	536	\N	Petit-Tenquin	57
22516	537	\N	Petite-Rosselle	57
22517	538	\N	Pettoncourt	57
22518	539	\N	Pévange	57
22519	540	\N	Phalsbourg	57
22520	541	\N	Philippsbourg	57
22521	542	\N	Piblange	57
22522	543	\N	Pierrevillers	57
22523	544	\N	Plaine-de-Walsch	57
22524	545	\N	Plappeville	57
22525	546	\N	Plesnois	57
22526	547	\N	Pommérieux	57
22527	548	\N	Pontoy	57
22528	549	\N	Pontpierre	57
22529	550	\N	Porcelette	57
22530	551	\N	Postroff	57
22531	552	\N	Pouilly	57
22532	553	\N	Pournoy-la-Chétive	57
22533	554	\N	Pournoy-la-Grasse	57
22534	555	\N	Prévocourt	57
22535	556	\N	Puttelange-aux-Lacs	57
22536	557	\N	Puttelange-lès-Thionville	57
22537	558	\N	Puttigny	57
22538	559	\N	Puzieux	57
22539	560	\N	Racrange	57
22540	561	\N	Rahling	57
22541	562	\N	Ranguevaux	57
22542	563	\N	Raville	57
22543	564	\N	Réchicourt-le-Château	57
22544	565	\N	Rédange	57
22545	566	\N	Réding	57
22546	567	\N	Rémelfang	57
22547	568	\N	Rémelfing	57
22548	569	\N	Rémeling	57
22549	570	\N	Rémering	57
22550	571	\N	Rémering-lès-Puttelange	57
22551	572	\N	Rémilly	57
22552	573	\N	Réning	57
22553	574	\N	Basse-Rentgen	57
22554	575	\N	Retonfey	57
22555	576	\N	Rettel	57
22556	577	\N	Reyersviller	57
22557	578	\N	Rezonville	57
22558	579	\N	Rhodes	57
22559	580	\N	Riche	57
22560	581	\N	Richeling	57
22561	582	\N	Richemont	57
22562	583	\N	Richeval	57
22563	584	\N	Rimling	57
22564	585	\N	Ritzing	57
22565	586	\N	Rochonvillers	57
22566	587	\N	Rodalbe	57
22567	588	\N	Rodemack	57
22568	589	\N	Rohrbach-lès-Bitche	57
22569	590	\N	Rolbing	57
22570	591	\N	Rombas	57
22571	592	\N	Romelfing	57
22572	593	\N	Roncourt	57
22573	594	\N	Roppeviller	57
22574	595	\N	Rorbach-lès-Dieuze	57
22575	596	\N	Rosbruck	57
22576	597	\N	Rosselange	57
22577	598	\N	Rouhling	57
22578	599	\N	Roupeldange	57
22579	600	\N	Roussy-le-Village	57
22580	601	\N	Rozérieulles	57
22581	602	\N	Rurange-lès-Thionville	57
22582	603	\N	Russange	57
22583	604	\N	Rustroff	57
22584	605	\N	Sailly-Achâtel	57
22585	606	\N	Saint-Avold	57
22586	607	\N	Sainte-Barbe	57
22587	609	\N	Saint-Epvre	57
22588	610	\N	Saint-François-Lacroix	57
22589	611	\N	Saint-Georges	57
22590	612	\N	Saint-Hubert	57
22591	613	\N	Saint-Jean-de-Bassel	57
22592	614	\N	Saint-Jean-Kourtzerode	57
22593	615	\N	Saint-Jean-Rohrbach	57
22594	616	\N	Saint-Julien-lès-Metz	57
22595	617	\N	Saint-Jure	57
22596	618	\N	Saint-Louis	57
22597	619	\N	Saint-Louis-lès-Bitche	57
22598	620	\N	Sainte-Marie-aux-Chênes	57
22599	621	\N	Saint-Médard	57
22600	622	\N	Saint-Privat-la-Montagne	57
22601	623	\N	Saint-Quirin	57
22602	624	\N	Sainte-Ruffine	57
22603	625	\N	Salonnes	57
22604	626	\N	Sanry-lès-Vigy	57
22605	627	\N	Sanry-sur-Nied	57
22606	628	\N	Sarralbe	57
22607	629	\N	Sarraltroff	57
22608	630	\N	Sarrebourg	57
22609	631	\N	Sarreguemines	57
22610	633	\N	Sarreinsming	57
22611	634	\N	Saulny	57
22612	635	\N	Schalbach	57
22613	636	\N	Schmittviller	57
22614	637	\N	Schneckenbusch	57
22615	638	\N	Schneck	57
22616	639	\N	Schorbach	57
22617	640	\N	Schwerdorff	57
22618	641	\N	Schweyen	57
22619	642	\N	Scy-Chazelles	57
22620	643	\N	Secourt	57
22621	644	\N	Seingbouse	57
22622	645	\N	Semécourt	57
22623	647	\N	Serémange-Erzange	57
22624	648	\N	Servigny-lès-Raville	57
22625	649	\N	Servigny-lès-Sainte-Barbe	57
22626	650	\N	Sierck-les-Bains	57
22627	651	\N	Siersthal	57
22628	652	\N	Sillegny	57
22629	653	\N	Silly-en-Saulnois	57
22630	654	\N	Silly-sur-Nied	57
22631	655	\N	Solgne	57
22632	656	\N	Sorbey	57
22633	657	\N	Sotzeling	57
22634	658	\N	Soucht	57
22635	659	\N	Spicheren	57
22636	660	\N	Stiring-Wendel	57
22637	661	\N	Sturzelbronn	57
22638	662	\N	Suisse	57
22639	663	\N	Talange	57
22640	664	\N	Tarquimpol	57
22641	665	\N	Tenteling	57
22642	666	\N	Terville	57
22643	667	\N	Téterchen	57
22644	668	\N	Teting-sur-Nied	57
22645	669	\N	Théding	57
22646	670	\N	Thicourt	57
22647	671	\N	Thimonville	57
22648	672	\N	Thionville	57
22649	673	\N	Thonville	57
22650	674	\N	Tincry	57
22651	675	\N	Torcheville	57
22652	676	\N	Tragny	57
22653	677	\N	Trémery	57
22654	678	\N	Tressange	57
22655	679	\N	Tritteling-Redlach	57
22656	680	\N	Troisfontaines	57
22657	681	\N	Tromborn	57
22658	682	\N	Turquestein-Blancrupt	57
22659	683	\N	Uckange	57
22660	684	\N	Vahl-Ebersing	57
22661	685	\N	Vahl-lès-Bénestroff	57
22662	686	\N	Vahl-lès-Faulquemont	57
22663	687	\N	Vallerange	57
22664	689	\N	Valmestroff	57
22665	690	\N	Valmont	57
22666	691	\N	Valmunster	57
22667	692	\N	Vannecourt	57
22668	693	\N	Vantoux	57
22669	694	\N	Vany	57
22670	695	\N	Varize	57
22671	696	\N	Varsberg	57
22672	697	\N	Vasperviller	57
22673	698	\N	Vatimont	57
22674	700	\N	Vaudreching	57
22675	701	\N	Vaux	57
22676	702	\N	Vaxy	57
22677	703	\N	Veckersviller	57
22678	704	\N	Veckring	57
22679	705	\N	Velving	57
22680	706	\N	Vergaville	57
22681	707	\N	Vernéville	57
22682	708	\N	Verny	57
22683	709	\N	Vescheim	57
22684	711	\N	Vibersviller	57
22685	712	\N	Vic-sur-Seille	57
22686	713	\N	Vieux-Lixheim	57
22687	714	\N	Haute-Vigneulles	57
22688	715	\N	Vigny	57
22689	716	\N	Vigy	57
22690	717	\N	Viller	57
22691	718	\N	Villers-Stoncourt	57
22692	719	\N	Villers-sur-Nied	57
22693	720	\N	Villing	57
22694	721	\N	Vilsberg	57
22695	722	\N	Vionville	57
22696	723	\N	Virming	57
22697	724	\N	Vitry-sur-Orne	57
22698	725	\N	Vittersbourg	57
22699	726	\N	Vittoncourt	57
22700	727	\N	Viviers	57
22701	728	\N	Voimhaut	57
22702	730	\N	Volmerange-lès-Boulay	57
22703	731	\N	Volmerange-les-Mines	57
22704	732	\N	Volmunster	57
22705	733	\N	Volstroff	57
22706	734	\N	Voyer	57
22707	736	\N	Vry	57
22708	737	\N	Vulmont	57
22709	738	\N	Waldhouse	57
22710	739	\N	Waldweistroff	57
22711	740	\N	Waldwisse	57
22712	741	\N	Walschbronn	57
22713	742	\N	Walscheid	57
22714	743	\N	Waltembourg	57
22715	745	\N	Wiesviller	57
22716	746	\N	Willerwald	57
22717	747	\N	Wintersbourg	57
22718	748	\N	Wittring	57
22719	749	\N	Vlfling-lès-Bouzonville	57
22720	750	\N	Wlfling-lès-Sarreguemines	57
22721	751	\N	Woippy	57
22722	752	\N	Woustviller	57
22723	753	\N	Wuisse	57
22724	754	\N	Xanrey	57
22725	755	\N	Xocourt	57
22726	756	\N	Xouaxange	57
22727	757	\N	Yutz	57
22728	759	\N	Zarbeling	57
22729	760	\N	Zetting	57
22730	761	\N	Zilling	57
22731	762	\N	Zimming	57
22732	763	\N	Zommange	57
22733	764	\N	Zoufftgen	57
22734	765	\N	Diesen	57
22735	767	\N	Stuckange	57
22736	001	\N	Achun	58
22737	002	\N	Alligny-Cosne	58
22738	003	\N	Alligny-en-Morvan	58
22739	004	\N	Alluy	58
22740	005	\N	Amazy	58
22741	006	\N	Anlezy	58
22742	007	\N	Annay	58
22743	008	\N	Anthien	58
22744	009	\N	Arbourse	58
22745	010	\N	Arleuf	58
22746	011	\N	Armes	58
22747	012	\N	Arquian	58
22748	013	\N	Arthel	58
22749	014	\N	Arzembouy	58
22750	015	\N	Asnan	58
22751	016	\N	Asnois	58
22752	017	\N	Aunay-en-Bazois	58
22753	018	\N	Authiou	58
22754	019	\N	Avrée	58
22755	020	\N	Avril-sur-Loire	58
22756	021	\N	Azy-le-Vif	58
22757	022	\N	Balleray	58
22758	023	\N	Bazoches	58
22759	024	\N	Bazolles	58
22760	025	\N	Béard	58
22761	026	\N	Beaulieu	58
22762	027	\N	Beaumont-la-Ferrière	58
22763	028	\N	Beaumont-Sardolles	58
22764	029	\N	Beuvron	58
22765	030	\N	Biches	58
22766	031	\N	Billy-Chevannes	58
22767	032	\N	Billy-sur-Oisy	58
22768	033	\N	Bitry	58
22769	034	\N	Blismes	58
22770	035	\N	Bona	58
22771	036	\N	Bouhy	58
22772	037	\N	Brassy	58
22773	038	\N	Breugnon	58
22774	039	\N	Brèves	58
22775	040	\N	Brinay	58
22776	041	\N	Brinon-sur-Beuvron	58
22777	042	\N	Bulcy	58
22778	043	\N	Bussy-la-Pesle	58
22779	044	\N	Celle-sur-Loire	58
22780	045	\N	Celle-sur-Nièvre	58
22781	046	\N	Cercy-la-Tour	58
22782	047	\N	Cervon	58
22783	048	\N	Cessy-les-Bois	58
22784	049	\N	Chalaux	58
22785	050	\N	Challement	58
22786	051	\N	Challuy	58
22787	052	\N	Champallement	58
22788	053	\N	Champlemy	58
22789	054	\N	Champlin	58
22790	055	\N	Champvert	58
22791	056	\N	Champvoux	58
22792	057	\N	Chantenay-Saint-Imbert	58
22793	058	\N	Chapelle-Saint-André	58
22794	059	\N	Charité-sur-Loire	58
22795	060	\N	Charrin	58
22796	061	\N	Chasnay	58
22797	062	\N	Château-Chinon(Ville)	58
22798	063	\N	Château-Chinon(Campagne)	58
22799	064	\N	Châteauneuf-Val-de-Bargis	58
22800	065	\N	Châtillon-en-Bazois	58
22801	066	\N	Châtin	58
22802	067	\N	Chaulgnes	58
22803	068	\N	Chaumard	58
22804	069	\N	Chaumot	58
22805	070	\N	Chazeuil	58
22806	071	\N	Chevannes-Changy	58
22807	072	\N	Chevenon	58
22808	073	\N	Chevroches	58
22809	074	\N	Chiddes	58
22810	075	\N	Chitry-les-Mines	58
22811	076	\N	Chougny	58
22812	077	\N	Ciez	58
22813	078	\N	Cizely	58
22814	079	\N	Clamecy	58
22815	080	\N	Collancelle	58
22816	081	\N	Colméry	58
22817	082	\N	Corancy	58
22818	083	\N	Corbigny	58
22819	084	\N	Corvol-d'Embernard	58
22820	085	\N	Corvol-l'Orgueilleux	58
22821	086	\N	Cosne-Cours-sur-Loire	58
22822	087	\N	Cossaye	58
22823	088	\N	Coulanges-lès-Nevers	58
22824	089	\N	Couloutre	58
22825	090	\N	Courcelles	58
22826	092	\N	Crux-la-Ville	58
22827	093	\N	Cuncy-lès-Varzy	58
22828	094	\N	Dampierre-sous-Bouhy	58
22829	095	\N	Decize	58
22830	096	\N	Devay	58
22831	097	\N	Diennes-Aubigny	58
22832	098	\N	Dirol	58
22833	099	\N	Dommartin	58
22834	100	\N	Dompierre-sur-Héry	58
22835	101	\N	Dompierre-sur-Nièvre	58
22836	102	\N	Donzy	58
22837	103	\N	Dornecy	58
22838	104	\N	Dornes	58
22839	105	\N	Druy-Parigny	58
22840	106	\N	Dun-les-Places	58
22841	107	\N	Dun-sur-Grandry	58
22842	108	\N	Empury	58
22843	109	\N	Entrains-sur-Nohain	58
22844	110	\N	Epiry	58
22845	111	\N	Fâchin	58
22846	112	\N	Fermeté	58
22847	113	\N	Fertrève	58
22848	114	\N	Fléty	58
22849	115	\N	Fleury-sur-Loire	58
22850	116	\N	Flez-Cuzy	58
22851	117	\N	Fourchambault	58
22852	118	\N	Fours	58
22853	119	\N	Frasnay-Reugny	58
22854	120	\N	Gâcogne	58
22855	121	\N	Garchizy	58
22856	122	\N	Garchy	58
22857	123	\N	Germenay	58
22858	124	\N	Germigny-sur-Loire	58
22859	125	\N	Gien-sur-Cure	58
22860	126	\N	Gimouille	58
22861	127	\N	Giry	58
22862	128	\N	Glux-en-Glenne	58
22863	129	\N	Gouloux	58
22864	130	\N	Grenois	58
22865	131	\N	Guérigny	58
22866	132	\N	Guipy	58
22867	133	\N	Héry	58
22868	134	\N	Imphy	58
22869	135	\N	Isenay	58
22870	136	\N	Jailly	58
22871	137	\N	Lamenay-sur-Loire	58
22872	138	\N	Langeron	58
22873	139	\N	Lanty	58
22874	140	\N	Larochemillay	58
22875	141	\N	Lavault-de-Frétoy	58
22876	142	\N	Limanton	58
22877	143	\N	Limon	58
22878	144	\N	Livry	58
22879	145	\N	Lormes	58
22880	146	\N	Lucenay-lès-Aix	58
22881	147	\N	Lurcy-le-Bourg	58
22882	148	\N	Luthenay-Uxeloup	58
22883	149	\N	Luzy	58
22884	150	\N	Lys	58
22885	151	\N	Machine	58
22886	152	\N	Magny-Cours	58
22887	153	\N	Magny-Lormes	58
22888	154	\N	Maison-Dieu	58
22889	155	\N	Marche	58
22890	156	\N	Marcy	58
22891	157	\N	Marigny-l'Église	58
22892	158	\N	Mars-sur-Allier	58
22893	159	\N	Marigny-sur-Yonne	58
22894	160	\N	Marzy	58
22895	161	\N	Maux	58
22896	162	\N	Menestreau	58
22897	163	\N	Menou	58
22898	164	\N	Mesves-sur-Loire	58
22899	165	\N	Metz-le-Comte	58
22900	166	\N	Mhère	58
22901	167	\N	Michaugues	58
22902	168	\N	Millay	58
22903	169	\N	Moissy-Moulinot	58
22904	170	\N	Monceaux-le-Comte	58
22905	171	\N	Montapas	58
22906	172	\N	Montambert	58
22907	173	\N	Montaron	58
22908	174	\N	Montenoison	58
22909	175	\N	Mont-et-Marré	58
22910	176	\N	Montigny-aux-Amognes	58
22911	177	\N	Montigny-en-Morvan	58
22912	178	\N	Montigny-sur-Canne	58
22913	179	\N	Montreuillon	58
22914	180	\N	Montsauche-les-Settons	58
22915	181	\N	Moraches	58
22916	182	\N	Moulins-Engilbert	58
22917	183	\N	Mouron-sur-Yonne	58
22918	184	\N	Moussy	58
22919	185	\N	Moux-en-Morvan	58
22920	186	\N	Murlin	58
22921	187	\N	Myennes	58
22922	188	\N	Nannay	58
22923	189	\N	Narcy	58
22924	190	\N	Neuffontaines	58
22925	191	\N	Neuilly	58
22926	192	\N	Neuville-lès-Decize	58
22927	193	\N	Neuvy-sur-Loire	58
22928	194	\N	Nevers	58
22929	195	\N	Nocle-Maulaix	58
22930	196	\N	Nolay	58
22931	197	\N	Nuars	58
22932	198	\N	Oisy	58
22933	199	\N	Onlay	58
22934	200	\N	Ouagne	58
22935	201	\N	Oudan	58
22936	202	\N	Ougny	58
22937	203	\N	Oulon	58
22938	204	\N	Ourouër	58
22939	205	\N	Ouroux-en-Morvan	58
22940	206	\N	Parigny-la-Rose	58
22941	207	\N	Parigny-les-Vaux	58
22942	208	\N	Pazy	58
22943	209	\N	Perroy	58
22944	210	\N	Planchez	58
22945	211	\N	Poil	58
22946	212	\N	Poiseux	58
22947	213	\N	Pougny	58
22948	214	\N	Pougues-les-Eaux	58
22949	215	\N	Pouilly-sur-Loire	58
22950	216	\N	Pouques-Lormes	58
22951	217	\N	Pousseaux	58
22952	218	\N	Prémery	58
22953	219	\N	Préporché	58
22954	220	\N	Raveau	58
22955	221	\N	Rémilly	58
22956	222	\N	Rix	58
22957	223	\N	Rouy	58
22958	224	\N	Ruages	58
22959	225	\N	Saincaize-Meauce	58
22960	226	\N	Saint-Agnan	58
22961	227	\N	Saint-Amand-en-Puisaye	58
22962	228	\N	Saint-Andelain	58
22963	229	\N	Saint-André-en-Morvan	58
22964	230	\N	Saint-Aubin-des-Chaumes	58
22965	231	\N	Saint-Aubin-les-Forges	58
22966	232	\N	Saint-Benin-d'Azy	58
22967	233	\N	Saint-Benin-des-Bois	58
22968	234	\N	Saint-Bonnot	58
22969	235	\N	Saint-Brisson	58
22970	236	\N	Sainte-Colombe-des-Bois	58
22971	237	\N	Saint-Didier	58
22972	238	\N	Saint-Éloi	58
22973	239	\N	Saint-Firmin	58
22974	240	\N	Saint-Franchy	58
22975	241	\N	Saint-Germain-Chassenay	58
22976	242	\N	Saint-Germain-des-Bois	58
22977	243	\N	Saint-Gratien-Savigny	58
22978	244	\N	Saint-Hilaire-en-Morvan	58
22979	245	\N	Saint-Hilaire-Fontaine	58
22980	246	\N	Saint-Honoré-les-Bains	58
22981	247	\N	Saint-Jean-aux-Amognes	58
22982	248	\N	Saint-Laurent-l'Abbaye	58
22983	249	\N	Saint-Léger-de-Fougeret	58
22984	250	\N	Saint-Léger-des-Vignes	58
22985	251	\N	Saint-Loup	58
22986	252	\N	Saint-Malo-en-Donziois	58
22987	253	\N	Sainte-Marie	58
22988	254	\N	Saint-Martin-d'Heuille	58
22989	255	\N	Saint-Martin-du-Puy	58
22990	256	\N	Saint-Martin-sur-Nohain	58
22991	257	\N	Saint-Maurice	58
22992	258	\N	Saint-Ouen-sur-Loire	58
22993	259	\N	Saint-Parize-en-Viry	58
22994	260	\N	Saint-Parize-le-Châtel	58
22995	261	\N	Saint-Père	58
22996	262	\N	Saint-Péreuse	58
22997	263	\N	Saint-Pierre-du-Mont	58
22998	264	\N	Saint-Pierre-le-Moûtier	58
22999	265	\N	Saint-Quentin-sur-Nohain	58
23000	266	\N	Saint-Révérien	58
23001	267	\N	Saint-Saulge	58
23002	268	\N	Saint-Seine	58
23003	269	\N	Saint-Sulpice	58
23004	270	\N	Saint-Vérain	58
23005	271	\N	Saizy	58
23006	272	\N	Sardy-lès-Épiry	58
23007	273	\N	Sauvigny-les-Bois	58
23008	274	\N	Savigny-Poil-Fol	58
23009	275	\N	Saxi-Bourdon	58
23010	276	\N	Sémelay	58
23011	277	\N	Sermages	58
23012	278	\N	Sermoise-sur-Loire	58
23013	279	\N	Sichamps	58
23014	280	\N	Sougy-sur-Loire	58
23015	281	\N	Suilly-la-Tour	58
23016	282	\N	Surgy	58
23017	283	\N	Taconnay	58
23018	284	\N	Talon	58
23019	285	\N	Tamnay-en-Bazois	58
23020	286	\N	Tannay	58
23021	287	\N	Tazilly	58
23022	288	\N	Teigny	58
23023	289	\N	Ternant	58
23024	290	\N	Thaix	58
23025	291	\N	Thianges	58
23026	292	\N	Tintury	58
23027	293	\N	Toury-Lurcy	58
23028	294	\N	Toury-sur-Jour	58
23029	295	\N	Tracy-sur-Loire	58
23030	296	\N	Tresnay	58
23031	297	\N	Trois-Vèvres	58
23032	298	\N	Tronsanges	58
23033	299	\N	Trucy-l'Orgueilleux	58
23034	300	\N	Urzy	58
23035	301	\N	Vandenesse	58
23036	302	\N	Varennes-lès-Narcy	58
23037	303	\N	Varennes-Vauzelles	58
23038	304	\N	Varzy	58
23039	305	\N	Vauclaix	58
23040	306	\N	Verneuil	58
23041	307	\N	Vielmanay	58
23042	308	\N	Vignol	58
23043	309	\N	Villapourçon	58
23044	310	\N	Villiers-le-Sec	58
23045	311	\N	Ville-Langy	58
23046	312	\N	Villiers-sur-Yonne	58
23047	313	\N	Vitry-Laché	58
23048	001	\N	Abancourt	59
23049	002	\N	Abscon	59
23050	003	\N	Aibes	59
23051	004	\N	Aix	59
23052	005	\N	Allennes-les-Marais	59
23053	006	\N	Amfroipret	59
23054	007	\N	Anhiers	59
23055	008	\N	Aniche	59
23056	009	\N	Villeneuve-d'Ascq	59
23057	010	\N	Anneux	59
23058	011	\N	Annullin	59
23059	012	\N	Anor	59
23060	013	\N	Anstaing	59
23061	014	\N	Anzin	59
23062	015	\N	Arleux	59
23063	016	\N	Armbouts-Cappel	59
23064	017	\N	Armentières	59
23065	018	\N	Arnèke	59
23066	019	\N	Artres	59
23067	021	\N	Assevent	59
23068	022	\N	Attiches	59
23069	023	\N	Aubencheul-au-Bac	59
23070	024	\N	Auberchicourt	59
23071	025	\N	Aubers	59
23072	026	\N	Aubigny-au-Bac	59
23073	027	\N	Aubry-du-Hainaut	59
23074	028	\N	Auby	59
23075	029	\N	Auchy-lez-Orchies	59
23076	031	\N	Audignies	59
23077	032	\N	Aulnoy-lez-Valenciennes	59
23078	033	\N	Aulnoye-Aymeries	59
23079	034	\N	Avelin	59
23080	035	\N	Avesnelles	59
23081	036	\N	Avesnes-sur-Helpe	59
23082	037	\N	Avesnes-les-Aubert	59
23083	038	\N	Avesnes-le-Sec	59
23084	039	\N	Awoingt	59
23085	041	\N	Bachant	59
23086	042	\N	Bachy	59
23087	043	\N	Bailleul	59
23088	044	\N	Baisieux	59
23089	045	\N	Baives	59
23090	046	\N	Bambecque	59
23091	047	\N	Banteux	59
23092	048	\N	Bantigny	59
23093	049	\N	Bantouzelle	59
23094	050	\N	Bas-Lieu	59
23095	051	\N	Bassée	59
23096	052	\N	Bauvin	59
23097	053	\N	Bavay	59
23098	054	\N	Bavinchove	59
23099	055	\N	Bazuel	59
23100	056	\N	Beaucamps-Ligny	59
23101	057	\N	Beaudignies	59
23102	058	\N	Beaufort	59
23103	059	\N	Beaumont-en-Cambrésis	59
23104	060	\N	Beaurain	59
23105	061	\N	Beaurepaire-sur-Sambre	59
23106	062	\N	Beaurieux	59
23107	063	\N	Beauvois-en-Cambrésis	59
23108	064	\N	Bellaing	59
23109	065	\N	Bellignies	59
23110	066	\N	Bérelles	59
23111	067	\N	Bergues	59
23112	068	\N	Berlaimont	59
23113	069	\N	Bermerain	59
23114	070	\N	Bermeries	59
23115	071	\N	Bersée	59
23116	072	\N	Bersillies	59
23117	073	\N	Berthen	59
23118	074	\N	Bertry	59
23119	075	\N	Béthencourt	59
23120	076	\N	Bettignies	59
23121	077	\N	Bettrechies	59
23122	078	\N	Beugnies	59
23123	079	\N	Beuvrages	59
23124	080	\N	Beuvry-la-Forêt	59
23125	081	\N	Bévillers	59
23126	082	\N	Bierne	59
23127	083	\N	Bissezeele	59
23128	084	\N	Blaringhem	59
23129	085	\N	Blécourt	59
23130	086	\N	Boeschepe	59
23131	087	\N	Boëseghem	59
23132	088	\N	Bois-Grenier	59
23133	089	\N	Bollezeele	59
23134	090	\N	Bondues	59
23135	091	\N	Borre	59
23136	092	\N	Bouchain	59
23137	093	\N	Boulogne-sur-Helpe	59
23138	094	\N	Bourbourg	59
23139	096	\N	Bourghelles	59
23140	097	\N	Boursies	59
23141	098	\N	Bousbecque	59
23142	099	\N	Bousies	59
23143	100	\N	Bousignies	59
23144	101	\N	Bousignies-sur-Roc	59
23145	102	\N	Boussières-en-Cambrésis	59
23146	103	\N	Boussières-sur-Sambre	59
23147	104	\N	Boussois	59
23148	105	\N	Bouvignies	59
23149	106	\N	Bouvines	59
23150	107	\N	Bray-Dunes	59
23151	108	\N	Briastre	59
23152	109	\N	Brillon	59
23153	110	\N	Brouckerque	59
23154	111	\N	Broxeele	59
23155	112	\N	Bruay-sur-l'Escaut	59
23156	113	\N	Bruille-lez-Marchiennes	59
23157	114	\N	Bruille-Saint-Amand	59
23158	115	\N	Brunémont	59
23159	116	\N	Bry	59
23160	117	\N	Bugnicourt	59
23161	118	\N	Busigny	59
23162	119	\N	Buysscheure	59
23163	120	\N	Caëstre	59
23164	121	\N	Cagnoncles	59
23165	122	\N	Cambrai	59
23166	123	\N	Camphin-en-Carembault	59
23167	124	\N	Camphin-en-Pévèle	59
23168	125	\N	Cantaing-sur-Escaut	59
23169	126	\N	Cantin	59
23170	127	\N	Capelle	59
23171	128	\N	Capinghem	59
23172	129	\N	Cappelle-en-Pévèle	59
23173	130	\N	Cappelle-Brouck	59
23174	131	\N	Cappelle-la-Grande	59
23175	132	\N	Carnières	59
23176	133	\N	Carnin	59
23177	134	\N	Cartignies	59
23178	135	\N	Cassel	59
23179	136	\N	Cateau-Cambrésis	59
23180	137	\N	Catillon-sur-Sambre	59
23181	138	\N	Cattenières	59
23182	139	\N	Caudry	59
23183	140	\N	Caullery	59
23184	141	\N	Cauroir	59
23185	142	\N	Cerfontaine	59
23186	143	\N	Chapelle-d'Armentières	59
23187	144	\N	Château-l'Abbaye	59
23188	145	\N	Chemy	59
23189	146	\N	Chéreng	59
23190	147	\N	Choisies	59
23191	148	\N	Clairfayts	59
23192	149	\N	Clary	59
23193	150	\N	Cobrieux	59
23194	151	\N	Colleret	59
23195	152	\N	Comines	59
23196	153	\N	Condé-sur-l'Escaut	59
23197	154	\N	Coudekerque	59
23198	155	\N	Coudekerque-Branche	59
23199	156	\N	Courchelettes	59
23200	157	\N	Cousolre	59
23201	158	\N	Coutiches	59
23202	159	\N	Craywick	59
23203	160	\N	Crespin	59
23204	161	\N	Crèvecur-sur-l'Escaut	59
23205	162	\N	Crochte	59
23206	163	\N	Croix	59
23207	164	\N	Croix-Caluyau	59
23208	165	\N	Cuincy	59
23209	166	\N	Curgies	59
23210	167	\N	Cuvillers	59
23211	168	\N	Cysoing	59
23212	169	\N	Damousies	59
23213	170	\N	Dechy	59
23214	171	\N	Dehéries	59
23215	172	\N	Denain	59
23216	173	\N	Deûlémont	59
23217	174	\N	Dimechaux	59
23218	175	\N	Dimont	59
23219	176	\N	Doignies	59
23220	177	\N	Dompierre-sur-Helpe	59
23221	178	\N	Douai	59
23222	179	\N	Douchy-les-Mines	59
23223	180	\N	Doulieu	59
23224	181	\N	Dourlers	59
23225	182	\N	Drincham	59
23226	183	\N	Dunkerque	59
23227	184	\N	Ebblinghem	59
23228	185	\N	Écaillon	59
23229	186	\N	Eccles	59
23230	187	\N	Éclaibes	59
23231	188	\N	Écuélin	59
23232	189	\N	Eecke	59
23233	190	\N	Élesmes	59
23234	191	\N	Élincourt	59
23235	192	\N	Émerchicourt	59
23236	193	\N	Emmerin	59
23237	194	\N	Englefontaine	59
23238	195	\N	Englos	59
23239	196	\N	Ennetières-en-Weppes	59
23240	197	\N	Ennevelin	59
23241	198	\N	Eppe-Sauvage	59
23242	199	\N	Erchin	59
23243	200	\N	Eringhem	59
23244	201	\N	Erquinghem-le-Sec	59
23245	202	\N	Erquinghem-Lys	59
23246	203	\N	Erre	59
23247	204	\N	Escarmain	59
23248	205	\N	Escaudain	59
23249	206	\N	Escauduvres	59
23250	207	\N	Escautpont	59
23251	208	\N	Escobecques	59
23252	209	\N	Esnes	59
23253	210	\N	Esquelbecq	59
23254	211	\N	Esquerchin	59
23255	212	\N	Estaires	59
23256	213	\N	Estourmel	59
23257	214	\N	Estrées	59
23258	215	\N	Estreux	59
23259	216	\N	Eswars	59
23260	217	\N	Eth	59
23261	218	\N	Étrungt	59
23262	219	\N	Estrun	59
23263	220	\N	Faches-Thumesnil	59
23264	221	\N	Famars	59
23265	222	\N	Faumont	59
23266	223	\N	Favril	59
23267	224	\N	Féchain	59
23268	225	\N	Feignies	59
23269	226	\N	Felleries	59
23270	227	\N	Fenain	59
23271	228	\N	Férin	59
23272	229	\N	Féron	59
23273	230	\N	Ferrière-la-Grande	59
23274	231	\N	Ferrière-la-Petite	59
23275	232	\N	Flamengrie	59
23276	233	\N	Flaumont-Waudrechies	59
23277	234	\N	Flers-en-Escrebieux	59
23278	236	\N	Flesquières	59
23279	237	\N	Flêtre	59
23280	238	\N	Flines-lès-Mortagne	59
23281	239	\N	Flines-lez-Raches	59
23282	240	\N	Floursies	59
23283	241	\N	Floyon	59
23284	242	\N	Fontaine-au-Bois	59
23285	243	\N	Fontaine-au-Pire	59
23286	244	\N	Fontaine-Notre-Dame	59
23287	246	\N	Forest-en-Cambrésis	59
23288	247	\N	Forest-sur-Marque	59
23289	248	\N	Fort-Mardyck	59
23290	249	\N	Fourmies	59
23291	250	\N	Fournes-en-Weppes	59
23292	251	\N	Frasnoy	59
23293	252	\N	Frelinghien	59
23294	253	\N	Fresnes-sur-Escaut	59
23295	254	\N	Fressain	59
23296	255	\N	Fressies	59
23297	256	\N	Fretin	59
23298	257	\N	Fromelles	59
23299	258	\N	Genech	59
23300	259	\N	Ghissignies	59
23301	260	\N	Ghyvelde	59
23302	261	\N	Glageon	59
23303	262	\N	Godewaersvelde	59
23304	263	\N	Gulzin	59
23305	264	\N	Gognies-Chaussée	59
23306	265	\N	Gommegnies	59
23307	266	\N	Gondecourt	59
23308	267	\N	Gonnelieu	59
23309	268	\N	Gorgue	59
23310	269	\N	Gouzeaucourt	59
23311	270	\N	Grand-Fayt	59
23312	271	\N	Grande-Synthe	59
23313	272	\N	Grand-Fort-Philippe	59
23314	273	\N	Gravelines	59
23315	274	\N	Groise	59
23316	275	\N	Gruson	59
23317	276	\N	Guesnain	59
23318	277	\N	Gussignies	59
23319	278	\N	Hallennes-lez-Haubourdin	59
23320	279	\N	Halluin	59
23321	280	\N	Hamel	59
23322	281	\N	Hantay	59
23323	282	\N	Hardifort	59
23324	283	\N	Hargnies	59
23325	284	\N	Hasnon	59
23326	285	\N	Haspres	59
23327	286	\N	Haubourdin	59
23328	287	\N	Haucourt-en-Cambrésis	59
23329	288	\N	Haulchin	59
23330	289	\N	Haussy	59
23331	290	\N	Haut-Lieu	59
23332	291	\N	Hautmont	59
23333	292	\N	Haveluy	59
23334	293	\N	Haverskerque	59
23335	294	\N	Haynecourt	59
23336	295	\N	Hazebrouck	59
23337	296	\N	Hecq	59
23338	297	\N	Hélesmes	59
23339	299	\N	Hem	59
23340	300	\N	Hem-Lenglet	59
23341	301	\N	Hergnies	59
23342	302	\N	Hérin	59
23343	303	\N	Herlies	59
23344	304	\N	Herrin	59
23345	305	\N	Herzeele	59
23346	306	\N	Hestrud	59
23347	307	\N	Holque	59
23348	308	\N	Hondeghem	59
23349	309	\N	Hondschoote	59
23350	310	\N	Hon-Hergies	59
23351	311	\N	Honnechy	59
23352	312	\N	Honnecourt-sur-Escaut	59
23353	313	\N	Hordain	59
23354	314	\N	Hornaing	59
23355	315	\N	Houdain-lez-Bavay	59
23356	316	\N	Houplin-Ancoisne	59
23357	317	\N	Houplines	59
23358	318	\N	Houtkerque	59
23359	319	\N	Hoymille	59
23360	320	\N	Illies	59
23361	321	\N	Inchy	59
23362	322	\N	Iwuy	59
23363	323	\N	Jenlain	59
23364	324	\N	Jeumont	59
23365	325	\N	Jolimetz	59
23366	326	\N	Killem	59
23367	327	\N	Lallaing	59
23368	328	\N	Lambersart	59
23369	329	\N	Lambres-lez-Douai	59
23370	330	\N	Landas	59
23371	331	\N	Landrecies	59
23372	332	\N	Lannoy	59
23373	333	\N	Larouillies	59
23374	334	\N	Lauwin-Planque	59
23375	335	\N	Lecelles	59
23376	336	\N	Lécluse	59
23377	337	\N	Lederzeele	59
23378	338	\N	Ledringhem	59
23379	339	\N	Leers	59
23380	340	\N	Leffrinckoucke	59
23381	341	\N	Lesdain	59
23382	342	\N	Lez-Fontaine	59
23383	343	\N	Lesquin	59
23384	344	\N	Leval	59
23385	345	\N	Lewarde	59
23386	346	\N	Lezennes	59
23387	347	\N	Liessies	59
23388	348	\N	Lieu-Saint-Amand	59
23389	349	\N	Ligny-en-Cambrésis	59
23390	350	\N	Lille	59
23391	351	\N	Limont-Fontaine	59
23392	352	\N	Linselles	59
23393	353	\N	Locquignol	59
23394	354	\N	Loffre	59
23395	356	\N	Lompret	59
23396	357	\N	Longueville	59
23397	358	\N	Looberghe	59
23398	359	\N	Loon-Plage	59
23399	360	\N	Loos	59
23400	361	\N	Lourches	59
23401	363	\N	Louvignies-Quesnoy	59
23402	364	\N	Louvil	59
23403	365	\N	Louvroil	59
23404	366	\N	Lynde	59
23405	367	\N	Lys-lez-Lannoy	59
23406	368	\N	Madeleine	59
23407	369	\N	Maing	59
23408	370	\N	Mairieux	59
23409	371	\N	Maisnil	59
23410	372	\N	Malincourt	59
23411	374	\N	Marbaix	59
23412	375	\N	Marchiennes	59
23413	377	\N	Marcoing	59
23414	378	\N	Marcq-en-Barul	59
23415	379	\N	Marcq-en-Ostrevent	59
23416	381	\N	Maresches	59
23417	382	\N	Maretz	59
23418	383	\N	Marly	59
23419	384	\N	Maroilles	59
23420	385	\N	Marpent	59
23421	386	\N	Marquette-lez-Lille	59
23422	387	\N	Marquette-en-Ostrevant	59
23423	388	\N	Marquillies	59
23424	389	\N	Masnières	59
23425	390	\N	Masny	59
23426	391	\N	Mastaing	59
23427	392	\N	Maubeuge	59
23428	393	\N	Maulde	59
23429	394	\N	Maurois	59
23430	395	\N	Mazinghien	59
23431	396	\N	Mecquignies	59
23432	397	\N	Merckeghem	59
23433	398	\N	Mérignies	59
23434	399	\N	Merris	59
23435	400	\N	Merville	59
23436	401	\N	Méteren	59
23437	402	\N	Millam	59
23438	403	\N	Millonfosse	59
23439	404	\N	Moëres	59
23440	405	\N	Muvres	59
23441	406	\N	Monceau-Saint-Waast	59
23442	407	\N	Monchaux-sur-Écaillon	59
23443	408	\N	Moncheaux	59
23444	409	\N	Monchecourt	59
23445	410	\N	Mons-en-Barul	59
23446	411	\N	Mons-en-Pévèle	59
23447	412	\N	Montay	59
23448	413	\N	Montigny-en-Cambrésis	59
23449	414	\N	Montigny-en-Ostrevent	59
23450	415	\N	Montrécourt	59
23451	416	\N	Morbecque	59
23452	418	\N	Mortagne-du-Nord	59
23453	419	\N	Mouchin	59
23454	420	\N	Moustier-en-Fagne	59
23455	421	\N	Mouvaux	59
23456	422	\N	Naves	59
23457	423	\N	Neuf-Berquin	59
23458	424	\N	Neuf-Mesnil	59
23459	425	\N	Neuville-en-Avesnois	59
23460	426	\N	Neuville-en-Ferrain	59
23461	427	\N	Neuville	59
23462	428	\N	Neuville-Saint-Rémy	59
23463	429	\N	Neuville-sur-Escaut	59
23464	430	\N	Neuvilly	59
23465	431	\N	Nieppe	59
23466	432	\N	Niergnies	59
23467	433	\N	Nieurlet	59
23468	434	\N	Nivelle	59
23469	435	\N	Nomain	59
23470	436	\N	Noordpeene	59
23471	437	\N	Noyelles-lès-Seclin	59
23472	438	\N	Noyelles-sur-Escaut	59
23473	439	\N	Noyelles-sur-Sambre	59
23474	440	\N	Noyelles-sur-Selle	59
23475	441	\N	Obies	59
23476	442	\N	Obrechies	59
23477	443	\N	Ochtezeele	59
23478	444	\N	Odomez	59
23479	445	\N	Ohain	59
23480	446	\N	Oisy	59
23481	447	\N	Onnaing	59
23482	448	\N	Oost-Cappel	59
23483	449	\N	Orchies	59
23484	450	\N	Ors	59
23485	451	\N	Orsinval	59
23486	452	\N	Ostricourt	59
23487	453	\N	Oudezeele	59
23488	454	\N	Oxelaëre	59
23489	455	\N	Paillencourt	59
23490	456	\N	Pecquencourt	59
23491	457	\N	Pérenchies	59
23492	458	\N	Péronne-en-Mélantois	59
23493	459	\N	Petite-Forêt	59
23494	461	\N	Petit-Fayt	59
23495	462	\N	Phalempin	59
23496	463	\N	Pitgam	59
23497	464	\N	Poix-du-Nord	59
23498	465	\N	Pommereuil	59
23499	466	\N	Pont-à-Marcq	59
23500	467	\N	Pont-sur-Sambre	59
23501	468	\N	Potelle	59
23502	469	\N	Pradelles	59
23503	470	\N	Prémesques	59
23504	471	\N	Préseau	59
23505	472	\N	Preux-au-Bois	59
23506	473	\N	Preux-au-Sart	59
23507	474	\N	Prisches	59
23508	475	\N	Prouvy	59
23509	476	\N	Proville	59
23510	477	\N	Provin	59
23511	478	\N	Quaëdypre	59
23512	479	\N	Quarouble	59
23513	480	\N	Quérénaing	59
23514	481	\N	Quesnoy	59
23515	482	\N	Quesnoy-sur-Deûle	59
23516	483	\N	Quiévelon	59
23517	484	\N	Quiévrechain	59
23518	485	\N	Quiévy	59
23519	486	\N	Râches	59
23520	487	\N	Radinghem-en-Weppes	59
23521	488	\N	Raillencourt-Sainte-Olle	59
23522	489	\N	Raimbeaucourt	59
23523	490	\N	Rainsars	59
23524	491	\N	Raismes	59
23525	492	\N	Ramillies	59
23526	493	\N	Ramousies	59
23527	494	\N	Raucourt-au-Bois	59
23528	495	\N	Recquignies	59
23529	496	\N	Rejet-de-Beaulieu	59
23530	497	\N	Renescure	59
23531	498	\N	Reumont	59
23532	499	\N	Rexpoëde	59
23533	500	\N	Ribécourt-la-Tour	59
23534	501	\N	Rieulay	59
23535	502	\N	Rieux-en-Cambrésis	59
23536	503	\N	Robersart	59
23537	504	\N	Rulx	59
23538	505	\N	Rombies-et-Marchipont	59
23539	506	\N	Romeries	59
23540	507	\N	Ronchin	59
23541	508	\N	Roncq	59
23542	509	\N	Roost-Warendin	59
23543	511	\N	Rosult	59
23544	512	\N	Roubaix	59
23545	513	\N	Roucourt	59
23546	514	\N	Rousies	59
23547	515	\N	Rouvignies	59
23548	516	\N	Rubrouck	59
23549	517	\N	Rues-des-Vignes	59
23550	518	\N	Ruesnes	59
23551	519	\N	Rumegies	59
23552	520	\N	Rumilly-en-Cambrésis	59
23553	521	\N	Sailly-lez-Cambrai	59
23554	522	\N	Sailly-lez-Lannoy	59
23555	523	\N	Sainghin-en-Mélantois	59
23556	524	\N	Sainghin-en-Weppes	59
23557	525	\N	Sains-du-Nord	59
23558	526	\N	Saint-Amand-les-Eaux	59
23559	527	\N	Saint-André-lez-Lille	59
23560	528	\N	Saint-Aubert	59
23561	529	\N	Saint-Aubin	59
23562	530	\N	Saint-Aybert	59
23563	531	\N	Saint-Benin	59
23564	532	\N	Saint-Georges-sur-l'Aa	59
23565	533	\N	Saint-Hilaire-lez-Cambrai	59
23566	534	\N	Saint-Hilaire-sur-Helpe	59
23567	535	\N	Saint-Jans-Cappel	59
23568	536	\N	Sainte-Marie-Cappel	59
23569	537	\N	Saint-Martin-sur-Écaillon	59
23570	538	\N	Saint-Momelin	59
23571	539	\N	Saint-Pierre-Brouck	59
23572	540	\N	Saint-Pol-sur-Mer	59
23573	541	\N	Saint-Python	59
23574	542	\N	Saint-Remy-Chaussée	59
23575	543	\N	Saint-Remy-du-Nord	59
23576	544	\N	Saint-Saulve	59
23577	545	\N	Saint-Souplet	59
23578	546	\N	Saint-Sylvestre-Cappel	59
23579	547	\N	Saint-Vaast-en-Cambrésis	59
23580	548	\N	Saint-Waast	59
23581	549	\N	Salesches	59
23582	550	\N	Salomé	59
23583	551	\N	Saméon	59
23584	552	\N	Sancourt	59
23585	553	\N	Santes	59
23586	554	\N	Sars-et-Rosières	59
23587	555	\N	Sars-Poteries	59
23588	556	\N	Sassegnies	59
23589	557	\N	Saultain	59
23590	558	\N	Saulzoir	59
23591	559	\N	Sebourg	59
23592	560	\N	Seclin	59
23593	562	\N	Sémeries	59
23594	563	\N	Semousies	59
23595	564	\N	Sentinelle	59
23596	565	\N	Sepmeries	59
23597	566	\N	Sequedin	59
23598	567	\N	Séranvillers-Forenville	59
23599	568	\N	Sercus	59
23600	569	\N	Sin-le-Noble	59
23601	570	\N	Socx	59
23602	571	\N	Solesmes	59
23603	572	\N	Solre-le-Château	59
23604	573	\N	Solrinnes	59
23605	574	\N	Somain	59
23606	575	\N	Sommaing	59
23607	576	\N	Spycker	59
23608	577	\N	Staple	59
23609	578	\N	Steenbecque	59
23610	579	\N	Steene	59
23611	580	\N	Steenvoorde	59
23612	581	\N	Steenwerck	59
23613	582	\N	Strazeele	59
23614	583	\N	Taisnières-en-Thiérache	59
23615	584	\N	Taisnières-sur-Hon	59
23616	585	\N	Templemars	59
23617	586	\N	Templeuve	59
23618	587	\N	Terdeghem	59
23619	588	\N	Téteghem	59
23620	589	\N	Thiant	59
23621	590	\N	Thiennes	59
23622	591	\N	Thivencelle	59
23623	592	\N	Thumeries	59
23624	593	\N	Thun-l'Évêque	59
23625	594	\N	Thun-Saint-Amand	59
23626	595	\N	Thun-Saint-Martin	59
23627	596	\N	Tilloy-lez-Marchiennes	59
23628	597	\N	Tilloy-lez-Cambrai	59
23629	598	\N	Toufflers	59
23630	599	\N	Tourcoing	59
23631	600	\N	Tourmignies	59
23632	601	\N	Trélon	59
23633	602	\N	Tressin	59
23634	603	\N	Trith-Saint-Léger	59
23635	604	\N	Troisvilles	59
23636	605	\N	Uxem	59
23637	606	\N	Valenciennes	59
23638	607	\N	Vendegies-au-Bois	59
23639	608	\N	Vendegies-sur-Écaillon	59
23640	609	\N	Vendeville	59
23641	610	\N	Verchain-Maugré	59
23642	611	\N	Verlinghem	59
23643	612	\N	Vertain	59
23644	613	\N	Vicq	59
23645	614	\N	Viesly	59
23646	615	\N	Vieux-Berquin	59
23647	616	\N	Vieux-Condé	59
23648	617	\N	Vieux-Mesnil	59
23649	618	\N	Vieux-Reng	59
23650	619	\N	Villereau	59
23651	620	\N	Villers-au-Tertre	59
23652	622	\N	Villers-en-Cauchies	59
23653	623	\N	Villers-Guislain	59
23654	624	\N	Villers-Outréaux	59
23655	625	\N	Villers-Plouich	59
23656	626	\N	Villers-Pol	59
23657	627	\N	Villers-Sire-Nicole	59
23658	628	\N	Volckerinckhove	59
23659	629	\N	Vred	59
23660	630	\N	Wahagnies	59
23661	631	\N	Walincourt-Selvigny	59
23662	632	\N	Wallers	59
23663	633	\N	Wallers-Trélon	59
23664	634	\N	Wallon-Cappel	59
23665	635	\N	Wambaix	59
23666	636	\N	Wambrechies	59
23667	637	\N	Wandignies-Hamage	59
23668	638	\N	Wannehain	59
23669	639	\N	Wargnies-le-Grand	59
23670	640	\N	Wargnies-le-Petit	59
23671	641	\N	Warhem	59
23672	642	\N	Warlaing	59
23673	643	\N	Warneton	59
23674	645	\N	Wasnes-au-Bac	59
23675	646	\N	Wasquehal	59
23676	647	\N	Watten	59
23677	648	\N	Wattignies	59
23678	649	\N	Wattignies-la-Victoire	59
23679	650	\N	Wattrelos	59
23680	651	\N	Wavrechain-sous-Denain	59
23681	652	\N	Wavrechain-sous-Faulx	59
23682	653	\N	Wavrin	59
23683	654	\N	Waziers	59
23684	655	\N	Wemaers-Cappel	59
23685	656	\N	Wervicq-Sud	59
23686	657	\N	West-Cappel	59
23687	658	\N	Wicres	59
23688	659	\N	Wignehies	59
23689	660	\N	Willems	59
23690	661	\N	Willies	59
23691	662	\N	Winnezeele	59
23692	663	\N	Wormhout	59
23693	664	\N	Wulverdinghe	59
23694	665	\N	Wylder	59
23695	666	\N	Zegerscappel	59
23696	667	\N	Zermezeele	59
23697	668	\N	Zuydcoote	59
23698	669	\N	Zuytpeene	59
23699	670	\N	Don	59
23700	001	\N	Abancourt	60
23701	002	\N	Abbecourt	60
23702	003	\N	Abbeville-Saint-Lucien	60
23703	004	\N	Achy	60
23704	005	\N	Acy-en-Multien	60
23705	006	\N	Ageux	60
23706	007	\N	Agnetz	60
23707	008	\N	Airion	60
23708	009	\N	Allonne	60
23709	010	\N	Amblainville	60
23710	011	\N	Amy	60
23711	012	\N	Andeville	60
23712	013	\N	Angicourt	60
23713	014	\N	Angivillers	60
23714	015	\N	Angy	60
23715	016	\N	Ansacq	60
23716	017	\N	Ansauvillers	60
23717	018	\N	Anserville	60
23718	019	\N	Antheuil-Portes	60
23719	020	\N	Antilly	60
23720	021	\N	Appilly	60
23721	022	\N	Apremont	60
23722	023	\N	Armancourt	60
23723	024	\N	Arsy	60
23724	025	\N	Attichy	60
23725	026	\N	Auchy-la-Montagne	60
23726	027	\N	Auger-Saint-Vincent	60
23727	028	\N	Aumont-en-Halatte	60
23728	029	\N	Auneuil	60
23729	030	\N	Auteuil	60
23730	031	\N	Autheuil-en-Valois	60
23731	032	\N	Autrêches	60
23732	033	\N	Avilly-Saint-Léonard	60
23733	034	\N	Avrechy	60
23734	035	\N	Avricourt	60
23735	036	\N	Avrigny	60
23736	037	\N	Babuf	60
23737	038	\N	Bachivillers	60
23738	039	\N	Bacouël	60
23739	040	\N	Bailleul-le-Soc	60
23740	041	\N	Bailleul-sur-Thérain	60
23741	042	\N	Bailleval	60
23742	043	\N	Bailly	60
23743	044	\N	Balagny-sur-Thérain	60
23744	045	\N	Barbery	60
23745	046	\N	Bargny	60
23746	047	\N	Baron	60
23747	048	\N	Baugy	60
23748	049	\N	Bazancourt	60
23749	050	\N	Bazicourt	60
23750	051	\N	Beaudéduit	60
23751	052	\N	Beaugies-sous-Bois	60
23752	053	\N	Beaulieu-les-Fontaines	60
23753	054	\N	Beaumont-les-Nonains	60
23754	055	\N	Beaurains-lès-Noyon	60
23755	056	\N	Beaurepaire	60
23756	057	\N	Beauvais	60
23757	058	\N	Beauvoir	60
23758	059	\N	Béhéricourt	60
23759	060	\N	Belle-Église	60
23760	061	\N	Belloy	60
23761	062	\N	Berlancourt	60
23762	063	\N	Berneuil-en-Bray	60
23763	064	\N	Berneuil-sur-Aisne	60
23764	065	\N	Berthecourt	60
23765	066	\N	Béthancourt-en-Valois	60
23766	067	\N	Béthisy-Saint-Martin	60
23767	068	\N	Béthisy-Saint-Pierre	60
23768	069	\N	Betz	60
23769	070	\N	Bienville	60
23770	071	\N	Biermont	60
23771	072	\N	Bitry	60
23772	073	\N	Blacourt	60
23773	074	\N	Blaincourt-lès-Précy	60
23774	075	\N	Blancfossé	60
23775	076	\N	Blargies	60
23776	077	\N	Blicourt	60
23777	078	\N	Blincourt	60
23778	079	\N	Boissy-Fresnoy	60
23779	080	\N	Boissy-le-Bois	60
23780	081	\N	Bonlier	60
23781	082	\N	Bonneuil-les-Eaux	60
23782	083	\N	Bonneuil-en-Valois	60
23783	084	\N	Bonnières	60
23784	085	\N	Bonvillers	60
23785	086	\N	Boran-sur-Oise	60
23786	087	\N	Borest	60
23787	088	\N	Bornel	60
23788	089	\N	Boubiers	60
23789	090	\N	Bouconvillers	60
23790	091	\N	Bouillancy	60
23791	092	\N	Boullarre	60
23792	093	\N	Boulogne-la-Grasse	60
23793	094	\N	Boursonne	60
23794	095	\N	Boury-en-Vexin	60
23795	096	\N	Boutavent	60
23796	097	\N	Boutencourt	60
23797	098	\N	Bouvresse	60
23798	099	\N	Braisnes	60
23799	100	\N	Brasseuse	60
23800	101	\N	Brégy	60
23801	102	\N	Brenouille	60
23802	103	\N	Bresles	60
23803	104	\N	Breteuil	60
23804	105	\N	Brétigny	60
23805	106	\N	Breuil-le-Sec	60
23806	107	\N	Breuil-le-Vert	60
23807	108	\N	Briot	60
23808	109	\N	Brombos	60
23809	110	\N	Broquiers	60
23810	111	\N	Broyes	60
23811	112	\N	Brunvillers-la-Motte	60
23812	113	\N	Bucamps	60
23813	114	\N	Buicourt	60
23814	115	\N	Bulles	60
23815	116	\N	Bury	60
23816	117	\N	Bussy	60
23817	118	\N	Caisnes	60
23818	119	\N	Cambronne-lès-Ribécourt	60
23819	120	\N	Cambronne-lès-Clermont	60
23820	121	\N	Campagne	60
23821	122	\N	Campeaux	60
23822	123	\N	Campremy	60
23823	124	\N	Candor	60
23824	125	\N	Canly	60
23825	126	\N	Cannectancourt	60
23826	127	\N	Canny-sur-Matz	60
23827	128	\N	Canny-sur-Thérain	60
23828	129	\N	Carlepont	60
23829	130	\N	Catenoy	60
23830	131	\N	Catheux	60
23831	132	\N	Catigny	60
23832	133	\N	Catillon-Fumechon	60
23833	134	\N	Cauffry	60
23834	135	\N	Cauvigny	60
23835	136	\N	Cempuis	60
23836	137	\N	Cernoy	60
23837	138	\N	Chamant	60
23838	139	\N	Chambly	60
23839	140	\N	Chambors	60
23840	141	\N	Chantilly	60
23841	142	\N	Chapelle-en-Serval	60
23842	143	\N	Chaumont-en-Vexin	60
23843	144	\N	Chavençon	60
23844	145	\N	Chelles	60
23845	146	\N	Chepoix	60
23846	147	\N	Chevincourt	60
23847	148	\N	Chèvreville	60
23848	149	\N	Chevrières	60
23849	150	\N	Chiry-Ourscamp	60
23850	151	\N	Choisy-au-Bac	60
23851	152	\N	Choisy-la-Victoire	60
23852	153	\N	Choqueuse-les-Bénards	60
23853	154	\N	Cinqueux	60
23854	155	\N	Cires-lès-Mello	60
23855	156	\N	Clairoix	60
23856	157	\N	Clermont	60
23857	158	\N	Coivrel	60
23858	159	\N	Compiègne	60
23859	160	\N	Conchy-les-Pots	60
23860	161	\N	Conteville	60
23861	162	\N	Corbeil-Cerf	60
23862	163	\N	Cormeilles	60
23863	164	\N	Coudray-Saint-Germer	60
23864	165	\N	Coudray-sur-Thelle	60
23865	166	\N	Coudun	60
23866	167	\N	Couloisy	60
23867	168	\N	Courcelles-Epayelles	60
23868	169	\N	Courcelles-lès-Gisors	60
23869	170	\N	Courteuil	60
23870	171	\N	Courtieux	60
23871	172	\N	Coye-la-Forêt	60
23872	173	\N	Cramoisy	60
23873	174	\N	Crapeaumesnil	60
23874	175	\N	Creil	60
23875	176	\N	Crépy-en-Valois	60
23876	177	\N	Cressonsacq	60
23877	178	\N	Crèvecur-le-Grand	60
23878	179	\N	Crèvecur-le-Petit	60
23879	180	\N	Crillon	60
23880	181	\N	Crisolles	60
23881	182	\N	Crocq	60
23882	183	\N	Croissy-sur-Celle	60
23883	184	\N	Croutoy	60
23884	185	\N	Crouy-en-Thelle	60
23885	186	\N	Cuignières	60
23886	187	\N	Cuigy-en-Bray	60
23887	188	\N	Cuise-la-Motte	60
23888	189	\N	Cuts	60
23889	190	\N	Cuvergnon	60
23890	191	\N	Cuvilly	60
23891	192	\N	Cuy	60
23892	193	\N	Daméraucourt	60
23893	194	\N	Dargies	60
23894	195	\N	Delincourt	60
23895	196	\N	Déluge	60
23896	197	\N	Dieudonné	60
23897	198	\N	Dives	60
23898	199	\N	Doméliers	60
23899	200	\N	Domfront	60
23900	201	\N	Dompierre	60
23901	203	\N	Duvy	60
23902	204	\N	Écuvilly	60
23903	205	\N	Élencourt	60
23904	206	\N	Élincourt-Sainte-Marguerite	60
23905	207	\N	Éméville	60
23906	208	\N	Énencourt-Léage	60
23907	209	\N	Énencourt-le-Sec	60
23908	210	\N	Épineuse	60
23909	211	\N	Éragny-sur-Epte	60
23910	212	\N	Ercuis	60
23911	213	\N	Ermenonville	60
23912	214	\N	Ernemont-Boutavent	60
23913	215	\N	Erquery	60
23914	216	\N	Erquinvillers	60
23915	217	\N	Escames	60
23916	218	\N	Esches	60
23917	219	\N	Escles-Saint-Pierre	60
23918	220	\N	Espaubourg	60
23919	221	\N	Esquennoy	60
23920	222	\N	Essuiles	60
23921	223	\N	Estrées-Saint-Denis	60
23922	224	\N	Étavigny	60
23923	225	\N	Étouy	60
23924	226	\N	Ève	60
23925	227	\N	Évricourt	60
23926	228	\N	Fay-les-Étangs	60
23927	229	\N	Fayel	60
23928	230	\N	Fay-Saint-Quentin	60
23929	231	\N	Feigneux	60
23930	232	\N	Ferrières	60
23931	233	\N	Feuquières	60
23932	234	\N	Fitz-James	60
23933	235	\N	Flavacourt	60
23934	236	\N	Flavy-le-Meldeux	60
23935	237	\N	Fléchy	60
23936	238	\N	Fleurines	60
23937	239	\N	Fleury	60
23938	240	\N	Fontaine-Bonneleau	60
23939	241	\N	Fontaine-Chaalis	60
23940	242	\N	Fontaine-Lavaganne	60
23941	243	\N	Fontaine-Saint-Lucien	60
23942	244	\N	Fontenay-Torcy	60
23943	245	\N	Formerie	60
23944	246	\N	Fosseuse	60
23945	247	\N	Fouilleuse	60
23946	248	\N	Fouilloy	60
23947	249	\N	Foulangues	60
23948	250	\N	Fouquenies	60
23949	251	\N	Fouquerolles	60
23950	252	\N	Fournival	60
23951	253	\N	Francastel	60
23952	254	\N	Francières	60
23953	255	\N	Fréniches	60
23954	256	\N	Fresneaux-Montchevreuil	60
23955	257	\N	Fresne-Léguillon	60
23956	258	\N	Fresnières	60
23957	259	\N	Fresnoy-en-Thelle	60
23958	260	\N	Fresnoy-la-Rivière	60
23959	261	\N	Fresnoy-le-Luat	60
23960	262	\N	Frestoy-Vaux	60
23961	263	\N	Frétoy-le-Château	60
23962	264	\N	Frocourt	60
23963	265	\N	Froissy	60
23964	267	\N	Gallet	60
23965	268	\N	Gannes	60
23966	269	\N	Gaudechart	60
23967	270	\N	Genvry	60
23968	271	\N	Gerberoy	60
23969	272	\N	Gilocourt	60
23970	273	\N	Giraumont	60
23971	274	\N	Glaignes	60
23972	275	\N	Glatigny	60
23973	276	\N	Godenvillers	60
23974	277	\N	Goincourt	60
23975	278	\N	Golancourt	60
23976	279	\N	Gondreville	60
23977	280	\N	Gourchelles	60
23978	281	\N	Gournay-sur-Aronde	60
23979	282	\N	Gouvieux	60
23980	283	\N	Gouy-les-Groseillers	60
23981	284	\N	Grandfresnoy	60
23982	285	\N	Grandvillers-aux-Bois	60
23983	286	\N	Grandvilliers	60
23984	287	\N	Grandrû	60
23985	288	\N	Grémévillers	60
23986	289	\N	Grez	60
23987	290	\N	Guignecourt	60
23988	291	\N	Guiscard	60
23989	292	\N	Gury	60
23990	293	\N	Hadancourt-le-Haut-Clocher	60
23991	294	\N	Hainvillers	60
23992	295	\N	Halloy	60
23993	296	\N	Hannaches	60
23994	297	\N	Hamel	60
23995	298	\N	Hanvoile	60
23996	299	\N	Hardivillers	60
23997	300	\N	Hardivillers-en-Vexin	60
23998	301	\N	Haucourt	60
23999	302	\N	Haudivillers	60
24000	303	\N	Hautbos	60
24001	304	\N	Haute-Épine	60
24002	305	\N	Hautefontaine	60
24003	306	\N	Hécourt	60
24004	307	\N	Heilles	60
24005	308	\N	Hémévillers	60
24006	309	\N	Hénonville	60
24007	310	\N	Herchies	60
24008	311	\N	Hérelle	60
24009	312	\N	Héricourt-sur-Thérain	60
24010	313	\N	Hermes	60
24011	314	\N	Hétomesnil	60
24012	315	\N	Hodenc-en-Bray	60
24013	316	\N	Hodenc-l'Évêque	60
24014	317	\N	Hondainville	60
24015	318	\N	Houdancourt	60
24016	319	\N	Houssoye	60
24017	320	\N	Ivors	60
24018	321	\N	Ivry-le-Temple	60
24019	322	\N	Jaméricourt	60
24020	323	\N	Janville	60
24021	324	\N	Jaulzy	60
24022	325	\N	Jaux	60
24023	326	\N	Jonquières	60
24024	327	\N	Jouy-sous-Thelle	60
24025	328	\N	Juvignies	60
24026	329	\N	Laberlière	60
24027	330	\N	Laboissière-en-Thelle	60
24028	331	\N	Labosse	60
24029	332	\N	Labruyère	60
24030	333	\N	Lachapelle-aux-Pots	60
24031	334	\N	Lachapelle-Saint-Pierre	60
24032	335	\N	Lachapelle-sous-Gerberoy	60
24033	336	\N	Lachaussée-du-Bois-d'Écu	60
24034	337	\N	Lachelle	60
24035	338	\N	Lacroix-Saint-Ouen	60
24036	339	\N	Lafraye	60
24037	340	\N	Lagny	60
24038	341	\N	Lagny-le-Sec	60
24039	342	\N	Laigneville	60
24040	343	\N	Lalande-en-Son	60
24041	344	\N	Lalandelle	60
24042	345	\N	Lamécourt	60
24043	346	\N	Lamorlaye	60
24044	347	\N	Lannoy-Cuillère	60
24045	348	\N	Larbroye	60
24046	350	\N	Lassigny	60
24047	351	\N	Lataule	60
24048	352	\N	Lattainville	60
24049	353	\N	Lavacquerie	60
24050	354	\N	Laverrière	60
24051	355	\N	Laversines	60
24052	356	\N	Lavilletertre	60
24053	357	\N	Léglantiers	60
24054	358	\N	Lévignen	60
24055	359	\N	Lhéraule	60
24056	360	\N	Liancourt	60
24057	361	\N	Liancourt-Saint-Pierre	60
24058	362	\N	Libermont	60
24059	363	\N	Lierville	60
24060	364	\N	Lieuvillers	60
24061	365	\N	Lihus	60
24062	366	\N	Litz	60
24063	367	\N	Loconville	60
24064	368	\N	Longueil-Annel	60
24065	369	\N	Longueil-Sainte-Marie	60
24066	370	\N	Lormaison	60
24067	371	\N	Loueuse	60
24068	372	\N	Luchy	60
24069	373	\N	Machemont	60
24070	374	\N	Maignelay-Montigny	60
24071	375	\N	Maimbeville	60
24072	376	\N	Maisoncelle-Saint-Pierre	60
24073	377	\N	Maisoncelle-Tuilerie	60
24074	378	\N	Marest-sur-Matz	60
24075	379	\N	Mareuil-la-Motte	60
24076	380	\N	Mareuil-sur-Ourcq	60
24077	381	\N	Margny-aux-Cerises	60
24078	382	\N	Margny-lès-Compiègne	60
24079	383	\N	Margny-sur-Matz	60
24080	385	\N	Marolles	60
24081	386	\N	Marquéglise	60
24082	387	\N	Marseille-en-Beauvaisis	60
24083	388	\N	Martincourt	60
24084	389	\N	Maucourt	60
24085	390	\N	Maulers	60
24086	391	\N	Maysel	60
24087	392	\N	Mélicocq	60
24088	393	\N	Mello	60
24089	394	\N	Ménévillers	60
24090	395	\N	Méru	60
24091	396	\N	Méry-la-Bataille	60
24092	397	\N	Mesnil-Conteville	60
24093	398	\N	Mesnil-en-Thelle	60
24094	399	\N	Mesnil-Saint-Firmin	60
24095	400	\N	Mesnil-sur-Bulles	60
24096	401	\N	Mesnil-Théribus	60
24097	402	\N	Meux	60
24098	403	\N	Milly-sur-Thérain	60
24099	404	\N	Mogneville	60
24100	405	\N	Moliens	60
24101	406	\N	Monceaux	60
24102	407	\N	Monceaux-l'Abbaye	60
24103	408	\N	Monchy-Humières	60
24104	409	\N	Monchy-Saint-Éloi	60
24105	410	\N	Mondescourt	60
24106	411	\N	Monneville	60
24107	412	\N	Montagny-en-Vexin	60
24108	413	\N	Montagny-Sainte-Félicité	60
24109	414	\N	Montataire	60
24110	415	\N	Montépilloy	60
24111	416	\N	Montgérain	60
24112	417	\N	Montherlant	60
24113	418	\N	Montiers	60
24114	420	\N	Montjavoult	60
24115	421	\N	Mont-l'Évêque	60
24116	422	\N	Montlognon	60
24117	423	\N	Montmacq	60
24118	424	\N	Montmartin	60
24119	425	\N	Montreuil-sur-Brêche	60
24120	426	\N	Montreuil-sur-Thérain	60
24121	427	\N	Monts	60
24122	428	\N	Mont-Saint-Adrien	60
24123	429	\N	Morangles	60
24124	430	\N	Morienval	60
24125	431	\N	Morlincourt	60
24126	432	\N	Mortefontaine	60
24127	433	\N	Mortefontaine-en-Thelle	60
24128	434	\N	Mortemer	60
24129	435	\N	Morvillers	60
24130	436	\N	Mory-Montcrux	60
24131	437	\N	Mouchy-le-Châtel	60
24132	438	\N	Moulin-sous-Touvent	60
24133	439	\N	Mouy	60
24134	440	\N	Moyenneville	60
24135	441	\N	Moyvillers	60
24136	442	\N	Muidorge	60
24137	443	\N	Muirancourt	60
24138	444	\N	Mureaumont	60
24139	445	\N	Nampcel	60
24140	446	\N	Nanteuil-le-Haudouin	60
24141	447	\N	Néry	60
24142	448	\N	Neufchelles	60
24143	449	\N	Neufvy-sur-Aronde	60
24144	450	\N	Neuilly-en-Thelle	60
24145	451	\N	Neuilly-sous-Clermont	60
24146	452	\N	Neuville-Bosc	60
24147	453	\N	Neuville-d'Aumont	60
24148	454	\N	Neuville-en-Hez	60
24149	455	\N	Neuville-Garnier	60
24150	456	\N	Neuville-Roy	60
24151	457	\N	Neuville-Saint-Pierre	60
24152	458	\N	Neuville-sur-Oudeuil	60
24153	459	\N	Neuville-sur-Ressons	60
24154	460	\N	Neuville-Vault	60
24155	461	\N	Nivillers	60
24156	462	\N	Noailles	60
24157	463	\N	Nogent-sur-Oise	60
24158	464	\N	Nointel	60
24159	465	\N	Noirémont	60
24160	466	\N	Noroy	60
24161	468	\N	Nourard-le-Franc	60
24162	469	\N	Novillers	60
24163	470	\N	Noyers-Saint-Martin	60
24164	471	\N	Noyon	60
24165	472	\N	Offoy	60
24166	473	\N	Ognes	60
24167	474	\N	Ognolles	60
24168	475	\N	Ognon	60
24169	476	\N	Omécourt	60
24170	477	\N	Ons-en-Bray	60
24171	478	\N	Ormoy-le-Davien	60
24172	479	\N	Ormoy-Villers	60
24173	480	\N	Oroër	60
24174	481	\N	Orrouy	60
24175	482	\N	Orry-la-Ville	60
24176	483	\N	Orvillers-Sorel	60
24177	484	\N	Oudeuil	60
24178	485	\N	Oursel-Maison	60
24179	486	\N	Paillart	60
24180	487	\N	Parnes	60
24181	488	\N	Passel	60
24182	489	\N	Péroy-les-Gombries	60
24183	490	\N	Pierrefitte-en-Beauvaisis	60
24184	491	\N	Pierrefonds	60
24185	492	\N	Pimprez	60
24186	493	\N	Pisseleu	60
24187	494	\N	Plailly	60
24188	495	\N	Plainval	60
24189	496	\N	Plainville	60
24190	497	\N	Plessier-sur-Bulles	60
24191	498	\N	Plessier-sur-Saint-Just	60
24192	499	\N	Plessis-de-Roye	60
24193	500	\N	Plessis-Belleville	60
24194	501	\N	Plessis-Brion	60
24195	502	\N	Plessis-Patte-d'Oie	60
24196	503	\N	Ployron	60
24197	504	\N	Ponchon	60
24198	505	\N	Pontarmé	60
24199	506	\N	Pont-l'Évêque	60
24200	507	\N	Pontoise-lès-Noyon	60
24201	508	\N	Pontpoint	60
24202	509	\N	Pont-Sainte-Maxence	60
24203	510	\N	Porcheux	60
24204	511	\N	Porquéricourt	60
24205	512	\N	Pouilly	60
24206	513	\N	Précy-sur-Oise	60
24207	514	\N	Prévillers	60
24208	515	\N	Pronleroy	60
24209	516	\N	Puiseux-en-Bray	60
24210	517	\N	Puiseux-le-Hauberger	60
24211	518	\N	Puits-la-Vallée	60
24212	519	\N	Quesmy	60
24213	520	\N	Quesnel-Aubry	60
24214	521	\N	Quincampoix-Fleuzy	60
24215	522	\N	Quinquempoix	60
24216	523	\N	Rainvillers	60
24217	524	\N	Rantigny	60
24218	525	\N	Raray	60
24219	526	\N	Ravenel	60
24220	527	\N	Réez-Fosse-Martin	60
24221	528	\N	Reilly	60
24222	529	\N	Rémécourt	60
24223	530	\N	Rémérangles	60
24224	531	\N	Remy	60
24225	532	\N	Ressons-l'Abbaye	60
24226	533	\N	Ressons-sur-Matz	60
24227	534	\N	Rethondes	60
24228	535	\N	Reuil-sur-Brêche	60
24229	536	\N	Rhuis	60
24230	537	\N	Ribécourt-Dreslincourt	60
24231	538	\N	Ricquebourg	60
24232	539	\N	Rieux	60
24233	540	\N	Rivecourt	60
24234	541	\N	Roberval	60
24235	542	\N	Rochy-Condé	60
24236	543	\N	Rocquemont	60
24237	544	\N	Rocquencourt	60
24238	545	\N	Romescamps	60
24239	546	\N	Rosières	60
24240	547	\N	Rosoy	60
24241	548	\N	Rosoy-en-Multien	60
24242	549	\N	Rotangy	60
24243	550	\N	Rothois	60
24244	551	\N	Rousseloy	60
24245	552	\N	Rouville	60
24246	553	\N	Rouvillers	60
24247	554	\N	Rouvres-en-Multien	60
24248	555	\N	Rouvroy-les-Merles	60
24249	556	\N	Royaucourt	60
24250	557	\N	Roy-Boissy	60
24251	558	\N	Roye-sur-Matz	60
24252	559	\N	Rue-Saint-Pierre	60
24253	560	\N	Rully	60
24254	561	\N	Russy-Bémont	60
24255	562	\N	Sacy-le-Grand	60
24256	563	\N	Sacy-le-Petit	60
24257	564	\N	Sains-Morainvillers	60
24258	565	\N	Saint-André-Farivillers	60
24259	566	\N	Saint-Arnoult	60
24260	567	\N	Saint-Aubin-en-Bray	60
24261	568	\N	Saint-Aubin-sous-Erquery	60
24262	569	\N	Saint-Crépin-aux-Bois	60
24263	570	\N	Saint-Crépin-Ibouvillers	60
24264	571	\N	Saint-Deniscourt	60
24265	572	\N	Saint-Étienne-Roilaye	60
24266	573	\N	Sainte-Eusoye	60
24267	574	\N	Saint-Félix	60
24268	575	\N	Sainte-Geneviève	60
24269	576	\N	Saint-Germain-la-Poterie	60
24270	577	\N	Saint-Germer-de-Fly	60
24271	578	\N	Saintines	60
24272	579	\N	Saint-Jean-aux-Bois	60
24273	581	\N	Saint-Just-en-Chaussée	60
24274	582	\N	Saint-Léger-aux-Bois	60
24275	583	\N	Saint-Léger-en-Bray	60
24276	584	\N	Saint-Leu-d'Esserent	60
24277	585	\N	Saint-Martin-aux-Bois	60
24278	586	\N	Saint-Martin-le-Nud	60
24279	587	\N	Saint-Martin-Longueau	60
24280	588	\N	Saint-Maur	60
24281	589	\N	Saint-Maximin	60
24282	590	\N	Saint-Omer-en-Chaussée	60
24283	591	\N	Saint-Paul	60
24284	592	\N	Saint-Pierre-es-Champs	60
24285	593	\N	Saint-Pierre-lès-Bitry	60
24286	594	\N	Saint-Quentin-des-Prés	60
24287	595	\N	Saint-Remy-en-l'Eau	60
24288	596	\N	Saint-Samson-la-Poterie	60
24289	597	\N	Saint-Sauveur	60
24290	598	\N	Saint-Sulpice	60
24291	599	\N	Saint-Thibault	60
24292	600	\N	Saint-Vaast-de-Longmont	60
24293	601	\N	Saint-Vaast-lès-Mello	60
24294	602	\N	Saint-Valery	60
24295	603	\N	Salency	60
24296	604	\N	Sarcus	60
24297	605	\N	Sarnois	60
24298	608	\N	Saulchoy	60
24299	609	\N	Savignies	60
24300	610	\N	Sempigny	60
24301	611	\N	Senantes	60
24302	612	\N	Senlis	60
24303	613	\N	Senots	60
24304	614	\N	Serans	60
24305	615	\N	Sérévillers	60
24306	616	\N	Sérifontaine	60
24307	617	\N	Sermaize	60
24308	618	\N	Séry-Magneval	60
24309	619	\N	Silly-le-Long	60
24310	620	\N	Silly-Tillard	60
24311	621	\N	Solente	60
24312	622	\N	Sommereux	60
24313	623	\N	Songeons	60
24314	624	\N	Sully	60
24315	625	\N	Suzoy	60
24316	626	\N	Talmontiers	60
24317	627	\N	Tartigny	60
24318	628	\N	Therdonne	60
24319	629	\N	Thérines	60
24320	630	\N	Thibivillers	60
24321	631	\N	Thiers-sur-Thève	60
24322	632	\N	Thiescourt	60
24323	633	\N	Thieuloy-Saint-Antoine	60
24324	634	\N	Thieux	60
24325	635	\N	Thiverny	60
24326	636	\N	Thourotte	60
24327	637	\N	Thury-en-Valois	60
24328	638	\N	Thury-sous-Clermont	60
24329	639	\N	Tillé	60
24330	640	\N	Tourly	60
24331	641	\N	Tracy-le-Mont	60
24332	642	\N	Tracy-le-Val	60
24333	643	\N	Tricot	60
24334	644	\N	Trie-Château	60
24335	645	\N	Trie-la-Ville	60
24336	646	\N	Troissereux	60
24337	647	\N	Trosly-Breuil	60
24338	648	\N	Troussencourt	60
24339	649	\N	Troussures	60
24340	650	\N	Trumilly	60
24341	651	\N	Ully-Saint-Georges	60
24342	652	\N	Valdampierre	60
24343	653	\N	Valescourt	60
24344	654	\N	Vandélicourt	60
24345	655	\N	Varesnes	60
24346	656	\N	Varinfroy	60
24347	657	\N	Vauchelles	60
24348	658	\N	Vauciennes	60
24349	659	\N	Vaudancourt	60
24350	660	\N	Vaumain	60
24351	661	\N	Vaumoise	60
24352	662	\N	Vauroux	60
24353	663	\N	Velennes	60
24354	664	\N	Vendeuil-Caply	60
24355	665	\N	Venette	60
24356	666	\N	Ver-sur-Launette	60
24357	667	\N	Verberie	60
24358	668	\N	Verderel-lès-Sauqueuse	60
24359	669	\N	Verderonne	60
24360	670	\N	Verneuil-en-Halatte	60
24361	671	\N	Versigny	60
24362	672	\N	Vez	60
24363	673	\N	Viefvillers	60
24364	674	\N	Vieux-Moulin	60
24365	675	\N	Vignemont	60
24366	676	\N	Ville	60
24367	677	\N	Villembray	60
24368	678	\N	Villeneuve-les-Sablons	60
24369	679	\N	Villeneuve-sous-Thury	60
24370	680	\N	Villeneuve-sur-Verberie	60
24371	681	\N	Villers-Saint-Barthélemy	60
24372	682	\N	Villers-Saint-Frambourg	60
24373	683	\N	Villers-Saint-Genest	60
24374	684	\N	Villers-Saint-Paul	60
24375	685	\N	Villers-Saint-Sépulcre	60
24376	686	\N	Villers-sous-Saint-Leu	60
24377	687	\N	Villers-sur-Auchy	60
24378	688	\N	Villers-sur-Bonnières	60
24379	689	\N	Villers-sur-Coudun	60
24380	690	\N	Villers-sur-Trie	60
24381	691	\N	Villers-Vermont	60
24382	692	\N	Villers-Vicomte	60
24383	693	\N	Villeselve	60
24384	694	\N	Villotran	60
24385	695	\N	Vineuil-Saint-Firmin	60
24386	697	\N	Vrocourt	60
24387	698	\N	Wacquemoulin	60
24388	699	\N	Wambez	60
24389	700	\N	Warluis	60
24390	701	\N	Wavignies	60
24391	702	\N	Welles-Pérennes	60
24392	703	\N	Marais	60
24393	001	\N	Alençon	61
24394	002	\N	Almenêches	61
24395	003	\N	Anceins	61
24396	004	\N	Antoigny	61
24397	005	\N	Appenai-sous-Bellême	61
24398	006	\N	Argentan	61
24399	007	\N	Athis-de-l'Orne	61
24400	008	\N	Aube	61
24401	009	\N	Aubry-en-Exmes	61
24402	010	\N	Aubry-le-Panthou	61
24403	011	\N	Aubusson	61
24404	012	\N	Auguaise	61
24405	013	\N	Aunay-les-Bois	61
24406	014	\N	Aunou-le-Faucon	61
24407	015	\N	Aunou-sur-Orne	61
24408	016	\N	Autheuil	61
24409	017	\N	Authieux-du-Puits	61
24410	018	\N	Avernes-Saint-Gourgon	61
24411	019	\N	Avernes-sous-Exmes	61
24412	020	\N	Avoine	61
24413	021	\N	Avrilly	61
24414	023	\N	Bailleul	61
24415	024	\N	Banvou	61
24416	025	\N	Baroche-sous-Lucé	61
24417	026	\N	Barville	61
24418	027	\N	Batilly	61
24419	028	\N	Bazoches-au-Houlme	61
24420	029	\N	Bazoches-sur-Hoëne	61
24421	030	\N	Bazoque	61
24422	031	\N	Beauchêne	61
24423	032	\N	Beaufai	61
24424	033	\N	Beaulandais	61
24425	034	\N	Beaulieu	61
24426	035	\N	Beauvain	61
24427	036	\N	Belfonds	61
24428	037	\N	Bellavilliers	61
24429	038	\N	Bellême	61
24430	039	\N	Bellière	61
24431	040	\N	Bellou-en-Houlme	61
24432	041	\N	Bellou-le-Trichard	61
24433	042	\N	Bellou-sur-Huisne	61
24434	043	\N	Berd'huis	61
24435	044	\N	Berjou	61
24436	045	\N	Bivilliers	61
24437	046	\N	Bizou	61
24438	047	\N	Bocquencé	61
24439	048	\N	Boëcé	61
24440	049	\N	Boissei-la-Lande	61
24441	050	\N	Boissy-Maugis	61
24442	051	\N	Boitron	61
24443	052	\N	Bonnefoi	61
24444	053	\N	Bonsmoulins	61
24445	054	\N	Bosc-Renoult	61
24446	055	\N	Boucé	61
24447	056	\N	Bouillon	61
24448	057	\N	Bourg-Saint-Léonard	61
24449	058	\N	Bréel	61
24450	059	\N	Bresolettes	61
24451	060	\N	Brethel	61
24452	061	\N	Bretoncelles	61
24453	062	\N	Brieux	61
24454	063	\N	Briouze	61
24455	064	\N	Brullemail	61
24456	065	\N	Bubertré	61
24457	066	\N	Buré	61
24458	067	\N	Bures	61
24459	068	\N	Bursard	61
24460	069	\N	Cahan	61
24461	070	\N	Caligny	61
24462	071	\N	Camembert	61
24463	072	\N	Canapville	61
24464	073	\N	Carneille	61
24465	074	\N	Carrouges	61
24466	075	\N	Ceaucé	61
24467	076	\N	Cercueil	61
24468	077	\N	Cerisé	61
24469	078	\N	Cerisy-Belle-Étoile	61
24470	079	\N	Ceton	61
24471	080	\N	Chahains	61
24472	081	\N	Chailloué	61
24473	082	\N	Chalange	61
24474	083	\N	Chambois	61
24475	084	\N	Champcerie	61
24476	085	\N	Champ-de-la-Pierre	61
24477	086	\N	Champeaux	61
24478	087	\N	Champeaux-sur-Sarthe	61
24479	088	\N	Champ-Haut	61
24480	089	\N	Champosoult	61
24481	090	\N	Champs	61
24482	091	\N	Champsecret	61
24483	092	\N	Chandai	61
24484	093	\N	Chanu	61
24485	094	\N	Chapelle-au-Moine	61
24486	095	\N	Chapelle-Biche	61
24487	096	\N	Chapelle-d'Andaine	61
24488	097	\N	Chapelle-Montligeon	61
24489	098	\N	Chapelle-près-Sées	61
24490	099	\N	Chapelle-Souëf	61
24491	100	\N	Chapelle-Viel	61
24492	101	\N	Château-d'Almenêches	61
24493	102	\N	Châtellier	61
24494	103	\N	Chaumont	61
24495	104	\N	Chaux	61
24496	105	\N	Chemilli	61
24497	106	\N	Chênedouit	61
24498	107	\N	Ciral	61
24499	108	\N	Cisai-Saint-Aubin	61
24500	109	\N	Clairefougère	61
24501	110	\N	Cochère	61
24502	111	\N	Colombiers	61
24503	112	\N	Colonard-Corubert	61
24504	113	\N	Comblot	61
24505	114	\N	Commeaux	61
24506	115	\N	Condeau	61
24507	116	\N	Condé-sur-Huisne	61
24508	117	\N	Condé-sur-Sarthe	61
24509	118	\N	Corbon	61
24510	120	\N	Coudehard	61
24511	121	\N	Coulimer	61
24512	122	\N	Coulmer	61
24513	123	\N	Coulonces	61
24514	124	\N	Coulonche	61
24515	125	\N	Coulonges-les-Sablons	61
24516	126	\N	Coulonges-sur-Sarthe	61
24517	127	\N	Courbe	61
24518	128	\N	Courcerault	61
24519	129	\N	Courgeon	61
24520	130	\N	Courgeoût	61
24521	131	\N	Courménil	61
24522	133	\N	Courtomer	61
24523	135	\N	Couterne	61
24524	136	\N	Couvains	61
24525	137	\N	Craménil	61
24526	138	\N	Croisilles	61
24527	139	\N	Crouttes	61
24528	140	\N	Crulai	61
24529	141	\N	Cuissai	61
24530	142	\N	Dame-Marie	61
24531	143	\N	Damigny	61
24532	144	\N	Dancé	61
24533	145	\N	Domfront	61
24534	146	\N	Dompierre	61
24535	147	\N	Dorceau	61
24536	148	\N	Durcet	61
24537	149	\N	Échalou	61
24538	150	\N	Échauffour	61
24539	151	\N	Écorcei	61
24540	152	\N	Écorches	61
24541	153	\N	Écouché	61
24542	154	\N	Eperrais	61
24543	155	\N	Épinay-le-Comte	61
24544	156	\N	Essay	61
24545	157	\N	Exmes	61
24546	158	\N	Faverolles	61
24547	159	\N	Fay	61
24548	160	\N	Feings	61
24549	161	\N	Fel	61
24550	162	\N	Ferrière-au-Doyen	61
24551	163	\N	Ferrière-aux-Étangs	61
24552	164	\N	Ferrière-Béchet	61
24553	165	\N	Ferrière-Bochard	61
24554	166	\N	Ferrières-la-Verrerie	61
24555	167	\N	Ferté-Frênel	61
24556	168	\N	Ferté-Macé	61
24557	169	\N	Flers	61
24558	170	\N	Fleuré	61
24559	171	\N	Fontaine-les-Bassets	61
24560	172	\N	Fontenai-les-Louvets	61
24561	173	\N	Fontenai-sur-Orne	61
24562	174	\N	Forêt-Auvray	61
24563	175	\N	Forges	61
24564	176	\N	Francheville	61
24565	177	\N	Frênes	61
24566	178	\N	Fresnaie-Fayel	61
24567	179	\N	Fresnaye-au-Sauvage	61
24568	180	\N	Fresnay-le-Samson	61
24569	181	\N	Gacé	61
24570	182	\N	Gandelain	61
24571	183	\N	Gâprée	61
24572	184	\N	Gauville	61
24573	185	\N	Gémages	61
24574	186	\N	Geneslay	61
24575	187	\N	Genettes	61
24576	188	\N	Genevraie	61
24577	189	\N	Giel-Courteilles	61
24578	190	\N	Ginai	61
24579	191	\N	Glos-la-Ferrière	61
24580	192	\N	Godisson	61
24581	193	\N	Gonfrière	61
24582	194	\N	Goulet	61
24583	195	\N	Grais	61
24584	196	\N	Gué-de-la-Chaîne	61
24585	197	\N	Guêprei	61
24586	198	\N	Guerquesalles	61
24587	199	\N	Habloville	61
24588	200	\N	Haleine	61
24589	201	\N	Haute-Chapelle	61
24590	202	\N	Hauterive	61
24591	203	\N	Héloup	61
24592	204	\N	Hermitière	61
24593	205	\N	Heugon	61
24594	206	\N	Hôme-Chamondot	61
24595	207	\N	Igé	61
24596	208	\N	Irai	61
24597	209	\N	Joué-du-Bois	61
24598	210	\N	Joué-du-Plain	61
24599	211	\N	Juvigny-sous-Andaine	61
24600	212	\N	Juvigny-sur-Orne	61
24601	213	\N	Lalacelle	61
24602	214	\N	Aigle	61
24603	215	\N	Laleu	61
24604	216	\N	Lande-de-Goult	61
24605	217	\N	Lande-de-Lougé	61
24606	218	\N	Lande-Patry	61
24607	219	\N	Lande-Saint-Siméon	61
24608	220	\N	Lande-sur-Eure	61
24609	221	\N	Landigou	61
24610	222	\N	Landisacq	61
24611	223	\N	Larchamp	61
24612	224	\N	Larré	61
24613	225	\N	Lignères	61
24614	226	\N	Lignerolles	61
24615	227	\N	Lignou	61
24616	228	\N	Livaie	61
24617	229	\N	Loisail	61
24618	230	\N	Longny-au-Perche	61
24619	231	\N	Longuenoë	61
24620	232	\N	Lonlay-l'Abbaye	61
24621	233	\N	Lonlay-le-Tesson	61
24622	234	\N	Lonrai	61
24623	235	\N	Loré	61
24624	236	\N	Loucé	61
24625	237	\N	Lougé-sur-Maire	61
24626	238	\N	Louvières-en-Auge	61
24627	239	\N	Lucé	61
24628	240	\N	Macé	61
24629	241	\N	Madeleine-Bouvet	61
24630	242	\N	Mage	61
24631	243	\N	Magny-le-Désert	61
24632	244	\N	Mahéru	61
24633	245	\N	Maison-Maugis	61
24634	246	\N	Mâle	61
24635	247	\N	Malétable	61
24636	248	\N	Mantilly	61
24637	249	\N	Marcei	61
24638	250	\N	Marchainville	61
24639	251	\N	Marchemaisons	61
24640	252	\N	Mardilly	61
24641	253	\N	Marmouillé	61
24642	255	\N	Mauves-sur-Huisne	61
24643	256	\N	Médavy	61
24644	257	\N	Méhoudin	61
24645	258	\N	Mêle-sur-Sarthe	61
24646	259	\N	Ménil-Bérard	61
24647	260	\N	Ménil-de-Briouze	61
24648	261	\N	Ménil-Broût	61
24649	262	\N	Ménil-Ciboult	61
24650	263	\N	Ménil-Erreux	61
24651	264	\N	Ménil-Froger	61
24652	265	\N	Ménil-Gondouin	61
24653	266	\N	Ménil-Guyon	61
24654	267	\N	Ménil-Hermei	61
24655	268	\N	Ménil-Hubert-en-Exmes	61
24656	269	\N	Ménil-Hubert-sur-Orne	61
24657	270	\N	Ménil-Jean	61
24658	271	\N	Ménil-Scelleur	61
24659	272	\N	Ménil-Vicomte	61
24660	273	\N	Ménil-Vin	61
24661	274	\N	Menus	61
24662	275	\N	Merlerault	61
24663	276	\N	Merri	61
24664	277	\N	Mesnière	61
24665	278	\N	Messei	61
24666	279	\N	Mieuxcé	61
24667	280	\N	Monceaux-au-Perche	61
24668	281	\N	Moncy	61
24669	282	\N	Monnai	61
24670	283	\N	Montabard	61
24671	284	\N	Montchevrel	61
24672	285	\N	Montgaroult	61
24673	286	\N	Montgaudry	61
24674	287	\N	Montilly-sur-Noireau	61
24675	288	\N	Montmerrei	61
24676	289	\N	Mont-Ormel	61
24677	290	\N	Montreuil-au-Houlme	61
24678	291	\N	Montreuil-la-Cambe	61
24679	292	\N	Montsecret	61
24680	293	\N	Mortagne-au-Perche	61
24681	294	\N	Mortrée	61
24682	295	\N	Motte-Fouquet	61
24683	296	\N	Moulicent	61
24684	297	\N	Moulins-la-Marche	61
24685	298	\N	Moulins-sur-Orne	61
24686	299	\N	Moussonvilliers	61
24687	300	\N	Moutiers-au-Perche	61
24688	301	\N	Neauphe-sous-Essai	61
24689	302	\N	Neauphe-sur-Dive	61
24690	303	\N	Nécy	61
24691	304	\N	Neuilly-le-Bisson	61
24692	305	\N	Neuilly-sur-Eure	61
24693	306	\N	Neuville-près-Sées	61
24694	307	\N	Neuville-sur-Touques	61
24695	308	\N	Neuvy-au-Houlme	61
24696	309	\N	Nocé	61
24697	310	\N	Nonant-le-Pin	61
24698	311	\N	Normandel	61
24699	313	\N	Notre-Dame-du-Rocher	61
24700	314	\N	Occagnes	61
24701	315	\N	Omméel	61
24702	316	\N	Ommoy	61
24703	317	\N	Orgères	61
24704	318	\N	Origny-le-Butin	61
24705	319	\N	Origny-le-Roux	61
24706	320	\N	Orville	61
24707	321	\N	Pacé	61
24708	322	\N	Parfondeval	61
24709	323	\N	Pas-Saint-l'Homer	61
24710	324	\N	Passais	61
24711	325	\N	Perrière	61
24712	326	\N	Perrou	61
24713	327	\N	Pervenchères	61
24714	328	\N	Pin-au-Haras	61
24715	329	\N	Pin-la-Garenne	61
24716	330	\N	Planches	61
24717	331	\N	Plantis	61
24718	332	\N	Pointel	61
24719	333	\N	Pontchardon	61
24720	335	\N	Poterie-au-Perche	61
24721	336	\N	Pouvrai	61
24722	337	\N	Préaux-du-Perche	61
24723	338	\N	Prépotin	61
24724	339	\N	Putanges-Pont-Écrepin	61
24725	340	\N	Rabodanges	61
24726	341	\N	Radon	61
24727	342	\N	Rai	61
24728	343	\N	Randonnai	61
24729	344	\N	Rânes	61
24730	345	\N	Rémalard	61
24731	346	\N	Renouard	61
24732	347	\N	Résenlieu	61
24733	348	\N	Réveillon	61
24734	349	\N	Ri	61
24735	350	\N	Roche-Mabile	61
24736	351	\N	Roiville	61
24737	352	\N	Rônai	61
24738	353	\N	Ronfeugerai	61
24739	354	\N	Rotours	61
24740	355	\N	Rouellé	61
24741	356	\N	Rouge	61
24742	357	\N	Rouperroux	61
24743	358	\N	Sai	61
24744	359	\N	Saint-Agnan-sur-Erre	61
24745	360	\N	Saint-Agnan-sur-Sarthe	61
24746	361	\N	Saint-André-de-Briouze	61
24747	362	\N	Saint-André-de-Messei	61
24748	363	\N	Saint-Aquilin-de-Corbion	61
24749	364	\N	Saint-Aubert-sur-Orne	61
24750	365	\N	Saint-Aubin-d'Appenai	61
24751	366	\N	Saint-Aubin-de-Bonneval	61
24752	367	\N	Saint-Aubin-de-Courteraie	61
24753	368	\N	Saint-Aubin-des-Grois	61
24754	369	\N	Saint-Bômer-les-Forges	61
24755	370	\N	Saint-Brice	61
24756	371	\N	Saint-Brice-sous-Rânes	61
24757	372	\N	Saint-Céneri-le-Gérei	61
24758	373	\N	Sainte-Céronne-lès-Mortagne	61
24759	374	\N	Saint-Christophe-de-Chaulieu	61
24760	375	\N	Saint-Christophe-le-Jajolet	61
24761	376	\N	Saint-Clair-de-Halouze	61
24762	377	\N	Saint-Cornier-des-Landes	61
24763	378	\N	Sainte-Croix-sur-Orne	61
24764	379	\N	Saint-Cyr-la-Rosière	61
24765	380	\N	Saint-Denis-de-Villenette	61
24766	381	\N	Saint-Denis-sur-Huisne	61
24767	382	\N	Saint-Denis-sur-Sarthon	61
24768	383	\N	Saint-Didier-sous-Écouves	61
24769	384	\N	Saint-Ellier-les-Bois	61
24770	385	\N	Saint-Evroult-de-Montfort	61
24771	386	\N	Saint-Evroult-Notre-Dame-du-Bois	61
24772	387	\N	Saint-Fraimbault	61
24773	388	\N	Saint-Fulgent-des-Ormes	61
24774	389	\N	Sainte-Gauburge-Sainte-Colombe	61
24775	390	\N	Saint-Georges-d'Annebecq	61
24776	391	\N	Saint-Georges-des-Groseillers	61
24777	392	\N	Saint-Germain-d'Aunay	61
24778	393	\N	Saint-Germain-de-Clairefeuille	61
24779	394	\N	Saint-Germain-de-la-Coudre	61
24780	395	\N	Saint-Germain-des-Grois	61
24781	396	\N	Saint-Germain-de-Martigny	61
24782	397	\N	Saint-Germain-du-Corbéis	61
24783	398	\N	Saint-Germain-le-Vieux	61
24784	399	\N	Saint-Gervais-des-Sablons	61
24785	400	\N	Saint-Gervais-du-Perron	61
24786	401	\N	Saint-Gilles-des-Marais	61
24787	402	\N	Saint-Hilaire-de-Briouze	61
24788	403	\N	Saint-Hilaire-la-Gérard	61
24789	404	\N	Saint-Hilaire-le-Châtel	61
24790	405	\N	Saint-Hilaire-sur-Erre	61
24791	406	\N	Saint-Hilaire-sur-Risle	61
24792	407	\N	Sainte-Honorine-la-Chardonne	61
24793	408	\N	Sainte-Honorine-la-Guillaume	61
24794	409	\N	Saint-Jean-de-la-Forêt	61
24795	410	\N	Saint-Jean-des-Bois	61
24796	411	\N	Saint-Jouin-de-Blavou	61
24797	412	\N	Saint-Julien-sur-Sarthe	61
24798	413	\N	Saint-Lambert-sur-Dive	61
24799	414	\N	Saint-Langis-lès-Mortagne	61
24800	415	\N	Saint-Léger-sur-Sarthe	61
24801	416	\N	Saint-Léonard-des-Parcs	61
24802	417	\N	Saint-Loyer-des-Champs	61
24803	418	\N	Saint-Mard-de-Réno	61
24804	419	\N	Sainte-Marguerite-de-Carrouges	61
24805	420	\N	Sainte-Marie-la-Robert	61
24806	421	\N	Saint-Mars-d'Égrenne	61
24807	422	\N	Aspres	61
24808	423	\N	Saint-Martin-d'Écublei	61
24809	424	\N	Saint-Martin-des-Landes	61
24810	425	\N	Saint-Martin-des-Pézerits	61
24811	426	\N	Saint-Martin-du-Vieux-Bellême	61
24812	427	\N	Saint-Martin-l'Aiguillon	61
24813	428	\N	Saint-Maurice-du-Désert	61
24814	429	\N	Saint-Maurice-lès-Charencey	61
24815	430	\N	Saint-Maurice-sur-Huisne	61
24816	431	\N	Saint-Michel-des-Andaines	61
24817	432	\N	Saint-Michel-Tubuf	61
24818	433	\N	Saint-Nicolas-des-Bois	61
24819	434	\N	Saint-Nicolas-des-Laitiers	61
24820	435	\N	Saint-Nicolas-de-Sommaire	61
24821	436	\N	Sainte-Opportune	61
24822	437	\N	Saint-Ouen-de-la-Cour	61
24823	438	\N	Saint-Ouen-de-Sécherouvre	61
24824	439	\N	Saint-Ouen-le-Brisoult	61
24825	440	\N	Saint-Ouen-sur-Iton	61
24826	441	\N	Saint-Ouen-sur-Maire	61
24827	442	\N	Saint-Patrice-du-Désert	61
24828	443	\N	Saint-Paul	61
24829	444	\N	Saint-Philbert-sur-Orne	61
24830	445	\N	Saint-Pierre-d'Entremont	61
24831	446	\N	Saint-Pierre-des-Loges	61
24832	447	\N	Saint-Pierre-du-Regard	61
24833	448	\N	Saint-Pierre-la-Bruyère	61
24834	449	\N	Saint-Pierre-la-Rivière	61
24835	450	\N	Saint-Quentin-de-Blavou	61
24836	451	\N	Saint-Quentin-les-Chardonnets	61
24837	452	\N	Saint-Roch-sur-Égrenne	61
24838	453	\N	Saint-Sauveur-de-Carrouges	61
24839	454	\N	Sainte-Scolasse-sur-Sarthe	61
24840	455	\N	Saint-Siméon	61
24841	456	\N	Saint-Sulpice-sur-Risle	61
24842	457	\N	Saint-Symphorien-des-Bruyères	61
24843	458	\N	Saint-Victor-de-Réno	61
24844	459	\N	Saires-la-Verrerie	61
24845	460	\N	Sap	61
24846	461	\N	Sap-André	61
24847	462	\N	Sarceaux	61
24848	463	\N	Sauvagère	61
24849	464	\N	Sées	61
24850	465	\N	Ségrie-Fontaine	61
24851	466	\N	Selle-la-Forge	61
24852	467	\N	Semallé	61
24853	468	\N	Sentilly	61
24854	469	\N	Sept-Forges	61
24855	470	\N	Serans	61
24856	471	\N	Sérigny	61
24857	472	\N	Sévigny	61
24858	473	\N	Sevrai	61
24859	474	\N	Silly-en-Gouffern	61
24860	475	\N	Soligny-la-Trappe	61
24861	476	\N	Suré	61
24862	477	\N	Survie	61
24863	478	\N	Taillebois	61
24864	479	\N	Tanques	61
24865	480	\N	Tanville	61
24866	481	\N	Tellières-le-Plessis	61
24867	482	\N	Tessé-Froulay	61
24868	483	\N	Bagnoles-de-l'Orne	61
24869	484	\N	Theil	61
24870	485	\N	Ticheville	61
24871	486	\N	Tinchebray	61
24872	487	\N	Torchamp	61
24873	488	\N	Touquettes	61
24874	489	\N	Tourailles	61
24875	490	\N	Tournai-sur-Dive	61
24876	491	\N	Tourouvre	61
24877	492	\N	Trémont	61
24878	493	\N	Trinité-des-Laitiers	61
24879	494	\N	Trun	61
24880	496	\N	Urou-et-Crennes	61
24881	497	\N	Valframbert	61
24882	498	\N	Vaunoise	61
24883	499	\N	Ventes-de-Bourse	61
24884	500	\N	Ventrouze	61
24885	501	\N	Verrières	61
24886	502	\N	Vidai	61
24887	503	\N	Vieux-Pont	61
24888	504	\N	Villebadin	61
24889	505	\N	Villedieu-lès-Bailleul	61
24890	506	\N	Villers-en-Ouche	61
24891	507	\N	Villiers-sous-Mortagne	61
24892	508	\N	Vimoutiers	61
24893	509	\N	Vingt-Hanaps	61
24894	510	\N	Vitrai-sous-Laigle	61
24895	511	\N	Vrigny	61
24896	512	\N	Yveteaux	61
24897	513	\N	Yvrandes	61
24898	001	\N	Ablain-Saint-Nazaire	62
24899	002	\N	Ablainzevelle	62
24900	003	\N	Acheville	62
24901	004	\N	Achicourt	62
24902	005	\N	Achiet-le-Grand	62
24903	006	\N	Achiet-le-Petit	62
24904	007	\N	Acq	62
24905	008	\N	Acquin-Westbécourt	62
24906	009	\N	Adinfer	62
24907	010	\N	Affringues	62
24908	011	\N	Agnez-lès-Duisans	62
24909	012	\N	Agnières	62
24910	013	\N	Agny	62
24911	014	\N	Aire-sur-la-Lys	62
24912	015	\N	Airon-Notre-Dame	62
24913	016	\N	Airon-Saint-Vaast	62
24914	017	\N	Aix-en-Ergny	62
24915	018	\N	Aix-en-Issart	62
24916	019	\N	Aix-Noulette	62
24917	020	\N	Alembon	62
24918	021	\N	Alette	62
24919	022	\N	Alincthun	62
24920	023	\N	Allouagne	62
24921	024	\N	Alquines	62
24922	025	\N	Ambleteuse	62
24923	026	\N	Ambricourt	62
24924	027	\N	Ambrines	62
24925	028	\N	Ames	62
24926	029	\N	Amettes	62
24927	030	\N	Amplier	62
24928	031	\N	Andres	62
24929	032	\N	Angres	62
24930	033	\N	Annay	62
24931	034	\N	Annequin	62
24932	035	\N	Annezin	62
24933	036	\N	Anvin	62
24934	037	\N	Anzin-Saint-Aubin	62
24935	038	\N	Ardres	62
24936	039	\N	Arleux-en-Gohelle	62
24937	040	\N	Arques	62
24938	041	\N	Arras	62
24939	042	\N	Athies	62
24940	043	\N	Attaques	62
24941	044	\N	Attin	62
24942	045	\N	Aubigny-en-Artois	62
24943	046	\N	Aubin-Saint-Vaast	62
24944	047	\N	Aubrometz	62
24945	048	\N	Auchel	62
24946	049	\N	Auchy-au-Bois	62
24947	050	\N	Auchy-lès-Hesdin	62
24948	051	\N	Auchy-les-Mines	62
24949	052	\N	Audembert	62
24950	053	\N	Audincthun	62
24951	054	\N	Audinghen	62
24952	055	\N	Audrehem	62
24953	056	\N	Audresselles	62
24954	057	\N	Audruicq	62
24955	058	\N	Aumerval	62
24956	059	\N	Autingues	62
24957	060	\N	Auxi-le-Château	62
24958	061	\N	Averdoingt	62
24959	062	\N	Avesnes	62
24960	063	\N	Avesnes-le-Comte	62
24961	064	\N	Avesnes-lès-Bapaume	62
24962	065	\N	Avion	62
24963	066	\N	Avondance	62
24964	067	\N	Avroult	62
24965	068	\N	Ayette	62
24966	069	\N	Azincourt	62
24967	070	\N	Bailleul-aux-Cornailles	62
24968	071	\N	Bailleul-lès-Pernes	62
24969	072	\N	Bailleulmont	62
24970	073	\N	Bailleul-Sir-Berthoult	62
24971	074	\N	Bailleulval	62
24972	075	\N	Baincthun	62
24973	076	\N	Bainghen	62
24974	077	\N	Bajus	62
24975	078	\N	Balinghem	62
24976	079	\N	Bancourt	62
24977	080	\N	Bapaume	62
24978	081	\N	Baralle	62
24979	082	\N	Barastre	62
24980	083	\N	Barlin	62
24981	084	\N	Barly	62
24982	085	\N	Basseux	62
24983	086	\N	Bavincourt	62
24984	087	\N	Bayenghem-lès-Éperlecques	62
24985	088	\N	Bayenghem-lès-Seninghem	62
24986	089	\N	Bazinghen	62
24987	090	\N	Béalencourt	62
24988	091	\N	Beaudricourt	62
24989	092	\N	Beaufort-Blavincourt	62
24990	093	\N	Beaulencourt	62
24991	094	\N	Beaumerie-Saint-Martin	62
24992	095	\N	Beaumetz-lès-Aire	62
24993	096	\N	Beaumetz-lès-Cambrai	62
24994	097	\N	Beaumetz-lès-Loges	62
24995	099	\N	Beaurains	62
24996	100	\N	Beaurainville	62
24997	101	\N	Beauvois	62
24998	102	\N	Bécourt	62
24999	103	\N	Béhagnies	62
25000	104	\N	Bellebrune	62
25001	105	\N	Belle-et-Houllefort	62
25002	106	\N	Bellonne	62
25003	107	\N	Bénifontaine	62
25004	108	\N	Berck	62
25005	109	\N	Bergueneuse	62
25006	111	\N	Berlencourt-le-Cauroy	62
25007	112	\N	Berles-au-Bois	62
25008	113	\N	Berles-Monchel	62
25009	114	\N	Bermicourt	62
25010	115	\N	Berneville	62
25011	116	\N	Bernieulles	62
25012	117	\N	Bertincourt	62
25013	118	\N	Béthonsart	62
25014	119	\N	Béthune	62
25015	120	\N	Beugin	62
25016	121	\N	Beugnâtre	62
25017	122	\N	Beugny	62
25018	123	\N	Beussent	62
25019	124	\N	Beutin	62
25020	125	\N	Beuvrequen	62
25021	126	\N	Beuvry	62
25022	127	\N	Bezinghem	62
25023	128	\N	Biache-Saint-Vaast	62
25024	129	\N	Biefvillers-lès-Bapaume	62
25025	130	\N	Bienvillers-au-Bois	62
25026	131	\N	Bihucourt	62
25027	132	\N	Billy-Berclau	62
25028	133	\N	Billy-Montigny	62
25029	134	\N	Bimont	62
25030	135	\N	Blairville	62
25031	137	\N	Blangerval-Blangermont	62
25032	138	\N	Blangy-sur-Ternoise	62
25033	139	\N	Blendecques	62
25034	140	\N	Bléquin	62
25035	141	\N	Blessy	62
25036	142	\N	Blingel	62
25037	143	\N	Boffles	62
25038	144	\N	Boiry-Becquerelle	62
25039	145	\N	Boiry-Notre-Dame	62
25040	146	\N	Boiry-Saint-Martin	62
25041	147	\N	Boiry-Sainte-Rictrude	62
25042	148	\N	Bois-Bernard	62
25043	149	\N	Boisdinghem	62
25044	150	\N	Boisjean	62
25045	151	\N	Boisleux-au-Mont	62
25046	152	\N	Boisleux-Saint-Marc	62
25047	153	\N	Bomy	62
25048	154	\N	Bonnières	62
25049	155	\N	Bonningues-lès-Ardres	62
25050	156	\N	Bonningues-lès-Calais	62
25051	157	\N	Boubers-lès-Hesmond	62
25052	158	\N	Boubers-sur-Canche	62
25053	160	\N	Boulogne-sur-Mer	62
25054	161	\N	Bouquehault	62
25055	162	\N	Bourecq	62
25056	163	\N	Bouret-sur-Canche	62
25057	164	\N	Bourlon	62
25058	165	\N	Bournonville	62
25059	166	\N	Bours	62
25060	167	\N	Boursin	62
25061	168	\N	Bourthes	62
25062	169	\N	Bouvelinghem	62
25063	170	\N	Bouvigny-Boyeffles	62
25064	171	\N	Boyaval	62
25065	172	\N	Boyelles	62
25066	173	\N	Brebières	62
25067	174	\N	Brêmes	62
25068	175	\N	Brévillers	62
25069	176	\N	Bréxent-Énocq	62
25070	177	\N	Brimeux	62
25071	178	\N	Bruay-la-Buissière	62
25072	179	\N	Brunembert	62
25073	180	\N	Brias	62
25074	181	\N	Bucquoy	62
25075	182	\N	Buire-au-Bois	62
25076	183	\N	Buire-le-Sec	62
25077	184	\N	Buissy	62
25078	185	\N	Bullecourt	62
25079	186	\N	Bully-les-Mines	62
25080	187	\N	Buneville	62
25081	188	\N	Burbure	62
25082	189	\N	Bus	62
25083	190	\N	Busnes	62
25084	191	\N	Caffiers	62
25085	192	\N	Cagnicourt	62
25086	193	\N	Calais	62
25087	194	\N	Calonne-Ricouart	62
25088	195	\N	Calonne-sur-la-Lys	62
25089	196	\N	Calotterie	62
25090	197	\N	Camblain-Châtelain	62
25091	198	\N	Cambligneul	62
25092	199	\N	Camblain-l'Abbé	62
25093	200	\N	Cambrin	62
25094	201	\N	Camiers	62
25095	202	\N	Campagne-lès-Boulonnais	62
25096	203	\N	Campagne-lès-Guines	62
25097	204	\N	Campagne-lès-Hesdin	62
25098	205	\N	Campagne-lès-Wardrecques	62
25099	206	\N	Campigneulles-les-Grandes	62
25100	207	\N	Campigneulles-les-Petites	62
25101	208	\N	Canettemont	62
25102	209	\N	Canlers	62
25103	210	\N	Canteleux	62
25104	211	\N	Capelle-Fermont	62
25105	212	\N	Capelle-lès-Hesdin	62
25106	213	\N	Carency	62
25107	214	\N	Carly	62
25108	215	\N	Carvin	62
25109	216	\N	Cauchie	62
25110	217	\N	Cauchy-à-la-Tour	62
25111	218	\N	Caucourt	62
25112	219	\N	Caumont	62
25113	220	\N	Cavron-Saint-Martin	62
25114	221	\N	Chelers	62
25115	222	\N	Chériennes	62
25116	223	\N	Chérisy	62
25117	224	\N	Chocques	62
25118	225	\N	Clairmarais	62
25119	226	\N	Clarques	62
25120	227	\N	Clenleu	62
25121	228	\N	Clerques	62
25122	229	\N	Cléty	62
25123	230	\N	Colembert	62
25124	231	\N	Colline-Beaumont	62
25125	232	\N	Comté	62
25126	233	\N	Conchil-le-Temple	62
25127	234	\N	Conchy-sur-Canche	62
25128	235	\N	Condette	62
25129	236	\N	Contes	62
25130	237	\N	Conteville-lès-Boulogne	62
25131	238	\N	Conteville-en-Ternois	62
25132	239	\N	Coquelles	62
25133	240	\N	Corbehem	62
25134	241	\N	Cormont	62
25135	242	\N	Couin	62
25136	243	\N	Coullemont	62
25137	244	\N	Coulogne	62
25138	245	\N	Coulomby	62
25139	246	\N	Coupelle-Neuve	62
25140	247	\N	Coupelle-Vieille	62
25141	248	\N	Courcelles-le-Comte	62
25142	249	\N	Courcelles-lès-Lens	62
25143	250	\N	Courrières	62
25144	251	\N	Courset	62
25145	252	\N	Couture	62
25146	253	\N	Couturelle	62
25147	254	\N	Coyecques	62
25148	255	\N	Crémarest	62
25149	256	\N	Crépy	62
25150	257	\N	Créquy	62
25151	258	\N	Croisette	62
25152	259	\N	Croisilles	62
25153	260	\N	Croix-en-Ternois	62
25154	261	\N	Cucq	62
25155	262	\N	Cuinchy	62
25156	263	\N	Dainville	62
25157	264	\N	Dannes	62
25158	265	\N	Delettes	62
25159	266	\N	Denier	62
25160	267	\N	Dennebrucq	62
25161	268	\N	Desvres	62
25162	269	\N	Diéval	62
25163	270	\N	Divion	62
25164	271	\N	Dohem	62
25165	272	\N	Douchy-lès-Ayette	62
25166	273	\N	Doudeauville	62
25167	274	\N	Dourges	62
25168	275	\N	Douriez	62
25169	276	\N	Douvrin	62
25170	277	\N	Drocourt	62
25171	278	\N	Drouvin-le-Marais	62
25172	279	\N	Duisans	62
25173	280	\N	Dury	62
25174	281	\N	Echinghen	62
25175	282	\N	Éclimeux	62
25176	283	\N	Écoivres	62
25177	284	\N	Écourt-Saint-Quentin	62
25178	285	\N	Écoust-Saint-Mein	62
25179	286	\N	Ecquedecques	62
25180	288	\N	Ecques	62
25181	289	\N	Écuires	62
25182	290	\N	Écurie	62
25183	291	\N	Éleu-dit-Leauwette	62
25184	292	\N	Elnes	62
25185	293	\N	Embry	62
25186	294	\N	Enguinegatte	62
25187	295	\N	Enquin-les-Mines	62
25188	296	\N	Enquin-sur-Baillons	62
25189	297	\N	Éperlecques	62
25190	298	\N	Épinoy	62
25191	299	\N	Eps	62
25192	300	\N	Équihen-Plage	62
25193	301	\N	Équirre	62
25194	302	\N	Ergny	62
25195	303	\N	Érin	62
25196	304	\N	Erny-Saint-Julien	62
25197	306	\N	Ervillers	62
25198	307	\N	Escalles	62
25199	308	\N	Escuilles	62
25200	309	\N	Esquerdes	62
25201	310	\N	Essars	62
25202	311	\N	Estevelles	62
25203	312	\N	Estrée	62
25204	313	\N	Estrée-Blanche	62
25205	314	\N	Estrée-Cauchy	62
25206	315	\N	Estréelles	62
25207	316	\N	Estrée-Wamin	62
25208	317	\N	Étaing	62
25209	318	\N	Étaples	62
25210	319	\N	Éterpigny	62
25211	320	\N	Étrun	62
25212	321	\N	Évin-Malmaison	62
25213	322	\N	Famechon	62
25214	323	\N	Fampoux	62
25215	324	\N	Farbus	62
25216	325	\N	Fauquembergues	62
25217	326	\N	Favreuil	62
25218	327	\N	Febvin-Palfart	62
25219	328	\N	Ferfay	62
25220	329	\N	Ferques	62
25221	330	\N	Festubert	62
25222	331	\N	Feuchy	62
25223	332	\N	Ficheux	62
25224	333	\N	Fiefs	62
25225	334	\N	Fiennes	62
25226	335	\N	Fillièvres	62
25227	336	\N	Fléchin	62
25228	337	\N	Flers	62
25229	338	\N	Fleurbaix	62
25230	339	\N	Fleury	62
25231	340	\N	Floringhem	62
25232	341	\N	Foncquevillers	62
25233	342	\N	Fontaine-lès-Boulans	62
25234	343	\N	Fontaine-lès-Croisilles	62
25235	344	\N	Fontaine-lès-Hermans	62
25236	345	\N	Fontaine-l'Étalon	62
25237	346	\N	Fortel-en-Artois	62
25238	347	\N	Fosseux	62
25239	348	\N	Foufflin-Ricametz	62
25240	349	\N	Fouquereuil	62
25241	350	\N	Fouquières-lès-Béthune	62
25242	351	\N	Fouquières-lès-Lens	62
25243	352	\N	Framecourt	62
25244	353	\N	Frémicourt	62
25245	354	\N	Frencq	62
25246	355	\N	Fresnes-lès-Montauban	62
25247	356	\N	Fresnicourt-le-Dolmen	62
25248	357	\N	Fresnoy	62
25249	358	\N	Fresnoy-en-Gohelle	62
25250	359	\N	Fressin	62
25251	360	\N	Fréthun	62
25252	361	\N	Frévent	62
25253	362	\N	Frévillers	62
25254	363	\N	Frévin-Capelle	62
25255	364	\N	Fruges	62
25256	365	\N	Galametz	62
25257	366	\N	Gauchin-Légal	62
25258	367	\N	Gauchin-Verloingt	62
25259	368	\N	Gaudiempré	62
25260	369	\N	Gavrelle	62
25261	370	\N	Gennes-Ivergny	62
25262	371	\N	Givenchy-en-Gohelle	62
25263	372	\N	Givenchy-le-Noble	62
25264	373	\N	Givenchy-lès-la-Bassée	62
25265	374	\N	Gomiécourt	62
25266	375	\N	Gommecourt	62
25267	376	\N	Gonnehem	62
25268	377	\N	Gosnay	62
25269	378	\N	Gouves	62
25270	379	\N	Gouy-en-Artois	62
25271	380	\N	Gouy-Servins	62
25272	381	\N	Gouy-en-Ternois	62
25273	382	\N	Gouy-Saint-André	62
25274	383	\N	Gouy-sous-Bellonne	62
25275	384	\N	Graincourt-lès-Havrincourt	62
25276	385	\N	Grand-Rullecourt	62
25277	386	\N	Grenay	62
25278	387	\N	Grévillers	62
25279	388	\N	Grigny	62
25280	389	\N	Grincourt-lès-Pas	62
25281	390	\N	Groffliers	62
25282	391	\N	Guarbecque	62
25283	392	\N	Guémappe	62
25284	393	\N	Guemps	62
25285	395	\N	Guigny	62
25286	396	\N	Guinecourt	62
25287	397	\N	Guînes	62
25288	398	\N	Guisy	62
25289	399	\N	Habarcq	62
25290	400	\N	Haillicourt	62
25291	401	\N	Haisnes	62
25292	402	\N	Halinghen	62
25293	403	\N	Hallines	62
25294	404	\N	Halloy	62
25295	405	\N	Hamblain-les-Prés	62
25296	406	\N	Hamelincourt	62
25297	407	\N	Ham-en-Artois	62
25298	408	\N	Hames-Boucres	62
25299	409	\N	Hannescamps	62
25300	410	\N	Haplincourt	62
25301	411	\N	Haravesnes	62
25302	412	\N	Hardinghen	62
25303	413	\N	Harnes	62
25304	414	\N	Haucourt	62
25305	415	\N	Haute-Avesnes	62
25306	416	\N	Hautecloque	62
25307	418	\N	Hauteville	62
25308	419	\N	Haut-Loquin	62
25309	421	\N	Havrincourt	62
25310	422	\N	Hébuterne	62
25311	423	\N	Helfaut	62
25312	424	\N	Hendecourt-lès-Cagnicourt	62
25313	425	\N	Hendecourt-lès-Ransart	62
25314	426	\N	Héninel	62
25315	427	\N	Hénin-Beaumont	62
25316	428	\N	Hénin-sur-Cojeul	62
25317	429	\N	Henneveux	62
25318	430	\N	Hénu	62
25319	431	\N	Herbelles	62
25320	432	\N	Herbinghen	62
25321	433	\N	Héricourt	62
25322	434	\N	Herlière	62
25323	435	\N	Herlincourt	62
25324	436	\N	Herlin-le-Sec	62
25325	437	\N	Herly	62
25326	438	\N	Hermaville	62
25327	439	\N	Hermelinghen	62
25328	440	\N	Hermies	62
25329	441	\N	Hermin	62
25330	442	\N	Hernicourt	62
25331	443	\N	Hersin-Coupigny	62
25332	444	\N	Hervelinghen	62
25333	445	\N	Hesdigneul-lès-Béthune	62
25334	446	\N	Hesdigneul-lès-Boulogne	62
25335	447	\N	Hesdin	62
25336	448	\N	Hesdin-l'Abbé	62
25337	449	\N	Hesmond	62
25338	450	\N	Hestrus	62
25339	451	\N	Heuchin	62
25340	452	\N	Heuringhem	62
25341	453	\N	Hézecques	62
25342	454	\N	Hinges	62
25343	455	\N	Hocquinghen	62
25344	456	\N	Houchin	62
25345	457	\N	Houdain	62
25346	458	\N	Houlle	62
25347	459	\N	Houvin-Houvigneul	62
25348	460	\N	Hubersent	62
25349	461	\N	Huby-Saint-Leu	62
25350	462	\N	Huclier	62
25351	463	\N	Hucqueliers	62
25352	464	\N	Hulluch	62
25353	465	\N	Humbercamps	62
25354	466	\N	Humbert	62
25355	467	\N	Humeruille	62
25356	468	\N	Humières	62
25357	469	\N	Inchy-en-Artois	62
25358	470	\N	Incourt	62
25359	471	\N	Inghem	62
25360	472	\N	Inxent	62
25361	473	\N	Isbergues	62
25362	474	\N	Isques	62
25363	475	\N	Ivergny	62
25364	476	\N	Izel-lès-Équerchin	62
25365	477	\N	Izel-les-Hameaux	62
25366	478	\N	Journy	62
25367	479	\N	Labeuvrière	62
25368	480	\N	Labourse	62
25369	481	\N	Labroye	62
25370	483	\N	Lacres	62
25371	484	\N	Lagnicourt-Marcel	62
25372	485	\N	Laires	62
25373	486	\N	Lambres	62
25374	487	\N	Landrethun-le-Nord	62
25375	488	\N	Landrethun-lès-Ardres	62
25376	489	\N	Lapugnoy	62
25377	490	\N	Lattre-Saint-Quentin	62
25378	491	\N	Laventie	62
25379	492	\N	Lebiez	62
25380	493	\N	Lebucquière	62
25381	494	\N	Léchelle	62
25382	495	\N	Ledinghem	62
25383	496	\N	Lefaux	62
25384	497	\N	Leforest	62
25385	498	\N	Lens	62
25386	499	\N	Lépine	62
25387	500	\N	Lespesses	62
25388	501	\N	Lespinoy	62
25389	502	\N	Lestrem	62
25390	503	\N	Leubringhen	62
25391	504	\N	Leulinghem	62
25392	505	\N	Leulinghen-Bernes	62
25393	506	\N	Licques	62
25394	507	\N	Liencourt	62
25395	508	\N	Lières	62
25396	509	\N	Liettres	62
25397	510	\N	Liévin	62
25398	511	\N	Lignereuil	62
25399	512	\N	Ligny-lès-Aire	62
25400	513	\N	Ligny-sur-Canche	62
25401	514	\N	Ligny-Saint-Flochel	62
25402	515	\N	Ligny-Thilloy	62
25403	516	\N	Lillers	62
25404	517	\N	Linghem	62
25405	518	\N	Linzeux	62
25406	519	\N	Lisbourg	62
25407	520	\N	Locon	62
25408	521	\N	Loge	62
25409	522	\N	Loison-sur-Créquoise	62
25410	523	\N	Loison-sous-Lens	62
25411	524	\N	Longfossé	62
25412	525	\N	Longuenesse	62
25413	526	\N	Longueville	62
25414	527	\N	Longvilliers	62
25415	528	\N	Loos-en-Gohelle	62
25416	529	\N	Lorgies	62
25417	530	\N	Lottinghen	62
25418	531	\N	Louches	62
25419	532	\N	Lozinghem	62
25420	533	\N	Lugy	62
25421	534	\N	Lumbres	62
25422	535	\N	Madelaine-sous-Montreuil	62
25423	536	\N	Magnicourt-en-Comte	62
25424	537	\N	Magnicourt-sur-Canche	62
25425	538	\N	Maintenay	62
25426	539	\N	Maisnil	62
25427	540	\N	Maisnil-lès-Ruitz	62
25428	541	\N	Maisoncelle	62
25429	542	\N	Maizières	62
25430	543	\N	Mametz	62
25431	544	\N	Manin	62
25432	545	\N	Maninghem	62
25433	546	\N	Maninghen-Henne	62
25434	547	\N	Marant	62
25435	548	\N	Marck	62
25436	549	\N	Marconne	62
25437	550	\N	Marconnelle	62
25438	551	\N	Marenla	62
25439	552	\N	Maresquel-Ecquemicourt	62
25440	553	\N	Marest	62
25441	554	\N	Maresville	62
25442	555	\N	Marles-les-Mines	62
25443	556	\N	Marles-sur-Canche	62
25444	557	\N	Maruil	62
25445	558	\N	Marquay	62
25446	559	\N	Marquion	62
25447	560	\N	Marquise	62
25448	561	\N	Martinpuich	62
25449	562	\N	Matringhem	62
25450	563	\N	Mazingarbe	62
25451	564	\N	Mazinghem	62
25452	565	\N	Mencas	62
25453	566	\N	Menneville	62
25454	567	\N	Mentque-Nortbécourt	62
25455	568	\N	Mercatel	62
25456	569	\N	Merck-Saint-Liévin	62
25457	570	\N	Méricourt	62
25458	571	\N	Merlimont	62
25459	572	\N	Metz-en-Couture	62
25460	573	\N	Meurchin	62
25461	574	\N	Mingoval	62
25462	576	\N	Moncheaux-lès-Frévent	62
25463	577	\N	Monchel-sur-Canche	62
25464	578	\N	Monchiet	62
25465	579	\N	Monchy-au-Bois	62
25466	580	\N	Monchy-Breton	62
25467	581	\N	Monchy-Cayeux	62
25468	582	\N	Monchy-le-Preux	62
25469	583	\N	Mondicourt	62
25470	584	\N	Mont-Bernanchon	62
25471	585	\N	Montcavrel	62
25472	586	\N	Montenescourt	62
25473	587	\N	Montigny-en-Gohelle	62
25474	588	\N	Montreuil	62
25475	589	\N	Mont-Saint-Éloi	62
25476	590	\N	Monts-en-Ternois	62
25477	591	\N	Morchies	62
25478	592	\N	Moringhem	62
25479	593	\N	Morval	62
25480	594	\N	Mory	62
25481	595	\N	Moulle	62
25482	596	\N	Mouriez	62
25483	597	\N	Moyenneville	62
25484	598	\N	Muncq-Nieurlet	62
25485	599	\N	Nabringhen	62
25486	600	\N	Nédon	62
25487	601	\N	Nédonchel	62
25488	602	\N	Nempont-Saint-Firmin	62
25489	603	\N	Nesles	62
25490	604	\N	Neufchâtel-Hardelot	62
25491	605	\N	Neulette	62
25492	606	\N	Neuve-Chapelle	62
25493	607	\N	Neuville-au-Cornet	62
25494	608	\N	Neuville-Bourjonval	62
25495	609	\N	Neuville-Saint-Vaast	62
25496	610	\N	Neuville-sous-Montreuil	62
25497	611	\N	Neuville-Vitasse	62
25498	612	\N	Neuvireuil	62
25499	613	\N	Nielles-lès-Bléquin	62
25500	614	\N	Nielles-lès-Ardres	62
25501	615	\N	Nielles-lès-Calais	62
25502	616	\N	Nux-lès-Auxi	62
25503	617	\N	Nux-les-Mines	62
25504	618	\N	Nordausques	62
25505	619	\N	Noreuil	62
25506	620	\N	Norrent-Fontes	62
25507	621	\N	Nortkerque	62
25508	622	\N	Nort-Leulinghem	62
25509	623	\N	Nouvelle-Église	62
25510	624	\N	Noyelles-Godault	62
25511	625	\N	Noyelles-lès-Humières	62
25512	626	\N	Noyelles-lès-Vermelles	62
25513	627	\N	Noyelles-sous-Bellonne	62
25514	628	\N	Noyelles-sous-Lens	62
25515	629	\N	Noyellette	62
25516	630	\N	Noyelle-Vion	62
25517	631	\N	Nuncq-Hautecôte	62
25518	632	\N	Oblinghem	62
25519	633	\N	uf-en-Ternois	62
25520	634	\N	Offekerque	62
25521	635	\N	Offin	62
25522	636	\N	Offrethun	62
25523	637	\N	Oignies	62
25524	638	\N	Oisy-le-Verger	62
25525	639	\N	Oppy	62
25526	640	\N	Orville	62
25527	641	\N	Ostreville	62
25528	642	\N	Ourton	62
25529	643	\N	Outreau	62
25530	644	\N	Ouve-Wirquin	62
25531	645	\N	Oye-Plage	62
25532	646	\N	Palluel	62
25533	647	\N	Parcq	62
25534	648	\N	Parenty	62
25535	649	\N	Pas-en-Artois	62
25536	650	\N	Pelves	62
25537	651	\N	Penin	62
25538	652	\N	Pernes	62
25539	653	\N	Pernes-lès-Boulogne	62
25540	654	\N	Peuplingues	62
25541	655	\N	Pierremont	62
25542	656	\N	Pihem	62
25543	657	\N	Pihen-lès-Guînes	62
25544	658	\N	Pittefaux	62
25545	659	\N	Planques	62
25546	660	\N	Plouvain	62
25547	661	\N	Bouin-Plumoison	62
25548	662	\N	Polincove	62
25549	663	\N	Pommera	62
25550	664	\N	Pommier	62
25551	665	\N	Ponchel	62
25552	666	\N	Pont-à-Vendin	62
25553	667	\N	Portel	62
25554	668	\N	Prédefin	62
25555	669	\N	Pressy	62
25556	670	\N	Preures	62
25557	671	\N	Pronville	62
25558	672	\N	Puisieux	62
25559	673	\N	Quéant	62
25560	674	\N	Quelmes	62
25561	675	\N	Quercamps	62
25562	676	\N	Quernes	62
25563	677	\N	Quesnoy-en-Artois	62
25564	678	\N	Quesques	62
25565	679	\N	Questrecques	62
25566	680	\N	Quiéry-la-Motte	62
25567	681	\N	Quiestède	62
25568	682	\N	Quilen	62
25569	683	\N	Quux-Haut-Maînil	62
25570	684	\N	Racquinghem	62
25571	685	\N	Radinghem	62
25572	686	\N	Ramecourt	62
25573	688	\N	Rang-du-Fliers	62
25574	689	\N	Ransart	62
25575	690	\N	Raye-sur-Authie	62
25576	691	\N	Rebecques	62
25577	692	\N	Rebergues	62
25578	693	\N	Rebreuve-Ranchicourt	62
25579	694	\N	Rebreuve-sur-Canche	62
25580	695	\N	Rebreuviette	62
25581	696	\N	Reclinghem	62
25582	697	\N	Récourt	62
25583	698	\N	Recques-sur-Course	62
25584	699	\N	Recques-sur-Hem	62
25585	700	\N	Regnauville	62
25586	701	\N	Rely	62
25587	702	\N	Remilly-Wirquin	62
25588	703	\N	Rémy	62
25589	704	\N	Renty	62
25590	705	\N	Rety	62
25591	706	\N	Richebourg	62
25592	708	\N	Riencourt-lès-Bapaume	62
25593	709	\N	Riencourt-lès-Cagnicourt	62
25594	710	\N	Rimboval	62
25595	711	\N	Rinxent	62
25596	712	\N	Rivière	62
25597	713	\N	Robecq	62
25598	714	\N	Roclincourt	62
25599	715	\N	Rocquigny	62
25600	716	\N	Rodelinghem	62
25601	717	\N	Roëllecourt	62
25602	718	\N	Rux	62
25603	719	\N	Rollancourt	62
25604	720	\N	Rombly	62
25605	721	\N	Roquetoire	62
25606	722	\N	Rougefay	62
25607	723	\N	Roussent	62
25608	724	\N	Rouvroy	62
25609	725	\N	Royon	62
25610	726	\N	Ruisseauville	62
25611	727	\N	Ruitz	62
25612	728	\N	Rumaucourt	62
25613	729	\N	Rumilly	62
25614	730	\N	Ruminghem	62
25615	731	\N	Ruyaulcourt	62
25616	732	\N	Sachin	62
25617	733	\N	Sailly-au-Bois	62
25618	734	\N	Sailly-en-Ostrevent	62
25619	735	\N	Sailly-Labourse	62
25620	736	\N	Sailly-sur-la-Lys	62
25621	737	\N	Sains-en-Gohelle	62
25622	738	\N	Sains-lès-Fressin	62
25623	739	\N	Sains-lès-Marquion	62
25624	740	\N	Sains-lès-Pernes	62
25625	741	\N	Saint-Amand	62
25626	742	\N	Saint-Aubin	62
25627	743	\N	Sainte-Austreberthe	62
25628	744	\N	Sainte-Catherine	62
25629	745	\N	Saint-Denux	62
25630	746	\N	Saint-Étienne-au-Mont	62
25631	747	\N	Saint-Floris	62
25632	748	\N	Saint-Folquin	62
25633	749	\N	Saint-Georges	62
25634	750	\N	Saint-Hilaire-Cottes	62
25635	751	\N	Saint-Inglevert	62
25636	752	\N	Saint-Josse	62
25637	753	\N	Saint-Laurent-Blangy	62
25638	754	\N	Saint-Léger	62
25639	755	\N	Saint-Léonard	62
25640	756	\N	Sainte-Marie-Kerque	62
25641	757	\N	Saint-Martin-au-Laërt	62
25642	758	\N	Saint-Martin-Boulogne	62
25643	759	\N	Saint-Martin-Choquel	62
25644	760	\N	Saint-Martin-d'Hardinghem	62
25645	761	\N	Saint-Martin-sur-Cojeul	62
25646	762	\N	Saint-Michel-sous-Bois	62
25647	763	\N	Saint-Michel-sur-Ternoise	62
25648	764	\N	Saint-Nicolas	62
25649	765	\N	Saint-Omer	62
25650	766	\N	Saint-Omer-Capelle	62
25651	767	\N	Saint-Pol-sur-Ternoise	62
25652	768	\N	Saint-Rémy-au-Bois	62
25653	769	\N	Saint-Tricat	62
25654	770	\N	Saint-Venant	62
25655	771	\N	Sallaumines	62
25656	772	\N	Salperwick	62
25657	773	\N	Samer	62
25658	774	\N	Sangatte	62
25659	775	\N	Sanghen	62
25660	776	\N	Sapignies	62
25661	777	\N	Sars	62
25662	778	\N	Sars-le-Bois	62
25663	779	\N	Sarton	62
25664	780	\N	Sauchy-Cauchy	62
25665	781	\N	Sauchy-Lestrée	62
25666	782	\N	Saudemont	62
25667	783	\N	Saulchoy	62
25668	784	\N	Saulty	62
25669	785	\N	Savy-Berlette	62
25670	786	\N	Selles	62
25671	787	\N	Sempy	62
25672	788	\N	Seninghem	62
25673	789	\N	Senlecques	62
25674	790	\N	Senlis	62
25675	791	\N	Séricourt	62
25676	792	\N	Serques	62
25677	793	\N	Servins	62
25678	794	\N	Setques	62
25679	795	\N	Sibiville	62
25680	796	\N	Simencourt	62
25681	797	\N	Siracourt	62
25682	798	\N	Sombrin	62
25683	799	\N	Sorrus	62
25684	800	\N	Souastre	62
25685	801	\N	Souchez	62
25686	802	\N	Souich	62
25687	803	\N	Surques	62
25688	804	\N	Sus-Saint-Léger	62
25689	805	\N	Tangry	62
25690	806	\N	Tardinghen	62
25691	807	\N	Tatinghem	62
25692	808	\N	Teneur	62
25693	809	\N	Ternas	62
25694	810	\N	Thélus	62
25695	811	\N	Thérouanne	62
25696	812	\N	Thiembronne	62
25697	813	\N	Thieuloye	62
25698	814	\N	Thièvres	62
25699	815	\N	Tigny-Noyelle	62
25700	816	\N	Tilloy-lès-Hermaville	62
25701	817	\N	Tilloy-lès-Mofflaines	62
25702	818	\N	Tilly-Capelle	62
25703	819	\N	Tilques	62
25704	820	\N	Tincques	62
25705	821	\N	Tingry	62
25706	822	\N	Tollent	62
25707	823	\N	Torcy	62
25708	824	\N	Tortefontaine	62
25709	825	\N	Tortequesne	62
25710	826	\N	Touquet-Paris-Plage	62
25711	827	\N	Tournehem-sur-la-Hem	62
25712	828	\N	Tramecourt	62
25713	829	\N	Transloy	62
25714	830	\N	Trescault	62
25715	831	\N	Troisvaux	62
25716	832	\N	Tubersent	62
25717	833	\N	Vacquerie-le-Boucq	62
25718	834	\N	Vacqueriette-Erquières	62
25719	835	\N	Valhuon	62
25720	836	\N	Vaudricourt	62
25721	837	\N	Vaudringhem	62
25722	838	\N	Vaulx	62
25723	839	\N	Vaulx-Vraucourt	62
25724	840	\N	Vélu	62
25725	841	\N	Vendin-lès-Béthune	62
25726	842	\N	Vendin-le-Vieil	62
25727	843	\N	Verchin	62
25728	844	\N	Verchocq	62
25729	845	\N	Verlincthun	62
25730	846	\N	Vermelles	62
25731	848	\N	Verquin	62
25732	849	\N	Verton	62
25733	850	\N	Vieil-Hesdin	62
25734	851	\N	Vieille-Chapelle	62
25735	852	\N	Vieille-Église	62
25736	853	\N	Vieil-Moutier	62
25737	854	\N	Villers-au-Bois	62
25738	855	\N	Villers-au-Flos	62
25739	856	\N	Villers-Brûlin	62
25740	857	\N	Villers-Châtel	62
25741	858	\N	Villers-lès-Cagnicourt	62
25742	859	\N	Villers-l'Hôpital	62
25743	860	\N	Villers-Sir-Simon	62
25744	861	\N	Vimy	62
25745	862	\N	Vincly	62
25746	863	\N	Violaines	62
25747	864	\N	Vis-en-Artois	62
25748	865	\N	Vitry-en-Artois	62
25749	866	\N	Waben	62
25750	867	\N	Wacquinghen	62
25751	868	\N	Wail	62
25752	869	\N	Wailly	62
25753	870	\N	Wailly-Beaucamp	62
25754	871	\N	Wambercourt	62
25755	872	\N	Wamin	62
25756	873	\N	Wancourt	62
25757	874	\N	Wanquetin	62
25758	875	\N	Wardrecques	62
25759	876	\N	Warlencourt-Eaucourt	62
25760	877	\N	Warlincourt-lès-Pas	62
25761	878	\N	Warlus	62
25762	879	\N	Warluzel	62
25763	880	\N	Wast	62
25764	881	\N	Beauvoir-Wavans	62
25765	882	\N	Wavrans-sur-l'Aa	62
25766	883	\N	Wavrans-sur-Ternoise	62
25767	885	\N	Westrehem	62
25768	886	\N	Wicquinghem	62
25769	887	\N	Widehem	62
25770	888	\N	Wierre-au-Bois	62
25771	889	\N	Wierre-Effroy	62
25772	890	\N	Willeman	62
25773	891	\N	Willencourt	62
25774	892	\N	Willerval	62
25775	893	\N	Wimereux	62
25776	894	\N	Wimille	62
25777	895	\N	Wingles	62
25778	896	\N	Wirwignes	62
25779	897	\N	Wismes	62
25780	898	\N	Wisques	62
25781	899	\N	Wissant	62
25782	900	\N	Witternesse	62
25783	901	\N	Wittes	62
25784	902	\N	Wizernes	62
25785	903	\N	Zoteux	62
25786	904	\N	Zouafques	62
25787	905	\N	Zudausques	62
25788	906	\N	Zutkerque	62
25789	907	\N	Libercourt	62
25790	908	\N	Capelle-lès-Boulogne	62
25791	909	\N	Ytres	62
25792	001	\N	Aigueperse	63
25793	002	\N	Aix-la-Fayette	63
25794	003	\N	Ambert	63
25795	004	\N	Ancizes-Comps	63
25796	005	\N	Antoingt	63
25797	006	\N	Anzat-le-Luguet	63
25798	007	\N	Apchat	63
25799	008	\N	Arconsat	63
25800	009	\N	Ardes	63
25801	010	\N	Arlanc	63
25802	011	\N	Ars-les-Favets	63
25803	012	\N	Artonne	63
25804	013	\N	Aubiat	63
25805	014	\N	Aubière	63
25806	015	\N	Aubusson-d'Auvergne	63
25807	016	\N	Augerolles	63
25808	017	\N	Augnat	63
25809	018	\N	Aulhat-Saint-Privat	63
25810	019	\N	Aulnat	63
25811	020	\N	Aurières	63
25812	021	\N	Authezat	63
25813	022	\N	Auzat-la-Combelle	63
25814	023	\N	Auzelles	63
25815	024	\N	Avèze	63
25816	025	\N	Ayat-sur-Sioule	63
25817	026	\N	Aydat	63
25818	027	\N	Baffie	63
25819	028	\N	Bagnols	63
25820	029	\N	Bansat	63
25821	030	\N	Bas-et-Lezat	63
25822	031	\N	Beaulieu	63
25823	032	\N	Beaumont	63
25824	033	\N	Beaumont-lès-Randan	63
25825	034	\N	Beauregard-l'Évêque	63
25826	035	\N	Beauregard-Vendon	63
25827	036	\N	Bergonne	63
25828	037	\N	Bertignat	63
25829	038	\N	Besse-et-Saint-Anastaise	63
25830	039	\N	Beurières	63
25831	040	\N	Billom	63
25832	041	\N	Biollet	63
25833	042	\N	Blanzat	63
25834	043	\N	Blot-l'Église	63
25835	044	\N	Bongheat	63
25836	045	\N	Bort-l'Étang	63
25837	046	\N	Boudes	63
25838	047	\N	Bourboule	63
25839	048	\N	Bourg-Lastic	63
25840	049	\N	Bouzel	63
25841	050	\N	Brassac-les-Mines	63
25842	051	\N	Brenat	63
25843	052	\N	Breuil-sur-Couze	63
25844	053	\N	Briffons	63
25845	054	\N	Broc	63
25846	055	\N	Bromont-Lamothe	63
25847	056	\N	Brousse	63
25848	057	\N	Brugeron	63
25849	058	\N	Bulhon	63
25850	059	\N	Busséol	63
25851	060	\N	Bussières	63
25852	061	\N	Bussières-et-Pruns	63
25853	062	\N	Buxières-sous-Montaigut	63
25854	063	\N	Cébazat	63
25855	064	\N	Celle	63
25856	065	\N	Ceilloux	63
25857	066	\N	Celles-sur-Durolle	63
25858	067	\N	Cellette	63
25859	068	\N	Cellule	63
25860	069	\N	Cendre	63
25861	070	\N	Ceyrat	63
25862	071	\N	Ceyssat	63
25863	072	\N	Chabreloche	63
25864	073	\N	Chadeleuf	63
25865	074	\N	Chalus	63
25866	075	\N	Chamalières	63
25867	076	\N	Chambon-sur-Dolore	63
25868	077	\N	Chambon-sur-Lac	63
25869	078	\N	Chaméane	63
25870	079	\N	Champagnat-le-Jeune	63
25871	080	\N	Champeix	63
25872	081	\N	Champétières	63
25873	082	\N	Champs	63
25874	083	\N	Chanat-la-Mouteyre	63
25875	084	\N	Chanonat	63
25876	085	\N	Chapdes-Beaufort	63
25877	086	\N	Chapelle-Agnon	63
25878	087	\N	Chapelle-Marcousse	63
25879	088	\N	Chapelle-sur-Usson	63
25880	089	\N	Chappes	63
25881	090	\N	Chaptuzat	63
25882	091	\N	Charbonnier-les-Mines	63
25883	092	\N	Charbonnières-les-Varennes	63
25884	093	\N	Charbonnières-les-Vieilles	63
25885	094	\N	Charensat	63
25886	095	\N	Charnat	63
25887	096	\N	Chas	63
25888	097	\N	Chassagne	63
25889	098	\N	Chastreix	63
25890	099	\N	Châteaugay	63
25891	100	\N	Châteauneuf-les-Bains	63
25892	101	\N	Château-sur-Cher	63
25893	102	\N	Châteldon	63
25894	103	\N	Châtelguyon	63
25895	104	\N	Chaulme	63
25896	105	\N	Chaumont-le-Bourg	63
25897	106	\N	Chauriat	63
25898	107	\N	Chavaroux	63
25899	108	\N	Cheix	63
25900	109	\N	Chidrac	63
25901	110	\N	Cisternes-la-Forêt	63
25902	111	\N	Clémensat	63
25903	112	\N	Clerlande	63
25904	113	\N	Clermont-Ferrand	63
25905	114	\N	Collanges	63
25906	115	\N	Combrailles	63
25907	116	\N	Combronde	63
25908	117	\N	Compains	63
25909	118	\N	Condat-en-Combraille	63
25910	119	\N	Condat-lès-Montboissier	63
25911	120	\N	Corent	63
25912	121	\N	Coudes	63
25913	122	\N	Courgoul	63
25914	123	\N	Cournols	63
25915	124	\N	Cournon-d'Auvergne	63
25916	125	\N	Courpière	63
25917	126	\N	Crest	63
25918	127	\N	Creste	63
25919	128	\N	Crevant-Laveine	63
25920	129	\N	Cros	63
25921	130	\N	Crouzille	63
25922	131	\N	Culhat	63
25923	132	\N	Cunlhat	63
25924	133	\N	Dallet	63
25925	134	\N	Dauzat-sur-Vodable	63
25926	135	\N	Davayat	63
25927	136	\N	Domaize	63
25928	137	\N	Doranges	63
25929	138	\N	Dorat	63
25930	139	\N	Dore-l'Église	63
25931	140	\N	Durmignat	63
25932	141	\N	Durtol	63
25933	142	\N	Échandelys	63
25934	143	\N	Effiat	63
25935	144	\N	Égliseneuve-d'Entraigues	63
25936	145	\N	Égliseneuve-des-Liards	63
25937	146	\N	Égliseneuve-près-Billom	63
25938	147	\N	Églisolles	63
25939	148	\N	Ennezat	63
25940	149	\N	Entraigues	63
25941	150	\N	Enval	63
25942	151	\N	Escoutoux	63
25943	152	\N	Espinasse	63
25944	153	\N	Espinchal	63
25945	154	\N	Espirat	63
25946	155	\N	Estandeuil	63
25947	156	\N	Esteil	63
25948	157	\N	Fayet-le-Château	63
25949	158	\N	Fayet-Ronaye	63
25950	159	\N	Fernoël	63
25951	160	\N	Flat	63
25952	161	\N	Forie	63
25953	162	\N	Fournols	63
25954	163	\N	Gelles	63
25955	164	\N	Gerzat	63
25956	165	\N	Giat	63
25957	166	\N	Gignat	63
25958	167	\N	Gimeaux	63
25959	168	\N	Glaine-Montaigut	63
25960	169	\N	Godivelle	63
25961	170	\N	Goutelle	63
25962	171	\N	Gouttières	63
25963	172	\N	Grandeyrolles	63
25964	173	\N	Grandrif	63
25965	174	\N	Grandval	63
25966	175	\N	Herment	63
25967	176	\N	Heume-l'Église	63
25968	177	\N	Isserteaux	63
25969	178	\N	Issoire	63
25970	179	\N	Job	63
25971	180	\N	Joze	63
25972	181	\N	Joserand	63
25973	182	\N	Jumeaux	63
25974	183	\N	Labessette	63
25975	184	\N	Lachaux	63
25976	185	\N	Lamontgie	63
25977	186	\N	Landogne	63
25978	187	\N	Lapeyrouse	63
25979	188	\N	Laps	63
25980	189	\N	Laqueuille	63
25981	190	\N	Larodde	63
25982	191	\N	Lastic	63
25983	192	\N	Tour-d'Auvergne	63
25984	193	\N	Lempdes	63
25985	194	\N	Lempty	63
25986	195	\N	Lezoux	63
25987	196	\N	Limons	63
25988	197	\N	Lisseuil	63
25989	198	\N	Loubeyrat	63
25990	199	\N	Ludesse	63
25991	200	\N	Lussat	63
25992	201	\N	Luzillat	63
25993	202	\N	Madriat	63
25994	203	\N	Malauzat	63
25995	204	\N	Malintrat	63
25996	205	\N	Manglieu	63
25997	206	\N	Manzat	63
25998	207	\N	Marat	63
25999	208	\N	Marcillat	63
26000	209	\N	Mareugheol	63
26001	210	\N	Maringues	63
26002	211	\N	Marsac-en-Livradois	63
26003	212	\N	Marsat	63
26004	213	\N	Martres-d'Artière	63
26005	214	\N	Martres-de-Veyre	63
26006	215	\N	Martres-sur-Morge	63
26007	216	\N	Mauzun	63
26008	218	\N	Mayres	63
26009	219	\N	Mazaye	63
26010	220	\N	Mazoires	63
26011	221	\N	Medeyrolles	63
26012	222	\N	Meilhaud	63
26013	223	\N	Menat	63
26014	224	\N	Ménétrol	63
26015	225	\N	Messeix	63
26016	226	\N	Mezel	63
26017	227	\N	Mirefleurs	63
26018	228	\N	Miremont	63
26019	229	\N	Moissat	63
26020	230	\N	Monestier	63
26021	231	\N	Monnerie-le-Montel	63
26022	232	\N	Mons	63
26023	233	\N	Montaigut	63
26024	234	\N	Montaigut-le-Blanc	63
26025	235	\N	Montcel	63
26026	236	\N	Mont-Dore	63
26027	237	\N	Montel-de-Gelat	63
26028	238	\N	Montfermy	63
26029	239	\N	Montmorin	63
26030	240	\N	Montpensier	63
26031	241	\N	Montpeyroux	63
26032	242	\N	Moriat	63
26033	243	\N	Moureuille	63
26034	244	\N	Moutade	63
26035	245	\N	Mozac	63
26036	246	\N	Murat-le-Quaire	63
26037	247	\N	Murol	63
26038	248	\N	Nébouzat	63
26039	249	\N	Néronde-sur-Dore	63
26040	250	\N	Neschers	63
26041	251	\N	Neuf-Église	63
26042	252	\N	Neuville	63
26043	253	\N	Noalhat	63
26044	254	\N	Nohanent	63
26045	255	\N	Nonette	63
26046	256	\N	Novacelles	63
26047	257	\N	Olby	63
26048	258	\N	Olliergues	63
26049	259	\N	Olloix	63
26050	260	\N	Olmet	63
26051	261	\N	Orbeil	63
26052	262	\N	Orcet	63
26053	263	\N	Orcines	63
26054	264	\N	Orcival	63
26055	265	\N	Orléat	63
26056	266	\N	Orsonnette	63
26057	267	\N	Palladuc	63
26058	268	\N	Pardines	63
26059	269	\N	Parent	63
26060	270	\N	Parentignat	63
26061	271	\N	Paslières	63
26062	272	\N	Pérignat-lès-Sarliève	63
26063	273	\N	Pérignat-sur-Allier	63
26064	274	\N	Perpezat	63
26065	275	\N	Perrier	63
26066	276	\N	Peschadoires	63
26067	277	\N	Peslières	63
26068	278	\N	Pessat-Villeneuve	63
26069	279	\N	Picherande	63
26070	280	\N	Pignols	63
26071	281	\N	Pionsat	63
26072	282	\N	Plauzat	63
26073	283	\N	Pontaumur	63
26074	284	\N	Pont-du-Château	63
26075	285	\N	Pontgibaud	63
26076	286	\N	Pouzol	63
26077	287	\N	Pradeaux	63
26078	288	\N	Prompsat	63
26079	289	\N	Prondines	63
26080	290	\N	Pulvérières	63
26081	291	\N	Puy-Guillaume	63
26082	292	\N	Puy-Saint-Gulmier	63
26083	293	\N	Quartier	63
26084	294	\N	Queuille	63
26085	295	\N	Randan	63
26086	296	\N	Ravel	63
26087	297	\N	Reignat	63
26088	298	\N	Renaudie	63
26089	299	\N	Rentières	63
26090	300	\N	Riom	63
26091	301	\N	Ris	63
26092	302	\N	Roche-Blanche	63
26093	303	\N	Roche-Charles-la-Mayrand	63
26094	304	\N	Roche-d'Agoux	63
26095	305	\N	Rochefort-Montagne	63
26096	306	\N	Roche-Noire	63
26097	307	\N	Romagnat	63
26098	308	\N	Royat	63
26099	309	\N	Saillant	63
26100	310	\N	Sainte-Agathe	63
26101	311	\N	Saint-Agoulin	63
26102	312	\N	Saint-Alyre-d'Arlanc	63
26103	313	\N	Saint-Alyre-ès-Montagne	63
26104	314	\N	Saint-Amant-Roche-Savine	63
26105	315	\N	Saint-Amant-Tallende	63
26106	317	\N	Saint-André-le-Coq	63
26107	318	\N	Saint-Angel	63
26108	319	\N	Saint-Anthème	63
26109	320	\N	Saint-Avit	63
26110	321	\N	Saint-Babel	63
26111	322	\N	Saint-Beauzire	63
26112	323	\N	Saint-Bonnet-le-Bourg	63
26113	324	\N	Saint-Bonnet-le-Chastel	63
26114	325	\N	Saint-Bonnet-lès-Allier	63
26115	326	\N	Saint-Bonnet-près-Orcival	63
26116	327	\N	Saint-Bonnet-près-Riom	63
26117	328	\N	Sainte-Catherine	63
26118	329	\N	Sainte-Christine	63
26119	330	\N	Saint-Cirgues-sur-Couze	63
26120	331	\N	Saint-Clément-de-Valorgue	63
26121	332	\N	Saint-Clément-de-Régnat	63
26122	333	\N	Saint-Denis-Combarnazat	63
26123	334	\N	Saint-Dier-d'Auvergne	63
26124	335	\N	Saint-Diéry	63
26125	336	\N	Saint-Donat	63
26126	337	\N	Saint-Éloy-la-Glacière	63
26127	338	\N	Saint-Éloy-les-Mines	63
26128	339	\N	Saint-Étienne-des-Champs	63
26129	340	\N	Saint-Étienne-sur-Usson	63
26130	341	\N	Saint-Ferréol-des-Côtes	63
26131	342	\N	Saint-Floret	63
26132	343	\N	Saint-Flour	63
26133	344	\N	Saint-Gal-sur-Sioule	63
26134	345	\N	Saint-Genès-Champanelle	63
26135	346	\N	Saint-Genès-Champespe	63
26136	347	\N	Saint-Genès-du-Retz	63
26137	348	\N	Saint-Genès-la-Tourette	63
26138	349	\N	Saint-Georges-de-Mons	63
26139	350	\N	Saint-Georges-sur-Allier	63
26140	351	\N	Saint-Germain-près-Herment	63
26141	352	\N	Saint-Germain-Lembron	63
26142	353	\N	Saint-Germain-l'Herm	63
26143	354	\N	Saint-Gervais-d'Auvergne	63
26144	355	\N	Saint-Gervais-sous-Meymont	63
26145	356	\N	Saint-Gervazy	63
26146	357	\N	Saint-Hérent	63
26147	358	\N	Saint-Hilaire-la-Croix	63
26148	359	\N	Saint-Hilaire-les-Monges	63
26149	360	\N	Saint-Hilaire	63
26150	362	\N	Saint-Ignat	63
26151	363	\N	Saint-Jacques-d'Ambur	63
26152	364	\N	Saint-Jean-d'Heurs	63
26153	365	\N	Saint-Jean-des-Ollières	63
26154	366	\N	Saint-Jean-en-Val	63
26155	367	\N	Saint-Jean-Saint-Gervais	63
26156	368	\N	Saint-Julien-de-Coppel	63
26157	369	\N	Saint-Julien-la-Geneste	63
26158	370	\N	Saint-Julien-Puy-Lavèze	63
26159	371	\N	Saint-Just	63
26160	372	\N	Saint-Laure	63
26161	373	\N	Saint-Maigner	63
26162	374	\N	Saint-Martin-des-Olmes	63
26163	375	\N	Saint-Martin-des-Plains	63
26164	376	\N	Saint-Martin-d'Ollières	63
26165	377	\N	Saint-Maurice-près-Pionsat	63
26166	378	\N	Saint-Maurice	63
26167	379	\N	Saint-Myon	63
26168	380	\N	Saint-Nectaire	63
26169	381	\N	Saint-Ours	63
26170	382	\N	Saint-Pardoux	63
26171	383	\N	Saint-Pierre-Colamine	63
26172	384	\N	Saint-Pierre-la-Bourlhonne	63
26173	385	\N	Saint-Pierre-le-Chastel	63
26174	386	\N	Saint-Pierre-Roche	63
26175	387	\N	Saint-Priest-Bramefant	63
26176	388	\N	Saint-Priest-des-Champs	63
26177	389	\N	Saint-Quentin-sur-Sauxillanges	63
26178	390	\N	Saint-Quintin-sur-Sioule	63
26179	391	\N	Saint-Rémy-de-Blot	63
26180	392	\N	Saint-Rémy-de-Chargnat	63
26181	393	\N	Saint-Rémy-sur-Durolle	63
26182	394	\N	Saint-Romain	63
26183	395	\N	Saint-Sandoux	63
26184	396	\N	Saint-Saturnin	63
26185	397	\N	Saint-Sauves-d'Auvergne	63
26186	398	\N	Saint-Sauveur-la-Sagne	63
26187	399	\N	Saint-Sulpice	63
26188	400	\N	Saint-Sylvestre-Pragoulin	63
26189	401	\N	Saint-Victor-la-Rivière	63
26190	402	\N	Saint-Victor-Montvianeix	63
26191	403	\N	Saint-Vincent	63
26192	404	\N	Saint-Yvoine	63
26193	405	\N	Sallèdes	63
26194	406	\N	Sardon	63
26195	407	\N	Saulzet-le-Froid	63
26196	408	\N	Sauret-Besserve	63
26197	409	\N	Saurier	63
26198	410	\N	Sauvagnat	63
26199	411	\N	Sauvagnat-Sainte-Marthe	63
26200	412	\N	Sauvessanges	63
26201	413	\N	Sauvetat	63
26202	414	\N	Sauviat	63
26203	415	\N	Sauxillanges	63
26204	416	\N	Savennes	63
26205	417	\N	Sayat	63
26206	418	\N	Sermentizon	63
26207	419	\N	Servant	63
26208	420	\N	Seychalles	63
26209	421	\N	Singles	63
26210	422	\N	Solignat	63
26211	423	\N	Sugères	63
26212	424	\N	Surat	63
26213	425	\N	Tallende	63
26214	426	\N	Tauves	63
26215	427	\N	Teilhède	63
26216	428	\N	Teilhet	63
26217	429	\N	Ternant-les-Eaux	63
26218	430	\N	Thiers	63
26219	431	\N	Thiolières	63
26220	432	\N	Thuret	63
26221	433	\N	Tortebesse	63
26222	434	\N	Tours-sur-Meymont	63
26223	435	\N	Tourzel-Ronzières	63
26224	436	\N	Tralaigues	63
26225	437	\N	Trémouille-Saint-Loup	63
26226	438	\N	Trézioux	63
26227	439	\N	Usson	63
26228	440	\N	Valbeleix	63
26229	441	\N	Valcivières	63
26230	442	\N	Valz-sous-Châteauneuf	63
26231	443	\N	Varennes-sur-Morge	63
26232	444	\N	Varennes-sur-Usson	63
26233	445	\N	Vassel	63
26234	446	\N	Vensat	63
26235	447	\N	Vergheas	63
26236	448	\N	Vernet-la-Varenne	63
26237	449	\N	Vernet-Sainte-Marguerite	63
26238	450	\N	Verneugheol	63
26239	451	\N	Vernines	63
26240	452	\N	Verrières	63
26241	453	\N	Vertaizon	63
26242	454	\N	Vertolaye	63
26243	455	\N	Veyre-Monton	63
26244	456	\N	Vichel	63
26245	457	\N	Vic-le-Comte	63
26246	458	\N	Villeneuve	63
26247	459	\N	Villeneuve-les-Cerfs	63
26248	460	\N	Villosanges	63
26249	461	\N	Vinzelles	63
26250	462	\N	Virlet	63
26251	463	\N	Viscomtat	63
26252	464	\N	Vitrac	63
26253	465	\N	Viverols	63
26254	466	\N	Vodable	63
26255	467	\N	Voingt	63
26256	468	\N	Vollore-Montagne	63
26257	469	\N	Vollore-Ville	63
26258	470	\N	Volvic	63
26259	471	\N	Youx	63
26260	472	\N	Yronde-et-Buron	63
26261	473	\N	Yssac-la-Tourette	63
26262	001	\N	Aast	64
26263	002	\N	Abère	64
26264	003	\N	Abidos	64
26265	004	\N	Abitain	64
26266	005	\N	Abos	64
26267	006	\N	Accous	64
26268	007	\N	Agnos	64
26269	008	\N	Ahaxe-Alciette-Bascassan	64
26270	009	\N	Ahetze	64
26271	010	\N	Aïcirits-Camou-Suhast	64
26272	011	\N	Aincille	64
26273	012	\N	Ainharp	64
26274	013	\N	Ainhice-Mongelos	64
26275	014	\N	Ainhoa	64
26276	015	\N	Alçay-Alçabéhéty-Sunharette	64
26277	016	\N	Aldudes	64
26278	017	\N	Alos-Sibas-Abense	64
26279	018	\N	Amendeuix-Oneix	64
26280	019	\N	Amorots-Succos	64
26281	020	\N	Ance	64
26282	021	\N	Andoins	64
26283	022	\N	Andrein	64
26284	023	\N	Angaïs	64
26285	024	\N	Anglet	64
26286	025	\N	Angous	64
26287	026	\N	Anhaux	64
26288	027	\N	Anos	64
26289	028	\N	Anoye	64
26290	029	\N	Aramits	64
26291	031	\N	Arancou	64
26292	032	\N	Araujuzon	64
26293	033	\N	Araux	64
26294	034	\N	Arbérats-Sillègue	64
26295	035	\N	Arbonne	64
26296	036	\N	Arbouet-Sussaute	64
26297	037	\N	Arbus	64
26298	038	\N	Arcangues	64
26299	039	\N	Aren	64
26300	040	\N	Arette	64
26301	041	\N	Aressy	64
26302	042	\N	Argagnon	64
26303	043	\N	Argelos	64
26304	044	\N	Arget	64
26305	045	\N	Arhansus	64
26306	046	\N	Armendarits	64
26307	047	\N	Arnéguy	64
26308	048	\N	Arnos	64
26309	049	\N	Aroue-Ithorots-Olhaïby	64
26310	050	\N	Arrast-Larrebieu	64
26311	051	\N	Arraute-Charritte	64
26312	052	\N	Arricau-Bordes	64
26313	053	\N	Arrien	64
26314	054	\N	Arros-de-Nay	64
26315	056	\N	Arrosès	64
26316	057	\N	Arthez-de-Béarn	64
26317	058	\N	Arthez-d'Asson	64
26318	059	\N	Artigueloutan	64
26319	060	\N	Artiguelouve	64
26320	061	\N	Artix	64
26321	062	\N	Arudy	64
26322	063	\N	Arzacq-Arraziguet	64
26323	064	\N	Asasp-Arros	64
26324	065	\N	Ascain	64
26325	066	\N	Ascarat	64
26326	067	\N	Assat	64
26327	068	\N	Asson	64
26328	069	\N	Aste-Béon	64
26329	070	\N	Astis	64
26330	071	\N	Athos-Aspis	64
26331	072	\N	Aubertin	64
26332	073	\N	Aubin	64
26333	074	\N	Aubous	64
26334	075	\N	Audaux	64
26335	077	\N	Auga	64
26336	078	\N	Auriac	64
26337	079	\N	Aurions-Idernes	64
26338	080	\N	Aussevielle	64
26339	081	\N	Aussurucq	64
26340	082	\N	Auterrive	64
26341	083	\N	Autevielle-Saint-Martin-Bideren	64
26342	084	\N	Aydie	64
26343	085	\N	Aydius	64
26344	086	\N	Ayherre	64
26345	087	\N	Baigts-de-Béarn	64
26346	088	\N	Balansun	64
26347	089	\N	Baleix	64
26348	090	\N	Baliracq-Maumusson	64
26349	091	\N	Baliros	64
26350	092	\N	Banca	64
26351	093	\N	Barcus	64
26352	094	\N	Bardos	64
26353	095	\N	Barinque	64
26354	096	\N	Barraute-Camu	64
26355	097	\N	Barzun	64
26356	098	\N	Bassillon-Vauzé	64
26357	099	\N	Bastanès	64
26358	100	\N	Bassussarry	64
26359	101	\N	Baudreix	64
26360	102	\N	Bayonne	64
26361	103	\N	Bédeille	64
26362	104	\N	Bedous	64
26363	105	\N	Béguios	64
26364	106	\N	Béhasque-Lapiste	64
26365	107	\N	Béhorléguy	64
26366	108	\N	Bellocq	64
26367	109	\N	Bénéjacq	64
26368	110	\N	Béost	64
26369	111	\N	Bentayou-Sérée	64
26370	112	\N	Bérenx	64
26371	113	\N	Bergouey-Viellenave	64
26372	114	\N	Bernadets	64
26373	115	\N	Berrogain-Laruns	64
26374	116	\N	Bescat	64
26375	117	\N	Bésingrand	64
26376	118	\N	Bétracq	64
26377	119	\N	Beuste	64
26378	120	\N	Beyrie-sur-Joyeuse	64
26379	121	\N	Beyrie-en-Béarn	64
26380	122	\N	Biarritz	64
26381	123	\N	Bidache	64
26382	124	\N	Bidarray	64
26383	125	\N	Bidart	64
26384	126	\N	Bidos	64
26385	127	\N	Bielle	64
26386	128	\N	Bilhères	64
26387	129	\N	Billère	64
26388	130	\N	Biriatou	64
26389	131	\N	Biron	64
26390	132	\N	Bizanos	64
26391	133	\N	Boeil-Bezing	64
26392	134	\N	Bonloc	64
26393	135	\N	Bonnut	64
26394	136	\N	Borce	64
26395	137	\N	Bordères	64
26396	138	\N	Bordes	64
26397	139	\N	Bosdarros	64
26398	140	\N	Boucau	64
26399	141	\N	Boueilh-Boueilho-Lasque	64
26400	142	\N	Bougarber	64
26401	143	\N	Bouillon	64
26402	144	\N	Boumourt	64
26403	145	\N	Bourdettes	64
26404	146	\N	Bournos	64
26405	147	\N	Briscous	64
26406	148	\N	Bruges-Capbis-Mifaget	64
26407	149	\N	Bugnein	64
26408	150	\N	Bunus	64
26409	151	\N	Burgaronne	64
26410	152	\N	Buros	64
26411	153	\N	Burosse-Mendousse	64
26412	154	\N	Bussunarits-Sarrasquette	64
26413	155	\N	Bustince-Iriberry	64
26414	156	\N	Buziet	64
26415	157	\N	Buzy	64
26416	158	\N	Cabidos	64
26417	159	\N	Cadillon	64
26418	160	\N	Cambo-les-Bains	64
26419	161	\N	Came	64
26420	162	\N	Camou-Cihigue	64
26421	165	\N	Cardesse	64
26422	166	\N	Caro	64
26423	167	\N	Carrère	64
26424	168	\N	Carresse-Cassaber	64
26425	170	\N	Castagnède	64
26426	171	\N	Casteide-Cami	64
26427	172	\N	Casteide-Candau	64
26428	173	\N	Casteide-Doat	64
26429	174	\N	Castéra-Loubix	64
26430	175	\N	Castet	64
26431	176	\N	Castetbon	64
26432	177	\N	Castétis	64
26433	178	\N	Castetnau-Camblong	64
26434	179	\N	Castetner	64
26435	180	\N	Castetpugon	64
26436	181	\N	Castillon(Canton d'Arthez-de-Béarn)	64
26437	182	\N	Castillon(Canton de Lembeye)	64
26438	183	\N	Caubios-Loos	64
26439	184	\N	Cescau	64
26440	185	\N	Cette-Eygun	64
26441	186	\N	Charre	64
26442	187	\N	Charritte-de-Bas	64
26443	188	\N	Chéraute	64
26444	189	\N	Ciboure	64
26445	190	\N	Claracq	64
26446	191	\N	Coarraze	64
26447	192	\N	Conchez-de-Béarn	64
26448	193	\N	Corbère-Abères	64
26449	194	\N	Coslédaà-Lube-Boast	64
26450	195	\N	Coublucq	64
26451	196	\N	Crouseilles	64
26452	197	\N	Cuqueron	64
26453	198	\N	Denguin	64
26454	199	\N	Diusse	64
26455	200	\N	Doazon	64
26456	201	\N	Dognen	64
26457	202	\N	Domezain-Berraute	64
26458	203	\N	Doumy	64
26459	204	\N	Eaux-Bonnes	64
26460	205	\N	Escos	64
26461	206	\N	Escot	64
26462	207	\N	Escou	64
26463	208	\N	Escoubès	64
26464	209	\N	Escout	64
26465	210	\N	Escurès	64
26466	211	\N	Eslourenties-Daban	64
26467	212	\N	Espéchède	64
26468	213	\N	Espelette	64
26469	214	\N	Espès-Undurein	64
26470	215	\N	Espiute	64
26471	216	\N	Espoey	64
26472	217	\N	Esquiule	64
26473	218	\N	Estérençuby	64
26474	219	\N	Estialescq	64
26475	220	\N	Estos	64
26476	221	\N	Etcharry	64
26477	222	\N	Etchebar	64
26478	223	\N	Etsaut	64
26479	224	\N	Eysus	64
26480	225	\N	Féas	64
26481	226	\N	Fichous-Riumayou	64
26482	227	\N	Gabaston	64
26483	228	\N	Gabat	64
26484	229	\N	Gamarthe	64
26485	230	\N	Gan	64
26486	231	\N	Garindein	64
26487	232	\N	Garlède-Mondebat	64
26488	233	\N	Garlin	64
26489	234	\N	Garos	64
26490	235	\N	Garris	64
26491	236	\N	Gayon	64
26492	237	\N	Gelos	64
26493	238	\N	Ger	64
26494	239	\N	Gerderest	64
26495	240	\N	Gère-Bélesten	64
26496	241	\N	Géronce	64
26497	242	\N	Gestas	64
26498	243	\N	Géus-d'Arzacq	64
26499	244	\N	Geüs-d'Oloron	64
26500	245	\N	Goès	64
26501	246	\N	Gomer	64
26502	247	\N	Gotein-Libarrenx	64
26503	249	\N	Guéthary	64
26504	250	\N	Guiche	64
26505	251	\N	Guinarthe-Parenties	64
26506	252	\N	Gurmençon	64
26507	253	\N	Gurs	64
26508	254	\N	Hagetaubin	64
26509	255	\N	Halsou	64
26510	256	\N	Hasparren	64
26511	257	\N	Haut-de-Bosdarros	64
26512	258	\N	Haux	64
26513	259	\N	Hélette	64
26514	260	\N	Hendaye	64
26515	261	\N	Herrère	64
26516	262	\N	Higuères-Souye	64
26517	263	\N	Hôpital-d'Orion	64
26518	264	\N	Hôpital-Saint-Blaise	64
26519	265	\N	Hosta	64
26520	266	\N	Hours	64
26521	267	\N	Ibarrolle	64
26522	268	\N	Idaux-Mendy	64
26523	269	\N	Idron	64
26524	270	\N	Igon	64
26525	271	\N	Iholdy	64
26526	272	\N	Ilharre	64
26527	273	\N	Irissarry	64
26528	274	\N	Irouléguy	64
26529	275	\N	Ispoure	64
26530	276	\N	Issor	64
26531	277	\N	Isturits	64
26532	279	\N	Itxassou	64
26533	280	\N	Izeste	64
26534	281	\N	Jasses	64
26535	282	\N	Jatxou	64
26536	283	\N	Jaxu	64
26537	284	\N	Jurançon	64
26538	285	\N	Juxue	64
26539	286	\N	Laà-Mondrans	64
26540	287	\N	Laàs	64
26541	288	\N	Labastide-Cézéracq	64
26542	289	\N	Bastide-Clairence	64
26543	290	\N	Labastide-Monréjeau	64
26544	291	\N	Labastide-Villefranche	64
26545	292	\N	Labatmale	64
26546	293	\N	Labatut	64
26547	294	\N	Labets-Biscay	64
26548	295	\N	Labeyrie	64
26549	296	\N	Lacadée	64
26550	297	\N	Lacarre	64
26551	298	\N	Lacarry-Arhan-Charritte-de-Haut	64
26552	299	\N	Lacommande	64
26553	300	\N	Lacq	64
26554	301	\N	Lagor	64
26555	302	\N	Lagos	64
26556	303	\N	Laguinge-Restoue	64
26557	304	\N	Lahonce	64
26558	305	\N	Lahontan	64
26559	306	\N	Lahourcade	64
26560	307	\N	Lalongue	64
26561	308	\N	Lalonquette	64
26562	309	\N	Lamayou	64
26563	310	\N	Lanne-en-Barétous	64
26564	311	\N	Lannecaube	64
26565	312	\N	Lanneplaà	64
26566	313	\N	Lantabat	64
26567	314	\N	Larceveau-Arros-Cibits	64
26568	315	\N	Laroin	64
26569	316	\N	Larrau	64
26570	317	\N	Larressore	64
26571	318	\N	Larreule	64
26572	319	\N	Larribar-Sorhapuru	64
26573	320	\N	Laruns	64
26574	321	\N	Lasclaveries	64
26575	322	\N	Lasse	64
26576	323	\N	Lasserre	64
26577	324	\N	Lasseube	64
26578	325	\N	Lasseubetat	64
26579	326	\N	Lay-Lamidou	64
26580	327	\N	Lecumberry	64
26581	328	\N	Ledeuix	64
26582	329	\N	Lée	64
26583	330	\N	Lées-Athas	64
26584	331	\N	Lembeye	64
26585	332	\N	Lème	64
26586	334	\N	Léren	64
26587	335	\N	Lescar	64
26588	336	\N	Lescun	64
26589	337	\N	Lespielle	64
26590	338	\N	Lespourcy	64
26591	339	\N	Lestelle-Bétharram	64
26592	340	\N	Lichans-Sunhar	64
26593	341	\N	Lichos	64
26594	342	\N	Licq-Athérey	64
26595	343	\N	Limendous	64
26596	344	\N	Livron	64
26597	345	\N	Lohitzun-Oyhercq	64
26598	346	\N	Lombia	64
26599	347	\N	Lonçon	64
26600	348	\N	Lons	64
26601	349	\N	Loubieng	64
26602	350	\N	Louhossoa	64
26603	351	\N	Lourdios-Ichère	64
26604	352	\N	Lourenties	64
26605	353	\N	Louvie-Juzon	64
26606	354	\N	Louvie-Soubiron	64
26607	355	\N	Louvigny	64
26608	356	\N	Luc-Armau	64
26609	357	\N	Lucarré	64
26610	358	\N	Lucgarier	64
26611	359	\N	Lucq-de-Béarn	64
26612	360	\N	Lurbe-Saint-Christau	64
26613	361	\N	Lussagnet-Lusson	64
26614	362	\N	Luxe-Sumberraute	64
26615	363	\N	Lys	64
26616	364	\N	Macaye	64
26617	365	\N	Malaussanne	64
26618	366	\N	Mascaraàs-Haron	64
26619	367	\N	Maslacq	64
26620	368	\N	Masparraute	64
26621	369	\N	Maspie-Lalonquère-Juillacq	64
26622	370	\N	Maucor	64
26623	371	\N	Mauléon-Licharre	64
26624	372	\N	Maure	64
26625	373	\N	Mazères-Lezons	64
26626	374	\N	Mazerolles	64
26627	375	\N	Méharin	64
26628	376	\N	Meillon	64
26629	377	\N	Mendionde	64
26630	378	\N	Menditte	64
26631	379	\N	Mendive	64
26632	380	\N	Méracq	64
26633	381	\N	Méritein	64
26634	382	\N	Mesplède	64
26635	383	\N	Mialos	64
26636	385	\N	Miossens-Lanusse	64
26637	386	\N	Mirepeix	64
26638	387	\N	Momas	64
26639	388	\N	Momy	64
26640	389	\N	Monassut-Audiracq	64
26641	390	\N	Moncaup	64
26642	391	\N	Moncayolle-Larrory-Mendibieu	64
26643	392	\N	Moncla	64
26644	393	\N	Monein	64
26645	394	\N	Monpezat	64
26646	395	\N	Monségur	64
26647	396	\N	Mont	64
26648	397	\N	Montagut	64
26649	398	\N	Montaner	64
26650	399	\N	Montardon	64
26651	400	\N	Montaut	64
26652	401	\N	Mont-Disse	64
26653	403	\N	Montfort	64
26654	404	\N	Montory	64
26655	405	\N	Morlaàs	64
26656	406	\N	Morlanne	64
26657	407	\N	Mouguerre	64
26658	408	\N	Mouhous	64
26659	409	\N	Moumour	64
26660	410	\N	Mourenx	64
26661	411	\N	Musculdy	64
26662	412	\N	Nabas	64
26663	413	\N	Narcastet	64
26664	414	\N	Narp	64
26665	415	\N	Navailles-Angos	64
26666	416	\N	Navarrenx	64
26667	417	\N	Nay	64
26668	418	\N	Noguères	64
26669	419	\N	Nousty	64
26670	420	\N	Ogenne-Camptort	64
26671	421	\N	Ogeu-les-Bains	64
26672	422	\N	Oloron-Sainte-Marie	64
26673	423	\N	Oraàs	64
26674	424	\N	Ordiarp	64
26675	425	\N	Orègue	64
26676	426	\N	Orin	64
26677	427	\N	Orion	64
26678	428	\N	Orriule	64
26679	429	\N	Orsanco	64
26680	430	\N	Orthez	64
26681	431	\N	Os-Marsillon	64
26682	432	\N	Ossas-Suhare	64
26683	433	\N	Osse-en-Aspe	64
26684	434	\N	Ossenx	64
26685	435	\N	Osserain-Rivareyte	64
26686	436	\N	Ossès	64
26687	437	\N	Ostabat-Asme	64
26688	438	\N	Ouillon	64
26689	439	\N	Ousse	64
26690	440	\N	Ozenx-Montestrucq	64
26691	441	\N	Pagolle	64
26692	442	\N	Parbayse	64
26693	443	\N	Pardies	64
26694	444	\N	Pardies-Piétat	64
26695	445	\N	Pau	64
26696	446	\N	Peyrelongue-Abos	64
26697	447	\N	Piets-Plasence-Moustrou	64
26698	448	\N	Poey-de-Lescar	64
26699	449	\N	Poey-d'Oloron	64
26700	450	\N	Pomps	64
26701	451	\N	Ponson-Debat-Pouts	64
26702	452	\N	Ponson-Dessus	64
26703	453	\N	Pontacq	64
26704	454	\N	Pontiacq-Viellepinte	64
26705	455	\N	Portet	64
26706	456	\N	Pouliacq	64
26707	457	\N	Poursiugues-Boucoue	64
26708	458	\N	Préchacq-Josbaig	64
26709	459	\N	Préchacq-Navarrenx	64
26710	460	\N	Précilhon	64
26711	461	\N	Puyoô	64
26712	462	\N	Ramous	64
26713	463	\N	Rébénacq	64
26714	464	\N	Ribarrouy	64
26715	465	\N	Riupeyrous	64
26716	466	\N	Rivehaute	64
26717	467	\N	Rontignon	64
26718	468	\N	Roquiague	64
26719	469	\N	Saint-Abit	64
26720	470	\N	Saint-Armou	64
26721	471	\N	Saint-Boès	64
26722	472	\N	Saint-Castin	64
26723	473	\N	Sainte-Colome	64
26724	474	\N	Saint-Dos	64
26725	475	\N	Sainte-Engrâce	64
26726	476	\N	Saint-Esteben	64
26727	477	\N	Saint-Étienne-de-Baïgorry	64
26728	478	\N	Saint-Faust	64
26729	479	\N	Saint-Girons	64
26730	480	\N	Saint-Gladie-Arrive-Munein	64
26731	481	\N	Saint-Goin	64
26732	482	\N	Saint-Jammes	64
26733	483	\N	Saint-Jean-de-Luz	64
26734	484	\N	Saint-Jean-le-Vieux	64
26735	485	\N	Saint-Jean-Pied-de-Port	64
26736	486	\N	Saint-Jean-Poudge	64
26737	487	\N	Saint-Just-Ibarre	64
26738	488	\N	Saint-Laurent-Bretagne	64
26739	489	\N	Saint-Martin-d'Arberoue	64
26740	490	\N	Saint-Martin-d'Arrossa	64
26741	491	\N	Saint-Médard	64
26742	492	\N	Saint-Michel	64
26743	493	\N	Saint-Palais	64
26744	494	\N	Saint-Pé-de-Léren	64
26745	495	\N	Saint-Pée-sur-Nivelle	64
26746	496	\N	Saint-Pierre-d'Irube	64
26747	498	\N	Saint-Vincent	64
26748	499	\N	Salies-de-Béarn	64
26749	500	\N	Salles-Mongiscard	64
26750	501	\N	Sallespisse	64
26751	502	\N	Sames	64
26752	503	\N	Samsons-Lion	64
26753	504	\N	Sare	64
26754	505	\N	Sarpourenx	64
26755	506	\N	Sarrance	64
26756	507	\N	Saubole	64
26757	508	\N	Saucède	64
26758	509	\N	Sauguis-Saint-Étienne	64
26759	510	\N	Sault-de-Navailles	64
26760	511	\N	Sauvagnon	64
26761	512	\N	Sauvelade	64
26762	513	\N	Sauveterre-de-Béarn	64
26763	514	\N	Séby	64
26764	515	\N	Sedze-Maubecq	64
26765	516	\N	Sedzère	64
26766	517	\N	Séméacq-Blachon	64
26767	518	\N	Sendets	64
26768	519	\N	Serres-Castet	64
26769	520	\N	Serres-Morlaàs	64
26770	521	\N	Serres-Sainte-Marie	64
26771	522	\N	Sévignacq-Meyracq	64
26772	523	\N	Sévignacq	64
26773	524	\N	Simacourbe	64
26774	525	\N	Siros	64
26775	526	\N	Soumoulou	64
26776	527	\N	Souraïde	64
26777	528	\N	Suhescun	64
26778	529	\N	Sus	64
26779	530	\N	Susmiou	64
26780	531	\N	Tabaille-Usquain	64
26781	532	\N	Tadousse-Ussau	64
26782	533	\N	Tardets-Sorholus	64
26783	534	\N	Taron-Sadirac-Viellenave	64
26784	535	\N	Tarsacq	64
26785	536	\N	Thèze	64
26786	537	\N	Trois-Villes	64
26787	538	\N	Uhart-Cize	64
26788	539	\N	Uhart-Mixe	64
26789	540	\N	Urcuit	64
26790	541	\N	Urdès	64
26791	542	\N	Urdos	64
26792	543	\N	Urepel	64
26793	544	\N	Urost	64
26794	545	\N	Urrugne	64
26795	546	\N	Urt	64
26796	547	\N	Ustaritz	64
26797	548	\N	Uzan	64
26798	549	\N	Uzein	64
26799	550	\N	Uzos	64
26800	551	\N	Verdets	64
26801	552	\N	Vialer	64
26802	554	\N	Viellenave-d'Arthez	64
26803	555	\N	Viellenave-de-Navarrenx	64
26804	556	\N	Vielleségure	64
26805	557	\N	Vignes	64
26806	558	\N	Villefranque	64
26807	559	\N	Viodos-Abense-de-Bas	64
26808	560	\N	Viven	64
26809	001	\N	Adast	65
26810	002	\N	Adé	65
26811	003	\N	Adervielle-Pouchergues	65
26812	004	\N	Agos-Vidalos	65
26813	005	\N	Allier	65
26814	006	\N	Ancizan	65
26815	007	\N	Andrest	65
26816	009	\N	Anères	65
26817	010	\N	Angos	65
26818	011	\N	Angles	65
26819	012	\N	Anla	65
26820	013	\N	Ansost	65
26821	014	\N	Antichan	65
26822	015	\N	Antin	65
26823	016	\N	Antist	65
26824	017	\N	Aragnouet	65
26825	018	\N	Arbéost	65
26826	019	\N	Arcizac-Adour	65
26827	020	\N	Arcizac-ez-Angles	65
26828	021	\N	Arcizans-Avant	65
26829	022	\N	Arcizans-Dessus	65
26830	023	\N	Ardengost	65
26831	024	\N	Argelès-Bagnères	65
26832	025	\N	Argelès-Gazost	65
26833	026	\N	Aries-Espénan	65
26834	027	\N	Armenteule	65
26835	028	\N	Arné	65
26836	029	\N	Arras-en-Lavedan	65
26837	031	\N	Arreau	65
26838	032	\N	Arrens-Marsous	65
26839	033	\N	Arrodets-ez-Angles	65
26840	034	\N	Arrodets	65
26841	035	\N	Artagnan	65
26842	036	\N	Artalens-Souin	65
26843	037	\N	Artiguemy	65
26844	038	\N	Artigues	65
26845	039	\N	Aspin-Aure	65
26846	040	\N	Aspin-en-Lavedan	65
26847	041	\N	Asque	65
26848	042	\N	Asté	65
26849	043	\N	Astugue	65
26850	044	\N	Aubarède	65
26851	045	\N	Aucun	65
26852	046	\N	Aulon	65
26853	047	\N	Aureilhan	65
26854	048	\N	Aurensan	65
26855	049	\N	Auriébat	65
26856	050	\N	Avajan	65
26857	051	\N	Aventignan	65
26858	052	\N	Averan	65
26859	053	\N	Aveux	65
26860	054	\N	Avezac-Prat-Lahitte	65
26861	055	\N	Ayros-Arbouix	65
26862	056	\N	Ayzac-Ost	65
26863	057	\N	Azereix	65
26864	058	\N	Azet	65
26865	059	\N	Bagnères-de-Bigorre	65
26866	060	\N	Banios	65
26867	061	\N	Barbachen	65
26868	062	\N	Barbazan-Debat	65
26869	063	\N	Barbazan-Dessus	65
26870	064	\N	Bareilles	65
26871	065	\N	Barlest	65
26872	066	\N	Barrancoueu	65
26873	067	\N	Barry	65
26874	068	\N	Barthe	65
26875	069	\N	Barthe-de-Neste	65
26876	070	\N	Bartrès	65
26877	071	\N	Batsère	65
26878	072	\N	Bazet	65
26879	073	\N	Bazillac	65
26880	074	\N	Bazordan	65
26881	075	\N	Bazus-Aure	65
26882	076	\N	Bazus-Neste	65
26883	077	\N	Beaucens	65
26884	078	\N	Beaudéan	65
26885	079	\N	Bégole	65
26886	080	\N	Bénac	65
26887	081	\N	Benqué	65
26888	082	\N	Berbérust-Lias	65
26889	083	\N	Bernac-Debat	65
26890	084	\N	Bernac-Dessus	65
26891	085	\N	Bernadets-Debat	65
26892	086	\N	Bernadets-Dessus	65
26893	087	\N	Bertren	65
26894	088	\N	Betbèze	65
26895	089	\N	Betpouey	65
26896	090	\N	Betpouy	65
26897	091	\N	Bettes	65
26898	092	\N	Beyrède-Jumet	65
26899	093	\N	Bize	65
26900	094	\N	Bizous	65
26901	095	\N	Bonnefont	65
26902	096	\N	Bonnemazon	65
26903	097	\N	Bonrepos	65
26904	098	\N	Boô-Silhen	65
26905	099	\N	Bordères-Louron	65
26906	100	\N	Bordères-sur-l'Échez	65
26907	101	\N	Bordes	65
26908	102	\N	Bouilh-Devant	65
26909	103	\N	Bouilh-Péreuilh	65
26910	104	\N	Boulin	65
26911	105	\N	Bourg-de-Bigorre	65
26912	106	\N	Bourisp	65
26913	107	\N	Bourréac	65
26914	108	\N	Bours	65
26915	109	\N	Bramevaque	65
26916	110	\N	Bugard	65
26917	111	\N	Bulan	65
26918	112	\N	Bun	65
26919	113	\N	Burg	65
26920	114	\N	Buzon	65
26921	115	\N	Cabanac	65
26922	116	\N	Cadéac	65
26923	117	\N	Cadeilhan-Trachère	65
26924	118	\N	Caharet	65
26925	119	\N	Caixon	65
26926	120	\N	Calavanté	65
26927	121	\N	Camalès	65
26928	122	\N	Camous	65
26929	123	\N	Campan	65
26930	124	\N	Camparan	65
26931	125	\N	Campistrous	65
26932	126	\N	Campuzan	65
26933	127	\N	Capvern	65
26934	128	\N	Castelbajac	65
26935	129	\N	Castelnau-Magnoac	65
26936	130	\N	Castelnau-Rivière-Basse	65
26937	131	\N	Castelvieilh	65
26938	132	\N	Castéra-Lanusse	65
26939	133	\N	Castéra-Lou	65
26940	134	\N	Casterets	65
26941	135	\N	Castillon	65
26942	136	\N	Caubous	65
26943	137	\N	Caussade-Rivière	65
26944	138	\N	Cauterets	65
26945	139	\N	Cazarilh	65
26946	140	\N	Cazaux-Debat	65
26947	141	\N	Cazaux-Fréchet-Anéran-Camors	65
26948	142	\N	Chelle-Debat	65
26949	143	\N	Chelle-Spou	65
26950	144	\N	Cheust	65
26951	145	\N	Chèze	65
26952	146	\N	Chis	65
26953	147	\N	Cieutat	65
26954	148	\N	Cizos	65
26955	149	\N	Clarac	65
26956	150	\N	Clarens	65
26957	151	\N	Collongues	65
26958	153	\N	Coussan	65
26959	154	\N	Créchets	65
26960	155	\N	Devèze	65
26961	156	\N	Dours	65
26962	157	\N	Ens	65
26963	158	\N	Esbareich	65
26964	159	\N	Escala	65
26965	160	\N	Escaunets	65
26966	161	\N	Escondeaux	65
26967	162	\N	Esconnets	65
26968	163	\N	Escots	65
26969	164	\N	Escoubès-Pouts	65
26970	165	\N	Esparros	65
26971	166	\N	Espèche	65
26972	167	\N	Espieilh	65
26973	168	\N	Esquièze-Sère	65
26974	169	\N	Estaing	65
26975	170	\N	Estampures	65
26976	171	\N	Estarvielle	65
26977	172	\N	Estensan	65
26978	173	\N	Esterre	65
26979	174	\N	Estirac	65
26980	175	\N	Ferrère	65
26981	176	\N	Ferrières	65
26982	177	\N	Fontrailles	65
26983	178	\N	Fréchède	65
26984	179	\N	Fréchendets	65
26985	180	\N	Fréchet-Aure	65
26986	181	\N	Fréchou-Fréchet	65
26987	182	\N	Gaillagos	65
26988	183	\N	Galan	65
26989	184	\N	Galez	65
26990	185	\N	Gardères	65
26991	186	\N	Gaudent	65
26992	187	\N	Gaussan	65
26993	188	\N	Gavarnie	65
26994	189	\N	Gayan	65
26995	190	\N	Gazave	65
26996	191	\N	Gazost	65
26997	192	\N	Gèdre	65
26998	193	\N	Gembrie	65
26999	194	\N	Générest	65
27000	195	\N	Génos	65
27001	196	\N	Gensac	65
27002	197	\N	Ger	65
27003	198	\N	Gerde	65
27004	199	\N	Germ	65
27005	200	\N	Germs-sur-l'Oussouet	65
27006	201	\N	Geu	65
27007	202	\N	Gez	65
27008	203	\N	Gez-ez-Angles	65
27009	204	\N	Gonez	65
27010	205	\N	Gouaux	65
27011	206	\N	Goudon	65
27012	207	\N	Gourgue	65
27013	208	\N	Grailhen	65
27014	209	\N	Grézian	65
27015	210	\N	Grust	65
27016	211	\N	Guchan	65
27017	212	\N	Guchen	65
27018	213	\N	Guizerix	65
27019	214	\N	Hachan	65
27020	215	\N	Hagedet	65
27021	216	\N	Hauban	65
27022	217	\N	Hautaget	65
27023	218	\N	Hèches	65
27024	219	\N	Hères	65
27025	220	\N	Hibarette	65
27026	221	\N	Hiis	65
27027	222	\N	Hitte	65
27028	223	\N	Horgues	65
27029	224	\N	Houeydets	65
27030	225	\N	Hourc	65
27031	226	\N	Ibos	65
27032	228	\N	Ilhet	65
27033	229	\N	Ilheu	65
27034	230	\N	Izaourt	65
27035	231	\N	Izaux	65
27036	232	\N	Jacque	65
27037	233	\N	Jarret	65
27038	234	\N	Jézeau	65
27039	235	\N	Juillan	65
27040	236	\N	Julos	65
27041	237	\N	Juncalas	65
27042	238	\N	Labassère	65
27043	239	\N	Labastide	65
27044	240	\N	Labatut-Rivière	65
27045	241	\N	Laborde	65
27046	242	\N	Lacassagne	65
27047	243	\N	Lafitole	65
27048	244	\N	Lagarde	65
27049	245	\N	Lagrange	65
27050	247	\N	Arrayou-Lahitte	65
27051	248	\N	Lahitte-Toupière	65
27052	249	\N	Lalanne	65
27053	250	\N	Lalanne-Trie	65
27054	251	\N	Laloubère	65
27055	252	\N	Lamarque-Pontacq	65
27056	253	\N	Lamarque-Rustaing	65
27057	254	\N	Laméac	65
27058	255	\N	Lançon	65
27059	256	\N	Lanespède	65
27060	257	\N	Lanne	65
27061	258	\N	Lannemezan	65
27062	259	\N	Lansac	65
27063	260	\N	Lapeyre	65
27064	261	\N	Laran	65
27065	262	\N	Larreule	65
27066	263	\N	Larroque	65
27067	264	\N	Lascazères	65
27068	265	\N	Laslades	65
27069	266	\N	Lassales	65
27070	267	\N	Lau-Balagnas	65
27071	268	\N	Layrisse	65
27072	269	\N	Lescurry	65
27073	270	\N	Lespouey	65
27074	271	\N	Lézignan	65
27075	272	\N	Lhez	65
27076	273	\N	Liac	65
27077	274	\N	Libaros	65
27078	275	\N	Lies	65
27079	276	\N	Lizos	65
27080	277	\N	Lombrès	65
27081	278	\N	Lomné	65
27082	279	\N	Lortet	65
27083	280	\N	Loubajac	65
27084	281	\N	Loucrup	65
27085	282	\N	Loudenvielle	65
27086	283	\N	Loudervielle	65
27087	284	\N	Louey	65
27088	285	\N	Louit	65
27089	286	\N	Lourdes	65
27090	287	\N	Loures-Barousse	65
27091	288	\N	Lubret-Saint-Luc	65
27092	289	\N	Luby-Betmont	65
27093	290	\N	Luc	65
27094	291	\N	Lugagnan	65
27095	292	\N	Luquet	65
27096	293	\N	Lustar	65
27097	294	\N	Lutilhous	65
27098	295	\N	Luz-Saint-Sauveur	65
27099	296	\N	Madiran	65
27100	297	\N	Mansan	65
27101	298	\N	Marquerie	65
27102	299	\N	Marsac	65
27103	300	\N	Marsas	65
27104	301	\N	Marseillan	65
27105	303	\N	Mascaras	65
27106	304	\N	Maubourguet	65
27107	305	\N	Mauléon-Barousse	65
27108	306	\N	Mauvezin	65
27109	307	\N	Mazères-de-Neste	65
27110	308	\N	Mazerolles	65
27111	309	\N	Mazouau	65
27112	310	\N	Mérilheu	65
27113	311	\N	Mingot	65
27114	312	\N	Molère	65
27115	313	\N	Momères	65
27116	314	\N	Monfaucon	65
27117	315	\N	Monléon-Magnoac	65
27118	316	\N	Monlong	65
27119	317	\N	Mont	65
27120	318	\N	Montastruc	65
27121	319	\N	Montégut	65
27122	320	\N	Montgaillard	65
27123	321	\N	Montignac	65
27124	322	\N	Montoussé	65
27125	323	\N	Montsérié	65
27126	324	\N	Moulédous	65
27127	325	\N	Moumoulous	65
27128	326	\N	Mun	65
27129	327	\N	Nestier	65
27130	328	\N	Neuilh	65
27131	329	\N	Nistos	65
27132	330	\N	Nouilhan	65
27133	331	\N	Odos	65
27134	332	\N	Oléac-Debat	65
27135	333	\N	Oléac-Dessus	65
27136	334	\N	Omex	65
27137	335	\N	Ordizan	65
27138	336	\N	Organ	65
27139	337	\N	Orieux	65
27140	338	\N	Orignac	65
27141	339	\N	Orincles	65
27142	340	\N	Orleix	65
27143	341	\N	Oroix	65
27144	342	\N	Osmets	65
27145	343	\N	Ossen	65
27146	344	\N	Ossun	65
27147	345	\N	Ossun-ez-Angles	65
27148	346	\N	Oueilloux	65
27149	347	\N	Ourde	65
27150	348	\N	Ourdis-Cotdoussan	65
27151	349	\N	Ourdon	65
27152	350	\N	Oursbelille	65
27153	351	\N	Ousté	65
27154	352	\N	Ouzous	65
27155	353	\N	Ozon	65
27156	354	\N	Pailhac	65
27157	355	\N	Paréac	65
27158	356	\N	Péré	65
27159	357	\N	Peyraube	65
27160	358	\N	Peyret-Saint-André	65
27161	359	\N	Peyriguère	65
27162	360	\N	Peyrouse	65
27163	361	\N	Peyrun	65
27164	362	\N	Pierrefitte-Nestalas	65
27165	363	\N	Pinas	65
27166	364	\N	Pintac	65
27167	366	\N	Poueyferré	65
27168	367	\N	Poumarous	65
27169	368	\N	Pouy	65
27170	369	\N	Pouyastruc	65
27171	370	\N	Pouzac	65
27172	371	\N	Préchac	65
27173	372	\N	Pujo	65
27174	373	\N	Puntous	65
27175	374	\N	Puydarrieux	65
27176	375	\N	Rabastens-de-Bigorre	65
27177	376	\N	Recurt	65
27178	377	\N	Réjaumont	65
27179	378	\N	Ricaud	65
27180	379	\N	Ris	65
27181	380	\N	Sabalos	65
27182	381	\N	Sabarros	65
27183	382	\N	Sacoué	65
27184	383	\N	Sadournin	65
27185	384	\N	Sailhan	65
27186	385	\N	Saint-Arroman	65
27187	386	\N	Saint-Créac	65
27188	387	\N	Saint-Lanne	65
27189	388	\N	Saint-Lary-Soulan	65
27190	389	\N	Saint-Laurent-de-Neste	65
27191	390	\N	Saint-Lézer	65
27192	391	\N	Sainte-Marie	65
27193	392	\N	Saint-Martin	65
27194	393	\N	Saint-Pastous	65
27195	394	\N	Saint-Paul	65
27196	395	\N	Saint-Pé-de-Bigorre	65
27197	396	\N	Saint-Savin	65
27198	397	\N	Saint-Sever-de-Rustan	65
27199	398	\N	Saléchan	65
27200	399	\N	Saligos	65
27201	400	\N	Salles	65
27202	401	\N	Salles-Adour	65
27203	402	\N	Samuran	65
27204	403	\N	Sanous	65
27205	404	\N	Sariac-Magnoac	65
27206	405	\N	Sarlabous	65
27207	406	\N	Sarniguet	65
27208	407	\N	Sarp	65
27209	408	\N	Sarrancolin	65
27210	409	\N	Sarriac-Bigorre	65
27211	410	\N	Sarrouilles	65
27212	411	\N	Sassis	65
27213	412	\N	Sauveterre	65
27214	413	\N	Sazos	65
27215	414	\N	Ségalas	65
27216	415	\N	Ségus	65
27217	416	\N	Seich	65
27218	417	\N	Séméac	65
27219	418	\N	Sénac	65
27220	419	\N	Sentous	65
27221	420	\N	Sère-en-Lavedan	65
27222	421	\N	Sère-Lanso	65
27223	422	\N	Séron	65
27224	423	\N	Sère-Rustaing	65
27225	424	\N	Sers	65
27226	425	\N	Siarrouy	65
27227	426	\N	Sinzos	65
27228	427	\N	Siradan	65
27229	428	\N	Sireix	65
27230	429	\N	Sombrun	65
27231	430	\N	Soréac	65
27232	431	\N	Sost	65
27233	432	\N	Soublecause	65
27234	433	\N	Soues	65
27235	435	\N	Soulom	65
27236	436	\N	Souyeaux	65
27237	437	\N	Tajan	65
27238	438	\N	Talazac	65
27239	439	\N	Tarasteix	65
27240	440	\N	Tarbes	65
27241	441	\N	Thèbe	65
27242	442	\N	Thermes-Magnoac	65
27243	443	\N	Thuy	65
27244	444	\N	Tibiran-Jaunac	65
27245	445	\N	Tilhouse	65
27246	446	\N	Tostat	65
27247	447	\N	Tournay	65
27248	448	\N	Tournous-Darré	65
27249	449	\N	Tournous-Devant	65
27250	450	\N	Tramezaïgues	65
27251	451	\N	Trébons	65
27252	452	\N	Trie-sur-Baïse	65
27253	453	\N	Troubat	65
27254	454	\N	Trouley-Labarthe	65
27255	455	\N	Tuzaguet	65
27256	456	\N	Uglas	65
27257	457	\N	Ugnouas	65
27258	458	\N	Uz	65
27259	459	\N	Uzer	65
27260	460	\N	Vic-en-Bigorre	65
27261	461	\N	Vidou	65
27262	462	\N	Vidouze	65
27263	463	\N	Viella	65
27264	464	\N	Vielle-Adour	65
27265	465	\N	Vielle-Aure	65
27266	466	\N	Vielle-Louron	65
27267	467	\N	Vier-Bordes	65
27268	468	\N	Vieuzos	65
27269	469	\N	Viey	65
27270	470	\N	Viger	65
27271	471	\N	Vignec	65
27272	472	\N	Villefranque	65
27273	473	\N	Villelongue	65
27274	474	\N	Villembits	65
27275	475	\N	Villemur	65
27276	476	\N	Villenave-près-Béarn	65
27277	477	\N	Villenave-près-Marsac	65
27278	478	\N	Viscos	65
27279	479	\N	Visker	65
27280	480	\N	Vizos	65
27281	481	\N	Barèges	65
27282	482	\N	Cantaous	65
27283	001	\N	Albère	66
27284	002	\N	Alénya	66
27285	003	\N	Amélie-les-Bains-Palalda	66
27286	004	\N	Angles	66
27287	005	\N	Angoustrine-Villeneuve-des-Escaldes	66
27288	006	\N	Ansignan	66
27289	007	\N	Arboussols	66
27290	008	\N	Argelès-sur-Mer	66
27291	009	\N	Arles-sur-Tech	66
27292	010	\N	Ayguatébia-Talau	66
27293	011	\N	Bages	66
27294	012	\N	Baho	66
27295	013	\N	Baillestavy	66
27296	014	\N	Baixas	66
27297	015	\N	Banyuls-dels-Aspres	66
27298	016	\N	Banyuls-sur-Mer	66
27299	017	\N	Barcarès	66
27300	018	\N	Bastide	66
27301	019	\N	Bélesta	66
27302	020	\N	Bolquère	66
27303	021	\N	Bompas	66
27304	022	\N	Boule-d'Amont	66
27305	023	\N	Bouleternère	66
27306	024	\N	Boulou	66
27307	025	\N	Bourg-Madame	66
27308	026	\N	Brouilla	66
27309	027	\N	Cabanasse	66
27310	028	\N	Cabestany	66
27311	029	\N	Caixas	66
27312	030	\N	Calce	66
27313	032	\N	Calmeilles	66
27314	033	\N	Camélas	66
27315	034	\N	Campôme	66
27316	035	\N	Campoussy	66
27317	036	\N	Canaveilles	66
27318	037	\N	Canet-en-Roussillon	66
27319	038	\N	Canohès	66
27320	039	\N	Caramany	66
27321	040	\N	Casefabre	66
27322	041	\N	Cases-de-Pène	66
27323	042	\N	Cassagnes	66
27324	043	\N	Casteil	66
27325	044	\N	Castelnou	66
27326	045	\N	Catllar	66
27327	046	\N	Caudiès-de-Fenouillèdes	66
27328	047	\N	Caudiès-de-Conflent	66
27329	048	\N	Cerbère	66
27330	049	\N	Céret	66
27331	050	\N	Claira	66
27332	051	\N	Clara	66
27333	052	\N	Codalet	66
27334	053	\N	Collioure	66
27335	054	\N	Conat	66
27336	055	\N	Corbère	66
27337	056	\N	Corbère-les-Cabanes	66
27338	057	\N	Corneilla-de-Conflent	66
27339	058	\N	Corneilla-la-Rivière	66
27340	059	\N	Corneilla-del-Vercol	66
27341	060	\N	Corsavy	66
27342	061	\N	Coustouges	66
27343	062	\N	Dorres	66
27344	063	\N	Cluses	66
27345	064	\N	Égat	66
27346	065	\N	Elne	66
27347	066	\N	Enveitg	66
27348	067	\N	Err	66
27349	068	\N	Escaro	66
27350	069	\N	Espira-de-l'Agly	66
27351	070	\N	Espira-de-Conflent	66
27352	071	\N	Estagel	66
27353	072	\N	Estavar	66
27354	073	\N	Estoher	66
27355	074	\N	Eus	66
27356	075	\N	Eyne	66
27357	076	\N	Felluns	66
27358	077	\N	Fenouillet	66
27359	078	\N	Fillols	66
27360	079	\N	Finestret	66
27361	080	\N	Fontpédrouse	66
27362	081	\N	Fontrabiouse	66
27363	082	\N	Formiguères	66
27364	083	\N	Fosse	66
27365	084	\N	Fourques	66
27366	085	\N	Fuilla	66
27367	086	\N	Glorianes	66
27368	088	\N	Ille-sur-Têt	66
27369	089	\N	Joch	66
27370	090	\N	Jujols	66
27371	091	\N	Lamanère	66
27372	092	\N	Lansac	66
27373	093	\N	Laroque-des-Albères	66
27374	094	\N	Latour-Bas-Elne	66
27375	095	\N	Latour-de-Carol	66
27376	096	\N	Latour-de-France	66
27377	097	\N	Lesquerde	66
27378	098	\N	Llagonne	66
27379	099	\N	Llauro	66
27380	100	\N	Llo	66
27381	101	\N	Llupia	66
27382	102	\N	Mantet	66
27383	103	\N	Marquixanes	66
27384	104	\N	Masos	66
27385	105	\N	Matemale	66
27386	106	\N	Maureillas-las-Illas	66
27387	107	\N	Maury	66
27388	108	\N	Millas	66
27389	109	\N	Molitg-les-Bains	66
27390	111	\N	Montalba-le-Château	66
27391	112	\N	Montauriol	66
27392	113	\N	Montbolo	66
27393	114	\N	Montescot	66
27394	115	\N	Montesquieu-des-Albères	66
27395	116	\N	Montferrer	66
27396	117	\N	Mont-Louis	66
27397	118	\N	Montner	66
27398	119	\N	Mosset	66
27399	120	\N	Nahuja	66
27400	121	\N	Néfiach	66
27401	122	\N	Nohèdes	66
27402	123	\N	Nyer	66
27403	124	\N	Font-Romeu-Odeillo-Via	66
27404	125	\N	Olette	66
27405	126	\N	Oms	66
27406	127	\N	Opoul-Périllos	66
27407	128	\N	Oreilla	66
27408	129	\N	Ortaffa	66
27409	130	\N	Osséja	66
27410	132	\N	Palau-de-Cerdagne	66
27411	133	\N	Palau-del-Vidre	66
27412	134	\N	Passa	66
27413	136	\N	Perpignan	66
27414	137	\N	Perthus	66
27415	138	\N	Peyrestortes	66
27416	139	\N	Pézilla-de-Conflent	66
27417	140	\N	Pézilla-la-Rivière	66
27418	141	\N	Pia	66
27419	142	\N	Planès	66
27420	143	\N	Planèzes	66
27421	144	\N	Pollestres	66
27422	145	\N	Ponteilla	66
27423	146	\N	Porta	66
27424	147	\N	Porté-Puymorens	66
27425	148	\N	Port-Vendres	66
27426	149	\N	Prades	66
27427	150	\N	Prats-de-Mollo-la-Preste	66
27428	151	\N	Prats-de-Sournia	66
27429	152	\N	Prugnanes	66
27430	153	\N	Prunet-et-Belpuig	66
27431	154	\N	Puyvalador	66
27432	155	\N	Py	66
27433	156	\N	Rabouillet	66
27434	157	\N	Railleu	66
27435	158	\N	Rasiguères	66
27436	159	\N	Réal	66
27437	160	\N	Reynès	66
27438	161	\N	Ria-Sirach	66
27439	162	\N	Rigarda	66
27440	164	\N	Rivesaltes	66
27441	165	\N	Rodès	66
27442	166	\N	Sahorre	66
27443	167	\N	Saillagouse	66
27444	168	\N	Saint-André	66
27445	169	\N	Saint-Arnac	66
27446	170	\N	Sainte-Colombe-de-la-Commanderie	66
27447	171	\N	Saint-Cyprien	66
27448	172	\N	Saint-Estève	66
27449	173	\N	Saint-Féliu-d'Amont	66
27450	174	\N	Saint-Féliu-d'Avall	66
27451	175	\N	Saint-Génis-des-Fontaines	66
27452	176	\N	Saint-Hippolyte	66
27453	177	\N	Saint-Jean-Lasseille	66
27454	178	\N	Saint-Jean-Pla-de-Corts	66
27455	179	\N	Saint-Laurent-de-Cerdans	66
27456	180	\N	Saint-Laurent-de-la-Salanque	66
27457	181	\N	Sainte-Léocadie	66
27458	182	\N	Sainte-Marie	66
27459	183	\N	Saint-Marsal	66
27460	184	\N	Saint-Martin	66
27461	185	\N	Saint-Michel-de-Llotes	66
27462	186	\N	Saint-Nazaire	66
27463	187	\N	Saint-Paul-de-Fenouillet	66
27464	188	\N	Saint-Pierre-dels-Forcats	66
27465	189	\N	Saleilles	66
27466	190	\N	Salses-le-Château	66
27467	191	\N	Sansa	66
27468	192	\N	Sauto	66
27469	193	\N	Serdinya	66
27470	194	\N	Serralongue	66
27471	195	\N	Soler	66
27472	196	\N	Sorède	66
27473	197	\N	Souanyas	66
27474	198	\N	Sournia	66
27475	199	\N	Taillet	66
27476	201	\N	Tarerach	66
27477	202	\N	Targassonne	66
27478	203	\N	Taulis	66
27479	204	\N	Taurinya	66
27480	205	\N	Tautavel	66
27481	206	\N	Tech	66
27482	207	\N	Terrats	66
27483	208	\N	Théza	66
27484	209	\N	Thuès-Entre-Valls	66
27485	210	\N	Thuir	66
27486	211	\N	Tordères	66
27487	212	\N	Torreilles	66
27488	213	\N	Toulouges	66
27489	214	\N	Tresserre	66
27490	215	\N	Trévillach	66
27491	216	\N	Trilla	66
27492	217	\N	Trouillas	66
27493	218	\N	Ur	66
27494	219	\N	Urbanya	66
27495	220	\N	Valcebollère	66
27496	221	\N	Valmanya	66
27497	222	\N	Vernet-les-Bains	66
27498	223	\N	Villefranche-de-Conflent	66
27499	224	\N	Villelongue-de-la-Salanque	66
27500	225	\N	Villelongue-dels-Monts	66
27501	226	\N	Villemolaque	66
27502	227	\N	Villeneuve-de-la-Raho	66
27503	228	\N	Villeneuve-la-Rivière	66
27504	230	\N	Vinça	66
27505	231	\N	Vingrau	66
27506	232	\N	Vira	66
27507	233	\N	Vivès	66
27508	234	\N	Vivier	66
27509	001	\N	Achenheim	67
27510	002	\N	Adamswiller	67
27511	003	\N	Albé	67
27512	004	\N	Allenwiller	67
27513	005	\N	Alteckendorf	67
27514	006	\N	Altenheim	67
27515	008	\N	Altorf	67
27516	009	\N	Altwiller	67
27517	010	\N	Andlau	67
27518	011	\N	Artolsheim	67
27519	012	\N	Aschbach	67
27520	013	\N	Asswiller	67
27521	014	\N	Auenheim	67
27522	016	\N	Avolsheim	67
27523	017	\N	Baerendorf	67
27524	018	\N	Balbronn	67
27525	019	\N	Baldenheim	67
27526	020	\N	Barembach	67
27527	021	\N	Barr	67
27528	022	\N	Bassemberg	67
27529	023	\N	Batzendorf	67
27530	025	\N	Beinheim	67
27531	026	\N	Bellefosse	67
27532	027	\N	Belmont	67
27533	028	\N	Benfeld	67
27534	029	\N	Berg	67
27535	030	\N	Bergbieten	67
27536	031	\N	Bernardswiller	67
27537	032	\N	Bernardvillé	67
27538	033	\N	Bernolsheim	67
27539	034	\N	Berstett	67
27540	035	\N	Berstheim	67
27541	036	\N	Bettwiller	67
27542	037	\N	Biblisheim	67
27543	038	\N	Bietlenheim	67
27544	039	\N	Bilwisheim	67
27545	040	\N	Bindernheim	67
27546	041	\N	Birkenwald	67
27547	043	\N	Bischheim	67
27548	044	\N	Bischholtz	67
27549	045	\N	Bischoffsheim	67
27550	046	\N	Bischwiller	67
27551	047	\N	Bissert	67
27552	048	\N	Bitschhoffen	67
27553	049	\N	Blaesheim	67
27554	050	\N	Blancherupt	67
27555	051	\N	Blienschwiller	67
27556	052	\N	Brsch	67
27557	053	\N	Bsenbiesen	67
27558	054	\N	Bolsenheim	67
27559	055	\N	Boofzheim	67
27560	056	\N	Bootzheim	67
27561	058	\N	Bossendorf	67
27562	059	\N	Bourg-Bruche	67
27563	060	\N	Bourgheim	67
27564	061	\N	Bouxwiller	67
27565	062	\N	Breitenau	67
27566	063	\N	Breitenbach	67
27567	065	\N	Breuschwickersheim	67
27568	066	\N	Broque	67
27569	067	\N	Brumath	67
27570	068	\N	Buswiller	67
27571	069	\N	Buhl	67
27572	070	\N	Burbach	67
27573	071	\N	Bust	67
27574	072	\N	Butten	67
27575	073	\N	Châtenois	67
27576	074	\N	Cleebourg	67
27577	075	\N	Climbach	67
27578	076	\N	Colroy-la-Roche	67
27579	077	\N	Cosswiller	67
27580	078	\N	Crastatt	67
27581	079	\N	Crttwiller	67
27582	080	\N	Dachstein	67
27583	081	\N	Dahlenheim	67
27584	082	\N	Dalhunden	67
27585	083	\N	Dambach	67
27586	084	\N	Dambach-la-Ville	67
27587	085	\N	Dangolsheim	67
27588	086	\N	Daubensand	67
27589	087	\N	Dauendorf	67
27590	088	\N	Dehlingen	67
27591	089	\N	Dettwiller	67
27592	090	\N	Diebolsheim	67
27593	091	\N	Diedendorf	67
27594	092	\N	Dieffenbach-au-Val	67
27595	093	\N	Dieffenbach-lès-Wrth	67
27596	094	\N	Dieffenthal	67
27597	095	\N	Diemeringen	67
27598	096	\N	Dimbsthal	67
27599	097	\N	Dingsheim	67
27600	098	\N	Dinsheim-sur-Bruche	67
27601	099	\N	Domfessel	67
27602	100	\N	Donnenheim	67
27603	101	\N	Dorlisheim	67
27604	102	\N	Dossenheim-Kochersberg	67
27605	103	\N	Dossenheim-sur-Zinsel	67
27606	104	\N	Drachenbronn-Birlenbach	67
27607	105	\N	Drulingen	67
27608	106	\N	Drusenheim	67
27609	107	\N	Duntzenheim	67
27610	108	\N	Duppigheim	67
27611	109	\N	Durningen	67
27612	110	\N	Durrenbach	67
27613	111	\N	Durstel	67
27614	112	\N	Duttlenheim	67
27615	113	\N	Eberbach-Seltz	67
27616	115	\N	Ebersheim	67
27617	116	\N	Ebersmunster	67
27618	117	\N	Eckartswiller	67
27619	118	\N	Eckbolsheim	67
27620	119	\N	Eckwersheim	67
27621	120	\N	Eichhoffen	67
27622	121	\N	Elsenheim	67
27623	122	\N	Wangenbourg-Engenthal	67
27624	123	\N	Engwiller	67
27625	124	\N	Entzheim	67
27626	125	\N	Epfig	67
27627	126	\N	Erckartswiller	67
27628	127	\N	Ergersheim	67
27629	128	\N	Ernolsheim-Bruche	67
27630	129	\N	Ernolsheim-lès-Saverne	67
27631	130	\N	Erstein	67
27632	131	\N	Eschau	67
27633	132	\N	Eschbach	67
27634	133	\N	Eschbourg	67
27635	134	\N	Eschwiller	67
27636	135	\N	Ettendorf	67
27637	136	\N	Eywiller	67
27638	137	\N	Fegersheim	67
27639	138	\N	Fessenheim-le-Bas	67
27640	139	\N	Flexbourg	67
27641	140	\N	Forstfeld	67
27642	141	\N	Forstheim	67
27643	142	\N	Fort-Louis	67
27644	143	\N	Fouchy	67
27645	144	\N	Fouday	67
27646	145	\N	Friedolsheim	67
27647	146	\N	Friesenheim	67
27648	147	\N	Frschwiller	67
27649	148	\N	Frohmuhl	67
27650	149	\N	Furchhausen	67
27651	150	\N	Furdenheim	67
27652	151	\N	Gambsheim	67
27653	152	\N	Geispolsheim	67
27654	153	\N	Geiswiller	67
27655	154	\N	Gerstheim	67
27656	155	\N	Gertwiller	67
27657	156	\N	Geudertheim	67
27658	158	\N	Gingsheim	67
27659	159	\N	Grlingen	67
27660	160	\N	Grsdorf	67
27661	161	\N	Gottenhouse	67
27662	162	\N	Gottesheim	67
27663	163	\N	Gougenheim	67
27664	164	\N	Goxwiller	67
27665	165	\N	Grandfontaine	67
27666	166	\N	Grassendorf	67
27667	167	\N	Grendelbruch	67
27668	168	\N	Gresswiller	67
27669	169	\N	Gries	67
27670	172	\N	Griesheim-près-Molsheim	67
27671	173	\N	Griesheim-sur-Souffel	67
27672	174	\N	Gumbrechtshoffen	67
27673	176	\N	Gundershoffen	67
27674	177	\N	Gunstett	67
27675	178	\N	Gungwiller	67
27676	179	\N	Haegen	67
27677	180	\N	Haguenau	67
27678	181	\N	Handschuheim	67
27679	182	\N	Hangenbieten	67
27680	183	\N	Harskirchen	67
27681	184	\N	Hatten	67
27682	185	\N	Hattmatt	67
27683	186	\N	Hegeney	67
27684	187	\N	Heidolsheim	67
27685	188	\N	Heiligenberg	67
27686	189	\N	Heiligenstein	67
27687	190	\N	Hengwiller	67
27688	191	\N	Herbitzheim	67
27689	192	\N	Herbsheim	67
27690	194	\N	Herrlisheim	67
27691	195	\N	Hessenheim	67
27692	196	\N	Hilsenheim	67
27693	197	\N	Hindisheim	67
27694	198	\N	Hinsbourg	67
27695	199	\N	Hinsingen	67
27696	200	\N	Hipsheim	67
27697	201	\N	Hirschland	67
27698	202	\N	Hochfelden	67
27699	203	\N	Hochstett	67
27700	204	\N	Hnheim	67
27701	205	\N	Hrdt	67
27702	206	\N	Hoffen	67
27703	207	\N	Hohatzenheim	67
27704	208	\N	Hohengft	67
27705	209	\N	Hohfrankenheim	67
27706	210	\N	Hohwald	67
27707	212	\N	Holtzheim	67
27708	213	\N	Hunspach	67
27709	214	\N	Hurtigheim	67
27710	215	\N	Huttendorf	67
27711	216	\N	Huttenheim	67
27712	217	\N	Ichtratzheim	67
27713	218	\N	Illkirch-Graffenstaden	67
27714	220	\N	Ingenheim	67
27715	221	\N	Ingolsheim	67
27716	222	\N	Ingwiller	67
27717	223	\N	Innenheim	67
27718	225	\N	Issenhausen	67
27719	226	\N	Ittenheim	67
27720	227	\N	Itterswiller	67
27721	228	\N	Neugartheim-Ittlenheim	67
27722	229	\N	Jetterswiller	67
27723	230	\N	Kaltenhouse	67
27724	231	\N	Kauffenheim	67
27725	232	\N	Keffenach	67
27726	233	\N	Kertzfeld	67
27727	234	\N	Keskastel	67
27728	235	\N	Kesseldorf	67
27729	236	\N	Kienheim	67
27730	237	\N	Kilstett	67
27731	238	\N	Kindwiller	67
27732	239	\N	Kintzheim	67
27733	240	\N	Kirchheim	67
27734	241	\N	Kirrberg	67
27735	242	\N	Kirrwiller-Bosselshausen	67
27736	244	\N	Kleingft	67
27737	245	\N	Knrsheim	67
27738	246	\N	Kogenheim	67
27739	247	\N	Kolbsheim	67
27740	248	\N	Krautergersheim	67
27741	249	\N	Krautwiller	67
27742	250	\N	Kriegsheim	67
27743	252	\N	Kurtzenhouse	67
27744	253	\N	Kuttolsheim	67
27745	254	\N	Kutzenhausen	67
27746	255	\N	Lalaye	67
27747	256	\N	Lampertheim	67
27748	257	\N	Lampertsloch	67
27749	258	\N	Landersheim	67
27750	259	\N	Langensoultzbach	67
27751	260	\N	Laubach	67
27752	261	\N	Lauterbourg	67
27753	263	\N	Lembach	67
27754	264	\N	Leutenheim	67
27755	265	\N	Lichtenberg	67
27756	266	\N	Limersheim	67
27757	267	\N	Lingolsheim	67
27758	268	\N	Lipsheim	67
27759	269	\N	Littenheim	67
27760	270	\N	Lixhausen	67
27761	271	\N	Lobsann	67
27762	272	\N	Lochwiller	67
27763	273	\N	Lohr	67
27764	274	\N	Lorentzen	67
27765	275	\N	Lupstein	67
27766	276	\N	Lutzelhouse	67
27767	277	\N	Mackenheim	67
27768	278	\N	Mackwiller	67
27769	279	\N	Maennolsheim	67
27770	280	\N	Maisonsgoutte	67
27771	281	\N	Marckolsheim	67
27772	282	\N	Marlenheim	67
27773	283	\N	Marmoutier	67
27774	285	\N	Matzenheim	67
27775	286	\N	Meistratzheim	67
27776	287	\N	Melsheim	67
27777	288	\N	Memmelshoffen	67
27778	289	\N	Menchhoffen	67
27779	290	\N	Merkwiller-Pechelbronn	67
27780	291	\N	Mertzwiller	67
27781	292	\N	Mietesheim	67
27782	293	\N	Minversheim	67
27783	295	\N	Mittelbergheim	67
27784	296	\N	Mittelhausbergen	67
27785	297	\N	Mittelhausen	67
27786	298	\N	Mittelschaeffolsheim	67
27787	299	\N	Mollkirch	67
27788	300	\N	Molsheim	67
27789	301	\N	Mommenheim	67
27790	302	\N	Monswiller	67
27791	303	\N	Morsbronn-les-Bains	67
27792	304	\N	Morschwiller	67
27793	305	\N	Mothern	67
27794	306	\N	Muhlbach-sur-Bruche	67
27795	307	\N	Mulhausen	67
27796	308	\N	Munchhausen	67
27797	309	\N	Mundolsheim	67
27798	310	\N	Mussig	67
27799	311	\N	Muttersholtz	67
27800	312	\N	Mutzenhouse	67
27801	313	\N	Mutzig	67
27802	314	\N	Natzwiller	67
27803	315	\N	Neewiller-près-Lauterbourg	67
27804	317	\N	Neubois	67
27805	319	\N	Neuhaeusel	67
27806	320	\N	Neuve-Église	67
27807	321	\N	Neuviller-la-Roche	67
27808	322	\N	Neuwiller-lès-Saverne	67
27809	324	\N	Niederbronn-les-Bains	67
27810	325	\N	Niederhaslach	67
27811	326	\N	Niederhausbergen	67
27812	327	\N	Niederlauterbach	67
27813	328	\N	Niedermodern	67
27814	329	\N	Niedernai	67
27815	330	\N	Niederrdern	67
27816	331	\N	Niederschaeffolsheim	67
27817	333	\N	Niedersoultzbach	67
27818	334	\N	Niedersteinbach	67
27819	335	\N	Nordheim	67
27820	336	\N	Nordhouse	67
27821	337	\N	Nothalten	67
27822	338	\N	Obenheim	67
27823	339	\N	Betschdorf	67
27824	340	\N	Oberbronn	67
27825	341	\N	Oberdorf-Spachbach	67
27826	342	\N	Oberhaslach	67
27827	343	\N	Oberhausbergen	67
27828	344	\N	Oberhoffen-lès-Wissembourg	67
27829	345	\N	Oberhoffen-sur-Moder	67
27830	346	\N	Oberlauterbach	67
27831	347	\N	Obermodern-Zutzendorf	67
27832	348	\N	Obernai	67
27833	349	\N	Oberrdern	67
27834	350	\N	Oberschaeffolsheim	67
27835	351	\N	Seebach	67
27836	352	\N	Obersoultzbach	67
27837	353	\N	Obersteinbach	67
27838	354	\N	Odratzheim	67
27839	355	\N	Oermingen	67
27840	356	\N	Offendorf	67
27841	358	\N	Offwiller	67
27842	359	\N	Ohlungen	67
27843	360	\N	Ohnenheim	67
27844	361	\N	Olwisheim	67
27845	362	\N	Orschwiller	67
27846	363	\N	Osthoffen	67
27847	364	\N	Osthouse	67
27848	365	\N	Ostwald	67
27849	366	\N	Ottersthal	67
27850	367	\N	Otterswiller	67
27851	368	\N	Ottrott	67
27852	369	\N	Ottwiller	67
27853	370	\N	Petersbach	67
27854	371	\N	Petite-Pierre	67
27855	372	\N	Pfaffenhoffen	67
27856	373	\N	Pfalzweyer	67
27857	374	\N	Pfettisheim	67
27858	375	\N	Pfulgriesheim	67
27859	377	\N	Plaine	67
27860	378	\N	Plobsheim	67
27861	379	\N	Preuschdorf	67
27862	380	\N	Printzheim	67
27863	381	\N	Puberg	67
27864	382	\N	Quatzenheim	67
27865	383	\N	Rangen	67
27866	384	\N	Ranrupt	67
27867	385	\N	Ratzwiller	67
27868	386	\N	Rauwiller	67
27869	387	\N	Reichsfeld	67
27870	388	\N	Reichshoffen	67
27871	389	\N	Reichstett	67
27872	391	\N	Reinhardsmunster	67
27873	392	\N	Reipertswiller	67
27874	394	\N	Retschwiller	67
27875	395	\N	Reutenbourg	67
27876	396	\N	Rexingen	67
27877	397	\N	Rhinau	67
27878	398	\N	Richtolsheim	67
27879	400	\N	Riedseltz	67
27880	401	\N	Rimsdorf	67
27881	402	\N	Ringeldorf	67
27882	403	\N	Ringendorf	67
27883	404	\N	Rittershoffen	67
27884	405	\N	Rschwoog	67
27885	406	\N	Rohr	67
27886	407	\N	Rohrwiller	67
27887	408	\N	Romanswiller	67
27888	409	\N	Roppenheim	67
27889	410	\N	Rosenwiller	67
27890	411	\N	Rosheim	67
27891	412	\N	Rossfeld	67
27892	413	\N	Rosteig	67
27893	414	\N	Rothau	67
27894	415	\N	Rothbach	67
27895	416	\N	Rott	67
27896	417	\N	Rottelsheim	67
27897	418	\N	Rountzenheim	67
27898	420	\N	Russ	67
27899	421	\N	Saales	67
27900	422	\N	Saasenheim	67
27901	423	\N	Saessolsheim	67
27902	424	\N	Saint-Blaise-la-Roche	67
27903	425	\N	Saint-Jean-Saverne	67
27904	426	\N	Saint-Martin	67
27905	427	\N	Saint-Maurice	67
27906	428	\N	Saint-Nabor	67
27907	429	\N	Saint-Pierre	67
27908	430	\N	Saint-Pierre-Bois	67
27909	431	\N	Salenthal	67
27910	432	\N	Salmbach	67
27911	433	\N	Sand	67
27912	434	\N	Sarre-Union	67
27913	435	\N	Sarrewerden	67
27914	436	\N	Saulxures	67
27915	437	\N	Saverne	67
27916	438	\N	Schaeffersheim	67
27917	439	\N	Schaffhouse-sur-Zorn	67
27918	440	\N	Schaffhouse-près-Seltz	67
27919	441	\N	Schalkendorf	67
27920	442	\N	Scharrachbergheim-Irmstett	67
27921	443	\N	Scheibenhard	67
27922	444	\N	Scherlenheim	67
27923	445	\N	Scherwiller	67
27924	446	\N	Schillersdorf	67
27925	447	\N	Schiltigheim	67
27926	448	\N	Schirmeck	67
27927	449	\N	Schirrhein	67
27928	450	\N	Schirrhoffen	67
27929	451	\N	Schleithal	67
27930	452	\N	Schnersheim	67
27931	453	\N	Schnau	67
27932	454	\N	Schnbourg	67
27933	455	\N	Schnenbourg	67
27934	456	\N	Schopperten	67
27935	458	\N	Schweighouse-sur-Moder	67
27936	459	\N	Schwenheim	67
27937	460	\N	Schwindratzheim	67
27938	461	\N	Schwobsheim	67
27939	462	\N	Sélestat	67
27940	463	\N	Seltz	67
27941	464	\N	Sermersheim	67
27942	465	\N	Sessenheim	67
27943	466	\N	Siegen	67
27944	467	\N	Siewiller	67
27945	468	\N	Siltzheim	67
27946	469	\N	Singrist	67
27947	470	\N	Solbach	67
27948	471	\N	Souffelweyersheim	67
27949	472	\N	Soufflenheim	67
27950	473	\N	Soultz-les-Bains	67
27951	474	\N	Soultz-sous-Forêts	67
27952	475	\N	Sparsbach	67
27953	476	\N	Stattmatten	67
27954	477	\N	Steige	67
27955	478	\N	Steinbourg	67
27956	479	\N	Steinseltz	67
27957	480	\N	Still	67
27958	481	\N	Stotzheim	67
27959	482	\N	Strasbourg	67
27960	483	\N	Struth	67
27961	484	\N	Stundwiller	67
27962	485	\N	Stutzheim-Offenheim	67
27963	486	\N	Sundhouse	67
27964	487	\N	Surbourg	67
27965	488	\N	Thal-Drulingen	67
27966	489	\N	Thal-Marmoutier	67
27967	490	\N	Thanvillé	67
27968	491	\N	Tieffenbach	67
27969	492	\N	Traenheim	67
27970	493	\N	Triembach-au-Val	67
27971	494	\N	Trimbach	67
27972	495	\N	Truchtersheim	67
27973	496	\N	Uberach	67
27974	497	\N	Uhlwiller	67
27975	498	\N	Uhrwiller	67
27976	499	\N	Urbeis	67
27977	500	\N	Urmatt	67
27978	501	\N	Uttenheim	67
27979	502	\N	Uttenhoffen	67
27980	503	\N	Uttwiller	67
27981	504	\N	Valff	67
27982	505	\N	Vancelle	67
27983	506	\N	Vendenheim	67
27984	507	\N	Villé	67
27985	508	\N	Vllerdingen	67
27986	509	\N	Volksberg	67
27987	510	\N	Wahlenheim	67
27988	511	\N	Walbourg	67
27989	512	\N	Walck	67
27990	513	\N	Waldersbach	67
27991	514	\N	Waldhambach	67
27992	515	\N	Waldolwisheim	67
27993	516	\N	Waltenheim-sur-Zorn	67
27994	517	\N	Wangen	67
27995	519	\N	Wantzenau	67
27996	520	\N	Wasselonne	67
27997	521	\N	Weinbourg	67
27998	522	\N	Weislingen	67
27999	523	\N	Weitbruch	67
28000	524	\N	Weiterswiller	67
28001	525	\N	Westhoffen	67
28002	526	\N	Westhouse	67
28003	527	\N	Westhouse-Marmoutier	67
28004	528	\N	Weyer	67
28005	529	\N	Weyersheim	67
28006	530	\N	Wickersheim-Wilshausen	67
28007	531	\N	Wildersbach	67
28008	532	\N	Willgottheim	67
28009	534	\N	Wilwisheim	67
28010	535	\N	Wimmenau	67
28011	536	\N	Windstein	67
28012	537	\N	Wingen	67
28013	538	\N	Wingen-sur-Moder	67
28014	539	\N	Wingersheim	67
28015	540	\N	Wintershouse	67
28016	541	\N	Wintzenbach	67
28017	542	\N	Wintzenheim-Kochersberg	67
28018	543	\N	Wisches	67
28019	544	\N	Wissembourg	67
28020	545	\N	Witternheim	67
28021	546	\N	Wittersheim	67
28022	547	\N	Wittisheim	67
28023	548	\N	Wiwersheim	67
28024	550	\N	Wrth	67
28025	551	\N	Wolfisheim	67
28026	552	\N	Wolfskirchen	67
28027	553	\N	Wolschheim	67
28028	554	\N	Wolxheim	67
28029	555	\N	Zehnacker	67
28030	556	\N	Zeinheim	67
28031	557	\N	Zellwiller	67
28032	558	\N	Zinswiller	67
28033	559	\N	Zittersheim	67
28034	560	\N	Zbersdorf	67
28035	001	\N	Algolsheim	68
28036	002	\N	Altenach	68
28037	004	\N	Altkirch	68
28038	005	\N	Ammerschwihr	68
28039	006	\N	Ammerzwiller	68
28040	007	\N	Andolsheim	68
28041	008	\N	Appenwihr	68
28042	009	\N	Artzenheim	68
28043	010	\N	Aspach	68
28044	011	\N	Aspach-le-Bas	68
28045	012	\N	Aspach-le-Haut	68
28046	013	\N	Attenschwiller	68
28047	014	\N	Aubure	68
28048	015	\N	Baldersheim	68
28049	016	\N	Balgau	68
28050	017	\N	Ballersdorf	68
28051	018	\N	Balschwiller	68
28052	019	\N	Baltzenheim	68
28053	020	\N	Bantzenheim	68
28054	021	\N	Bartenheim	68
28055	022	\N	Battenheim	68
28056	023	\N	Beblenheim	68
28057	024	\N	Bellemagny	68
28058	025	\N	Bendorf	68
28059	026	\N	Bennwihr	68
28060	027	\N	Berentzwiller	68
28061	028	\N	Bergheim	68
28062	029	\N	Bergholtz	68
28063	030	\N	Bergholtzzell	68
28064	031	\N	Bernwiller	68
28065	032	\N	Berrwiller	68
28066	033	\N	Bettendorf	68
28067	034	\N	Bettlach	68
28068	035	\N	Biederthal	68
28069	036	\N	Biesheim	68
28070	037	\N	Biltzheim	68
28071	038	\N	Bischwihr	68
28072	039	\N	Bisel	68
28073	040	\N	Bitschwiller-lès-Thann	68
28074	041	\N	Blodelsheim	68
28075	042	\N	Blotzheim	68
28076	043	\N	Bollwiller	68
28077	044	\N	Bonhomme	68
28078	045	\N	Bourbach-le-Bas	68
28079	046	\N	Bourbach-le-Haut	68
28080	049	\N	Bouxwiller	68
28081	050	\N	Bréchaumont	68
28082	051	\N	Breitenbach-Haut-Rhin	68
28083	052	\N	Bretten	68
28084	054	\N	Brinckheim	68
28085	055	\N	Bruebach	68
28086	056	\N	Brunstatt	68
28087	057	\N	Buethwiller	68
28088	058	\N	Buhl	68
28089	059	\N	Burnhaupt-le-Bas	68
28090	060	\N	Burnhaupt-le-Haut	68
28091	061	\N	Buschwiller	68
28092	062	\N	Carspach	68
28093	063	\N	Cernay	68
28094	064	\N	Chalampé	68
28095	065	\N	Chavannes-sur-l'Étang	68
28096	066	\N	Colmar	68
28097	067	\N	Courtavon	68
28098	068	\N	Dannemarie	68
28099	069	\N	Dessenheim	68
28100	070	\N	Didenheim	68
28101	071	\N	Diefmatten	68
28102	072	\N	Dietwiller	68
28103	073	\N	Dolleren	68
28104	074	\N	Durlinsdorf	68
28105	075	\N	Durmenach	68
28106	076	\N	Durrenentzen	68
28107	077	\N	Eglingen	68
28108	078	\N	Eguisheim	68
28109	079	\N	Elbach	68
28110	080	\N	Emlingen	68
28111	081	\N	Saint-Bernard	68
28112	082	\N	Ensisheim	68
28113	083	\N	Eschbach-au-Val	68
28114	084	\N	Eschentzwiller	68
28115	085	\N	Eteimbes	68
28116	086	\N	Falkwiller	68
28117	087	\N	Feldbach	68
28118	088	\N	Feldkirch	68
28119	089	\N	Fellering	68
28120	090	\N	Ferrette	68
28121	091	\N	Fessenheim	68
28122	092	\N	Fislis	68
28123	093	\N	Flaxlanden	68
28124	094	\N	Folgensbourg	68
28125	095	\N	Fortschwihr	68
28126	096	\N	Franken	68
28127	097	\N	Fréland	68
28128	098	\N	Friesen	68
28129	099	\N	Frningen	68
28130	100	\N	Fulleren	68
28131	101	\N	Galfingue	68
28132	102	\N	Geishouse	68
28133	103	\N	Geispitzen	68
28134	104	\N	Geiswasser	68
28135	105	\N	Gildwiller	68
28136	106	\N	Goldbach-Altenbach	68
28137	107	\N	Gommersdorf	68
28138	108	\N	Grentzingen	68
28139	109	\N	Griesbach-au-Val	68
28140	110	\N	Grussenheim	68
28141	111	\N	Gueberschwihr	68
28142	112	\N	Guebwiller	68
28143	113	\N	Guémar	68
28144	114	\N	Guevenatten	68
28145	115	\N	Guewenheim	68
28146	116	\N	Gundolsheim	68
28147	117	\N	Gunsbach	68
28148	118	\N	Habsheim	68
28149	119	\N	Hagenbach	68
28150	120	\N	Hagenthal-le-Bas	68
28151	121	\N	Hagenthal-le-Haut	68
28152	122	\N	Hartmannswiller	68
28153	123	\N	Hattstatt	68
28154	124	\N	Hausgauen	68
28155	125	\N	Hecken	68
28156	126	\N	Hégenheim	68
28157	127	\N	Heidwiller	68
28158	128	\N	Heimersdorf	68
28159	129	\N	Heimsbrunn	68
28160	130	\N	Heiteren	68
28161	131	\N	Heiwiller	68
28162	132	\N	Helfrantzkirch	68
28163	133	\N	Henflingen	68
28164	134	\N	Herrlisheim-près-Colmar	68
28165	135	\N	Hésingue	68
28166	136	\N	Hettenschlag	68
28167	137	\N	Hindlingen	68
28168	138	\N	Hirsingue	68
28169	139	\N	Hirtzbach	68
28170	140	\N	Hirtzfelden	68
28171	141	\N	Hochstatt	68
28172	142	\N	Hohrod	68
28173	143	\N	Holtzwihr	68
28174	144	\N	Hombourg	68
28175	145	\N	Horbourg-Wihr	68
28176	146	\N	Houssen	68
28177	147	\N	Hunawihr	68
28178	148	\N	Hundsbach	68
28179	149	\N	Huningue	68
28180	150	\N	Husseren-les-Châteaux	68
28181	151	\N	Husseren-Wesserling	68
28182	152	\N	Illfurth	68
28183	153	\N	Illhaeusern	68
28184	154	\N	Illzach	68
28185	155	\N	Ingersheim	68
28186	156	\N	Issenheim	68
28187	157	\N	Jebsheim	68
28188	158	\N	Jettingen	68
28189	159	\N	Jungholtz	68
28190	160	\N	Kappelen	68
28191	161	\N	Katzenthal	68
28192	162	\N	Kaysersberg	68
28193	163	\N	Kembs	68
28194	164	\N	Kientzheim	68
28195	165	\N	Kiffis	68
28196	166	\N	Kingersheim	68
28197	167	\N	Kirchberg	68
28198	168	\N	Knringue	68
28199	169	\N	Kstlach	68
28200	170	\N	Ktzingue	68
28201	171	\N	Kruth	68
28202	172	\N	Kunheim	68
28203	173	\N	Labaroche	68
28204	174	\N	Landser	68
28205	175	\N	Lapoutroie	68
28206	176	\N	Largitzen	68
28207	177	\N	Lautenbach	68
28208	178	\N	Lautenbachzell	68
28209	179	\N	Lauw	68
28210	180	\N	Leimbach	68
28211	181	\N	Levoncourt	68
28212	182	\N	Leymen	68
28213	183	\N	Liebenswiller	68
28214	184	\N	Liebsdorf	68
28215	185	\N	Lièpvre	68
28216	186	\N	Ligsdorf	68
28217	187	\N	Linsdorf	68
28218	188	\N	Linthal	68
28219	189	\N	Logelheim	68
28220	190	\N	Lucelle	68
28221	191	\N	Luemschwiller	68
28222	192	\N	Valdieu-Lutran	68
28223	193	\N	Luttenbach-près-Munster	68
28224	194	\N	Lutter	68
28225	195	\N	Lutterbach	68
28226	196	\N	Magny	68
28227	197	\N	Magstatt-le-Bas	68
28228	198	\N	Magstatt-le-Haut	68
28229	199	\N	Malmerspach	68
28230	200	\N	Manspach	68
28231	201	\N	Masevaux	68
28232	202	\N	Mertzen	68
28233	203	\N	Merxheim	68
28234	204	\N	Metzeral	68
28235	205	\N	Meyenheim	68
28236	206	\N	Michelbach	68
28237	207	\N	Michelbach-le-Bas	68
28238	208	\N	Michelbach-le-Haut	68
28239	209	\N	Mittelwihr	68
28240	210	\N	Mittlach	68
28241	211	\N	Mitzach	68
28242	212	\N	Mrnach	68
28243	213	\N	Mollau	68
28244	214	\N	Montreux-Jeune	68
28245	215	\N	Montreux-Vieux	68
28246	216	\N	Mooslargue	68
28247	217	\N	Moosch	68
28248	218	\N	Morschwiller-le-Bas	68
28249	219	\N	Mortzwiller	68
28250	221	\N	Muespach	68
28251	222	\N	Muespach-le-Haut	68
28252	223	\N	Muhlbach-sur-Munster	68
28253	224	\N	Mulhouse	68
28254	225	\N	Munchhouse	68
28255	226	\N	Munster	68
28256	227	\N	Muntzenheim	68
28257	228	\N	Munwiller	68
28258	229	\N	Murbach	68
28259	230	\N	Nambsheim	68
28260	231	\N	Neuf-Brisach	68
28261	232	\N	Neuwiller	68
28262	233	\N	Niederbruck	68
28263	234	\N	Niederentzen	68
28264	235	\N	Niederhergheim	68
28265	237	\N	Niedermorschwihr	68
28266	238	\N	Niffer	68
28267	239	\N	Oberbruck	68
28268	240	\N	Oberdorf	68
28269	241	\N	Oberentzen	68
28270	242	\N	Oberhergheim	68
28271	243	\N	Oberlarg	68
28272	244	\N	Obermorschwihr	68
28273	245	\N	Obermorschwiller	68
28274	246	\N	Obersaasheim	68
28275	247	\N	Oderen	68
28276	248	\N	Oltingue	68
28277	249	\N	Orbey	68
28278	250	\N	Orschwihr	68
28279	251	\N	Osenbach	68
28280	252	\N	Ostheim	68
28281	253	\N	Ottmarsheim	68
28282	254	\N	Petit-Landau	68
28283	255	\N	Pfaffenheim	68
28284	256	\N	Pfastatt	68
28285	257	\N	Pfetterhouse	68
28286	258	\N	Pulversheim	68
28287	259	\N	Raedersdorf	68
28288	260	\N	Raedersheim	68
28289	261	\N	Rammersmatt	68
28290	262	\N	Ranspach	68
28291	263	\N	Ranspach-le-Bas	68
28292	264	\N	Ranspach-le-Haut	68
28293	265	\N	Rantzwiller	68
28294	266	\N	Réguisheim	68
28295	267	\N	Reiningue	68
28296	268	\N	Retzwiller	68
28297	269	\N	Ribeauvillé	68
28298	270	\N	Richwiller	68
28299	271	\N	Riedisheim	68
28300	272	\N	Riedwihr	68
28301	273	\N	Riespach	68
28302	274	\N	Rimbach-près-Guebwiller	68
28303	275	\N	Rimbach-près-Masevaux	68
28304	276	\N	Rimbachzell	68
28305	277	\N	Riquewihr	68
28306	278	\N	Rixheim	68
28307	279	\N	Roderen	68
28308	280	\N	Rodern	68
28309	281	\N	Roggenhouse	68
28310	282	\N	Romagny	68
28311	283	\N	Rombach-le-Franc	68
28312	284	\N	Roppentzwiller	68
28313	285	\N	Rorschwihr	68
28314	286	\N	Rosenau	68
28315	287	\N	Rouffach	68
28316	288	\N	Ruederbach	68
28317	289	\N	Ruelisheim	68
28318	290	\N	Rustenhart	68
28319	291	\N	Rumersheim-le-Haut	68
28320	292	\N	Saint-Amarin	68
28321	293	\N	Saint-Cosme	68
28322	294	\N	Sainte-Croix-aux-Mines	68
28323	295	\N	Sainte-Croix-en-Plaine	68
28324	296	\N	Saint-Hippolyte	68
28325	297	\N	Saint-Louis	68
28326	298	\N	Sainte-Marie-aux-Mines	68
28327	299	\N	Saint-Ulrich	68
28328	300	\N	Sausheim	68
28329	301	\N	Schlierbach	68
28330	302	\N	Schweighouse-Thann	68
28331	303	\N	Schwoben	68
28332	304	\N	Sentheim	68
28333	305	\N	Seppois-le-Bas	68
28334	306	\N	Seppois-le-Haut	68
28335	307	\N	Sewen	68
28336	308	\N	Sickert	68
28337	309	\N	Sierentz	68
28338	310	\N	Sigolsheim	68
28339	311	\N	Sondernach	68
28340	312	\N	Sondersdorf	68
28341	313	\N	Soppe-le-Bas	68
28342	314	\N	Soppe-le-Haut	68
28343	315	\N	Soultz-Haut-Rhin	68
28344	316	\N	Soultzbach-les-Bains	68
28345	317	\N	Soultzeren	68
28346	318	\N	Soultzmatt	68
28347	319	\N	Spechbach-le-Bas	68
28348	320	\N	Spechbach-le-Haut	68
28349	321	\N	Staffelfelden	68
28350	322	\N	Steinbach	68
28351	323	\N	Steinbrunn-le-Bas	68
28352	324	\N	Steinbrunn-le-Haut	68
28353	325	\N	Steinsoultz	68
28354	326	\N	Sternenberg	68
28355	327	\N	Stetten	68
28356	328	\N	Storckensohn	68
28357	329	\N	Stosswihr	68
28358	330	\N	Strueth	68
28359	331	\N	Sundhoffen	68
28360	332	\N	Tagolsheim	68
28361	333	\N	Tagsdorf	68
28362	334	\N	Thann	68
28363	335	\N	Thannenkirch	68
28364	336	\N	Traubach-le-Bas	68
28365	337	\N	Traubach-le-Haut	68
28366	338	\N	Turckheim	68
28367	340	\N	Ueberstrass	68
28368	341	\N	Uffheim	68
28369	342	\N	Uffholtz	68
28370	343	\N	Ungersheim	68
28371	344	\N	Urbès	68
28372	345	\N	Urschenheim	68
28373	347	\N	Vieux-Ferrette	68
28374	348	\N	Vieux-Thann	68
28375	349	\N	Village-Neuf	68
28376	350	\N	Vgtlinshoffen	68
28377	351	\N	Vogelgrun	68
28378	352	\N	Volgelsheim	68
28379	353	\N	Wahlbach	68
28380	354	\N	Walbach	68
28381	355	\N	Waldighofen	68
28382	356	\N	Walheim	68
28383	357	\N	Waltenheim	68
28384	358	\N	Wasserbourg	68
28385	359	\N	Wattwiller	68
28386	360	\N	Weckolsheim	68
28387	361	\N	Wegscheid	68
28388	362	\N	Wentzwiller	68
28389	363	\N	Werentzhouse	68
28390	364	\N	Westhalten	68
28391	365	\N	Wettolsheim	68
28392	366	\N	Wickerschwihr	68
28393	367	\N	Widensolen	68
28394	368	\N	Wihr-au-Val	68
28395	370	\N	Wildenstein	68
28396	371	\N	Willer	68
28397	372	\N	Willer-sur-Thur	68
28398	373	\N	Winkel	68
28399	374	\N	Wintzenheim	68
28400	375	\N	Wittelsheim	68
28401	376	\N	Wittenheim	68
28402	377	\N	Wittersdorf	68
28403	378	\N	Wolfersdorf	68
28404	379	\N	Wolfgantzen	68
28405	380	\N	Wolschwiller	68
28406	381	\N	Wuenheim	68
28407	382	\N	Zaessingue	68
28408	383	\N	Zellenberg	68
28409	384	\N	Zillisheim	68
28410	385	\N	Zimmerbach	68
28411	386	\N	Zimmersheim	68
28412	001	\N	Affoux	69
28413	002	\N	Aigueperse	69
28414	003	\N	Albigny-sur-Saône	69
28415	004	\N	Alix	69
28416	005	\N	Ambérieux	69
28417	006	\N	Amplepuis	69
28418	007	\N	Ampuis	69
28419	008	\N	Ancy	69
28420	009	\N	Anse	69
28421	010	\N	Arbresle	69
28422	012	\N	Ardillats	69
28423	013	\N	Arnas	69
28424	014	\N	Aveize	69
28425	015	\N	Avenas	69
28426	016	\N	Azolette	69
28427	017	\N	Bagnols	69
28428	018	\N	Beaujeu	69
28429	019	\N	Belleville	69
28430	020	\N	Belmont-d'Azergues	69
28431	021	\N	Bessenay	69
28432	022	\N	Bibost	69
28433	023	\N	Blacé	69
28434	024	\N	Bois-d'Oingt	69
28435	025	\N	Bourg-de-Thizy	69
28436	026	\N	Breuil	69
28437	027	\N	Brignais	69
28438	028	\N	Brindas	69
28439	029	\N	Bron	69
28440	030	\N	Brullioles	69
28441	031	\N	Brussieu	69
28442	032	\N	Bully	69
28443	033	\N	Cailloux-sur-Fontaines	69
28444	034	\N	Caluire-et-Cuire	69
28445	035	\N	Cenves	69
28446	036	\N	Cercié	69
28447	037	\N	Chambost-Allières	69
28448	038	\N	Chambost-Longessaigne	69
28449	039	\N	Chamelet	69
28450	040	\N	Champagne-au-Mont-d'Or	69
28451	041	\N	Chapelle-de-Mardore	69
28452	042	\N	Chapelle-sur-Coise	69
28453	043	\N	Chaponost	69
28454	044	\N	Charbonnières-les-Bains	69
28455	045	\N	Charentay	69
28456	046	\N	Charly	69
28457	047	\N	Charnay	69
28458	048	\N	Chassagny	69
28459	049	\N	Chasselay	69
28460	050	\N	Châtillon	69
28461	051	\N	Chaussan	69
28462	052	\N	Chazay-d'Azergues	69
28463	053	\N	Chénas	69
28464	054	\N	Chénelette	69
28465	055	\N	Chères	69
28466	056	\N	Chessy	69
28467	057	\N	Chevinay	69
28468	058	\N	Chiroubles	69
28469	059	\N	Civrieux-d'Azergues	69
28470	060	\N	Claveisolles	69
28471	061	\N	Cogny	69
28472	062	\N	Coise	69
28473	063	\N	Collonges-au-Mont-d'Or	69
28474	064	\N	Condrieu	69
28475	065	\N	Corcelles-en-Beaujolais	69
28476	066	\N	Cours-la-Ville	69
28477	067	\N	Courzieu	69
28478	068	\N	Couzon-au-Mont-d'Or	69
28479	069	\N	Craponne	69
28480	070	\N	Cublize	69
28481	071	\N	Curis-au-Mont-d'Or	69
28482	072	\N	Dardilly	69
28483	073	\N	Dareizé	69
28484	074	\N	Denicé	69
28485	075	\N	Dième	69
28486	076	\N	Dommartin	69
28487	077	\N	Dracé	69
28488	078	\N	Duerne	69
28489	080	\N	Échalas	69
28490	081	\N	Écully	69
28491	082	\N	Émeringes	69
28492	083	\N	Éveux	69
28493	084	\N	Fleurie	69
28494	085	\N	Fleurieu-sur-Saône	69
28495	086	\N	Fleurieux-sur-l'Arbresle	69
28496	087	\N	Fontaines-Saint-Martin	69
28497	088	\N	Fontaines-sur-Saône	69
28498	089	\N	Francheville	69
28499	090	\N	Frontenas	69
28500	091	\N	Givors	69
28501	092	\N	Gleizé	69
28502	093	\N	Grandris	69
28503	094	\N	Grézieu-la-Varenne	69
28504	095	\N	Grézieu-le-Marché	69
28505	096	\N	Grigny	69
28506	097	\N	Haies	69
28507	098	\N	Halles	69
28508	099	\N	Haute-Rivoire	69
28509	100	\N	Irigny	69
28510	101	\N	Jarnioux	69
28511	102	\N	Joux	69
28512	103	\N	Juliénas	69
28513	104	\N	Jullié	69
28514	105	\N	Lacenas	69
28515	106	\N	Lachassagne	69
28516	107	\N	Lamure-sur-Azergues	69
28517	108	\N	Lancié	69
28518	109	\N	Lantignié	69
28519	110	\N	Larajasse	69
28520	111	\N	Légny	69
28521	112	\N	Lentilly	69
28522	113	\N	Létra	69
28523	114	\N	Liergues	69
28524	115	\N	Limas	69
28525	116	\N	Limonest	69
28526	117	\N	Lissieu	69
28527	118	\N	Loire-sur-Rhône	69
28528	119	\N	Longes	69
28529	120	\N	Longessaigne	69
28530	121	\N	Lozanne	69
28531	122	\N	Lucenay	69
28532	123	\N	Lyon	69
28533	124	\N	Marchampt	69
28534	125	\N	Marcilly-d'Azergues	69
28535	126	\N	Marcy	69
28536	127	\N	Marcy-l'Étoile	69
28537	128	\N	Mardore	69
28538	129	\N	Marnand	69
28539	130	\N	Meaux-la-Montagne	69
28540	131	\N	Messimy	69
28541	132	\N	Meys	69
28542	133	\N	Millery	69
28543	134	\N	Moiré	69
28544	135	\N	Monsols	69
28545	136	\N	Montagny	69
28546	137	\N	Montmelas-Saint-Sorlin	69
28547	138	\N	Montromant	69
28548	139	\N	Montrottier	69
28549	140	\N	Morancé	69
28550	141	\N	Mornant	69
28551	142	\N	Mulatière	69
28552	143	\N	Neuville-sur-Saône	69
28553	144	\N	Nuelles	69
28554	145	\N	Odenas	69
28555	146	\N	Oingt	69
28556	147	\N	Olmes	69
28557	148	\N	Orliénas	69
28558	149	\N	Oullins	69
28559	150	\N	Ouroux	69
28560	151	\N	Perréon	69
28561	152	\N	Pierre-Bénite	69
28562	153	\N	Poleymieux-au-Mont-d'Or	69
28563	154	\N	Pollionnay	69
28564	155	\N	Pomeys	69
28565	156	\N	Pommiers	69
28566	157	\N	Pontcharra-sur-Turdine	69
28567	158	\N	Pont-Trambouze	69
28568	159	\N	Pouilly-le-Monial	69
28569	160	\N	Poule-les-Écharmeaux	69
28570	161	\N	Propières	69
28571	162	\N	Quincié-en-Beaujolais	69
28572	163	\N	Quincieux	69
28573	164	\N	Ranchal	69
28574	165	\N	Régnié-Durette	69
28575	166	\N	Riverie	69
28576	167	\N	Rivolet	69
28577	168	\N	Rochetaillée-sur-Saône	69
28578	169	\N	Ronno	69
28579	170	\N	Rontalon	69
28580	171	\N	Sain-Bel	69
28581	172	\N	Salles-Arbuissonnas-en-Beaujolais	69
28582	173	\N	Sarcey	69
28583	174	\N	Sauvages	69
28584	175	\N	Savigny	69
28585	176	\N	Soucieu-en-Jarrest	69
28586	177	\N	Sourcieux-les-Mines	69
28587	178	\N	Souzy	69
28588	179	\N	Saint-Andéol-le-Château	69
28589	180	\N	Saint-André-la-Côte	69
28590	181	\N	Saint-Appolinaire	69
28591	182	\N	Saint-Bonnet-des-Bruyères	69
28592	183	\N	Saint-Bonnet-le-Troncy	69
28593	184	\N	Sainte-Catherine	69
28594	185	\N	Saint-Christophe	69
28595	186	\N	Saint-Clément-de-Vers	69
28596	187	\N	Saint-Clément-les-Places	69
28597	188	\N	Saint-Clément-sur-Valsonne	69
28598	189	\N	Sainte-Colombe	69
28599	190	\N	Sainte-Consorce	69
28600	191	\N	Saint-Cyr-au-Mont-d'Or	69
28601	192	\N	Saint-Cyr-le-Chatoux	69
28602	193	\N	Saint-Cyr-sur-le-Rhône	69
28603	194	\N	Saint-Didier-au-Mont-d'Or	69
28604	195	\N	Saint-Didier-sous-Riverie	69
28605	196	\N	Saint-Didier-sur-Beaujeu	69
28606	197	\N	Saint-Étienne-des-Oullières	69
28607	198	\N	Saint-Étienne-la-Varenne	69
28608	199	\N	Saint-Fons	69
28609	200	\N	Saint-Forgeux	69
28610	201	\N	Sainte-Foy-l'Argentière	69
28611	202	\N	Sainte-Foy-lès-Lyon	69
28612	203	\N	Saint-Genis-l'Argentière	69
28613	204	\N	Saint-Genis-Laval	69
28614	205	\N	Saint-Genis-les-Ollières	69
28615	206	\N	Saint-Georges-de-Reneins	69
28616	207	\N	Saint-Germain-au-Mont-d'Or	69
28617	208	\N	Saint-Germain-sur-l'Arbresle	69
28618	209	\N	Saint-Igny-de-Vers	69
28619	210	\N	Saint-Jacques-des-Arrêts	69
28620	211	\N	Saint-Jean-d'Ardières	69
28621	212	\N	Saint-Jean-des-Vignes	69
28622	213	\N	Saint-Jean-de-Touslas	69
28623	214	\N	Saint-Jean-la-Bussière	69
28624	215	\N	Saint-Julien	69
28625	216	\N	Saint-Julien-sur-Bibost	69
28626	217	\N	Saint-Just-d'Avray	69
28627	218	\N	Saint-Lager	69
28628	219	\N	Saint-Laurent-d'Agny	69
28629	220	\N	Saint-Laurent-de-Chamousset	69
28630	221	\N	Saint-Laurent-de-Vaux	69
28631	222	\N	Saint-Laurent-d'Oingt	69
28632	223	\N	Saint-Loup	69
28633	224	\N	Saint-Mamert	69
28634	225	\N	Saint-Marcel-l'Éclairé	69
28635	227	\N	Saint-Martin-en-Haut	69
28636	228	\N	Saint-Maurice-sur-Dargoire	69
28637	229	\N	Saint-Nizier-d'Azergues	69
28638	230	\N	Sainte-Paule	69
28639	231	\N	Saint-Pierre-la-Palud	69
28640	233	\N	Saint-Romain-au-Mont-d'Or	69
28641	234	\N	Saint-Romain-de-Popey	69
28642	235	\N	Saint-Romain-en-Gal	69
28643	236	\N	Saint-Romain-en-Gier	69
28644	237	\N	Saint-Sorlin	69
28645	238	\N	Saint-Symphorien-sur-Coise	69
28646	239	\N	Saint-Vérand	69
28647	240	\N	Saint-Vincent-de-Reins	69
28648	241	\N	Taluyers	69
28649	242	\N	Taponas	69
28650	243	\N	Tarare	69
28651	244	\N	Tassin-la-Demi-Lune	69
28652	245	\N	Ternand	69
28653	246	\N	Theizé	69
28654	247	\N	Thel	69
28655	248	\N	Thizy	69
28656	249	\N	Thurins	69
28657	250	\N	Tour-de-Salvagny	69
28658	251	\N	Trades	69
28659	252	\N	Trèves	69
28660	253	\N	Tupin-et-Semons	69
28661	254	\N	Valsonne	69
28662	255	\N	Vaugneray	69
28663	256	\N	Vaulx-en-Velin	69
28664	257	\N	Vaux-en-Beaujolais	69
28665	258	\N	Vauxrenard	69
28666	259	\N	Vénissieux	69
28667	260	\N	Vernaison	69
28668	261	\N	Vernay	69
28669	263	\N	Villechenève	69
28670	264	\N	Villefranche-sur-Saône	69
28671	265	\N	Ville-sur-Jarnioux	69
28672	266	\N	Villeurbanne	69
28673	267	\N	Villié-Morgon	69
28674	268	\N	Vourles	69
28675	269	\N	Yzeron	69
28676	270	\N	Chaponnay	69
28677	271	\N	Chassieu	69
28678	272	\N	Communay	69
28679	273	\N	Corbas	69
28680	275	\N	Décines-Charpieu	69
28681	276	\N	Feyzin	69
28682	277	\N	Genas	69
28683	278	\N	Genay	69
28684	279	\N	Jonage	69
28685	280	\N	Jons	69
28686	281	\N	Marennes	69
28687	282	\N	Meyzieu	69
28688	283	\N	Mions	69
28689	284	\N	Montanay	69
28690	285	\N	Pusignan	69
28691	286	\N	Rillieux-la-Pape	69
28692	287	\N	Saint-Bonnet-de-Mure	69
28693	288	\N	Saint-Laurent-de-Mure	69
28694	289	\N	Saint-Pierre-de-Chandieu	69
28695	290	\N	Saint-Priest	69
28696	291	\N	Saint-Symphorien-d'Ozon	69
28697	292	\N	Sathonay-Camp	69
28698	293	\N	Sathonay-Village	69
28699	294	\N	Sérézin-du-Rhône	69
28700	295	\N	Simandres	69
28701	296	\N	Solaize	69
28702	297	\N	Ternay	69
28703	298	\N	Toussieu	69
28704	299	\N	Colombier-Saugnieu	69
28705	001	\N	Abelcourt	70
28706	002	\N	Aboncourt-Gesincourt	70
28707	003	\N	Achey	70
28708	004	\N	Adelans-et-le-Val-de-Bithaine	70
28709	005	\N	Aillevans	70
28710	006	\N	Aillevillers-et-Lyaumont	70
28711	007	\N	Ailloncourt	70
28712	008	\N	Ainvelle	70
28713	009	\N	Aisey-et-Richecourt	70
28714	010	\N	Alaincourt	70
28715	011	\N	Amage	70
28716	012	\N	Amance	70
28717	013	\N	Ambiévillers	70
28718	014	\N	Amblans-et-Velotte	70
28719	015	\N	Amoncourt	70
28720	016	\N	Amont-et-Effreney	70
28721	017	\N	Anchenoncourt-et-Chazel	70
28722	018	\N	Ancier	70
28723	019	\N	Andelarre	70
28724	020	\N	Andelarrot	70
28725	021	\N	Andornay	70
28726	022	\N	Angirey	70
28727	023	\N	Anjeux	70
28728	024	\N	Apremont	70
28729	025	\N	Arbecey	70
28730	026	\N	Arc-lès-Gray	70
28731	027	\N	Argillières	70
28732	028	\N	Aroz	70
28733	029	\N	Arpenans	70
28734	030	\N	Arsans	70
28735	031	\N	Athesans-Étroitefontaine	70
28736	032	\N	Attricourt	70
28737	035	\N	Augicourt	70
28738	036	\N	Aulx-lès-Cromary	70
28739	037	\N	Autet	70
28740	038	\N	Authoison	70
28741	039	\N	Autoreille	70
28742	040	\N	Autrey-lès-Cerre	70
28743	041	\N	Autrey-lès-Gray	70
28744	042	\N	Autrey-le-Vay	70
28745	043	\N	Auvet-et-la-Chapelotte	70
28746	044	\N	Auxon	70
28747	045	\N	Avrigney-Virey	70
28748	046	\N	Aynans	70
28749	047	\N	Baignes	70
28750	048	\N	Bard-lès-Pesmes	70
28751	049	\N	Barges	70
28752	050	\N	Barre	70
28753	051	\N	Basse-Vaivre	70
28754	052	\N	Bassigney	70
28755	053	\N	Bâties	70
28756	054	\N	Battrans	70
28757	055	\N	Baudoncourt	70
28758	056	\N	Baulay	70
28759	057	\N	Bay	70
28760	058	\N	Beaujeu-Saint-Vallier-Pierrejux-et-Quitteur	70
28761	059	\N	Beaumotte-Aubertans	70
28762	060	\N	Beaumotte-lès-Pin	70
28763	061	\N	Belfahy	70
28764	062	\N	Belmont	70
28765	063	\N	Belonchamp	70
28766	064	\N	Belverne	70
28767	065	\N	Besnans	70
28768	066	\N	Betaucourt	70
28769	067	\N	Betoncourt-lès-Brotte	70
28770	069	\N	Betoncourt-Saint-Pancras	70
28771	070	\N	Betoncourt-sur-Mance	70
28772	071	\N	Beulotte-Saint-Laurent	70
28773	072	\N	Beveuge	70
28774	074	\N	Blondefontaine	70
28775	075	\N	Bonboillon	70
28776	076	\N	Bonnevent-Velloreille	70
28777	077	\N	Borey	70
28778	078	\N	Bougey	70
28779	079	\N	Bougnon	70
28780	080	\N	Bouhans-et-Feurg	70
28781	081	\N	Bouhans-lès-Lure	70
28782	082	\N	Bouhans-lès-Montbozon	70
28783	083	\N	Bouligney	70
28784	084	\N	Boulot	70
28785	085	\N	Boult	70
28786	086	\N	Bourbévelle	70
28787	087	\N	Bourguignon-lès-Conflans	70
28788	088	\N	Bourguignon-lès-la-Charité	70
28789	089	\N	Bourguignon-lès-Morey	70
28790	090	\N	Boursières	70
28791	091	\N	Bousseraucourt	70
28792	092	\N	Bresilley	70
28793	093	\N	Breuches	70
28794	094	\N	Breuchotte	70
28795	095	\N	Breurey-lès-Faverney	70
28796	096	\N	Brevilliers	70
28797	097	\N	Briaucourt	70
28798	098	\N	Brotte-lès-Luxeuil	70
28799	099	\N	Brotte-lès-Ray	70
28800	100	\N	Broye-les-Loups-et-Verfontaine	70
28801	101	\N	Broye-Aubigney-Montseugny	70
28802	102	\N	Brussey	70
28803	103	\N	Bruyère	70
28804	104	\N	Bucey-lès-Gy	70
28805	105	\N	Bucey-lès-Traves	70
28806	106	\N	Buffignécourt	70
28807	107	\N	Bussières	70
28808	109	\N	Buthiers	70
28809	111	\N	Calmoutier	70
28810	112	\N	Cemboing	70
28811	113	\N	Cenans	70
28812	114	\N	Cendrecourt	70
28813	115	\N	Cerre-lès-Noroy	70
28814	116	\N	Chagey	70
28815	117	\N	Châlonvillars	70
28816	118	\N	Chambornay-lès-Bellevaux	70
28817	119	\N	Chambornay-lès-Pin	70
28818	120	\N	Champagney	70
28819	121	\N	Champey	70
28820	122	\N	Champlitte	70
28821	124	\N	Champtonnay	70
28822	125	\N	Champvans	70
28823	126	\N	Chancey	70
28824	127	\N	Chantes	70
28825	128	\N	Chapelle-lès-Luxeuil	70
28826	129	\N	Chapelle-Saint-Quillain	70
28827	130	\N	Charcenne	70
28828	132	\N	Chargey-lès-Gray	70
28829	133	\N	Chargey-lès-Port	70
28830	134	\N	Chariez	70
28831	135	\N	Charmes-Saint-Valbert	70
28832	136	\N	Charmoille	70
28833	137	\N	Chassey-lès-Montbozon	70
28834	138	\N	Chassey-lès-Scey	70
28835	140	\N	Châteney	70
28836	141	\N	Châtenois	70
28837	142	\N	Chaumercenne	70
28838	143	\N	Chauvirey-le-Châtel	70
28839	144	\N	Chauvirey-le-Vieil	70
28840	145	\N	Chaux-la-Lotière	70
28841	146	\N	Chaux-lès-Port	70
28842	147	\N	Chavanne	70
28843	148	\N	Chemilly	70
28844	149	\N	Chenebier	70
28845	150	\N	Chenevrey-et-Morogne	70
28846	151	\N	Chevigney	70
28847	152	\N	Choye	70
28848	153	\N	Cintrey	70
28849	154	\N	Cirey	70
28850	155	\N	Citers	70
28851	156	\N	Citey	70
28852	157	\N	Clairegoutte	70
28853	158	\N	Clans	70
28854	159	\N	Cognières	70
28855	160	\N	Coisevaux	70
28856	162	\N	Colombe-lès-Vesoul	70
28857	163	\N	Colombier	70
28858	164	\N	Colombotte	70
28859	165	\N	Combeaufontaine	70
28860	166	\N	Comberjon	70
28861	167	\N	Conflandey	70
28862	168	\N	Conflans-sur-Lanterne	70
28863	169	\N	Confracourt	70
28864	170	\N	Contréglise	70
28865	171	\N	Corbenay	70
28866	172	\N	Corbière	70
28867	174	\N	Cordonnet	70
28868	175	\N	Cornot	70
28869	176	\N	Corravillers	70
28870	177	\N	Corre	70
28871	178	\N	Côte	70
28872	179	\N	Coulevon	70
28873	180	\N	Courchaton	70
28874	181	\N	Courcuire	70
28875	182	\N	Courmont	70
28876	183	\N	Courtesoult-et-Gatey	70
28877	184	\N	Couthenans	70
28878	185	\N	Cresancey	70
28879	186	\N	Creuse	70
28880	187	\N	Crevans-et-la-Chapelle-lès-Granges	70
28881	188	\N	Creveney	70
28882	189	\N	Cromary	70
28883	190	\N	Cubry-lès-Faverney	70
28884	192	\N	Cugney	70
28885	193	\N	Cult	70
28886	194	\N	Cuve	70
28887	195	\N	Dambenoît-lès-Colombe	70
28888	196	\N	Dampierre-lès-Conflans	70
28889	197	\N	Dampierre-sur-Linotte	70
28890	198	\N	Dampierre-sur-Salon	70
28891	199	\N	Dampvalley-lès-Colombe	70
28892	200	\N	Dampvalley-Saint-Pancras	70
28893	201	\N	Delain	70
28894	202	\N	Demangevelle	70
28895	203	\N	Demie	70
28896	204	\N	Denèvre	70
28897	205	\N	Échavanne	70
28898	206	\N	Échenans-sous-Mont-Vaudois	70
28899	207	\N	Échenoz-la-Méline	70
28900	208	\N	Échenoz-le-Sec	70
28901	210	\N	Écromagny	70
28902	211	\N	Écuelle	70
28903	213	\N	Éhuns	70
28904	214	\N	Équevilley	70
28905	215	\N	Errevet	70
28906	216	\N	Esboz-Brest	70
28907	217	\N	Esmoulières	70
28908	218	\N	Esmoulins	70
28909	219	\N	Esprels	70
28910	220	\N	Essertenne-et-Cecey	70
28911	221	\N	Étobon	70
28912	222	\N	Étrelles-et-la-Montbleuse	70
28913	224	\N	Étuz	70
28914	225	\N	Fahy-lès-Autrey	70
28915	226	\N	Fallon	70
28916	227	\N	Faucogney-et-la-Mer	70
28917	228	\N	Faverney	70
28918	229	\N	Faymont	70
28919	230	\N	Fédry	70
28920	231	\N	Ferrières-lès-Ray	70
28921	232	\N	Ferrières-lès-Scey	70
28922	233	\N	Fessey	70
28923	234	\N	Filain	70
28924	235	\N	Flagy	70
28925	236	\N	Fleurey-lès-Faverney	70
28926	237	\N	Fleurey-lès-Lavoncourt	70
28927	238	\N	Fleurey-lès-Saint-Loup	70
28928	239	\N	Fondremand	70
28929	240	\N	Fontaine-lès-Luxeuil	70
28930	242	\N	Fontenois-la-Ville	70
28931	243	\N	Fontenois-lès-Montbozon	70
28932	244	\N	Fouchécourt	70
28933	245	\N	Fougerolles	70
28934	247	\N	Fouvent-Saint-Andoche	70
28935	248	\N	Frahier-et-Chatebier	70
28936	249	\N	Francalmont	70
28937	250	\N	Franchevelle	70
28938	251	\N	Francourt	70
28939	252	\N	Framont	70
28940	253	\N	Frasne-le-Château	70
28941	254	\N	Frédéric-Fontaine	70
28942	255	\N	Fresne-Saint-Mamès	70
28943	256	\N	Fresse	70
28944	257	\N	Fretigney-et-Velloreille	70
28945	258	\N	Froideconche	70
28946	259	\N	Froideterre	70
28947	260	\N	Frotey-lès-Lure	70
28948	261	\N	Frotey-lès-Vesoul	70
28949	262	\N	Genevreuille	70
28950	263	\N	Genevrey	70
28951	264	\N	Georfans	70
28952	265	\N	Germigney	70
28953	267	\N	Gevigney-et-Mercey	70
28954	268	\N	Gézier-et-Fontenelay	70
28955	269	\N	Girefontaine	70
28956	271	\N	Gouhenans	70
28957	272	\N	Gourgeon	70
28958	273	\N	Grammont	70
28959	274	\N	Grandecourt	70
28960	275	\N	Grandvelle-et-le-Perrenot	70
28961	276	\N	Granges-la-Ville	70
28962	277	\N	Granges-le-Bourg	70
28963	278	\N	Grattery	70
28964	279	\N	Gray	70
28965	280	\N	Gray-la-Ville	70
28966	281	\N	Greucourt	70
28967	282	\N	Gy	70
28968	283	\N	Haut-du-Them-Château-Lambert	70
28969	284	\N	Hautevelle	70
28970	285	\N	Héricourt	70
28971	286	\N	Hugier	70
28972	287	\N	Hurecourt	70
28973	288	\N	Hyet	70
28974	289	\N	Igny	70
28975	290	\N	Jasney	70
28976	291	\N	Jonvelle	70
28977	292	\N	Jussey	70
28978	293	\N	Lambrey	70
28979	294	\N	Lantenot	70
28980	295	\N	Lanterne-et-les-Armonts	70
28981	296	\N	Larians-et-Munans	70
28982	297	\N	Larret	70
28983	298	\N	Lavigney	70
28984	299	\N	Lavoncourt	70
28985	301	\N	Lieffrans	70
28986	302	\N	Lieucourt	70
28987	303	\N	Liévans	70
28988	304	\N	Linexert	70
28989	305	\N	Luilley	70
28990	306	\N	Lomont	70
28991	307	\N	Longevelle	70
28992	308	\N	Longine	70
28993	309	\N	Loulans-Verchamp	70
28994	310	\N	Lure	70
28995	311	\N	Luxeuil-les-Bains	70
28996	312	\N	Luze	70
28997	313	\N	Lyoffans	70
28998	314	\N	Magnivray	70
28999	315	\N	Magnoncourt	70
29000	316	\N	Magnoray	70
29001	317	\N	Magny	70
29002	318	\N	Magny-Danigon	70
29003	319	\N	Magny-Jobert	70
29004	320	\N	Magny-lès-Jussey	70
29005	321	\N	Magny-Vernois	70
29006	322	\N	Mailleroncourt-Charette	70
29007	323	\N	Mailleroncourt-Saint-Pancras	70
29008	324	\N	Mailley-et-Chazelot	70
29009	325	\N	Maizières	70
29010	326	\N	Malachère	70
29011	327	\N	Malans	70
29012	328	\N	Malbouhans	70
29013	329	\N	Malvillers	70
29014	330	\N	Mandrevillars	70
29015	331	\N	Mantoche	70
29016	332	\N	Marast	70
29017	334	\N	Marnay	70
29018	335	\N	Maussans	70
29019	336	\N	Mélecey	70
29020	337	\N	Melin	70
29021	338	\N	Melincourt	70
29022	339	\N	Mélisey	70
29023	340	\N	Membrey	70
29024	341	\N	Menoux	70
29025	342	\N	Mercey-sur-Saône	70
29026	343	\N	Mersuay	70
29027	344	\N	Meurcourt	70
29028	345	\N	Miellin	70
29029	347	\N	Mignavillers	70
29030	348	\N	Moffans-et-Vacheresse	70
29031	349	\N	Moimay	70
29032	350	\N	Molay	70
29033	351	\N	Mollans	70
29034	352	\N	Montagne	70
29035	353	\N	Montagney	70
29036	355	\N	Montarlot-lès-Rioz	70
29037	356	\N	Montboillon	70
29038	357	\N	Montbozon	70
29039	358	\N	Montcey	70
29040	359	\N	Montcourt	70
29041	360	\N	Montdoré	70
29042	361	\N	Montessaux	70
29043	362	\N	Montigny-lès-Cherlieu	70
29044	363	\N	Montigny-lès-Vesoul	70
29045	364	\N	Montjustin-et-Velotte	70
29046	366	\N	Villers-Chemin-et-Mont-lès-Étrelles	70
29047	367	\N	Mont-le-Vernois	70
29048	368	\N	Montot	70
29049	369	\N	Mont-Saint-Léger	70
29050	371	\N	Montureux-et-Prantigny	70
29051	372	\N	Montureux-lès-Baulay	70
29052	373	\N	Roche-Morey	70
29053	374	\N	Motey-Besuche	70
29054	375	\N	Motey-sur-Saône	70
29055	376	\N	Nantilly	70
29056	378	\N	Navenne	70
29057	380	\N	Neurey-en-Vaux	70
29058	381	\N	Neurey-lès-la-Demie	70
29059	383	\N	Neuvelle-lès-Cromary	70
29060	384	\N	Neuvelle-lès-la-Charité	70
29061	385	\N	Neuvelle-lès-Lure	70
29062	386	\N	Neuvelle-lès-Scey	70
29063	387	\N	Noidans-le-Ferroux	70
29064	388	\N	Noidans-lès-Vesoul	70
29065	389	\N	Noiron	70
29066	390	\N	Noroy-le-Bourg	70
29067	392	\N	Oigney	70
29068	393	\N	Oiselay-et-Grachaux	70
29069	394	\N	Onay	70
29070	395	\N	Oppenans	70
29071	396	\N	Oricourt	70
29072	397	\N	Ormenans	70
29073	398	\N	Ormoiche	70
29074	399	\N	Ormoy	70
29075	400	\N	Ouge	70
29076	401	\N	Ovanches	70
29077	402	\N	Oyrières	70
29078	403	\N	Palante	70
29079	404	\N	Passavant-la-Rochère	70
29080	405	\N	Pennesières	70
29081	406	\N	Percey-le-Grand	70
29082	407	\N	Perrouse	70
29083	408	\N	Pesmes	70
29084	409	\N	Pierrecourt	70
29085	410	\N	Pin	70
29086	411	\N	Pisseure	70
29087	412	\N	Plainemont	70
29088	413	\N	Plancher-Bas	70
29089	414	\N	Plancher-les-Mines	70
29090	415	\N	Polaincourt-et-Clairefontaine	70
29091	416	\N	Pomoy	70
29092	417	\N	Pontcey	70
29093	418	\N	Pont-de-Planches	70
29094	419	\N	Pont-du-Bois	70
29095	420	\N	Pont-sur-l'Ognon	70
29096	421	\N	Port-sur-Saône	70
29097	422	\N	Poyans	70
29098	423	\N	Preigney	70
29099	425	\N	Proiselière-et-Langle	70
29100	426	\N	Provenchère	70
29101	427	\N	Purgerot	70
29102	428	\N	Pusey	70
29103	429	\N	Pusy-et-Épenoux	70
29104	430	\N	Quarte	70
29105	431	\N	Quenoche	70
29106	432	\N	Quers	70
29107	433	\N	Quincey	70
29108	435	\N	Raddon-et-Chapendu	70
29109	436	\N	Raincourt	70
29110	437	\N	Ranzevelle	70
29111	438	\N	Ray-sur-Saône	70
29112	439	\N	Raze	70
29113	440	\N	Recologne	70
29114	441	\N	Recologne-lès-Rioz	70
29115	442	\N	Renaucourt	70
29116	443	\N	Grande-Résie	70
29117	444	\N	Résie-Saint-Martin	70
29118	445	\N	Rignovelle	70
29119	446	\N	Rigny	70
29120	447	\N	Rioz	70
29121	448	\N	Roche-et-Raucourt	70
29122	449	\N	Roche-sur-Linotte-et-Sorans-les-Cordiers	70
29123	450	\N	Rochelle	70
29124	451	\N	Ronchamp	70
29125	452	\N	Rosey	70
29126	453	\N	Rosière	70
29127	454	\N	Rosières-sur-Mance	70
29128	455	\N	Roye	70
29129	456	\N	Ruhans	70
29130	457	\N	Rupt-sur-Saône	70
29131	459	\N	Saint-Barthélemy	70
29132	460	\N	Saint-Bresson	70
29133	461	\N	Saint-Broing	70
29134	462	\N	Saint-Ferjeux	70
29135	463	\N	Saint-Gand	70
29136	464	\N	Saint-Germain	70
29137	466	\N	Saint-Loup-Nantouard	70
29138	467	\N	Saint-Loup-sur-Semouse	70
29139	468	\N	Saint-Marcel	70
29140	469	\N	Sainte-Marie-en-Chanois	70
29141	470	\N	Sainte-Marie-en-Chaux	70
29142	471	\N	Sainte-Reine	70
29143	472	\N	Saint-Remy	70
29144	473	\N	Saint-Sauveur	70
29145	474	\N	Saint-Sulpice	70
29146	475	\N	Saint-Valbert	70
29147	476	\N	Saponcourt	70
29148	477	\N	Saulnot	70
29149	478	\N	Saulx	70
29150	479	\N	Sauvigney-lès-Gray	70
29151	480	\N	Sauvigney-lès-Pesmes	70
29152	481	\N	Savoyeux	70
29153	482	\N	Scey-sur-Saône-et-Saint-Albin	70
29154	483	\N	Scye	70
29155	484	\N	Secenans	70
29156	485	\N	Selles	70
29157	486	\N	Semmadon	70
29158	487	\N	Senargent-Mignafans	70
29159	488	\N	Senoncourt	70
29160	489	\N	Servance	70
29161	490	\N	Servigney	70
29162	491	\N	Seveux	70
29163	492	\N	Soing-Cubry-Charentenay	70
29164	493	\N	Sorans-lès-Breurey	70
29165	494	\N	Sornay	70
29166	496	\N	Tartécourt	70
29167	497	\N	Tavey	70
29168	498	\N	Ternuay-Melay-et-Saint-Hilaire	70
29169	499	\N	Theuley	70
29170	500	\N	Thieffrans	70
29171	501	\N	Thiénans	70
29172	502	\N	Tincey-et-Pontrebeau	70
29173	503	\N	Traitiéfontaine	70
29174	504	\N	Traves	70
29175	505	\N	Tremblois	70
29176	506	\N	Trémoins	70
29177	507	\N	Trésilley	70
29178	509	\N	Tromarey	70
29179	510	\N	Vadans	70
29180	511	\N	Vaite	70
29181	512	\N	Vaivre	70
29182	513	\N	Vaivre-et-Montoille	70
29183	514	\N	Valay	70
29184	515	\N	Val-de-Gouhenans	70
29185	516	\N	Vallerois-le-Bois	70
29186	517	\N	Vallerois-Lorioz	70
29187	518	\N	Val-Saint-Éloi	70
29188	519	\N	Vandelans	70
29189	520	\N	Vanne	70
29190	521	\N	Vantoux-et-Longevelle	70
29191	522	\N	Varogne	70
29192	523	\N	Vars	70
29193	524	\N	Vauchoux	70
29194	525	\N	Vauconcourt-Nervezain	70
29195	526	\N	Vauvillers	70
29196	527	\N	Vaux-le-Moncelot	70
29197	528	\N	Velesmes-Échevanne	70
29198	529	\N	Velet	70
29199	530	\N	Vellechevreux-et-Courbenans	70
29200	531	\N	Velleclaire	70
29201	532	\N	Vellefaux	70
29202	533	\N	Vellefrey-et-Vellefrange	70
29203	534	\N	Vellefrie	70
29204	535	\N	Velleguindry-et-Levrecey	70
29205	536	\N	Velle-le-Châtel	70
29206	537	\N	Velleminfroy	70
29207	538	\N	Vellemoz	70
29208	539	\N	Vellexon-Queutrey-et-Vaudey	70
29209	540	\N	Velloreille-lès-Choye	70
29210	541	\N	Velorcey	70
29211	542	\N	Venère	70
29212	544	\N	Vergenne	70
29213	545	\N	Venisey	70
29214	546	\N	Vereux	70
29215	547	\N	Verlans	70
29216	548	\N	Vernois-sur-Mance	70
29217	549	\N	Vernotte	70
29218	550	\N	Vesoul	70
29219	551	\N	Vezet	70
29220	552	\N	Villafans	70
29221	553	\N	Villargent	70
29222	554	\N	Villars-le-Pautel	70
29223	555	\N	Villedieu-en-Fontenette	70
29224	557	\N	Villefrancon	70
29225	558	\N	Villeneuve-Bellenoye-et-la-Maize	70
29226	559	\N	Villeparois	70
29227	560	\N	Villers-Bouton	70
29228	561	\N	Villersexel	70
29229	562	\N	Villers-la-Ville	70
29230	563	\N	Villers-le-Sec	70
29231	564	\N	Villers-lès-Luxeuil	70
29232	565	\N	Villers-Pater	70
29233	566	\N	Villers-sur-Port	70
29234	567	\N	Villers-sur-Saulnot	70
29235	568	\N	Villers-Vaudey	70
29236	569	\N	Vilory	70
29237	571	\N	Visoncourt	70
29238	572	\N	Vitrey-sur-Mance	70
29239	573	\N	Voivre	70
29240	574	\N	Volon	70
29241	575	\N	Voray-sur-l'Ognon	70
29242	576	\N	Vougécourt	70
29243	577	\N	Vouhenans	70
29244	578	\N	Vregille	70
29245	579	\N	Vyans-le-Val	70
29246	580	\N	Vy-le-Ferroux	70
29247	581	\N	Vy-lès-Lure	70
29248	582	\N	Vy-lès-Rupt	70
29249	583	\N	Vy-lès-Filain	70
29250	001	\N	Abergement-de-Cuisery	71
29251	002	\N	Abergement-Sainte-Colombe	71
29252	003	\N	Allerey-sur-Saône	71
29253	004	\N	Allériot	71
29254	005	\N	Aluze	71
29255	006	\N	Amanzé	71
29256	007	\N	Ameugny	71
29257	008	\N	Anglure-sous-Dun	71
29258	009	\N	Anost	71
29259	010	\N	Antully	71
29260	011	\N	Anzy-le-Duc	71
29261	012	\N	Artaix	71
29262	013	\N	Authumes	71
29263	014	\N	Autun	71
29264	015	\N	Auxy	71
29265	016	\N	Azé	71
29266	017	\N	Ballore	71
29267	018	\N	Bantanges	71
29268	019	\N	Barizey	71
29269	020	\N	Barnay	71
29270	021	\N	Baron	71
29271	022	\N	Baudemont	71
29272	023	\N	Baudrières	71
29273	024	\N	Baugy	71
29274	025	\N	Beaubery	71
29275	026	\N	Beaumont-sur-Grosne	71
29276	027	\N	Beaurepaire-en-Bresse	71
29277	028	\N	Beauvernois	71
29278	029	\N	Bellevesvre	71
29279	030	\N	Bergesserin	71
29280	031	\N	Berzé-le-Châtel	71
29281	032	\N	Berzé-la-Ville	71
29282	033	\N	Bey	71
29283	034	\N	Bissey-sous-Cruchaud	71
29284	035	\N	Bissy-la-Mâconnaise	71
29285	036	\N	Bissy-sous-Uxelles	71
29286	037	\N	Bissy-sur-Fley	71
29287	038	\N	Bizots	71
29288	039	\N	Blanot	71
29289	040	\N	Blanzy	71
29290	041	\N	Bois-Sainte-Marie	71
29291	042	\N	Bonnay	71
29292	043	\N	Bordes	71
29293	044	\N	Bosjean	71
29294	045	\N	Bouhans	71
29295	046	\N	Boulaye	71
29296	047	\N	Bourbon-Lancy	71
29297	048	\N	Bourg-le-Comte	71
29298	050	\N	Bourgvilain	71
29299	051	\N	Bouzeron	71
29300	052	\N	Boyer	71
29301	054	\N	Bragny-sur-Saône	71
29302	055	\N	Brandon	71
29303	056	\N	Branges	71
29304	057	\N	Bray	71
29305	058	\N	Bresse-sur-Grosne	71
29306	059	\N	Breuil	71
29307	060	\N	Briant	71
29308	061	\N	Brienne	71
29309	062	\N	Brion	71
29310	063	\N	Broye	71
29311	064	\N	Bruailles	71
29312	065	\N	Buffières	71
29313	066	\N	Burgy	71
29314	067	\N	Burnand	71
29315	068	\N	Burzy	71
29316	069	\N	Bussières	71
29317	070	\N	Buxy	71
29318	071	\N	Céron	71
29319	072	\N	Cersot	71
29320	073	\N	Chagny	71
29321	074	\N	Chaintré	71
29322	075	\N	Chalmoux	71
29323	076	\N	Chalon-sur-Saône	71
29324	077	\N	Chambilly	71
29325	078	\N	Chamilly	71
29326	079	\N	Champagnat	71
29327	080	\N	Champagny-sous-Uxelles	71
29328	081	\N	Champforgeuil	71
29329	082	\N	Champlecy	71
29330	084	\N	Chânes	71
29331	085	\N	Change	71
29332	086	\N	Changy	71
29333	087	\N	Chapaize	71
29334	088	\N	Chapelle-au-Mans	71
29335	089	\N	Chapelle-de-Bragny	71
29336	090	\N	Chapelle-de-Guinchay	71
29337	091	\N	Chapelle-du-Mont-de-France	71
29338	092	\N	Chapelle-Naude	71
29339	093	\N	Chapelle-Saint-Sauveur	71
29340	094	\N	Chapelle-sous-Brancion	71
29341	095	\N	Chapelle-sous-Dun	71
29342	096	\N	Chapelle-sous-Uchon	71
29343	097	\N	Chapelle-Thècle	71
29344	098	\N	Charbonnat	71
29345	099	\N	Charbonnières	71
29346	100	\N	Chardonnay	71
29347	101	\N	Charette-Varennes	71
29348	102	\N	Charmée	71
29349	103	\N	Charmoy	71
29350	104	\N	Charnay-lès-Chalon	71
29351	105	\N	Charnay-lès-Mâcon	71
29352	106	\N	Charolles	71
29353	107	\N	Charrecey	71
29354	108	\N	Chasselas	71
29355	109	\N	Chassey-le-Camp	71
29356	110	\N	Chassigny-sous-Dun	71
29357	111	\N	Chassy	71
29358	112	\N	Château	71
29359	113	\N	Châteauneuf	71
29360	115	\N	Châtel-Moron	71
29361	116	\N	Châtenay	71
29362	117	\N	Châtenoy-en-Bresse	71
29363	118	\N	Châtenoy-le-Royal	71
29364	119	\N	Chaudenay	71
29365	120	\N	Chauffailles	71
29366	121	\N	Chaux	71
29367	122	\N	Cheilly-lès-Maranges	71
29368	123	\N	Chenay-le-Châtel	71
29369	124	\N	Chenôves	71
29370	125	\N	Chérizet	71
29371	126	\N	Chevagny-les-Chevrières	71
29372	127	\N	Chevagny-sur-Guye	71
29373	128	\N	Chiddes	71
29374	129	\N	Chissey-en-Morvan	71
29375	130	\N	Chissey-lès-Mâcon	71
29376	131	\N	Ciel	71
29377	132	\N	Ciry-le-Noble	71
29378	133	\N	Clayette	71
29379	134	\N	Clermain	71
29380	135	\N	Clessé	71
29381	136	\N	Clessy	71
29382	137	\N	Cluny	71
29383	138	\N	Clux	71
29384	139	\N	Collonge-en-Charollais	71
29385	140	\N	Collonge-la-Madeleine	71
29386	141	\N	Colombier-en-Brionnais	71
29387	142	\N	Comelle	71
29388	143	\N	Condal	71
29389	144	\N	Cordesse	71
29390	145	\N	Cormatin	71
29391	146	\N	Cortambert	71
29392	147	\N	Cortevaix	71
29393	148	\N	Coublanc	71
29394	149	\N	Couches	71
29395	150	\N	Crêches-sur-Saône	71
29396	151	\N	Créot	71
29397	152	\N	Cressy-sur-Somme	71
29398	153	\N	Creusot	71
29399	154	\N	Crissey	71
29400	155	\N	Cronat	71
29401	156	\N	Cruzille	71
29402	157	\N	Cuiseaux	71
29403	158	\N	Cuisery	71
29404	159	\N	Culles-les-Roches	71
29405	160	\N	Curbigny	71
29406	161	\N	Curdin	71
29407	162	\N	Curgy	71
29408	163	\N	Curtil-sous-Buffières	71
29409	164	\N	Curtil-sous-Burnand	71
29410	165	\N	Cussy-en-Morvan	71
29411	166	\N	Cuzy	71
29412	167	\N	Damerey	71
29413	168	\N	Dampierre-en-Bresse	71
29414	169	\N	Davayé	71
29415	170	\N	Demigny	71
29416	171	\N	Dennevy	71
29417	172	\N	Dettey	71
29418	173	\N	Devrouze	71
29419	174	\N	Dezize-lès-Maranges	71
29420	175	\N	Diconne	71
29421	176	\N	Digoin	71
29422	177	\N	Dommartin-lès-Cuiseaux	71
29423	178	\N	Dompierre-les-Ormes	71
29424	179	\N	Dompierre-sous-Sanvignes	71
29425	180	\N	Donzy-le-National	71
29426	181	\N	Donzy-le-Pertuis	71
29427	182	\N	Dracy-le-Fort	71
29428	183	\N	Dracy-lès-Couches	71
29429	184	\N	Dracy-Saint-Loup	71
29430	185	\N	Dyo	71
29431	186	\N	Écuelles	71
29432	187	\N	Écuisses	71
29433	188	\N	Épertully	71
29434	189	\N	Épervans	71
29435	190	\N	Épinac	71
29436	191	\N	Essertenne	71
29437	192	\N	Étang-sur-Arroux	71
29438	193	\N	Étrigny	71
29439	194	\N	Farges-lès-Chalon	71
29440	195	\N	Farges-lès-Mâcon	71
29441	196	\N	Fay	71
29442	198	\N	Flacey-en-Bresse	71
29443	199	\N	Flagy	71
29444	200	\N	Fleury-la-Montagne	71
29445	201	\N	Fley	71
29446	202	\N	Fontaines	71
29447	203	\N	Fontenay	71
29448	204	\N	Fragnes	71
29449	205	\N	Frangy-en-Bresse	71
29450	206	\N	Frette	71
29451	207	\N	Fretterans	71
29452	208	\N	Frontenard	71
29453	209	\N	Frontenaud	71
29454	210	\N	Fuissé	71
29455	212	\N	Génelard	71
29456	213	\N	Genête	71
29457	214	\N	Genouilly	71
29458	215	\N	Gergy	71
29459	216	\N	Germagny	71
29460	217	\N	Germolles-sur-Grosne	71
29461	218	\N	Gibles	71
29462	219	\N	Gigny-sur-Saône	71
29463	220	\N	Gilly-sur-Loire	71
29464	221	\N	Givry	71
29465	222	\N	Gourdon	71
29466	223	\N	Grande-Verrière	71
29467	224	\N	Grandvaux	71
29468	225	\N	Granges	71
29469	226	\N	Grevilly	71
29470	227	\N	Grury	71
29471	228	\N	Guerfand	71
29472	229	\N	Guerreaux	71
29473	230	\N	Gueugnon	71
29474	231	\N	Guiche	71
29475	232	\N	Hautefond	71
29476	233	\N	Hôpital-le-Mercier	71
29477	234	\N	Huilly-sur-Seille	71
29478	235	\N	Hurigny	71
29479	236	\N	Igé	71
29480	237	\N	Igornay	71
29481	238	\N	Iguerande	71
29482	239	\N	Issy-l'Évêque	71
29483	240	\N	Jalogny	71
29484	241	\N	Jambles	71
29485	242	\N	Joncy	71
29486	243	\N	Joudes	71
29487	244	\N	Jouvençon	71
29488	245	\N	Jugy	71
29489	246	\N	Juif	71
29490	247	\N	Jully-lès-Buxy	71
29491	248	\N	Lacrost	71
29492	249	\N	Laives	71
29493	250	\N	Laizé	71
29494	251	\N	Laizy	71
29495	252	\N	Lalheue	71
29496	253	\N	Lans	71
29497	254	\N	Lays-sur-le-Doubs	71
29498	255	\N	Lesme	71
29499	256	\N	Lessard-en-Bresse	71
29500	257	\N	Lessard-le-National	71
29501	258	\N	Leynes	71
29502	259	\N	Ligny-en-Brionnais	71
29503	261	\N	Loisy	71
29504	262	\N	Longepierre	71
29505	263	\N	Louhans	71
29506	264	\N	Lournand	71
29507	265	\N	Loyère	71
29508	266	\N	Lucenay-l'Évêque	71
29509	267	\N	Lugny	71
29510	268	\N	Lugny-lès-Charolles	71
29511	269	\N	Lux	71
29512	270	\N	Mâcon	71
29513	271	\N	Mailly	71
29514	272	\N	Malay	71
29515	273	\N	Maltat	71
29516	274	\N	Mancey	71
29517	275	\N	Marcigny	71
29518	276	\N	Marcilly-la-Gueurce	71
29519	277	\N	Marcilly-lès-Buxy	71
29520	278	\N	Marigny	71
29521	279	\N	Marizy	71
29522	280	\N	Marly-sous-Issy	71
29523	281	\N	Marly-sur-Arroux	71
29524	282	\N	Marmagne	71
29525	283	\N	Marnay	71
29526	284	\N	Martailly-lès-Brancion	71
29527	285	\N	Martigny-le-Comte	71
29528	286	\N	Mary	71
29529	287	\N	Massilly	71
29530	288	\N	Massy	71
29531	289	\N	Matour	71
29532	290	\N	Mazille	71
29533	291	\N	Melay	71
29534	292	\N	Mellecey	71
29535	293	\N	Ménetreuil	71
29536	294	\N	Mercurey	71
29537	295	\N	Mervans	71
29538	296	\N	Messey-sur-Grosne	71
29539	297	\N	Mesvres	71
29540	299	\N	Milly-Lamartine	71
29541	300	\N	Miroir	71
29542	301	\N	Mont	71
29543	302	\N	Montagny-lès-Buxy	71
29544	303	\N	Montagny-près-Louhans	71
29545	304	\N	Montagny-sur-Grosne	71
29546	305	\N	Montbellet	71
29547	306	\N	Montceau-les-Mines	71
29548	307	\N	Montceaux-l'Étoile	71
29549	308	\N	Montceaux-Ragny	71
29550	309	\N	Montcenis	71
29551	310	\N	Montchanin	71
29552	311	\N	Montcony	71
29553	312	\N	Montcoy	71
29554	313	\N	Monthelon	71
29555	314	\N	Montjay	71
29556	315	\N	Mont-lès-Seurre	71
29557	316	\N	Montmelard	71
29558	317	\N	Montmort	71
29559	318	\N	Montpont-en-Bresse	71
29560	319	\N	Montret	71
29561	320	\N	Mont-Saint-Vincent	71
29562	321	\N	Morey	71
29563	322	\N	Morlet	71
29564	323	\N	Mornay	71
29565	324	\N	Moroges	71
29566	325	\N	Motte-Saint-Jean	71
29567	326	\N	Mouthier-en-Bresse	71
29568	327	\N	Mussy-sous-Dun	71
29569	328	\N	Nanton	71
29570	329	\N	Navilly	71
29571	330	\N	Neuvy-Grandchamp	71
29572	331	\N	Nochize	71
29573	332	\N	Ormes	71
29574	333	\N	Oslon	71
29575	334	\N	Oudry	71
29576	335	\N	Ouroux-sous-le-Bois-Sainte-Marie	71
29577	336	\N	Ouroux-sur-Saône	71
29578	337	\N	Oyé	71
29579	338	\N	Ozenay	71
29580	339	\N	Ozolles	71
29581	340	\N	Palinges	71
29582	341	\N	Palleau	71
29583	342	\N	Paray-le-Monial	71
29584	343	\N	Paris-l'Hôpital	71
29585	344	\N	Passy	71
29586	345	\N	Péronne	71
29587	346	\N	Perrecy-les-Forges	71
29588	347	\N	Perreuil	71
29589	348	\N	Perrigny-sur-Loire	71
29590	349	\N	Petite-Verrière	71
29591	350	\N	Pierreclos	71
29592	351	\N	Pierre-de-Bresse	71
29593	352	\N	Planois	71
29594	353	\N	Plottes	71
29595	354	\N	Poisson	71
29596	355	\N	Pontoux	71
29597	356	\N	Pouilloux	71
29598	357	\N	Pourlans	71
29599	358	\N	Pressy-sous-Dondin	71
29600	359	\N	Préty	71
29601	360	\N	Prissé	71
29602	361	\N	Prizy	71
29603	362	\N	Pruzilly	71
29604	363	\N	Puley	71
29605	364	\N	Racineuse	71
29606	365	\N	Rancy	71
29607	366	\N	Ratenelle	71
29608	367	\N	Ratte	71
29609	368	\N	Reclesne	71
29610	369	\N	Remigny	71
29611	370	\N	Rigny-sur-Arroux	71
29612	371	\N	Roche-Vineuse	71
29613	372	\N	Romanèche-Thorins	71
29614	373	\N	Romenay	71
29615	374	\N	Rosey	71
29616	375	\N	Rousset	71
29617	376	\N	Roussillon-en-Morvan	71
29618	377	\N	Royer	71
29619	378	\N	Rully	71
29620	379	\N	Sagy	71
29621	380	\N	Saillenard	71
29622	381	\N	Sailly	71
29623	382	\N	Saint-Agnan	71
29624	383	\N	Saint-Albain	71
29625	384	\N	Saint-Ambreuil	71
29626	385	\N	Saint-Amour-Bellevue	71
29627	386	\N	Saint-André-en-Bresse	71
29628	387	\N	Saint-André-le-Désert	71
29629	388	\N	Saint-Aubin-en-Charollais	71
29630	389	\N	Saint-Aubin-sur-Loire	71
29631	390	\N	Saint-Berain-sous-Sanvignes	71
29632	391	\N	Saint-Bérain-sur-Dheune	71
29633	392	\N	Saint-Boil	71
29634	393	\N	Saint-Bonnet-de-Cray	71
29635	394	\N	Saint-Bonnet-de-Joux	71
29636	395	\N	Saint-Bonnet-de-Vieille-Vigne	71
29637	396	\N	Saint-Bonnet-en-Bresse	71
29638	397	\N	Sainte-Cécile	71
29639	398	\N	Saint-Christophe-en-Bresse	71
29640	399	\N	Saint-Christophe-en-Brionnais	71
29641	400	\N	Saint-Clément-sur-Guye	71
29642	401	\N	Sainte-Croix	71
29643	402	\N	Saint-Cyr	71
29644	403	\N	Saint-Denis-de-Vaux	71
29645	404	\N	Saint-Désert	71
29646	405	\N	Saint-Didier-en-Bresse	71
29647	406	\N	Saint-Didier-en-Brionnais	71
29648	407	\N	Saint-Didier-sur-Arroux	71
29649	408	\N	Saint-Edmond	71
29650	409	\N	Saint-Émiland	71
29651	410	\N	Saint-Étienne-en-Bresse	71
29652	411	\N	Saint-Eugène	71
29653	412	\N	Saint-Eusèbe	71
29654	413	\N	Saint-Firmin	71
29655	414	\N	Saint-Forgeot	71
29656	415	\N	Sainte-Foy	71
29657	416	\N	Saint-Gengoux-de-Scissé	71
29658	417	\N	Saint-Gengoux-le-National	71
29659	419	\N	Saint-Germain-du-Bois	71
29660	420	\N	Saint-Germain-du-Plain	71
29661	421	\N	Saint-Germain-en-Brionnais	71
29662	422	\N	Saint-Germain-lès-Buxy	71
29663	423	\N	Saint-Gervais-en-Vallière	71
29664	424	\N	Saint-Gervais-sur-Couches	71
29665	425	\N	Saint-Gilles	71
29666	426	\N	Sainte-Hélène	71
29667	427	\N	Saint-Huruge	71
29668	428	\N	Saint-Igny-de-Roche	71
29669	430	\N	Saint-Jean-de-Vaux	71
29670	431	\N	Saint-Jean-de-Trézy	71
29671	433	\N	Saint-Julien-de-Civry	71
29672	434	\N	Saint-Julien-de-Jonzy	71
29673	435	\N	Saint-Julien-sur-Dheune	71
29674	436	\N	Saint-Laurent-d'Andenay	71
29675	437	\N	Saint-Laurent-en-Brionnais	71
29676	438	\N	Saint-Léger-du-Bois	71
29677	439	\N	Saint-Léger-lès-Paray	71
29678	440	\N	Saint-Léger-sous-Beuvray	71
29679	441	\N	Saint-Léger-sous-la-Bussière	71
29680	442	\N	Saint-Léger-sur-Dheune	71
29681	443	\N	Saint-Loup-Géanges	71
29682	444	\N	Saint-Loup-de-Varennes	71
29683	445	\N	Saint-Marcel	71
29684	446	\N	Saint-Marcelin-de-Cray	71
29685	447	\N	Saint-Mard-de-Vaux	71
29686	448	\N	Saint-Martin-Belle-Roche	71
29687	449	\N	Saint-Martin-d'Auxy	71
29688	450	\N	Saint-Martin-de-Commune	71
29689	451	\N	Saint-Martin-de-Lixy	71
29690	452	\N	Saint-Martin-de-Salencey	71
29691	453	\N	Saint-Martin-du-Lac	71
29692	454	\N	Saint-Martin-du-Mont	71
29693	455	\N	Saint-Martin-du-Tartre	71
29694	456	\N	Saint-Martin-en-Bresse	71
29695	457	\N	Saint-Martin-en-Gâtinois	71
29696	458	\N	Saint-Martin-la-Patrouille	71
29697	459	\N	Saint-Martin-sous-Montaigu	71
29698	460	\N	Saint-Maurice-de-Satonnay	71
29699	461	\N	Saint-Maurice-des-Champs	71
29700	462	\N	Saint-Maurice-en-Rivière	71
29701	463	\N	Saint-Maurice-lès-Châteauneuf	71
29702	464	\N	Saint-Maurice-lès-Couches	71
29703	465	\N	Saint-Micaud	71
29704	466	\N	Saint-Nizier-sur-Arroux	71
29705	468	\N	Saint-Pierre-de-Varennes	71
29706	469	\N	Saint-Pierre-le-Vieux	71
29707	470	\N	Saint-Point	71
29708	471	\N	Saint-Privé	71
29709	472	\N	Saint-Prix	71
29710	473	\N	Saint-Racho	71
29711	474	\N	Sainte-Radegonde	71
29712	475	\N	Saint-Rémy	71
29713	477	\N	Saint-Romain-sous-Gourdon	71
29714	478	\N	Saint-Romain-sous-Versigny	71
29715	479	\N	Saint-Sernin-du-Bois	71
29716	480	\N	Saint-Sernin-du-Plain	71
29717	481	\N	Saint-Symphorien-d'Ancelles	71
29718	482	\N	Saint-Symphorien-de-Marmagne	71
29719	483	\N	Saint-Symphorien-des-Bois	71
29720	484	\N	Saint-Usuge	71
29721	485	\N	Saint-Vallerin	71
29722	486	\N	Saint-Vallier	71
29723	487	\N	Saint-Vérand	71
29724	488	\N	Saint-Vincent-des-Prés	71
29725	489	\N	Saint-Vincent-en-Bresse	71
29726	490	\N	Saint-Vincent-Bragny	71
29727	491	\N	Saint-Yan	71
29728	492	\N	Saint-Ythaire	71
29729	493	\N	Saisy	71
29730	494	\N	Salle	71
29731	495	\N	Salornay-sur-Guye	71
29732	496	\N	Sampigny-lès-Maranges	71
29733	497	\N	Sancé	71
29734	498	\N	Santilly	71
29735	499	\N	Sanvignes-les-Mines	71
29736	500	\N	Sarry	71
29737	501	\N	Sassangy	71
29738	502	\N	Sassenay	71
29739	503	\N	Saules	71
29740	504	\N	Saunières	71
29741	505	\N	Savianges	71
29742	506	\N	Savigny-en-Revermont	71
29743	507	\N	Savigny-sur-Grosne	71
29744	508	\N	Savigny-sur-Seille	71
29745	509	\N	Celle-en-Morvan	71
29746	510	\N	Semur-en-Brionnais	71
29747	512	\N	Sennecey-le-Grand	71
29748	513	\N	Senozan	71
29749	514	\N	Sens-sur-Seille	71
29750	515	\N	Sercy	71
29751	516	\N	Serley	71
29752	517	\N	Sermesse	71
29753	518	\N	Serrières	71
29754	519	\N	Serrigny-en-Bresse	71
29755	520	\N	Sevrey	71
29756	521	\N	Sigy-le-Châtel	71
29757	522	\N	Simandre	71
29758	523	\N	Simard	71
29759	524	\N	Sivignon	71
29760	525	\N	Sologny	71
29761	526	\N	Solutré-Pouilly	71
29762	527	\N	Sommant	71
29763	528	\N	Sornay	71
29764	529	\N	Suin	71
29765	530	\N	Sully	71
29766	531	\N	Tagnière	71
29767	532	\N	Taizé	71
29768	533	\N	Tancon	71
29769	534	\N	Tartre	71
29770	535	\N	Tavernay	71
29771	537	\N	Thil-sur-Arroux	71
29772	538	\N	Thurey	71
29773	539	\N	Tintry	71
29774	540	\N	Torcy	71
29775	541	\N	Torpes	71
29776	542	\N	Toulon-sur-Arroux	71
29777	543	\N	Tournus	71
29778	544	\N	Toutenant	71
29779	545	\N	Tramayes	71
29780	546	\N	Trambly	71
29781	547	\N	Trivy	71
29782	548	\N	Tronchy	71
29783	549	\N	Truchère	71
29784	550	\N	Uchizy	71
29785	551	\N	Uchon	71
29786	552	\N	Uxeau	71
29787	553	\N	Vareilles	71
29788	554	\N	Varenne-l'Arconce	71
29789	555	\N	Varennes-le-Grand	71
29790	556	\N	Varennes-lès-Mâcon	71
29791	557	\N	Varenne-Saint-Germain	71
29792	558	\N	Varennes-Saint-Sauveur	71
29793	559	\N	Varennes-sous-Dun	71
29794	561	\N	Vauban	71
29795	562	\N	Vaudebarrier	71
29796	563	\N	Vaux-en-Pré	71
29797	564	\N	Vendenesse-lès-Charolles	71
29798	565	\N	Vendenesse-sur-Arroux	71
29799	566	\N	Verdun-sur-le-Doubs	71
29800	567	\N	Vergisson	71
29801	568	\N	Vérissey	71
29802	570	\N	Verjux	71
29803	571	\N	Verosvres	71
29804	572	\N	Vers	71
29805	573	\N	Versaugues	71
29806	574	\N	Verzé	71
29807	576	\N	Villars	71
29808	577	\N	Villegaudin	71
29809	578	\N	Villeneuve	71
29810	579	\N	Villeneuve-en-Montagne	71
29811	580	\N	Vincelles	71
29812	581	\N	Vindecy	71
29813	582	\N	Vineuse	71
29814	583	\N	Vinzelles	71
29815	584	\N	Viré	71
29816	585	\N	Virey-le-Grand	71
29817	586	\N	Viry	71
29818	587	\N	Vitry-lès-Cluny	71
29819	588	\N	Vitry-en-Charollais	71
29820	589	\N	Vitry-sur-Loire	71
29821	590	\N	Volesvres	71
29822	591	\N	Fleurville	71
29823	001	\N	Aigné	72
29824	002	\N	Aillières-Beauvoir	72
29825	003	\N	Allonnes	72
29826	004	\N	Amné	72
29827	005	\N	Ancinnes	72
29828	006	\N	Arçonnay	72
29829	007	\N	Ardenay-sur-Mérize	72
29830	008	\N	Arnage	72
29831	009	\N	Arthezé	72
29832	010	\N	Asnières-sur-Vègre	72
29833	011	\N	Assé-le-Boisne	72
29834	012	\N	Assé-le-Riboul	72
29835	013	\N	Aubigné-Racan	72
29836	015	\N	Aulneaux	72
29837	016	\N	Auvers-le-Hamon	72
29838	017	\N	Auvers-sous-Montfaucon	72
29839	018	\N	Avesnes-en-Saosnois	72
29840	019	\N	Avessé	72
29841	020	\N	Avezé	72
29842	021	\N	Avoise	72
29843	022	\N	Bailleul	72
29844	023	\N	Ballon	72
29845	024	\N	Bazoge	72
29846	025	\N	Bazouges-sur-le-Loir	72
29847	026	\N	Beaufay	72
29848	027	\N	Beaumont-sur-Dême	72
29849	028	\N	Beaumont-Pied-de-Buf	72
29850	029	\N	Beaumont-sur-Sarthe	72
29851	031	\N	Beillé	72
29852	032	\N	Berfay	72
29853	033	\N	Bernay	72
29854	034	\N	Bérus	72
29855	035	\N	Bessé-sur-Braye	72
29856	036	\N	Béthon	72
29857	037	\N	Blèves	72
29858	038	\N	Boëssé-le-Sec	72
29859	039	\N	Bonnétable	72
29860	040	\N	Bosse	72
29861	041	\N	Bouër	72
29862	042	\N	Bouloire	72
29863	043	\N	Bourg-le-Roi	72
29864	044	\N	Bousse	72
29865	045	\N	Brains-sur-Gée	72
29866	046	\N	Breil-sur-Mérize	72
29867	047	\N	Brette-les-Pins	72
29868	048	\N	Briosne-lès-Sables	72
29869	049	\N	Bruère-sur-Loir	72
29870	050	\N	Brûlon	72
29871	051	\N	Cérans-Foulletourte	72
29872	052	\N	Chahaignes	72
29873	053	\N	Challes	72
29874	054	\N	Champagné	72
29875	056	\N	Champfleur	72
29876	057	\N	Champrond	72
29877	058	\N	Changé	72
29878	059	\N	Chantenay-Villedieu	72
29879	060	\N	Chapelle-aux-Choux	72
29880	061	\N	Chapelle-d'Aligné	72
29881	062	\N	Chapelle-du-Bois	72
29882	063	\N	Chapelle-Gaugain	72
29883	064	\N	Chapelle-Huon	72
29884	065	\N	Chapelle-Saint-Aubin	72
29885	066	\N	Chapelle-Saint-Fray	72
29886	067	\N	Chapelle-Saint-Rémy	72
29887	068	\N	Chartre-sur-le-Loir	72
29888	069	\N	Chassé	72
29889	070	\N	Chassillé	72
29890	071	\N	Château-du-Loir	72
29891	072	\N	Château-l'Hermitage	72
29892	073	\N	Chaufour-Notre-Dame	72
29893	074	\N	Chemiré-en-Charnie	72
29894	075	\N	Chemiré-le-Gaudin	72
29895	076	\N	Chenay	72
29896	077	\N	Chenu	72
29897	078	\N	Chérancé	72
29898	079	\N	Chérisay	72
29899	080	\N	Cherré	72
29900	081	\N	Cherreau	72
29901	082	\N	Chevain	72
29902	083	\N	Chevillé	72
29903	084	\N	Clermont-Créans	72
29904	085	\N	Cogners	72
29905	086	\N	Commerveil	72
29906	087	\N	Conflans-sur-Anille	72
29907	088	\N	Congé-sur-Orne	72
29908	089	\N	Conlie	72
29909	090	\N	Connerré	72
29910	091	\N	Contilly	72
29911	093	\N	Cormes	72
29912	094	\N	Coudrecieux	72
29913	095	\N	Coulaines	72
29914	096	\N	Coulans-sur-Gée	72
29915	097	\N	Coulombiers	72
29916	098	\N	Coulongé	72
29917	099	\N	Courcebufs	72
29918	100	\N	Courcelles-la-Forêt	72
29919	101	\N	Courcemont	72
29920	102	\N	Courcival	72
29921	103	\N	Courdemanche	72
29922	104	\N	Courgains	72
29923	105	\N	Courgenard	72
29924	106	\N	Courtillers	72
29925	107	\N	Crannes-en-Champagne	72
29926	108	\N	Cré	72
29927	109	\N	Crissé	72
29928	110	\N	Crosmières	72
29929	111	\N	Cures	72
29930	112	\N	Dangeul	72
29931	113	\N	Degré	72
29932	114	\N	Dehault	72
29933	115	\N	Dissay-sous-Courcillon	72
29934	116	\N	Dissé-sous-Ballon	72
29935	117	\N	Dissé-sous-le-Lude	72
29936	118	\N	Dollon	72
29937	119	\N	Domfront-en-Champagne	72
29938	120	\N	Doucelles	72
29939	121	\N	Douillet	72
29940	122	\N	Duneau	72
29941	123	\N	Dureil	72
29942	124	\N	Écommoy	72
29943	125	\N	Écorpain	72
29944	126	\N	Épineu-le-Chevreuil	72
29945	127	\N	Étival-lès-le-Mans	72
29946	128	\N	Évaillé	72
29947	129	\N	Fatines	72
29948	130	\N	Fay	72
29949	131	\N	Fercé-sur-Sarthe	72
29950	132	\N	Ferté-Bernard	72
29951	133	\N	Fillé	72
29952	134	\N	Flée	72
29953	135	\N	Fontaine-Saint-Martin	72
29954	136	\N	Fontenay-sur-Vègre	72
29955	137	\N	Fresnaye-sur-Chédouet	72
29956	138	\N	Fresnay-sur-Sarthe	72
29957	139	\N	Fyé	72
29958	141	\N	Gesnes-le-Gandelin	72
29959	142	\N	Grandchamp	72
29960	143	\N	Grand-Lucé	72
29961	144	\N	Gréez-sur-Roc	72
29962	145	\N	Grez	72
29963	146	\N	Guécélard	72
29964	147	\N	Guierche	72
29965	148	\N	Jauzé	72
29966	149	\N	Joué-en-Charnie	72
29967	150	\N	Joué-l'Abbé	72
29968	151	\N	Juigné-sur-Sarthe	72
29969	152	\N	Juillé	72
29970	153	\N	Jupilles	72
29971	154	\N	Flèche	72
29972	155	\N	Laigné-en-Belin	72
29973	156	\N	Lamnay	72
29974	157	\N	Lavardin	72
29975	158	\N	Lavaré	72
29976	159	\N	Lavenay	72
29977	160	\N	Lavernat	72
29978	161	\N	Lhomme	72
29979	162	\N	Lignières-la-Carelle	72
29980	163	\N	Ligron	72
29981	164	\N	Livet-en-Saosnois	72
29982	165	\N	Lombron	72
29983	166	\N	Longnes	72
29984	167	\N	Louailles	72
29985	168	\N	Loué	72
29986	169	\N	Louplande	72
29987	170	\N	Louvigny	72
29988	171	\N	Louzes	72
29989	172	\N	Luart	72
29990	173	\N	Luceau	72
29991	174	\N	Lucé-sous-Ballon	72
29992	175	\N	Luché-Pringé	72
29993	176	\N	Lude	72
29994	177	\N	Maigné	72
29995	178	\N	Maisoncelles	72
29996	179	\N	Malicorne-sur-Sarthe	72
29997	180	\N	Mamers	72
29998	181	\N	Mans	72
29999	182	\N	Mansigné	72
30000	183	\N	Marçon	72
30001	184	\N	Mareil-en-Champagne	72
30002	185	\N	Mareil-sur-Loir	72
30003	186	\N	Maresché	72
30004	187	\N	Marigné-Laillé	72
30005	188	\N	Marollette	72
30006	189	\N	Marolles-les-Braults	72
30007	190	\N	Marolles-lès-Saint-Calais	72
30008	191	\N	Mayet	72
30009	192	\N	Mées	72
30010	193	\N	Melleray	72
30011	194	\N	Meurcé	72
30012	195	\N	Mézeray	72
30013	196	\N	Mézières-sur-Ponthouin	72
30014	197	\N	Mézières-sous-Lavardin	72
30015	198	\N	Milesse	72
30016	199	\N	Moitron-sur-Sarthe	72
30017	200	\N	Moncé-en-Belin	72
30018	201	\N	Moncé-en-Saosnois	72
30019	202	\N	Monhoudou	72
30020	203	\N	Montabon	72
30021	204	\N	Montaillé	72
30022	205	\N	Montbizot	72
30023	207	\N	Montigny	72
30024	208	\N	Montmirail	72
30025	209	\N	Montreuil-le-Chétif	72
30026	210	\N	Montreuil-le-Henri	72
30027	211	\N	Mont-Saint-Jean	72
30028	212	\N	Moulins-le-Carbonnel	72
30029	213	\N	Mulsanne	72
30030	214	\N	Nauvay	72
30031	215	\N	Neufchâtel-en-Saosnois	72
30032	216	\N	Neuvillalais	72
30033	217	\N	Neuville-sur-Sarthe	72
30034	218	\N	Neuvillette-en-Charnie	72
30035	219	\N	Neuvy-en-Champagne	72
30036	220	\N	Nogent-le-Bernard	72
30037	221	\N	Nogent-sur-Loir	72
30038	222	\N	Nouans	72
30039	223	\N	Noyen-sur-Sarthe	72
30040	224	\N	Nuillé-le-Jalais	72
30041	225	\N	Oisseau-le-Petit	72
30042	226	\N	Oizé	72
30043	227	\N	Panon	72
30044	228	\N	Parcé-sur-Sarthe	72
30045	229	\N	Parennes	72
30046	230	\N	Parigné-le-Pôlin	72
30047	231	\N	Parigné-l'Évêque	72
30048	232	\N	Notre-Dame-du-Pé	72
30049	233	\N	Peray	72
30050	234	\N	Pezé-le-Robert	72
30051	235	\N	Piacé	72
30052	236	\N	Pincé	72
30053	237	\N	Pirmil	72
30054	238	\N	Pizieux	72
30055	239	\N	Poillé-sur-Vègre	72
30056	240	\N	Poncé-sur-le-Loir	72
30057	241	\N	Montfort-le-Gesnois	72
30058	243	\N	Pontvallain	72
30059	244	\N	Précigné	72
30060	245	\N	Préval	72
30061	246	\N	Prévelles	72
30062	247	\N	Pruillé-le-Chétif	72
30063	248	\N	Pruillé-l'Éguillé	72
30064	249	\N	Quinte	72
30065	250	\N	Rahay	72
30066	251	\N	René	72
30067	252	\N	Requeil	72
30068	253	\N	Roézé-sur-Sarthe	72
30069	254	\N	Rouessé-Fontaine	72
30070	255	\N	Rouessé-Vassé	72
30071	256	\N	Rouez	72
30072	257	\N	Rouillon	72
30073	258	\N	Roullée	72
30074	259	\N	Rouperroux-le-Coquet	72
30075	260	\N	Ruaudin	72
30076	261	\N	Ruillé-en-Champagne	72
30077	262	\N	Ruillé-sur-Loir	72
30078	264	\N	Sablé-sur-Sarthe	72
30079	265	\N	Saint-Aignan	72
30080	266	\N	Saint-Aubin-de-Locquenay	72
30081	267	\N	Saint-Aubin-des-Coudrais	72
30082	268	\N	Saint-Biez-en-Belin	72
30083	269	\N	Saint-Calais	72
30084	270	\N	Saint-Calez-en-Saosnois	72
30085	271	\N	Saint-Célerin	72
30086	272	\N	Sainte-Cérotte	72
30087	273	\N	Saint-Christophe-du-Jambet	72
30088	274	\N	Saint-Christophe-en-Champagne	72
30089	275	\N	Saint-Corneille	72
30090	276	\N	Saint-Cosme-en-Vairais	72
30091	277	\N	Saint-Denis-des-Coudrais	72
30092	278	\N	Saint-Denis-d'Orques	72
30093	279	\N	Saint-Georges-de-la-Couée	72
30094	280	\N	Saint-Georges-du-Bois	72
30095	281	\N	Saint-Georges-du-Rosay	72
30096	282	\N	Saint-Georges-le-Gaultier	72
30097	283	\N	Saint-Germain-d'Arcé	72
30098	284	\N	Saint-Germain-sur-Sarthe	72
30099	286	\N	Saint-Gervais-de-Vic	72
30100	287	\N	Saint-Gervais-en-Belin	72
30101	288	\N	Saint-Hilaire-le-Lierru	72
30102	289	\N	Sainte-Jamme-sur-Sarthe	72
30103	290	\N	Saint-Jean-d'Assé	72
30104	291	\N	Saint-Jean-de-la-Motte	72
30105	292	\N	Saint-Jean-des-Échelles	72
30106	293	\N	Saint-Jean-du-Bois	72
30107	294	\N	Saint-Léonard-des-Bois	72
30108	295	\N	Saint-Longis	72
30109	296	\N	Saint-Maixent	72
30110	297	\N	Saint-Marceau	72
30111	298	\N	Saint-Mars-de-Locquenay	72
30112	299	\N	Saint-Mars-d'Outillé	72
30113	300	\N	Saint-Mars-la-Brière	72
30114	301	\N	Saint-Mars-sous-Ballon	72
30115	302	\N	Saint-Martin-des-Monts	72
30116	303	\N	Saint-Michel-de-Chavaignes	72
30117	304	\N	Sainte-Osmane	72
30118	305	\N	Saint-Ouen-de-Mimbré	72
30119	306	\N	Saint-Ouen-en-Belin	72
30120	307	\N	Saint-Ouen-en-Champagne	72
30121	308	\N	Saint-Paterne	72
30122	309	\N	Saint-Paul-le-Gaultier	72
30123	310	\N	Saint-Pavace	72
30124	311	\N	Saint-Pierre-de-Chevillé	72
30125	312	\N	Saint-Pierre-des-Bois	72
30126	313	\N	Saint-Pierre-des-Ormes	72
30127	314	\N	Saint-Pierre-du-Lorouër	72
30128	315	\N	Saint-Rémy-de-Sillé	72
30129	316	\N	Saint-Rémy-des-Monts	72
30130	317	\N	Saint-Rémy-du-Val	72
30131	318	\N	Saint-Rigomer-des-Bois	72
30132	319	\N	Sainte-Sabine-sur-Longève	72
30133	320	\N	Saint-Saturnin	72
30134	321	\N	Saint-Symphorien	72
30135	322	\N	Saint-Ulphace	72
30136	323	\N	Saint-Victeur	72
30137	324	\N	Saint-Vincent-des-Prés	72
30138	325	\N	Saint-Vincent-du-Lorouër	72
30139	326	\N	Saosnes	72
30140	327	\N	Sarcé	72
30141	328	\N	Sargé-lès-le-Mans	72
30142	329	\N	Savigné-l'Évêque	72
30143	330	\N	Savigné-sous-le-Lude	72
30144	331	\N	Sceaux-sur-Huisne	72
30145	332	\N	Ségrie	72
30146	333	\N	Semur-en-Vallon	72
30147	334	\N	Sillé-le-Guillaume	72
30148	335	\N	Sillé-le-Philippe	72
30149	336	\N	Solesmes	72
30150	337	\N	Sougé-le-Ganelon	72
30151	338	\N	Souillé	72
30152	339	\N	Souligné-Flacé	72
30153	340	\N	Souligné-sous-Ballon	72
30154	341	\N	Soulitré	72
30155	342	\N	Souvigné-sur-Même	72
30156	343	\N	Souvigné-sur-Sarthe	72
30157	344	\N	Spay	72
30158	345	\N	Surfonds	72
30159	346	\N	Suze-sur-Sarthe	72
30160	347	\N	Tassé	72
30161	348	\N	Tassillé	72
30162	349	\N	Teillé	72
30163	350	\N	Teloché	72
30164	351	\N	Tennie	72
30165	352	\N	Terrehault	72
30166	353	\N	Théligny	72
30167	354	\N	Thoigné	72
30168	355	\N	Thoiré-sous-Contensor	72
30169	356	\N	Thoiré-sur-Dinan	72
30170	357	\N	Thorée-les-Pins	72
30171	358	\N	Thorigné-sur-Dué	72
30172	359	\N	Torcé-en-Vallée	72
30173	360	\N	Trangé	72
30174	361	\N	Tresson	72
30175	362	\N	Tronchet	72
30176	363	\N	Tuffé	72
30177	364	\N	Vaas	72
30178	366	\N	Valennes	72
30179	367	\N	Vallon-sur-Gée	72
30180	368	\N	Vancé	72
30181	369	\N	Verneil-le-Chétif	72
30182	370	\N	Vernie	72
30183	372	\N	Vezot	72
30184	373	\N	Vibraye	72
30185	374	\N	Villaines-la-Carelle	72
30186	375	\N	Villaines-la-Gonais	72
30187	376	\N	Villaines-sous-Lucé	72
30188	377	\N	Villaines-sous-Malicorne	72
30189	378	\N	Vion	72
30190	379	\N	Viré-en-Champagne	72
30191	380	\N	Vivoin	72
30192	381	\N	Voivres-lès-le-Mans	72
30193	382	\N	Volnay	72
30194	383	\N	Vouvray-sur-Huisne	72
30195	384	\N	Vouvray-sur-Loir	72
30196	385	\N	Yvré-le-Pôlin	72
30197	386	\N	Yvré-l'Évêque	72
30198	001	\N	Aiguebelette-le-Lac	73
30199	002	\N	Aiguebelle	73
30200	003	\N	Aigueblanche	73
30201	004	\N	Aillon-le-Jeune	73
30202	005	\N	Aillon-le-Vieux	73
30203	006	\N	Aime	73
30204	007	\N	Aiton	73
30205	008	\N	Aix-les-Bains	73
30206	010	\N	Albens	73
30207	011	\N	Albertville	73
30208	012	\N	Albiez-le-Jeune	73
30209	013	\N	Albiez-Montrond	73
30210	014	\N	Allondaz	73
30211	015	\N	Allues	73
30212	017	\N	Apremont	73
30213	018	\N	Arbin	73
30214	019	\N	Argentine	73
30215	020	\N	Arith	73
30216	021	\N	Arvillard	73
30217	022	\N	Attignat-Oncin	73
30218	023	\N	Aussois	73
30219	024	\N	Avanchers-Valmorel	73
30220	025	\N	Avressieux	73
30221	026	\N	Avrieux	73
30222	027	\N	Ayn	73
30223	028	\N	Balme	73
30224	029	\N	Barberaz	73
30225	030	\N	Barby	73
30226	031	\N	Bassens	73
30227	032	\N	Bâthie	73
30228	033	\N	Bauche	73
30229	034	\N	Beaufort	73
30230	036	\N	Bellecombe-en-Bauges	73
30231	038	\N	Bellentre	73
30232	039	\N	Belmont-Tramonet	73
30233	040	\N	Bessans	73
30234	041	\N	Betton-Bettonet	73
30235	042	\N	Billième	73
30236	043	\N	Biolle	73
30237	045	\N	Bois	73
30238	046	\N	Bonneval	73
30239	047	\N	Bonneval-sur-Arc	73
30240	048	\N	Bonvillard	73
30241	049	\N	Bonvillaret	73
30242	050	\N	Bourdeau	73
30243	051	\N	Bourget-du-Lac	73
30244	052	\N	Bourget-en-Huile	73
30245	053	\N	Bourgneuf	73
30246	054	\N	Bourg-Saint-Maurice	73
30247	055	\N	Bozel	73
30248	056	\N	Bramans	73
30249	057	\N	Brides-les-Bains	73
30250	058	\N	Bridoire	73
30251	059	\N	Brison-Saint-Innocent	73
30252	061	\N	Césarches	73
30253	062	\N	Cessens	73
30254	063	\N	Cevins	73
30255	064	\N	Challes-les-Eaux	73
30256	065	\N	Chambéry	73
30257	067	\N	Chambre	73
30258	068	\N	Chamousset	73
30259	069	\N	Chamoux-sur-Gelon	73
30260	070	\N	Champagneux	73
30261	071	\N	Champagny-en-Vanoise	73
30262	072	\N	Champ-Laurent	73
30263	073	\N	Chanaz	73
30264	074	\N	Chapelle	73
30265	075	\N	Chapelle-Blanche	73
30266	076	\N	Chapelle-du-Mont-du-Chat	73
30267	077	\N	Chapelles	73
30268	078	\N	Chapelle-Saint-Martin	73
30269	079	\N	Châteauneuf	73
30270	080	\N	Châtel	73
30271	081	\N	Châtelard	73
30272	082	\N	Chavanne	73
30273	083	\N	Chavannes-en-Maurienne	73
30274	084	\N	Chignin	73
30275	085	\N	Chindrieux	73
30276	086	\N	Cléry	73
30277	087	\N	Cognin	73
30278	088	\N	Cohennoz	73
30279	089	\N	Coise-Saint-Jean-Pied-Gauthier	73
30280	090	\N	Compôte	73
30281	091	\N	Conjux	73
30282	092	\N	Corbel	73
30283	093	\N	Côte-d'Aime	73
30284	094	\N	Crest-Voland	73
30285	095	\N	Croix-de-la-Rochette	73
30286	096	\N	Cruet	73
30287	097	\N	Curienne	73
30288	098	\N	Déserts	73
30289	099	\N	Détrier	73
30290	100	\N	Domessin	73
30291	101	\N	Doucy-en-Bauges	73
30292	103	\N	Drumettaz-Clarafond	73
30293	104	\N	Dullin	73
30294	105	\N	Échelles	73
30295	106	\N	École	73
30296	107	\N	Entremont-le-Vieux	73
30297	108	\N	Épersy	73
30298	109	\N	Épierre	73
30299	110	\N	Esserts-Blay	73
30300	111	\N	Étable	73
30301	112	\N	Feissons-sur-Isère	73
30302	113	\N	Feissons-sur-Salins	73
30303	114	\N	Flumet	73
30304	115	\N	Fontaine-le-Puits	73
30305	116	\N	Fontcouverte-la Toussuire	73
30306	117	\N	Fourneaux	73
30307	118	\N	Francin	73
30308	119	\N	Freney	73
30309	120	\N	Fréterive	73
30310	121	\N	Frontenex	73
30311	122	\N	Gerbaix	73
30312	123	\N	Giettaz	73
30313	124	\N	Gilly-sur-Isère	73
30314	126	\N	Granier	73
30315	127	\N	Gresin	73
30316	128	\N	Grésy-sur-Aix	73
30317	129	\N	Grésy-sur-Isère	73
30318	130	\N	Grignon	73
30319	131	\N	Hautecour	73
30320	132	\N	Hauteluce	73
30321	133	\N	Hauteville	73
30322	135	\N	Hermillon	73
30323	137	\N	Jacob-Bellecombette	73
30324	138	\N	Jarrier	73
30325	139	\N	Jarsy	73
30326	140	\N	Jongieux	73
30327	141	\N	Laissaud	73
30328	142	\N	Landry	73
30329	143	\N	Lanslebourg-Mont-Cenis	73
30330	144	\N	Lanslevillard	73
30331	145	\N	Lépin-le-Lac	73
30332	146	\N	Lescheraines	73
30333	147	\N	Loisieux	73
30334	149	\N	Lucey	73
30335	150	\N	Mâcot-la-Plagne	73
30336	151	\N	Marches	73
30337	152	\N	Marcieux	73
30338	153	\N	Marthod	73
30339	154	\N	Mercury	73
30340	155	\N	Méry	73
30341	156	\N	Meyrieux-Trouet	73
30342	157	\N	Modane	73
30343	158	\N	Mognard	73
30344	159	\N	Mollettes	73
30345	160	\N	Montagnole	73
30346	161	\N	Montagny	73
30347	162	\N	Montailleur	73
30348	163	\N	Montaimont	73
30349	164	\N	Montcel	73
30350	166	\N	Montendry	73
30351	167	\N	Montgellafrey	73
30352	168	\N	Montgilbert	73
30353	169	\N	Montgirod	73
30354	170	\N	Monthion	73
30355	171	\N	Montmélian	73
30356	173	\N	Montricher-Albanne	73
30357	175	\N	Montsapey	73
30358	176	\N	Montvalezan	73
30359	177	\N	Montvernier	73
30360	178	\N	Motte-en-Bauges	73
30361	179	\N	Motte-Servolex	73
30362	180	\N	Motz	73
30363	181	\N	Moûtiers	73
30364	182	\N	Mouxy	73
30365	183	\N	Myans	73
30366	184	\N	Nances	73
30367	186	\N	Notre-Dame-de-Bellecombe	73
30368	187	\N	Léchère	73
30369	188	\N	Notre-Dame-des-Millières	73
30370	189	\N	Notre-Dame-du-Cruet	73
30371	190	\N	Notre-Dame-du-Pré	73
30372	191	\N	Novalaise	73
30373	192	\N	Noyer	73
30374	193	\N	Ontex	73
30375	194	\N	Orelle	73
30376	196	\N	Pallud	73
30377	197	\N	Peisey-Nancroix	73
30378	198	\N	Perrière	73
30379	200	\N	Planaise	73
30380	201	\N	Planay	73
30381	202	\N	Plancherine	73
30382	203	\N	Pontamafrey-Montpascal	73
30383	204	\N	Pont-de-Beauvoisin	73
30384	205	\N	Pontet	73
30385	206	\N	Pralognan-la-Vanoise	73
30386	207	\N	Presle	73
30387	208	\N	Pugny-Chatenod	73
30388	210	\N	Puygros	73
30389	211	\N	Queige	73
30390	212	\N	Randens	73
30391	213	\N	Ravoire	73
30392	214	\N	Rochefort	73
30393	215	\N	Rochette	73
30394	216	\N	Rognaix	73
30395	217	\N	Rotherens	73
30396	218	\N	Ruffieux	73
30397	219	\N	Saint-Alban-de-Montbel	73
30398	220	\N	Saint-Alban-des-Hurtières	73
30399	221	\N	Saint-Alban-des-Villards	73
30400	222	\N	Saint-Alban-Leysse	73
30401	223	\N	Saint-André	73
30402	224	\N	Saint-Avre	73
30403	225	\N	Saint-Baldoph	73
30404	226	\N	Saint-Béron	73
30405	227	\N	Saint-Bon-Tarentaise	73
30406	228	\N	Saint-Cassin	73
30407	229	\N	Saint-Christophe	73
30408	230	\N	Saint-Colomban-des-Villards	73
30409	231	\N	Saint-Etienne-de-Cuines	73
30410	232	\N	Sainte-Foy-Tarentaise	73
30411	233	\N	Saint-Franc	73
30412	234	\N	Saint-François-de-Sales	73
30413	235	\N	Saint-François-Longchamp	73
30414	236	\N	Saint-Genix-sur-Guiers	73
30415	237	\N	Saint-Georges-des-Hurtières	73
30416	238	\N	Saint-Germain-la-Chambotte	73
30417	239	\N	Saint-Girod	73
30418	240	\N	Sainte-Hélène-du-Lac	73
30419	241	\N	Sainte-Hélène-sur-Isère	73
30420	242	\N	Saint-Jean-d'Arves	73
30421	243	\N	Saint-Jean-d'Arvey	73
30422	244	\N	Saint-Jean-de-Belleville	73
30423	245	\N	Saint-Jean-de-Chevelu	73
30424	246	\N	Saint-Jean-de-Couz	73
30425	247	\N	Saint-Jean-de-la-Porte	73
30426	248	\N	Saint-Jean-de-Maurienne	73
30427	249	\N	Saint-Jeoire-Prieuré	73
30428	250	\N	Saint-Julien-Mont-Denis	73
30429	252	\N	Saint-Léger	73
30430	253	\N	Saint-Marcel	73
30431	254	\N	Sainte-Marie-d'Alvey	73
30432	255	\N	Sainte-Marie-de-Cuines	73
30433	256	\N	Saint-Martin-d'Arc	73
30434	257	\N	Saint-Martin-de-Belleville	73
30435	258	\N	Saint-Martin-de-la-Porte	73
30436	259	\N	Saint-Martin-sur-la-Chambre	73
30437	260	\N	Saint-Maurice-de-Rotherens	73
30438	261	\N	Saint-Michel-de-Maurienne	73
30439	262	\N	Saint-Nicolas-la-Chapelle	73
30440	263	\N	Saint-Offenge-Dessous	73
30441	264	\N	Saint-Offenge-Dessus	73
30442	265	\N	Saint-Ours	73
30443	266	\N	Saint-Oyen	73
30444	267	\N	Saint-Pancrace	73
30445	268	\N	Saint-Paul-sur-Isère	73
30446	269	\N	Saint-Paul	73
30447	270	\N	Saint-Pierre-d'Albigny	73
30448	271	\N	Saint-Pierre-d'Alvey	73
30449	272	\N	Saint-Pierre-de-Belleville	73
30450	273	\N	Saint-Pierre-de-Curtille	73
30451	274	\N	Saint-Pierre-d'Entremont	73
30452	275	\N	Saint-Pierre-de-Genebroz	73
30453	276	\N	Saint-Pierre-de-Soucy	73
30454	277	\N	Sainte-Reine	73
30455	278	\N	Saint-Rémy-de-Maurienne	73
30456	280	\N	Saint-Sorlin-d'Arves	73
30457	281	\N	Saint-Sulpice	73
30458	282	\N	Saint-Thibaud-de-Couz	73
30459	283	\N	Saint-Vital	73
30460	284	\N	Salins-les-Thermes	73
30461	285	\N	Séez	73
30462	286	\N	Serrières-en-Chautagne	73
30463	287	\N	Sollières-Sardières	73
30464	288	\N	Sonnaz	73
30465	289	\N	Table	73
30466	290	\N	Termignon	73
30467	292	\N	Thénésol	73
30468	293	\N	Thoiry	73
30469	294	\N	Thuile	73
30470	296	\N	Tignes	73
30471	297	\N	Tournon	73
30472	298	\N	Tours-en-Savoie	73
30473	299	\N	Traize	73
30474	300	\N	Tresserve	73
30475	301	\N	Trévignin	73
30476	302	\N	Trinité	73
30477	303	\N	Ugine	73
30478	304	\N	Val-d'Isère	73
30479	305	\N	Valezan	73
30480	306	\N	Valloire	73
30481	307	\N	Valmeinier	73
30482	308	\N	Venthon	73
30483	309	\N	Verel-de-Montbel	73
30484	310	\N	Verel-Pragondran	73
30485	311	\N	Verneil	73
30486	312	\N	Verrens-Arvey	73
30487	313	\N	Verthemex	73
30488	314	\N	Villard-d'Héry	73
30489	315	\N	Villard-Léger	73
30490	316	\N	Villard-Sallet	73
30491	317	\N	Villard-sur-Doron	73
30492	318	\N	Villarembert	73
30493	320	\N	Villargondran	73
30494	321	\N	Villarlurin	73
30495	322	\N	Villarodin-Bourget	73
30496	323	\N	Villaroger	73
30497	324	\N	Villaroux	73
30498	326	\N	Vimines	73
30499	327	\N	Vions	73
30500	328	\N	Viviers-du-Lac	73
30501	329	\N	Voglans	73
30502	330	\N	Yenne	73
30503	001	\N	Abondance	74
30504	002	\N	Alby-sur-Chéran	74
30505	003	\N	Alex	74
30506	004	\N	Allèves	74
30507	005	\N	Allinges	74
30508	006	\N	Allonzier-la-Caille	74
30509	007	\N	Amancy	74
30510	008	\N	Ambilly	74
30511	009	\N	Andilly	74
30512	010	\N	Annecy	74
30513	011	\N	Annecy-le-Vieux	74
30514	012	\N	Annemasse	74
30515	013	\N	Anthy-sur-Léman	74
30516	014	\N	Arâches-la-Frasse	74
30517	015	\N	Arbusigny	74
30518	016	\N	Archamps	74
30519	018	\N	Arenthon	74
30520	019	\N	Argonay	74
30521	020	\N	Armoy	74
30522	021	\N	Arthaz-Pont-Notre-Dame	74
30523	022	\N	Aviernoz	74
30524	024	\N	Ayse	74
30525	025	\N	Ballaison	74
30526	026	\N	Balme-de-Sillingy	74
30527	027	\N	Balme-de-Thuy	74
30528	029	\N	Bassy	74
30529	030	\N	Baume	74
30530	031	\N	Beaumont	74
30531	032	\N	Bellevaux	74
30532	033	\N	Bernex	74
30533	034	\N	Biot	74
30534	035	\N	Bloye	74
30535	036	\N	Bluffy	74
30536	037	\N	Boëge	74
30537	038	\N	Bogève	74
30538	040	\N	Bonne	74
30539	041	\N	Bonnevaux	74
30540	042	\N	Bonneville	74
30541	043	\N	Bons-en-Chablais	74
30542	044	\N	Bossey	74
30543	045	\N	Bouchet	74
30544	046	\N	Boussy	74
30545	048	\N	Brenthonne	74
30546	049	\N	Brizon	74
30547	050	\N	Burdignin	74
30548	051	\N	Cercier	74
30549	052	\N	Cernex	74
30550	053	\N	Cervens	74
30551	054	\N	Chainaz-les-Frasses	74
30552	055	\N	Challonges	74
30553	056	\N	Chamonix-Mont-Blanc	74
30554	057	\N	Champanges	74
30555	058	\N	Chapelle-d'Abondance	74
30556	059	\N	Chapelle-Rambaud	74
30557	060	\N	Chapelle-Saint-Maurice	74
30558	061	\N	Chapeiry	74
30559	062	\N	Charvonnex	74
30560	063	\N	Châtel	74
30561	064	\N	Châtillon-sur-Cluses	74
30562	065	\N	Chaumont	74
30563	066	\N	Chavannaz	74
30564	067	\N	Chavanod	74
30565	068	\N	Chêne-en-Semine	74
30566	069	\N	Chênex	74
30567	070	\N	Chens-sur-Léman	74
30568	071	\N	Chessenaz	74
30569	072	\N	Chevaline	74
30570	073	\N	Chevenoz	74
30571	074	\N	Chevrier	74
30572	075	\N	Chilly	74
30573	076	\N	Choisy	74
30574	077	\N	Clarafond	74
30575	078	\N	Clermont	74
30576	079	\N	Clefs	74
30577	080	\N	Clusaz	74
30578	081	\N	Cluses	74
30579	082	\N	Collonges-sous-Salève	74
30580	083	\N	Combloux	74
30581	084	\N	Cons-Sainte-Colombe	74
30582	085	\N	Contamines-Montjoie	74
30583	086	\N	Contamine-Sarzin	74
30584	087	\N	Contamine-sur-Arve	74
30585	088	\N	Copponex	74
30586	089	\N	Cordon	74
30587	090	\N	Cornier	74
30588	091	\N	Côte-d'Arbroz	74
30589	093	\N	Cran-Gevrier	74
30590	094	\N	Cranves-Sales	74
30591	095	\N	Crempigny-Bonneguête	74
30592	096	\N	Cruseilles	74
30593	097	\N	Cusy	74
30594	098	\N	Cuvat	74
30595	099	\N	Demi-Quartier	74
30596	100	\N	Desingy	74
30597	101	\N	Dingy-en-Vuache	74
30598	102	\N	Dingy-Saint-Clair	74
30599	103	\N	Domancy	74
30600	104	\N	Doussard	74
30601	105	\N	Douvaine	74
30602	106	\N	Draillant	74
30603	107	\N	Droisy	74
30604	108	\N	Duingt	74
30605	109	\N	Éloise	74
30606	110	\N	Entremont	74
30607	111	\N	Entrevernes	74
30608	112	\N	Épagny	74
30609	114	\N	Essert-Romand	74
30610	116	\N	Etaux	74
30611	117	\N	Étercy	74
30612	118	\N	Étrembières	74
30613	119	\N	Évian-les-Bains	74
30614	120	\N	Évires	74
30615	121	\N	Excenevex	74
30616	122	\N	Faucigny	74
30617	123	\N	Faverges	74
30618	124	\N	Feigères	74
30619	126	\N	Fessy	74
30620	127	\N	Féternes	74
30621	128	\N	Fillinges	74
30622	129	\N	Forclaz	74
30623	130	\N	Franclens	74
30624	131	\N	Frangy	74
30625	133	\N	Gaillard	74
30626	134	\N	Gets	74
30627	135	\N	Giez	74
30628	136	\N	Grand-Bornand	74
30629	137	\N	Groisy	74
30630	138	\N	Gruffy	74
30631	139	\N	Habère-Lullin	74
30632	140	\N	Habère-Poche	74
30633	141	\N	Hauteville-sur-Fier	74
30634	142	\N	Héry-sur-Alby	74
30635	143	\N	Houches	74
30636	144	\N	Jonzier-Épagny	74
30637	145	\N	Juvigny	74
30638	146	\N	Larringes	74
30639	147	\N	Lathuile	74
30640	148	\N	Leschaux	74
30641	150	\N	Loisin	74
30642	151	\N	Lornay	74
30643	152	\N	Lovagny	74
30644	153	\N	Lucinges	74
30645	154	\N	Lugrin	74
30646	155	\N	Lullin	74
30647	156	\N	Lully	74
30648	157	\N	Lyaud	74
30649	158	\N	Machilly	74
30650	159	\N	Magland	74
30651	160	\N	Manigod	74
30652	161	\N	Marcellaz-Albanais	74
30653	162	\N	Marcellaz	74
30654	163	\N	Margencel	74
30655	164	\N	Marignier	74
30656	165	\N	Marigny-Saint-Marcel	74
30657	166	\N	Marin	74
30658	167	\N	Marlens	74
30659	168	\N	Marlioz	74
30660	169	\N	Marnaz	74
30661	170	\N	Massingy	74
30662	171	\N	Massongy	74
30663	172	\N	Maxilly-sur-Léman	74
30664	173	\N	Megève	74
30665	174	\N	Mégevette	74
30666	175	\N	Meillerie	74
30667	176	\N	Menthon-Saint-Bernard	74
30668	177	\N	Menthonnex-en-Bornes	74
30669	178	\N	Menthonnex-sous-Clermont	74
30670	179	\N	Mésigny	74
30671	180	\N	Messery	74
30672	181	\N	Metz-Tessy	74
30673	182	\N	Meythet	74
30674	183	\N	Mieussy	74
30675	184	\N	Minzier	74
30676	185	\N	Monnetier-Mornex	74
30677	186	\N	Montagny-les-Lanches	74
30678	187	\N	Montmin	74
30679	188	\N	Montriond	74
30680	189	\N	Mont-Saxonnex	74
30681	190	\N	Morillon	74
30682	191	\N	Morzine	74
30683	192	\N	Moye	74
30684	193	\N	Muraz	74
30685	194	\N	Mûres	74
30686	195	\N	Musièges	74
30687	196	\N	Nancy-sur-Cluses	74
30688	197	\N	Nangy	74
30689	198	\N	Nâves-Parmelan	74
30690	199	\N	Nernier	74
30691	200	\N	Neuvecelle	74
30692	201	\N	Neydens	74
30693	202	\N	Nonglard	74
30694	203	\N	Novel	74
30695	204	\N	Ollières	74
30696	205	\N	Onnion	74
30697	206	\N	Orcier	74
30698	208	\N	Passy	74
30699	209	\N	Peillonnex	74
30700	210	\N	Perrignier	74
30701	211	\N	Pers-Jussy	74
30702	212	\N	Petit-Bornand-les-Glières	74
30703	213	\N	Poisy	74
30704	215	\N	Praz-sur-Arly	74
30705	216	\N	Présilly	74
30706	217	\N	Pringy	74
30707	218	\N	Publier	74
30708	219	\N	Quintal	74
30709	220	\N	Reignier	74
30710	221	\N	Reposoir	74
30711	222	\N	Reyvroz	74
30712	223	\N	Rivière-Enverse	74
30713	224	\N	Roche-sur-Foron	74
30714	225	\N	Rumilly	74
30715	226	\N	Saint-André-de-Boëge	74
30716	228	\N	Saint-Blaise	74
30717	229	\N	Saint-Cergues	74
30718	231	\N	Saint-Eusèbe	74
30719	232	\N	Saint-Eustache	74
30720	233	\N	Saint-Félix	74
30721	234	\N	Saint-Ferréol	74
30722	235	\N	Saint-Germain-sur-Rhône	74
30723	236	\N	Saint-Gervais-les-Bains	74
30724	237	\N	Saint-Gingolph	74
30725	238	\N	Saint-Jean-d'Aulps	74
30726	239	\N	Saint-Jean-de-Sixt	74
30727	240	\N	Saint-Jean-de-Tholome	74
30728	241	\N	Saint-Jeoire	74
30729	242	\N	Saint-Jorioz	74
30730	243	\N	Saint-Julien-en-Genevois	74
30731	244	\N	Saint-Laurent	74
30732	245	\N	Saint-Martin-Bellevue	74
30733	249	\N	Saint-Paul-en-Chablais	74
30734	250	\N	Saint-Pierre-en-Faucigny	74
30735	252	\N	Saint-Sigismond	74
30736	253	\N	Saint-Sixt	74
30737	254	\N	Saint-Sylvestre	74
30738	255	\N	Sales	74
30739	256	\N	Sallanches	74
30740	257	\N	Sallenôves	74
30741	258	\N	Samoëns	74
30742	259	\N	Sappey	74
30743	260	\N	Savigny	74
30744	261	\N	Saxel	74
30745	262	\N	Scientrier	74
30746	263	\N	Sciez	74
30747	264	\N	Scionzier	74
30748	265	\N	Serraval	74
30749	266	\N	Servoz	74
30750	267	\N	Sévrier	74
30751	268	\N	Seynod	74
30752	269	\N	Seyssel	74
30753	270	\N	Seythenex	74
30754	271	\N	Seytroux	74
30755	272	\N	Sillingy	74
30756	273	\N	Sixt-Fer-à-Cheval	74
30757	274	\N	Val-de-Fier	74
30758	275	\N	Talloires	74
30759	276	\N	Taninges	74
30760	278	\N	Thyez	74
30761	279	\N	Thollon-les-Mémises	74
30762	280	\N	Thônes	74
30763	281	\N	Thonon-les-Bains	74
30764	282	\N	Thorens-Glières	74
30765	283	\N	Thusy	74
30766	284	\N	Tour	74
30767	285	\N	Usinens	74
30768	286	\N	Vacheresse	74
30769	287	\N	Vailly	74
30770	288	\N	Valleiry	74
30771	289	\N	Vallières	74
30772	290	\N	Vallorcine	74
30773	291	\N	Vanzy	74
30774	292	\N	Vaulx	74
30775	293	\N	Veigy-Foncenex	74
30776	294	\N	Verchaix	74
30777	295	\N	Vernaz	74
30778	296	\N	Vers	74
30779	297	\N	Versonnex	74
30780	298	\N	Vétraz-Monthoux	74
30781	299	\N	Veyrier-du-Lac	74
30782	301	\N	Villard	74
30783	302	\N	Villards-sur-Thônes	74
30784	303	\N	Villaz	74
30785	304	\N	Ville-en-Sallaz	74
30786	305	\N	Ville-la-Grand	74
30787	306	\N	Villy-le-Bouveret	74
30788	307	\N	Villy-le-Pelloux	74
30789	308	\N	Vinzier	74
30790	309	\N	Viry	74
30791	310	\N	Viuz-la-Chiésaz	74
30792	311	\N	Viuz-en-Sallaz	74
30793	312	\N	Vougy	74
30794	313	\N	Vovray-en-Bornes	74
30795	314	\N	Vulbens	74
30796	315	\N	Yvoire	74
30797	056	\N	Paris	75
30798	001	\N	Allouville-Bellefosse	76
30799	002	\N	Alvimare	76
30800	004	\N	Ambrumesnil	76
30801	005	\N	Amfreville-la-Mi-Voie	76
30802	006	\N	Amfreville-les-Champs	76
30803	007	\N	Anceaumeville	76
30804	008	\N	Ancourt	76
30805	009	\N	Ancourteville-sur-Héricourt	76
30806	010	\N	Ancretiéville-Saint-Victor	76
30807	011	\N	Ancretteville-sur-Mer	76
30808	012	\N	Angerville-Bailleul	76
30809	013	\N	Angerville-la-Martel	76
30810	014	\N	Angerville-l'Orcher	76
30811	015	\N	Angiens	76
30812	016	\N	Anglesqueville-la-Bras-Long	76
30813	017	\N	Anglesqueville-l'Esneval	76
30814	018	\N	Val-de-Saâne	76
30815	019	\N	Anneville-sur-Scie	76
30816	020	\N	Anneville-Ambourville	76
30817	021	\N	Annouville-Vilmesnil	76
30818	022	\N	Anquetierville	76
30819	023	\N	Anvéville	76
30820	024	\N	Ardouval	76
30821	025	\N	Argueil	76
30822	026	\N	Arques-la-Bataille	76
30823	027	\N	Assigny	76
30824	028	\N	Aubéguimont	76
30825	029	\N	Aubermesnil-aux-Érables	76
30826	030	\N	Aubermesnil-Beaumais	76
30827	031	\N	Auberville-la-Campagne	76
30828	032	\N	Auberville-la-Manuel	76
30829	033	\N	Auberville-la-Renault	76
30830	034	\N	Auffay	76
30831	035	\N	Aumale	76
30832	036	\N	Auppegard	76
30833	037	\N	Auquemesnil	76
30834	038	\N	Authieux-Ratiéville	76
30835	039	\N	Authieux-sur-le-Port-Saint-Ouen	76
30836	040	\N	Autigny	76
30837	041	\N	Autretot	76
30838	042	\N	Auvilliers	76
30839	043	\N	Auzebosc	76
30840	044	\N	Auzouville-Auberbosc	76
30841	045	\N	Auzouville-l'Esneval	76
30842	046	\N	Auzouville-sur-Ry	76
30843	047	\N	Auzouville-sur-Saâne	76
30844	048	\N	Avesnes-en-Bray	76
30845	049	\N	Avesnes-en-Val	76
30846	050	\N	Avremesnil	76
30847	051	\N	Bacqueville-en-Caux	76
30848	052	\N	Bailleul-Neuville	76
30849	053	\N	Baillolet	76
30850	054	\N	Bailly-en-Rivière	76
30851	055	\N	Baons-le-Comte	76
30852	056	\N	Bardouville	76
30853	057	\N	Barentin	76
30854	058	\N	Baromesnil	76
30855	059	\N	Bazinval	76
30856	060	\N	Beaubec-la-Rosière	76
30857	062	\N	Beaumont-le-Hareng	76
30858	063	\N	Beauval-en-Caux	76
30859	064	\N	Beaurepaire	76
30860	065	\N	Beaussault	76
30861	066	\N	Beautot	76
30862	067	\N	Beauvoir-en-Lyons	76
30863	068	\N	Bec-de-Mortagne	76
30864	069	\N	Belbeuf	76
30865	070	\N	Bellencombre	76
30866	071	\N	Bellengreville	76
30867	072	\N	Belleville-en-Caux	76
30868	073	\N	Belleville-sur-Mer	76
30869	074	\N	Bellière	76
30870	075	\N	Belmesnil	76
30871	076	\N	Bénarville	76
30872	077	\N	Bénesville	76
30873	078	\N	Bennetot	76
30874	079	\N	Bénouville	76
30875	080	\N	Bermonville	76
30876	081	\N	Berneval-le-Grand	76
30877	082	\N	Bernières	76
30878	083	\N	Bertheauville	76
30879	084	\N	Bertreville	76
30880	085	\N	Bertreville-Saint-Ouen	76
30881	086	\N	Bertrimont	76
30882	087	\N	Berville	76
30883	088	\N	Berville-sur-Seine	76
30884	089	\N	Betteville	76
30885	090	\N	Beuzeville-la-Grenier	76
30886	091	\N	Beuzeville-la-Guérard	76
30887	092	\N	Beuzevillette	76
30888	093	\N	Bézancourt	76
30889	094	\N	Bierville	76
30890	095	\N	Bihorel	76
30891	096	\N	Biville-la-Baignarde	76
30892	097	\N	Biville-la-Rivière	76
30893	098	\N	Biville-sur-Mer	76
30894	099	\N	Blacqueville	76
30895	100	\N	Blainville-Crevon	76
30896	101	\N	Blangy-sur-Bresle	76
30897	103	\N	Bonsecours	76
30898	104	\N	Blosseville	76
30899	105	\N	Bocasse	76
30900	106	\N	Bois-d'Ennebourg	76
30901	107	\N	Bois-Guilbert	76
30902	108	\N	Bois-Guillaume	76
30903	109	\N	Bois-Héroult	76
30904	110	\N	Bois-Himont	76
30905	111	\N	Bois-l'Évêque	76
30906	112	\N	Bois-Robert	76
30907	113	\N	Boissay	76
30908	114	\N	Bolbec	76
30909	115	\N	Bolleville	76
30910	116	\N	Boos	76
30911	117	\N	Bordeaux-Saint-Clair	76
30912	118	\N	Bornambusc	76
30913	119	\N	Bosc-Bérenger	76
30914	120	\N	Bosc-Bordel	76
30915	121	\N	Bosc-Édeline	76
30916	122	\N	Callengeville	76
30917	123	\N	Bosc-Guérard-Saint-Adrien	76
30918	124	\N	Bosc-Hyons	76
30919	125	\N	Bosc-le-Hard	76
30920	126	\N	Bosc-Mesnil	76
30921	127	\N	Bosc-Roger-sur-Buchy	76
30922	128	\N	Bosville	76
30923	129	\N	Boudeville	76
30924	130	\N	Bouelles	76
30925	131	\N	Bouille	76
30926	132	\N	Bourdainville	76
30927	133	\N	Bourg-Dun	76
30928	134	\N	Bourville	76
30929	135	\N	Bouville	76
30930	136	\N	Brachy	76
30931	137	\N	Bracquemont	76
30932	138	\N	Bracquetuit	76
30933	139	\N	Bradiancourt	76
30934	140	\N	Brametot	76
30935	141	\N	Bréauté	76
30936	142	\N	Brémontier-Merval	76
30937	143	\N	Bretteville-du-Grand-Caux	76
30938	144	\N	Bretteville-Saint-Laurent	76
30939	145	\N	Brunville	76
30940	146	\N	Buchy	76
30941	147	\N	Bully	76
30942	148	\N	Bures-en-Bray	76
30943	149	\N	Butot	76
30944	151	\N	Cailleville	76
30945	152	\N	Cailly	76
30946	153	\N	Calleville-les-Deux-Églises	76
30947	154	\N	Campneuseville	76
30948	155	\N	Canehan	76
30949	156	\N	Canouville	76
30950	157	\N	Canteleu	76
30951	158	\N	Canville-les-Deux-Églises	76
30952	159	\N	Cany-Barville	76
30953	160	\N	Carville-la-Folletière	76
30954	161	\N	Carville-Pot-de-Fer	76
30955	162	\N	Catelier	76
30956	163	\N	Catenay	76
30957	164	\N	Caudebec-en-Caux	76
30958	165	\N	Caudebec-lès-Elbeuf	76
30959	166	\N	Caule-Sainte-Beuve	76
30960	167	\N	Cauville-sur-Mer	76
30961	168	\N	Cent-Acres	76
30962	169	\N	Cerlangue	76
30963	170	\N	Chapelle-du-Bourgay	76
30964	171	\N	Chapelle-Saint-Ouen	76
30965	172	\N	Chapelle-sur-Dun	76
30966	173	\N	Chaussée	76
30967	174	\N	Cideville	76
30968	175	\N	Clais	76
30969	176	\N	Clasville	76
30970	177	\N	Claville-Motteville	76
30971	178	\N	Cléon	76
30972	179	\N	Clères	76
30973	180	\N	Cleuville	76
30974	181	\N	Cléville	76
30975	182	\N	Cliponville	76
30976	183	\N	Colleville	76
30977	184	\N	Colmesnil-Manneville	76
30978	185	\N	Compainville	76
30979	186	\N	Conteville	76
30980	187	\N	Contremoulins	76
30981	188	\N	Cottévrard	76
30982	189	\N	Crasville-la-Mallet	76
30983	190	\N	Crasville-la-Rocquefort	76
30984	191	\N	Cressy	76
30985	192	\N	Criel-sur-Mer	76
30986	193	\N	Crique	76
30987	194	\N	Criquebeuf-en-Caux	76
30988	195	\N	Criquetot-le-Mauconduit	76
30989	196	\N	Criquetot-l'Esneval	76
30990	197	\N	Criquetot-sur-Longueville	76
30991	198	\N	Criquetot-sur-Ouville	76
30992	199	\N	Criquiers	76
30993	200	\N	Critot	76
30994	201	\N	Croisy-sur-Andelle	76
30995	202	\N	Croixdalle	76
30996	203	\N	Croix-Mare	76
30997	204	\N	Cropus	76
30998	205	\N	Crosville-sur-Scie	76
30999	206	\N	Cuverville	76
31000	207	\N	Cuverville-sur-Yères	76
31001	208	\N	Cuy-Saint-Fiacre	76
31002	209	\N	Dampierre-en-Bray	76
31003	210	\N	Dampierre-Saint-Nicolas	76
31004	211	\N	Dancourt	76
31005	212	\N	Darnétal	76
31006	213	\N	Daubeuf-Serville	76
31007	214	\N	Dénestanville	76
31008	215	\N	Derchigny	76
31009	216	\N	Déville-lès-Rouen	76
31010	217	\N	Dieppe	76
31011	218	\N	Doudeauville	76
31012	219	\N	Doudeville	76
31013	220	\N	Douvrend	76
31014	221	\N	Drosay	76
31015	222	\N	Duclair	76
31016	223	\N	Écalles-Alix	76
31017	224	\N	Écrainville	76
31018	225	\N	Écretteville-lès-Baons	76
31019	226	\N	Écretteville-sur-Mer	76
31020	227	\N	Ectot-l'Auber	76
31021	228	\N	Ectot-lès-Baons	76
31022	229	\N	Elbeuf-en-Bray	76
31023	230	\N	Elbeuf-sur-Andelle	76
31024	231	\N	Elbeuf	76
31025	232	\N	Életot	76
31026	233	\N	Ellecourt	76
31027	234	\N	Émanville	76
31028	235	\N	Envermeu	76
31029	236	\N	Envronville	76
31030	237	\N	Épinay-sur-Duclair	76
31031	238	\N	Épouville	76
31032	239	\N	Épretot	76
31033	240	\N	Épreville	76
31034	241	\N	Ermenouville	76
31035	242	\N	Ernemont-la-Villette	76
31036	243	\N	Ernemont-sur-Buchy	76
31037	244	\N	Esclavelles	76
31038	245	\N	Eslettes	76
31039	247	\N	Esteville	76
31040	248	\N	Estouteville-Écalles	76
31041	249	\N	Étaimpuis	76
31042	250	\N	Étainhus	76
31043	251	\N	Étalleville	76
31044	252	\N	Étalondes	76
31045	253	\N	Étoutteville	76
31046	254	\N	Étretat	76
31047	255	\N	Eu	76
31048	257	\N	Fallencourt	76
31049	258	\N	Fauville-en-Caux	76
31050	259	\N	Fécamp	76
31051	260	\N	Ferrières-en-Bray	76
31052	261	\N	Ferté-Saint-Samson	76
31053	262	\N	Fesques	76
31054	263	\N	Feuillie	76
31055	264	\N	Flamanville	76
31056	265	\N	Flamets-Frétils	76
31057	266	\N	Flocques	76
31058	267	\N	Folletière	76
31059	268	\N	Fongueusemare	76
31060	269	\N	Fontaine-en-Bray	76
31061	270	\N	Fontaine-la-Mallet	76
31062	271	\N	Fontaine-le-Bourg	76
31063	272	\N	Fontaine-le-Dun	76
31064	273	\N	Fontaine-sous-Préaux	76
31065	274	\N	Fontelaye	76
31066	275	\N	Fontenay	76
31067	276	\N	Forges-les-Eaux	76
31068	277	\N	Fossé	76
31069	278	\N	Foucarmont	76
31070	279	\N	Foucart	76
31071	280	\N	Fréauville	76
31072	281	\N	Frénaye	76
31073	282	\N	Freneuse	76
31074	283	\N	Fresles	76
31075	284	\N	Fresnay-le-Long	76
31076	285	\N	Fresne-le-Plan	76
31077	286	\N	Fresnoy-Folny	76
31078	287	\N	Fresquiennes	76
31079	288	\N	Freulleville	76
31080	289	\N	Fréville	76
31081	290	\N	Frichemesnil	76
31082	291	\N	Froberville	76
31083	292	\N	Fry	76
31084	293	\N	Fultot	76
31085	294	\N	Gaillarde	76
31086	295	\N	Gaillefontaine	76
31087	296	\N	Gainneville	76
31088	297	\N	Gancourt-Saint-Étienne	76
31089	298	\N	Ganzeville	76
31090	299	\N	Gerponville	76
31091	300	\N	Gerville	76
31092	301	\N	Glicourt	76
31093	302	\N	Goderville	76
31094	303	\N	Gommerville	76
31095	304	\N	Gonfreville-Caillot	76
31096	305	\N	Gonfreville-l'Orcher	76
31097	306	\N	Gonnetot	76
31098	307	\N	Gonneville-la-Mallet	76
31099	308	\N	Gonneville-sur-Scie	76
31100	309	\N	Gonzeville	76
31101	310	\N	Gouchaupre	76
31102	311	\N	Goupillières	76
31103	312	\N	Gournay-en-Bray	76
31104	313	\N	Gouy	76
31105	314	\N	Graimbouville	76
31106	315	\N	Grainville-la-Teinturière	76
31107	316	\N	Grainville-sur-Ry	76
31108	317	\N	Grainville-Ymauville	76
31109	318	\N	Grand-Camp	76
31110	319	\N	Grand-Couronne	76
31111	320	\N	Grandcourt	76
31112	321	\N	Grandes-Ventes	76
31113	322	\N	Grand-Quevilly	76
31114	323	\N	Graval	76
31115	324	\N	Grèges	76
31116	325	\N	Grémonville	76
31117	326	\N	Greny	76
31118	327	\N	Greuville	76
31119	328	\N	Grigneuseville	76
31120	329	\N	Gruchet-le-Valasse	76
31121	330	\N	Gruchet-Saint-Siméon	76
31122	331	\N	Grugny	76
31123	332	\N	Grumesnil	76
31124	333	\N	Guerville	76
31125	334	\N	Gueures	76
31126	335	\N	Gueutteville	76
31127	336	\N	Gueutteville-les-Grès	76
31128	337	\N	Guilmécourt	76
31129	338	\N	Hallotière	76
31130	339	\N	Hanouard	76
31131	340	\N	Harcanville	76
31132	341	\N	Harfleur	76
31133	342	\N	Hattenville	76
31134	343	\N	Haucourt	76
31135	344	\N	Haudricourt	76
31136	345	\N	Haussez	76
31137	346	\N	Hautot-l'Auvray	76
31138	347	\N	Hautot-le-Vatois	76
31139	348	\N	Hautot-Saint-Sulpice	76
31140	349	\N	Hautot-sur-Mer	76
31141	350	\N	Hautot-sur-Seine	76
31142	351	\N	Havre	76
31143	352	\N	Haye	76
31144	353	\N	Héberville	76
31145	354	\N	Hénouville	76
31146	355	\N	Héricourt-en-Caux	76
31147	356	\N	Hermanville	76
31148	357	\N	Hermeville	76
31149	358	\N	Héron	76
31150	359	\N	Héronchelles	76
31151	360	\N	Heugleville-sur-Scie	76
31152	361	\N	Heuqueville	76
31153	362	\N	Heurteauville	76
31154	363	\N	Hodeng-au-Bosc	76
31155	364	\N	Hodeng-Hodenger	76
31156	365	\N	Houdetot	76
31157	366	\N	Houlme	76
31158	367	\N	Houppeville	76
31159	368	\N	Houquetot	76
31160	369	\N	Houssaye-Béranger	76
31161	370	\N	Hugleville-en-Caux	76
31162	371	\N	Ifs	76
31163	372	\N	Illois	76
31164	373	\N	Imbleville	76
31165	374	\N	Incheville	76
31166	375	\N	Ingouville	76
31167	376	\N	Intraville	76
31168	377	\N	Isneauville	76
31169	378	\N	Jumièges	76
31170	379	\N	Lamberville	76
31171	380	\N	Lammerville	76
31172	381	\N	Landes-Vieilles-et-Neuves	76
31173	382	\N	Lanquetot	76
31174	383	\N	Lestanville	76
31175	384	\N	Lillebonne	76
31176	385	\N	Limésy	76
31177	386	\N	Limpiville	76
31178	387	\N	Lindebeuf	76
31179	388	\N	Lintot	76
31180	389	\N	Lintot-les-Bois	76
31181	390	\N	Loges	76
31182	391	\N	Londe	76
31183	392	\N	Londinières	76
31184	393	\N	Longmesnil	76
31185	394	\N	Longroy	76
31186	395	\N	Longueil	76
31187	396	\N	Longuerue	76
31188	397	\N	Longueville-sur-Scie	76
31189	398	\N	Louvetot	76
31190	399	\N	Lucy	76
31191	400	\N	Luneray	76
31192	401	\N	Mailleraye-sur-Seine	76
31193	402	\N	Malaunay	76
31194	403	\N	Malleville-les-Grès	76
31195	404	\N	Manéglise	76
31196	405	\N	Manéhouville	76
31197	406	\N	Maniquerville	76
31198	407	\N	Manneville-ès-Plains	76
31199	408	\N	Manneville-la-Goupil	76
31200	409	\N	Mannevillette	76
31201	410	\N	Maromme	76
31202	411	\N	Marques	76
31203	412	\N	Martainville-Épreville	76
31204	413	\N	Martigny	76
31205	414	\N	Martin-Église	76
31206	415	\N	Massy	76
31207	416	\N	Mathonville	76
31208	417	\N	Maucomble	76
31209	418	\N	Maulévrier-Sainte-Gertrude	76
31210	419	\N	Mauny	76
31211	420	\N	Mauquenchy	76
31212	421	\N	Mélamare	76
31213	422	\N	Melleville	76
31214	423	\N	Ménerval	76
31215	424	\N	Ménonval	76
31216	425	\N	Mentheville	76
31217	426	\N	Mésangueville	76
31218	427	\N	Mesnières-en-Bray	76
31219	428	\N	Mesnil-Durdent	76
31220	429	\N	Mesnil-Esnard	76
31221	430	\N	Mesnil-Follemprise	76
31222	431	\N	Mesnil-Lieubray	76
31223	432	\N	Mesnil-Mauger	76
31224	433	\N	Mesnil-Panneville	76
31225	434	\N	Mesnil-Raoul	76
31226	435	\N	Mesnil-Réaume	76
31227	436	\N	Mesnil-sous-Jumièges	76
31228	437	\N	Meulers	76
31229	438	\N	Millebosc	76
31230	439	\N	Mirville	76
31231	440	\N	Molagnies	76
31232	441	\N	Monchaux-Soreng	76
31233	442	\N	Monchy-sur-Eu	76
31234	443	\N	Mont-Cauvaire	76
31235	444	\N	Mont-de-l'If	76
31236	445	\N	Montérolier	76
31237	446	\N	Montigny	76
31238	447	\N	Montivilliers	76
31239	448	\N	Montmain	76
31240	449	\N	Montreuil-en-Caux	76
31241	450	\N	Montroty	76
31242	451	\N	Mont-Saint-Aignan	76
31243	452	\N	Montville	76
31244	453	\N	Morgny-la-Pommeraye	76
31245	454	\N	Mortemer	76
31246	455	\N	Morville-sur-Andelle	76
31247	456	\N	Motteville	76
31248	457	\N	Moulineaux	76
31249	458	\N	Muchedent	76
31250	459	\N	Nesle-Hodeng	76
31251	460	\N	Nesle-Normandeuse	76
31252	461	\N	Neufbosc	76
31253	462	\N	Neufchâtel-en-Bray	76
31254	463	\N	Neuf-Marché	76
31255	464	\N	Neuville-Chant-d'Oisel	76
31256	465	\N	Neuville-Ferrières	76
31257	467	\N	Néville	76
31258	468	\N	Nointot	76
31259	469	\N	Nolléval	76
31260	470	\N	Normanville	76
31261	471	\N	Norville	76
31262	472	\N	Notre-Dame-d'Aliermont	76
31263	473	\N	Notre-Dame-de-Bliquetuit	76
31264	474	\N	Notre-Dame-de-Bondeville	76
31265	475	\N	Franqueville-Saint-Pierre	76
31266	476	\N	Notre-Dame-de-Gravenchon	76
31267	477	\N	Notre-Dame-du-Bec	76
31268	478	\N	Notre-Dame-du-Parc	76
31269	479	\N	Nullemont	76
31270	480	\N	Ocqueville	76
31271	481	\N	Octeville-sur-Mer	76
31272	482	\N	Offranville	76
31273	483	\N	Oherville	76
31274	484	\N	Oissel	76
31275	485	\N	Omonville	76
31276	486	\N	Orival	76
31277	487	\N	Osmoy-Saint-Valery	76
31278	488	\N	Ouainville	76
31279	489	\N	Oudalle	76
31280	490	\N	Ourville-en-Caux	76
31281	491	\N	Ouville-l'Abbaye	76
31282	492	\N	Ouville-la-Rivière	76
31283	493	\N	Paluel	76
31284	494	\N	Parc-d'Anxtot	76
31285	495	\N	Pavilly	76
31286	496	\N	Penly	76
31287	497	\N	Petit-Couronne	76
31288	498	\N	Petit-Quevilly	76
31289	499	\N	Petiville	76
31290	500	\N	Pierrecourt	76
31291	501	\N	Pierrefiques	76
31292	502	\N	Pierreval	76
31293	503	\N	Pissy-Pôville	76
31294	504	\N	Pleine-Sève	76
31295	505	\N	Pommereux	76
31296	506	\N	Pommeréval	76
31297	507	\N	Ponts-et-Marais	76
31298	508	\N	Poterie-Cap-d'Antifer	76
31299	509	\N	Préaux	76
31300	510	\N	Prétot-Vicquemare	76
31301	511	\N	Preuseville	76
31302	512	\N	Puisenval	76
31303	513	\N	Quevillon	76
31304	514	\N	Quévreville-la-Poterie	76
31305	515	\N	Quiberville	76
31306	516	\N	Quièvrecourt	76
31307	517	\N	Quincampoix	76
31308	518	\N	Raffetot	76
31309	519	\N	Rainfreville	76
31310	520	\N	Réalcamp	76
31311	521	\N	Rebets	76
31312	522	\N	Remuée	76
31313	523	\N	Rétonval	76
31314	524	\N	Reuville	76
31315	525	\N	Ricarville	76
31316	526	\N	Ricarville-du-Val	76
31317	527	\N	Richemont	76
31318	528	\N	Rieux	76
31319	529	\N	Riville	76
31320	530	\N	Robertot	76
31321	531	\N	Rocquefort	76
31322	532	\N	Rocquemont	76
31323	533	\N	Rogerville	76
31324	534	\N	Rolleville	76
31325	535	\N	Roncherolles-en-Bray	76
31326	536	\N	Roncherolles-sur-le-Vivier	76
31327	537	\N	Ronchois	76
31328	538	\N	Rosay	76
31329	540	\N	Rouen	76
31330	541	\N	Roumare	76
31331	542	\N	Routes	76
31332	543	\N	Rouville	76
31333	544	\N	Rouvray-Catillon	76
31334	545	\N	Rouxmesnil-Bouteilles	76
31335	546	\N	Royville	76
31336	547	\N	Rue-Saint-Pierre	76
31337	548	\N	Ry	76
31338	549	\N	Saâne-Saint-Just	76
31339	550	\N	Sahurs	76
31340	551	\N	Sainneville	76
31341	552	\N	Sainte-Adresse	76
31342	553	\N	Sainte-Agathe-d'Aliermont	76
31343	554	\N	Saint-Aignan-sur-Ry	76
31344	555	\N	Saint-André-sur-Cailly	76
31345	556	\N	Saint-Antoine-la-Forêt	76
31346	557	\N	Saint-Arnoult	76
31347	558	\N	Saint-Aubin-Celloville	76
31348	559	\N	Saint-Aubin-de-Crétot	76
31349	560	\N	Saint-Aubin-Épinay	76
31350	561	\N	Saint-Aubin-lès-Elbeuf	76
31351	562	\N	Saint-Aubin-le-Cauf	76
31352	563	\N	Saint-Aubin-Routot	76
31353	564	\N	Saint-Aubin-sur-Mer	76
31354	565	\N	Saint-Aubin-sur-Scie	76
31355	566	\N	Sainte-Austreberthe	76
31356	567	\N	Sainte-Beuve-en-Rivière	76
31357	568	\N	Saint-Clair-sur-les-Monts	76
31358	569	\N	Sainte-Colombe	76
31359	570	\N	Saint-Crespin	76
31360	571	\N	Sainte-Croix-sur-Buchy	76
31361	572	\N	Saint-Denis-d'Aclon	76
31362	573	\N	Saint-Denis-le-Thiboult	76
31363	574	\N	Saint-Denis-sur-Scie	76
31364	575	\N	Saint-Étienne-du-Rouvray	76
31365	576	\N	Saint-Eustache-la-Forêt	76
31366	577	\N	Sainte-Foy	76
31367	578	\N	Sainte-Geneviève	76
31368	580	\N	Saint-Georges-sur-Fontaine	76
31369	581	\N	Saint-Germain-des-Essourts	76
31370	582	\N	Saint-Germain-d'Étables	76
31371	583	\N	Saint-Germain-sous-Cailly	76
31372	584	\N	Saint-Germain-sur-Eaulne	76
31373	585	\N	Saint-Gilles-de-Crétot	76
31374	586	\N	Saint-Gilles-de-la-Neuville	76
31375	587	\N	Sainte-Hélène-Bondeville	76
31376	588	\N	Saint-Hellier	76
31377	589	\N	Saint-Honoré	76
31378	590	\N	Saint-Jacques-d'Aliermont	76
31379	591	\N	Saint-Jacques-sur-Darnétal	76
31380	592	\N	Saint-Jean-de-Folleville	76
31381	593	\N	Saint-Jean-de-la-Neuville	76
31382	594	\N	Saint-Jean-du-Cardonnay	76
31383	595	\N	Saint-Jouin-Bruneval	76
31384	596	\N	Saint-Laurent-de-Brèvedent	76
31385	597	\N	Saint-Laurent-en-Caux	76
31386	598	\N	Saint-Léger-aux-Bois	76
31387	599	\N	Saint-Léger-du-Bourg-Denis	76
31388	600	\N	Saint-Léonard	76
31389	602	\N	Saint-Maclou-de-Folleville	76
31390	603	\N	Saint-Maclou-la-Brière	76
31391	604	\N	Saint-Mards	76
31392	605	\N	Sainte-Marguerite-sur-Mer	76
31393	606	\N	Morienne	76
31394	607	\N	Sainte-Marguerite-sur-Fauville	76
31395	608	\N	Sainte-Marguerite-sur-Duclair	76
31396	609	\N	Sainte-Marie-au-Bosc	76
31397	610	\N	Sainte-Marie-des-Champs	76
31398	611	\N	Saint-Martin-aux-Arbres	76
31399	612	\N	Saint-Martin-au-Bosc	76
31400	613	\N	Saint-Martin-aux-Buneaux	76
31401	614	\N	Saint-Martin-de-Boscherville	76
31402	615	\N	Saint-Martin-du-Bec	76
31403	616	\N	Saint-Martin-du-Manoir	76
31404	617	\N	Saint-Martin-du-Vivier	76
31405	618	\N	Saint-Martin-en-Campagne	76
31406	619	\N	Saint-Martin-le-Gaillard	76
31407	620	\N	Saint-Martin-l'Hortier	76
31408	621	\N	Saint-Martin-Osmonville	76
31409	622	\N	Saint-Maurice-d'Ételan	76
31410	623	\N	Saint-Michel-d'Halescourt	76
31411	624	\N	Saint-Nicolas-d'Aliermont	76
31412	625	\N	Saint-Nicolas-de-Bliquetuit	76
31413	626	\N	Saint-Nicolas-de-la-Haie	76
31414	627	\N	Saint-Nicolas-de-la-Taille	76
31415	628	\N	Saint-Ouen-du-Breuil	76
31416	629	\N	Saint-Ouen-le-Mauger	76
31417	630	\N	Saint-Ouen-sous-Bailly	76
31418	631	\N	Saint-Paër	76
31419	632	\N	Saint-Pierre-Bénouville	76
31420	634	\N	Saint-Pierre-de-Manneville	76
31421	635	\N	Saint-Pierre-des-Jonquières	76
31422	636	\N	Saint-Pierre-de-Varengeville	76
31423	637	\N	Saint-Pierre-en-Port	76
31424	638	\N	Saint-Pierre-en-Val	76
31425	639	\N	Saint-Pierre-Lavis	76
31426	640	\N	Saint-Pierre-lès-Elbeuf	76
31427	641	\N	Saint-Pierre-le-Vieux	76
31428	642	\N	Saint-Pierre-le-Viger	76
31429	643	\N	Saint-Quentin-au-Bosc	76
31430	644	\N	Saint-Rémy-Boscrocourt	76
31431	645	\N	Saint-Riquier-en-Rivière	76
31432	646	\N	Saint-Riquier-ès-Plains	76
31433	647	\N	Saint-Romain-de-Colbosc	76
31434	648	\N	Saint-Saëns	76
31435	649	\N	Saint-Saire	76
31436	650	\N	Saint-Sauveur-d'Émalleville	76
31437	651	\N	Saint-Sylvain	76
31438	652	\N	Saint-Vaast-d'Équiqueville	76
31439	653	\N	Saint-Vaast-Dieppedalle	76
31440	654	\N	Saint-Vaast-du-Val	76
31441	655	\N	Saint-Valery-en-Caux	76
31442	656	\N	Saint-Victor-l'Abbaye	76
31443	657	\N	Saint-Vigor-d'Ymonville	76
31444	658	\N	Saint-Vincent-Cramesnil	76
31445	659	\N	Saint-Wandrille-Rançon	76
31446	660	\N	Sandouville	76
31447	662	\N	Sassetot-le-Malgardé	76
31448	663	\N	Sassetot-le-Mauconduit	76
31449	664	\N	Sasseville	76
31450	665	\N	Sauchay	76
31451	666	\N	Saumont-la-Poterie	76
31452	667	\N	Sauqueville	76
31453	668	\N	Saussay	76
31454	669	\N	Sausseuzemare-en-Caux	76
31455	670	\N	Senneville-sur-Fécamp	76
31456	671	\N	Sept-Meules	76
31457	672	\N	Serqueux	76
31458	673	\N	Servaville-Salmonville	76
31459	674	\N	Sévis	76
31460	675	\N	Sierville	76
31461	676	\N	Sigy-en-Bray	76
31462	677	\N	Smermesnil	76
31463	678	\N	Sommery	76
31464	679	\N	Sommesnil	76
31465	680	\N	Sorquainville	76
31466	681	\N	Sotteville-lès-Rouen	76
31467	682	\N	Sotteville-sous-le-Val	76
31468	683	\N	Sotteville-sur-Mer	76
31469	684	\N	Tancarville	76
31470	685	\N	Thérouldeville	76
31471	686	\N	Theuville-aux-Maillots	76
31472	688	\N	Thiergeville	76
31473	689	\N	Thiétreville	76
31474	690	\N	Thil-Manneville	76
31475	691	\N	Thil-Riberpré	76
31476	692	\N	Thiouville	76
31477	693	\N	Tilleul	76
31478	694	\N	Tocqueville-en-Caux	76
31479	695	\N	Tocqueville-les-Murs	76
31480	696	\N	Tocqueville-sur-Eu	76
31481	697	\N	Torcy-le-Grand	76
31482	698	\N	Torcy-le-Petit	76
31483	699	\N	Torp-Mesnil	76
31484	700	\N	Tôtes	76
31485	701	\N	Touffreville-la-Cable	76
31486	702	\N	Touffreville-la-Corbeline	76
31487	703	\N	Touffreville-sur-Eu	76
31488	704	\N	Tourville-la-Chapelle	76
31489	705	\N	Tourville-la-Rivière	76
31490	706	\N	Tourville-les-Ifs	76
31491	707	\N	Tourville-sur-Arques	76
31492	708	\N	Toussaint	76
31493	709	\N	Trait	76
31494	710	\N	Trémauville	76
31495	711	\N	Tréport	76
31496	712	\N	Trinité-du-Mont	76
31497	713	\N	Triquerville	76
31498	714	\N	Trois-Pierres	76
31499	715	\N	Trouville	76
31500	716	\N	Turretot	76
31501	717	\N	Val-de-la-Haye	76
31502	718	\N	Valliquerville	76
31503	719	\N	Valmont	76
31504	720	\N	Varengeville-sur-Mer	76
31505	721	\N	Varneville-Bretteville	76
31506	723	\N	Vassonville	76
31507	724	\N	Vatierville	76
31508	725	\N	Vattetot-sous-Beaumont	76
31509	726	\N	Vattetot-sur-Mer	76
31510	727	\N	Vatteville-la-Rue	76
31511	728	\N	Vaupalière	76
31512	729	\N	Veauville-lès-Baons	76
31513	730	\N	Veauville-lès-Quelles	76
31514	731	\N	Vénestanville	76
31515	732	\N	Butot-Vénesville	76
31516	733	\N	Ventes-Saint-Rémy	76
31517	734	\N	Vergetot	76
31518	735	\N	Veules-les-Roses	76
31519	736	\N	Veulettes-sur-Mer	76
31520	737	\N	Vibeuf	76
31521	738	\N	Vieux-Manoir	76
31522	739	\N	Vieux-Rouen-sur-Bresle	76
31523	740	\N	Vieux-Rue	76
31524	741	\N	Villainville	76
31525	742	\N	Villequier	76
31526	743	\N	Villers-Écalles	76
31527	744	\N	Villers-sous-Foucarmont	76
31528	745	\N	Villy-sur-Yères	76
31529	746	\N	Vinnemerville	76
31530	747	\N	Virville	76
31531	748	\N	Vittefleur	76
31532	749	\N	Wanchy-Capval	76
31533	750	\N	Yainville	76
31534	751	\N	Yébleron	76
31535	752	\N	Yerville	76
31536	753	\N	Ymare	76
31537	754	\N	Yport	76
31538	755	\N	Ypreville-Biville	76
31539	756	\N	Yquebeuf	76
31540	757	\N	Yvecrique	76
31541	758	\N	Yvetot	76
31542	759	\N	Yville-sur-Seine	76
31543	001	\N	Achères-la-Forêt	77
31544	002	\N	Amillis	77
31545	003	\N	Amponville	77
31546	004	\N	Andrezel	77
31547	005	\N	Annet-sur-Marne	77
31548	006	\N	Arbonne-la-Forêt	77
31549	007	\N	Argentières	77
31550	008	\N	Armentières-en-Brie	77
31551	009	\N	Arville	77
31552	010	\N	Aubepierre-Ozouer-le-Repos	77
31553	011	\N	Aufferville	77
31554	012	\N	Augers-en-Brie	77
31555	013	\N	Aulnoy	77
31556	014	\N	Avon	77
31557	015	\N	Baby	77
31558	016	\N	Bagneaux-sur-Loing	77
31559	018	\N	Bailly-Romainvilliers	77
31560	019	\N	Balloy	77
31561	020	\N	Bannost-Villegagnon	77
31562	021	\N	Barbey	77
31563	022	\N	Barbizon	77
31564	023	\N	Barcy	77
31565	024	\N	Bassevelle	77
31566	025	\N	Bazoches-lès-Bray	77
31567	026	\N	Beauchery-Saint-Martin	77
31568	027	\N	Beaumont-du-Gâtinais	77
31569	028	\N	Beautheil	77
31570	029	\N	Beauvoir	77
31571	030	\N	Bellot	77
31572	031	\N	Bernay-Vilbert	77
31573	032	\N	Beton-Bazoches	77
31574	033	\N	Bezalles	77
31575	034	\N	Blandy	77
31576	035	\N	Blennes	77
31577	036	\N	Boisdon	77
31578	037	\N	Bois-le-Roi	77
31579	038	\N	Boissettes	77
31580	039	\N	Boissise-la-Bertrand	77
31581	040	\N	Boissise-le-Roi	77
31582	041	\N	Boissy-aux-Cailles	77
31583	042	\N	Boissy-le-Châtel	77
31584	043	\N	Boitron	77
31585	044	\N	Bombon	77
31586	045	\N	Bougligny	77
31587	046	\N	Boulancourt	77
31588	047	\N	Bouleurs	77
31589	048	\N	Bourron-Marlotte	77
31590	049	\N	Boutigny	77
31591	050	\N	Bransles	77
31592	051	\N	Bray-sur-Seine	77
31593	052	\N	Bréau	77
31594	053	\N	Brie-Comte-Robert	77
31595	054	\N	Brosse-Montceaux	77
31596	055	\N	Brou-sur-Chantereine	77
31597	056	\N	Burcy	77
31598	057	\N	Bussières	77
31599	058	\N	Bussy-Saint-Georges	77
31600	059	\N	Bussy-Saint-Martin	77
31601	060	\N	Buthiers	77
31602	061	\N	Cannes-Écluse	77
31603	062	\N	Carnetin	77
31604	063	\N	Celle-sur-Morin	77
31605	065	\N	Cély	77
31606	066	\N	Cerneux	77
31607	067	\N	Cesson	77
31608	068	\N	Cessoy-en-Montois	77
31609	069	\N	Chailly-en-Bière	77
31610	070	\N	Chailly-en-Brie	77
31611	071	\N	Chaintreaux	77
31612	072	\N	Chalautre-la-Grande	77
31613	073	\N	Chalautre-la-Petite	77
31614	075	\N	Chalifert	77
31615	076	\N	Chalmaison	77
31616	077	\N	Chambry	77
31617	078	\N	Chamigny	77
31618	079	\N	Champagne-sur-Seine	77
31619	080	\N	Champcenest	77
31620	081	\N	Champdeuil	77
31621	082	\N	Champeaux	77
31622	083	\N	Champs-sur-Marne	77
31623	084	\N	Changis-sur-Marne	77
31624	085	\N	Chanteloup-en-Brie	77
31625	086	\N	Chapelle-Gauthier	77
31626	087	\N	Chapelle-Iger	77
31627	088	\N	Chapelle-la-Reine	77
31628	089	\N	Chapelle-Rablais	77
31629	090	\N	Chapelle-Saint-Sulpice	77
31630	091	\N	Chapelles-Bourbon	77
31631	093	\N	Chapelle-Moutils	77
31632	094	\N	Charmentray	77
31633	095	\N	Charny	77
31634	096	\N	Chartrettes	77
31635	097	\N	Chartronges	77
31636	098	\N	Châteaubleau	77
31637	099	\N	Château-Landon	77
31638	100	\N	Châtelet-en-Brie	77
31639	101	\N	Châtenay-sur-Seine	77
31640	102	\N	Châtenoy	77
31641	103	\N	Châtillon-la-Borde	77
31642	104	\N	Châtres	77
31643	106	\N	Chauffry	77
31644	107	\N	Chaumes-en-Brie	77
31645	108	\N	Chelles	77
31646	109	\N	Chenoise	77
31647	110	\N	Chenou	77
31648	111	\N	Chessy	77
31649	112	\N	Chevrainvilliers	77
31650	113	\N	Chevru	77
31651	114	\N	Chevry-Cossigny	77
31652	115	\N	Chevry-en-Sereine	77
31653	116	\N	Choisy-en-Brie	77
31654	117	\N	Citry	77
31655	118	\N	Claye-Souilly	77
31656	119	\N	Clos-Fontaine	77
31657	120	\N	Cocherel	77
31658	121	\N	Collégien	77
31659	122	\N	Combs-la-Ville	77
31660	123	\N	Compans	77
31661	124	\N	Conches-sur-Gondoire	77
31662	125	\N	Condé-Sainte-Libiaire	77
31663	126	\N	Congis-sur-Thérouanne	77
31664	127	\N	Coubert	77
31665	128	\N	Couilly-Pont-aux-Dames	77
31666	129	\N	Coulombs-en-Valois	77
31667	130	\N	Coulommes	77
31668	131	\N	Coulommiers	77
31669	132	\N	Coupvray	77
31670	133	\N	Courcelles-en-Bassée	77
31671	134	\N	Courchamp	77
31672	135	\N	Courpalay	77
31673	136	\N	Courquetaine	77
31674	137	\N	Courtacon	77
31675	138	\N	Courtomer	77
31676	139	\N	Courtry	77
31677	140	\N	Coutençon	77
31678	141	\N	Coutevroult	77
31679	142	\N	Crécy-la-Chapelle	77
31680	143	\N	Crégy-lès-Meaux	77
31681	144	\N	Crèvecur-en-Brie	77
31682	145	\N	Crisenoy	77
31683	146	\N	Croissy-Beaubourg	77
31684	147	\N	Croix-en-Brie	77
31685	148	\N	Crouy-sur-Ourcq	77
31686	149	\N	Cucharmoy	77
31687	150	\N	Cuisy	77
31688	151	\N	Dagny	77
31689	152	\N	Dammarie-les-Lys	77
31690	153	\N	Dammartin-en-Goële	77
31691	154	\N	Dammartin-sur-Tigeaux	77
31692	155	\N	Dampmart	77
31693	156	\N	Darvault	77
31694	157	\N	Dhuisy	77
31695	158	\N	Diant	77
31696	159	\N	Donnemarie-Dontilly	77
31697	161	\N	Dormelles	77
31698	162	\N	Doue	77
31699	163	\N	Douy-la-Ramée	77
31700	164	\N	Échouboulains	77
31701	165	\N	Écrennes	77
31702	166	\N	Écuelles	77
31703	167	\N	Égligny	77
31704	168	\N	Égreville	77
31705	169	\N	Émerainville	77
31706	170	\N	Épisy	77
31707	171	\N	Esbly	77
31708	172	\N	Esmans	77
31709	173	\N	Étrépilly	77
31710	174	\N	Everly	77
31711	175	\N	Évry-Grégy-sur-Yerre	77
31712	176	\N	Faremoutiers	77
31713	177	\N	Favières	77
31714	178	\N	Faÿ-lès-Nemours	77
31715	179	\N	Féricy	77
31716	180	\N	Férolles-Attilly	77
31717	181	\N	Ferrières-en-Brie	77
31718	182	\N	Ferté-Gaucher	77
31719	183	\N	Ferté-sous-Jouarre	77
31720	184	\N	Flagy	77
31721	185	\N	Fleury-en-Bière	77
31722	186	\N	Fontainebleau	77
31723	187	\N	Fontaine-Fourches	77
31724	188	\N	Fontaine-le-Port	77
31725	190	\N	Fontains	77
31726	191	\N	Fontenailles	77
31727	192	\N	Fontenay-Trésigny	77
31728	193	\N	Forfry	77
31729	194	\N	Forges	77
31730	195	\N	Fouju	77
31731	196	\N	Fresnes-sur-Marne	77
31732	197	\N	Frétoy	77
31733	198	\N	Fromont	77
31734	199	\N	Fublaines	77
31735	200	\N	Garentreville	77
31736	201	\N	Gastins	77
31737	202	\N	Genevraye	77
31738	203	\N	Germigny-l'Évêque	77
31739	204	\N	Germigny-sous-Coulombs	77
31740	205	\N	Gesvres-le-Chapitre	77
31741	206	\N	Giremoutiers	77
31742	207	\N	Gironville	77
31743	208	\N	Gouaix	77
31744	209	\N	Gouvernes	77
31745	210	\N	Grande-Paroisse	77
31746	211	\N	Grandpuits-Bailly-Carrois	77
31747	212	\N	Gravon	77
31748	214	\N	Gressy	77
31749	215	\N	Gretz-Armainvilliers	77
31750	216	\N	Grez-sur-Loing	77
31751	217	\N	Grisy-Suisnes	77
31752	218	\N	Grisy-sur-Seine	77
31753	219	\N	Guérard	77
31754	220	\N	Guercheville	77
31755	221	\N	Guermantes	77
31756	222	\N	Guignes	77
31757	223	\N	Gurcy-le-Châtel	77
31758	224	\N	Hautefeuille	77
31759	225	\N	Haute-Maison	77
31760	226	\N	Héricy	77
31761	227	\N	Hermé	77
31762	228	\N	Hondevilliers	77
31763	229	\N	Houssaye-en-Brie	77
31764	230	\N	Ichy	77
31765	231	\N	Isles-les-Meldeuses	77
31766	232	\N	Isles-lès-Villenoy	77
31767	233	\N	Iverny	77
31768	234	\N	Jablines	77
31769	235	\N	Jaignes	77
31770	236	\N	Jaulnes	77
31771	237	\N	Jossigny	77
31772	238	\N	Jouarre	77
31773	239	\N	Jouy-le-Châtel	77
31774	240	\N	Jouy-sur-Morin	77
31775	241	\N	Juilly	77
31776	242	\N	Jutigny	77
31777	243	\N	Lagny-sur-Marne	77
31778	244	\N	Larchant	77
31779	245	\N	Laval-en-Brie	77
31780	246	\N	Léchelle	77
31781	247	\N	Lescherolles	77
31782	248	\N	Lesches	77
31783	249	\N	Lésigny	77
31784	250	\N	Leudon-en-Brie	77
31785	251	\N	Lieusaint	77
31786	252	\N	Limoges-Fourches	77
31787	253	\N	Lissy	77
31788	254	\N	Liverdy-en-Brie	77
31789	255	\N	Livry-sur-Seine	77
31790	256	\N	Lizines	77
31791	257	\N	Lizy-sur-Ourcq	77
31792	258	\N	Lognes	77
31793	259	\N	Longperrier	77
31794	260	\N	Longueville	77
31795	261	\N	Lorrez-le-Bocage-Préaux	77
31796	262	\N	Louan-Villegruis-Fontaine	77
31797	263	\N	Luisetaines	77
31798	264	\N	Lumigny-Nesles-Ormeaux	77
31799	265	\N	Luzancy	77
31800	266	\N	Machault	77
31801	267	\N	Madeleine-sur-Loing	77
31802	268	\N	Magny-le-Hongre	77
31803	269	\N	Maincy	77
31804	270	\N	Maisoncelles-en-Brie	77
31805	271	\N	Maisoncelles-en-Gâtinais	77
31806	272	\N	Maison-Rouge	77
31807	273	\N	Marchémoret	77
31808	274	\N	Marcilly	77
31809	275	\N	Marêts	77
31810	276	\N	Mareuil-lès-Meaux	77
31811	277	\N	Marles-en-Brie	77
31812	278	\N	Marolles-en-Brie	77
31813	279	\N	Marolles-sur-Seine	77
31814	280	\N	Mary-sur-Marne	77
31815	281	\N	Mauperthuis	77
31816	282	\N	Mauregard	77
31817	283	\N	May-en-Multien	77
31818	284	\N	Meaux	77
31819	285	\N	Mée-sur-Seine	77
31820	286	\N	Meigneux	77
31821	287	\N	Meilleray	77
31822	288	\N	Melun	77
31823	289	\N	Melz-sur-Seine	77
31824	290	\N	Méry-sur-Marne	77
31825	291	\N	Mesnil-Amelot	77
31826	292	\N	Messy	77
31827	293	\N	Misy-sur-Yonne	77
31828	294	\N	Mitry-Mory	77
31829	295	\N	Moisenay	77
31830	296	\N	Moissy-Cramayel	77
31831	297	\N	Mondreville	77
31832	298	\N	Mons-en-Montois	77
31833	299	\N	Montarlot	77
31834	300	\N	Montceaux-lès-Meaux	77
31835	301	\N	Montceaux-lès-Provins	77
31836	302	\N	Montcourt-Fromonville	77
31837	303	\N	Montdauphin	77
31838	304	\N	Montenils	77
31839	305	\N	Montereau-Fault-Yonne	77
31840	306	\N	Montereau-sur-le-Jard	77
31841	307	\N	Montévrain	77
31842	308	\N	Montgé-en-Goële	77
31843	309	\N	Monthyon	77
31844	310	\N	Montigny-le-Guesdier	77
31845	311	\N	Montigny-Lencoup	77
31846	312	\N	Montigny-sur-Loing	77
31847	313	\N	Montmachoux	77
31848	314	\N	Montolivet	77
31849	315	\N	Montry	77
31850	316	\N	Moret-sur-Loing	77
31851	317	\N	Mormant	77
31852	318	\N	Mortcerf	77
31853	319	\N	Mortery	77
31854	320	\N	Mouroux	77
31855	321	\N	Mousseaux-lès-Bray	77
31856	322	\N	Moussy-le-Neuf	77
31857	323	\N	Moussy-le-Vieux	77
31858	325	\N	Mouy-sur-Seine	77
31859	326	\N	Nandy	77
31860	327	\N	Nangis	77
31861	328	\N	Nanteau-sur-Essonne	77
31862	329	\N	Nanteau-sur-Lunain	77
31863	330	\N	Nanteuil-lès-Meaux	77
31864	331	\N	Nanteuil-sur-Marne	77
31865	332	\N	Nantouillet	77
31866	333	\N	Nemours	77
31867	335	\N	Chauconin-Neufmontiers	77
31868	336	\N	Neufmoutiers-en-Brie	77
31869	337	\N	Noisiel	77
31870	338	\N	Noisy-Rudignon	77
31871	339	\N	Noisy-sur-École	77
31872	340	\N	Nonville	77
31873	341	\N	Noyen-sur-Seine	77
31874	342	\N	Obsonville	77
31875	343	\N	Ocquerre	77
31876	344	\N	Oissery	77
31877	345	\N	Orly-sur-Morin	77
31878	347	\N	Ormes-sur-Voulzie	77
31879	348	\N	Ormesson	77
31880	349	\N	Othis	77
31881	350	\N	Ozoir-la-Ferrière	77
31882	352	\N	Ozouer-le-Voulgis	77
31883	353	\N	Paley	77
31884	354	\N	Pamfou	77
31885	355	\N	Paroy	77
31886	356	\N	Passy-sur-Seine	77
31887	357	\N	Pécy	77
31888	358	\N	Penchard	77
31889	359	\N	Perthes	77
31890	360	\N	Pézarches	77
31891	361	\N	Pierre-Levée	77
31892	363	\N	Pin	77
31893	364	\N	Plessis-aux-Bois	77
31894	365	\N	Plessis-Feu-Aussoux	77
31895	366	\N	Plessis-l'Évêque	77
31896	367	\N	Plessis-Placy	77
31897	368	\N	Poigny	77
31898	369	\N	Poincy	77
31899	370	\N	Poligny	77
31900	371	\N	Pommeuse	77
31901	372	\N	Pomponne	77
31902	373	\N	Pontault-Combault	77
31903	374	\N	Pontcarré	77
31904	376	\N	Précy-sur-Marne	77
31905	377	\N	Presles-en-Brie	77
31906	378	\N	Pringy	77
31907	379	\N	Provins	77
31908	380	\N	Puisieux	77
31909	381	\N	Quiers	77
31910	382	\N	Quincy-Voisins	77
31911	383	\N	Rampillon	77
31912	384	\N	Réau	77
31913	385	\N	Rebais	77
31914	386	\N	Recloses	77
31915	387	\N	Remauville	77
31916	388	\N	Reuil-en-Brie	77
31917	389	\N	Rochette	77
31918	390	\N	Roissy-en-Brie	77
31919	391	\N	Rouilly	77
31920	392	\N	Rouvres	77
31921	393	\N	Rozay-en-Brie	77
31922	394	\N	Rubelles	77
31923	395	\N	Rumont	77
31924	396	\N	Rupéreux	77
31925	397	\N	Saâcy-sur-Marne	77
31926	398	\N	Sablonnières	77
31927	399	\N	Saint-Ange-le-Viel	77
31928	400	\N	Saint-Augustin	77
31929	401	\N	Sainte-Aulde	77
31930	402	\N	Saint-Barthélemy	77
31931	403	\N	Saint-Brice	77
31932	404	\N	Sainte-Colombe	77
31933	405	\N	Saint-Cyr-sur-Morin	77
31934	406	\N	Saint-Denis-lès-Rebais	77
31935	407	\N	Saint-Fargeau-Ponthierry	77
31936	408	\N	Saint-Fiacre	77
31937	409	\N	Saint-Germain-Laval	77
31938	410	\N	Saint-Germain-Laxis	77
31939	411	\N	Saint-Germain-sous-Doue	77
31940	412	\N	Saint-Germain-sur-École	77
31941	413	\N	Saint-Germain-sur-Morin	77
31942	414	\N	Saint-Hilliers	77
31943	415	\N	Saint-Jean-les-Deux-Jumeaux	77
31944	416	\N	Saint-Just-en-Brie	77
31945	417	\N	Saint-Léger	77
31946	418	\N	Saint-Loup-de-Naud	77
31947	419	\N	Saint-Mammès	77
31948	420	\N	Saint-Mard	77
31949	421	\N	Saint-Mars-Vieux-Maisons	77
31950	423	\N	Saint-Martin-des-Champs	77
31951	424	\N	Saint-Martin-du-Boschet	77
31952	425	\N	Saint-Martin-en-Bière	77
31953	426	\N	Saint-Méry	77
31954	427	\N	Saint-Mesmes	77
31955	428	\N	Saint-Ouen-en-Brie	77
31956	429	\N	Saint-Ouen-sur-Morin	77
31957	430	\N	Saint-Pathus	77
31958	431	\N	Saint-Pierre-lès-Nemours	77
31959	432	\N	Saint-Rémy-la-Vanne	77
31960	433	\N	Saints	77
31961	434	\N	Saint-Sauveur-lès-Bray	77
31962	435	\N	Saint-Sauveur-sur-École	77
31963	436	\N	Saint-Siméon	77
31964	437	\N	Saint-Soupplets	77
31965	438	\N	Saint-Thibault-des-Vignes	77
31966	439	\N	Salins	77
31967	440	\N	Sammeron	77
31968	441	\N	Samois-sur-Seine	77
31969	442	\N	Samoreau	77
31970	443	\N	Sancy	77
31971	444	\N	Sancy-lès-Provins	77
31972	445	\N	Savigny-le-Temple	77
31973	446	\N	Savins	77
31974	447	\N	Seine-Port	77
31975	448	\N	Sept-Sorts	77
31976	449	\N	Serris	77
31977	450	\N	Servon	77
31978	451	\N	Signy-Signets	77
31979	452	\N	Sigy	77
31980	453	\N	Sivry-Courtry	77
31981	454	\N	Sognolles-en-Montois	77
31982	455	\N	Soignolles-en-Brie	77
31983	456	\N	Soisy-Bouy	77
31984	457	\N	Solers	77
31985	458	\N	Souppes-sur-Loing	77
31986	459	\N	Sourdun	77
31987	460	\N	Tancrou	77
31988	461	\N	Thénisy	77
31989	462	\N	Thieux	77
31990	463	\N	Thomery	77
31991	464	\N	Thorigny-sur-Marne	77
31992	465	\N	Thoury-Férottes	77
31993	466	\N	Tigeaux	77
31994	467	\N	Tombe	77
31995	468	\N	Torcy	77
31996	469	\N	Touquin	77
31997	470	\N	Tournan-en-Brie	77
31998	471	\N	Tousson	77
31999	472	\N	Trétoire	77
32000	473	\N	Treuzy-Levelay	77
32001	474	\N	Trilbardou	77
32002	475	\N	Trilport	77
32003	476	\N	Trocy-en-Multien	77
32004	477	\N	Ury	77
32005	478	\N	Ussy-sur-Marne	77
32006	479	\N	Vaires-sur-Marne	77
32007	480	\N	Valence-en-Brie	77
32008	481	\N	Vanvillé	77
32009	482	\N	Varennes-sur-Seine	77
32010	483	\N	Varreddes	77
32011	484	\N	Vaucourtois	77
32012	485	\N	Vaudoué	77
32013	486	\N	Vaudoy-en-Brie	77
32014	487	\N	Vaux-le-Pénil	77
32015	489	\N	Vaux-sur-Lunain	77
32016	490	\N	Vendrest	77
32017	491	\N	Veneux-les-Sablons	77
32018	492	\N	Verdelot	77
32019	493	\N	Verneuil-l'Étang	77
32020	494	\N	Vernou-la-Celle-sur-Seine	77
32021	495	\N	Vert-Saint-Denis	77
32022	496	\N	Vieux-Champagne	77
32023	498	\N	Vignely	77
32024	500	\N	Villebéon	77
32025	501	\N	Villecerf	77
32026	504	\N	Villemaréchal	77
32027	505	\N	Villemareuil	77
32028	506	\N	Villemer	77
32029	507	\N	Villenauxe-la-Petite	77
32030	508	\N	Villeneuve-le-Comte	77
32031	509	\N	Villeneuve-les-Bordes	77
32032	510	\N	Villeneuve-Saint-Denis	77
32033	511	\N	Villeneuve-sous-Dammartin	77
32034	512	\N	Villeneuve-sur-Bellot	77
32035	513	\N	Villenoy	77
32036	514	\N	Villeparisis	77
32037	515	\N	Villeroy	77
32038	516	\N	Ville-Saint-Jacques	77
32039	517	\N	Villevaudé	77
32040	518	\N	Villiers-en-Bière	77
32041	519	\N	Villiers-Saint-Georges	77
32042	520	\N	Villiers-sous-Grez	77
32043	521	\N	Villiers-sur-Morin	77
32044	522	\N	Villiers-sur-Seine	77
32045	523	\N	Villuis	77
32046	524	\N	Vimpelles	77
32047	525	\N	Vinantes	77
32048	526	\N	Vincy-Manuvre	77
32049	527	\N	Voinsles	77
32050	528	\N	Voisenon	77
32051	529	\N	Voulangis	77
32052	530	\N	Voulton	77
32053	531	\N	Voulx	77
32054	532	\N	Vulaines-lès-Provins	77
32055	533	\N	Vulaines-sur-Seine	77
32056	534	\N	Yèbles	77
32057	003	\N	Ablis	78
32058	005	\N	Achères	78
32059	006	\N	Adainville	78
32060	007	\N	Aigremont	78
32061	009	\N	Allainville	78
32062	010	\N	Alluets-le-Roi	78
32063	013	\N	Andelu	78
32064	015	\N	Andrésy	78
32065	020	\N	Arnouville-lès-Mantes	78
32066	029	\N	Aubergenville	78
32067	030	\N	Auffargis	78
32068	031	\N	Auffreville-Brasseuil	78
32069	033	\N	Aulnay-sur-Mauldre	78
32070	034	\N	Auteuil	78
32071	036	\N	Autouillet	78
32072	043	\N	Bailly	78
32073	048	\N	Bazainville	78
32074	049	\N	Bazemont	78
32075	050	\N	Bazoches-sur-Guyonne	78
32076	053	\N	Béhoust	78
32077	057	\N	Bennecourt	78
32078	062	\N	Beynes	78
32079	068	\N	Blaru	78
32080	070	\N	Boinville-en-Mantois	78
32081	071	\N	Boinville-le-Gaillard	78
32082	072	\N	Boinvilliers	78
32083	073	\N	Bois-d'Arcy	78
32084	076	\N	Boissets	78
32085	077	\N	Boissière-École	78
32086	082	\N	Boissy-Mauvoisin	78
32087	084	\N	Boissy-sans-Avoir	78
32088	087	\N	Bonnelles	78
32089	089	\N	Bonnières-sur-Seine	78
32090	090	\N	Bouafle	78
32091	092	\N	Bougival	78
32092	096	\N	Bourdonné	78
32093	104	\N	Breuil-Bois-Robert	78
32094	107	\N	Bréval	78
32095	108	\N	Bréviaires	78
32096	113	\N	Brueil-en-Vexin	78
32097	117	\N	Buc	78
32098	118	\N	Buchelay	78
32099	120	\N	Bullion	78
32100	123	\N	Carrières-sous-Poissy	78
32101	124	\N	Carrières-sur-Seine	78
32102	125	\N	Celle-les-Bordes	78
32103	126	\N	Celle-Saint-Cloud	78
32104	128	\N	Cernay-la-Ville	78
32105	133	\N	Chambourcy	78
32106	138	\N	Chanteloup-les-Vignes	78
32107	140	\N	Chapet	78
32108	143	\N	Châteaufort	78
32109	146	\N	Chatou	78
32110	147	\N	Chaufour-lès-Bonnières	78
32111	152	\N	Chavenay	78
32112	158	\N	Chesnay	78
32113	160	\N	Chevreuse	78
32114	162	\N	Choisel	78
32115	163	\N	Civry-la-Forêt	78
32116	164	\N	Clairefontaine-en-Yvelines	78
32117	165	\N	Clayes-sous-Bois	78
32118	168	\N	Coignières	78
32119	171	\N	Condé-sur-Vesgre	78
32120	172	\N	Conflans-Sainte-Honorine	78
32121	185	\N	Courgent	78
32122	188	\N	Cravent	78
32123	189	\N	Crespières	78
32124	190	\N	Croissy-sur-Seine	78
32125	192	\N	Dammartin-en-Serve	78
32126	193	\N	Dampierre-en-Yvelines	78
32127	194	\N	Dannemarie	78
32128	196	\N	Davron	78
32129	202	\N	Drocourt	78
32130	206	\N	Ecquevilly	78
32131	208	\N	Élancourt	78
32132	209	\N	Émancé	78
32133	217	\N	Épône	78
32134	220	\N	Essarts-le-Roi	78
32135	224	\N	Étang-la-Ville	78
32136	227	\N	Évecquemont	78
32137	230	\N	Falaise	78
32138	231	\N	Favrieux	78
32139	233	\N	Feucherolles	78
32140	234	\N	Flacourt	78
32141	236	\N	Flexanville	78
32142	237	\N	Flins-Neuve-Église	78
32143	238	\N	Flins-sur-Seine	78
32144	239	\N	Follainville-Dennemont	78
32145	242	\N	Fontenay-le-Fleury	78
32146	245	\N	Fontenay-Mauvoisin	78
32147	246	\N	Fontenay-Saint-Père	78
32148	251	\N	Fourqueux	78
32149	255	\N	Freneuse	78
32150	261	\N	Gaillon-sur-Montcient	78
32151	262	\N	Galluis	78
32152	263	\N	Gambais	78
32153	264	\N	Gambaiseuil	78
32154	265	\N	Garancières	78
32155	267	\N	Gargenville	78
32156	269	\N	Gazeran	78
32157	276	\N	Gommecourt	78
32158	278	\N	Goupillières	78
32159	281	\N	Goussonville	78
32160	283	\N	Grandchamp	78
32161	285	\N	Gressey	78
32162	289	\N	Grosrouvre	78
32163	290	\N	Guernes	78
32164	291	\N	Guerville	78
32165	296	\N	Guitrancourt	78
32166	297	\N	Guyancourt	78
32167	299	\N	Hardricourt	78
32168	300	\N	Hargeville	78
32169	302	\N	Hauteville	78
32170	305	\N	Herbeville	78
32171	307	\N	Hermeray	78
32172	310	\N	Houdan	78
32173	311	\N	Houilles	78
32174	314	\N	Issou	78
32175	317	\N	Jambville	78
32176	320	\N	Jeufosse	78
32177	321	\N	Jouars-Pontchartrain	78
32178	322	\N	Jouy-en-Josas	78
32179	324	\N	Jouy-Mauvoisin	78
32180	325	\N	Jumeauville	78
32181	327	\N	Juziers	78
32182	329	\N	Lainville-en-Vexin	78
32183	334	\N	Lévis-Saint-Nom	78
32184	335	\N	Limay	78
32185	337	\N	Limetz-Villez	78
32186	343	\N	Loges-en-Josas	78
32187	344	\N	Lommoye	78
32188	346	\N	Longnes	78
32189	349	\N	Longvilliers	78
32190	350	\N	Louveciennes	78
32191	354	\N	Magnanville	78
32192	356	\N	Magny-les-Hameaux	78
32193	358	\N	Maisons-Laffitte	78
32194	361	\N	Mantes-la-Jolie	78
32195	362	\N	Mantes-la-Ville	78
32196	364	\N	Marcq	78
32197	366	\N	Mareil-le-Guyon	78
32198	367	\N	Mareil-Marly	78
32199	368	\N	Mareil-sur-Mauldre	78
32200	372	\N	Marly-le-Roi	78
32201	380	\N	Maule	78
32202	381	\N	Maulette	78
32203	382	\N	Maurecourt	78
32204	383	\N	Maurepas	78
32205	384	\N	Médan	78
32206	385	\N	Ménerville	78
32207	389	\N	Méré	78
32208	391	\N	Méricourt	78
32209	396	\N	Mesnil-le-Roi	78
32210	397	\N	Mesnil-Saint-Denis	78
32211	398	\N	Mesnuls	78
32212	401	\N	Meulan	78
32213	402	\N	Mézières-sur-Seine	78
32214	403	\N	Mézy-sur-Seine	78
32215	404	\N	Millemont	78
32216	406	\N	Milon-la-Chapelle	78
32217	407	\N	Mittainville	78
32218	410	\N	Moisson	78
32219	413	\N	Mondreville	78
32220	415	\N	Montainville	78
32221	416	\N	Montalet-le-Bois	78
32222	417	\N	Montchauvet	78
32223	418	\N	Montesson	78
32224	420	\N	Montfort-l'Amaury	78
32225	423	\N	Montigny-le-Bretonneux	78
32226	431	\N	Morainvilliers	78
32227	437	\N	Mousseaux-sur-Seine	78
32228	439	\N	Mulcent	78
32229	440	\N	Mureaux	78
32230	442	\N	Neauphle-le-Château	78
32231	443	\N	Neauphle-le-Vieux	78
32232	444	\N	Neauphlette	78
32233	451	\N	Nézel	78
32234	455	\N	Noisy-le-Roi	78
32235	460	\N	Oinville-sur-Montcient	78
32236	464	\N	Orcemont	78
32237	465	\N	Orgerus	78
32238	466	\N	Orgeval	78
32239	470	\N	Orphin	78
32240	472	\N	Orsonville	78
32241	474	\N	Orvilliers	78
32242	475	\N	Osmoy	78
32243	478	\N	Paray-Douaville	78
32244	481	\N	Pecq	78
32245	484	\N	Perdreauville	78
32246	486	\N	Perray-en-Yvelines	78
32247	490	\N	Plaisir	78
32248	497	\N	Poigny-la-Forêt	78
32249	498	\N	Poissy	78
32250	499	\N	Ponthévrard	78
32251	501	\N	Porcheville	78
32252	502	\N	Port-Marly	78
32253	503	\N	Port-Villez	78
32254	505	\N	Prunay-le-Temple	78
32255	506	\N	Prunay-en-Yvelines	78
32256	513	\N	Queue-les-Yvelines	78
32257	516	\N	Raizeux	78
32258	517	\N	Rambouillet	78
32259	518	\N	Rennemoulin	78
32260	520	\N	Richebourg	78
32261	522	\N	Rochefort-en-Yvelines	78
32262	524	\N	Rocquencourt	78
32263	528	\N	Rolleboise	78
32264	530	\N	Rosay	78
32265	531	\N	Rosny-sur-Seine	78
32266	536	\N	Sailly	78
32267	537	\N	Saint-Arnoult-en-Yvelines	78
32268	545	\N	Saint-Cyr-l'École	78
32269	548	\N	Saint-Forget	78
32270	550	\N	Saint-Germain-de-la-Grange	78
32271	551	\N	Saint-Germain-en-Laye	78
32272	557	\N	Saint-Hilarion	78
32273	558	\N	Saint-Illiers-la-Ville	78
32274	559	\N	Saint-Illiers-le-Bois	78
32275	561	\N	Saint-Lambert	78
32276	562	\N	Saint-Léger-en-Yvelines	78
32277	564	\N	Saint-Martin-de-Bréthencourt	78
32278	565	\N	Saint-Martin-des-Champs	78
32279	567	\N	Saint-Martin-la-Garenne	78
32280	569	\N	Sainte-Mesme	78
32281	571	\N	Saint-Nom-la-Bretèche	78
32282	575	\N	Saint-Rémy-lès-Chevreuse	78
32283	576	\N	Saint-Rémy-l'Honoré	78
32284	586	\N	Sartrouville	78
32285	588	\N	Saulx-Marchais	78
32286	590	\N	Senlisse	78
32287	591	\N	Septeuil	78
32288	597	\N	Soindres	78
32289	601	\N	Sonchamp	78
32290	605	\N	Tacoignières	78
32291	606	\N	Tartre-Gaudran	78
32292	608	\N	Tertre-Saint-Denis	78
32293	609	\N	Tessancourt-sur-Aubette	78
32294	615	\N	Thiverval-Grignon	78
32295	616	\N	Thoiry	78
32296	618	\N	Tilly	78
32297	620	\N	Toussus-le-Noble	78
32298	621	\N	Trappes	78
32299	623	\N	Tremblay-sur-Mauldre	78
32300	624	\N	Triel-sur-Seine	78
32301	638	\N	Vaux-sur-Seine	78
32302	640	\N	Vélizy-Villacoublay	78
32303	642	\N	Verneuil-sur-Seine	78
32304	643	\N	Vernouillet	78
32305	644	\N	Verrière	78
32306	646	\N	Versailles	78
32307	647	\N	Vert	78
32308	650	\N	Vésinet	78
32309	653	\N	Vicq	78
32310	655	\N	Vieille-Église-en-Yvelines	78
32311	668	\N	Villeneuve-en-Chevrie	78
32312	672	\N	Villennes-sur-Seine	78
32313	674	\N	Villepreux	78
32314	677	\N	Villette	78
32315	681	\N	Villiers-le-Mahieu	78
32316	683	\N	Villiers-Saint-Fréderic	78
32317	686	\N	Viroflay	78
32318	688	\N	Voisins-le-Bretonneux	78
32319	001	\N	Absie	79
32320	002	\N	Adilly	79
32321	003	\N	Aiffres	79
32322	004	\N	Aigonnay	79
32323	005	\N	Airvault	79
32324	006	\N	Alleuds	79
32325	007	\N	Allonne	79
32326	008	\N	Amailloux	79
32327	009	\N	Amuré	79
32328	010	\N	Arçais	79
32329	011	\N	Ardilleux	79
32330	012	\N	Ardin	79
32331	013	\N	Argenton-Château	79
32332	014	\N	Argenton-l'Église	79
32333	015	\N	Asnières-en-Poitou	79
32334	016	\N	Assais-les-Jumeaux	79
32335	018	\N	Aubigné	79
32336	019	\N	Aubigny	79
32337	020	\N	Augé	79
32338	022	\N	Availles-Thouarsais	79
32339	023	\N	Avon	79
32340	024	\N	Azay-le-Brûlé	79
32341	025	\N	Azay-sur-Thouet	79
32342	027	\N	Bataille	79
32343	029	\N	Beaulieu-sous-Parthenay	79
32344	030	\N	Beaussais	79
32345	031	\N	Beauvoir-sur-Niort	79
32346	032	\N	Béceleuf	79
32347	033	\N	Belleville	79
32348	034	\N	Bessines	79
32349	035	\N	Beugnon	79
32350	037	\N	Boësse	79
32351	038	\N	Boismé	79
32352	039	\N	Boisserolles	79
32353	040	\N	Boissière-en-Gâtine	79
32354	042	\N	Bougon	79
32355	043	\N	Bouillé-Loretz	79
32356	044	\N	Bouillé-Saint-Paul	79
32357	045	\N	Bouin	79
32358	046	\N	Bourdet	79
32359	047	\N	Boussais	79
32360	048	\N	Crèche	79
32361	049	\N	Bressuire	79
32362	050	\N	Bretignolles	79
32363	051	\N	Breuil-Bernard	79
32364	053	\N	Breuil-sous-Argenton	79
32365	054	\N	Brie	79
32366	055	\N	Brieuil-sur-Chizé	79
32367	056	\N	Brion-près-Thouet	79
32368	057	\N	Brioux-sur-Boutonne	79
32369	058	\N	Brûlain	79
32370	059	\N	Busseau	79
32371	060	\N	Caunay	79
32372	061	\N	Celles-sur-Belle	79
32373	062	\N	Cerizay	79
32374	063	\N	Cersay	79
32375	064	\N	Chail	79
32376	066	\N	Champdeniers-Saint-Denis	79
32377	068	\N	Chantecorps	79
32378	069	\N	Chanteloup	79
32379	070	\N	Chapelle-Bâton	79
32380	071	\N	Chapelle-Bertrand	79
32381	072	\N	Chapelle-Gaudin	79
32382	074	\N	Chapelle-Pouilloux	79
32383	075	\N	Chapelle-Saint-Étienne	79
32384	076	\N	Chapelle-Saint-Laurent	79
32385	077	\N	Chapelle-Thireuil	79
32386	078	\N	Prissé-la-Charrière	79
32387	079	\N	Mauléon	79
32388	080	\N	Châtillon-sur-Thouet	79
32389	081	\N	Chauray	79
32390	083	\N	Chef-Boutonne	79
32391	084	\N	Chenay	79
32392	085	\N	Chérigné	79
32393	086	\N	Cherveux	79
32394	087	\N	Chey	79
32395	088	\N	Chiché	79
32396	089	\N	Chillou	79
32397	090	\N	Chizé	79
32398	091	\N	Cirières	79
32399	092	\N	Clavé	79
32400	094	\N	Clessé	79
32401	095	\N	Clussais-la-Pommeraie	79
32402	096	\N	Combrand	79
32403	098	\N	Couarde	79
32404	099	\N	Coudre	79
32405	100	\N	Coulon	79
32406	101	\N	Coulonges-sur-l'Autize	79
32407	102	\N	Coulonges-Thouarsais	79
32408	103	\N	Courlay	79
32409	104	\N	Cours	79
32410	105	\N	Coutières	79
32411	106	\N	Couture-d'Argenson	79
32412	107	\N	Crézières	79
32413	108	\N	Doux	79
32414	109	\N	Échiré	79
32415	111	\N	Ensigné	79
32416	112	\N	Épannes	79
32417	113	\N	Étusson	79
32418	114	\N	Exireuil	79
32419	115	\N	Exoudun	79
32420	116	\N	Faye-l'Abbesse	79
32421	117	\N	Faye-sur-Ardin	79
32422	118	\N	Fénery	79
32423	119	\N	Fenioux	79
32424	120	\N	Ferrière-en-Parthenay	79
32425	121	\N	Fomperron	79
32426	122	\N	Fontenille-Saint-Martin-d'Entraigues	79
32427	123	\N	Forêt-sur-Sèvre	79
32428	124	\N	Forges	79
32429	125	\N	Fors	79
32430	126	\N	Fosses	79
32431	127	\N	Foye-Monjault	79
32432	128	\N	François	79
32433	129	\N	Fressines	79
32434	130	\N	Frontenay-Rohan-Rohan	79
32435	131	\N	Geay	79
32436	132	\N	Genneton	79
32437	133	\N	Germond-Rouvre	79
32438	134	\N	Glénay	79
32439	135	\N	Gourgé	79
32440	136	\N	Gournay-Loizé	79
32441	137	\N	Granzay-Gript	79
32442	139	\N	Groseillers	79
32443	140	\N	Hanc	79
32444	141	\N	Irais	79
32445	142	\N	Juillé	79
32446	144	\N	Juscorps	79
32447	145	\N	Lageon	79
32448	147	\N	Largeasse	79
32449	148	\N	Lezay	79
32450	149	\N	Lhoumois	79
32451	150	\N	Limalonges	79
32452	152	\N	Lorigné	79
32453	153	\N	Loubigné	79
32454	154	\N	Loubillé	79
32455	156	\N	Louin	79
32456	157	\N	Louzy	79
32457	158	\N	Luché-sur-Brioux	79
32458	159	\N	Luché-Thouarsais	79
32459	160	\N	Lusseray	79
32460	161	\N	Luzay	79
32461	162	\N	Magné	79
32462	163	\N	Mairé-Levescault	79
32463	164	\N	Maisonnay	79
32464	165	\N	Maisontiers	79
32465	166	\N	Marigny	79
32466	167	\N	Marnes	79
32467	168	\N	Massais	79
32468	170	\N	Mauzé-sur-le-Mignon	79
32469	171	\N	Mauzé-Thouarsais	79
32470	172	\N	Mazières-en-Gâtine	79
32471	173	\N	Mazières-sur-Béronne	79
32472	174	\N	Melle	79
32473	175	\N	Melleran	79
32474	176	\N	Ménigoute	79
32475	177	\N	Messé	79
32476	178	\N	Missé	79
32477	179	\N	Moncoutant	79
32478	180	\N	Montalembert	79
32479	183	\N	Montravers	79
32480	184	\N	Mothe-Saint-Héray	79
32481	185	\N	Mougon	79
32482	187	\N	Moutiers-sous-Argenton	79
32483	188	\N	Moutiers-sous-Chantemerle	79
32484	189	\N	Nanteuil	79
32485	190	\N	Neuvy-Bouin	79
32486	191	\N	Niort	79
32487	195	\N	Nueil-les-Aubiers	79
32488	196	\N	Oiron	79
32489	197	\N	Oroux	79
32490	198	\N	Paizay-le-Chapt	79
32491	199	\N	Paizay-le-Tort	79
32492	200	\N	Pamplie	79
32493	201	\N	Pamproux	79
32494	202	\N	Parthenay	79
32495	203	\N	Pas-de-Jeu	79
32496	204	\N	Périgné	79
32497	205	\N	Pers	79
32498	207	\N	Petite-Boissière	79
32499	208	\N	Peyratte	79
32500	209	\N	Pierrefitte	79
32501	210	\N	Pin	79
32502	211	\N	Pioussay	79
32503	212	\N	Pliboux	79
32504	213	\N	Pompaire	79
32505	214	\N	Pouffonds	79
32506	215	\N	Pougne-Hérisson	79
32507	216	\N	Prahecq	79
32508	217	\N	Prailles	79
32509	218	\N	Pressigny	79
32510	219	\N	Priaires	79
32511	220	\N	Prin-Deyrançon	79
32512	222	\N	Pugny	79
32513	223	\N	Puihardy	79
32514	225	\N	Reffannes	79
32515	226	\N	Retail	79
32516	229	\N	Rochénard	79
32517	230	\N	Rom	79
32518	231	\N	Romans	79
32519	235	\N	Saint-Amand-sur-Sèvre	79
32520	236	\N	Saint-André-sur-Sèvre	79
32521	238	\N	Saint-Aubin-du-Plain	79
32522	239	\N	Saint-Aubin-le-Cloud	79
32523	240	\N	Sainte-Blandine	79
32524	241	\N	Saint-Christophe-sur-Roc	79
32525	242	\N	Saint-Clémentin	79
32526	243	\N	Saint-Coutant	79
32527	244	\N	Saint-Cyr-la-Lande	79
32528	246	\N	Sainte-Eanne	79
32529	247	\N	Saint-Étienne-la-Cigogne	79
32530	249	\N	Saint-Gelais	79
32531	250	\N	Sainte-Gemme	79
32532	251	\N	Saint-Génard	79
32533	252	\N	Saint-Généroux	79
32534	253	\N	Saint-Georges-de-Noisné	79
32535	254	\N	Saint-Georges-de-Rex	79
32536	255	\N	Saint-Germain-de-Longue-Chaume	79
32537	256	\N	Saint-Germier	79
32538	257	\N	Saint-Hilaire-la-Palud	79
32539	258	\N	Saint-Jacques-de-Thouars	79
32540	259	\N	Saint-Jean-de-Thouars	79
32541	260	\N	Saint-Jouin-de-Marnes	79
32542	261	\N	Saint-Jouin-de-Milly	79
32543	263	\N	Saint-Laurs	79
32544	264	\N	Saint-Léger-de-la-Martinière	79
32545	265	\N	Saint-Léger-de-Montbrun	79
32546	267	\N	Saint-Lin	79
32547	268	\N	Saint-Loup-Lamairé	79
32548	269	\N	Saint-Maixent-de-Beugné	79
32549	270	\N	Saint-Maixent-l'École	79
32550	271	\N	Saint-Marc-la-Lande	79
32551	273	\N	Saint-Martin-de-Bernegoue	79
32552	274	\N	Saint-Martin-de-Mâcon	79
32553	276	\N	Saint-Martin-de-Saint-Maixent	79
32554	277	\N	Saint-Martin-de-Sanzay	79
32555	278	\N	Saint-Martin-du-Fouilloux	79
32556	279	\N	Saint-Martin-lès-Melle	79
32557	280	\N	Saint-Maurice-la-Fougereuse	79
32558	281	\N	Saint-Maxire	79
32559	282	\N	Saint-Médard	79
32560	283	\N	Sainte-Néomaye	79
32561	284	\N	Sainte-Ouenne	79
32562	285	\N	Saint-Pardoux	79
32563	286	\N	Saint-Paul-en-Gâtine	79
32564	289	\N	Saint-Pierre-des-Échaubrognes	79
32565	290	\N	Saint-Pompain	79
32566	292	\N	Sainte-Radegonde	79
32567	293	\N	Saint-Rémy	79
32568	294	\N	Saint-Romans-des-Champs	79
32569	295	\N	Saint-Romans-lès-Melle	79
32570	297	\N	Sainte-Soline	79
32571	298	\N	Saint-Symphorien	79
32572	299	\N	Saint-Varent	79
32573	300	\N	Sainte-Verge	79
32574	301	\N	Saint-Vincent-la-Châtre	79
32575	302	\N	Saivres	79
32576	303	\N	Salles	79
32577	304	\N	Sansais	79
32578	305	\N	Sanzay	79
32579	306	\N	Saurais	79
32580	307	\N	Sauzé-Vaussais	79
32581	308	\N	Sciecq	79
32582	309	\N	Scillé	79
32583	310	\N	Secondigné-sur-Belle	79
32584	311	\N	Secondigny	79
32585	312	\N	Séligné	79
32586	313	\N	Sepvret	79
32587	314	\N	Sompt	79
32588	316	\N	Soudan	79
32589	318	\N	Soutiers	79
32590	319	\N	Souvigné	79
32591	320	\N	Surin	79
32592	321	\N	Taizé	79
32593	322	\N	Tallud	79
32594	325	\N	Tessonnière	79
32595	326	\N	Thénezay	79
32596	327	\N	Thorigné	79
32597	328	\N	Thorigny-sur-le-Mignon	79
32598	329	\N	Thouars	79
32599	330	\N	Tillou	79
32600	331	\N	Tourtenay	79
32601	332	\N	Trayes	79
32602	333	\N	Ulcot	79
32603	334	\N	Usseau	79
32604	335	\N	Vallans	79
32605	336	\N	Vançais	79
32606	337	\N	Vanneau-Irleau	79
32607	338	\N	Vanzay	79
32608	339	\N	Vasles	79
32609	340	\N	Vausseroux	79
32610	341	\N	Vautebis	79
32611	342	\N	Vernoux-en-Gâtine	79
32612	343	\N	Vernoux-sur-Boutonne	79
32613	345	\N	Verruyes	79
32614	346	\N	Vert	79
32615	347	\N	Viennay	79
32616	348	\N	Villefollet	79
32617	349	\N	Villemain	79
32618	350	\N	Villiers-en-Bois	79
32619	351	\N	Villiers-en-Plaine	79
32620	352	\N	Villiers-sur-Chizé	79
32621	353	\N	Vitré	79
32622	354	\N	Vouhé	79
32623	355	\N	Vouillé	79
32624	356	\N	Voultegon	79
32625	357	\N	Xaintray	79
32626	001	\N	Abbeville	80
32627	002	\N	Ablaincourt-Pressoir	80
32628	003	\N	Acheux-en-Amiénois	80
32629	004	\N	Acheux-en-Vimeu	80
32630	005	\N	Agenville	80
32631	006	\N	Agenvillers	80
32632	008	\N	Aigneville	80
32633	009	\N	Ailly-le-Haut-Clocher	80
32634	010	\N	Ailly-sur-Noye	80
32635	011	\N	Ailly-sur-Somme	80
32636	013	\N	Airaines	80
32637	014	\N	Aizecourt-le-Bas	80
32638	015	\N	Aizecourt-le-Haut	80
32639	016	\N	Albert	80
32640	017	\N	Allaines	80
32641	018	\N	Allenay	80
32642	019	\N	Allery	80
32643	020	\N	Allonville	80
32644	021	\N	Amiens	80
32645	022	\N	Andainville	80
32646	023	\N	Andechy	80
32647	024	\N	Arguves	80
32648	025	\N	Argoules	80
32649	026	\N	Arguel	80
32650	027	\N	Armancourt	80
32651	028	\N	Arquèves	80
32652	029	\N	Arrest	80
32653	030	\N	Arry	80
32654	031	\N	Arvillers	80
32655	032	\N	Assainvillers	80
32656	033	\N	Assevillers	80
32657	034	\N	Athies	80
32658	035	\N	Aubercourt	80
32659	036	\N	Aubigny	80
32660	037	\N	Aubvillers	80
32661	038	\N	Auchonvillers	80
32662	039	\N	Ault	80
32663	040	\N	Aumâtre	80
32664	041	\N	Aumont	80
32665	042	\N	Autheux	80
32666	043	\N	Authie	80
32667	044	\N	Authieule	80
32668	045	\N	Authuille	80
32669	046	\N	Avelesges	80
32670	047	\N	Aveluy	80
32671	048	\N	Avesnes-Chaussoy	80
32672	049	\N	Ayencourt	80
32673	050	\N	Bacouel-sur-Selle	80
32674	051	\N	Bailleul	80
32675	052	\N	Baizieux	80
32676	053	\N	Balâtre	80
32677	054	\N	Barleux	80
32678	055	\N	Barly	80
32679	056	\N	Bavelincourt	80
32680	057	\N	Bayencourt	80
32681	058	\N	Bayonvillers	80
32682	059	\N	Bazentin	80
32683	060	\N	Béalcourt	80
32684	061	\N	Beaucamps-le-Jeune	80
32685	062	\N	Beaucamps-le-Vieux	80
32686	063	\N	Beauchamps	80
32687	064	\N	Beaucourt-en-Santerre	80
32688	065	\N	Beaucourt-sur-l'Ancre	80
32689	066	\N	Beaucourt-sur-l'Hallue	80
32690	067	\N	Beaufort-en-Santerre	80
32691	068	\N	Beaumetz	80
32692	069	\N	Beaumont-Hamel	80
32693	070	\N	Beauquesne	80
32694	071	\N	Beauval	80
32695	073	\N	Bécordel-Bécourt	80
32696	074	\N	Becquigny	80
32697	076	\N	Béhen	80
32698	077	\N	Béhencourt	80
32699	078	\N	Bellancourt	80
32700	079	\N	Belleuse	80
32701	080	\N	Belloy-en-Santerre	80
32702	081	\N	Belloy-Saint-Léonard	80
32703	082	\N	Belloy-sur-Somme	80
32704	083	\N	Bergicourt	80
32705	084	\N	Bermesnil	80
32706	085	\N	Bernâtre	80
32707	086	\N	Bernaville	80
32708	087	\N	Bernay-en-Ponthieu	80
32709	088	\N	Bernes	80
32710	089	\N	Berneuil	80
32711	090	\N	Berny-en-Santerre	80
32712	092	\N	Bertangles	80
32713	093	\N	Berteaucourt-les-Dames	80
32714	094	\N	Berteaucourt-lès-Thennes	80
32715	095	\N	Bertrancourt	80
32716	096	\N	Béthencourt-sur-Mer	80
32717	097	\N	Béthencourt-sur-Somme	80
32718	098	\N	Bettembos	80
32719	099	\N	Bettencourt-Rivière	80
32720	100	\N	Bettencourt-Saint-Ouen	80
32721	101	\N	Beuvraignes	80
32722	102	\N	Biaches	80
32723	103	\N	Biarre	80
32724	104	\N	Biencourt	80
32725	105	\N	Billancourt	80
32726	106	\N	Blangy-sous-Poix	80
32727	107	\N	Blangy-Tronville	80
32728	108	\N	Boisbergues	80
32729	109	\N	Boisle	80
32730	110	\N	Boismont	80
32731	112	\N	Bonnay	80
32732	113	\N	Bonneville	80
32733	114	\N	Bosquel	80
32734	115	\N	Bouchavesnes-Bergen	80
32735	116	\N	Bouchoir	80
32736	117	\N	Bouchon	80
32737	118	\N	Boufflers	80
32738	119	\N	Bougainville	80
32739	120	\N	Bouillancourt-en-Séry	80
32740	121	\N	Bouillancourt-la-Bataille	80
32741	122	\N	Bouquemaison	80
32742	123	\N	Bourdon	80
32743	124	\N	Bourseville	80
32744	125	\N	Boussicourt	80
32745	126	\N	Bouttencourt	80
32746	127	\N	Bouvaincourt-sur-Bresle	80
32747	128	\N	Bouvincourt-en-Vermandois	80
32748	129	\N	Bouzincourt	80
32749	130	\N	Bovelles	80
32750	131	\N	Boves	80
32751	132	\N	Braches	80
32752	133	\N	Brailly-Cornehotte	80
32753	134	\N	Brassy	80
32754	135	\N	Bray-lès-Mareuil	80
32755	136	\N	Bray-sur-Somme	80
32756	137	\N	Breilly	80
32757	138	\N	Bresle	80
32758	139	\N	Breuil	80
32759	140	\N	Brévillers	80
32760	141	\N	Brie	80
32761	142	\N	Briquemesnil-Floxicourt	80
32762	143	\N	Brocourt	80
32763	144	\N	Brouchy	80
32764	145	\N	Brucamps	80
32765	146	\N	Brutelles	80
32766	147	\N	Buigny-l'Abbé	80
32767	148	\N	Buigny-lès-Gamaches	80
32768	149	\N	Buigny-Saint-Maclou	80
32769	150	\N	Buire-Courcelles	80
32770	151	\N	Buire-sur-l'Ancre	80
32771	152	\N	Bus-la-Mésière	80
32772	153	\N	Bus-lès-Artois	80
32773	154	\N	Bussu	80
32774	155	\N	Bussus-Bussuel	80
32775	156	\N	Bussy-lès-Daours	80
32776	157	\N	Bussy-lès-Poix	80
32777	158	\N	Buverchy	80
32778	159	\N	Cachy	80
32779	160	\N	Cagny	80
32780	161	\N	Cahon	80
32781	162	\N	Caix	80
32782	163	\N	Cambron	80
32783	164	\N	Camon	80
32784	165	\N	Camps-en-Amiénois	80
32785	166	\N	Canaples	80
32786	167	\N	Canchy	80
32787	168	\N	Candas	80
32788	169	\N	Cannessières	80
32789	170	\N	Cantigny	80
32790	171	\N	Caours	80
32791	172	\N	Cappy	80
32792	173	\N	Cardonnette	80
32793	174	\N	Cardonnois	80
32794	175	\N	Carnoy	80
32795	176	\N	Carrépuis	80
32796	177	\N	Cartigny	80
32797	179	\N	Caulières	80
32798	180	\N	Cavillon	80
32799	181	\N	Cayeux-en-Santerre	80
32800	182	\N	Cayeux-sur-Mer	80
32801	183	\N	Cerisy-Buleux	80
32802	184	\N	Cerisy	80
32803	185	\N	Champien	80
32804	186	\N	Chaulnes	80
32805	187	\N	Chaussée-Tirancourt	80
32806	188	\N	Chaussoy-Epagny	80
32807	189	\N	Chavatte	80
32808	190	\N	Chépy	80
32809	191	\N	Chilly	80
32810	192	\N	Chipilly	80
32811	193	\N	Chirmont	80
32812	194	\N	Chuignes	80
32813	195	\N	Chuignolles	80
32814	196	\N	Citerne	80
32815	197	\N	Cizancourt	80
32816	198	\N	Clairy-Saulchoix	80
32817	199	\N	Cléry-sur-Somme	80
32818	200	\N	Cocquerel	80
32819	201	\N	Coigneux	80
32820	202	\N	Coisy	80
32821	203	\N	Colincamps	80
32822	204	\N	Combles	80
32823	205	\N	Condé-Folie	80
32824	206	\N	Contalmaison	80
32825	207	\N	Contay	80
32826	208	\N	Conteville	80
32827	209	\N	Contoire	80
32828	210	\N	Contre	80
32829	211	\N	Conty	80
32830	212	\N	Corbie	80
32831	213	\N	Cottenchy	80
32832	214	\N	Coullemelle	80
32833	215	\N	Coulonvillers	80
32834	216	\N	Courcelette	80
32835	217	\N	Courcelles-au-Bois	80
32836	218	\N	Courcelles-sous-Moyencourt	80
32837	219	\N	Courcelles-sous-Thoix	80
32838	220	\N	Courtemanche	80
32839	221	\N	Cramont	80
32840	222	\N	Crécy-en-Ponthieu	80
32841	223	\N	Crémery	80
32842	224	\N	Cressy-Omencourt	80
32843	225	\N	Creuse	80
32844	226	\N	Croix-Moligneaux	80
32845	227	\N	Croixrault	80
32846	228	\N	Crotoy	80
32847	229	\N	Crouy-Saint-Pierre	80
32848	230	\N	Curchy	80
32849	231	\N	Curlu	80
32850	232	\N	Damery	80
32851	233	\N	Dancourt-Popincourt	80
32852	234	\N	Daours	80
32853	235	\N	Dargnies	80
32854	236	\N	Davenescourt	80
32855	237	\N	Démuin	80
32856	238	\N	Dernancourt	80
32857	239	\N	Devise	80
32858	240	\N	Doingt	80
32859	241	\N	Domart-en-Ponthieu	80
32860	242	\N	Domart-sur-la-Luce	80
32861	243	\N	Domesmont	80
32862	244	\N	Dominois	80
32863	245	\N	Domléger-Longvillers	80
32864	246	\N	Dommartin	80
32865	247	\N	Dompierre-Becquincourt	80
32866	248	\N	Dompierre-sur-Authie	80
32867	249	\N	Domqueur	80
32868	250	\N	Domvast	80
32869	251	\N	Doudelainville	80
32870	252	\N	Douilly	80
32871	253	\N	Doullens	80
32872	256	\N	Dreuil-lès-Amiens	80
32873	258	\N	Driencourt	80
32874	259	\N	Dromesnil	80
32875	260	\N	Drucat	80
32876	261	\N	Dury	80
32877	262	\N	Eaucourt-sur-Somme	80
32878	263	\N	Échelle-Saint-Aurin	80
32879	264	\N	Éclusier-Vaux	80
32880	265	\N	Embreville	80
32881	266	\N	Englebelmer	80
32882	267	\N	Ennemain	80
32883	268	\N	Épagne-Épagnette	80
32884	269	\N	Épaumesnil	80
32885	270	\N	Épécamps	80
32886	271	\N	Épehy	80
32887	272	\N	Épénancourt	80
32888	273	\N	Éplessier	80
32889	274	\N	Eppeville	80
32890	275	\N	Équancourt	80
32891	276	\N	Équennes-Éramecourt	80
32892	278	\N	Erches	80
32893	279	\N	Ercheu	80
32894	280	\N	Ercourt	80
32895	281	\N	Ergnies	80
32896	282	\N	Érondelle	80
32897	283	\N	Esclainvillers	80
32898	284	\N	Esmery-Hallon	80
32899	285	\N	Essertaux	80
32900	287	\N	Estrébuf	80
32901	288	\N	Estrées-Deniécourt	80
32902	290	\N	Estrées-lès-Crécy	80
32903	291	\N	Estrées-sur-Noye	80
32904	292	\N	Étalon	80
32905	293	\N	Ételfay	80
32906	294	\N	Éterpigny	80
32907	295	\N	Étinehem	80
32908	296	\N	Étoile	80
32909	297	\N	Étréjust	80
32910	298	\N	Étricourt-Manancourt	80
32911	299	\N	Faloise	80
32912	300	\N	Falvy	80
32913	301	\N	Famechon	80
32914	302	\N	Faverolles	80
32915	303	\N	Favières	80
32916	304	\N	Fay	80
32917	305	\N	Ferrières	80
32918	306	\N	Fescamps	80
32919	307	\N	Feuillères	80
32920	308	\N	Feuquières-en-Vimeu	80
32921	310	\N	Fienvillers	80
32922	311	\N	Fignières	80
32923	312	\N	Fins	80
32924	313	\N	Flaucourt	80
32925	314	\N	Flers	80
32926	315	\N	Flers-sur-Noye	80
32927	316	\N	Flesselles	80
32928	317	\N	Fleury	80
32929	318	\N	Flixecourt	80
32930	319	\N	Fluy	80
32931	320	\N	Folies	80
32932	321	\N	Folleville	80
32933	322	\N	Fonches-Fonchette	80
32934	324	\N	Fontaine-le-Sec	80
32935	325	\N	Fontaine-lès-Cappy	80
32936	326	\N	Fontaine-sous-Montdidier	80
32937	327	\N	Fontaine-sur-Maye	80
32938	328	\N	Fontaine-sur-Somme	80
32939	329	\N	Forceville	80
32940	330	\N	Forceville-en-Vimeu	80
32941	331	\N	Forest-l'Abbaye	80
32942	332	\N	Forest-Montiers	80
32943	333	\N	Fort-Mahon-Plage	80
32944	334	\N	Fossemanant	80
32945	335	\N	Foucaucourt-en-Santerre	80
32946	336	\N	Foucaucourt-Hors-Nesle	80
32947	337	\N	Fouencamps	80
32948	338	\N	Fouilloy	80
32949	339	\N	Fouquescourt	80
32950	340	\N	Fourcigny	80
32951	341	\N	Fourdrinoy	80
32952	342	\N	Framerville-Rainecourt	80
32953	343	\N	Framicourt	80
32954	344	\N	Francières	80
32955	345	\N	Franleu	80
32956	346	\N	Franqueville	80
32957	347	\N	Fransart	80
32958	348	\N	Fransu	80
32959	349	\N	Fransures	80
32960	350	\N	Franvillers	80
32961	351	\N	Fréchencourt	80
32962	352	\N	Frémontiers	80
32963	353	\N	Fresnes-Mazancourt	80
32964	354	\N	Fresnes-Tilloloy	80
32965	355	\N	Fresneville	80
32966	356	\N	Fresnoy-Andainville	80
32967	357	\N	Fresnoy-au-Val	80
32968	358	\N	Fresnoy-en-Chaussée	80
32969	359	\N	Fresnoy-lès-Roye	80
32970	360	\N	Fressenneville	80
32971	361	\N	Frettecuisse	80
32972	362	\N	Frettemeule	80
32973	364	\N	Friaucourt	80
32974	365	\N	Fricamps	80
32975	366	\N	Fricourt	80
32976	367	\N	Frise	80
32977	368	\N	Friville-Escarbotin	80
32978	369	\N	Frohen-le-Grand	80
32979	370	\N	Frohen-le-Petit	80
32980	371	\N	Froyelles	80
32981	372	\N	Frucourt	80
32982	373	\N	Gamaches	80
32983	374	\N	Gapennes	80
32984	375	\N	Gauville	80
32985	376	\N	Gentelles	80
32986	377	\N	Gézaincourt	80
32987	378	\N	Ginchy	80
32988	379	\N	Glisy	80
32989	380	\N	Gorenflos	80
32990	381	\N	Gorges	80
32991	383	\N	Goyencourt	80
32992	384	\N	Grandcourt	80
32993	385	\N	Grand-Laviers	80
32994	386	\N	Gratibus	80
32995	387	\N	Grattepanche	80
32996	388	\N	Grébault-Mesnil	80
32997	389	\N	Grécourt	80
32998	390	\N	Grivesnes	80
32999	391	\N	Grivillers	80
33000	392	\N	Grouches-Luchuel	80
33001	393	\N	Gruny	80
33002	395	\N	Guerbigny	80
33003	396	\N	Gueschart	80
33004	397	\N	Gueudecourt	80
33005	399	\N	Guignemicourt	80
33006	400	\N	Guillaucourt	80
33007	401	\N	Guillemont	80
33008	402	\N	Guizancourt	80
33009	403	\N	Guyencourt-sur-Noye	80
33010	404	\N	Guyencourt-Saulcourt	80
33011	405	\N	Hailles	80
33012	406	\N	Hallencourt	80
33013	407	\N	Hallivillers	80
33014	408	\N	Halloy-lès-Pernois	80
33015	409	\N	Hallu	80
33016	410	\N	Ham	80
33017	411	\N	Hamel	80
33018	412	\N	Hamelet	80
33019	413	\N	Hancourt	80
33020	414	\N	Hangard	80
33021	415	\N	Hangest-en-Santerre	80
33022	416	\N	Hangest-sur-Somme	80
33023	417	\N	Harbonnières	80
33024	418	\N	Hardecourt-aux-Bois	80
33025	419	\N	Hargicourt	80
33026	420	\N	Harponville	80
33027	421	\N	Hattencourt	80
33028	422	\N	Hautvillers-Ouville	80
33029	423	\N	Havernas	80
33030	424	\N	Hébécourt	80
33031	425	\N	Hédauville	80
33032	426	\N	Heilly	80
33033	427	\N	Hem-Hardinval	80
33034	428	\N	Hem-Monacu	80
33035	429	\N	Hénencourt	80
33036	430	\N	Herbécourt	80
33037	431	\N	Hérissart	80
33038	432	\N	Herleville	80
33039	433	\N	Herly	80
33040	434	\N	Hervilly	80
33041	435	\N	Hesbécourt	80
33042	436	\N	Hescamps	80
33043	437	\N	Heucourt-Croquoison	80
33044	438	\N	Heudicourt	80
33045	439	\N	Heuzecourt	80
33046	440	\N	Hiermont	80
33047	442	\N	Hombleux	80
33048	443	\N	Hornoy-le-Bourg	80
33049	444	\N	Huchenneville	80
33050	445	\N	Humbercourt	80
33051	446	\N	Huppy	80
33052	447	\N	Hyencourt-le-Grand	80
33053	449	\N	Ignaucourt	80
33054	450	\N	Inval-Boiron	80
33055	451	\N	Irles	80
33056	452	\N	Jumel	80
33057	453	\N	Laboissière-en-Santerre	80
33058	455	\N	Lachapelle	80
33059	456	\N	Lafresguimont-Saint-Martin	80
33060	458	\N	Lahoussoye	80
33061	459	\N	Laleu	80
33062	460	\N	Lamaronde	80
33063	461	\N	Lamotte-Brebière	80
33064	462	\N	Lamotte-Buleux	80
33065	463	\N	Lamotte-Warfusée	80
33066	464	\N	Lanchères	80
33067	465	\N	Languevoisin-Quiquery	80
33068	466	\N	Lanches-Saint-Hilaire	80
33069	467	\N	Laucourt	80
33070	468	\N	Laviéville	80
33071	469	\N	Lawarde-Mauger-l'Hortoy	80
33072	470	\N	Léalvillers	80
33073	472	\N	Lesbufs	80
33074	473	\N	Liancourt-Fosse	80
33075	474	\N	Licourt	80
33076	475	\N	Liéramont	80
33077	476	\N	Liercourt	80
33078	477	\N	Ligescourt	80
33079	478	\N	Lignières	80
33080	479	\N	Lignières-Châtelain	80
33081	480	\N	Lignières-en-Vimeu	80
33082	481	\N	Lihons	80
33083	482	\N	Limeux	80
33084	484	\N	Liomer	80
33085	485	\N	Luilly	80
33086	486	\N	Long	80
33087	487	\N	Longavesnes	80
33088	488	\N	Longpré-les-Corps-Saints	80
33089	489	\N	Longueau	80
33090	490	\N	Longueval	80
33091	491	\N	Longuevillette	80
33092	493	\N	Louvencourt	80
33093	494	\N	Louvrechy	80
33094	495	\N	Lucheux	80
33095	496	\N	Machiel	80
33096	497	\N	Machy	80
33097	498	\N	Mailly-Maillet	80
33098	499	\N	Mailly-Raineval	80
33099	500	\N	Maisnières	80
33100	501	\N	Maison-Ponthieu	80
33101	502	\N	Maison-Roland	80
33102	503	\N	Maizicourt	80
33103	504	\N	Malpart	80
33104	505	\N	Mametz	80
33105	507	\N	Marcelcave	80
33106	508	\N	Marché-Allouarde	80
33107	509	\N	Marchélepot	80
33108	511	\N	Marestmontiers	80
33109	512	\N	Mareuil-Caubert	80
33110	513	\N	Maricourt	80
33111	514	\N	Marieux	80
33112	515	\N	Marlers	80
33113	516	\N	Marquaix	80
33114	517	\N	Marquivillers	80
33115	518	\N	Martainneville	80
33116	519	\N	Matigny	80
33117	520	\N	Maucourt	80
33118	521	\N	Maurepas	80
33119	522	\N	Mazis	80
33120	523	\N	Méaulte	80
33121	524	\N	Méharicourt	80
33122	525	\N	Meigneux	80
33123	526	\N	Meillard	80
33124	527	\N	Méneslies	80
33125	528	\N	Méréaucourt	80
33126	529	\N	Mérélessart	80
33127	530	\N	Méricourt-l'Abbé	80
33128	531	\N	Méricourt-en-Vimeu	80
33129	532	\N	Méricourt-sur-Somme	80
33130	533	\N	Mers-les-Bains	80
33131	535	\N	Mesge	80
33132	536	\N	Mesnil-Bruntel	80
33133	537	\N	Mesnil-Domqueur	80
33134	538	\N	Mesnil-en-Arrouaise	80
33135	540	\N	Mesnil-Martinsart	80
33136	541	\N	Mesnil-Saint-Georges	80
33137	542	\N	Mesnil-Saint-Nicaise	80
33138	543	\N	Métigny	80
33139	544	\N	Mézerolles	80
33140	545	\N	Mézières-en-Santerre	80
33141	546	\N	Miannay	80
33142	547	\N	Millencourt	80
33143	548	\N	Millencourt-en-Ponthieu	80
33144	549	\N	Miraumont	80
33145	550	\N	Mirvaux	80
33146	551	\N	Misery	80
33147	552	\N	Moislains	80
33148	553	\N	Molliens-au-Bois	80
33149	554	\N	Molliens-Dreuil	80
33150	555	\N	Monchy-Lagache	80
33151	556	\N	Mons-Boubert	80
33152	557	\N	Estrées-Mons	80
33153	558	\N	Monsures	80
33154	559	\N	Montagne-Fayel	80
33155	560	\N	Montauban-de-Picardie	80
33156	561	\N	Montdidier	80
33157	562	\N	Montigny-sur-l'Hallue	80
33158	563	\N	Montigny-les-Jongleurs	80
33159	565	\N	Montonvillers	80
33160	566	\N	Fieffes-Montrelet	80
33161	568	\N	Morchain	80
33162	569	\N	Morcourt	80
33163	570	\N	Moreuil	80
33164	571	\N	Morisel	80
33165	572	\N	Morlancourt	80
33166	573	\N	Morvillers-Saint-Saturnin	80
33167	574	\N	Mouflers	80
33168	575	\N	Mouflières	80
33169	576	\N	Moyencourt	80
33170	577	\N	Moyencourt-lès-Poix	80
33171	578	\N	Moyenneville	80
33172	579	\N	Muille-Villette	80
33173	580	\N	Nampont	80
33174	582	\N	Namps-Maisnil	80
33175	583	\N	Nampty	80
33176	584	\N	Naours	80
33177	585	\N	Nesle	80
33178	586	\N	Nesle-l'Hôpital	80
33179	587	\N	Neslette	80
33180	588	\N	Neufmoulin	80
33181	589	\N	Neuilly-le-Dien	80
33182	590	\N	Neuilly-l'Hôpital	80
33183	591	\N	Neuville-au-Bois	80
33184	592	\N	Neuville-Coppegueule	80
33185	593	\N	Neuville-lès-Bray	80
33186	594	\N	Neuville-lès-Luilly	80
33187	595	\N	Neuville-Sire-Bernard	80
33188	596	\N	Neuvillette	80
33189	597	\N	Nibas	80
33190	598	\N	Nouvion	80
33191	599	\N	Noyelles-en-Chaussée	80
33192	600	\N	Noyelles-sur-Mer	80
33193	601	\N	Nurlu	80
33194	602	\N	Occoches	80
33195	603	\N	Ochancourt	80
33196	604	\N	Offignies	80
33197	605	\N	Offoy	80
33198	606	\N	Oisemont	80
33199	607	\N	Oissy	80
33200	608	\N	Omiécourt	80
33201	609	\N	Oneux	80
33202	611	\N	Oresmaux	80
33203	613	\N	Oust-Marest	80
33204	614	\N	Outrebois	80
33205	615	\N	Ovillers-la-Boisselle	80
33206	616	\N	Pargny	80
33207	617	\N	Parvillers-le-Quesnoy	80
33208	618	\N	Pendé	80
33209	619	\N	Pernois	80
33210	620	\N	Péronne	80
33211	621	\N	Pertain	80
33212	622	\N	Picquigny	80
33213	623	\N	Piennes-Onvillers	80
33214	624	\N	Pierregot	80
33215	625	\N	Pierrepont-sur-Avre	80
33216	626	\N	Pissy	80
33217	627	\N	Plachy-Buyon	80
33218	628	\N	Plessier-Rozainvillers	80
33219	629	\N	Puilly	80
33220	630	\N	Poix-de-Picardie	80
33221	631	\N	Ponches-Estruval	80
33222	632	\N	Pont-de-Metz	80
33223	633	\N	Ponthoile	80
33224	634	\N	Pont-Noyelles	80
33225	635	\N	Pont-Remy	80
33226	637	\N	Port-le-Grand	80
33227	638	\N	Potte	80
33228	639	\N	Poulainville	80
33229	640	\N	Pozières	80
33230	642	\N	Prouville	80
33231	643	\N	Prouzel	80
33232	644	\N	Proyart	80
33233	645	\N	Puchevillers	80
33234	646	\N	Punchy	80
33235	647	\N	Puzeaux	80
33236	648	\N	Pys	80
33237	649	\N	Quend	80
33238	650	\N	Querrieu	80
33239	651	\N	Quesne	80
33240	652	\N	Quesnel	80
33241	654	\N	Quesnoy-le-Montant	80
33242	655	\N	Quesnoy-sur-Airaines	80
33243	656	\N	Quevauvillers	80
33244	657	\N	Quiry-le-Sec	80
33245	658	\N	Quivières	80
33246	659	\N	Raincheval	80
33247	661	\N	Rainneville	80
33248	662	\N	Ramburelles	80
33249	663	\N	Rambures	80
33250	664	\N	Rancourt	80
33251	665	\N	Regnière-Écluse	80
33252	666	\N	Remaisnil	80
33253	667	\N	Remaugies	80
33254	668	\N	Remiencourt	80
33255	669	\N	Rethonvillers	80
33256	670	\N	Revelles	80
33257	671	\N	Ribeaucourt	80
33258	672	\N	Ribemont-sur-Ancre	80
33259	673	\N	Riencourt	80
33260	674	\N	Rivery	80
33261	675	\N	Rogy	80
33262	676	\N	Roiglise	80
33263	677	\N	Roisel	80
33264	678	\N	Rollot	80
33265	679	\N	Ronssoy	80
33266	680	\N	Rosières-en-Santerre	80
33267	681	\N	Rouvrel	80
33268	682	\N	Rouvroy-en-Santerre	80
33269	683	\N	Rouy-le-Grand	80
33270	684	\N	Rouy-le-Petit	80
33271	685	\N	Roye	80
33272	686	\N	Rubempré	80
33273	687	\N	Rubescourt	80
33274	688	\N	Rue	80
33275	690	\N	Rumigny	80
33276	691	\N	Saigneville	80
33277	692	\N	Sailly-Flibeaucourt	80
33278	693	\N	Sailly-Laurette	80
33279	694	\N	Sailly-le-Sec	80
33280	695	\N	Sailly-Saillisel	80
33281	696	\N	Sains-en-Amiénois	80
33282	697	\N	Saint-Acheul	80
33283	698	\N	Saint-Aubin-Montenoy	80
33284	699	\N	Saint-Aubin-Rivière	80
33285	700	\N	Saint-Blimont	80
33286	701	\N	Saint-Christ-Briost	80
33287	702	\N	Saint-Fuscien	80
33288	703	\N	Saint-Germain-sur-Bresle	80
33289	704	\N	Saint-Gratien	80
33290	705	\N	Saint-Léger-lès-Authie	80
33291	706	\N	Saint-Léger-lès-Domart	80
33292	707	\N	Saint-Léger-sur-Bresle	80
33293	708	\N	Saint-Mard	80
33294	709	\N	Saint-Maulvis	80
33295	710	\N	Saint-Maxent	80
33296	711	\N	Saint-Ouen	80
33297	713	\N	Saint-Quentin-en-Tourmont	80
33298	714	\N	Saint-Quentin-la-Motte-Croix-au-Bailly	80
33299	716	\N	Saint-Riquier	80
33300	717	\N	Saint-Sauflieu	80
33301	718	\N	Saint-Sauveur	80
33302	719	\N	Sainte-Segrée	80
33303	721	\N	Saint-Valery-sur-Somme	80
33304	722	\N	Saint-Vaast-en-Chaussée	80
33305	723	\N	Saisseval	80
33306	724	\N	Saleux	80
33307	725	\N	Salouël	80
33308	726	\N	Sancourt	80
33309	728	\N	Saulchoy-sous-Poix	80
33310	729	\N	Sauvillers-Mongival	80
33311	730	\N	Saveuse	80
33312	732	\N	Senarpont	80
33313	733	\N	Senlis-le-Sec	80
33314	734	\N	Sentelie	80
33315	735	\N	Seux	80
33316	736	\N	Sorel-en-Vimeu	80
33317	737	\N	Sorel	80
33318	738	\N	Soues	80
33319	740	\N	Sourdon	80
33320	741	\N	Soyécourt	80
33321	742	\N	Surcamps	80
33322	743	\N	Suzanne	80
33323	744	\N	Tailly	80
33324	746	\N	Talmas	80
33325	747	\N	Templeux-la-Fosse	80
33326	748	\N	Templeux-le-Guérard	80
33327	749	\N	Terramesnil	80
33328	750	\N	Tertry	80
33329	751	\N	Thennes	80
33330	752	\N	Thézy-Glimont	80
33331	753	\N	Thiepval	80
33332	754	\N	Thieulloy-l'Abbaye	80
33333	755	\N	Thieulloy-la-Ville	80
33334	756	\N	Thièvres	80
33335	757	\N	Thoix	80
33336	758	\N	Thory	80
33337	759	\N	Tilloloy	80
33338	760	\N	Tilloy-Floriville	80
33339	761	\N	Tilloy-lès-Conty	80
33340	762	\N	Tincourt-Boucly	80
33341	763	\N	Titre	80
33342	764	\N	Tufles	80
33343	765	\N	Tours-en-Vimeu	80
33344	766	\N	Toutencourt	80
33345	767	\N	Translay	80
33346	769	\N	Treux	80
33347	770	\N	Tully	80
33348	771	\N	Ugny-l'Équipée	80
33349	773	\N	Vadencourt	80
33350	774	\N	Vaire-sous-Corbie	80
33351	775	\N	Valines	80
33352	776	\N	Varennes	80
33353	777	\N	Vauchelles-lès-Authie	80
33354	778	\N	Vauchelles-lès-Domart	80
33355	779	\N	Vauchelles-les-Quesnoy	80
33356	780	\N	Vaudricourt	80
33357	781	\N	Vauvillers	80
33358	782	\N	Vaux-en-Amiénois	80
33359	783	\N	Vaux-Marquenneville	80
33360	784	\N	Vaux-sur-Somme	80
33361	785	\N	Vecquemont	80
33362	786	\N	Velennes	80
33363	787	\N	Vercourt	80
33364	788	\N	Vergies	80
33365	789	\N	Vermandovillers	80
33366	790	\N	Verpillières	80
33367	791	\N	Vers-sur-Selles	80
33368	792	\N	Vicogne	80
33369	793	\N	Vignacourt	80
33370	794	\N	Villecourt	80
33371	795	\N	Ville-le-Marclet	80
33372	796	\N	Villeroy	80
33373	797	\N	Villers-aux-Érables	80
33374	798	\N	Villers-Bocage	80
33375	799	\N	Villers-Bretonneux	80
33376	800	\N	Villers-Campsart	80
33377	801	\N	Villers-Carbonnel	80
33378	802	\N	Villers-Faucon	80
33379	803	\N	Villers-lès-Roye	80
33380	804	\N	Villers-sous-Ailly	80
33381	805	\N	Villers-Tournelle	80
33382	806	\N	Villers-sur-Authie	80
33383	807	\N	Ville-sur-Ancre	80
33384	808	\N	Vironchaux	80
33385	809	\N	Vismes	80
33386	810	\N	Vitz-sur-Authie	80
33387	811	\N	Voyennes	80
33388	812	\N	Vraignes-en-Vermandois	80
33389	813	\N	Vraignes-lès-Hornoy	80
33390	814	\N	Vrély	80
33391	815	\N	Vron	80
33392	819	\N	Wargnies	80
33393	820	\N	Warloy-Baillon	80
33394	821	\N	Warlus	80
33395	822	\N	Warsy	80
33396	823	\N	Warvillers	80
33397	824	\N	Wiencourt-l'Équipée	80
33398	825	\N	Wiry-au-Mont	80
33399	826	\N	Woignarue	80
33400	827	\N	Woincourt	80
33401	828	\N	Woirel	80
33402	829	\N	Y	80
33403	830	\N	Yaucourt-Bussus	80
33404	832	\N	Yvrench	80
33405	833	\N	Yvrencheux	80
33406	834	\N	Yzengremer	80
33407	835	\N	Yzeux	80
33408	836	\N	Yonval	80
33409	001	\N	Aguts	81
33410	002	\N	Aiguefonde	81
33411	003	\N	Alban	81
33412	004	\N	Albi	81
33413	005	\N	Albine	81
33414	006	\N	Algans	81
33415	007	\N	Alos	81
33416	008	\N	Almayrac	81
33417	009	\N	Amarens	81
33418	010	\N	Ambialet	81
33419	011	\N	Ambres	81
33420	012	\N	Andillac	81
33421	013	\N	Andouque	81
33422	014	\N	Anglès	81
33423	015	\N	Appelle	81
33424	016	\N	Arfons	81
33425	017	\N	Arifat	81
33426	018	\N	Arthès	81
33427	019	\N	Assac	81
33428	020	\N	Aussac	81
33429	021	\N	Aussillon	81
33430	022	\N	Bannières	81
33431	023	\N	Barre	81
33432	024	\N	Beauvais-sur-Tescou	81
33433	025	\N	Belcastel	81
33434	026	\N	Bellegarde	81
33435	027	\N	Belleserre	81
33436	028	\N	Berlats	81
33437	029	\N	Bernac	81
33438	030	\N	Bertre	81
33439	031	\N	Bez	81
33440	032	\N	Blan	81
33441	033	\N	Blaye-les-Mines	81
33442	034	\N	Boissezon	81
33443	035	\N	Bournazel	81
33444	036	\N	Bout-du-Pont-de-Larn	81
33445	037	\N	Brassac	81
33446	038	\N	Brens	81
33447	039	\N	Briatexte	81
33448	040	\N	Brousse	81
33449	041	\N	Broze	81
33450	042	\N	Burlats	81
33451	043	\N	Busque	81
33452	044	\N	Cabanès	81
33453	045	\N	Cabannes	81
33454	046	\N	Cadalen	81
33455	047	\N	Cadix	81
33456	048	\N	Cagnac-les-Mines	81
33457	049	\N	Cahuzac	81
33458	050	\N	Cambon-lès-Lavaur	81
33459	051	\N	Cahuzac-sur-Vère	81
33460	052	\N	Cambon	81
33461	053	\N	Cambounès	81
33462	054	\N	Cambounet-sur-le-Sor	81
33463	055	\N	Cammazes	81
33464	056	\N	Campagnac	81
33465	058	\N	Carbes	81
33466	059	\N	Carlus	81
33467	060	\N	Carmaux	81
33468	061	\N	Castanet	81
33469	062	\N	Castelnau-de-Brassac	81
33470	063	\N	Castelnau-de-Lévis	81
33471	064	\N	Castelnau-de-Montmiral	81
33472	065	\N	Castres	81
33473	066	\N	Caucalières	81
33474	067	\N	Cestayrols	81
33475	068	\N	Combefa	81
33476	069	\N	Cordes-sur-Ciel	81
33477	070	\N	Coufouleux	81
33478	071	\N	Courris	81
33479	072	\N	Crespin	81
33480	073	\N	Crespinet	81
33481	074	\N	Cunac	81
33482	075	\N	Cuq	81
33483	076	\N	Cuq-Toulza	81
33484	077	\N	Curvalle	81
33485	078	\N	Damiatte	81
33486	079	\N	Dénat	81
33487	080	\N	Donnazac	81
33488	081	\N	Dourgne	81
33489	082	\N	Dourn	81
33490	083	\N	Durfort	81
33491	084	\N	Escoussens	81
33492	085	\N	Escroux	81
33493	086	\N	Espérausses	81
33494	087	\N	Fayssac	81
33495	088	\N	Fauch	81
33496	089	\N	Faussergues	81
33497	090	\N	Fénols	81
33498	091	\N	Ferrières	81
33499	092	\N	Fiac	81
33500	093	\N	Florentin	81
33501	094	\N	Fraissines	81
33502	095	\N	Frausseilles	81
33503	096	\N	Fraysse	81
33504	097	\N	Fréjairolles	81
33505	098	\N	Fréjeville	81
33506	099	\N	Gaillac	81
33507	100	\N	Garrevaques	81
33508	101	\N	Garric	81
33509	102	\N	Garrigues	81
33510	103	\N	Gijounet	81
33511	104	\N	Giroussens	81
33512	105	\N	Graulhet	81
33513	106	\N	Grazac	81
33514	107	\N	Guitalens	81
33515	108	\N	Itzac	81
33516	109	\N	Jonquières	81
33517	110	\N	Jouqueviel	81
33518	111	\N	Labarthe-Bleys	81
33519	112	\N	Labastide-de-Lévis	81
33520	113	\N	Labastide-Dénat	81
33521	114	\N	Labastide-Gabausse	81
33522	115	\N	Labastide-Rouairoux	81
33523	116	\N	Labastide-Saint-Georges	81
33524	117	\N	Labessière-Candeil	81
33525	118	\N	Laboulbène	81
33526	119	\N	Laboutarie	81
33527	120	\N	Labruguière	81
33528	121	\N	Lacabarède	81
33529	122	\N	Lacapelle-Pinet	81
33530	123	\N	Lacapelle-Ségalar	81
33531	124	\N	Lacaune	81
33532	125	\N	Lacaze	81
33533	126	\N	Lacougotte-Cadoul	81
33534	127	\N	Lacroisille	81
33535	128	\N	Lacrouzette	81
33536	129	\N	Lagardiolle	81
33537	130	\N	Lagarrigue	81
33538	131	\N	Lagrave	81
33539	132	\N	Lalbarède	81
33540	133	\N	Lamillarié	81
33541	134	\N	Lamontélarié	81
33542	135	\N	Laparrouquial	81
33543	136	\N	Larroque	81
33544	137	\N	Lasfaillades	81
33545	138	\N	Lasgraisses	81
33546	139	\N	Lautrec	81
33547	140	\N	Lavaur	81
33548	141	\N	Lédas-et-Penthiès	81
33549	142	\N	Lempaut	81
33550	143	\N	Lescout	81
33551	144	\N	Lescure-d'Albigeois	81
33552	145	\N	Lisle-sur-Tarn	81
33553	146	\N	Livers-Cazelles	81
33554	147	\N	Lombers	81
33555	148	\N	Loubers	81
33556	149	\N	Loupiac	81
33557	150	\N	Lugan	81
33558	151	\N	Magrin	81
33559	152	\N	Mailhoc	81
33560	153	\N	Margnès	81
33561	154	\N	Marnaves	81
33562	155	\N	Marsal	81
33563	156	\N	Marssac-sur-Tarn	81
33564	157	\N	Marzens	81
33565	158	\N	Masnau-Massuguiès	81
33566	159	\N	Massac-Séran	81
33567	160	\N	Massaguel	81
33568	161	\N	Massals	81
33569	162	\N	Maurens-Scopont	81
33570	163	\N	Mazamet	81
33571	164	\N	Mézens	81
33572	165	\N	Milhars	81
33573	166	\N	Milhavet	81
33574	167	\N	Miolles	81
33575	168	\N	Mirandol-Bourgnounac	81
33576	169	\N	Missècle	81
33577	170	\N	Monestiés	81
33578	171	\N	Montans	81
33579	172	\N	Montauriol	81
33580	173	\N	Montcabrier	81
33581	174	\N	Montdragon	81
33582	175	\N	Montdurausse	81
33583	176	\N	Montels	81
33584	177	\N	Montfa	81
33585	178	\N	Montgaillard	81
33586	179	\N	Montgey	81
33587	180	\N	Montirat	81
33588	181	\N	Montpinier	81
33589	182	\N	Montredon-Labessonnié	81
33590	183	\N	Mont-Roc	81
33591	184	\N	Montrosier	81
33592	185	\N	Montvalen	81
33593	186	\N	Moularès	81
33594	187	\N	Moulayrès	81
33595	188	\N	Moulin-Mage	81
33596	189	\N	Mouzens	81
33597	190	\N	Mouzieys-Teulet	81
33598	191	\N	Mouzieys-Panens	81
33599	192	\N	Murat-sur-Vèbre	81
33600	193	\N	Nages	81
33601	195	\N	Navès	81
33602	196	\N	Noailhac	81
33603	197	\N	Noailles	81
33604	198	\N	Orban	81
33605	199	\N	Padiès	81
33606	200	\N	Palleville	81
33607	201	\N	Pampelonne	81
33608	202	\N	Parisot	81
33609	203	\N	Paulinet	81
33610	204	\N	Payrin-Augmontel	81
33611	205	\N	Péchaudier	81
33612	206	\N	Penne	81
33613	207	\N	Peyregoux	81
33614	208	\N	Peyrole	81
33615	209	\N	Pont-de-Larn	81
33616	210	\N	Poudis	81
33617	211	\N	Poulan-Pouzols	81
33618	212	\N	Prades	81
33619	213	\N	Pratviel	81
33620	214	\N	Puéchoursi	81
33621	215	\N	Puybegon	81
33622	216	\N	Puycalvel	81
33623	217	\N	Puycelci	81
33624	218	\N	Puygouzon	81
33625	219	\N	Puylaurens	81
33626	220	\N	Rabastens	81
33627	221	\N	Rayssac	81
33628	222	\N	Réalmont	81
33629	223	\N	Rialet	81
33630	224	\N	Riols	81
33631	225	\N	Rivières	81
33632	226	\N	Ronel	81
33633	227	\N	Roquecourbe	81
33634	228	\N	Roquemaure	81
33635	229	\N	Roquevidal	81
33636	230	\N	Rosières	81
33637	231	\N	Rouairoux	81
33638	232	\N	Rouffiac	81
33639	233	\N	Roumégoux	81
33640	234	\N	Roussayrolles	81
33641	235	\N	Saint-Affrique-les-Montagnes	81
33642	236	\N	Saint-Agnan	81
33643	237	\N	Saint-Amancet	81
33644	238	\N	Saint-Amans-Soult	81
33645	239	\N	Saint-Amans-Valtoret	81
33646	240	\N	Saint-André	81
33647	241	\N	Saint-Antonin-de-Lacalm	81
33648	242	\N	Saint-Avit	81
33649	243	\N	Saint-Beauzile	81
33650	244	\N	Saint-Benoît-de-Carmaux	81
33651	245	\N	Saint-Christophe	81
33652	246	\N	Sainte-Cécile-du-Cayrou	81
33653	247	\N	Saint-Cirgue	81
33654	248	\N	Saint-Gauzens	81
33655	249	\N	Sainte-Gemme	81
33656	250	\N	Saint-Genest-de-Contest	81
33657	251	\N	Saint-Germain-des-Prés	81
33658	252	\N	Saint-Germier	81
33659	253	\N	Saint-Grégoire	81
33660	254	\N	Saint-Jean-de-Marcel	81
33661	255	\N	Saint-Jean-de-Rives	81
33662	256	\N	Saint-Jean-de-Vals	81
33663	257	\N	Saint-Juéry	81
33664	258	\N	Saint-Julien-du-Puy	81
33665	259	\N	Saint-Julien-Gaulène	81
33666	260	\N	Saint-Lieux-Lafenasse	81
33667	261	\N	Saint-Lieux-lès-Lavaur	81
33668	262	\N	Saint-Marcel-Campes	81
33669	263	\N	Saint-Martin-Laguépie	81
33670	264	\N	Saint-Michel-Labadié	81
33671	265	\N	Saint-Michel-de-Vax	81
33672	266	\N	Saint-Paul-Cap-de-Joux	81
33673	267	\N	Saint-Pierre-de-Trivisy	81
33674	268	\N	Saint-Salvi-de-Carcavès	81
33675	269	\N	Saint-Salvy-de-la-Balme	81
33676	270	\N	Saint-Sernin-lès-Lavaur	81
33677	271	\N	Saint-Sulpice	81
33678	272	\N	Saint-Urcisse	81
33679	273	\N	Saïx	81
33680	274	\N	Saliès	81
33681	275	\N	Salles	81
33682	276	\N	Salvagnac	81
33683	277	\N	Saussenac	81
33684	278	\N	Sauveterre	81
33685	279	\N	Sauzière-Saint-Jean	81
33686	280	\N	Ségur	81
33687	281	\N	Sémalens	81
33688	282	\N	Senaux	81
33689	283	\N	Senouillac	81
33690	284	\N	Sequestre	81
33691	285	\N	Sérénac	81
33692	286	\N	Serviès	81
33693	287	\N	Sieurac	81
33694	288	\N	Sorèze	81
33695	289	\N	Soual	81
33696	290	\N	Souel	81
33697	291	\N	Taïx	81
33698	292	\N	Tanus	81
33699	293	\N	Tauriac	81
33700	294	\N	Técou	81
33701	295	\N	Teillet	81
33702	296	\N	Terre-Clapier	81
33703	297	\N	Terssac	81
33704	298	\N	Teulat	81
33705	299	\N	Teyssode	81
33706	300	\N	Tonnac	81
33707	301	\N	Travet	81
33708	302	\N	Tréban	81
33709	303	\N	Trébas	81
33710	304	\N	Trévien	81
33711	305	\N	Vabre	81
33712	306	\N	Valderiès	81
33713	307	\N	Valdurenque	81
33714	308	\N	Valence-d'Albigeois	81
33715	309	\N	Vaour	81
33716	310	\N	Veilhes	81
33717	311	\N	Vénès	81
33718	312	\N	Verdalle	81
33719	313	\N	Verdier	81
33720	314	\N	Viane	81
33721	315	\N	Vielmur-sur-Agout	81
33722	316	\N	Vieux	81
33723	317	\N	Villefranche-d'Albigeois	81
33724	318	\N	Villeneuve-lès-Lavaur	81
33725	319	\N	Villeneuve-sur-Vère	81
33726	320	\N	Vindrac-Alayrac	81
33727	321	\N	Vintrou	81
33728	322	\N	Virac	81
33729	323	\N	Viterbe	81
33730	324	\N	Viviers-lès-Lavaur	81
33731	325	\N	Viviers-lès-Montagnes	81
33732	326	\N	Sainte-Croix	81
33733	001	\N	Albefeuille-Lagarde	82
33734	002	\N	Albias	82
33735	003	\N	Angeville	82
33736	004	\N	Asques	82
33737	005	\N	Aucamville	82
33738	006	\N	Auterive	82
33739	007	\N	Auty	82
33740	008	\N	Auvillar	82
33741	009	\N	Balignac	82
33742	010	\N	Bardigues	82
33743	011	\N	Barry-d'Islemade	82
33744	012	\N	Barthes	82
33745	013	\N	Beaumont-de-Lomagne	82
33746	014	\N	Beaupuy	82
33747	015	\N	Belbèse	82
33748	016	\N	Belvèze	82
33749	017	\N	Bessens	82
33750	018	\N	Bioule	82
33751	019	\N	Boudou	82
33752	020	\N	Bouillac	82
33753	021	\N	Bouloc	82
33754	022	\N	Bourg-de-Visa	82
33755	023	\N	Bourret	82
33756	024	\N	Brassac	82
33757	025	\N	Bressols	82
33758	026	\N	Bruniquel	82
33759	027	\N	Campsas	82
33760	028	\N	Canals	82
33761	029	\N	Castanet	82
33762	030	\N	Castelferrus	82
33763	031	\N	Castelmayran	82
33764	032	\N	Castelsagrat	82
33765	033	\N	Castelsarrasin	82
33766	034	\N	Castéra-Bouzet	82
33767	035	\N	Caumont	82
33768	036	\N	Causé	82
33769	037	\N	Caussade	82
33770	038	\N	Caylus	82
33771	039	\N	Cayrac	82
33772	040	\N	Cayriech	82
33773	041	\N	Cazals	82
33774	042	\N	Cazes-Mondenard	82
33775	043	\N	Comberouger	82
33776	044	\N	Corbarieu	82
33777	045	\N	Cordes-Tolosannes	82
33778	046	\N	Coutures	82
33779	047	\N	Cumont	82
33780	048	\N	Dieupentale	82
33781	049	\N	Donzac	82
33782	050	\N	Dunes	82
33783	051	\N	Durfort-Lacapelette	82
33784	052	\N	Escatalens	82
33785	053	\N	Escazeaux	82
33786	054	\N	Espalais	82
33787	055	\N	Esparsac	82
33788	056	\N	Espinas	82
33789	057	\N	Fabas	82
33790	058	\N	Fajolles	82
33791	059	\N	Faudoas	82
33792	060	\N	Fauroux	82
33793	061	\N	Féneyrols	82
33794	062	\N	Finhan	82
33795	063	\N	Garganvillar	82
33796	064	\N	Gariès	82
33797	065	\N	Gasques	82
33798	066	\N	Génébrières	82
33799	067	\N	Gensac	82
33800	068	\N	Gimat	82
33801	069	\N	Ginals	82
33802	070	\N	Glatens	82
33803	071	\N	Goas	82
33804	072	\N	Golfech	82
33805	073	\N	Goudourville	82
33806	074	\N	Gramont	82
33807	075	\N	Grisolles	82
33808	076	\N	Honor-de-Cos	82
33809	077	\N	Labarthe	82
33810	078	\N	Labastide-de-Penne	82
33811	079	\N	Labastide-Saint-Pierre	82
33812	080	\N	Labastide-du-Temple	82
33813	081	\N	Labourgade	82
33814	082	\N	Lacapelle-Livron	82
33815	083	\N	Lachapelle	82
33816	084	\N	Lacour	82
33817	085	\N	Lacourt-Saint-Pierre	82
33818	086	\N	Lafitte	82
33819	087	\N	Lafrançaise	82
33820	088	\N	Laguépie	82
33821	089	\N	Lamagistère	82
33822	090	\N	Lamothe-Capdeville	82
33823	091	\N	Lamothe-Cumont	82
33824	092	\N	Lapenche	82
33825	093	\N	Larrazet	82
33826	094	\N	Lauzerte	82
33827	095	\N	Lavaurette	82
33828	096	\N	Ville-Dieu-du-Temple	82
33829	097	\N	Lavit	82
33830	098	\N	Léojac	82
33831	099	\N	Lizac	82
33832	100	\N	Loze	82
33833	101	\N	Malause	82
33834	102	\N	Mansonville	82
33835	103	\N	Marignac	82
33836	104	\N	Marsac	82
33837	105	\N	Mas-Grenier	82
33838	106	\N	Maubec	82
33839	107	\N	Maumusson	82
33840	108	\N	Meauzac	82
33841	109	\N	Merles	82
33842	110	\N	Mirabel	82
33843	111	\N	Miramont-de-Quercy	82
33844	112	\N	Moissac	82
33845	113	\N	Molières	82
33846	114	\N	Monbéqui	82
33847	115	\N	Monclar-de-Quercy	82
33848	116	\N	Montagudet	82
33849	117	\N	Montaigu-de-Quercy	82
33850	118	\N	Montaïn	82
33851	119	\N	Montalzat	82
33852	120	\N	Montastruc	82
33853	121	\N	Montauban	82
33854	122	\N	Montbarla	82
33855	123	\N	Montbartier	82
33856	124	\N	Montbeton	82
33857	125	\N	Montech	82
33858	126	\N	Monteils	82
33859	127	\N	Montesquieu	82
33860	128	\N	Montfermier	82
33861	129	\N	Montgaillard	82
33862	130	\N	Montjoi	82
33863	131	\N	Montpezat-de-Quercy	82
33864	132	\N	Montricoux	82
33865	133	\N	Mouillac	82
33866	134	\N	Nègrepelisse	82
33867	135	\N	Nohic	82
33868	136	\N	Orgueil	82
33869	137	\N	Parisot	82
33870	138	\N	Perville	82
33871	139	\N	Pin	82
33872	140	\N	Piquecos	82
33873	141	\N	Pommevic	82
33874	142	\N	Pompignan	82
33875	143	\N	Poupas	82
33876	144	\N	Puycornet	82
33877	145	\N	Puygaillard-de-Quercy	82
33878	146	\N	Puygaillard-de-Lomagne	82
33879	147	\N	Puylagarde	82
33880	148	\N	Puylaroque	82
33881	149	\N	Réalville	82
33882	150	\N	Reyniès	82
33883	151	\N	Roquecor	82
33884	152	\N	Saint-Aignan	82
33885	153	\N	Saint-Amans-du-Pech	82
33886	154	\N	Saint-Amans-de-Pellagal	82
33887	155	\N	Saint-Antonin-Noble-Val	82
33888	156	\N	Saint-Arroumex	82
33889	157	\N	Saint-Beauzeil	82
33890	158	\N	Saint-Cirice	82
33891	159	\N	Saint-Cirq	82
33892	160	\N	Saint-Clair	82
33893	161	\N	Saint-Etienne-de-Tulmont	82
33894	162	\N	Saint-Georges	82
33895	163	\N	Saint-Jean-du-Bouzet	82
33896	164	\N	Sainte-Juliette	82
33897	165	\N	Saint-Loup	82
33898	166	\N	Saint-Michel	82
33899	167	\N	Saint-Nauphary	82
33900	168	\N	Saint-Nazaire-de-Valentane	82
33901	169	\N	Saint-Nicolas-de-la-Grave	82
33902	170	\N	Saint-Paul-d'Espis	82
33903	171	\N	Saint-Porquier	82
33904	172	\N	Saint-Projet	82
33905	173	\N	Saint-Sardos	82
33906	174	\N	Saint-Vincent	82
33907	175	\N	Saint-Vincent-Lespinasse	82
33908	176	\N	Salvetat-Belmontet	82
33909	177	\N	Sauveterre	82
33910	178	\N	Savenès	82
33911	179	\N	Septfonds	82
33912	180	\N	Sérignac	82
33913	181	\N	Sistels	82
33914	182	\N	Touffailles	82
33915	183	\N	Tréjouls	82
33916	184	\N	Vaïssac	82
33917	185	\N	Valeilles	82
33918	186	\N	Valence	82
33919	187	\N	Varen	82
33920	188	\N	Varennes	82
33921	189	\N	Vazerac	82
33922	190	\N	Verdun-sur-Garonne	82
33923	191	\N	Verfeil	82
33924	192	\N	Verlhac-Tescou	82
33925	193	\N	Vigueron	82
33926	194	\N	Villebrumier	82
33927	195	\N	Villemade	82
33928	001	\N	Adrets-de-l'Estérel	83
33929	002	\N	Aiguines	83
33930	003	\N	Ampus	83
33931	004	\N	Arcs	83
33932	005	\N	Artignosc-sur-Verdon	83
33933	006	\N	Artigues	83
33934	007	\N	Aups	83
33935	008	\N	Bagnols-en-Forêt	83
33936	009	\N	Bandol	83
33937	010	\N	Bargème	83
33938	011	\N	Bargemon	83
33939	012	\N	Barjols	83
33940	013	\N	Bastide	83
33941	014	\N	Baudinard-sur-Verdon	83
33942	015	\N	Bauduen	83
33943	016	\N	Beausset	83
33944	017	\N	Belgentier	83
33945	018	\N	Besse-sur-Issole	83
33946	019	\N	Bormes-les-Mimosas	83
33947	020	\N	Bourguet	83
33948	021	\N	Bras	83
33949	022	\N	Brenon	83
33950	023	\N	Brignoles	83
33951	025	\N	Brue-Auriac	83
33952	026	\N	Cabasse	83
33953	027	\N	Cadière-d'Azur	83
33954	028	\N	Callas	83
33955	029	\N	Callian	83
33956	030	\N	Camps-la-Source	83
33957	031	\N	Cannet-des-Maures	83
33958	032	\N	Carcès	83
33959	033	\N	Carnoules	83
33960	034	\N	Carqueiranne	83
33961	035	\N	Castellet	83
33962	036	\N	Cavalaire-sur-Mer	83
33963	037	\N	Celle	83
33964	038	\N	Châteaudouble	83
33965	039	\N	Châteauvert	83
33966	040	\N	Châteauvieux	83
33967	041	\N	Claviers	83
33968	042	\N	Cogolin	83
33969	043	\N	Collobrières	83
33970	044	\N	Comps-sur-Artuby	83
33971	045	\N	Correns	83
33972	046	\N	Cotignac	83
33973	047	\N	Crau	83
33974	048	\N	Croix-Valmer	83
33975	049	\N	Cuers	83
33976	050	\N	Draguignan	83
33977	051	\N	Entrecasteaux	83
33978	052	\N	Esparron	83
33979	053	\N	Évenos	83
33980	054	\N	Farlède	83
33981	055	\N	Fayence	83
33982	056	\N	Figanières	83
33983	057	\N	Flassans-sur-Issole	83
33984	058	\N	Flayosc	83
33985	059	\N	Forcalqueiret	83
33986	060	\N	Fox-Amphoux	83
33987	061	\N	Fréjus	83
33988	062	\N	Garde	83
33989	063	\N	Garde-Freinet	83
33990	064	\N	Garéoult	83
33991	065	\N	Gassin	83
33992	066	\N	Ginasservis	83
33993	067	\N	Gonfaron	83
33994	068	\N	Grimaud	83
33995	069	\N	Hyères	83
33996	070	\N	Lavandou	83
33997	071	\N	Londe-les-Maures	83
33998	072	\N	Lorgues	83
33999	073	\N	Luc	83
34000	074	\N	Martre	83
34001	075	\N	Mayons	83
34002	076	\N	Mazaugues	83
34003	077	\N	Méounes-lès-Montrieux	83
34004	078	\N	Moissac-Bellevue	83
34005	079	\N	Môle	83
34006	080	\N	Mons	83
34007	081	\N	Montauroux	83
34008	082	\N	Montferrat	83
34009	083	\N	Montfort-sur-Argens	83
34010	084	\N	Montmeyan	83
34011	085	\N	Motte	83
34012	086	\N	Muy	83
34013	087	\N	Nans-les-Pins	83
34014	088	\N	Néoules	83
34015	089	\N	Ollières	83
34016	090	\N	Ollioules	83
34017	091	\N	Pierrefeu-du-Var	83
34018	092	\N	Pignans	83
34019	093	\N	Plan-d'Aups-Sainte-Baume	83
34020	094	\N	Plan-de-la-Tour	83
34021	095	\N	Pontevès	83
34022	096	\N	Pourcieux	83
34023	097	\N	Pourrières	83
34024	098	\N	Pradet	83
34025	099	\N	Puget-sur-Argens	83
34026	100	\N	Puget-Ville	83
34027	101	\N	Ramatuelle	83
34028	102	\N	Régusse	83
34029	103	\N	Revest-les-Eaux	83
34030	104	\N	Rians	83
34031	105	\N	Riboux	83
34032	106	\N	Rocbaron	83
34033	107	\N	Roquebrune-sur-Argens	83
34034	108	\N	Roquebrussanne	83
34035	109	\N	Roque-Esclapon	83
34036	110	\N	Rougiers	83
34037	111	\N	Sainte-Anastasie-sur-Issole	83
34038	112	\N	Saint-Cyr-sur-Mer	83
34039	113	\N	Saint-Julien	83
34040	114	\N	Saint-Martin	83
34041	115	\N	Sainte-Maxime	83
34042	116	\N	Saint-Maximin-la-Sainte-Baume	83
34043	117	\N	Saint-Paul-en-Forêt	83
34044	118	\N	Saint-Raphaël	83
34045	119	\N	Saint-Tropez	83
34046	120	\N	Saint-Zacharie	83
34047	121	\N	Salernes	83
34048	122	\N	Salles-sur-Verdon	83
34049	123	\N	Sanary-sur-Mer	83
34050	124	\N	Seillans	83
34051	125	\N	Seillons-Source-d'Argens	83
34052	126	\N	Seyne-sur-Mer	83
34053	127	\N	Signes	83
34054	128	\N	Sillans-la-Cascade	83
34055	129	\N	Six-Fours-les-Plages	83
34056	130	\N	Solliès-Pont	83
34057	131	\N	Solliès-Toucas	83
34058	132	\N	Solliès-Ville	83
34059	133	\N	Tanneron	83
34060	134	\N	Taradeau	83
34061	135	\N	Tavernes	83
34062	136	\N	Thoronet	83
34063	137	\N	Toulon	83
34064	138	\N	Tourrettes	83
34065	139	\N	Tourtour	83
34066	140	\N	Tourves	83
34067	141	\N	Trans-en-Provence	83
34068	142	\N	Trigance	83
34069	143	\N	Val	83
34070	144	\N	Valette-du-Var	83
34071	145	\N	Varages	83
34072	146	\N	Verdière	83
34073	147	\N	Vérignon	83
34074	148	\N	Vidauban	83
34075	149	\N	Villecroze	83
34076	150	\N	Vinon-sur-Verdon	83
34077	151	\N	Vins-sur-Caramy	83
34078	152	\N	Rayol-Canadel-sur-Mer	83
34079	153	\N	Saint-Mandrier-sur-Mer	83
34080	154	\N	Saint-Antonin-du-Var	83
34081	001	\N	Althen-des-Paluds	84
34082	002	\N	Ansouis	84
34083	003	\N	Apt	84
34084	004	\N	Aubignan	84
34085	005	\N	Aurel	84
34086	006	\N	Auribeau	84
34087	007	\N	Avignon	84
34088	008	\N	Barroux	84
34089	009	\N	Bastide-des-Jourdans	84
34090	010	\N	Bastidonne	84
34091	011	\N	Beaucet	84
34092	012	\N	Beaumes-de-Venise	84
34093	013	\N	Beaumettes	84
34094	014	\N	Beaumont-de-Pertuis	84
34095	015	\N	Beaumont-du-Ventoux	84
34096	016	\N	Bédarrides	84
34097	017	\N	Bédoin	84
34098	018	\N	Blauvac	84
34099	019	\N	Bollène	84
34100	020	\N	Bonnieux	84
34101	021	\N	Brantes	84
34102	022	\N	Buisson	84
34103	023	\N	Buoux	84
34104	024	\N	Cabrières-d'Aigues	84
34105	025	\N	Cabrières-d'Avignon	84
34106	026	\N	Cadenet	84
34107	027	\N	Caderousse	84
34108	028	\N	Cairanne	84
34109	029	\N	Camaret-sur-Aigues	84
34110	030	\N	Caromb	84
34111	031	\N	Carpentras	84
34112	032	\N	Caseneuve	84
34113	033	\N	Castellet	84
34114	034	\N	Caumont-sur-Durance	84
34115	035	\N	Cavaillon	84
34116	036	\N	Châteauneuf-de-Gadagne	84
34117	037	\N	Châteauneuf-du-Pape	84
34118	038	\N	Cheval-Blanc	84
34119	039	\N	Courthézon	84
34120	040	\N	Crestet	84
34121	041	\N	Crillon-le-Brave	84
34122	042	\N	Cucuron	84
34123	043	\N	Entraigues-sur-la-Sorgue	84
34124	044	\N	Entrechaux	84
34125	045	\N	Faucon	84
34126	046	\N	Flassan	84
34127	047	\N	Gargas	84
34128	048	\N	Gignac	84
34129	049	\N	Gigondas	84
34130	050	\N	Gordes	84
34131	051	\N	Goult	84
34132	052	\N	Grambois	84
34133	053	\N	Grillon	84
34134	054	\N	Isle-sur-la-Sorgue	84
34135	055	\N	Jonquerettes	84
34136	056	\N	Jonquières	84
34137	057	\N	Joucas	84
34138	058	\N	Lacoste	84
34139	059	\N	Lafare	84
34140	060	\N	Lagarde-d'Apt	84
34141	061	\N	Lagarde-Paréol	84
34142	062	\N	Lagnes	84
34143	063	\N	Lamotte-du-Rhône	84
34144	064	\N	Lapalud	84
34145	065	\N	Lauris	84
34146	066	\N	Lioux	84
34147	067	\N	Loriol-du-Comtat	84
34148	068	\N	Lourmarin	84
34149	069	\N	Malaucène	84
34150	070	\N	Malemort-du-Comtat	84
34151	071	\N	Maubec	84
34152	072	\N	Mazan	84
34153	073	\N	Ménerbes	84
34154	074	\N	Mérindol	84
34155	075	\N	Méthamis	84
34156	076	\N	Mirabeau	84
34157	077	\N	Modène	84
34158	078	\N	Mondragon	84
34159	079	\N	Monieux	84
34160	080	\N	Monteux	84
34161	081	\N	Morières-lès-Avignon	84
34162	082	\N	Mormoiron	84
34163	083	\N	Mornas	84
34164	084	\N	Motte-d'Aigues	84
34165	085	\N	Murs	84
34166	086	\N	Oppède	84
34167	087	\N	Orange	84
34168	088	\N	Pernes-les-Fontaines	84
34169	089	\N	Pertuis	84
34170	090	\N	Peypin-d'Aigues	84
34171	091	\N	Piolenc	84
34172	092	\N	Pontet	84
34173	093	\N	Puget	84
34174	094	\N	Puyméras	84
34175	095	\N	Puyvert	84
34176	096	\N	Rasteau	84
34177	097	\N	Richerenches	84
34178	098	\N	Roaix	84
34179	099	\N	Robion	84
34180	100	\N	Roque-Alric	84
34181	101	\N	Roque-sur-Pernes	84
34182	102	\N	Roussillon	84
34183	103	\N	Rustrel	84
34184	104	\N	Sablet	84
34185	105	\N	Saignon	84
34186	106	\N	Sainte-Cécile-les-Vignes	84
34187	107	\N	Saint-Christol	84
34188	108	\N	Saint-Didier	84
34189	109	\N	Saint-Hippolyte-le-Graveyron	84
34190	110	\N	Saint-Léger-du-Ventoux	84
34191	111	\N	Saint-Marcellin-lès-Vaison	84
34192	112	\N	Saint-Martin-de-Castillon	84
34193	113	\N	Saint-Martin-de-la-Brasque	84
34194	114	\N	Saint-Pantaléon	84
34195	115	\N	Saint-Pierre-de-Vassols	84
34196	116	\N	Saint-Romain-en-Viennois	84
34197	117	\N	Saint-Roman-de-Malegarde	84
34198	118	\N	Saint-Saturnin-lès-Apt	84
34199	119	\N	Saint-Saturnin-lès-Avignon	84
34200	120	\N	Saint-Trinit	84
34201	121	\N	Sannes	84
34202	122	\N	Sarrians	84
34203	123	\N	Sault	84
34204	124	\N	Saumane-de-Vaucluse	84
34205	125	\N	Savoillan	84
34206	126	\N	Séguret	84
34207	127	\N	Sérignan-du-Comtat	84
34208	128	\N	Sivergues	84
34209	129	\N	Sorgues	84
34210	130	\N	Suzette	84
34211	131	\N	Taillades	84
34212	132	\N	Thor	84
34213	133	\N	Tour-d'Aigues	84
34214	134	\N	Travaillan	84
34215	135	\N	Uchaux	84
34216	136	\N	Vacqueyras	84
34217	137	\N	Vaison-la-Romaine	84
34218	138	\N	Valréas	84
34219	139	\N	Fontaine-de-Vaucluse	84
34220	140	\N	Vaugines	84
34221	141	\N	Vedène	84
34222	142	\N	Velleron	84
34223	143	\N	Venasque	84
34224	144	\N	Viens	84
34225	145	\N	Villars	84
34226	146	\N	Villedieu	84
34227	147	\N	Villelaure	84
34228	148	\N	Villes-sur-Auzon	84
34229	149	\N	Violès	84
34230	150	\N	Visan	84
34231	151	\N	Vitrolles-en-Lubéron	84
34232	001	\N	Aiguillon-sur-Mer	85
34233	002	\N	Aiguillon-sur-Vie	85
34234	003	\N	Aizenay	85
34235	004	\N	Angles	85
34236	005	\N	Antigny	85
34237	006	\N	Apremont	85
34238	008	\N	Aubigny	85
34239	009	\N	Auzay	85
34240	010	\N	Avrillé	85
34241	011	\N	Barbâtre	85
34242	012	\N	Barre-de-Monts	85
34243	013	\N	Bazoges-en-Paillers	85
34244	014	\N	Bazoges-en-Pareds	85
34245	015	\N	Beaufou	85
34246	016	\N	Beaulieu-sous-la-Roche	85
34247	017	\N	Beaurepaire	85
34248	018	\N	Beauvoir-sur-Mer	85
34249	019	\N	Belleville-sur-Vie	85
34250	020	\N	Benet	85
34251	021	\N	Bernardière	85
34252	022	\N	Bernard	85
34253	023	\N	Bessay	85
34254	024	\N	Bois-de-Céné	85
34255	025	\N	Boissière-de-Montaigu	85
34256	026	\N	Boissière-des-Landes	85
34257	027	\N	Boufféré	85
34258	028	\N	Bouillé-Courdault	85
34259	029	\N	Bouin	85
34260	030	\N	Boulogne	85
34261	031	\N	Boupère	85
34262	033	\N	Bourneau	85
34263	034	\N	Bournezeau	85
34264	035	\N	Bretignolles-sur-Mer	85
34265	036	\N	Bretonnière-la-Claye	85
34266	037	\N	Breuil-Barret	85
34267	038	\N	Brouzils	85
34268	039	\N	Bruffière	85
34269	040	\N	Caillère-Saint-Hilaire	85
34270	041	\N	Cezais	85
34271	042	\N	Chaillé-les-Marais	85
34272	043	\N	Chaillé-sous-les-Ormeaux	85
34273	044	\N	Chaix	85
34274	045	\N	Chaize-Giraud	85
34275	046	\N	Chaize-le-Vicomte	85
34276	047	\N	Challans	85
34277	048	\N	Chambretaud	85
34278	049	\N	Champagné-les-Marais	85
34279	050	\N	Champ-Saint-Père	85
34280	051	\N	Chantonnay	85
34281	052	\N	Chapelle-Achard	85
34282	053	\N	Chapelle-aux-Lys	85
34283	054	\N	Chapelle-Hermier	85
34284	055	\N	Chapelle-Palluau	85
34285	056	\N	Chapelle-Thémer	85
34286	058	\N	Chasnais	85
34287	059	\N	Châtaigneraie	85
34288	060	\N	Château-d'Olonne	85
34289	061	\N	Château-Guibert	85
34290	062	\N	Châteauneuf	85
34291	063	\N	Châtelliers-Châteaumur	85
34292	064	\N	Chauché	85
34293	065	\N	Chavagnes-en-Paillers	85
34294	066	\N	Chavagnes-les-Redoux	85
34295	067	\N	Cheffois	85
34296	069	\N	Clouzeaux	85
34297	070	\N	Coëx	85
34298	071	\N	Commequiers	85
34299	072	\N	Copechagnière	85
34300	073	\N	Corpe	85
34301	074	\N	Couture	85
34302	076	\N	Cugand	85
34303	077	\N	Curzon	85
34304	078	\N	Damvix	85
34305	080	\N	Doix	85
34306	081	\N	Dompierre-sur-Yon	85
34307	082	\N	Epesses	85
34308	083	\N	Épine	85
34309	084	\N	Essarts	85
34310	086	\N	Falleron	85
34311	087	\N	Faymoreau	85
34312	088	\N	Fenouiller	85
34313	089	\N	Ferrière	85
34314	090	\N	Flocellière	85
34315	091	\N	Fontaines	85
34316	092	\N	Fontenay-le-Comte	85
34317	093	\N	Fougeré	85
34318	094	\N	Foussais-Payré	85
34319	095	\N	Froidfond	85
34320	096	\N	Garnache	85
34321	097	\N	Gaubretière	85
34322	098	\N	Génétouze	85
34323	099	\N	Girouard	85
34324	100	\N	Givrand	85
34325	101	\N	Givre	85
34326	102	\N	Grand'Landes	85
34327	103	\N	Grosbreuil	85
34328	104	\N	Grues	85
34329	105	\N	Gué-de-Velluire	85
34330	106	\N	Guérinière	85
34331	107	\N	Guyonnière	85
34332	108	\N	Herbergement	85
34333	109	\N	Herbiers	85
34334	110	\N	Hermenault	85
34335	111	\N	Île-d'Elle	85
34336	112	\N	Île-d'Olonne	85
34337	113	\N	Île-d'Yeu	85
34338	114	\N	Jard-sur-Mer	85
34339	115	\N	Jaudonnière	85
34340	116	\N	Jonchère	85
34341	117	\N	Lairoux	85
34342	118	\N	Landeronde	85
34343	119	\N	Landes-Genusson	85
34344	120	\N	Landevieille	85
34345	121	\N	Langon	85
34346	123	\N	Liez	85
34347	125	\N	Loge-Fougereuse	85
34348	126	\N	Longèves	85
34349	127	\N	Longeville-sur-Mer	85
34350	128	\N	Luçon	85
34351	129	\N	Lucs-sur-Boulogne	85
34352	130	\N	Maché	85
34353	131	\N	Magnils-Reigniers	85
34354	132	\N	Maillé	85
34355	133	\N	Maillezais	85
34356	134	\N	Mallièvre	85
34357	135	\N	Mareuil-sur-Lay-Dissais	85
34358	136	\N	Marillet	85
34359	137	\N	Marsais-Sainte-Radégonde	85
34360	138	\N	Martinet	85
34361	139	\N	Mazeau	85
34362	140	\N	Meilleraie-Tillay	85
34363	141	\N	Menomblet	85
34364	142	\N	Merlatière	85
34365	143	\N	Mervent	85
34366	144	\N	Mesnard-la-Barotière	85
34367	145	\N	Monsireigne	85
34368	146	\N	Montaigu	85
34369	147	\N	Montournais	85
34370	148	\N	Montreuil	85
34371	149	\N	Moreilles	85
34372	150	\N	Mormaison	85
34373	151	\N	Mortagne-sur-Sèvre	85
34374	152	\N	Mothe-Achard	85
34375	153	\N	Mouchamps	85
34376	154	\N	Mouilleron-en-Pareds	85
34377	155	\N	Mouilleron-le-Captif	85
34378	156	\N	Moutiers-les-Mauxfaits	85
34379	157	\N	Moutiers-sur-le-Lay	85
34380	158	\N	Mouzeuil-Saint-Martin	85
34381	159	\N	Nalliers	85
34382	160	\N	Nesmy	85
34383	161	\N	Nieul-le-Dolent	85
34384	162	\N	Nieul-sur-l'Autise	85
34385	163	\N	Noirmoutier-en-l'Île	85
34386	164	\N	Notre-Dame-de-Monts	85
34387	165	\N	Oie	85
34388	166	\N	Olonne-sur-Mer	85
34389	167	\N	Orbrie	85
34390	168	\N	Oulmes	85
34391	169	\N	Palluau	85
34392	171	\N	Péault	85
34393	172	\N	Perrier	85
34394	174	\N	Petosse	85
34395	175	\N	Pineaux	85
34396	176	\N	Pissotte	85
34397	177	\N	Poiré-sur-Velluire	85
34398	178	\N	Poiré-sur-Vie	85
34399	179	\N	Poiroux	85
34400	180	\N	Pommeraie-sur-Sèvre	85
34401	181	\N	Pouillé	85
34402	182	\N	Pouzauges	85
34403	184	\N	Puy-de-Serre	85
34404	185	\N	Puyravault	85
34405	186	\N	Rabatelière	85
34406	187	\N	Réaumur	85
34407	188	\N	Réorthe	85
34408	189	\N	Notre-Dame-de-Riez	85
34409	190	\N	Rocheservière	85
34410	191	\N	Roche-sur-Yon	85
34411	192	\N	Rochetrejoux	85
34412	193	\N	Rosnay	85
34413	194	\N	Sables-d'Olonne	85
34414	196	\N	Saint-André-Goule-d'Oie	85
34415	197	\N	Saint-André-Treize-Voies	85
34416	198	\N	Saint-Aubin-des-Ormeaux	85
34417	199	\N	Saint-Aubin-la-Plaine	85
34418	200	\N	Saint-Avaugourd-des-Landes	85
34419	201	\N	Saint-Benoist-sur-Mer	85
34420	202	\N	Sainte-Cécile	85
34421	204	\N	Saint-Christophe-du-Ligneron	85
34422	205	\N	Saint-Cyr-des-Gâts	85
34423	206	\N	Saint-Cyr-en-Talmondais	85
34424	207	\N	Saint-Denis-du-Payré	85
34425	208	\N	Saint-Denis-la-Chevasse	85
34426	209	\N	Saint-Étienne-de-Brillouet	85
34427	210	\N	Saint-Étienne-du-Bois	85
34428	211	\N	Sainte-Flaive-des-Loups	85
34429	212	\N	Sainte-Florence	85
34430	213	\N	Saint-Florent-des-Bois	85
34431	214	\N	Sainte-Foy	85
34432	215	\N	Saint-Fulgent	85
34433	216	\N	Sainte-Gemme-la-Plaine	85
34434	217	\N	Saint-Georges-de-Montaigu	85
34435	218	\N	Saint-Georges-de-Pointindoux	85
34436	219	\N	Saint-Germain-l'Aiguiller	85
34437	220	\N	Saint-Germain-de-Prinçay	85
34438	221	\N	Saint-Gervais	85
34439	222	\N	Saint-Gilles-Croix-de-Vie	85
34440	223	\N	Sainte-Hermine	85
34441	224	\N	Saint-Hilaire-de-Loulay	85
34442	226	\N	Saint-Hilaire-de-Riez	85
34443	227	\N	Saint-Hilaire-des-Loges	85
34444	229	\N	Saint-Hilaire-de-Voust	85
34445	231	\N	Saint-Hilaire-la-Forêt	85
34446	232	\N	Saint-Hilaire-le-Vouhis	85
34447	233	\N	Saint-Jean-de-Beugné	85
34448	234	\N	Saint-Jean-de-Monts	85
34449	235	\N	Saint-Juire-Champgillon	85
34450	236	\N	Saint-Julien-des-Landes	85
34451	237	\N	Saint-Laurent-de-la-Salle	85
34452	238	\N	Saint-Laurent-sur-Sèvre	85
34453	239	\N	Saint-Maixent-sur-Vie	85
34454	240	\N	Saint-Malô-du-Bois	85
34455	242	\N	Saint-Mars-la-Réorthe	85
34456	243	\N	Brem-sur-Mer	85
34457	244	\N	Saint-Martin-de-Fraigneau	85
34458	245	\N	Saint-Martin-des-Fontaines	85
34459	246	\N	Saint-Martin-des-Noyers	85
34460	247	\N	Saint-Martin-des-Tilleuls	85
34461	248	\N	Saint-Martin-Lars-en-Sainte-Hermine	85
34462	250	\N	Saint-Mathurin	85
34463	251	\N	Saint-Maurice-des-Noues	85
34464	252	\N	Saint-Maurice-le-Girard	85
34465	254	\N	Saint-Mesmin	85
34466	255	\N	Saint-Michel-en-l'Herm	85
34467	256	\N	Saint-Michel-le-Cloucq	85
34468	257	\N	Saint-Michel-Mont-Mercure	85
34469	259	\N	Saint-Paul-en-Pareds	85
34470	260	\N	Saint-Paul-Mont-Penit	85
34471	261	\N	Sainte-Pexine	85
34472	262	\N	Saint-Philbert-de-Bouaine	85
34473	264	\N	Saint-Pierre-du-Chemin	85
34474	265	\N	Saint-Pierre-le-Vieux	85
34475	266	\N	Saint-Prouant	85
34476	267	\N	Sainte-Radégonde-des-Noyers	85
34477	268	\N	Saint-Révérend	85
34478	269	\N	Saint-Sigismond	85
34479	271	\N	Saint-Sulpice-en-Pareds	85
34480	272	\N	Saint-Sulpice-le-Verdon	85
34481	273	\N	Saint-Urbain	85
34482	274	\N	Saint-Valérien	85
34483	276	\N	Saint-Vincent-Sterlanges	85
34484	277	\N	Saint-Vincent-sur-Graon	85
34485	278	\N	Saint-Vincent-sur-Jard	85
34486	279	\N	Saligny	85
34487	280	\N	Sallertaine	85
34488	281	\N	Sérigné	85
34489	282	\N	Sigournais	85
34490	284	\N	Soullans	85
34491	285	\N	Tablier	85
34492	286	\N	Taillée	85
34493	287	\N	Tallud-Sainte-Gemme	85
34494	288	\N	Talmont-Saint-Hilaire	85
34495	289	\N	Tardière	85
34496	290	\N	Thiré	85
34497	291	\N	Thorigny	85
34498	292	\N	Thouarsais-Bouildroux	85
34499	293	\N	Tiffauges	85
34500	294	\N	Tranche-sur-Mer	85
34501	295	\N	Treize-Septiers	85
34502	296	\N	Treize-Vents	85
34503	297	\N	Triaize	85
34504	298	\N	Vairé	85
34505	299	\N	Velluire	85
34506	300	\N	Venansault	85
34507	301	\N	Vendrennes	85
34508	302	\N	Verrie	85
34509	303	\N	Vix	85
34510	304	\N	Vouillé-les-Marais	85
34511	305	\N	Vouvant	85
34512	306	\N	Xanton-Chassenon	85
34513	307	\N	Faute-sur-Mer	85
34514	001	\N	Adriers	86
34515	002	\N	Amberre	86
34516	003	\N	Anché	86
34517	004	\N	Angles-sur-l'Anglin	86
34518	005	\N	Angliers	86
34519	006	\N	Antigny	86
34520	007	\N	Antran	86
34521	008	\N	Arçay	86
34522	009	\N	Archigny	86
34523	010	\N	Aslonnes	86
34524	011	\N	Asnières-sur-Blour	86
34525	012	\N	Asnois	86
34526	013	\N	Aulnay	86
34527	014	\N	Availles-en-Châtellerault	86
34528	015	\N	Availles-Limouzine	86
34529	016	\N	Avanton	86
34530	017	\N	Ayron	86
34531	018	\N	Basses	86
34532	019	\N	Beaumont	86
34533	020	\N	Bellefonds	86
34534	021	\N	Benassay	86
34535	022	\N	Berrie	86
34536	023	\N	Berthegon	86
34537	024	\N	Béruges	86
34538	025	\N	Béthines	86
34539	026	\N	Beuxes	86
34540	027	\N	Biard	86
34541	028	\N	Bignoux	86
34542	029	\N	Blanzay	86
34543	030	\N	Blaslay	86
34544	031	\N	Bonnes	86
34545	032	\N	Bonneuil-Matours	86
34546	034	\N	Bouresse	86
34547	035	\N	Bourg-Archambault	86
34548	036	\N	Bournand	86
34549	037	\N	Brigueil-le-Chantre	86
34550	038	\N	Brion	86
34551	039	\N	Brux	86
34552	040	\N	Bussière	86
34553	041	\N	Buxerolles	86
34554	042	\N	Buxeuil	86
34555	043	\N	Ceaux-en-Couhé	86
34556	044	\N	Ceaux-en-Loudun	86
34557	045	\N	Celle-Lévescault	86
34558	046	\N	Cenon-sur-Vienne	86
34559	047	\N	Cernay	86
34560	048	\N	Chabournay	86
34561	049	\N	Chalais	86
34562	050	\N	Chalandray	86
34563	051	\N	Champagné-le-Sec	86
34564	052	\N	Champagné-Saint-Hilaire	86
34565	053	\N	Champigny-le-Sec	86
34566	054	\N	Champniers	86
34567	055	\N	Chapelle-Bâton	86
34568	056	\N	Chapelle-Montreuil	86
34569	058	\N	Chapelle-Moulière	86
34570	059	\N	Chapelle-Viviers	86
34571	060	\N	Charrais	86
34572	061	\N	Charroux	86
34573	062	\N	Chasseneuil-du-Poitou	86
34574	063	\N	Chatain	86
34575	064	\N	Château-Garnier	86
34576	065	\N	Château-Larcher	86
34577	066	\N	Châtellerault	86
34578	067	\N	Châtillon	86
34579	068	\N	Chaunay	86
34580	069	\N	Chaussée	86
34581	070	\N	Chauvigny	86
34582	071	\N	Cheneché	86
34583	072	\N	Chenevelles	86
34584	073	\N	Cherves	86
34585	074	\N	Chiré-en-Montreuil	86
34586	075	\N	Chouppes	86
34587	076	\N	Cissé	86
34588	077	\N	Civaux	86
34589	078	\N	Civray	86
34590	079	\N	Roche-Rigault	86
34591	080	\N	Cloué	86
34592	081	\N	Colombiers	86
34593	082	\N	Couhé	86
34594	083	\N	Coulombiers	86
34595	084	\N	Coulonges	86
34596	085	\N	Coussay	86
34597	086	\N	Coussay-les-Bois	86
34598	087	\N	Craon	86
34599	088	\N	Croutelle	86
34600	089	\N	Cuhon	86
34601	090	\N	Curçay-sur-Dive	86
34602	091	\N	Curzay-sur-Vonne	86
34603	092	\N	Dangé-Saint-Romain	86
34604	093	\N	Dercé	86
34605	094	\N	Dienné	86
34606	095	\N	Dissay	86
34607	096	\N	Doussay	86
34608	097	\N	Ferrière-Airoux	86
34609	098	\N	Fleix	86
34610	099	\N	Fleuré	86
34611	100	\N	Fontaine-le-Comte	86
34612	102	\N	Frozes	86
34613	103	\N	Gençay	86
34614	104	\N	Genouillé	86
34615	105	\N	Gizay	86
34616	106	\N	Glénouze	86
34617	107	\N	Gouex	86
34618	108	\N	Grimaudière	86
34619	109	\N	Guesnes	86
34620	110	\N	Haims	86
34621	111	\N	Ingrandes	86
34622	112	\N	Isle-Jourdain	86
34623	113	\N	Iteuil	86
34624	114	\N	Jardres	86
34625	115	\N	Jaunay-Clan	86
34626	116	\N	Jazeneuil	86
34627	117	\N	Jouhet	86
34628	118	\N	Journet	86
34629	119	\N	Joussé	86
34630	120	\N	Lathus-Saint-Rémy	86
34631	121	\N	Latillé	86
34632	122	\N	Lauthiers	86
34633	123	\N	Lavausseau	86
34634	124	\N	Lavoux	86
34635	125	\N	Leigné-les-Bois	86
34636	126	\N	Leignes-sur-Fontaine	86
34637	127	\N	Leigné-sur-Usseau	86
34638	128	\N	Lencloître	86
34639	129	\N	Lésigny	86
34640	130	\N	Leugny	86
34641	131	\N	Lhommaizé	86
34642	132	\N	Liglet	86
34643	133	\N	Ligugé	86
34644	134	\N	Linazay	86
34645	135	\N	Liniers	86
34646	136	\N	Lizant	86
34647	137	\N	Loudun	86
34648	138	\N	Luchapt	86
34649	139	\N	Lusignan	86
34650	140	\N	Lussac-les-Châteaux	86
34651	141	\N	Magné	86
34652	142	\N	Maillé	86
34653	143	\N	Mairé	86
34654	144	\N	Maisonneuve	86
34655	145	\N	Marçay	86
34656	146	\N	Marigny-Brizay	86
34657	147	\N	Marigny-Chemereau	86
34658	148	\N	Marnay	86
34659	149	\N	Martaizé	86
34660	150	\N	Massognes	86
34661	151	\N	Maulay	86
34662	152	\N	Mauprévoir	86
34663	153	\N	Mazerolles	86
34664	154	\N	Mazeuil	86
34665	156	\N	Messemé	86
34666	157	\N	Mignaloux-Beauvoir	86
34667	158	\N	Migné-Auxances	86
34668	159	\N	Millac	86
34669	160	\N	Mirebeau	86
34670	161	\N	Moncontour	86
34671	162	\N	Mondion	86
34672	163	\N	Montamisé	86
34673	164	\N	Monthoiron	86
34674	165	\N	Montmorillon	86
34675	166	\N	Montreuil-Bonnin	86
34676	167	\N	Monts-sur-Guesnes	86
34677	169	\N	Morton	86
34678	170	\N	Moulismes	86
34679	171	\N	Moussac	86
34680	172	\N	Mouterre-sur-Blourde	86
34681	173	\N	Mouterre-Silly	86
34682	174	\N	Naintré	86
34683	175	\N	Nalliers	86
34684	176	\N	Nérignac	86
34685	177	\N	Neuville-de-Poitou	86
34686	178	\N	Nieuil-l'Espoir	86
34687	180	\N	Nouaillé-Maupertuis	86
34688	181	\N	Nueil-sous-Faye	86
34689	182	\N	Orches	86
34690	183	\N	Ormes	86
34691	184	\N	Ouzilly	86
34692	186	\N	Oyré	86
34693	187	\N	Paizay-le-Sec	86
34694	188	\N	Payré	86
34695	189	\N	Payroux	86
34696	190	\N	Persac	86
34697	191	\N	Pindray	86
34698	192	\N	Plaisance	86
34699	193	\N	Pleumartin	86
34700	194	\N	Poitiers	86
34701	195	\N	Port-de-Piles	86
34702	196	\N	Pouançay	86
34703	197	\N	Pouant	86
34704	198	\N	Pouillé	86
34705	200	\N	Pressac	86
34706	201	\N	Prinçay	86
34707	202	\N	Puye	86
34708	203	\N	Queaux	86
34709	204	\N	Quinçay	86
34710	205	\N	Ranton	86
34711	206	\N	Raslay	86
34712	207	\N	Roche-Posay	86
34713	208	\N	Rochereau	86
34714	209	\N	Roches-Prémarie-Andillé	86
34715	210	\N	Roiffé	86
34716	211	\N	Romagne	86
34717	213	\N	Rouillé	86
34718	214	\N	Saint-Benoît	86
34719	217	\N	Saint-Christophe	86
34720	218	\N	Saint-Clair	86
34721	219	\N	Saint-Cyr	86
34722	220	\N	Saint-Gaudent	86
34723	221	\N	Saint-Genest-d'Ambière	86
34724	222	\N	Saint-Georges-lès-Baillargeaux	86
34725	223	\N	Saint-Germain	86
34726	224	\N	Saint-Gervais-les-Trois-Clochers	86
34727	225	\N	Saint-Jean-de-Sauves	86
34728	226	\N	Saint-Julien-l'Ars	86
34729	227	\N	Saint-Laon	86
34730	228	\N	Saint-Laurent-de-Jourdes	86
34731	229	\N	Saint-Léger-de-Montbrillais	86
34732	230	\N	Saint-Léomer	86
34733	231	\N	Saint-Macoux	86
34734	233	\N	Valdivienne	86
34735	234	\N	Saint-Martin-l'Ars	86
34736	235	\N	Saint-Maurice-la-Clouère	86
34737	236	\N	Saint-Pierre-de-Maillé	86
34738	237	\N	Saint-Pierre-d'Exideuil	86
34739	239	\N	Sainte-Radégonde	86
34740	241	\N	Saint-Rémy-sur-Creuse	86
34741	242	\N	Saint-Romain	86
34742	244	\N	Saint-Sauvant	86
34743	245	\N	Saint-Sauveur	86
34744	246	\N	Saint-Savin	86
34745	247	\N	Saint-Saviol	86
34746	248	\N	Saint-Secondin	86
34747	249	\N	Saires	86
34748	250	\N	Saix	86
34749	252	\N	Sammarçolles	86
34750	253	\N	Sanxay	86
34751	254	\N	Saulgé	86
34752	255	\N	Savigné	86
34753	256	\N	Savigny-Lévescault	86
34754	257	\N	Savigny-sous-Faye	86
34755	258	\N	Scorbé-Clairvaux	86
34756	259	\N	Senillé	86
34757	260	\N	Sérigny	86
34758	261	\N	Sèvres-Anxaumont	86
34759	262	\N	Sillars	86
34760	263	\N	Smarves	86
34761	264	\N	Sommières-du-Clain	86
34762	265	\N	Sossais	86
34763	266	\N	Surin	86
34764	268	\N	Tercé	86
34765	269	\N	Ternay	86
34766	270	\N	Thollet	86
34767	271	\N	Thurageau	86
34768	272	\N	Thuré	86
34769	273	\N	Trimouille	86
34770	274	\N	Trois-Moutiers	86
34771	275	\N	Usseau	86
34772	276	\N	Usson-du-Poitou	86
34773	277	\N	Varennes	86
34774	278	\N	Vaux	86
34775	279	\N	Vaux-sur-Vienne	86
34776	280	\N	Vellèches	86
34777	281	\N	Vendeuvre-du-Poitou	86
34778	284	\N	Vernon	86
34779	285	\N	Verrières	86
34780	286	\N	Verrue	86
34781	287	\N	Vézières	86
34782	288	\N	Vicq-sur-Gartempe	86
34783	289	\N	Vigeant	86
34784	290	\N	Villedieu-du-Clain	86
34785	291	\N	Villemort	86
34786	292	\N	Villiers	86
34787	293	\N	Vivonne	86
34788	294	\N	Vouillé	86
34789	295	\N	Voulême	86
34790	296	\N	Voulon	86
34791	297	\N	Vouneuil-sous-Biard	86
34792	298	\N	Vouneuil-sur-Vienne	86
34793	299	\N	Vouzailles	86
34794	300	\N	Yversay	86
34795	001	\N	Aixe-sur-Vienne	87
34796	002	\N	Ambazac	87
34797	003	\N	Arnac-la-Poste	87
34798	004	\N	Augne	87
34799	005	\N	Aureil	87
34800	006	\N	Azat-le-Ris	87
34801	007	\N	Balledent	87
34802	008	\N	Bazeuge	87
34803	009	\N	Beaumont-du-Lac	87
34804	011	\N	Bellac	87
34805	012	\N	Berneuil	87
34806	013	\N	Bersac-sur-Rivalier	87
34807	014	\N	Bessines-sur-Gartempe	87
34808	015	\N	Beynac	87
34809	016	\N	Billanges	87
34810	017	\N	Blanzac	87
34811	018	\N	Blond	87
34812	019	\N	Boisseuil	87
34813	020	\N	Bonnac-la-Côte	87
34814	021	\N	Bosmie-l'Aiguille	87
34815	022	\N	Breuilaufa	87
34816	023	\N	Buis	87
34817	024	\N	Bujaleuf	87
34818	025	\N	Burgnac	87
34819	026	\N	Bussière-Boffy	87
34820	027	\N	Bussière-Galant	87
34821	028	\N	Bussière-Poitevine	87
34822	029	\N	Cars	87
34823	030	\N	Chaillac-sur-Vienne	87
34824	031	\N	Chalard	87
34825	032	\N	Châlus	87
34826	033	\N	Chamboret	87
34827	034	\N	Champagnac-la-Rivière	87
34828	035	\N	Champnétery	87
34829	036	\N	Champsac	87
34830	037	\N	Chapelle-Montbrandeix	87
34831	038	\N	Chaptelat	87
34832	039	\N	Château-Chervix	87
34833	040	\N	Châteauneuf-la-Forêt	87
34834	041	\N	Châteauponsac	87
34835	042	\N	Châtenet-en-Dognon	87
34836	043	\N	Cheissoux	87
34837	044	\N	Chéronnac	87
34838	045	\N	Cieux	87
34839	046	\N	Cognac-la-Forêt	87
34840	047	\N	Compreignac	87
34841	048	\N	Condat-sur-Vienne	87
34842	049	\N	Coussac-Bonneval	87
34843	050	\N	Couzeix	87
34844	051	\N	Croisille-sur-Briance	87
34845	052	\N	Croix-sur-Gartempe	87
34846	053	\N	Cromac	87
34847	054	\N	Cussac	87
34848	055	\N	Darnac	87
34849	056	\N	Dinsac	87
34850	057	\N	Dompierre-les-Églises	87
34851	058	\N	Domps	87
34852	059	\N	Dorat	87
34853	060	\N	Dournazac	87
34854	061	\N	Droux	87
34855	062	\N	Eybouleuf	87
34856	063	\N	Eyjeaux	87
34857	064	\N	Eymoutiers	87
34858	065	\N	Feytiat	87
34859	066	\N	Flavignac	87
34860	067	\N	Folles	87
34861	068	\N	Fromental	87
34862	069	\N	Gajoubert	87
34863	070	\N	Geneytouse	87
34864	071	\N	Glandon	87
34865	072	\N	Glanges	87
34866	073	\N	Gorre	87
34867	074	\N	Grands-Chézeaux	87
34868	075	\N	Isle	87
34869	076	\N	Jabreilles-les-Bordes	87
34870	077	\N	Janailhac	87
34871	078	\N	Javerdat	87
34872	079	\N	Jonchère-Saint-Maurice	87
34873	080	\N	Jouac	87
34874	081	\N	Jourgnac	87
34875	082	\N	Ladignac-le-Long	87
34876	083	\N	Laurière	87
34877	084	\N	Lavignac	87
34878	085	\N	Limoges	87
34879	086	\N	Linards	87
34880	087	\N	Lussac-les-Églises	87
34881	088	\N	Magnac-Bourg	87
34882	089	\N	Magnac-Laval	87
34883	090	\N	Mailhac-sur-Benaize	87
34884	091	\N	Maisonnais-sur-Tardoire	87
34885	092	\N	Marval	87
34886	093	\N	Masléon	87
34887	094	\N	Meilhac	87
34888	095	\N	Meuzac	87
34889	096	\N	Meyze	87
34890	097	\N	Mézières-sur-Issoire	87
34891	099	\N	Moissannes	87
34892	100	\N	Montrol-Sénard	87
34893	101	\N	Mortemart	87
34894	103	\N	Nantiat	87
34895	104	\N	Nedde	87
34896	105	\N	Neuvic-Entier	87
34897	106	\N	Nexon	87
34898	107	\N	Nieul	87
34899	108	\N	Nouic	87
34900	109	\N	Oradour-Saint-Genest	87
34901	110	\N	Oradour-sur-Glane	87
34902	111	\N	Oradour-sur-Vayres	87
34903	112	\N	Pageas	87
34904	113	\N	Palais-sur-Vienne	87
34905	114	\N	Panazol	87
34906	115	\N	Pensol	87
34907	116	\N	Peyrat-de-Bellac	87
34908	117	\N	Peyrat-le-Château	87
34909	118	\N	Peyrilhac	87
34910	119	\N	Pierre-Buffière	87
34911	120	\N	Porcherie	87
34912	121	\N	Rancon	87
34913	122	\N	Razès	87
34914	123	\N	Rempnat	87
34915	124	\N	Rilhac-Lastours	87
34916	125	\N	Rilhac-Rancon	87
34917	126	\N	Rochechouart	87
34918	127	\N	Roche-l'Abeille	87
34919	128	\N	Roussac	87
34920	129	\N	Royères	87
34921	130	\N	Roziers-Saint-Georges	87
34922	131	\N	Saillat-sur-Vienne	87
34923	132	\N	Saint-Amand-le-Petit	87
34924	133	\N	Saint-Amand-Magnazeix	87
34925	134	\N	Sainte-Anne-Saint-Priest	87
34926	135	\N	Saint-Auvent	87
34927	136	\N	Saint-Barbant	87
34928	137	\N	Saint-Bazile	87
34929	138	\N	Saint-Bonnet-Briance	87
34930	139	\N	Saint-Bonnet-de-Bellac	87
34931	140	\N	Saint-Brice-sur-Vienne	87
34932	141	\N	Saint-Cyr	87
34933	142	\N	Saint-Denis-des-Murs	87
34934	143	\N	Saint-Gence	87
34935	144	\N	Saint-Genest-sur-Roselle	87
34936	145	\N	Saint-Georges-les-Landes	87
34937	146	\N	Saint-Germain-les-Belles	87
34938	147	\N	Saint-Gilles-les-Forêts	87
34939	148	\N	Saint-Hilaire-Bonneval	87
34940	149	\N	Saint-Hilaire-la-Treille	87
34941	150	\N	Saint-Hilaire-les-Places	87
34942	151	\N	Saint-Jean-Ligoure	87
34943	152	\N	Saint-Jouvent	87
34944	153	\N	Saint-Julien-le-Petit	87
34945	154	\N	Saint-Junien	87
34946	155	\N	Saint-Junien-les-Combes	87
34947	156	\N	Saint-Just-le-Martel	87
34948	157	\N	Saint-Laurent-les-Églises	87
34949	158	\N	Saint-Laurent-sur-Gorre	87
34950	159	\N	Saint-Léger-la-Montagne	87
34951	160	\N	Saint-Léger-Magnazeix	87
34952	161	\N	Saint-Léonard-de-Noblat	87
34953	162	\N	Sainte-Marie-de-Vaux	87
34954	163	\N	Saint-Martial-sur-Isop	87
34955	164	\N	Saint-Martin-de-Jussac	87
34956	165	\N	Saint-Martin-le-Mault	87
34957	166	\N	Saint-Martin-le-Vieux	87
34958	167	\N	Saint-Martin-Terressus	87
34959	168	\N	Saint-Mathieu	87
34960	169	\N	Saint-Maurice-les-Brousses	87
34961	170	\N	Saint-Méard	87
34962	172	\N	Saint-Ouen-sur-Gartempe	87
34963	173	\N	Saint-Pardoux	87
34964	174	\N	Saint-Paul	87
34965	176	\N	Saint-Priest-Ligoure	87
34966	177	\N	Saint-Priest-sous-Aixe	87
34967	178	\N	Saint-Priest-Taurion	87
34968	179	\N	Saint-Sornin-la-Marche	87
34969	180	\N	Saint-Sornin-Leulac	87
34970	181	\N	Saint-Sulpice-Laurière	87
34971	182	\N	Saint-Sulpice-les-Feuilles	87
34972	183	\N	Saint-Sylvestre	87
34973	184	\N	Saint-Symphorien-sur-Couze	87
34974	185	\N	Saint-Victurnien	87
34975	186	\N	Saint-Vitte-sur-Briance	87
34976	187	\N	Saint-Yrieix-la-Perche	87
34977	188	\N	Saint-Yrieix-sous-Aixe	87
34978	189	\N	Salles-Lavauguyon	87
34979	190	\N	Sauviat-sur-Vige	87
34980	191	\N	Séreilhac	87
34981	192	\N	Solignac	87
34982	193	\N	Surdoux	87
34983	194	\N	Sussac	87
34984	195	\N	Tersannes	87
34985	196	\N	Thiat	87
34986	197	\N	Thouron	87
34987	198	\N	Vaulry	87
34988	199	\N	Vayres	87
34989	200	\N	Verneuil-Moustiers	87
34990	201	\N	Verneuil-sur-Vienne	87
34991	202	\N	Veyrac	87
34992	203	\N	Vicq-sur-Breuilh	87
34993	204	\N	Videix	87
34994	205	\N	Vigen	87
34995	206	\N	Villefavard	87
34996	001	\N	Ableuvenettes	88
34997	002	\N	Ahéville	88
34998	003	\N	Aingeville	88
34999	004	\N	Ainvelle	88
35000	005	\N	Allarmont	88
35001	006	\N	Ambacourt	88
35002	007	\N	Ameuvelle	88
35003	008	\N	Anglemont	88
35004	009	\N	Anould	88
35005	010	\N	Aouze	88
35006	011	\N	Arches	88
35007	012	\N	Archettes	88
35008	013	\N	Aroffe	88
35009	014	\N	Arrentès-de-Corcieux	88
35010	015	\N	Attignéville	88
35011	016	\N	Attigny	88
35012	017	\N	Aulnois	88
35013	018	\N	Aumontzey	88
35014	019	\N	Autigny-la-Tour	88
35015	020	\N	Autreville	88
35016	021	\N	Autrey	88
35017	022	\N	Auzainvilliers	88
35018	023	\N	Avillers	88
35019	024	\N	Avrainville	88
35020	025	\N	Avranville	88
35021	026	\N	Aydoilles	88
35022	027	\N	Badménil-aux-Bois	88
35023	028	\N	Baffe	88
35024	029	\N	Bains-les-Bains	88
35025	030	\N	Bainville-aux-Saules	88
35026	031	\N	Balléville	88
35027	032	\N	Ban-de-Laveline	88
35028	033	\N	Ban-de-Sapt	88
35029	035	\N	Barbey-Seroux	88
35030	036	\N	Barville	88
35031	037	\N	Basse-sur-le-Rupt	88
35032	038	\N	Battexey	88
35033	039	\N	Baudricourt	88
35034	040	\N	Bayecourt	88
35035	041	\N	Bazegney	88
35036	042	\N	Bazien	88
35037	043	\N	Bazoilles-et-Ménil	88
35038	044	\N	Bazoilles-sur-Meuse	88
35039	045	\N	Beaufremont	88
35040	046	\N	Beauménil	88
35041	047	\N	Begnécourt	88
35042	048	\N	Bellefontaine	88
35043	049	\N	Belmont-lès-Darney	88
35044	050	\N	Belmont-sur-Buttant	88
35045	051	\N	Belmont-sur-Vair	88
35046	052	\N	Belrupt	88
35047	053	\N	Belval	88
35048	054	\N	Bertrimoutier	88
35049	055	\N	Bettegney-Saint-Brice	88
35050	056	\N	Bettoncourt	88
35051	057	\N	Beulay	88
35052	058	\N	Biécourt	88
35053	059	\N	Biffontaine	88
35054	060	\N	Blémerey	88
35055	061	\N	Bleurville	88
35056	062	\N	Blevaincourt	88
35057	063	\N	Bocquegney	88
35058	064	\N	Bois-de-Champ	88
35059	065	\N	Bonvillet	88
35060	066	\N	Boulaincourt	88
35061	068	\N	Bourgonce	88
35062	069	\N	Bouxières-aux-Bois	88
35063	070	\N	Bouxurulles	88
35064	071	\N	Bouzemont	88
35065	073	\N	Brantigny	88
35066	074	\N	Brechainville	88
35067	075	\N	Bresse	88
35068	076	\N	Brouvelieures	88
35069	077	\N	Brû	88
35070	078	\N	Bruyères	88
35071	079	\N	Bulgnéville	88
35072	080	\N	Bult	88
35073	081	\N	Bussang	88
35074	082	\N	Celles-sur-Plaine	88
35075	083	\N	Certilleux	88
35076	084	\N	Chamagne	88
35077	085	\N	Champdray	88
35078	086	\N	Champ-le-Duc	88
35079	087	\N	Chantraine	88
35080	088	\N	Chapelle-aux-Bois	88
35081	089	\N	Chapelle-devant-Bruyères	88
35082	090	\N	Charmes	88
35083	091	\N	Charmois-devant-Bruyères	88
35084	092	\N	Charmois-l'Orgueilleux	88
35085	093	\N	Châtas	88
35086	094	\N	Châtel-sur-Moselle	88
35087	095	\N	Châtenois	88
35088	096	\N	Châtillon-sur-Saône	88
35089	097	\N	Chauffecourt	88
35090	098	\N	Chaumousey	88
35091	099	\N	Chavelot	88
35092	100	\N	Chef-Haut	88
35093	101	\N	Cheniménil	88
35094	102	\N	Chermisey	88
35095	103	\N	Circourt	88
35096	104	\N	Circourt-sur-Mouzon	88
35097	105	\N	Claudon	88
35098	106	\N	Ban-sur-Meurthe-Clefcy	88
35099	107	\N	Clérey-la-Côte	88
35100	108	\N	Clerjus	88
35101	109	\N	Cleurie	88
35102	110	\N	Clézentaine	88
35103	111	\N	Coinches	88
35104	112	\N	Colroy-la-Grande	88
35105	113	\N	Combrimont	88
35106	114	\N	Contrexéville	88
35107	115	\N	Corcieux	88
35108	116	\N	Cornimont	88
35109	117	\N	Courcelles-sous-Châtenois	88
35110	118	\N	Coussey	88
35111	119	\N	Crainvilliers	88
35112	120	\N	Croix-aux-Mines	88
35113	121	\N	Damas-aux-Bois	88
35114	122	\N	Damas-et-Bettegney	88
35115	123	\N	Damblain	88
35116	124	\N	Darney	88
35117	125	\N	Darney-aux-Chênes	88
35118	126	\N	Darnieulles	88
35119	127	\N	Deinvillers	88
35120	128	\N	Denipaire	88
35121	129	\N	Derbamont	88
35122	130	\N	Destord	88
35123	131	\N	Deycimont	88
35124	132	\N	Deyvillers	88
35125	133	\N	Dignonville	88
35126	134	\N	Dinozé	88
35127	135	\N	Docelles	88
35128	136	\N	Dogneville	88
35129	137	\N	Dolaincourt	88
35130	138	\N	Dombasle-devant-Darney	88
35131	139	\N	Dombasle-en-Xaintois	88
35132	140	\N	Dombrot-le-Sec	88
35133	141	\N	Dombrot-sur-Vair	88
35134	142	\N	Domèvre-sur-Avière	88
35135	143	\N	Domèvre-sur-Durbion	88
35136	144	\N	Domèvre-sous-Montfort	88
35137	145	\N	Domfaing	88
35138	146	\N	Domjulien	88
35139	147	\N	Dommartin-aux-Bois	88
35140	148	\N	Dommartin-lès-Remiremont	88
35141	149	\N	Dommartin-lès-Vallois	88
35142	150	\N	Dommartin-sur-Vraine	88
35143	151	\N	Dompaire	88
35144	152	\N	Dompierre	88
35145	153	\N	Domptail	88
35146	154	\N	Domrémy-la-Pucelle	88
35147	155	\N	Domvallier	88
35148	156	\N	Doncières	88
35149	157	\N	Dounoux	88
35150	158	\N	Éloyes	88
35151	159	\N	Entre-deux-Eaux	88
35152	160	\N	Épinal	88
35153	161	\N	Escles	88
35154	162	\N	Esley	88
35155	163	\N	Essegney	88
35156	164	\N	Estrennes	88
35157	165	\N	Étival-Clairefontaine	88
35158	166	\N	Évaux-et-Ménil	88
35159	167	\N	Faucompierre	88
35160	168	\N	Fauconcourt	88
35161	169	\N	Fays	88
35162	170	\N	Ferdrupt	88
35163	171	\N	Fignévelle	88
35164	172	\N	Fiménil	88
35165	173	\N	Florémont	88
35166	174	\N	Fomerey	88
35167	175	\N	Fontenay	88
35168	176	\N	Fontenoy-le-Château	88
35169	177	\N	Forge	88
35170	178	\N	Forges	88
35171	179	\N	Fouchécourt	88
35172	180	\N	Frain	88
35173	181	\N	Fraize	88
35174	182	\N	Frapelle	88
35175	183	\N	Frebécourt	88
35176	184	\N	Fremifontaine	88
35177	185	\N	Frenelle-la-Grande	88
35178	186	\N	Frenelle-la-Petite	88
35179	187	\N	Frénois	88
35180	188	\N	Fresse-sur-Moselle	88
35181	189	\N	Fréville	88
35182	190	\N	Frizon	88
35183	192	\N	Gelvécourt-et-Adompt	88
35184	193	\N	Gemaingoutte	88
35185	194	\N	Gemmelaincourt	88
35186	195	\N	Gendreville	88
35187	196	\N	Gérardmer	88
35188	197	\N	Gerbamont	88
35189	198	\N	Gerbépal	88
35190	199	\N	Gignéville	88
35191	200	\N	Gigney	88
35192	201	\N	Girancourt	88
35193	202	\N	Gircourt-lès-Viéville	88
35194	203	\N	Girecourt-sur-Durbion	88
35195	204	\N	Girmont	88
35196	205	\N	Girmont-Val-d'Ajol	88
35197	206	\N	Gironcourt-sur-Vraine	88
35198	208	\N	Godoncourt	88
35199	209	\N	Golbey	88
35200	210	\N	Gorhey	88
35201	212	\N	Grand	88
35202	213	\N	Grande-Fosse	88
35203	214	\N	Grandrupt-de-Bains	88
35204	215	\N	Grandrupt	88
35205	216	\N	Grandvillers	88
35206	218	\N	Granges-sur-Vologne	88
35207	219	\N	Greux	88
35208	220	\N	Grignoncourt	88
35209	221	\N	Gruey-lès-Surance	88
35210	222	\N	Gugnécourt	88
35211	223	\N	Gugney-aux-Aulx	88
35212	224	\N	Hadigny-les-Verrières	88
35213	225	\N	Hadol	88
35214	226	\N	Hagécourt	88
35215	227	\N	Hagnéville-et-Roncourt	88
35216	228	\N	Haillainville	88
35217	229	\N	Harchéchamp	88
35218	230	\N	Hardancourt	88
35219	231	\N	Haréville	88
35220	232	\N	Harmonville	88
35221	233	\N	Harol	88
35222	234	\N	Harsault	88
35223	235	\N	Hautmougey	88
35224	236	\N	Haye	88
35225	237	\N	Hennecourt	88
35226	238	\N	Hennezel	88
35227	239	\N	Hergugney	88
35228	240	\N	Herpelmont	88
35229	241	\N	Houécourt	88
35230	242	\N	Houéville	88
35231	243	\N	Housseras	88
35232	244	\N	Houssière	88
35233	245	\N	Hurbache	88
35234	246	\N	Hymont	88
35235	247	\N	Igney	88
35236	248	\N	Isches	88
35237	249	\N	Jainvillotte	88
35238	250	\N	Jarménil	88
35239	251	\N	Jeanménil	88
35240	252	\N	Jésonville	88
35241	253	\N	Jeuxey	88
35242	254	\N	Jorxey	88
35243	255	\N	Jubainville	88
35244	256	\N	Jussarupt	88
35245	257	\N	Juvaincourt	88
35246	258	\N	Lamarche	88
35247	259	\N	Landaville	88
35248	260	\N	Langley	88
35249	261	\N	Laval-sur-Vologne	88
35250	262	\N	Laveline-devant-Bruyères	88
35251	263	\N	Laveline-du-Houx	88
35252	264	\N	Légéville-et-Bonfays	88
35253	265	\N	Lemmecourt	88
35254	266	\N	Lépanges-sur-Vologne	88
35255	267	\N	Lerrain	88
35256	2