/*-------------------------------------------------------------------------
 *
 * proc.h--
 *
 *
 *
 * Copyright (c) 1994, Regents of the University of California
 *
 * $Id: proc.h,v 1.2 2001/06/12 14:51:59 johnsond Exp $
 *
 *
 *	Note(wb):
 *		The process management has been extended to allow a single process
 *		to wait for more than one lock. Up to now, the whole process 
 *		descriptor is inserted into the lock queue. It contains data
 *		pertaining to the process, as well as to the lock queue. However,
 *		when a process has to wait in several lock queues, 
 *		this scheme is not adequate. To this end, I have introduced "process
 *		shadows", which are structures containing data pertaining to a lock 
 *		queue, and a reference to a process structure, which contains the
 *		process description(e.g. semaphore). Processes are no longer
 *		inserted into the lock queues, process shadows take their place.
 *		This allows us to have several process shadows in different lock 
 *		queues which are, however, all pointing to the same process descriptor.
 *
 *
 *-------------------------------------------------------------------------
 */
#ifndef _PROC_H_
#define _PROC_H_


//#define PROC_DEBUG
#include <storage/lock.h>
#include "replication/replication.h"




typedef struct sema
{
  int			sleeplock;
  int			semNum;
  IpcSemaphoreId semId;
  IpcSemaphoreKey semKey;
} SEMA;

typedef enum abortFlagType
{
  NO_ABORT,
  MUST_ABORT,
  ABORTING
} abortFlagType;

/*
 * Each backend has:
 */
typedef struct proc
{
  
  /* proc->links MUST BE THE FIRST ELEMENT OF STRUCT (see ProcWakeup()) */
  
  SHM_QUEUE	links;		/* only for the GlobalProc 
				   freelist, not for lock wait queues */
  SEMA		sem;		/* ONE semaphore to sleep on */
  SEMA		sem_multiple;  /* bk: i need a second semaphore, to be able to
							* distinguish in the write phase between a normal
							* lock acquisition on a system table or the multiple
							* lock acquisition on the user tables */
  int		errType;	/* error code tells why we woke up */
  
  int		critSects;	/* If critSects > 0, we are in sensitive
				 * routines that cannot be recovered when
				 * the process fails. */
  
  //int		prio;		/* priority for sleep queue */
  
  TransactionId xid;		/* transaction currently being executed by
				 * this proc */
  
#ifdef LowLevelLocking
  TransactionId xmin;		/* minimal running XID as it was when we
				 * were starting our xact: vacuum must not
				 * remove tuples deleted by xid >= xmin ! */
#endif
  
  //LOCK	*waitLock;	/* Lock we're sleeping on */
  //int		token;		/* info for proc wakeup routines */
  int		pid;		/* This procs process id */
  short		sLocks[MAX_SPINS]; /* Spin lock stats */
  SHM_QUEUE	shadowList;	   /*shadows associated with current transaction*/
  SHM_QUEUE	lockQueue;	   /* locks associated with current
				    * transaction */
  /*
   *	(wb) State info is used to determine wether the transaction
   *	associated to this process has to be aborted because a write set
   *	belonging to another txn has arrived before its own ws. The replication 
   *	protocol enforces the abortion of a txn that is in read phase and has not
   *	yet set its write locks if a write set of another txn requests a lock held
   *	by the current txn.
   */
  rc_state_t	state;
   rc_connection_t  connection_state;
  abortFlagType		abortFlag;
} PROC;


/*
 *	The shadow tag is the hastable key for the process shadow hash table
 */
typedef struct tag{
  int		pid;  /*the process id associated with this shadow*/
  SHMEM_OFFSET 	lock; /*the lock id this shadow was created for and sleeps on*/
} SHDWTAG;

/*
 *	For each (might be more than one) lock the backend is waiting on,
 *	a PROC_SHADOW struct is inserted into the lock waiting list.
 */
typedef struct proc_shadow{
  SHDWTAG	tag;	/*shadow hashtable key*/
  SHM_QUEUE 	waitLinks; 	/* next shadow in this lock queue*/
  SHM_QUEUE	shadowLinks;	/*shadows associated with a proc*/
  XIDLookupEnt  *xidentP;       /* bk: pointer to corresponding XID-entry */
  int		prio;	/*priority for this sleep queue*/
  REQ_MODE  req_mode;  /* bk: single: by ProcSleep, multiple: by ProcSleepOnWriteSet */
  LOCK		*waitLock;  	/*lock this shadow sleeps on*/
  int 		token;		/*info for wakeup routines*/
  SHMEM_OFFSET	proc;			/*proc associated to this shadow*/
} PROC_SHADOW;

/*
 * MAX_PROC_SEMS is the maximum number of per-process semaphores (those used
 * by the lock mgr) we can keep track of. PROC_NSEMS_PER_SET is the number
 * of semaphores in each (sys-V) semaphore set allocated. (Be careful not
 * to set it to greater 32. Otherwise, the bitmap will overflow.)
 */
#define  MAX_PROC_SEMS			128
#define  PROC_NSEMS_PER_SET		16

typedef struct procglobal
{
  SHMEM_OFFSET freeProcs;
  IPCKey		currKey;
  int32		freeSemMap[MAX_PROC_SEMS / PROC_NSEMS_PER_SET];
} PROC_HDR;

extern PROC *MyProc;

#define PROC_INCR_SLOCK(lock) if (MyProc) (MyProc->sLocks[(lock)])++
#define PROC_DECR_SLOCK(lock) if (MyProc) (MyProc->sLocks[(lock)])--

/*
 * flags explaining why process woke up
 */
#define NO_ERROR		0
#define ERR_TIMEOUT		1
#define ERR_BUFFER_IO	2
#define	ERR_INT_ABORT	3 //(wb) internal abort

#define MAX_PRIO		50
#define MIN_PRIO		(-1)

extern SPINLOCK ProcStructLock;

/*
 * Function Prototypes
 */
extern void InitProcess(IPCKey key);
extern void ProcReleaseLocks(void);
extern bool ProcRemove(int pid);
extern void ProcRemoveWaiting(PROC *proc);
extern void ProcRemoveAllWaitingLocks();

/* extern bool ProcKill(int exitStatus, int pid); */
/* make static in storage/lmgr/proc.c -- jolly */

extern void ProcQueueInit(PROC_QUEUE *queue);
extern void ProcInsertWaiting(PROC_QUEUE *queue, int token,
		  int prio, LOCK *lock, TransactionId xid, REQ_MODE reqmode, XIDLookupEnt *xidentP);
extern int ProcSleep(SPINLOCK spinlock, LOCK *lock);
extern int ProcSleepOnWriteSet(Oid wokenRelIds[]);
extern int ProcLockWakeup(PROC_QUEUE *queue, LOCKMETHOD lockmethod,
			   LOCK *lock);
extern void ProcAddLock(SHM_QUEUE *elem);
extern void ProcReleaseSpins(PROC *proc);
extern void ProcFreeAllSemaphores(void);


#endif	 /* PROC_H */
