/*
*	(wb) bufferedSock.c provides buffered sockets
*		ATTN: has NEVER been tested with AF_INET sockets!!!!
*
*  Created by: Win Bausch and Bettina Kemme
*
*/

#include "postgres.h"

#include <stdio.h>
#if defined(HAVE_STRING_H)
#include <string.h>
#else
#include <strings.h>
#endif
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>		/* for ttyname() */
// DAR HACK #include <sys/filio.h> 
#include <sys/ioctl.h>
#include <sys/time.h>


#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/file.h>

#if defined(linux)
#ifndef SOMAXCONN
#define SOMAXCONN 5				/* from Linux listen(2) man page */
#endif	 /* SOMAXCONN */
#endif	 /* linux */

#include "miscadmin.h"
#include "libpq/bufferedSock.h"
#include "libpq/pqsignal.h"
//#include "libpq/auth.h"
#include "libpq/libpq.h"
#include "libpq/pqcomm.h"		
#include "storage/ipc.h"
#ifdef MULTIBYTE
#include "mb/pg_wchar.h"
#endif
#include "utils/trace.h"
#include "utils/elog.h"

//#define EBUF_DEBUG

/* --------------------------------------------------------------------- */
/* sockInit: initializes a buffered socket
 */
bufsockptr
sockInit(int bufsize)
{
	bufsockptr bsock = (bufsockptr) malloc(sizeof(buf_sock));
	 
	MemSet((char *) bsock, 0, sizeof(buf_sock));
	/*all variables in buf_sock are now zero'ed*/
	
	bsock->inBuffer = (char *) malloc(bufsize);
	bsock->outBuffer = (char *) malloc(bufsize);
	if (bsock->inBuffer == NULL || bsock->outBuffer == NULL)
	{
		sockDestroy(bsock);
		return NULL;
	}
	bsock->sock = -1;
	bsock->status = NOT_CONNECTED;
	bsock->inBufSize = bufsize;
	bsock->outBufSize = bufsize;
	return bsock;
}

/* --------------------------------------------------------------------- */
/* sockDestroy: destroys a buffered socket
 */
void
sockDestroy(bufsockptr bsock)
{
	if(!bsock)
		return;
	if (bsock->sock >= 0)
#ifdef WIN32
	closesocket(bsock->sock);
#else
	close(bsock->sock);
#endif
	if(bsock->inBuffer)
		free(bsock->inBuffer);
	if(bsock->outBuffer)
		free(bsock->outBuffer);
	free(bsock);
}
/* --------------------------------------------------------------------- */
/* sockGetc:
   get a character from the connection

   All these routines return 0 on success, EOF on error.
   Note that for the Get routines, EOF only means there is not enough
   data in the buffer, not that there is necessarily a hard error.
*/
int
sockGetc(char *result, bufsockptr bsock)
{
	if (bsock->inCursor >= bsock->inEnd)
		return EOF;

	*result = bsock->inBuffer[bsock->inCursor++];

	return 0;
}


/* --------------------------------------------------------------------- */
/* sockPutBytes: local routine to write N bytes to the connection,
   with buffering
 */
static int
sockPutBytes(const char *s, int nbytes, bufsockptr bsock)
{
	int			avail = 0;
	
	//elog(NOTICE, "sockPutBytes: avail=%d, nbytes=%d", avail, nbytes);
	avail = bsock->outBufSize - bsock->outCount;
	while (nbytes > avail)
	{
		memcpy(bsock->outBuffer + bsock->outCount, s, avail);
		bsock->outCount += avail;
		s += avail;
		nbytes -= avail;
		if (sockFlush(bsock))
			return EOF;
		avail = bsock->outBufSize;
	}

	memcpy(bsock->outBuffer + bsock->outCount, s, nbytes);
	bsock->outCount += nbytes;

	return 0;
}

/* --------------------------------------------------------------------- */
/* sockGets:
   get a null-terminated string from the connection,
   and store it in a buffer of size maxlen bytes.
   If the incoming string is >= maxlen bytes, all of it is read,
   but the excess characters are silently discarded.
*/
int
sockGets(char *s, int maxlen, bufsockptr bsock)
{
	/* Copy conn data to locals for faster search loop */
	char	   *inBuffer = bsock->inBuffer;
	int			inCursor = bsock->inCursor;
	int			inEnd = bsock->inEnd;
	int			slen;

	while (inCursor < inEnd && inBuffer[inCursor])
		inCursor++;

	if (inCursor >= inEnd)
	{	
		return EOF;
	}
	
	slen = inCursor - bsock->inCursor;
	if (slen < maxlen)
		strcpy(s, inBuffer + bsock->inCursor);
	else
	{
		strncpy(s, inBuffer + bsock->inCursor, maxlen - 1);
		s[maxlen - 1] = '\0';
	}

	bsock->inCursor = ++inCursor;

	return 0;
}

/* --------------------------------------------------------------------- */
/* sockPuts:
	put a null-terminated string into the buffer

*/
int
sockPuts(const char *s, bufsockptr bsock)
{
	if (sockPutBytes(s, strlen(s) + 1, bsock))
		return EOF;

	return 0;
}

/* --------------------------------------------------------------------- */
/* sockGetnchar:
   get a string of exactly len bytes in buffer s, no null termination
*/
int
sockGetnchar(char *s, int len, bufsockptr bsock)
{
	if (len < 0 || len > bsock->inEnd - bsock->inCursor)
		return EOF;
		
	//elog(NOTICE, "sockGetnchar: %d bytes unprocessed",sockNumLeft(bsock));
	memcpy(s, bsock->inBuffer + bsock->inCursor, len);
	/* no terminating null */

	bsock->inCursor += len;
	//elog(NOTICE, "sockGetnchar: %d bytes unprocessed",sockNumLeft(bsock));
	
	return 0;
}

/* --------------------------------------------------------------------- */
/* sockPutnchar:
   send a string of exactly len bytes, no null termination needed
*/
int
sockPutnchar(const char *s, int len, bufsockptr bsock)
{
	if (sockPutBytes(s, len, bsock))
		return EOF;
		
	return 0;
}

/* --------------------------------------------------------------------- */
/* sockGetInt
   read a 2 or 4 byte integer and convert from network byte order
   to local byte order
*/
int
sockGetInt(int *result, int bytes, bufsockptr bsock)
{
	uint16		tmp2;
	uint32		tmp4;

	//elog(NOTICE, "sockGetInt: %d bytes unprocessed",sockNumLeft(bsock));
	switch (bytes)
	{
		case 2:
			if (bsock->inCursor + 2 > bsock->inEnd)
				return EOF;
			memcpy(&tmp2, bsock->inBuffer + bsock->inCursor, 2);
			bsock->inCursor += 2;
			*result = (int) ntohs(tmp2);
			break;
		case 4:
			if (bsock->inCursor + 4 > bsock->inEnd)
				return EOF;
			memcpy(&tmp4, bsock->inBuffer + bsock->inCursor, 4);
			bsock->inCursor += 4;
			*result = (int) ntohl(tmp4);
			break;
		default:
			return EOF;
	}
	//elog(NOTICE, "sockGetInt: %d bytes unprocessed",sockNumLeft(bsock));

	return 0;
}

/* --------------------------------------------------------------------- */
/* sockPutInt
   send an integer of 2 or 4 bytes, converting from host byte order
   to network byte order.
*/
int
sockPutInt(int value, int bytes, bufsockptr bsock)
{
	uint16		tmp2;
	uint32		tmp4;

	switch (bytes)
	{
		case 2:
			tmp2 = htons((uint16) value);
			if (sockPutBytes((const char *) &tmp2, 2, bsock))
				return EOF;
			break;
		case 4:
			tmp4 = htonl((uint32) value);
			if (sockPutBytes((const char *) &tmp4, 4, bsock))
				return EOF;
			break;
		default:
			sprintf(bsock->errorMessage,
					"pqPutInt: int size %d not supported\n", bytes);
			return EOF;
	}

	return 0;
}

/* --------------------------------------------------------------------- */
/* sockCheck: is select() saying the file is ready to read/write or is
*  there an exception pending?
 */
int
sockCheck(bufsockptr bsock, bool *forRead, bool *forWrite, bool *forExcept)
{
	fd_set		read_mask,
				write_mask,
				error_mask;
	struct timeval timeout;

	if (bsock->sock < 0)
		return (-2);

	FD_ZERO(&read_mask);
	FD_ZERO(&write_mask);
	FD_ZERO(&error_mask);
	if(*forRead)
		FD_SET(bsock->sock, &read_mask);
	if(*forWrite)
		FD_SET(bsock->sock, &write_mask);
	if(*forExcept)
		FD_SET(bsock->sock, &error_mask);
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
	if (select(bsock->sock + 1, &read_mask, &write_mask, &error_mask,
			   &timeout) < 0)
	{
		return STATUS_ERROR;
	}
	*forRead = FALSE;
	*forWrite = FALSE;
	*forExcept = FALSE;
	if(FD_ISSET(bsock->sock, &read_mask))
		*forRead = TRUE;
	if(FD_ISSET(bsock->sock, &write_mask))
		*forWrite = TRUE;
	if(FD_ISSET(bsock->sock, &error_mask))
		*forExcept = TRUE;
	return STATUS_OK;
}

/* --------------------------------------------------------------------- */
/* sockReadData: read more data, if any is available
 * Possible return values:
 *	 1: successfully loaded at least one more byte
 *	 0: no data is presently available, but no error detected
 *	-1: error detected (including EOF = connection closure);
 *	
 * NOTE: callers must not assume that pointers or indexes into bsock->inBuffer
 * remain valid across this call!
 *
 *	useLimit - if TRUE, then maxbytes are read at most, not more
 *			   if FALSE all avalable bytes are read
 */
int
sockReadData(bufsockptr bsock, int maxbytes, bool useLimit)
{
	int			nread = 0,
				npass = 0,
				status;
	bool		forRead = TRUE;

	if(useLimit && (maxbytes == 0))
		return 0;
		
	if (bsock->sock < 0)
	{
		strcpy(bsock->errorMessage, "sockReadData() -- connection not open\n");
		return -1;
	}

	/* Left-justify any data in the buffer to make room */
	if ((bsock->inStart != 0) && (bsock->inStart < bsock->inEnd))
	{
		memmove(bsock->inBuffer, bsock->inBuffer + bsock->inStart,
				bsock->inEnd - bsock->inStart);
		bsock->inEnd -= bsock->inStart;
		bsock->inCursor -= bsock->inStart;
		bsock->inStart = 0;
	}
	if(bsock->inStart >= bsock->inEnd)
		bsock->inStart = bsock->inCursor = bsock->inEnd = 0;

	/*
	 * If the buffer is fairly full, enlarge it. We need to be able to
	 * enlarge the buffer in case a single message exceeds the initial
	 * buffer size.  We enlarge before filling the buffer entirely so as
	 * to avoid asking the kernel for a partial packet. New buffer is
	 *	allocated if there is less than a quarter of the initial space left
	 */
tryAgain:

	 printf("sockReadData:  checking buffer size\n");
	 fflush(stdout);
	
	if (bsock->inBufSize - bsock->inEnd < (bsock->inBufSize/REALLOC_FRACTION) ||
		(useLimit && (maxbytes > bsock->inBufSize - bsock->inEnd)))
	{
		int			newSize = bsock->inBufSize;
		char	   *newBuf;
		
		
		printf("sockReadData: try allocating new buffer, size is %d\n", bsock->inBufSize);
		fflush(stdout);
	
		do
		{
			newSize	*= 2;
		}
		while(useLimit && (maxbytes > newSize - bsock->inEnd));

		newBuf = (char *) realloc(bsock->inBuffer, newSize);
	
	
	
		if (newBuf)
		{
			printf("sockReadData: allocated new buffer, size is %d\n", newSize);
			 fflush(stdout);
			bsock->inBuffer = newBuf;
			bsock->inBufSize = newSize;
		}
	}
	
	if(!useLimit)
		maxbytes = bsock->inBufSize - bsock->inEnd;
	/* OK, try to read some data */
	printf("i am before recv\n");
	fflush(stdout);
	nread = recv(bsock->sock, bsock->inBuffer + bsock->inEnd,
				 maxbytes, 0);
		

	printf("sockReadData: read %d bytes\n", nread);
	fflush(stdout);
	npass++;
	if (nread < 0)
	{
		if (errno == EINTR)
		{
			goto tryAgain;
		}
		
		/* Some systems return EAGAIN/EWOULDBLOCK for no data */
#ifdef EAGAIN
		if (errno == EAGAIN)
		{
			return 0;
		}
#endif
#if defined(EWOULDBLOCK) && (!defined(EAGAIN) || (EWOULDBLOCK != EAGAIN))
		if (errno == EWOULDBLOCK)
		{	
			return 0;
		}
#endif
		/* We might get ECONNRESET here if using TCP and backend died */
#ifdef ECONNRESET
		if (errno == ECONNRESET)
		{
			printf("Error is %d Backend died\n", errno);
			fflush(stdout);
			goto definitelyFailed;
		}
#endif
		sprintf(bsock->errorMessage,
				"pqReadData() --  read() failed: errno=%d\n%s\n",
				errno, strerror(errno));
		return -1;
	}
	if (nread > 0)
	{
		bsock->inEnd += nread;
		elog(NOTICE, "sockReadData: %d bytes unprocessed", sockNumLeft(bsock));
		return nread;
	}
	/*
	*	If we read nothing, the socket might have been closed
	*	or there might be nothing to read. Check this...
	*/
checkAgain:
	status = sockCheck(bsock, &forRead, FALSE, FALSE);
	if (status == STATUS_ERROR){
		if(errno == EINTR)
			goto checkAgain;
		elog(NOTICE, "sockReadData: status is %d and errno is %d", status, errno);
		goto definitelyFailed;
	}else if(status == STATUS_OK){
		if(!forRead)
			return 0;
//		else if(npass > 1)
//			return (-1);
		goto tryAgain;
	}

definitelyFailed:
	sprintf(bsock->errorMessage,
			"pqReadData() -- SRD: backend closed the channel unexpectedly.\n"
			"\tThis probably means the backend terminated abnormally"
			" before or while processing the request.\n");
	bsock->status = CONNECTION_ERROR;		/* No more connection to backend */
#ifdef WIN32
	closesocket(bsock->sock);
#else
	close(bsock->sock);
#endif
	bsock->sock = -1;

	return -1;
}


/* --------------------------------------------------------------------- */
/* sockFlush: send any data waiting in the output buffer
*
*/
int
sockFlush(bufsockptr bsock)
{
	char	   *ptr = bsock->outBuffer;
	int			len = bsock->outCount;

	if (bsock->sock < 0)
	{
		strcpy(bsock->errorMessage, "pqFlush() -- connection not open\n");
		return EOF;
	}

	while (len > 0)
	{
		/* Prevent being SIGPIPEd if backend has closed the connection. */
#ifndef WIN32
		pqsigfunc	oldsighandler = pqsignal(SIGPIPE, SIG_IGN);
#endif

		int			sent = send(bsock->sock, ptr, len, 0);
#ifndef WIN32
		pqsignal(SIGPIPE, oldsighandler);
#endif

		if (sent < 0)
		{
			/*
			 * Anything except EAGAIN or EWOULDBLOCK is trouble.
			 * If it's EPIPE or ECONNRESET, assume we've lost the
			 *  connection permanently.
			 */
			switch (errno)
			{
#ifdef EAGAIN
				case EAGAIN:
					break;
#endif
#if defined(EWOULDBLOCK) && (!defined(EAGAIN) || (EWOULDBLOCK != EAGAIN))
				case EWOULDBLOCK:
					break;
#endif
				case EPIPE:
#ifdef ECONNRESET
				case ECONNRESET:
#endif
					sprintf(bsock->errorMessage,
							"pqFlush() -- backend closed the channel unexpectedly.\n"
							"\tThis probably means the backend terminated abnormally"
							" before or while processing the request.\n");
					bsock->status = CONNECTION_ERROR; /* No more connection */
#ifdef WIN32
					closesocket(bsock->sock);
#else
					close(bsock->sock);
#endif
					bsock->sock = -1;
					return EOF;
				default:
					sprintf(bsock->errorMessage,
							"pqFlush() --  couldn't send data: errno=%d\n%s\n",
							errno, strerror(errno));
					/* We don't assume it's a fatal error... */
					return EOF;
			}
		}
		else
		{
			ptr += sent;
			len -= sent;
		}
		if (len > 0)
		{
			/* We didn't send it all, wait till we can send more */
			if (sockWait(FALSE, TRUE, bsock))
				return EOF;
		}
	}

	bsock->outCount = 0;
	return 0;
}

/* --------------------------------------------------------------------- */
/* sockWait: wait until we can read or write the connection socket
 */
int
sockWait(bool forRead, bool forWrite, bufsockptr bsock)
{
	fd_set		input_mask;
	fd_set		output_mask;

	if (bsock->sock < 0)
	{
		strcpy(bsock->errorMessage, "sockWait() -- connection not open\n");
		return EOF;
	}

	/* loop in case select returns EINTR */
	for (;;)
	{
		FD_ZERO(&input_mask);
		FD_ZERO(&output_mask);
		if (forRead)
			FD_SET(bsock->sock, &input_mask);
		if (forWrite)
			FD_SET(bsock->sock, &output_mask);
		//elog(NOTICE, "sockWait: waiting...");
		if (select(bsock->sock + 1, &input_mask, &output_mask, (fd_set *) NULL,
				   (struct timeval *) NULL) < 0)
		{
			if (errno == EINTR)
				continue;
			sprintf(bsock->errorMessage,
					"sockWait() -- select() failed: errno=%d\n%s\n",
					errno, strerror(errno));
			return EOF;
		}
		//elog(NOTICE, "sockWait: ...done");
		/* On nonerror return, assume we're done */
		break;
	}

	return 0;
}

/* --------------------------------------------------------------------- */
/* sockServerPort: sets up server port
 */
int
sockServerPort(char *hostName, short portName, int *fdP, struct sockaddr *serverAddr)
{
	SockAddr	saddr;
	int			fd,
				err,
				family;
	size_t		len;
	int			one = 1;
#ifdef HAVE_FCNTL_SETLK
	int			lock_fd;
#endif

	elog(DEBUG, "bufferedSock: serverport: hostName=%s & portName=%d", \
						hostName, portName);
	family = ((hostName != NULL) ? AF_INET : AF_UNIX);

	if ((fd = socket(family, SOCK_STREAM, 0)) < 0)
	{
		elog(DEBUG, "bufferedSock: serverport: socket failed");
		return STATUS_ERROR;
	}
	if ((setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *) &one,
					sizeof(one))) == -1)
	{
		elog(DEBUG, "bufferedSock: serverport: setsockopt failed");
		return STATUS_ERROR;
	}
	MemSet((char *) &saddr, 0, sizeof(saddr));
	saddr.sa.sa_family = family;
	if (family == AF_UNIX)
	{
		len = UNIXSOCK_PATH(saddr.un, portName);
		memmove(serverAddr, &saddr, sizeof(saddr));

		/*
		 * If the socket exists but nobody has an advisory lock on it we
		 * can safely delete the file.
		 */
#ifdef HAVE_FCNTL_SETLK
		if ((lock_fd = open(sock_path, O_WRONLY | O_NONBLOCK, 0666)) >= 0)
		{
			struct flock	lck;
			
			lck.l_whence = SEEK_SET; lck.l_start = lck.l_len = 0;
			lck.l_type = F_WRLCK;
			if (fcntl(lock_fd, F_SETLK, &lck) == 0)
			{
				TPRINTF(TRACE_VERBOSE, "flock on %s, deleting", sock_path);
				unlink(sock_path);
			}
			else
				TPRINTF(TRACE_VERBOSE, "flock failed for %s", sock_path);
			close(lock_fd);
		}
#endif /* HAVE_FCNTL_SETLK */
	}
	else
	{
		saddr.in.sin_addr.s_addr = htonl(INADDR_ANY);
		saddr.in.sin_port = htons(portName);
		len = sizeof(struct sockaddr_in);
	}
	err = bind(fd, &saddr.sa, len);
	if (err < 0)
	{
		elog(DEBUG, "bufferedSock: ServerPort: bind failed");
		return STATUS_ERROR;
	}

	if (family == AF_UNIX)
	{
		//on_proc_exit(sockUnlink, null);

		/*
		 * Open the socket file and get an advisory lock on it. The
		 * lock_fd is left open to keep the lock.
		 */
#ifdef HAVE_FCNTL_SETLK
		if ((lock_fd = open(sock_path, O_WRONLY | O_NONBLOCK, 0666)) >= 0)
		{
			struct flock	lck;
			
			lck.l_whence = SEEK_SET; lck.l_start = lck.l_len = 0;
			lck.l_type = F_WRLCK;
			if (fcntl(lock_fd, F_SETLK, &lck) != 0)
				TPRINTF(TRACE_VERBOSE, "flock error for %s", sock_path);
		}
#endif /* HAVE_FCNTL_SETLK */
	}

	listen(fd, SOMAXCONN);

	/*
	 * MS: I took this code from Dillon's version.  It makes the listening
	 * port non-blocking.  That is not necessary (and may tickle kernel
	 * bugs).
	 *
	 * fcntl(fd, F_SETFD, 1); fcntl(fd, F_SETFL, FNDELAY);
	 */

	*fdP = fd;
	if (family == AF_UNIX)
		chmod(saddr.un.sun_path, 0777);
	return STATUS_OK;
}

/* --------------------------------------------------------------------- */
/* sockServerAccept: accepts a pending client connection
 */
int
sockServerAccept(int server_fd, bufsockptr bsock)
{

	int			len,
				flag = 1;
	SOCKET_SIZE_TYPE	addrlen;
	int			family = 0; 

	/* accept connection (and fill in the client (remote) address) */
	if(bsock == NULL)
		return STATUS_ERROR;
	addrlen = sizeof(bsock->raddr.sa);
	if ((bsock->sock = accept(server_fd,
							 &bsock->raddr.sa,
							 &addrlen)) < 0)
	{
		elog(NOTICE, "bufferedSock: serverAccept: accept: %m");
		bsock->status = CONNECTION_ERROR;
		return STATUS_ERROR;
	}
	family = bsock->raddr.sa.sa_family;
	len = family == AF_INET ?
		sizeof(struct sockaddr_in) : sizeof(struct sockaddr_un);
	/* fill in the server (local) address */
	addrlen = len;
	if (getsockname(bsock->sock, (struct sockaddr *) & bsock->laddr,
					&addrlen) < 0)
	{
		elog(NOTICE, "bufferedSock getsockname: getsockname: %m");
		bsock->status = CONNECTION_ERROR;
		return STATUS_ERROR;
	}
	if (family == AF_INET)
	{
		struct protoent *pe;
		int			on = 1;

		pe = getprotobyname("TCP");
		if (pe == NULL)
		{
			elog(NOTICE, "bufferedSock: getprotobyname failed");
			bsock->status = CONNECTION_ERROR;
			return STATUS_ERROR;
		}
		if (setsockopt(bsock->sock, pe->p_proto, TCP_NODELAY,
					   &on, sizeof(on)) < 0)
		{
			elog(NOTICE, "bufferedSock: setsockopt failed");
			bsock->status = CONNECTION_ERROR;
			return STATUS_ERROR;
		}
	}
	/* reset to non-blocking */
	//fcntl(bsock->sock, F_SETFL, 1);
	if(ioctl(bsock->sock, FIONBIO, &flag)<0)
		return STATUS_ERROR;
	
	bsock->status = NOT_CONNECTED;
	return STATUS_OK;
}

/* --------------------------------------------------------------------- */
/* sockClientConnect: connects to a server 
 */
int
sockClientConnect(char *hostName, short portName, bufsockptr bsock)
{
	SOCKET_SIZE_TYPE	len;
	int			err;
	struct hostent *hp;
	extern int	errno;

	/* set up the server (remote) address */
	MemSet((char *) &bsock->raddr, 0, sizeof(bsock->raddr));
	if (hostName)
	{
		if (!(hp = gethostbyname(hostName)) || hp->h_addrtype != AF_INET)
		{
			elog(NOTICE, "bufferedSock: ClientConnect:gethostbyname failed");
			bsock->status = CONNECTION_ERROR;
			return STATUS_ERROR;
		}
		memmove((char *) &(bsock->raddr.in.sin_addr),
				(char *) hp->h_addr,
				hp->h_length);
		bsock->raddr.in.sin_family = AF_INET;
		bsock->raddr.in.sin_port = htons(portName);
		len = sizeof(struct sockaddr_in);
	}
	else
	{
		bsock->raddr.un.sun_family = AF_UNIX;
		len = UNIXSOCK_PATH(bsock->raddr.un, portName);
	}
	/* connect to the server */
	if ((bsock->sock = socket(bsock->raddr.sa.sa_family, SOCK_STREAM, 0)) < 0)
	{
		elog(DEBUG, "bufferedSock: ClientConnect: socket call failed");
		bsock->status = CONNECTION_ERROR;
		return STATUS_ERROR;
	}
	/*elog(NOTICE,"got socket");*/
	err = connect(bsock->sock, &bsock->raddr.sa, len);
	if (err < 0)
	{
		elog(DEBUG, "bufferedSock: ClientConnect: connect failed %d:%d", err,errno);
		bsock->status = CONNECTION_ERROR;
		return STATUS_ERROR;
	}

	/* fill in the client address */
	if (getsockname(bsock->sock, &bsock->laddr.sa, &len) < 0)
	{
		elog(DEBUG, "bufferedSock: clientConnect: getsockname failed");
		bsock->status = CONNECTION_ERROR;
		return STATUS_ERROR;
	}
	
	bsock->status = CONNECTION_OK;
	return STATUS_OK;
}

/* --------------------------------------------------------------------- */
/* sockSClose: closes buffered socket
 */
void
sockClose(bufsockptr bsock)
{
	close(bsock->sock);
	bsock->status = NOT_CONNECTED;
}

/* --------------------------------------------------------------------- */
/* sockUnlink: deletes file associated to AF_UNIX socket type
 */
void
sockUnlink(char path[])
{
	Assert(path[0]);
	unlink(path);
}

/* --------------------------------------------------------------------- */
/* sockResetBuffers: resets the input and output buffer indexes of 
		a socket
 */
void
sockResetBuffers(bool in, bool out, bufsockptr bsock)
{
	if(!bsock)
		return;
	if(in){
		bsock->inStart = 0;
		bsock->inEnd = 0;
		bsock->inCursor = 0;
	}
	if(out)
		bsock->outCount = 0;
}

/* --------------------------------------------------------------------- */
/* sockIsConnected: tells if socket has active connection
 */
int
sockIsConnected(bufsockptr bsock)
{
	if(bsock->status == NOT_CONNECTED)
		return (-1);
	else if(bsock->status == CONNECTION_ERROR)
		return 0;
	else
		return 1;
}

/* --------------------------------------------------------------------- */
/* sockNumLeft: gives the number of unread bytes in the 'incoming' buffer
 */
int
sockNumLeft(bufsockptr bsock)
{
	int nleft = 0;
	
	if(!bsock)
		return -1;
	
	nleft = bsock->inEnd - bsock->inCursor;
	if(nleft <= 0)
	{
		return 0;
	}
	return nleft;
}

/* --------------------------------------------------------------------- */
/* sockPeekHeader: reads the first n bytes in the 'incoming' buffer
			without marking them as read.
 */
int
sockPeekHeader(int nbytes, bufsockptr bsock, char *header)
{
	if (nbytes < 0 || nbytes > bsock->inEnd - bsock->inCursor)
		return EOF;

	memcpy(header, bsock->inBuffer + bsock->inCursor, nbytes);
	return 0;
}

/* --------------------------------------------------------------------- */
/* sockTransfer: transfers data from one buffered socket to another without
		unnecessary copying 
 */
int
sockTransfer(bufsockptr dest, bufsockptr src, int len)
{
	if(sockPutnchar(src->inBuffer + src->inCursor, len, dest) == 0)
		src->inCursor += len;
		return 0;
	return 1;
}

/* --------------------------------------------------------------------- */
/* sockIgnoreData: discards the first n bytes in the 'incoming' buffer
 */
int
sockIgnoreData(int nbytes, bufsockptr bsock)
{
	if(bsock->inCursor + nbytes <= bsock->inEnd)
		bsock->inCursor += nbytes;
		return 0;
	return 1;
}

/* --------------------------------------------------------------------- */
/* sockConsume: marks the bytes in the 'incoming' buffer that have been 
	read as consumed
 */
void
sockConsume(bufsockptr bsock)
{
	if(bsock->inStart < bsock->inCursor)
		bsock->inStart = bsock->inCursor;
}

/* --------------------------------------------------------------------- */
/* sockResetCursor: resets the reading cursor of the 'incoming' buffer
 */
void
sockResetCursor(bufsockptr bsock)
{
	if(bsock->inStart < bsock->inCursor)
		bsock->inCursor = bsock->inStart;
}

/* --------------------------------------------------------------------- */
/* sockGetSlen: gives the length of the null-terminated string beginning 
	at the read cursor position
 */
int
sockGetSlen(bufsockptr bsock)
{
	return strlen(bsock->inBuffer + bsock->inCursor);
}

/*
	extern buffer routines
*/

/* --------------------------------------------------------------------- */
/* _checkEbufLen: checks if an external buffer has enough room left to hold
	n bytes, if not, the buffer is reallocated
 */
static void
_checkEbufLen(int bytes, ebufptr extbuf)
{
	char *newbuf = NULL;
	int	 newsize = 0;

	if(!extbuf->buf)
	{
		if(extbuf->buflen != 0)
		{
			extbuf->buf = (char *) malloc(extbuf->buflen * sizeof(char));
		}
		else
		{
			extbuf->buflen = EBUF_DEFAULT_SIZE;
			extbuf->buf = (char *) malloc(extbuf->buflen * sizeof(char));
		}
#ifdef EBUF_DEBUG
		elog(DEBUG, "_checkEbufLen: Allocated ebuf (len = %d, buf_addr = %x)"
								, extbuf->buflen, extbuf->buf);
#endif
	}
	else
	{
		int avail = extbuf->buflen - extbuf->currindex;
		if(avail < bytes)
		{
			newsize = extbuf->buflen;
			
			do{
				newsize *= 2;
			}
			while(bytes < newsize - extbuf->currindex);
			
			newbuf = realloc(extbuf->buf, extbuf->buflen);
			
			if(newbuf)
			{
				extbuf->buf = newbuf;
				extbuf->buflen = newsize;
#ifdef EBUF_DEBUG
				elog(DEBUG, "_checkEbufLen: reallocated ebuf (len = %d, buf_addr = %x)"
											, extbuf->buflen, extbuf->buf);
#endif
			}
			else
			{
				elog(DEBUG, "EBuffer realloc failed");
			}
		}
	}
}

/* --------------------------------------------------------------------- */
/* sockInitE: initializes an external buffer
 */
void
sockInitE(ebufptr extbuf, int buflen)
{
	if(!extbuf)
		return;
	
	extbuf->buf = (char *) malloc(buflen * sizeof(char));
	if(!extbuf->buf)
	{
		elog(DEBUG, "Ebuf allocation failed");
		return;
	}
	extbuf->currindex = 0;
	extbuf->buflen = buflen;
#ifdef EBUF_DEBUG
	elog(DEBUG, "Initialized ebuf (len=%d, addr=%x)", extbuf->buflen, extbuf);
#endif
}

/* --------------------------------------------------------------------- */
/* sockDestroyE: discards an external buffer
 */
void
sockDestroyE(ebufptr extbuf)
{
	if(!extbuf)
		return;
		
	free(extbuf->buf);
	extbuf->buf = NULL;
	extbuf->buflen = 0;
	extbuf->currindex = 0;
#ifdef EBUF_DEBUG
	elog(DEBUG, "destroyed ebuf (addr=%x)", extbuf);
#endif
}

/* --------------------------------------------------------------------- */
/* sockGetCountE: gives the number of data bytes in the buffer
 */
int
sockGetCountE(ebufptr extbuf)
{
	return extbuf->currindex;
}

/* --------------------------------------------------------------------- */
/* sockFlushE: flushes an external buffer into a buffered socket
 */
void
sockFlushE(ebufptr extbuf, bufsockptr bsock, bool flushBeforePipe)
{
	char	*outBufSave;
	int		outCountSave;
	
#ifdef EBUF_DEBUG
	elog(DEBUG, "Flushing ebuf (sock=%d, ebuf_addr=%x, outCount=%d)"
							, bsock->sock, extbuf, extbuf->currindex);
#endif
	if(flushBeforePipe)
		sockFlush(bsock);
		
	outBufSave = bsock->outBuffer;
	outCountSave = bsock->outCount;
	
	bsock->outBuffer = extbuf->buf;
	bsock->outCount = extbuf->currindex;
	sockFlush(bsock);
	
	bsock->outBuffer = outBufSave;
	bsock->outCount = outCountSave;
}

/* --------------------------------------------------------------------- */
/* sockPutIntE: writes an int into an external buffer
 */
void
sockPutIntE(int value, int bytes, ebufptr extbuf)
{
	uint16		tmp2;
	uint32		tmp4;
	
	
	if(    extbuf->buflen < 0
		|| extbuf->currindex < 0 
		|| extbuf->buflen < extbuf->currindex
		|| (bytes != 4 && bytes != 2)
	  )
	{
		elog(DEBUG, "sockPutIntE: bad ebuf");	
		return;
	}
	
	_checkEbufLen(bytes, extbuf);

	switch (bytes)
	{
		case 2:
			tmp2 = htons((uint16) value);
			memcpy(extbuf->buf + extbuf->currindex, (char *) &tmp2, bytes);
			break;
		case 4:
			tmp4 = htonl((uint32) value);
			memcpy(extbuf->buf + extbuf->currindex, (char *) &tmp4, bytes);
			break;
	}
	extbuf->currindex += bytes;
#ifdef EBUF_DEBUG
	elog(DEBUG, "sockPutIntE (ebuf_addr=%x): put %d bytes (index=%d, buflen=%d)"
						, extbuf, bytes, extbuf->currindex, extbuf->buflen);
#endif
	return;
}

/* --------------------------------------------------------------------- */
/* sockPutsE: writes a null-terminated string into an external buffer
 */
void
sockPutsE(const char *s, ebufptr extbuf)
{
	int len = strlen(s) + 1;
	
	if(extbuf->buflen <= 0 || extbuf->currindex < 0 
		|| extbuf->buflen < extbuf->currindex)
	{
		elog(DEBUG, "sockPutsE: bad ebuf");	
		return;
	}
	
	_checkEbufLen(len, extbuf);
	
	strncpy(extbuf->buf + extbuf->currindex, s, len);
	extbuf->currindex += len;
#ifdef EBUF_DEBUG
	elog(DEBUG, "sockPutsE (ebuf_addr=%x) : put %d bytes (index=%d, buflen=%d)"
						, extbuf, len, extbuf->currindex, extbuf->buflen);
#endif
}

/* --------------------------------------------------------------------- */
/* sockPutncharE: writes n characters into an external buffer
 */
void
sockPutncharE(const char *s, int len, ebufptr extbuf)
{
	if(extbuf->buflen <= 0 || extbuf->currindex < 0 
		|| extbuf->buflen < extbuf->currindex)
	{
		elog(DEBUG, "sockPutncharE: bad ebuf");	
		return;
	}
	
	_checkEbufLen(len, extbuf);
	
	memcpy(extbuf->buf + extbuf->currindex, s, len);
	extbuf->currindex += len;
#ifdef EBUF_DEBUG
	elog(DEBUG, "sockPutncharE (ebuf_addr=%x) : put %d bytes (index=%d, buflen=%d)"
						, extbuf, len, extbuf->currindex, extbuf->buflen);
#endif
}

