package pgclibrary.dataaccess;

import java.sql.*;
import java.util.*;
import java.io.*;
import java.text.*;

/**
 * This class initializes the database connection.
 * It's realizes an abstraction of the connection classes.
 *
 * <H4> This object is not be able to manage batch processes.
 *      If you need to make process as transactions you must clone
 * this object creating one MainConnection from anoter MainConnection.
 * The one of them can be non autoflushed.
 * </H4>
 *
 * @author Nestor Marsollier
 * @version 1.1
 */
public class MainConnection extends Object implements Serializable {
    public static final String defaultDriver = "org.postgresql.Driver";
    public static final String defaultURI = "jdbc:postgresql://localhost:5432/prueba1";
    
    private String driver = null;                  // Driver utilizado en la conexion
    private String URI = null;                      // Origen de datos utilizado
    private String userName = null;            // Nombre del usuario conectado
    private String password = null;             // Contrase\u00f1a utilizada en la conexion
    private Connection connection = null;   // java.sql.Connection utilizado
    private Statement statement = null;
    
    private DatabaseQuerys databaseQG = null;  // Database query generator
    
    private Vector error_text = new Vector();
    private long error_code = 0;
    
    private double database_version = -1;
    
    /*
     * Default Constructor
     */
    public MainConnection(){}
    /**
     * Complete constructor.
     * This initis the database connection from the parameters specified.
     *
     * @param _driver  - Database Driver
     * @param _URI  - URL location to connect
     * @param _userName  - Database user name
     * @param _password  - User password
     */
    public MainConnection(String _driver, String _URI, String _userName, String _password){
        error_text = new Vector();
        error_code = 0;
        
        if((_driver != null) && (_URI != null) && (_userName != null)
        && (_password != null)){
            driver = new String(_driver);
            URI = new String(_URI);
            userName = new String(_userName);
            password = new String(_password);
            
            Properties props = new Properties();
            props.put("user",userName);
            props.put("password",password);
            
            connect(props);
        }
    }
    
    /* Makes the connection and manage the posible errors.
     **/
    private boolean connect(Properties props){
        boolean retorno = false;
        
        Connection temp = null;
        
        if((driver != null) && (URI != null) && (userName != null) && (password != null)){
            try{
                Class.forName(driver);
                temp = DriverManager.getConnection(URI, props);
                temp.setAutoCommit(true);
                SQLWarning sqlw = temp.getWarnings();
                if(sqlw != null && sqlw.getErrorCode() == 28002){
                    error_text.add("Incorrect password.");
                    error_code = sqlw.getErrorCode();
                }
            }catch(Exception e){
                error_text.add(e.getMessage());
                temp = null;
            }
        }
        if(temp != null){
            connection = temp;
            try{
                statement = connection.createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_UPDATABLE);
                retorno = true;
            } catch (Exception e){
                error_text.add(e.getMessage());
                connection = null;
                retorno = false;
            }
        }
        return retorno;
    }
    
    /**
     * Inits the predeterminated driver and url connection
     * the a defined user
     *
     * @param _userName  - Database user name
     * @param _password  - User password
     */
    public MainConnection(String _userName, String _password){
        this(defaultDriver, defaultURI, _userName, _password);
    }
    
    /**
     * Clones the database connection
     * @param _mainConnection MainConnection to clone
     */
    public MainConnection(MainConnection _mainConnection){
        this(_mainConnection.getDriver(),
        _mainConnection.getURI(),
        _mainConnection.getUserName(),
        _mainConnection.getPassword());
    }
    
    public Vector getError(){
        return error_text;
    }
    public long getErrorCode(){
        return error_code;
    }
    
    /**
     * Retrieve the user name of the connection
     *
     * @return User named used
     */
    public String getUserName(){
        return userName;
    }
    
    /**
     * Retrieve the password used..
     *
     * @return User password
     */
    private String getPassword(){
        return password;
    }
    
    /**
     * Retrieves the Connection object used
     *
     * @return Objeto java.sql.Connection.
     * @throws SQLException If not connected.
     */
    public Connection getConnection() throws SQLException{
        if (!isConnected()){
            throw new SQLException("The connection has not be initialized");
        }
        return connection;
    }
    
    /**
     * Retrieves the Driver name used.
     *
     * @return String Driver name
     */
    public String getDriver(){
        return driver;
    }
    
    /**
     * Retrieves the URL of the connection
     *
     * @return String Data location.
     */
    public String getURI(){
        return URI;
    }
    
    /**
     * Retrieves if the connection is ok
     *
     * @return true OKey
     * @return false Not connected
     */
    public boolean isConnected(){
        return connection != null;
    }
    
    /**
     * Retrieves a new statement from the connection
     * The statement is readOnly.
     *
     * @return Statement
     * @throws SQLException If not connected
     */
    public Statement getStatementReadOnly() throws SQLException{
        Statement st = null;
        st = getConnection().createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE,
        ResultSet.CONCUR_READ_ONLY);
        return st;
    }
    
    /**
     * Makes a new statement and return this
     *
     * @return Statement
     * @throws SQLException If not connected
     */
    public Statement getStatement() throws SQLException{
        Statement st = null;
        st = getConnection().createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY);
        return st;
    }
    
    /**
     * Makes a new statement from database and return it
     *
     * @return Statement
     * @param tipo ResultSet type
     * @param concur Concurrency type
     * @throws SQLException If not connected
     */
    public Statement getStatement(int tipo, int concur) throws SQLException{
        Statement st = null;
        st = getConnection().createStatement(tipo, concur);
        return st;
    }
    
    /**
     * Execute an SQL statement and return the rows number modifieds
     * This is an abstraction for Statement.executeUpdate
     *
     * @return int count of modified rows
     * @param sql SQL command
     * @throws SQLException For an error
     */
    public int executeUpdate(String sql) throws SQLException{
        if(statement != null){
            return statement.executeUpdate(sql);
        } else {
            return 0;
        }
    }
    
    /**
     * Executes a query and return the resultset
     *
     * This is a Statement.executeQuery() abstraction
     *
     * @return ResultSet
     * @param sql SQL query to run
     * @throws SQLException Generated by statement
     */
    public ResultSet executeQuery(String sql) throws SQLException {
        if(statement != null){
            return statement.executeQuery(sql);
        } else {
            return null;
        }
    }
    
    /**
     * Its makes a new PreparedStatemetn
     *
     * Its an Connection.prepareStatement() abstraction
     *
     * @return PreparedStatement.
     * @param sql Query to prepare
     * @throws SQLException Connection exceptions
     */
    public PreparedStatement prepareStatement(String sql) throws SQLException{
        if (!isConnected()){
            throw new SQLException("Connection nor initialized");
        }
        return getConnection().prepareStatement(sql, ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY);
    }
    
    /**
     * Closes the databse connection
     * It must be called before MainConnection release
     */
    public void close(){
        if (isConnected()){
            try{
                statement.close();
                connection.close();
            } catch (SQLException e){};
        }
    }
    
    protected void finalize() throws Throwable{
        close();
    }
    
    public static boolean isQuery(String _query){
        String tmp = _query.toUpperCase();
        return (tmp.indexOf("SELECT") >= 0)
        && (tmp.indexOf("UPDATE") < 0)
        && (tmp.indexOf("DELETE") < 0)
        && (tmp.indexOf("GRANT") < 0)
        && (tmp.indexOf("REVOKE") < 0)
        && (tmp.indexOf("INSERT") < 0);
    }
    
    public String getDatabaseName(){
        String retorno = new String();
        if(isConnected()){
            StringTokenizer stok = new StringTokenizer(getURI(), "/");
            
            while(stok.hasMoreTokens()){
                retorno = stok.nextToken();
            }
        }
        return retorno;
    }
    public String getProductName(){
        String retorno = new String();
        try{
            if(isConnected()){
                retorno = getConnection().getMetaData().getDatabaseProductName() + " " +
                getConnection().getMetaData().getDatabaseProductVersion();
            }
        } catch (SQLException e){}
        return retorno;
    }
    
    public String getDriverVersion(){
        String retorno = new String();
        try{
            if(isConnected()){
                retorno = getConnection().getMetaData().getDriverVersion();
            }
        } catch (SQLException e){}
        return retorno;
    }
    
    public double getDatabaseVersion(){
        if (database_version<0){
            try{
                if(isConnected()){
                    StringTokenizer stk = new StringTokenizer(getConnection().getMetaData().getDatabaseProductVersion(), ".");
                    if(stk.hasMoreTokens()){
                        long acum = 1;
                        database_version = (double) Integer.parseInt(stk.nextToken());
                        while (stk.hasMoreTokens()){
                            String next = stk.nextToken();
                            acum *=  java.lang.Math.pow(10, next.length());
                            if(next.length() > 0){
                                int tmp = Integer.parseInt(next);
                                database_version += ((double)  tmp) / acum;
                            }
                        }
                    }
                }
            } catch (SQLException e){}
        }
        return database_version;
    }
    
    public String getURL(){
        String retorno = new String();
        try{
            if(isConnected()){
                retorno = getConnection().getMetaData().getURL();
            }
        } catch (SQLException e){}
        return retorno;
    }
    
    public DatabaseQuerys getQueryGenerator(){
        if(databaseQG == null){
            databaseQG = DatabaseQuerys.getDefaultQueryGenerator(this);
        }
        return databaseQG;
    }
    
    
    static {
        Locale.setDefault(Locale.US);
    }
}


