/**
 * Class copyright 2003 by the Ravensfield Digital Resource Group, Ltd, Granville, OH.
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without a written agreement
 * is hereby granted, provided that the above copyright notice and this
 * paragraph and the following two paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE RAVENSFIELD DIGITAL RESOURCE GROUP, LTD BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING
 * LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 * DOCUMENTATION, EVEN IF THE RAVENSFIELD DIGITAL RESOURCE GROUP, LTD HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * THE RAVENSFIELD DIGITAL RESOURCE GROUP, LTD SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE RAVENSFIELD DIGITAL RESOURCE GROUP, LTD HAS NO OBLIGATIONS TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 * (Quick readers will recognize that as the stock BSD license)
 */

package org.postgresql.ers;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;

import java.io.LineNumberReader;
import java.sql.Connection;

import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.StringTokenizer;

/**
 *  Class to add a table to a replication scheme. This is a rewrite of ers_addtable.
 *
 * @author     Andrew Rawnsley
 * @created    August 29, 2003
 */
public abstract class ERS {

	/**
	 *  Verbose flag
	 */
	protected static boolean quiet = false;

	/**
	 *  Create a connection
	 *
	 * @param  connectionString  Connection string
	 * @param  user              User ID
	 * @param  passwd            Password
	 * @return                   New connection, or null on exception
	 */
	public static Connection getConnection(String connectionString, String user, String passwd) {

		try {
			Class.forName("org.postgresql.Driver");
			return DriverManager.getConnection(connectionString, user, passwd);
		} catch (ClassNotFoundException ex) {
			System.err.println(connectionString);
			ex.printStackTrace();
			return null;
		} catch (SQLException sx) {
			System.err.println(connectionString);
			sx.printStackTrace();
			return null;
		}

	}

	/**
	 *  Splits a property into a name/value pair
	 *
	 * @param  s  A property string (x=y)
	 * @return    The property value
	 */
	public static String[] getProperty(String s) {
		int pos;
		String[] prop = new String[2];

		if ((pos = s.indexOf("=")) >= 0) {
			prop[0] = s.substring(0, pos);
			prop[1] = s.substring(pos + 1);
			return prop;
		}
		return null;
	}

	/**
	 *  Sets the quiet attribute
	 *
	 * @param  b  The new quiet value
	 */
	public static void setQuiet(boolean b) {
		quiet = b;
	}

	/**
	 * Take a delimited list as a string and tokenize it into an array
	 *
	 * @param  s      String to split
	 * @param  delim  Delimiter
	 * @return        Array of tokens
	 */
	public static String[] split(String s, String delim) {
		if (s == null) {
			return new String[0];
		}
		StringTokenizer tok = new StringTokenizer(s, delim);
		String[] list = new String[tok.countTokens()];
		for (int i = 0; i < list.length; i++) {
			list[i] = tok.nextToken();
		}

		return list;
	}

	/**
	 *  Copy a table from primary to replicant
	 *
	 * @param  table       Table to copy
	 * @param  primary     Connection to the primary
	 * @param  replicant   Connection to the replicant
	 * @param  uniqColumn  Name of the unique column
	 */
	protected void copyTable(String table, String uniqColumn, Connection primary, Connection replicant) {

		Statement stmnt = null;
		Statement seqStmnt = null;
		ResultSet rs;
		DatabaseMetaData metaData = null;
		String columnName;
		String sequenceName;
		String columnType;
		String defaultClause;
		int columnSize;
		int columnDec;
		int nullable;
		int pos1;
		int pos2;
		int pos;
		String indexName;
		String tableName = null;

		tableName = table;

		try {

			// Prepare the statement to add sequences to the replicant
			seqStmnt = replicant.createStatement();

			// Get the column definitions for the table in question
			metaData = primary.getMetaData();
			rs = metaData.getColumns(null, null, tableName, "%");

			// Start the CREATE TABLE statement
			StringBuffer sb = new StringBuffer("CREATE TABLE ").append(table).append(" (");

			boolean first = true;

			if (!quiet) {
				System.out.println("Creating table on replicant...");
			}
			// Step through each column
			while (rs.next()) {

				if (!first) {
					sb.append(", ");
				} else {
					first = false;
				}

				columnName = rs.getString(4);
				columnType = rs.getString(6);
				columnSize = rs.getInt(7);
				columnDec = rs.getInt(9);
				nullable = rs.getInt(11);
				defaultClause = rs.getString(13);

				sb.append(columnName).append(' ').append(columnType);

				// Deal with size/precision
				if (columnType.equals("varchar") || columnType.equals("bpchar")) {
					sb.append('(').append(columnSize).append(") ");
				} else if (columnType.equals("numeric")) {
					sb.append('(').append(columnSize).append(',').append(columnDec).append(") ");
				} else {
					sb.append(' ');
				}

				// nullable check
				if (nullable == 0) {
					sb.append(" NOT NULL ");
				}

				// Check the default clause to see if its a sequence. If so, isolate the sequence name and create the sequence
				if (defaultClause != null) {
					if (defaultClause.toLowerCase().indexOf("nextval") >= 0) {
						pos1 = defaultClause.indexOf("'");
						pos2 = defaultClause.lastIndexOf("'");
						if (pos1 != -1 && pos2 != pos1) {
							sequenceName = defaultClause.substring(pos1 + 1, pos2).replace('"', ' ').trim();
							if (!relationExists(sequenceName, replicant)) {
								seqStmnt.execute("CREATE SEQUENCE " + sequenceName);
							}
						}
					}

					sb.append(" DEFAULT ").append(defaultClause);

				}
			}

			// Add the primary key portion, if necessary
			rs = metaData.getPrimaryKeys(null, null, tableName);

			first = true;
			while (rs.next()) {

				if (first) {
					sb.append(", PRIMARY KEY (");
					first = false;
				} else {
					sb.append(',');
				}

				sb.append(rs.getString(4));

			}

			// if there were no primary keys, first won't be set to false, and we don't need to add an ending ')'
			if (!first) {
				sb.append(')');
			}

			// end the statement
			sb.append(')');

			// Create the table

			stmnt = replicant.createStatement();
			stmnt.execute(sb.toString());
			stmnt.close();

			// Copy data from one to the other using pg_dump through a temp file
			try {
				if (!quiet) {
					System.out.println("Copying data...");
				}
				stmnt = replicant.createStatement();

				File f = File.createTempFile("ers", null);
				String url = metaData.getURL();
				String db = url.substring(url.lastIndexOf("/") + 1);

				// Dump data, then read in and execute the INSERT statements. Sssllooowww....
				String exec = "pg_dump -Da -f " + f.getPath() + " -U " + metaData.getUserName() + " -t " + table + " " + db;

				try {
					Process p = Runtime.getRuntime().exec(exec);
					p.waitFor();
					// wait for it....
				} catch (Exception ex) {
					ex.printStackTrace();
				}

				String line;
				LineNumberReader in = new LineNumberReader(new FileReader(f));
				// Load data into replicant
				while ((line = in.readLine()) != null) {
					if (line.startsWith("INSERT")) {
						stmnt.execute(line);
					}
				}
				stmnt.close();
				f.delete();
			} catch (IOException iox) {
				iox.printStackTrace();
			}

			// Recreate indexes
			seqStmnt = primary.createStatement();
			stmnt = replicant.createStatement();
			rs = seqStmnt.executeQuery("SELECT * FROM pg_indexes WHERE tablename='" + tableName + "'");
			while (rs.next()) {
				indexName = rs.getString(3);
				if (!indexName.endsWith("_pkey")) {
					// skip primary keys
					stmnt.execute(rs.getString(4));
				}
			}

			rs.close();

			// Insert the table/column into the control tables
			stmnt = replicant.createStatement();

			stmnt.execute("INSERT INTO _rserv_slave_tables_ (tname, cname, reloid, key) " +
					"SELECT r.relname, a.attname, r.oid, a.attnum  " +
					"FROM pg_class r, pg_attribute a  " +
					"WHERE r.oid = a.attrelid " +
					"AND r.relname = '" + table + "' " +
					"AND a.attname = '" + uniqColumn + "'");
			stmnt.close();

		} catch (SQLException sx) {
			sx.printStackTrace();
		} finally {

			// Close the statements, if necessary
			try {
				if (stmnt != null) {
					stmnt.close();
				}
			} catch (SQLException ex) {

			}
			try {
				if (seqStmnt != null) {
					seqStmnt.close();
				}
			} catch (SQLException ex) {

			}
		}

	}

	/**
	 *  Check for the existance of the eRServer unique ID
	 *
	 * @param  table       Table to check
	 * @param  c           Connection to use
	 * @param  columnName  Description of the Parameter
	 * @return             Yes/no
	 */
	protected boolean hasERSUniq(String table, String columnName, Connection c) {

		Statement stmnt = null;

		try {
			stmnt = c.createStatement();
			ResultSet rs = stmnt.executeQuery("SELECT pga.attname " +
					"FROM pg_class pgc, pg_attribute pga  " +
					"WHERE pgc.relname = '" + table + "' " +
					"AND pgc.oid = pga.attrelid " +
					"AND pga.attname = '" + columnName + "'");
			return rs.next();
		} catch (SQLException ex) {
			ex.printStackTrace();
			return false;
		} finally {
			try {
				stmnt.close();
			} catch (SQLException sx) {}
		}

	}

	/**
	 *  Check to see if a relation exists
	 *
	 * @param  table  Table to check
	 * @param  c      Connection to use
	 * @return        True/false
	 */
	protected boolean relationExists(String table, Connection c) {

		Statement stmnt = null;

		try {
			stmnt = c.createStatement();
			ResultSet rs = stmnt.executeQuery("SELECT relname FROM pg_class WHERE relname = '" + table + "'");
			return rs.next();
		} catch (SQLException ex) {
			ex.printStackTrace();
			return false;
		} finally {
			try {
				stmnt.close();
			} catch (SQLException sx) {}
		}

	}


	/**
	 *  Create the necessary tables/views on a new replicated server 
	 *
	 * @param  replicant         Connection to use
	 * @exception  SQLException  
	 */
	protected void setupReplicant(Connection replicant) throws SQLException {
		Statement stmnt = null;
		try {
			stmnt = replicant.createStatement();
			stmnt.execute("CREATE TABLE _rserv_slave_tables_ (tname name, cname name, reloid oid, key int4)");
			stmnt.execute("CREATE TABLE _rserv_slave_sync_ (syncid int4, synctime timestamp)");
			// This is a convenience view to deal with getting the oid/attnum when adding tables with schema support. Its basically
			// the pg_tables view with the oid thrown in
//			stmnt.execute("CREATE VIEW _ers_class_attr AS SELECT n.nspname AS schemaname, c.oid , c.relname AS tablename, n.nspname || '.' || c.relname AS relname " +
//					" FROM (pg_class c LEFT JOIN pg_namespace n ON ((n.oid = c.relnamespace))) WHERE ((c.relkind = 'r'::\"char\") OR (c.relkind = 's'::\"char\"))");
		} finally {
			try {
				if (stmnt != null) {
					stmnt.close();
				}
			} catch (SQLException sx) {}
		}
	}

	/**
	 *  Check to see if a table exists. This is different than relationExists in that it supports schemas.
	 *
	 * @param  table  Table to check
	 * @param  c      Connection to use
	 * @return        True/false
	 */
	protected boolean tableExists(String table, Connection c) {
		return relationExists(table,c);

		/*Statement stmnt = null;
		String schema;
		String tableName;
		int pos;

		// Separate the table name to schema and table, if necessary
		if ((pos = table.indexOf(".")) >= 0) {
			schema = table.substring(0, pos);
			tableName = table.substring(pos + 1);
		} else {
			tableName = table;
			schema = "public";
		}
		try {
			stmnt = c.createStatement();
			ResultSet rs = stmnt.executeQuery("SELECT tablename FROM pg_tables WHERE schemaname = '" + schema + "' AND tablename = '" + tableName + "'");
			return rs.next();
		} catch (SQLException ex) {
			ex.printStackTrace();
			return false;
		} finally {
			try {
				stmnt.close();
			} catch (SQLException sx) {}
		}*/

	}

}

