/*-------
 * Module:			dlg_specific.c
 *
 * Description:		This module contains any specific code for handling
 *					dialog boxes such as driver/datasource options.  Both the
 *					ConfigDSN() and the SQLDriverConnect() functions use
 *					functions in this module.  If you were to add a new option
 *					to any dialog box, you would most likely only have to change
 *					things in here rather than in 2 separate places as before.
 *
 * Classes:			none
 *
 * API functions:	none
 *
 * Comments:		See "notice.txt" for copyright and license information.
 *-------
 */
/* Multibyte support	Eiji Tokuya 2001-03-15 */

#include "dlg_specific.h"

#include "convert.h"

#include "multibyte.h"
#include "pgapifunc.h"

#ifndef BOOL
#define BOOL	int
#endif
#ifndef FALSE
#define FALSE	(BOOL)0
#endif
#ifndef TRUE
#define TRUE	(BOOL)1
#endif

extern GLOBAL_VALUES globals;

void
makeConnectString(char *connect_string, const ConnInfo *ci, UWORD len)
{
	char		got_dsn = (ci->dsn[0] != '\0');
	char		encoded_conn_settings[LARGE_REGISTRY_LEN];
	UWORD		hlen;
	/*BOOL		abbrev = (len <= 400);*/
	BOOL		abbrev = (len < 1024);

	/* fundamental info */
	sprintf(connect_string, "%s=%s;DATABASE=%s;SERVER=%s;PORT=%s;UID=%s;PWD=%s",
			got_dsn ? "DSN" : "DRIVER",
			got_dsn ? ci->dsn : ci->drivername,
			ci->database,
			ci->server,
			ci->port,
			ci->username,
			ci->password);

	encode(ci->conn_settings, encoded_conn_settings);

	/* extra info */
	hlen = strlen(connect_string);
	if (!abbrev)
		sprintf(&connect_string[hlen],
				";%s=%s;%s=%s;%s=%s;%s=%s;%s=%s;%s=%s;%s=%s;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%s;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d;%s=%d",
				INI_READONLY,
				ci->onlyread,
				INI_PROTOCOL,
				ci->protocol,
				INI_FAKEOIDINDEX,
				ci->fake_oid_index,
				INI_SHOWOIDCOLUMN,
				ci->show_oid_column,
				INI_ROWVERSIONING,
				ci->row_versioning,
				INI_SHOWSYSTEMTABLES,
				ci->show_system_tables,
				INI_CONNSETTINGS,
				encoded_conn_settings,
				INI_FETCH,
				ci->drivers.fetch_max,
				INI_SOCKET,
				ci->drivers.socket_buffersize,
				INI_UNKNOWNSIZES,
				ci->drivers.unknown_sizes,
				INI_MAXVARCHARSIZE,
				ci->drivers.max_varchar_size,
				INI_MAXLONGVARCHARSIZE,
				ci->drivers.max_longvarchar_size,
				INI_DEBUG,
				ci->drivers.debug,
				INI_COMMLOG,
				ci->drivers.commlog,
				INI_OPTIMIZER,
				ci->drivers.disable_optimizer,
				INI_KSQO,
				ci->drivers.ksqo,
				INI_USEDECLAREFETCH,
				ci->drivers.use_declarefetch,
				INI_TEXTASLONGVARCHAR,
				ci->drivers.text_as_longvarchar,
				INI_UNKNOWNSASLONGVARCHAR,
				ci->drivers.unknowns_as_longvarchar,
				INI_BOOLSASCHAR,
				ci->drivers.bools_as_char,
				INI_PARSE,
				ci->drivers.parse,
				INI_CANCELASFREESTMT,
				ci->drivers.cancel_as_freestmt,
				INI_EXTRASYSTABLEPREFIXES,
				ci->drivers.extra_systable_prefixes,
				INI_LFCONVERSION,
				ci->lf_conversion,
				INI_UPDATABLECURSORS,
				ci->allow_keyset,
				INI_DISALLOWPREMATURE,
				ci->disallow_premature,
				INI_TRUEISMINUS1,
				ci->true_is_minus1,
				INI_INT8AS,
				ci->int8_as,
				INI_BYTEAASLONGVARBINARY,
				ci->bytea_as_longvarbinary,
				INI_USESERVERSIDEPREPARE,
				ci->use_server_side_prepare);
	/* Abbrebiation is needed ? */
	if (abbrev || strlen(connect_string) >= len)
	{
		unsigned long flag = 0;
		if (ci->disallow_premature)
			flag |= BIT_DISALLOWPREMATURE;
		if (ci->allow_keyset)
			flag |= BIT_UPDATABLECURSORS;
		if (ci->lf_conversion)
			flag |= BIT_LFCONVERSION;
		if (ci->drivers.unique_index)
			flag |= BIT_UNIQUEINDEX;
		if (strncmp(ci->protocol, PG64, strlen(PG64)) == 0)
			flag |= BIT_PROTOCOL_64;
		else if (strncmp(ci->protocol, PG63, strlen(PG63)) == 0)
			flag |= BIT_PROTOCOL_63;
		switch (ci->drivers.unknown_sizes)
		{
			case UNKNOWNS_AS_DONTKNOW:
				flag |= BIT_UNKNOWN_DONTKNOW;
				break;
			case UNKNOWNS_AS_MAX:
				flag |= BIT_UNKNOWN_ASMAX;
				break;
		}
		if (ci->drivers.disable_optimizer)
			flag |= BIT_OPTIMIZER;
		if (ci->drivers.ksqo)
			flag |= BIT_KSQO;
		if (ci->drivers.commlog)
			flag |= BIT_COMMLOG;
		if (ci->drivers.debug)
			flag |= BIT_DEBUG;
		if (ci->drivers.parse)
			flag |= BIT_PARSE;
		if (ci->drivers.cancel_as_freestmt)
			flag |= BIT_CANCELASFREESTMT;
		if (ci->drivers.use_declarefetch)
			flag |= BIT_USEDECLAREFETCH;
		if (ci->onlyread[0] == '1')
			flag |= BIT_READONLY;
		if (ci->drivers.text_as_longvarchar)
			flag |= BIT_TEXTASLONGVARCHAR;
		if (ci->drivers.unknowns_as_longvarchar)
			flag |= BIT_UNKNOWNSASLONGVARCHAR;
		if (ci->drivers.bools_as_char)
			flag |= BIT_BOOLSASCHAR;
		if (ci->row_versioning[0] == '1')
			flag |= BIT_ROWVERSIONING;
		if (ci->show_system_tables[0] == '1')
			flag |= BIT_SHOWSYSTEMTABLES;
		if (ci->show_oid_column[0] == '1')
			flag |= BIT_SHOWOIDCOLUMN;
		if (ci->fake_oid_index[0] == '1')
			flag |= BIT_FAKEOIDINDEX;
		if (ci->true_is_minus1)
			flag |= BIT_TRUEISMINUS1;
		if (ci->bytea_as_longvarbinary)
			flag |= BIT_BYTEAASLONGVARBINARY;
		if (ci->use_server_side_prepare)
			flag |= BIT_USESERVERSIDEPREPARE;

		sprintf(&connect_string[hlen],
				";A6=%s;A7=%d;A8=%d;B0=%d;B1=%d;%s=%d;C2=%s;CX=%02x%lx",
				encoded_conn_settings,
				ci->drivers.fetch_max,
				ci->drivers.socket_buffersize,
				ci->drivers.max_varchar_size,
				ci->drivers.max_longvarchar_size,
				INI_INT8AS,
				ci->int8_as,
				ci->drivers.extra_systable_prefixes,
				EFFECTIVE_BIT_COUNT,
				flag);
	}
}

static void
unfoldCXAttribute(ConnInfo *ci, const char *value)
{
	int		count;
	unsigned long	flag;

	if (strlen(value) < 2)
	{
		count = 3;
		sscanf(value, "%lx", &flag);
	}
	else
	{
		char	cnt[8];
		memcpy(cnt, value, 2);
		cnt[2] = '\0';
		sscanf(cnt, "%x", &count);
		sscanf(value + 2, "%lx", &flag);
	}
	ci->disallow_premature = (char)((flag & BIT_DISALLOWPREMATURE) != 0);
	ci->allow_keyset = (char)((flag & BIT_UPDATABLECURSORS) != 0);
	ci->lf_conversion = (char)((flag & BIT_LFCONVERSION) != 0);
	if (count < 4)
		return;
	ci->drivers.unique_index = (char)((flag & BIT_UNIQUEINDEX) != 0);
	if ((flag & BIT_PROTOCOL_64) != 0)
		strcpy(ci->protocol, PG64);
	else if ((flag & BIT_PROTOCOL_63) != 0)
		strcpy(ci->protocol, PG63);
	else
		strcpy(ci->protocol, PG62);
	if ((flag & BIT_UNKNOWN_DONTKNOW) != 0)
		ci->drivers.unknown_sizes = UNKNOWNS_AS_DONTKNOW;
	else if ((flag & BIT_UNKNOWN_ASMAX) != 0)
		ci->drivers.unknown_sizes = UNKNOWNS_AS_MAX;
	else 
		ci->drivers.unknown_sizes = UNKNOWNS_AS_LONGEST;
	ci->drivers.disable_optimizer = (char)((flag & BIT_OPTIMIZER) != 0);
	ci->drivers.ksqo = (char)((flag & BIT_KSQO) != 0);
	ci->drivers.commlog = (char)((flag & BIT_COMMLOG) != 0);
	ci->drivers.debug = (char)((flag & BIT_DEBUG) != 0);
	ci->drivers.parse = (char)((flag & BIT_PARSE) != 0);
	ci->drivers.cancel_as_freestmt = (char)((flag & BIT_CANCELASFREESTMT) != 0);
	ci->drivers.use_declarefetch = (char)((flag & BIT_USEDECLAREFETCH) != 0);
	sprintf(ci->onlyread, "%d", (char)((flag & BIT_READONLY) != 0));
	ci->drivers.text_as_longvarchar = (char)((flag & BIT_TEXTASLONGVARCHAR) !=0);
	ci->drivers.unknowns_as_longvarchar = (char)((flag & BIT_UNKNOWNSASLONGVARCHAR) !=0);
	ci->drivers.bools_as_char = (char)((flag & BIT_BOOLSASCHAR) != 0);
	sprintf(ci->row_versioning, "%d", (char)((flag & BIT_ROWVERSIONING) != 0));
	sprintf(ci->show_system_tables, "%d", (char)((flag & BIT_SHOWSYSTEMTABLES) != 0));
	sprintf(ci->show_oid_column, "%d", (char)((flag & BIT_SHOWOIDCOLUMN) != 0));
	sprintf(ci->fake_oid_index, "%d", (char)((flag & BIT_FAKEOIDINDEX) != 0));
	ci->true_is_minus1 = (char)((flag & BIT_TRUEISMINUS1) != 0);
	ci->bytea_as_longvarbinary = (char)((flag & BIT_BYTEAASLONGVARBINARY) != 0);
	ci->use_server_side_prepare = (char)((flag & BIT_USESERVERSIDEPREPARE) != 0);
}
void
copyAttributes(ConnInfo *ci, const char *attribute, const char *value)
{
	if (stricmp(attribute, "DSN") == 0)
		strcpy(ci->dsn, value);

	else if (stricmp(attribute, "driver") == 0)
		strcpy(ci->drivername, value);

	else if (stricmp(attribute, INI_DATABASE) == 0)
		strcpy(ci->database, value);

	else if (stricmp(attribute, INI_SERVER) == 0 || stricmp(attribute, "server") == 0)
		strcpy(ci->server, value);

	else if (stricmp(attribute, INI_USER) == 0 || stricmp(attribute, "uid") == 0)
		strcpy(ci->username, value);

	else if (stricmp(attribute, INI_PASSWORD) == 0 || stricmp(attribute, "pwd") == 0)
		strcpy(ci->password, value);

	else if (stricmp(attribute, INI_PORT) == 0)
		strcpy(ci->port, value);

	else if (stricmp(attribute, INI_READONLY) == 0 || stricmp(attribute, "A0") == 0)
		strcpy(ci->onlyread, value);

	else if (stricmp(attribute, INI_PROTOCOL) == 0 || stricmp(attribute, "A1") == 0)
		strcpy(ci->protocol, value);

	else if (stricmp(attribute, INI_SHOWOIDCOLUMN) == 0 || stricmp(attribute, "A3") == 0)
		strcpy(ci->show_oid_column, value);

	else if (stricmp(attribute, INI_FAKEOIDINDEX) == 0 || stricmp(attribute, "A2") == 0)
		strcpy(ci->fake_oid_index, value);

	else if (stricmp(attribute, INI_ROWVERSIONING) == 0 || stricmp(attribute, "A4") == 0)
		strcpy(ci->row_versioning, value);

	else if (stricmp(attribute, INI_SHOWSYSTEMTABLES) == 0 || stricmp(attribute, "A5") == 0)
		strcpy(ci->show_system_tables, value);

	else if (stricmp(attribute, INI_CONNSETTINGS) == 0 || stricmp(attribute, "A6") == 0)
	{
		decode(value, ci->conn_settings);
		/* strcpy(ci->conn_settings, value); */
	}
	else if (stricmp(attribute, INI_DISALLOWPREMATURE) == 0 || stricmp(attribute, "C3") == 0)
		ci->disallow_premature = atoi(value);
	else if (stricmp(attribute, INI_UPDATABLECURSORS) == 0 || stricmp(attribute, "C4") == 0)
		ci->allow_keyset = atoi(value);
	else if (stricmp(attribute, INI_LFCONVERSION) == 0)
		ci->lf_conversion = atoi(value);
	else if (stricmp(attribute, INI_TRUEISMINUS1) == 0)
		ci->true_is_minus1 = atoi(value);
	else if (stricmp(attribute, INI_INT8AS) == 0)
		ci->int8_as = atoi(value);
	else if (stricmp(attribute, INI_BYTEAASLONGVARBINARY) == 0)
		ci->bytea_as_longvarbinary = atoi(value);
	else if (stricmp(attribute, INI_USESERVERSIDEPREPARE) == 0)
		ci->use_server_side_prepare = atoi(value);
	else if (stricmp(attribute, "CX") == 0)
		unfoldCXAttribute(ci, value);

	mylog("copyAttributes: DSN='%s',server='%s',dbase='%s',user='%s',passwd='%s',port='%s',onlyread='%s',protocol='%s',conn_settings='%s',disallow_premature=%d)\n", ci->dsn, ci->server, ci->database, ci->username, ci->password ? "xxxxx" : "", ci->port, ci->onlyread, ci->protocol, ci->conn_settings, ci->disallow_premature);
}

void
copyCommonAttributes(ConnInfo *ci, const char *attribute, const char *value)
{
	if (stricmp(attribute, INI_FETCH) == 0 || stricmp(attribute, "A7") == 0)
		ci->drivers.fetch_max = atoi(value);
	else if (stricmp(attribute, INI_SOCKET) == 0 || stricmp(attribute, "A8") == 0)
		ci->drivers.socket_buffersize = atoi(value);
	else if (stricmp(attribute, INI_DEBUG) == 0 || stricmp(attribute, "B2") == 0)
		ci->drivers.debug = atoi(value);
	else if (stricmp(attribute, INI_COMMLOG) == 0 || stricmp(attribute, "B3") == 0)
		ci->drivers.commlog = atoi(value);
	else if (stricmp(attribute, INI_OPTIMIZER) == 0 || stricmp(attribute, "B4") == 0)
		ci->drivers.disable_optimizer = atoi(value);
	else if (stricmp(attribute, INI_KSQO) == 0 || stricmp(attribute, "B5") == 0)
		ci->drivers.ksqo = atoi(value);

	/*
	 * else if (stricmp(attribute, INI_UNIQUEINDEX) == 0 ||
	 * stricmp(attribute, "UIX") == 0) ci->drivers.unique_index =
	 * atoi(value);
	 */
	else if (stricmp(attribute, INI_UNKNOWNSIZES) == 0 || stricmp(attribute, "A9") == 0)
		ci->drivers.unknown_sizes = atoi(value);
	else if (stricmp(attribute, INI_LIE) == 0)
		ci->drivers.lie = atoi(value);
	else if (stricmp(attribute, INI_PARSE) == 0 || stricmp(attribute, "C0") == 0)
		ci->drivers.parse = atoi(value);
	else if (stricmp(attribute, INI_CANCELASFREESTMT) == 0 || stricmp(attribute, "C1") == 0)
		ci->drivers.cancel_as_freestmt = atoi(value);
	else if (stricmp(attribute, INI_USEDECLAREFETCH) == 0 || stricmp(attribute, "B6") == 0)
		ci->drivers.use_declarefetch = atoi(value);
	else if (stricmp(attribute, INI_MAXVARCHARSIZE) == 0 || stricmp(attribute, "B0") == 0)
		ci->drivers.max_varchar_size = atoi(value);
	else if (stricmp(attribute, INI_MAXLONGVARCHARSIZE) == 0 || stricmp(attribute, "B1") == 0)
		ci->drivers.max_longvarchar_size = atoi(value);
	else if (stricmp(attribute, INI_TEXTASLONGVARCHAR) == 0 || stricmp(attribute, "B7") == 0)
		ci->drivers.text_as_longvarchar = atoi(value);
	else if (stricmp(attribute, INI_UNKNOWNSASLONGVARCHAR) == 0 || stricmp(attribute, "B8") == 0)
		ci->drivers.unknowns_as_longvarchar = atoi(value);
	else if (stricmp(attribute, INI_BOOLSASCHAR) == 0 || stricmp(attribute, "B9") == 0)
		ci->drivers.bools_as_char = atoi(value);
	else if (stricmp(attribute, INI_EXTRASYSTABLEPREFIXES) == 0 || stricmp(attribute, "C2") == 0)
		strcpy(ci->drivers.extra_systable_prefixes, value);
	mylog("CopyCommonAttributes: A7=%d;A8=%d;A9=%d;B0=%d;B1=%d;B2=%d;B3=%d;B4=%d;B5=%d;B6=%d;B7=%d;B8=%d;B9=%d;C0=%d;C1=%d;C2=%s",
		  ci->drivers.fetch_max,
		  ci->drivers.socket_buffersize,
		  ci->drivers.unknown_sizes,
		  ci->drivers.max_varchar_size,
		  ci->drivers.max_longvarchar_size,
		  ci->drivers.debug,
		  ci->drivers.commlog,
		  ci->drivers.disable_optimizer,
		  ci->drivers.ksqo,
		  ci->drivers.use_declarefetch,
		  ci->drivers.text_as_longvarchar,
		  ci->drivers.unknowns_as_longvarchar,
		  ci->drivers.bools_as_char,
		  ci->drivers.parse,
		  ci->drivers.cancel_as_freestmt,
		  ci->drivers.extra_systable_prefixes);
}


void
getDSNdefaults(ConnInfo *ci)
{
	if (ci->port[0] == '\0')
		strcpy(ci->port, DEFAULT_PORT);

	if (ci->onlyread[0] == '\0')
		sprintf(ci->onlyread, "%d", globals.onlyread);

	if (ci->protocol[0] == '\0')
		strcpy(ci->protocol, globals.protocol);

	if (ci->fake_oid_index[0] == '\0')
		sprintf(ci->fake_oid_index, "%d", DEFAULT_FAKEOIDINDEX);

	if (ci->show_oid_column[0] == '\0')
		sprintf(ci->show_oid_column, "%d", DEFAULT_SHOWOIDCOLUMN);

	if (ci->show_system_tables[0] == '\0')
		sprintf(ci->show_system_tables, "%d", DEFAULT_SHOWSYSTEMTABLES);

	if (ci->row_versioning[0] == '\0')
		sprintf(ci->row_versioning, "%d", DEFAULT_ROWVERSIONING);

	if (ci->disallow_premature < 0)
		ci->disallow_premature = DEFAULT_DISALLOWPREMATURE;
	if (ci->allow_keyset < 0)
		ci->allow_keyset = DEFAULT_UPDATABLECURSORS;
	if (ci->lf_conversion < 0)
		ci->lf_conversion = DEFAULT_LFCONVERSION;
	if (ci->true_is_minus1 < 0)
		ci->true_is_minus1 = DEFAULT_TRUEISMINUS1;
	if (ci->int8_as < -100)
		ci->int8_as = DEFAULT_INT8AS;
	if (ci->bytea_as_longvarbinary < 0)
		ci->bytea_as_longvarbinary = DEFAULT_BYTEAASLONGVARBINARY;
	if (ci->use_server_side_prepare < 0)
		ci->use_server_side_prepare = DEFAULT_USESERVERSIDEPREPARE;
}

int
getDriverNameFromDSN(const char *dsn, char *driver_name, int namelen)
{
	return SQLGetPrivateProfileString(ODBC_DATASOURCES, dsn, "", driver_name, namelen, ODBC_INI);
}

void
getDSNinfo(ConnInfo *ci, char overwrite)
{
	char	   *DSN = ci->dsn;
	char		encoded_conn_settings[LARGE_REGISTRY_LEN],
				temp[SMALL_REGISTRY_LEN];

/*
 *	If a driver keyword was present, then dont use a DSN and return.
 *	If DSN is null and no driver, then use the default datasource.
 */
	if (DSN[0] == '\0')
	{
		if (ci->drivername[0] != '\0')
			return;
		else
			strcpy(DSN, INI_DSN);
	}

	/* brute-force chop off trailing blanks... */
	while (*(DSN + strlen(DSN) - 1) == ' ')
		*(DSN + strlen(DSN) - 1) = '\0';

	if (ci->drivername[0] == '\0' || overwrite)
	{
		getDriverNameFromDSN(DSN, ci->drivername, sizeof(ci->drivername));
		if (ci->drivername[0] && stricmp(ci->drivername, DBMS_NAME))
			getCommonDefaults(ci->drivername, ODBCINST_INI, ci);
	}

	/* Proceed with getting info for the given DSN. */

	if (ci->desc[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_KDESC, "", ci->desc, sizeof(ci->desc), ODBC_INI);

	if (ci->server[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_SERVER, "", ci->server, sizeof(ci->server), ODBC_INI);

	if (ci->database[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_DATABASE, "", ci->database, sizeof(ci->database), ODBC_INI);

	if (ci->username[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_USER, "", ci->username, sizeof(ci->username), ODBC_INI);

	if (ci->password[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_PASSWORD, "", ci->password, sizeof(ci->password), ODBC_INI);

	if (ci->port[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_PORT, "", ci->port, sizeof(ci->port), ODBC_INI);

	if (ci->onlyread[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_READONLY, "", ci->onlyread, sizeof(ci->onlyread), ODBC_INI);

	if (ci->show_oid_column[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_SHOWOIDCOLUMN, "", ci->show_oid_column, sizeof(ci->show_oid_column), ODBC_INI);

	if (ci->fake_oid_index[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_FAKEOIDINDEX, "", ci->fake_oid_index, sizeof(ci->fake_oid_index), ODBC_INI);

	if (ci->row_versioning[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_ROWVERSIONING, "", ci->row_versioning, sizeof(ci->row_versioning), ODBC_INI);

	if (ci->show_system_tables[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_SHOWSYSTEMTABLES, "", ci->show_system_tables, sizeof(ci->show_system_tables), ODBC_INI);

	if (ci->protocol[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_PROTOCOL, "", ci->protocol, sizeof(ci->protocol), ODBC_INI);

	if (ci->conn_settings[0] == '\0' || overwrite)
	{
		SQLGetPrivateProfileString(DSN, INI_CONNSETTINGS, "", encoded_conn_settings, sizeof(encoded_conn_settings), ODBC_INI);
		decode(encoded_conn_settings, ci->conn_settings);
	}

	if (ci->translation_dll[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_TRANSLATIONDLL, "", ci->translation_dll, sizeof(ci->translation_dll), ODBC_INI);

	if (ci->translation_option[0] == '\0' || overwrite)
		SQLGetPrivateProfileString(DSN, INI_TRANSLATIONOPTION, "", ci->translation_option, sizeof(ci->translation_option), ODBC_INI);

	if (ci->disallow_premature < 0 || overwrite)
	{
		SQLGetPrivateProfileString(DSN, INI_DISALLOWPREMATURE, "", temp, sizeof(temp), ODBC_INI);
		if (temp[0])
			ci->disallow_premature = atoi(temp);
	}

	if (ci->allow_keyset < 0 || overwrite)
	{
		SQLGetPrivateProfileString(DSN, INI_UPDATABLECURSORS, "", temp, sizeof(temp), ODBC_INI);
		if (temp[0])
			ci->allow_keyset = atoi(temp);
	}

	if (ci->lf_conversion < 0 || overwrite)
	{
		SQLGetPrivateProfileString(DSN, INI_LFCONVERSION, "", temp, sizeof(temp), ODBC_INI);
		if (temp[0])
			ci->lf_conversion = atoi(temp);
	}

	if (ci->true_is_minus1 < 0 || overwrite)
	{
		SQLGetPrivateProfileString(DSN, INI_TRUEISMINUS1, "", temp, sizeof(temp), ODBC_INI);
		if (temp[0])
			ci->true_is_minus1 = atoi(temp);
	}

	if (ci->int8_as < -100 || overwrite)
	{
		SQLGetPrivateProfileString(DSN, INI_INT8AS, "", temp, sizeof(temp), ODBC_INI);
		if (temp[0])
			ci->int8_as = atoi(temp);
	}

	if (ci->bytea_as_longvarbinary < 0 || overwrite)
	{
		SQLGetPrivateProfileString(DSN, INI_BYTEAASLONGVARBINARY, "", temp, sizeof(temp), ODBC_INI);
		if (temp[0])
			ci->bytea_as_longvarbinary = atoi(temp);
	}

	if (ci->use_server_side_prepare < 0 || overwrite)
	{
		SQLGetPrivateProfileString(DSN, INI_USESERVERSIDEPREPARE, "", temp, sizeof(temp), ODBC_INI);
		if (temp[0])
			ci->use_server_side_prepare = atoi(temp);
	}

	/* Allow override of odbcinst.ini parameters here */
	getCommonDefaults(DSN, ODBC_INI, ci);

	qlog("DSN info: DSN='%s',server='%s',port='%s',dbase='%s',user='%s',passwd='%s'\n",
		 DSN,
		 ci->server,
		 ci->port,
		 ci->database,
		 ci->username,
		 ci->password ? "xxxxx" : "");
	qlog("          onlyread='%s',protocol='%s',showoid='%s',fakeoidindex='%s',showsystable='%s'\n",
		 ci->onlyread,
		 ci->protocol,
		 ci->show_oid_column,
		 ci->fake_oid_index,
		 ci->show_system_tables);

	check_client_encoding(ci->conn_settings);
	qlog("          conn_settings='%s',conn_encoding='%s'\n",
		 ci->conn_settings,
		 check_client_encoding(ci->conn_settings));
	qlog("          translation_dll='%s',translation_option='%s'\n",
		 ci->translation_dll,
		 ci->translation_option);
}

/*
 *	This function writes any global parameters (that can be manipulated)
 *	to the ODBCINST.INI portion of the registry
 */
void
writeDriverCommoninfo(const char *fileName, const char *sectionName,
			 const GLOBAL_VALUES *comval)
{
	char		tmp[128];

	if (ODBCINST_INI == fileName && NULL == sectionName)
		sectionName = DBMS_NAME;
 
	sprintf(tmp, "%d", comval->fetch_max);
	SQLWritePrivateProfileString(sectionName,
								 INI_FETCH, tmp, fileName);

	sprintf(tmp, "%d", comval->commlog);
	SQLWritePrivateProfileString(sectionName,
								 INI_COMMLOG, tmp, fileName);

	sprintf(tmp, "%d", comval->debug);
	SQLWritePrivateProfileString(sectionName,
								 INI_DEBUG, tmp, fileName);

	sprintf(tmp, "%d", comval->disable_optimizer);
	SQLWritePrivateProfileString(sectionName,
								 INI_OPTIMIZER, tmp, fileName);

	sprintf(tmp, "%d", comval->ksqo);
	SQLWritePrivateProfileString(sectionName,
								 INI_KSQO, tmp, fileName);

	sprintf(tmp, "%d", comval->unique_index);
	SQLWritePrivateProfileString(sectionName, INI_UNIQUEINDEX, tmp, fileName);
	/*
	 * Never update the onlyread from this module.
	 */
	if (ODBCINST_INI == fileName)
	{
		sprintf(tmp, "%d", comval->onlyread);
		SQLWritePrivateProfileString(sectionName, INI_READONLY, tmp,
									 fileName);
	}

	sprintf(tmp, "%d", comval->use_declarefetch);
	SQLWritePrivateProfileString(sectionName,
								 INI_USEDECLAREFETCH, tmp, fileName);

	sprintf(tmp, "%d", comval->unknown_sizes);
	SQLWritePrivateProfileString(sectionName,
								 INI_UNKNOWNSIZES, tmp, fileName);

	sprintf(tmp, "%d", comval->text_as_longvarchar);
	SQLWritePrivateProfileString(sectionName,
								 INI_TEXTASLONGVARCHAR, tmp, fileName);

	sprintf(tmp, "%d", comval->unknowns_as_longvarchar);
	SQLWritePrivateProfileString(sectionName,
							   INI_UNKNOWNSASLONGVARCHAR, tmp, fileName);

	sprintf(tmp, "%d", comval->bools_as_char);
	SQLWritePrivateProfileString(sectionName,
								 INI_BOOLSASCHAR, tmp, fileName);

	sprintf(tmp, "%d", comval->parse);
	SQLWritePrivateProfileString(sectionName,
								 INI_PARSE, tmp, fileName);

	sprintf(tmp, "%d", comval->cancel_as_freestmt);
	SQLWritePrivateProfileString(sectionName,
								 INI_CANCELASFREESTMT, tmp, fileName);

	sprintf(tmp, "%d", comval->max_varchar_size);
	SQLWritePrivateProfileString(sectionName,
								 INI_MAXVARCHARSIZE, tmp, fileName);

	sprintf(tmp, "%d", comval->max_longvarchar_size);
	SQLWritePrivateProfileString(sectionName,
								 INI_MAXLONGVARCHARSIZE, tmp, fileName);

	SQLWritePrivateProfileString(sectionName,
	INI_EXTRASYSTABLEPREFIXES, comval->extra_systable_prefixes, fileName);

	/*
	 * Never update the conn_setting from this module
	 * SQLWritePrivateProfileString(sectionName, INI_CONNSETTINGS,
	 * comval->conn_settings, fileName);
	 */
}

/*	This is for datasource based options only */
void
writeDSNinfo(const ConnInfo *ci)
{
	const char *DSN = ci->dsn;
	char		encoded_conn_settings[LARGE_REGISTRY_LEN],
				temp[SMALL_REGISTRY_LEN];

	encode(ci->conn_settings, encoded_conn_settings);

	SQLWritePrivateProfileString(DSN,
								 INI_KDESC,
								 ci->desc,
								 ODBC_INI);

	SQLWritePrivateProfileString(DSN,
								 INI_DATABASE,
								 ci->database,
								 ODBC_INI);

	SQLWritePrivateProfileString(DSN,
								 INI_SERVER,
								 ci->server,
								 ODBC_INI);

	SQLWritePrivateProfileString(DSN,
								 INI_PORT,
								 ci->port,
								 ODBC_INI);

	SQLWritePrivateProfileString(DSN,
								 INI_USER,
								 ci->username,
								 ODBC_INI);

	SQLWritePrivateProfileString(DSN,
								 INI_PASSWORD,
								 ci->password,
								 ODBC_INI);

	SQLWritePrivateProfileString(DSN,
								 INI_READONLY,
								 ci->onlyread,
								 ODBC_INI);

	SQLWritePrivateProfileString(DSN,
								 INI_SHOWOIDCOLUMN,
								 ci->show_oid_column,
								 ODBC_INI);

	SQLWritePrivateProfileString(DSN,
								 INI_FAKEOIDINDEX,
								 ci->fake_oid_index,
								 ODBC_INI);

	SQLWritePrivateProfileString(DSN,
								 INI_ROWVERSIONING,
								 ci->row_versioning,
								 ODBC_INI);

	SQLWritePrivateProfileString(DSN,
								 INI_SHOWSYSTEMTABLES,
								 ci->show_system_tables,
								 ODBC_INI);

	SQLWritePrivateProfileString(DSN,
								 INI_PROTOCOL,
								 ci->protocol,
								 ODBC_INI);

	SQLWritePrivateProfileString(DSN,
								 INI_CONNSETTINGS,
								 encoded_conn_settings,
								 ODBC_INI);

	sprintf(temp, "%d", ci->disallow_premature);
	SQLWritePrivateProfileString(DSN,
								 INI_DISALLOWPREMATURE,
								 temp,
								 ODBC_INI);
	sprintf(temp, "%d", ci->allow_keyset);
	SQLWritePrivateProfileString(DSN,
								 INI_UPDATABLECURSORS,
								 temp,
								 ODBC_INI);
	sprintf(temp, "%d", ci->lf_conversion);
	SQLWritePrivateProfileString(DSN,
								 INI_LFCONVERSION,
								 temp,
								 ODBC_INI);
	sprintf(temp, "%d", ci->true_is_minus1);
	SQLWritePrivateProfileString(DSN,
								 INI_TRUEISMINUS1,
								 temp,
								 ODBC_INI);
	sprintf(temp, "%d", ci->int8_as);
	SQLWritePrivateProfileString(DSN,
								 INI_INT8AS,
								 temp,
								 ODBC_INI);
	sprintf(temp, "%d", ci->bytea_as_longvarbinary);
	SQLWritePrivateProfileString(DSN,
								 INI_BYTEAASLONGVARBINARY,
								 temp,
								 ODBC_INI);
	sprintf(temp, "%d", ci->use_server_side_prepare);
	SQLWritePrivateProfileString(DSN,
								 INI_USESERVERSIDEPREPARE,
								 temp,
								 ODBC_INI);
}


/*
 *	This function reads the ODBCINST.INI portion of
 *	the registry and gets any driver defaults.
 */
void
getCommonDefaults(const char *section, const char *filename, ConnInfo *ci)
{
	char		temp[256];
	GLOBAL_VALUES *comval;

	if (ci)
		comval = &(ci->drivers);
	else
		comval = &globals;
	/* Fetch Count is stored in driver section */
	SQLGetPrivateProfileString(section, INI_FETCH, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
	{
		comval->fetch_max = atoi(temp);
		/* sanity check if using cursors */
		if (comval->fetch_max <= 0)
			comval->fetch_max = FETCH_MAX;
	}
	else if (!ci)
		comval->fetch_max = FETCH_MAX;

	/* Socket Buffersize is stored in driver section */
	SQLGetPrivateProfileString(section, INI_SOCKET, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->socket_buffersize = atoi(temp);
	else if (!ci)
		comval->socket_buffersize = SOCK_BUFFER_SIZE;

	/* Debug is stored in the driver section */
	SQLGetPrivateProfileString(section, INI_DEBUG, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->debug = atoi(temp);
	else if (!ci)
		comval->debug = DEFAULT_DEBUG;

	/* CommLog is stored in the driver section */
	SQLGetPrivateProfileString(section, INI_COMMLOG, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->commlog = atoi(temp);
	else if (!ci)
		comval->commlog = DEFAULT_COMMLOG;

	if (!ci)
		logs_on_off(0, 0, 0);
	/* Optimizer is stored in the driver section only */
	SQLGetPrivateProfileString(section, INI_OPTIMIZER, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->disable_optimizer = atoi(temp);
	else if (!ci)
		comval->disable_optimizer = DEFAULT_OPTIMIZER;

	/* KSQO is stored in the driver section only */
	SQLGetPrivateProfileString(section, INI_KSQO, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->ksqo = atoi(temp);
	else if (!ci)
		comval->ksqo = DEFAULT_KSQO;

	/* Recognize Unique Index is stored in the driver section only */
	SQLGetPrivateProfileString(section, INI_UNIQUEINDEX, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->unique_index = atoi(temp);
	else if (!ci)
		comval->unique_index = DEFAULT_UNIQUEINDEX;


	/* Unknown Sizes is stored in the driver section only */
	SQLGetPrivateProfileString(section, INI_UNKNOWNSIZES, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->unknown_sizes = atoi(temp);
	else if (!ci)
		comval->unknown_sizes = DEFAULT_UNKNOWNSIZES;


	/* Lie about supported functions? */
	SQLGetPrivateProfileString(section, INI_LIE, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->lie = atoi(temp);
	else if (!ci)
		comval->lie = DEFAULT_LIE;

	/* Parse statements */
	SQLGetPrivateProfileString(section, INI_PARSE, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->parse = atoi(temp);
	else if (!ci)
		comval->parse = DEFAULT_PARSE;

	/* SQLCancel calls SQLFreeStmt in Driver Manager */
	SQLGetPrivateProfileString(section, INI_CANCELASFREESTMT, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->cancel_as_freestmt = atoi(temp);
	else if (!ci)
		comval->cancel_as_freestmt = DEFAULT_CANCELASFREESTMT;

	/* UseDeclareFetch is stored in the driver section only */
	SQLGetPrivateProfileString(section, INI_USEDECLAREFETCH, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->use_declarefetch = atoi(temp);
	else if (!ci)
		comval->use_declarefetch = DEFAULT_USEDECLAREFETCH;

	/* Max Varchar Size */
	SQLGetPrivateProfileString(section, INI_MAXVARCHARSIZE, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->max_varchar_size = atoi(temp);
	else if (!ci)
		comval->max_varchar_size = MAX_VARCHAR_SIZE;

	/* Max TextField Size */
	SQLGetPrivateProfileString(section, INI_MAXLONGVARCHARSIZE, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->max_longvarchar_size = atoi(temp);
	else if (!ci)
		comval->max_longvarchar_size = TEXT_FIELD_SIZE;

	/* Text As LongVarchar	*/
	SQLGetPrivateProfileString(section, INI_TEXTASLONGVARCHAR, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->text_as_longvarchar = atoi(temp);
	else if (!ci)
		comval->text_as_longvarchar = DEFAULT_TEXTASLONGVARCHAR;

	/* Unknowns As LongVarchar	*/
	SQLGetPrivateProfileString(section, INI_UNKNOWNSASLONGVARCHAR, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->unknowns_as_longvarchar = atoi(temp);
	else if (!ci)
		comval->unknowns_as_longvarchar = DEFAULT_UNKNOWNSASLONGVARCHAR;

	/* Bools As Char */
	SQLGetPrivateProfileString(section, INI_BOOLSASCHAR, "",
							   temp, sizeof(temp), filename);
	if (temp[0])
		comval->bools_as_char = atoi(temp);
	else if (!ci)
		comval->bools_as_char = DEFAULT_BOOLSASCHAR;

	/* Extra Systable prefixes */

	/*
	 * Use @@@ to distinguish between blank extra prefixes and no key
	 * entry
	 */
	SQLGetPrivateProfileString(section, INI_EXTRASYSTABLEPREFIXES, "@@@",
							   temp, sizeof(temp), filename);
	if (strcmp(temp, "@@@"))
		strcpy(comval->extra_systable_prefixes, temp);
	else if (!ci)
		strcpy(comval->extra_systable_prefixes, DEFAULT_EXTRASYSTABLEPREFIXES);

	mylog("globals.extra_systable_prefixes = '%s'\n", comval->extra_systable_prefixes);


	/* Dont allow override of an override! */
	if (!ci)
	{
		/*
		 * ConnSettings is stored in the driver section and per datasource
		 * for override
		 */
		SQLGetPrivateProfileString(section, INI_CONNSETTINGS, "",
		 comval->conn_settings, sizeof(comval->conn_settings), filename);

		/* Default state for future DSN's Readonly attribute */
		SQLGetPrivateProfileString(section, INI_READONLY, "",
								   temp, sizeof(temp), filename);
		if (temp[0])
			comval->onlyread = atoi(temp);
		else
			comval->onlyread = DEFAULT_READONLY;

		/*
		 * Default state for future DSN's protocol attribute This isn't a
		 * real driver option YET.	This is more intended for
		 * customization from the install.
		 */
		SQLGetPrivateProfileString(section, INI_PROTOCOL, "@@@",
								   temp, sizeof(temp), filename);
		if (strcmp(temp, "@@@"))
			strcpy(comval->protocol, temp);
		else
			strcpy(comval->protocol, DEFAULT_PROTOCOL);
	}
}
