/*
 * $Header: /home/t-ishii/repository/pgstattuple/pgstattuple.c,v 1.2 2001/08/30 06:21:48 t-ishii Exp $
 *
 * Copyright (c) 2001  Tatsuo Ishii
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without a
 * written agreement is hereby granted, provided that the above
 * copyright notice and this paragraph and the following two
 * paragraphs appear in all copies.
 */

#include "postgres.h"
#include "fmgr.h"
#include "access/heapam.h"
#include "access/transam.h"

PG_FUNCTION_INFO_V1(pgstattuple);

extern Datum pgstattuple(PG_FUNCTION_ARGS);

/* ----------
 * pgstattuple:
 * returns the percentage of dead tuples
 *
 * C FUNCTION definition
 * pgstattuple(NAME) returns FLOAT8
 * ----------
 */
Datum
pgstattuple(PG_FUNCTION_ARGS)
{
    Name	p = PG_GETARG_NAME(0);

    Relation	rel;
    HeapScanDesc	scan;
    HeapTuple	tuple;
    BlockNumber nblocks;
    double	table_len;
    uint64	tuple_len = 0;
    uint64	dead_tuple_len = 0;
    uint32	tuple_count = 0;
    uint32	dead_tuple_count = 0;
    double	tuple_percent;
    double	dead_tuple_percent;

    rel = heap_openr(NameStr(*p), NoLock);
    nblocks = RelationGetNumberOfBlocks(rel);
    scan = heap_beginscan(rel, false, SnapshotAny, 0, NULL);

    while ((tuple = heap_getnext(scan,0)))
    {
	if (HeapTupleSatisfiesNow(tuple->t_data))
	{
	    tuple_len += tuple->t_len;
	    tuple_count++;
	}
	else
	{
	    dead_tuple_len += tuple->t_len;
	    dead_tuple_count++;
	}
    }
    heap_endscan(scan);
    heap_close(rel, NoLock);

    table_len = (double)nblocks*BLCKSZ;

    if (nblocks == 0)
    {
	tuple_percent = 0.0;
	dead_tuple_percent = 0.0;
    }
    else
    {
	tuple_percent = (double)tuple_len*100.0/table_len;
	dead_tuple_percent = (double)dead_tuple_len*100.0/table_len;
    }

    elog(NOTICE,"physical length: %.2fMB live tuples: %u (%.2fMB, %.2f%%) dead tuples: %u (%.2fMB, %.2f%%) overhead: %.2f%%",

	 table_len/1024/1024,

	 tuple_count,
	 (double)tuple_len/1024/1024,
	 tuple_percent,

	 dead_tuple_count,
	 (double)dead_tuple_len/1024/1024,
	 dead_tuple_percent,

	 (nblocks == 0)?0.0: 100.0 - tuple_percent - dead_tuple_percent);

    PG_RETURN_FLOAT8(dead_tuple_percent);
}
