/*
* $Header: /home/t-ishii/repository/pgpool/pool_stream.c,v 1.9 2003/08/09 02:27:17 t-ishii Exp $
*
* pgpool: a language independent connection pool server for PostgreSQL 
* written by Tatsuo Ishii
*
* Copyright (c) 2003	Tatsuo Ishii
*
* Permission to use, copy, modify, and distribute this software and
* its documentation for any purpose and without fee is hereby
* granted, provided that the above copyright notice appear in all
* copies and that both that copyright notice and this permission
* notice appear in supporting documentation, and that the name of the
* author not be used in advertising or publicity pertaining to
* distribution of the software without specific, written prior
* permission. The author makes no representations about the
* suitability of this software for any purpose.  It is provided "as
* is" without express or implied warranty.
*
* pool_stream.c: stream I/O modules
*
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "pool.h"

#define READBUFSZ 1024

static int mystrlen(char *str, int upper, int *flag);
static int mystrlinelen(char *str, int upper, int *flag);

/*
* open read/write file descriptors.
* returns POOL_CONNECTION on success otherwise NULL.
*/
POOL_CONNECTION *pool_open(int fd)
{
	POOL_CONNECTION *cp;

	cp = (POOL_CONNECTION *)malloc(sizeof(POOL_CONNECTION));
	if (cp == NULL)
	{
		pool_error("pool_open: malloc failed: %s", strerror(errno));
		return NULL;
	}

	memset(cp, 0, sizeof(*cp));

	cp->read_fd = fdopen(fd, "r");
	if (cp->read_fd == NULL)
	{
		pool_error("pool_open: fdopen failed: %s",strerror(errno));
		free(cp);
		return NULL;
	}

	cp->write_fd = fdopen(fd, "w");
	if (cp->write_fd == NULL)
	{
		pool_error("pool_open: fdopen failed: %s",strerror(errno));
		fclose(cp->read_fd);
		free(cp);
		return NULL;
	}

	/* initialize pending data buffer */
	cp->hp = malloc(READBUFSZ);
	if (cp->hp == NULL)
	{
		pool_error("pool_open: malloc failed");
		return NULL;
	}
	cp->bufsz = READBUFSZ;
	cp->po = 0;
	cp->len = 0;

	cp->fd = fd;
	return cp;
}

/*
* close read/write file descriptors.
*/
void pool_close(POOL_CONNECTION *cp)
{
	fclose(cp->read_fd);
	fclose(cp->write_fd);
	close(cp->fd);
	free(cp);
}

/*
* read len bytes from cp
* returns 0 on success otherwise -1.
*/
int pool_read(POOL_CONNECTION *cp, void *buf, int len)
{
	int readlen;

	if (cp->len)
	{
		/* consume pending data */
		if (cp->len >= len)
		{
			/* there is enough pending data */
			memmove(buf, cp->hp+cp->po, len);
			cp->po += len;
			cp->len -= len;
			return 0;
		}
		else
		{
			memmove(buf, cp->hp+cp->po, cp->len);
			len -= cp->len;
			buf += cp->len;
			cp->len = 0;
		}
	}

	while (len > 0)
	{
		readlen = read(cp->fd, buf, len);
		if (readlen == -1)
		{
			pool_error("pool_read: read failed (%s)", strerror(errno));

			if (cp->isbackend)
			{
			    /* fatal error, notice to parent and exit */
			    notice_backend_error();
			    exit(1);
			}
			else
			{
			    return -1;
			}
		}
		else if (readlen == 0)
		{
			pool_error("pool_read: EOF encountered");

			if (cp->isbackend)
			{
			    /* fatal error, notice to parent and exit */
			    notice_backend_error();
				exit(1);
			}
			else
			{
				return -1;
			}
		}
		else if (len > readlen)
		{
			pool_debug("pool_read: expected %d bytes got %d retrying...", len, readlen);
			buf += readlen;
			len -= readlen;
		}
		else
		{
			break;
		}
	}

#ifdef DEBUG
	if (readlen == 1)
	{
		pool_debug("pool read %d byte (%c)", len, *((char *)buf));
	}
#endif

	return 0;
}

/*
* write len bytes from cp
* returns 0 on success otherwise -1.
*/
int pool_write(POOL_CONNECTION *cp, void *buf, int len)
{
	fwrite(buf, len, 1, cp->write_fd);

	return 0;
}

/*
* flush write buffer
*/
int pool_flush(POOL_CONNECTION *cp)
{
	if (fflush(cp->write_fd) != 0)
	{
		pool_error("pool_flush: fflush failed (%s)", strerror(errno));

		if (cp->isbackend)
		{
		    notice_backend_error();
		    exit(1);
		}
		else
		{
		    return -1;
		}
	}
	return 0;
}

/*
* combo of pool_write and pool_flush
*/
int pool_write_and_flush(POOL_CONNECTION *cp, void *buf, int len)
{
	if (pool_write(cp, buf, len))
		return -1;
	return pool_flush(cp);
}

/* 
 * read a string until EOF or NULL is encountered.
 * if line is not 0, read until new line is encountered.
*/
char *pool_read_string(POOL_CONNECTION *cp, int *len, int line)
{
	static char *buf = NULL;
	static int bufsz = 0;

	int readp;
	int readsize;
	int readlen;
	int strlength;
	int flag;

#ifdef DEBUG
	static char pbuf[READBUFSZ];
#endif

	*len = 0;
	readp = 0;

	/* initialize read buffer */
	if (bufsz == 0)
	{
		buf = malloc(READBUFSZ);
		if (buf == NULL)
		{
			pool_error("pool_read_string: malloc failed");
			return NULL;
		}
		bufsz = READBUFSZ;
		*buf = '\0';
	}

	/* any pending data? */
	if (cp->len)
	{
		if (line)
			strlength = mystrlinelen(cp->hp+cp->po, cp->len, &flag);
		else
			strlength = mystrlen(cp->hp+cp->po, cp->len, &flag);

		/* buffer is too small? */
		if ((strlength + 1) > bufsz)
		{
			bufsz = ((strlength+1)/READBUFSZ+1)*READBUFSZ;
			buf = realloc(buf, bufsz);
			if (buf == NULL)
			{
				pool_error("pool_read_string: realloc failed");
				return NULL;
			}
		}

		/* copy pending data to buffer */
		memmove(buf, cp->hp+cp->po, strlength);

#ifdef DEBUG
		snprintf(pbuf, strlength, buf);

		pool_debug("copy partial pending data :%s: po: %d remain: %d len: %d", pbuf, cp->po, cp->len, strlength);
#endif
		cp->po += strlength;
		cp->len -= strlength;
		*len = strlength;

		/* is the string null terminated? */
		if (cp->len <= 0 && !flag)
		{
			/* not null or line terminated */
			cp->len = 0;
			/* we need to read more since we have not encountered NULL yet */
			readsize = bufsz - strlength;
			readp = strlength;
		}
		else
		{
			return buf;
		}
	} else
	{
		readsize = bufsz;
	}


	for (;;)
	{
		pool_debug("pool_read_string: readsize: %d readp: %d", readsize, readp);

		readlen = read(cp->fd, buf+readp, readsize);
		if (readlen == -1)
		{
			pool_error("pool_read_string: read() failed. reason:%s", strerror(errno));

			if (cp->isbackend)
			{
			    notice_backend_error();
			    exit(1);
			}
			else
			{
			    return NULL;
			}
		}

		pool_debug("pool_read_string: buf:%s: readlen: %d", buf, readlen);

		/* check overrun */
		if (line)
			strlength = mystrlinelen(buf+readp, readlen, &flag);
		else
			strlength = mystrlen(buf+readp, readlen, &flag);

		pool_debug("strlength: %d", strlength);

		if (strlength < readlen)
		{
			/* save pending data */
			cp->len = readlen - strlength;

			/* pending buffer is enough? */
			if (cp->len > cp->bufsz)
			{
				/* too small, enlarge it. we assume that pending data size is never larger than bufsz */

				pool_debug("pool_read_string: enlarge pending buffer. old: %d new: %d", cp->bufsz, bufsz);

				cp->hp = realloc(cp->hp, bufsz);
				if (cp->hp == NULL)
				{
					pool_error("pool_read_string: realloc failed");
					return NULL;
				}
			}

			memmove(cp->hp, buf+readp+strlength, cp->len);
			cp->po = 0;
			*len += strlength;
			pool_debug("pending data saved: :%s: length: %d return data: :%s: length: %d", cp->hp, cp->len, buf, *len);

			return buf;
		}

		*len += readlen;

		pool_debug("pool_read_string: string: :%s:", buf+readp);
		pool_debug("pool_read_string: next string: :%s:", buf+readp+strlen(buf+readp)+1);
		
		/* enountered null or newline? */
		if (flag)
		{
			/* ok we have read all data */
			pool_debug("pool_read_string: final read %d string: :%s:", readlen, buf+readp);
			pool_debug("pool_read_string: total result %d string: :%s:", *len, buf);
			break;
		}

		pool_debug("pool_read_string: read more data %d string: :%s:", *len, buf+readp);

		readp = bufsz;
		bufsz += READBUFSZ;

		buf = realloc(buf, bufsz);
		if (buf == NULL)
		{
			pool_error("pool_read_string: realloc failed");
			return NULL;
		}
	}
	return buf;
}


/*
 * returns the byte length of str, including \0, no more than upper.
 * if encountered \0, flag is set to non 0.
 * example:
 *	mystrlen("abc", 2) returns 2
 *	mystrlen("abc", 3) returns 3
 *	mystrlen("abc", 4) returns 4
 *	mystrlen("abc", 5) returns 4
 */
static int mystrlen(char *str, int upper, int *flag)
{
	int len;

	*flag = 0;

	for (len = 0;len < upper; len++, str++)
	{
	    if (!*str)
	    {
			len++;
			*flag = 1;
			break;
	    }
	}
	return len;
}

/*
 * returns the byte length of str terminated by \n or \0 (including \n or \0), no more than upper.
 * if encountered \0 or \n, flag is set to non 0.
 * example:
 *	mystrlinelen("abc", 2) returns 2
 *	mystrlinelen("abc", 3) returns 3
 *	mystrlinelen("abc", 4) returns 4
 *	mystrlinelen("abc", 5) returns 4
 *	mystrlinelen("abcd\nefg", 4) returns 4
 *	mystrlinelen("abcd\nefg", 5) returns 5
 *	mystrlinelen("abcd\nefg", 5) returns 5
 *	mystrlinelen("abcd\nefg", 6) returns 5
 */
static int mystrlinelen(char *str, int upper, int *flag)
{
	int len;

	*flag = 0;

	for (len = 0;len < upper; len++, str++)
	{
	    if (!*str || *str == '\n')
	    {
			len++;
			*flag = 1;
			break;
	    }
	}
	return len;
}
