/* -*-pgsql-c-*- */
/*
 * $Header: /home/t-ishii/repository/pgpool/child.c,v 1.9 2003/07/20 14:44:37 t-ishii Exp $
 *
 * pgpool: a language independent connection pool server for PostgreSQL 
 * written by Tatsuo Ishii
 *
 * Copyright (c) 2003	Tatsuo Ishii
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of the
 * author not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. The author makes no representations about the
 * suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * child.c: child process main
 *
 */
#include "config.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/un.h>

#include <signal.h>

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#include "pool.h"

#ifdef NONE_BLOCK
static void set_nonblock(int fd);
#endif
static void unset_nonblock(int fd);

static POOL_CONNECTION *do_accept(int unix_fd, int inet_fd);
static StartupPacket *read_startup_packet(POOL_CONNECTION *cp);
static int send_startup_packet(POOL_CONNECTION *cp, StartupPacket *sp);
static void reset_backend(POOL_CONNECTION *backend);

/*
* child main loop
*/
void do_child(int unix_fd, int inet_fd)
{
	StartupPacket *sp;
	POOL_CONNECTION *frontend;
	POOL_CONNECTION_POOL *backend;

	pool_debug("I am %d", getpid());

	/* set up signal handlers */
	signal(SIGALRM, SIG_DFL);
	signal(SIGTERM, SIG_DFL);
	signal(SIGINT, SIG_DFL);
	signal(SIGCHLD, SIG_DFL);
	signal(SIGUSR1, SIG_DFL);

#ifdef NONE_BLOCK
	/* set listen fds to none block */
	set_nonblock(unix_fd);
	if (inet_fd)
	{
		set_nonblock(inet_fd);
	}
#endif

	/* initialize connection pool */
	if (pool_init_cp())
	{
		exit(1);
	}

	for (;;)
	{
		int connection_reuse = 1;

		/* perform accept() */
		frontend = do_accept(unix_fd, inet_fd);
		if (frontend == NULL)
		{
			/* accept() failed. return to the accept() loop */
			continue;
		}

		/* unset frontend fd tp none block */
		unset_nonblock(frontend->fd);

		/* read the startup packet */
		sp = read_startup_packet(frontend);
		if (sp == NULL)
		{
			/* failed to read the startup packet. return to the
			   accept() loop */
			pool_close(frontend);
			continue;
		}

		/*
		 * if there's no connection associated with user and database,
		 * we need to connect to the backend and send the startup packet.
		 */
		if ((backend = pool_get_cp(sp->user, sp->database)) == NULL)
		{
			connection_reuse = 0;

			/* connect to the backend and send the startup packect here */
			backend = pool_create_cp(sp->user, sp->database);
			if (backend == NULL)
			{
				static char *msg = "Sorry, too many clients already";

				pool_error("do_child: cannot not create backend connection");

				pool_write(frontend, "E", 1);
				pool_write_and_flush(frontend, msg, strlen(msg)+1);
				pool_close(frontend);
				continue;
			}

			/* mark this is a backend connection */
			backend->con->isbackend = 1;

			/* send startup packet */
			if (send_startup_packet(backend->con, sp) < 0)
			{
				pool_error("do_child: fails to send startup packet to the backend");
				pool_close(frontend);
				continue;
			}

			/*
			 * do authentication stuff
			 */
			if (pool_do_auth(frontend, backend))
			{
				pool_close(frontend);
				pool_discard_cp(sp->user, sp->database);
				continue;
			}
		}
		else
		{
			reset_backend(backend->con);

			/* send Auth OK and Ready for Query to the frontend */
			if (pool_send_auth_ok(frontend, backend->pid, backend->key) != POOL_CONTINUE)
			{
				pool_close(frontend);
				continue;
			}
			if (pool_write_and_flush(frontend, "Z", 1) < 0)
			{
				pool_close(frontend);
				continue;
			}
		}

		/* query process loop */
		for (;;)
		{
			int status;

			status = pool_process_query(frontend, backend->con, 
										connection_reuse);
			switch (status)
			{
				/* client exits */
				case POOL_END:
					/* do not cache connection to template0, template1, regression */
					if (!strcmp(sp->database, "template0") || !strcmp(sp->database, "template1") ||
						!strcmp(sp->database, "regression"))
						pool_discard_cp(sp->user, sp->database);

					pool_close(frontend);

					pool_connection_pool_timer(backend);
					break;
				
				/* error occured. discard backend connection pool
                   and disconnect connection to the frontend */
				case POOL_ERROR:
					pool_discard_cp(sp->user, sp->database);
					pool_close(frontend);
					break;

				/* fatal error occured. just exit myself... */
				case POOL_FATAL:
					exit(1);
					break;

				/* not implemented yet */
				case POOL_IDLE:
					do_accept(unix_fd, inet_fd);
					pool_debug("accept while idle");
					break;
			}

			if (status != POOL_CONTINUE)
				break;
		}

	}
	exit(0);
}

/* -------------------------------------------------------------------
 * private functions
 * -------------------------------------------------------------------
 */

#ifdef NONE_BLOCK
/*
 * set non-block flag
 */
static void set_nonblock(int fd)
{
	int var;

	/* set fd to none blocking */
	var = fcntl(fd, F_GETFL, 0);
	if (var == -1)
	{
		pool_error("fcntl failed. %s", strerror(errno));
		exit(1);
	}
	if (fcntl(fd, F_SETFL, var | O_NONBLOCK) == -1)
	{
		pool_error("fcntl failed. %s", strerror(errno));
		exit(1);
	}
}
#endif

/*
 * unset non-block flag
 */
static void unset_nonblock(int fd)
{
	int var;

	/* set fd to none blocking */
	var = fcntl(fd, F_GETFL, 0);
	if (var == -1)
	{
		pool_error("fcntl failed. %s", strerror(errno));
		exit(1);
	}
	if (fcntl(fd, F_SETFL, var & ~O_NONBLOCK) == -1)
	{
		pool_error("fcntl failed. %s", strerror(errno));
		exit(1);
	}
}

/*
* perform accept() and returns new fd
*/
static POOL_CONNECTION *do_accept(int unix_fd, int inet_fd)
{
    fd_set	readmask;
    fd_set	exceptmask;
    int fds;

	struct sockaddr addr;
	socklen_t addrlen;
	int fd = 0;
	int afd;
	POOL_CONNECTION *cp;

	FD_ZERO(&readmask);
	FD_ZERO(&exceptmask);
	FD_SET(unix_fd, &readmask);
	if (inet_fd)
		FD_SET(inet_fd, &readmask);
	FD_SET(unix_fd, &exceptmask);
	if (inet_fd)
		FD_SET(inet_fd, &exceptmask);

	fds = select(Max(unix_fd, inet_fd)+1,
				 &readmask, NULL, &exceptmask, NULL);

	if (fds == -1)
	{
		if (errno == EINTR)
			return NULL;

		pool_error("select() failed. reason %s", strerror(errno));
		return NULL;
	}

	if (fds == 0)
		return NULL;

	if (FD_ISSET(unix_fd, &readmask))
	{
		fd = unix_fd;
	}

	if (FD_ISSET(inet_fd, &readmask))
	{
		fd = inet_fd;
	}

	/*
	 * Note that some SysV systems do not work here. For those
	 * systems, we need some locking mechanism for the fd.
	 */
	addrlen = sizeof(addr);
	afd = accept(fd, &addr, &addrlen);
	if (afd < 0)
	{
		pool_error("accept() failed. reason: %s", strerror(errno));
		return NULL;
	}
	pool_debug("I am %d accept fd %d", getpid(), afd);

	if ((cp = pool_open(afd)) == NULL)
	{
		close(afd);
		return NULL;
	}
	return cp;
}

/*
* read startup packet
*/
static StartupPacket *read_startup_packet(POOL_CONNECTION *cp)
{
	static StartupPacket sp;
	int len;

	memset((char *)&sp, 0, sizeof(sp));

	if (pool_read(cp, &len, sizeof(len)))
	{
		return NULL;
	}
	len = ntohl(len);
	len -= sizeof(len);

	if (pool_read(cp, &sp, len))
	{
		return NULL;
	}

	pool_debug("Protocol Version: %08x",ntohl(sp.protoVersion));
	pool_debug("Protocol Major: %d Minor:%d",
		   ntohl(sp.protoVersion)>>16,
			   ntohl(sp.protoVersion) & 0x0000ffff);
	pool_debug("database: %s",sp.database);
	pool_debug("user: %s",sp.user);

	return &sp;
}

/*
* send startup packet
*/
static int send_startup_packet(POOL_CONNECTION *cp, StartupPacket *sp)
{
	int len;

	len = htonl(sizeof(len) + sizeof(StartupPacket));

	pool_write(cp, &len, sizeof(len));
	return pool_write_and_flush(cp, sp, sizeof(StartupPacket));
}

/*
 * reset backend status
 */
static void reset_backend(POOL_CONNECTION *backend)
{
	char kind;
	int len;

#ifdef NO_RESET_ALL
	static char *queries[] = {"ABORT"};
#else
	static char *queries[] = {"ABORT", "RESET ALL"};
#endif

	int i;

	for (i=0;i<sizeof(queries)/sizeof(char *);i++)
	{
		pool_write(backend, "Q", 1);
		pool_write_and_flush(backend, queries[i], strlen(queries[i])+1);
		pool_read(backend, &kind, 1);

		switch(kind)
		{
			case 'C':
			case 'E':
			case 'N':
				/* read and discard notice or error message */
				pool_read_string(backend, &len, 0);

				if (kind != 'C')
				{
					/* read and discard complete command response */
					pool_read(backend, &kind, 1);
					pool_read_string(backend, &len, 0);
				}

				/* read and discard ready for query message */
				pool_read(backend, &kind, 1);
				break;

			default:
				pool_error("reset_backend: Unknown response");
				exit(1);
		}
	}
}
