/*
 *   IPC package for CygWin32
 *
 *   Copyright (C) 1997 Philippe CHAPUY
 *   Copyright (C) 1998 Ludovic LANGE
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   HISTORY:
 *   --------
 *
 *   13/05/1998 : Version 1.00 released
 *                First public release
 *                adress any comments to llange@capgemini.fr
 *
 */



/*
 * sem.c
 * Copyright (C) 1997 Philippe Chapuy
 *
 */

#define EXTERN
#include "IpcNtExt.h"
#include <unistd.h>
#include "strace.h"

static int newary (key_t, int, int);
static int findkey (key_t key);
static void freeary (int id);

static struct semid_ds   **semary           ;
static int                used_sems      = 0;
static int                used_semids    = 0;
static int                max_semid      = 0;

static struct semid_ds    *ptrsem           ;		/*PCPC*/
static CYGWIN32_IPCNT_SEMSTR *shareadrsem         ;		/*PCPC*/

static unsigned short     sem_seq        = 0;
static int		  GFirstSem	 = 0;		/*PCPC*/
static int		  GFdSem	    ;		/*PCPC*/

/************************************************************************/
/* Demande d'acces a la zone partagee de gestion des semaphores		*/
/************************************************************************/

static int IsGSemSemExist()
{
    GSemSem = OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE, CYGWIN32_IPCNT_SEMSEM) ;
    if( GSemSem != NULL )
    {
	return (1) ;
    } else {
	return (0) ;
    }
}

static int sem_connect(void)
{
    int LRet ;

    if( GFirstSem == 0 )
    {
	if( IsGSemSemExist() )
	{
	    GFirstSem = 1 ;	    
	    GFdSem  = open(CYGWIN32_IPCNT_FILESEM, O_RDWR) ;
	    shareadrsem = (CYGWIN32_IPCNT_SEMSTR *) mmap(
		    0, sizeof(CYGWIN32_IPCNT_SEMSTR), PROT_WRITE|PROT_READ,
		    MAP_SHARED, GFdSem, 0) ;
		    
	    if( shareadrsem == (CYGWIN32_IPCNT_SEMSTR *) -1 )
	    {
		close (GFdSem) ;
		return (0) ;
    	    }
	    ptrsem = &(shareadrsem->sem[0]) ;
	    semary = &(shareadrsem->semary[0]);
	} else {
	    /* RAF unmap */
	    return(0) ;
	}
    }


    LRet = WaitForSingleObject(GSemSem, INFINITE) ;


    used_sems = shareadrsem->used_sems ;
    used_semids = shareadrsem->used_semids ;
    max_semid = shareadrsem->max_semid ;
    sem_seq = shareadrsem->sem_seq ;


    return(1) ;
}

/************************************************************************/
/*PCPC									*/
/************************************************************************/
static void sem_deconnect()
{
    long LPrevious ;


    shareadrsem->used_sems = used_sems ;
    shareadrsem->used_semids = used_semids ;
    shareadrsem->max_semid = max_semid ;
    shareadrsem->sem_seq = sem_seq ;

    ReleaseSemaphore(GSemSem, (LONG) 1, &LPrevious) ;


    if( LPrevious != 0 )
    {
	printf ("Error semaphore semaphore not equal 0\n") ;
    }
}

static int findkey (key_t key)
{
	int id;
	struct semid_ds *sma;

	for (id = 0; id <= max_semid; id++) {

		if (semary[id] == IPC_UNUSED)
		{
			continue;
		}
		sma = (struct semid_ds *) &(ptrsem[id]) ;
		if (key == sma->sem_perm.key)
		{
			CYGWIN32_IPCNT_RETURN (id);
		}

	}
	CYGWIN32_IPCNT_RETURN (-1);
}

static void FDestroySem(int Index, int Id)
{
    HANDLE LHandle ;
    char   LBuff[256] ;

    itoa(100*Id+Index, LBuff) ;

    LHandle = OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE, LBuff) ;

    if( LHandle != NULL )
    {
	CloseHandle ( LHandle ) ;
    }

    return ;
}

/************************************************************************/
static HANDLE FCreateSem(int Index, int Id)
{
    HANDLE LHandle ;
    char   LBuff[256] ;


    itoa(100*Id+Index, LBuff) ;

    LHandle = OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE, LBuff) ;
    if( LHandle != NULL )
    {
	CloseHandle ( LHandle ) ;
    }
    LHandle = CreateSemaphore(NULL, 0, 0x7FFFFFFF, LBuff) ;
    if( LHandle == NULL )
    {
	printf( "Creation de Semaphore \"Sem\" impossible\n" ) ;
	goto endko ;
    }
    return (LHandle) ;
endko:;
    return (NULL) ;
}

static int newary (key_t key, int nsems, int semflg)
{
	int id;
	struct semid_ds *sma;
	struct ipc_perm *ipcp;
	int size;
	int LIndex ;
	HANDLE LHandle ;

debug_printf("newary : key=%p nsems=%X semflg=%X\n",key,nsems,semflg);
	if (!nsems)
	{
debug_printf("newary : return -EINVAL\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-EINVAL) ;
	}
	if (used_sems + nsems > SEMMNS)
	{
debug_printf("newary : return -ENOSPC\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-ENOSPC);
	}


	for (id = 0; id < SEMMNI; id++)
		if (!semary[id] || semary[id] == (struct semid_ds *) IPC_UNUSED) {
			semary[id] = (struct semid_ds *) IPC_NOID;
			goto found;
		}

	/****************************************************************/
	/*PCPC On doit liberer le semaphore de gestion			*/
	/****************************************************************/
debug_printf("newary : return -ENOSPC\n");
	CYGWIN32_IPCNT_RETURN_DECONNECT (-ENOSPC);

found:

	size = sizeof (struct semid_ds) ;
	sma = (struct semid_ds *) &(ptrsem[id]) ;

	memset (sma, 0, size);
	ipcp = &sma->sem_perm;
	ipcp->mode = semflg ;
	ipcp->key = key;
	ipcp->seq = sem_seq;
	sma->sem_nsems = nsems;

	/****************************************************************/
	if (id > max_semid)
	{
		max_semid = id;
	}
	used_semids++;
	semary[id] = (struct semid_ds *)
			((char *) sma - (int) shareadrsem);

	/****************************************************************/
	/* On cree nsems semaphores NT					*/
	/****************************************************************/
	for (LIndex=0; LIndex < nsems; LIndex++)
	{
	    shareadrsem->current_nb[id].current_nb[LIndex] = 0 ;
	    LHandle = FCreateSem(LIndex, id)  ;
	    if( LHandle == NULL )
	    {
		semary[id] = (struct semid_ds *) IPC_UNUSED;
		used_sems -= nsems;
		/********************************************************/
		/*PCPC On doit liberer le semaphore de gestion		*/
		/********************************************************/
debug_printf("newary : return -ENOMEM\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-ENOMEM);
	    }
	}

	/****************************************************************/
	/*PCPC On doit liberer le semaphore de gestion			*/
	/****************************************************************/
debug_printf("newary : return %p\n",(sma->sem_perm.seq) * SEMMNI + id);
	CYGWIN32_IPCNT_RETURN_DECONNECT_WAIT1(
		(sma->sem_perm.seq) * SEMMNI + id );
}

int semget (key_t key, int nsems, int semflg)
{
	int id;
	struct semid_ds *sma;

debug_printf("semget : key=%p nsems=%X semflg=%X\n",key,nsems,semflg);

	if (nsems < 0 || nsems > SEMMSL)
	{
debug_printf("semget : return -EINVAL\n");
	    CYGWIN32_IPCNT_RETURN (-EINVAL);
	}

	if (sem_connect() == 0)
	{
debug_printf("semget : return -EACCES\n");
	    CYGWIN32_IPCNT_RETURN (-EACCES) ;
	}

	if (key == IPC_PRIVATE)
	{
debug_printf("semget -> newary\n");
		return newary(key, nsems, semflg);
	}
	
	if ((id = findkey (key)) == -1)
	{  /* key not used */
		if (!(semflg & IPC_CREAT))
		{
debug_printf("semget : return -ENOENT\n");
			CYGWIN32_IPCNT_RETURN_DECONNECT (-ENOENT) ;
		}
debug_printf("semget -> newary\n");
		return newary(key, nsems, semflg);
	}
	if (semflg & IPC_CREAT && semflg & IPC_EXCL)
	{
debug_printf("semget : return -EEXIST\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-EEXIST);
	}
	sma = (struct semid_ds *)
			((char *) semary[id] + (int) semary);
	if (nsems > sma->sem_nsems)
	{
debug_printf("semget : return -EINVAL\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-EINVAL);
	}

debug_printf("semget : return %p\n",(sma->sem_perm.seq) * SEMMNI + id);
	CYGWIN32_IPCNT_RETURN_DECONNECT ( (sma->sem_perm.seq) * SEMMNI + id) ;
}

static int do_semop (struct semid_ds * sma, struct sembuf * sops, int nsops,
		     int id)
{
	int i, LRet;
	char LStrSem[100] ;
	HANDLE LHandle ;
	long LVal ;

debug_printf("do_semop : sma=%p, sops=%p, nsops=%X, id=%X\n",sma,sops,nsops,id);

	for (i = 0; i < nsops; i++) {
		struct sembuf * sop = &sops[i];

		/********************************************************/
		/* PCPC							*/
		/********************************************************/
 		itoa(100*id+sop->sem_num, LStrSem) ;
		LHandle = OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE, LStrSem);
		if( sop->sem_op >= 0 )
		{
		    if (sem_connect() == 0)
		    {
debug_printf("do_semop : return -EACCES\n");
			CYGWIN32_IPCNT_RETURN (-EACCES) ;
		    }
		    ReleaseSemaphore(LHandle, sop->sem_op, &LVal) ;
	    	    shareadrsem->current_nb[id].current_nb[sop->sem_num] +=
					sop->sem_op ;
		    sem_deconnect() ;
		} else {
		    if( sop->sem_flg == IPC_NOWAIT )
		    {
			LRet = WaitForSingleObject(LHandle, 0) ;
			if( LRet == WAIT_TIMEOUT )
			{
debug_printf("do_semop : return -EAGAIN\n");
			    CYGWIN32_IPCNT_RETURN (-EAGAIN) ;
			}
			if (sem_connect() == 0)
			{
debug_printf("do_semop : return -EACCES\n");
			    CYGWIN32_IPCNT_RETURN (-EACCES) ;
			}
	    		shareadrsem->current_nb[id].current_nb[sop->sem_num] -= 1 ;
			sem_deconnect() ;
		    } else {
			LRet = WaitForSingleObject(LHandle, INFINITE) ;
			if (sem_connect() == 0)
			{
debug_printf("do_semop : return -EACCES\n");
			    CYGWIN32_IPCNT_RETURN (-EACCES) ;
			}
			    shareadrsem->current_nb[id].current_nb[sop->sem_num] -= 1 ;
			    sem_deconnect() ;
		    }
		}
		/********************************************************/
		/* FIN PCPC						*/
		/********************************************************/
	}
debug_printf("do_semop : return 0\n");
   CYGWIN32_IPCNT_RETURN (0) ;
}

/* Free a semaphore set. */
static void freeary (int id)
{
	struct semid_ds *sma ;
	int LIndex ;

	sma = (struct semid_ds *)
			((char *) semary[id] + (int) semary);

	for (LIndex=0 ; LIndex < sma->sem_nsems; LIndex++)
	{
         FDestroySem(LIndex, id) ;
	}

	/* Invalidate this semaphore set */
	sma->sem_perm.seq++;
	sem_seq = (sem_seq+1) % ((unsigned)(1<<31)/SEMMNI); /* increment, but avoid overflow */
	used_sems -= sma->sem_nsems;
	if (id == max_semid)
	{
		while (max_semid && (semary[--max_semid] == IPC_UNUSED));
	}
	/*On laisse le pointeur */
	shareadrsem->state[id] = 1 ;
	used_semids--;
}

int semctl (int semid, int semnum, int cmd, union semun arg)
{
	struct semid_ds *buf = NULL;
	struct semid_ds tbuf;
	int i, id;
	struct semid_ds *sma;
	struct ipc_perm *ipcp;
	unsigned int nsems;
	ushort *array = NULL;
	char LBuff[100] ;
	HANDLE LHandle ;
	long LPrevious ;
	int LIndex;

debug_printf("semctl : semid=%X semnum=%X cmd=0x%02X arg=%p\n",semid,semnum,cmd,arg);
	if (semid < 0 || semnum < 0 || cmd < 0)
	{
debug_printf("semctl : return -EINVAL\n");
		CYGWIN32_IPCNT_RETURN (-EINVAL);
	}

	id = (unsigned int) semid % SEMMNI;

	if (sem_connect() == 0)
	{
debug_printf("semctl : return -EACCES\n");
	 CYGWIN32_IPCNT_RETURN (-EACCES) ;
	}

	switch (cmd) {
	case IPC_INFO:
	case SEM_INFO:
	{

		struct seminfo seminfo, *tmp = arg.__buf;


		seminfo.semmni = SEMMNI;
		seminfo.semmns = SEMMNS;
		seminfo.semmsl = SEMMSL;
		seminfo.semopm = SEMOPM;
		seminfo.semvmx = SEMVMX;
		seminfo.semmnu = SEMMNU;
		seminfo.semmap = SEMMAP;
		seminfo.semume = SEMUME;
		seminfo.semusz = SEMUSZ;
		seminfo.semaem = SEMAEM;
		if (cmd == SEM_INFO) {
			seminfo.semusz = used_semids;
			seminfo.semaem = used_sems;
		}
		memcpy (tmp, &seminfo, sizeof(struct seminfo));

debug_printf("semctl : xxx_INFO : return %X\n",max_semid);
		CYGWIN32_IPCNT_RETURN_DECONNECT (max_semid) ;
	}

	case SEM_STAT:

		buf = arg.buf;
		if (semid > max_semid)
		{
debug_printf("semctl : SEM_STAT : return -EINVAL\n");
			CYGWIN32_IPCNT_RETURN_DECONNECT (-EINVAL);
		}
		if (semary[id] == IPC_UNUSED || semary[id] == IPC_NOID)
		{
debug_printf("semctl : SEM_STAT : return -EINVAL\n");
			CYGWIN32_IPCNT_RETURN_DECONNECT (-EINVAL);
		}
		sma = (struct semid_ds *)
			((char *) semary[id] + (int) semary);

		id = (unsigned int) sma->sem_perm.seq * SEMMNI + semid;
		memcpy ((&tbuf.sem_perm), &(sma->sem_perm),
				sizeof(struct ipc_perm) ) ;
		tbuf.sem_otime  = sma->sem_otime;
		tbuf.sem_ctime  = sma->sem_ctime;
		tbuf.sem_nsems  = sma->sem_nsems;
		memcpy (buf, &tbuf, sizeof(struct semid_ds));

debug_printf("semctl : SEM_STAT : return %X\n",id);
		CYGWIN32_IPCNT_RETURN_DECONNECT (id);
	}


	if (semary[id] == IPC_UNUSED || semary[id] == IPC_NOID)
	{

debug_printf("semctl : return -EINVAL\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-EINVAL);
	}

	sma = (struct semid_ds *)
			((char *) semary[id] + (int) semary);

	ipcp = &sma->sem_perm;
	nsems = sma->sem_nsems;
	if (sma->sem_perm.seq != (unsigned int) semid / SEMMNI)
	{

debug_printf("semctl : return -EIDRM\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-EIDRM);
	}

	switch (cmd) {
	case GETPID:
	case GETNCNT:
	case GETZCNT:
		if ((unsigned int)semnum >= nsems)
		{
debug_printf("semctl : GETPID : return -EINVAL\n");
			CYGWIN32_IPCNT_RETURN_DECONNECT (-EINVAL);
		}
debug_printf("semctl : GETPID : return -ENOSYS\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-ENOSYS);
		break;
	case SETVAL:
		if ((unsigned int)semnum >= nsems)
		{
debug_printf("semctl : SETVAL : return -EINVAL\n");
			CYGWIN32_IPCNT_RETURN_DECONNECT (-EINVAL);
		}
		break;
	}

	switch (cmd) {

	case GETVAL:
		if ((unsigned int)semnum > nsems)
		{
debug_printf("semctl : GETVAL : return -EINVAL\n");
			CYGWIN32_IPCNT_RETURN_DECONNECT (-EINVAL);
		}
debug_printf("semctl : GETVAL : return %X\n",shareadrsem->current_nb[id].current_nb[semnum]);
		CYGWIN32_IPCNT_RETURN_DECONNECT (shareadrsem->current_nb[id].current_nb[semnum]);

	case SETVAL:
		itoa(100*id+semnum, LBuff) ;

		LHandle =
		OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE, LBuff) ;
		if( LHandle != NULL )
		{
		    if( arg.val > shareadrsem->current_nb[id].current_nb[semnum] )
		    {
			ReleaseSemaphore(LHandle,
			arg.val-shareadrsem->current_nb[id].current_nb[semnum],
			&LPrevious) ;
		    }
		    else if (arg.val <
		             shareadrsem->current_nb[id].current_nb[semnum] )
		    {
			for( LIndex = arg.val;
			LIndex < shareadrsem->current_nb[id].current_nb[semnum];
			LIndex++ )
			{
			    WaitForSingleObject(LHandle, 0) ;
			}
		    }
            	    shareadrsem->current_nb[id].current_nb[semnum] = arg.val ;
		}
debug_printf("semctl : SETVAL : return 0\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (0);
		break;

	case IPC_RMID:
		freeary (id);
debug_printf("semctl : IPC_RMID : return 0\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (0);
		break;

	case SETALL: /* arg is a pointer to an array of ushort */
		array = arg.array ;
		for (i=0; i<semnum; i++)
		{
 		    itoa(100*id+i, LBuff) ;

		    LHandle =
			OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE, LBuff) ;
		    if( LHandle != NULL )
		    {
			ReleaseSemaphore(LHandle, array[i], &LPrevious) ;
	        	shareadrsem->current_nb[id].current_nb[i] +=
			    array[i] ;
		    }
		}
debug_printf("semctl : SETALL : return 0\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (0);
		break;
	case IPC_STAT:
debug_printf("semctl : IPC_STAT : return -ENOSYS\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-ENOSYS);
		break;
	case IPC_SET:
debug_printf("semctl : IPC_SET : return -EINVAL\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-ENOSYS);
		break;
	}

	if (semary[id] == IPC_UNUSED || semary[id] == IPC_NOID)
	{
debug_printf("semctl : return -EIDRM\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-EIDRM);
	}
	if (sma->sem_perm.seq != (unsigned int) semid / SEMMNI)
	{
debug_printf("semctl : return -EIDRM\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-EIDRM);
	}

	switch (cmd) {
	case GETALL:
debug_printf("semctl : GETALL return -1\n");
			CYGWIN32_IPCNT_RETURN_DECONNECT (-1);
		break;
	case SETALL:
debug_printf("semctl : SETALL return -1\n");
			CYGWIN32_IPCNT_RETURN_DECONNECT (-1);
		break;
	default:
debug_printf("semctl : return -EINVAL\n");
		CYGWIN32_IPCNT_RETURN_DECONNECT (-EINVAL);
	}
debug_printf("semctl : return 0\n");
	CYGWIN32_IPCNT_RETURN_DECONNECT (0);
}

int semop (int semid, struct sembuf *tsops, unsigned nsops)
{
	int id, error;
	unsigned int i;
	struct semid_ds *sma;
	struct sembuf sops[SEMOPM], *sop;

debug_printf("semop : semid=%X tsops=%p nsops=%X\n",semid,tsops,nsops);
	if (nsops < 1 || semid < 0)
	{
debug_printf("semop : return -EINVAL\n");
		CYGWIN32_IPCNT_RETURN (-EINVAL);
	}
	if (nsops > SEMOPM)
	{
debug_printf("semop : return -E2BIG\n");
		CYGWIN32_IPCNT_RETURN (-E2BIG);
	}
	if (!tsops)
	{
debug_printf("semop : return -EFAULT\n");
		CYGWIN32_IPCNT_RETURN (-EFAULT);
	}

	memcpy (sops, tsops, nsops * sizeof(struct sembuf));

	if (sem_connect() == 0)
	{
debug_printf("semop : return -EACCES\n");
	 CYGWIN32_IPCNT_RETURN (-EACCES) ;
	}
	sem_deconnect() ;

	id = (unsigned int) semid % SEMMNI;
	if (semary[id] == IPC_UNUSED || semary[id] == IPC_NOID)
	{
debug_printf("semop : return -EINVAL\n");
		CYGWIN32_IPCNT_RETURN (-EINVAL);
	}
	sma = (struct semid_ds *)
			((char *) semary[id] + (int) semary);
	if (sma->sem_perm.seq != (unsigned int) semid / SEMMNI)
	{
debug_printf("semop : return -EIDRM\n");
		CYGWIN32_IPCNT_RETURN (-EIDRM);
	}
	for (i = 0; i < nsops; i++) {
		sop = &sops[i];
		if (sop->sem_num >= sma->sem_nsems)
		{
debug_printf("semop : return -EFBIG\n");
			CYGWIN32_IPCNT_RETURN (-EFBIG);
		}
	}

	error = do_semop(sma, sops, nsops, id);
debug_printf("semop : return %d\n",error);
	return (error) ;
}
