/* ============================================================
* File  : streamsniff.c
* Author: Eric Giesselbach <ericgies@kabelfoon.nl>
* Date  : 2008-08-24
* Description : sniffs network for stream url's
*
* Copyright 2005-2008 by Eric Giesselbach

* This program is free software; you can redistribute it
* and/or modify it under the terms of the GNU General
* Public License as published bythe Free Software Foundation;
* either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* ============================================================ */

/*
Todo:
	OK if a stream url is detected, trace back (http) packets between
		same server and client looking for (more stable) playlist url
	OK cache all http packets. Find stream url in non-GET http packet
		data and trace back same connection (host, port) for GET.
	OK create separate protocol handlers with proper interface.
	OK Windows port
	KINDOF BSD port

	2008 aug 24:	detects x-shockwave flash
						detects HTTP 200 OK payload starting with FLV

*/ 

#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <time.h>
#include <sys/types.h>

#define VERSION "0.05"

#ifdef linux
#include <sys/socket.h>
#include <linux/if_ether.h>
#include <linux/in.h>
#endif

#ifdef WIN32
#include <winsock2.h>
#define SIO_RCVALL _WSAIOW(IOC_VENDOR,1)
#endif

#include "protocols/extern.c"
#include "protocols/mms.c"
#include "protocols/icy.c"
#include "protocols/rtsp.c"
#include "protocols/http.c"

#ifdef linux 
int sock; // raw socket
#define socket_cleanup(s) (close((s)))
#endif 

#ifdef WIN32 
SOCKET sock;
#define socket_cleanup(s) (closesocket((s)))
#endif
		
char *testpoint;

struct proto_ip
{
char sep2[9];  // ignore bit 4-8 20 byte, no options assumed
char proto[1]; // 0x06 TCP, 0x11 UDP
char sep3[2];
struct in_addr saddr;
struct in_addr daddr;
} __attribute__ ((packed));


struct proto_udp
{
char sep[8];
} __attribute__ ((packed));


struct proto_tcp
{
unsigned short sport; // use ntohs()
unsigned short dport;
char sep1[8];
unsigned char offset[1];  // tcp header size = offset >> 4 * 4;
char sep2[7];
// + options, padding
} __attribute__ ((packed));


struct proto_http
{
	char command[5];
} __attribute__ ((packed));


void wipeall(int nogo)
{
socket_cleanup(sock);
exit(1);
}

enum parserstate
{
	searching = 0,
	streamfound,
	nostreamfound
};

#define eq(s1, s2) ( (strcmp((s1),(s2)) == 0) )

int main(int argc, char *argv[])
{
		enum parserstate state;
				
		//unsigned int test;
		
		struct connection _connection;
		struct connection *m_connection = &_connection;
	
		struct streamurl  *url;
		struct proto_ip   *prot_ip;
		//struct proto_udp  *prot_udp;
		struct proto_tcp  *prot_tcp;
		struct proto_http *prot_http;
		
		char *buffer; //, *curchar, *endchar;
		char buf[BUFLEN]; //, host[200];
		int readcnt, i;
		size_t cursize;
		
		char *localip  = NULL;
		int use2 = 0;  // dump to stdout AND stderr
		int help = 0;
		

		i = 1;
		while (argv[i])
		{
			if ( eq(argv[i], "-b") ) localip  = argv[i+1];
			if ( eq(argv[i], "--stderr") ) use2 = 1;
			if ( eq(argv[i], "-h") || eq(argv[i], "--help") ) help = 1;
			i++;
		}

		if ( help )
		{
			printf("streamsniff v%s\n", VERSION);
			printf("sniff for stream url's\n\nusage: \n");
			printf(  "  streamsniff \n");
			printf(  "  streamsniff -d \"local ip adress\"\n");
			printf(  "  streamsniff -d \"local ip adress\" --stderr > dump.txt\n");
			printf("\n  -d            local ip address to bind to, required on windows\n");

			printf(  "  --stderr      dump to stdout AND stderr (use when stdout is \n");
			printf(  "                redirected to file)\n\n");
			exit(1);
		}


#ifdef WIN32
		WSADATA wsd;
		DWORD  dwBytesRet;
		DWORD BytesRecv;
		struct sockaddr_in SockAddr;
		int I = 1;
	
		if (!localip)
		{
			printf("streamsniff v%s\n\nrequired option on windows: \n", VERSION);
			printf("  streamsniff -b \"local ip address\"\n");
			exit(0);
		}

		if ( ( WSAStartup(MAKEWORD(2,1),&wsd)) != 0 )
		{
				printf("Unable to initialise WinSock\n");
				printf("error %i\n", WSAGetLastError());
				exit(1);
		}

		if ( (sock = socket(PF_INET, SOCK_RAW, IPPROTO_IP)) == INVALID_SOCKET )
		{
				printf("error %i\n", WSAGetLastError());
				printf("cannot open raw socket (you root?)\n");
				exit(-1);
		}

		memset(&SockAddr, 0, sizeof(struct sockaddr));
		SockAddr.sin_addr.s_addr = inet_addr(localip);
		SockAddr.sin_family = AF_INET;
		SockAddr.sin_port = 0;

		if (bind(sock, (struct sockaddr *)&SockAddr, sizeof(struct sockaddr)) == SOCKET_ERROR)
		{
				printf("bind(%s) failed.\n", localip);
				exit(-1);
		}

		if (WSAIoctl(sock, SIO_RCVALL, &I, sizeof(I), NULL, 0, &BytesRecv, NULL, NULL) == SOCKET_ERROR)
		{
				printf("WSAIoctl() error %i\n", WSAGetLastError());
				//exit(-1);
		}
#endif

#ifdef linux      
						
		if ( (sock = socket(PF_PACKET, SOCK_DGRAM, htons(ETH_P_ALL))) < 0 )
		{
			fprintf (stderr, "error opening raw socket: %s\n", strerror (errno));
			printf("cannot open raw socket (you root?)\n");
			exit(-1);
		}
		
#endif

		buffer = buf;
		buffer[BUFLEN - 1] = 0;
		
		int headerindex = 0;
		
		for (i=0; i < 100; i++) {
		headers[i].header[0] = 0;
		headers[i].headerlen = 0;
		}
		
		signal(SIGINT,  wipeall);
		
		printf("looking for streams...\n\n");
		
		while (1)
		{
			state = searching;
			url = 0;
			
			readcnt = recvfrom(sock, buffer, BUFLEN - 1, 0, NULL, NULL);
			
			cursize = 0;
			//printf("%i bytes read\n", readcnt);
			
	// check min packet size
			if (readcnt < (cursize + sizeof(struct proto_ip)) )
			{
			printf("incomplete ip header (reported size: %i bytes), aborting\n", readcnt);
			socket_cleanup(sock);
			exit(-1);
			}
			
	// make sure char data terminates
			buffer[readcnt] = 0;

	// map eth/ip structure (both assumed)
			prot_ip = (struct proto_ip*)buffer;
			cursize += sizeof(struct proto_ip);  // should allow for options...
			
	// map tcp structure (prot_ip->proto == 0x06 assumed)
			if ( readcnt < cursize + sizeof(struct proto_tcp) )
				state = nostreamfound;
			else
			{
				prot_tcp = (struct proto_tcp*)(buffer + cursize);
				cursize += (prot_tcp->offset[0] >> 4) * 4; // tcp header size
			}
			
	// register connection endpoints
			if (state == searching)
			{
				
				strcpy(m_connection->saddr, (char*)inet_ntoa(prot_ip->saddr));
				strcpy(m_connection->daddr, (char*)inet_ntoa(prot_ip->daddr));
				
				m_connection->sport = prot_tcp->sport;
				m_connection->dport = prot_tcp->dport;
			}
			
	// check http
			if ( state == searching )
			{
				if (readcnt < cursize + sizeof(struct proto_http) )
					state = nostreamfound;
				else
				{
					prot_http = (struct proto_http*)(buffer + cursize);
				
					// store http headers
					if ( strncmp(prot_http->command, "GET " , 4) == 0 ||
						strncmp(prot_http->command, "HTTP/", 5) == 0   )
					{
						if (++headerindex > 99) headerindex = 0;
						
						strcpy(headers[headerindex].conn.saddr, (char*)inet_ntoa(prot_ip->saddr));
						strcpy(headers[headerindex].conn.daddr, (char*)inet_ntoa(prot_ip->daddr));
					
						headers[headerindex].conn.sport = prot_tcp->sport;
						headers[headerindex].conn.dport = prot_tcp->dport;
					
						headers[headerindex].timestamp = time(0);
						strncpy(headers[headerindex].header, prot_http->command, readcnt - cursize);
						headers[headerindex].headerlen = readcnt - cursize;
					}
				}
			}

	// note: headerindex increments on http packets only
		if (state == searching)
		{
			
			url = check_rtsp(buffer + cursize, readcnt - cursize, headerindex);
			
			if ( url->type == none )
				url = check_icy(buffer + cursize, readcnt - cursize, headerindex, m_connection);
					
			if ( url->type == none )
				url = check_mms(buffer + cursize, readcnt - cursize, headerindex, m_connection);
				
			if ( url->type == none )
				url = check_http(buffer + cursize, readcnt - cursize, headerindex, m_connection);
		}
		
			
			
			if ( url && url->type != none)
			{
				printf ("%s: %s%s%s\n", names[url->type], url->prot, url->host, url->path);
				if (url->plst[0] != 0) printf ("    : %s\n", url->plst);
				printf("\n");
				
				if (use2)
				{
				fprintf(stderr, "%s: %s%s%s\n", names[url->type], url->prot, url->host, url->path);
				if (url->plst[0] != 0) fprintf(stderr, "    : %s\n", url->plst);
				printf("\n");
				}

				url->type = none;
			}
		
		}
				
} 
