/**************************************************************************
     zlib.cpp                                                        v.13
     Free software licensed under GNU General Public License v.2
     source:  http://kornelix.squarespace.com/utilities
***************************************************************************/

#include "zlib.h"

/**************************************************************************
      Linux system utilities
***************************************************************************/

//  get string associated with error number
//  if zero, use last system error = global variable "errno"

char * syserrText(int err)
{
   static char    *ptext, buff[200];
   if (err == 0) err = errno;
   ptext = strerror_r(err,buff,200);
   return ptext;
}


//  Fatal exit with error message to stdout.  
//  Works like printf.

void appcrash(char *pMess, ... )
{
   va_list  arglist;
   char     message[200];

   va_start(arglist,pMess);
   vsnprintf(message,200,pMess,arglist);
   va_end(arglist);

   printf("appcrash: %s \n",message);
   exit(1);
}


//  Output a message to stdout and wait for user ACK.
//  Works like printf.

void apppause(char *pMess, ... )
{
   va_list  arglist;
   char     message[200];

   va_start(arglist,pMess);
   vsnprintf(message,200,pMess,arglist);
   va_end(arglist);

   printf("pause: %s \n",message);
   printf("*** press return to continue: ");
   getchar();
   return;
}

void apppause()
{
   printf("*** pause, press return to continue: ");
   getchar();
   return;
}


//  start a timer or get elapsed time with millisecond resolution.

void start_timer(timeval *time0)
{
   gettimeofday(time0,0);
   return;
}

double get_timer(timeval *time0)
{
   double   elapsed;
   timeval  time1;

   gettimeofday(&time1,0);
   elapsed = time1.tv_sec - time0->tv_sec
           + 0.000001 * (time1.tv_usec - time0->tv_usec);
   return elapsed;
}


//  get elapsed CPU time used by current process
//  returns seconds with millisecond resolution

double CPUtime()
{
   clock_t ctime = clock();
   double dtime = ctime / 1000000.0;
   return dtime;
}


//  start a process CPU timer or get elapsed process CPU time
//  returns seconds with millisecond resolution

void start_CPUtimer(double *time0)
{
   *time0 = CPUtime();
   return;
}

double get_CPUtimer(double *time0)
{
   return CPUtime() - *time0;
}


//  Synchronize execution of multiple threads.
//  Simultaneously resume NT calling threads.
//  from main():        synch_threads(NT)          //  setup to synch NT threads
//  from each thread:   synch_threads()            //  suspend, resume simultaneously

void synch_threads(int NT)
{
   static pthread_barrier_t  barrier;
   static int  bflag = 0;

   if (NT) {                                                               //  main(), initialize
      if (bflag) pthread_barrier_destroy(&barrier);
      pthread_barrier_init(&barrier,null,NT);
      bflag = 1;
      return;
   }

   pthread_barrier_wait(&barrier);                                         //  thread(), block
   return;                                                                 //  unblock
}


//  sleep for specified time in seconds (double)
//  small intervals will be increased by OS to a minimum value
//  (currently about 0.008 seconds for Linux kernel 2.6.xx)

void zsleep(double dsecs)
{
   unsigned    isecs, nsecs;
   timespec    tsecs;

   if (dsecs == 0.0) return;   
   isecs = unsigned(dsecs);
   nsecs = unsigned(1000000000.0 * (dsecs - isecs));
   tsecs.tv_sec = isecs;
   tsecs.tv_nsec = nsecs;
   nanosleep(&tsecs,null);
   return;
}


//  malloc() and free() wrappers with auto crash on failure and log option.
//  extra 8 bytes are allocated to track size of allocated blocks   v.07

unsigned ztotalmem = 0;
#define zmallocadd 8

char * zmalloc(size_t bytes, int log)
{
   char *maddr = (char *) malloc(bytes + zmallocadd);
   if (! maddr) appcrash("zmalloc() failure, request: %u",bytes);
   ztotalmem += bytes;
   unsigned *bmaddr = (unsigned *) maddr;
   bmaddr[0] = bytes;
   bmaddr[1] = 6789;                                                       //  v.08 trap bad zfree calls
   char *raddr = maddr + zmallocadd;
   if (log) printf("zmalloc loc: %p  bytes: %u  total: %u \n",raddr,bytes,ztotalmem);
   return raddr;
}

void zfree(void *raddr, int log)
{
   char *maddr = (char *) raddr;
   maddr -= zmallocadd;
   unsigned *bmaddr = (unsigned *) maddr;
   unsigned bytes = bmaddr[0];
   ztotalmem -= bytes;
   if (log) printf("zfree loc: %p  bytes: %u  total: %u \n",raddr,bytes,ztotalmem);
   if (bmaddr[1] != 6789) appcrash("zfree bad address");
   bmaddr[1] = 0;
   free(maddr);
}


//  create a subprocess to run a shell command and read its outputs        //  v.03
//  return:    0: OK    +N: error creating subprocess

FILE    *CPFID = 0;

int createProc(char *command)
{
   char     *command2;
   
   command2 = zmalloc(strlen(command)+20);
   strcpy(command2,command);
   strcat(command2,"; echo CPSTAT: $? ");                                  //  add: get status
   CPFID = popen(command2,"r");
   zfree(command2);
   if (! CPFID) return 1;
   return 0;
}


//  get output records from process created via createProc()               //  v.03
//  bcc:     buff cc
//  return:  0: not EOF, output rec. available  
//           1: EOF, cstat = command status available
//          -1: unknown status
//  cstat:   0: OK,  -1: unknown,  +N: error

int getProcOutput(char *buff, int bcc, int &cstat)
{
   char     *pp;

   cstat = -1;

   if (! CPFID) return -1;                                                 //  no command pending

   pp = fgets_trim(buff,bcc,CPFID,1);                                      //  next output record, less trailing \n

   if (! pp) {                                                             //  EOF
      pclose(CPFID);                                                       //  command status unknown
      return -1;
   }
   
   pp = strstr(pp,"CPSTAT:");
   if (! pp) return 0;                                                     //  normal output record

   pclose(CPFID);                                                          //  status record, close pipe
   CPFID = 0;
   cstat = atoi(pp+7);                                                     //  return command status
   return -1;                                                              //  return EOF
}


//  signal a subprocess to pause, resume, or terminate
//  return:  0: OK  +-N: error

int signalProc(char *pname, char *signal)
{
   pid_t       pid;
   FILE        *fid;
   char        buff[100], *pp;
   int         err, nsignal = 0;

   sprintf(buff,"ps -C %s h o pid > /tmp/signalProc-temp",pname);
   err = system(buff);
   if (err) { err = 2; goto cleanup; }
   
   fid = fopen("/tmp/signalProc-temp","r");
   if (! fid) { err = 3; goto cleanup; }

   pp = fgets(buff,100,fid);
   fclose(fid);
   if (! pp) { err = 4; goto cleanup; }

   pid = atoi(buff);
   if (! pid) { err = 5; goto cleanup; }

   if (strEqu(signal,"pause")) nsignal = SIGSTOP; 
   if (strEqu(signal,"resume")) nsignal = SIGCONT; 
   if (strEqu(signal,"kill")) nsignal = SIGKILL; 
   err = kill(pid,nsignal);   

cleanup:
   system("rm -f /tmp/signalProc-temp");
   return err;
}


//  fgets with additional feature: trailing \n \r are removed
//  optional bf flag: true if trailing blanks are to be removed

char * fgets_trim(char *buff, int maxcc, FILE *fid, int bf)
{
   int      cc;
   char     *pp;
   
   pp = fgets(buff,maxcc,fid);
   if (! pp) return pp;
   cc = strlen(buff);
   if (bf) while (cc && (buff[cc-1] <= ' ')) --cc;
   else    while (cc && (buff[cc-1] < ' ')) --cc;
   buff[cc] = 0;
   return pp;
}


//  random number generators with explicit context
//  and improved randomness over a small series

int lrandz(int64 *seed)                                                    //  returns 0 to 0x7fffffff
{
   *seed = *seed ^ (*seed << 17);
   *seed = *seed ^ (*seed << 20);
   return nrand48((unsigned int16 *) seed);
}

double drandz(int64 *seed)                                                 //  returns 0.0 to 0.99999...
{
   *seed = *seed ^ (*seed << 17);
   *seed = *seed ^ (*seed << 20);
   return erand48((unsigned int16 *) seed);
}


//  return true if both files are in the same directory           v.08
//  both files may be files or directories

int samedirk(const char *file1, const char *file2)
{
   int      cc1, cc2;
   char     *pp1, *pp2;
   
   if (! file1 || ! file2) return 0;
   pp1 = strrchr(file1,'/');
   pp2 = strrchr(file2,'/');
   if (! pp1 && ! pp2) return 1;
   if (pp1 && ! pp2) return 0;
   if (! pp1 && pp2) return 0;
   cc1 = pp1 - file1;
   cc2 = pp2 - file2;
   if (cc1 != cc2) return 0;
   if (strncmp(file1,file2,cc1) == 0) return 1;
   return 0;
}


/**************************************************************************
      string utilities
***************************************************************************

    strField()                             v.09

    char * strField(const char *string, const char *delim, int Nth)

    Get the Nth field in input string, which contains at least N fields 
    delimited by the character(s) in delim (e.g. blank, comma).
    
    Returns a pointer to the found field (actually a pointer to a
    copy of the found field, with a null terminator appended).
    
    If a delimiter is immediately followed by another delimiter, it is 
    considered a field with zero length, and the string "" is returned.

    Leading blanks in a field are omitted from the returned field.
    A field with only blanks is returned as a single blank.

    The last field may be terminated by null or a delimiter.
    
    Characters within quotes (" or ') are treated as data within a
    field, i.e. blanks and delimiters are not processed as such.
    The quotes are removed from the returned field.

    If there are less than N fields, a null pointer is returned.
    
    The last 100 fields are saved and recycled in a circular manner.
    The caller does not have to free memory. If more memory depth is 
    needed, caller must copy the returned data elsewhere.
    
    The input string must be < 1000 characters.
    
    Example: input string: ,a,bb,  cc,   ,dd"ee,ff"ggg,
             (first and last characters are comma)
             delimiter: comma
             Nth   returned string
              1:   (null string)
              2:   a
              3:   bb
              4:   cc
              5:   (one blank)
              6:   ddee,ffggg
              7:   (null pointer >> no more fields)

***************************************************************************/

char * strField(const char *string, const char *delim, int Nth)
{
   static int     ftf = 1, nret = 0;
   static char    *retf[100]; 
   char           *pf1, pf2[1000];
   char           quote1 = '\'', quote2 = '"';
   int            ii, nf, fcc;
   
   if (ftf)                                                                //  overall first call
   {
      ftf = 0;
      for (ii = 0; ii < 100; ii++) retf[ii] = 0;
   }
   
   if (strlen(string) > 999) return 0;
   if (Nth < 1) return 0;

   pf1 = (char *) string - 1;                                              //  start parse
   nf = 0;
   
   while (nf < Nth)
   {
      pf1++;                                                               //  start field
      nf++;
      fcc = 0;

      while (*pf1 == ' ') pf1++;                                           //  skip leading blanks

      while (true)
      {
         if (*pf1 == quote1) {                                             //  pass chars between single quotes
            pf1++;                                                         //  (but without the quotes)
            while (*pf1 && *pf1 != quote1) pf2[fcc++] = *pf1++;
            if (*pf1 == quote1) pf1++;
         }

         else if (*pf1 == quote2) {                                        //  same for double quotes
            pf1++;
            while (*pf1 && *pf1 != quote2) pf2[fcc++] = *pf1++;
            if (*pf1 == quote2) pf1++;
         }
         
         else if (strchr(delim,*pf1) || *pf1 == 0) break;                  //  found delimiter or null
         
         else pf2[fcc++] = *pf1++;                                         //  pass normal character
      }

      if (*pf1 == 0) break;
   }
      
   if (nf < Nth) return 0;                                                 //  no Nth field
   
   if (fcc == 0) {                                                         //  empty field
      if (*string && pf1[-1] == ' ' && !strchr(delim,' '))                 //  if all blanks and blank no delimiter
         return " ";                                                       //     return one blank   v.09
      if (*pf1 == 0) return 0;                                             //  no field
      return "";                                                           //  return null string
   }

   if (++nret == 100) nret = 0;                                            //  use next return slot
   if (retf[nret]) zfree(retf[nret]);
   retf[nret] = zmalloc(fcc+2);
   strncpy0(retf[nret],pf2,fcc+1);
   return retf[nret];
}

char * strField(const char *string, const char delim, int Nth)             //  alternative with one delimiter
{
   char     delims[2] = "x";
   
   *delims = delim;
   return strField(string,delims,Nth);
}


/**************************************************************************

   stat = strParms(begin, input, pname, maxcc, pval)                       v.08

   Parse an input string with parameter names and values:
     "pname1=pval1 | pname2 | pname3=pval3 | pname4 ..."                   v.11 change delimiter
   
   begin    int &          must be 1 to start new string, is modified
   input    const char *   input string
   pname    char *         output parameter name
   maxcc    int            max. length for pname, including null
   pval     double &       output parameter value
   stat     int            status: 0=OK, -1=EOL, 1=parse error
   
   Each call returns the next pname and pval.
   A pname with no pval is assigned a value of 1 (present).
   Input format:  pname1 | pname2=pval2 | pname3 ... null
   Leading blanks are ignored, and pnames may have imbedded blanks.
   pvals must convert to double using convSD (accepts decimal point or comma)

***************************************************************************/

int strParms(int &begin, const char *input, char *pname, int maxcc, double &pval)
{
   static int     ii, beginx = 3579246;
   const char     *pnamex, *delim;
   int            cc, err;

   if (begin == 1) {                                                       //  start new string
      begin = ++beginx;
      ii = 0;
   }

   if (begin != beginx) zappcrash("strParms call error");                  //  thread safe, not reentrant
   
   *pname = 0;                                                             //  initz. outputs to nothing    v.11
   pval = 0;
   
   while (input[ii] == ' ') ii++;                                          //  skip leading blanks
   if (input[ii] == 0) return -1;                                          //  no more data

   pnamex = input + ii;                                                    //  next pname
   
   for (cc = 0; ; cc++)
   {                                                                       //  look for delimiter
      if (pnamex[cc] == '=') break;
      if (pnamex[cc] == '|') break;                                        //  v.11
      if (pnamex[cc] == 0) break;
   }
   
   if (cc == 0) return 1;                                                  //  err: 2 delimiters
   if (cc >= maxcc) return 1;                                              //  err: pname too big

   strncpy0(pname,pnamex,cc+1);                                            //  pname >> caller
   strTrim(pname);                                                         //  remove trailing blanks    v.11

   if (pnamex[cc] == 0) {                                                  //  pname + null
      ii += cc;                                                            //  position for next call
      pval = 1.0;                                                          //  pval = 1 >> caller
      return 0;
   }

   if (pnamex[cc] == '|') {                                                //  pname + |
      ii += cc + 1;                                                        //  position for next call
      pval = 1.0;                                                          //  pval = 1 >> caller
      return 0;
   }

   ii += cc + 1;                                                           //  pname = pval
   err = convSD(input + ii, pval, &delim);                                 //  parse pval   (was strtod()   v.11)
   if (err > 1) return 1;
   while (*delim == ' ') delim++;                                          //  skip poss. trailing blanks
   if (*delim && *delim != '|') return 1;                                  //  err: delimiter not | or null
   ii = delim - input;
   if (*delim) ii++;                                                       //  position for next call
   return 0;
}


//  Produce random value from hashed input string.
//  Output range is 0 to max-1.

int strHash(const char *string, int max)
{
   int      hash, ii, cc;
   long     *lstring = (long *) string;

   cc = strlen(string);
   if (cc > 99) appcrash("strHash, too long",null);

   cc = (cc + 3) / 4;
   hash = 0x12345678;

   for (ii = 0; ii < cc; ii++) {
      hash = hash ^ lstring[ii];
      hash = hash ^ (hash << 5);
      hash = hash ^ (hash >> 7);
      hash = hash ^ (hash << 9);
      hash = hash ^ (hash >> 11);
   }

   hash = hash & 0x3FFFFFFF;
   return (hash % max);
}


//  Hash an input string into a random printable (a-z) output string.      v.08
//  Returns outcc character random printable string in allocated memory.
//  Caller owns the output string, which is subject to zfree().
//  Every output character is randomized from the entire input string.

const char * strHash2(const char *instring, int outcc)
{
   int      incc, ii, jj, rani = 0;
   int64    seed = 13579;
   char     *outstring;

   incc = strlen(instring);
   outstring = zmalloc(outcc+1);
   
   for (ii = 0; ii < outcc; ii++)
   {
      for (jj = 0; jj < incc; jj++)
      {
         seed = seed + instring[jj];
         rani = lrandz(&seed);
      }
      outstring[ii] = 'a' + rani % 26;
   }

   outstring[ii] = 0;
   return outstring;
}


//  Copy string with specified max. length (including null terminator).
//  truncate if needed. null terminator is always supplied.

void strncpy0(char *dest, const char *source, int cc)
{
   strncpy(dest,source,cc);
   dest[cc-1] = 0;
}


//  Copy string with blank pad to specified length.  No null.

void strnPad(char *dest, const char *source, int cc)
{
   strncpy(dest,source,cc);
   int ii = strlen(source);
   for (int jj = ii; jj < cc; jj++) dest[jj] = ' ';
}


//  Remove trailing blanks from a string. Returns remaining length.

int strTrim(char *dest, const char *source)
{
   if (dest != source) strcpy(dest,source);
   return strTrim(dest);
}

int strTrim(char *dest)
{
   int  ii = strlen(dest);
   while (ii && (dest[ii-1] == ' ')) dest[--ii] = 0;
   return ii;
}


//  Remove imbedded blanks from a string. Returns remaining length.

int strCompress(char *dest, const char *source)
{
   if (dest != source) strcpy(dest,source);
   return strCompress(dest);
}

int strCompress(char *string)
{
   int   ii, jj;

   for (ii = jj = 0; string[ii]; ii++)
   {
      if (string[ii] != ' ')
      {
         string[jj] = string[ii];
         jj++;
      }
   }
   string[jj] = 0;
   return jj;
}


//  Concatenate multiple strings, staying within a specified overall length.
//  The destination string is also the first source string. 
//  Null marks the end of the source strings (omission --> crash).
//  Output is truncated to fit within the specified length.
//  A final null is assured and is included in the length.
//  Returns 0 if OK, 1 if truncation was needed.
//  v.03: truncate instead of do nothing if string does not fit.

int strncatv(char *dest, int maxcc, const char *source, ...)
{
   const char  *ps;
   va_list     arglist;

   maxcc = maxcc - strlen(dest) - 1;
   if (maxcc < 0) return 1;
   va_start(arglist,source);
   ps = source;

   while (ps)
   {
      strncat(dest,ps,maxcc);
      maxcc = maxcc - strlen(ps);
      if (maxcc < 0) break;
      ps = va_arg(arglist,const char *);
   }
   
   va_end(arglist);
   if (maxcc < 0) return 1;
   return 0;
}


//  Match 1st string to N additional strings.
//  Return matching string number 1 to N or 0 if no match.
//  Supply a null argument for end of list.

int strcmpv(const char *string, ...)
{
   int      match = 0;
   char     *stringN;
   va_list  arglist;

   va_start(arglist,string);

   while (1)
   {
      stringN = va_arg(arglist, char *);
      if (stringN == null)
      {
         va_end(arglist);
         return 0;
      }

      match++;
      if (strcmp(string,stringN) == 0)
      {
         va_end(arglist);
         return match;
      }
   }
}


//  convert string to upper case

void strToUpper(char *string)
{
   int         ii;
   char        jj;
   const int   delta = 'A' - 'a';

   for (ii = 0; (jj = string[ii]); ii++)
        if ((jj >= 'a') && (jj <= 'z')) string[ii] += delta;
}

void strToUpper(char *dest, const char *source)
{
   strcpy(dest,source);
   strToUpper(dest);
}


//  convert string to lower case

void strToLower(char *string)
{
   int         ii;
   char        jj;
   const int   delta = 'a' - 'A';

   for (ii = 0; (jj = string[ii]); ii++)
        if ((jj >= 'A') && (jj <= 'Z')) string[ii] += delta;
}

void strToLower(char *dest, const char *source)
{
   strcpy(dest,source);
   strToLower(dest);
}


//  copy string strin to strout, replacing every occurrence
//    of the substring ssin with the substring ssout

int repl_1str(char *strin, char *strout, char *ssin, char *ssout)
{
   int      ccc, cc1, cc2, nfound;
   char     *ppp;
   
   cc1 = strlen(ssin);
   cc2 = strlen(ssout);
   nfound = 0;
   
   while ((ppp = strstr(strin,ssin)))
   {
      nfound++;
      ccc = ppp - strin;
      strncpy(strout,strin,ccc);
      strout += ccc;
      strin += ccc;
      strncpy(strout,ssout,cc2);
      strin += cc1;
      strout += cc2;
   }

   strcpy(strout,strin);
   return nfound;
}


//  like repl_1str, but multiple pairs of substrings are processed
//   (... ssin1, ssout1, ssin2, ssout2, ... null) 

int repl_Nstrs(char *strin, char *strout, ...)
{
   va_list     arglist;
   char        *ssin, *ssout;
   char        ftemp[maxfcc];
   int         ftf, nfound;
   
   ftf = 1;
   nfound = 0;
   va_start(arglist,strout);
   
   while (true)
   {
      ssin = va_arg(arglist, char *);
      if (! ssin) break;
      ssout = va_arg(arglist, char *);

      if (ftf) {
         ftf = 0;
         nfound += repl_1str(strin,strout,ssin,ssout);
      }

      else {
         strcpy(ftemp,strout);
         nfound += repl_1str(ftemp,strout,ssin,ssout);
      }
   }

   va_end(arglist);
   return nfound;
}


//  Copy and convert string to hex string.

void strncpyx(char *out, const char *in, int ccin)
{
   int      ii, jj, c1, c2;
   char     cx[] = "0123456789ABCDEF";

   for (ii = 0, jj = 0; ii < ccin; ii++, jj += 3)
   {
      c1 = in[ii] > 4;                                                     //  bugfix v.08
      c2 = in[ii] & 15;
      out[jj] = cx[c1];
      out[jj+1] = cx[c2];
      out[jj+2] = ' ';
   }
   out[jj] = 0;
   return;
}


//  Strip trailing zeros from ascii floating numbers
//    (e.g. 1.230000E+02  -->  1.23E+02)

void StripZeros(char *pNum)
{
   int     ii, ll;
   int     pp, k1, k2;
   char    work[20];

   ll = strlen(pNum);
   if (ll >= 20) return;

   for (ii = 0; ii < ll; ii++)
   {
      if (pNum[ii] == '.')
      {
         pp = ii;
         k1 = k2 = 0;
         for (++ii; ii < ll; ii++)
         {
            if (pNum[ii] == '0')
            {
               if (! k1) k1 = k2 = ii;
               else k2 = ii;
               continue;
            }

            if ((pNum[ii] >= '1') && (pNum[ii] <= '9'))
            {
               k1 = 0;
               continue;
            }

            break;
         }

         if (! k1) return;

         if (k1 == pp + 1) k1++;
         if (k2 < k1) return;
         strcpy(work,pNum);
         strcpy(work+k1,pNum+k2+1);
         strcpy(pNum,work);
         return;
      }
   }
}


//  test for blank/null string                                             v.08

int is_blank_null(const char *string)
{
   if (! string) return 1;                                                 //  null string
   if (! *string) return 1;                                                //  zero length string
   int cc = strlen(string);
   for (int ii = 0; ii < cc; ii++)
      if (string[ii] != ' ') return 0;                                     //  non-blank string
   return 1;                                                               //  blank string
}


/**************************************************************************
   bitmap functions                                            v.09
***************************************************************************/

//  create a new bitmap with specified bit length. 
//  initially all bits are false.

bitmap * bitmap_new(int nbits)
{
   int      cc, ii;
   bitmap   *bm;
   
   bm = (bitmap *) zmalloc(sizeof(bitmap));
   bm->nbits = nbits;
   cc = (nbits + 7) / 8;
   bm->bits = (uchar *) zmalloc(cc);
   for (ii = 0; ii < cc; ii++) bm->bits[ii] = 0;
   return bm;
}


//  set bit in bitmap to true or false

void bitmap_set(bitmap *bm, int bit, bool value)
{
   int      ii, jj;
   uchar    bit1;

   if (bit >= bm->nbits) zappcrash("bitmap, bit %d too big",bit);
   ii = bit / 8;
   jj = bit % 8;
   bit1 = 0x80 >> jj;

   if (value) bm->bits[ii] = bm->bits[ii] | bit1;
   else {
      bit1 = bit1 ^ 0xff;
      bm->bits[ii] = bm->bits[ii] & bit1;
   }

   return;
}


//  fetch bitmap bit, return true or false

bool bitmap_get(bitmap *bm, int bit)
{
   int      ii, jj;
   uchar    bit1;

   ii = bit / 8;
   jj = bit % 8;
   bit1 = bm->bits[ii] << jj;
   if (bit1 < 127) return false;
   else return true;
}


//  delete bitmap

void bitmap_delete(bitmap *bm)
{
   zfree(bm->bits);
   zfree(bm);
   return;
}


/**************************************************************************
   variable list functions - array or list of strings          v.07
***************************************************************************/

//  create new variable list with specified capacity

pvlist * pvlist_create(int max)
{
   pvlist      *pv;

   pv = (pvlist *) zmalloc(sizeof(pvlist));
   pv->max = max;
   pv->act = 0;
   pv->list = (char **) zmalloc(max * sizeof(char *));
   return pv;
}

//  free memory for variable list
   
void pvlist_free(pvlist *pv)
{
   int      ii;
   
   for (ii = 0; ii < pv->act; ii++)
      zfree(pv->list[ii]);
   zfree(pv->list);
   zfree(pv);
}

//  append new entry to end of list (optional if unique)
//  if list if full, first entry is removed and rest are packed down
//  return: N >= 0: new entry added at position N
//          N = -1: not unique, not added
   
int pvlist_append(pvlist *pv, const char *entry, int unique)
{
   int      ii;

   if (unique && pvlist_find(pv,entry) >= 0) return -1;                    //  not unique

   if (pv->act == pv->max) pvlist_remove(pv,0);                            //  if list full, remove 1st entry

   ii = pv->act;
   pv->list[ii] = zmalloc(strlen(entry)+1);                                //  add to end of list
   strcpy(pv->list[ii],entry);
   pv->act++;
   return ii;
}

//  prepend new entry to list (optional if unique)
//  prior list entries are pushed down to make room
//  if list is full, last entry is removed first
//  return: N = 0: new entry added at position 0
//          N = -1: not unique, not added
   
int pvlist_prepend(pvlist *pv, const char *entry, int unique)
{
   int      ii;
   
   if (unique && pvlist_find(pv,entry) >= 0) return -1;                    //  not unique

   if (pv->act == pv->max) pvlist_remove(pv,pv->act-1);                    //  if list full, remove last entry

   for (ii = pv->act; ii > 0; ii--)                                        //  push all list entries down
      pv->list[ii] = pv->list[ii-1];
   pv->list[0] = zmalloc(strlen(entry)+1);                                 //  add to start of list
   strcpy(pv->list[0],entry);
   pv->act++;
   return 0;
}

//  find list entry by name, return -1 if not found
   
int pvlist_find(pvlist *pv, const char *entry)
{
   int      ii;

   for (ii = 0; ii < pv->act; ii++)
      if (strEqu(entry,pv->list[ii])) break;
   if (ii < pv->act) return ii;
   return -1;
}

//  remove an entry by name and repack list
   
int pvlist_remove(pvlist *pv, const char *entry)
{
   int      ii;
   
   ii = pvlist_find(pv,entry);
   if (ii < 0) return -1;
   pvlist_remove(pv,ii);
   return ii;
}

//  remove an entry by number and repack list
   
int pvlist_remove(pvlist *pv, int ii)
{
   if (ii < 0 || ii >= pv->act) return -1;
   zfree(pv->list[ii]);
   for (++ii; ii < pv->act; ii++)
      pv->list[ii-1] = pv->list[ii];
   pv->act--;
   return 0;
}


//  return entry count

int pvlist_count(pvlist *pv)
{
   return pv->act;
}


//  replace Nth entry with new one           v.09

int pvlist_replace(pvlist * pv, int ii, char *entry)
{
   if (ii < 0 || ii >= pv->act) return -1;
   zfree(pv->list[ii]);
   pv->list[ii] = zmalloc(strlen(entry)+1);
   strcpy(pv->list[ii],entry);
   return 0;
}


//  return Nth entry or null

char * pvlist_get(pvlist *pv, int Nth)
{
   if (Nth >= pv->act) return 0;
   return pv->list[Nth];
}


//  sort list in ascending order       v.09

int pvlist_sort(pvlist *pv)
{
   HeapSort(pv->list,pv->act);
   return 0;
}

/**************************************************************************

   Conversion Utilities

   convSI(string, inum, delim)                     string to int
   convSI(string, inum, low, high, delim)          string to int with range check

   convSD(string, dnum, delim)                     string to double
   convSD(string, dnum, low, high, delim)          string to double with range check

   convIS(inum, string, cc)                        int to string with returned cc

   convDS(fnum, digits, string, cc)                double to string with specified 
                                                     digits of precision and returned cc
   
   string      input (const char *) or output (char *)
   inum        input (int) or output (int &)
   dnum        input (double) or output (double &)
   delim       optional output pointer to delimiter (null or const char **)   v.11
   low, high   input range check (int or double)
   cc          output string length (int &)
   digits      input digits of precision (int) to be used for output string

   function status returned:
       0    normal conversion, no invalid digits, blank/null termination
       1    successful converstion, but trailing non-numeric found
       2    conversion OK, but outside specified limits
       3    null or blank string, converted to zero
       4    conversion error, invalid data in string
   overlapping statuses have following precedence: 4 3 2 1 0

***************************************************************************/

#define  max10  (0x7fffffff / 10)


//  Convert string to integer

int convSI(const char *string, int &inum, const char **delim)
{
   char        ch;
   int         sign = 0, digits = 0, tnb = 0;
   const char  *pch = string;

   inum = 0;

   while ((ch = *pch) == ' ') pch++;                                       //  skip leading blanks

   if (ch == '-') sign = -1;                                               //  process leading +/- sign
   if (ch == '+') sign = 1;                                                //  (at most one sign character)
   if (sign) pch++;

   while ((*pch >= '0') && (*pch <= '9'))                                  //  process digits 0 - 9
   {
      if (inum > max10) goto conv_err;                                     //  value too big
      inum = 10 * inum + *pch - '0';
      digits++;
      pch++;
   }

   if (delim) *delim = pch;                                                //  terminating delimiter     v.11
   if (*pch && (*pch != ' ')) tnb++;                                       //  not null or blank

   if (! digits)                                                           //  no digits found
   {
      if (tnb) return 4;                                                   //  non-numeric (invalid) string
      else return 3;                                                       //  null or blank string
   }

   if (sign == -1) inum = -inum;                                           //  negate if - sign

   if (! tnb) return 0;                                                    //  no trailing non-numerics
   else return 1;                                                          //  trailing non-numerics

conv_err:
   inum = 0;
   return 4;
}


int convSI(const char *string, int & inum, int lolim, int hilim, const char **delim)
{
   int   stat = convSI(string,inum,delim);

   if (stat > 2) return stat;                                              //  invalid or null/blank
   if (inum < lolim) return 2;                                             //  return 2 if out of limits
   if (inum > hilim) return 2;                                             //  (has precedence over status 1)
   return stat;                                                            //  limits OK, return 0 or 1
}


//  Convert string to double.

int convSD(const char *string, double &dnum, const char **delim)
{
   char        ch;
   int         ii, sign = 0, digits = 0, ndec = 0;
   int         exp = 0, esign = 0, edigits = 0, tnb = 0;
   const char  *pch = string;

   static int  first = 1;
   static double  decimals[21], exponents[74];

   if (first)                                                              //  first-time called
   {
      first = 0;                                                           //  pre-calculate constants
      for (ii = 1; ii <= 20; ii++) decimals[ii] = pow(10.0,-ii);
      for (ii = -36; ii <= 36; ii++) exponents[ii+37] = pow(10.0,ii);
   }

   dnum = 0.0;

   while ((ch = *pch) == ' ') pch++;                                       //  skip leading blanks

   if (ch == '-') sign = -1;                                               //  process leading +/- sign
   if (ch == '+') sign = 1;                                                //  (at most one sign character)
   if (sign) pch++;

get_digits:

   while ((*pch >= '0') && (*pch <= '9'))                                  //  process digits 0 - 9
   {
      dnum = 10.0 * dnum + (*pch - '0');
      pch++;
      digits++;
      if (ndec) ndec++;
   }

   if ((*pch == '.') || (*pch == ','))                                     //  process decimal point
   {                                                                       //  (allow comma or period)  v.07
      if (ndec) goto conv_err;
      ndec++;
      pch++;
      goto get_digits;
   }

   if ((*pch == 'e') || (*pch == 'E'))                                     //  process optional exponent
   {
      pch++;
      if (*pch == '+') esign = 1;                                          //  optional +/- sign
      if (*pch == '-') esign = -1;
      if (esign) pch++;

      if ((*pch < '0') || (*pch > '9')) goto conv_err;                     //  1st digit
      exp = *pch - '0';
      edigits++;
      pch++;

      if ((*pch >= '0') && (*pch <= '9'))                                  //  optional 2nd digit
      {
         exp = 10 * exp + (*pch - '0');
         edigits++;
         pch++;
      }

      if ((exp < -36) || (exp > 36)) goto conv_err;                        //  exponent too big 
   }

   if (delim) *delim = pch;                                                //  terminating delimiter     v.11
   if (*pch && (*pch != ' ')) tnb++;                                       //  not null or blank

   if (!(digits + edigits))                                                //  no digits found
   {
      if (tnb) return 4;                                                   //  non-numeric (invalid) string
      else return 3;                                                       //  null or blank string
   }

   if (ndec > 1) dnum = dnum * decimals[ndec-1];                           //  compensate for decimal places

   if (sign == -1) dnum = - dnum;                                          //  negate if negative

   if (exp)                                                
   {
      if (esign == -1) exp = -exp;                                         //  process exponent
      dnum = dnum * exponents[exp+37];
   }

   if (! tnb) return 0;                                                    //  no trailing non-numerics
   else return 1;                                                          //  trailing non-numerics

conv_err:
   dnum = 0.0;
   return 4;
}


int convSD(const char *string, double &dnum, double lolim, double hilim, const char **delim)
{
   int stat = convSD(string,dnum,delim);

   if (stat > 2) return stat;                                              //  invalid or null/blank
   if (dnum < lolim) return 2;                                             //  return 2 if out of limits
   if (dnum > hilim) return 2;                                             //  (has precedence over status 1)
   return stat;                                                            //  limits OK, return 0 or 1
}


//  Convert int to string with returned length.

int convIS(int inum, char *string, int & cc)
{
   cc = sprintf(string,"%d",inum);
   return 0;
}


//  Convert double to string with specified digits of precision.
//  Shortest length format (f/e) will be used.  
//  Output length is returned in cc.

int convDS(double dnum, int digits, char *string, int &cc)
{
   char  *pstr;
   
   sprintf(string,"%.*g",digits,dnum);

   pstr = strstr(string,"e+");
   if (pstr) strcpy(pstr+1,pstr+2);

   pstr = strstr(string,"e0");            
   if (pstr) strcpy(pstr+1,pstr+2);

   pstr = strstr(string,"e0");            
   if (pstr) strcpy(pstr+1,pstr+2);

   pstr = strstr(string,"e-0");
   if (pstr) strcpy(pstr+2,pstr+3);

   pstr = strstr(string,"e-0");
   if (pstr) strcpy(pstr+2,pstr+3);

   cc = strlen(string);
   return 0;
}


/**************************************************************************

    Wildcard string match

    Match candidate string to wildcard string containing any number of 
    '*' or '?' wildcard characters. '*' matches any number of characters, 
    including zero characters. '?' matches any one character.

    Returns 0 if match, 1 if no match.

***************************************************************************/

int MatchWild(const char *pWild, const char *pString)
{
   int   ii, star;

new_segment:

   star = 0;
   while (pWild[0] == '*')
   {
      star = 1;
      pWild++;
   }

test_match:

   for (ii = 0; pWild[ii] && (pWild[ii] != '*'); ii++)
   {
      if (pWild[ii] != pString[ii])
      {
         if (! pString[ii]) return 1;
         if (pWild[ii] == '?') continue;
         if (! star) return 1;
         pString++;
         goto test_match;
      }
   }

   if (pWild[ii] == '*')
   {
      pString += ii;
      pWild += ii;
      goto new_segment;
   }

   if (! pString[ii]) return 0;
   if (ii && pWild[ii-1] == '*') return 0;
   if (! star) return 1;
   pString++;
   goto test_match;
}


/**************************************************************************

   SearchWild  - wildcard file search

   Find all files with total /pathname/filename matching a pattern,
   which may have any number of the wildcard characters '*' and '?'
   in either or both the pathname and filename.

   const char * SearchWild(const char *wfilespec, int &flag)
   
   inputs:  flag = 1 to start a new search
            flag = 2 abort a running search
            *** do not modify flag within a search ***

            wfilespec = filespec to search with optional wildcards
               e.g. "/name1/na*me2/nam??e3/name4*.ext?"
               
   return:  a pointer to one matching file is returned per call,
            or null when there are no more matching files.
             
   The search may be aborted before completion, but make a final 
   call with flag = 2 to clean up temp file. A new search with 
   flag = 1 will also finish the cleanup.
   
   NOT THREAD SAFE - do not use in parallel threads
   
   shell find command is used for the initial search because this
   is much faster than recursive use of readdir() (why?). 

   (#) is used in place of (*) in comments below to prevent 
   compiler from interpreting (#/) as end of comments

   GNU find peculiarities: 
     find /path/#      omits "." files
     find /path/       includes "." files
     find /path/#      recurses directories under /path/
     find /path/#.txt  does not recurse directories
     find /path/#/     finds all files under /path/
     find /path/#/#    finds files >= 1 directory level under /path/
     find /path/xxx#   never finds anything

   SearchWild uses simpler and more intuitive matching: 
     '/' and '.' are matched by '#'
     /path/#.txt finds all .txt files under /path/ at any directory level
     there is no exclusion of "." files
   
***************************************************************************/

const char * SearchWild(const char *wpath, int &uflag)
{
   static FILE    *fid = 0;
   static char    tempfile[60];
   static char    matchfile[maxfcc];
   char           searchpath[maxfcc];
   char           command[maxfcc];
   int            cc, err;
   char           *pp;
   pid_t          pid;
   
   if ((uflag == 1) || (uflag == 2)) {                                     //  first call or stop flag
      if (fid) {
         fclose(fid);                                                      //  if file open, close it
         fid = 0;
         sprintf(command,"rm -f %s",tempfile);                             //  and delete it
         system(command);
      }
   }
   
   if (uflag == 2) return 0;                                               //  kill flag, done
      
   if (uflag == 1)                                                         //  first call flag
   {
      cc = strlen(wpath);
      if (cc == 0) return 0;
      if (cc > maxfcc-20) appcrash("SearchWild: wpath > max");
      
      pp = (char *) wpath;
      repl_Nstrs(pp,searchpath,"$","\\$","\"","\\\"",0);                   //  initial search path, escape $ and "

      pp = strchr(searchpath,'*');
      if (pp) {                                                            //  not efficient but foolproof 
         while ((*pp != '/') && (pp > searchpath)) pp--;                   //  /aaa/bbb/cc*cc... >>> /aaa/bbb/
         if (pp > searchpath) *(pp+1) = 0;
      }

      pid = getpid();
      sprintf(tempfile,"/tmp/searchwild-%u",pid);                          //  unique temp file name
      sprintf(command,"find \"%s\" -type f -or -type l > %s",              //  find files (ordinary, symlink)
                                 searchpath,tempfile);                     //  output to temp file
      err = system(command);                                               //  ignore error
      fid = fopen(tempfile,"r");                                           //  open temp file
      uflag = 763568954;                                                   //  begin search
   }

   if (uflag != 763568954) appcrash("SearchWild, uflag invalid");
   
   while (true)
   {
      pp = fgets(matchfile,maxfcc-2,fid);                                  //  next matching file
      if (! pp) {
         fclose(fid);                                                      //  no more
         fid = 0;
         sprintf(command,"rm -f %s \n",tempfile);
         system(command);
         return 0;
      }

      cc = strlen(matchfile);                                              //  get rid of trailing \n
      matchfile[cc-1] = 0;

      err = MatchWild(wpath,matchfile);                                    //  wildcard match?
      if (err) continue;                                                   //  no

      return matchfile;                                                    //  return file
   }
}


/**************************************************************************/

//  perform a binary search on sorted list of integers                     //  2006.12.17
//  return matching element or -1 if not found

int bsearch(int element, int nn, int list[])
{
   int      ii, jj, kk, rkk;

   ii = nn / 2;                                                            //  next element to search
   jj = (ii + 1) / 2;                                                      //  next increment
   nn--;                                                                   //  last element
   rkk = 0;

   while (true)
   {
      kk = list[ii] - element;                                             //  check element

      if (kk > 0)
      {
         ii -= jj;                                                         //  too high, go down
         if (ii < 0) return -1;
      }

      else if (kk < 0)
      {
         ii += jj;                                                         //  too low, go up
         if (ii > nn) return -1;
      }

      else if (kk == 0) return ii;                                         //  matched

      jj = jj / 2;                                                         //  reduce increment

      if (jj == 0)
      {
         jj = 1;                                                           //  step by 1 element
         if (! rkk) rkk = kk;                                              //  save direction
         else
         {
            if (rkk > 0) { if (kk < 0) return -1; }                        //  if change direction, fail
            else if (kk > 0) return -1;
         }
      }
   }
}


/**************************************************************************
   heap sort functions
***************************************************************************/

#define SWAP(x,y) (temp = (x), (x) = (y), (y) = temp)


//  heapsort for array of integers 

static void adjust(int vv[], int n1, int n2)
{
   int   *bb, jj, kk, temp;

   bb = vv - 1;
   jj = n1;
   kk = n1 * 2;

   while (kk <= n2)
   {
      if (kk < n2 && bb[kk] < bb[kk+1]) kk++;
      if (bb[jj] < bb[kk]) SWAP(bb[jj],bb[kk]);
      jj = kk;
      kk *= 2;
   }
}

void HeapSort(int vv[], int nn)
{
   int   *bb, jj, temp;

   for (jj = nn/2; jj > 0; jj--) adjust(vv,jj,nn);

   bb = vv - 1;

   for (jj = nn-1; jj > 0; jj--)
   {
      SWAP(bb[1], bb[jj+1]);
      adjust(vv,1,jj);
   }
}


//  heapsort for array of floats 

static void adjust(float vv[], int n1, int n2)
{
   float    *bb, temp;
   int      jj, kk;

   bb = vv - 1;
   jj = n1;
   kk = n1 * 2;

   while (kk <= n2)
   {
      if (kk < n2 && bb[kk] < bb[kk+1]) kk++;
      if (bb[jj] < bb[kk]) SWAP(bb[jj],bb[kk]);
      jj = kk;
      kk *= 2;
   }
}

void HeapSort(float vv[], int nn)
{
   float    *bb, temp;
   int      jj;

   for (jj = nn/2; jj > 0; jj--) adjust(vv,jj,nn);

   bb = vv - 1;

   for (jj = nn-1; jj > 0; jj--)
   {
      SWAP(bb[1], bb[jj+1]);
      adjust(vv,1,jj);
   }
}


//  heapsort for array of doubles 

static void adjust(double vv[], int n1, int n2)
{
   double   *bb, temp;
   int      jj, kk;

   bb = vv - 1;
   jj = n1;
   kk = n1 * 2;

   while (kk <= n2)
   {
      if (kk < n2 && bb[kk] < bb[kk+1]) kk++;
      if (bb[jj] < bb[kk]) SWAP(bb[jj],bb[kk]);
      jj = kk;
      kk *= 2;
   }
}

void HeapSort(double vv[], int nn)
{
   double   *bb, temp;
   int      jj;

   for (jj = nn/2; jj > 0; jj--) adjust(vv,jj,nn);

   bb = vv - 1;

   for (jj = nn-1; jj > 0; jj--)
   {
      SWAP(bb[1], bb[jj+1]);
      adjust(vv,1,jj);
   }
}


//  heapsort array of pointers to strings in ascending order of strings
//  pointers are sorted, strings are not changed.

static void adjust(char *vv[], int n1, int n2)
{
   char     **bb, *temp;
   int      jj, kk;

   bb = vv - 1;
   jj = n1;
   kk = n1 * 2;

   while (kk <= n2)
   {
      if (kk < n2 && strcmp(bb[kk],bb[kk+1]) < 0) kk++;
      if (strcmp(bb[jj],bb[kk]) < 0) SWAP(bb[jj],bb[kk]);
      jj = kk;
      kk *= 2;
   }
}

void HeapSort(char *vv[], int nn)
{
   char     **bb, *temp;
   int      jj;

   for (jj = nn/2; jj > 0; jj--) adjust(vv,jj,nn);

   bb = vv;

   for (jj = nn-1; jj > 0; jj--)
   {
      SWAP(bb[0], bb[jj]);
      adjust(vv,1,jj);
   }
}


//  heapsort array of pointers to strings in user-defined order.           v.06
//  pointers are sorted, strings are not changed.

static void adjust(char *vv[], int n1, int n2, HeapSortUcomp fcomp)
{
   char     **bb, *temp;
   int      jj, kk;

   bb = vv - 1;
   jj = n1;
   kk = n1 * 2;

   while (kk <= n2)
   {
      if (kk < n2 && fcomp(bb[kk],bb[kk+1]) < 0) kk++;
      if (fcomp(bb[jj],bb[kk]) < 0) SWAP(bb[jj],bb[kk]);
      jj = kk;
      kk *= 2;
   }
}

void HeapSort(char *vv[], int nn, HeapSortUcomp fcomp)
{
   char     **bb, *temp;
   int      jj;

   for (jj = nn/2; jj > 0; jj--) adjust(vv,jj,nn,fcomp);

   bb = vv;

   for (jj = nn-1; jj > 0; jj--)
   {
      SWAP(bb[0], bb[jj]);
      adjust(vv,1,jj,fcomp);
   }
}


//  heapsort for array of strings or records, 
//  using caller-supplied record compare function.
//  HeapSortUcomp returns [ -1 0 +1 ]  for  rec1 [ < = > ] rec2
//  method: build array of pointers and sort these, then
//  use this sorted array to re-order the records at the end.

static int     *vv1, *vv2;

static void adjust(char *recs, int RL, int n1, int n2, HeapSortUcomp fcomp)
{
   int      *bb, jj, kk, temp;
   char     *rec1, *rec2;

   bb = vv1 - 1;
   jj = n1;
   kk = n1 * 2;

   while (kk <= n2)
   {
      rec1 = recs + RL * bb[kk];
      rec2 = recs + RL * bb[kk+1];
      if (kk < n2 && fcomp(rec1,rec2) < 0) kk++;
      rec1 = recs + RL * bb[jj];
      rec2 = recs + RL * bb[kk];
      if (fcomp(rec1,rec2) < 0) SWAP(bb[jj],bb[kk]);
      jj = kk;
      kk *= 2;
   }
}

void HeapSort(char *recs, int RL, int NR, HeapSortUcomp fcomp)
{
   int      *bb, jj, kk, temp, flag;
   char     *vvrec;

   vv1 = new int[NR];
   for (jj = 0; jj < NR; jj++) vv1[jj] = jj;

   for (jj = NR/2; jj > 0; jj--) adjust(recs,RL,jj,NR,fcomp);

   bb = vv1 - 1;

   for (jj = NR-1; jj > 0; jj--)
   {
      SWAP(bb[1], bb[jj+1]);
      adjust(recs,RL,1,jj,fcomp);
   }

   vv2 = new int[NR];
   for (jj = 0; jj < NR; jj++) vv2[vv1[jj]] = jj;

   vvrec = new char[RL];
   flag = 1;
   while (flag)
   {
      flag = 0;
      for (jj = 0; jj < NR; jj++)
      {
         kk = vv2[jj];
         if (kk == jj) continue;
         memmove(vvrec,recs+jj*RL,RL);
         memmove(recs+jj*RL,recs+kk*RL,RL);
         memmove(recs+kk*RL,vvrec,RL);
         SWAP(vv2[jj],vv2[kk]);
         flag = 1;
      }
   }

   delete vv1;
   delete vv2;
   delete vvrec;
}


/**************************************************************************

         int MemSort (char *RECS, int RL, int NR, int KEYS[][3], int NK)

         RECS is an array of records, to be sorted in-place.
         (record length = RL, record count = NR)

         KEYS[NK,3]  is an integer array defined as follows:
              [N,0]    starting position of Nth key field in RECS
              [N,1]    length of Nth key field in RECS
              [N,2]    type of sort for Nth key:
                        1 = char ascending
                        2 = char descending
                        3 = int*4 ascending (int, long)
                        4 = int*4 descending
                        5 = float*4 ascending (float)
                        6 = float*4 descending
                        7 = float*8 ascending (double)
                        8 = float*8 descending

***************************************************************************/

int MemSortComp(const char *rec1, const char *rec2);
int MemSortKeys[10][3], MemSortNK;

int MemSort(char *RECS, int RL, int NR, int KEYS[][3], int NK)
{
   int   ii;

   if (NR < 2) return 1;

   if (NK > 10) appcrash("MemSort, bad NK");
   if (NK < 1) appcrash("MemSort, bad NK");

   MemSortNK = NK;

   for (ii = 0; ii < NK; ii++)
   {
      MemSortKeys[ii][0] = KEYS[ii][0];
      MemSortKeys[ii][1] = KEYS[ii][1];
      MemSortKeys[ii][2] = KEYS[ii][2];
   }

   HeapSort(RECS,RL,NR,MemSortComp);
   return 1;
}

int MemSortComp(const char *rec1, const char *rec2)
{
   int            ii, stat, kpos, ktype, kleng;
   int            inum1, inum2;
   float          rnum1, rnum2;
   double         dnum1, dnum2;
   const char     *p1, *p2;

   for (ii = 0; ii < MemSortNK; ii++)                                      //  loop each key
   {
      kpos = MemSortKeys[ii][0];                                           //  relative position
      kleng = MemSortKeys[ii][1];                                          //  length
      ktype = MemSortKeys[ii][2];                                          //  type

      p1 = rec1 + kpos;                                                    //  absolute position
      p2 = rec2 + kpos;

      switch (ktype)
      {
         case 1:                                                           //  char ascending
            stat = strncmp(p1,p2,kleng);                                   //  compare 2 key values
            if (stat) return stat;                                         //  + if rec1 > rec2, - if rec1 < rec2
            break;                                                         //  2 keys are equal, check next key

         case 2:                                                           //  char descending
            stat = strncmp(p1,p2,kleng);
            if (stat) return -stat;
            break;

         case 3:                                                           //  int ascending
            memmove(&inum1,p1,4);
            memmove(&inum2,p2,4);
            if (inum1 > inum2) return 1;
            if (inum1 < inum2) return -1;
            break;

         case 4:                                                           //  int descending
            memmove(&inum1,p1,4);
            memmove(&inum2,p2,4);
            if (inum1 > inum2) return -1;
            if (inum1 < inum2) return 1;
            break;

         case 5:                                                           //  float ascending
            memmove(&rnum1,p1,4);
            memmove(&rnum2,p2,4);
            if (rnum1 > rnum2) return 1;
            if (rnum1 < rnum2) return -1;
            break;

         case 6:                                                           //  float descending
            memmove(&rnum1,p1,4);
            memmove(&rnum2,p2,4);
            if (rnum1 > rnum2) return -1;
            if (rnum1 < rnum2) return 1;
            break;

         case 7:                                                           //  double ascending
            memmove(&dnum1,p1,8);
            memmove(&dnum2,p2,8);
            if (dnum1 > dnum2) return 1;
            if (dnum1 < dnum2) return -1;
            break;

         case 8:                                                           //  double descending
            memmove(&dnum1,p1,8);
            memmove(&dnum2,p2,8);
            if (dnum1 > dnum2) return -1;
            if (dnum1 < dnum2) return 1;
            break;

         default:                                                          //  key type not 1-8
            appcrash("MemSort, bad KEYS sort type");
      }
   }

   return 0;                                                               //  records match on all keys
}


/**************************************************************************
   Quick Math Functions                v.02

   These functions duplicate C library standard functions but provide 
   faster execution at the expense of precision. The precision is 
   generally 5 or 6 digits, but test and use with caution.
   
   They all build a table of values on first call (1-2 millisecs),
   which is used to interpolate values in subsequent calls.

   The argument range is also limited - see comments in each function.
   
   Following benchmarks done with Intel Core 2 Duo at 2.4 GHz

   double qsine(double)       3.2 times as fast as sin()
   double qcosine(double)     3.2 times as fast as cos()
   double qarcsine(double)    5.1 times as fast as asin()
   double qsqrt(double)       2.1 times as fast as sqrt()

***************************************************************************/

double qsine(double x)                                                     //  sine(x) for x = -pi to +pi
{                                                                          //  (seg. fault if out of range)
   int            ii;
   static int     ftf = 1;
   double         v1, s1, s2, ans;
   static double  base = 3.15;                                             //  pi + a bit more
   static double  K1 = base / 500.0;
   static double  K2 = 500.0 / base;
   static double  v[1002], sinv[1002];

   if (ftf) {
      ftf = 0;
      for (ii = 0; ii < 1002; ii++)
      {
         v1 = ii * K1 - base;                                              //  -pi to +pi (a bit more each end)
         v[ii] = v1;
         sinv[ii] = sin(v1);
      }
   }
   
   ii = int(K2 * (x + base));
   v1 = v[ii];
   s1 = sinv[ii];
   s2 = sinv[ii+1];
   ans = s1 + (s2 - s1) * (x - v1) * K2;
   return ans;
}

double qcosine(double x)                                                   //  cosine(x) for x = -pi to +pi
{                                                                          //  (seg. fault if out of range)
   int            ii;
   static int     ftf = 1;
   double         v1, s1, s2, ans;
   static double  base = 3.15;                                             //  pi + a bit more
   static double  K1 = base / 500.0;
   static double  K2 = 500.0 / base;
   static double  v[1002], cosinv[1002];

   if (ftf) {
      ftf = 0;
      for (ii = 0; ii < 1002; ii++)
      {
         v1 = ii * K1 - base;                                              //  -pi to +pi (a bit more each end)
         v[ii] = v1;
         cosinv[ii] = cos(v1);
      }
   }
   
   ii = int(K2 * (x + base));
   v1 = v[ii];
   s1 = cosinv[ii];
   s2 = cosinv[ii+1];
   ans = s1 + (s2 - s1) * (x - v1) * K2;
   return ans;
}

double qarcsine(double x)                                                  //  arcsine(x) for x = -1 to +1
{                                                                          //  (seg. fault if out of range)
   int            ii;
   static int     ftf = 1;
   double         v1, s1, s2, ans;
   static double  base = 1.0;
   static double  K1 = base / 1000.0;
   static double  K2 = 1000.0 / base;
   static double  v[2001], arcsinv[2001];

   if (ftf) {
      ftf = 0;
      for (ii = 0; ii < 2001; ii++)
      {
         v1 = ii * K1 - base;                                              //  -1 to +1, exactly
         v[ii] = v1;
         arcsinv[ii] = asin(v1);
      }
   }

   ii = int(K2 * (x + base));
   if (ii < 50 || ii > 1950) return asin(x);                               //  improve accuracy near 90 deg.  v.06
   v1 = v[ii];
   s1 = arcsinv[ii];
   s2 = arcsinv[ii+1];
   ans = s1 + (s2 - s1) * (x - v1) * K2;
   return ans;
}

double qsqrt(double x)                                                     //  sqrt(x) for x = 0 to 1       v.06
{                                                                          //  (seg. fault if out of range)
   int               ii;
   static int        ftf = 1;
   static double     *sqrtv;

   if (ftf) {
      ftf = 0;
      sqrtv = (double *) zmalloc(100001 * sizeof(double));
      for (ii = 0; ii < 100001; ii++)
            sqrtv[ii] = sqrt(0.00001 * ii + 0.000005);
   }

   ii = int(100000.0 * x);
   if (ii < 100) return sqrt(x);                                           //  improve accuracy near zero

   return sqrtv[ii];                                                       //  interpolation slower than sqrt()
}                                                                          //  (done in cpu microcode)


/**************************************************************************
   GTK windowing system utilities
***************************************************************************/

//  crash with error message in popup window and in stdout file

void zappcrash(char *pMess, ... )
{
   GtkWidget   *dialog;
   va_list     arglist;
   char        message[200];

   va_start(arglist,pMess);
   vsnprintf(message,199,pMess,arglist);
   va_end(arglist);

   printf("zappcrash: %s \n",message);

   dialog = gtk_message_dialog_new(null,GTK_DIALOG_MODAL,
                     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, message);
   gtk_dialog_run(GTK_DIALOG(dialog));
   gtk_widget_destroy(dialog);
   gtk_main_quit();
   exit(1);
}


//  functions to lock GTK calls from threads only

#define     tmax 10                                                        //  max. simultaneous GTK threads
pthread_t   tid_main = 0;
pthread_t   tids[tmax];
int         tlocks[tmax];
int         zinit = 0;
mutex       zmutex;                                                        //  mutex 2006.07.20

void zlockInit()                                                           //  initz. call from main()
{
   int      ii;
   tid_main = pthread_self();
   for (ii = 0; ii < tmax; ii++) tids[ii] = tlocks[ii] = 0;
   mutex_init(&zmutex,null);
   zinit++;
   return;
}

void zlock()                                                               //  lock GTK if in a thread
{
   int         ii;
   pthread_t   tid_me;

   if (zinit != 1) zappcrash("zlock(): zinit() not done");

   tid_me = pthread_self();
   if (pthread_equal(tid_main,tid_me)) return;                             //  main() thread, do nothing

   mutex_lock(&zmutex);

   for (ii = 0; ii < tmax; ii++)                                           //  find my thread slot
         if (pthread_equal(tids[ii],tid_me)) goto zret;
   for (ii = 0; ii < tmax; ii++)                                           //  find a free slot
         if (tids[ii] == 0) goto zret;
   zappcrash("zlock(): too many threads");

zret:
   tids[ii] = tid_me;
   ++tlocks[ii];
   mutex_unlock(&zmutex);

   if (tlocks[ii] == 1) gdk_threads_enter();                               //  1st lock, lock GTK
   return;
}

void zunlock()                                                             //  unlock GTK if in a thread
{
   int         ii;
   pthread_t   tid_me;

   tid_me = pthread_self();
   if (pthread_equal(tid_main,tid_me)) return;                             //  main() thread, do nothing

   for (ii = 0; ii < tmax; ii++)                                           //  find this thread
         if (pthread_equal(tids[ii],tid_me)) goto zret;
   zappcrash("zunlock(): not locked");

zret:
   --tlocks[ii];                                                           //  decrement locks
   if (tlocks[ii] == 0) {
      tids[ii] = 0;                                                        //  last lock removed, free slot
      gdk_flush();
      gdk_threads_leave();                                                 //  unlock GTK
   }
   return;
}


//  iterate main loop, but only if in main() thread

void zmainloop()
{
   pthread_t   tid_me;

   if (zinit != 1) zappcrash("zmainloop(): zinit() not done");
   tid_me = pthread_self();
   if (pthread_equal(tid_main,tid_me)) {
      while (gtk_events_pending()) 
            gtk_main_iteration(); 
   }
   return;
}


//  display message box and wait for user OK
//  *** be careful about calling this before gtk_main() is started *** 

void zmessageACK(char *pMess, ... )
{
   GtkWidget  *dialog;
   va_list     arglist;
   char        message[200];
   char        *blanks = "                           ";

   va_start(arglist,pMess);
   vsnprintf(message,200,pMess,arglist);
   va_end(arglist);
   
   if (strlen(message) < 20) strcat(message,blanks);                       //  v.13

   zlock();
   dialog = gtk_message_dialog_new(null,GTK_DIALOG_MODAL,
                    GTK_MESSAGE_INFO,GTK_BUTTONS_OK, message);
   gtk_dialog_run(GTK_DIALOG(dialog));
   gtk_widget_destroy(dialog);
   zunlock();

   return;
}


//  display message box and wait for user Yes or No response
//  *** be careful about calling this before gtk_main() is started *** 

int zmessageYN(char *pMess, ... )
{
   GtkWidget  *dialog;
   va_list     arglist;
   char        message[200];
   int         stat;

   va_start(arglist,pMess);
   vsnprintf(message,200,pMess,arglist);
   va_end(arglist);

   zlock();
   dialog = gtk_message_dialog_new(null,GTK_DIALOG_MODAL,
                     GTK_MESSAGE_QUESTION,GTK_BUTTONS_YES_NO, message);
   stat = gtk_dialog_run(GTK_DIALOG(dialog));
   gtk_widget_destroy(dialog);
   zunlock();

   if (stat == GTK_RESPONSE_YES) return 1;
   else return 0;
}


//  get text input from a popup dialog

const char * dialogText(const char *title, const char *inittext)
{
   zdialog  *zd;
   
   zd = zdialog_new(title,"OK","cancel",0);
   zdialog_add_widget(zd,"frame","fred","dialog");
   zdialog_add_widget(zd,"edit","edit","fred");
   zdialog_put_data(zd,"edit",inittext);

   int zstat = zdialog_run(zd);
   zdialog_destroy(zd);
   if (zstat != 1) return 0;
   return zdialog_get_data(zd,"edit");
}


//  write message to text view window
//  line:   +N    existing lines from top (replace)
//          -N    existing lines from bottom (replace)
//           0    next line (add new line at bottom)
//  scroll logic assumes only one \n per message

void wprintx(GtkWidget *mLog, int line, const char *message, const char *font)
{
   static GtkTextMark      *endMark = 0;
   GtkTextBuffer           *textBuff;
   GtkTextIter             iter1, iter2;
   static GtkTextTag       *fontag, *fontags[20];
   static char             *fontmem[20];
   static int              nfonts = 0;
   int                     ii, fcc, nlines, scroll = 0;

   zlock();
   
   textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mLog));
   
   endMark = gtk_text_buffer_get_mark(textBuff,"wpxend");                  //  new buffer?
   if (! endMark) {
      gtk_text_buffer_get_end_iter(textBuff,&iter1);                       //  yes, set my end mark      v.13
      endMark = gtk_text_buffer_create_mark(textBuff,"wpxend",&iter1,0);
      for (ii = 0; ii < nfonts; ii++) zfree(fontmem[ii]);                  //  free font tags memory
      nfonts = 0;
   }

   if (font) {                                                             //  v.07
      for (ii = 0; ii < nfonts; ii++) 
            if (strEqu(font,fontmem[ii])) break;                           //  use existing font tag if poss.
      if (ii == nfonts) {
         if (nfonts == 20) zappcrash("wprintx: exceed 20 font tags");
         fcc = strlen(font);                                               //  create a new font tag
         fontmem[ii] = zmalloc(fcc+1);
         strcpy(fontmem[ii],font);
         fontags[ii] = gtk_text_buffer_create_tag(textBuff,0,"font",font,0);
         nfonts++;
      }
      fontag = fontags[ii];
   }
   
   nlines = gtk_text_buffer_get_line_count(textBuff);                      //  lines now in buffer

   if (line == 0) scroll++;                                                //  auto scroll is on

   if (line < 0) {
      line = nlines + line + 1;                                            //  last lines: -1, -2 ...
      if (line < 1) line = 1;                                              //  above top, use line 1
   }
   
   if (line > nlines) line = 0;                                            //  below bottom, treat as append

   if (line == 0) gtk_text_buffer_get_end_iter(textBuff,&iter1);           //  append new line
   
   if (line > 0) {
      gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line-1);            //  old line start
      if (line < nlines)
          gtk_text_buffer_get_iter_at_line(textBuff,&iter2,line);          //  old line end
      if (line == nlines)                                                  //    or buffer end
          gtk_text_buffer_get_end_iter(textBuff,&iter2);                   
      gtk_text_buffer_delete(textBuff,&iter1,&iter2);                      //  delete old line
   }
                                                                           //  insert new line
   if (font)                                                               //    with optional font   v.07
         gtk_text_buffer_insert_with_tags(textBuff,&iter1,message,-1,fontag,null);
   else  gtk_text_buffer_insert(textBuff,&iter1,message,-1);

   if (scroll)                                                             //  scroll line into view
      gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(mLog),endMark,0,0,1,1);

   zmainloop();
   zunlock();
   return;
}

void wprintf(GtkWidget *mLog, int line, char *mess, ... )                  //  "printf" version
{
   va_list  arglist;
   char     message[1000];
   int      cc;

   va_start(arglist,mess);
   cc = vsnprintf(message,999,mess,arglist);
   va_end(arglist);

   wprintx(mLog,line,message);
   return;
}

void wprintf(GtkWidget *mLog, char *mess, ... )                            //  "printf", scrolling output
{
   va_list  arglist;
   char     message[1000];
   int      cc;

   va_start(arglist,mess);
   cc = vsnprintf(message,999,mess,arglist);                               //  v.09 stop overflow, remove warning
   va_end(arglist);

   wprintx(mLog,0,message);
   return;
}


//  clear a text view window and get a new buffer (a kind of defrag)

void wclear(GtkWidget *mLog)
{
   GtkTextBuffer  *buff;

   zlock();
   buff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mLog));
   gtk_text_buffer_set_text(buff,"",-1);
   zunlock();
   return;
}


//  clear a text view window from designated line to end of buffer

void wclear(GtkWidget *mLog, int line)
{
   GtkTextBuffer           *textBuff;
   GtkTextIter             iter1, iter2;

   zlock();

   textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mLog));
   gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line-1);               //  iter at line start
   gtk_text_buffer_get_end_iter(textBuff,&iter2);
   gtk_text_buffer_delete(textBuff,&iter1,&iter2);                         //  delete existing line

   zmainloop();
   zunlock();
   return;
}


//  get text records from a text view window, one per call
//  removes trailing new line characters ( \n )

char * wscanf(GtkWidget *mLog, int & ftf)
{
   GtkTextBuffer  *textBuff;
   GtkTextIter    iter1, iter2;
   static char    *precs = 0, *prec1, *pret;
   static int     cc;
   
   if (ftf)
   {                                                                       //  get all window text
      ftf = 0;
      if (precs) g_free(precs);                                            //  free prior memory if there
      zlock();
      textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mLog));            //  get all text
      gtk_text_buffer_get_bounds(textBuff,&iter1,&iter2);
      precs = gtk_text_buffer_get_text(textBuff,&iter1,&iter2,0);
      prec1 = precs;                                                       //  1st record
      zunlock();
   }
   
   if (! precs || (*prec1 == 0))                                           //  no more records
   {
      if (precs) g_free(precs);
      precs = 0;
      return 0;
   }

   cc = 0;
   while ((prec1[cc] != 0) && (prec1[cc] != '\n')) cc++;                   //  scan for terminator
   pret = prec1;
   prec1 = prec1 + cc;                                                     //  next record
   if (*prec1 == '\n') prec1++;
   pret[cc] = 0;                                                           //  replace \n with 0
   return pret;
}


//  dump text window into file
//  return:  0: OK  +N: error

int   wfiledump_maxcc = 0;                                                 //  v.07

int wfiledump(GtkWidget *mLog, char *filespec)
{
   FILE        *fid;
   char        *prec;
   int         ftf, err, cc;

   fid = fopen(filespec,"w");                                              //  open file
   if (! fid) { 
      zmessageACK("cannot open file %s",filespec);
      return 1; 
   }
   
   wfiledump_maxcc = 0;
   
   ftf = 1;
   while (true)
   {
      prec = wscanf(mLog,ftf);                                             //  get text line
      if (! prec) break;
      fprintf(fid,"%s\n",prec);                                            //  output with \n
      cc = strlen(prec);
      if (cc > wfiledump_maxcc) wfiledump_maxcc = cc;
   }
   
   err = fclose(fid);                                                      //  close file
   if (err) { zmessageACK("file close error"); return 2; }
   else return 0;
}


//  save text window to file, via file chooser dialog

void wfilesave(GtkWidget *mLog)
{
   int      err;
   char     *file;

   file = zgetfile("save screen to file","screen-save.txt","save");
   if (! file) return;
   err = wfiledump(mLog,file);
   if (err) zmessageACK(strerror_r(errno,file,200));
   zfree(file);
   return;
}


//  print text window to default printer
//  use landscape mode if max. print line > A4 width     v.07

void wprintp(GtkWidget *mLog)
{
   int      pid, err;
   char     tempfile[50], command[200];

   pid = getpid();
   snprintf(tempfile,49,"/tmp/wprintp-%d",pid);
   err = wfiledump(mLog,tempfile);
   if (err) return;

   if (wfiledump_maxcc < 97)                                               //  v.07
      snprintf(command,199,"lp -o %s -o %s -o %s -o %s -o %s -o %s %s",
                     "cpi=14","lpi=8","page-left=50","page-top=50",
                     "page-right=40","page-bottom=40",tempfile);

   else
      snprintf(command,199,"lp -o %s -o %s -o %s -o %s -o %s -o %s -o %s %s",
                     "landscape","cpi=14","lpi=8","page-left=50","page-top=50",
                     "page-right=40","page-bottom=40",tempfile);

   err = system(command);
   if (err) zmessageACK("print error %d",err);
   return;
}


//  File chooser dialog. Action is "open" or "save".              v.10
//  buttx: optional button "hidden" or "quality" or null
//  Memory for returned filespec should be freed via zfree().
//  "hidden" or "hidden files" button toggles display of hidden files
//  "quality" button returns optional global variable JPGquality

char        JPGquality[4] = "80";                                          //  gdk_pixbuf_save() quality (extern)

char * zgetfile(char *title, char *initfile, char *action, char *buttx)
{
   GtkWidget      *dialog;
   int            fcstat, err, fcc, bcode = 0, qnum, hide = 0;
   char           *file, *dirk, *fname, *pp, initfile2[maxfcc];
   char           *gfile, *button1 = 0;
   const char     *qual;
   struct stat    initfilestat;

   zlock();

   GtkFileChooserAction fcact = GTK_FILE_CHOOSER_ACTION_OPEN;
   button1 = "open";

   if (strEqu(action,"save")) {
      fcact = GTK_FILE_CHOOSER_ACTION_SAVE;
      button1 = "save";
   }
   
   if (buttx) {
      if (strnEqu(buttx,"hidden",6)) bcode = 103;
      if (strEqu(buttx,"quality")) bcode = 104;
   }
   
   dialog = gtk_file_chooser_dialog_new(title, null, fcact,                //  create file selection dialog
                              button1, GTK_RESPONSE_ACCEPT, 
                              "cancel", GTK_RESPONSE_CANCEL, 
                              buttx, bcode, null);

   file = dirk = fname = 0;

   if (initfile && *initfile) {                                            //  inspect initfile    v.09 simplify
      strncpy0(initfile2,initfile,maxfcc-1);
      pp = strrchr(initfile2,'/');
      if (pp) {
         *pp = 0;
         file = initfile;                                                  //  poss. whole file path
         dirk = initfile2;                                                 //  poss. directory part
         fname = pp + 1;                                                   //  poss. file name part
      }
      else fname = initfile;                                               //  file name part only
      
      err = stat(initfile,&initfilestat);                                  //  test initfile status
      if (err) file = 0;
      else if (S_ISDIR(initfilestat.st_mode)) {
         file = 0;
         dirk = initfile;                                                  //  have directory only
         fname = 0;
      }
   }

   if (file) gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(dialog),file);
   if (dirk && !file) gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog),dirk);
   if (fname && !file) gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog),fname);

   gtk_file_chooser_set_show_hidden(GTK_FILE_CHOOSER(dialog),0);           //  default: no show hidden   v.08
                                                                           //  (gtk bug: shown anyway) ***
   if (strEqu(action,"save"))
      gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(dialog),1);

   while (true)
   {
      fcstat = gtk_dialog_run(GTK_DIALOG(dialog));                         //  run dialog, get status button

      if (fcstat == 103) {                                                 //  show/hide hidden files
         hide = gtk_file_chooser_get_show_hidden(GTK_FILE_CHOOSER(dialog));
         hide = 1 - hide;
         gtk_file_chooser_set_show_hidden(GTK_FILE_CHOOSER(dialog),hide);
      }

      else if (fcstat == 104) {                                            //  get JPG quality parameter   v.10
         while (true) {
            qual = dialogText("JPG quality 0-100",JPGquality);
            if (! qual) break;                                             //  cancel = no change
            err = convSI(qual,qnum,0,100);
            if (err) continue;                                             //  enforce 0-100
            snprintf(JPGquality,4,"%d",qnum);
            break;
         }
      }

      else break;                                                          //  some other button 
   }

   if (fcstat == GTK_RESPONSE_ACCEPT) 
      gfile = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));     //  get chosen file
   else  gfile = 0;

   gtk_widget_destroy(dialog);

   if (gfile) {                                                            //  copy into own memory
      fcc = strlen(gfile);
      file = zmalloc(fcc+1);
      strcpy(file,gfile);
      g_free(gfile);                                                       //  free GTK resource
   }
   else file = 0;
   
   zunlock();
   return file;
}


/**************************************************************************
   initialize application files according to following conventions:
    - application name is from executable file name (less .x or .exe)
    - help file is appname-guide.pdf in execution directory
    - application files are in directory /home/user/.appname/
      (which is created if not already there)
    - initial application data files are in execution directory,
      and these are copied to /home/user/.appname/xxxxx
      (but only if new /home/user/.appname/ directory was created)
    - if not already attached to a terminal, directs stdout/stderr 
      to log file at /home/user/.appname/appname.log
***************************************************************************/

char     zexedirk[1000], zappdirk[200], zappname[40], zapplog[100];

void initz_appfiles(const char *appfile, ...)  //  null terminator         v.07
{
   char           command[1200], *pp;
   int            fcc, err;
   struct stat    statdat;
   va_list        arglist;

   fcc = readlink("/proc/self/exe",zexedirk,999);                          //  get /dirk-path/appname.x
   if (fcc < 3 || fcc > 997) zappcrash("/proc/self/exe absurd");
   zexedirk[fcc] = 0;
   pp = strrchr(zexedirk,'/');
   if (! pp) zappcrash("/proc/self/exe absurd");
   *pp = 0;

   strncpy0(zappname,pp+1,39);                                             //  appname
   pp = strrchr(zappname,'.');
   if (pp) *pp = 0;
   
   pp = cuserid(0);                                                        //  get /home/user/.appname/
   if (strEqu(pp,"root")) snprintf(zappdirk,199,"/root/.%s",zappname);
   else snprintf(zappdirk,199,"/home/%s/.%s",pp,zappname);
   
   err = stat(zappdirk,&statdat);                                          //  does it exist already?
   if (err) {
      snprintf(command,1199,"mkdir -m 0700 %s",zappdirk);                  //  no, create it
      err = system(command);
      if (err) printf("%s  %s \n",syserrText(err),zappdirk);
      if (err) return;
   }

   va_start(arglist,appfile);                                              //  process list of application files
   while (appfile)
   {
      snprintf(command,1199,"%s/%s",zappdirk,appfile);                     //  if application file does not already
      err = stat(command,&statdat);                                        //    exist, copy from exe directory
      if (err) {                                                           //      to /home/user/.appname/xxxxx
         snprintf(command,1199,"cp %s/%s %s",zexedirk,appfile,zappdirk);   //                         v.09
         system(command);
      }
      appfile = va_arg(arglist, const char *);
   }
   va_end(arglist);

   if (! isatty(1)) {                                                      //  if not attached to a terminal,
      snprintf(zapplog,99,"%s/%s.log",zappdirk,zappname);                  //    redirect output to log file
      freopen(zapplog,"w",stdout);                                         //      /home/user/.fotox/fotox.log
      freopen(zapplog,"w",stderr);
      printf("%s message and error log \n",zappname);
   }

   return;
}

char * getz_exedirk()                                                      //  get execution directory  v.07
{
   if (*zexedirk) return zexedirk;
   zappcrash("initz_appfiles not done");
   return 0;
}

char * getz_appdirk()                                                      //  get app directory   v.07
{
   if (*zappdirk) return zappdirk;
   zappcrash("initz_appfiles not done");
   return 0;
}

void showz_helpfile()                                                      //  launch help file in new process
{
   char     *PDFviewers[5] = { 
                  "evince --page-label=1",                                 //  alternative PDF file viewers
                  "acroread -geometry 500x600",
                  "kpdf", "xpdf", "gpdf" };

   int      ii, err;
   char     command[1000];

   for (ii = 0; ii < 5; ii++) {
      snprintf(command,999,"%s %s/%s-guide.pdf",PDFviewers[ii],zexedirk,zappname);
      err = system(command);
      if (! err) return;
   }
   
   zmessageACK("please install one of the following PDF readers:\n"
               "evince acroread kpdf xpdf gpdf");
   return;
}


/**************************************************************************
   simplified menu bar, tool bar, status bar functions     v.05
***************************************************************************/

static int     Nmtbar = 0, mtbar0 = 2754;
static int     Nmitem = 0, mitem0 = 7453;
static int     tbIconSize = 24;
static void    *wmtbar[10], *wttips[10];
static void    *wmitem[100], *wmsub[100];


//  create menu bar and add to vertical packing box

int create_menubar(GtkWidget *vbox)
{
   if (Nmtbar >= 10) zappcrash("exceed 10 menu/tool/status bars");
   wmtbar[Nmtbar] = (void *) gtk_menu_bar_new(); 
   gtk_box_pack_start(GTK_BOX(vbox),(GtkWidget *) wmtbar[Nmtbar],0,0,0);
   return mtbar0 + Nmtbar++;
}


//  add menu item to menu bar

int add_menubar_item(int mbar, const char *mname, respFunc func)
{
   mbar -= mtbar0;
   if (mbar < 0 || mbar >= Nmtbar) zappcrash("bad menu bar ID");
   if (Nmitem >= 100) zappcrash("exceeded 100 menu + tool items");
   
   wmitem[Nmitem] = (void *) gtk_menu_item_new_with_label(mname);
   gtk_menu_shell_append(GTK_MENU_SHELL(wmtbar[mbar]),(GtkWidget *) wmitem[Nmitem]);
   wmsub[Nmitem] = 0;
   if (func) G_SIGNAL(wmitem[Nmitem],"activate",func,mname);
   return mitem0 + Nmitem++;
}


//  add submenu item to menu item

int add_submenu_item(int mitem, const char *subname, respFunc func)
{
   GtkWidget   *submenu;

   mitem -= mitem0;
   if (mitem < 0 || mitem >= Nmitem) zappcrash("bad menu item");
   if (Nmitem >= 100) zappcrash("exceeded 100 menu + tool items");
   
   if (wmsub[mitem] == 0) {
      submenu = gtk_menu_new();
      gtk_menu_item_set_submenu(GTK_MENU_ITEM(wmitem[mitem]),submenu);
      wmsub[mitem] = (void *) submenu;
   }
   
   submenu = (GtkWidget *) wmsub[mitem];
   wmitem[Nmitem] = (void *) gtk_menu_item_new_with_label(subname);
   gtk_menu_shell_append(GTK_MENU_SHELL(submenu),(GtkWidget *) wmitem[Nmitem]);
   wmsub[Nmitem] = 0;
   if (func) G_SIGNAL(wmitem[Nmitem],"activate",func,subname);
   return mitem0 + Nmitem++;
}


//  create toolbar and add to vertical packing box

int create_toolbar(GtkWidget *box, int iconsize, int vert)
{
   if (Nmtbar >= 10) zappcrash("exceed 10 menu/tool/status bars");
   wmtbar[Nmtbar] = (void *) gtk_toolbar_new();
   wttips[Nmtbar] = (void *) gtk_tooltips_new();
   if (vert) gtk_toolbar_set_orientation(GTK_TOOLBAR(wmtbar[Nmtbar]),GTK_ORIENTATION_VERTICAL);
   gtk_box_pack_start(GTK_BOX(box),(GtkWidget *) wmtbar[Nmtbar],0,0,0);
   if (iconsize) tbIconSize = iconsize;
   return mtbar0 + Nmtbar++;
}


//  add toolbar button with stock icon ("gtk-quit") or custom icon ("iconfile.png")
//  icon file must be in same directory as program file (/proc/self/exe)

int add_toolbar_button(int tbar, const char *blab, const char *btip, const char *icon, respFunc func)
{
   GtkToolItem    *tbutton;
   char           iconpath[1000];
   GError         **gerror = 0;
   GdkPixbuf      *pixbuf;
   GtkWidget      *wicon;

   tbar -= mtbar0;
   if (tbar < 0 || tbar >= Nmtbar) zappcrash("bad toolbar ID");

   if (icon == 0 || *icon == 0) 
      tbutton = gtk_tool_button_new(0,0);

   else if (strnEqu(icon,"gtk-",4)) 
      tbutton = gtk_tool_button_new_from_stock(icon);

   else {
      *iconpath = 0;
      strncatv(iconpath,999,getz_exedirk(),"/",icon,0);
      pixbuf = gdk_pixbuf_new_from_file_at_scale(iconpath,tbIconSize,tbIconSize,1,gerror);
      if (pixbuf) {
         wicon = gtk_image_new_from_pixbuf(pixbuf);
         tbutton = gtk_tool_button_new(wicon,0);  }
      else tbutton = gtk_tool_button_new_from_stock("gtk-missing-image");
   }

   gtk_tool_button_set_label(GTK_TOOL_BUTTON(tbutton),blab);
   gtk_tool_item_set_tooltip(GTK_TOOL_ITEM(tbutton),GTK_TOOLTIPS(wttips[tbar]),btip,"");
   gtk_toolbar_insert(GTK_TOOLBAR(wmtbar[tbar]),GTK_TOOL_ITEM(tbutton),-1);
   G_SIGNAL(tbutton,"clicked",func,blab);

   return 1;
}


//  add toolbar button with an attached menu
//  menuID returned can be used with add_submenu_item()
//  *** GTK lib does not work, outputs no errors ***   v.09

int add_toolbar_menu(int tbar, const char *blab, const char *icon)
{
   GtkToolItem    *tbutton;
   char           iconpath[1000];
   GError         **gerror = 0;
   GdkPixbuf      *pixbuf;
   GtkWidget      *wicon, *menu;

   tbar -= mtbar0;
   if (tbar < 0 || tbar >= Nmtbar) zappcrash("bad toolbar ID");
   if (Nmitem >= 100) zappcrash("exceeded 100 menu + tool items");

   if (icon == 0 || *icon == 0) 
      tbutton = gtk_menu_tool_button_new(0,0);

   else if (strnEqu(icon,"gtk-",4)) 
      tbutton = gtk_menu_tool_button_new_from_stock(icon);

   else {
      *iconpath = 0;
      strncatv(iconpath,999,getz_exedirk(),"/",icon,0);
      pixbuf = gdk_pixbuf_new_from_file_at_scale(iconpath,tbIconSize,tbIconSize,1,gerror);
      if (pixbuf) {
         wicon = gtk_image_new_from_pixbuf(pixbuf);
         tbutton = gtk_menu_tool_button_new(wicon,0);  }
      else tbutton = gtk_menu_tool_button_new_from_stock("gtk-missing-image");
   }

   gtk_tool_button_set_label(GTK_TOOL_BUTTON(tbutton),blab);
   gtk_toolbar_insert(GTK_TOOLBAR(wmtbar[tbar]),GTK_TOOL_ITEM(tbutton),-1);
   
   menu = gtk_menu_new();
   gtk_menu_tool_button_set_menu(GTK_MENU_TOOL_BUTTON(tbutton),menu);
   wmitem[Nmitem] = (void *) menu;
   wmsub[Nmitem] = (void *) menu;
   return mitem0 + Nmitem++;
}


//  create a status bar and add to the start of a packing box              v.07

int create_stbar(GtkWidget *pbox)
{
   GtkWidget      *stbar;
   static PangoFontDescription    *fontdesc;

   stbar = gtk_statusbar_new(); 
   fontdesc = pango_font_description_from_string("Monospace 9");
   gtk_widget_modify_font(stbar,fontdesc);                                 //  v.09  *** GTK does not work ***
   gtk_box_pack_start(GTK_BOX(pbox),stbar,0,0,0);
   gtk_widget_show(stbar);

   if (Nmtbar >= 10) zappcrash("exceeded 10 menu/tool/status bars");
   wmtbar[Nmtbar] = (void *) stbar;
   return mtbar0 + Nmtbar++;
}


//  display message in status bar - callable from threads

int stbar_message(int stbar, const char *message)
{
   static int     ctx = -1;

   stbar -= mtbar0;
   if (stbar < 0 || stbar >= Nmtbar) zappcrash("bad stbar ID");
   
   zlock();
   if (ctx == -1) ctx = gtk_statusbar_get_context_id(GTK_STATUSBAR(wmtbar[stbar]),"all");
   gtk_statusbar_pop(GTK_STATUSBAR(wmtbar[stbar]),ctx);
   gtk_statusbar_push(GTK_STATUSBAR(wmtbar[stbar]),ctx,message);
   zmainloop();
   zunlock();

   return 0;
}


/**************************************************************************
   simplified dialog functions
***************************************************************************/

//  private functions for interception of widget events and dialog completion

void  zdialog_widget_event(GtkWidget *, zdialog *zd);
void  zdialog_response_event(GtkWidget *, int stat, zdialog *zd);


//  create a new zdialog dialog
//  optional arguments: list of 0-5 button names, null
//  returned dialog status:  N = button N (1 to 5)
//                          -4 = "X" destroy button

zdialog * zdialog_new(const char *title, ...)
{
   zdialog     *zd;
   GtkWidget   *dialog;
   char        *butt[5];
   int         ii;
   va_list     arglist;

   zd = (zdialog *) zmalloc(sizeof(zdialog));
   
   for (ii = 0; ii < 5; ii++) butt[ii] = 0;

   va_start(arglist,title);

   for (ii = 0; ii < 5; ii++)
   {
      butt[ii] = va_arg(arglist, char *);                                  //  get up to 5 buttons
      if (! butt[ii]) break;
   }

   va_end(arglist);

   zlock();                                                                //  create dialog box
   dialog = gtk_dialog_new_with_buttons(title,null,(GtkDialogFlags) 0,
                  butt[0],1,butt[1],2,butt[2],3,butt[3],4,butt[4],5,0);
   gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(dialog)->vbox),5);

   zunlock();

   zd->eventCB = 0;                                                        //  no user event callback
   zd->complCB = 0;                                                        //  no user dialog completion callback
   zd->zstat = 0;                                                          //  no zdialog status

   zd->widget[0].type = "dialog";                                          //  set up 1st widget = dialog
   zd->widget[0].name = "dialog";
   zd->widget[0].pname = 0;
   zd->widget[0].data = zmalloc(strlen(title)+1);
   strcpy(zd->widget[0].data,title);
   zd->widget[0].cblist = 0;
   zd->widget[0].widget = dialog;

   zd->widget[1].type = 0;                                                 //  eof - no contained widgets yet

   return zd;
}


//  add widget to existing zdialog

int zdialog_add_widget (
     zdialog *zd, const char *type, const char *name, const char *pname,   //  mandatory args
     const char *data, int scc, int homog, int expand, int space)          //  optional args (default = 0)
{
   GtkWidget      *widget = 0, *pwidget = 0;
   GtkTextBuffer  *editBuff = 0;
   const char     *ptype = 0;
   char           *pp, vdata[30];
   double         min, max, step, val;
   int            iiw, iip, kk, err;

   static PangoFontDescription    *monofont = 0;

   for (iiw = 1; zd->widget[iiw].type; iiw++);                             //  find next avail. slot
   if (iiw > 98) zappcrash("zdialog_add_widget: exceed 99 widgets");
   
   zd->widget[iiw].type = type;                                            //  init. widget struct
   zd->widget[iiw].name = name;
   zd->widget[iiw].pname = pname;
   zd->widget[iiw].data = 0;
   zd->widget[iiw].cblist = 0;
   zd->widget[iiw].scc = scc;
   zd->widget[iiw].homog = homog;
   zd->widget[iiw].expand = expand;
   zd->widget[iiw].space = space;
   zd->widget[iiw].widget = 0;

   zd->widget[iiw+1].type = 0;                                             //  new eof

   if (strcmpv(type,"dialog","hbox","vbox","hsep","vsep","frame","scrwin",
                    "label","entry","edit","button","togbutt","check",
                    "combo","radio","spin","hscale","vscale",null) == 0)
      zappcrash("zdialog, bad widget type: %s",type);

   for (iip = iiw-1; iip >= 0; iip--)                                      //  find parent (container) widget
      if (strEqu(pname,zd->widget[iip].name)) break;
   if (iip < 0) zappcrash("zdialog, no parent for widget: %s",name);

   pwidget = zd->widget[iip].widget;                                       //  parent widget, type
   ptype = zd->widget[iip].type;
   
   if (strcmpv(ptype,"dialog","hbox","vbox","frame","scrwin",null) == 0)
      zappcrash("zdialog, bad widget parent type: %s",ptype);

   zlock();
      
   if (! monofont) monofont = pango_font_description_from_string("Monospace");

   if (strEqu(type,"hbox")) widget = gtk_hbox_new(homog,space);            //  expandable container boxes
   if (strEqu(type,"vbox")) widget = gtk_vbox_new(homog,space);

   if (strEqu(type,"hsep")) widget = gtk_hseparator_new();                 //  horiz. & vert. separators
   if (strEqu(type,"vsep")) widget = gtk_vseparator_new();
         
   if (strEqu(type,"frame")) {                                             //  frame around contained widgets
      widget = gtk_frame_new(data);
      gtk_frame_set_shadow_type(GTK_FRAME(widget),GTK_SHADOW_IN);
   }

   if (strEqu(type,"scrwin")) widget = gtk_scrolled_window_new(0,0);       //  scrolled window container

   if (strEqu(type,"label")) widget = gtk_label_new(data);                 //  label (static text)

   if (strEqu(type,"entry")) {                                             //  1-line text entry
      widget = gtk_entry_new();
      if (data) gtk_entry_set_text(GTK_ENTRY(widget),data);
      if (scc) gtk_entry_set_width_chars(GTK_ENTRY(widget),scc);
      gtk_widget_modify_font(widget,monofont);
      G_SIGNAL(widget,"changed",zdialog_widget_event,zd)
   }
      
   if (strEqu(type,"edit")) {                                              //  multiline edit box
      widget = gtk_text_view_new();
      editBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(widget));
      if (data) gtk_text_buffer_set_text(editBuff,data,-1);
      gtk_text_view_set_editable(GTK_TEXT_VIEW(widget),1);
      gtk_widget_modify_font(widget,monofont);
      G_SIGNAL(editBuff,"changed",zdialog_widget_event,zd)                 //  buffer signals, not widget
   }
      
   if (strEqu(type,"button")) {                                            //  button
      widget = gtk_button_new_with_label(data);
      G_SIGNAL(widget,"clicked",zdialog_widget_event,zd)
   }

   if (strEqu(type,"togbutt")) {                                           //  toggle button
      widget = gtk_toggle_button_new_with_label(data);
      G_SIGNAL(widget,"toggled",zdialog_widget_event,zd)
   }

   if (strEqu(type,"check")) {                                             //  checkbox
      widget = gtk_check_button_new_with_label(data);
      G_SIGNAL(widget,"toggled",zdialog_widget_event,zd)
   }
      
   if (strEqu(type,"combo")) {                                             //  combo box
      widget = gtk_combo_box_entry_new_text();
      zd->widget[iiw].cblist = pvlist_create(zdcbmax);                     //  for drop-down list   v.07
      if (! is_blank_null(data)) {
         gtk_entry_set_text(GTK_ENTRY(GTK_BIN(widget)->child),data);       //  entry = initial data
         pvlist_append(zd->widget[iiw].cblist,data);                       //  add data to drop-down list
         gtk_combo_box_append_text(GTK_COMBO_BOX(widget),data);
      }
      gtk_widget_modify_font(widget,monofont);
      G_SIGNAL(widget,"changed",zdialog_widget_event,zd)
   }
      
   if (strEqu(type,"radio")) {                                             //  radio button
      for (kk = iip+1; kk <= iiw; kk++) 
         if (strEqu(zd->widget[kk].pname,pname) &&                         //  find first radio button
             strEqu(zd->widget[kk].type,"radio")) break;                   //    with same container
      if (kk == iiw) 
         widget = gtk_radio_button_new_with_label(null,data);              //  this one is first
      else 
         widget = gtk_radio_button_new_with_label_from_widget              //  not first, add to group
              (GTK_RADIO_BUTTON(zd->widget[kk].widget),data);
      G_SIGNAL(widget,"toggled",zdialog_widget_event,zd)
   }

   if (strcmpv(type,"spin","hscale","vscale",null)) {                      //  spin button or sliding scale
      pp = strField(data,'|',1); err = convSD(pp,min);                     //  locale fix    v.09
      pp = strField(data,'|',2); err += convSD(pp,max);
      pp = strField(data,'|',3); err += convSD(pp,step);
      pp = strField(data,'|',4); err += convSD(pp,val);
      if (err) { min = 0; max = 100; step = 1; val = 50; }

      if (*type == 's') {
         widget = gtk_spin_button_new_with_range(min,max,step);
         gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget),val);
      }
      if (*type == 'h') {
         widget = gtk_hscale_new_with_range(min,max,step);
         gtk_range_set_value(GTK_RANGE(widget),val);
         gtk_scale_set_draw_value(GTK_SCALE(widget),0);
      }
      if (*type == 'v') {
         widget = gtk_vscale_new_with_range(min,max,step);
         gtk_range_set_value(GTK_RANGE(widget),val);
         gtk_scale_set_draw_value(GTK_SCALE(widget),0);
      }
      G_SIGNAL(widget,"value-changed",zdialog_widget_event,zd)
      sprintf(vdata,"%g",val);
      data = vdata;
   }
      
   //  all widget types come here

   zd->widget[iiw].widget = widget;                                        //  set widget in zdialog

   if (strEqu(ptype,"hbox") || strEqu(ptype,"vbox"))                       //  add to hbox/vbox
      gtk_box_pack_start(GTK_BOX(pwidget),widget,expand,expand,space);
   if (strEqu(ptype,"frame"))                                              //  add to frame
      gtk_container_add(GTK_CONTAINER(pwidget),widget);
   if (strEqu(ptype,"scrwin"))                                             //  add to scroll window
      gtk_container_add(GTK_CONTAINER(pwidget),widget);
   if (strEqu(ptype,"dialog"))                                             //  add to dialog box
      gtk_box_pack_start(GTK_BOX(GTK_DIALOG(pwidget)->vbox),
                                       widget,expand,expand,space);
   if (data) {                                                             //  put initial data into
      zd->widget[iiw].data = zmalloc(strlen(data)+1);                      //    allocable/modifiable memory
      strcpy(zd->widget[iiw].data,data);
   }

   zunlock();
   return 0;
}


//  add widget to existing zdialog - alternative form (clearer and easier code)        v.08
//  options: "scc=nn | homog | expand | space=nn"  (all optional, any order)           v.11 change delimiter

int zdialog_add_widget(zdialog *zd, const char *type, const char *name, 
                       const char *parent, const char *data, const char *options)                                
{
   int      stat, scc = 0, homog = 0, expand = 0, space = 0, begin = 1;
   char     pname[8];
   double   pval;
   
   while (true)
   {
      stat = strParms(begin,options,pname,8,pval);
      if (stat == -1) break;
      if (stat == 1) zappcrash("bad zdialog options: %s",options);
      if (strEqu(pname,"scc")) scc = (int(pval));
      else if (strEqu(pname,"homog")) homog = 1;
      else if (strEqu(pname,"expand")) expand = 1;
      else if (strEqu(pname,"space")) space = (int(pval));
      else zappcrash("bad zdialog options: %s",options);
   }
   
   stat = zdialog_add_widget(zd,type,name,parent,data,scc,homog,expand,space);
   return stat;
}


//  resize dialog to a size greater than initial size
//  (as determined by the included widgets)

int zdialog_resize(zdialog *zd, int width, int height)
{
   zlock();
   GtkWidget *window = zd->widget[0].widget;
   gtk_window_set_default_size(GTK_WINDOW(window),width,height);
   zunlock();
   return 0;
}


//  put data into a zdialog widget

int zdialog_put_data(zdialog *zd, const char *name, const char *data)
{
   int            ii, kk, cc;
   const char     *type;
   char           *wdata;
   GtkWidget      *widget;
   GtkTextBuffer  *textBuff;
   double         val;

   for (ii = 1; zd->widget[ii].type; ii++)                                 //  find widget
      if (strEqu(zd->widget[ii].name,name)) break;
   if (! zd->widget[ii].type) return 0;
   
   type = zd->widget[ii].type;
   widget = zd->widget[ii].widget;

   wdata = zd->widget[ii].data;
   if (wdata) zfree(wdata);                                                //  free prior data memory
   zd->widget[ii].data = 0;

   if (data) {
      cc = strlen(data);
      wdata = zmalloc(cc+1);                                               //  set new data for widget
      strcpy(wdata,data);
      zd->widget[ii].data = wdata;
   }

   zlock();   

   if (strEqu(type,"label")) 
      gtk_label_set_text(GTK_LABEL(widget),data);

   if (strEqu(type,"entry")) 
      gtk_entry_set_text(GTK_ENTRY(widget),data);

   if (strEqu(type,"edit")) {
      textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(widget));
      gtk_text_buffer_set_text(textBuff,data,-1);
   }
   
   if (strcmpv(type,"togbutt","check","radio",null)) {
      kk = convSI(data,ii);                                                //  v.11
      if (kk != 0) ii = 0;
      if (ii <= 0) ii = 0; else ii = 1;
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget),ii);
   }
   
   if (strEqu(type,"spin")) {
      kk = convSD(data,val);                                               //  v.11
      if (kk != 0) val = 0.0;
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget),val);
   }
   
   if (strcmpv(type,"hscale","vscale",null)) {
      kk = convSD(data,val);                                               //  v.11
      if (kk != 0) val = 0.0;
      gtk_range_set_value(GTK_RANGE(widget),val);
   }
   
   if (strEqu(type,"combo")) {
      gtk_entry_set_text(GTK_ENTRY(GTK_BIN(widget)->child),data);          //  stuff entry box with new data
      if (! is_blank_null(data)) {
         kk = pvlist_prepend(zd->widget[ii].cblist,data,1);                //  add to drop-down list  v.07
         if (kk == 0)                                                      //  (only if unique)
            gtk_combo_box_prepend_text(GTK_COMBO_BOX(widget),data);
      }
   }

   zunlock();
   return ii;
}


//  get data from a dialog widget based on its name

const char * zdialog_get_data(zdialog *zd, const char *name)
{
   for (int ii = 1; zd->widget[ii].type; ii++)
      if (strEqu(zd->widget[ii].name,name)) return zd->widget[ii].data;
   return 0;
}


//  get GTK widget from zdialog and widget name

GtkWidget * zdialog_widget(zdialog *zd, const char *name)
{
   for (int ii = 1; zd->widget[ii].type; ii++)
   if (strEqu(zd->widget[ii].name,name)) return zd->widget[ii].widget;
   return 0;
}


//  run the zdialog
//  if modal, return after complete with status of button (OK, cancel ...)
//  if not modal, return immediately with dialog active

int zdialog_run(zdialog *zd, zdialog_event evfunc, zdialog_compl compfunc)
{
   int         ii, zstat;
   GtkWidget   *widget, *dialog;

   zlock();

   dialog = zd->widget[0].widget;
   gtk_widget_show_all(dialog);                                            //  activate dialog

   for (ii = 1; zd->widget[ii].type; ii++)                                 //  *** stop auto-selection
   {                                                                       //  (GTK "feature")
      if (strEqu(zd->widget[ii].type,"entry")) {
         widget = zd->widget[ii].widget;
         gtk_editable_set_position(GTK_EDITABLE(widget),-1);
         break;
      }

      if (strEqu(zd->widget[ii].type,"combo")) {
         widget = zd->widget[ii].widget;
         gtk_editable_set_position(GTK_EDITABLE(GTK_BIN(widget)->child),-1);
         break;
      }
   }

   if (evfunc) zd->eventCB = (void *) evfunc;                              //  link to user event callback

   if (compfunc) {
      G_SIGNAL(dialog,"response",zdialog_response_event,zd);               //  internal dialog response function
      zd->complCB = (void *) compfunc;                                     //  link to user completion callback
      zunlock();
      return 0;                                                            //  return now, dialog is non-modal
   }

   else zstat = gtk_dialog_run(GTK_DIALOG(dialog));                        //  modal dialog, return when complete
   zd->zstat = zstat;                                                      //  set zdialog status (from button)
   zunlock();
   return zstat;                                                           //  and return status
}


//  zdialog event handler - private function called when a widget is edited.
//  update data in zdialog, call user callback function (if present).
//  this function always runs in main() thread, so zlock() unnecessary.

void zdialog_widget_event(GtkWidget *widget, zdialog *zd)
{
   int               ii, nn;
   const char        *name, *type;
   char              *wdata, sdata[20];
   GtkTextView       *textView = 0;
   GtkTextBuffer     *textBuff = 0;
   GtkTextIter       iter1, iter2;
   double            dval;
   static GtkWidget  *lastwidget = 0;
   static int        cbadded = 0, wbusy = 0;

   zdialog_event  *callbackfunc = 0;                                       //  user event callback function
   
   if (wbusy) return;                                                      //  stop re-entrance from my own updates
   wbusy++;                                                                //  v.12

   for (ii = 1; zd->widget[ii].type; ii++)                                 //  find widget in zdialog
      if (zd->widget[ii].widget == widget) goto found_widget;

   for (ii = 1; zd->widget[ii].type; ii++) {                               //  failed, test if buffer
      if (strEqu(zd->widget[ii].type,"edit")) {                            //    of text view widget
         textView = GTK_TEXT_VIEW(zd->widget[ii].widget);
         textBuff = gtk_text_view_get_buffer(textView);
         if (widget == (GtkWidget *) textBuff) goto found_widget;
      }
   }

   wbusy = 0;
   return;                                                                 //  widget not found, ignore

found_widget:

   name = zd->widget[ii].name;
   type = zd->widget[ii].type;
   wdata = 0;

   if (strEqu(type,"button")) wdata = "clicked";

   if (strEqu(type,"entry"))
         wdata = (char *) gtk_entry_get_text(GTK_ENTRY(widget));
         
   if (strEqu(type,"edit")) {
      gtk_text_buffer_get_bounds(textBuff,&iter1,&iter2);
      wdata = gtk_text_buffer_get_text(textBuff,&iter1,&iter2,0);
   }

   if (strcmpv(type,"radio","check","togbutt",null)) 
   {
      nn = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
      if (nn == 0) wdata = "0";
      else wdata = "1";
   }

   if (strEqu(type,"combo"))
   {
      if (widget == lastwidget && cbadded) {
         pvlist_remove(zd->widget[ii].cblist,0);                           //  detect multiple edits (keystrokes)
         gtk_combo_box_remove_text(GTK_COMBO_BOX(widget),0);               //    and replace prior entry with new
      }
      wdata = (char *) gtk_entry_get_text(GTK_ENTRY(GTK_BIN(widget)->child));
      cbadded = 0;
      if (! is_blank_null(wdata)) {
         nn = pvlist_prepend(zd->widget[ii].cblist,wdata,1);               //  add entry to drop-down list  v.07
         if (nn == 0) {                                                    //  (only if unique)
            gtk_combo_box_prepend_text(GTK_COMBO_BOX(widget),wdata);
            cbadded = 1;
         }
      }
   }
   
   if (strEqu(type,"spin"))
   {
      dval = gtk_spin_button_get_value(GTK_SPIN_BUTTON(widget));
      sprintf(sdata,"%g",dval);
      wdata = sdata;
   }
   
   if (strcmpv(type,"hscale","vscale",null))
   {
      dval = gtk_range_get_value(GTK_RANGE(widget));
      sprintf(sdata,"%g",dval);
      wdata = sdata;
   }
   
   //  all widgets come here

   if (zd->widget[ii].data) zfree(zd->widget[ii].data);                    //  clear prior data
   zd->widget[ii].data = 0;

   if (wdata) {
      zd->widget[ii].data = zmalloc(strlen(wdata)+1);                      //  set new data
      strcpy(zd->widget[ii].data,wdata);
   }
   
   lastwidget = widget;                                                    //  remember last widget updated

   if (zd->eventCB) {
      callbackfunc = (zdialog_event *) zd->eventCB;                        //  do user callback function
      callbackfunc(zd,name);
   }

   wbusy = 0;
   return;
}


//  zdialog response handler - private function 
//  called when dialog is completed
//  status corresponds to completion button from user (OK, cancel ...)

void zdialog_response_event(GtkWidget *, int zstat, zdialog *zd)
{
   zdialog_compl  *callbackfunc = 0;

   zd->zstat = zstat;                                                      //  set zdialog status 

   if (zd->complCB) {
      callbackfunc = (zdialog_compl *) zd->complCB;                        //  do user callback function
      callbackfunc(zd,zstat);
   }

   return;
}


//  destroy the zdialog - must be done by zdialog_run() caller
//  (else dialog continues active even after completion button)

int zdialog_destroy(zdialog *zd)
{
   zlock();
   if (zd->widget[0].widget) 
         gtk_widget_destroy(zd->widget[0].widget);                         //  destroy GTK dialog
   zd->widget[0].widget = 0;
   zunlock();

   return 0;
}


//  free zdialog memory    v.07

int zdialog_free(zdialog *zd)
{
   zdialog_destroy(zd);                                                    //  destroy GTK dialog if there

   for (int ii = 1; zd->widget[ii].type; ii++)                             //  loop through widgets
   {
      if (zd->widget[ii].data) zfree(zd->widget[ii].data);                 //  free data 
      if (strEqu(zd->widget[ii].type,"combo"))                             //  if combo box, free drop-down list
         pvlist_free(zd->widget[ii].cblist);
   }
   
   zfree(zd);
   return 0;
}


//  convenience functions for stuffing and retrieving widget data

int zdialog_stuff(zdialog *zd, const char *name, const char *data)         //  stuff a string
{
   zdialog_put_data(zd, name, data);
   return 1;
}

int zdialog_stuff(zdialog *zd, const char *name, int idata)                //  stuff an integer
{
   char  string[16];

   sprintf(string,"%d",idata);
   zdialog_put_data(zd,name,string);
   return 1;
}

int zdialog_stuff(zdialog *zd, const char *name, double ddata)             //  stuff a double
{
   char  string[32];
   
   snprintf(string,31,"%g",ddata);                                         //  outputs decimal point or comma
   zdialog_put_data(zd,name,string);                                       //  (per locale)
   return 1;
}

int zdialog_fetch(zdialog *zd, const char *name, char *data, int maxcc)    //  fetch string data
{
   const char  *zdata;

   zdata = zdialog_get_data(zd,name);

   if (! zdata) {
      *data = 0;
      return 0;
   }
   
   strncpy0(data,zdata,maxcc);
   return 1;
}

int zdialog_fetch(zdialog *zd, const char *name, int &idata)               //  fetch an integer
{
   const char  *zdata;

   zdata = zdialog_get_data(zd,name);
   
   if (! zdata) {
      idata = 0;
      return 0;
   }
   
   idata = atoi(zdata);
   return 1;
}

int zdialog_fetch(zdialog *zd, const char *name, double &ddata)            //  fetch a double
{
   int         stat;
   const char  *zdata;

   zdata = zdialog_get_data(zd,name);
   
   if (! zdata) {
      ddata = 0;
      return 0;
   }
   
   stat = convSD(zdata,ddata);                                             //  period or comma decimal point OK
   if (stat < 4) return 1;
   return 0;
}

//  append new item to combo box list without changing entry box

int zdialog_cb_app(zdialog *zd, const char *name, const char *data)
{
   int         ii, nn;

   if (is_blank_null(data)) return 0;                                      //  find widget
   for (ii = 1; zd->widget[ii].type; ii++) 
      if (strEqu(zd->widget[ii].name,name)) break;
   if (! zd->widget[ii].type) return 0;                                    //  not found
   if (strNeq(zd->widget[ii].type,"combo")) return 0;                      //  not combo box

   nn = pvlist_append(zd->widget[ii].cblist,data,1);                       //  append unique   v.07
   if (nn >= 0) 
      gtk_combo_box_append_text(GTK_COMBO_BOX(zd->widget[ii].widget),data);
   return 1;
}


//  prepend new item to combo box list without changing entry box

int zdialog_cb_prep(zdialog *zd, const char *name, const char *data)
{
   int         ii, nn;

   if (is_blank_null(data)) return 0;
   for (ii = 1; zd->widget[ii].type; ii++) 
      if (strEqu(zd->widget[ii].name,name)) break;
   if (! zd->widget[ii].type) return 0;
   if (strNeq(zd->widget[ii].type,"combo")) return 0;
   nn = pvlist_prepend(zd->widget[ii].cblist,data,1);                      //  append unique v.07
   if (nn == 0) 
      gtk_combo_box_prepend_text(GTK_COMBO_BOX(zd->widget[ii].widget),data);
   return 1;
}


//  get combo box drop-down list entry                                     //  v.07

char * zdialog_cb_get(zdialog *zd, const char *name, int Nth)
{
   int      ii;

   for (ii = 1; zd->widget[ii].type; ii++) 
      if (strEqu(zd->widget[ii].name,name)) break;
   if (! zd->widget[ii].type) return 0;
   if (strNeq(zd->widget[ii].type,"combo")) return 0;
   return pvlist_get(zd->widget[ii].cblist,Nth);
}


/**************************************************************************
    parameter management functions
***************************************************************************/

struct t_parmlist {                                                        //  parameter list in memory
   int      max;                                                           //  max parameter count
   int      count;                                                         //  actual parameter count
   char     **name;                                                        //  pointer to names (list of char *)
   double   *value;                                                        //  pointer to values (list of double)
} parmlist;

int      parmlistvalid = 0;                                                //  flag
char     zparmfile[1000];                                                  //  last used parm file


//  initialize parameter list - must be called first

int initParmlist(int max)
{
   if (! parmlistvalid)                                                    //  start with default parms file
      strncatv(zparmfile,999,getz_appdirk(),"/parameters.txt",0);          //   /home/user/.appname/parameters.txt

   if (parmlistvalid) {                                                    //  delete old parms
      delete [] parmlist.name;
      delete [] parmlist.value;
   }

   parmlist.max = max;
   parmlist.count = 0;
   char **names = new char*[max];                                          //  allocate max pointers for names
   double *values = new double[max];                                       //  allocate max doubles for values
   parmlist.name = names;
   parmlist.value = values;
   parmlistvalid = 1;
   return 0;
}


//  load parameters from a file, with file selection dialog

int loadParms()
{
   char     *pfile;
   int      np;
   
   if (! parmlistvalid) zappcrash("parmlistvalid = 0");

   pfile = zgetfile("load parameters from a file",zparmfile,"open","hidden files");
   if (! pfile) return 0;
   
   np = loadParms(pfile);
   zfree(pfile);

   return np;
}
   

//  load parameters from a file
//  use default parameters.txt if file is "use default"                    //  v.07  
//  returns no. parameters loaded

int loadParms(char *pfile)
{
   FILE        *fid;
   int         Nth, np1, np2 = 0, err;
   char        buff[100], *fgs, *pp, *pname, *pvalue;
   double      dvalue;
   
   if (! parmlistvalid) zappcrash("parmlistvalid = 0");

   if (! pfile) pfile = zparmfile;                                         //  v.09
   if (strEqu(pfile,"use default")) pfile = zparmfile;
   
   if (*pfile != '/') {                                                    //  if parm file name only,
      pp = strrchr(zparmfile,'/');                                         //    make complete absolute path
      if (pp) strcpy(pp+1,pfile);                                          //      in same directory as prior
      pfile = zparmfile;
   }

   fid = fopen(pfile,"r");
   if (! fid) return loadParms();                                          //  bad file, ask user to select

   strncpy0(zparmfile,pfile,999);                                          //  set current parm file

   while (true)                                                            //  read file
   {
      fgs = fgets_trim(buff,99,fid,1);
      if (! fgs) break;                                                    //  EOF

      pp = strchr(buff,'#');                                               //  eliminate comments
      if (pp) *pp = 0;

      Nth = 1;                                                             //  parse parm name, value
      pname = strField(buff,' ',Nth++);
      if (! pname) continue;
      pvalue = strField(buff,' ',Nth);
      if (! pvalue) continue;
      err = convSD(pvalue,dvalue);                                         //  v.07
      if (err) continue;
      np1 = setParm(pname,dvalue);                                         //  set the parameter
      if (! np1) continue;
      np2++;
   }
   
   fclose(fid);                                                            //  close file
   return np2;                                                             //  return parameter count
}


//  save parameters to a file, with file selection dialog

int saveParms()
{
   char     *pfile;
   int      np;

   if (! parmlistvalid) zappcrash("parmlistvalid = 0");

   pfile = zgetfile("save parameters to a file",zparmfile,"save","hidden files");
   if (! pfile) return 0;

   np = saveParms(pfile);
   zfree(pfile);

   return np;
}


//  save parameters to a file

int saveParms(char *pfile)
{
   FILE     *fid;
   int      np;

   if (! parmlistvalid) zappcrash("parmlistvalid = 0");

   fid = fopen(pfile,"w");
   if (! fid) { 
      zmessageACK("cannot open: %s",pfile);
      return 0;
   }

   strncpy0(zparmfile,pfile,999);

   for (np = 0; np < parmlist.count; np++)
      fprintf(fid," \"%s\"  %.12g \n",parmlist.name[np],parmlist.value[np]);
   
   fclose(fid);
   return np;
}


//  create a new paramater or change value of existing parameter

int setParm(const char *parmname, double parmval)
{
   int      ii, cc;
   char     *ppname;

   if (! parmlistvalid) zappcrash("parmlistvalid = 0");

   for (ii = 0; ii < parmlist.count; ii++)
      if (strEqu(parmlist.name[ii],parmname)) break;

   if (ii == parmlist.max) return 0;

   if (ii == parmlist.count) {
      parmlist.count++;
      cc = strlen(parmname);
      ppname = new char[cc+1];
      strcpy(ppname,parmname);
      parmlist.name[ii] = ppname;
   }

   parmlist.value[ii] = parmval;
   return parmlist.count;
}


//  get parameter value from parameter name

double getParm(const char *parmname)
{
   if (! parmlistvalid) zappcrash("parmlistvalid = 0");

   for (int ii = 0; ii < parmlist.count; ii++)
   {
      if (strNeq(parmlist.name[ii],parmname)) continue;
      return parmlist.value[ii];
   }

   return NAN;
}


//  get Nth parameter name (zero-based)

char * getParm(int Nth)
{
   if (! parmlistvalid) zappcrash("parmlistvalid = 0");
   if (Nth >= parmlist.count) return null;
   return parmlist.name[Nth];
}


//  list parameters in supplied text entry window

int listParms(GtkWidget *textWin)
{
   int            ii;
   const char     *pname;
   double         pvalue;

   for (ii = 0; ii < parmlist.count; ii++)
   {
      pname = getParm(ii);
      pvalue = getParm(pname);
      wprintf(textWin," %s  %.12g \n",pname,pvalue);
   }
   
   return parmlist.count;
}


//  edit parameters with a GUI
//  textWin != null enables button to list parameters in window
//  addp != 0 enables button to add new parameters

int editParms(GtkWidget *textWin, int addp)
{
   GtkWidget      *peDialog, *peLabel[100], *peEdit[100], *peHbox[100];
   char           ptemp[20];
   const char     *pchval, *pname;
   double         pvalue;
   int            ii, err, iie = -1, stat;
   
   if (! parmlistvalid) zappcrash("parmlistvalid = 0");

   build_dialog:                                                           //  build parameter edit dialog

   if (parmlist.count > 100) zappcrash("more than 100 parameters");
   
   if (textWin && addp) peDialog = gtk_dialog_new_with_buttons
                        ("edit parameters", null, GTK_DIALOG_MODAL, 
                         "done",2,  "load\nfile",3, "save\nfile",4, 
                         "list\nall",5, "add\nnew",6, null);

   else if (textWin) peDialog = gtk_dialog_new_with_buttons
                        ("edit parameters", null, GTK_DIALOG_MODAL, 
                         "done",2,  "load\nfile",3, "save\nfile",4, 
                         "list\nall",5, null);

   else if (addp) peDialog = gtk_dialog_new_with_buttons
                        ("edit parameters", null, GTK_DIALOG_MODAL, 
                         "done",2,  "load\nfile",3, "save\nfile",4, 
                         "add\nnew",6, null);

   else peDialog = gtk_dialog_new_with_buttons
                        ("edit parameters", null, GTK_DIALOG_MODAL, 
                         "done",2,  "load\nfile",3, "save\nfile",4, 
                          null);

   for (ii = 0; ii < parmlist.count; ii++)                                 //  labels and edit boxes side by side
   {                                                                       //  (parm names and parm values)
      peLabel[ii] = gtk_label_new(parmlist.name[ii]);
      gtk_misc_set_alignment(GTK_MISC(peLabel[ii]),1,0.5);
      gtk_label_set_width_chars(GTK_LABEL(peLabel[ii]),30);
      peEdit[ii] = gtk_entry_new();
      gtk_entry_set_width_chars(GTK_ENTRY(peEdit[ii]),12);
      sprintf(ptemp,"%.12g",parmlist.value[ii]);
      gtk_entry_set_text(GTK_ENTRY(peEdit[ii]),ptemp);
      peHbox[ii] = gtk_hbox_new(0,0);
      gtk_box_pack_start(GTK_BOX(peHbox[ii]),peLabel[ii],0,0,5);
      gtk_box_pack_start(GTK_BOX(peHbox[ii]),peEdit[ii],0,0,5);
      gtk_box_pack_start(GTK_BOX(GTK_DIALOG(peDialog)->vbox),peHbox[ii],1,1,2);
   }

   run_dialog:                                                             //  display dialog and get inputs
   
   if (iie > -1)
   {
      gtk_editable_select_region(GTK_EDITABLE(peEdit[iie]),0,-1);          //  focus on new or bad parameter
      gtk_widget_grab_focus(peEdit[iie]);
      iie = -1;
   }

   gtk_widget_show_all(peDialog);
   stat = gtk_dialog_run(GTK_DIALOG(peDialog));

   if (stat < 2)                                                           //  kill, cancel
   {
      gtk_widget_destroy(peDialog);
      return parmlist.count;
   }
   
   if (stat == 3)                                                          //  load from file
   {
      loadParms();
      gtk_widget_destroy(peDialog);
      goto build_dialog;
   }
   
   for (ii = 0; ii < parmlist.count; ii++)                                 //  capture inputs and check if OK
   {
      pchval = gtk_entry_get_text(GTK_ENTRY(peEdit[ii]));
      err = convSD(pchval,pvalue);
      if (err && iie < 0) iie = ii;                                        //  remember 1st error
      parmlist.value[ii] = pvalue;
   }

   if (iie >= 0) goto run_dialog;                                          //  re-get bad input

   if (stat == 2)                                                          //  done
   {
      gtk_widget_destroy(peDialog);
      return parmlist.count;
   }

   if (stat == 4)                                                          //  save to file
   {
      saveParms();
      goto run_dialog;
   }
   
   if (stat == 5)                                                          //  list parameters
   {
      listParms(textWin);
      goto run_dialog;
   }
   
   if (stat == 6)                                                          //  add parameter
   {
      pname = dialogText("add parameter","(new parm name)");
      if (! pname) goto run_dialog;
      setParm(pname,0.0);
      iie = parmlist.count - 1;                                            //  focus on new parm
      gtk_widget_destroy(peDialog);
      goto build_dialog;
   }
   
   gtk_widget_destroy(peDialog);                                           //  unknown status
   return 0;
}


/**************************************************************************
   GDK graphics utilities
***************************************************************************/

/**************************************************************************

     GdkPixbuf * gdk_pixbuf_rotate(GdkPixbuf *pixbuf, double angle)

     Rotate a pixbuf through an arbitrary angle (degrees).

     The returned image has the same size as the original, but the
     pixbuf envelope is increased to accomodate the rotated original
     (e.g. a 100x100 pixbuf rotated 45 deg. needs a 142x142 pixbuf).

     The space added around the rotated image is black (RGB 0,0,0).
     Angle is in degrees. Positive direction is clockwise.
     Pixbuf must have 8 bits per channel and 3 or 4 channels.
     Loss of resolution is < 1 pixel.
     Speed is about 2.4 million pixels/sec. for a 2 GHz CPU.
     
     NULL is returned if the function fails for one of the following:
         - pixbuf not 8 bits/channel or < 3 channels
         - unable to create output pixbuf (lack of memory?)
    
     Algorithm:
         create output pixbuf big enough for rotated input pixbuf
         loop all output pixels
            get next output pixel (px2,py2)
            compute (R,theta) from center of pixbuf
            rotate theta by -angle
            (R,theta) is now within the closest input pixel
            convert to input pixel (px1,py1)
            if outside of pixmap
               output pixel = black
               continue
            for 4 input pixels based at (px0,py0) = (int(px1),int(py1))
               compute overlap (0 to 1) with (px1,py1)
               sum RGB values * overlap
            output aggregate RGB to pixel (px2,py2)

***************************************************************************/

GdkPixbuf * gdk_pixbuf_rotate(GdkPixbuf *pixbuf1, double angle)
{
   typedef unsigned char  *pixel;                                          //  3 RGB values, 0-255 each

   GdkPixbuf      *pixbuf2;
   GdkColorspace  color;

   int      nch, nbits, alpha;
   int      pbW1, pbH1, pbR1, pbW2, pbH2, pbR2;
   int      px2, py2, px0, py0;
   pixel    ppix1, ppix2, pix0, pix1, pix2, pix3;
   double   rx1, ry1, rx2, ry2, R, theta, px1, py1;
   double   f0, f1, f2, f3, red, green, blue;
   double   pi = 3.141592654;

   nch = gdk_pixbuf_get_n_channels(pixbuf1);
   nbits = gdk_pixbuf_get_bits_per_sample(pixbuf1);
   if (nch < 3) return 0;                                                  //  must have 3+ channels (colors)
   if (nbits != 8) return 0;                                               //  must be 8 bits per channel

   color = gdk_pixbuf_get_colorspace(pixbuf1);                             //  get input pixbuf1 attributes
   alpha = gdk_pixbuf_get_has_alpha(pixbuf1);
   pbW1 = gdk_pixbuf_get_width(pixbuf1);
   pbH1 = gdk_pixbuf_get_height(pixbuf1);
   pbR1 = gdk_pixbuf_get_rowstride(pixbuf1);

   while (angle < -180) angle += 360;                                      //  normalize, -180 to +180
   while (angle > 180) angle -= 360;
   angle = angle * pi / 180;                                               //  radians, -pi to +pi

   pbW2 = int(pbW1*fabs(cos(angle)) + pbH1*fabs(sin(angle)));              //  rectangle containing rotated image
   pbH2 = int(pbW1*fabs(sin(angle)) + pbH1*fabs(cos(angle)));

   pixbuf2 = gdk_pixbuf_new(color,alpha,nbits,pbW2,pbH2);                  //  create output pixbuf2
   if (! pixbuf2) return 0;
   pbR2 = gdk_pixbuf_get_rowstride(pixbuf2);
   
   ppix1 = gdk_pixbuf_get_pixels(pixbuf1);                                 //  input pixel array
   ppix2 = gdk_pixbuf_get_pixels(pixbuf2);                                 //  output pixel array
   
   for (px2 = 0; px2 < pbW2; px2++)                                        //  loop through output pixels
   for (py2 = 0; py2 < pbH2; py2++)
   {
      rx2 = px2 - 0.5 * pbW2;                                              //  (rx2,ry2) = center of pixel
      ry2 = py2 - 0.5 * pbH2;
      R = sqrt(rx2*rx2 + ry2*ry2);                                         //  convert to (R,theta)
      if (R < 0.1) theta = 0;
      else theta = qarcsine(ry2 / R);                                      //  quick arc sine
      if (rx2 < 0) {
         if (theta < 0) theta = - pi - theta;                              //  adjust for quandrant
         else theta = pi - theta;
      }

      theta = theta - angle;                                               //  rotate theta backwards
      if (theta > pi) theta -= 2 * pi;                                     //  range -pi to +pi
      if (theta < -pi) theta += 2 * pi;

      rx1 = R * qcosine(theta);                                            //  quick cosine, sine
      ry1 = R * qsine(theta);
      px1 = rx1 + 0.5 * pbW1;                                              //  (px1,py1) = corresponding
      py1 = ry1 + 0.5 * pbH1;                                              //    point within input pixels

      px0 = int(px1);                                                      //  pixel containing (px1,py1)
      py0 = int(py1);
      
      if (px1 < 0 || px0 >= pbW1-1 || py1 < 0 || py0 >= pbH1-1) {          //  if outside input pixel array
         pix2 = ppix2 + py2 * pbR2 + px2 * nch;                            //    output is black
         pix2[0] = pix2[1] = pix2[2] = 0;
         continue;
      }

      pix0 = ppix1 + py0 * pbR1 + px0 * nch;                               //  4 input pixels based at (px0,py0)
      pix1 = pix0 + pbR1;
      pix2 = pix0 + nch;
      pix3 = pix0 + pbR1 + nch;

      f0 = (px0+1 - px1) * (py0+1 - py1);                                  //  overlap of (px1,py1)
      f1 = (px0+1 - px1) * (py1 - py0);                                    //    in each of the 4 pixels
      f2 = (px1 - px0) * (py0+1 - py1);
      f3 = (px1 - px0) * (py1 - py0);
   
      red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];   //  sum the weighted inputs
      green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
      blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
      
      pix2 = ppix2 + py2 * pbR2 + px2 * nch;                               //  output pixel
      pix2[0] = int(red);
      pix2[1] = int(green);
      pix2[2] = int(blue);
   }
      
   return pixbuf2;
}


/**************************************************************************

   functions for navigtion of image files in a directory    v.08
      - get first or last image, prior or next image
      - create a window of thumbnail images
      - use thumbnail window to navigate and select images

**************************************************************************/

namespace image_navi {

   #define flimit 10000                                                    //  max image files in one directory
   #define indexfont "sans 8"                                              //  font for thumbnail labels
   #define thumbfilesize 256                                               //  thumbnail file image size

   #define nodither GDK_RGB_DITHER_NONE,0,0
   #define interp GDK_INTERP_BILINEAR
   #define colorspace GDK_COLORSPACE_RGB

   char        dirkx[maxfcc];                                              //  image directory
   int         nfiles = 0;                                                 //  image file count
   int         cxfile = 0;                                                 //  current image file 1...nfiles
   char        **flist = 0;                                                //  image file list
   txfunc      *xfunc;                                                     //  callback function for clicked image

   GtkWidget   *windx = 0, *vboxx, *dwindx;                                //  thumbnail index and drawing window
   GdkGC       *gdkgc = 0;                                                 //  graphics context
   GError      **gerror = 0;

   char        *filex = 0;                                                 //  index window anchor file
   int         xwinW, xwinH;                                               //  index window size (dwindx)
   int         thumbsize = thumbfilesize;                                  //  thumbnail image <= thumbnail file
   int         thumbW, thumbH;                                             //  index window thumbnail cell size
   int         xrows, xcols;                                               //  index window thumbnail rows and cols
   int         xmargW, xmargH;                                             //  cell margin from left and top edge

   void  menufuncx(GtkWidget *win, const char *menu);                      //  menu function, index window
   void  windx_paint();                                                    //  index window paint function
   void  windx_destroy();                                                  //  index window destroy event function
   void  draw_xtext(GtkWidget *win, char *text, int x, int y);             //  draw text in index window
   void  mouse_xevent(GtkWidget *, GdkEventButton *, void *);              //  index window mouse event function
   GdkPixbuf * get_xthumb(char *filename);                                 //  get thumbnail for index window
}

using namespace image_navi;

/**************************************************************************

   char * track_image_files(char *filez, char *action, int Nth)      v.08

   get (prev/next) image file in the same directory as given filez
   action: init      initialize file list, return 0
           add       add file to list (in sort order), return 0
           remove    remove file from list, return 0
           prev      return previous file
           next      return next file
           first     return first file
           last      return last file
   if Nth > 0 then get Nth prev/next file (fotox thumbnail page)
   returned file belongs to caller and is a subject for zfree()

***************************************************************************/

char * track_image_files(char *filez, char *action, int Nth)
{
   char           buff[1000], *pp, *file2;
   int            err, eof, ii, nn, cstat;
   struct stat    statbuf;

   cxfile = 0;                                                             //  track current file, unknown

   if (Nth < 0) return 0;   
   if (Nth == 0) Nth = 1;
   
   if (strEqu(action,"init")) {                                            //  init
      for (int ii = 0; ii < nfiles; ii++) 
            zfree(flist[ii]);                                              //  free memory for prior list
      if (flist) zfree(flist);
      flist = 0;
      nfiles = 0;

      flist = (char **) zmalloc(flimit * sizeof(char *));                  //  list of file pointers < flimit

      strncpy0(dirkx,filez,maxfcc-2);                                      //  get directory of given file

      err = stat(dirkx,&statbuf);
      if (err) {
         pp = strrchr(dirkx,'/');                                          //  bad file, check directory part  
         if (! pp) return 0;
         pp[1] = 0;
         err = stat(dirkx,&statbuf);
         if (err) return 0;
      }

      if (! S_ISDIR(statbuf.st_mode)) {                                    //  if not directory, get directory part
         pp = strrchr(dirkx,'/');                                          //  v.09
         if (! pp) return 0;
         pp[1] = 0;
      }
      
      snprintf(buff,999,"find \"%s\" -maxdepth 1 -type f",dirkx);          //  find all files in this directory
      err = createProc(buff);
      if (err) return 0;
      
      while (true)
      {
         eof = getProcOutput(buff,999,cstat);                              //  next file
         if (eof) break;
         pp = strrchr(buff,'.');
         if (! pp) continue;
         pp = strcasestr(".jpeg .jpg .png .tif .tiff .bmp .gif",pp);       //  screen for image file     v.09
         if (! pp) continue;
         flist[nfiles] = zmalloc(strlen(buff)+1);
         strcpy(flist[nfiles],buff);                                       //  add to file list
         nfiles++;
         if (nfiles == flimit) zappcrash("more than %d files",flimit);
      }

      if (nfiles == 0) return 0;

      HeapSort(flist,nfiles,strcasecmp);                                   //  Heap Sort - pointers to strings
   }
   
   for (ii = 0, nn = 1; ii < nfiles; ii++) {                               //  search for filez in file list
      nn = strcasecmp(flist[ii],filez);                                    //  = 0: filez = flist[ii]
      if (nn >= 0) break;                                                  //  > 0: flist[ii-1] < filez < flist[ii]
   }

   cxfile = ii + 1;                                                        //  track current file, 1-nfiles
   if (nfiles == 0) cxfile = 0;
   
   if (strEqu(action,"init")) return 0;

   if (strEqu(action,"add")) {                                             //  add new file to list
      if (nn == 0) return 0;                                               //  same as old file
      if (! samedirk(filez,dirkx)) return 0;                               //  in another directory
      if (nfiles == flimit) return 0;                                      //  no room
      for (nn = nfiles; nn > ii; nn--)
         flist[nn] = flist[nn-1];                                          //  make a hole
      flist[ii] = zmalloc(strlen(filez)+1);
      strcpy(flist[ii],filez);                                             //  insert file
      nfiles++;
      return 0;
   }

   if (strEqu(action,"remove")) {                                          //  remove file from list
      if (nn != 0) return 0;                                               //  not found
      zfree(flist[ii]);                                                    //  remove file
      for (nn = ii; nn < nfiles-1; nn++)
         flist[nn] = flist[nn+1];                                          //  close hole
      nfiles--;
      return 0;
   }
   
   if (nfiles == 0) return 0;                                              //  no first/last/prev/next

   if (strEqu(action,"first")) ii = 0;                                     //  first

   if (strEqu(action,"last")) ii = nfiles-1;                               //  last
   
   if (strEqu(action,"prev")) {                                            //  prev
      if (ii == 0) return 0;                                               //  filez <= first in list
      ii -= Nth;
      if (ii < 0) ii = 0;
   }
   
   if (strEqu(action,"next")) {                                            //  next
      if (ii == nfiles-1) return 0;                                        //  filez >= last in list
      if (nn == 0) ii += Nth;
      if (ii > nfiles-1) ii = nfiles-1;
   }
   
   cxfile = ii + 1;                                                        //  track current file, 1-nfiles

   file2 = zmalloc(strlen(flist[ii])+1);                                   //  copy file into new memory
   strcpy(file2,flist[ii]);
   return file2;                                                           //  return file
}


//  get current image position and total count

void get_image_counts(int &posn, int &count)
{
   posn = cxfile;
   count = nfiles;
   return;
}


/**************************************************************************

   void thumbnail_index(char *file, char *action, txfunc func)             v.08
      - make window of thumbnails starting with file
      - handle window buttons (up row, down page, etc.)
      - call func() when thumbnail is clicked
      - action: init: create or refresh existing window
              update: refresh existing window (or do nothing)
   void func(char *file)
      - receives filename of selected thumbnail

***************************************************************************/

void thumbnail_index(char *filez, char *action, txfunc func)
{
   int            tbarx, err;
   struct stat    statbuf;

   if (func) xfunc = func;                                                 //  save callback function

   if (filez) {
      if (! samedirk(filez,filex)) {
         track_image_files(filez,"init");                                  //  if new directory, get new file list
         if (filex) zfree(filex);
         filex = 0;                                                        //  no anchor file yet
      }
      err = stat(filez,&statbuf);
      if (! err) {                                                         //  if input file OK,  
         if (filex) zfree(filex);                                          //    set new anchor file
         filex = zmalloc(strlen(filez)+1);
         strcpy(filex,filez);
      }
   }

   if (strEqu(action,"update") && ! windx) return;                         //  no window, do nothing

   if (windx) {
      windx_paint();                                                       //  repaint existing window
      if (strEqu(action,"init"))
         gtk_window_present(GTK_WINDOW(windx));                            //  bring window to top
      return;
   }

   windx = gtk_window_new(GTK_WINDOW_TOPLEVEL);                            //  create index window
   gtk_window_set_position(GTK_WINDOW(windx),GTK_WIN_POS_CENTER);
   gtk_window_set_default_size(GTK_WINDOW(windx),1100,660);

   vboxx = gtk_vbox_new(0,0);                                              //  vertical packing box
   gtk_container_add(GTK_CONTAINER(windx),vboxx);                          //  add to main window

   tbarx = create_toolbar(vboxx,48);                                       //  add toolbar and buttons

   add_toolbar_button(tbarx,"change\n size","change thumbnail image size",0,menufuncx);
   add_toolbar_button(tbarx,"prev row","previous row","icons/prev-row.png",menufuncx);
   add_toolbar_button(tbarx,"next row","next row","icons/next-row.png",menufuncx);
   add_toolbar_button(tbarx,"prev page","previous page","icons/prev-page.png",menufuncx);
   add_toolbar_button(tbarx,"next page","next page","icons/next-page.png",menufuncx);
   add_toolbar_button(tbarx,"first file","jump to first file","icons/goto-top.png",menufuncx);
   add_toolbar_button(tbarx,"last file","jump to last file","icons/goto-end.png",menufuncx);
   add_toolbar_button(tbarx,"any file","jump to any file or directory","icons/open.png",menufuncx);

   dwindx = gtk_drawing_area_new();                                        //  add drawing window
   gtk_container_add(GTK_CONTAINER(vboxx),dwindx);                         //  add to main window

   gtk_widget_show_all(windx);                                             //  show all widgets (will paint)

   gdkgc = gdk_gc_new(dwindx->window);                                     //  initz. graphics context

   G_SIGNAL(windx,"destroy",windx_destroy,0)                               //  connect window events
   G_SIGNAL(dwindx,"expose-event",windx_paint,0)

   gtk_widget_add_events(dwindx,GDK_BUTTON_PRESS_MASK);                    //  connect mouse events
   G_SIGNAL(dwindx,"button-press-event",mouse_xevent,0)
   
   return;
}


//  private function
//  paint index window - draw all thumbnail images that can fit

void image_navi::windx_paint()
{
   GdkPixbuf      *pxbT;
   int            x, y, row, col, err;
   char           *fileC, *fileN, *pp, *fname;
   struct stat    statbuf;
   
   gtk_window_set_title(GTK_WINDOW(windx),dirkx);
   gdk_window_clear(dwindx->window);

   xwinW = dwindx->allocation.width;                                       //  curr. index window size
   xwinH = dwindx->allocation.height;

   thumbW = thumbsize + 10;                                                //  thumbnail cell size
   thumbH = thumbsize + 30;

   xmargW = xmargH = 5;                                                    //  edge margins
   
   xrows = int(0.5 + 1.0 * xwinH / thumbH);                                //  get thumbnail rows and cols that
   xcols = int(0.4 + 1.0 * xwinW / thumbW);                                //    (almost) fit in window     v.09
   if (xrows < 1) xrows = 1;
   if (xcols < 1) xcols = 1;

   if (! filex) return;                                                    //  check anchor file      v.09
   err = stat(filex,&statbuf);
   if (err) return;

   if (S_ISDIR(statbuf.st_mode)) {                                         //  if directory, anchor = first file
      fileN = track_image_files(filex,"first");                            //  v.09
      if (! fileN) return;
      zfree(filex);
      filex = fileN;
   }

   fileC = zmalloc(strlen(filex)+2);
   strcpy(fileC,filex);

   for (row = 0; row < xrows; row++)                                       //  draw thumbnails
   for (col = 0; col < xcols; col++)
   {
      x = col * thumbW + xmargW;
      y = row * thumbH + xmargH;

      pp = strrchr(fileC,'/');                                             //  draw file name 
      if (pp) fname = pp + 1;
      else fname = fileC;
      draw_xtext(dwindx,fname,x,y);

      pxbT = get_xthumb(fileC);                                            //  get thumbnail
      if (pxbT) {
         gdk_draw_pixbuf(dwindx->window,0,pxbT,0,0,x,y+20,-1,-1,nodither); //  draw thumbnail
         g_object_unref(pxbT);
      }

      fileN = track_image_files(fileC,"next");                             //  get next image file
      zfree(fileC);
      if (! fileN) return;
      fileC = fileN;
   }

   return;
}


//  private function
//  Get thumbnail image for given file, from cache if found.
//  Generate and add missing thumbnails to thumbnail cache.
//  Returned thumbnail belongs to caller: g_object_unref() necessary.

GdkPixbuf * image_navi::get_xthumb(char *fpath)
{
   char              bpath[maxfcc], *pfile;
   const char        *pthumb;
   GdkPixbuf         *thumbpxb, *temppxb;
   int               err, size, sizew, sizeh;
   struct stat       statf, statb;
   
   strcpy(bpath,fpath);                                                    //  image file full path
   pfile = strrchr(bpath,'/');
   if (! pfile) return 0;

   err = stat(fpath,&statf);
   if (err) return 0;

   pfile++;                                                                //  file name part
   pthumb = strHash2(pfile,12);                                            //  thumbnail name xxxx... from filename
   strcpy(pfile,".thumbnails/");                                           //  construct thumbnail file name
   strcpy(pfile + 12,pthumb);                                              //  /image dirk/.thumbnails/xxxx...png
   strcpy(pfile + 24,".png");
   zfree((void *) pthumb);

   size = thumbfilesize;   

   thumbpxb = gdk_pixbuf_new_from_file_at_size(bpath,size,size,gerror);    //  get thumbnail /.thumbnails/****.png
   if (thumbpxb) {
      err = stat(bpath,&statb);                                            //  found, compare date to image date
      if (err || (statb.st_mtime < statf.st_mtime)) {
         g_object_unref(thumbpxb);                                         //  stale - regenerate
         thumbpxb = 0;
      }
   }

   if (! thumbpxb) {                                                       //  generate thumbnail from image file
      thumbpxb = gdk_pixbuf_new_from_file_at_size(fpath,size,size,gerror);
      if (! thumbpxb) return 0;                                            //  hopeless
      gdk_pixbuf_save(thumbpxb,bpath,"png",gerror,null);                   //  save to thumbnail cache (if present)
   }

   if (thumbsize == thumbfilesize) return thumbpxb;                        //  return thumbnail if right size

   sizew = gdk_pixbuf_get_width(thumbpxb) * thumbsize / thumbfilesize;     //  rescale to right size     v.09
   sizeh = gdk_pixbuf_get_height(thumbpxb) * thumbsize / thumbfilesize;
   temppxb = gdk_pixbuf_scale_simple(thumbpxb,sizew,sizeh,interp);
   g_object_unref(thumbpxb);
   return temppxb;
}


//  private function
//  write text for thumbnail limited by width of thumbnail

void image_navi::draw_xtext(GtkWidget *win, char *text, int x, int y)
{
   static PangoFontDescription   *pfont = 0;
   static PangoLayout            *playout = 0;

   int            ww, hh, cc;
   char           text2[100];

   if (! pfont) {
      pfont = pango_font_description_from_string(indexfont);
      playout = gtk_widget_create_pango_layout(win,0);
      pango_layout_set_font_description(playout,pfont);
   }

   strncpy0(text2,text,99);
   pango_layout_set_text(playout,text2,-1);
   pango_layout_get_pixel_size(playout,&ww,&hh);

   if (ww > thumbsize) {                                                   //  size > thumbnail width
      cc = strlen(text2) * thumbsize / ww;                                 //  ratio cc down
      text2[cc] = 0;
      pango_layout_set_text(playout,text2,-1);
   }

   gdk_draw_layout(win->window,gdkgc,x,y,playout);
   return;
}


//  private function
//  index window destroy event - track if window is active or not

void image_navi::windx_destroy()
{
   windx = 0;                                                              //  no window
   if (filex) zfree(filex);                                                //  no anchor file
   filex = 0;
   return;
}


//  private function
//  menu function for index window - scroll window as requested

void image_navi::menufuncx(GtkWidget *win, const char *menu)
{
   char     *filez, *action = 0;
   int      count = 0;
   int      size1 = thumbfilesize, size2 = size1/2;

   if (strEqu(menu,"change\n size"))  {
      if (thumbsize == size2) thumbsize = size1;
      else thumbsize = int(thumbsize * 0.707);                             //  1/2 size in 2 steps
      if (thumbsize < size2) thumbsize = size2;                            //  (256 >> 180 >> 128)
      windx_paint();
      return;
   }

   if (strEqu(menu,"any file")) {
      filez = zgetfile("select new file",filex,"open");                    //  file chooser dialog
      if (filez) {
         if (! samedirk(filez,filex))                                      //  if directory changed, 
            track_image_files(filez,"init");                               //    get new file list
         zfree(filex);
         filex = filez;                                                    //  set new anchor file
         windx_paint();
      }
      return;
   }

   if (! filex) return;                                                    //  current anchor file

   if (strEqu(menu,"prev row")) {
      action = "prev";
      count = xcols;
   }

   if (strEqu(menu,"next row")) {
      action = "next";
      count = xcols;
   }

   if (strEqu(menu,"prev page")) {
      action = "prev";
      count = xcols * xrows;
   }

   if (strEqu(menu,"next page")) {
      action = "next";
      count = xcols * xrows;
   }

   if (strEqu(menu,"first file")) {
      action = "first";
      count = 0;
   }

   if (strEqu(menu,"last file")) {
      action = "last";
      count = 0;
   }

   filez = track_image_files(filex,action,count);                          //  jump to file at prev/next row/page
   if (! filez) return;
   zfree(filex);
   filex = filez;                                                          //  new anchor file
   
   windx_paint();
   return;
}


//  private function
//  mouse event function for index window - get selected thumbnail and file

void image_navi::mouse_xevent(GtkWidget *, GdkEventButton *event, void *)
{
   int      mousex, mousey;
   int      row, col, Nth;
   char     *filez;

   mousex = int(event->x);
   mousey = int(event->y);

   row = (mousey - xmargH) / thumbH;                                       //  find selected row, col
   col = (mousex - xmargW) / thumbW;

   if (row < 0 || row >= xrows) return;
   if (col < 0 || col >= xcols) return;
   
   Nth = xcols * row + col;

   if (Nth == 0) {
      filez = zmalloc(strlen(filex)+1);                                    //  anchor file selected
      strcpy(filez,filex);
   }
   else {
      filez = track_image_files(filex,"next",Nth);                         //  else get Nth file after anchor
      if (! filez) return;
   }
   
   if (xfunc) xfunc(filez);
   return;
}


/**************************************************************************
      xstring class (dynamic length string)
***************************************************************************/

#define  wmiv  1648734981

int   xstring::tcount = 0;                                                 //  initz. static members
int   xstring::tmem = 0;


xstring::xstring(int cc)                                                   //  new xstring(cc)
{
   wmi = wmiv;
   xmem = (cc & 0x7ffffff8) + 8;                                           //  mod 8 length
   xpp = new char[xmem];                                                   //  allocate
   if (! xpp) appcrash("xstring NEW failure",null);
   tcount++;                                                               //  incr. object count
   tmem += xmem;                                                           //  incr. allocated memory
   xcc = 0;                                                                //  string cc = 0
   *xpp = 0;                                                               //  string = null
}


xstring::xstring(const char *string)                                       //  new xstring("initial string")
{
   wmi = wmiv;
   xcc = 0;
   if (string) xcc = strlen(string);                                       //  string length
   xmem = (xcc & 0x7ffffff8) + 8;                                          //  mod 8 length
   xpp = new char[xmem];                                                   //  allocate
   if (! xpp) appcrash("xstring NEW failure",null);
   tcount++;                                                               //  incr. object count
   tmem += xmem;                                                           //  incr. allocated memory
   *xpp = 0;
   if (xcc) strcpy(xpp,string);                                            //  copy string
}


xstring::xstring(const xstring & xstr)                                     //  new xstring2(xstring1)
{
   wmi = wmiv;
   xmem = xstr.xmem;                                                       //  allocate same length
   xcc = xstr.xcc;
   xpp = new char[xmem];
   if (! xpp) appcrash("xstring NEW failure",null);
   tcount++;                                                               //  incr. object count
   tmem += xmem;                                                           //  incr. allocated memory
   strcpy(xpp,xstr.xpp);                                                   //  copy string
}


xstring::~xstring()                                                        //  delete xstring
{  
   validate();
   delete[] xpp;                                                           //  release allocated memory
   xpp = 0;
   tcount--;                                                               //  decr. object count
   tmem -= xmem;                                                           //  decr. allocated memory
   if (tcount < 0) appcrash("xstring count < 0",null);
   if (tmem < 0) appcrash("xstring memory < 0",null);
   if (tcount == 0 && tmem > 0) appcrash("xstring memory leak",null);
}


xstring xstring::operator= (const xstring & xstr)                          //  xstring2 = xstring1
{
   validate();
   xstr.validate();
   if (this == &xstr) return *this;
   xcc = xstr.xcc;
   if (xmem < xcc+1)
   {
      delete[] xpp;                                                        //  expand memory if needed
      tmem -= xmem;
      xmem = (xcc & 0x7ffffff8) + 8;                                       //  mod 8 length
      xpp = new char[xmem];
      if (! xpp) appcrash("xstring NEW failure",null);
      tmem += xmem;
   }
   strcpy(xpp,xstr.xpp);                                                   //  copy string
   return *this;
}


xstring xstring::operator= (const char *str)                               //  xstring = "some string"
{
   validate();
   xcc = 0;
   *xpp = 0;
   if (str) xcc = strlen(str);
   if (xmem < xcc+1)
   {
      delete[] xpp;                                                        //  expand memory if needed
      tmem -= xmem;
      xmem = (xcc & 0x7ffffff8) + 8;                                       //  mod 8 length
      xpp = new char[xmem];
      if (! xpp) appcrash("xstring NEW failure",null);
      tmem += xmem;
   }
   if (xcc) strcpy(xpp,str);                                               //  copy string
   return *this;
}


xstring operator+ (const xstring & x1, const xstring & x2)                 //  xstring1 + xstring2
{
   x1.validate();
   x2.validate();
   xstring temp(x1.xcc + x2.xcc);                                          //  build temp xstring
   strcpy(temp.xpp,x1.xpp);                                                //    with both input strings
   strcpy(temp.xpp + x1.xcc, x2.xpp);
   temp.xcc = x1.xcc + x2.xcc;
   temp.validate();
   return temp;
}


xstring operator+ (const xstring & x1, const char *s2)                     //  xstring + "some string"
{
   x1.validate();
   int cc2 = 0;
   if (s2) cc2 = strlen(s2);
   xstring temp(x1.xcc + cc2);                                             //  build temp xstring
   strcpy(temp.xpp,x1.xpp);                                                //    with both input strings
   if (s2) strcpy(temp.xpp + x1.xcc, s2);
   temp.xcc = x1.xcc + cc2;
   temp.validate();
   return temp;
}


xstring operator+ (const char *s1, const xstring & x2)                     //  "some string" + xstring
{
   x2.validate();
   int cc1 = 0;
   if (s1) cc1 = strlen(s1);
   xstring temp(cc1 + x2.xcc);                                             //  build temp xstring
   if (s1) strcpy(temp.xpp,s1);                                            //    with both input strings
   strcpy(temp.xpp + cc1, x2.xpp);
   temp.xcc = cc1 + x2.xcc;
   temp.validate();
   return temp;
}


void xstring::insert(int pos, const char *string, int cc)                  //  insert cc chars from string at pos
{                                                                          //  pad if pos > xcc or cc > string
   validate();

   int scc = strlen(string);
   if (! cc) cc = scc;

   int pad = pos - xcc;
   if (pad < 0) pad = 0;                                

   if (xmem < xcc + cc + pad + 1)                                          //  allocate more memory if needed
   {
      int newmem = xcc + cc + pad;
      newmem = (newmem & 0x7ffffff8) + 8;                                  //  mod 8 length
      char * xpp2 = new char[newmem];
      if (! xpp2) appcrash("xstring NEW failure",null);
      strcpy(xpp2,xpp);                                                    //  copy to new space
      delete[] xpp;
      xpp = xpp2;
      tmem += newmem - xmem;
      xmem = newmem;
   }

   if (pad) memset(xpp+xcc,' ',pad);                                       //  add blanks up to pos

   for (int ii = xcc + pad; ii >= pos; ii--)                               //  make hole for inserted string
           *(xpp+ii+cc) = *(xpp+ii);

   if (cc > scc) memset(xpp+pos+scc,' ',cc-scc);                           //  blank pad if cc > string
   if (cc < scc) scc = cc;
   strncpy(xpp+pos,string,scc);                                            //  insert string, without null

   xcc += cc + pad;                                                        //  set new length
   xpp[xcc] = 0;
   validate();
}


void xstring::overlay(int pos, const char *string, int cc)                 //  overlay substring
{
   validate();

   int scc = strlen(string);
   if (! cc) cc = scc;

   if (xmem < pos + cc + 1)                                                //  allocate more memory if needed
   {
      int newmem = pos + cc;
      newmem = (newmem & 0x7ffffff8) + 8;                                  //  mod 8 length
      char * xpp2 = new char[newmem];
      if (! xpp2) appcrash("xstring NEW failure",null);
      strcpy(xpp2,xpp);                                                    //  copy to new space
      delete[] xpp;
      xpp = xpp2;
      tmem += newmem - xmem;
      xmem = newmem;
   }

   if (pos > xcc) memset(xpp+xcc,' ',pos-xcc);                             //  add blanks up to pos
   
   if (cc > scc) memset(xpp+pos+scc,' ',cc-scc);                           //  blank pad if cc > string
   if (cc < scc) scc = cc;
   strncpy(xpp+pos,string,scc);                                            //  insert string, without null

   if (pos + cc > xcc) xcc = pos + cc;                                     //  set new length 
   xpp[xcc] = 0;
   validate();
}


void xstring::getStats(int & tcount2, int & tmem2)                         //  get statistics
{
   tcount2 = tcount;
   tmem2 = tmem;
}


void xstring::validate() const                                             //  validate integrity
{
   if (wmi != wmiv) appcrash("xstring bad wmi",null);
   if (xmem < xcc+1) appcrash("xstring xmem < xcc+1",null);
   if (xcc != (int) strlen(xpp)) appcrash("xstring xcc != strlen(xpp)",null);
}


/**************************************************************************
      Vxstring class (array or vector of xstring)
***************************************************************************/

Vxstring::Vxstring(int ii)                                                 //  constructor
{  
   pdata = 0;
   nd = ii;
   if (nd) pdata = new xstring[nd];
   if (nd && !pdata) appcrash("Vxstring NEW fail",null);
}


Vxstring::~Vxstring()                                                      //  destructor
{
   if (nd) delete[] pdata;
   pdata = 0;
   nd = 0;
}


Vxstring::Vxstring(const Vxstring & pold)                                  //  copy constructor
{
   pdata = 0;
   nd = pold.nd;                                                           //  set size
   if (nd) pdata = new xstring[nd];                                        //  allocate memory
   if (nd && !pdata) appcrash("Vxstring NEW fail");
   for (int ii = 0; ii < nd; ii++) pdata[ii] = pold[ii];                   //  copy defined elements
}


Vxstring Vxstring::operator= (const Vxstring & vdstr)                      //  operator =
{
   if (nd) delete[] pdata;                                                 //  delete old memory
   pdata = 0;
   nd = vdstr.nd;
   if (nd) pdata = new xstring[nd];                                        //  allocate new memory
   if (nd && !pdata) appcrash("Vxstring NEW fail",null);
   for (int ii = 0; ii < nd; ii++) pdata[ii] = vdstr.pdata[ii];            //  copy elements
   return *this;
}


xstring & Vxstring::operator[] (int ii)                                    //  operator []
{
   static xstring xnull(0);
   if (ii < nd) return pdata[ii];                                          //  return reference
   appcrash("Vxstring index invalid %d %d",nd,ii,null);
   return xnull;
}


const xstring & Vxstring::operator[] (int ii) const                        //  operator []
{
   static xstring xnull(0);
   if (ii < nd) return pdata[ii];                                          //  return reference
   appcrash("Vxstring index invalid %d %d",nd,ii,null);
   return xnull;
}


int Vxstring::search(const char *string)                                   //  find element in unsorted Vxstring
{
   for (int ii = 0; ii < nd; ii++)
        if (strEqu(pdata[ii],string)) return ii;
   return -1;
}


int Vxstring::bsearch(const char *string)                                  //  find element in sorted Vxstring
{                                                                          //   (binary search)
   int   nn, ii, jj, kk, rkk;

   nn = nd;
   if (! nn) return 0;                                                     //  empty list

   ii = nn / 2;                                                            //  next element to search
   jj = (ii + 1) / 2;                                                      //  next increment
   nn--;                                                                   //  last element
   rkk = 0;

   while (1)
   {
      kk = strcmp(pdata[ii],string);                                       //  check element

      if (kk > 0) 
      {
         ii -= jj;                                                         //  too high, go down
         if (ii < 0) return -1;
      }

      else if (kk < 0) 
      {
         ii += jj;                                                         //  too low, go up
         if (ii > nn) return -1;
      }

      else if (kk == 0) return ii;                                         //  matched

      jj = jj / 2;                                                         //  reduce increment

      if (jj == 0) 
      {
         jj = 1;                                                           //  step by 1 element
         if (! rkk) rkk = kk;                                              //  save direction
         else 
         {
            if (rkk > 0) { if (kk < 0) return -1; }                        //  if change direction, fail
            else if (kk > 0) return -1;
         }
      }
   }
}


static int  VDsortKeys[10][3], VDsortNK;

int Vxstring::sort(int NK, int keys[][3])                                  //  sort elements by subfields
{                                                                          //  key[ii][0] = position
   int     NR, RL, ii;                                                     //         [1] = length
   HeapSortUcomp  VDsortComp;                                              //         [2] = 1/2 = ascending/desc.
                                                                           //             = 3/4 =  + ignore case
   NR = nd;
   if (NR < 2) return 1;

   RL = sizeof(xstring);

   if (NK < 1) appcrash("Vxstring::sort, bad NK",null);
   if (NK > 10) appcrash("Vxstring::sort, bad NK",null);
   VDsortNK = NK;

   for (ii = 0; ii < NK; ii++)
   {
      VDsortKeys[ii][0] = keys[ii][0];
      VDsortKeys[ii][1] = keys[ii][1];
      VDsortKeys[ii][2] = keys[ii][2];
   }

   HeapSort((char *) pdata,RL,NR,VDsortComp);

   return 1;
}


int VDsortComp(const char *r1, const char *r2)
{
   xstring      *d1, *d2;
   const char   *p1, *p2;
   int          ii, stat, kpos, ktype, kleng;

   d1 = (xstring *) r1;
   d2 = (xstring *) r2;
   p1 = *d1;
   p2 = *d2;

   for (ii = 0; ii < VDsortNK; ii++)                                       //  compare each key
   {
      kpos = VDsortKeys[ii][0];
      kleng = VDsortKeys[ii][1];
      ktype = VDsortKeys[ii][2];

      if (ktype == 1)
      {
         stat = strncmp(p1+kpos,p2+kpos,kleng);
         if (stat) return stat;
         continue;
      }

      else if (ktype == 2)
      {
         stat = strncmp(p1+kpos,p2+kpos,kleng);
         if (stat) return -stat;
         continue;
      }

      else if (ktype == 3)
      {
         stat = strncasecmp(p1+kpos,p2+kpos,kleng);
         if (stat) return stat;
         continue;
      }

      else if (ktype == 4)
      {
         stat = strncasecmp(p1+kpos,p2+kpos,kleng);
         if (stat) return -stat;
         continue;
      }

      appcrash("Vxstring::sort, bad KEYS sort type",null);
   }

   return 0;
}


int Vxstring::sort(int pos, int cc)                                        //  sort elements ascending
{
   int   key[3];

   if (! cc) cc = 999999;
   key[0] = pos;
   key[1] = cc;
   key[2] = 1;

   sort(1,&key);

   return 1;
}


/**************************************************************************
     Hash Table class
***************************************************************************/

//  static members (robust for tables up to 60% full)

int HashTab::trys1 = 100;                                                  //  Add() tries
int HashTab::trys2 = 200;                                                  //  Find() tries


HashTab::HashTab(int _cc, int _cap)                                        //  constructor
{
   cc = 4 * (_cc + 4) / 4;                                                 //  + 1 + mod 4 length
   cap = _cap;
   int len = cc * cap;
   table = new char [len];
   if (! table) appcrash("HashTab() new %d fail",len,null);
   memset(table,0,len);
}


HashTab::~HashTab()                                                        //  destructor
{
   delete [] table;
   table = 0;
}


//  Add a new string to table

int HashTab::Add(const char *string)
{
   int   pos, fpos, trys;

   pos = strHash(string,cap);                                              //  get random position
   pos = pos * cc;

   for (trys = 0, fpos = -1;                                               //  find next free slot 
        trys < trys1;                                                      //   at/after position
        trys++, pos += cc)
   {
      if (pos >= cap * cc) pos = 0;                                        //  last position wraps to 1st

      if (! table[pos])                                                    //  empty slot: string not found
      {
         if (fpos != -1) pos = fpos;                                       //  use prior deleted slot if there
         strncpy(table+pos,string,cc);                                     //  insert new string
         table[pos+cc-1] = 0;                                              //  insure null terminator
         return (pos/cc);                                                  //  return rel. table entry
      }

      if (table[pos] == -1)                                                //  deleted slot
      {
         if (fpos == -1) fpos = pos;                                       //  remember 1st one found
         continue;
      }

      if (strEqu(string,table+pos)) return -2;                             //  string already present
   }   

   return -3;                                                              //  table full (trys1 exceeded)
}


//  Delete a string from table

int HashTab::Del(const char *string)
{
   int   pos, trys;

   pos = strHash(string,cap);                                              //  get random position
   pos = pos * cc;

   for (trys = 0;                                                          //  search for string
        trys < trys2;                                                      //   at/after position
        trys++, pos += cc)
   {
      if (pos >= cap * cc) pos = 0;                                        //  last position wraps to 1st

      if (! table[pos]) return -1;                                         //  empty slot, string not found

      if (strEqu(string,table+pos))                                        //  string found
      {
         table[pos] = -1;                                                  //  delete table entry
         return (pos/cc);                                                  //  return rel. table entry
      }
   }   

   appcrash("HashTab::Del() bug",null);                                    //  exceed trys2, must not happen
   return 0;                                                               //  (table too full to function)
}


//  Find a table entry.

int HashTab::Find(const char *string)
{
   int   pos, trys;

   pos = strHash(string,cap);                                              //  get random position
   pos = pos * cc;

   for (trys = 0;                                                          //  search for string
        trys < trys2;                                                      //   at/after position
        trys++, pos += cc)
   {
      if (pos >= cap * cc) pos = 0;                                        //  last position wraps to 1st
      if (! table[pos]) return -1;                                         //  empty slot, string not found
      if (strEqu(string,table+pos)) return (pos/cc);                       //  string found, return rel. entry
   }   

   appcrash("HashTab::Find() bug",null);                                   //  cannot happen
   return 0;
}


//  return first or next table entry

int HashTab::GetNext(int & ftf, char *string)
{
   static int    pos;

   if (ftf)                                                                //  initial call
   {
      pos = 0;
      ftf = 0;
   }

   while (pos < (cap * cc))
   {
      if ((table[pos] == 0) || (table[pos] == -1))
      {
         pos += cc;
         continue;
      }

      strcpy(string,table+pos);                                            //  return string
      pos += cc;
      return 1;
   }

   return -4;                                                              //  EOF
}


int HashTab::Dump()
{
   int   ii, pos;

   for (ii = 0; ii < cap; ii++)
   {
      pos = ii * cc;
      if (table[pos] > 0) printf("%d %s \n", pos, table + pos);
      if (table[pos] == -1) printf("%d deleted \n", pos);
   }        
   return 1;
}


/**************************************************************************
     class for queue of dynamic strings
***************************************************************************/

Queue::Queue(int cap)                                                      //  constructor
{
   int   err;
   
   err = mutex_init(&qmutex, 0);                                           //  create mutex = queue lock
   if (err) appcrash("Queue(), mutex init fail",null);

   qcap = cap;                                                             //  queue capacity
   ent1 = entN = qcount = 0;                                               //  state = empty
   vd = new Vxstring(qcap);                                                //  create vector of xstring's
   if (! vd) appcrash("Queue(), NEW fail %d",cap,null);
   strcpy(wmi,"queue");
   return;
}


Queue::~Queue()                                                            //  destructor
{
   if (strNeq(wmi,"queue")) appcrash("~Queue wmi fail",null);
   wmi[0] = 0;
   mutex_destroy(&qmutex);                                                 //  destroy mutex
   qcount = qcap = ent1 = entN = -1;
   delete vd;
   vd = 0;
   return;
}


void Queue::lock()                                                         //  lock queue (private)
{
   int   err;
   err = mutex_lock(&qmutex);                                              //  reserve mutex or suspend until avail.
   if (err) appcrash("Queue mutex lock fail",null);
   return;
}


void Queue::unlock()                                                       //  unlock queue (private)
{
   int   err;
   err = mutex_unlock(&qmutex);                                            //  release mutex
   if (err) appcrash("Queue mutex unlock fail",null);
   return;
}


int Queue::getCount()                                                      //  get current entry count
{
   if (strNeq(wmi,"queue")) appcrash("Queue getCount wmi fail",null);
   return qcount;
}


int Queue::push(const xstring *newEnt, double wait)                        //  add entry to queue, with max. wait
{
   double  elaps = 0.0;
   int     count;
   
   if (strNeq(wmi,"queue")) appcrash("Queue::push wmi fail",null);

   lock();                                                                 //  lock queue
   while (qcount == qcap) {                                                //  queue full
      unlock();                                                            //  unlock queue
      if (elaps >= wait) return -1;                                        //  too long, return -1 status
      usleep(1000);                                                        //  sleep in 1 millisec. steps
      elaps += 0.001;                                                      //  until queue not full
      lock();                                                              //  lock queue
   }

   (* vd)[entN] = *newEnt;                                                 //  copy new entry into queue
   entN++;                                                                 //  incr. end pointer
   if (entN == qcap) entN = 0;
   qcount++;                                                               //  incr. queue count
   count = qcount;
   unlock();                                                               //  unlock queue
   return count;                                                           //  return curr. queue count
}


xstring *Queue::pop1()                                                     //  get 1st (oldest) entry and remove
{
   xstring    *entry;
   
   if (strNeq(wmi,"queue")) appcrash("Queue::pop1 wmi fail",null);

   lock();                                                                 //  lock queue

   if (qcount == 0) entry = 0;                                             //  queue empty
   else {
      entry = &(* vd)[ent1];                                               //  get first entry
      ent1++;                                                              //  index pointer to next
      if (ent1 == qcap) ent1 = 0;
      qcount--;                                                            //  decr. queue count
   }

   unlock();                                                               //  unlock queue
   return entry;
}


xstring *Queue::popN()                                                     //  get last (newest) entry and remove
{
   xstring   *entry;
   
   if (strNeq(wmi,"queue")) appcrash("Queue::popN wmi fail",null);

   lock();                                                                 //  lock queue

   if (qcount == 0) entry = 0;                                             //  queue empty
   else {
      if (entN == 0) entN = qcap;                                          //  index pointer to prior
      entN--;
      qcount--;                                                            //  decr. queue count
      entry = &(* vd)[entN];                                               //  get last entry
   }

   unlock();                                                               //  unlock queue
   return entry;
}


/**************************************************************************

   Tree class, tree-structured data storage without limits

   Store any amount of data at any depth within a tree-structure with named nodes.
   Data can be found using an ordered list of node names or node numbers.

   Node numbers are in the sequence added using put() with names,
   or the same as those numbers used in put() with numbers.

   Internal code conventions: 
      - caller level is node 0, next level is node 1, etc.
      - node names and numbers in calls to get() and put() refer to next levels
      - number of levels = 1+nn, where nn is max. in calls to put(...nodes[], nn)

***************************************************************************/

#define wmid 1374602859


//  constructor

Tree::Tree(char *name)
{
   wmi = wmid;
   tname = 0;
   tmem = 0;
   tdata = 0;
   nsub = 0;
   psub = 0;

   if (name) 
   {
      int cc = strlen(name);
      tname = new char[cc+1];
      if (! tname) appcrash("Tree, no memory",null);
      strcpy(tname,name);
   }
}


//  destructor

Tree::~Tree()
{
   if (wmi != wmid) appcrash("not a Tree",null);
   if (tname) delete [] tname;
   tname = 0;
   if (tmem) free(tdata);
   tmem = 0;
   tdata = 0;
   for (int ii = 0; ii < nsub; ii++) delete psub[ii];
   if (psub) free(psub);
   nsub = 0;
   psub = 0;
}


//  put data by node names[]

int Tree::put(void *data, int dd, char *nodes[], int nn)
{
   Tree    *tnode;
   
   if (wmi != wmid) appcrash("not a Tree",null);
   tnode = make(nodes,nn);
   if (tnode->tdata) free(tnode->tdata);
   tnode->tdata = new char[dd];
   if (! tnode->tdata) appcrash("Tree, no memory",null);
   tnode->tmem = dd;
   memmove(tnode->tdata,data,dd);
   return 1;
}


//  put data by node numbers[]

int Tree::put(void *data, int dd, int nodes[], int nn)
{
   Tree    *tnode;
   
   if (wmi != wmid) appcrash("not a Tree",null);
   tnode = make(nodes,nn);
   if (tnode->tdata) free(tnode->tdata);
   tnode->tdata = new char[dd];
   if (! tnode->tdata) appcrash("Tree, no memory",null);
   tnode->tmem = dd;
   memmove(tnode->tdata,data,dd);
   return 1;
}


//  get data by node names[]

int Tree::get(void *data, int dd, char *nodes[], int nn)
{
   Tree *tnode = find(nodes,nn);
   if (! tnode) return 0;
   if (! tnode->tmem) return 0;
   if (dd > tnode->tmem) dd = tnode->tmem;
   memmove(data,tnode->tdata,dd);
   return dd;
}


//  get data by node numbers[]

int Tree::get(void *data, int dd, int nodes[], int nn)
{
   Tree *tnode = find(nodes,nn);
   if (! tnode) return 0;
   if (! tnode->tmem) return 0;
   if (dd > tnode->tmem) dd = tnode->tmem;
   memmove(data,tnode->tdata,dd);
   return dd;
}


//  find a given node by names[]

Tree * Tree::find(char *nodes[], int nn)
{
   int      ii;
   
   for (ii = 0; ii < nsub; ii++)
      if (psub[ii]->tname && strEqu(nodes[0],psub[ii]->tname)) break;
   if (ii == nsub) return 0;
   if (nn == 1) return psub[ii];
   return psub[ii]->find(&nodes[1],nn-1);
}


//  find a given node by numbers[]

Tree * Tree::find(int nodes[], int nn)
{
   int ii = nodes[0];
   if (ii >= nsub) return 0;
   if (! psub[ii]) return 0;
   if (nn == 1) return psub[ii];
   return psub[ii]->find(&nodes[1],nn-1);
}


//  find or create a given node by names[]

Tree * Tree::make(char *nodes[], int nn)
{
   int      ii;
   Tree   **psub2;
   
   for (ii = 0; ii < nsub; ii++)
      if (psub[ii]->tname && strEqu(nodes[0],psub[ii]->tname)) break;

   if (ii == nsub)
   {
      psub2 = new Tree * [nsub+1];
      if (! psub2) appcrash("Tree, no memory",null);
      for (ii = 0; ii < nsub; ii++) psub2[ii] = psub[ii];
      delete [] psub;
      psub = psub2;
      nsub++;
      psub[ii] = new Tree(nodes[0]);
      if (! psub[ii]) appcrash("Tree, no memory",null);
   }

   if (nn == 1) return psub[ii];
   return psub[ii]->make(&nodes[1],nn-1);
}


//  find or create a given node by numbers[]

Tree * Tree::make(int nodes[], int nn)
{
   Tree   **psub2;
   int      ii, jj;

   ii = nodes[0];
   if ((ii < nsub) && psub[ii])
   {
      if (nn == 1) return psub[ii];
      return psub[ii]->make(&nodes[1],nn-1);
   }

   if (ii >= nsub)
   {
      psub2 = new Tree * [ii+1];
      if (! psub2) appcrash("Tree, no memory",null);
      for (jj = 0; jj < nsub; jj++) psub2[jj] = psub[jj];
      for (jj = nsub; jj < ii; jj++) psub2[jj] = 0;
      delete [] psub;
      psub = psub2;
      nsub = ii + 1;
   }

   psub[ii] = new Tree("noname");
   if (! psub[ii]) appcrash("Tree, no memory",null);

   if (nn == 1) return psub[ii];
   return psub[ii]->make(&nodes[1],nn-1);
}


//  dump tree data to stdout (call with level 0)

void Tree::dump(int level)
{
   char    *name;
   
   if (! tname) name = "noname";
   else name = tname;
   printf("%*s level: %d  name: %s  subs: %d  mem: %d \n",
                        level*2,"",level,name,nsub,tmem);
   for (int ii = 0; ii < nsub; ii++) 
         if (psub[ii]) psub[ii]->dump(level+1);
}


//  get node counts and total data per level
//  level 0 + nn more levels, as given in calls to put(...nodes[],nn)
//  caller must initialize counters to zero

void Tree::stats(int nn[], int nd[])
{
   nn[0] += 1;
   nd[0] += tmem;
   for (int ii = 0; ii < nsub; ii++) 
      if (psub[ii]) psub[ii]->stats(&nn[1],&nd[1]);
}



